# windeployqt-to-nsh
#
# Windeployqt-to-nsh - Convert the output of windeployqt to an equivalent set of
# NSIS "File" function calls.
#
# Copyright 2014 Gerald Combs <gerald@wireshark.org>
#
# Wireshark - Network traffic analyzer
# By Gerald Combs <gerald@wireshark.org>
# Copyright 1998 Gerald Combs
#
# SPDX-License-Identifier: GPL-2.0-or-later

#requires -version 2

<#
.SYNOPSIS
Creates NSIS "File" function calls required for Qt packaging.

.DESCRIPTION
This script creates an NSIS-compatible file based on the following Qt
versions:

  - 5.3 and later: A list of DLLs and directories based on the output of the
    "windeployqt" utility. Windeployqt lists the DLLs required to run a Qt
    application. (The initial version that shipped with Qt 5.2 is unusable.)

  - 5.2 and earlier: A hard-coded list of Qt DLLs and directories appropriate
    for earlier Qt versions.

  - None: A dummy file.

If building with Qt, QMake must be in your PATH.

.PARAMETER Executable
The path to a Qt application. It will be examined for dependent DLLs.

.PARAMETER FilePath
Output filename.

.PARAMETER DebugConfig
Assume debug binaries.

.INPUTS
-Executable Path to the Qt application.
-FilePath Output NSIS file.

.OUTPUTS
List of NSIS commands required to package supporting DLLs.

.EXAMPLE
C:\PS> .\windeployqt-to-nsis.ps1 windeployqt.exe ..\..\staging\wireshark.exe qt-dll-manifest.nsh [-DebugConfig]
#>

Param(
    [Parameter(Mandatory=$true, Position=0)]
    [String] $Executable,

    [Parameter(Position=1)]
    [String] $FilePath = "qt-dll-manifest.nsh",

    [Parameter(Mandatory=$false)]
    [Switch] $DebugConfig
)


try {
    $qtVersion = [version](qmake -query QT_VERSION)
    $nsisCommands = @("# Qt version " + $qtVersion ; "#")

    if ($qtVersion -ge "5.3") {
        # Qt 5.3 or later. Windeployqt is present and works

        $DebugOrRelease = If ($DebugConfig) {"--debug"} Else {"--release"}

        # windeployqt lists translation files that it don't exist (e.g.
        # qtbase_ar.qm), so we handle those by hand.
        # https://bugreports.qt.io/browse/QTBUG-65974
        $wdqtList = windeployqt `
            $DebugOrRelease `
            --no-compiler-runtime `
            --no-translations `
            --list relative `
            $Executable

        $basePath = Split-Path -Parent $Executable
        $currentDir = ""

        foreach ($entry in $wdqtList) {
            $dir = Split-Path -Parent $entry
            if ($dir -and $dir -ne $currentDir) {
                $nsisCommands += "SetOutPath `"`$INSTDIR\$dir`""
                $currentDir = $dir
            }
            $nsisCommands += "File `"$basePath\$entry`""
        }
    } elseif ($qtVersion -ge "5.0") {
        # Qt 5.0 - 5.2. Windeployqt is buggy or not present

        $nsisCommands += @"
File "..\..\wireshark-qt-release\Qt5Core.dll"
File "..\..\wireshark-qt-release\Qt5Gui.dll"
File "..\..\wireshark-qt-release\Qt5Widgets.dll"
File "..\..\wireshark-qt-release\Qt5PrintSupport.dll"
File /r "..\..\wireshark-qt-release\platforms"
"@

    }
}

catch {

    $nsisCommands = @"
# Qt not configured
#
"@

}

Set-Content $FilePath @"
#
# Automatically generated by $($MyInvocation.MyCommand.Name)
#
"@

Add-Content $FilePath $nsisCommands
