// Copyright (c) 2015, Sandia Corporation.
// Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
// the U.S. Government retains certain rights in this software.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//     * Redistributions of source code must retain the above copyright
//       notice, this list of conditions and the following disclaimer.
//
//     * Redistributions in binary form must reproduce the above
//       copyright notice, this list of conditions and the following
//       disclaimer in the documentation and/or other materials provided
//       with the distribution.
//
//     * Neither the name of Sandia Corporation nor the names of its
//       contributors may be used to endorse or promote products derived
//       from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

#include "FaceTestingUtils.hpp"
#include "ioUtils.hpp"
#include <stk_mesh/base/MetaData.hpp>
#include <stk_mesh/base/BulkData.hpp>
#include <stk_mesh/base/SkinBoundary.hpp>
#include <stk_mesh/base/Field.hpp>
#include <stk_mesh/baseImpl/ForEachEntityLoopAbstractions.hpp>
#include <stk_mesh/base/GetEntities.hpp>
#include <stk_mesh/base/CreateFaces.hpp>
#include <stk_io/StkMeshIoBroker.hpp>
#include <stk_util/diag/StringUtil.hpp>
#include <stk_mesh/base/FEMHelpers.hpp>

unsigned count_sides_in_mesh(const stk::mesh::BulkData& mesh)
{
    std::vector<size_t> countVec;
    stk::mesh::count_entities(mesh.mesh_meta_data().universal_part(), mesh, countVec);
    return countVec[mesh.mesh_meta_data().side_rank()];
}

unsigned read_file_create_faces_count_sides(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    stk::mesh::create_all_sides(mesh, meta.universal_part(), {}, false);
    return count_sides_in_mesh(mesh);
}

unsigned read_file_count_sides(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    return count_sides_in_mesh(mesh);
}

bool is_face_fully_connected(const stk::mesh::BulkData& mesh, stk::mesh::MeshIndex faceMeshIndex)
{
    const unsigned num_expected_faces = faceMeshIndex.bucket->topology().num_faces();
    if (num_expected_faces != mesh.num_faces(stk::mesh::impl::get_entity(faceMeshIndex)))
        return false;
    return true;
}

bool fully_connected_elements_to_faces(const stk::mesh::BulkData& mesh)
{
    bool fully_connected = true;
    stk::mesh::impl::for_each_entity_run(mesh, stk::topology::ELEMENT_RANK,
        [&fully_connected](const stk::mesh::BulkData& mesh, const stk::mesh::MeshIndex& meshIndex)
        {
          fully_connected &= is_face_fully_connected(mesh,meshIndex);
        }
    );
    return fully_connected;
}

unsigned read_file_create_faces_fully_connected_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    stk::mesh::create_all_sides(mesh, meta.universal_part(), {}, false);
    return fully_connected_elements_to_faces(mesh);
}

unsigned read_file_fully_connected_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    return fully_connected_elements_to_faces(mesh);
}

bool is_face_shared_between_different_elements(const stk::mesh::BulkData& mesh, stk::mesh::Entity face)
{
    stk::mesh::Entity const * elements = mesh.begin_elements(face);
    for (unsigned elem_count = 0; elem_count < mesh.num_elements(face); ++elem_count)
        for (unsigned other_elem_count = elem_count; other_elem_count < mesh.num_elements(face); ++other_elem_count)
            if ((elem_count != other_elem_count) && (elements[elem_count] != elements[other_elem_count]))
                return true;
    return false;
}

unsigned count_shared_faces_between_different_elements(const stk::mesh::BulkData& mesh)
{
    unsigned shared_face_count = 0;
    stk::mesh::impl::for_each_entity_run(mesh, stk::topology::FACE_RANK,
        [&shared_face_count](const stk::mesh::BulkData& mesh, const stk::mesh::MeshIndex& meshIndex)
        {
          if (is_face_shared_between_different_elements(mesh,stk::mesh::impl::get_entity(meshIndex)))
              ++shared_face_count;
        }
    );
    return shared_face_count;
}

unsigned read_file_create_faces_shared_faces_different_elements_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    stk::mesh::create_all_sides(mesh, meta.universal_part(), {}, false);
    return count_shared_faces_between_different_elements(mesh);
}

unsigned read_file_shared_faces_different_elements_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    return count_shared_faces_between_different_elements(mesh);
}

bool is_face_shared_between_same_element(const stk::mesh::BulkData& mesh, stk::mesh::Entity face)
{
    stk::mesh::Entity const * elements = mesh.begin_elements(face);
    for (unsigned elem_count = 0; elem_count < mesh.num_elements(face); ++elem_count)
        for (unsigned other_elem_count = elem_count; other_elem_count < mesh.num_elements(face); ++other_elem_count)
            if ((elem_count != other_elem_count) && (elements[elem_count] == elements[other_elem_count]))
                return true;
    return false;
}

unsigned count_shared_faces_between_same_element(const stk::mesh::BulkData& mesh)
{
    unsigned shared_face_count = 0;
    stk::mesh::impl::for_each_entity_run(mesh, stk::topology::FACE_RANK,
      [&shared_face_count](const stk::mesh::BulkData& mesh, const stk::mesh::MeshIndex& meshIndex)
      {
        stk::mesh::Entity face = stk::mesh::impl::get_entity(meshIndex);

        if (is_face_shared_between_same_element(mesh,face))
            ++shared_face_count;
      }
    );
    return shared_face_count;
}

unsigned read_file_create_faces_shared_faces_same_elements_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    stk::mesh::create_all_sides(mesh, meta.universal_part(), {}, false);
    return count_shared_faces_between_same_element(mesh);
}

unsigned read_file_shared_faces_same_elements_stk(std::string filename)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    return count_shared_faces_between_same_element(mesh);
}

bool is_node_not_at_x_equal_half(const stk::mesh::BulkData& mesh, stk::mesh::Entity node)
{
    double *xyz = static_cast<double *>(stk::mesh::field_data(*mesh.mesh_meta_data().coordinate_field(), node));
    return (xyz[0] != 0.5);
}

bool is_face_at_x_equal_half(const stk::mesh::BulkData& mesh, stk::mesh::Entity face)
{
    stk::mesh::Entity const * node = mesh.begin_nodes(face);
    for (unsigned node_count = 0; node_count < mesh.num_nodes(face); ++node_count)
        if (is_node_not_at_x_equal_half(mesh,node[node_count]))
            return false;
    return true;
}

stk::mesh::EntityVector get_faces_at_x_equal_half(const stk::mesh::BulkData& mesh)
{
    stk::mesh::EntityVector faces_at_x_equal_half;
    stk::mesh::impl::for_each_entity_run(mesh, stk::topology::FACE_RANK,
      [&faces_at_x_equal_half](const stk::mesh::BulkData& mesh, const stk::mesh::MeshIndex& meshIndex)
      {
        if (is_face_at_x_equal_half(mesh,stk::mesh::impl::get_entity(meshIndex)))
            faces_at_x_equal_half.push_back(stk::mesh::impl::get_entity(meshIndex));
      }
    );
    return faces_at_x_equal_half;
}

std::set<unsigned> get_face_connectivity_at_x_equal_half(const stk::mesh::BulkData& mesh)
{
    std::set<unsigned> faces;
    for(stk::mesh::Entity face : get_faces_at_x_equal_half(mesh))
        faces.insert(mesh.num_elements(face));
    return faces;
}

std::ostream& operator<<(std::ostream& os, const std::set<unsigned>& data)
{
    os << "{ " << stk::util::join(data.begin(),data.end(),", ") << " }";
    return os;
}

bool check_face_elem_connectivity(const stk::mesh::BulkData& mesh, const std::set<unsigned>& gold_faces)
{
    std::set<unsigned> current_faces = get_face_connectivity_at_x_equal_half(mesh);
    if (current_faces == gold_faces) {
        return true;
    }
    std::cout << "gold_faces = " << gold_faces << ", current_faces = " << current_faces << std::endl;
    return false;
}

bool read_file_create_faces_check_face_elem_connectivity_stk(std::string filename, const std::set<unsigned>& counts)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    stk::mesh::create_all_sides(mesh, meta.universal_part(), {}, false);
    return check_face_elem_connectivity(mesh, counts);

}

bool read_file_check_face_elem_connectivity_stk(std::string filename, const std::set<unsigned>& counts)
{
    stk::mesh::MetaData meta;
    stk::mesh::BulkData mesh(meta, MPI_COMM_WORLD);
    stk::io::fill_mesh(filename, mesh);
    return check_face_elem_connectivity(mesh, counts);

}

namespace stk
{
namespace unit_test_util
{

stk::mesh::Entity declare_element_side_with_nodes(stk::mesh::BulkData &mesh,
                                                  stk::mesh::Entity elem,
                                                  const stk::mesh::EntityVector &nodes,
                                                  stk::mesh::EntityId globalId,
                                                  stk::mesh::Part &part)
{
    std::pair<stk::mesh::ConnectivityOrdinal, stk::mesh::Permutation> ordinalAndPermutation = get_ordinal_and_permutation(mesh, elem, mesh.mesh_meta_data().side_rank(), nodes);
    return mesh.declare_element_side(elem, ordinalAndPermutation.first, stk::mesh::ConstPartVector{&part});
}

stk::mesh::Entity declare_element_to_edge_with_nodes(stk::mesh::BulkData &mesh, stk::mesh::Entity elem, const stk::mesh::EntityVector &sub_topology_nodes,
        stk::mesh::EntityId global_sub_topology_id, stk::mesh::Part &part)
{
    std::pair<stk::mesh::ConnectivityOrdinal, stk::mesh::Permutation> ordinalAndPermutation =
            get_ordinal_and_permutation(mesh, elem, stk::topology::EDGE_RANK, sub_topology_nodes);

    if((ordinalAndPermutation.first == stk::mesh::ConnectivityOrdinal::INVALID_CONNECTIVITY_ORDINAL) || (ordinalAndPermutation.second
            == stk::mesh::Permutation::INVALID_PERMUTATION))
    {
        stk::mesh::Entity invalid;
        invalid = stk::mesh::Entity::InvalidEntity;
        return invalid;
    }

    stk::mesh::Entity side = mesh.get_entity(stk::topology::EDGE_RANK, global_sub_topology_id);
    if(!mesh.is_valid(side))
    {
        side = mesh.declare_edge(global_sub_topology_id, {&part});
        for(unsigned i = 0; i < sub_topology_nodes.size(); ++i)
            mesh.declare_relation(side, sub_topology_nodes[i], i);
    }
    else
    {
        const stk::mesh::Entity* sideNodes = mesh.begin_nodes(side);
        unsigned numNodes = mesh.num_nodes(side);
        ThrowRequireMsg(sub_topology_nodes.size() == numNodes,
                        "declare_element_to_sub_topology_with_nodes ERROR, side already exists with different number of nodes");
        for(unsigned i = 0; i < numNodes; ++i)
        {
            ThrowRequireMsg(sub_topology_nodes[i] == sideNodes[i],
                            "declare_element_to_sub_topology_with_nodes ERROR, side already exists with different node connectivity");
        }
    }

    mesh.declare_relation(elem, side, ordinalAndPermutation.first, ordinalAndPermutation.second);
    return side;
}
}}
