#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: MBI

  Description: @{shortdesc}@
    @{longdesc}@

  Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: @{p2pfeature}@
  P2P!nonblocking: @{ip2pfeature}@
  P2P!persistent: Lacking
  COLL!basic: Lacking
  COLL!nonblocking: Lacking
  COLL!persistent: Lacking
  COLL!tools: Lacking
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 3 ${EXE}
  | @{outcome}@
  | @{errormsg}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>

#define N 10

int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;
  int dest, src;
  int i=0;
  int root = 0;
  int stag = 0, rtag = 0;
  int buff_size = 1;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs < 3)
    printf("MBI ERROR: This test needs at least 3 processes to produce a bug!\\n");

  int dbs = sizeof(int)*nprocs; /* Size of the dynamic buffers for alltoall and friends */
  MPI_Comm newcom = MPI_COMM_WORLD;
  MPI_Datatype type = MPI_INT;
  MPI_Op op = MPI_SUM;

  @{init0}@
  @{init1a}@
  @{init1b}@
  @{init2}@

  if (rank == 0) {
    dest = 1; stag = 1;
    @{operation0}@
    @{fini0}@
  } else if (rank == 1) {
    src = MPI_ANY_SOURCE;
    rtag = @{tag1}@;
    @{operation1a}@
    @{fini1a}@
    rtag = @{tag2}@;
    @{operation1b}@ @{tagerror}@
    @{fini1b}@
  } else if (rank == 2) {
    dest = 1; stag = 2;
    @{operation2}@
    @{fini2}@
  }

  @{free0}@
  @{free1a}@
  @{free1b}@
  @{free2}@

  MPI_Finalize();
  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""

# To be correct, this benchmark must be use wildcard on second recv
# tag, or no wildcard on first recv tag and second recv tag must be
# different.
#
# |-----+-----+----+----|
# | x\y | ANY | 1  | 2  |
# |-----+-----+----+----|
# | ANY | OK  |  - |  - |
# |   1 | OK  |  - | OK |
# |   2 | OK  | OK |  - |
# |-----+-----+----+----|

for s in gen.send:
    for r in gen.recv:
        for x, y in [('MPI_ANY_TAG', 'MPI_ANY_TAG'), # OK
                     ('MPI_ANY_TAG', '1'),           # NOK
                     ('1', 'MPI_ANY_TAG'),           # OK
                     ('1', '2'),                     # OK
                     ('2', '2')]:                    # NOK
            patterns = {}
            patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
            patterns['p2pfeature'] = 'Yes' if s in gen.send or r in gen.recv else 'Lacking'
            patterns['ip2pfeature'] = 'Yes' if s in gen.isend or r in gen.irecv else 'Lacking'
            patterns['s'] = s
            patterns['r'] = r

            patterns['tag1'] = x
            patterns['tag2'] = y

            patterns['init0'] = gen.init[s]("0")
            patterns['operation0'] = gen.operation[s]("0")
            patterns['fini0'] = gen.fini[s]("0")
            patterns['free0'] = gen.free[s]("0")

            patterns['init1a'] = gen.init[r]("1a")
            patterns['init1b'] = gen.init[r]("1b")
            patterns['operation1a'] = gen.operation[r]("1a")
            patterns['operation1b'] = gen.operation[r]("1b")
            patterns['fini1a'] = gen.fini[r]("1a")
            patterns['fini1b'] = gen.fini[r]("1b")
            patterns['free1a'] = gen.free[r]("1a")
            patterns['free1b'] = gen.free[r]("1b")

            patterns['init2'] = gen.init[s]("2")
            patterns['operation2'] = gen.operation[s]("2")
            patterns['fini2'] = gen.fini[s]("2")
            patterns['free2'] = gen.free[s]("2")
            patterns['tagerror'] = '/* MBIERROR */'

            if y == 'MPI_ANY_TAG' or (x != 'MPI_ANY_TAG' and x != y):
                # Generate the correct matching because of the conditional
                replace = patterns.copy()
                replace['shortdesc'] = 'Message race'
                replace['longdesc'] = 'Correct code without message race.'
                replace['outcome'] = 'OK'
                replace['errormsg'] = 'OK'
                replace['tagerror'] = ''
                gen.make_file(template, f'MessageRace_tag_{x}_{y}_{s}_{r}_ok.c', replace)
            else:
                # Generate the incorrect matching because of the conditional
                replace = patterns.copy()
                replace['shortdesc'] = 'Message race'
                replace['longdesc'] = 'Message race in @{r}@ with @{s}@.'
                replace['outcome'] = 'ERROR: MessageRace'
                replace['errormsg'] = 'Message race. The use of wildcard receive calls @{r}@ at @{filename}@:@{line:MBIERROR}@ and incorrect tag matching.'
                gen.make_file(template, f'MessageRace_tag_{x}_{y}_{s}_{r}_nok.c', replace)
