#!/usr/bin/env python2
# -*- coding: utf-8 -*-

# SAT plugin to detect language (experimental)
# Copyright (C) 2009-2018 Jérôme Poisson (goffi@goffi.org)

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.

# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from sat.core.i18n import _
from sat.core import exceptions
from sat.core.constants import Const as C
from sat.core.log import getLogger
log = getLogger(__name__)
from sat.tools import utils
from sat.tools.common import uri as uri_parse
from twisted.internet import defer
from twisted.words.protocols.jabber import jid, error
from twisted.words.xish import domish
from wokkel import pubsub


PLUGIN_INFO = {
    C.PI_NAME: "Event plugin",
    C.PI_IMPORT_NAME: "EVENTS",
    C.PI_TYPE: "EXP",
    C.PI_PROTOCOLS: [],
    C.PI_DEPENDENCIES: ["XEP-0060"],
    C.PI_RECOMMENDATIONS: ["INVITATIONS", "XEP-0277"],
    C.PI_MAIN: "Events",
    C.PI_HANDLER: "no",
    C.PI_DESCRIPTION: _("""Experimental implementation of XMPP events management""")
}

NS_EVENT = 'org.salut-a-toi.event:0'


class Events(object):
    """Q&D module to handle event attendance answer, experimentation only"""

    def __init__(self, host):
        log.info(_(u"Event plugin initialization"))
        self.host = host
        self._p = self.host.plugins["XEP-0060"]
        self._i = self.host.plugins.get("INVITATIONS")
        self._b = self.host.plugins.get("XEP-0277")
        host.bridge.addMethod("eventGet", ".plugin",
                              in_sign='ssss', out_sign='(ia{ss})',
                              method=self._eventGet,
                              async=True)
        host.bridge.addMethod("eventCreate", ".plugin",
                              in_sign='ia{ss}ssss', out_sign='s',
                              method=self._eventCreate,
                              async=True)
        host.bridge.addMethod("eventModify", ".plugin",
                              in_sign='sssia{ss}s', out_sign='',
                              method=self._eventModify,
                              async=True)
        host.bridge.addMethod("eventInviteeGet", ".plugin",
                              in_sign='sss', out_sign='a{ss}',
                              method=self._eventInviteeGet,
                              async=True)
        host.bridge.addMethod("eventInviteeSet", ".plugin",
                              in_sign='ssa{ss}s', out_sign='',
                              method=self._eventInviteeSet,
                              async=True)
        host.bridge.addMethod("eventInviteesList", ".plugin",
                              in_sign='sss', out_sign='a{sa{ss}}',
                              method=self._eventInviteesList,
                              async=True),
        host.bridge.addMethod("eventInvite", ".plugin", in_sign='ssssassssssss', out_sign='',
                              method=self._invite,
                              async=True)

    def _eventGet(self, service, node, id_=u'', profile_key=C.PROF_KEY_NONE):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventGet(client, service, node, id_)

    @defer.inlineCallbacks
    def eventGet(self, client, service, node, id_=NS_EVENT):
        """Retrieve event data

        @param service(unicode, None): PubSub service
        @param node(unicode): PubSub node of the event
        @param id_(unicode): id_ with even data
        @return (tuple[int, dict[unicode, unicode]): event data:
            - timestamp of the event
            - event metadata where key can be:
                location: location of the event
                image: URL of a picture to use to represent event
                background-image: URL of a picture to use in background
        """
        if not id_:
            id_ = NS_EVENT
        items, metadata = yield self._p.getItems(client, service, node, item_ids=[id_])
        try:
            event_elt = next(items[0].elements(NS_EVENT, u'event'))
        except IndexError:
            raise exceptions.NotFound(_(u"No event with this id has been found"))

        try:
            timestamp = utils.date_parse(next(event_elt.elements(NS_EVENT, "date")))
        except StopIteration:
            timestamp = -1

        data = {}

        for key in (u'name',):
            try:
                data[key] = event_elt[key]
            except KeyError:
                continue

        for elt_name in (u'description',):
            try:
                elt = next(event_elt.elements(NS_EVENT, elt_name))
            except StopIteration:
                continue
            else:
                data[elt_name] = unicode(elt)

        for elt_name in (u'image', 'background-image'):
            try:
                image_elt = next(event_elt.elements(NS_EVENT, elt_name))
                data[elt_name] = image_elt['src']
            except StopIteration:
                continue
            except KeyError:
                log.warning(_(u'no src found for image'))

        for uri_type in (u'invitees', u'blog'):
            try:
                elt = next(event_elt.elements(NS_EVENT, uri_type))
                uri = data[uri_type + u'_uri'] = elt['uri']
                uri_data = uri_parse.parseXMPPUri(uri)
                if uri_data[u'type'] != u'pubsub':
                    raise ValueError
            except StopIteration:
                log.warning(_(u"no {uri_type} element found!").format(uri_type=uri_type))
            except KeyError:
                log.warning(_(u"incomplete {uri_type} element").format(uri_type=uri_type))
            except ValueError:
                log.warning(_(u"bad {uri_type} element").format(uri_type=uri_type))
            else:
                data[uri_type + u'_service'] = uri_data[u'path']
                data[uri_type + u'_node'] = uri_data[u'node']

        for meta_elt in event_elt.elements(NS_EVENT, 'meta'):
            key = meta_elt[u'name']
            if key in data:
                log.warning(u'Ignoring conflicting meta element: {xml}'.format(xml=meta_elt.toXml()))
                continue
            data[key] = unicode(meta_elt)

        defer.returnValue((timestamp, data))

    def _eventCreate(self, timestamp, data, service, node, id_=u'', profile_key=C.PROF_KEY_NONE):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventCreate(client, timestamp, data, service, node, id_ or NS_EVENT)

    @defer.inlineCallbacks
    def eventCreate(self, client, timestamp, data, service, node=None, item_id=NS_EVENT):
        """Create or replace an event

        @param service(jid.JID, None): PubSub service
        @param node(unicode, None): PubSub node of the event
            None will create instant node.
        @param item_id(unicode): ID of the item to create.
        @param timestamp(timestamp, None)
        @param data(dict[unicode, unicode]): data to update
            dict will be cleared, do a copy if data are still needed
            key can be:
                - name: name of the event
                - description: details
                - image: main picture of the event
                - background-image: image to use as background
        @return (unicode): created node
        """
        if not item_id:
            raise ValueError(_(u"item_id must be set"))
        if not service:
            service = client.jid.userhostJID()
        event_elt = domish.Element((NS_EVENT, 'event'))
        if timestamp is not None and timestamp != -1:
            formatted_date = utils.xmpp_date(timestamp)
            event_elt.addElement((NS_EVENT, 'date'), content=formatted_date)
        for key in (u'name',):
            if key in data:
                event_elt[key] = data.pop(key)
        for key in (u'description',):
            if key in data:
                event_elt.addElement((NS_EVENT, key), content=data.pop(key))
        for key in (u'image', u'background-image'):
            if key in data:
                elt = event_elt.addElement((NS_EVENT, key))
                elt['src'] = data.pop(key)

        # we first create the invitees and blog nodes (if not specified in data)
        for uri_type in (u'invitees', u'blog'):
            key = uri_type + u'_uri'
            for to_delete in (u'service', u'node'):
                k = uri_type + u'_' + to_delete
                if k in data:
                    del data[k]
            if key not in data:
                # FIXME: affiliate invitees
                uri_node = yield self._p.createNode(client, service)
                yield self._p.setConfiguration(client, service, uri_node, {self._p.OPT_ACCESS_MODEL: self._p.ACCESS_WHITELIST})
                uri_service = service
            else:
                uri = data.pop(key)
                uri_data = uri_parse.parseXMPPUri(uri)
                if uri_data[u'type'] != u'pubsub':
                    raise ValueError(_(u'The given URI is not valid: {uri}').format(uri=uri))
                uri_service = jid.JID(uri_data[u'path'])
                uri_node = uri_data[u'node']

            elt = event_elt.addElement((NS_EVENT, uri_type))
            elt['uri'] = uri_parse.buildXMPPUri('pubsub', path=uri_service.full(), node=uri_node)

        # remaining data are put in <meta> elements
        for key in data.keys():
            elt = event_elt.addElement((NS_EVENT, 'meta'), content = data.pop(key))
            elt['name'] = key

        item_elt = pubsub.Item(id=item_id, payload=event_elt)
        try:
            # TODO: check auto-create, no need to create node first if available
            node = yield self._p.createNode(client, service, nodeIdentifier=node)
        except error.StanzaError as e:
            if e.condition == u'conflict':
                log.debug(_(u"requested node already exists"))

        yield self._p.publish(client, service, node, items=[item_elt])

        defer.returnValue(node)

    def _eventModify(self, service, node, id_, timestamp_update, data_update, profile_key=C.PROF_KEY_NONE):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventModify(client, service, node, id_ or NS_EVENT, timestamp_update or None, data_update)

    @defer.inlineCallbacks
    def eventModify(self, client, service, node, id_=NS_EVENT, timestamp_update=None, data_update=None):
        """Update an event

        Similar as create instead that it update existing item instead of
        creating or replacing it. Params are the same as for [eventCreate].
        """
        event_timestamp, event_metadata = yield self.eventGet(client, service, node, id_)
        new_timestamp = event_timestamp if timestamp_update is None else timestamp_update
        new_data = event_metadata
        if data_update:
            for k, v in data_update.iteritems():
                new_data[k] = v
        yield self.eventCreate(client, new_timestamp, new_data, service, node, id_)

    def _eventInviteeGet(self, service, node, profile_key):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventInviteeGet(client, service, node)

    @defer.inlineCallbacks
    def eventInviteeGet(self, client, service, node):
        """Retrieve attendance from event node

        @param service(unicode, None): PubSub service
        @param node(unicode): PubSub node of the event
        @return (dict): a dict with current attendance status,
            an empty dict is returned if nothing has been answered yed
        """
        items, metadata = yield self._p.getItems(client, service, node, item_ids=[client.jid.userhost()])
        try:
            event_elt = next(items[0].elements(NS_EVENT, u'invitee'))
        except IndexError:
            # no item found, event data are not set yet
            defer.returnValue({})
        data = {}
        for key in (u'attend', u'guests'):
            try:
                data[key] = event_elt[key]
            except KeyError:
                continue
        defer.returnValue(data)

    def _eventInviteeSet(self, service, node, event_data,  profile_key):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventInviteeSet(client, service, node, event_data)

    def eventInviteeSet(self, client, service, node, data):
        """Set or update attendance data in event node

        @param service(unicode, None): PubSub service
        @param node(unicode): PubSub node of the event
        @param data(dict[unicode, unicode]): data to update
            key can be:
                attend: one of "yes", "no", "maybe"
                guests: an int
        """
        event_elt = domish.Element((NS_EVENT, 'invitee'))
        for key in (u'attend', u'guests'):
            try:
                event_elt[key] = data.pop(key)
            except KeyError:
                pass
        item_elt = pubsub.Item(id=client.jid.userhost(), payload=event_elt)
        return self._p.publish(client, service, node, items=[item_elt])

    def _eventInviteesList(self, service, node, profile_key):
        service = jid.JID(service) if service else None
        node = node if node else NS_EVENT
        client = self.host.getClient(profile_key)
        return self.eventInviteesList(client, service, node)

    @defer.inlineCallbacks
    def eventInviteesList(self, client, service, node):
        """Retrieve attendance from event node

        @param service(unicode, None): PubSub service
        @param node(unicode): PubSub node of the event
        @return (dict): a dict with current attendance status,
            an empty dict is returned if nothing has been answered yed
        """
        items, metadata = yield self._p.getItems(client, service, node)
        invitees = {}
        for item in items:
            try:
                event_elt = next(item.elements(NS_EVENT, u'invitee'))
            except IndexError:
                # no item found, event data are not set yet
                log.warning(_(u"no data found for {item_id} (service: {service}, node: {node})".format(
                    item_id=item['id'],
                    service=service,
                    node=node
                    )))
            data = {}
            for key in (u'attend', u'guests'):
                try:
                    data[key] = event_elt[key]
                except KeyError:
                    continue
            invitees[item['id']] = data
        defer.returnValue(invitees)

    def _invite(self, service, node, id_=NS_EVENT, email=u'', emails_extra=None, name=u'', host_name=u'', language=u'', url_template=u'',
        message_subject=u'', message_body=u'', profile_key=C.PROF_KEY_NONE):
        client = self.host.getClient(profile_key)
        kwargs = {u'profile': client.profile,
                  u'emails_extra': [unicode(e) for e in emails_extra]
                 }
        for key in ("email", "name", "host_name", "language", "url_template", "message_subject", "message_body"):
            value = locals()[key]
            kwargs[key] = unicode(value)
        return self.invite(client,
                           jid.JID(service) if service else None,
                           node,
                           id_ or NS_EVENT,
                           **kwargs)

    @defer.inlineCallbacks
    def invite(self, client, service, node, id_=NS_EVENT, **kwargs):
        """High level method to create an email invitation to an event

        @param service(unicode, None): PubSub service
        @param node(unicode): PubSub node of the event
        @param id_(unicode): id_ with even data
        """
        if self._i is None:
            raise exceptions.FeatureNotFound(_(u'"Invitations" plugin is needed for this feature'))
        if self._b is None:
            raise exceptions.FeatureNotFound(_(u'"XEP-0277" (blog) plugin is needed for this feature'))
        event_service = (service or client.jid.userhostJID())
        event_uri = uri_parse.buildXMPPUri('pubsub',
            path=event_service.full(),
            node=node,
            item=id_)
        kwargs['extra'] = {u'event_uri': event_uri}
        invitation_data = yield self._i.create(**kwargs)
        invitee_jid = invitation_data[u'jid']
        log.debug(_(u'invitation created'))
        yield self._p.setNodeAffiliations(client, event_service, node, {invitee_jid: u'member'})
        log.debug(_(u'affiliation set on event node'))
        dummy, event_data = yield self.eventGet(client, service, node, id_)
        log.debug(_(u'got event data'))
        invitees_service = jid.JID(event_data['invitees_service'])
        invitees_node = event_data['invitees_node']
        blog_service = jid.JID(event_data['blog_service'])
        blog_node = event_data['blog_node']
        yield self._p.setNodeAffiliations(client, invitees_service, invitees_node, {invitee_jid: u'publisher'})
        log.debug(_(u'affiliation set on invitee node'))
        yield self._p.setNodeAffiliations(client, blog_service, blog_node, {invitee_jid: u'member'})
        # FIXME: what follow is crazy, we have no good way to handle comments affiliations for blog
        blog_items, dummy = yield self._b.mbGet(client, blog_service, blog_node, None)

        for item in blog_items:
            comments_service = jid.JID(item['comments_service'])
            comments_node = item['comments_node']
            yield self._p.setNodeAffiliations(client, comments_service, comments_node, {invitee_jid: u'publisher'})
        log.debug(_(u'affiliation set on blog and comments nodes'))


