# --------------------------------------------------------
#
# This file is part of Rheolef.
#
# Copyright (C) 2000-2009 Pierre Saramito 
#
# Rheolef is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Rheolef is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Rheolef; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# -------------------------------------------------------------------------
#
# surface mesh viewer
#
# author: pierre.saramito@imag.fr
#
# note: under tests
# --------------------------------------------------------
# inspirated from 
#  * vtk-3.2/graphic/examplesTcl/Decimate.tcl
#  * Effective Tcl/Tk Programming
#    Mark Harrison and Michael McLennan
#    Addison-Wesley Professional Computing Series
# --------------------------------------------------------
#
# contents
#  part 1.
#	1.1  includes
#	1.x  ressources
#	1.2  global variables and default flags values
#	1.3  instances of vtk object: polydata, filters
#	1.4  menus bar
#	1.x  error msg
# 	1.5  animation control
#	1.6  the vtk rendering window
#	1.7  the status bar
#	1.8  graphic objects: cameras, light, renderer
#	1.9  procedures
#	     1.8.1 to set particular view
#	     1.8.2 input/output
#	     1.8.3 undo
#	     1.8.4 release memory of filters
#	1.10 create graphic pipeline
#	     1.9.1 welcome banner
#	     1.9.2 actor for comparison
#	     1.9.3 edges 
#	     1.9.3 set camera zoom 
#  part 2.
#	2.1 procedures
#	     2.1 updata data statistics and gui
#	     2.2 comparison split screen
#	2.2 popup menu and procedures
#	     2.1 decimate
#	     2.2 smooth
#	     2.3 clean degenerated cells
#	     2.4 connected components
#	     2.5 triangulate
#	     2.6 normals to the surface
#	     2.7 set background color
#	     2.8 set surface properties
#  part 3.
#	3.1 gauge
#
# ------------------------------------------------------------------------------
# TODO:
#  - package vtk variables (reader,...) in the tk/vtk $window object
#  - complete vtk objects : axes, outline, edit colortable, 
#	height and topography filters, transparency, etc...
# ------------------------------------------------------------------------------
# dependencies: include or use a wrapper
# ------------------------------------------------------------------------------
#if {![winfo exists PKGDATADIR]} {
#    set PKGDATADIR /disk/pierre/dvt/rheolef/fem/vtk
#}
#source $PKGDATADIR/vtk_interactor.tcl
#source $PKGDATADIR/vtk_widget_object.tcl
#source $PKGDATADIR/vtk_volume_interactor.tcl
#source $PKGDATADIR/tcl_toolbar.tcl
#source $PKGDATADIR/tcl_balloon.tcl
#source $PKGDATADIR/tcl_percent.tcl
#source $PKGDATADIR/tcl_animate.tcl
#source $PKGDATADIR/tcl_font.tcl
#source $PKGDATADIR/tcl_textbar.tcl

set color_wheel_file $PKGDATADIR/vtk_color_wheel.ppm

# ------------------------------------------------------------------------------
# ressources
# ------------------------------------------------------------------------------

option add *Rheolef*foreground                  black           interactive
option add *Rheolef*background                  gray80          interactive

option add *Rheolef*titleFontFamily             helvetica       interactive
option add *Rheolef*titleFontSlant              roman           interactive
option add *Rheolef*titleFontWeight             bold            interactive
option add *Rheolef*titleFontSize               10              interactive

bind . <Control-q> exit
bind . <q>         exit
tk_setPalette gray80

# ------------------------
# global variables
# ------------------------
set deci_reduction 0.0
set deci_preserve  1
set view           Left
set graphic_width  300 
set graphic_height 300 

# ------------------------------
# effect flags: default settings
# ------------------------------
set flag_surface        	1
set flag_elevation        	1;	# elevation z = u(x,y,t) by values of u(t)
set flag_feature_edge 		0
set flag_boundary_edges 	0
set flag_non_manifold_edges  	0
set flag_compare 		0
set flag_edge_splitting 	1
set flag_flip_normals 		0

# ------------------------------
# command-line settings
# ------------------------------
set do_elevation		1

# ------------------------
# instances of vtk objects
# ------------------------
vtkPolyData poly_data
			    	# poly_data GlobalWarningDisplayOff;
vtkPolyData  previous_polydata
vtkPolyData  tmp_polydata
vtkCellTypes cell_types

vtkDecimatePro deci
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	deci AddObserver StartEvent    {start_gauge_progress deci "Decimating..."}
	deci AddObserver ProgressEvent {show_gauge_progress  deci "Decimating..."}
	deci AddObserver EndEvent      end_gauge_progress
    } else {
	deci SetStartMethod    {start_gauge_progress deci "Decimating..."}
	deci SetProgressMethod {show_gauge_progress  deci "Decimating..."}
	deci SetEndMethod      end_gauge_progress
    }

vtkSmoothPolyDataFilter smooth
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	smooth AddObserver StartEvent    {start_gauge_progress smooth "Smoothing..."}
	smooth AddObserver ProgressEvent {show_gauge_progress  smooth "Smoothing..."}
	smooth AddObserver EndEvent      end_gauge_progress
    } else {
	smooth SetStartMethod    {start_gauge_progress smooth "Smoothing..."}
	smooth SetProgressMethod {show_gauge_progress  smooth "Smoothing..."}
	smooth SetEndMethod      end_gauge_progress
    }

vtkCleanPolyData cleaner
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	cleaner AddObserver StartEvent    {start_gauge_progress cleaner "Cleaning..."}
	cleaner AddObserver ProgressEvent {show_gauge_progress  cleaner "Cleaning..."}
	cleaner AddObserver EndEvent      end_gauge_progress
    } else {
	cleaner SetStartMethod    {start_gauge_progress cleaner "Cleaning..."}
	cleaner SetProgressMethod {show_gauge_progress  cleaner "Cleaning..."}
	cleaner SetEndMethod      end_gauge_progress
    }

vtkPolyDataConnectivityFilter connect
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	connect AddObserver StartEvent    {start_gauge_progress connect "Connectivity..."}
	connect AddObserver ProgressEvent {show_gauge_progress  connect "Connectivity..."}
	connect AddObserver EndEvent      end_gauge_progress
    } else {
	connect SetStartMethod    {start_gauge_progress connect "Connectivity..."}
	connect SetProgressMethod {show_gauge_progress  connect "Connectivity..."}
	connect SetEndMethod      end_gauge_progress
    }

vtkTriangleFilter tri
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	tri AddObserver StartEvent    {start_gauge_progress tri "Triangulating..."}
	tri AddObserver ProgressEvent {show_gauge_progress  tri "Triangulating..."}
	tri AddObserver EndEvent      end_gauge_progress
    } else {
	tri SetStartMethod    {start_gauge_progress tri "Triangulating..."}
	tri SetProgressMethod {show_gauge_progress  tri "Triangulating..."}
	tri SetEndMethod      end_gauge_progress
    }

vtkPolyDataNormals normals
    if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	normals AddObserver StartEvent    {start_gauge_progress normals "Generating Normals..."}
	normals AddObserver ProgressEvent {show_gauge_progress  normals "Generating Normals..."}
	normals AddObserver EndEvent      end_gauge_progress
    } else {
	normals SetStartMethod    {start_gauge_progress normals "Generating Normals..."}
	normals SetProgressMethod {show_gauge_progress  normals "Generating Normals..."}
	normals SetEndMethod      end_gauge_progress
    }

# ----------------------------------------------------------------------------
# Create top-level GUI
# ----------------------------------------------------------------------------

wm title . "rheolef animation"
frame .top -class Rheolef
pack .top -fill both -expand 1
set top .top

set mbar $top.mbar
frame $mbar -relief raised -bd 2
pack $mbar -side top -fill x

menubutton $mbar.file    -text File    -menu $mbar.file.menu
menubutton $mbar.edit    -text Edit    -menu $mbar.edit.menu
menubutton $mbar.view    -text View    -menu $mbar.view.menu
menubutton $mbar.options -text Options -menu $mbar.options.menu
menubutton $mbar.help    -text Help    -menu $mbar.help.menu
pack $mbar.file $mbar.edit $mbar.view $mbar.options -side left
pack $mbar.help -side right

menu $mbar.file.menu
     $mbar.file.menu add command -label Open -command open_file_menu
     $mbar.file.menu add command -label Save -command save_file -state disabled
     $mbar.file.menu add command -label Exit -command exit

menu $mbar.edit.menu
     $mbar.edit.menu add command -label Clean        -command proc_clean       -state disabled
     $mbar.edit.menu add command -label Connectivity -command proc_connectivity -state disabled
     $mbar.edit.menu add command -label Decimate     -command proc_decimate    -state disabled
     $mbar.edit.menu add command -label Normals      -command proc_normals     -state disabled
     $mbar.edit.menu add command -label Smooth       -command proc_smooth      -state disabled
     $mbar.edit.menu add command -label Triangulate  -command proc_triangulate -state disabled
     $mbar.edit.menu add command -label "Undo/Redo"  -command proc_undo

menu $mbar.view.menu
     $mbar.view.menu add checkbutton 						\
				-label    "Object Surface" 			\
				-variable flag_surface				\
	  			-command  {update_gui; $render_window Render}
     $mbar.view.menu add checkbutton 						\
				-label    "Elevation"	 			\
				-variable flag_elevation			\
	  			-command  {update_gui; $render_window Render}
     $mbar.view.menu add checkbutton 						\
				-label    "Feature Edges" 			\
				-variable flag_feature_edge			\
	  			-command  {update_gui; $render_window Render}
     $mbar.view.menu add checkbutton 						\
				-label    "Boundary Edges" 			\
				-variable flag_boundary_edges			\
	  			-command  {update_gui; $render_window Render}
     $mbar.view.menu add checkbutton 						\
				-label    "Non-manifold Edges" 			\
				-variable flag_non_manifold_edges 		\
	  			-command  {update_gui; $render_window Render}
     $mbar.view.menu add separator
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Front 				\
				-value    Front					\
            			-command {update_view 1 0 0 0 1 0}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Back 					\
				-value    Back					\
            			-command {update_view -1 0 0 0 1 0}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Left 					\
				-value    Left					\
            			-command {update_view 0 0 1 0 1 0}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Right 				\
				-value    Right					\
            			-command {update_view 0 0 -1 0 1 0}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Top 					\
				-value    Top					\
            			-command {update_view 0 1 0 0 0 1}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Bottom 				\
				-value    Bottom				\
            			-command {update_view 0 -1 0 0 0 1}
     $mbar.view.menu add radiobutton 						\
				-variable view 					\
				-label    Isometric 				\
        			-value    Isometric 				\
				-command {update_view 3 3 1 1 1 3}

menu $mbar.options.menu
     $mbar.options.menu add command 						\
				-label   "Compare Results" 			\
				-command set_compare 				\
	    			-state   disabled
     $mbar.options.menu add command 						\
				-label   "Background Color..." 			\
	    			-command set_background_color
     $mbar.options.menu add command 						\
				-label   "Surface Properties..." 		\
	    			-command set_properties

menu $mbar.help.menu
     $mbar.help.menu add command 						\
				-label {"please, read the documentation"}

# ---------------------
# error msg
# ---------------------
proc error_message {msg} {
    # TODO: pop-up window
    puts "$msg"
}
# ---------------------
# The animation control
# ---------------------
proc draw_bitmap {name} {

    global PKGDATADIR

    return [image create bitmap -foreground black -background white \
        -file     [file join $PKGDATADIR $name.xbm] \
        -maskfile [file join $PKGDATADIR ${name}.xbm]]
}
proc show_field {name} {

    global render_window

    reader SetScalarsName $name
        # if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	#     reader AddObserver StartEvent    {start_gauge_progress reader "Reading $name..."}
	#     reader AddObserver ProgressEvent {show_gauge_progress  reader "Reading $name..."}
	#     reader AddObserver EndEvent       end_gauge_progress
        # } else {
	#     reader SetStartMethod    {start_gauge_progress reader "Reading $name..."}
	#     reader SetProgressMethod {show_gauge_progress  reader "Reading $name..."}
	#     reader SetEndMethod       end_gauge_progress
        # }

    update_undo "reader"
    update_gui
    $render_window Render
}
set  record_field_id 0
proc record_field {name} {

    global render_window
    global record_field_id

    show_field $name
    set basename "output-[format "%.5d" $record_field_id]"
    save_as $render_window $basename
    incr record_field_id
}
set pending_animate 0
set pager ""

proc anim_mode {mode} {

    global record_field_id
    global pending_animate
    global pager
    global anim

    if { $mode == "play" } {
	set list [textbar_get $pager -marked]
	textbar_configure $pager -disable-callback
        set pending_animate [animate_create 100 $list { \
		show_field %v; \
		textbar_configure $pager -select %v ; \
	    }]
	animate_configure $pending_animate -callback-at-end "toolbar_configure $anim.bar stop"
	animate_play $pending_animate
    } elseif { $mode == "repeat" } {
	set list [textbar_get $pager -marked]
	textbar_configure $pager -disable-callback
        set pending_animate [animate_create 100 $list { \
		show_field %v; \
		textbar_configure $pager -select %v ; \
	    }]
	animate_configure $pending_animate -do-infinite-loop 1
	animate_play $pending_animate
    } elseif { $mode == "record" } {
	set record_field_id 0
	set list [textbar_get $pager -marked]
	textbar_configure $pager -disable-callback
        set pending_animate [animate_create 100 $list { \
		record_field %v; \
		textbar_configure $pager -select %v ; \
	    }]
	animate_configure $pending_animate -callback-at-end "toolbar_configure $anim.bar stop"
	animate_play $pending_animate
    } elseif { $mode == "stop" } {
    	animate_stop $pending_animate
	if { $pager != "" } {
	    textbar_configure $pager -enable-callback
	}
    } elseif { $mode == "back" } {
	set list [textbar_get $pager -marked]
	set value [lindex $list 0]
	show_field $value
	textbar_configure $pager -select $value
    } elseif { $mode == "fwrd" } {
	set list [textbar_get $pager -marked]
	set last [expr [llength $list] - 1]
	set value [lindex $list $last]
	show_field $value
	textbar_configure $pager -select $value
    } elseif { $mode == "quit" } {
	exit 0
    } else {
    	puts "anim_mode \"$mode\": not yet supported"
    }
}
set anim $top.anim
frame $anim -borderwidth 1 -relief sunken
pack  $anim -side top -fill y

toolbar_create $anim.bar left {anim_mode %t}
pack  $anim.bar -side top -expand yes -fill both -padx 4 -pady 4

# TODO ? prev, next, pause, random
foreach tool {
    {stop 	{stop}}
    {record 	{record}}
    {back 	{back}}
    {play 	{play}}
    {fwrd 	{forward}}
    {repeat 	{repeat}}
    {quit 	{quit}}
} {
    set name [lindex $tool 0]
    set imh [draw_bitmap $name]

    set win [toolbar_add $anim.bar $name $imh]
    set help [lindex $tool 1]
    balloonhelp_for $win $help
}
toolbar_select $anim.bar stop

# -------------------------------------------
# The textbar control for list of fields
# -------------------------------------------
frame $top.dis
set dis $top.dis
pack $dis -side left -fill both -expand 1

set pager $dis.pager
textbar_create $pager
textbar_bind $pager Select {
    show_field %v
}
grid $pager -row 0 -column 0 -sticky ns
grid columnconfigure $dis 0 -minsize
grid rowconfigure    $dis 0 -weight 1
# --------------------
# The rendering widget
# --------------------
set window $dis.window
vtkTkRenderWidget $window -width $graphic_width -height $graphic_height 
    BindTkRenderWidget $window

grid $window -row 0 -column 1 -sticky nsew
grid columnconfigure $dis 1 -weight 1
grid rowconfigure    $dis 0 -weight 1

# --------------------
# Gauge progress
# --------------------
frame .gauge -relief sunken -borderwidth 3
label .gauge.status -text "(No data)" -borderwidth 0
pack  .gauge.status -side top -anchor w -expand 1 -fill x -padx 0 -pady 0
pack  .gauge  -side top -anchor sw -fill x -padx 0 -pady 0

# --------------------
# Graphics objects
# --------------------
vtkCamera camera
vtkLight  light
vtkRenderer renderer
    renderer SetActiveCamera camera
    renderer AddLight light
vtkRenderer comprare_renderer
    comprare_renderer SetViewport 0.0 0.0 0.5 1.0
    comprare_renderer SetActiveCamera camera
    comprare_renderer AddLight light
set render_window [$window GetRenderWindow]
$render_window AddRenderer renderer

# ---------------------------------
# Procedure to set particular views
# ---------------------------------
proc update_view {x y z vx vy vz} {

    global render_window

    set camera [renderer GetActiveCamera]
    $camera SetFocalPoint 0 0 0
    $camera SetPosition $x $y $z
    $camera SetViewUp $vx $vy $vz
    renderer ResetCamera
    Render $render_window
}
# -------------------------------------
# Procedure opens file and resets view
# -------------------------------------
proc open_file {filename} {
    	global anim
    	global pager
    	global window
    	global render_window 
    	global flag_elevation
    	global do_elevation
    	global view

	textbar_clear $pager

        if { [info commands reader] != "" } {reader Delete}
        if { [string match *.g $filename] } {
            vtkBYUReader reader
            reader SetGeometryFileName $filename
        } elseif { [string match *.stl $filename] } {
            vtkSTLReader reader
            reader SetFileName $filename
        } elseif { [string match *.vtk $filename] } {
            vtkPolyDataReader reader
            reader SetFileName $filename
        } elseif { [string match *.cyb $filename] } {
            vtkCyberReader reader
            reader SetFileName $filename
        } elseif { [string match *.tri $filename] } {
            vtkMCubesReader reader
            reader SetFileName $filename
        } elseif { [string match *.obj $filename] } {
            vtkOBJReader reader
            reader SetFileName $filename
        } else {
            error_message "unrecognized file suffix"
            return
        }
        if { [get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0 } {
	    reader AddObserver StartEvent    {start_gauge_progress reader "Reading..."}
	    reader AddObserver ProgressEvent {show_gauge_progress  reader "Reading..."}
	    reader AddObserver EndEvent       end_gauge_progress
	} else {
	    reader SetStartMethod    {start_gauge_progress reader "Reading..."}
	    reader SetProgressMethod {show_gauge_progress  reader "Reading..."}
	    reader SetEndMethod       end_gauge_progress
	}

	reader Update

	if {[get_vtk_major_version] >= 3 && [get_vtk_minor_version] >= 2} {
	    # scan data for animations
	    set n_field [reader GetNumberOfScalarsInFile]
	    for {set i 0} {$i < $n_field} {incr i} {
    	        set name [reader GetScalarsNameInFile $i]
	        textbar_insert $pager $name
	    }
	} else {
	    # scan with a "grep SCALARS"
	    set grep_output [eval exec [list grep SCALARS $filename]]
	    foreach {scalars name float} $grep_output {
	        textbar_insert $pager $name
	    }
	}
	textbar_redraw $pager

        update_undo "reader"
	update_gui

	set filename "rheolef animation: [file tail $filename]"
	wm title . $filename

        renderer ResetCamera
        $render_window Render

	# set elevation with values by default if surface is flat
        set  bounds [[reader GetOutput] GetBounds]
        scan $bounds "%f %f %f %f %f %f" minx maxx miny maxy minz maxz
	if {[expr $maxz - $minz] == 0.0 && $do_elevation} {
    	    set flag_elevation 1
	    set view Isometric
	    update_view 3 3 1 1 1 3
	} else {
    	    set flag_elevation 0
	    set view Left
            update_view 0 0 1 0 1 0
	}
}
proc open_file_menu {} {

    set types {
        {{Visualization Toolkit (polygonal)}    {.vtk}        }
        {{Stereo-Lithography}                   {.stl}        }
        {{BYU}                                  {.g}          }
        {{Cyberware (Laser Scanner)}            {.cyb}        }
        {{Marching Cubes}                       {.tri}        }
        {{Wavefront}                            {.obj}        }
        {{All Files }                           *             }
    }
    set filename [tk_getOpenFile -filetypes $types]
    if { $filename != "" } {
	open_file $filename
    }
}
# ----------------------------
# Procedure saves data to file
# ----------------------------
proc save_file {} {

    global poly_data
    global render_window

    set types {
        {{Visualization Toolkit (polygonal)}    {.vtk}        }
        {{VRML}                                 {.wrl}        }
        {{Stereo-Lithography}                   {.stl}        }
        {{BYU}                                  {.g}          }
        {{Marching Cubes}                       {.tri}        }
        {{RIB (Renderman)}                      {.rib}        }
        {{Wavefront OBJ}                        {.obj}        }
        {{All Files }                           *             }
    }
    set filename [tk_getSaveFile -filetypes $types]
    if { $filename != "" } {
        if { [info commands writer] != "" } {writer Delete}
        if { [string match *.g $filename] } {
            vtkBYUWriter writer
            writer SetGeometryFileName $filename
            writer SetInput poly_data
        } elseif { [string match *.stl $filename] } {
            vtkSTLWriter writer
            writer SetFileName $filename
            writer SetInput poly_data
        } elseif { [string match *.vtk $filename] } {
            vtkPolyDataWriter writer
            writer SetFileName $filename
            writer SetInput poly_data
        } elseif { [string match *.tri $filename] } {
            vtkMCubesWriter writer
            writer SetFileName $filename
            writer SetInput poly_data
        } elseif { [string match *.wrl $filename] } {
            vtkVRMLExporter writer
	    writer SetRenderWindow $render_window
            writer SetFileName $filename
        } elseif { [string match *.obj $filename] } {
            vtkOBJExporter writer
	    writer SetRenderWindow $render_window
            writer SetFilePrefix [file rootname $filename]
        } elseif { [string match *.rib $filename] } {
            vtkRIBExporter writer
	    writer SetRenderWindow $render_window
            writer SetFilePrefix [file rootname $filename]
        } else {
            error_message "Can't write this file"
            return
        }
        writer Write
    }
}
# ------------------------------------------------
# Enable the undo procedure after filter execution
# ------------------------------------------------
proc update_undo {filter} {

    global current_filter

    set current_filter $filter
    $filter Update

    previous_polydata CopyStructure poly_data
    [previous_polydata GetPointData] PassData [poly_data GetPointData]
    previous_polydata Modified

    poly_data CopyStructure [$filter GetOutput]
    [poly_data GetPointData] PassData [[$filter GetOutput] GetPointData]
    poly_data Modified

    release_data
}
# --------------
# Undo last edit
# --------------
proc proc_undo {} {

    global render_window

    tmp_polydata CopyStructure poly_data
    [tmp_polydata GetPointData] PassData [poly_data GetPointData]

    poly_data CopyStructure previous_polydata
    [poly_data GetPointData] PassData [previous_polydata GetPointData]
    poly_data Modified

    previous_polydata CopyStructure tmp_polydata
    [previous_polydata GetPointData] PassData [tmp_polydata GetPointData]
    previous_polydata Modified

    update_gui
    $render_window Render
}
# ---------------------------------------------------------------
# Procedure initializes filters so that they release their memory
# ---------------------------------------------------------------
proc release_data {} {
    [deci    GetOutput] Initialize
    [smooth  GetOutput] Initialize
    [cleaner GetOutput] Initialize
    [connect GetOutput] Initialize
    [tri     GetOutput] Initialize
    [smooth  GetOutput] Initialize
}
# -------------------
# Create pipeline
# -------------------

set scale_factor 1.0

vtkWarpScalar apply_elevation
  apply_elevation SetInput poly_data
  apply_elevation SetNormal 0 0 1
  apply_elevation SetScaleFactor $scale_factor
  apply_elevation UseNormalOn

vtkPolyDataMapper   mapper
    mapper SetInput [apply_elevation GetOutput]

vtkProperty property
    property SetColor 0.8900 0.8100 0.3400
    property SetSpecularColor 1 1 1
    property SetSpecular 0.3
    property SetSpecularPower 20
    property SetAmbient 0.2
    property SetDiffuse 0.8

vtkActor actor
    actor SetMapper mapper
    actor SetProperty property

# Welcome banner
vtkTextMapper banner
    banner SetInput "rheolef surface mesh viewer\nversion 1.0"
    if {[get_vtk_major_version] >= 4 && [get_vtk_minor_version] > 0} {
	vtkTextProperty banner_text_property
	banner_text_property SetFontFamilyToArial
	banner_text_property SetFontSize 18
	banner_text_property ItalicOn
	banner_text_property SetJustificationToCentered
	banner SetTextProperty banner_text_property
    } else {
	banner SetFontFamilyToArial
	banner SetFontSize 18
	banner ItalicOn
	banner SetJustificationToCentered
    }

vtkActor2D bannerActor
    bannerActor SetMapper banner
    [bannerActor GetProperty] SetColor 0 1 0
    [bannerActor GetPositionCoordinate] SetCoordinateSystemToNormalizedDisplay
    [bannerActor GetPositionCoordinate] SetValue 0.5 0.5
renderer AddProp bannerActor

# -------------------------------------------
# Actor used for side-by-side data comparison
# -------------------------------------------
vtkPolyDataMapper CompareMapper
    CompareMapper SetInput previous_polydata

vtkActor CompareActor
    CompareActor SetMapper CompareMapper
    CompareActor SetProperty property
comprare_renderer AddActor CompareActor

# -----------
# Edges
# -----------
vtkFeatureEdges FeatureEdges
    FeatureEdges SetInput poly_data
    FeatureEdges BoundaryEdgesOff
    FeatureEdges NonManifoldEdgesOff
    FeatureEdges ManifoldEdgesOff
    FeatureEdges FeatureEdgesOff

vtkPolyDataMapper FEdgesMapper
    FEdgesMapper SetInput [FeatureEdges GetOutput]
    FEdgesMapper SetScalarModeToUseCellData

vtkActor FEdgesActor
    FEdgesActor SetMapper FEdgesMapper

renderer ResetCamera
[renderer GetActiveCamera] Zoom 1.25

# -----------------------------------------------
# Procedure updates data statistics and GUI menus
# -----------------------------------------------
proc update_gui {} {
    global flag_surface
    global flag_elevation
    global render_window
    global flag_feature_edge 
    global flag_boundary_edges
    global flag_non_manifold_edges
    global render_window
    global mbar
    global view

    set number_of_nodes [poly_data GetNumberOfPoints]
    set number_of_elements [poly_data GetNumberOfCells]
    poly_data GetCellTypes cell_types

    renderer RemoveActor bannerActor
    renderer RemoveActor actor
    renderer RemoveActor FEdgesActor

    # Check to see whether to add surface model
    if { [poly_data GetNumberOfCells] <= 0 } {
	renderer AddActor bannerActor
	$mbar.edit.menu entryconfigure 1 -state disabled
	$mbar.edit.menu entryconfigure 2 -state disabled
	$mbar.edit.menu entryconfigure 3 -state disabled
	$mbar.edit.menu entryconfigure 4 -state disabled
	$mbar.edit.menu entryconfigure 5 -state disabled
	$mbar.file.menu entryconfigure 1 -state disabled
	$mbar.options.menu entryconfigure 1 -state disabled
        set s "(None)"

    } else {
	if { $flag_surface } {renderer AddActor actor}

	if { $flag_elevation } {
	    mapper SetInput [apply_elevation GetOutput]
        } else {
	    mapper SetInput poly_data
	}
	if { $flag_feature_edge || $flag_boundary_edges || $flag_non_manifold_edges  } {
	    renderer AddActor FEdgesActor
	    FeatureEdges SetBoundaryEdges $flag_boundary_edges
	    FeatureEdges SetFeatureEdges $flag_feature_edge
	    FeatureEdges SetNonManifoldEdges $flag_non_manifold_edges 
	}

	$mbar.edit.menu entryconfigure 1 -state normal
        if { [cell_types GetNumberOfTypes] != 1 || [cell_types GetCellType 0] != 5 } {
	    $mbar.edit.menu entryconfigure 2 -state disabled
            $mbar.edit.menu entryconfigure 6 -state normal
            set s [format "vertices:%d    elements:%d" \
                   $number_of_nodes $number_of_elements]
	} else {
            $mbar.edit.menu entryconfigure 2 -state normal
            $mbar.edit.menu entryconfigure 6 -state disabled
            set s [format "vertices:%d    triangles:%d" \
                   $number_of_nodes $number_of_elements]
	}
	$mbar.edit.menu entryconfigure 3 -state normal
	$mbar.edit.menu entryconfigure 4 -state normal
	$mbar.edit.menu entryconfigure 5 -state normal
	$mbar.file.menu entryconfigure 2 -state normal
	$mbar.options.menu entryconfigure 1 -state normal
    }
    .gauge.status configure -text $s
}
# -------------------------------------------------------
# Procedure manages splitting screen and comparing data
# -------------------------------------------------------
proc set_compare {} {

    global flag_compare
    global render_window
    global mbar

    if { $flag_compare == 0} {
	$render_window AddRenderer comprare_renderer
	renderer SetViewport 0.5 0.0 1.0 1.0
	$mbar.options.menu entryconfigure 1 -label "Uncompare Results"
	set flag_compare 1

    } else {
	$render_window RemoveRenderer comprare_renderer
	renderer SetViewport 0.0 0.0 1.0 1.0
	$mbar.options.menu entryconfigure 1 -label "Compare Results"
	set flag_compare 0
    }
    $render_window Render
}
# -------------------
# The decimation GUI
# -------------------

# Procedure defines GUI and behavior for decimating data

proc proc_decimate {} {
    update_decimation_gui
    wm deiconify .decimate
}
proc close_decimate {} {
    wm withdraw .decimate
}
toplevel .decimate
wm withdraw .decimate
wm title .decimate "Decimate"
wm protocol .decimate WM_DELETE_WINDOW {wm withdraw .decimate}

frame .decimate.f1
checkbutton .decimate.f1.preserve 					\
				-variable  deci_preserve 		\
				-text     "Preserve Topology"
scale .decimate.f1.red 							\
				-label    "Requested Number Of Polygons" \
				-from 0 -to 100000 			\
				-length     3.0i 			\
				-orient     horizontal 			\
				-resolution 1 				\
				-command    set_deci_polygons
.decimate.f1.red set 4000
pack .decimate.f1.preserve .decimate.f1.red \
	-pady 0.1i -side top -anchor w

frame .decimate.fb
button .decimate.fb.apply  -text Apply  -command apply_decimation
button .decimate.fb.cancel -text Cancel -command close_decimate
pack .decimate.fb.apply .decimate.fb.cancel -side left -expand 1 -fill x
pack .decimate.f1 .decimate.fb -side top -fill both -expand 1

proc update_decimation_gui {} {

   set numPolys [poly_data GetNumberOfCells]
   .decimate.f1.red configure -to $numPolys

   set_deci_polygons [.decimate.f1.red get]
}
proc apply_decimation {} {
    global deci_reduction deci_preserve render_window

    deci SetInput poly_data    

    deci SetTargetReduction $deci_reduction
    deci SetPreserveTopology $deci_preserve

    update_undo "deci"
    update_gui

    $render_window Render
    close_decimate
}  
proc set_deci_polygons value {
    global deci_reduction

    set numInPolys [poly_data GetNumberOfCells]
    if { $numInPolys <= 0 } {return}
    set deci_reduction [expr (double($numInPolys) - $value) / $numInPolys]
}
# -----------------------------
# The smooth poly data GUI
# -----------------------------

proc proc_smooth {} {
    update_smooth_gui
    wm deiconify .smooth
}
proc close_smooth {} {
    wm withdraw .smooth
}
toplevel    .smooth
wm withdraw .smooth
wm title    .smooth "Smooth"
wm protocol .smooth WM_DELETE_WINDOW {wm withdraw .smooth}

frame .smooth.f1
scale .smooth.f1.num -label "Number Of Iterations" \
	-from 1 -to 1000 -length 3.0i -orient horizontal \
	-resolution 1
.smooth.f1.num set 100
scale .smooth.f1.fact -label "RelaxationFactor" \
	-from 0.00 -to 1.00 -length 3.0i -orient horizontal \
	-resolution 0.01
.smooth.f1.fact set 0.01
pack .smooth.f1.num .smooth.f1.fact \
	-pady 0.1i -side top -anchor w

frame .smooth.fb
button .smooth.fb.apply  -text Apply  -command apply_smooth
button .smooth.fb.cancel -text Cancel -command close_smooth
pack .smooth.fb.apply .smooth.fb.cancel -side left -expand 1 -fill x
pack .smooth.f1 .smooth.fb -side top -fill both -expand 1

proc update_smooth_gui {} {

   .smooth.f1.num set [smooth GetNumberOfIterations]
   .smooth.f1.fact set [smooth GetRelaxationFactor]
}
proc apply_smooth {} {
    global render_window

    smooth SetInput poly_data    

    smooth SetNumberOfIterations [.smooth.f1.num get]
    smooth SetRelaxationFactor [.smooth.f1.fact get]

    update_undo "smooth"
    update_gui

    $render_window Render
    close_smooth
}  
# ---------------
# The clean GUI
# ---------------

# Procedure defines GUI and behavior for cleaning data. Cleaning means
# removing degenerate polygons and eliminating coincident or unused points.

proc proc_clean {} {
    update_clean_gui
    wm deiconify .clean
}
proc close_clean {} {
    wm withdraw .clean
}

toplevel .clean
wm withdraw .clean
wm title .clean "Clean Data"
wm protocol .clean WM_DELETE_WINDOW {wm withdraw .clean}

frame .clean.f1
scale .clean.f1.s -label "Tolerance" \
	-from 0.000 -to 1.000 -length 3.0i -orient horizontal\
	-resolution 0.001 -digits 3
.clean.f1.s set 0.000
pack .clean.f1.s -side top -anchor w

frame .clean.fb
button .clean.fb.apply -text Apply -command apply_clean
button .clean.fb.cancel -text Cancel -command close_clean
pack .clean.fb.apply .clean.fb.cancel -side left -expand 1 -fill x
pack .clean.f1 .clean.fb -side top -fill both -expand 1

proc update_clean_gui {} {
    .clean.f1.s set [cleaner GetTolerance]
}

proc apply_clean {} {
    global render_window

    cleaner SetInput poly_data
    cleaner SetTolerance [.clean.f1.s get]

    update_undo "cleaner"
    update_gui

    $render_window Render
    close_clean
}
# --------------------
# The connectivity GUI
# --------------------

# Procedure defines GUI and behavior for extracting connected data. Connecting
# means extracting all cells joined at a vertex.

proc proc_connectivity {} {
    update_connectivity_gui
    wm deiconify .connect
}
proc close_connectivity {} {
    wm withdraw .connect
}

toplevel .connect
wm withdraw .connect
wm title .connect "Extract Connected Data"
wm protocol .connect WM_DELETE_WINDOW {wm withdraw .connect}

frame .connect.fb
button .connect.fb.apply -text Apply -command apply_connectivity
button .connect.fb.cancel -text Cancel -command close_connectivity
pack .connect.fb.apply .connect.fb.cancel -side left -expand 1 -fill x
pack .connect.fb -side top -fill both -expand 1

proc update_connectivity_gui {} {
}

proc apply_connectivity {} {
    global render_window

    connect SetInput poly_data

    update_undo "connect"
    update_gui

    $render_window Render
    close_connectivity
}
# -------------------
# The triangulate GUI
# -------------------

# Procedure defines GUI and behavior for triangulating data. This will
# convert all polygons into triangles

proc proc_triangulate {} {
    UpdateTriGUI
    wm deiconify .tri
}
proc CloseTri {} {
    wm withdraw .tri
}
toplevel .tri
wm withdraw .tri
wm title .tri "Triangulate Data"
wm protocol .tri WM_DELETE_WINDOW {wm withdraw .tri}

frame .tri.fb
button .tri.fb.apply  -text Apply  -command ApplyTri
button .tri.fb.cancel -text Cancel -command CloseTri
pack .tri.fb.apply .tri.fb.cancel -side left -expand 1 -fill x
pack .tri.fb -side top -fill both -expand 1

proc UpdateTriGUI {} {
}
proc ApplyTri {} {
    global render_window

    tri SetInput poly_data

    update_undo "tri"
    update_gui

    $render_window Render
    CloseTri
}
# ------------------------
# The surface normals GUI
# ------------------------

# Procedure defines GUI and behavior for generating surface normals. This will
# convert all polygons into triangles.

proc proc_normals {} {
    UpdateNormalsGUI
    wm deiconify .normals
}
proc CloseNormals {} {
    wm withdraw .normals
}

toplevel .normals
wm withdraw .normals
wm title .normals "Generate Surface Normals"
wm protocol .normals WM_DELETE_WINDOW {wm withdraw .normals}

frame .normals.f1
scale .normals.f1.fangle -label "Feature Angle" \
	-from 0 -to 180 -length 3.0i -orient horizontal -resolution 1
checkbutton .normals.f1.split -variable flag_edge_splitting \
	-text "Edge Splitting"
checkbutton .normals.f1.flip -variable flag_flip_normals \
	-text "Flip Normals"
pack .normals.f1.fangle .normals.f1.split .normals.f1.flip \
	-pady 0.1i -side top -anchor w

frame .normals.fb
button .normals.fb.apply -text Apply -command ApplyNormals
button .normals.fb.cancel -text Cancel -command CloseNormals
pack .normals.fb.apply .normals.fb.cancel -side left -expand 1 -fill x
pack .normals.f1 .normals.fb -side top -fill both -expand 1

proc UpdateNormalsGUI {} {
    .normals.f1.fangle set [normals GetFeatureAngle]
}
proc ApplyNormals {} {
    global flag_edge_splitting flag_flip_normals
    global render_window

    normals SetFeatureAngle [.normals.f1.fangle get]
    normals SetSplitting $flag_edge_splitting
    normals SetFlipNormals $flag_flip_normals

    normals SetInput poly_data

    update_undo "normals"
    update_gui

    $render_window Render
    CloseNormals
}
# ------------------------
# Setting background color
# ------------------------

proc set_background_color {} {

    set background [renderer GetBackground]
    .back.f1.l.r set [expr [lindex $background 0] * 255.0]
    .back.f1.l.g set [expr [lindex $background 1] * 255.0]
    .back.f1.l.b set [expr [lindex $background 2] * 255.0]
    wm deiconify .back
}
proc CloseBackground {} {
    wm withdraw .back
}
toplevel .back
wm withdraw .back
wm title .back "Select Background Color"
wm protocol .back WM_DELETE_WINDOW {wm withdraw .back}
frame .back.f1

frame .back.f1.l  -relief raised -borderwidth 3
scale .back.f1.l.r -from 255 -to 0 -orient vertical -background #f00 \
	-command SetColor
scale .back.f1.l.g -from 255 -to 0 -orient vertical -background #0f0 \
	-command SetColor
scale .back.f1.l.b -from 255 -to 0 -orient vertical -background #00f \
	-command SetColor
pack .back.f1.l.r .back.f1.l.g .back.f1.l.b -side left -fill both

frame .back.f1.m -relief raised -borderwidth 3
label .back.f1.m.sample -highlightthickness 0 -text "  Background Color  "
pack .back.f1.m.sample -fill both -expand 1

frame .back.f1.r -relief raised -borderwidth 3
image create photo color_wheel -file $color_wheel_file
label .back.f1.r.wheel -image color_wheel -highlightthickness 0
bind .back.f1.r.wheel <Button-1> {
    scan [color_wheel get %x %y] "%%f %%f %%f" r g b
    .back.f1.l.r set $r
    .back.f1.l.g set $g
    .back.f1.l.b set $b
}
pack .back.f1.r.wheel -fill both
pack .back.f1.l .back.f1.m .back.f1.r -side left -expand 1 -fill both

frame .back.fb
button .back.fb.apply -text Apply -command ApplyBackground
button .back.fb.cancel -text Cancel -command CloseBackground
pack .back.fb.apply .back.fb.cancel -side left -expand 1 -fill x
pack .back.f1 .back.fb -side top -fill both -expand 1

proc SetColor {value} {
    set color [format #%02x%02x%02x [.back.f1.l.r get] [.back.f1.l.g get]\
	    [.back.f1.l.b get]]
    .back.f1.m.sample config -background $color
}
proc ApplyBackground {} {
    global render_window

    renderer SetBackground [expr [.back.f1.l.r get]/255.0] \
	    [expr [.back.f1.l.g get]/255.0] \
	    [expr [.back.f1.l.b get]/255.0]
    comprare_renderer SetBackground [expr [.back.f1.l.r get]/255.0] \
	    [expr [.back.f1.l.g get]/255.0] \
	    [expr [.back.f1.l.b get]/255.0]
    Render $render_window
}
# -------------------------
# Set surface properties
# -------------------------
proc set_properties {} {

    set color [property GetColor]
    .prop.f1.l.r set [expr [lindex $color 0] * 255.0]
    .prop.f1.l.g set [expr [lindex $color 1] * 255.0]
    .prop.f1.l.b set [expr [lindex $color 2] * 255.0]
    .prop.sliders.amb set [property GetAmbient]
    .prop.sliders.diff set [property GetDiffuse]
    .prop.sliders.spec set [property GetSpecular]
    .prop.sliders.power set [property GetSpecularPower]

    wm deiconify .prop
}
proc CloseProperties {} {
    wm withdraw .prop
}
toplevel .prop
wm withdraw .prop
wm title .prop "Set Surface Properties"
wm protocol .prop WM_DELETE_WINDOW {wm withdraw .prop}
frame .prop.f1

frame .prop.f1.l  -relief raised -borderwidth 3
scale .prop.f1.l.r -from 255 -to 0 -orient vertical -background #f00 \
	-command SetSurfaceColor
scale .prop.f1.l.g -from 255 -to 0 -orient vertical -background #0f0 \
	-command SetSurfaceColor
scale .prop.f1.l.b -from 255 -to 0 -orient vertical -background #00f \
	-command SetSurfaceColor
pack .prop.f1.l.r .prop.f1.l.g .prop.f1.l.b -side left -fill both

frame .prop.f1.m -relief raised -borderwidth 3
label .prop.f1.m.sample -highlightthickness 0 -text "   Surface Color    "
pack .prop.f1.m.sample -fill both -expand 1

frame .prop.f1.r -relief raised -borderwidth 3
image create photo color_wheel -file $color_wheel_file
label .prop.f1.r.wheel -image color_wheel -highlightthickness 0
bind .prop.f1.r.wheel <Button-1> {
    scan [color_wheel get %x %y] "%%f %%f %%f" r g b
    .prop.f1.l.r set $r
    .prop.f1.l.g set $g
    .prop.f1.l.b set $b
}
pack .prop.f1.r.wheel -fill both
pack .prop.f1.l .prop.f1.m .prop.f1.r -side left -expand 1 -fill both

frame .prop.sliders
scale .prop.sliders.amb -from 0.00 -to 1.00 -orient horizontal\
	-resolution 0.01 -label Ambient
scale .prop.sliders.diff -from 0.00 -to 1.00 -orient horizontal\
	-resolution 0.01 -label Diffuse
scale .prop.sliders.spec -from 0.00 -to 1.00 -orient horizontal\
	-resolution 0.01 -label Specular
scale .prop.sliders.power -from 0 -to 100 -orient horizontal\
	-resolution 1 -label "Specular Power"
pack .prop.sliders.spec .prop.sliders.power .prop.sliders.amb\
	.prop.sliders.diff -side top -fill both

frame .prop.fb
button .prop.fb.apply -text Apply -command ApplyProperties
button .prop.fb.cancel -text Cancel -command CloseProperties
pack .prop.fb.apply .prop.fb.cancel -side left -expand 1 -fill x
pack .prop.f1 .prop.sliders .prop.fb -side top -fill both -expand 1

proc SetSurfaceColor {value} {
    set color [format #%02x%02x%02x [.prop.f1.l.r get] [.prop.f1.l.g get]\
	    [.prop.f1.l.b get]]
    .prop.f1.m.sample config -background $color
}
proc ApplyProperties {} {
    global render_window

    property SetColor [expr [.prop.f1.l.r get]/255.0] \
	    [expr [.prop.f1.l.g get]/255.0] \
	    [expr [.prop.f1.l.b get]/255.0]
    property SetAmbient [.prop.sliders.amb get]
    property SetDiffuse [.prop.sliders.diff get]
    property SetSpecular [.prop.sliders.spec get]
    property SetSpecularPower [.prop.sliders.power get]
    Render $render_window
}
# -----------------
# gauge progress
# -----------------
proc start_gauge_progress {filter label} {

    global gauge_bar_id
    global gauge_text_id
 
    set height [winfo height .gauge.status]
    set width [winfo width .gauge.status]
 
    if { ![winfo exists .gauge.canvas] } {
       canvas .gauge.canvas -height $height -width $width -borderwidth 0\
 	    -highlightthickness 0
    } else {
       .gauge.canvas configure -height $height -width $width
       .gauge.canvas delete $gauge_bar_id
       .gauge.canvas delete $gauge_text_id
    }
 
    set gauge_bar_id [.gauge.canvas create rect 0 0 0 $height -fill #888]
    set gauge_text_id [.gauge.canvas create text [expr $width/2] [expr $height/2] \
 	   -anchor center -justify center -text $label]
    pack forget .gauge.status
    pack .gauge.canvas -padx 0 -pady 0
 
    update
    return 0
}
proc show_gauge_progress {filter label} {
    global gauge_bar_id
    global gauge_text_id
 
    set progress [$filter GetProgress]
    set height [winfo height .gauge.status]
    set width [winfo width .gauge.status]
 
    .gauge.canvas delete $gauge_bar_id
    .gauge.canvas delete $gauge_text_id
    set gauge_bar_id [.gauge.canvas create rect 0 0 [expr $progress*$width] $height \
 	 -fill #888]
    set gauge_text_id [.gauge.canvas create text [expr $width/2] [expr $height/2] \
 	   -anchor center -justify center -text $label]
 
    update
    return 0
}
proc end_gauge_progress {} {
    pack forget .gauge.canvas
    pack .gauge.status -side top -anchor w -expand 1 -fill x
 
    update
    return 0
}
# -----------------------------------------------------------------------------------------
# prevent vtk-4.0 from prompting user when debugging leaks...
# -----------------------------------------------------------------------------------------
if {[get_vtk_major_version] >= 4} {
  vtkDebugLeaks debug_leaks
    debug_leaks PromptUserOff
#   debug_leaks DeleteTable
}
# -----------------------------------------------------------------------------------------
# usage
# -----------------------------------------------------------------------------------------
proc usage {} {
   puts "usage: editor \[\[no\]elevation\] <file>"
   exit 0
}
# -----------------------------------------------------------------------------------------
# parse command line arguments
# -----------------------------------------------------------------------------------------
set filename ""
# puts "argc = $argc";
for {set i 0} {$i < $argc} {incr i} {
        set argi [lindex $argv $i];
        # puts "argv($i) = `$argi'";
        if {$argi == "--help" || $argi == "-h"} {
            usage
        } elseif {$argi == "-elevation"} {
	    set do_elevation 1
    	    set flag_elevation 1
        } elseif {$argi == "-noelevation"} {
	    set do_elevation 0
    	    set flag_elevation 0
        } else {
            # puts "filename $argi"
            set filename $argi
        }
}
if {$filename != ""} {
        open_file $filename 
}

