# ifndef _SKIT_DNS_H
# define _SKIT_DNS_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Class:csr
NAME: @code{dns} - dense matrix format 
DESCRIPTION:       
 The class implements a float matrix in dense format.
 A declaration whithout any parametrers correspond to a matrix with null size:
 @example
          dns<float> m;
 @end example

 The constructor can be invocated whith row, column:
 @example
          dns<float> m(nrow, ncol);
 @end example

 Notes that the constructor can be invocated with an initializer:
 @example
          dns<float> m = m2;
 @end example
 
 Explicit conversion is allowed from @ref{csr class}:
 @example
          csr<float> a = ...
          m = dns(a);
 @end example
 
 Affectation from a scalar
 @example
          m = 3.14;
 @end example

NOTE:
 The @code{dns} class supports also a STL-like interface.
 This interface is not yet stabilized.

AUTHOR: 
     Pierre Saramito
   | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:   21 january 1997
End:
*/

//<dns:
#include "rheolef/array.h"
namespace rheolef { 

template<class T>
class dns : public Array<T> {
public:
// typedefs:

    typedef          T                        element_type; 
    typedef typename Array<T>::size_type      size_type; 
    typedef typename Array<T>::iterator       iterator; 
    typedef typename Array<T>::const_iterator const_iterator; 

// allocators/deallocators:

	explicit dns(Index nr = 0, Index nc = 0)
	: Array<T>(nr*nc), NROW(nr), NCOL(nc) {}
	
	dns(const dns& d) 
	: Array<T>(d), NROW(d.NROW), NCOL(d.NCOL) {}

       	explicit dns (const csr<T>& a);

// assignments:

       	const dns<T>& operator = (const csr<T>& a);
    	dns<T> operator = (const T& lambda);

// accessors:

	Index nrow() const	{ return NROW; }
	Index ncol() const	{ return NCOL; }
        Index nnz () const;
	
	size_type size() const  { return nrow()*ncol(); }
	bool empty() const { return (size() == 0); }

	T operator() (Index i, Index j) const
	{ 
           assert_macro (i < nrow(), "index #1 " << i << " out of range 0.." << nrow() - 1);
           assert_macro (j < ncol(), "index #2 " << i << " out of range 0.." << ncol() - 1);
           return data().at(nrow()*j + i); 
	}
	T& operator() (Index i, Index j)
	{
           assert_macro (i < nrow(), "index #1 " << i << " out of range 0.." << nrow() - 1);
           assert_macro (j < ncol(), "index #2 " << i << " out of range 0.." << ncol() - 1);
           return data().at(nrow()*j + i); 
	}
protected:
//data:
       	Index NROW;	// row dimension
       	Index NCOL;	// column dimension
	const vector_rep<T>& data() const { return Array<T>::data(); }
	vector_rep<T>& data() { return Array<T>::data(); }
};
// io routines
template<class T> std::istream& operator >> (std::istream&, dns<T>&);
template<class T> std::ostream& operator << (std::ostream&, const dns<T>&);
//>dns:
}// namespace rheolef
# endif /* _SKIT_DNS_H */
