# pyOCD debugger
# Copyright (c) 2018-2019 Arm Limited
# Copyright (c) 2021 Chris Reed
# SPDX-License-Identifier: Apache-2.0
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging

from ...coresight.coresight_target import CoreSightTarget
from ...core.memory_map import (FlashRegion, RamRegion, MemoryMap)
from ...debug.svd.loader import SVDFile

LOG = logging.getLogger(__name__)

RESET_MASK = 0x50021104
RESET_MASK_SYSRSTREQ0_EN = 1 << 4
RESET_MASK_SYSRSTREQ1_EN = 1 << 5

FLASH_ALGO_QSPI = {
    'load_address' : 0x20000000,
    'instructions': [
    0xE00ABE00, 0x062D780D, 0x24084068, 0xD3000040, 0x1E644058, 0x1C49D1FA, 0x2A001E52, 0x4770D1F2,
    0x4604b570, 0x4616460d, 0x44484867, 0xb9e86800, 0x44784866, 0x44494966, 0x46086008, 0x44494965,
    0x46086008, 0x44494964, 0x68096008, 0xf0006808, 0x4961fc96, 0x68084449, 0xf0002101, 0xb108f9bb,
    0xbd702001, 0x49582001, 0x60084449, 0xe7f82000, 0x4604b510, 0x4854b984, 0x68004448, 0xd10b2801,
    0x44494955, 0xf0006808, 0xb108fa1f, 0xbd102001, 0x494d2000, 0x60084449, 0xe7f82000, 0x494eb510,
    0x22004449, 0x46116808, 0xfba4f000, 0x2001b108, 0x2000bd10, 0xb510e7fc, 0xf0244604, 0x4604407f,
    0x44494945, 0x3280f44f, 0x46216808, 0xfb92f000, 0x2001b108, 0x2000bd10, 0xe92de7fc, 0x460441f0,
    0x4617460d, 0x407ff024, 0x493b4604, 0x462b4449, 0x6808463a, 0xf0004621, 0x4606faf2, 0x4630b116,
    0x81f0e8bd, 0xe7fb2000, 0x43f8e92d, 0x46884607, 0xf0274615, 0xbf00467f, 0xe0232400, 0x4449492e,
    0x466a2304, 0x46316808, 0xfa8bf000, 0x0000f89d, 0x42885d29, 0xf89dd111, 0x1c601001, 0x42815c28,
    0xf89dd10b, 0x1ca01002, 0x42815c28, 0xf89dd105, 0x1ce01003, 0x42815c28, 0x1930d002, 0x83f8e8bd,
    0x1d241d36, 0xd3d94544, 0xe7f72000, 0x43f8e92d, 0x460f4606, 0xf0264614, 0xbf00457f, 0x0800f04f,
    0x4915e01d, 0x23044449, 0x6808466a, 0xf0004629, 0xf89dfa58, 0x42a00000, 0xf89dd10b, 0x42a00001,
    0xf89dd107, 0x42a00002, 0xf89dd103, 0x42a00003, 0x2001d002, 0x83f8e8bd, 0xf1081d2d, 0x45b80804,
    0x2000d3df, 0x0000e7f6, 0x00000004, 0x00000d9a, 0x00000008, 0x0000000c, 0x00000020, 0xb118b570,
    0x2d2018d5, 0xb902d800, 0x2a20bd70, 0x6001d101, 0x2501e7fa, 0x1e6c4095, 0xfa046805, 0x43b5f603,
    0x0604ea01, 0x4335409e, 0xbf006005, 0xb510e7ec, 0x21064604, 0xf0006820, 0xbd10fd7b, 0x4604b57f,
    0x2000460d, 0x90009003, 0x90029001, 0xaa032301, 0x68202165, 0xfd7ef000, 0xb1164606, 0xb0044630,
    0xb125bd70, 0xd00d2d01, 0xd1212d02, 0xf89de015, 0xf040000c, 0x90030040, 0x000cf89d, 0x0080f040,
    0xe0179003, 0x000cf89d, 0x0040f020, 0xf89d9003, 0xf040000c, 0x90030080, 0xf89de00c, 0xf040000c,
    0x90030040, 0x000cf89d, 0x0080f020, 0xe0019003, 0xe7d42001, 0x4620bf00, 0xffb9f7ff, 0x90002000,
    0x90029001, 0xaa032301, 0x68202161, 0xfd5ff000, 0xb10e4606, 0xe7c24630, 0x462a462b, 0x68204629,
    0xfb83f000, 0xb10e4606, 0xe7b84630, 0xe7b62000, 0x4604b57f, 0x2000460e, 0x90009003, 0x90029001,
    0xaa032301, 0x68202185, 0xfd24f000, 0xb1154605, 0xb0044628, 0x2304bd70, 0x4631461a, 0xf7ffa803,
    0x4620ff6d, 0xff83f7ff, 0x90002000, 0x90029001, 0xaa032301, 0x68202181, 0xfd29f000, 0xb10d4605,
    0xe7e64628, 0xe7e42000, 0x4605b57f, 0x90032000, 0x2000e00e, 0x90019000, 0x23019002, 0x2170aa03,
    0xf0006828, 0x4604fcf7, 0x4620b114, 0xbd70b004, 0x000cf89d, 0x280009c0, 0x2000d0eb, 0xe92de7f6,
    0xb0844dff, 0x46924606, 0x9c10469b, 0xf5c0b2e0, 0xe9dd7580, 0xe9cd1011, 0x90024100, 0x4652462b,
    0x99056830, 0xfcf3f000, 0xb11f4607, 0xb0084638, 0x8df0e8bd, 0x442c44aa, 0xf7ff4630, 0x4680ffc5,
    0x0f00f1b8, 0x4640d001, 0x4630e7f1, 0xff2ff7ff, 0x1011e9dd, 0x0305ebab, 0x4100e9cd, 0x46529002,
    0x99056830, 0xfcd3f000, 0xb10f4607, 0xe7de4638, 0xe7dc2000, 0x4df3e92d, 0x4604b082, 0xb1209803,
    0xd00b2801, 0xd11b2802, 0x2700e011, 0x0a03f04f, 0xf04f2600, 0x20000b02, 0xe0159001, 0xf04f2700,
    0x26080a0b, 0x0b02f04f, 0x90012000, 0x2702e00c, 0x0a6bf04f, 0xf04f2608, 0x20000b32, 0xe0039001,
    0xb0042001, 0x8df0e8bd, 0x4639bf00, 0xf7ff4620, 0x4680fef5, 0x0f00f1b8, 0x4640d001, 0x4631e7f1,
    0xf7ff4620, 0x4680ff45, 0x0f00f1b8, 0x4640d001, 0xbf00e7e7, 0xf0006820, 0x2800fa40, 0x6820d0fa,
    0xfa85f000, 0x682068a1, 0xfa91f000, 0xb10d4605, 0xe7d64628, 0x46514632, 0xf0006820, 0x4605faff,
    0x4628b10d, 0x6820e7cd, 0x9a014659, 0xfb01f000, 0xb10d4605, 0xe7c44628, 0x7180f44f, 0xf0006820,
    0x4605fb03, 0x4628b10d, 0x2103e7bb, 0xf0006820, 0x4605fb14, 0x4628b10d, 0x6820e7b3, 0xfa5ff000,
    0x74209803, 0xe7ac2000, 0x4604b570, 0x46202100, 0xfea4f7ff, 0xb10d4605, 0xbd704628, 0x46202100,
    0xfef6f7ff, 0xb10d4605, 0xe7f64628, 0x6820bf00, 0xf9f3f000, 0xd0fa2800, 0xf0006820, 0x6820fa38,
    0xfb30f000, 0xf0006820, 0x2000fa3a, 0xbf007420, 0xe92de7e3, 0x46044df0, 0x46174688, 0x6820461d,
    0x68406800, 0xfbb51c46, 0xb107fbf6, 0x2001b915, 0x8df0e8bd, 0x0005eb08, 0x428868e1, 0x2006d301,
    0xf04fe7f6, 0xe00d0a00, 0x68204641, 0xfadef000, 0x46384632, 0xf0006861, 0x1badfc29, 0x44b04437,
    0x0a01f10a, 0xd3ef45da, 0x4641b145, 0xf0006820, 0x462afacd, 0x68614638, 0xfc18f000, 0xf0006820,
    0x2000fadc, 0xe92de7d4, 0x46044dfc, 0x4692460f, 0x6820461e, 0x68406800, 0x0801f100, 0x0f00f1ba,
    0xb916d000, 0xe8bd2001, 0x19b88dfc, 0x428868e1, 0x2006d301, 0xf007e7f7, 0x1b7f0503, 0xfbb01970,
    0x9001f0f8, 0x90002000, 0x4639e01e, 0xf0006820, 0xeba8fa9d, 0x68610205, 0x46511948, 0xfbe6f000,
    0x0005eba8, 0xeba81a36, 0x44820005, 0x25004447, 0xf7ff4620, 0x4683fe99, 0x0f00f1bb, 0x4658d001,
    0x9800e7d1, 0x90001c40, 0x0100e9dd, 0xd3dc4288, 0x4639b196, 0xf0006820, 0x6861fa79, 0x46321948,
    0xf0004651, 0x4620fbc3, 0xfe7ef7ff, 0xf1bb4683, 0xd0010f00, 0xe7b64658, 0xf0006820, 0x2000fa7e,
    0xe92de7b1, 0xb0844dff, 0x460e4605, 0x08f8461f, 0x7c289003, 0x2801b160, 0x2802d005, 0xf04fd107,
    0x24080a6b, 0xf04fe008, 0x24080a0b, 0xbf00e004, 0x0a03f04f, 0xbf002400, 0xf04fbf00, 0xe0180b00,
    0xe9cd2003, 0x94026000, 0x68282308, 0x9a064651, 0xfb50f000, 0xf1b84680, 0xd0030f00, 0xb0084640,
    0x8df0e8bd, 0x30089806, 0x36089006, 0xf10b3f08, 0x98030b01, 0xd3e34583, 0x2003b17f, 0x6000e9cd,
    0x463b9402, 0x46516828, 0xf0009a06, 0x4680fb33, 0x0f00f1b8, 0x4640d001, 0x2000e7e1, 0xe92de7df,
    0xb0864dff, 0x460c4680, 0x08f0461e, 0xf8989005, 0xb1480010, 0xd0062801, 0xd1032802, 0x0b32f04f,
    0xe0052500, 0xbf00bf00, 0x0b02f04f, 0xbf002500, 0x2000bf00, 0xe0379004, 0xf7ff4640, 0x0a21fd78,
    0xebb11de0, 0xd00f2f10, 0x23082003, 0x0501e9cd, 0x46599400, 0x9a084640, 0xfe11f7ff, 0xb1a74607,
    0xb00a4638, 0x8df0e8bd, 0xe9cd2003, 0x94000501, 0xf8d82308, 0x46590000, 0xf0009a08, 0x4682fb08,
    0x0f00f1ba, 0x4650d001, 0x4640e7eb, 0xfddcf7ff, 0xb10f4607, 0xe7e44638, 0x30089808, 0x34089008,
    0x98043e08, 0x90041c40, 0x0104e9dd, 0xd3c34288, 0x4640b376, 0xfd3bf7ff, 0x19a00a21, 0xebb11e40,
    0xd00d2f10, 0x46332003, 0x0501e9cd, 0x46599400, 0x9a084640, 0xfdd3f7ff, 0xb1974607, 0xe7c04638,
    0xe9cd2003, 0x94000501, 0xf8d84633, 0x46590000, 0xf0009a08, 0x4682facc, 0x0f00f1ba, 0x4650d001,
    0x4640e7af, 0xfda0f7ff, 0xb10f4607, 0xe7a84638, 0xe7a62000, 0x4df0e92d, 0x4607b086, 0x4693460c,
    0xf7ff4638, 0xf1bbfd04, 0xd0090f00, 0x5f80f5bb, 0xf5bbd01d, 0xd0124f00, 0x3f80f5bb, 0xe007d11f,
    0x2005b11c, 0xe8bdb006, 0x25c78df0, 0xe0182600, 0x260325d8, 0xb108b2a0, 0xe7f32005, 0x2552e011,
    0xf3c42603, 0xb108000e, 0xe7eb2005, 0x2520e009, 0xf3c42603, 0xb108000b, 0xe7e32005, 0x2001e001,
    0xbf00e7e0, 0x42a068f8, 0x2006d801, 0x2000e7da, 0xc151a901, 0x90009004, 0x461a2300, 0x68384629,
    0xf986f000, 0xf1b84680, 0xd0010f00, 0xe7c94640, 0xf7ff4638, 0x4682fd49, 0x0f00f1ba, 0x4640d001,
    0x2000e7c0, 0x0000e7be, 0xb118b570, 0x2d2018d5, 0xb902d800, 0x2a20bd70, 0x6001d101, 0x2501e7fa,
    0x1e6c4095, 0xfa046805, 0x43b5f603, 0x0604ea01, 0x4335409e, 0xbf006005, 0x4601e7ec, 0x68026808,
    0x0fc06810, 0xe92d4770, 0x460545f8, 0x4614468a, 0x6828461e, 0xb10e6807, 0xe0011d38, 0x0008f107,
    0xf8d84680, 0x90000000, 0x4628bf00, 0xffe5f7ff, 0xd0fa2800, 0xd9022c1f, 0xe8bd2001, 0x230085f8,
    0x46512208, 0xf7ff4668, 0x2318ffbf, 0x46212205, 0xf7ff4668, 0x9800ffb9, 0x0000f8c8, 0xe7ec2000,
    0xb1214601, 0xd0042901, 0xd1062902, 0x2000e003, 0x20014770, 0x2002e7fc, 0xf04fe7fa, 0xe7f730ff,
    0x68084601, 0x68106802, 0x0001f000, 0x46014770, 0x6810680a, 0xf0226802, 0x60020201, 0x46014770,
    0x6810680a, 0xf0426802, 0x60020201, 0xb5704770, 0x460c4605, 0x68066828, 0x4628bf00, 0xff9df7ff,
    0xd0fa2800, 0x0001f004, 0x2c02b918, 0x2c20d301, 0x2001d901, 0x2001bd70, 0x0154ebc0, 0x22042313,
    0xf7ff4630, 0x2000ff71, 0xe92de7f4, 0xb0824dff, 0x460f4682, 0xf8da4693, 0x68040000, 0x90016860,
    0x900068a0, 0x4650bf00, 0xff77f7ff, 0xd0fa2800, 0xf7ff4638, 0x4605ffa5, 0xb9181c68, 0xb0062001,
    0x8df0e8bd, 0x2308b13f, 0x46292202, 0xf7ff1d20, 0x2000ff4b, 0x4658e7f3, 0xff92f7ff, 0x98054680,
    0xff8ef7ff, 0xf1084606, 0xb1080001, 0xb9081c70, 0xe7e42001, 0x22022308, 0xa8014629, 0xff34f7ff,
    0x2202230c, 0xa8014641, 0xff2ef7ff, 0x22022310, 0xa8014631, 0xff28f7ff, 0x2202230c, 0x46684641,
    0xff22f7ff, 0x22022310, 0x46684631, 0xff1cf7ff, 0x60609801, 0x60a09800, 0xe7c02000, 0x4604b570,
    0x4616460d, 0x46322301, 0x46204629, 0xff2bf7ff, 0xb570bd70, 0x460d4604, 0x23004616, 0x46294632,
    0xf7ff4620, 0xbd70ff20, 0x4604b570, 0x6820460d, 0xbf006806, 0xf7ff4620, 0x2800ff10, 0xf5b5d0fa,
    0xd3015f80, 0xbd702001, 0x220c2304, 0xf1064629, 0xf7ff0014, 0x2000fee9, 0xb570e7f5, 0x460c4605,
    0x68066828, 0x4628bf00, 0xfef7f7ff, 0xd0fa2800, 0x2c10b10c, 0x2001d901, 0x1e61bd70, 0x22042300,
    0x0014f106, 0xfed0f7ff, 0xe7f52000, 0x4604b570, 0x6820460d, 0x46206803, 0xff22f7ff, 0xb1164606,
    0xf7ff4620, 0x625dff24, 0xf4406818, 0x60183080, 0x4620b116, 0xff23f7ff, 0xb530bd70, 0x68184603,
    0x46186804, 0xff0cf7ff, 0xb1154605, 0xf7ff4618, 0x6820ff0e, 0x3080f420, 0xb1156020, 0xf7ff4618,
    0xbd30ff0e, 0x680a4601, 0x4a8e6810, 0x22036002, 0x22026042, 0x4a8c6082, 0x22006142, 0xf8c06242,
    0xf8c02090, 0xf8c02094, 0xf8c020a8, 0x477020ac, 0x4dffe92d, 0x4616b086, 0xf8dd461d, 0xe9dda054,
    0x98068712, 0x68046800, 0x90052000, 0xb10db116, 0xe0002001, 0x46832000, 0x0f00f1b8, 0xb10fd002,
    0xe0002001, 0x90042000, 0x0f00f1ba, 0x2001d001, 0x2000e000, 0x20009003, 0x90029001, 0xd9032d08,
    0xb00a2001, 0x8df0e8bd, 0xd9012f08, 0xe7f82001, 0x0f04f1ba, 0x2001d901, 0x9816e7f3, 0xd901281f,
    0xe7ee2001, 0x0f00f1bb, 0x9804d003, 0x2001b108, 0x2318e7e7, 0xa8052208, 0xf7ff9907, 0xf1bbfe45,
    0xd0090f00, 0xf4409805, 0x90050000, 0x23141e69, 0xa8052203, 0xfe38f7ff, 0xb3289804, 0xf4409805,
    0x90054000, 0x230c1e79, 0xa8052203, 0xfe2cf7ff, 0x0003f008, 0xf007b968, 0xb9500003, 0x0000f8d8,
    0x00a8f8c4, 0xd10f2f08, 0x0004f8d8, 0x00acf8c4, 0x463ae00a, 0xa8014641, 0xf898f000, 0xf8c49801,
    0x980200a8, 0x00acf8c4, 0xb1689803, 0xf4409805, 0x90052000, 0xf8c49814, 0xf1aa0094, 0x23100101,
    0xa8052202, 0xfe00f7ff, 0x22052307, 0x9916a805, 0xfdfaf7ff, 0xf8449805, 0x68200f90, 0x0001f040,
    0x0990f844, 0xf8d4bf00, 0xf3c00090, 0x28000040, 0xf1bbd1f9, 0xd0190f00, 0x0003f006, 0xf005b958,
    0xb9400003, 0x00a0f8d4, 0x2d086030, 0xf8d4d10e, 0x607000a4, 0xf8d4e00a, 0x900100a0, 0x00a4f8d4,
    0x462a9002, 0x4630a901, 0xf850f000, 0xe7682000, 0xb085b530, 0x460d4604, 0x90012000, 0x90039002,
    0x46039004, 0x46294602, 0x46209000, 0xff30f7ff, 0xbd30b005, 0x4df0e92d, 0x4606b086, 0x4614460f,
    0xe9dd461d, 0xf8ddab0f, 0xb1048038, 0x2001b91d, 0xe8bdb006, 0x20008df0, 0xe8a1a901, 0x462b0d01,
    0x46394622, 0x46309000, 0xff12f7ff, 0xe92de7f0, 0xb0864df0, 0x460f4606, 0x461d4614, 0xab0fe9dd,
    0x8038f8dd, 0xb91db104, 0xb0062001, 0x8df0e8bd, 0x461a2300, 0x46304639, 0x0d30e88d, 0xfef8f7ff,
    0x0000e7f3, 0x80780081, 0x00101002, 0x0301ea40, 0xd003079b, 0xc908e009, 0xc0081f12, 0xd2fa2a04,
    0xf811e003, 0xf8003b01, 0x1e523b01, 0x4770d2f9, 0x52800000, 0x0003ffff, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000004, 0x00800000, 0x00000000, 0x00000000
    ],

    # Function addresses
    'pc_init': 0x20000021,
    'pc_unInit': 0x20000071,
    'pc_program_page': 0x200000db,
    'pc_erase_sector': 0x200000b7,
    'pc_eraseAll': 0x2000009d,

    'static_base' : 0x20000000 + 0x00000020 + 0x00000db8,
    'begin_stack' : 0x20001000,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x100,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20001000, 0x20001100],   # Enable double buffering
    'min_program_length' : 0x100,

    # Flash information
    'flash_start': 0x0,
    'flash_size': 0x800000,
    'sector_sizes': (
        (0x0, 0x10000),
    )
    }

FLASH_ALGO_EFLASH = {
    'load_address' : 0x20000000,

    # Flash algorithm as a hex string
    'instructions': [
    0xE00ABE00, 0x062D780D, 0x24084068, 0xD3000040, 0x1E644058, 0x1C49D1FA, 0x2A001E52, 0x4770D1F2,
    0x4604b570, 0x4616460d, 0x44484866, 0xb9986800, 0x44484865, 0x44494965, 0x48656048, 0x60084448,
    0x49644608, 0x60084449, 0x49636808, 0xf8dbf000, 0x495c2001, 0x60084449, 0xbd702000, 0xb9414601,
    0x44484858, 0x28016800, 0x2000d103, 0x444a4a55, 0x20006010, 0xb5104770, 0x44494956, 0x68082202,
    0xf0002100, 0xb108fa2b, 0xbd102001, 0xe7fc2000, 0x4604b510, 0x6020f1a4, 0x4449494b, 0x1e496849,
    0x0401ea00, 0x4449494b, 0x68082200, 0xf0004621, 0xb108fa15, 0xbd102001, 0xe7fc2000, 0x43f8e92d,
    0x460d4604, 0xf1a44616, 0x493f6020, 0x68494449, 0xea001e49, 0x20040401, 0xea4f9000, 0x27000895,
    0x493ce00e, 0x466b4449, 0x68084632, 0xf0004621, 0xb110f94b, 0xe8bd2001, 0x1d3683f8, 0x1c7f1d24,
    0xd3ee4547, 0xe7f62000, 0x41fce92d, 0x460f4604, 0xf1a44690, 0x492c6020, 0x68494449, 0xea001e49,
    0x20040401, 0x25009001, 0x492ae01a, 0xab014449, 0x6808466a, 0xf0004621, 0x2600f8c1, 0xf81de00c,
    0x19a90006, 0x1001f818, 0xd0044288, 0x6020f104, 0xe8bd4428, 0x1c7681fc, 0xd3f02e04, 0x1d2d1d24,
    0xd3e242bd, 0xe7f42000, 0x41fce92d, 0x460e4604, 0xf1a44617, 0x49146020, 0x68494449, 0xea001e49,
    0x20040401, 0xf04f9001, 0xe0160800, 0x44494911, 0x466aab01, 0x46216808, 0xf890f000, 0xe0072500,
    0x0005f81d, 0xd00242b8, 0xe8bd2001, 0x1c6d81fc, 0xd3f52d04, 0xf1081d24, 0x45b00804, 0x2000d3e6,
    0x0000e7f3, 0x00000004, 0x0000000c, 0x00000014, 0x00000008, 0x0000001c, 0x02710000, 0x6804b510,
    0xb11a6823, 0x041ff001, 0xe002601c, 0x041ff001, 0xbd10605c, 0x4604b510, 0x68096821, 0x5080f501,
    0xf9eaf000, 0xb570bd10, 0x460d4604, 0x211f2200, 0xf7ff4620, 0x6821ffe3, 0xf5016809, 0x46295080,
    0xf9cef000, 0xf7ff4620, 0x6861ffe5, 0x20016048, 0x70086861, 0xb570bd70, 0x25004604, 0x69a0bf00,
    0x001cf000, 0xd0fa1e05, 0x0010f005, 0x4620b140, 0xf9b3f000, 0x69a0bf00, 0x000cf000, 0xd0fa1e05,
    0xbd704628, 0x4604b570, 0xbf00460d, 0xf00069a0, 0x28000001, 0x61e5d1fa, 0x61602001, 0xf7ff4620,
    0xbd70ffda, 0x4604b5f0, 0x462e460d, 0x0020f104, 0x2a1018c7, 0x2010d901, 0x4610e000, 0x18d04602,
    0xd9002810, 0x21001ad2, 0x7838e004, 0x1c7f7030, 0x1c491c76, 0xd3f84291, 0xbdf04610, 0x4dfce92d,
    0x460f4604, 0x461d4692, 0x68006820, 0x682e9001, 0x0b00f04f, 0x200046d8, 0x68609000, 0xb9107800,
    0xe8bd2001, 0x68288dfc, 0x68614438, 0x42886849, 0x2005d901, 0x9801e7f5, 0xb1086980, 0xe7f02002,
    0x68096821, 0x5080f501, 0xf963f000, 0xd0012880, 0xe7e62003, 0x4639e015, 0xf7ff9801, 0x4683ffa3,
    0x000ff007, 0x46329000, 0xe9dd4651, 0xf7ff3000, 0x4680ffa9, 0x0608eba6, 0x44c24447, 0xf0009801,
    0xb11ef93c, 0x0008f00b, 0xd0e42800, 0x0008f00b, 0x4446b100, 0x1b806828, 0xf00b6028, 0xb1080008,
    0xe7be2006, 0xe7bc2000, 0x41f0e92d, 0x460d4604, 0x61e54616, 0x20026226, 0x46206160, 0xff5bf7ff,
    0x46384607, 0x81f0e8bd, 0x4df0e92d, 0x460e4605, 0x461c4690, 0xf8d06828, 0xf04fb000, 0x68270a00,
    0x78006868, 0x2001b910, 0x8df0e8bd, 0x44306820, 0x68496869, 0xd9014288, 0xe7f52005, 0x0018f8db,
    0x2002b108, 0xf006e7f0, 0xb9180003, 0xf0007820, 0xb1080003, 0xe7e72007, 0x4631e00d, 0xf8d84658,
    0xf7ff2000, 0x4682ffc1, 0xf1081d36, 0x1f3f0804, 0xf0004658, 0xb11ff8e2, 0x0008f00a, 0xd0ec2800,
    0x0008f00a, 0x1d3fb100, 0x1bc06820, 0xf00a6020, 0xb1080008, 0xe7c72006, 0xe7c52000, 0x4ff8e92d,
    0x460e4607, 0x461c4692, 0x68056838, 0x90002000, 0x8000f8d4, 0x0b02f04f, 0x78006878, 0x2001b910,
    0x8ff8e8bd, 0x44306820, 0x68496879, 0xd9014288, 0xe7f52005, 0xb10869a8, 0xe7f12002, 0x0003f006,
    0x7820b918, 0x0003f000, 0x2007b108, 0xe023e7e8, 0xf8da61ee, 0x62280000, 0x61682003, 0x9800e00c,
    0x0008f000, 0x6820b140, 0x0008eba0, 0x46286020, 0xf893f000, 0xe7d32006, 0x900069a8, 0x000bea00,
    0xd1ec4558, 0x0b06f04f, 0xf10a1d36, 0xf1a80a04, 0x46280804, 0xf881f000, 0x0f00f1b8, 0xbf00d1d8,
    0xf00069a8, 0x28000004, 0x6820d0fa, 0x0008eba0, 0x46286020, 0xf871f000, 0xe7b12000, 0x41f0e92d,
    0x460e4605, 0x68284617, 0xf04f6804, 0x68680800, 0xb9107800, 0xe8bd2001, 0x69a081f0, 0x2002b108,
    0xb127e7f9, 0xd0132f01, 0xd12a2f02, 0x6868e01c, 0x42b06840, 0x2005d201, 0x61e6e7ed, 0x61602004,
    0xf7ff4620, 0x4680fe88, 0xf0004620, 0xe01af846, 0x61e02000, 0x61602007, 0xf7ff4620, 0x4680fe7c,
    0xf0004620, 0xe00ef83a, 0x1000f44f, 0x200761e0, 0x46206160, 0xfe6ff7ff, 0x46204680, 0xf82df000,
    0x2004e001, 0xbf00e7c7, 0x0f08f1b8, 0x2006d101, 0x2000e7c1, 0x4601e7bf, 0x68026808, 0xf3c06990,
    0x47701040, 0x4604b510, 0x68096821, 0x5080f501, 0xf826f000, 0xb510bd10, 0x68214604, 0xf5016809,
    0xf0005080, 0xbd10f821, 0x4604b510, 0x68096821, 0x5080f501, 0xf81bf000, 0x211fbd10, 0x477060c1,
    0x4601460a, 0x600b4b0c, 0x604b4b0c, 0x608b4b0c, 0x46014770, 0x47702080, 0xf44f4601, 0x47701000,
    0xf44f4601, 0x47704080, 0x20034601, 0x46014770, 0x6990460a, 0x00004770, 0x11082801, 0x64050208,
    0x0a0a0a08, 0x00000000, 0x00000000, 0x52400000, 0x00000000, 0x00400000, 0x00000000, 0x00000000,
    0x00000000
    ],

    # Relative function addresses
    'pc_init': 0x20000021,
    'pc_unInit': 0x2000005d,
    'pc_program_page': 0x200000bd,
    'pc_erase_sector': 0x20000091,
    'pc_eraseAll': 0x20000077,

    'static_base' : 0x20000000 + 0x00000020 + 0x000005e4,
    'begin_stack' : 0x20000900,
    'begin_data' : 0x20000000 + 0x1000,
    'page_size' : 0x4000,
    'analyzer_supported' : False,
    'analyzer_address' : 0x00000000,
    'page_buffers' : [0x20001000, 0x20005000],   # Enable double buffering
    'min_program_length' : 0x4000,

    # Flash information
    'flash_start': 0xa000000,
    'flash_size': 0x400000,
    'sector_sizes': (
        (0x0, 0x4000),
    )
    }

class MuscaB1(CoreSightTarget):

    VENDOR = "Arm"

    MEMORY_MAP = MemoryMap(
        FlashRegion(name='neflash',     start=0x0A000000, length=0x00200000, access='rx',
                        blocksize=0x4000,
                        page_size=0x4000,
                        is_boot_memory=False,
                        is_default=False,
                        algo=FLASH_ALGO_EFLASH),
        FlashRegion(name='seflash',     start=0x1A000000, length=0x00200000, access='rxs',
                        blocksize=0x4000,
                        page_size=0x4000,
                        is_boot_memory=False,
                        is_default=False,
                        algo=FLASH_ALGO_EFLASH,
                        alias='neflash'),
        FlashRegion(name='nqspi',       start=0x00000000, length=0x00800000, access='rx',
                        blocksize=0x10000,
                        page_size=0x10000,
                        is_boot_memory=True,
                        is_external=True,
                        algo=FLASH_ALGO_QSPI),
        FlashRegion(name='sqspi',       start=0x10000000, length=0x00800000, access='rxs',
                        blocksize=0x10000,
                        page_size=0x10000,
                        is_boot_memory=True,
                        is_external=True,
                        algo=FLASH_ALGO_QSPI,
                        alias='nqspi'),
        RamRegion(  name='ncoderam',    start=0x0A400000, length=0x00080000, access='rwx'),
        RamRegion(  name='scoderam',    start=0x1A400000, length=0x00080000, access='rwxs',
                        alias='ncoderam'),
        # Due to an errata, the first 8 kB of sysram is not accessible to the debugger.
        RamRegion(  name='nsysram',     start=0x20002000, length=0x0007e000, access='rwx'),
        RamRegion(  name='ssysram',     start=0x30002000, length=0x0007e000, access='rwxs',
                        alias='nsysram'),
        )

    def __init__(self, session):
        super(MuscaB1, self).__init__(session, self.MEMORY_MAP)
        self._svd_location = SVDFile.from_builtin("Musca_B1.svd")

    def create_init_sequence(self):
        seq = super(MuscaB1, self).create_init_sequence()

        seq.insert_before('halt_on_connect',
            ('enable_sysresetreq',        self._enable_sysresetreq),
            )

        return seq

    def _enable_sysresetreq(self):
        LOG.info("Enabling SYSRSTREQ0_EN and SYSRSTREQ1_EN")
        reset_mask = self.read32(RESET_MASK)
        reset_mask |= RESET_MASK_SYSRSTREQ0_EN | RESET_MASK_SYSRSTREQ1_EN
        self.write32(RESET_MASK, reset_mask)


