/*
 * Copyright © 2011 Intel Corporation
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the next
 * paragraph) shall be included in all copies or substantial portions of the
 * Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 *
 *
 */

/**
 * @file ext_packed_float/getteximage-invalid-format-for-packed-type.c
 *
 * Section 8.4.4.2 Special Interpretations in Section 8.4 Pixel Rectangles of
 * the OpenGL 4.5 core spec (30.10.2014) says:
 *
 *	"A type matching one of the types in table 8.5 is a special case in
 *	which all the components of each group are packed into a single
 *	unsigned byte, unsigned short, or unsigned int, depending on the
 *	type....
 *	The number of components per packed pixel is fixed by the type, and
 *	must match the number of components per group indicated by the format
 *	parameter, as listed in table 8.5.
 *	An INVALID_OPERATION error is generated by any command
 *	processing pixel rectangles if a mismatch occurs."
 *
 * Table 8.5 Packed pixel formats:
 *
 *"type Parameter Token Name      ...  Matching Pixel Formats"
 * UNSIGNED_BYTE_3_3_2                  RGB, RGB_INTEGER
 * UNSIGNED_BYTE_2_3_3_REV              RGB, RGB_INTEGER
 * UNSIGNED_SHORT_5_6_5                 RGB, RGB_INTEGER
 * UNSIGNED_SHORT_5_6_5_REV             RGB, RGB_INTEGER
 * UNSIGNED_SHORT_4_4_4_4               RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_SHORT_4_4_4_4_REV           RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_SHORT_5_5_5_1               RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_SHORT_1_5_5_5_REV           RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_INT_8_8_8_8                 RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_INT_8_8_8_8_REV             RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_INT_10_10_10_2              RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_INT_2_10_10_10_REV          RGBA, BGRA, RGBA_INTEGER,
 *                                      BGRA_INTEGER
 * UNSIGNED_INT_24_8                    DEPTH_STENCIL
 * UNSIGNED_INT_10F_11F_11F_REV         RGB
 * UNSIGNED_INT_5_9_9_9_REV             RGB
 * FLOAT_32_UNSIGNED_INT_24_8_REV       DEPTH_STENCIL"
 *
 *
 */

#include "piglit-util-gl.h"

PIGLIT_GL_TEST_CONFIG_BEGIN

	config.supports_gl_compat_version = 10;

	config.window_visual = PIGLIT_GL_VISUAL_RGB | PIGLIT_GL_VISUAL_DOUBLE;

PIGLIT_GL_TEST_CONFIG_END

#define TEXSIZE 4


/* For simplicity, we are only testing the following types: */
static const GLenum testedTypes[] = {
	GL_UNSIGNED_BYTE_3_3_2,
	GL_UNSIGNED_BYTE_2_3_3_REV,
	GL_UNSIGNED_SHORT_5_6_5,
	GL_UNSIGNED_SHORT_5_6_5_REV,
	GL_UNSIGNED_INT_10F_11F_11F_REV,
};

/* As Table 8.5 states, our testedTypes[] only work with GL_RGB below. */
static const GLenum formatTypes[] = {
	GL_RGBA,
	GL_RGB,
	GL_RED,
	GL_GREEN,
	GL_BLUE,
	GL_ALPHA,
	GL_LUMINANCE,
	GL_LUMINANCE_ALPHA,
};

void
piglit_init(int argc, char **argv)
{
	piglit_require_extension("GL_EXT_packed_float");
}

enum piglit_result
subtest(GLenum format, GLenum type)
{
	GLfloat pxBuffer[TEXSIZE * TEXSIZE * 4];
	enum piglit_result result;

	glGetTexImage(GL_TEXTURE_2D, 0, format, type, pxBuffer);
	if (format == GL_RGB) {
		if (!piglit_check_gl_error(GL_NO_ERROR))
			result = PIGLIT_FAIL;
		else
			result = PIGLIT_PASS;
	}
	else {
		if (!piglit_check_gl_error(GL_INVALID_OPERATION))
			result = PIGLIT_FAIL;
		else
			result = PIGLIT_PASS;
	}

	piglit_report_subtest_result(result, "%s, %s",
		piglit_get_gl_enum_name(type),
		piglit_get_gl_enum_name(format));

	return result;
}

enum piglit_result
piglit_display(void)
{
	int i, j;
	enum piglit_result result = PIGLIT_PASS;
	enum piglit_result subtest_result;
	GLuint tex;
	GLfloat texData[TEXSIZE][TEXSIZE][3];

	memset(texData, 0, sizeof(texData));

	/* Setup a packed float texture (contents not significant) */
	glGenTextures(1, &tex);
	glBindTexture(GL_TEXTURE_2D, tex);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_R11F_G11F_B10F_EXT,
		     TEXSIZE, TEXSIZE, 0, GL_RGB, GL_FLOAT, texData);

	for (j = 0; j < ARRAY_SIZE(testedTypes); j++) {
		for (i = 0; i < ARRAY_SIZE(formatTypes); i++) {
			subtest_result = subtest(formatTypes[i],
						 testedTypes[j]);
			if (subtest_result != PIGLIT_PASS)
				result = PIGLIT_FAIL;
		}
	}

	return result;
}
