/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Function_p.h"

#include <cstdarg>

#include <llvm/DerivedTypes.h>
#include <llvm/Module.h>
#include <llvm/Type.h>

#include "Parameter.h"

#include "Debug.h"
#include "Macros_p.h"
#include "ModuleData_p.h"
#include "ScopedName.h"
#include "Value.h"
#include "Type.h"
#include "Type_p.h"

using namespace GTLCore;

Function::Data::Data( const std::vector< Parameter >& _parameters, int minimumParameters) : m_parameters(_parameters), m_function(0), m_module(0), m_minimumParameters(minimumParameters), m_maximumParameters(m_parameters.size())
{
  if( minimumParameters == -1) m_minimumParameters = (unsigned int)m_maximumParameters;
}

void Function::Data::setFunction( llvm::Function* _functions )
{
  GTL_ASSERT(not m_function);
  m_function = _functions;
  GTL_ASSERT(m_function);
}
void Function::Data::setModule( ModuleData* _module )
{
  m_module = _module;
}

llvm::Function* Function::Data::function() const
{
  return m_function;
}

GTLCore::String Function::Data::symbolName( const ScopedName& _functionName, const GTLCore::Type* _returnType, const std::vector< Parameter >& _parameters )
{
  GTLCore::String functionSymbolName = _functionName.nameSpace() + "_" + _functionName.name() + "_" + _returnType->d->symbolName();
  
  foreach( const Parameter& param, _parameters)
  {
    functionSymbolName += "_" + param.type()->d->symbolName();
  }
  
  return functionSymbolName;
}

Function* Function::Private::createExternalFunction(ModuleData* _module, llvm::Module* _llvmModule, llvm::LLVMContext& _context, const Function* _functionToCopy )
{
  GTLCore::String _symbolName = GTLCore::Function::Data::symbolName( _functionToCopy->name(), _functionToCopy->returnType(), _functionToCopy->parameters() );
  GTL_DEBUG(_symbolName);
  const llvm::Function* func = _functionToCopy->d->data->function();
  llvm::FunctionType *FTy =
    llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>(func->getType())->getElementType());
  llvm::Function* function = llvm::dyn_cast<llvm::Function>( _llvmModule->getOrInsertFunction( (const std::string&)_symbolName, FTy) );
  GTLCore::Function::Data* data = new GTLCore::Function::Data( _functionToCopy->parameters(), _functionToCopy->d->data->minimumParameters() );
  data->setFunction( function );
  data->setModule( _module );
  return new GTLCore::Function( _functionToCopy->name(), _functionToCopy->returnType(), data);
}

GTLCore::Function* Function::Private::createExternalFunction(ModuleData* _module, llvm::Module* _llvmModule, llvm::LLVMContext& _context, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, ExternalFunctionParameters efpParameter, int _count, ...)
{
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, _count);
  switch(efpParameter)
  {
    case EFP_ONLY_TYPE:
    {
      for(int i = 0; i < _count; ++i)
      {
        const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
        arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
      }
    }
    break;
    case EFP_TYPE_AND_DEFAULT:
    {
      for(int i = 0; i < _count; ++i)
      {
        const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
        GTLCore::Value* value = va_arg(argp, GTLCore::Value*);
        if(value)
        {
          arguments.push_back(GTLCore::Parameter("", type, false, false, *value ) );
        } else {
          arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
        }
      }
    }
    break;
  }
  va_end(argp);
  return createExternalFunction(_module, _llvmModule, _context, _name, _symbolName, retType, arguments );
}

GTLCore::Function* Function::Private::createExternalFunction(ModuleData* _module, llvm::Module* _llvmModule, llvm::LLVMContext& _context, const GTLCore::String& _name, const GTLCore::String& _symbolName, const GTLCore::Type* retType, const std::vector<GTLCore::Parameter>& arguments)
{
  int minimumParameters = -1;
  std::vector<llvm::Type*> llvmArguments;
  for(unsigned int i = 0; i < arguments.size(); ++i)
  {
    llvm::Type* type = Type::Private::d( arguments[i].type() )->asArgumentType(_context);
    if(arguments[i].isOutput())
    {
      type = llvm::PointerType::get( type, 0 );
    }
    if(arguments[i].defaultValue().type() != Type::Undefined and minimumParameters == -1)
    {
      minimumParameters = i;
    }
    llvmArguments.push_back( type );
  }
  llvm::Function* function = llvm::dyn_cast<llvm::Function>( _llvmModule->getOrInsertFunction( (const std::string&)_symbolName, llvm::FunctionType::get( retType->d->asArgumentType(_context), llvmArguments, false ) ) );
  GTLCore::Function::Data* data = new GTLCore::Function::Data(arguments, minimumParameters == -1 ? arguments.size() : minimumParameters );
  data->setFunction( function );
  data->setModule( _module );
  return new GTLCore::Function( GTLCore::ScopedName("", _name), retType, data );
}

GTLCore::Function* Function::Private::createInternalFunction(ModuleData* _module, llvm::LLVMContext& _context, const GTLCore::String& _name, llvm::Function* _function, const GTLCore::Type* retType, ExternalFunctionParameters efpParameter, int _count, ...)
{
  GTL_ASSERT( _function );
  std::vector<GTLCore::Parameter> arguments;
  va_list argp;
  va_start(argp, _count);
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  const llvm::FunctionType *FTy =
      llvm::cast<llvm::FunctionType>(llvm::cast<llvm::PointerType>(_function->getType())->getElementType());
#endif
  switch(efpParameter)
  {
    case EFP_ONLY_TYPE:
    {
      for(int i = 0; i < _count; ++i)
      {
        const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
        arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
        GTL_DEBUG( *Type::Private::d( type )->asArgumentType(_context) << " == " << *FTy->getParamType(i) );
        GTL_ASSERT( Type::Private::d( type )->asArgumentType(_context) == FTy->getParamType(i) );
#endif
      }
    }
    break;
    case EFP_TYPE_AND_DEFAULT:
    {
      for(int i = 0; i < _count; ++i)
      {
        const GTLCore::Type* type = va_arg(argp, const GTLCore::Type*);
        GTLCore::Value* value = va_arg(argp, GTLCore::Value*);
        if(value)
        {
          arguments.push_back(GTLCore::Parameter("", type, false, false, *value ) );
        } else {
          arguments.push_back(GTLCore::Parameter("", type, false, false, GTLCore::Value() ) );
        }
        delete value;
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
        GTL_DEBUG( *Type::Private::d( type )->asArgumentType(_context) << " == " << *FTy->getParamType(i) );
        GTL_ASSERT( Type::Private::d( type )->asArgumentType(_context) == FTy->getParamType(i) );
#endif
      }
    }
    break;
  }
  va_end(argp);
  GTLCore::Function::Data* data = new GTLCore::Function::Data(arguments, arguments.size() );
  data->setFunction( _function );
  data->setModule( _module );
  return new GTLCore::Function( GTLCore::ScopedName("", _name), retType, data );
}

bool Function::Private::isReturnedAsPointer() const
{
#ifdef OPENGTL_32_BITS
  return returnType->dataType() == Type::VECTOR and returnType->vectorSize() > 2;
#else
  return returnType->dataType() == Type::VECTOR and returnType->vectorSize() % 2 == 1;
#endif
}
