-----------------------------------------------------------------------
--          GtkAda - Ada95 binding for the Gimp Toolkit              --
--                                                                   --
--                        Copyright (C) 2000                         --
--        Emmanuel Briot, Joel Brobecker and Arnaud Charlet          --
--                                                                   --
-- This library is free software; you can redistribute it and/or     --
-- modify it under the terms of the GNU General Public               --
-- License as published by the Free Software Foundation; either      --
-- version 2 of the License, or (at your option) any later version.  --
--                                                                   --
-- This library is distributed in the hope that it will be useful,   --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of    --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU --
-- General Public License for more details.                          --
--                                                                   --
-- You should have received a copy of the GNU General Public         --
-- License along with this library; if not, write to the             --
-- Free Software Foundation, Inc., 59 Temple Place - Suite 330,      --
-- Boston, MA 02111-1307, USA.                                       --
--                                                                   --
-- As a special exception, if other files instantiate generics from  --
-- this unit, or you link this unit with other files to produce an   --
-- executable, this  unit  does not  by itself cause  the resulting  --
-- executable to be covered by the GNU General Public License. This  --
-- exception does not however invalidate any other reasons why the   --
-- executable file  might be covered by the  GNU Public License.     --
-----------------------------------------------------------------------

with Interfaces.C.Strings;

package body Intl is

   use Interfaces.C;
   use Interfaces.C.Strings;

   -------------
   -- Gettext --
   -------------

   function Gettext (Msg : String) return String is
      function Internal (msgid : in char_array)
                        return chars_ptr;
      pragma Import (C, Internal, "gettext");
      R : constant chars_ptr := Internal (To_C (Msg));
   begin
      if R = Null_Ptr then
         raise Locale_Error;
      end if;
      return Value (R);
   end Gettext;

   --------------
   -- Dgettext --
   --------------

   function Dgettext (Domain : String; Msg : String) return String is
      function Internal (domainname : in char_array;
                         msgid      : in char_array)
                        return chars_ptr;
      pragma Import (C, Internal, "dgettext");
      R : constant chars_ptr := Internal (To_C (Domain), To_C (Msg));
   begin
      if R = Null_Ptr then
         raise Locale_Error;
      end if;
      return Value (R);
   end Dgettext;

   ---------------
   -- Dcgettext --
   ---------------

   function Dcgettext
     (Domain : String; Msg : String; Category : Integer) return String
   is
      function Internal (domainname : in char_array;
                         msgid      : in char_array;
                         category   : in int)
                        return chars_ptr;
      pragma Import (C, Internal, "dcgettext");
      R : constant chars_ptr := Internal (To_C (Domain), To_C (Msg),
                                          int (Category));
   begin
      if R = Null_Ptr then
         raise Locale_Error;
      end if;
      return Value (R);
   end Dcgettext;

   -------------------------
   -- Default_Text_Domain --
   -------------------------

   function Default_Text_Domain return String is
      function Internal (domainname : in chars_ptr)
                        return chars_ptr;
      pragma Import (C, Internal, "textdomain");
      R : constant chars_ptr := Internal (Null_Ptr);
   begin
      if R = Null_Ptr then
         raise Locale_Error;
      end if;
      return Value (R);
   end Default_Text_Domain;

   -----------------
   -- Text_Domain --
   -----------------

   procedure Text_Domain (Domain : String := "") is
      function Internal (domainname : in char_array)
                        return chars_ptr;
      pragma Import (C, Internal, "textdomain");
   begin
      if Internal (To_C (Domain)) = Null_Ptr then
         raise Locale_Error;
      end if;
   end Text_Domain;

   ----------------------
   -- Bind_Text_Domain --
   ----------------------

   procedure Bind_Text_Domain (Domain : String; Dirname : String) is
      function Internal (domainname : in char_array;
                         dirname    : in char_array)
                        return chars_ptr;
      pragma Import (C, Internal, "bindtextdomain");
   begin
      if Internal (To_C (Domain), To_C (Dirname)) = Null_Ptr then
         raise Locale_Error;
      end if;
   end Bind_Text_Domain;

   procedure Set_Locale is
      function Internal (category : in int;
                         locale   : in char_array)
                        return chars_ptr;
      pragma Import (C, Internal, "setlocale");
      LC_ALL : constant := 6;
   begin
      if Internal (LC_ALL, To_C ("")) = Null_Ptr then
         raise Locale_Error;
      end if;
   end Set_Locale;

end Intl;
