package latexDraw.parsers.pstricks;

import java.awt.Color;
import java.awt.geom.Point2D;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JProgressBar;
import javax.swing.SwingUtilities;

import latexDraw.figures.FramedBox;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.parsers.*;
import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.dialog.ExceptionFrameDialog;



/** 
 * This class defines a PSTricks parser.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 
 * 09/06/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.5<br>
 */
public class PSTricksParser extends Parser
{
	/** Manages the actions to do when a command is parsed. */
	PSTricksParserActionsManager texParserActions;
	
	/** The progress bar */
	private JProgressBar progressB;
	
	/** Allows to stop the thread (if there is a one) */
	private boolean stopThread;
	
	private static final Hashtable<String, String> latexCommandsWithBrackets;
	
	private static final Hashtable<String, String> latexCommandsWithoutBrack;
	
	static
	{
		latexCommandsWithBrackets = new Hashtable<String, String>();
		
		latexCommandsWithBrackets.put("textit", "textit");//$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("textbf", "textbf");//$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("address", "address");//$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("acute", "acute");//$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("\"", "\"");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("’", "’");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put(".", ".");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("=", "="); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("b", "b");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("‘", "‘");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("newtie", "newtie");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("^", "^"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("c", "c");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("d", "d");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("G", "G");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("H", "H"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("U", "U");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("t", "t");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("r", "r");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("k", "k"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("u", "u");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("v", "v");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("~", "~");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("vdots", "vdots"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("author", "author");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("arabic", "arabic");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("alph", "alph"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("overbrace", "overbrace"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("widetilde", "widetilde");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("overline", "overline");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("overleftarrow", "overleftarrow");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("underbrace", "underbrace"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("widehat", "widehat");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("underline", "underline");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("overriglatexCommandsWithBracketsarrow", "overriglatexCommandsWithBracketsarrow");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("bigcap", "bigcap"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("bigcup", "bigcup"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("bigodot", "bigodot");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("bigoplus", "vdash"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("bigotimes", "bigotimes");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("coprod", "coprod"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("bigwedge", "bigwedge"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("bigsqcup", "bigsqcup");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("biguplus", "biguplus");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("bigvee", "bigvee");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("int", "int"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithBrackets.put("prod", "prod");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithBrackets.put("sum", "sum"); //$NON-NLS-1$ //$NON-NLS-2$
		
		latexCommandsWithoutBrack = new Hashtable<String, String>();
		
		latexCommandsWithoutBrack.put("ldots", "ldots");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ldotp", "ldotp");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("vdash", "vdash");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("vdots", "vdots"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("aa", "aa");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("accent", "accent");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ae", "ae");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("aleph", "aleph");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("alpha", "alpha");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("angle", "angle"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("approx", "approx");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("arccos", "arccos");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("arcsin", "arcsin"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("arctan", "arctan");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("arg", "arg"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("arrowvert", "arrowvert");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("asymp", "asymp");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("atop", "atop");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("atopwithdelims", "atopwithdelims"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("AA", "AA");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("AE", "AE"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("DH", "DH");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("l", "l");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("L", "L");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("dj", "dj"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("DJ", "DJ");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("dh", "dh"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("NG", "NG");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ng", "ng");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("o", "o");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("O", "O"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("OE", "OE"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("gamma", "gamma");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("beta", "beta"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("delta", "delta");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("epsilon", "epsilon");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Gamma", "Gamma");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Delta", "Delta"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Lambda", "Lambda");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Theta", "Theta"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("vartheta", "vartheta");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("theta", "theta");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("eta", "eta");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("zeta", "zeta"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("varepsilon", "varepsilon"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("iota", "iota");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("kappa", "kappa"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("lambda", "lambda");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("mu", "mu");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("nu", "nu");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Xi", "Xi"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Pi", "Pi");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Upsilon", "Upsilon"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Sigma", "Sigma");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("rho", "rho");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("varpi", "varpi");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("pio", "pio"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("xi", "xi"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("varrho", "varrho");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("sigma", "sigma");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("varsigma", "varsigma"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("tau", "tau");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("upsilon", "upsilon"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Phi", "Phi");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Psi", "Psi");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Omega", "Omega");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("omega", "omega"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("psi", "psi"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("chi", "chi");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("varphi", "varphi"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("phi", "phi"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("bowtie", "bowtie");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("cong", "cong");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("dashv", "dashv");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("doteq", "doteq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("equiv", "equiv");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("frown", "frown"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("models", "models");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("mid", "mid");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ll", "ll");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("leq", "leq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Join", "Join"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("in", "in");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("gg", "gg");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("geq", "geq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("neq", "neq");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ni", "ni"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("parallel", "parallel");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("perp", "perp");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("prec", "prec");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("preceq", "preceq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("propto", "propto"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("sim", "sim");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("subseteq", "subseteq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("subset", "subset"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("smile", "smile");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("simeq", "simeq");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("succ", "succ");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("succeq", "succeq"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("supset", "supset"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("supseteq", "supseteq");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("vdash", "vdash"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("rmoustache", "rmoustache");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("lmoustache", "lmoustache"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("rgroup", "rgroup"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("lgroup", "lgroup");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("arrowvert", "arrowvert");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("bracevert", "bracevert");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Arrowvert", "Arrowvert"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("bot", "bot"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("top", "top"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("forall", "forall");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("exists", "exists"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("Im", "Im"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("nabla", "nabla");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Re", "simeq");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("infty", "infty");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ell", "ell"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("neg", "neg"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("prime", "prime");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("wp", "wp"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("hbar", "hbar");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("backslash", "backslash"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("cos", "cos"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("cosh", "cosh");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("cot", "cot");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("inf", "inf");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("hom", "hom"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("gcd", "gcd");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("exp", "exp"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("dim", "dim");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("det", "det"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("deg", "deg"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("csc", "csc");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("coth", "coth");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("ker", "ker");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("lg", "lg"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("lim", "lim"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("liminf", "liminf");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("limsup", "limsup"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("ln", "ln"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("log", "csc");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("max", "max");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("min", "min");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("Pr", "Pr"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("sec", "sec"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("sin", "sin"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("sinh", "sinh");  //$NON-NLS-1$//$NON-NLS-2$
		latexCommandsWithoutBrack.put("sup", "sup"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("tan", "tan"); //$NON-NLS-1$ //$NON-NLS-2$
		latexCommandsWithoutBrack.put("tanh", "tanh");  //$NON-NLS-1$//$NON-NLS-2$
	}
	
	
	
	/**
	 * The constructor.
	 * @param texParserAc The actions manager.
	 */
	public PSTricksParser(PSTricksParserActionsManager texParserAc)
	{
		texParserActions = texParserAc;
		stopThread = true;
	}
	
	

	public synchronized void setStopThread(boolean state)
	{
		stopThread = state;
	}
	
	
	
	/**
	 * Allows to parse a TeX file.
	 * @param f The TeX file to parse.
	 * @param parent The main frame: Useful to display message dialog (can be equal to null: no message).
	 */
	public int parse(File f, JFrame parent, JProgressBar pg)
	{
		if(f==null)
			throw new IllegalArgumentException();
		
		try
		{
			setStopThread(false);
			
			if(!f.exists() || !f.isFile() || !f.canRead() || !f.exists())
				throw new IOException();
			
			FileReader fr 	  = new FileReader(f);
			BufferedReader br = new BufferedReader(fr);
			String eol = System.getProperty("line.separator");//$NON-NLS-1$
			String s = "";//$NON-NLS-1$
			String line;

			while( (line=br.readLine())!=null)
				s+=line+eol;

			br.close();
			fr.close();

			return parse(s, parent, pg);
			
		}catch(IOException e)
		{
			String msg = LaTeXDrawLang.getString1_6("PSTricksParser.0"); //$NON-NLS-1$
			
			if(parent==null)
				System.out.println(msg);
			else
				JOptionPane.showMessageDialog(parent, msg, LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.86"),  //$NON-NLS-1$
		 		    JOptionPane.ERROR_MESSAGE); 
			
			return ERROR_PARSING;
		}
	}
	
	
	/**
	 * Allows to update the progress bar.
	 * @param newValue its new value.
	 */
	public synchronized void updateBar(int newValue) 
	{
		if(progressB!=null)
			progressB.setValue(newValue);
	}
	
	
	/**
	 * Allows to parse a TeX file.
	 * @param file The TeX code to parse.
	 * @param parent The main frame: useful to display message dialog (can be equal to null: no message).
	 * @param progressBar The progress bar of the parser. If null, it will not be used.
	 */
	public int parse(String file, JFrame parent, JProgressBar progressBar)
	{
		try
		{
			PSTricksParameters.reinitialize();
			stopThread = false;
			progressB = progressBar;
			
			/* *********************************************************************** */
			/* We remove big spaces and comments */
			
			char eol = System.getProperty("line.separator").charAt(0);//$NON-NLS-1$
			char[] f2 = new char[file.length()];
			
			int i=0, lgth = file.length(), cpt=0;
			boolean zero = false;
			while(i<lgth)
			{
				if(i<lgth && (file.charAt(i)==' ' || file.charAt(i)==9)) //9 = '\t'
				{
					if(!zero) 
					{
						f2[cpt++] = ' ';
						zero=true;
					}
				}
				else
				{
					if(file.charAt(i)=='%') 
					{
						if(i>0)
							if(file.charAt(i-1)=='\\')
							{
								f2[cpt++] = file.charAt(i);
								zero = false;
							}
							else
							{
								while(i<lgth && eol!=file.charAt(i))
									i++;
								
								if(i<lgth) f2[cpt++] = file.charAt(i);// We copy the eol
							}
					}
					else 
					{
						f2[cpt++] = file.charAt(i);
						zero = false;
					}
				}
				i++;
			} // while(i<lgth)
			
			
			String f = String.valueOf(f2, 0, cpt) + ' ';//We terminate the code by a space
			
			if(progressBar!=null)
			{
				progressB.setMinimum(0);
				progressB.setMaximum(f.length());
			}
			
			/* ********************************************************************/
			Vector<PSTricksParameters> params = new Vector<PSTricksParameters>();
			boolean started = false;
			i = 0;
			int i2;
			lgth = f.length();
			int nLine = 1;
			Vector<ParserCmdMessage> parserMsgs = new Vector<ParserCmdMessage>();
			boolean parsingFinished = false;
			boolean lastTokenIsEOL = false, isFramedBox = false;
			int iOld = i;
			boolean startedWithBeginPs = true;
			
			while(i<lgth && !parsingFinished)
			{
				if(progressBar!=null) 
				{
					final int cpti = progressB.getValue()+i-iOld;
					SwingUtilities.invokeAndWait(
					new Runnable() { public void run() {updateBar(cpti);} });
				}
				
				try {testEndThread(); }
				catch(Exception e) { return CANCEL_PARSING;}
				
				iOld = i;
				try
				{
				try
				{
					if(f.charAt(i)!=eol) lastTokenIsEOL = false;
					
					switch(f.charAt(i))
					{
						case '$':
							i++;
							
							if(started)
							{
								PSTricksParameters p = params.lastElement();
								p.textParsed += '$';
								
								while(i<lgth && f.charAt(i)!='$')
								{
									p.textParsed += f.charAt(i);
									i++;
								}
								
								p.textParsed += '$';
							}
							else
								while(i<lgth && f.charAt(i)!='$')
									i++;

							if(i>=lgth) throw new UnexceptedEndOfFileException();
							
							i++;
							break;
							
						case '\\' :
							// ********************** //
							/* Parsing of the command */
							
							i++;
							if(i>=lgth)  throw new UnexceptedEndOfFileException();
							
							switch(f.charAt(i))
							{
								case 'p' : 
									i2 = i;
									if(i+1>=lgth) 
									{
										if(started)  throw new UnexceptedEndOfFileException();
										throw new IgnoreCommandException(nLine);
									}
									i++;
									
									switch(f.charAt(i))
									{
										case 's' ://ps
											
											if(i+1>=lgth) throw new UnexceptedEndOfFileException();
											i++;
											switch(f.charAt(i))
											{
												case 'c' : // psc
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													i++;
													
													switch(f.charAt(i))
													{
														case 'o' :// pscoil
															if(i+3>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='i' && f.charAt(++i)=='l' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	throw new NotYetManagedException("pscoil", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
															
														case 'i' : // psci 
															if(i+4>=lgth) throw new UnexceptedEndOfFileException();
															
															if(f.charAt(++i)=='r' && f.charAt(++i)=='c' && f.charAt(++i)=='l' &&
																f.charAt(++i)=='e') // pscircle
															{
																if(++i>=lgth) throw new UnexceptedEndOfFileException();
																
																if((f.charAt(i)<'a' || f.charAt(i)>'z') && 
																(f.charAt(i)<'A' || f.charAt(i)>'Z'))//pscircle 
																{
																	int end = texParserActions.actionsOnCircle(nLine, f.charAt(i)=='*', 
																			params, f.substring(i, f.length()));
																	if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																}
																else
																{
																	if(i+3>=lgth) throw new UnexceptedEndOfFileException();
																	if(f.charAt(i)=='b' && f.charAt(++i)=='o' && 
																		f.charAt(++i)=='x' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
																		(f.charAt(i)<'A' || f.charAt(i)>'Z')) // pscriclebox
																	{
																		PSTricksParameters tp = params.lastElement();
																		
																		if(tp.psBox.isEmpty())
																			tp.psBoxText.setText("");//$NON-NLS-1$
																		int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																				params, f.substring(i, f.length()), 
																				f.substring(i, f.length()), 1);
																		if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																		isFramedBox = true;
																	}
																	else
																	{
																		i=i2;
																		throw new UnknownCommandException(nLine);
																	}
																}
															}
															else
															{
																i=i2;
																throw new UnexceptedEndOfFileException();
															}
															break;
															
														case 'c' : // psccurve
															if(i+10>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='c' && f.charAt(++i)=='u' && f.charAt(++i)=='r' && 
																f.charAt(++i)=='v' && f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	throw new NotYetManagedException("psccurve", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
															
														case 'u' : // pscu
															if(i+1>=lgth) throw new UnexceptedEndOfFileException();
															
															switch(f.charAt(++i))
															{
																case 's'://pscus
																	if(i+4>=lgth) throw new UnexceptedEndOfFileException();
																	if(f.charAt(++i)=='t' && f.charAt(++i)=='o' && f.charAt(++i)=='m' && 
																		(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
																		f.charAt(i)>'Z'))//pscustom
																	{
																		int end = texParserActions.parsePsCustomCommand(
																				nLine, f.charAt(i)=='*', params, 
																				f.substring((f.charAt(i)=='*'?i+1:i), f.length()));
																		if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																		
																	}
																	else
																	{
																		i=i2;
																		throw new UnknownCommandException(nLine);
																	}
																	break;
																	
																case 'r'://pscur
																	if(i+3>=lgth) throw new UnexceptedEndOfFileException();
																	if(f.charAt(++i)=='v' &&f.charAt(++i)=='e' &&
																		(f.charAt(++i)<'a' || f.charAt(i)>'z') &&
																		(f.charAt(i)<'A' || f.charAt(i)>'Z'))// pscurve
																		throw new NotYetManagedException("pscurve", nLine);//$NON-NLS-1$
																	
																	i=i2;
																	throw new UnknownCommandException(nLine);
																	
																default:
																	i=i2;
																	throw new UnknownCommandException(nLine);
															}
															break;
													}
													break;
													
												case 'a' : // psa
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth)throw new UnexceptedEndOfFileException();
													i++;
													switch(f.charAt(i))
													{
														case 'x' :// psaxes
															if(i+3>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='e' && f.charAt(++i)=='s' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																if(f.charAt(i)=='*')
																	i++;
																int end = texParserActions.actionsOnGridAxes(nLine, params, 
																		PSTricksParameters.pictureSWPt, PSTricksParameters.pictureNEPt, 
																		f.substring(i, f.length()), false);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														case 'd' : // psaddtolength
															if(i+10>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='d' && f.charAt(++i)=='t' && f.charAt(++i)=='o' && 
																f.charAt(++i)=='l' && f.charAt(++i)=='e' && f.charAt(++i)=='n' &&
																f.charAt(++i)=='g' && f.charAt(++i)=='t' && f.charAt(++i)=='t' &&
																(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
																f.charAt(i)>'Z'))
																	throw new NotYetManagedException("psaddtolength", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
															
														case 'r' : // psar
															if(i+1>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='c')
															{
																if(i+1>=lgth) throw new UnexceptedEndOfFileException();
																if((f.charAt(++i)<'a' || f.charAt(i)>'z') &&
																		(f.charAt(i)<'A' || f.charAt(i)>'Z'))//psarc
																{
																	int end = texParserActions.actionsOnArcWedge(nLine, 
																			f.charAt(i)=='*', params, f.substring(i, f.length()),true);
																	if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																}
																else
																{
																	if(f.charAt(i)=='n' && (f.charAt(++i)<'a' || f.charAt(i)>'z') &&
																		(f.charAt(i)<'A' || f.charAt(i)>'Z'))// psarcn
																		throw new NotYetManagedException("psarcn", nLine);//$NON-NLS-1$
																}
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
														
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'p' : // psp
													if(i+1>=lgth)
													{
														if(started)	throw new UnexceptedEndOfFileException();
														throw new IgnoreCommandException(nLine);
													}
													i++;
													
													switch(f.charAt(i))
													{
														case 'o' :// pspolygon
															if(!started)  throw new IgnoreCommandException(nLine);
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='l' && f.charAt(++i)=='y' && f.charAt(++i)=='g' && 
																f.charAt(++i)=='o' && f.charAt(++i)=='n' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																int end = texParserActions.actionsOnPolygon(nLine, f.charAt(i)=='*', 
																		params, f.substring(i, f.length()));
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
																
														case 'i' : // pspicture
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='c' && f.charAt(++i)=='t' && f.charAt(++i)=='u' && 
																f.charAt(++i)=='r' && f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																if(started)
																	throw new FatalErrorParsingException(
																	LaTeXDrawLang.getString1_6("PSTricksParser.1")+ //$NON-NLS-1$
																	LaTeXDrawLang.getString1_6("PSTricksParser.2")); //$NON-NLS-1$
																started = true;
																startedWithBeginPs = false;
																Point2D.Double pt1, pt2;
																String sub = f.substring(i);
																try 
																{
																	pt1 = PSTricksParserActionsManager.readOneCoordinate(sub, 0, nLine, true);
																	i += sub.indexOf(")", 0);//$NON-NLS-1$
																}catch(Exception e)
																{ 
																	throw new FatalErrorParsingException("Invalid command \\begin{pspicture}");//$NON-NLS-1$
																}
																
																int j, k = ++i;
																try 
																{ // optional
																	while(k<lgth && (PSTricksParserActionsManager.isCharToIgnore(f.charAt(k)) ||
																			f.charAt(k)=='\\'))
																		k++;
																
																	if(f.charAt(k)!='(') throw new IgnoreCommandException(-1);
																	j = f.indexOf(')', k);
																	sub = f.substring(k, j+1);
																	pt2 = PSTricksParserActionsManager.readOneCoordinate(sub, 0, nLine, true);
																	i = j;
																}catch(IgnoreCommandException e)
																{
																	pt2 = pt1;
																	pt1 = new Point2D.Double(0,0);
																}
																catch(Exception e)
																{
																	throw new FatalErrorParsingException("Invalid command \\begin{pspicture}");//$NON-NLS-1$
																}
																
																PSTricksParameters.pictureNEPt = pt2;
																PSTricksParameters.pictureSWPt = pt1;
																params.add(new PSTricksParameters());
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														case 'a' : // psparabola
															if(!started)  throw new IgnoreCommandException(nLine);
															if(i+7>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='r' && f.charAt(++i)=='a' && f.charAt(++i)=='b' && 
																f.charAt(++i)=='o' && f.charAt(++i)=='l' && f.charAt(++i)=='a' &&
																(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
																f.charAt(i)>'Z'))
																	throw new NotYetManagedException("psparabola", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
														
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 's' : // pss
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth)throw new UnexceptedEndOfFileException();
													i++;
													switch(f.charAt(i))
													{
														case 'e' :// psset
															if(i+1>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='t')
															{
																if(i+1>=lgth) throw new UnexceptedEndOfFileException();
																if((f.charAt(++i)<'a' || f.charAt(i)>'z') && 
																	(f.charAt(i)<'A' || f.charAt(i)>'Z'))//psset
																{
																	int j = i;
																	int cptBrack=0;
																	
																	while(j<lgth && f.charAt(j)!='{')
																		j++;
																	j++;
																	
																	while(j<lgth && (f.charAt(j)!='}' || cptBrack>0))
																	{
																		if(f.charAt(j)=='{') 
																			cptBrack++;
																		else if(f.charAt(j)=='}' && cptBrack>0)
																			cptBrack--;
																		j++; // We search the '}'
																	}
																	if(j>=lgth) throw new UnexceptedEndOfFileException();
																	j++;
																	String str = f.substring(i, j);
																	PSTricksParserActionsManager.parsepssetCommand(str, params.lastElement());
																	i = j;
																}
																else
																{
																	if(i+3>=lgth) throw new UnexceptedEndOfFileException();
																	if(f.charAt(i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' && 
																		(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
																		f.charAt(i)>'Z'))//pssetbox
																			throw new NotYetManagedException("pssetbox", nLine);//$NON-NLS-1$
																}
															} // if(f.charAt(++i)=='t')
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
											
														case 'h' : // psshadowbox
															if(i+8>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='a' && f.charAt(++i)=='d' && f.charAt(++i)=='o' && 
																f.charAt(++i)=='w' && f.charAt(++i)=='b' && f.charAt(++i)=='o' && 
																f.charAt(++i)=='x' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																PSTricksParameters tp = params.lastElement();
																
																if(tp.psBox.isEmpty())
																	tp.psBoxText.setText("");//$NON-NLS-1$
																int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																		params, f.substring(i, f.length()), 
																		f.substring(i, f.length()), 6);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																isFramedBox = true;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'f' : // psf
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+4>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='r' && f.charAt(++i)=='a' && f.charAt(++i)=='m' &&
														f.charAt(++i)=='e')//psframe
													{
														if(i+1>=lgth) throw new UnexceptedEndOfFileException();
														i++;
														if((f.charAt(i)<'a' || f.charAt(i)>'z') && 
															(f.charAt(i)<'A' || f.charAt(i)>'Z'))//psframe
														{
															int end = texParserActions.actionsOnFrameEllipse(nLine, 
																	f.charAt(i)=='*', params, f.substring(i, f.length()), true);
															if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
														}
														else
														{
															if(i+3>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' && 
																(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' ||
																f.charAt(i)>'Z'))//psframebox
															{
																PSTricksParameters tp = params.lastElement();
																
																if(tp.psBox.isEmpty())
																	tp.psBoxText.setText("");//$NON-NLS-1$
																int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																		params, f.substring(i, f.length()), 
																		f.substring(i, f.length()), 0);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) 
																{
																	i+=end;
																	isFramedBox = true;
																}
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
														}
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'l' : // psl
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													i++;
													switch(f.charAt(i))
													{
														case 'i' :// psline
															if(i+3>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='n' && f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																int end = texParserActions.actionsOnLine(nLine, f.charAt(i)=='*', 
																		params, f.substring(i, f.length()));
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														case 'o' : // pslongbox
															if(i+10>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='n' && f.charAt(++i)=='g' && f.charAt(++i)=='b' && 
																f.charAt(++i)=='o' && f.charAt(++i)=='x' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	throw new NotYetManagedException("pslongbox", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
														
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'd' : //psd
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													i++;
													switch(f.charAt(i))
													{
														case 'o' :// psdots
															if(++i>=lgth) throw new UnexceptedEndOfFileException();
															
															if(f.charAt(i)=='t') // psdot
															{
																if(++i>=lgth) throw new UnexceptedEndOfFileException();
																if((f.charAt(i)<'a' || f.charAt(i)>'z') && 
																   (f.charAt(i)<'A' || f.charAt(i)>'Z')) //psdot command
															   {
																	int end = texParserActions.actionsOnDot(nLine, params, f.substring(i, f.length()),true);
																	if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
															   }else
															   {
																   if(i+1>=lgth) throw new UnexceptedEndOfFileException();
																   if(f.charAt(i)=='s' && (f.charAt(++i)<'a' || 
																	  f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	{ // psdots
																		int end = texParserActions.actionsOnDot(nLine, params, f.substring(i, f.length()),false);
																		if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																	}else
																	{
																		i=i2;
																		throw new UnknownCommandException(nLine);
																	}
															   }
															}else 
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														case 'b' : // psdblframebox
															if(i+10>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='l' && f.charAt(++i)=='f' && f.charAt(++i)=='r' &&
																f.charAt(++i)=='a' && f.charAt(++i)=='m' && f.charAt(++i)=='e' &&
																f.charAt(++i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' &&
																(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																PSTricksParameters tp = params.lastElement();
																
																if(tp.psBox.isEmpty())
																	tp.psBoxText.setText("");//$NON-NLS-1$
																int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																		params, f.substring(i, f.length()), 
																		f.substring(i, f.length()), 5);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																isFramedBox = true;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
														break;
															
														case 'i' : // psdi
															if(i+1>=lgth) throw new UnexceptedEndOfFileException();
															
															if(f.charAt(++i)=='a')//psdia
															{
																if(i+5>=lgth) throw new UnexceptedEndOfFileException();
																if(f.charAt(++i)=='m' && f.charAt(++i)=='o' &&
																   f.charAt(++i)=='n' && f.charAt(++i)=='d' && (f.charAt(++i)<'a' || 
																   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																{// psdiamond
																	int end = texParserActions.actionsOnDiamondTriangle(nLine, 
																			f.charAt(i)=='*', params, f.substring(i, f.length()), true);
																	if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																}
																else
																{
																	if(i+3>=lgth) throw new UnexceptedEndOfFileException();
																	if(f.charAt(i)=='b' && f.charAt(++i)=='o' &&
																	   f.charAt(++i)=='x' && (f.charAt(++i)<'a' || 
																	   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	{//psdiabox
																		PSTricksParameters tp = params.lastElement();
																		
																		if(tp.psBox.isEmpty())
																			tp.psBoxText.setText("");//$NON-NLS-1$
																		int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																				params, f.substring(i, f.length()), 
																				f.substring(i, f.length()), 3);
																		if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																		isFramedBox = true;
																	}
																	else
																	{
																		i=i2;
																		throw new UnknownCommandException(nLine);
																	}
																}
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'e' : // pse
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													i++;
													switch(f.charAt(i))
													{
														case 'l' :// psellipse
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='l' && f.charAt(++i)=='i' && f.charAt(++i)=='p' &&
																f.charAt(++i)=='s' && f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{
																int end = texParserActions.actionsOnFrameEllipse(nLine, f.charAt(i)=='*', 
																		params, f.substring(i, f.length()), false);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
															
														case 'c' : // psecurve
															if(i+5>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='u' && f.charAt(++i)=='r' && f.charAt(++i)=='v' &&
																f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																	throw new NotYetManagedException("psecurve", nLine);//$NON-NLS-1$
															i=i2;
															throw new UnknownCommandException(nLine);
														
														default :
															i=i2;
															throw new UnknownCommandException(nLine);
													}
													break;
													
												case 't' : // pst
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+2>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='r' && f.charAt(++i)=='i')
													{//pstri
														if(i+6>=lgth) throw new UnexceptedEndOfFileException();
														if(f.charAt(++i)=='a' && f.charAt(++i)=='n' && f.charAt(++i)=='g' && 
															f.charAt(++i)=='l' && f.charAt(++i)=='e' &&
															(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														{//pstriangle
															int end = texParserActions.actionsOnDiamondTriangle(nLine, f.charAt(i)=='*', 
																						params, f.substring(i, f.length()), false);
															if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
														}
														else
														{
															if(i+3>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' &&
																(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//pstribox
																PSTricksParameters tp = params.lastElement();
																
																if(tp.psBox.isEmpty())
																	tp.psBoxText.setText("");//$NON-NLS-1$
																int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																		params, f.substring(i, f.length()), 
																		f.substring(i, f.length()), 2);
																if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
																isFramedBox = true;
															}
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
														}
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'C' ://psCoil
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+4>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='o' && f.charAt(++i)=='i' && f.charAt(++i)=='l' &&
														(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															throw new NotYetManagedException("psCoil", nLine);//$NON-NLS-1$
													i=i2;
													throw new UnknownCommandException(nLine);
													
												case 'b' : // psbezier
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+6>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='e' && f.charAt(++i)=='z' && f.charAt(++i)=='i' &&
														f.charAt(++i)=='e' && f.charAt(++i)=='r' && (f.charAt(++i)<'a' || 
														f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
													{
														int end = texParserActions.actionsOnBezierCurve(nLine, f.charAt(i)=='*', 
																params, f.substring(i, f.length()));
														if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'w' : // pswedge
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+5>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='e' && f.charAt(++i)=='d' && f.charAt(++i)=='g' &&
														f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
														f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
													{
														int end = texParserActions.actionsOnArcWedge(nLine, f.charAt(i)=='*', 
																params, f.substring(i, f.length()),false);
														if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'g' : // psgrid
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+4>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='r' && f.charAt(++i)=='i' && f.charAt(++i)=='d' &&
														(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
													{
														int end = texParserActions.actionsOnGridAxes(nLine, params, 
																PSTricksParameters.pictureSWPt, PSTricksParameters.pictureNEPt, 
																f.substring(i, f.length()), true);
														if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'o' : // psovalbox
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+7>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='v' && f.charAt(++i)=='a' && f.charAt(++i)=='l' &&
														f.charAt(++i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' &&
														(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' ||
														f.charAt(i)>'Z'))
													{
														PSTricksParameters tp = params.lastElement();
														
														if(tp.psBox.isEmpty())
															tp.psBoxText.setText("");//$NON-NLS-1$
														int end = texParserActions.actionsOnFramedBox(nLine, f.charAt(i)=='*',
																params, f.substring(i, f.length()), 
																f.substring(i, f.length()), 4);
														if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
														isFramedBox = true;
													}
													else
													{
														i=i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'z' : // pszigzag
													if(!started)  throw new IgnoreCommandException(nLine);
													if(i+6>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='i' && f.charAt(++i)=='g' && f.charAt(++i)=='z' &&
														f.charAt(++i)=='a' && f.charAt(++i)=='g' && (f.charAt(++i)<'a' || 
														f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															throw new NotYetManagedException("pszigzag", nLine);//$NON-NLS-1$
													i=i2;
													throw new UnknownCommandException(nLine);
													
												default :
													i=i2;
													throw new UnknownCommandException(nLine);
											} // switch(f.charAt(i))
											break;
											
										case 'u':// pu
											if(i+1>=lgth) throw new UnexceptedEndOfFileException();
											if(f.charAt(++i)=='t' && //put
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int j = i;
												
												while(j<lgth && f.charAt(j)!='{') // We extract the command the string : put(...){...}
													j++; 
												
												if(++j>=lgth) throw new UnexceptedEndOfFileException();
												
												String putS = f.substring(i, j);
												try{PSTricksParserActionsManager.parseputCommand(putS, params);}
												catch(Exception e){ e.printStackTrace(); throw new InvalidFormatCommandException("put", nLine); }//$NON-NLS-1$
												i = j;
											}
											else
											{
												i=i2;
												throw new UnknownCommandException(nLine);
											}
											
											break;
											
										case 'o' : //polar
											if(i+4>=lgth)throw new UnexceptedEndOfFileException();
											if(f.charAt(++i)=='l' && f.charAt(++i)=='a' && f.charAt(++i)=='r' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' ||
												f.charAt(i)>'Z'))
												throw new WillNeverBeManagedException("polar", nLine);//$NON-NLS-1$
										break;
										
										default :
											i=i2;
											throw new UnknownCommandException(nLine);
									} // switch(f.charAt(i))
									break;
									
									
								case 'n' :
									i2 = i;
									if(i+1>=lgth)
									{
										if(!started) throw new IgnoreCommandException(nLine);
										throw new UnexceptedEndOfFileException();
									}
									i++;
									
									switch(f.charAt(i))
									{
										case 'e' ://ne
											if(i+1>=lgth)
											{
												i=i2;
												if(!started) throw new IgnoreCommandException(nLine);
												throw new UnexceptedEndOfFileException();
											}
											i++;
											if(f.charAt(i)=='w')//new
											{
												if(i+1>=lgth)
												{
													i=i2;
													if(!started) throw new IgnoreCommandException(nLine);
													throw new UnexceptedEndOfFileException();
												}
												i++;
												
												switch(f.charAt(i))
												{
													case 'p' : //newp
														if(i+1>=lgth)
														{
															i=i2;
															if(!started) throw new IgnoreCommandException(nLine);
															throw new UnexceptedEndOfFileException();
														}
														i++;
														switch(f.charAt(i))
														{
															case 's'://newps
																if(i+1>=lgth)
																{
																	i=i2;
																	if(!started) throw new IgnoreCommandException(nLine);
																	throw new UnexceptedEndOfFileException();
																}
																i++;
																switch(f.charAt(i))
																{
																	case 's' ://newpsstyle
																		if(i+5>=lgth)
																		{
																			if(!started) throw new IgnoreCommandException(nLine);
																			throw new UnexceptedEndOfFileException();
																		}
																		if(f.charAt(++i)=='t' && f.charAt(++i)=='y' && f.charAt(++i)=='l' &&
																			f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
																			f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																			throw new NotYetManagedException("newpsstyle", nLine);//$NON-NLS-1$
																		i=i2;
																		if(started) throw new UnknownCommandException(nLine);
																		throw new IgnoreCommandException(nLine);
																	
																	case 'o' : //newpsobject
																		if(i+1>=lgth)
																		{
																			if(!started) throw new IgnoreCommandException(nLine);
																			throw new UnexceptedEndOfFileException();
																		}
																		i++;
																		if(f.charAt(++i)=='b' && f.charAt(++i)=='j' && f.charAt(++i)=='e' &&
																			f.charAt(++i)=='c' && f.charAt(++i)=='t' &&	(f.charAt(++i)<'a' || 
																			f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																			throw new NotYetManagedException("newpsobject", nLine);//$NON-NLS-1$
																		i=i2;
																		if(started) throw new UnknownCommandException(nLine);
																		throw new IgnoreCommandException(nLine);
																	
																	default :
																		i=i2;
																		if(started) throw new UnknownCommandException(nLine);
																		throw new IgnoreCommandException(nLine);
																} // switch(f.charAt(i))
														
															case 'a'://newpa
																if(!started) throw new IgnoreCommandException(nLine);
																if(i+3>=lgth)
																	throw new UnexceptedEndOfFileException();
																
																if(f.charAt(++i)=='t' && f.charAt(++i)=='h'&& (f.charAt(++i)<'a' || 
																	f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
																{//newpath
																	texParserActions.actionsOnNewPath(nLine);
																}
																else
																{
																	i=i2;
																	throw new UnknownCommandException(nLine);
																}
																break;
																
															default:
																i=i2;
																if(started) throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
														}
														break;
														
													case 'c':// newcmykcolor
														if(i + 8 >= lgth)
														{
															if(!started)
																throw new IgnoreCommandException(nLine);
															throw new UnexceptedEndOfFileException();
														}
														
														if(f.charAt(++i) == 'm' && f.charAt(++i) == 'y' && f.charAt(++i) == 'k' && 
															f.charAt(++i) == 'c' && f.charAt(++i) == 'o' && f.charAt(++i) == 'l' && 
															f.charAt(++i) == 'o' && f.charAt(++i) == 'r' && (f.charAt(++i) < 'a' || 
																	f.charAt(i) > 'z') && (f.charAt(i) < 'A' || f.charAt(i) > 'Z'))
														{
															int j = i, cptBracket = 0;

															while(cptBracket < 2 && j < f.length())
															{
																if(f.charAt(j) == '}')
																	cptBracket++;
																j++;
															}

															if(cptBracket < 2)
																throw new InvalidFormatCommandException("newcmykcolor", nLine);//$NON-NLS-1$
															
															String p = f.substring(i, j);
															
															try { PSTricksParserActionsManager.parseNewcmykcolorParameters(p); }
															catch(Exception e)
															{
																e.printStackTrace();
																throw new InvalidFormatCommandException("newcmykcolor", nLine);//$NON-NLS-1$
															}
															
															i = j;
														}
														else
														{
															i = i2;
															if(started)
																throw new UnknownCommandException(nLine);
															throw new IgnoreCommandException(nLine);
														}
														break; 
													
													case 'h':// newhsbcolor
														if(i + 8 >= lgth)
														{
															if(!started)
																throw new IgnoreCommandException(nLine);
															throw new UnexceptedEndOfFileException();
														}
														
														if(f.charAt(++i) == 's' && f.charAt(++i) == 'b' && f.charAt(++i) == 'c' && 
															f.charAt(++i) == 'o' && f.charAt(++i) == 'l' && f.charAt(++i) == 'o' && 
															f.charAt(++i) == 'r' && (f.charAt(++i) < 'a' || f.charAt(i) > 'z') && 
															(f.charAt(i) < 'A' || f.charAt(i) > 'Z'))
														{
															int j = i, cptBracket = 0;

															while(cptBracket < 2 && j < f.length())
															{
																if(f.charAt(j) == '}')
																	cptBracket++;
																j++;
															}

															if(cptBracket < 2)
																throw new InvalidFormatCommandException("newhsbcolor", nLine);//$NON-NLS-1$
															
															String p = f.substring(i, j);
															
															try { PSTricksParserActionsManager.parseNewhsbcolorParameters(p); }
															catch(Exception e)
															{
																e.printStackTrace();
																throw new InvalidFormatCommandException("newhsbcolor", nLine);//$NON-NLS-1$
															}
															
															i = j;
														}
														else
														{
															i = i2;
															if(started)
																throw new UnknownCommandException(nLine);
															throw new IgnoreCommandException(nLine);
														}
														break; 
														
													  case 'r':// newrgbcolor
															if(i + 8 >= lgth)
															{
																if(!started)
																	throw new IgnoreCommandException(nLine);
																throw new UnexceptedEndOfFileException();
															}
															
															if(f.charAt(++i) == 'g' && f.charAt(++i) == 'b' && f.charAt(++i) == 'c' && 
																f.charAt(++i) == 'o' && f.charAt(++i) == 'l' && f.charAt(++i) == 'o' && 
																f.charAt(++i) == 'r' && (f.charAt(++i) < 'a' || f.charAt(i) > 'z') && 
																(f.charAt(i) < 'A' || f.charAt(i) > 'Z'))
															{
																int j = i, cptBracket = 0;

																while(cptBracket < 2 && j < f.length())
																{
																	if(f.charAt(j) == '}')
																		cptBracket++;
																	j++;
																}

																if(cptBracket < 2)
																	throw new InvalidFormatCommandException("newrgbcolor", nLine);//$NON-NLS-1$
																
																String p = f.substring(i, j);
																
																try { PSTricksParserActionsManager.parseNewrgbcolorParameters(p); }
																catch(Exception e)
																{
																	e.printStackTrace();
																	throw new InvalidFormatCommandException("newrgbcolor", nLine);//$NON-NLS-1$
																}
																
																i = j;
															}
															else
															{
																i = i2;
																if(started)
																	throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
															}
															break; 
														
													  case 'g':// newgray
															if(i + 4 >= lgth)
															{
																if(!started)
																	throw new IgnoreCommandException(nLine);
																throw new UnexceptedEndOfFileException();
															}
															
															if(f.charAt(++i) == 'r' && f.charAt(++i) == 'a' && f.charAt(++i) == 'y' && 
																(f.charAt(++i) < 'a' || f.charAt(i) > 'z') && (f.charAt(i) < 'A' || 
																f.charAt(i) > 'Z'))
															{
																int j = i, cptBracket = 0;

																while(cptBracket < 2 && j < f.length())
																{
																	if(f.charAt(j) == '}')
																		cptBracket++;
																	j++;
																}

																if(cptBracket < 2)
																	throw new InvalidFormatCommandException("newgray", nLine);//$NON-NLS-1$
																
																String p = f.substring(i, j);
																
																try { PSTricksParserActionsManager.parseNewgrayParameters(p); }
																catch(Exception e)
																{
																	e.printStackTrace();
																	throw new InvalidFormatCommandException("newgray", nLine);//$NON-NLS-1$
																}
																
																i = j;
															}
															else
															{
																i = i2;
																if(started)
																	throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
															}
															break;
												
													default :
														i=i2;
														if(started) throw new UnknownCommandException(nLine);
														throw new IgnoreCommandException(nLine);
												} // switch(f.charAt(i))
											}// if(f.charAt(i)=='w')
											else
											{
												i=i2;
												if(started) throw new UnknownCommandException(nLine);
												throw new IgnoreCommandException(nLine);
											}
											break;
											
										case 'o' :// no
											if(!started)   throw new IgnoreCommandException(nLine);
											if(i+9>=lgth) throw new UnexceptedEndOfFileException();

											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='r' && f.charAt(++i)=='m' && f.charAt(++i)=='a' && f.charAt(++i)=='l')
											{// normal
												if(i+1>=lgth) throw new UnexceptedEndOfFileException();
												++i;
												
												switch(f.charAt(i))
												{
													case 'c': // normalcolor
														if(i+5>=lgth) throw new UnexceptedEndOfFileException();
														
														if(f.charAt(++i)=='o' && f.charAt(++i)=='l' && f.charAt(++i)=='o' && 
															f.charAt(++i)=='r' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
															(f.charAt(i)<'A' || f.charAt(i)>'Z'))
																tp.textColor = Color.BLACK;
															else
															{
																i=i2;
																throw new UnknownCommandException(nLine);
															}
															break;
														
													case 's':// normalsize
														if(i+4>=lgth) throw new UnexceptedEndOfFileException();
														
														if(f.charAt(++i)=='i' && f.charAt(++i)=='z' && 
															f.charAt(++i)=='e' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
															(f.charAt(i)<'A' || f.charAt(i)>'Z'))
																tp.textSize = PSTricksConstants.COMMAND_TEXT_NORMAL;
														else
														{
															i=i2;
															throw new UnknownCommandException(nLine);
														}
														break;
														
													default :
														i=i2;
														throw new UnknownCommandException(nLine);
												}
											}
											else
											{
												i=i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i=i2;
											if(started) throw new UnknownCommandException(nLine);
											throw new IgnoreCommandException(nLine);
									}
									break;
								
									
								case 'r' : 
									i2 = i;
									i++;
									if(!started) { i=i2; throw new IgnoreCommandException(nLine); }
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 'o' : //ro
											if(i+4>=lgth)throw new UnexceptedEndOfFileException();
											
											if(f.charAt(++i)=='t' && f.charAt(++i)=='a' && f.charAt(++i)=='t' &&
												f.charAt(++i)=='e')//rotate
											{
												i++;
												if(i>=lgth)throw new UnexceptedEndOfFileException();
												switch(f.charAt(i))
												{
													case 'b' : //rotatebox
														if(i+3>=lgth)throw new UnexceptedEndOfFileException();
														if(f.charAt(++i)=='o' && f.charAt(++i)=='x' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
														  (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														{
															int nb=0, j = i;
															
															while(j < lgth && nb<2)
															{
																if(f.charAt(j)=='{')
																	nb++;
																j++;
															}

															if(j >= lgth)
															{
																j = i2;
																throw new InvalidFormatCommandException("rotatebox", nLine);//$NON-NLS-1$
															}
															
															PSTricksParserActionsManager.parseRotateBoxCommand(f.substring(i, j), params);

															i = j+1;
														}
														else
														{
															i=i2;
															throw new UnknownCommandException(nLine);
														}
														break;
														
													case 'l': // rotateleft
														if(i + 4 >= lgth)
															throw new UnexceptedEndOfFileException();
														if(f.charAt(++i) == 'e' && f.charAt(++i) == 'f' && f.charAt(++i) == 't' && 
															(f.charAt(++i) < 'a' || f.charAt(i) > 'z') && (f.charAt(i) < 'A' || f.charAt(i) > 'Z'))
														{
															params.add(new PSTricksParameters(params.lastElement()));

															while(i < lgth && PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																i++;

															if(i >= lgth || f.charAt(i) != '{')
															{
																i = i2;
																throw new InvalidFormatCommandException("rotateleft", nLine);//$NON-NLS-1$
															}

															i++;
															params.lastElement().setPosition(0, 0, true, 90, 0, 0, 0., 0.);
														}
														else
														{
															i = i2;
															throw new UnknownCommandException(nLine);
														}
														break; 
													
													case 'r' : // rotateright
														if(i+5>=lgth)throw new UnexceptedEndOfFileException();
														if(f.charAt(++i)=='i' && f.charAt(++i)=='g' && f.charAt(++i)=='h' && f.charAt(++i)=='t' &&
													      (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														{
															params.add(new PSTricksParameters(params.lastElement()));

															while(i < lgth && PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																i++;

															if(i >= lgth || f.charAt(i) != '{')
															{
																i = i2;
																throw new InvalidFormatCommandException("rotateright", nLine);//$NON-NLS-1$
															}

															i++;
															params.lastElement().setPosition(0, 0, true, -90, 0, 0, 0., 0.);
														}
														else
														{
															i = i2;
															throw new UnknownCommandException(nLine);
														}
														break; 	
															
													case 'd' : // rotatedown
														if(i+4>=lgth)throw new UnexceptedEndOfFileException();
														if(f.charAt(++i)=='o' && f.charAt(++i)=='w' && f.charAt(++i)=='n' &&
													      (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														{
															params.add(new PSTricksParameters(params.lastElement()));

															while(i < lgth && PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																i++;

															if(i >= lgth || f.charAt(i) != '{')
															{
																i = i2;
																throw new InvalidFormatCommandException("rotatedown", nLine);//$NON-NLS-1$
															}

															i++;
															params.lastElement().setPosition(0, 0, true, 180, 0, 0, 0., 0.);
														}
														else
														{
															i = i2;
															throw new UnknownCommandException(nLine);
														}
														break;	
													
													default :
														i=i2;
														throw new UnknownCommandException(nLine);	
												} // switch(f.charAt(i))
											} // if(f.charAt(++i)=='t' && f.
											else
											{
												i=i2;
												throw new UnknownCommandException(nLine);
											}
											break;
										
										case 'e' : // re
											i++;
											if(i>=lgth)
											{
												i=i2;
												throw new IgnoreCommandException(nLine);
											}
											
											switch(f.charAt(i))
											{
												case 'd' : // red
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();

													PSTricksParameters tp = params.lastElement();
													if((f.charAt(++i)<'a' || f.charAt(i)>'z') && 
														(f.charAt(i)<'A' || f.charAt(i)>'Z'))
														tp.textColor = Color.RED;
													else
													{
														i=i2;
														throw new IgnoreCommandException(nLine);
													}
													break;
													
												case 's' : // resizebox
													if(i+7>=lgth)throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='i' && f.charAt(++i)=='z' && f.charAt(++i)=='e' &&
														f.charAt(++i)=='b' && f.charAt(++i)=='o' && f.charAt(++i)=='x' &&
														(f.charAt(++i)<'a' || f.charAt(i)>'z') && 
														(f.charAt(i)<'A' || f.charAt(i)>'Z'))
														throw new NotYetManagedException("resizebox", nLine);//$NON-NLS-1$
													
													i=i2;
													throw new UnknownCommandException(nLine);
													
												default :
													i=i2;
													throw new IgnoreCommandException(nLine);
											}
											break;
											
										
										case 'a' : // radians
											if(i+6>=lgth) throw new UnexceptedEndOfFileException();
											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='d' && f.charAt(++i)=='i' && f.charAt(++i)=='a' &&
												f.charAt(++i)=='n' && f.charAt(++i)=='s' && (f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												tp.onRadians = true;
												params.lastElement().degrees=Math.PI*2;
											}
											else
											{
												i=i2;
												throw new IgnoreCommandException(nLine);
											}
											break;
											
										case 'p' : // rput
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();
											if(f.charAt(++i)=='u' && f.charAt(++i)=='t' && (f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int j = i;
												boolean end = false, hasStar = f.charAt(i)=='*';
												
												while(j<lgth && ((f.charAt(j)!='{' || !end))) 
												{ // We extract the command the string : rput*{..}(..){...
													if(f.charAt(j)=='(') end = true;
													else if(f.charAt(j)=='{') end = true;
													j++; 
												}
												if(++j>=lgth) throw new UnexceptedEndOfFileException();
												
												String rputS = f.substring(i, j);
												try{PSTricksParserActionsManager.parserputCommand(rputS, hasStar, params);}
												catch(Exception e){ e.printStackTrace(); throw new InvalidFormatCommandException("rput", nLine); }//$NON-NLS-1$
												i = j;
											}else
											{
												i=i2;
												throw new IgnoreCommandException(nLine);
											}
											break;
											
										default : 
											i=i2;
											throw new IgnoreCommandException(nLine);
									}
									break;
									
									
									
								case 's' :
								if(!started) throw new IgnoreCommandException(nLine);
								i2 = i;	
								i++;
								if(i>=lgth) throw new UnexceptedEndOfFileException();
								
								switch(f.charAt(i))
								{
									case 'c' : //sc
										if(i+1>=lgth) throw new UnexceptedEndOfFileException();
										i++;
										switch(f.charAt(i))
										{
											case 'a' : //sca
												if(i+5>=lgth) throw new UnexceptedEndOfFileException();
												if(f.charAt(++i)=='l' && f.charAt(++i)=='e' && f.charAt(++i)=='b' &&
													f.charAt(++i)=='o' && f.charAt(++i)=='x')
												{
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													if((f.charAt(++i)<'a' || f.charAt(i)>'z') && 
														(f.charAt(i)<'A' || f.charAt(i)>'Z'))
														throw new NotYetManagedException("scalebox", nLine);//$NON-NLS-1$
													
													if(i+3>=lgth) throw new UnexceptedEndOfFileException();
													if(f.charAt(++i)=='t' && f.charAt(++i)=='o' &&  
														(f.charAt(++i)<'a' || f.charAt(i)>'z') && 
														(f.charAt(i)<'A' || f.charAt(i)>'Z'))
															throw new NotYetManagedException("scaleboxto", nLine);//$NON-NLS-1$
												}
												i=i2;
												throw new UnknownCommandException(nLine);
											
											case 'r' : // scriptsize
												if(i+8>=lgth) throw new UnexceptedEndOfFileException();
												PSTricksParameters tp = params.lastElement();
												if(f.charAt(++i)=='i' && f.charAt(++i)=='p' && f.charAt(++i)=='t' &&
													f.charAt(++i)=='s' && f.charAt(++i)=='i' && f.charAt(++i)=='z' && 
													f.charAt(++i)=='e' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
													(f.charAt(i)<'A' || f.charAt(i)>'Z'))
													tp.textSize = PSTricksConstants.COMMAND_TEXT_SCRIPT;
												else
												{
													i=i2;
													throw new UnknownCommandException(nLine);
												}
												break;
												
											default :
												i=i2;
												throw new UnknownCommandException(nLine);
										}
										break;
									
									case 'l' : // sl
										if(i+1>=lgth) throw new UnexceptedEndOfFileException();

										PSTricksParameters tp = params.lastElement();
										if((f.charAt(++i)<'a' || f.charAt(i)>'z') && 
											(f.charAt(i)<'A' || f.charAt(i)>'Z'))
											tp.textItalic = true;
										else
										{
											i = i2;
											throw new UnknownCommandException(nLine);
										}
										break;
										
									case 'm' : // small
										if(i+4>=lgth) throw new UnexceptedEndOfFileException();

										tp = params.lastElement();
										if(f.charAt(++i)=='a' && f.charAt(++i)=='l' && f.charAt(++i)=='l' &&
											(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
										   f.charAt(i)>'Z'))
											tp.textSize = PSTricksConstants.COMMAND_TEXT_SMALL;
										else
										{
											i = i2;
											throw new UnknownCommandException(nLine);
										}
										break;
									
									case 't' : // st
										if(i+2>=lgth) throw new UnexceptedEndOfFileException();
										
										if(f.charAt(++i)=='r') // str
										{
											i++;
											if(f.charAt(i)=='o') // stro
											{
												if(i+3>=lgth) throw new UnexceptedEndOfFileException();
												if(f.charAt(++i)=='k' && f.charAt(++i)=='e' && 
													(f.charAt(++i)<'a' || f.charAt(i)>'z') &&
													(f.charAt(i)<'A' || f.charAt(i)>'Z')) // stroke
													throw new NotYetManagedException("stroke", nLine);//$NON-NLS-1$
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											else if(f.charAt(i)=='i') // stri
											{
												if(i+3>=lgth) throw new UnexceptedEndOfFileException();
												if(f.charAt(++i)=='n' && f.charAt(++i)=='g' && 
													(f.charAt(++i)<'a' || f.charAt(i)>'z') &&
													(f.charAt(i)<'A' || f.charAt(i)>'Z')) // string
												{
													// nothing to do
												}else
												{
													i = i2;
													throw new UnknownCommandException(nLine);
												}
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
										}
										else
										{
											i = i2;
											throw new UnknownCommandException(nLine);
										}
										break;
									
									case 'w' : // swapaxes
										if(i+7>=lgth) throw new UnexceptedEndOfFileException();
										if(f.charAt(++i)=='a' && f.charAt(++i)=='p' && f.charAt(++i)=='a' &&
											f.charAt(++i)=='x' && f.charAt(++i)=='e' && f.charAt(++i)=='s' &&
											(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
										   f.charAt(i)>'Z'))
											throw new NotYetManagedException("swapaxes", nLine);//$NON-NLS-1$
										i = i2;
										throw new UnknownCommandException(nLine);
									
									default :
										i=i2;
										throw new UnknownCommandException(nLine);
								}								
								break ;
								
								
									
								case 'b' :
									i2 = i;
									i++;
									if(i>=lgth)
									{
										if(!started) throw new IgnoreCommandException(nLine);
										throw new UnexceptedEndOfFileException();
									}
									
									switch(f.charAt(i))
									{
										case 'l' : //bl
											if(!started) { i=i2; throw new IgnoreCommandException(nLine);}
											i++;
											if(i>=lgth) throw new UnexceptedEndOfFileException();
											
											switch(f.charAt(i))
											{
												case 'a' : // black
													if(i+3>=lgth) throw new UnexceptedEndOfFileException();

													PSTricksParameters tp = params.lastElement();
													if(f.charAt(++i)=='c' && f.charAt(++i)=='k' && (f.charAt(++i)<'a' || 
														f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														tp.textColor = Color.BLACK;
													else
													{
														i = i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												case 'u' : // blue
													if(i+2>=lgth) throw new UnexceptedEndOfFileException();

													tp = params.lastElement();
													if(f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
														f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
														tp.textColor = Color.BLUE;
													else
													{
														i = i2;
														throw new UnknownCommandException(nLine);
													}
													break;
													
												default :
													i=i2;
													if(started)throw new UnknownCommandException(nLine);
											}
											break;
										

										case 'e' : //be
											if(i+4>=lgth) 
											{
												if(!started) throw new IgnoreCommandException(nLine);
												throw new UnexceptedEndOfFileException();
											}
											if(f.charAt(++i)=='g' && f.charAt(++i)=='i' && f.charAt(++i)=='n' && 
											   (f.charAt(++i)<'a'  || f.charAt(i)>'z')&& (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
											{//begin
												while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
												{
													if(f.charAt(i)==eol) nLine++;
													i++;
													if(i>=lgth) throw new UnexceptedEndOfFileException();
												}
												if(f.charAt(i)=='{')
												{//begin{
													i++;
													if(i>=lgth) throw new UnexceptedEndOfFileException();
													while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
													{
														if(f.charAt(i)==eol) nLine++;
														i++;
														if(i>=lgth) throw new UnexceptedEndOfFileException();
													}
													if(i+1>=lgth) throw new UnexceptedEndOfFileException();
													
													switch(f.charAt(i))
													{
														case 'p'://begin{p
															if(i+9>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='s' && f.charAt(++i)=='p' &&
															   f.charAt(++i)=='i' && f.charAt(++i)=='c' && f.charAt(++i)=='t' &&
															   f.charAt(++i)=='u' && f.charAt(++i)=='r' && f.charAt(++i)=='e' &&
															   (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//begin{pspicture
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}')
																{//begin{pspicture}
																	if(started)
																		throw new FatalErrorParsingException(
																		LaTeXDrawLang.getString1_6("PSTricksParser.1")+ //$NON-NLS-1$
																		LaTeXDrawLang.getString1_6("PSTricksParser.2")); //$NON-NLS-1$
																	started = true;
																	startedWithBeginPs = true;
																	Point2D.Double pt1, pt2;
																	String sub = f.substring(i);
																	try 
																	{
																		pt1 = PSTricksParserActionsManager.readOneCoordinate(sub, 0, nLine, true);
																		i += sub.indexOf(")", 0);//$NON-NLS-1$
																	}catch(Exception e)
																	{ 
																		throw new FatalErrorParsingException("Invalid command begin{pspicture}");//$NON-NLS-1$
																	}
																	
																	int j, k = ++i;
																	try 
																	{ // optional
																		while(k<lgth && (PSTricksParserActionsManager.isCharToIgnore(f.charAt(k)) || 
																				f.charAt(k)=='\\'))
																			k++;
																	
																		if(f.charAt(k)!='(') throw new IgnoreCommandException(-1);
																		j = f.indexOf(')', k);
																		sub = f.substring(k, j+1);
																		pt2 = PSTricksParserActionsManager.readOneCoordinate(sub, 0, nLine, true);
																		i = j;
																	}catch(IgnoreCommandException e)
																	{
																		pt2 = pt1;
																		pt1 = new Point2D.Double(0,0);
																	}
																	catch(Exception e)
																	{
																		throw new FatalErrorParsingException("Invalid command begin{pspicture}");//$NON-NLS-1$
																	}
																	
																	PSTricksParameters.pictureNEPt = pt2;
																	PSTricksParameters.pictureSWPt = pt1;
																	params.add(new PSTricksParameters());
																	
																}else throw new InvalidFormatCommandException("\\begin{pspicture}", nLine);//$NON-NLS-1$
															}else
															{
																i = i2;
																if(started) throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
															}
															break;
															
														case 'c'://begin{c
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='e' && f.charAt(++i)=='n' &&
															   f.charAt(++i)=='t' && f.charAt(++i)=='e' && 
															   f.charAt(++i)=='r' && (f.charAt(++i)<'a' || 
															   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//begin{center
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}') //begin{center}
																{
																	if(started && !parsingFinished)
																		throw new FatalErrorParsingException(LaTeXDrawLang.getString1_9("PSTricksParser.0")); //$NON-NLS-1$
																	
																	PSTricksParameters.isCentered = true;
																}
																else 
																	throw new InvalidFormatCommandException("\\begin{center}", nLine);//$NON-NLS-1$
															}else
															{
																i = i2;
																if(started) throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
															}
															break;
															
														case 'f'://begin{f
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='i' && f.charAt(++i)=='g' &&
															   f.charAt(++i)=='u' && f.charAt(++i)=='r' && 
															   f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
															   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//begin{figure
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}') //begin{figure}
																	i += PSTricksParserActionsManager.parseBeginFigureCmd(f.substring(++i), nLine);
																else throw new InvalidFormatCommandException("\\begin{pspicture}", nLine);//$NON-NLS-1$
																	
															}else
															{
																i = i2;
																if(started) throw new UnknownCommandException(nLine);
																throw new IgnoreCommandException(nLine);
															}
															break;
															
														default :
															i = i2;
															if(!started) throw new IgnoreCommandException(nLine);
															throw new IgnoreCommandException(nLine);
													}
												}
												else
												{
													i = i2;
													if(!started) throw new IgnoreCommandException(nLine);
													throw new IgnoreCommandException(nLine);
												}
											}else
											{
												i = i2;
												if(!started) throw new IgnoreCommandException(nLine);
												throw new IgnoreCommandException(nLine);
											}
											break;
										
										
										case 'f' : // bf
											if(!started) { i=i2; throw new IgnoreCommandException(nLine);}
											if(i>=lgth) throw new UnexceptedEndOfFileException();
											
											PSTricksParameters tp = params.lastElement();
											if((f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												tp.textBold = true;
											else
											{
												i = i2;
												throw new IgnoreCommandException(nLine);
											}
											break;
											
										case 'o' : // box
											if(!started)  { i=i2; throw new IgnoreCommandException(nLine); }
											if(i+2>=lgth) throw new UnexceptedEndOfFileException();
											
											if(f.charAt(++i)=='x' && (f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
												throw new NotYetManagedException("box", nLine);//$NON-NLS-1$
												
											i = i2;
											throw new IgnoreCommandException(nLine);
										
										default :
											i=i2;
											if(started) throw new UnknownCommandException(nLine);
											throw new IgnoreCommandException(nLine);
									}
									break;
									
									
								case 'l' :
									if(!started) throw new IgnoreCommandException(nLine);
									i2 = i;
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 'a' : // large
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();

											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='r' && f.charAt(++i)=='g' && f.charAt(++i)=='e' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												tp.textSize = PSTricksConstants.COMMAND_TEXT_LARGE1;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'd' : // ldots
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();

											if(f.charAt(++i)=='o' && f.charAt(++i)=='t' && f.charAt(++i)=='s' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												throw new NotYetManagedException("ldots", nLine);//$NON-NLS-1$
												
											i = i2;
											throw new UnknownCommandException(nLine);
											
										case 'i'://li
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();

											if(f.charAt(++i)=='n' && f.charAt(++i)=='e' && f.charAt(++i)=='t' &&
												f.charAt(++i)=='o' &&(f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int end = texParserActions.actionsOnLineTo(nLine, 
														params, f.substring(i, f.length()));
												if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i=i2;
											throw new UnknownCommandException(nLine);
									}
								break;

								case '%' :
									if(started)
									{
										PSTricksParameters p = params.lastElement();
										p.textParsed+='%';
									}
									i++;
									break;
								
									
								case 'e' :
									i2 = i;
									
									if(i+3>=lgth) 
									{
										if(started) throw new UnexceptedEndOfFileException();
										throw new IgnoreCommandException(nLine);
									}
									
									if(f.charAt(++i)=='n' && f.charAt(++i)=='d')
									{//end
										++i;
										
										if(f.charAt(i)=='p')
										{//endp
											if(f.charAt(++i)=='s' && f.charAt(++i)=='p' && f.charAt(++i)=='i' && f.charAt(++i)=='c' &&
												f.charAt(++i)=='t' && f.charAt(++i)=='u' && f.charAt(++i)=='r' && f.charAt(++i)=='e' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{//endpspicture
												if(!started || (started && startedWithBeginPs))
													throw new FatalErrorParsingException("\\endpspicture must come after \\pspicture");//$NON-NLS-1$
												
												parsingFinished = true;
												i++;
											}
											else
											{
												if(started) throw new UnknownCommandException(nLine);
												throw new IgnoreCommandException(nLine);
											}
										}
										else
										{
											 if((f.charAt(i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											 {
												while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
												{
													if(f.charAt(i)==eol) nLine++;
													i++;
													if(i>=lgth) throw new UnexceptedEndOfFileException();
												}
												if(f.charAt(i)=='{')
												{//end{
													i++;
													if(i>=lgth) throw new UnexceptedEndOfFileException();
													while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
													{
														if(f.charAt(i)==eol) nLine++;
														i++;
														if(i>=lgth) throw new UnexceptedEndOfFileException();
													}
													
													switch(f.charAt(i))
													{
														case 'p'://end{p
															if(i+9>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='s' && f.charAt(++i)=='p' &&
															   f.charAt(++i)=='i' && f.charAt(++i)=='c' && f.charAt(++i)=='t' &&
															   f.charAt(++i)=='u' && f.charAt(++i)=='r' && f.charAt(++i)=='e' &&
															   (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
															   f.charAt(i)>'Z'))
															{//end{pspicture
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}')
																{//end{pspicture}
																	if(!started || (started && !startedWithBeginPs))
																		throw new FatalErrorParsingException("end{pspicture} must come after begin{pspicture}");//$NON-NLS-1$
																	
																	parsingFinished = true;
																	i++;
																}
																else 
																	throw new InvalidFormatCommandException("\\end{pspicture}", nLine);//$NON-NLS-1$
															} //if(f.charAt(i)=='p' && f.charAt...
															else
															{
																i = i2;
																throw new UnexceptedEndOfFileException();
															}
														break;
														
														case 'c'://end{c
															if(i+6>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='e' && f.charAt(++i)=='n' &&
															   f.charAt(++i)=='t' && f.charAt(++i)=='e' && 
															   f.charAt(++i)=='r' && (f.charAt(++i)<'a' || 
															   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//end{center
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}') //end{center} 
																{
																	if(started && !parsingFinished)
																		throw new FatalErrorParsingException(LaTeXDrawLang.getString1_9("PSTricksParser.1")); //$NON-NLS-1$
																	
																	if(started)
																		parsingFinished = true;
																	else
																		PSTricksParameters.isCentered = false;
																}
																else 
																	throw new InvalidFormatCommandException("\\end{center}", nLine);//$NON-NLS-1$
															}else
															{
																i = i2;
																throw new UnexceptedEndOfFileException();
															}
															break;
															
														case 'f'://end{f 
															if(i+9>=lgth) throw new UnexceptedEndOfFileException();
															if(f.charAt(++i)=='i' && f.charAt(++i)=='g' &&
															   f.charAt(++i)=='u' && f.charAt(++i)=='r' && 
															   f.charAt(++i)=='e' && (f.charAt(++i)<'a' || 
															   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
															{//end{figure
																while(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
																{
																	if(f.charAt(i)==eol) nLine++;
																	i++;
																	if(i>=lgth) throw new UnexceptedEndOfFileException();
																}
																if(f.charAt(i)=='}') //end{figure} 
																{
																	if(started && !parsingFinished)
																		throw new FatalErrorParsingException(LaTeXDrawLang.getString1_9("PSTricksParser.2")); //$NON-NLS-1$
																	
																	if(started)
																		parsingFinished = true;
																	else
																		PSTricksParameters.tokenPosition = "";//$NON-NLS-1$
																}
																else 
																	throw new InvalidFormatCommandException("\\end{figure}", nLine);//$NON-NLS-1$
															}else
															{
																i = i2;
																throw new UnexceptedEndOfFileException();
															}
															break;
													}
												}
												else
												{
													i = i2;
													throw new UnknownCommandException(nLine);
												}
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
										}// if(f.charAt(++i)=='n' && f.charAt(++i)=...
									}
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'd' : 
									if(!started) throw new IgnoreCommandException(nLine);
									i2 = i;
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									if(f.charAt(i)!='e')
									{
										i=i2;
										throw new UnknownCommandException(nLine);
									}
									
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();

									switch(f.charAt(i))
									{
										case 'g' : // degrees
											if(i+5>=lgth) throw new UnexceptedEndOfFileException();

											if(f.charAt(++i)=='r' && f.charAt(++i)=='e' && f.charAt(++i)=='e' &&
												f.charAt(++i)=='s' && (f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int id = texParserActions.parseDegreesCommand(f.substring(i),params.lastElement(), nLine);
												i += id;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'f' : // definecolor
											if(i+9>=lgth) throw new UnexceptedEndOfFileException();

											if(f.charAt(++i)=='i' && f.charAt(++i)=='n' && f.charAt(++i)=='e' &&
												f.charAt(++i)=='c' && (f.charAt(++i)=='o' && f.charAt(++i)=='l' &&
												f.charAt(++i)=='o' && f.charAt(++i)=='r' && f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int id1 = f.indexOf("{", i);//$NON-NLS-1$
												int id2 = f.indexOf("}", i);//$NON-NLS-1$
												
												if(id1==-1 || id2==-1) 
												{
													i=i2;
													throw new UnknownCommandException(nLine);
												}
												
												int[] tmp = {0};
												String name = PSTricksParserActionsManager.readString(f.substring(id1, id2), tmp, true);
												
												if(name==null || name.length()==0)
												{
													i=i2;
													throw new UnknownCommandException(nLine);
												}
												
												id1 = f.indexOf("{", id2);//$NON-NLS-1$
												id2 = f.indexOf("}", id2+1);//$NON-NLS-1$
												
												if(id1==-1 || id2==-1) 
												{
													i=i2;
													throw new UnknownCommandException(nLine);
												}
												tmp[0]=0;
												String type = PSTricksParserActionsManager.readString(f.substring(id1, id2), tmp, true);
												
												if(name.length()==0)
												{
													i=i2;
													throw new UnknownCommandException(nLine);
												}
												
												id1 = f.indexOf("{", id2);//$NON-NLS-1$
												id2 = f.indexOf("}", id2+1);//$NON-NLS-1$
												tmp[0] = 0;
												String subStr = f.substring(id1+1, id2);
												
												if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_NAMED))
												{ // definecolor{alias}{named}{colorName}
													String val1 = PSTricksParserActionsManager.readString(subStr, tmp, true);
													
													Color c = DviPsColors.getColour(val1);
													if(c!=null)
														DviPsColors.addUserColour(c, name);
												}
												else
												{
													double val1 = PSTricksParserActionsManager.readReal(subStr, tmp, true);
													if(val1<0 || val1>1) throw new UnknownCommandException(nLine);
													
													if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_GRAY)) // definecolor{name}{gray}{val}
														DviPsColors.addUserColour(new Color((float)val1, (float)val1, (float)val1), name);	
													else
													{
														PSTricksParserActionsManager.readChar(subStr, ',', tmp);
														double val2 = PSTricksParserActionsManager.readReal(subStr, tmp, true);
														PSTricksParserActionsManager.readChar(subStr, ',', tmp);
														double val3 = PSTricksParserActionsManager.readReal(subStr, tmp, true);
														
														if(val2<0 || val2>1 || val3<0 || val3>1) 
															throw new UnknownCommandException(nLine);
														
														if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_RGB))// definecolor{name}{rgb}{r g b}
															DviPsColors.addUserColour(new Color((float)val1, (float)val2, (float)val3), name);
														else 
															if(type.equals(PSTricksConstants.TOKEN_COLOR_TYPE_HSB))
																DviPsColors.addUserColour(Color.getHSBColor((float)val1, (float)val2, (float)val3), name);
															else
																if(type.compareTo(PSTricksConstants.TOKEN_COLOR_TYPE_CMYK)==0)
	                                                            {
																	PSTricksParserActionsManager.readChar(subStr, ',', tmp);
	                                                                double val4 = PSTricksParserActionsManager.readReal(subStr, tmp, true);
	                                                              
	                                                                if(val4<0 || val4>1)
	                                                                    throw new InvalidFormatCommandException(nLine);
	                                                              
	                                                                DviPsColors.addUserColour(DviPsColors.CMYK2RGB(val1, val2, val3, val4), name);
	                                                             }
	                                                             else
	                                                                 throw new UnknownCommandException(nLine); 
													}
												}
												i = id2+2;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i-=2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'g' :
									if(!started) throw new IgnoreCommandException(nLine);
									i2 = i;
								
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									
									if(f.charAt(i)!='r') 
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();

									switch(f.charAt(i))
									{
										case 'a' : // gray
											if(i+2>=lgth) throw new UnexceptedEndOfFileException();

											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='y' && (f.charAt(++i)<'a' || 
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
												tp.textColor = Color.GRAY;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'e' : // green
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();

											i2 = i;
											tp = params.lastElement();
											if(f.charAt(++i)=='e' && f.charAt(++i)=='n' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												tp.textColor = Color.GREEN;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i=i2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'L' :
									if(!started) throw new IgnoreCommandException(nLine);
								
									i2 = i;
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 'A' : // LARGE
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();

											i2 = i;
											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='R' && f.charAt(++i)=='G' && f.charAt(++i)=='E' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												tp.textSize = PSTricksConstants.COMMAND_TEXT_LARGE3;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'a' : // Large
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();
											
											i2 = i;
											tp = params.lastElement();
											if(f.charAt(++i)=='r' && f.charAt(++i)=='g' && f.charAt(++i)=='e' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
												f.charAt(i)>'Z'))
												tp.textSize = PSTricksConstants.COMMAND_TEXT_LARGE2;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i=i2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'q' :
								
									if(!started) throw new IgnoreCommandException(nLine);
									i2 = i;
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 'l' : // qline
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();
											i2 = i;
											if(f.charAt(++i)=='i' && f.charAt(++i)=='n' && f.charAt(++i)=='e' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												throw new NotYetManagedException("qline", nLine);//$NON-NLS-1$
											i = i2;
											throw new UnknownCommandException(nLine);
											
										case 'd' : // qdisk
											if(i+4>=lgth) throw new UnexceptedEndOfFileException();
											i2 = i;
											if(f.charAt(++i)=='i' && f.charAt(++i)=='s' && f.charAt(++i)=='k' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												throw new NotYetManagedException("qdisk", nLine);//$NON-NLS-1$
											i = i2;
											throw new UnknownCommandException(nLine);
											
										default :
											i = i2;
											throw new UnknownCommandException(nLine);
												
									}

								case 'c' :
									if(!started) throw new IgnoreCommandException(nLine);
								
									i2 = i;
									i++;
									if(i>=lgth) throw new IgnoreCommandException(nLine);

									switch(f.charAt(i))
									{
										case 'p' : // cput
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();
											
											i2 = i;
											if(f.charAt(++i)=='u' && f.charAt(++i)=='t' && (f.charAt(++i)=='*' ||
												f.charAt(i)=='[' || f.charAt(i)=='{'))
												throw new NotYetManagedException("cput", nLine);//$NON-NLS-1$
											i = i2;
											throw new UnknownCommandException(nLine);
										
										case 'y' : // cyan
											if(i+3>=lgth) throw new IgnoreCommandException(nLine);

											i2 = i;
											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='a' && f.charAt(++i)=='n' && 
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
												f.charAt(i)>'Z'))
												tp.textColor = Color.CYAN;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'o': // color
											if(i+4>lgth) throw new UnexceptedEndOfFileException();
											
											if(f.charAt(++i)=='l' && f.charAt(++i)=='o' && 
											   f.charAt(++i)=='r' && (f.charAt(++i)=='{' || PSTricksParserActionsManager.isCharToIgnore(f.charAt(i))))
											{
												while(i<lgth && f.charAt(i)!='{') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												int[] id = {0};
												tp = params.lastElement();
												String tmp = PSTricksParserActionsManager.readString(f.substring(i), id, true);
												Color c = DviPsColors.getColour(tmp); // We search the color
												if(c!=null) tp.textColor = c;
												while(i<lgth && f.charAt(i)!='}') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'u'://cu
											if(i+6>=lgth) throw new UnexceptedEndOfFileException();
											
											i2 = i;
											if(f.charAt(++i)=='r' && f.charAt(++i)=='v' && f.charAt(++i)=='e' &&
												f.charAt(++i)=='t' && f.charAt(++i)=='o' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
												   f.charAt(i)>'Z'))//curveto
											{
												int end = texParserActions.actionsOnCurveTo(nLine, 
																params, f.substring(i, f.length()));
												if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'l'://cl
											if(i+8>=lgth) throw new UnexceptedEndOfFileException();
											
											i2 = i;
											if(f.charAt(++i)=='o' && f.charAt(++i)=='s' && f.charAt(++i)=='e' &&
												f.charAt(++i)=='p' && f.charAt(++i)=='a' &&
												f.charAt(++i)=='t' && f.charAt(++i)=='h' &&
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
												   f.charAt(i)>'Z'))//closepath
												texParserActions.actionsOnClosePathCommand(nLine);
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i = i2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 't' :
									if(!started) throw new IgnoreCommandException(nLine);
									
									i2 = i;
									i++;
									if(i>=lgth) throw new UnexceptedEndOfFileException();

									switch(f.charAt(i))
									{
										case 'r' ://translate
											if(i+8>=lgth) throw new UnexceptedEndOfFileException();
											
											i2 = i;
											if(f.charAt(++i)=='a' && f.charAt(++i)=='n' && f.charAt(++i)=='s' && 
											   f.charAt(++i)=='l' && f.charAt(++i)=='a' && f.charAt(++i)=='t' && 
											   f.charAt(++i)=='e' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
											   (f.charAt(i)<'A' || f.charAt(i)>'Z'))
												throw new NotYetManagedException("translate", nLine);//$NON-NLS-1$
											
											i = i2;
											throw new UnknownCommandException(nLine);
											
											
										case 'i' : //tiny
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();

											i2 = i;
											PSTricksParameters tp = params.lastElement();
											if(f.charAt(++i)=='n' && f.charAt(++i)=='y' && 
											  (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || 
											   f.charAt(i)>'Z'))
												tp.textSize = PSTricksConstants.COMMAND_TEXT_TINY;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i = i2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'u' :
									if(!started) throw new IgnoreCommandException(nLine);
									i2 = i;
									if(++i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 'p' : // uput
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();
											
											if(f.charAt(++i)=='u' && f.charAt(++i)=='t' &&
											   (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
											   (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int j = i;
												boolean brack = false, end=false;//, hasStar = f.charAt(i)=='*';
												
												while(j<lgth && !end) 
												{ // We extract the command the string : uput*{..}[..]{..}(..){...
													if(f.charAt(j)=='(') brack = true;
													else if(f.charAt(j)=='{' && brack) end = true;
													j++; 
												}
												if(++j>=lgth) throw new UnexceptedEndOfFileException();

											//	String uputS = f.substring(i, j);
											//	try{AbstractPSTricksParserActions.parseuputCommand(uputS, hasStar, params);}
											//	catch(Exception e){ e.printStackTrace(); throw new InvalidFormatCommandException("uput", nLine); }
												i = j-2;
												throw new NotYetManagedException("uput", nLine);//$NON-NLS-1$
											}//else
											//{
												i=i2;
												throw new IgnoreCommandException(nLine);
											//}
										//	break;
											
										case 's' : // usefont{enc}{family}{series}{shape}
											if(i+6>lgth) { i = i2; throw new UnknownCommandException(nLine); }
											
											if(f.charAt(++i)=='e' && f.charAt(++i)=='f' && 
											   f.charAt(++i)=='o' && f.charAt(++i)=='n' && f.charAt(++i)=='t' && 
											   (f.charAt(++i)=='{' || PSTricksParserActionsManager.isCharToIgnore(f.charAt(i))))
											{
												PSTricksParameters tp = params.lastElement();
												while(i<lgth && f.charAt(i)!='{') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												
												int[] tmp = {0};
												tp.textEnc = PSTricksParserActionsManager.readString(f.substring(i), tmp, true);
												while(i<lgth && f.charAt(i)!='}') i++; // We search the '}'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												while(i<lgth && f.charAt(i)!='{') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												tmp[0] = 0;
												tp.textFamily = PSTricksParserActionsManager.readString(f.substring(i), tmp, true);
												while(i<lgth && f.charAt(i)!='}') i++; // We search the '}'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												while(i<lgth && f.charAt(i)!='{') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												tmp[0] = 0;
												tp.textSeries = PSTricksParserActionsManager.readString(f.substring(i), tmp, true);
												while(i<lgth && f.charAt(i)!='}') i++; // We search the '}'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												while(i<lgth && f.charAt(i)!='{') i++; // We search the '{'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
												tmp[0] = 0;
												tp.textShape = PSTricksParserActionsManager.readString(f.substring(i), tmp, true);
												while(i<lgth && f.charAt(i)!='}') i++; // We search the '}'
												if(i>=lgth) throw new UnexceptedEndOfFileException();
												i++;
											}else 
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default :
											i = i2;
											throw new UnknownCommandException(nLine);
									} // switch(f.charAt(i))
									break;
									
								
								case 'H' :  // Huge
									if(!started)  throw new IgnoreCommandException(nLine);
									if(i+4>=lgth) throw new UnexceptedEndOfFileException();
									
									i2 = i;
									PSTricksParameters tp = params.lastElement();
									if(f.charAt(++i)=='u' && f.charAt(++i)=='g' && f.charAt(++i)=='e' &&
									  (f.charAt(++i)<'a' ||  f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.textSize = PSTricksConstants.COMMAND_TEXT_HUGE2;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
										
								case 'N' : //NormalCoor
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+11>=lgth) throw new UnexceptedEndOfFileException();
									
									i2 = i;
									tp = params.lastElement();
									if(f.charAt(++i)=='o' && f.charAt(++i)=='r' && f.charAt(++i)=='m' && 
										f.charAt(++i)=='a' && f.charAt(++i)=='l' && 
										f.charAt(++i)=='C' && f.charAt(++i)=='o' && f.charAt(++i)=='o' && 
										f.charAt(++i)=='r' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
										(f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.specialCoor = false;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'S' : //SpecialCoor
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+11>=lgth) throw new UnexceptedEndOfFileException();
								
									i2 = i;
									tp = params.lastElement();
									if(f.charAt(++i)=='p' && f.charAt(++i)=='e' && f.charAt(++i)=='c' && 
										f.charAt(++i)=='i' && f.charAt(++i)=='a' && f.charAt(++i)=='l' && 
										f.charAt(++i)=='C' && f.charAt(++i)=='o' && f.charAt(++i)=='o' && 
										f.charAt(++i)=='r' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
										(f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.specialCoor = true;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'f': //f
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+1>=lgth) throw new UnexceptedEndOfFileException();
									i2 = i;
									
									switch(f.charAt(++i))
									{
										case 'o'://fo
											if(i+11>=lgth) throw new UnexceptedEndOfFileException();
										
											tp = params.lastElement();
											if(f.charAt(++i)=='o' && f.charAt(++i)=='t' &&// footnodesize
												f.charAt(++i)=='n' && f.charAt(++i)=='o' && f.charAt(++i)=='t' &&
												f.charAt(++i)=='e' && f.charAt(++i)=='s' && f.charAt(++i)=='i' &&
												f.charAt(++i)=='z' && f.charAt(++i)=='e' &&(f.charAt(++i)<'a' ||  
												f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
												tp.textSize = PSTricksConstants.COMMAND_TEXT_FOOTNOTE;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'i'://fi
											if(i+3>=lgth) throw new UnexceptedEndOfFileException();

											if(f.charAt(++i)=='l' && f.charAt(++i)=='l' && 
												(f.charAt(++i)<'a' || f.charAt(i)>'z') && 
												(f.charAt(i)<'A' || f.charAt(i)>'Z'))
												throw new NotYetManagedException("fill", nLine);//$NON-NLS-1$
												
											i = i2;
											throw new UnknownCommandException(nLine);
											
										default:
											i = i2;
											throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'h' :  // huge
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+4>=lgth) throw new UnexceptedEndOfFileException();

									i2 = i;
									tp = params.lastElement();
									if(f.charAt(++i)=='u' && f.charAt(++i)=='g' && f.charAt(++i)=='e' &&
									  (f.charAt(++i)<'a' ||  f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.textSize = PSTricksConstants.COMMAND_TEXT_HUGE1;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'm' : // m
									if(!started)   throw new IgnoreCommandException(nLine);
									
									if(i+1>=lgth) throw new UnexceptedEndOfFileException();
									i2 = i;
									
									switch(f.charAt(++i))
									{
										case 'a'://ma
											if(i+5>=lgth) throw new UnexceptedEndOfFileException();
		
											tp = params.lastElement();
											if(f.charAt(++i)=='g' && f.charAt(++i)=='e' &&//magenta
											   f.charAt(++i)=='n' && f.charAt(++i)=='t' && (f.charAt(++i)<'a' || 
											   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
												tp.textColor = Color.MAGENTA;
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										case 'o'://mo
											if(i+5>=lgth) throw new UnexceptedEndOfFileException();
											
											tp = params.lastElement();
											if(f.charAt(++i)=='v' && f.charAt(++i)=='e' &&
											   f.charAt(++i)=='t' && f.charAt(++i)=='o' && (f.charAt(++i)<'a' || 
											   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												int end = texParserActions.actionsOnMoveTo(nLine, 
																params, f.substring(i, f.length()));
												if(end!=PSTricksParserActionsManager.ERR_END_CMD_INDEX) i+=end;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
											
										default:
												i = i2;
												throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'y' : // yellow
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+6>=lgth) throw new UnexceptedEndOfFileException();
								
									i2 = i;
									tp = params.lastElement();
									if(f.charAt(++i)=='e' && f.charAt(++i)=='l' && f.charAt(++i)=='l' &&
									   f.charAt(++i)=='o' && f.charAt(++i)=='w' && (f.charAt(++i)<'a' || 
									   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.textColor = Color.YELLOW;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
								case 'i' : 
									i2 = i;
									if(!started)   throw new IgnoreCommandException(nLine);
									if(++i>=lgth) throw new UnexceptedEndOfFileException();
									
									switch(f.charAt(i))
									{
										case 't' : // it
											if(i+1>=lgth) throw new UnexceptedEndOfFileException();
										
											if((f.charAt(++i)<'a' || f.charAt(i)>'z') && 
											   (f.charAt(i)<'A' || f.charAt(i)>'Z'))
											{
												tp = params.lastElement();
												tp.textItalic = true;
											}
											else
											{
												i = i2;
												throw new UnknownCommandException(nLine);
											}
											break;
										
										case 'n' : // in
											if(i+14>=lgth) throw new UnexceptedEndOfFileException();
											
											if(f.charAt(++i)=='c' && f.charAt(++i)=='l' && f.charAt(++i)=='u' &&
												f.charAt(++i)=='d' && f.charAt(++i)=='e' && f.charAt(++i)=='g' &&
												f.charAt(++i)=='r' && f.charAt(++i)=='a' && f.charAt(++i)=='p' &&
												f.charAt(++i)=='h' && f.charAt(++i)=='i' && f.charAt(++i)=='c' &&
												f.charAt(++i)=='s' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
											   (f.charAt(i)<'A' || f.charAt(i)>'Z')) // includegraphics
											{
												int z = texParserActions.actionsOnPicture(nLine, params, f.substring(i));
												if(z>0) i+=z;
												throw new NotYetManagedException("includegraphics", nLine); //$NON-NLS-1$
											}
											i = i2;
											throw new UnknownCommandException(nLine);
											
										default :
											i = i2;
											throw new UnknownCommandException(nLine);
									}
									
								break;
									
								case 'w' :// white
									if(!started)   throw new IgnoreCommandException(nLine);
									if(i+5>=lgth) throw new UnexceptedEndOfFileException();

									i2 = i;
									tp = params.lastElement();
									if(f.charAt(++i)=='h' && f.charAt(++i)=='i' && f.charAt(++i)=='t' &&
									   f.charAt(++i)=='e' && (f.charAt(++i)<'a' || f.charAt(i)>'z') && 
									   (f.charAt(i)<'A' || f.charAt(i)>'Z'))
										tp.textColor = Color.WHITE;
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
									
									
								case 'z' : //z
									if(!started) throw new IgnoreCommandException(nLine);
									
									i2 = i;//zigzag
									if(i+6>=lgth) throw new UnexceptedEndOfFileException();
									if(f.charAt(++i)=='i' && f.charAt(++i)=='g' && f.charAt(++i)=='z' &&
									   f.charAt(++i)=='a' && f.charAt(++i)=='g' && (f.charAt(++i)<'a' || 
									   f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
										 throw new NotYetManagedException("zigzag", nLine);//$NON-NLS-1$
									
									i = i2;
									throw new UnknownCommandException(nLine);
									
								case 'o'://o
									if(!started) throw new IgnoreCommandException(nLine);
									
									i2 = i;
									if(i+6>=lgth) throw new UnexceptedEndOfFileException();
									if(f.charAt(++i)=='p' && f.charAt(++i)=='e' && f.charAt(++i)=='n' &&
									   f.charAt(++i)=='s' && f.charAt(++i)=='h' && f.charAt(++i)=='a' &&
									   f.charAt(++i)=='d' && f.charAt(++i)=='o' && f.charAt(++i)=='w' &&
									   (f.charAt(++i)<'a' || f.charAt(i)>'z') && (f.charAt(i)<'A' || f.charAt(i)>'Z'))
									{
										int end = texParserActions.parseOpenShadowCommand(params.lastElement(), nLine,
																	f.substring(i));
										if(end!=-1)
											i+=end;
									}
									else
									{
										i = i2;
										throw new UnknownCommandException(nLine);
									}
									break;
									
								default :
									if(f.charAt(i)==eol)
									{
										nLine++;
										i++;
									}
									else 
									{
										if(started)// throw new UnknownCommandException(nLine);
										{
											if(!PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
											{
												PSTricksParameters p = params.lastElement();
												String str = "\\" + String.valueOf(f.charAt(i)); //$NON-NLS-1$
												
												p.textForFrame += str;
												p.textParsed   += str;
												
												if(p.fromFramedBox)
												{
													boolean again = true;
													int k = params.size()-2;
													
													while(again && k>=0)
													{
														p = params.elementAt(k);
														p.textForFrame += str;
														
														if(p.getPosition()!=null)
															again = false;
														else 
															k--;
													}
												}
											}
											i++;
										}
										else throw new IgnoreCommandException(nLine);
									}
							} //switch(f.charAt(i))
							
							// Any space after a command must not be taken like a text.
							while(i<lgth && PSTricksParserActionsManager.isCharToIgnore(f.charAt(i))) i++;
							break;
						
							
						case '{' :
							if(started)
								params.add(new PSTricksParameters(params.lastElement()));
							
							if(isFramedBox)
							{
								params.lastElement().fromFramedBox = true;
								isFramedBox = false;
							}
							i++;
							break;
							
							
						case '}' :
							if(started)
							{
								if(params.size()<2)
									throw new TooManyBracketsException(nLine);
									
								PSTricksParameters tp = params.lastElement();
								
								if(tp.fromFramedBox)
								{
									PSTricksParameters pp = null;
									int k = params.size()-1;
									
									while(pp==null && k>=0)
										if(params.elementAt(k).getPosition()==null)
											k--;
										else
											pp = params.elementAt(k);
									
									if(pp==null)
										pp = params.firstElement();
									
									pp.psBox.lastElement().setEnd(pp.textForFrame.length());
									pp.psBox.add(0, pp.psBox.remove(pp.psBox.size()-1));
								}
								else
									if(tp.getPosition()!=null && !tp.psBox.isEmpty())
									{
										Vector<FramedBox> fbs = new Vector<FramedBox>();
										while(!tp.psBox.isEmpty())
											fbs.add(tp.psBox.remove(tp.psBox.size()-1));
										tp.psBox = fbs;
											
										tp.psBoxText.setText(tp.textForFrame);
										texParserActions.actionsOnTerminatedFramedBoxes(params);
									}
									else
										if(tp.psCustomP.fromPsCustom)
										{
											try
											{
												texParserActions.actionsOnterminatedPsCustom(params.lastElement());
												i++;
												if(params.isEmpty())//A enlever quand la commande sera totalement gérée.
													throw new TooManyBracketsException(nLine);
												
											}catch(NotFullyManagedException e)
											{//A enlever quand la commande sera totalement gérée.
												i++;
												if(params.isEmpty())
													throw new TooManyBracketsException(nLine);
												
												params.removeElementAt(params.size()-1);
												throw new NotFullyManagedException("pscustom", nLine);//$NON-NLS-1$
											}
										}
										else
											try {texParserActions.actionsOnText(params);}
											catch(InvalidFormatCommandException e) { /* Nothing to do.*/ }
								
								params.removeElementAt(params.size()-1);
							}
							i++;
							break;
							
						case ' ' :
						case 9 ://'\t'
							if(started)
							{
								PSTricksParameters p = params.lastElement();
								
								p.textForFrame+=" ";//$NON-NLS-1$
								p.textParsed+=" ";//$NON-NLS-1$
								
								if(p.fromFramedBox)
								{
									boolean again=true;
									int k = params.size()-2;
									
									while(again && k>=0)
									{
										p = params.elementAt(k);
										p.textForFrame+=" ";//$NON-NLS-1$
										
										if(p.getPosition()==null)
											k--;
										else
											again = false;
									}
								}
							}
							i++;
							break;
							
							
						default : 
							if(PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
							{
								if(started)
								{
									if(!lastTokenIsEOL && f.charAt(i)!=10)
									{
										if(params.isEmpty())
											throw new TooManyBracketsException(nLine);
										PSTricksParameters p = params.lastElement();
										p.textForFrame+=" ";//$NON-NLS-1$
										p.textParsed+=" ";//$NON-NLS-1$
									
										if(p.fromFramedBox)
										{
											boolean again = true;
											int k = params.size()-2;
											while(again && k>=0)
											{
												p = params.elementAt(k);
												p.textForFrame+=" ";//$NON-NLS-1$
												
												if(p.getPosition()!=null)
													again = false;
												else 
													k--;
											}
										}
										lastTokenIsEOL = true;
									}
								}
								i++;
								if(f.charAt(i)!=10) nLine++;
							}
							else
							{
								if(started)
								{
									PSTricksParameters p = params.lastElement();
									String str = String.valueOf(f.charAt(i));
									
									p.textForFrame+=str;
									p.textParsed+=str;
									if(p.fromFramedBox)
									{
										boolean again = true;
										int k = params.size()-2;
										while(again && k>=0)
										{
											p = params.elementAt(k);
											p.textForFrame+=str;
											
											if(p.getPosition()!=null)
												again = false;
											else 
												k--;
										}
									}
								}//if
								i++;
							}//else
							break;
					} // switch(f.charAt(i))
					
				}catch(InvalidFormatCommandException e)
				{	
					parserMsgs.add(new ParserCmdMessage(e.getNLine(), e, e.getCommand()));
					
				}catch(NotYetManagedException e)
				{
					parserMsgs.add(new ParserCmdMessage(e.getNLine(), e, e.getCommand()));
					
				}catch(UnknownCommandException e)
				{
					String cmd = String.valueOf(f.charAt(i));
					i++;
					while(i<lgth && (f.charAt(i)>='a' && f.charAt(i)<='z') || (f.charAt(i)>='A' && f.charAt(i)<='Z'))
						cmd+=f.charAt(i++);
					
					if(latexCommandsWithBrackets.containsValue(cmd))
					{
						PSTricksParameters p = params.lastElement();
						
						p.textParsed+="\\"+cmd; //$NON-NLS-1$
						
						while(i<lgth && PSTricksParserActionsManager.isCharToIgnore(f.charAt(i)))
							i++;
						
						if(f.charAt(i)=='{')
						{
							boolean despecialize = false;
							int cptBraces = 1;
							
							p.textParsed += f.charAt(i++);
							
							while(i<lgth && cptBraces>0)
							{
								if(!despecialize)
								{
									if(f.charAt(i)=='}')
										cptBraces--;
									else if(f.charAt(i)=='{')
											cptBraces++;
								}
								
								despecialize = f.charAt(i)!='\\' || despecialize ? false : true;
								p.textParsed += f.charAt(i++);
							}
						}
					}
					else
					{
						if(latexCommandsWithoutBrack.containsValue(cmd))
							params.lastElement().textParsed+="\\"+cmd; //$NON-NLS-1$
						else
						{
							e.printStackTrace();
							parserMsgs.add(new ParserCmdMessage(e.getNLine(), e, cmd));
						}
					}
				}
				}catch(IgnoreCommandException e)
				{
					while(i<lgth && (f.charAt(i)>='a' && f.charAt(i)<='z') || (f.charAt(i)>='A' && f.charAt(i)<='Z'))
						i++;
				}
			}//while(i<lgth)
			
			if(params.size()>1) throw new UnclosedBracketsException();
			
			if(started && !parsingFinished)
				throw new FatalErrorParsingException(LaTeXDrawLang.getString1_9("PSTricksParser.3")); //$NON-NLS-1$
			
			if(progressB!=null)
				progressB.setValue(progressB.getMaximum());
			
			if(!parserMsgs.isEmpty())
				ParserMessagesFrame.showParserMessagesFrame(parent, true, parserMsgs);
			
			return OK_PARSING;
			
		}catch(FatalErrorParsingException e)
		{
			String title = LaTeXDrawLang.getString1_6("PSTricksParser.3"); //$NON-NLS-1$
			
			if(parent!=null)
				JOptionPane.showMessageDialog(parent, e.toString(), title, JOptionPane.ERROR_MESSAGE); 
			else 
				System.out.println(title+" "+e.toString()); //$NON-NLS-1$
			
			return ERROR_PARSING;
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
			return ERROR_PARSING;
		}
	} //public int parse(String f, JFrame parent)
	
	
	
	
	
	/**
	 * Allows to know if the code contains the tag : \begin{pspicture} or \pspicture
	 * @param code The code to analyse
	 * @return 1, if the code contains the tag \begin{pspicture}<br>
	 * 2, if the code contains the tag \pspicture<br>
	 * 0 else.
	 */
	public static int containsBeginPSPicture(String code)
	{
		char eol = System.getProperty("line.separator").charAt(0);//$NON-NLS-1$
		char[] f2 = new char[code.length()];
		int i=0, lgth = code.length(), cpt=0;
		boolean zero = false;
		
		while(i<lgth)
		{
			if(i<lgth && (code.charAt(i)==' ' || code.charAt(i)==9))
			{
				if(!zero) 
				{
					f2[cpt++] = ' ';
					zero=true;
				}
			}
			else
			{
				if(code.charAt(i)=='%') 
				{
					if(i>0 && code.charAt(i-1)!='\\')
					{
						while(i<lgth && eol!=code.charAt(i))
							i++;
						
						if(i<lgth) f2[cpt++] = code.charAt(i);// We copy the eol
					}
					else if(cpt>0)cpt--;
				}
				else 
				{
					f2[cpt++] = code.charAt(i);
					zero = false;
				}
			}
			i++;
		} // while(i<lgth)
		
		String code2 = String.valueOf(f2, 0, cpt) + ' ';//We terminate the code by a space
		
		if(code2.contains("\\pspicture"))//$NON-NLS-1$
			return 2;
		
		String begin = "\\begin";//$NON-NLS-1$
		lgth = begin.length();
		boolean end = false;
		int start=0;
		
		while(!end)
		{
			int[] id = {code2.indexOf(begin, start)};
			
			if(id[0]==-1) end = true;
			else
				try
				{
					id[0]+=begin.length();
					PSTricksParserActionsManager.goToNextChar(code2, id);
					start = id[0];
					
					if(code2.charAt(id[0])!='{')
						continue;
		
					id[0]++;
					PSTricksParserActionsManager.goToNextChar(code2, id);
					
					if(code2.charAt(id[0])!='p' || code2.charAt(++id[0])!='s' || code2.charAt(++id[0])!='p' || 
						code2.charAt(++id[0])!='i' || code2.charAt(++id[0])!='c' || code2.charAt(++id[0])!='t' || 
						code2.charAt(++id[0])!='u' || code2.charAt(++id[0])!='r' || code2.charAt(++id[0])!='e')
						start = id[0]+lgth;
					else
					{
						id[0]++;
						PSTricksParserActionsManager.goToNextChar(code2, id);
						
						if(code2.charAt(id[0])=='}')
							return 1;
						return 0;
					}
				}catch (IgnoreCommandException e) { return 0; }
		}
		return 0;
	}
	
	
	
	/**
	 * Allows to test if the parsing if finished or stopped
	 * @throws InterruptedException If the parsing is stopped or finished
	 */
	public synchronized void testEndThread() throws InterruptedException 
	{
        if(stopThread) throw new InterruptedException();
	} 

	
	
	/**
	 * Allows to stop the parsing
	 */
	public synchronized void stop() 
	{
        stopThread = true;
	}
}
