/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (SM2 precomputed)
//
//
*/
#include "owncp.h"
#include "gfpec/pcpgfpecstuff.h"


#define OPERAND_BITSIZE (256)
#define LEN_P256        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* SM2 affine point */
typedef struct{
   BNU_CHUNK_T X[LEN_P256];
   BNU_CHUNK_T Y[LEN_P256];
} SM2_POINT_AFFINE;

extern const __ALIGN64 SM2_POINT_AFFINE precomputed_ec_sm2[37][64];


#if defined( _IPP_DATA )

#if !defined(_DISABLE_ECP_SM2_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

const __ALIGN64 SM2_POINT_AFFINE precomputed_ec_sm2[37][64] = {
/* digit=0 base_pwr=2^0 */
{
   {{LL(0xf418029e,0x61328990),LL(0xdca6c050,0x3e7981ed),LL(0xac24c3c3,0xd6a1ed99),LL(0xe1c13b05,0x91167a5e)}, {LL(0x3c2d0ddd,0xc1354e59),LL(0x8d3295fa,0xc1f5e578),LL(0x6e2a48f8,0x8d4cfb06),LL(0x81d735bd,0x63cd65d4)}},
   {{LL(0xbc3be46a,0x0af037bf),LL(0x2d8fa938,0x83bdc9ba),LL(0x5788cd24,0x5349d94b),LL(0xcaa5736a,0x0d7e9c18)}, {LL(0x69db9ac1,0x6a7e1a1d),LL(0xc4a8e82b,0xccbd8d37),LL(0x9b7157ac,0xc7b14516),LL(0x6c21bdf5,0x947e7465)}},
   {{LL(0xab589e4a,0x1cda54fd),LL(0xdb4f0a0d,0x26765289),LL(0x8ceb4a0a,0x0a265a30),LL(0xfe887c64,0x3019fd6b)}, {LL(0x4b2fc190,0x0a10fbe9),LL(0x87cbce60,0xf40aa52b),LL(0x6dc13c97,0xcc496bfa),LL(0x5bb3fbb4,0x28ad3478)}},
   {{LL(0x98615060,0x393f7c5a),LL(0xe9016209,0x487ea27f),LL(0xa09f9020,0x8a86bcb4),LL(0xc899dbe1,0x50dc8e3a)}, {LL(0xfd619998,0xfc099043),LL(0x7c7383bd,0x1de135ea),LL(0x32cf70ed,0x4d0bd556),LL(0x25bce9e3,0x6ffc31c5)}},
   {{LL(0x6a9c8162,0x9a575633),LL(0x21dfcc53,0x15aa58f2),LL(0x1ef5f4c5,0x7ad354bf),LL(0x63f875b9,0x0f443ef3)}, {LL(0xd3450133,0x2e81d68f),LL(0xe3607d18,0xb30f4bbd),LL(0x362258ef,0xb1826a4c),LL(0x142a6768,0x7b415276)}},
   {{LL(0x0acd72ba,0x136a9c4c),LL(0x5e7ec73c,0xb1274a25),LL(0x5de34db6,0xf15a876e),LL(0x8cba8047,0x85e74ca0)}, {LL(0xb469eb37,0x08454cdd),LL(0xc99754f8,0x8fbf6d1f),LL(0xec30e984,0x1060e7f8),LL(0x4b8c598a,0xb568bc97)}},
   {{LL(0x81f06784,0xaa3531c7),LL(0x07132520,0x0b894193),LL(0xacfe18c5,0x84ee5b69),LL(0xd9fbec28,0xbbf492e0)}, {LL(0xe5f6186d,0x313a35c1),LL(0x757a01b8,0x0e449a2e),LL(0x2bd99baf,0x96c9b992),LL(0x3b84d777,0x2ba05a8f)}},
   {{LL(0x09122670,0xde523a1c),LL(0x22cc810c,0x90be6f2a),LL(0x4387df9e,0x086e6341),LL(0xd9c44134,0x115c2fc0)}, {LL(0x8799302a,0x9334430d),LL(0xe27b7ea4,0x693b3500),LL(0x9a8f3382,0xcbe1136f),LL(0xb5778247,0xe77fd5f2)}},
   {{LL(0x30fbde86,0x98e795c3),LL(0xab21af8f,0x8e5e0495),LL(0xb48669b4,0x3925bf83),LL(0x469522c8,0x77d88740)}, {LL(0x987b04ce,0x8fbf8b5b),LL(0x3aff4428,0x63c563a8),LL(0x53a6e969,0x5dc11165),LL(0x32697f4c,0x822a6c24)}},
   {{LL(0x642cb143,0x6774298a),LL(0x2d110e71,0xecdb60d8),LL(0x1388728e,0xe810b11b),LL(0xd8603a8a,0x2e8237d8)}, {LL(0x50aeeae1,0x673968fc),LL(0x6746a3f4,0x08c65d19),LL(0xd7dd7165,0x7a61a6b5),LL(0xa9b6df3a,0xe31bbfd9)}},
   {{LL(0x3421e115,0x2b252ad0),LL(0xc6affc01,0x7557c8c7),LL(0x8a509267,0xd90c19fd),LL(0xe0d871c8,0x483da168)}, {LL(0xc10729bf,0x72d6f9b3),LL(0x15b7061e,0x5dd84021),LL(0x9f2c587d,0x9bfea2db),LL(0x98641ec2,0x528398a7)}},
   {{LL(0xf3afdd62,0x18a65d8d),LL(0xd6d7e4e4,0x89f38500),LL(0x9d8d4f07,0x65708c6a),LL(0xd0bdc7f4,0xb90ea13c)}, {LL(0x8c3e2b32,0x58985855),LL(0xbcfad3a1,0xfa48d5c5),LL(0x62385ffa,0x5c3544e7),LL(0x7e72aeb7,0xb6bd39ed)}},
   {{LL(0x8000fe4e,0x34e51c6a),LL(0x89c46941,0x7da2bdfd),LL(0xe1bc2b2e,0x667ba91d),LL(0x10a73e5c,0x3c80c9d0)}, {LL(0xc7f5c64d,0x4fadebbe),LL(0x3ea35052,0xaef09eb4),LL(0x26ec55f9,0x167ee11b),LL(0x85189260,0x45fa508a)}},
   {{LL(0x22542fc3,0xa0e9a439),LL(0xddac78dc,0x3f194a6c),LL(0x6f74d053,0xa75ae72d),LL(0x097c6617,0x0f8babeb)}, {LL(0x4303c247,0x1d12bc5c),LL(0xbd1e246c,0xfe0c027a),LL(0xb69b55ad,0xe9ca1a99),LL(0x117cd63a,0xff6cd2b0)}},
   {{LL(0xdde97d4d,0xf3489343),LL(0xbbb2ce1f,0x9c14e38a),LL(0xcfddf221,0x25866911),LL(0x460efef1,0x0df89411)}, {LL(0x73ae8326,0xf713f30e),LL(0xcdd274a1,0xd9be66a8),LL(0x36885947,0xdf915ae2),LL(0x7878b781,0x2c5c1e9e)}},
   {{LL(0x39e8a120,0xf71560c9),LL(0x7273b59a,0x7121d6b8),LL(0x8ef4639d,0x649535ce),LL(0x14cc6d58,0xcd01076e)}, {LL(0x96e74f8a,0x2705729a),LL(0x5533037e,0xb07e3230),LL(0x663c5c62,0x0846dcc1),LL(0x10fc3ac1,0x6a4759c1)}},
   {{LL(0xcfbdfeff,0x3c126193),LL(0x4996d845,0x4a31dd20),LL(0x19f2b658,0x48a76ba0),LL(0x8890a8bc,0xbe330142)}, {LL(0x308aa041,0x287b34e1),LL(0x813adf29,0xcbf5da24),LL(0xcdcdc439,0xcdfc5a58),LL(0x198a6075,0xbda3bda2)}},
   {{LL(0x1497fe38,0x639f92bc),LL(0xd58bd278,0x8ed8eeac),LL(0xb417bfe4,0xcf5d7ce6),LL(0x44400c59,0xf617c54e)}, {LL(0x7d8dc939,0xde635635),LL(0x241baaff,0x2e6a3a75),LL(0xe07e8e97,0x02f324e5),LL(0x70f9fc9d,0xeb715487)}},
   {{LL(0x86712116,0xbefd3380),LL(0x884efe46,0x9b9e9707),LL(0x8c9e513f,0x611a1eec),LL(0x3b6dbcec,0xe2d8e3f5)}, {LL(0x4f8964e4,0x7cedab1c),LL(0xf4e139f8,0xee12d062),LL(0x9a9af4f3,0x8e63c9c0),LL(0x8b907b23,0xe3246dbb)}},
   {{LL(0x71099001,0x70d5bda2),LL(0x15fae7dd,0x3d876d4a),LL(0x7b69c20e,0xaba0500f),LL(0x9834adf2,0xa8e3e094)}, {LL(0x980b21b9,0x69db851b),LL(0x788c2a30,0x274c1de2),LL(0xd47d153d,0x5caa5336),LL(0x57cef318,0xada69877)}},
   {{LL(0xa0551c80,0x83879486),LL(0x658e61be,0x1611dea0),LL(0x1b935068,0x1fe95c82),LL(0x5b229223,0x8f01e019)}, {LL(0x7e93c389,0x23017e05),LL(0x9840dd64,0xce4ac99d),LL(0x1de86399,0xddc9b900),LL(0x88015785,0x6abe5cc3)}},
   {{LL(0xb3c50898,0xc09545a9),LL(0x6c05d902,0xbd443361),LL(0x2c6bcc8c,0xed71f70c),LL(0xbdf8e908,0x8dbc0b88)}, {LL(0x4fcbcd9a,0x56eb5b98),LL(0x08114397,0xafb6fedc),LL(0xb35f7927,0x0500ce5b),LL(0x95efe710,0x7005bcf9)}},
   {{LL(0x2eba7f39,0x125cbed2),LL(0x6c488d44,0xc7c42e76),LL(0x676915c4,0xdb8991f9),LL(0x9183839f,0xdf6ae594)}, {LL(0xc79f8bd1,0x4f69c304),LL(0xaa1662fa,0x638cb070),LL(0xba6f2599,0xc7f68c72),LL(0x1f6edfa9,0x11bb84d9)}},
   {{LL(0xa215fda2,0x9ed156ec),LL(0x20c5ddb6,0x19de7a91),LL(0x0668c65d,0xc1ed949d),LL(0xd0826f6a,0x96683044)}, {LL(0x1adaa8ff,0x1e6325e0),LL(0x07ac392a,0xbc53bc24),LL(0xd9f06e44,0x2c342db5),LL(0x30db8c1a,0x3f529385)}},
   {{LL(0xe7492326,0xc5957d29),LL(0x0663f829,0x3addc3df),LL(0x728cfdc1,0x8faa3169),LL(0x6b975134,0xde53aa7c)}, {LL(0xefddc764,0xf481759b),LL(0x09edaff3,0xd605474b),LL(0x653d48c9,0xc7df1eb9),LL(0xc5040212,0xa71e6854)}},
   {{LL(0xafe945b5,0x136d8342),LL(0xe9d239c7,0x91707e7d),LL(0xfb2e80de,0xeda23dc5),LL(0xff614966,0x892bed73)}, {LL(0x838dc12d,0x2ded2367),LL(0xb002bd9c,0x73fd298c),LL(0x2c4629df,0xc548b426),LL(0x8f7e03b7,0x93605d17)}},
   {{LL(0xd37c24cc,0x32861816),LL(0xe427975a,0x5bb54ee2),LL(0x32f943a9,0x6da013d2),LL(0x9bc202e5,0x0746a77a)}, {LL(0xcd1def5b,0x6db07a84),LL(0x861d9f9b,0x9421fe7f),LL(0x692181fb,0x71767292),LL(0xc9d2441d,0x0560e7e5)}},
   {{LL(0x4d7e922a,0xf1496afd),LL(0xe11fa533,0x67f42a3f),LL(0x977956cd,0x9f903e5b),LL(0x1eb49608,0x37671e24)}, {LL(0x21fb2047,0x967950a0),LL(0x35da3c6b,0x141f96fb),LL(0xd27bba59,0xe07c3c40),LL(0x0e1af754,0xbde5ed1d)}},
   {{LL(0x54f1f257,0xdc64c4b0),LL(0xb01196dc,0xecb033c8),LL(0x8202d5bd,0x54e65f4d),LL(0x2b2fd451,0x63afcc93)}, {LL(0x30640fb7,0x1e929a39),LL(0x5b361718,0xdc91387e),LL(0xf8f0bbe8,0x10aadecb),LL(0x0977e2bb,0x81d8f466)}},
   {{LL(0xbd64cd96,0xdcaa3790),LL(0xcee698d3,0xbc8ac152),LL(0xa1143c45,0xde7192f7),LL(0xf5fb9ea0,0xf7c9d826)}, {LL(0xc9468f50,0x54aea92e),LL(0xcc427ed4,0x340f4459),LL(0x02ad5467,0x3fec5be9),LL(0x2cc6c8b5,0xec780d9c)}},
   {{LL(0xb889c78a,0x7b179a8b),LL(0x0aca32c5,0x069a7ab9),LL(0x591b9a36,0xe4e5215e),LL(0x3bd54630,0x7802fb3e)}, {LL(0x233c6eeb,0x9a479313),LL(0x4e1cbabc,0x18c612ad),LL(0xc0e36f3b,0x28a29273),LL(0x7d3deb26,0xf4e2dfb1)}},
   {{LL(0xadbb3c8b,0xa6c11369),LL(0x4c8ec378,0xd78af40b),LL(0x03f0a982,0xffb3a80d),LL(0xa83be50a,0x550e3e71)}, {LL(0x418ee45b,0x845c0fb2),LL(0x0791b964,0x5297cf43),LL(0xcc47e33b,0x676b638c),LL(0xfecf85b2,0xb1c52fac)}},
   {{LL(0x3dba2c0e,0xf011b5e5),LL(0x026d4f11,0xa6c68448),LL(0xc3f206fb,0x11596db3),LL(0x29414a3c,0xc91c76dc)}, {LL(0xb94ddc7c,0x1839b9d1),LL(0x56ae8610,0xdfb20ce7),LL(0xd8734400,0x3e2b1cd9),LL(0xf01ea540,0x59f9329a)}},
   {{LL(0x2351a2a9,0x7d4c140c),LL(0xbf4c9823,0x575c1e1b),LL(0x31068df9,0x8f11c2ca),LL(0x05e6def0,0xe3c17aa0)}, {LL(0x501c8630,0xe6281c70),LL(0xc88a412e,0xad240917),LL(0x390492d7,0x6f21bfb7),LL(0xc3a3ccb7,0x61ea1385)}},
   {{LL(0x33733cbc,0x60494a83),LL(0x27ed8157,0x8da622a0),LL(0x0471ad90,0x0022b154),LL(0xd3568003,0x3bd0a4c5)}, {LL(0xd932df23,0xdc8e2d03),LL(0x7a1f5159,0x859ed940),LL(0x2a375b0f,0xad670e63),LL(0x9520db97,0x15922fae)}},
   {{LL(0x59eb1a9b,0xfb73d16f),LL(0x8511e541,0x3ee8cc1f),LL(0x1590c321,0x20d72d59),LL(0x3bd075d4,0x62eab566)}, {LL(0xfae123ab,0xac07a7c7),LL(0x1f10af6e,0x83b89abf),LL(0x1da8ac5d,0x469962ec),LL(0x8c58c3b3,0x09761c35)}},
   {{LL(0x7da90fc9,0x2c086d5e),LL(0x5cc27782,0x458e5ffd),LL(0xb9268939,0xc3f48611),LL(0xde4b9110,0x39fed873)}, {LL(0xfda698cc,0x16ef8f78),LL(0xa973bb50,0xb028dc21),LL(0xe29b725b,0x45eb849e),LL(0x14c6eae9,0xd41b5b6d)}},
   {{LL(0xc55d5720,0x5e931b21),LL(0xa0e40b19,0xb628ccb2),LL(0x000651a5,0x42044ffe),LL(0x076544e7,0x2130b4de)}, {LL(0x3677c70f,0x38428594),LL(0xf8945d86,0xfdcdb038),LL(0x4169ae44,0xfb2e3d4c),LL(0x0d13bce2,0xd4695e9b)}},
   {{LL(0x039d646d,0x45191390),LL(0xb12ba339,0x983b7a2e),LL(0x5923e7d6,0xdfd30d3e),LL(0xba9d206a,0xae3590f0)}, {LL(0xb6d5e62a,0x7d58d334),LL(0x7e402b12,0xb15b0544),LL(0x62ae8e01,0xac57e113),LL(0xf473edee,0x4d83804c)}},
   {{LL(0xc81bc828,0x2faa7c4d),LL(0xfb62561f,0xb16ed9d7),LL(0x49c2fa93,0x4c9da270),LL(0xb311d90d,0x3b014c73)}, {LL(0xf5443332,0xd29c5d65),LL(0xeebdb7c2,0xb6457d54),LL(0x4cce9480,0xc6a0bf3a),LL(0x85355854,0xd434a3b0)}},
   {{LL(0x8b2c703c,0x178ca01b),LL(0x0ab71a51,0x605bba53),LL(0x3db948d5,0x2140948e),LL(0x5fb6b8c1,0xc45b2689)}, {LL(0xf17b47bd,0x421f66de),LL(0x2e9b3ee5,0x57627a5a),LL(0x66614339,0xedf3920a),LL(0x4b638a46,0x7ea61903)}},
   {{LL(0x3c030643,0x7203d542),LL(0x5e631461,0x7112bb3d),LL(0x2bc3da9c,0x2604eac7),LL(0x32d2541b,0x2e4964e7)}, {LL(0xe8b6482a,0x940faf46),LL(0x24d27c9e,0x8f772fcb),LL(0xca7c5f88,0x125c34d7),LL(0xd1f47795,0x9903eadb)}},
   {{LL(0xe2147129,0x11aaa417),LL(0xf88a0a30,0x3ccef5c2),LL(0x90283f97,0x78d5207a),LL(0xd25226b6,0xba1261e9)}, {LL(0xd1e7a01c,0xbfc79248),LL(0x941ab2bd,0x373f1cd5),LL(0x19a0668b,0xf0881e21),LL(0x1f77bf0a,0x7b793789)}},
   {{LL(0x63d4523d,0x49c2769b),LL(0xf0399eaf,0xf8df2cba),LL(0x22a2a74d,0x5ae94c69),LL(0xefd1e193,0xd08f8d45)}, {LL(0xc681f376,0x64341fc4),LL(0xec918711,0x3a8e25c8),LL(0x0608f50f,0xdf35304d),LL(0x9a973742,0x9b4c6967)}},
   {{LL(0xbfba043b,0xb5c1f5d3),LL(0xe975f03b,0xaff4f896),LL(0xae2cbb01,0xea1f39bd),LL(0xa62915ff,0x4cc1c4cb)}, {LL(0x89e943b8,0x5eb4afa3),LL(0x154e565a,0x8c4d27e5),LL(0x7f2bced6,0x4e2e5a7e),LL(0x4487f6a3,0x7af408e2)}},
   {{LL(0x97a60de7,0xe5dacbae),LL(0x4401b0ad,0x9774834c),LL(0x8a9113f9,0x7683bb00),LL(0x42b2ba67,0xc6fe7e8b)}, {LL(0x54e760c8,0xc0c0564d),LL(0x118606c2,0xf7b05401),LL(0xec3cd7b9,0x554a9b0f),LL(0x27916a21,0xce75ecfb)}},
   {{LL(0x12118abd,0xf6638997),LL(0x097da3a7,0x2ba6e754),LL(0x0fdf9985,0x1df82085),LL(0x546c864a,0xbf73502a)}, {LL(0xc02d9ce0,0xdfde9323),LL(0xe4dd0e7d,0x580491e2),LL(0xae43b9b4,0xe71522d2),LL(0x6a231a41,0x876e3627)}},
   {{LL(0xb36362ec,0xfa8ff511),LL(0x34085959,0x11c5a9f6),LL(0x9770c62b,0x272b86f2),LL(0x7c7e8827,0xf0626225)}, {LL(0xea1e13eb,0x929168bf),LL(0xce59b0f5,0xdb892971),LL(0x4f826f34,0x6769e31d),LL(0x0a955cec,0xfa1dd934)}},
   {{LL(0xa294d7ea,0x123d9ca2),LL(0x4492569b,0x8699063b),LL(0xa8dd86c3,0x6a50eae9),LL(0x12c06c38,0x3d757d10)}, {LL(0x3e41e556,0x5a92c2c0),LL(0x6330c21a,0xa64595eb),LL(0xe184d925,0x70d8141a),LL(0xa2f10304,0x8543f2ce)}},
   {{LL(0x9eaca504,0x4559b0a2),LL(0x2617bc9b,0xb9843a4b),LL(0x1b641003,0x5b28d4ee),LL(0x4ced538a,0x3e9af8e1)}, {LL(0x7bdf7dc2,0x3790fe89),LL(0xc32549ee,0xc7c74941),LL(0xabcd2f42,0xdcc8295b),LL(0xead078b6,0x48b29a4f)}},
   {{LL(0x2040178e,0x8e8b28e3),LL(0x971725fc,0xceff8f3e),LL(0xfcee2cc1,0x4a97b6fa),LL(0xbac85b56,0x775df6a9)}, {LL(0xd28a21cc,0x32e5cbe6),LL(0xae2b82db,0xe8b86ada),LL(0x86e38e96,0x44dfbb50),LL(0x1afc2d4b,0x45d3fe7d)}},
   {{LL(0xd23f620d,0x838b356e),LL(0x4592fe4b,0x2e8fa8ac),LL(0x3af5b1d8,0x1396e1b3),LL(0xcbf50fb0,0x9c0c2ef3)}, {LL(0x836e93e9,0xd9efb6c9),LL(0x0899163f,0xe6eb5870),LL(0xdca00d1b,0x3a2f6d77),LL(0xb40ba0d6,0x36f55f89)}},
   {{LL(0x32866e57,0xf3b1701f),LL(0x59de0f2e,0xf0768473),LL(0xab57962d,0xe55d7aed),LL(0x2b60cabb,0x45004985)}, {LL(0xd5498888,0x8d539d6e),LL(0xa5e0ff6a,0x176ce1a0),LL(0xdc088c50,0xcb7c15ef),LL(0xc9a9ae2f,0x90393d7a)}},
   {{LL(0xd396bdce,0xd9c1a140),LL(0x6fb2800f,0x4215b78b),LL(0x2f76b0df,0x8939109f),LL(0x2adb40a8,0x0f250897)}, {LL(0x3a86e009,0x4db0007c),LL(0xf968a635,0x6ef0ad95),LL(0x8eaefa78,0x58a82d4b),LL(0x493604a4,0xe8a181cb)}},
   {{LL(0x520d216d,0x36c84e34),LL(0xc666171c,0x2b2ef6b5),LL(0x2ce29d37,0x9469b91f),LL(0xc15f20aa,0x3ecd84e7)}, {LL(0x292edd2c,0xf1090635),LL(0x7c3447f6,0x6d439362),LL(0x3eea3fdf,0x51b9a0a9),LL(0x9e57e450,0x68e0d1f8)}},
   {{LL(0x7380931e,0x25d249d5),LL(0x2011a45b,0x87f03fad),LL(0xefde1ca3,0x89df0324),LL(0x9a9b4330,0x52ae43cd)}, {LL(0xa1867c1b,0xfe48bc64),LL(0x9866920e,0xdd874f66),LL(0xfcf50251,0x6942a7e4),LL(0x9c5f6298,0xf5c10048)}},
   {{LL(0x00973d66,0x305183eb),LL(0x95baf07c,0x1ce66760),LL(0x74822e13,0x74c9d971),LL(0x76b5e6ef,0x2ccd7fbb)}, {LL(0xa3e1ca18,0x51688b49),LL(0xa603f2f1,0x1beb5bbb),LL(0x962534b6,0x09a231d1),LL(0xafa92f75,0x70417ce1)}},
   {{LL(0xe154bc00,0xb86b6d82),LL(0x895483e5,0x5a0b19e8),LL(0xa0ff1e44,0xb15f6c05),LL(0xfdd8615d,0x2938b88a)}, {LL(0x971615c3,0x81800a05),LL(0xc03d2039,0x6be6d56b),LL(0xc476ce64,0xff3e57d2),LL(0x6f583ee8,0x5b509b7b)}},
   {{LL(0x7c1f5d3b,0x1d92c36c),LL(0xe11df757,0x1e60b19b),LL(0xe37e36f6,0x20261501),LL(0x29bc86e3,0xb68a9aaa)}, {LL(0xf61d23ca,0xfba81eaa),LL(0xd5adaa18,0x63440834),LL(0xa5f93bb8,0xa80d76ed),LL(0x5a728480,0x3264283d)}},
   {{LL(0xe4b8c48e,0xd6171111),LL(0xde557cca,0x3ee227a1),LL(0x3cb59841,0x2bebc09a),LL(0x99bf6205,0x2f8047fe)}, {LL(0x4c43845f,0xb78b243e),LL(0x46d3b5e0,0x484ac183),LL(0x0314524d,0xa07be476),LL(0x1ab4c447,0xc0a3aa35)}},
   {{LL(0x9c341f84,0x2f302d58),LL(0x84f130ba,0x264911a7),LL(0x3ee64343,0x30bed408),LL(0x5dc5868a,0xd7d6e92d)}, {LL(0x80adb3fb,0x92074568),LL(0xa133123e,0x005ab33c),LL(0x42e1da50,0x105119fd),LL(0xb7f6b1e8,0x6987117d)}},
   {{LL(0xc2bccb7a,0xa2315af3),LL(0x8672c98a,0x95ddd3ee),LL(0x5f48f607,0xa9032645),LL(0xc5273603,0x76861e62)}, {LL(0x88817217,0x71aaa35f),LL(0x2892afac,0x57e95b6c),LL(0x9e84c791,0xf65e909b),LL(0xaa52f3b1,0x257bcc2d)}},
   {{LL(0x865c665a,0xd5f6110a),LL(0x30c08b4c,0xddc3afe1),LL(0xefec26fc,0x4df3d04a),LL(0xb035af5d,0xf229bddf)}, {LL(0xd191b439,0x364913cf),LL(0x5a7fa8a4,0xf41b8f6d),LL(0x6f6c1219,0x677cc51b),LL(0x148b7f64,0x593afe4a)}},
   {{LL(0x0d038ad4,0x80ffa5ae),LL(0x36256c8f,0xf44d3df3),LL(0xbc978dce,0x0a3077c8),LL(0x745b8317,0xf3d9b4b0)}, {LL(0xb6b1852c,0x8bbf4484),LL(0x0e78ff07,0x0cd02ed4),LL(0x49c24238,0x91cb827e),LL(0xdaa3cb55,0x58adaee5)}},
},
/* digit=1 base_pwr=2^7 */
{
   {{LL(0x033fc12a,0x07e6ce4d),LL(0x4886f316,0xba4f98a1),LL(0xe66f3f11,0xb24b38f3),LL(0x5ea4bde3,0xe3f6205a)}, {LL(0xa77b998f,0x00705387),LL(0x9549f3b1,0x2c9b4457),LL(0x533a61d6,0xdef6625b),LL(0x7e4f781a,0x4eda7e2a)}},
   {{LL(0xfd120134,0xe9730aaa),LL(0xc057309c,0xb22b9089),LL(0x84726ce7,0x98e79565),LL(0xd635a584,0x0e1431a0)}, {LL(0xe834ffa6,0xbd387023),LL(0x036ab1ae,0x64198ddf),LL(0x9124b684,0x46e5ebb1),LL(0x233b3c6d,0xa316fa44)}},
   {{LL(0x84782513,0xec2a9325),LL(0x2903d20b,0xd67c8ab7),LL(0x157f9aee,0x6b65b262),LL(0x69f964a2,0x547be60c)}, {LL(0xee0419db,0x001bf327),LL(0xf20c7005,0x92fa0800),LL(0xcdc1ccda,0x1e11e745),LL(0xe471f822,0xa785ec10)}},
   {{LL(0xa1371aa4,0xbc970210),LL(0x54b5424e,0xaff481a0),LL(0x0e64269b,0xbcdf91fd),LL(0xb02fc7cf,0x18bb37bb)}, {LL(0x6f69d439,0xd99edd79),LL(0x169514b2,0x4e27a58f),LL(0x66e19ae4,0x80eca1ca),LL(0x0788696d,0x0470e965)}},
   {{LL(0x8c9d34f6,0xa6b1992f),LL(0xa5ed969e,0xaf062ffe),LL(0x3a6d7ae2,0xbca2580d),LL(0xc8999158,0xf30cd9e6)}, {LL(0x49d1ab0d,0x93e57897),LL(0x30214280,0xcfa3aa4d),LL(0x0a814831,0x0ca8b4fd),LL(0x0b10097c,0xdad179db)}},
   {{LL(0x3dfdc228,0x63778bfc),LL(0xb9648a36,0xc0bae0ad),LL(0x015a99b5,0xda8cb8ab),LL(0x8366b58a,0xb045cccb)}, {LL(0x4164cebd,0x74ef8ef4),LL(0xc5e00e5f,0x41e71fc8),LL(0x4479468e,0x753cf906),LL(0x332ea72d,0x78b5223f)}},
   {{LL(0xddebafa2,0x8fc3e370),LL(0x351f9f04,0x15ffcce0),LL(0x45b0efdc,0x3fbd5f5c),LL(0x0fe3b460,0xb8216623)}, {LL(0x533c7db6,0xe8322fbd),LL(0x00a243ff,0xf3866d15),LL(0xa0e5aaea,0xf1194ae2),LL(0xb9287b3d,0x3e93eb01)}},
   {{LL(0x5876d6e8,0x528a9e2f),LL(0xd2b622d7,0x93c48f85),LL(0x3e5411d7,0x88d9eac8),LL(0x00a70e91,0xb7e4a6ba)}, {LL(0xf1c43b2e,0xaf18e5ba),LL(0xa2f347de,0x46578c7e),LL(0xf009638f,0x19ca736d),LL(0xbd1acd29,0xa6563f1e)}},
   {{LL(0x2f4126e7,0xdf9bcd3a),LL(0xd62efebd,0xecc22d13),LL(0x10943242,0xd9b29b4b),LL(0x670136f9,0x499ffa74)}, {LL(0x2b889952,0xa2a9ad2c),LL(0xfd132dad,0x945f306e),LL(0x15cebfd7,0xfd05b884),LL(0xc7a5627f,0x653e70af)}},
   {{LL(0x577dae35,0xfefc54b5),LL(0xaac3a655,0x9d2f0546),LL(0xfac31d00,0xb96bd298),LL(0xee69563d,0x3328a51c)}, {LL(0x43195f4e,0x5e19098e),LL(0xa998010b,0x657f4ba5),LL(0x4047ccb6,0x45f29a84),LL(0x6e423da6,0x833a5443)}},
   {{LL(0xca33f42b,0x97e480c6),LL(0x06e52a05,0x20a51033),LL(0x0a9be572,0x85e87255),LL(0xb988b582,0xe8bc857a)}, {LL(0xc183c500,0x782495e8),LL(0xfee0ae2f,0xf33a87fd),LL(0xc64d1dec,0xf069fe20),LL(0xf4b08816,0x0b6dd98a)}},
   {{LL(0x99229a90,0x6e6cf298),LL(0x1d71d532,0xa6840bc8),LL(0x71e3a8b7,0x803e5407),LL(0x6afd9a0e,0xd5611ee4)}, {LL(0xbbbefa73,0xd739ca0e),LL(0xc5ec48b7,0x6082dbab),LL(0xbbdea0ec,0xa0ab10df),LL(0xf1633e03,0xb1b7ebe4)}},
   {{LL(0x7be26441,0xfa752496),LL(0x0ef683e6,0xf52cb1b6),LL(0x39dd611d,0x1c96401f),LL(0x7bb19083,0x09c5a35b)}, {LL(0x00a5d5a1,0xa2f002b8),LL(0xacf4e8ed,0x4e300ddd),LL(0xb4cc58c6,0x0d26b600),LL(0x50062651,0x5a53863a)}},
   {{LL(0xad1cac22,0x62e64475),LL(0xc7e11395,0x2008653e),LL(0xd9479c4a,0xa875ad01),LL(0x804b30d1,0x3e6cf633)}, {LL(0xb6b06e46,0x58b3ef6e),LL(0xf7b8410b,0x74c45dbe),LL(0xc278458d,0x02675759),LL(0xacd30bd1,0xb2ef4956)}},
   {{LL(0x339aae8d,0x1a4a5773),LL(0x0c0fe175,0xa775b952),LL(0x5d5d5ac1,0x7b39ac1b),LL(0x11a511b6,0x3f183d49)}, {LL(0x045ac045,0x9524e286),LL(0x4934c52f,0x0498d296),LL(0x9b636528,0x1fec5474),LL(0xc3e9b84b,0xec6f7a37)}},
   {{LL(0x12ee579d,0x870b12dd),LL(0x06dd62d6,0x2a9a12ab),LL(0x071d7582,0xbcd52599),LL(0xa869c457,0x7a36193a)}, {LL(0xe976ae5b,0xd29e6592),LL(0xadfecd58,0xe82c8712),LL(0xf714686d,0xbc83a440),LL(0x0c21e3ba,0xfe19344a)}},
   {{LL(0xd7a191ae,0x2a32c989),LL(0x4e58caca,0x00a25163),LL(0xe4a11597,0x2c6501b8),LL(0x7f1891e6,0xb3e45d09)}, {LL(0x659fd516,0xb7f532b1),LL(0xa7002930,0x99cf64de),LL(0xf2cd2d4d,0x56357ed4),LL(0x3447951f,0xa94cf5c5)}},
   {{LL(0x76a164be,0x26c7f244),LL(0xa72e974c,0xbd83e20b),LL(0xda31de06,0x64e9c241),LL(0x1cdb203d,0x022bc0f0)}, {LL(0x55c0601f,0x5eec4fcb),LL(0xb168a484,0xa1504f91),LL(0x1243026d,0xb9cf98b1),LL(0xfb3e5a1c,0x6a009deb)}},
   {{LL(0x60657650,0xf1df3752),LL(0xcb1b8d9e,0xa5bbd8f5),LL(0x81b6af13,0x9e0d9447),LL(0x624cb828,0x8572cecf)}, {LL(0xd003617a,0x28319d57),LL(0x996fde09,0x175c4766),LL(0x04878e13,0x168514b2),LL(0xec83a771,0x58a541d7)}},
   {{LL(0x29fb000f,0xafdaad3b),LL(0xc20f56f5,0x1977a8de),LL(0xc5b7ba77,0x450faf6f),LL(0xe5954518,0x93253964)}, {LL(0x644c3385,0x11ee0f31),LL(0xa8a57bad,0x6c24de9d),LL(0x5533a7ba,0xe8ff408c),LL(0xeace56fa,0x660a74d9)}},
   {{LL(0x8cc2a866,0xb4b2543b),LL(0xeffc0cbf,0x69f23f18),LL(0x5308b9b1,0x0db4682a),LL(0x17037e08,0xce7fac53)}, {LL(0x0a885b01,0xf02098c4),LL(0xb2e4eb6e,0xd375c03d),LL(0x70d4b81b,0xb6d4f6c1),LL(0x7ce5f297,0xa2b5e9cd)}},
   {{LL(0xb8a233c3,0x787229cc),LL(0x3419867f,0x44ef5dd8),LL(0x79d3d8dc,0x00316d22),LL(0x90bb1410,0xdcf32003)}, {LL(0x835d2264,0x62ad0125),LL(0x0ed6605c,0x768c8658),LL(0xfc44e760,0xa31abf17),LL(0xbb22e570,0xc91848ac)}},
   {{LL(0xb16c805f,0xad1882f5),LL(0x7ccf9e9a,0xb74cc0ed),LL(0x7b122dd7,0x9635af23),LL(0x5c3cd11b,0x48a20903)}, {LL(0x34c1eb54,0xa24820b6),LL(0x5284dcb3,0x31a3c330),LL(0x069c2ee4,0xd966cf59),LL(0xb3ff9335,0xa74eec6f)}},
   {{LL(0x4620e739,0xf44eeb99),LL(0xf4159a9a,0x7663a596),LL(0xb4b745b1,0x79c54f42),LL(0x59db9482,0xa8d34937)}, {LL(0x579501df,0x35fad92a),LL(0x289d7c2b,0x1d81bbe3),LL(0xddf3d371,0x1d60a274),LL(0x46df1233,0xf08e23e5)}},
   {{LL(0xf3a95f04,0x4bc4c079),LL(0xa8626015,0x0b43e660),LL(0x246ae3ac,0xedb31526),LL(0x41247209,0xa8536eb6)}, {LL(0xfdfacc62,0x6893a7df),LL(0xc557777b,0xf3de226f),LL(0x0d7f4265,0xa68c8d8c),LL(0x15c685e3,0x55a628eb)}},
   {{LL(0x5ecec6ee,0x8cad8f87),LL(0x2a06c242,0x4aefda2d),LL(0x57f00a7d,0x46a21033),LL(0x7ed125cf,0x91910c3a)}, {LL(0x541165d2,0x0b7f0e4a),LL(0x553eeec1,0x15ed1b93),LL(0xd24e020b,0xadf5b4db),LL(0xa7493b8b,0xf05307a3)}},
   {{LL(0x62070042,0x725548dc),LL(0xc274916a,0x74d71526),LL(0x6f098d01,0x3269851e),LL(0xf9ec928c,0xb2e01cb7)}, {LL(0x2b4368cd,0x96c2d922),LL(0xa0ec45d1,0x8eb84b03),LL(0x26e5b3ac,0x733ad068),LL(0x93c5a962,0xced3679e)}},
   {{LL(0xdd6eb876,0x23c6a22d),LL(0xa343dc3b,0xbd98ad9a),LL(0x56054515,0x61933d03),LL(0xe45cd744,0x4a64b769)}, {LL(0x12586de6,0x617a63f3),LL(0x7976e7d1,0x04984a9f),LL(0xcd2a0a6b,0xb00ba446),LL(0x7d059d46,0x5b64e7f5)}},
   {{LL(0x6a4b08e6,0x8801ce04),LL(0xb13bbe9c,0x66f31460),LL(0x4d87114e,0xb174e887),LL(0xf348e94f,0xb2fee192)}, {LL(0x7c822d05,0xfede2283),LL(0x8f82b14a,0x8d50c49c),LL(0x0f5f1b5d,0x21ea4f6e),LL(0xc1818095,0x68627cf0)}},
   {{LL(0x8a7b2458,0xc1c0650c),LL(0x8bbc6aff,0x82ab62bb),LL(0x6ce6989d,0x7b3665d7),LL(0x7579e973,0x2ad7991f)}, {LL(0x7e9e8510,0x701287aa),LL(0x0a18da53,0xb296a038),LL(0x2bf00fdc,0xf8c3af86),LL(0xb220dc06,0x55776951)}},
   {{LL(0x7d7dd541,0x4e6e4b4f),LL(0xfe5c7431,0x812feac7),LL(0x340297b1,0x6bdfa63e),LL(0x98009910,0xecc11e55)}, {LL(0xb25b98c0,0xee4c6165),LL(0x02c5939c,0x8a07b0d2),LL(0x23147c40,0x9b36c176),LL(0xde2eab3a,0x396054a2)}},
   {{LL(0x2c439171,0x1f41010b),LL(0xe8139388,0x3ff85ee6),LL(0x8f077633,0x4ada4c7d),LL(0x824e6023,0x9976011a)}, {LL(0xeaf49f63,0xa2501197),LL(0xd60b0c4c,0xdff2122f),LL(0xbab3df90,0x1a6a3abb),LL(0xb66ffd5f,0x854bbcc6)}},
   {{LL(0x728572c1,0x525964a9),LL(0xfadbd14b,0x8a4923a2),LL(0xcd90b61b,0x03830df9),LL(0x79c2afe9,0xcdb00f4a)}, {LL(0xa6c3f13d,0xff2f84bb),LL(0x5c0de4dd,0xdee45c30),LL(0xfba2e933,0x3e1dd748),LL(0x7c51124c,0xe9dcc690)}},
   {{LL(0x28e11f62,0x725177af),LL(0x8a64fdf5,0xc8e120a1),LL(0xf24fb357,0x82ab73df),LL(0x44724879,0x2d5d1618)}, {LL(0x96c66b86,0x09627e26),LL(0xc81d38c1,0x1d547cae),LL(0xd0f76658,0xbe8991a4),LL(0xcf11a976,0xf1508662)}},
   {{LL(0x3be3e582,0xa5dafebd),LL(0x07399295,0xd9f545ba),LL(0x676f9598,0xd9f564a4),LL(0x9294431e,0xec00bddf)}, {LL(0xc1fdc758,0xc1971113),LL(0x69a001de,0xe32f572f),LL(0xb907f044,0x048d7776),LL(0x5ca10e67,0x4a474e6e)}},
   {{LL(0x3039a4b7,0x6476dd40),LL(0x018ee2b8,0x85de9baa),LL(0xfd7365f2,0x0c945aeb),LL(0x96c7267e,0x2b47dc0d)}, {LL(0x0410de25,0xb12b48a7),LL(0x177242c1,0x3ba7a11a),LL(0x6504ff87,0x44e6cee7),LL(0x9d19f26c,0xb2605ff6)}},
   {{LL(0x50fb1b6b,0xa56bb589),LL(0x71d2fb53,0x98dc1180),LL(0xa1b78e04,0xa4fdc6f8),LL(0x39d9349d,0xbea745b0)}, {LL(0x62d7eb73,0xac474229),LL(0x8b808ac3,0x7b765138),LL(0xd0ca219f,0x882370af),LL(0x9d1c23e8,0x28dcff7b)}},
   {{LL(0x3872f0a9,0xc6dc70eb),LL(0xdfb642b1,0xb2f21248),LL(0x65bbdfc9,0x86838f0f),LL(0x40b28364,0x1d04a8b5)}, {LL(0x1e4d8d58,0xd4fa229d),LL(0xfad0a9cd,0x74ee5e20),LL(0x5a40ec4a,0x25a59aae),LL(0x3727d6cd,0x73be91f3)}},
   {{LL(0xd63f64eb,0x9c31405e),LL(0x91d2f1c1,0x9943c34c),LL(0x4fcdbf34,0x70ad75d7),LL(0xb239e00d,0xa6ce7145)}, {LL(0xcd04b9e9,0x136bceed),LL(0x44ed7f96,0xb9ebeb8d),LL(0x5d136280,0x068b43a5),LL(0x4c559b6b,0x2e1b6624)}},
   {{LL(0x5472d67b,0xe3808e72),LL(0xce74546e,0x73450378),LL(0xea1d58f7,0xc1b1b66e),LL(0xe34c2a7d,0x2b576e4f)}, {LL(0x2f732803,0xc2b1bdf7),LL(0x9f8e48c3,0x37aea390),LL(0x944f1cf3,0x8bbbb61e),LL(0x86c59643,0x5cc7ccaa)}},
   {{LL(0x8d5b000a,0xaf4c18e3),LL(0x2b6d561c,0x23b0edd0),LL(0x0d6cbe27,0x11b67ef0),LL(0xb1b50e9d,0x679d789b)}, {LL(0x372c4015,0xda198336),LL(0x65781ea7,0x5da50baf),LL(0x550201ba,0x00b3a6d4),LL(0xecfffc72,0x988b89f7)}},
   {{LL(0x25948852,0xf2f08a09),LL(0x406d1a34,0x4036bbb7),LL(0x23d2dd87,0x1cd57f08),LL(0x4704dac3,0x11a4387e)}, {LL(0xc5413b59,0xb8091a7a),LL(0x09b5fa71,0xe58940c6),LL(0x6a75397c,0x70fd5154),LL(0x5c59ff75,0xea544375)}},
   {{LL(0xac25bd3a,0x15e5bed3),LL(0x6b17971e,0x1bed3c33),LL(0xbaa96968,0x046fc1cd),LL(0x7090256f,0xda1b010d)}, {LL(0xe6677d20,0xeec55752),LL(0x24c9bb42,0x8eac5d06),LL(0x8f4120e0,0xc2f6270e),LL(0x07748faa,0xd9ae9fff)}},
   {{LL(0xe8f7b7af,0x5a1b2634),LL(0x81b1612d,0x1fcd743d),LL(0x3d420398,0x6b065aa2),LL(0x41e06643,0xe758b9c7)}, {LL(0x7f111b3b,0xe1e52b53),LL(0x83498731,0xb9ee0a5d),LL(0xea8154f4,0x49c19631),LL(0xe1c08746,0x8f5a3479)}},
   {{LL(0x65dd5561,0xe032d7c1),LL(0x442bef09,0x6c3420fe),LL(0xa64eff47,0x1d390561),LL(0x902763bf,0x0d8fbf07)}, {LL(0xa4bc6856,0x0262f26d),LL(0x9f4f2101,0x7c1b59a7),LL(0x51240642,0x663d9b38),LL(0x77ce53cc,0x39a0b4c2)}},
   {{LL(0x61f5e655,0x1c896beb),LL(0x9f4bfd2d,0x75c4c049),LL(0x10111b02,0xb8799a15),LL(0xa4c2fa0e,0xc76f8641)}, {LL(0x185fc036,0xd77ff7fd),LL(0xf5acbd16,0x53212bd6),LL(0x0408cff8,0x4ef7431f),LL(0xfb082c4b,0x45aa9d99)}},
   {{LL(0xf0438565,0x22c1fa8e),LL(0x4cb43ab5,0x8e3a2ee3),LL(0x232081d1,0x457df338),LL(0x482ff47b,0xd1293d9b)}, {LL(0x68106365,0x802a300e),LL(0xe51978c9,0xa8f27aa1),LL(0xa6a6a4d3,0x6ca0edda),LL(0x24c9526a,0x4cab1223)}},
   {{LL(0x56730245,0x26234b2e),LL(0xe1b54be4,0x9a04c15d),LL(0xee89282b,0x153fb6cf),LL(0xd79d81ad,0x5901ca12)}, {LL(0x7c3c5ffd,0xbe6853d8),LL(0x35e1942a,0x16d3efb5),LL(0x3b56bece,0x3491f207),LL(0x5b818cfd,0x0d75e0c1)}},
   {{LL(0x40969df4,0x79a0e319),LL(0x9ae34b31,0x75e4632c),LL(0x68e8df30,0x4a47585c),LL(0x2a495467,0x4a4a40e4)}, {LL(0x2762eae9,0x92b8a6f5),LL(0xc9a3d133,0xa204cd80),LL(0xd1ff23cf,0xa441ecfd),LL(0x4550ee57,0xd06feb58)}},
   {{LL(0xdc032002,0xe14ca6a1),LL(0x05505a36,0x9a780e57),LL(0x08cb2b29,0xad93852e),LL(0x008b00c4,0xa54deaab)}, {LL(0xe1042350,0x8cd2c71a),LL(0xa8915596,0x2014b85d),LL(0x97ddd1dc,0x1228b3e4),LL(0x4a3b3ab7,0xa97282ce)}},
   {{LL(0x0f1559ad,0xd978cd73),LL(0x86b14d3c,0x2e877fa2),LL(0x3660f189,0x01d3dc94),LL(0x0d2b4ddd,0x90ad950d)}, {LL(0x92245e3e,0xa8d26760),LL(0x4964245d,0xfc1bf8d5),LL(0xac3d97eb,0x31206c72),LL(0xa58c64cb,0x39dfd972)}},
   {{LL(0x7efbbd16,0xd631907f),LL(0x174f1fd5,0x4fdc84e2),LL(0x5c277996,0xe81e89b3),LL(0x5f79f1de,0xcb277b4e)}, {LL(0x2ed1962c,0x2eff44b3),LL(0x72883505,0xbe36a640),LL(0x545e69a0,0x14a1fac0),LL(0xd6f658b9,0x76dbbcbd)}},
   {{LL(0x78e2e86d,0x0720c584),LL(0xcaeead35,0x52fccffb),LL(0x587fd1b2,0x06f28c72),LL(0x9e48bf69,0xec36a912)}, {LL(0xdaa3cdbd,0x74874436),LL(0xcdc2f2a3,0xb3f7409f),LL(0x1951c078,0x0e50d7fa),LL(0xee8949f0,0xd97ff34e)}},
   {{LL(0x742d7b1d,0x00db635e),LL(0x29f0d0f9,0x5c0b280e),LL(0xeabf9b35,0xafa7e616),LL(0x2c8a76e8,0x7341e2c7)}, {LL(0x2e96f198,0x9679e34d),LL(0x90ee26ca,0x8c2661c0),LL(0x67a6516e,0x9c6dab35),LL(0x46b4b34f,0x7c8edc4b)}},
   {{LL(0x2afba4fe,0xc502cf2f),LL(0x6776dbf1,0x76847ae0),LL(0xa2c3c83e,0xace02706),LL(0x4601c550,0x0012645f)}, {LL(0xef6189bd,0x1940e14a),LL(0x2cdf5e89,0xba7f615f),LL(0x438a3781,0x698101aa),LL(0xa9e22357,0xf568a45d)}},
   {{LL(0x1f913210,0x83af640e),LL(0x8d505edc,0x529a29fd),LL(0xd6b0c85a,0xdf3d3090),LL(0x6897ea43,0x46e23886)}, {LL(0x416577ae,0x97cca980),LL(0x9aa08fc3,0x1f5a96a8),LL(0x56c05c30,0xcb014b33),LL(0x05ec9be4,0x1944765a)}},
   {{LL(0xddc4371d,0x2d6789cc),LL(0xf3618fc2,0xd768f5a6),LL(0x3da93c1c,0x77065e11),LL(0x0e27b3eb,0x4ea3fbc3)}, {LL(0x11ba30e9,0x7c1bfba0),LL(0x1036ebe6,0xfc6fba67),LL(0xd3231aed,0x0053a30c),LL(0xee3ac524,0x7f0613d9)}},
   {{LL(0x63093df6,0x95ec2fd9),LL(0x7c0eea52,0xfbc63768),LL(0x8b64ea48,0xf767b286),LL(0xf75bc633,0x6959b0ec)}, {LL(0xc9f63154,0x47e34c3b),LL(0xa524bc76,0xd616b19f),LL(0x632ac100,0xefc9bb54),LL(0xb4d96a7d,0xd9abba10)}},
   {{LL(0xfe2ad7e8,0x3b7dd91a),LL(0xb4ebf343,0x29134cd7),LL(0x152864fd,0x49d1c305),LL(0x80efc220,0x3afd83d0)}, {LL(0x3f2f0d27,0x3552517e),LL(0xfda48969,0x0a2b5006),LL(0x3c3e8ec9,0x568863ed),LL(0x891edec9,0xd99d5c62)}},
   {{LL(0xd1c9d6ee,0xb0ddc129),LL(0x57db23b4,0x373dad74),LL(0xb416c7df,0x7c178b0b),LL(0x4f8a7153,0x77431dac)}, {LL(0x41c1367e,0xf5288888),LL(0xb838c91c,0xf1518939),LL(0x541f3281,0x81e17838),LL(0x65b2bde5,0x70030244)}},
   {{LL(0x7350251c,0xdc309424),LL(0x7c811130,0xfac0c6ad),LL(0x6a141269,0x3817aa1a),LL(0xe10b4a6d,0x1aa5a92f)}, {LL(0x34648a96,0x996cca7f),LL(0x4e2a4f52,0x517a25b9),LL(0x38b1547c,0xff95ac42),LL(0xd9b9cd4f,0x01d981b6)}},
   {{LL(0x88d60eba,0xcc34d15e),LL(0xa0ea1a51,0x45851bf4),LL(0x82854ee0,0x5d5f9b30),LL(0x176ea156,0x914be21f)}, {LL(0x2a05c368,0xecac86d1),LL(0x73a666a8,0x255cb9c0),LL(0x78c0eec5,0x5e4799d9),LL(0x8fc05a71,0x40ed8989)}},
   {{LL(0x8ae03edd,0x54888ac2),LL(0xa83b554b,0xef3e9865),LL(0xb7612fe4,0x47b41822),LL(0x8f76cd2e,0xf6e16fd5)}, {LL(0xa977b5dd,0x091c7b12),LL(0x8f99d4aa,0x7051bf6b),LL(0xfed218fe,0x9f737902),LL(0xb752c612,0xd8112477)}},
   {{LL(0x18d13bd3,0xbb45c287),LL(0x23c6dd1a,0xbbf3a894),LL(0x13b9cf87,0xc8171c5e),LL(0x34f5348d,0x2dfc7792)}, {LL(0x985cabd4,0x9b9a662d),LL(0x4d971de0,0x588a6ebc),LL(0x574cba64,0xda9fd894),LL(0x651e6e67,0x7e0f0cca)}},
},
/* digit=2 base_pwr=2^14 */
{
   {{LL(0x4b8bceb8,0x88ca276c),LL(0x752d1106,0x6d4ec101),LL(0xf834dcbf,0x2ad98063),LL(0xdfff19d0,0x4da81d19)}, {LL(0x3a9828ff,0x4ccc7cd2),LL(0x2e64b332,0xf1e389b0),LL(0x7308b81c,0xe2fb6a6c),LL(0x5bcc0ac6,0xc6df66b2)}},
   {{LL(0xe1c58c80,0x5ccb8c75),LL(0x83fcc95a,0x2ba9de04),LL(0xdfccbcf9,0xccdeb0ee),LL(0x70f3d3ad,0x1d667d4f)}, {LL(0x36269820,0xc6aa14a5),LL(0x0fe87940,0x329a308b),LL(0xede5cfb2,0x39869970),LL(0xf601bb2c,0xc33c3068)}},
   {{LL(0xa1a8781b,0x3087444a),LL(0x5cff3cbf,0x6cb5b706),LL(0x83082714,0x7673a8e4),LL(0x0842a792,0xc4bce015)}, {LL(0x53e2a531,0xae71a033),LL(0x8b5315f9,0x147b28f8),LL(0x6c5ab37a,0xcc460133),LL(0x540dde16,0xb1dd088b)}},
   {{LL(0x11c09289,0xec250455),LL(0x164079c9,0x83042ba7),LL(0x6e3879a2,0x4881640c),LL(0x802452ee,0x77c5babc)}, {LL(0x7088f360,0x7a7759a6),LL(0xb74be7e9,0x02da352c),LL(0xe0338289,0x15800cdb),LL(0x501688c6,0xad69f7c9)}},
   {{LL(0x57ae1213,0xb7d35063),LL(0x97024ecd,0xd536753a),LL(0x24938196,0x9d680716),LL(0x44ed6d4e,0xac1bee4c)}, {LL(0x33e95503,0x6dd9c9bf),LL(0x88fc1c3d,0x5ee9f1fd),LL(0x21654473,0x4a701ff4),LL(0xbd2ffe36,0x9a316450)}},
   {{LL(0x103b5fa2,0xe9130a63),LL(0x8eee983b,0xe97f7120),LL(0xe8749cba,0x54b7f85b),LL(0xbb1bca55,0x69976910)}, {LL(0xf4e621d3,0x9ec4034f),LL(0x695e17da,0xaad567ed),LL(0xcedb2ea8,0x7647f054),LL(0x09fc7433,0xf85f944c)}},
   {{LL(0xb95eeddd,0x30af23b3),LL(0x89985f3d,0xfd1d565a),LL(0x4c254738,0xfbb53173),LL(0x171170a4,0xb07ba56a)}, {LL(0x294d55d1,0x5069882c),LL(0x792694c1,0xae0385c4),LL(0x11225dc6,0x0a0c7927),LL(0xe22867c9,0xadcc5f08)}},
   {{LL(0xaee03999,0x164ac67f),LL(0x79ff7f91,0x4de174d3),LL(0x548da6ea,0x063e4943),LL(0xdb7ccdf7,0x5264880b)}, {LL(0x49b992cc,0x4a18f34b),LL(0x14065870,0xe16b6f4d),LL(0x4cdb0e21,0xd32479ac),LL(0x162bc9f8,0xce8151f6)}},
   {{LL(0xe8f78545,0x0f8d9a2f),LL(0x3145b086,0x091643db),LL(0x23a1bcc9,0x5915a582),LL(0x8a280fc7,0x97348efd)}, {LL(0x65eccf5d,0x3f9d6236),LL(0x01ac8146,0xd1a34937),LL(0x8ad0d5c1,0x1b8e5128),LL(0xd581dd11,0x5cbcc9ef)}},
   {{LL(0xed059f1d,0x947ceaff),LL(0x7460a186,0xf5754d03),LL(0x0164ff7b,0x37698fa6),LL(0x35805339,0x630900d2)}, {LL(0xeddd6bbc,0xe467a6be),LL(0x5e36b12e,0xc53bffec),LL(0xf831fc7d,0x06dfd3f9),LL(0xdaef86ac,0xd995fcc4)}},
   {{LL(0x32d5b2e3,0x7d148468),LL(0x6335f566,0x7796b94c),LL(0x6769b8bd,0x693983d6),LL(0xed5244fa,0xff0306aa)}, {LL(0x89b8e801,0x2e90d41a),LL(0x39e732f3,0x1af09d86),LL(0x320ccb1d,0x96d14e1f),LL(0xc05dcece,0xbaf21c6f)}},
   {{LL(0xc216cf37,0x8ae82a1c),LL(0x773828bf,0xac437f45),LL(0x9d51a85b,0x8c12ff18),LL(0x34c16578,0xfeb563be)}, {LL(0xc6706966,0x9d9353b6),LL(0x0cda8733,0xcdc6eb5a),LL(0x3e4953db,0x033c186e),LL(0xb2e37f7c,0x2ba46a66)}},
   {{LL(0xb9f3ee06,0xb32115e2),LL(0xdd6346a9,0x1bc12cec),LL(0x321242fe,0x6b9c2142),LL(0x5c68ea06,0xcf9b9bb3)}, {LL(0x920d49bc,0x7fe554ac),LL(0x37aedebb,0x90b3a9b4),LL(0x7695af86,0xacb181e0),LL(0xfd567fea,0xd1c99c55)}},
   {{LL(0xfccf76eb,0xb7c18083),LL(0xf93113a3,0xc693bdbb),LL(0x66e03205,0x215ff05d),LL(0xf76d2a12,0x4424aaea)}, {LL(0xe7f30891,0xb23f2782),LL(0x062db479,0xad814d5e),LL(0x4aea78c3,0x347ec1d0),LL(0x6a2332f2,0x3d0f0a7e)}},
   {{LL(0x8ad9e323,0x02ecefa6),LL(0x0d45e0c9,0x16c81248),LL(0x2757306c,0xd4b6253d),LL(0x81e42d04,0xe90203a3)}, {LL(0xc13782f0,0xbcef10fb),LL(0x156267d4,0x823efe5d),LL(0xfddb0092,0x18add11a),LL(0xb104561a,0x27068a29)}},
   {{LL(0xda0abf3e,0x7eb7f516),LL(0x61b3381f,0x3c92ac94),LL(0xd3418870,0xbad7320e),LL(0x07dbe066,0xbab7a126)}, {LL(0x2def303f,0xe7ce59be),LL(0x2d1e0c9f,0x0bf1f237),LL(0x38f418dc,0x12c18d1e),LL(0xb85bb676,0x7fcc5e3e)}},
   {{LL(0x1b038ac6,0x0bcf25ad),LL(0xddf1becb,0x35388760),LL(0x8a1ad137,0x5734bf37),LL(0xb7645056,0x92f3a250)}, {LL(0x718a5ace,0x6ed926a4),LL(0xb967f1cf,0x8e63f0a2),LL(0xd835fe33,0x6d9cccc9),LL(0x31f82e18,0xb1b5efee)}},
   {{LL(0x24f2c6b1,0x997aa2a4),LL(0x9e536a91,0xde87114f),LL(0x0f819ec8,0x01938bd2),LL(0xef772a43,0x012e9031)}, {LL(0x77aa9256,0x1578eb4c),LL(0x61a0c8ed,0x052b4088),LL(0x6ab5a380,0x1153a330),LL(0x132f5675,0xa3e7f085)}},
   {{LL(0x909b8a41,0x5e946e00),LL(0x16a3c156,0x55f7d231),LL(0x8ac8f8e3,0xcd952464),LL(0xcd8d67f7,0x7c5184d4)}, {LL(0x39ef93a7,0xb3468964),LL(0x4e9058c8,0xf4aa0b7e),LL(0x4b7c713f,0xa409403e),LL(0x41a83e50,0x9d55e33c)}},
   {{LL(0x7e1754b8,0x9efee704),LL(0xb1c0027b,0x54085471),LL(0x45af4e6d,0xc5e7a6fa),LL(0x30048569,0xb4d3cd58)}, {LL(0xf3ae8e79,0xd2c20014),LL(0x849f3f23,0xd0b6af13),LL(0x17982a8c,0x3a1db915),LL(0xf9ffcf90,0xea3c8099)}},
   {{LL(0x7bada472,0x25d9eb82),LL(0x09afd498,0xff84d98c),LL(0x56ff21f4,0x5e2c1ffe),LL(0x2f2f3a94,0xafd07201)}, {LL(0xcdb673bc,0xb0227fe6),LL(0xfe8d7326,0x58fc0e7e),LL(0x191bfd4d,0xb988d3eb),LL(0x2474d8b6,0x82499093)}},
   {{LL(0x68caff21,0xd1ef53cb),LL(0x5074160a,0x3cff018c),LL(0x98f982fc,0x609a4688),LL(0x562a099e,0xee5caaac)}, {LL(0xf8c6cfd7,0xf650365b),LL(0x9cbc10ee,0x2652aa23),LL(0x6ab86f4e,0x904fd66e),LL(0x2d82f3d8,0x6a25bbc3)}},
   {{LL(0x19c7a275,0xd3e6ecad),LL(0x3604b2dd,0x05ed0451),LL(0x00c71863,0xdd1d87e2),LL(0x8cd23356,0xd9fc8793)}, {LL(0x0036b81f,0x3337f8ba),LL(0xb5300622,0x63b5a762),LL(0xce8800e3,0x4cf696f1),LL(0x07e3cbc3,0x12cb3261)}},
   {{LL(0x0fa12b5b,0x18eac953),LL(0x77d159b5,0x45ccf073),LL(0x6e844a0d,0xa7480444),LL(0xd77d1c18,0x4404e6c6)}, {LL(0xce1af18f,0x003e43a6),LL(0x17fdffcc,0x8a828081),LL(0xcabf3d17,0x91b63c11),LL(0xad26f286,0xa4dedc21)}},
   {{LL(0x1a2b1579,0x6bf62b69),LL(0xceeb29ff,0x3b67b87b),LL(0x40d4b996,0x451ffadb),LL(0x080978f8,0x10c6ae50)}, {LL(0x2c242dc5,0x959d47e2),LL(0x5423e158,0xced9e922),LL(0x8d8a68f1,0x9a212d4c),LL(0x3708393f,0xeff3d644)}},
   {{LL(0xfbaffded,0x43f51810),LL(0x0f6fd7c3,0x3886ccb4),LL(0x13c31946,0xb939247b),LL(0xaa1fd72a,0xbc1ee613)}, {LL(0x631fd790,0x6d40140a),LL(0xd26b3fd9,0x9382e3ba),LL(0xb3af96c3,0xff414370),LL(0xe0ea9ad6,0x38c813cf)}},
   {{LL(0x157594da,0xf8844c3c),LL(0xcac628bd,0x2a7b514f),LL(0xc08c5107,0xc023e4e2),LL(0x3f2722fe,0x6c164496)}, {LL(0xc03a22ad,0x842e1d06),LL(0x37ddae0d,0x5dbc2865),LL(0x0342bc72,0x46dfc88d),LL(0xa4a3c65c,0x873c805c)}},
   {{LL(0x60aa5c14,0xd202853b),LL(0x3850cc05,0x1dc35d34),LL(0x0cabccfd,0x8014357e),LL(0xc5a5225a,0x1aa44ce9)}, {LL(0x3a8444b4,0xa3cef920),LL(0xc95384b1,0xcf3f91b3),LL(0xc9e5da54,0x1d625ba1),LL(0xb1d0f46a,0xbf1fba37)}},
   {{LL(0xfb3f4885,0xdcef4fad),LL(0x3267f912,0xa49debb2),LL(0x1e121cb8,0x6417d37a),LL(0x533e94c9,0xa6d871fc)}, {LL(0x2e4834fb,0x89f80208),LL(0xb353452f,0x27e83f0f),LL(0xe1f8f322,0xaf009f3c),LL(0x89319fd8,0xa5b77a77)}},
   {{LL(0xedf71900,0x0a89e741),LL(0xd514d93f,0xd679b841),LL(0xb0a03702,0x8878577f),LL(0x85a209ae,0xc9607b78)}, {LL(0x59432a28,0xb7bd0616),LL(0xed567145,0x0da060a2),LL(0x1a449f52,0x44e35a7a),LL(0xbbaccc0f,0x9c9a2c82)}},
   {{LL(0xd83701f4,0x83abd436),LL(0x4bb9cbe8,0x56e8bfe8),LL(0xc631cd1e,0x5b545cc8),LL(0x955aca7d,0x6d03426f)}, {LL(0x2f8db817,0x049fc9fa),LL(0xdc59675f,0xfcec1799),LL(0x0455f095,0xa00ed392),LL(0xe5096b18,0x6d7cfa5f)}},
   {{LL(0xe30ae90b,0x2cda5cae),LL(0xcaabea0d,0x2cc34290),LL(0x41e67856,0x564afcd9),LL(0xcf6ef8b7,0x210c7a09)}, {LL(0xf82a591d,0xc316d352),LL(0xab43d2a1,0x5fe8cc4d),LL(0x8b4e9470,0xd8ebce97),LL(0xba321a07,0x26c78f44)}},
   {{LL(0xd75e509c,0xa63f4b34),LL(0xb9a6c63b,0x9122bbc5),LL(0x17942443,0x8bf792a3),LL(0x7f4f70f9,0x95b05d68)}, {LL(0x13b70dc8,0x57d7dee5),LL(0xfc376fdd,0xe84259ed),LL(0xe3e313b4,0xf8c4c4ff),LL(0x13fa8ff1,0xf8e2d3da)}},
   {{LL(0xf8e4eede,0x9692c390),LL(0x0e95a902,0x3b514551),LL(0x7360623e,0x45c1670c),LL(0x6abd2a82,0xf7a74f55)}, {LL(0x24e8e721,0x99b16e7e),LL(0x512f1401,0xae52fa2a),LL(0x3f3a09d5,0x46c60e80),LL(0x0750e968,0xf803d1b3)}},
   {{LL(0x1791644e,0x17840d2f),LL(0x3b7981e6,0x3e32b3db),LL(0xd3dfae10,0x2d0830a5),LL(0x6cc6dd0d,0x1b28d118)}, {LL(0x78368274,0x944a9889),LL(0x55b1bf81,0x310da94a),LL(0x0d739056,0x503061ec),LL(0xb4d73288,0x1947e940)}},
   {{LL(0x0228346f,0x760ee846),LL(0xc5cff077,0x108765b3),LL(0xbeb12160,0x22092b39),LL(0xb63001af,0xa631d553)}, {LL(0x0af3d43a,0x9340cac4),LL(0x60d338a3,0xe6cbfb54),LL(0x7ca3f604,0x2280ff0c),LL(0x3ba738cb,0xaf48f86b)}},
   {{LL(0x47d372ff,0x7435dd78),LL(0xf005c006,0xbf9c7149),LL(0x7a8d0e81,0x624084b9),LL(0x4840496c,0x50b578f3)}, {LL(0xb52a4266,0x414ca2c1),LL(0x5535ef0b,0xa3c30275),LL(0xb50f7f47,0xd4b808c1),LL(0x9a199920,0xe6781ae2)}},
   {{LL(0x27a91ef8,0x50141234),LL(0x5b77d060,0x2f4f5937),LL(0xc2dcb03d,0x1be8269e),LL(0x9f65043f,0xa293017c)}, {LL(0x8caac401,0x1678dfe0),LL(0x968b1716,0x4942d8ce),LL(0xae36e201,0xa9b55fae),LL(0xd5279632,0xcfe4bde3)}},
   {{LL(0x79d637e6,0x6126d744),LL(0xd63b4aad,0x8491f1a8),LL(0x9816b82c,0xdf97b736),LL(0x796408c1,0xafca2c36)}, {LL(0x7a8e8058,0xc17f3f01),LL(0xe74705e2,0xb3335a24),LL(0x46e3e3b0,0xee200023),LL(0x40630e08,0x07bce061)}},
   {{LL(0xee8f9dfc,0x46b42c00),LL(0x9e9b7f58,0x3b8e8509),LL(0xd36e8e89,0x83df4b18),LL(0x15d50555,0x09631af5)}, {LL(0xef1ee3f1,0xb7906b77),LL(0x4bd1e17b,0x8272dc83),LL(0x4903faac,0xf160bfd9),LL(0x0dc71e59,0x7fe9e999)}},
   {{LL(0xe714187d,0x6ee9b790),LL(0x9d5a656f,0x7391ec2a),LL(0xe10b20f0,0xcbb55ec6),LL(0xec3645d6,0xbba3b57b)}, {LL(0xe18322e8,0x9c3265bc),LL(0x93328c91,0xdb49b0f3),LL(0x49c2bbec,0xa911db72),LL(0x6e5bd229,0xf71b4df3)}},
   {{LL(0x7ba27baa,0xdccede33),LL(0x4b712a97,0x1af4476a),LL(0x8a8683ad,0xf0aaabec),LL(0x6fa8e84c,0x138cdac5)}, {LL(0xdc78b1ad,0xd2d50b00),LL(0x696442b9,0x26fc0b72),LL(0x125bf11b,0x12cd5d8b),LL(0xc4f82ca6,0x2a2ce980)}},
   {{LL(0x52e00dd3,0x9921c0a6),LL(0xf1d7e1af,0x98e8707a),LL(0xdf03b040,0xaa7aa8b8),LL(0xdff6bd74,0xb3ba8b23)}, {LL(0x31db8c0b,0x2fd0faab),LL(0x2819b732,0x4697e9bf),LL(0x0425b866,0x2dc3a5d0),LL(0xd97816f1,0x4b9e7899)}},
   {{LL(0x4c756c70,0x1355c412),LL(0x0fa089af,0x2d4c4eee),LL(0x3b8a01b5,0x4d8425a8),LL(0xa3531d3a,0xcc26b8a9)}, {LL(0x7ebd9eea,0x6eebe11b),LL(0x92c0f858,0xd511a797),LL(0xec49a0c8,0xaa863f01),LL(0xa8242995,0x7fb65625)}},
   {{LL(0x3dbc00c3,0x9de9d3f4),LL(0x3f7d61ab,0xb846152f),LL(0xd0d74549,0xc060fdbd),LL(0x7b273702,0xe722aab2)}, {LL(0xd81b6f6e,0x9e54f098),LL(0x9e2fde1f,0x32dbaa5f),LL(0x9ebbc796,0x14cc9995),LL(0x0eb83921,0x4ca6686c)}},
   {{LL(0x10a0c0bd,0x6e65d7c6),LL(0xb3c0f6cd,0x1f6930d7),LL(0x4d783d6f,0xe4e0a933),LL(0x70b20ad4,0xc945ee7f)}, {LL(0x034b0265,0x521bd135),LL(0x0fa9be95,0xeb5d96e0),LL(0x357ef592,0x834c28c2),LL(0xb81df99f,0x08ab5b4c)}},
   {{LL(0xf464825d,0x6be99d80),LL(0x9a0c1293,0x1cc83719),LL(0xe7e43c6a,0x76616803),LL(0x91cc47ac,0x6fa33715)}, {LL(0xdbfc08b9,0xc3fdb99b),LL(0x68e2b249,0x66e1ef2d),LL(0x64a4a438,0xd3d8ef7f),LL(0xa6f25b00,0x775a70fc)}},
   {{LL(0xa0cb5443,0x2444c682),LL(0x4b743ee7,0x264c2662),LL(0xa303eb20,0xd7a1adc4),LL(0xf60a5b98,0x3f14821b)}, {LL(0x1a1d7661,0xa439102d),LL(0x8d8a5a1a,0x47c25a37),LL(0xa34c66a9,0xdf4a48db),LL(0x4c828c73,0xab467364)}},
   {{LL(0x3459cc8b,0xd3caad73),LL(0x181b16c2,0x08eeb442),LL(0x70600d33,0x3444abbb),LL(0xcd0f8e70,0xaa2a39c4)}, {LL(0x24836d70,0x5fc6ae8f),LL(0x47d32fd4,0xc119be84),LL(0x0d6000ce,0x2b3f3771),LL(0xe602337a,0x439893a8)}},
   {{LL(0xc1e8e564,0x4b75ff6e),LL(0xe451cf42,0x6185413c),LL(0x162c3150,0x0276d3b6),LL(0x3aea9c55,0x844539e0)}, {LL(0x42e9d70b,0xfc629ee6),LL(0x0be610c9,0x4eb9b7e6),LL(0x39ca3d92,0x8c53fda1),LL(0x14c2e9e2,0xd2e4cfa6)}},
   {{LL(0xf14b31b2,0x3c1f6895),LL(0xeb951fad,0xad42d951),LL(0xb8f10fc1,0x5b20a169),LL(0x586c61cd,0x284810bd)}, {LL(0xe863d781,0x0c4a89aa),LL(0x9c235d5c,0x2eda4847),LL(0xe6005150,0x8e141950),LL(0x52785efa,0x75716e1b)}},
   {{LL(0x8305624e,0x290ced58),LL(0x06650920,0x398956a8),LL(0xdb5bd5b6,0xd057a47b),LL(0xbe9e119c,0xf2d85299)}, {LL(0x7c5fc039,0x4783095c),LL(0x05363915,0x72f7e7cd),LL(0xdf3e2968,0xe46b90d1),LL(0xaaea2e53,0xaadb3dae)}},
   {{LL(0x0cc4f426,0xf5d37496),LL(0x59d78369,0xa59bffa8),LL(0xf0a46b04,0x7ad4cc11),LL(0xb8e21b9e,0xcbd63351)}, {LL(0x5653ebbf,0x60d255e6),LL(0x4d6b5843,0x3eaa59af),LL(0x9e1df2e2,0x90049d25),LL(0xe56aa105,0x9a185a6d)}},
   {{LL(0x80e3d909,0xbd31c5cf),LL(0xa1f034d1,0x30caad3b),LL(0xd9c7c342,0xaca74fa1),LL(0x9565cf8a,0xac722cfc)}, {LL(0x5b42e582,0x8b172ce6),LL(0x9b0607b2,0x9e99e4e5),LL(0x9446ca45,0x284eb579),LL(0xc57c9feb,0x6c5464ba)}},
   {{LL(0xe511bc3b,0x1437fc95),LL(0x834d0889,0x22d7bc16),LL(0xc5071c43,0x62e545b2),LL(0x8cb4acd6,0x4c644d48)}, {LL(0x68246492,0xd9efbe50),LL(0xcbd8ad0e,0xc9d169e7),LL(0x798ae01f,0xcb7365dc),LL(0x6d0dea3a,0x5783f98f)}},
   {{LL(0xec454423,0x9b4a7e38),LL(0x96ff4c8c,0x27405d08),LL(0x0c462f7c,0x9769f097),LL(0x7dc946aa,0xcbda5412)}, {LL(0xe7dd5146,0xdacb510f),LL(0x30507b37,0x9c9a0d39),LL(0x05ded0ac,0xa605730b),LL(0x6c6c7b5b,0x7e683472)}},
   {{LL(0x7c952984,0xb378d92c),LL(0x72ae34d6,0xec76370d),LL(0xacda665b,0x1fde0bde),LL(0xb931afc1,0xc8f648f4)}, {LL(0xb960f6ce,0x2b55adb2),LL(0x7336a643,0x71b3bdd4),LL(0x73cc39e7,0xf66e77bf),LL(0x2fa3999a,0xf582c5e8)}},
   {{LL(0xaf986d1d,0x30ecd0c7),LL(0x4557dd65,0xa2ae53ed),LL(0x7d618a1d,0x97ebccfb),LL(0x11eed889,0xcbf54149)}, {LL(0xd8f2bdd4,0xdd0ff0e7),LL(0xfa769e74,0x6ac4a9fb),LL(0x93e5abab,0xdfdfc7e9),LL(0xdffc6fcc,0x0c7151c5)}},
   {{LL(0x5cbae56c,0x6d75e962),LL(0x96dccb89,0x77fae152),LL(0x6cc0e535,0x275c4946),LL(0x81781318,0xc4a400a9)}, {LL(0x77ba50e6,0x8b9f872c),LL(0xa138eeb4,0x971b6cb3),LL(0x53f552a7,0xa7e7d1f9),LL(0x8447c730,0x360512ce)}},
   {{LL(0xc5454439,0xf0c73bbb),LL(0xa3a24b5c,0x7f1b9b18),LL(0x51fa7d6b,0xc5bb48dc),LL(0x8b05a553,0xd264d6ec)}, {LL(0xe9371f83,0x123caaf2),LL(0xb149f564,0xdf5da393),LL(0x853b9bab,0x38e02eb6),LL(0x95bf6647,0xc6aab96e)}},
   {{LL(0x3141219b,0x4890be89),LL(0x7883fe8e,0x7afe4c2f),LL(0x59b86241,0xc27bd13c),LL(0xaacebdc9,0x1b9720f5)}, {LL(0xf6b2174c,0xa054e203),LL(0x60f6de8e,0xd4e7b952),LL(0xf4558633,0xcf7b1aea),LL(0xbefa40a6,0x43fc1881)}},
   {{LL(0xe23cef63,0x592164dd),LL(0xf7b4aaf2,0xfe57d6e8),LL(0xe8aef9bc,0x38a5e2c9),LL(0x1ac2b10b,0x576bd78c)}, {LL(0x14309d10,0x2357944c),LL(0xed0ed94a,0x9933d7ed),LL(0x0339f299,0xb8792ea3),LL(0x87fd9bd1,0xcfb44322)}},
   {{LL(0x92966739,0x864f2fd5),LL(0xd3cfd83e,0x7435ecc5),LL(0xec4249f2,0x8516d277),LL(0xfc158b34,0xaa7e1a8a)}, {LL(0xfbe640a1,0xfc0fc22b),LL(0x91121fec,0xf287767f),LL(0x3f590dcb,0x0ce48273),LL(0xf087c249,0x5e994e2f)}},
   {{LL(0x65604726,0x681a38c7),LL(0x247a421e,0x4f8c6ae3),LL(0x1294956e,0x1a51eaa0),LL(0x47c9b324,0x0984b1ef)}, {LL(0x597b7696,0x3749bd0d),LL(0x08e57ee7,0x9d432b78),LL(0x2ba112d2,0x3092afe1),LL(0x16c5a7f5,0x89ccee49)}},
},
/* digit=3 base_pwr=2^21 */
{
   {{LL(0x54089685,0x355e9d7b),LL(0x40818349,0x9f0ec68f),LL(0x3861b80f,0x4cf4d8cd),LL(0xc1f5fa14,0xcce669fd)}, {LL(0x1788f9da,0xea212509),LL(0xf3ccf239,0x32953613),LL(0x50027f3b,0x1048d092),LL(0x4270fbcb,0xe807b39d)}},
   {{LL(0x95e388c3,0x5099dc55),LL(0xea44e3ea,0xd0670ff5),LL(0x61b41f7b,0xd212c993),LL(0xfaf13305,0x4f594af9)}, {LL(0x05c01232,0xbc508bf2),LL(0x39ff08a5,0x76833536),LL(0xb837741a,0xa1cf70bd),LL(0xaaf7bd2a,0xba8e6616)}},
   {{LL(0xdef27938,0xde04c343),LL(0x48cee32a,0x3f15ca91),LL(0x9dd142da,0xcb61573b),LL(0x126dd9bc,0xc094eefd)}, {LL(0x136bb4da,0x5d42f1a5),LL(0xdb2f3449,0x75693952),LL(0x5c16795e,0x98017cd6),LL(0x2afb67db,0x9e401530)}},
   {{LL(0x9b7c6c75,0x6376749f),LL(0xacbca35d,0x680eacdc),LL(0x5e145b32,0xe87fd5b5),LL(0x36b886af,0xeb20d1ba)}, {LL(0x779b12bb,0xca499055),LL(0x0be39fb7,0x6f290ff2),LL(0xf4a128ce,0x33ad6fe0),LL(0x9b31da81,0xf09e2a40)}},
   {{LL(0x12039372,0xb2ed3d70),LL(0x2ff46c13,0xb87e02c4),LL(0xfb27dce2,0x164246c6),LL(0xe6d95811,0xe34ee8f6)}, {LL(0x3ec1fde9,0x66cc601c),LL(0x80ffdd56,0x056b3194),LL(0x9626aa21,0xff009868),LL(0x2d931092,0xc3e4982c)}},
   {{LL(0xc3d42729,0xbc0da9c1),LL(0x720df0a0,0x4905da24),LL(0x45f6eadf,0x0e5e1fa0),LL(0x2aab7523,0xc02033f3)}, {LL(0xedde75e1,0x45ba916f),LL(0x75c68e52,0xf43919bd),LL(0x84892e6a,0x00e7c076),LL(0x70dfeb08,0x259f8488)}},
   {{LL(0xd8a869a0,0x3bfd5f2c),LL(0x574e7d67,0x1df48669),LL(0xe14cfd3b,0x16d6ed5a),LL(0xfcf78465,0x583aac2c)}, {LL(0x67da2ae9,0x67210e6b),LL(0xcfee511d,0x0b024e70),LL(0x13839a4f,0xf27e122c),LL(0xb79dfa97,0xfa5356c9)}},
   {{LL(0xf357999b,0xf0c24783),LL(0x26bfacb3,0x2c21474c),LL(0xd3ddb945,0xe3abed6a),LL(0x6031a5ea,0xbb21b764)}, {LL(0x8afc2a09,0x6db3b68b),LL(0x81306b71,0x1aac2f08),LL(0x852eb6f5,0x882c3371),LL(0xd98e9b6f,0xadfe0c1a)}},
   {{LL(0x7edcb9e5,0x0247ee7b),LL(0x1f29918b,0xe29ec013),LL(0x6099b6ce,0x5d1629e6),LL(0xcb534584,0x68587803)}, {LL(0x8ce551d3,0x6ccfeddb),LL(0xf85123a8,0x7ef98b72),LL(0xf9711dcd,0x19af4771),LL(0xfd80e4dd,0x8f67858b)}},
   {{LL(0x7d607ee3,0xa4c8c016),LL(0x4015a479,0x15db36d7),LL(0x9d28ea30,0x0cb58eee),LL(0xbecb7b4e,0xb3d469b0)}, {LL(0x6f476e2c,0x811081b9),LL(0x59c78fab,0x264da3aa),LL(0x3cd73147,0xd6e5813d),LL(0xe905362c,0xce9e34a4)}},
   {{LL(0xcb3afa55,0xe551ec2e),LL(0x4b05589c,0x2c9bef25),LL(0xbcd083bc,0xd36ddeb7),LL(0xddb54a24,0x1c180b52)}, {LL(0xc0961f32,0xb84220f3),LL(0xfe3ae670,0xa71103fb),LL(0x46902477,0x6a14d319),LL(0x778b8eee,0x516701d2)}},
   {{LL(0x4c3166d5,0x1cdb1025),LL(0x3d6fcb6e,0x3a0ba2c2),LL(0xb3820def,0xa218b4af),LL(0xbfe8a8f8,0xda6de958)}, {LL(0x4ceabdfa,0xc2b3c755),LL(0x8d73edcb,0xd3534691),LL(0x0ce17182,0x453b8e63),LL(0x01654263,0x6507a5b0)}},
   {{LL(0xd5da0e59,0xb2b8e424),LL(0x61ac4c2e,0x7e599c75),LL(0x41aff49a,0xc64cb4c3),LL(0xea3e378e,0x0e231e63)}, {LL(0xe08edace,0x707cc0e3),LL(0x5410779f,0x18918dd2),LL(0x2eef6bb3,0xcdd57690),LL(0xff758569,0x4c54d7d8)}},
   {{LL(0x2c89683c,0x49459204),LL(0x7827e518,0x93596a16),LL(0x2b20c939,0x6198954b),LL(0x8044d3ba,0x6672c94d)}, {LL(0x199b16dd,0x55e95fd3),LL(0x4185999a,0xa8484135),LL(0xfe36e449,0x5e8709c8),LL(0x91401957,0x47470e2e)}},
   {{LL(0x0874afce,0x0058bb09),LL(0x606c3e52,0x19fb1d56),LL(0x710903a0,0xe1208b2a),LL(0xd47dfd1c,0xecabc372)}, {LL(0x5e94818f,0xd9daa7f4),LL(0x5dc99882,0x1302ac8f),LL(0xc44d37be,0x7b4c6b15),LL(0x72d19e0d,0x0bcf6d4c)}},
   {{LL(0x3fd5a1de,0x1e0bf063),LL(0xa75b5b8c,0x5d05e901),LL(0xcb3c617a,0xbbbdb1ab),LL(0x1aef4706,0x44954a8c)}, {LL(0xff6a6e47,0xbc3ceea3),LL(0x0ded1275,0x6140f421),LL(0x4dabe95f,0xbb4b4c04),LL(0x7135e813,0xc55e87da)}},
   {{LL(0xd963dd6b,0x15ad105c),LL(0x666941a3,0x33d18f73),LL(0x5d9253d6,0x860ccabe),LL(0xd16e8b69,0x2af702fd)}, {LL(0x74e525c0,0x7e46aadd),LL(0xaf59f48f,0xd9958a44),LL(0x8e7de482,0xd8ca872f),LL(0xcf7d007d,0xc2270c14)}},
   {{LL(0xa200e574,0x87c6204e),LL(0x7b69e79e,0x0ee014cb),LL(0x82b23226,0x176ff378),LL(0x8dbbb2f3,0x802d829d)}, {LL(0xe0a4dc31,0xb902924f),LL(0x5fe522f2,0x1f1a9ec7),LL(0x4da7c04a,0xbcd95d85),LL(0xb1543c0c,0x3a3a2e63)}},
   {{LL(0xf3271bf8,0x9e70a3ff),LL(0xd2cd68ad,0xd2522d88),LL(0xa6b727b9,0xb777851b),LL(0x63ff5264,0x58953d6f)}, {LL(0xb65c70d2,0x5e111c22),LL(0xd3a5143f,0xaae73c5b),LL(0x85ef5dc0,0x2daa2bfc),LL(0xea13ded3,0x5e7258d2)}},
   {{LL(0x2e3ce423,0x4161127c),LL(0x6b1af415,0x7e35a0a2),LL(0xeed24b7b,0x004483a8),LL(0x9f9d44f1,0x2816180a)}, {LL(0x062829a1,0x214add93),LL(0x225e847c,0x262a0bef),LL(0x5d6c53c4,0x4bb1b1ce),LL(0x91d06e53,0xd02f829a)}},
   {{LL(0x784da27c,0xcdc8ba5c),LL(0x161b5836,0x78a6c0d2),LL(0x8373c6a4,0x6bea92c4),LL(0xa881f59a,0x815f1a30)}, {LL(0x227cb8e2,0x699c8642),LL(0x25a2b9d0,0x515d1e2b),LL(0x1787b3e5,0xcb5f1c6c),LL(0x104dddc6,0xc9a10260)}},
   {{LL(0x0f3811e5,0x18be4f2a),LL(0x71e727d3,0x8c05d3fc),LL(0xfa707140,0xecae3e5f),LL(0xd275b463,0x4bb05b16)}, {LL(0xb02a5ac8,0x74bad373),LL(0x520344ee,0x7232875a),LL(0x65059d8f,0x32cef98c),LL(0x54e1b11d,0x68e0fdb6)}},
   {{LL(0x3f3db43f,0x683424f3),LL(0xabf4a83f,0xf5f0878f),LL(0x4ac2c5c9,0x681350d9),LL(0x47dd3652,0x825e9ecb)}, {LL(0x20713db6,0x420743f0),LL(0xd1b082e5,0x95db7427),LL(0x1affa57d,0xa0e1117f),LL(0xf940f325,0x62c87b5e)}},
   {{LL(0x4e1d5d9a,0x6a65fda8),LL(0x345ccdef,0x0c0fe385),LL(0xd6d72c0a,0x19ff360f),LL(0xfb016131,0x1be1e8d7)}, {LL(0x025b45e1,0xe2f27e91),LL(0x05259bf1,0x25bec266),LL(0xe51cc67e,0xd7b8b4e7),LL(0xab80a20e,0x3a839aa5)}},
   {{LL(0x9f85320d,0x04a9b695),LL(0x98d669f3,0xb939cd83),LL(0xe6948957,0x24464ced),LL(0xa43928e8,0x463de507)}, {LL(0xf8755845,0x4e1844e7),LL(0x5447e61c,0xc9c71091),LL(0x599d4bd7,0x1798f394),LL(0x1e072c64,0x758f7630)}},
   {{LL(0x739b1925,0x83c93728),LL(0xfa8eb048,0x692017d7),LL(0x478d1ee3,0x4a3a2a59),LL(0x022640cd,0xb8e62912)}, {LL(0x8572b8d7,0x4689a4dd),LL(0x8f79da63,0x6281ddfe),LL(0x212a153c,0x788bf9aa),LL(0xb3438da6,0xb67e18f5)}},
   {{LL(0x31cebdb8,0x3fbafc51),LL(0xb042bd47,0x7f8ad590),LL(0xe3055004,0xf5d26c88),LL(0x3d7d6f5c,0x7f23a149)}, {LL(0x8758ccc0,0x2fee5428),LL(0xe1b80dfa,0xb08c91b7),LL(0xea0c0a53,0xf2bcc903),LL(0x04e684ff,0xcdf2eae0)}},
   {{LL(0xe1d9a693,0x354b2c07),LL(0x97a833a8,0x93b1fa2d),LL(0xe9e5f2b1,0x2dcd22c7),LL(0x18aa3163,0xf040a69c)}, {LL(0x76939874,0x4f9a4b29),LL(0x15e24d44,0x58e5947f),LL(0xb0c2ef6f,0x9b47a945),LL(0xf630e92c,0xc4a15b7d)}},
   {{LL(0x7b1d4141,0x8d7a33e7),LL(0x966486bc,0x44dabde9),LL(0xef31dc9d,0x387a6016),LL(0x1462ff61,0x76744b23)}, {LL(0x20cdd726,0x2ad63954),LL(0x0e7803da,0x9cff7e86),LL(0xfd124ed3,0xaf5b8b4a),LL(0x050c1149,0x466dbbbd)}},
   {{LL(0x06b296a3,0x68352636),LL(0x7f3fe1ef,0x0ab40080),LL(0x05bf08f8,0x1fc38951),LL(0x633c457f,0x69b54ae4)}, {LL(0x1a206c53,0x2ad428c6),LL(0x8b09b3f9,0xd6725687),LL(0x0bc619c9,0x552d4d0e),LL(0x3113c689,0x0e88b313)}},
   {{LL(0xe87a91b4,0xb2483b80),LL(0x0c75377b,0xb9f842d7),LL(0x5a78145e,0x50463f38),LL(0x830817a9,0xf2d3810d)}, {LL(0x39cc886a,0x1819261e),LL(0x8415699b,0x697de51d),LL(0x5cab106e,0x688a874e),LL(0xcb8692ec,0xde48f3bb)}},
   {{LL(0x38f4194d,0xffa1de97),LL(0x3b996b63,0x33d2726a),LL(0x0d2053a7,0x787c0ec3),LL(0xeecd5c0c,0x9447e9cb)}, {LL(0x284773c0,0x077f121c),LL(0x815829a1,0x496427e4),LL(0x94def08b,0x4b119786),LL(0x9c15a778,0x9e7b29e6)}},
   {{LL(0xfd4a8a47,0xa4d6d2be),LL(0x4333baef,0x4f000a12),LL(0x642c570b,0xc9049d86),LL(0x25e6aa6a,0x9424e8f9)}, {LL(0xe011cfec,0x84de7fe9),LL(0x1e8c83b0,0xf273f956),LL(0xa47a40a6,0x98960835),LL(0x0a13c27b,0xd91a20f1)}},
   {{LL(0xed703e13,0xaf08b4ef),LL(0xc9994946,0xefcbcf34),LL(0x2d53b069,0x019e6f38),LL(0x9b160894,0x3d62c3c0)}, {LL(0xadfc8f3b,0xac7ad700),LL(0x0042fce6,0x41cc0cc3),LL(0x21cf742c,0x0228ae75),LL(0xf4c9a1a9,0x56a1152a)}},
   {{LL(0xfebd27dc,0x5d8a3321),LL(0x7c525f7f,0x89bce700),LL(0x1c1039ee,0xe8f815a9),LL(0x62e86536,0x9f6db698)}, {LL(0x66fe804d,0x1ea6e7a6),LL(0x261aea16,0x652acc41),LL(0xf9df596b,0xde28e5d8),LL(0x1553a545,0x18f453c1)}},
   {{LL(0x84eeb5c8,0xa224f763),LL(0x835ba87e,0x8ac452f5),LL(0xc5f4c054,0x9b2b5939),LL(0x3ac1cdcc,0xb2577943)}, {LL(0x772c60dc,0x1ba2cd0d),LL(0xd7a9bd1c,0x1fa52c43),LL(0x60444f34,0x2efd4f4a),LL(0x2bdcfc9d,0x7d188c05)}},
   {{LL(0xe1913711,0x49ef6825),LL(0x600d6c46,0xbca95ded),LL(0xaf8d66d3,0x63916baa),LL(0x2dc837a8,0x04981202)}, {LL(0x0d3ae79d,0xb501e517),LL(0xb4edb859,0x99ff7864),LL(0xaf4ec081,0x5099edee),LL(0x964f4052,0x89574889)}},
   {{LL(0x52066d70,0x1690fdb8),LL(0x671f4e7f,0xb403207d),LL(0xd7413111,0x8ebc1d1b),LL(0xb4cfdf14,0x1432d7fe)}, {LL(0x65ad5d0e,0x9277666a),LL(0xa928e194,0xbd5ae578),LL(0xb64962fb,0x2f6c10d5),LL(0x2e794187,0xe3d756c0)}},
   {{LL(0xd3e6349c,0xf04fd82a),LL(0xcc7d39b6,0xde602dba),LL(0x044e7deb,0x0886e20a),LL(0xe9ba917e,0x6e30c75f)}, {LL(0x4a322ede,0x763961fc),LL(0x2324bb92,0x6df4a3cb),LL(0x8f2ac967,0x9fe82323),LL(0x2345372a,0x3c372afe)}},
   {{LL(0x50b66fec,0xbf7e9c55),LL(0x0c065cfa,0x5db7dd71),LL(0x50d459ea,0x3525e310),LL(0x8122941a,0xad7abe5a)}, {LL(0x122d92fa,0xc7aeba80),LL(0xefcc1c24,0x066c3765),LL(0x8ffd71b1,0xa6d767ca),LL(0x9cc16dbc,0x4a75fab5)}},
   {{LL(0xbb58b63d,0x9acf8b89),LL(0x6fc8c090,0x226cdcd3),LL(0xae7fbd0b,0x852965b7),LL(0xb8bfe65f,0x4cadd176)}, {LL(0xcfa2ac11,0x4ccc11d1),LL(0x800319ab,0x8abf7420),LL(0x88bb3ef1,0x24ab82cb),LL(0x524c0ce1,0x4d3db003)}},
   {{LL(0x3a431e8c,0x38483641),LL(0x792848ad,0xfc0c04a0),LL(0xa07701b0,0x2fc52bb8),LL(0xf29c72cb,0xdfdced3d)}, {LL(0x5280c2e0,0x677e3d84),LL(0xe98cbec5,0x2dda1451),LL(0xaec26be2,0xba28b181),LL(0x5ddea39b,0x16694717)}},
   {{LL(0x4b9aa9b0,0x911ec5f0),LL(0x3594ae7d,0x24b9aaa0),LL(0xc3c136a0,0x0ccfa661),LL(0xb7474319,0x5518964d)}, {LL(0x2175c3dc,0xf0b0427b),LL(0x966b7bad,0x08db4cfc),LL(0x5e888ad1,0x6f61428a),LL(0x57b52d37,0xfaa96176)}},
   {{LL(0x10aac879,0xe834013b),LL(0x95a62172,0x73397bb0),LL(0x33a244b2,0x97806839),LL(0xc3bec0d0,0x0ab3806c)}, {LL(0x2a72512d,0x4fc7a859),LL(0x0a4228b9,0x96474939),LL(0x4de4b4a5,0x8e5d79a8),LL(0x05d62667,0x5a60d1b0)}},
   {{LL(0x08d90c20,0xd31be21d),LL(0xcc14dbb1,0x3f7ed5f2),LL(0xd7d431c4,0xdc8f58f9),LL(0x82b5c63f,0x714f6dee)}, {LL(0x76d2de93,0x6b285466),LL(0xc39dd98c,0x3c2f5d8f),LL(0xea3760a2,0x9bba0075),LL(0x2411742e,0x75e0389a)}},
   {{LL(0x7ffdb955,0x87d6715a),LL(0x9efb199d,0x702108fc),LL(0x6c010f8a,0xf11db1f9),LL(0x7eb6871b,0xf52b1e0f)}, {LL(0x97c3ed9e,0xc49c0dc7),LL(0x77220a50,0x18846f95),LL(0x97afddcb,0xdb2273bc),LL(0xcc469f75,0x5b9a16d6)}},
   {{LL(0x3beedaf4,0xee364394),LL(0x528a9239,0x825e01d6),LL(0xffd0f17c,0xb60ba965),LL(0xb888384b,0xc00106b0)}, {LL(0x31751f74,0x6e24322f),LL(0x1821d05a,0xfe4d074c),LL(0xbf072932,0xf2493c73),LL(0x21089f21,0xa797e208)}},
   {{LL(0x2988abcd,0xf1b318af),LL(0x8e7da518,0xf887558f),LL(0x97836b57,0xb8b9939c),LL(0xc0a74cf3,0xf793e3b5)}, {LL(0x37684170,0xe191008a),LL(0x05cb453c,0x7708823b),LL(0x361beb2c,0xec221d40),LL(0xeb1b68f4,0x0e5a6cce)}},
   {{LL(0x9644e937,0x3dc535f0),LL(0xfda6c1b7,0xf506d720),LL(0xf99437bd,0xc78c0e0b),LL(0xcc9e2b09,0xa920b4d3)}, {LL(0xf089b0e0,0x550965fe),LL(0x0109d910,0xf9813492),LL(0x8c9d5d83,0xd2496f20),LL(0x3e3e661f,0x751b6900)}},
   {{LL(0x9e6ac190,0x921edbde),LL(0xf02d0e7a,0x75891359),LL(0x1c4da092,0xdeb0f83b),LL(0x4feb2375,0x7b427915)}, {LL(0x7c3a85c3,0x24637c72),LL(0x3f214ac3,0xbbfabf86),LL(0xae22fbfa,0xe8765740),LL(0x5f14045a,0x3a09fab0)}},
   {{LL(0x8190dd41,0x546d574f),LL(0x48b5a39f,0xdfcf0b73),LL(0x74097b2d,0xf26c69de),LL(0x3a7e3e90,0x37aa27ff)}, {LL(0x83bbe3df,0x0942447b),LL(0x9ab378aa,0xe779fe20),LL(0x91e2264f,0xad18ad23),LL(0xaaabd6d1,0xe1dad926)}},
   {{LL(0x5db5e5c8,0x9de0aa4f),LL(0xdb67e44e,0x45c3d73e),LL(0x5cd83936,0x440862a1),LL(0xffce9a79,0x9f2b9a88)}, {LL(0x76cc6265,0x63299069),LL(0x7a8830f5,0xf596a67f),LL(0x8d1d8284,0x7051c842),LL(0x3e5561fc,0xa00d05a8)}},
   {{LL(0x7a34d5bc,0x15ce42d5),LL(0xb0e37254,0x4d9b3f5f),LL(0x38841ab4,0x26e84094),LL(0x9a8ede27,0xa7afd35d)}, {LL(0x14835fa9,0x4e8bcdb8),LL(0x79493e39,0x85d04ddc),LL(0xdf8f65ae,0xbfa8fa79),LL(0xda6c7c62,0xe31d759a)}},
   {{LL(0x7600aea7,0x76f27e70),LL(0xd4d9acf5,0xbec79f15),LL(0x5eae2ff6,0x0f10bd0f),LL(0x7116a0c4,0x96c9eef1)}, {LL(0x0cb6f595,0x30add2cc),LL(0x943efe90,0x0c70b548),LL(0x0a05f4a8,0x2ce8026f),LL(0xb7c53c00,0xaa3da153)}},
   {{LL(0x8e385a26,0xcc46bf67),LL(0x99bae0f6,0x64bcf06e),LL(0x035dcb4c,0x49480a36),LL(0xe3cbae58,0x2cc1a299)}, {LL(0xb5480cb2,0x849f8633),LL(0x5607d83e,0x1d8fa56d),LL(0xcea9f22b,0xcc3f0eee),LL(0x1a23f3da,0x7d5ece29)}},
   {{LL(0x8ae66f00,0xc6f0a006),LL(0x78d648f0,0x2620157e),LL(0x40d2880f,0xfc717762),LL(0xbe105017,0x2e0e293c)}, {LL(0x854116f4,0xb320f214),LL(0x2d5cd4ec,0x5e4fa700),LL(0xdffc1c55,0x83fa0a23),LL(0xc9a9ca15,0x18fcb8d2)}},
   {{LL(0xd0ac70fe,0x9e9baccd),LL(0x27fe06ce,0x8ba02fb7),LL(0x3868fdd4,0x2708804c),LL(0x9ba83df4,0x355eaf0c)}, {LL(0xbe43993a,0x014089ba),LL(0x469cccd6,0xc8b59eda),LL(0xb893a5a7,0x77c94507),LL(0x8e517fd5,0x0dffd39b)}},
   {{LL(0x13dbeadf,0x71b6edb7),LL(0xfea2d0cb,0x1617b77f),LL(0x48ff989f,0xf7454736),LL(0xb618bfa3,0x27357890)}, {LL(0xa7181331,0xf08c70ac),LL(0xb8bc036d,0x33b6cfe5),LL(0x7163f883,0x75ed10f9),LL(0x47d1cbbd,0x979875fc)}},
   {{LL(0x7ad23938,0x6644b234),LL(0xd82e2bc8,0x0f09e7f1),LL(0x5588a39d,0x1e6c512b),LL(0xce8eae85,0xb44e6694)}, {LL(0xf392a4c7,0x107336e2),LL(0xdbcd7b43,0x2619b284),LL(0xb7f476a5,0x7b7ec516),LL(0x43081af2,0x0de74ef3)}},
   {{LL(0xd8d54faf,0x93d08bc6),LL(0xf2ae6c9f,0x88d343a7),LL(0xbc147c27,0x7cdb9003),LL(0x69248562,0xd740b19d)}, {LL(0x464b3b60,0x7f3c48bb),LL(0xc91d92c1,0xfc4cd7e9),LL(0xd7420ac9,0x8172af80),LL(0xb9a50be9,0x66907b77)}},
   {{LL(0x9ec8e974,0xed99fea1),LL(0x54f39b1c,0x624a8c94),LL(0xce9798d1,0x9c4d608a),LL(0xa4812277,0x81e1652e)}, {LL(0xf58b7db8,0xa2cf7509),LL(0x745e450e,0xef2cd193),LL(0x9d9da493,0x48ee8431),LL(0xb8ce96fd,0x7b471698)}},
   {{LL(0xe7553998,0x14dbaff8),LL(0x822de823,0xb0b14e4a),LL(0x429d7c51,0x11032354),LL(0xd572d20e,0xc1bb3327)}, {LL(0x6a9c189e,0xff473811),LL(0x9c7b3b83,0x7cf2354e),LL(0x7662df92,0x29681ff6),LL(0x51c297d1,0x09296227)}},
   {{LL(0x5e3da635,0x1b800b34),LL(0x745116e4,0xb5fd32d2),LL(0x2565abb0,0xdae17a1f),LL(0x1fec80c2,0x4f39d3d7)}, {LL(0x290c2f4b,0xb4a19cc2),LL(0x0b6e5ae0,0x1a1b049e),LL(0x6a823b6b,0x41be6e92),LL(0x969649ce,0x35648873)}},
   {{LL(0x2a8ed3d7,0xe85f995e),LL(0x2f319e47,0x9dc712e8),LL(0x536d98a2,0xc4402eff),LL(0x37521e35,0xca61e310)}, {LL(0xc3196672,0xfed39621),LL(0xff17e8a7,0x29e7743f),LL(0x412a7c49,0x47eca488),LL(0x33a2a6da,0xf0114513)}},
},
/* digit=4 base_pwr=2^28 */
{
   {{LL(0xe6880b5f,0x5675a12a),LL(0xe2606d25,0x9ba1e92c),LL(0xeb3b2125,0xb012facb),LL(0xc37b0099,0x3c50fdfb)}, {LL(0x9ce223e9,0xc9ce461c),LL(0xeefbd8ac,0xcb90bdd6),LL(0xc631ea8e,0xf657e5a4),LL(0x38a83ff6,0x6584520b)}},
   {{LL(0x635abcf0,0xd959f317),LL(0x99e17618,0xa516a43f),LL(0xce3bd99b,0xed90ccf2),LL(0xa9fb3290,0x2fc6d460)}, {LL(0x0cde4302,0xb61ebe09),LL(0xf908003b,0x5a3b061f),LL(0xf60f5787,0xf51bb736),LL(0x057efc2f,0x1717f6e9)}},
   {{LL(0x1ca260ef,0x565acf93),LL(0xf1811d23,0x7d6e797d),LL(0x783e42c8,0xe63c6920),LL(0x8dcb5158,0xdc9dbce8)}, {LL(0xc8e39022,0x1426dc7a),LL(0x30ebfe47,0xf3037f34),LL(0xf87d6395,0x75aa6845),LL(0x61f53539,0xbf792fd5)}},
   {{LL(0x6ddc3d83,0xa8bf2172),LL(0xd88207bb,0xf68deb6e),LL(0xcd03bd7e,0xa8eae2eb),LL(0x951f59a4,0x64c7f57e)}, {LL(0xa1786d57,0x8badb223),LL(0x71182790,0x2e7fda60),LL(0x9a5a9457,0x9dc90e36),LL(0xf4b07e07,0x6eca838b)}},
   {{LL(0x03264871,0xad2e235b),LL(0xb8b933de,0xb4c56243),LL(0x91354c8e,0xd9c2bdda),LL(0x6a73fc76,0x97d743ff)}, {LL(0xce88013e,0xbed4109d),LL(0xf3b3bf4f,0xa2428275),LL(0x011e761c,0x900d3560),LL(0xe24fd6c2,0x34925d7d)}},
   {{LL(0xa8198235,0x08b966ca),LL(0xed2d764a,0x355d098c),LL(0xa3d63f3a,0xfac27f7c),LL(0xd3edc140,0x3e553f6c)}, {LL(0x11ff4334,0x64d72c7f),LL(0xbc62cb57,0x48735aab),LL(0xeba21082,0xcf064294),LL(0xbb8d96fd,0xc1f9e456)}},
   {{LL(0x293cd945,0x1d24bdbc),LL(0xea254e36,0x76985bcb),LL(0x876fb485,0x3df2cb6a),LL(0xcd1f673d,0x0176969f)}, {LL(0x642133a7,0x8b41cacb),LL(0x373880e2,0x31ea88f8),LL(0xb3b1463f,0xccf1ff85),LL(0xaca74a27,0x88fffa15)}},
   {{LL(0x167cdd1f,0x9a4b9b92),LL(0xf879b894,0xa9118fc0),LL(0xc55479f5,0xf6e73387),LL(0xc626d292,0xfadf82ed)}, {LL(0x56e80e6a,0xa03bb761),LL(0xf27555d1,0x59a783f9),LL(0x3d087e43,0x027d63b6),LL(0x02fdeded,0x29f9ff32)}},
   {{LL(0x371d0ec5,0x88a9173d),LL(0x08c0227a,0x04ac4d0d),LL(0x9c7ec715,0x00213011),LL(0xd9d6b472,0x0d2b7c76)}, {LL(0x5050bdff,0xe678d53a),LL(0x65a5fcd5,0x8f929d57),LL(0x1dc3e712,0x0793920b),LL(0x4b073699,0x9a6a690f)}},
   {{LL(0x758bdc9d,0x329d9a81),LL(0x7d867b66,0xebbaadd9),LL(0xe6025f68,0x0d7e6b19),LL(0xc53dce26,0x50184374)}, {LL(0x3ed13916,0x298cb00f),LL(0xf5d45b26,0x835fe31e),LL(0xf5a7fb7a,0x373a9c49),LL(0x34d3d8a8,0x59ed7e23)}},
   {{LL(0x3baf6fa3,0x1a8dfe33),LL(0xda53714f,0x926ccce7),LL(0x18ef6fe2,0xda4feaed),LL(0xc3ca5cdd,0xeddaf090)}, {LL(0xbfe06d45,0xc39c2046),LL(0x0d7f549f,0x1d9e889e),LL(0x8d537d0a,0x209ace73),LL(0x0e31e1ce,0x6f182c88)}},
   {{LL(0x1b8c82e2,0x865e0761),LL(0xa659f2ab,0xcf11bcb9),LL(0x7c868143,0x1804bbeb),LL(0x453e36eb,0x2fa89a0e)}, {LL(0x2e17bad1,0x42d69d8f),LL(0xdc2ec741,0xe7fcea6f),LL(0x379ceb37,0xe7f19b45),LL(0x49bb35a0,0x84f0bd89)}},
   {{LL(0x5264b33d,0xa8a50678),LL(0xab1c9e26,0x8cfae763),LL(0xff9b931a,0x1e837dc3),LL(0x796ac029,0x76164be8)}, {LL(0x1266db27,0x26a8bb2b),LL(0x54822255,0xfba4ab83),LL(0x38524458,0x7a5adcfd),LL(0x44ee212c,0xa056c882)}},
   {{LL(0x55018577,0xe8db6fee),LL(0x91955960,0xf71256b6),LL(0x10abe8d8,0xeb1c118e),LL(0xd45a8426,0x984efc9f)}, {LL(0x00f2c6ed,0x4e1b323a),LL(0x331baae2,0x1759a7af),LL(0x2e00ba6c,0xf1587189),LL(0xbb385d39,0xbd8a877e)}},
   {{LL(0x57d6c1ae,0x440d1eae),LL(0xa957dc67,0x092abdef),LL(0x74554b3f,0x1065cbc6),LL(0x710566c7,0x67062382)}, {LL(0x6d04ae2b,0xd327679d),LL(0xb0340551,0x11507b00),LL(0xa2f52d80,0x2e571583),LL(0xe8578507,0x673628f4)}},
   {{LL(0x0cf4efe5,0xecb8f92d),LL(0x960e2d22,0x88c47214),LL(0x6059f079,0xca9549ef),LL(0x7016da7c,0xd0a3774a)}, {LL(0x1d001cab,0xd51c95f6),LL(0xa3feeec1,0x2d744def),LL(0x0afedf2b,0xb7c20cc2),LL(0x71d144a5,0xbf16c5f1)}},
   {{LL(0x3dc0d12e,0x00384727),LL(0xb01cc80f,0xaa95f450),LL(0xa6f8e927,0x19be3106),LL(0x0417ba8b,0x6d6e10aa)}, {LL(0x870e3491,0x149f120c),LL(0x026dde94,0x27380b41),LL(0xf29b04e6,0x97d00840),LL(0x4bf9eb19,0x21d5d7e3)}},
   {{LL(0xd5327f05,0xea1daad9),LL(0x9c88c17c,0xf1f45d94),LL(0x3f8ee0ab,0xc5f3dee2),LL(0x75238a56,0x706b777c)}, {LL(0xf834c60b,0xf7aee379),LL(0x13cfe17b,0x5c24dae6),LL(0x8091804b,0x354c82e5),LL(0x102a577b,0x0dec2fdf)}},
   {{LL(0x5253f8fc,0xbf3b7030),LL(0xd913c01c,0xe516fa69),LL(0xa105ba64,0x053afef4),LL(0xc89c1e76,0x91a1f36c)}, {LL(0x7e724e18,0x3375865c),LL(0x29327b2b,0x43132144),LL(0x6f7bb24e,0x9cb2fc3b),LL(0x6319e789,0x20a6a16d)}},
   {{LL(0x642c467a,0x20bfbd77),LL(0x259d50c8,0x3452bb12),LL(0xec7ffab2,0x0d3ba9c7),LL(0x3560e541,0xbbdb5454)}, {LL(0xd63ba04b,0xab1d6e22),LL(0x7d24f015,0xdf6f11d3),LL(0xf3df15fa,0x7c4d61d2),LL(0x40b3288c,0xd5269f79)}},
   {{LL(0x0e7c7b6c,0xf8516b9e),LL(0xc203dac8,0x48750d82),LL(0xa13d3083,0x89845d36),LL(0x280a131a,0xb3db3cfa)}, {LL(0xfbf752e6,0x40045401),LL(0x1432e856,0x0289f97b),LL(0x5fc1aa11,0x41a9f371),LL(0x8d464042,0xe5c1e5a5)}},
   {{LL(0x589b71a7,0xfbee2ea2),LL(0x5de7056c,0xdd6ee5bd),LL(0x8fd6b6de,0xcf8a4541),LL(0xb15e33b1,0xb47831dc)}, {LL(0x2064321e,0x126a2169),LL(0x6e517edd,0xa21d2d22),LL(0x5ba5a30b,0x1f8072be),LL(0xc6a24b7d,0x24cca576)}},
   {{LL(0x5c282027,0x57eab82f),LL(0x557344b3,0x1620f5e6),LL(0x460b3385,0x59e852e0),LL(0xf050816c,0xc906e3db)}, {LL(0x3eb398e8,0xc031f8cf),LL(0x507ac07f,0x9c25b69b),LL(0x9cf7bdbe,0x652baf2b),LL(0x5ad91107,0x06fedc53)}},
   {{LL(0x4c4b12c5,0xa8ca0be2),LL(0x28762d5d,0x633292b6),LL(0x827c0d5e,0xc04983f2),LL(0xc707ef03,0xcb6b867d)}, {LL(0xb9ac1124,0xa7fc0d5b),LL(0xaab7dcaf,0xa5ce085b),LL(0x1cfda998,0xb85e8f1c),LL(0x27822503,0x8208df42)}},
   {{LL(0xa8dd6d76,0xeaa82320),LL(0xad36eb73,0x7b2fb4ae),LL(0x97a7b040,0x24d73191),LL(0x4001e02f,0xc3ff64ae)}, {LL(0x88799d94,0xd5d87157),LL(0x93ceb95a,0x559142d0),LL(0x59c3009a,0x798a453c),LL(0x7d6c83a2,0x546b6fab)}},
   {{LL(0x5c76029a,0xe263b23a),LL(0x4ac62973,0x85630532),LL(0xecb007ac,0x14ee0643),LL(0x7ca60905,0xf9e06297)}, {LL(0x92f1f170,0x21b2fb23),LL(0x46528ab2,0x31c40918),LL(0x3395cfd2,0x43b53242),LL(0x6d14fb40,0x4042138f)}},
   {{LL(0x4464f342,0x80899c8c),LL(0x084be305,0x0f54c993),LL(0xfbf84810,0xfacecac3),LL(0x8ae5244f,0xa6585936)}, {LL(0x9a9f8d4a,0xb467c3c0),LL(0xfd394895,0x3e5f219c),LL(0x9bf85fa8,0x39f0767a),LL(0xd8ee6022,0xd97cc55d)}},
   {{LL(0xc83f86c4,0xc480938f),LL(0xe43bfcc6,0x6479b8ef),LL(0x38cabad7,0x8e6f2e22),LL(0x31f8c6aa,0x48e57fdd)}, {LL(0xcfbbdcac,0x66dd6a77),LL(0x50ece329,0xc7d9950b),LL(0x0747a937,0x2e31f205),LL(0xa07acb8a,0xc0f8a7e2)}},
   {{LL(0x15eaa686,0x578477bd),LL(0xf2f58b50,0xd72fb935),LL(0xd3a64d22,0xe9fdbc6f),LL(0x492dc89f,0xa3e42674)}, {LL(0xa8fb7d24,0x42410ffd),LL(0x52676ed7,0x08a37dfd),LL(0xcb5d6125,0x4607c41b),LL(0x4001fa42,0x7db48af8)}},
   {{LL(0x50cd30f0,0xe2264eb1),LL(0xe215f8d7,0xbb6fe952),LL(0x97e3fe73,0xf3ce2411),LL(0x37f19247,0xe52e2179)}, {LL(0x20c233c1,0x9c7fc8c0),LL(0xb383b101,0x91c7e721),LL(0xa7ac883f,0x1163c472),LL(0x9d3b0f1e,0xbe1c3b3a)}},
   {{LL(0xa3536baf,0x07be716f),LL(0x62e9c19a,0x764d9f4e),LL(0x8eaf19f4,0x15af3499),LL(0x38ea0ace,0x987a7c47)}, {LL(0x4a1f0117,0xb03740b8),LL(0xfe098a9f,0x5cd1164f),LL(0xc9d6fee5,0xaf952cef),LL(0x3c0ad28b,0x4e86dcbb)}},
   {{LL(0x677b7a8f,0x81125450),LL(0xe69273d2,0xba889fce),LL(0x582c5990,0x4a40a859),LL(0xf48934c3,0x836638b3)}, {LL(0xf3596ba6,0xe964e189),LL(0xde8b0754,0x2f417c0e),LL(0xd5f93f1b,0xd883169f),LL(0xd45bb389,0x0318fe4e)}},
   {{LL(0xdb03273e,0xe2c998a1),LL(0x33ec151b,0xc34f544d),LL(0xeb92d963,0xae0456b1),LL(0x9738857f,0xaab61ec4)}, {LL(0xf71d9c39,0x4fb6a34e),LL(0xd816ec44,0xaa9dbd8c),LL(0x5efdf950,0xf6532e37),LL(0x67109c55,0x7151dc44)}},
   {{LL(0x3f4e322c,0xb18b586a),LL(0x3553a18b,0x27b30066),LL(0x1ae4cd85,0xbd31ea24),LL(0xa64de69a,0xe8f88f4a)}, {LL(0x609c13bb,0x8c946a97),LL(0x0eebd9f0,0xbf8cc55a),LL(0x7a8892b9,0x446aa2e4),LL(0x65b98c31,0x660c0a55)}},
   {{LL(0xd3463522,0x568c56fc),LL(0xeb130aa5,0xfa6bf3a6),LL(0x008dc0da,0x16c1568b),LL(0xfed70a87,0x9c4132cc)}, {LL(0xd497fdff,0x3e983d09),LL(0xf0ebe6b0,0xd7a0e542),LL(0x68b542ca,0x193a07e0),LL(0x07c6ab4f,0x4909776b)}},
   {{LL(0x418acd7b,0x55b77ef4),LL(0x47a77d32,0x64ba62d3),LL(0x2d1f562e,0xaec1aa93),LL(0x10dc5999,0x3468725b)}, {LL(0x6ff0d478,0x422851b1),LL(0x8e7dddcc,0x15da8429),LL(0xb8ac5238,0x38567920),LL(0x2e3344d8,0xfd29eb4a)}},
   {{LL(0x4fc636b5,0x7b2af70c),LL(0x879e7640,0x242acfc8),LL(0xb5e25c7b,0x88e89786),LL(0x16ec1bfd,0x85576b1b)}, {LL(0x1891e595,0xb31c8253),LL(0xca5608a5,0x14315dfe),LL(0xb0c14fd9,0xb9d61b76),LL(0x734b6cab,0x5d5ad8a3)}},
   {{LL(0x44aee005,0xc2ea321d),LL(0x147ed658,0xd68abd2c),LL(0x893db877,0x31152d60),LL(0x281487b6,0x4807ac46)}, {LL(0x65da04b5,0x58ebd15e),LL(0xb2f9d1fd,0xf0f74fd4),LL(0x393c7d91,0x3d04aa65),LL(0x8e7e6a2c,0xb46fb59a)}},
   {{LL(0xae1eed5d,0x9236fdf1),LL(0x7810e2be,0x71936f56),LL(0x6d9ff147,0xa1ead7d5),LL(0x149a9b6d,0x32670ed8)}, {LL(0xcb58ea59,0x12772fdd),LL(0x9df52ddb,0xfce260b3),LL(0xccab1e97,0x3221f2fb),LL(0x57762484,0xf8ff7e37)}},
   {{LL(0x855512cf,0xb0a31a1c),LL(0xd71d4c4e,0x293a819e),LL(0xcd6a900f,0xc1ebc896),LL(0x9b9e0a4b,0xc727a646)}, {LL(0x0018f29f,0x06124fc0),LL(0x41b7730c,0x67bd8fed),LL(0xc77be72e,0xeeebf0f0),LL(0x474d747a,0x427fe6fe)}},
   {{LL(0x932ccbf0,0xa7fb9a4f),LL(0x5f3d489f,0xabb9c85e),LL(0xbdf26442,0xe7e4f956),LL(0x38d17422,0xd014848e)}, {LL(0xd3e9bff6,0xae37d855),LL(0xca5aeb09,0x88fbae1d),LL(0xf025feaa,0x1a8a740b),LL(0xb9475ebb,0xc1a67821)}},
   {{LL(0xde2bf8a2,0xb6cb6acc),LL(0x66a0f14e,0x9b2ab1ca),LL(0x83b2ba59,0xcbfbc068),LL(0x68447934,0x336ab62c)}, {LL(0xf19719b8,0xd3a016a9),LL(0x0b5b9d6e,0x819a31bb),LL(0x3e1c6c0b,0x7b24be2b),LL(0x013f6821,0x10834b4a)}},
   {{LL(0x86f21d2c,0xe5e5de27),LL(0xe9e35ad5,0x56b46a2d),LL(0xe2111e59,0xfc4e861d),LL(0x6e37ca63,0x7472ce5e)}, {LL(0x27d2210e,0xafab9a71),LL(0x9ff6245a,0x1644a0a6),LL(0x8dbef51f,0xee498acb),LL(0x2e9604d3,0xd4c70da1)}},
   {{LL(0x6fecb64c,0xde487364),LL(0xd15fb62f,0xa8fda1fd),LL(0x088de028,0x97e2febe),LL(0xecdce095,0x4a769019)}, {LL(0x50a58ddb,0x4cb6a338),LL(0x17028d36,0x08df59d8),LL(0xb51722b7,0xfe3a80ff),LL(0x963c2383,0xa3cc2fe2)}},
   {{LL(0x53cc5341,0x40b2df49),LL(0xa3c4bf2f,0xf3e90d4c),LL(0x20f02731,0x3f25c5ec),LL(0x69065d9a,0xd84f5b5a)}, {LL(0x129921be,0x156d350e),LL(0x1b116922,0xe98787cc),LL(0x39e77b13,0xba5f9b82),LL(0x044449a5,0xee4d79f5)}},
   {{LL(0xdd6d852d,0xb54b7388),LL(0xf9ca5fdf,0xf7554c5c),LL(0x51228a81,0x864d1fbf),LL(0x9a80f90b,0x721e1add)}, {LL(0xad0efa62,0x89d4e297),LL(0x6dba9404,0x4e471a87),LL(0x1c1008b0,0x9a38158b),LL(0x95c47ec2,0x3dfe81a7)}},
   {{LL(0x28603026,0xcb02ce9b),LL(0x3bd357fc,0xfd3207aa),LL(0xf296f5f2,0xb3807bdd),LL(0x23c2ea7e,0x7895918d)}, {LL(0x88feb3ba,0xdc0eb62f),LL(0xbdd75674,0x024dfd84),LL(0x0a1e0496,0xe5bd3828),LL(0x24c8f30c,0xb8b1cd86)}},
   {{LL(0x674d10cf,0xb559e34d),LL(0x9f962ec5,0x6955bb69),LL(0x542af42d,0x8bf1ab6c),LL(0xdfa61256,0x3f2f33fa)}, {LL(0x73d1049e,0x32140195),LL(0xdfd7f39b,0xf5089278),LL(0xb4237be0,0xb42eb51c),LL(0x874d0e57,0xdf747f44)}},
   {{LL(0x77b5d475,0xbe64bb22),LL(0x28308634,0x2c3d5ecb),LL(0xcb999c46,0x936a2987),LL(0xe26489ea,0x5a30ddfa)}, {LL(0xc8eabf9c,0x8bfc782e),LL(0x74c8c6e3,0xb9995bb0),LL(0x391f5c5a,0x4f99c7ac),LL(0x5270c4ad,0x67f4092b)}},
   {{LL(0xe6e8135e,0x6771a29d),LL(0x6c698cec,0x988dfb2b),LL(0x77812aa1,0x7818600f),LL(0xfd98e1c1,0x04393c83)}, {LL(0x864ef146,0xe448232e),LL(0xa465ab71,0x9b70ecf4),LL(0xb13cc704,0x31df0531),LL(0x16e48426,0x401ae0b3)}},
   {{LL(0x7fc514ed,0xa8106630),LL(0xde4b1614,0xda798170),LL(0xc2c684fe,0xde892efc),LL(0x05d64eff,0xd5205bc1)}, {LL(0xe1d59ba5,0x84df4ead),LL(0x89bb2ea7,0x65245ca1),LL(0x64edbf51,0x3de6ca34),LL(0x56bcebf9,0x115296e4)}},
   {{LL(0x7fd52a3d,0x0851631f),LL(0x949ad4be,0x9881db71),LL(0x88caf772,0x4b2337dd),LL(0x33ec7979,0x02da59de)}, {LL(0xafe840df,0x2473c620),LL(0xa92ef1d0,0x2965ebff),LL(0x6fcd9651,0x2452854f),LL(0xbac2ed99,0x97092935)}},
   {{LL(0x08242246,0xf0743ce7),LL(0x6d1a8439,0x76fdd82c),LL(0x61079258,0x3627c890),LL(0x82b21983,0x312f98f1)}, {LL(0xe9173891,0xd87dceec),LL(0xd7a30e32,0xad16cfe0),LL(0xc9c7efaf,0xc404a1a6),LL(0xd6df357e,0x46e34671)}},
   {{LL(0x7a02aa32,0x92fec7c4),LL(0x5a6a7bb9,0x567fef7e),LL(0xa3f97b5d,0x35fd570c),LL(0x4a4b0dfa,0x456bad8c)}, {LL(0xa677f090,0x85a3f42c),LL(0x22a68d53,0x35060bb8),LL(0x53567530,0x1cea9d11),LL(0x8169fbce,0xf2cbc8dd)}},
   {{LL(0x86cde794,0xa3e1d52d),LL(0xb3bdf344,0x72a258cb),LL(0x31b8614d,0x2997cd59),LL(0x8164b632,0x31ce2ea4)}, {LL(0x0eba7545,0xe495e9b7),LL(0xbc4403b5,0xaad69130),LL(0x45760d9b,0x37f6389b),LL(0xb871b17d,0x00f4d58d)}},
   {{LL(0x4aa359d7,0x91973d4d),LL(0xc8dd0582,0x249f510c),LL(0x7608be27,0xef11ac87),LL(0xd940b1c7,0xce116714)}, {LL(0xfef20037,0xf34881f3),LL(0x98203f4e,0x26222472),LL(0xe9363154,0x4c9e98ed),LL(0x03a8158d,0xa806b3a6)}},
   {{LL(0x09d16ce3,0xdd974d66),LL(0x59ae977a,0xe1bcc513),LL(0x218464d6,0x0e6201c7),LL(0x9e35c7af,0x05789811)}, {LL(0x8b33a863,0xb1596f7a),LL(0x42bd8284,0x8fa93aeb),LL(0x46e11559,0xf197c202),LL(0xadd27d86,0x356b9c81)}},
   {{LL(0x1695cb70,0x3c4080fd),LL(0x20f20318,0xc10c28cc),LL(0xce1ffab9,0xe9d7ed93),LL(0x4f9de9bd,0xb23976b3)}, {LL(0x6d61a6f2,0x9b1b81dd),LL(0xf6318874,0x7537d729),LL(0x20cee7ab,0xb75022f4),LL(0xaa430952,0x425fddba)}},
   {{LL(0x1ccfb3fd,0x54c4306d),LL(0x46a30a37,0xf10a54f1),LL(0x74fd4925,0x2d332a29),LL(0x1438feb2,0x8d2fa921)}, {LL(0xfbb41bd2,0x46a9c6b5),LL(0xd30c65fd,0x87e98550),LL(0x66cd9a20,0xfbcb2ca6),LL(0x91719670,0xc176530e)}},
   {{LL(0xcec38056,0xdd4a1a18),LL(0x75544998,0xe6ef1793),LL(0x30583fb6,0xf58f69cf),LL(0xaa76bf2b,0x12197860)}, {LL(0x3bb686fc,0x717813e5),LL(0x0f06c403,0x9beeb1ae),LL(0x2782dc86,0xd83416ee),LL(0xb5500ccc,0x5fc89c01)}},
   {{LL(0x8e110ed3,0x063aee25),LL(0x45963073,0x1a87377c),LL(0x5110634f,0x86944f59),LL(0x0ba76459,0x50659ae1)}, {LL(0xde9eb40d,0xa00e48ff),LL(0xfe5b118c,0x49235afa),LL(0xc425ee38,0x81705008),LL(0x2d5f2f92,0x3c01abc8)}},
   {{LL(0x6fdf148b,0x4a21bc95),LL(0x0b7e6871,0xea5cc30e),LL(0x11713844,0x90b4abb6),LL(0xf2001af9,0x3b7d734f)}, {LL(0x782b2020,0xfc616b89),LL(0xd8b0e02d,0x68b3935c),LL(0x1cbb2de4,0x54cf5b8c),LL(0xa7c0f7ac,0x42b0432a)}},
   {{LL(0xbdffae5e,0xa04e06ef),LL(0xd4b636ee,0x36cac28e),LL(0xc3a98127,0x08a06b2f),LL(0x290c5385,0x1ef0b57b)}, {LL(0x27154c46,0x14e184b8),LL(0x60910b3e,0xa5dd3444),LL(0x3c67a74b,0xd0008ac4),LL(0xefed9fd1,0x2649cba4)}},
   {{LL(0xf0c1bb4f,0x26bc537a),LL(0xd06b90f5,0x37f376ff),LL(0xbe7c89cf,0x4d48d994),LL(0x8d572003,0x511c2158)}, {LL(0x088dda1e,0xc26fbac1),LL(0x7ad4934c,0xc3d55189),LL(0x85dcaf7c,0x5233c176),LL(0xa88b473d,0xec3a8a29)}},
},
/* digit=5 base_pwr=2^35 */
{
   {{LL(0xb1f0c175,0xfd96667a),LL(0x2ab99e7d,0xa256a611),LL(0x05e43f9d,0xff07c1ea),LL(0x7e1c9cd6,0x305700bc)}, {LL(0x2b2887a3,0x3f1e2546),LL(0xc772fd14,0xdd782f49),LL(0x38584057,0x9125f996),LL(0x16a02cf9,0x19fd0396)}},
   {{LL(0x4c58174d,0xa8d62bd3),LL(0xa900551f,0x872251d3),LL(0xf12802c3,0x06f5862d),LL(0xdd925555,0x5d93c48a)}, {LL(0xbd6006f8,0xc39b67d5),LL(0xf96ccc67,0xea6f756b),LL(0x543014db,0x140e853e),LL(0xe9de42c0,0x2bdc5674)}},
   {{LL(0xe01c073d,0x01dfda7b),LL(0xff9e1234,0x07a6bb65),LL(0x622cee4e,0x2a4f7f18),LL(0x50f0a3a7,0xdf4cead8)}, {LL(0x1b8c2903,0x152b3c8e),LL(0x5f2a89b3,0x9e82e999),LL(0x68ce7a3c,0x0e6cfa7e),LL(0x0ca0464c,0xebb34d90)}},
   {{LL(0xeda49f74,0xfa1a58fa),LL(0x0e4545a3,0xddb89957),LL(0xc74c07e4,0xd3576489),LL(0xb59b1008,0x64e4b39e)}, {LL(0xf66b546c,0x3b090340),LL(0xcdeb912f,0x0e0f4013),LL(0x01e55cca,0xbb00b46c),LL(0x99ad0768,0x55b61b34)}},
   {{LL(0xe8bbda5b,0xb06b71fc),LL(0xa24b0a63,0x8de64d84),LL(0xb5d4603f,0xb73dc262),LL(0x1965a916,0x5d5fa964)}, {LL(0x3bc98966,0xb48a4053),LL(0x6f564743,0xaa871863),LL(0xe76a6a3e,0x88b00822),LL(0xb38d9e0d,0x58c9e92e)}},
   {{LL(0xe989963e,0xc0d22337),LL(0xd3778d5a,0x2c4831ce),LL(0xee8c4178,0xd775c6a5),LL(0x9d0c2894,0xe2391654)}, {LL(0x5d0eb314,0xf7d4fe86),LL(0x8b2290d3,0x42801b8f),LL(0xcdcefa78,0x73e9b332),LL(0x3e877fea,0xc0d169d9)}},
   {{LL(0xffee23fa,0x29c8138b),LL(0xfb92e3b8,0xbff98230),LL(0x8fa75007,0x14077ad5),LL(0x88e61b81,0x4d3a6e10)}, {LL(0x3bcf733d,0x218a867d),LL(0x665e37fc,0x20ff6566),LL(0xda5cbf67,0xe39c0581),LL(0x8add8c4c,0x4a6e1d7c)}},
   {{LL(0x734a1327,0xcab02370),LL(0x1951afa8,0xa1df7afc),LL(0x42638b8a,0x581cfbaf),LL(0x2130eaa6,0x39db6d2b)}, {LL(0xda2f91a5,0x4bbc805b),LL(0xe569add8,0x3dcb0a7e),LL(0xd721fa7d,0x724ab65a),LL(0xf88f8008,0xa5152b95)}},
   {{LL(0x281615ba,0x7fe7f1b9),LL(0x41d5aa0c,0x419d1a53),LL(0x9fb0917e,0xafc556dc),LL(0x616ce893,0xab2a69f3)}, {LL(0xc0861e05,0xfb9a6eb1),LL(0x5eb02b8f,0x0b74ae11),LL(0x3b1e44fe,0xccff0ad5),LL(0x88824f53,0x86dfe0e6)}},
   {{LL(0x41177a46,0xedf38dc4),LL(0x7f039a7b,0xd9a955bb),LL(0x4d8ae7c2,0x4f152581),LL(0x4f848819,0x063c9f83)}, {LL(0x841e8783,0x54ea4526),LL(0xaa5f2b32,0xe86a4119),LL(0x19846dcf,0xb7529a3b),LL(0x35689d70,0x91356a07)}},
   {{LL(0x8f049ef8,0xbe66f5db),LL(0xc38dd5ed,0x0f5fd99e),LL(0x1b4ae7a7,0x1896d52b),LL(0x480b1ebb,0xf27c45c6)}, {LL(0x3fede5c1,0xd88cff4c),LL(0xda27560b,0x57d902c9),LL(0x52d57deb,0x84aa7f07),LL(0x08bb6028,0x8da4c7c8)}},
   {{LL(0x8910763e,0x658f4dea),LL(0x076a0f80,0x6e5fcb48),LL(0xab65f9b9,0x6a5447a4),LL(0xa75bb0c5,0xd7d863d4)}, {LL(0xe87e7916,0x806c34a7),LL(0xcd961e88,0x05391559),LL(0x74fe6aeb,0x5def2d88),LL(0xf9226ca1,0x8ac350b2)}},
   {{LL(0x12401813,0xffa8a649),LL(0x5337c55d,0xd6182762),LL(0x3be902e3,0xfce9d7ff),LL(0xea0dd7a5,0xb3b275d1)}, {LL(0x2cb48ac9,0x342620f4),LL(0xa8b38a74,0xc0369384),LL(0xc0695d3a,0x04b0ee6a),LL(0x94c5394d,0x4d025585)}},
   {{LL(0x81443d16,0xff9635d0),LL(0xa6cc364b,0x2342cbfa),LL(0x25bf8438,0x63b0a032),LL(0xa078d298,0x6ccd3ce5)}, {LL(0x91292fd3,0xf93bd108),LL(0x14073286,0xc887a31b),LL(0x9f62cd16,0xeb1275bf),LL(0x61578b46,0x0335bae3)}},
   {{LL(0x53348e4e,0x810d5efd),LL(0x63c74225,0xf9cd822a),LL(0xa426bf44,0x93d2e810),LL(0x019d36b3,0x95a47a97)}, {LL(0xd5d1f840,0x1da421b9),LL(0xd6c46e3c,0xe5b8a55f),LL(0xc9244881,0x2dd3a5e7),LL(0x70c1fd2f,0xd50f9cde)}},
   {{LL(0x614d9ff3,0xbee2aca7),LL(0x358f245a,0xd1f13b2c),LL(0xc46f62ab,0x9e92d83f),LL(0x827d7374,0xc1dd32dd)}, {LL(0xc3e566e7,0x1636d593),LL(0x04ccb02b,0x81c2f4e7),LL(0xcd35b652,0xb57782c6),LL(0x88210d42,0xad88787e)}},
   {{LL(0xfbd7d35a,0x3ad52d72),LL(0x37a2a095,0x4117f502),LL(0xd356b3b6,0xed03d415),LL(0x15ca6087,0x135d5a8c)}, {LL(0xef5dca2a,0xfbaba41f),LL(0xafb4787d,0x660e5cd0),LL(0xa55e9ef0,0xe0e66378),LL(0x69939f56,0xf24513cf)}},
   {{LL(0xab4f6bd9,0x0f38f09c),LL(0x922dcac1,0xec3037b4),LL(0x08a1a51e,0x706b201a),LL(0x8ffff040,0x15911351)}, {LL(0xccf63d87,0x239d7b6a),LL(0x5187f595,0xeca37dc8),LL(0xad5a0ab3,0x04ea79e4),LL(0xe9520e8f,0xcdd81522)}},
   {{LL(0xc35e1020,0x7fe6b6aa),LL(0x140ac884,0x57b63c9e),LL(0x33f19077,0xc45c23fc),LL(0xb71273c5,0x468d2c36)}, {LL(0xfc305ac2,0xeb6839d6),LL(0x0183793a,0xf6e310ff),LL(0x32da639d,0xbca206e4),LL(0x8518e27e,0x8eb5cac1)}},
   {{LL(0x66ed96f9,0xfeed0feb),LL(0xecc3a8dc,0x1632632e),LL(0x1455c8ae,0x90449363),LL(0x0aeada65,0x8d7619d4)}, {LL(0x9f630ee9,0x2f2fa898),LL(0x370db87c,0xd78caf0c),LL(0xc45898cf,0x46fa0fc9),LL(0x2d84244f,0xa509cc3e)}},
   {{LL(0xa5b099aa,0xbdbea4b4),LL(0x7592587b,0x8e8fe284),LL(0x42000897,0x0226d387),LL(0x36db5cd9,0xb6780551)}, {LL(0xca64f047,0xd8fe5eb1),LL(0xb77cf8cb,0x6f21474b),LL(0xee45ae34,0xab8fcae7),LL(0x1f19cd67,0x73eaf9eb)}},
   {{LL(0xee4df6ce,0x5bb96415),LL(0xa3ae4cf3,0xd1e27bcf),LL(0xc7f1868e,0x9bf7ace3),LL(0x82091dca,0xe821aa8b)}, {LL(0xd381b6c4,0xf732e6bc),LL(0xdd01864f,0x5feda346),LL(0xb6387846,0x0933b92c),LL(0xa0028029,0xbf1f1e83)}},
   {{LL(0xa3e38124,0x0848bf8c),LL(0x208fda8f,0xfe295fdf),LL(0x8913a1c4,0x73379239),LL(0x7e78564e,0x59354b24)}, {LL(0x32dcafbc,0x042b7529),LL(0xfa93c5c7,0x752173d3),LL(0x9737135a,0x6ffd4590),LL(0x0f983005,0x249712b0)}},
   {{LL(0x1f25da8b,0xdbba2874),LL(0x097ba4a9,0x14027f11),LL(0x34b8e4a2,0xe429b3c7),LL(0x056b4afc,0xd66a43e3)}, {LL(0x2ac351e6,0x158644ad),LL(0x164bc6cc,0xff4aecd9),LL(0xf6c615ee,0xbb5b0c87),LL(0xd7679b1b,0xc497d8ee)}},
   {{LL(0xf1c6e97a,0xf666c625),LL(0xc73a277f,0xe89f84b2),LL(0x746af4c0,0x2403d513),LL(0xb7101feb,0xe6858fdf)}, {LL(0x84f1dcb7,0x1a42c51b),LL(0x8202bc04,0xc57f12e0),LL(0x754df5ae,0xf8326a93),LL(0x81a46aef,0x3d3daf04)}},
   {{LL(0x01232d03,0x8bb8c276),LL(0xfb371cf1,0xd446c82e),LL(0xefa495f4,0xe5e8b639),LL(0x477e6493,0x51a7b34a)}, {LL(0x824f2b6e,0xffba5466),LL(0xf0eaa6a9,0xcc67ddad),LL(0xfee19b24,0xcf0f8ce1),LL(0x83b3df41,0x34309127)}},
   {{LL(0x9719a6cd,0xc8b13e8c),LL(0x619d5c33,0xb408e505),LL(0xa3158864,0x8c1b831b),LL(0x0b3d02bb,0x506b3c16)}, {LL(0xbf11ff8d,0xf23846bc),LL(0x16e0328a,0xf0f043e8),LL(0x65986a7a,0x30b7b9cd),LL(0x21b660cd,0x0951b102)}},
   {{LL(0x52bf29a5,0x72a26c5f),LL(0xb6534592,0xb513d669),LL(0x578195ea,0xb8ac15ad),LL(0xc0785f88,0xd6ed33ea)}, {LL(0xb9e33946,0x39e23dbf),LL(0xf43e88eb,0xeadb2453),LL(0x2746c34b,0x6d82fefa),LL(0xcc542b54,0xe9172aa0)}},
   {{LL(0xecb50699,0x8af6b819),LL(0x1c1d0af9,0x4af76939),LL(0x99dddb1a,0x5a7dbbbe),LL(0x891ea41d,0x97b0a3aa)}, {LL(0x6e35ea4f,0x32b457e6),LL(0x9d77b900,0xe2a21c2a),LL(0x2ac991cf,0xb18718d6),LL(0x740743cd,0xc4416237)}},
   {{LL(0x6a05ab55,0xcc3f76b6),LL(0x98091425,0x2ab4b29e),LL(0xb6478fc8,0xbf373ad1),LL(0x178b5844,0x8a1a9489)}, {LL(0x09daf4be,0xb5295edf),LL(0x4ed54766,0x07fbb119),LL(0x7d0b9d8f,0x6e44367b),LL(0xedb96a10,0x6dc4d8f6)}},
   {{LL(0x37fc19a3,0x2ba69106),LL(0x0b138296,0x522eba39),LL(0xfda58cf3,0x751544c7),LL(0x0ba33938,0xaba6fe16)}, {LL(0x94dac7d6,0x48e085be),LL(0x19f99faa,0x06c87014),LL(0x1a587f89,0x33b9a8d6),LL(0x3fd8d8fe,0xdae382ca)}},
   {{LL(0x150b0fcd,0xb5b383c6),LL(0xed9b0f4c,0xf948da80),LL(0xccd05413,0xcf075225),LL(0x4f62be64,0x3f31b12c)}, {LL(0x368c17f6,0x23b21fc8),LL(0x400bc690,0x423d5369),LL(0xdeac140e,0x5335dd1e),LL(0x9493ad61,0xe631c249)}},
   {{LL(0x32fe490a,0xc274c695),LL(0x6d8ebd70,0x42bcb4e1),LL(0x65d7a1d0,0x69059e1e),LL(0x29fdd109,0xf36dfe2f)}, {LL(0x0c4e6370,0xacfea1ec),LL(0x7a935ff4,0x97e7f722),LL(0xf8006bbd,0x83e7c7c3),LL(0x78e6792c,0x87a8b84d)}},
   {{LL(0x94d3d60f,0x5cbe4883),LL(0x91cbc054,0x6eba464d),LL(0x021c38fa,0xf9c880d0),LL(0x21af4942,0x6200faf1)}, {LL(0x5f03e261,0xd5b2b12d),LL(0xf3ea3e07,0x1659a0ac),LL(0x836757a8,0x8008f18d),LL(0x75a8f8e0,0xfb2f467b)}},
   {{LL(0x9c9b00cc,0x9a6183c7),LL(0x3bf842b0,0x82ca07e3),LL(0xee1f83d9,0xe7089191),LL(0x2d0cd2da,0xc41ecde4)}, {LL(0x4d1feacd,0x0ce421b0),LL(0x431c53f1,0xe80a1395),LL(0xe6bfccf5,0xae9b2018),LL(0x8b359c3b,0xdf9f86ad)}},
   {{LL(0xb6170a5f,0x9887e28f),LL(0xf3c0c30c,0xf5b85d21),LL(0x632af7a4,0x30861cf8),LL(0xbb4ec123,0x2fb670ad)}, {LL(0x3c425976,0x0668b84c),LL(0x02883af7,0x55c21b4e),LL(0xf8698d29,0x0fad58b5),LL(0x68b671c5,0xef210770)}},
   {{LL(0x23f232b8,0x534d510a),LL(0x49c99708,0xdb66fec1),LL(0x6d54721b,0xf1a6f3e7),LL(0x4480f858,0x8d37ab64)}, {LL(0xb0f7f354,0x7fcfca6c),LL(0x95bfd318,0x58c7ff5f),LL(0x903f9d91,0x3048e9af),LL(0x75357af0,0xe480bc0e)}},
   {{LL(0xa5a1162e,0x4f915e1c),LL(0x37efa40c,0xdd539c21),LL(0x789201c2,0x61a45c53),LL(0xe7890746,0x1bc2333d)}, {LL(0xbbed8f77,0xeed38f50),LL(0x178501a0,0xc1e93732),LL(0xa8fb8623,0xfed5b1d1),LL(0xdc3e1148,0xa3be3e2c)}},
   {{LL(0xa71a390a,0x62fc1633),LL(0x2891c4c3,0x4be2868e),LL(0x2a0c3c23,0x6573fe49),LL(0xde1589d0,0x182d0bd4)}, {LL(0x5a7aa63d,0x17c6a780),LL(0x12543191,0x9d84cfa8),LL(0x950c85c9,0xcdb22db7),LL(0x119010c4,0xd03589e0)}},
   {{LL(0x8220dee8,0xbcd02e8a),LL(0x705632fd,0xbd4d1f2a),LL(0x22f8e30b,0x00119bfd),LL(0x6eb97c32,0x06c6e73e)}, {LL(0x35abff53,0xa26f0a6a),LL(0x8564c37d,0x7d79a89f),LL(0x1b207495,0x0347bb17),LL(0xb5c8253a,0x1baf90e9)}},
   {{LL(0x37affc96,0x01059b5f),LL(0xffee0a60,0xbe76c578),LL(0x75d6b83c,0x45d7291b),LL(0xe0b58129,0x212ff131)}, {LL(0xaa5d46ed,0x4acc5748),LL(0x9193931b,0x9fc557d9),LL(0xda4eba9b,0x17568fcf),LL(0xa0edc975,0x2cf3690c)}},
   {{LL(0x953df6fd,0x0e8b0e7e),LL(0x62036a87,0x38ea7cea),LL(0x655c3685,0x57e01428),LL(0xc39d8a43,0xaedfee73)}, {LL(0x5fb27e0a,0xed7f6598),LL(0x946888e0,0x524c3653),LL(0xe949b72f,0xd84a299b),LL(0xb0c61ea4,0x76c1397a)}},
   {{LL(0x1afe465a,0xfd9f7ed0),LL(0xdbbaf852,0x832c69ad),LL(0x03713338,0xcd888c22),LL(0xe3306617,0x4e1fe026)}, {LL(0x23521b97,0xa87adf86),LL(0xf9fbb2a0,0x673d3625),LL(0x5d8f5b80,0xf29a1413),LL(0xd3526501,0x6e9be0c4)}},
   {{LL(0xe8bfd84d,0x6129f861),LL(0x77e35a47,0x1df491d6),LL(0xa84a82cb,0xefe0e9a9),LL(0x6d949612,0x972bc3bc)}, {LL(0x3a766eca,0x8d7795f5),LL(0x12fcc6d4,0x6119383f),LL(0xc95f0e21,0xa66d9836),LL(0x684e434b,0x77a0aa0a)}},
   {{LL(0x7dd7b05a,0x3d55d256),LL(0x0fed8362,0xda616243),LL(0x383e94fe,0x24bd0fe8),LL(0x6bfd0cd2,0xbc2b7334)}, {LL(0x321f7a70,0xf9497232),LL(0x6a3df54f,0x37a4c2f6),LL(0x4ddc49d6,0x7ba783bf),LL(0x04408c76,0x4d142317)}},
   {{LL(0x38b99f23,0x7502146b),LL(0x21992e8f,0x479ab73c),LL(0xd52c41d3,0xf605370a),LL(0x3a60435f,0x358b746d)}, {LL(0x5bc537b8,0xb2cbab94),LL(0xb99057d3,0x1fd24431),LL(0xb8510f3c,0xff2242a0),LL(0x0733bc53,0x74b4965d)}},
   {{LL(0x86edc9b2,0x30a3a634),LL(0x49c07c7f,0x99c9cf19),LL(0x5b0cd506,0x9d8a50c2),LL(0xbbcb3d65,0x0ed9da5a)}, {LL(0x013f88ec,0x6de1fb5e),LL(0x09086f8c,0xc9356bff),LL(0x2b8825d7,0xa272e1ac),LL(0xf2c5ba33,0x3ad83acb)}},
   {{LL(0x275bce43,0x721ca22c),LL(0xd24f78e8,0xf058b8a7),LL(0xeed46b97,0xd178eb57),LL(0x259fdb5b,0x4ad7d425)}, {LL(0x1b515fe5,0x669ed853),LL(0x76fa1b5e,0x9f14b8e5),LL(0x3da46b02,0xfaba8d0c),LL(0x338f7652,0x759c2c95)}},
   {{LL(0xb5c0ceb3,0x9a369cb0),LL(0x28a2a633,0xc1d2d1ab),LL(0xfcb48cd3,0x676190e3),LL(0xee09c3b1,0x9af97ab3)}, {LL(0xf7e918f5,0x39323719),LL(0xfd3cd554,0xc657cb57),LL(0xa2a02d5c,0x78a33d05),LL(0x64ada03f,0xda3b79df)}},
   {{LL(0x61b3a72a,0x7115ab5c),LL(0x337194fc,0xdd19f34b),LL(0x8f0a14c3,0x0f23bfec),LL(0xa60485d3,0x1fe19eec)}, {LL(0xa463dc9b,0x1ca308c3),LL(0x5e1ae8be,0x83e18dd0),LL(0xd928c0e7,0x849eabdf),LL(0x6bd3e7b3,0x2d131ff5)}},
   {{LL(0x45be4c14,0xc84cd284),LL(0xf8f4c719,0xdee94092),LL(0x3cb73831,0xe8f223ef),LL(0x18c2361e,0x24382f88)}, {LL(0xbe91c8dd,0x205366d0),LL(0x56024b95,0x1e17b50c),LL(0x742cabd3,0x3c3487da),LL(0x8bad494c,0xbe451387)}},
   {{LL(0x18ffaef0,0xfae6c0bf),LL(0x85ed1ede,0x2e7b0ee3),LL(0x125d1488,0x3cebaa05),LL(0x7c8b7fb8,0xcd0de0fe)}, {LL(0x464bc74a,0x59434d54),LL(0xa03fd77b,0x17472da2),LL(0x2c1a9edc,0xab23d042),LL(0xd9cf4b37,0x5390625e)}},
   {{LL(0x0531264e,0x43b85844),LL(0xee7aedca,0x8d71805e),LL(0xfbe643ad,0x4ace3068),LL(0x5f7d46c1,0xc98d1cd2)}, {LL(0xf59b3acd,0xd4888744),LL(0x27288b99,0xcf662d61),LL(0x5bce2649,0xf2704561),LL(0x206ae654,0x33a8f3f9)}},
   {{LL(0x9bce2b39,0xe834843f),LL(0xa90cfc7d,0x8de8e41d),LL(0xd81115b4,0x398800ed),LL(0xff2532da,0x4d33f7c5)}, {LL(0xdcc59e2c,0x5ae37fb2),LL(0x24015306,0xca27b622),LL(0x11e8d6e6,0x51beca89),LL(0xa9693774,0x08c0b7e2)}},
   {{LL(0x72fa713e,0x795e1a21),LL(0x4be58266,0x5ec1c123),LL(0x1be14fc3,0x5d8e87da),LL(0x80283ad5,0x82cefc1e)}, {LL(0xdab7865e,0x820a385b),LL(0xf3daf96c,0x11e32d62),LL(0x5835a989,0xf022ade7),LL(0x00242233,0x2cbc2554)}},
   {{LL(0xe7ce649c,0x653335a0),LL(0x6857eff7,0x8b30baef),LL(0xf3288377,0x7ea7c856),LL(0xe8572f5d,0x1387b347)}, {LL(0xbe10c0cf,0x8a6b0352),LL(0x037c97b9,0x2a74e834),LL(0x197b122e,0xfe10bf59),LL(0x1918aced,0xd1ee174c)}},
   {{LL(0x3958c20d,0x568e5fb9),LL(0x0484a92f,0x1188cbe6),LL(0x4b0d29e3,0x00ec14f4),LL(0x16a2796d,0x2b2e078e)}, {LL(0x20440444,0x48b8cffa),LL(0x661ab68d,0xd4b877a0),LL(0xc4b459fa,0x1f352ab1),LL(0xc53aa54c,0x33accbe6)}},
   {{LL(0x02bb383b,0xce4ff566),LL(0xfd62813d,0xcad561c6),LL(0x01dfc9a8,0x0927c348),LL(0x00fb9a61,0x0dde73fb)}, {LL(0xfce59f34,0xd859809f),LL(0x81872a46,0x225bd9b6),LL(0x0314bb90,0x2642add2),LL(0x0ae61eb8,0x82dc7958)}},
   {{LL(0x22d5b667,0x84c97478),LL(0x6214f46d,0xb2fe94d1),LL(0x12cb20de,0x834740f2),LL(0x8aa69c94,0x336dc7a7)}, {LL(0x939a33e6,0x8ca085a4),LL(0x75a94543,0xd59c9ae9),LL(0x3c47dd07,0x83c97f98),LL(0xe3177231,0x0985f73e)}},
   {{LL(0xebbc623d,0xe556c3fc),LL(0xb1b968fa,0x30a3242f),LL(0xbcd05a51,0x842ce9b0),LL(0x0ad576ce,0x241a35ed)}, {LL(0xbb4a793e,0x49ccaf3c),LL(0x4492a828,0x6e6c7a7b),LL(0xba53eb42,0x72f4f5fc),LL(0x3ea74dab,0x0ca4ba53)}},
   {{LL(0xbbaf9d5f,0xe7b5fb06),LL(0xb02d3b20,0xd49c2e17),LL(0x2d933cc8,0x4d31052a),LL(0x07299aec,0x5346e0b4)}, {LL(0x79aa99ec,0x952a6205),LL(0xecb34e97,0xaab9bc32),LL(0x58ffe9ae,0xd539d7e4),LL(0x9d994472,0x91599393)}},
   {{LL(0xe8822711,0x6b1d4868),LL(0x73d452b8,0x8857e282),LL(0xf08ed046,0xad59adfd),LL(0xc1c47abe,0xdb755d65)}, {LL(0x63275d49,0x2df8520b),LL(0x7f8a3249,0xc3c712ec),LL(0x5215ef57,0x55f2a108),LL(0x3ee2f149,0x955e07a3)}},
   {{LL(0x33f344f4,0x2194ff53),LL(0xbad16820,0xb455b9fe),LL(0x610b4e4c,0xfe69ea78),LL(0x8ab11fe8,0x2957be96)}, {LL(0x2ce14366,0x3efdee3c),LL(0x01eddf9f,0x42043f9f),LL(0x93524f6c,0xfb7998b1),LL(0xdfecf763,0x95ea64c0)}},
   {{LL(0x21afa86f,0xb23d2620),LL(0x86b11457,0xea757f03),LL(0xb0148d30,0x0bc4d2d1),LL(0x88ce4170,0x119b5535)}, {LL(0x0aa9c8f6,0xaab5bb67),LL(0x88e05de2,0xdfc9693c),LL(0xe3f1e9c3,0x6cae7e57),LL(0x6f6c3b9c,0x2b1ceb54)}},
},
/* digit=6 base_pwr=2^42 */
{
   {{LL(0x87636183,0x12e335ca),LL(0x719d1ca3,0x1461a65a),LL(0xb14161d8,0x8150080a),LL(0xc612e112,0x08da4ebf)}, {LL(0xa8498a9a,0xc95dfb6b),LL(0xba0f8dba,0x099cf91d),LL(0x4fb4f497,0x12d2ae14),LL(0x33cb7306,0xfa3a28b0)}},
   {{LL(0x0f01c7ce,0xc89fc5d0),LL(0x7283bdf0,0x6fc45ffd),LL(0x81151923,0x71dece81),LL(0xc433fcc9,0xed1cb14c)}, {LL(0xd3959bcf,0x4279612b),LL(0x35b5732f,0xe163880b),LL(0x71d0a1ca,0x35414ca7),LL(0x2c1e47f3,0xe8b9e651)}},
   {{LL(0xc8df0a74,0x4ff11b0c),LL(0xe095ea9a,0x346ba520),LL(0xcc2bc6c0,0x81dd2268),LL(0xc2701468,0x2fb2e99f)}, {LL(0x98053f0e,0x0d213361),LL(0xf7ae879a,0xe0b8280d),LL(0x952560f7,0xd92b7a75),LL(0x9723b62e,0x8d17dfad)}},
   {{LL(0x08b21362,0x5ce8a78a),LL(0xd9fe0b36,0xf37f5e7f),LL(0x2c87837c,0xdca66c7f),LL(0x0bf2e993,0x92524b94)}, {LL(0x71745788,0xfc0f020c),LL(0x3cbfbf4c,0x6018de46),LL(0xac3de1c8,0xa8446691),LL(0x5de5ae41,0xb194d419)}},
   {{LL(0x2ff27af2,0x1586cdff),LL(0xde26b5ef,0xee628535),LL(0xc682923e,0x58480040),LL(0x5e37da30,0x4dd4596b)}, {LL(0x2f64225f,0x247b9fd7),LL(0x51ca2121,0xdcc6de5f),LL(0x86e7ab9a,0x99fb41ac),LL(0x952b413a,0x54c782a0)}},
   {{LL(0x7298c7d9,0x7641190e),LL(0x716eda14,0x499c35ed),LL(0xbb764e90,0x316134bf),LL(0x884fc34e,0x4d23467e)}, {LL(0xf1d13484,0xfd1208a9),LL(0xcd298a74,0x089d9605),LL(0x73c4346a,0xb398c85a),LL(0xf37f13de,0x50064076)}},
   {{LL(0xa6ebb83d,0xfe10d25a),LL(0xa834b30d,0xc5e3edf8),LL(0x683e09ff,0x546b5d5c),LL(0xc6dc44c6,0x02f96218)}, {LL(0xc0edfc04,0x64528c55),LL(0xb0fc3058,0xb5a44a2c),LL(0xceeff21c,0x9f09b116),LL(0x6b0fbbcd,0x077bcd67)}},
   {{LL(0x9ce76a94,0x29aaa4a8),LL(0xc0725c97,0x847cd081),LL(0x97e16665,0x0c099e90),LL(0x8f7b1fc4,0xe409ffc9)}, {LL(0x690941ed,0xc0575b80),LL(0x92c0ee9d,0x8e25100a),LL(0x9b75837d,0x71662d27),LL(0xe56bb22b,0x6eeb9e97)}},
   {{LL(0x85c6a60b,0xf1d6333f),LL(0x1d7ccfaa,0x982fee9d),LL(0xd4634535,0x1c5e28e7),LL(0x94fec469,0xa76e1d27)}, {LL(0xafe377ec,0x1fe944d6),LL(0x2f68ae6b,0xbd579a25),LL(0xab6b515e,0x10eabb93),LL(0x31b4e4b8,0xa17b5f6c)}},
   {{LL(0xaf72c239,0x05e785fb),LL(0x8af42e92,0x597e2016),LL(0xb32ae6c9,0x663f5a72),LL(0x45541cc6,0x3040ff13)}, {LL(0xdeca6b32,0x6140081f),LL(0xc735001b,0xcdaccaf7),LL(0xdaef0e68,0x62de5066),LL(0xd837df99,0x056e9021)}},
   {{LL(0x16cd1be7,0xba399283),LL(0xcfacf7ad,0x2a486323),LL(0x277777ce,0x00c15730),LL(0xd49a254c,0x5d2f200f)}, {LL(0xdb68078d,0xf38a1f3b),LL(0x33604a36,0x595dea3f),LL(0x904b60b2,0x14749d8c),LL(0x246348ff,0xe70c96d8)}},
   {{LL(0x390e35da,0x04340d52),LL(0x27a9947c,0xc098e3d3),LL(0x9ecc7a3f,0xe6d78198),LL(0x23aa6df6,0x2c39102e)}, {LL(0x300f3cb1,0xb83fed0d),LL(0xdcfbe054,0xc0b1e356),LL(0x20cf45a8,0x3da2224c),LL(0x2f30deda,0x5be55df7)}},
   {{LL(0x2faa9530,0x4d31c29d),LL(0x49d42f79,0x1d5783ae),LL(0xf618b3f3,0xe588c224),LL(0xf8f5b65d,0x7d8a6f90)}, {LL(0x62d09174,0xa802a3d2),LL(0xbddd1cb7,0x4f1a93d9),LL(0x35a5c1dc,0xe08e1d3c),LL(0xf9d2958e,0x856b2323)}},
   {{LL(0x96f00090,0xefd1e3ba),LL(0x3e0d25de,0xd489943f),LL(0x30c8626f,0x082c40ae),LL(0xa4f428e0,0xf6e5b5ef)}, {LL(0x38a7f623,0x660414a3),LL(0x23eefed8,0xcd4e68de),LL(0xfc14e750,0x6dcadc62),LL(0xbeae89b6,0xcb78b3bc)}},
   {{LL(0x1d5e580e,0x445acc56),LL(0xc43abe19,0xbf6547ef),LL(0xc922d50f,0xd160a81b),LL(0xf68eed4e,0x3877c7f8)}, {LL(0xf8a9f64a,0x395745ea),LL(0x603350f6,0x9085b253),LL(0x8b1df366,0x2a4c71f1),LL(0xabe332dc,0x49b9e818)}},
   {{LL(0x528960b1,0xb3e76e66),LL(0xd84aecb3,0x445dc393),LL(0x1612ad64,0x13618436),LL(0x8c831e37,0x3ccbeccc)}, {LL(0x6121383c,0x0fb0bd41),LL(0x80d895a3,0x316164a3),LL(0x233f2f1e,0xc3d34153),LL(0xe0d92225,0x2905906f)}},
   {{LL(0x95456622,0xe12d66e2),LL(0xff554b13,0x10469942),LL(0xf7126c09,0xa894af86),LL(0xf581d3f5,0x448f3267)}, {LL(0xa2b5e758,0xb5512926),LL(0x43fddd90,0x08f02988),LL(0x8ba319e6,0x5f437035),LL(0x865b37e7,0xd254188e)}},
   {{LL(0x8a5cb63a,0x5b281b23),LL(0x6dd136c2,0xa15a2712),LL(0x169beae4,0x00fab229),LL(0xde31b4a1,0x400d3f37)}, {LL(0xf8545cb0,0x275877a4),LL(0x36df0277,0xb396a513),LL(0x838ced07,0xf9896978),LL(0x715cea8d,0x86e68167)}},
   {{LL(0x06a5a96d,0x0eb0f0de),LL(0x1fcf91ae,0x2c7a3672),LL(0x630eca3a,0x287bf614),LL(0xf60c9d2d,0x65347473)}, {LL(0x906efa7f,0xed15a229),LL(0xd549b5b3,0xe7be6381),LL(0x2ce9141b,0x23f32972),LL(0xfcf823f8,0x9618d9a1)}},
   {{LL(0xa3d89e15,0x3d0ef0d3),LL(0x0d07f5eb,0x4d5a30c9),LL(0x73e4887a,0xc359e310),LL(0xdbdec349,0x2f4c6b7e)}, {LL(0xba142643,0xc5a1d3e9),LL(0x11c794b6,0x8f4fd58e),LL(0x1810c63d,0xcad091d1),LL(0xf0bfa76c,0x5b616239)}},
   {{LL(0xa838792a,0xe3433562),LL(0x54148e98,0x4aead02b),LL(0xdb66f216,0x809f2baf),LL(0xeabfe5da,0x09cc90ff)}, {LL(0x63e8edad,0x69eb235a),LL(0xa7f95997,0x64f7acb5),LL(0xfae20f25,0xe999ea18),LL(0x3c4966b3,0xcd7ff208)}},
   {{LL(0x345c8929,0x595e0cc0),LL(0xde5e2498,0xfe43c73c),LL(0x503f216d,0x0cdefc98),LL(0xf98826fb,0x8e4e170d)}, {LL(0xb6c79b1c,0x1492247d),LL(0xef0532aa,0xf8e24b38),LL(0x044bc458,0x9f349d51),LL(0x1002d315,0x2ef04ead)}},
   {{LL(0xda60d581,0xaf322f23),LL(0x0681173f,0x07deaa88),LL(0xa78feca0,0x86b97444),LL(0xc633a54d,0x64d336ea)}, {LL(0x2a426cbf,0x10dd4b1f),LL(0x7af59869,0x08d97c15),LL(0x2d7fe97e,0xb8cc814b),LL(0x3bfb60fe,0x7eacd2e1)}},
   {{LL(0xb790881c,0x967dafb7),LL(0x3663e76c,0x2002b8e4),LL(0xf8e82490,0x3bd28ede),LL(0x4bb2a47a,0x44dd2e81)}, {LL(0xdbc3f2f8,0xde750dfe),LL(0x6e2eec70,0xd9b6e912),LL(0x1e4c4d2f,0xe8400e2f),LL(0x23217be2,0xd3325697)}},
   {{LL(0xd4231a1d,0x030b7e39),LL(0x613d17d8,0x1f72e8b1),LL(0x01857d37,0xcd423512),LL(0x0b4b7926,0x9ecd682c)}, {LL(0x8ec44636,0xfe4ac1c3),LL(0x9aacc091,0x4a030cbf),LL(0x0b133f54,0x12bb252e),LL(0xf970d266,0xbf90ea5d)}},
   {{LL(0xf5484410,0xe00d25f7),LL(0x2a922894,0xb4984eeb),LL(0x8e1e09ce,0x498102fd),LL(0xe1d731bf,0x8f8c9fcb)}, {LL(0x0b4983b7,0xdb897669),LL(0x7b2468f5,0x7a7767f9),LL(0x72f4d5f4,0x1a2f9fe8),LL(0xa6169daa,0x10e95aa9)}},
   {{LL(0x520166e9,0x9487e50f),LL(0xc4ee6a95,0x6f6b115b),LL(0xcf7560f8,0xaf29926f),LL(0x1f989e46,0x20a32458)}, {LL(0xd3bd2473,0x165a2232),LL(0xe9fecdf8,0x93d841ff),LL(0xbf9978c0,0x71d63fa7),LL(0xe7de184b,0x381bcf34)}},
   {{LL(0x347dfaef,0x317c8e40),LL(0x64464bf3,0x795b0f7d),LL(0x1364ec20,0x15dc99d6),LL(0x91600d3f,0xc07fce28)}, {LL(0xc8bebbda,0x9825338b),LL(0xa8547c03,0x5e5e89f6),LL(0x1a040b84,0x3c50032f),LL(0x2b3a533d,0xcea7544f)}},
   {{LL(0x43247e19,0xea26d87d),LL(0xfba8368e,0x7e753390),LL(0x3c7bcfc6,0xb35e75cb),LL(0x7e44aab3,0xf78cb5ce)}, {LL(0xa98d7104,0x4a3534e9),LL(0x6f5852ea,0x2b83ea6c),LL(0x68dced7c,0x11337fff),LL(0xd1a2a294,0xcca0f2c6)}},
   {{LL(0x426bf202,0xb547c662),LL(0x66194a34,0xec50423e),LL(0x78161e84,0x11d34865),LL(0x64f04dec,0x83508c06)}, {LL(0xf7732345,0xd1c72976),LL(0x18e77e0a,0xd624bacd),LL(0xba79bdd9,0x71344b75),LL(0x8d6c1474,0xe4bfe085)}},
   {{LL(0xcc5eb43a,0x505e8fd9),LL(0xdaaf0621,0x612ab1d0),LL(0xe6672368,0xde170783),LL(0x83788673,0xfee7df44)}, {LL(0xd119271d,0x364d6885),LL(0xe1b0cea2,0xdd70bae8),LL(0x5832adba,0xb4b873ad),LL(0x8c683749,0xad3ecc18)}},
   {{LL(0x4d217a2e,0x963d8793),LL(0x1fa4702d,0x099e8c56),LL(0xe6431f1b,0x6d91bc47),LL(0xa5f61399,0x3fd21287)}, {LL(0x682fa08b,0x2fc90bae),LL(0xc1ca371c,0x51699c85),LL(0x831c428f,0x16f29d74),LL(0x9fa2b504,0x0ecefb66)}},
   {{LL(0xa75c5a91,0xd04ac53f),LL(0x3bf0524b,0xcbe62421),LL(0xb8792826,0x91dcb3ce),LL(0x7885092a,0x28a6bf88)}, {LL(0x64c1e218,0x24798e59),LL(0x3fec97dc,0x18e848dc),LL(0x9da457b7,0x935e0f50),LL(0xb8f497a6,0x46b67ab7)}},
   {{LL(0x7651e4ff,0x15a38140),LL(0x4890cd7f,0x6ba6c617),LL(0x5fe253ca,0xa527b8d2),LL(0xff3d603b,0x945277b8)}, {LL(0x75392f01,0x10796155),LL(0x2bd9619d,0xcac8f713),LL(0xdebb8e28,0x71a87eca),LL(0x52ab1792,0xe8e6179e)}},
   {{LL(0xe33705e7,0x4ce3998b),LL(0x48ba56e4,0xf9a0661a),LL(0xd9e4e184,0x47f06b30),LL(0xf9f8f6df,0xda465f75)}, {LL(0xc0ad3e20,0xb05acbbe),LL(0x92bc2c13,0xec8776a4),LL(0x240a908b,0xbb3971b7),LL(0xbbd0cecc,0x80a14367)}},
   {{LL(0x086949bc,0x40911e50),LL(0x4064a19d,0x39b3ab69),LL(0x6b07eaa7,0x538c6d96),LL(0xd3723bde,0x38c05b47)}, {LL(0x080d2a64,0x1e669308),LL(0x2a77601f,0x6b44dbe5),LL(0xe7c6ce9a,0x35579681),LL(0xa16afa75,0xd2950b0e)}},
   {{LL(0xeaf7fafc,0xd228a3ba),LL(0xee878f00,0x9324e71d),LL(0xe413c1ce,0xa853bfc1),LL(0xdcf17d1c,0xfe916368)}, {LL(0x2546154b,0x8611383a),LL(0xe715b375,0xdbdf225d),LL(0x8dbb0651,0x874d70a6),LL(0x9ed56391,0x84e58895)}},
   {{LL(0x3776503c,0xca83d8ad),LL(0x46e82d65,0x2cf38b4e),LL(0xadf3a8d1,0x65af46e6),LL(0x1d31056d,0x4f09a4ab)}, {LL(0xcacc81d5,0xdba27b42),LL(0x5d6e1bdd,0xb6caa0ca),LL(0xf7198b26,0x1086e441),LL(0xac572f9b,0x15dfe6cb)}},
   {{LL(0xd2051dd5,0xd9444337),LL(0x834cd72d,0x6c34b236),LL(0x58df3f28,0x84783216),LL(0x2e921372,0x59b8808e)}, {LL(0x55835302,0x3b268249),LL(0x9f4863f1,0x3299cbe0),LL(0x4c4e213a,0x616e3cdd),LL(0x8c824317,0xa3c84868)}},
   {{LL(0x460ed764,0x884be61c),LL(0xb6041177,0x388df47b),LL(0x60b29b0b,0x27089763),LL(0xe502ba08,0xd66d7d53)}, {LL(0x5acbfaf4,0xadec85ca),LL(0x470c9519,0xfbacf9b7),LL(0xdbcda5b2,0x5d18b7f6),LL(0x0f228ed5,0x7615c036)}},
   {{LL(0xd6000825,0xdfcd8273),LL(0x9d440eb3,0xdacfcf11),LL(0x7738fa46,0xa8257834),LL(0x76281df3,0x7db548af)}, {LL(0x3e0b052c,0x71dd19f6),LL(0xd876613f,0x811feef2),LL(0xf9c3155c,0x7536e854),LL(0x4c8c1220,0x3e594973)}},
   {{LL(0x069b4228,0xf8c5c72d),LL(0x1f2f6b02,0xc077d394),LL(0x76393a52,0x0032dfb9),LL(0x706c3572,0x5e52c880)}, {LL(0xe003750f,0x4a9d6e9d),LL(0x6147cee7,0x3d54814d),LL(0x23b655fa,0x09ed7f77),LL(0xf1988361,0x14fff651)}},
   {{LL(0xfb263d48,0x742f3abd),LL(0x53699a0c,0xedb557dc),LL(0x7ecd0f61,0xc692a474),LL(0x058f0d17,0xdc64f583)}, {LL(0x3227d3ed,0x68a9ce75),LL(0x4601d083,0xfd0b0320),LL(0x9c2cee38,0x7167b330),LL(0x710e350d,0xef658993)}},
   {{LL(0x16910648,0x75a83be1),LL(0x2e7d446c,0x5b32e77d),LL(0xa86ba2de,0x8e0534e5),LL(0xb692aeee,0xc8a92eac)}, {LL(0xf663150f,0x3cf686eb),LL(0x01962baf,0x840eaade),LL(0xa264d877,0x3e06154f),LL(0x24630300,0xbbd04137)}},
   {{LL(0x58631468,0x0b0151bd),LL(0x9f99bbe5,0x570ef82c),LL(0xb30f7b96,0x03565f47),LL(0x98c04b24,0x000628e0)}, {LL(0xd6ccdb2b,0xd34a90ae),LL(0xa99a761c,0x1a584858),LL(0x65e29f1b,0xa640ddca),LL(0x728d3208,0xffb672f9)}},
   {{LL(0x5433abd8,0x550f6392),LL(0x13ff0107,0x4f35e116),LL(0xe731a37b,0xbb2b0fab),LL(0xc83d6e74,0x1e8a5a08)}, {LL(0xc6565e23,0xf617e177),LL(0x76da0795,0x8e370e5a),LL(0x03936a55,0xa5631e02),LL(0xd41293ad,0xe576bee9)}},
   {{LL(0x5381bc98,0xfcfd9bc7),LL(0x000a98ab,0x8a42ddfd),LL(0xb49463c4,0xd6091ca6),LL(0x9754ce07,0xf37f6b9f)}, {LL(0xfa399fda,0xe1543897),LL(0x8810063c,0x7b029ead),LL(0xec5a5b52,0xa98a46bd),LL(0xd50cc504,0xdd162811)}},
   {{LL(0x67a95e56,0x4d725c1d),LL(0x8e17af44,0xc36d6e8b),LL(0x313454c1,0x38ffb699),LL(0x991e4eaa,0x22c3da8c)}, {LL(0x0bb72dc0,0xfa36ee15),LL(0x4fd01d32,0x356bbf74),LL(0x3c7939a3,0x9ff71a30),LL(0x691786e9,0xa0ad2fb5)}},
   {{LL(0x283c34c1,0x7d7f4770),LL(0x2454a31c,0x0148a4f0),LL(0xeab3b646,0xdcbb138a),LL(0xf101223e,0x7834bdb9)}, {LL(0x965baa81,0x49de6cc3),LL(0x15471215,0x5462f15e),LL(0x81d17760,0xd77e7a56),LL(0x53f00de3,0xa08c5ad9)}},
   {{LL(0x397ed010,0x2e6e6686),LL(0x2bef28cb,0xe444a5a9),LL(0x4073cbe4,0x1ab9d616),LL(0x18f0b7dd,0x24c6b9e0)}, {LL(0xc2a93055,0x456482b6),LL(0xefbe8715,0x0f89129f),LL(0x62e5f6f2,0xb50818c3),LL(0x3d63c663,0x1d74a1ca)}},
   {{LL(0xd84bfa55,0x9a9124ee),LL(0x254b3f04,0x6cf81f41),LL(0xa1051109,0x7c9b7c3e),LL(0x640e8df2,0x71c3d6d6)}, {LL(0x62f6af2f,0x56571157),LL(0xe1bc9ae8,0x2ec61a3f),LL(0x2fcc848b,0x20caa2ff),LL(0xdc5c297e,0x71e30dac)}},
   {{LL(0xfbc0740a,0x11901efc),LL(0xb6e35fca,0x4994fc5f),LL(0x177de7dd,0x4dc09eba),LL(0x0494bebf,0xedfdd25c)}, {LL(0x0cbaeb8b,0xc4821ed9),LL(0x66788fbd,0xa9ef7a48),LL(0xd65efbc2,0x5b7a7ca5),LL(0xa9cb1fc6,0xe18feb42)}},
   {{LL(0x56b00ab5,0x2cc74b9c),LL(0xdb4bf3f4,0xf559a140),LL(0xb8a4b54e,0x283136d4),LL(0x37032aa2,0xe969e4f8)}, {LL(0xd85beed3,0x5635fb66),LL(0xa72a54bb,0x32bc4fdd),LL(0x4c386a49,0xc1e5ee2b),LL(0x795a0b08,0x979fd877)}},
   {{LL(0x431f0b8e,0x5acef24d),LL(0xd13cafef,0x9f1c4a80),LL(0x4659f447,0xf19ac70b),LL(0x10f561ae,0x82bab6b6)}, {LL(0xbbc879a7,0x1268e7f3),LL(0x79c37165,0x7e7d7141),LL(0x597e313f,0x491f049d),LL(0xecc98736,0x6ca7e73e)}},
   {{LL(0x80a31eb9,0xd7712aa4),LL(0x2d8b99d7,0xbf7376ca),LL(0x2b8e5f7b,0xc1166cdc),LL(0xf1a48c9b,0x562bf290)}, {LL(0x31c38c75,0xa6e72238),LL(0xb5f42def,0x51a9a100),LL(0xa100b75f,0xa0931d81),LL(0x967830be,0x7022479d)}},
   {{LL(0xc192bc29,0x53eaaa1f),LL(0x4123a9f2,0x09504e7a),LL(0x90671997,0xe897397f),LL(0x4294fda2,0xc56185d2)}, {LL(0x9819b185,0xb531f278),LL(0xe9dda4cc,0x390155ff),LL(0x14d26bf9,0x1258a5d9),LL(0x7e5f13a1,0x47d8f5ae)}},
   {{LL(0xe9591945,0xef9e05e3),LL(0x846441be,0x92d20e07),LL(0x1d897ad1,0x28cc70ef),LL(0x0bac861f,0xee962e74)}, {LL(0xbed368e4,0x9b7a4236),LL(0xe49b9720,0xe65ac22a),LL(0x22c1bd82,0x851f0032),LL(0x1e75ab15,0x771573ec)}},
   {{LL(0x702eb71a,0x2e0a4635),LL(0xee924cd9,0x65167c74),LL(0x10ccabb5,0xe16b3515),LL(0x10ea5907,0x63cf15c4)}, {LL(0x616f5290,0x59dacdc6),LL(0x8e372a43,0x19eb409b),LL(0xe3c36beb,0x5c879337),LL(0x0555fa1e,0x5841e7e2)}},
   {{LL(0xf346ec2d,0xce197347),LL(0x221db43d,0xe14818a8),LL(0xc935c025,0x1bf37115),LL(0xfee14ce0,0xb22bdb03)}, {LL(0xf0e3dfd6,0x893c5efa),LL(0xb1f95e1e,0x8fe9d56c),LL(0x407e4db0,0x6ee580e5),LL(0x9fb33911,0x0292bfc4)}},
   {{LL(0xb2869cac,0x598ce787),LL(0x4798690a,0xd2f77720),LL(0x9cb6015a,0x2689e0f3),LL(0x7127b0e8,0x8831ddeb)}, {LL(0x5c4fd58c,0x44db087b),LL(0xcda33285,0x04a0c68e),LL(0xe1a4c364,0xe55f00d7),LL(0xb6393b21,0xb3a2ea9a)}},
   {{LL(0x5e9d8db9,0x99ef4da3),LL(0x17b484bc,0xa01f11d9),LL(0x1767f6ca,0xc2283fbf),LL(0x9e77f389,0xbb524479)}, {LL(0x10149726,0xc4ea3c66),LL(0x82ec5f64,0x4b71d644),LL(0x642944c7,0x0fe49d52),LL(0x5a17a7bd,0x69fef689)}},
   {{LL(0x8c3dce23,0x2f3588fc),LL(0x68e0c237,0x9d429231),LL(0x86fa61d2,0x63756076),LL(0x729bc192,0x1d89c6b8)}, {LL(0x00d3ffd1,0x85e098d2),LL(0xde6f9109,0x5bf292c2),LL(0x3e7b8f23,0xb20dc994),LL(0x87c6eb54,0xcbe51bad)}},
   {{LL(0x0517b914,0x263fd862),LL(0x225499a7,0x447624ad),LL(0x71f807d9,0xfbb831bb),LL(0x2fe2e021,0x9514fe38)}, {LL(0x52418e9a,0x881e8763),LL(0xf1d9b43b,0x268e4655),LL(0x1f780af9,0xf917044a),LL(0x3d758ba5,0x3727b2d9)}},
   {{LL(0x68755cf3,0x8487eb90),LL(0x7fe12541,0x1887394e),LL(0x46af8ca8,0x2e4c65d4),LL(0xb9e119dc,0x72aae645)}, {LL(0x1ec6ad73,0x958e0094),LL(0x8ce4573e,0x84a7eec4),LL(0xf9254b96,0x3d6d00d4),LL(0x8e421732,0x4ef44f58)}},
},
/* digit=7 base_pwr=2^49 */
{
   {{LL(0x7d3ad2ac,0xf59de0f8),LL(0xc0f92c5c,0xd2670cb1),LL(0xc900b6a5,0x8f05944a),LL(0x8d9668eb,0x11aeed23)}, {LL(0x7c488ea6,0x21b038e4),LL(0x78083d75,0x406ea3f7),LL(0x3bd31fe2,0xd22197b4),LL(0x28a6ef9a,0xdc8f8ccb)}},
   {{LL(0x02887281,0x679a6483),LL(0x7f9de66f,0x13f3d39b),LL(0xf1a7dee2,0x289c3c50),LL(0x40b698b8,0xa510a53c)}, {LL(0x06f799ad,0xc566c3fb),LL(0xb5374650,0xcc95a879),LL(0x61c947b2,0xbd7343c0),LL(0x9543e51b,0xbbbff69d)}},
   {{LL(0xba75aba9,0xb80d38dc),LL(0xd150f881,0xe9b61ac6),LL(0xca7b47c5,0x9f56af52),LL(0x77fb3128,0x040300d9)}, {LL(0xc01fd0c1,0x36877184),LL(0x8b6e134b,0x40112a04),LL(0xccd71653,0x56daed90),LL(0xb74bd62e,0xec553aa6)}},
   {{LL(0x11476dde,0x0398381d),LL(0x1ea0923f,0x4959204d),LL(0x017745bd,0xd67427ad),LL(0x6935e467,0xef022a74)}, {LL(0x24e0380a,0x57e799f5),LL(0xb5f1a730,0x6ee2b64f),LL(0x521771d8,0x9aeaac48),LL(0x0992a13c,0x02c8521c)}},
   {{LL(0x48f6934a,0x25dd9f41),LL(0x3e38e31f,0x8d191988),LL(0x32899837,0x35548844),LL(0xf56b07d6,0xf87c696b)}, {LL(0x73e927b9,0xbc66e7d7),LL(0xbb0bedc5,0x04cdac77),LL(0x0bcd022a,0x1e6f2903),LL(0x22c574b5,0xafa637be)}},
   {{LL(0x55c1759b,0xcdca4b1e),LL(0xa6819d39,0x3d46ee3b),LL(0xb4b0fce9,0xf7497ade),LL(0xdcb613c3,0x54aef506)}, {LL(0x522ff464,0xbc11d721),LL(0x37bd3c51,0xf53f16f2),LL(0x485695b9,0x88f29955),LL(0xdac00fe3,0x428ce742)}},
   {{LL(0xa3520c27,0xd971fbd2),LL(0xe05e1b9d,0x2204fe54),LL(0xf0c15c89,0xb08be507),LL(0x901a15c3,0xfeeda919)}, {LL(0x84b60eb5,0x6576ad3b),LL(0x59e951da,0x40d4b9a1),LL(0x76244608,0xbe96e1b8),LL(0x58ef9f37,0x3af35ec9)}},
   {{LL(0x153adbf7,0xbfe5c43b),LL(0x80351fec,0x07a66edf),LL(0x3b109e60,0x3d804235),LL(0xa832c162,0x4dc97176)}, {LL(0xb1db1e5c,0x03fec75f),LL(0xa15b9900,0x6aa02da6),LL(0x4faa1cff,0x5f9e808f),LL(0xa6412a26,0x90aa28bd)}},
   {{LL(0xe041d453,0x2fb2c15b),LL(0x86c18ef0,0x2b847efa),LL(0x115b7318,0x84f5ee9d),LL(0x71699936,0xd568b080)}, {LL(0x6ea5c852,0x34658ae7),LL(0x126d10ce,0x99f918b3),LL(0x09b9407f,0x75e3d9cc),LL(0x7883f978,0x11c6a0bb)}},
   {{LL(0x876db0fd,0x522a6925),LL(0x7a9a4211,0xc5401ca1),LL(0x6789e755,0x89163b57),LL(0x0fd6f1b8,0xd2b2c99a)}, {LL(0xa7b452dc,0x427eea22),LL(0xef9db65c,0xce8e6682),LL(0xda9c80f7,0xfd835810),LL(0xea916411,0xdb91bfbb)}},
   {{LL(0x798b5051,0x7a5aefad),LL(0x42a0d4cd,0xbd7ebc88),LL(0xbac28520,0x958e327a),LL(0x7d010c77,0xfa8bf6d4)}, {LL(0x579752f4,0x8a7040aa),LL(0xe573d895,0x47974d84),LL(0xfe16f77b,0xfd2a0cdc),LL(0xdbf71fdc,0x0f08f86a)}},
   {{LL(0x2983bd4d,0xb19de6f1),LL(0x1e3a28b3,0xb990931e),LL(0x00cbc4f9,0x43b71b8b),LL(0xf4d75a0e,0x35d1ddd0)}, {LL(0xc653f111,0xc211951d),LL(0x88750928,0xbbc46824),LL(0x174803e3,0x0cf6e752),LL(0x8960d80c,0x81f5e0ac)}},
   {{LL(0x0c52fcf9,0xe03ca085),LL(0x0865ced4,0xa795382e),LL(0xe7117376,0x03bd561c),LL(0x3fd7184a,0x8608dde1)}, {LL(0xa2a98acc,0xfd48fd50),LL(0x11df74b0,0x902fa587),LL(0xfa73b8f1,0x683f101d),LL(0xe7c0efa6,0xc805d31b)}},
   {{LL(0x5b11d6c0,0xe5effb4e),LL(0x01c8374c,0xba30f747),LL(0x0c275aec,0x8733511b),LL(0x97354e36,0xf140b740)}, {LL(0x0341268e,0xb01ded69),LL(0x27eac17b,0x17bc3176),LL(0x8984992c,0x88097703),LL(0x3e05061c,0x37bfafab)}},
   {{LL(0x111d0eb8,0x7eca9f09),LL(0x8f243481,0xda7eb023),LL(0x59b5e491,0xac3cb2d6),LL(0x4f794842,0x56e725b1)}, {LL(0x45b2dff6,0x43245254),LL(0xef10ec78,0xeafe73b9),LL(0x78819dbf,0x0d3cb2bc),LL(0xe784eb22,0xff1cd617)}},
   {{LL(0x9ce0fcd6,0x0dbaf1c9),LL(0x3232a847,0x732ea65e),LL(0x6a75d822,0xdb2ce218),LL(0x3d2273ca,0x88ffd479)}, {LL(0xf2f26b61,0x89092ad2),LL(0x686706ab,0xfb7041bd),LL(0x5e23597c,0xe3d5fa75),LL(0x5995fc84,0xa2035bf8)}},
   {{LL(0x4514b8bd,0x1feecd2c),LL(0x434b9233,0x57cb78b4),LL(0x24215322,0x59bd2ad7),LL(0x1ce8daa0,0x41437de2)}, {LL(0x7147ce80,0x401bbece),LL(0x5abb61e8,0x5e462137),LL(0x5a3790eb,0xbbf21033),LL(0x5134dee3,0x9a791c09)}},
   {{LL(0xcedd2cc1,0xc8ded766),LL(0x6447b925,0xa3e48e9d),LL(0x69efa728,0xc73282a3),LL(0x8d408bec,0x8cb72c30)}, {LL(0x41cf92eb,0xfb4f2797),LL(0x26f2412e,0xef3f42a0),LL(0xa941ab5a,0xdbc0f972),LL(0x98337613,0xc7bd62dd)}},
   {{LL(0x4e45dcbd,0x31892744),LL(0xb51b7f91,0x3b2979cb),LL(0x29b27fec,0x41e002f5),LL(0x4dd51b0f,0x9007ee68)}, {LL(0x6e23d565,0x82f417a3),LL(0x77127820,0x3321f343),LL(0x199b32be,0x8d09d965),LL(0x5bc2017b,0x948429eb)}},
   {{LL(0x124eb248,0x22b639f9),LL(0x125f8c22,0xed097f74),LL(0x5f8bed34,0xdbc02517),LL(0x51aa29c3,0xb93f5b42)}, {LL(0xc7368c44,0x6fedd599),LL(0x2c772a9a,0x99a5a795),LL(0x7a5f156e,0x30b35ba7),LL(0x191c45af,0x9dc50978)}},
   {{LL(0xb5b4c4fe,0xe8d241f5),LL(0xb75f54f8,0xda89eac1),LL(0x9ef86ae5,0xb399dba0),LL(0x51c1b8c6,0x2337bb46)}, {LL(0x4d02f348,0xfe60b0c5),LL(0x0afc6cd0,0x709f1235),LL(0xb40fce18,0x8a0b458b),LL(0xe3929cfa,0xefe143aa)}},
   {{LL(0x0ca6cec0,0xab3a4b0d),LL(0x67246ec3,0xcb235374),LL(0x1ec2538f,0xdf9b0e89),LL(0x80c7b53d,0x3ec2ea13)}, {LL(0xd0ae3146,0x920c55f2),LL(0x43946090,0xd3ac4e1e),LL(0x97ebe7a4,0xeba72583),LL(0x393d401f,0x5031644a)}},
   {{LL(0x9714de1a,0x802c3409),LL(0xde5bacba,0xc62d66d0),LL(0x903b8148,0xb6c2abeb),LL(0x5bffe1c4,0x203531ef)}, {LL(0xe862ead7,0x186266de),LL(0x1a23bebb,0x21e643d5),LL(0x6edda603,0x15c13d11),LL(0xb1bebc77,0x39b4a3a3)}},
   {{LL(0xb9ac4754,0xdb456c1a),LL(0x4d3f305a,0xf497e6e9),LL(0x3fa62dc0,0x84d27e3a),LL(0x2524b94f,0xc18c3569)}, {LL(0xe380f5cb,0x92198954),LL(0x272ea458,0x81d8221c),LL(0x5f328491,0x6fa082f6),LL(0x8e304ccf,0x810ca5af)}},
   {{LL(0x0d76e6d5,0xda9f1c15),LL(0xb7abad72,0x4bd38afc),LL(0x08aa20f5,0x14b5cc26),LL(0x81061318,0x010a1af8)}, {LL(0x03c287c0,0xaf9d7a73),LL(0xbc4d40ab,0x9ba5105a),LL(0xb07937a6,0x99e4b824),LL(0xc869f63c,0x026d294c)}},
   {{LL(0xc910b521,0xaaebde75),LL(0xa7d5dd9c,0xc803ded4),LL(0x62764be2,0xc8b713b0),LL(0x92540cf3,0x5ea9ea2b)}, {LL(0x6930bd0e,0xbaa999c6),LL(0x1f4b254c,0x57052e53),LL(0x2b0b27ee,0xfb9fd699),LL(0x4cc83793,0x86b70932)}},
   {{LL(0xfba59bbe,0x09ab4dd7),LL(0x04f4609e,0x83204fee),LL(0x93934977,0x251cb390),LL(0x647c13e8,0x8add9e8b)}, {LL(0xe7ea7006,0x444815d3),LL(0xbd032c32,0x22333c0a),LL(0x4058b7cb,0xe7728dc8),LL(0xd1bc061f,0xde8eb503)}},
   {{LL(0x493d76c2,0x5d3ece2e),LL(0xd804954a,0xa425f3ae),LL(0xeac95eb8,0x49100271),LL(0x38b4be56,0x94e4dfa0)}, {LL(0x650f9930,0xa855893f),LL(0x50264765,0x1fa0a07d),LL(0x0d1d40be,0x37a3c169),LL(0x2eed2a0d,0xfedb51e4)}},
   {{LL(0x1b3348b4,0xa6e0c2b2),LL(0xc414464e,0x9e361f42),LL(0x176e109a,0x3e14e2ee),LL(0xf4af92fd,0x5f1a6bbe)}, {LL(0x84beb8e5,0xf15d4647),LL(0x97d36132,0xac3f01c1),LL(0x84ca42ae,0x36e669bf),LL(0xd9433ca1,0xf789bdbd)}},
   {{LL(0xf71e84d4,0x384f37f4),LL(0x59d6481a,0x57de9473),LL(0xf5e6fa70,0xa9a81f99),LL(0x6cb57bf3,0x26f0a64f)}, {LL(0x061d38fe,0xc07e1c13),LL(0x4a475732,0x6fae70e9),LL(0x840e595c,0x6cfb6b1d),LL(0x62848351,0xb23cf1f2)}},
   {{LL(0x4fcf8743,0xef6094c7),LL(0x05fab119,0x7dc42218),LL(0x5c220d15,0x3207463f),LL(0x22c4bfb2,0xdf51b3f0)}, {LL(0x1572735b,0x13db445b),LL(0x2f6db574,0xd7662537),LL(0x7796f888,0x692f1e05),LL(0x33f45593,0x9f3d7a5b)}},
   {{LL(0x313de667,0xb5deb892),LL(0x66a478a8,0x75c872d7),LL(0xc4992428,0xb67b5513),LL(0xf70fde09,0xf97e010e)}, {LL(0x60ee268c,0x49b0f053),LL(0xf67cd321,0x981b5141),LL(0x4fbc187c,0xb5a1ac8d),LL(0xc12e6da8,0x162417e2)}},
   {{LL(0x62914938,0x07bb6fff),LL(0x19f44438,0xd385285b),LL(0xa28904dc,0x05a610a1),LL(0x5a29b9f8,0xd80a7099)}, {LL(0xc177af4a,0x72ccb553),LL(0x5e3752f4,0xac0bd91b),LL(0x7ae838a0,0x8e8ae668),LL(0x1fdfe7c3,0xcaa5a46c)}},
   {{LL(0x93d34156,0x2cc2c1a5),LL(0x61fe4572,0x22beffb1),LL(0xfcdc7418,0x66f9f3ce),LL(0x6af66892,0xbaccda41)}, {LL(0x1af43f56,0x775c783d),LL(0x0ae04034,0x1b23b187),LL(0xe99b486b,0x5a9325f4),LL(0x8367ab05,0x36d5bfe9)}},
   {{LL(0xa519d028,0x17d8d2fb),LL(0xbe00e7e0,0x27b6beb2),LL(0x15a3f03a,0x8d51c36c),LL(0x5faac8dd,0xbf90e78b)}, {LL(0xb27ab69f,0x4e8c28e7),LL(0x4a553da4,0x37ecf0c7),LL(0x210fe720,0x3a39682f),LL(0x0b0cdd0c,0x60c62e80)}},
   {{LL(0xa16647cd,0x893aa225),LL(0x64ce0455,0xcffb728e),LL(0xc4f0fe79,0x81891d39),LL(0xf9c39f00,0x1abe3073)}, {LL(0xf961d05c,0x88336c27),LL(0xa5fc96df,0xc9033a88),LL(0x864b39f8,0x0d084405),LL(0x851e95c9,0x866aa904)}},
   {{LL(0x98bae4a8,0x0c36da08),LL(0xb5feb202,0x9f88d799),LL(0x8054e4da,0xcd9aeb4a),LL(0x1e9134cb,0x005206bf)}, {LL(0x17ee6649,0xd5f32bf8),LL(0x60847ad2,0x9431dcd8),LL(0x8a3e4050,0xbe6d62c7),LL(0xedf10d40,0x3ae68f7a)}},
   {{LL(0x4604d71f,0xa95c9ea0),LL(0x415f8028,0x01aa3fea),LL(0x5a41970a,0x3dd55ca5),LL(0x0b5776b4,0x05978ad4)}, {LL(0x787fe20c,0x7c9f5bdd),LL(0x75fdba0b,0x23b9fb7d),LL(0x5fcf3a0f,0xfb1a724d),LL(0x87817071,0xd63b3515)}},
   {{LL(0x44e40138,0xecae282d),LL(0x87605748,0x8732df23),LL(0xd11188cb,0x0ef49da0),LL(0x51146cc0,0xc0478138)}, {LL(0x46621921,0x4ba42323),LL(0x47dfa4eb,0x8836dd44),LL(0x8ec16442,0xdb6a0100),LL(0x9cdd2e87,0xabdd9b81)}},
   {{LL(0x502e26d1,0x205ee262),LL(0x3294e240,0xb961ef9c),LL(0x6da7733d,0x7178f1fb),LL(0x232ecf73,0x989b69fb)}, {LL(0x9a9bccae,0xb7278a35),LL(0x400a01f3,0xb1c81a0b),LL(0xa6b213ba,0x0781855a),LL(0x3429817e,0x8acc1b78)}},
   {{LL(0xfb4e1aae,0x527e3a9f),LL(0x4c0b0f4c,0xc18c1cfd),LL(0x1fa7d9f0,0x0676c365),LL(0x4454cc7c,0x3314509f)}, {LL(0xc7c48245,0xb0f45371),LL(0x695ef470,0x913fe759),LL(0xc8d3e0ad,0xbb676070),LL(0x902e1638,0x0db98fcc)}},
   {{LL(0xfc4dfaa8,0x42874e9c),LL(0x7084b2cb,0xcbf89462),LL(0x8a846ab8,0xd6d46f77),LL(0x14c183b1,0x9e4506ca)}, {LL(0xc53b7631,0xc2d0f9b7),LL(0x294d6c34,0xe47c3d8f),LL(0xc05d3f1c,0x04e3c868),LL(0xa5957fef,0xbacec4f3)}},
   {{LL(0x3b77893e,0x4f4530ba),LL(0x69a18bd9,0x4c234f54),LL(0x5071f5e3,0xb45aadd8),LL(0xd1bd0b86,0x73e4160a)}, {LL(0x1c474f64,0x43fcb30d),LL(0x617d1612,0xedef0769),LL(0x0eec330e,0x92076734),LL(0x5b0a21b5,0xd7767770)}},
   {{LL(0x183e26f4,0x4b7dea31),LL(0xc9fd2e70,0x59d6ff20),LL(0xd5d914f5,0x7bdea00f),LL(0x56376640,0xc736dd0d)}, {LL(0x38ae8300,0x593ae6ef),LL(0xdf0355bf,0xdafe49f1),LL(0x0db4491f,0x094ccd86),LL(0xfe4271ab,0x32295701)}},
   {{LL(0x5db7014e,0x2b7690e4),LL(0xd7766bfb,0x1bbc9c36),LL(0x7d986d0c,0xc52249f0),LL(0x324f20ae,0xc7eec37b)}, {LL(0x0e912329,0xd376afa3),LL(0x04268fa3,0xbc35e949),LL(0x9e91a4ac,0x617bf765),LL(0x1d483ecc,0xb1e932ed)}},
   {{LL(0xac493266,0xd4e31672),LL(0xecdafb85,0x1c779fe2),LL(0x06280499,0xed09eb4a),LL(0xcd4e1f33,0x3dd8d965)}, {LL(0xf34576dc,0x0fb4308d),LL(0x85781a43,0xa8ccbf5e),LL(0xce623a24,0x8dbf488a),LL(0x6118cfd4,0xb0e71d30)}},
   {{LL(0x8cc9d957,0xfc68da68),LL(0x83815670,0x7e5e6b65),LL(0x3f185dfe,0x2c16f5ef),LL(0x98952b33,0x23a4098b)}, {LL(0xd515f551,0x15a80298),LL(0xa7f8f341,0x71a2e7fc),LL(0x8cf4f7b6,0xed42b1b6),LL(0x1504d390,0x02743db2)}},
   {{LL(0x3016e513,0x2bded3a8),LL(0xfb0f7bfb,0xa3c508af),LL(0xaa2be716,0xa6a490de),LL(0xf4485b9f,0x5a04d9e5)}, {LL(0x6ad25b5d,0xd07b99d1),LL(0x65a72cb4,0xa1840109),LL(0x14c45a95,0xc8e2b32d),LL(0xe4f2ecff,0x0fae6e86)}},
   {{LL(0xd94b6fe7,0xd09f454b),LL(0x23006b62,0xa776a633),LL(0xd332b4b9,0x6c700a1c),LL(0xce016225,0x50c3fb34)}, {LL(0x8af71463,0x4b805bc3),LL(0x5f1fb3b7,0x049143e2),LL(0x5a6d1dd3,0xbcaf4b61),LL(0x4733abac,0x02093dd7)}},
   {{LL(0xdf59f061,0x1a23c3f6),LL(0x80c4efb7,0x87a6c141),LL(0xd88e4363,0x47635ae4),LL(0xbf8d2072,0x75e2089f)}, {LL(0xac83803b,0xa2bc1b27),LL(0xe2aafecf,0x8ae61522),LL(0xd0010193,0x4b459205),LL(0x9205f876,0x900f6a31)}},
   {{LL(0xf808f044,0x49cddbc9),LL(0x95094ead,0x94637692),LL(0xb87c9bbf,0x3c9c7c0c),LL(0x4e1844d1,0x1699670a)}, {LL(0xcbcf85c3,0xd8a978f2),LL(0x6a36e1c9,0x83e7b806),LL(0xfaff9c52,0x6f28a73f),LL(0xb71eaa80,0x51341222)}},
   {{LL(0x9328a676,0x195461da),LL(0x21766180,0xefcc93e5),LL(0x771a5485,0xed82c930),LL(0x205a8bff,0x34f15ce0)}, {LL(0xb8b3bfd8,0x88ab72cb),LL(0x8110fe55,0xbb59a5be),LL(0xc7d61a31,0x9ce8a082),LL(0x5b1c63d2,0xfe81d072)}},
   {{LL(0xe9ff8421,0x9fae0be1),LL(0x967e13a6,0x4254f89d),LL(0x35da926f,0x1c094620),LL(0x4a76583d,0x84eda272)}, {LL(0xe0e0ffb8,0xa4033064),LL(0xabc72d0c,0x47951945),LL(0xb72c32e7,0x0af6bb4c),LL(0xda797f9e,0x6c73357b)}},
   {{LL(0x2ac2e99d,0xd7a726c9),LL(0xcd62e7cc,0xf44b4731),LL(0xe6225822,0xf89f8e29),LL(0x8d713d92,0xa44bb9b0)}, {LL(0x9404f6c6,0x3291e8d3),LL(0x37bdb22d,0x50b7a4ff),LL(0x216a0f13,0xe008662e),LL(0xcf382547,0x150fa2d6)}},
   {{LL(0x3138acbc,0xe5e47c55),LL(0x40d7f3db,0x595cf1e2),LL(0x2ee1949d,0x2872392d),LL(0x8a4fb721,0xdbd15bf8)}, {LL(0x183351dc,0x30e78cdc),LL(0x6b294729,0xa39b8efb),LL(0xc7b553e8,0x0df4d23e),LL(0x659d3ffc,0x434f38fa)}},
   {{LL(0x55a0c931,0x1764115e),LL(0xa5c920a4,0x34ea18b9),LL(0xaf903710,0x6a099ddc),LL(0xe49f2c7a,0x4b937dc1)}, {LL(0x430f0a7e,0xacfc4a1a),LL(0x421dbe96,0x8f106a58),LL(0x1811d3fe,0x48ac7026),LL(0xb80f13c5,0x5484226a)}},
   {{LL(0x8da7ca79,0xf692e17b),LL(0x718691b9,0x4827aaa2),LL(0x5c5ea68c,0x881f1c38),LL(0x88bdf643,0x1620f5d6)}, {LL(0x0b9a5add,0xe5703cb2),LL(0xbe925061,0x392e6ea5),LL(0xb0bab0d5,0x2a66ce20),LL(0xf98e8dad,0x83a8e0e5)}},
   {{LL(0xdeec2329,0x53532223),LL(0x346eea96,0x6a740238),LL(0x1dde2a6a,0xa54afbdf),LL(0xf2b5b190,0x0e6ca8c1)}, {LL(0xf3cd4e46,0xcccaa3c6),LL(0x0eb7bb3c,0x168d66bd),LL(0x08d4f4e9,0xf1275144),LL(0x139811fc,0x2ae8c946)}},
   {{LL(0xc870713a,0x4973c726),LL(0xba54b13f,0x298465ee),LL(0x940f224f,0x9f901403),LL(0xb9950a40,0x5cd6a07b)}, {LL(0x069a8484,0x9d4095e6),LL(0xd4f8831f,0xe6bf3181),LL(0x39243da8,0x37ceb29a),LL(0x2693653c,0xb3887f31)}},
   {{LL(0x42c98a56,0x685d2172),LL(0x3969dd9a,0x350fbab8),LL(0xe8ac84ec,0x728edca9),LL(0x59bbb0c4,0xf42beab3)}, {LL(0x27d3c3fd,0x9793e746),LL(0xc732b37e,0xbf6016de),LL(0xdf0f248f,0x3688173a),LL(0x7ed59dfa,0x84fbd040)}},
   {{LL(0xa6731b1b,0x2bad638f),LL(0xb7355979,0x1c7b4b13),LL(0xb8e77093,0xf21550e0),LL(0x53efc63c,0x14d0bc9d)}, {LL(0xd56e1f29,0x119ae9fb),LL(0x4d60bc5a,0x3511309c),LL(0xe3574e43,0xec654f06),LL(0xbef6aea2,0x2d6695df)}},
   {{LL(0x5d6abff7,0x27ece611),LL(0x640c9ab8,0xa706d42d),LL(0x5a6f8fa6,0x7a7252d9),LL(0x349aaf8c,0x32be7195)}, {LL(0xff734e23,0xffb48a3d),LL(0x7d27b99c,0xa9b36c82),LL(0x0ccaedbc,0x85b7a57e),LL(0xc04f2394,0xb93b14fd)}},
   {{LL(0x160700e0,0x3a3a78c5),LL(0x961e4af8,0xbd7ae60a),LL(0xd9472cd7,0xe1deb736),LL(0x3880bbbe,0x276b51b7)}, {LL(0x1aa99bfb,0xcf0c4b9a),LL(0x689d7f58,0xaf949d5f),LL(0x65f33328,0x00878488),LL(0xe7d7b484,0x0f1a178c)}},
   {{LL(0x849e6d32,0xd44550f8),LL(0xfe16485e,0xe7bc29d4),LL(0x2f343924,0x29bbfec6),LL(0x40f2b5ce,0xeeb802f2)}, {LL(0xbbb64f33,0x2b337542),LL(0x9f9bdb3c,0x4c1d3a36),LL(0xc7a1cb88,0x1067cf3b),LL(0x4601fb6e,0x3f12a31d)}},
},
/* digit=8 base_pwr=2^56 */
{
   {{LL(0x1f8a4a91,0xb720a78f),LL(0x753dbe73,0x59e22211),LL(0xadd0991a,0x9f5ad99c),LL(0x7380726f,0x3a0db802)}, {LL(0x7dfb4f1c,0x37f0761c),LL(0x5ac819cd,0x68e7098a),LL(0x37ffe348,0x9683d610),LL(0x2b7b5140,0x5bf205e5)}},
   {{LL(0x61a97262,0x9846b5f6),LL(0x974a82f7,0xedf2cacb),LL(0xaf439654,0x3dfab85f),LL(0xc724ee09,0x43fb0ef9)}, {LL(0x53b0119a,0xd0d5016f),LL(0x5bc8fc81,0x68445363),LL(0x1f65d298,0x6d10b649),LL(0x21a4e64f,0x0f3c88c6)}},
   {{LL(0x7f34c517,0x320372a1),LL(0x2378bc27,0x5602bd16),LL(0x91aae024,0x666a592d),LL(0x317bbdaa,0x716886ab)}, {LL(0xe3045103,0xce55fe68),LL(0x7de1d701,0xf2c4b0b2),LL(0x7d724cb6,0x8da35885),LL(0x9ec47314,0x9aac623c)}},
   {{LL(0xb8529a01,0x824cff46),LL(0x4856b95c,0x6e4d82a2),LL(0xc65af7f7,0x58c6b833),LL(0xae110e53,0x8a6c4125)}, {LL(0x4f083340,0x38207c30),LL(0x176cdb31,0x71aa384b),LL(0x42882de1,0x1ada2941),LL(0xc16a2e4a,0x38b1ad2e)}},
   {{LL(0x142bcb30,0xbdda2720),LL(0xfaf604d1,0x56175263),LL(0xe6796314,0x086189c1),LL(0x5b04dd19,0xdab01c68)}, {LL(0xba8ed3c1,0xce54e4b0),LL(0xe281acfb,0xf616513b),LL(0x5e0a6319,0xaf179629),LL(0x328b587b,0x85e79ac9)}},
   {{LL(0xc9fd7da0,0x11d84588),LL(0x1238d0c4,0xa78682d0),LL(0x829d6475,0x333ddde0),LL(0x69de9e18,0x80c88440)}, {LL(0xc6d8176f,0x5d15f21a),LL(0xa509d470,0xdaff9434),LL(0x8bbbfcd5,0x0191bb0a),LL(0x08fc2688,0xff7732b8)}},
   {{LL(0x5ab3d89e,0x02fe772d),LL(0x9a786c91,0xf1580ec9),LL(0x5a323866,0x8fd83417),LL(0xbadec96f,0x93711d49)}, {LL(0x6b9b4a30,0x2020c34a),LL(0xb8b0de24,0xbf10e000),LL(0x28de3ce5,0x2a5f298d),LL(0xfe1a1c63,0x807a398e)}},
   {{LL(0x73f7c45c,0x9fb640cd),LL(0x0afe059c,0xeb1f87ad),LL(0x52b168d4,0xa3c3979a),LL(0x7b1e403f,0x6eef460c)}, {LL(0x2724bb3f,0x6d943e50),LL(0xf9d922d1,0x53f3f1bb),LL(0xcd538b4a,0x547e7a03),LL(0xd2c4145f,0x37631e20)}},
   {{LL(0xb1f810bf,0xe7e49922),LL(0xf2645825,0xacafdb0f),LL(0x15f35bda,0x0f22216a),LL(0xd85bd0b7,0x6f2b4d95)}, {LL(0xbedc9ecd,0x2f203db8),LL(0xb91e090d,0x26639ff6),LL(0x3486eb84,0x94cd6596),LL(0x42c05747,0x32747db3)}},
   {{LL(0xcebfa9f1,0xcd3e7a52),LL(0xfb2b3007,0x5e792d76),LL(0xb9ecce81,0x9669523d),LL(0x04f191e1,0x9263cc85)}, {LL(0x69655fe1,0x192019c0),LL(0x4d984e59,0x1c5cc5eb),LL(0xdf33f336,0x9ad10ed6),LL(0x41d94897,0x0ca48387)}},
   {{LL(0xf222476c,0xbd1ddf67),LL(0x12d6dc4d,0xb4ad7126),LL(0x93ed702a,0x5c327b18),LL(0xfa70cd9f,0x7e3a27b1)}, {LL(0xc0c4f415,0xdca750bd),LL(0x213a5d61,0x98197c90),LL(0x6f10fcc7,0x9bbd014a),LL(0x2ceed4fb,0xb06061e1)}},
   {{LL(0xa8ad25f9,0xaf6dbbe2),LL(0x7ade697d,0xe70e9f40),LL(0x6eb872d7,0xb829e016),LL(0x1b04173f,0xc330e15c)}, {LL(0x0d4763d8,0xd4868e29),LL(0x4c18c9fb,0x37867f72),LL(0x28019486,0x5fd2f47f),LL(0xb16e9bdd,0xe6bfdf81)}},
   {{LL(0x783e43c5,0xace2a977),LL(0x76eed46a,0xe1791288),LL(0xd1767739,0x3884a5b2),LL(0x427c50a3,0x14eddddb)}, {LL(0x1c9b1fcc,0xbeeed5ac),LL(0x4ecdb47a,0x50b1cb44),LL(0x0dcb78d5,0xcbf69555),LL(0xf2b17a99,0xe60bf9c7)}},
   {{LL(0x9e9ade95,0x0edae6b0),LL(0xcb78b1e1,0xb5c6e13d),LL(0x1c257848,0x32860fba),LL(0xef7f5080,0xfc9aa9f4)}, {LL(0x32aac870,0xccef8508),LL(0xfb5310a0,0x4b237704),LL(0xfeebb972,0x4c3cf970),LL(0x763d5f67,0x5dd3c7a0)}},
   {{LL(0xccbf29c6,0xa656797e),LL(0x5a76a56b,0x6d77f211),LL(0x0e3daff3,0xc627156b),LL(0x7646fb1c,0xa4bd37f5)}, {LL(0xa8cd3e5a,0x5fd7e286),LL(0x2f5fed51,0x3889951a),LL(0xe48c49be,0xf8186fc5),LL(0xc662ee38,0x0d3d308a)}},
   {{LL(0x970e164d,0xb7c9bf06),LL(0xbd3d3087,0xc27a88d8),LL(0xf4e7c899,0x8a37c9cd),LL(0xab411371,0x18494d5a)}, {LL(0xd9d8b29c,0x06532375),LL(0x915a2f74,0xb92dd45c),LL(0x515acb02,0x8a23f6bf),LL(0x435bfa89,0x0e69248c)}},
   {{LL(0x6866c5e4,0x8bf41ec3),LL(0x0999159d,0xf059e652),LL(0xd29d7cd8,0xf906838f),LL(0x3a269735,0xc30100f6)}, {LL(0x6280e70b,0xb7742bc8),LL(0x867b54e1,0x0067d971),LL(0xf544622a,0xafe9032b),LL(0x118a2042,0x6b441e39)}},
   {{LL(0xcdd66b70,0x905c8655),LL(0xc1e2110d,0xe88cce1b),LL(0xee674093,0x8cc23c0c),LL(0xb2ea3fc3,0x55ded4d9)}, {LL(0xb58dfbed,0xdd14502b),LL(0x49f698f8,0x523a4dd9),LL(0x01c83e5a,0xf843a501),LL(0xfe71ee1e,0xf11fd4c1)}},
   {{LL(0x162d7c0b,0xeedd7229),LL(0x4ccad713,0xd42d6a9e),LL(0x2b0c7b93,0xa082fffd),LL(0x2a5016b9,0xee3abd48)}, {LL(0xc117e22b,0x5079c95f),LL(0x814b8666,0x5d4b9169),LL(0x9bf90a6d,0x9e0f5e87),LL(0x744bf7ab,0x4346bd29)}},
   {{LL(0xbfb551b6,0x4d85af0e),LL(0x31f7a958,0xb48e3da8),LL(0x6f5bc50d,0x3b474ce6),LL(0xe7c8dced,0x9fdb47bc)}, {LL(0x53003272,0x2064450e),LL(0x839e69da,0x6bb230f3),LL(0x4d822be5,0xb6941512),LL(0xf11a9dc1,0xb51bc6aa)}},
   {{LL(0xb23047dc,0x866447f8),LL(0xe5f52c2d,0xe02dbd63),LL(0x02770a76,0xe6ea43cb),LL(0x56fa6c25,0x853f5fe3)}, {LL(0x960de6d5,0xfe9615f0),LL(0xf4b1b945,0x37c8b0c8),LL(0x4618629d,0xa6e83805),LL(0x23a2ac61,0x38fb5264)}},
   {{LL(0x01751c20,0x5dfd7005),LL(0xce72773a,0x7e100245),LL(0x0776794a,0xdf09f92a),LL(0x1b730fdc,0xc4a8de81)}, {LL(0xf0c7b031,0x72c302ab),LL(0x1283913b,0xdddff68e),LL(0xe32517b5,0x24889098),LL(0x856a2934,0x2483a0f5)}},
   {{LL(0xa1c3d56d,0xdf6d7dcc),LL(0x09afb797,0x07f9c00b),LL(0x083d9557,0xe90da23d),LL(0xcbc03826,0x80ae6e53)}, {LL(0x7c0e1b23,0x1fd6ff6d),LL(0xb1100226,0x1e90f3c8),LL(0x05a24e23,0xf179e00e),LL(0x946f16bd,0xe5361efe)}},
   {{LL(0x4c662091,0x50f12e4a),LL(0x28608585,0xdad2c7a3),LL(0xf7429473,0x55c66749),LL(0x045ea1b4,0x440b77de)}, {LL(0x91229927,0x9f707b49),LL(0xc6725715,0x3501e29e),LL(0x1225a8e6,0x5626fabb),LL(0x9507e709,0x270a9c2b)}},
   {{LL(0xbdcb9039,0xe0d629da),LL(0x20255b7c,0xb4d7cd22),LL(0x5ed874a6,0x10c8614b),LL(0x4e67d406,0x36891e70)}, {LL(0x1dce66fe,0x020da834),LL(0xabd64dea,0xae69e1e7),LL(0xcc71b37b,0x9cf153a1),LL(0x44771c7e,0xa6e9d024)}},
   {{LL(0x8840fc17,0xb15e31c7),LL(0x349124a4,0x57853112),LL(0xbac542ee,0x78a9d807),LL(0x38fe1188,0xe7b4d812)}, {LL(0xb3a3b801,0x874adc70),LL(0x4694cec2,0x80c0e02a),LL(0xe97805e1,0xd05c8c0e),LL(0x89d8cd40,0x8eaebceb)}},
   {{LL(0x378d055f,0x888c777b),LL(0xb104a132,0x6956795e),LL(0xbe8472d7,0xe4bce719),LL(0x5f51729e,0x23c9f0bf)}, {LL(0x36a3bf3e,0xfe7f7e19),LL(0x20a32d37,0xf8f5d2ca),LL(0x93b8a344,0xf383b467),LL(0x27a6e2c5,0x7eab76f5)}},
   {{LL(0x93b54bc1,0x86c31b0e),LL(0xfc4ecab2,0xb9405ea9),LL(0xa0f6d341,0x09485578),LL(0x4b77e8e7,0x88053bb8)}, {LL(0x29a07ddd,0xcde9b777),LL(0x97649102,0xec8ea63f),LL(0xc516777a,0xf74d082a),LL(0xbacf0dd3,0xf4e26d89)}},
   {{LL(0xd0b3b578,0x6a919da8),LL(0xa0b5f7d8,0x0bcc3b29),LL(0x9e55924b,0xbf4565e5),LL(0x7889dbb6,0x13b36187)}, {LL(0x533981bd,0xad0e59c6),LL(0x0bd0cb7a,0xea941b62),LL(0xa9e7aa7c,0xe5e35e9a),LL(0x088bfd7d,0x27f61727)}},
   {{LL(0x8b3c7fbc,0xda2a5a20),LL(0xba55cb48,0x33cdd403),LL(0x90e7ff36,0xb72b51cf),LL(0x6f215840,0x8cc4b553)}, {LL(0xd2671224,0xf7b80ad9),LL(0x6a45436b,0x560b4387),LL(0xff9e8fae,0xdca90694),LL(0xf97aa84e,0x2e7e9546)}},
   {{LL(0xf37cd717,0x71e9ff45),LL(0x0d73e98f,0x6edf335e),LL(0x9f715170,0xf355690c),LL(0x3f5a82bd,0xf74df40b)}, {LL(0x95e5b105,0x28b6d931),LL(0x2841a54c,0x8827f47c),LL(0x62b4312d,0x159cb943),LL(0x8db37edb,0x277943d7)}},
   {{LL(0x6113a9f8,0x561454fd),LL(0xe70e67e6,0x78ebe733),LL(0x903f2feb,0x8764360b),LL(0x97902f36,0x2ba3b3d8)}, {LL(0x87490b8a,0x28808cef),LL(0xf05f31b3,0xb1175954),LL(0x6c9b4f4d,0xbd5d6005),LL(0xdd254e60,0x12b13fca)}},
   {{LL(0x14959566,0x38d4e812),LL(0x36fe9a6c,0xe253b750),LL(0x809450c1,0x24b2c81a),LL(0x8fec36b1,0x0aa89966)}, {LL(0x053e97e7,0x9a99deb5),LL(0xe31d3a6e,0x5e57321c),LL(0x8dbe78a2,0xcd7a4f33),LL(0x3299e070,0x9f809d4f)}},
   {{LL(0xa26a9eca,0xd6de8cfa),LL(0xa158a735,0x33d5705b),LL(0xc2293743,0x08dd3fcc),LL(0x68bbbaea,0x1f8d0a46)}, {LL(0x61bc4105,0x53ff76f9),LL(0x7c4a8fc9,0x6445e88d),LL(0xc285d0e6,0xfd9a8d04),LL(0xfe62b449,0xf08d0d6b)}},
   {{LL(0xc062810c,0x08c27292),LL(0x6663fa28,0x955629f6),LL(0x9d86fee8,0xbaf96c0e),LL(0x46bb9894,0x1dbc5406)}, {LL(0x93dd45c7,0x8d6b6207),LL(0x3ee989fc,0xaf3baef6),LL(0x59b7b2f7,0xf66cfdb1),LL(0xda16c637,0x287fc2bf)}},
   {{LL(0x2d71f863,0xa44ca8fa),LL(0x84d5dee5,0xa1161962),LL(0x3957b610,0x5a5c8ce3),LL(0x17f50b57,0xdbb32253)}, {LL(0x76056358,0xc6a4eb7d),LL(0xc359d90f,0xff9eb424),LL(0xa88cb38c,0xdf4afe23),LL(0xa696b75d,0x2ae727cb)}},
   {{LL(0xd20a58c8,0x47cc63ef),LL(0xc492ab36,0xd319dc3a),LL(0x36c7f76e,0x887a7d83),LL(0xfcd4cf49,0x65ed5e3e)}, {LL(0xda301d39,0x0e6f2f34),LL(0x38ad4533,0xf2f7c102),LL(0xae834622,0x8a3a003b),LL(0xa060a0d4,0x94084169)}},
   {{LL(0x13c8a1eb,0xb673168b),LL(0x459f8da1,0x80109609),LL(0x5c82007b,0x68003fa1),LL(0x248e0430,0x9f634159)}, {LL(0xfb9b6510,0x188156ab),LL(0xe62844de,0xc35be1cc),LL(0xb0c84d39,0x21e8f908),LL(0xdad3ae53,0xa886d3eb)}},
   {{LL(0x82b0f5fd,0x9e20cd56),LL(0xc465c721,0xc0c12f0b),LL(0x6f913a6e,0xfeeb1051),LL(0xaa32d6fe,0x9e7c76b9)}, {LL(0xb8637b5f,0x820b49a0),LL(0xf4abccf0,0xe9ae172a),LL(0xfb270e67,0xccc050b1),LL(0x2269d1de,0x0b51d7e3)}},
   {{LL(0x678c8d8b,0xca772ec1),LL(0x77ae7c7b,0x74eea3f8),LL(0x1e1bcbd3,0x51550df1),LL(0x3458b249,0xa931c17c)}, {LL(0xf204aed5,0x192c3a45),LL(0xc993c881,0x93abf63d),LL(0x83421891,0xc60aa2cb),LL(0xf6b70284,0x11ce6735)}},
   {{LL(0x69e152e4,0x53e8a3ee),LL(0x0033da23,0x6889ece0),LL(0x7d585418,0xada56904),LL(0xf5e5abb9,0xaf81a877)}, {LL(0xdf515727,0x36e0267d),LL(0x3daad2a9,0xe04b532d),LL(0x1a11ced6,0x290e3ee7),LL(0x65e7a651,0x5be7c429)}},
   {{LL(0x8ef9b498,0xc0662cd3),LL(0x6c4dcbf9,0x0ec5fbf0),LL(0xce4d7e3a,0x26694c70),LL(0xfa52de99,0xc1699a93)}, {LL(0x6dae3e97,0x2e0d394b),LL(0x4c66e572,0xe3af28cf),LL(0xba1e27e4,0x9caf7bf8),LL(0xd5a4bdaa,0xd5c39337)}},
   {{LL(0x9ec8ad6d,0xbb5d9551),LL(0x609fc2e1,0xfb3bc1f1),LL(0x95fe12b5,0x0d95ad2a),LL(0x5341dc74,0xf6fd6e89)}, {LL(0x7537b803,0x1532991e),LL(0xeaf96f9c,0x77772fd3),LL(0xf832749a,0x4ed09840),LL(0x95f19d25,0x69a194ce)}},
   {{LL(0x041cc340,0x5464471a),LL(0x1c442289,0x26f7e550),LL(0xb5ce9706,0x38f2c20e),LL(0x8a44efd3,0xcf73f8f2)}, {LL(0x586e8f77,0x5176eda5),LL(0x63ece447,0x47e33844),LL(0x86b00be2,0x83826e8f),LL(0x539807b7,0x49cffcdb)}},
   {{LL(0x414d3fb1,0x543d1fad),LL(0x38b1ef44,0xd56aac6a),LL(0x96c89050,0x9980bb64),LL(0xb169b8a9,0xc300cb46)}, {LL(0x83413df4,0x5ab01a6b),LL(0xf3c91eda,0x179b8922),LL(0x43cccc06,0x4060b943),LL(0x9458ec1e,0x4f6adeb5)}},
   {{LL(0xe339e40e,0x0a4c6437),LL(0x02aefe83,0x9cb6c532),LL(0x23dce7ea,0xb072d02b),LL(0x59a9032f,0x2cd7b117)}, {LL(0x81dbfaef,0x01220cea),LL(0x0905332d,0xffe0026c),LL(0x0197adff,0x95ec2cb2),LL(0x4c3d0e49,0x853bf6f5)}},
   {{LL(0x25d78f7c,0x04ed54fb),LL(0xbb68cb9f,0x45aae3e1),LL(0xe32d7421,0xf4f1a2c6),LL(0x45a05771,0x646ade65)}, {LL(0x91eab45e,0xab241cfa),LL(0x7b214af0,0xb1cf204c),LL(0x851d311c,0x92dfb3e3),LL(0x144ae0da,0x56479ffb)}},
   {{LL(0x9a7a4ede,0xbf847444),LL(0xf5cfd20f,0xb26b1f15),LL(0x83b33b64,0xf380ed7d),LL(0x3d1998c9,0xa21f9564)}, {LL(0xa720e347,0xd985c7d3),LL(0x8bdf09d5,0x98078974),LL(0xce947692,0xa1f34ce2),LL(0xf419c385,0xf69e6144)}},
   {{LL(0x8c3adcc6,0xe1926526),LL(0x42746263,0x848974fb),LL(0x97791569,0xa731261f),LL(0x065b721b,0xfed39da2)}, {LL(0x836a7e20,0x8369b04c),LL(0x53c19f62,0x5758a761),LL(0x0ebea868,0x45746383),LL(0x3b7d71a8,0x20179927)}},
   {{LL(0x57632243,0xb466ed4f),LL(0x120577c9,0xc8d918cb),LL(0xeda40e9c,0xbab307e5),LL(0xd5f65d1b,0xe6dbc7d4)}, {LL(0x60619e10,0xcae0c649),LL(0x6b0df67c,0xffddf6d1),LL(0xb32ee5d1,0x60488755),LL(0x47164a55,0xcb278aaf)}},
   {{LL(0x0bfb732d,0x354c3392),LL(0x649bc125,0xcd4fc821),LL(0x770ffdb8,0xa8e1253f),LL(0x0ff0c37e,0xf7eec595)}, {LL(0x7149b102,0xe5a65279),LL(0xd0528224,0x1cbbb56b),LL(0xb51c5df4,0x40b1a8d9),LL(0x39e1ca25,0xccb43d26)}},
   {{LL(0xfdcfe8c5,0x48f74dc2),LL(0xfa5b8daf,0x3ccb31b6),LL(0x7de6300f,0x6f8dc5bc),LL(0xf247bc0b,0x2a373fd3)}, {LL(0x17825306,0xefe13539),LL(0xc50c47b4,0xeb253484),LL(0x3c739f02,0x4a7f2af3),LL(0x9a3c6746,0x3a3eb385)}},
   {{LL(0x588978e2,0xa90afa2a),LL(0x8d80894f,0x501fcebf),LL(0x6bf1a4cb,0x1de1d06d),LL(0x6cc42a07,0xb0f4a61d)}, {LL(0x78d406f0,0x975cb8de),LL(0xe3d293e3,0x560b0d7b),LL(0x32e686ca,0x5746227c),LL(0x3fcb0205,0xd12854f5)}},
   {{LL(0x499512e3,0x8c0eaba8),LL(0xade99108,0x8d97c229),LL(0xff2b5782,0xd80da38e),LL(0xaef08107,0xf8c30ba1)}, {LL(0x076b97c3,0x9068d7d0),LL(0xb1b7eba5,0x851d1cb9),LL(0x318e4675,0x02bb728c),LL(0x76ddc683,0x0efe9707)}},
   {{LL(0x6a248b04,0x6985d358),LL(0xf8969ed1,0x75eb6019),LL(0x606a0c64,0xecb66a20),LL(0xfe39b5e5,0xd1252f64)}, {LL(0x2aa222a9,0x93d5d61c),LL(0x1ffff8ec,0x16c0d6f9),LL(0x5dfab0fe,0x0f1f962d),LL(0xcedcccb0,0x88776fe1)}},
   {{LL(0xa32cbff1,0x410333c6),LL(0x093bcbda,0xca13ce28),LL(0x0e479259,0xd97b0684),LL(0xbf505c93,0x8b2b3ad8)}, {LL(0x71761412,0x42092d64),LL(0x918acf33,0x9d0c842d),LL(0xac9ade57,0x904d3add),LL(0xe0d5ef6a,0x025e4177)}},
   {{LL(0x0b33d4ed,0xce406ec0),LL(0x57b5c958,0xf73ac4da),LL(0x6ef70849,0x5f96cb8c),LL(0x77b32d5d,0x702ccc6f)}, {LL(0xcea6885c,0x75bda8d8),LL(0xc0c0432e,0xbfc3e62e),LL(0x54631c9a,0x46db9cc6),LL(0xba1d1550,0x1669075b)}},
   {{LL(0x2d227656,0x5ccc4e34),LL(0x02cb0644,0x0724e41b),LL(0x435601fb,0xc5e2077d),LL(0x68d6aee2,0x356155c5)}, {LL(0xfde58906,0x0ea00013),LL(0x37a9eda4,0x79fa13c3),LL(0x8f51a6a6,0x7d09479d),LL(0x1f979fed,0x86e955b7)}},
   {{LL(0xe39ab804,0x9cb22960),LL(0x03535a39,0x6aeae783),LL(0xb9909be6,0xeb4741de),LL(0x1a5f4139,0xb957c5da)}, {LL(0xedc1819f,0xafdb3e8b),LL(0xe7caa690,0x33545722),LL(0x8bb66ed0,0x0ef33e28),LL(0x70e667b5,0x59073742)}},
   {{LL(0x5c7773b8,0x0390fb3c),LL(0x286a809d,0xb80b4a2f),LL(0xfac46467,0xd17d6103),LL(0x91a48972,0x9a09a0d6)}, {LL(0x39e44585,0xa2124b62),LL(0x174d241a,0x14c8a671),LL(0xada8ba26,0x99abfa37),LL(0xfbb457ae,0x847f3040)}},
   {{LL(0x7529a18c,0x0587aaa4),LL(0x9bb45ee6,0x23b3f724),LL(0x2aa81155,0x4d7f5712),LL(0xa4f16d09,0xa9185804)}, {LL(0x3fc992d1,0xab638141),LL(0x0cad0bb0,0xb6c326fa),LL(0x60f2cb10,0xe21c3625),LL(0x2fac20a9,0x6c7af09e)}},
   {{LL(0xdc6f72ab,0x31e892fa),LL(0x21b81f7b,0x71d5c6a3),LL(0x298a0dd2,0xc3e2d70d),LL(0x13ecdc80,0xbc0c37e2)}, {LL(0xe6496ba4,0xd3191146),LL(0x35115466,0x15f81541),LL(0x07d1937f,0x162be77d),LL(0x7b176367,0x38b4d194)}},
   {{LL(0xb8cafbc9,0x4485966d),LL(0xf44c2a81,0x7cfc0d67),LL(0xe624cefe,0xe9e7ec4d),LL(0x581d4e48,0x4db8bec3)}, {LL(0x7fc8615a,0xe76edf00),LL(0x9a02cdb8,0x1b62c4a5),LL(0x83938a6d,0x8b565749),LL(0x50c86c19,0xd813864e)}},
   {{LL(0x16f55d40,0x7fc071ef),LL(0x9bb45ea5,0x70195438),LL(0xa35543ca,0x83cf09f2),LL(0x20554c19,0x07e91a84)}, {LL(0x62a9d06e,0x51ecd701),LL(0x2044a663,0x00e14c62),LL(0x00423dd9,0xb1317c13),LL(0xa46eab4c,0xf49431bc)}},
},
/* digit=9 base_pwr=2^63 */
{
   {{LL(0xd0614aa1,0x35118434),LL(0xd1418434,0x8bae9779),LL(0xb8c15b89,0xf5641d82),LL(0x416432eb,0x2383af56)}, {LL(0x2c73f990,0xa552d3f0),LL(0xa6bbdc7d,0x8df82e9e),LL(0xd75ec634,0x0f336aa8),LL(0x1603e53f,0xc42e3b2d)}},
   {{LL(0xbad830d2,0x4b33e020),LL(0x590dffb3,0x5c101f9e),LL(0xbc80ecb0,0xcd0e0498),LL(0x52aa293e,0x302787f8)}, {LL(0x220f8fc8,0xbfd64ced),LL(0xbe0ee377,0xcf5cebe0),LL(0x8913b128,0xdc03a038),LL(0xfde23279,0x4b096971)}},
   {{LL(0xd2d638ad,0xb0f8c0de),LL(0x4f299d5f,0x47fc8c77),LL(0x9b68d48e,0xd1720a92),LL(0xa1c6f103,0xf944e708)}, {LL(0xa146889b,0x36e34e04),LL(0xe74a2a28,0xb0aad2d6),LL(0xca52f53c,0xedbb034b),LL(0x87fb2713,0xe987a8e1)}},
   {{LL(0xf727ef3a,0x6c5389af),LL(0x33db88fb,0x95ffeb95),LL(0x9dae0777,0x27cb7042),LL(0x616dbf02,0xd20afe81)}, {LL(0x914bf706,0x0fab8e18),LL(0x0517cd09,0x3b1e66f3),LL(0x12e40644,0x24b46dce),LL(0x08f2d8fa,0x0ff10168)}},
   {{LL(0xea2d8d84,0xe08a10df),LL(0xe97dda79,0xe31f05e7),LL(0x4e9ab132,0xfe95f84a),LL(0x927e216f,0xacd6f7fc)}, {LL(0x83c5a3ea,0x025e27bd),LL(0x50f120fc,0xed010c0d),LL(0xb828101f,0x443b3b8a),LL(0x8cfc0dea,0xd8384819)}},
   {{LL(0x83dc5447,0xe55f34c8),LL(0x04e4e9a0,0xbe76243b),LL(0x819166a2,0x78fb4cbc),LL(0xae37f80a,0x0bdfb703)}, {LL(0xc217cda8,0xf869288e),LL(0x62af4156,0x2662bb71),LL(0x50ae9d30,0xce64f291),LL(0xdc0353c9,0xee0d4440)}},
   {{LL(0xbd25609c,0x3e61a9ea),LL(0xb3839c8b,0x4ccaea93),LL(0xe43736e2,0x721cefa3),LL(0xd0035908,0x229cb244)}, {LL(0x7f10aebb,0x936bc1dc),LL(0xb67332e7,0xc93a1002),LL(0xf98d1132,0xf4b53dd4),LL(0xd5a75030,0x7b99a196)}},
   {{LL(0xca9a9526,0xb13caadd),LL(0x69a303e9,0x701c63fa),LL(0xb0a50f3c,0xb97d667a),LL(0x68e6557f,0x27c03d7c)}, {LL(0xeb105607,0xab24e712),LL(0x8dd86ccb,0x4936aedd),LL(0x0a986d68,0x32196f8a),LL(0x248f5a65,0x0307b826)}},
   {{LL(0xfcadb2ad,0x20e14b4c),LL(0x8c3b8c23,0x4cb4a092),LL(0x1caa9db1,0x50fe3c1a),LL(0x81c0a4e9,0x23cc56e8)}, {LL(0x0867753f,0x5ab09199),LL(0xf9d47c55,0x5a253d19),LL(0x1a9bcc88,0x422b4e03),LL(0x671e4f36,0x4e1ce22b)}},
   {{LL(0xebbe949f,0x588f58b5),LL(0x6982215b,0xb7762296),LL(0xcff863c0,0x3cc83dd6),LL(0x01098f51,0x81ec094d)}, {LL(0xbe0432d0,0x214d69aa),LL(0x6455957d,0xe4e52a9c),LL(0xfadc1eab,0x94743ba8),LL(0x8176316f,0x2c395d97)}},
   {{LL(0xe6bb4d34,0xeab6400c),LL(0xc0d49bf2,0x7364dc55),LL(0xe6959c7e,0xd6fa6e40),LL(0x7eaae61c,0x7960a997)}, {LL(0x94ea77c2,0x918b3c63),LL(0x76866dd1,0x2cf4997f),LL(0xbcbba8ca,0xc4214abf),LL(0x7aa4aab2,0x349a6133)}},
   {{LL(0x99458b24,0xd64bab77),LL(0x2eba3064,0x6fe19e25),LL(0x74068f03,0x9aabd83d),LL(0x6fdf8655,0xaef81218)}, {LL(0x65593fef,0xf506d27b),LL(0xfaa457b2,0x0a1ad85d),LL(0xa303dff4,0x266d0f06),LL(0xabb416e3,0xe8114f4e)}},
   {{LL(0x6aa5a1b8,0xe743f617),LL(0x1b5b2bd6,0xaf84652d),LL(0x092e2c46,0x8b1beab1),LL(0xe2518383,0x7e857549)}, {LL(0xa9383124,0x6be2ece1),LL(0x7fc20081,0x8309442a),LL(0xc3046cab,0x1f00eb8b),LL(0x26f39f8c,0x959f3155)}},
   {{LL(0x8fc2ed93,0xaacfe2d3),LL(0x78f0f858,0x83446645),LL(0xdda35ec4,0x58de6f09),LL(0xf78c69b1,0x891e5ecd)}, {LL(0x91c13d67,0xff4a4ba9),LL(0x487d5575,0x6e78063d),LL(0x8d303a7e,0x226b621e),LL(0xc95987ed,0x5c9bc103)}},
   {{LL(0x5e3be13e,0x28980108),LL(0x414af955,0x5e8c0ac5),LL(0xeaaa71a5,0x0f08e93b),LL(0xce4524f0,0x1bc50407)}, {LL(0x921be66b,0x6a6a2e6a),LL(0xc27da9f2,0x37113baa),LL(0x52e90e29,0xc7b3c636),LL(0xc8558307,0xc075d178)}},
   {{LL(0x88a45b65,0x605f581a),LL(0x68e58c1c,0xcb789200),LL(0xbc5bfe1c,0x14cbed65),LL(0xf02b11d7,0xd1af7dc7)}, {LL(0xcd3a7cc8,0xb8341bc0),LL(0xa01a77b7,0x8e9aefe8),LL(0x5ae2a402,0x8eeafe87),LL(0xc11f3859,0x27a0698f)}},
   {{LL(0xf7af9756,0xc5e49f07),LL(0x9e5b871f,0xffd65bcc),LL(0x423eed7b,0x62a95357),LL(0xb2ec687c,0x93cf64d5)}, {LL(0xbe5da479,0x04b87dd7),LL(0x1a134c0b,0xdcceabd7),LL(0xc5c6925c,0xa4875091),LL(0x8e9c098d,0x3bf947df)}},
   {{LL(0x11d1323b,0xb2617271),LL(0xe4c6046d,0x7769247c),LL(0xcfa6aac3,0xf9c1aaab),LL(0x354492e2,0xf7f13317)}, {LL(0xe91befb6,0x4bd65afd),LL(0xf25b8f8d,0x3e78cd8c),LL(0xe60ff4d9,0x2adf53ed),LL(0x3d288d4c,0x81ec3853)}},
   {{LL(0xee3bf44a,0xda852a71),LL(0xd7b5c6da,0x39b4ef6c),LL(0x44f4e720,0x472e6996),LL(0x9191614b,0xbbd19d38)}, {LL(0x30c0e99d,0xa2bcc2ec),LL(0x57ba0582,0x29318d7b),LL(0xd315963a,0x322faf40),LL(0x0c0619d1,0x49ba5570)}},
   {{LL(0xe5dcd066,0xc28c1f81),LL(0xff9e3493,0x64d1268d),LL(0xbdf8992c,0xab0db38e),LL(0x320cce30,0xe3790c26)}, {LL(0x26e3e4b0,0x59b408a0),LL(0x6ab8504e,0xe9e5fe29),LL(0x83c9eaf3,0x45c827bd),LL(0x89518edb,0xc298e236)}},
   {{LL(0x8d3ab381,0xb79a8b15),LL(0xdb0bb7c0,0x6bb951e8),LL(0xbe4b3353,0x5ebd3854),LL(0x2eb3b0fe,0x107ba27d)}, {LL(0x46786cb4,0x9d01654d),LL(0xcf3a1aa2,0xf46d8352),LL(0xa1662f72,0xa8f669a0),LL(0x68a1d3e1,0xc979209f)}},
   {{LL(0x65471473,0xc64975fa),LL(0xff1f2aad,0x1f8eec02),LL(0x8d0dd401,0x1b520fcc),LL(0x15e14346,0xcd732092)}, {LL(0x8f878a76,0x616478d8),LL(0x7423e0f5,0x3579d49c),LL(0x1b2af15f,0x119f6d6e),LL(0xb08c2c8c,0xbbe33d81)}},
   {{LL(0x8534a355,0x051d99c9),LL(0x458b764b,0xe3f3ddd3),LL(0xbc8c03bd,0xbd7e51aa),LL(0xe8d42e38,0xcd7abf4a)}, {LL(0x3160e63f,0xf0d97428),LL(0x34d13871,0x258bba07),LL(0x3dcb885e,0x4fedb647),LL(0x50f0a645,0x009fca27)}},
   {{LL(0x99775c4e,0x3f06c146),LL(0xf66e7d05,0xb10a4ed3),LL(0x3a3ab903,0x9300e3ca),LL(0xde3c3e1f,0x0a5610e0)}, {LL(0x1af56fb7,0xe2827312),LL(0xd75d9a9c,0x7e2a2365),LL(0xf11f8963,0x9c3bb05a),LL(0x30c80488,0xdf94cac7)}},
   {{LL(0x2d1143f5,0xaff1682f),LL(0xb4d6ed7f,0x5837e83a),LL(0xb4bce662,0xf3e179be),LL(0x8caa5fbb,0xfa8d7862)}, {LL(0x59ea54c1,0xbdde016f),LL(0x3c1ac962,0xc488c829),LL(0x14b46863,0xabe8b367),LL(0x82897d1a,0xbcfde363)}},
   {{LL(0x8c152354,0x87ddf0ec),LL(0x7a953398,0xdec85db7),LL(0x0b57108f,0x927a8b10),LL(0x525f78f2,0xb38b732f)}, {LL(0xeb306d56,0x7e696084),LL(0x50269227,0x9befefef),LL(0xcaddfa11,0xfa86e376),LL(0x404be227,0xd50a08da)}},
   {{LL(0x03bb523c,0xb7408e33),LL(0xc093aaf1,0x6d21aa4a),LL(0xa85d6fcf,0x52aae4c9),LL(0xb726afa9,0xf5d057c9)}, {LL(0xf92ca5b2,0x7979bb5c),LL(0xc4e3e4f3,0x4b1f7936),LL(0x071ec517,0x2c534200),LL(0x67d3f86a,0x47b52ffe)}},
   {{LL(0x84d1c5b2,0x4a0b581d),LL(0x0dfa90cb,0xfc825a4a),LL(0x11c72996,0x2df2ec98),LL(0x7dde922e,0x82077a6e)}, {LL(0x9f28b584,0x89acda10),LL(0xe49fe66f,0x54578eb8),LL(0x4a1c29d7,0x90a5f700),LL(0xb3b44748,0x2de4719c)}},
   {{LL(0x18d85e6b,0x6944fe14),LL(0xde7b904f,0x90bd8208),LL(0xa4597137,0x5811f3b6),LL(0xd4ab5433,0x7ea43767)}, {LL(0xa204a36f,0x7ec39109),LL(0xa30fb76e,0xa43a4a57),LL(0xe090f2be,0x4fd514f8),LL(0xda1c97f8,0x3918138e)}},
   {{LL(0x15145a20,0x2b466ae2),LL(0xfbac86b7,0x28ccb2ce),LL(0x04106b98,0xb891b707),LL(0x29696a08,0xe40a2310)}, {LL(0x636d9e11,0x1210fed0),LL(0x2043caa1,0xdaea218d),LL(0x0aef7dcd,0x10c2ed0f),LL(0xffa5db7b,0x926be98a)}},
   {{LL(0x36abac30,0xe762191c),LL(0x8b75b5cb,0xe21acfaa),LL(0xd180cc32,0x4f5e6b9f),LL(0x55deffdd,0x01358309)}, {LL(0x992a66f3,0x1b1ab943),LL(0xceef1a9c,0x1ebe0246),LL(0x7a01dcb9,0xa24c9e25),LL(0x326505f5,0x3d45c4e3)}},
   {{LL(0xc8544885,0x9b805759),LL(0x7bfcad78,0xbe9b99ca),LL(0x2b8fe78e,0xd1db36e1),LL(0xd5387bcf,0x37255a2d)}, {LL(0xa150ad32,0x044b3a3e),LL(0x6671ae59,0xc65bc2a3),LL(0x1d52384b,0x41ce078e),LL(0x9e72c300,0x3115f1b1)}},
   {{LL(0xd0a358a0,0x487ff9da),LL(0x9c242aec,0x4b20c369),LL(0x1c7b145f,0x7813a44c),LL(0xd6f2d3ee,0x87c6bede)}, {LL(0x47d393b1,0x34d2a89b),LL(0x73f78679,0x1e9f97c6),LL(0x2edce91c,0xcb614fe0),LL(0x7e9a5fa9,0x62b96009)}},
   {{LL(0x58c34b85,0x7eb2aeb5),LL(0xcf5074fc,0xa256a478),LL(0x98c1de9b,0x73f23a56),LL(0x61ce6535,0xeffd490e)}, {LL(0x4a6c15c8,0x2569df2a),LL(0xfffc97a5,0x91e202a0),LL(0x28dc7a57,0xd83c428e),LL(0x9fc8dca8,0x03bc53c7)}},
   {{LL(0x9b60487b,0xed394cfa),LL(0xb483a686,0xa4259f91),LL(0x179a5cca,0x11f51779),LL(0x86c1d1c7,0x00b00ef0)}, {LL(0xf1231aed,0x6e596d2a),LL(0xd80eaa89,0x6c1a702b),LL(0xd23f1d64,0xd28f8c15),LL(0x6d01728f,0x93e85bea)}},
   {{LL(0x1c704922,0xd4288fb5),LL(0x8c1363c5,0xaadd1968),LL(0x52f2cc4e,0x9b5c42d7),LL(0xc298e794,0xf9e4bc96)}, {LL(0xaf0804ac,0xd604f076),LL(0xb3bb2628,0xa441140a),LL(0xd37bf6bd,0x761eabca),LL(0xbe1cf79c,0x7d192922)}},
   {{LL(0x3da7073d,0x2365739e),LL(0x8e2c078f,0xfb7423ea),LL(0x3adfb6f3,0x08f5132e),LL(0x710a26fe,0x470a4205)}, {LL(0x2b6c9b33,0xbe0afeb4),LL(0x3cd813bf,0x94d9edc8),LL(0x440a1699,0xa2c7a7a0),LL(0x4eaf0c10,0xbdc4ea3b)}},
   {{LL(0x52fdc8d3,0x5a5455db),LL(0xb2945868,0x60485f39),LL(0x00af0abe,0x54ce9567),LL(0xe8d15f54,0x17bff77b)}, {LL(0x0e14a7ec,0x0021c531),LL(0xdc9c800a,0x3efdb22d),LL(0x4d947ada,0x9a9e2747),LL(0xb37fc947,0x19417bc4)}},
   {{LL(0x8f02c551,0x71ca8da8),LL(0x074cebc0,0x782d5da4),LL(0xc1a43a2d,0x99094980),LL(0x24890d9b,0xe1b02ff0)}, {LL(0x45d82f7c,0x4eedaddb),LL(0x5061c943,0x7ae170a5),LL(0x4d47c578,0xaf8c7ea0),LL(0xad3a6eae,0xcad17044)}},
   {{LL(0x4f4c9c8b,0x51383e61),LL(0x9182fc81,0x78d17182),LL(0x90d72cb4,0xbed6f0d4),LL(0x7bea62f0,0x98761291)}, {LL(0xef3cd3fc,0x27594570),LL(0x91a8c364,0xf4759534),LL(0x2744eb2d,0xf5c607c5),LL(0xd8d8f337,0x0d6264eb)}},
   {{LL(0xa8701750,0xb54867a6),LL(0x87691191,0x1387e940),LL(0xbd2f75dc,0xc451f178),LL(0xd1da6080,0x31a099d3)}, {LL(0x49f87f03,0x0d0fcf97),LL(0x0af6273d,0x0b7585f8),LL(0x1142265d,0x3619cf2c),LL(0x05c559a4,0xf84d3f96)}},
   {{LL(0xb83f2cb9,0xc3d3c80e),LL(0x8f602141,0xf4ef0b54),LL(0xb9612928,0x3afb343d),LL(0x8db5c907,0x7abe5620)}, {LL(0xcf019b08,0xcd692215),LL(0x9ae12255,0x98d70b38),LL(0x8dfda5f2,0xb459e256),LL(0x8f3f403e,0x066a445e)}},
   {{LL(0x423fbbb6,0x5663e123),LL(0x5424d48f,0xcc55ce36),LL(0x3b6d5467,0x8bca99b9),LL(0x316fc956,0x299ff0ea)}, {LL(0xa0ceb256,0xd973a8d8),LL(0x6d9956b9,0x443ecdb9),LL(0x2f64f912,0x8c16a75d),LL(0xbbf7ab50,0x89e490c2)}},
   {{LL(0xb8dbf031,0x4bd00db0),LL(0x7d2cb92d,0x866e0bbe),LL(0x1dd3db2c,0xad36406e),LL(0xe4e3f194,0x969dc881)}, {LL(0x2a115bc8,0xcb3ac9e4),LL(0xe0a5ab75,0xb45efd5d),LL(0x55377d5c,0x1709c293),LL(0xde6bc25d,0x06d11ba4)}},
   {{LL(0xccf2d10b,0x84a09347),LL(0x08ee5aef,0x571cd4d9),LL(0xa450dd82,0x1379ac02),LL(0xae404542,0x5b7f02f5)}, {LL(0x2a7df4ce,0x17366e7f),LL(0x9830ebec,0x5bb3560c),LL(0x7c254726,0x5c582580),LL(0x70ab7b3d,0xea13f8fd)}},
   {{LL(0x314e2a25,0x868c0f8d),LL(0x0be90b12,0x4b3dad3a),LL(0x32aaffcf,0x09970da4),LL(0x8a6d894d,0xe711e9cf)}, {LL(0x0a80d07a,0x511521af),LL(0x8a2a2851,0xe3814716),LL(0x1de9183e,0xde76d41b),LL(0xaac779e5,0x8a9fc79a)}},
   {{LL(0x26879f8b,0xd7d1f235),LL(0xe37d5f9f,0xcc849c85),LL(0x6b9cd82f,0x26b5488a),LL(0x91099141,0x1b068e8d)}, {LL(0x35ee636f,0x040dc00f),LL(0xd84a9cbb,0xab40f94b),LL(0xdb303776,0x2e4cf65c),LL(0x78e8affb,0x42eaa12e)}},
   {{LL(0x876f8f38,0x7835e4e9),LL(0x090ca6b6,0xcd421d77),LL(0xad0604f7,0x71a1d12d),LL(0x1a22e872,0x51c2d158)}, {LL(0x429e45e9,0xfe7dfcc8),LL(0x48224b6f,0x20028f5c),LL(0x50abf907,0xf7afed37),LL(0xc4ce1a69,0x92183692)}},
   {{LL(0x2d128add,0x0b93365c),LL(0x13200221,0x883f43c3),LL(0x4d309b2d,0x9d3b5a53),LL(0xcf91a023,0x60f0db16)}, {LL(0x5b0e47be,0x20f0ebbd),LL(0x317d8b4b,0xcc20dde8),LL(0x5303dd7c,0xab033b48),LL(0x7a9c1974,0x6703eac7)}},
   {{LL(0x351c8f26,0x92f0b738),LL(0x08569159,0xadd39ac8),LL(0x61098dd5,0x80866e5e),LL(0xcae578f6,0x7d0c1c6f)}, {LL(0x975f59e4,0x13d89cee),LL(0x0092de2c,0x86006ed4),LL(0x819adda4,0xda825b0a),LL(0xde710934,0x74fefb46)}},
   {{LL(0xd3dc8683,0x7c2ec289),LL(0x690e5d08,0x25898cd8),LL(0xbcc67531,0x9bed0f32),LL(0xac762453,0x356ba80c)}, {LL(0x7680da5e,0xd3224c57),LL(0x3399d679,0xaae2597b),LL(0x68df6e74,0xb2a2a1e5),LL(0x2301b373,0x49d23e8c)}},
   {{LL(0x170dd677,0xcb89b484),LL(0x6b3ee110,0x36b1d3d1),LL(0x0d7b51b4,0xe50ada4f),LL(0xfd9afdbc,0xa2f4fb57)}, {LL(0xaa6dd8e8,0xb1b9b81d),LL(0x0be328aa,0x616056a0),LL(0xe12b07c8,0x8f6dd943),LL(0x325abaf6,0x4bb551c6)}},
   {{LL(0x68fbed5f,0xa5460380),LL(0x87ed0d37,0xa65d059f),LL(0x208aa8cc,0xff60beda),LL(0x33d2d65e,0xc91ff11b)}, {LL(0xf65f65d2,0x078c4e5e),LL(0xf347dccf,0xa92ed905),LL(0xf59e3be9,0x261ad25d),LL(0x3b69facc,0x95903d91)}},
   {{LL(0xe789d854,0xcf0a2f94),LL(0x10fbf531,0x9d39cd51),LL(0x6de44e3c,0x980ed5d4),LL(0x78425caa,0xaedbae37)}, {LL(0x7bd278b8,0x35804bc1),LL(0x6a2d7bee,0xf4bee96a),LL(0xa605671c,0xc6c553a6),LL(0x86f010d2,0x182c9238)}},
   {{LL(0x9cd6f37a,0x94343b7a),LL(0x237190a9,0xa71e3853),LL(0xa8a28451,0xfcbebde7),LL(0xd711d2be,0xfa928367)}, {LL(0xc3668951,0xba8fd2ea),LL(0x2d241329,0x00fad1ed),LL(0x5dbdffd1,0x61b82e19),LL(0x5a181dfe,0x0e5e5782)}},
   {{LL(0xc60f1799,0x1c1bf593),LL(0x64ef800f,0x388d6950),LL(0xce927a87,0xf78ef00f),LL(0x6abfff9f,0x2a010419)}, {LL(0xb0b7ffe2,0x13a7b08e),LL(0x6da4cc8f,0x4619da3e),LL(0x7937e0bd,0x8ac19190),LL(0x1af4f84c,0xf97d3fcb)}},
   {{LL(0x8ac425a1,0xaea2abd0),LL(0x4a02e136,0xc619c17d),LL(0x1b2c4acb,0xf09a57d3),LL(0x87b4eb40,0xc6fce6fc)}, {LL(0xb21b02f7,0xa161bb70),LL(0x95bcb925,0x075301fb),LL(0xe1b440ce,0x1d408003),LL(0x606b3142,0xb42a47af)}},
   {{LL(0x1c832c35,0xd4ad09c7),LL(0x0e17fb8f,0x5bebe913),LL(0x8b45b250,0xbf8efbcd),LL(0xe5ca21e4,0xbef3cafe)}, {LL(0x688076f1,0x08a18be7),LL(0x0c3a2648,0xabbb3fc5),LL(0xfb54720e,0xa77086e8),LL(0x19c29e8e,0x84277757)}},
   {{LL(0x5b95b36d,0x551768ca),LL(0xc7df6d3f,0x8850a9b0),LL(0x5008c00a,0xe5a2737f),LL(0xad076e3c,0x9a577c0d)}, {LL(0x2afa6a8a,0xbe7c611c),LL(0x04259dac,0x5dd9142a),LL(0x422bf3d1,0xd14253bb),LL(0x6805c78b,0x8c9dc4c6)}},
   {{LL(0xd430488c,0xb9837258),LL(0x7abc184b,0xf9fc178b),LL(0x0c5e6a11,0x035d3079),LL(0xfbc2182b,0x20cbe540)}, {LL(0x9d76812f,0x849994e2),LL(0xf7a85553,0x166a9279),LL(0x19d70aff,0x15ff0643),LL(0x4bc6a829,0x3c58e0b0)}},
   {{LL(0x84df75ff,0x3809904b),LL(0x67a7c214,0x454c63fd),LL(0x2d873c09,0x79e0ffde),LL(0xcef301bf,0x620a3293)}, {LL(0x237c2bdf,0x8f38c8e8),LL(0x13203c2c,0x61cf96de),LL(0xd0bef848,0xdff401d6),LL(0xee4bcbb6,0x3c8ed7ce)}},
   {{LL(0x07ff8f9a,0x3e128e2d),LL(0xad7e8d5e,0x0653c0b2),LL(0xb1930408,0x7bb30bb5),LL(0x4c158386,0x91d18705)}, {LL(0x80c21fb4,0xc4cf843c),LL(0x8a04133a,0x97a72d75),LL(0x4218755a,0x6b7c49f3),LL(0x68a40f68,0xc1a5a447)}},
   {{LL(0x15ca3037,0x0ab9650e),LL(0xac06feb0,0x16b1fa71),LL(0x0faa3dca,0x50179660),LL(0x1c1aaeae,0x368b2d89)}, {LL(0xb46f0310,0xf6fa652a),LL(0x79fcbc59,0x86a4d627),LL(0x6106a132,0x78169b8e),LL(0x9e387d16,0x40a741eb)}},
   {{LL(0x80eed542,0x14a45174),LL(0x3362ef7f,0xadd64561),LL(0xc5dd0396,0x39228bfc),LL(0xea0c538b,0xe9fdf903)}, {LL(0x74d235de,0x6bfd91ec),LL(0x24aa0a47,0x96ec2378),LL(0xaf8d6168,0xf5699241),LL(0xc548a60b,0x0a7b9be3)}},
},
/* digit=10 base_pwr=2^70 */
{
   {{LL(0x2ade9556,0xe5255c30),LL(0x75ba2e9b,0xe328af1b),LL(0x41ce9e47,0x9d3391ef),LL(0xfb0ffcc9,0xb74cd668)}, {LL(0xe3226acf,0xc67103e4),LL(0xd2959e42,0xa65ad22c),LL(0x99d490fc,0x3aaa8406),LL(0x9ecc6356,0x3e26a1c2)}},
   {{LL(0x4e92defc,0x71c975de),LL(0xd0089883,0x81aeb173),LL(0x2ec09190,0x8a30ce4a),LL(0x69a36d64,0x426e7838)}, {LL(0x309bd2d7,0x5899a0b6),LL(0x3cc1a4af,0x3b1c24af),LL(0x345163b3,0xb2aa8142),LL(0x2c78c86d,0xd2ad9a69)}},
   {{LL(0x8e7a4174,0xde59fe5d),LL(0xab3b0f3d,0xaedff2d2),LL(0x1f053470,0x4688e9e0),LL(0x97c95c7c,0x29ff3fb1)}, {LL(0x85e6a8df,0xffb930cc),LL(0x623b6068,0xaa4d3175),LL(0xf9716623,0x682101dd),LL(0x402db696,0xa3bc9f5f)}},
   {{LL(0xba4e4651,0xf67233c6),LL(0x0714d4ac,0x8cf95660),LL(0x71f1f8da,0xd70decc3),LL(0x7078370e,0xb674732a)}, {LL(0x4ccc773b,0x4270416d),LL(0xde87951e,0xc848ff35),LL(0x493dfb56,0xa61506a8),LL(0x767eb110,0xd8371ea9)}},
   {{LL(0x9e6b1d70,0xb468c382),LL(0x4cd025fb,0x1a901978),LL(0x5e6879e8,0x4bf50c7e),LL(0x71cf7119,0x6b862c0f)}, {LL(0x06240e95,0x6a53ce89),LL(0x04107ff4,0x3ddfaa0a),LL(0x65532b51,0x317093cc),LL(0x0e27b5fc,0xf1e0f859)}},
   {{LL(0xfe4674b4,0x96a97a12),LL(0xf7c97b12,0x2a132ae6),LL(0xa5f5ae91,0x5afcd087),LL(0xd4805ddb,0xfd1d3a32)}, {LL(0xd7b5c8bd,0x0a989dc0),LL(0x4429af19,0x35d186e4),LL(0x42935fba,0x65623ad2),LL(0xe79b867d,0x4e274314)}},
   {{LL(0x08aaba2a,0x47d92016),LL(0xf3f4c812,0x12b62343),LL(0x464f4b4c,0xb35bf043),LL(0xd8e8ba16,0xdc9391c0)}, {LL(0x5d460c0d,0xcc0f8c4a),LL(0xe20fc6ad,0x04ce64bf),LL(0xaa4b7db5,0xd0289df5),LL(0xe5299815,0xe0ea15c5)}},
   {{LL(0xda3adfe0,0xc066ee2f),LL(0x0c964e7d,0xce6a9bdc),LL(0x0c859476,0x04a0115b),LL(0x9c95699c,0xb5e02dc9)}, {LL(0x11377eb9,0xf8301f62),LL(0x172bca2e,0x57b245a2),LL(0x7b47cf1f,0xa7d9b470),LL(0x1774b1c1,0x1b469bab)}},
   {{LL(0xda2dce70,0xbb9ec3e9),LL(0xd29bcdda,0x02d5353e),LL(0xb215233c,0xc193244a),LL(0xd27a4e2a,0xb8d5727f)}, {LL(0xb6c5b114,0x79e56194),LL(0x5ce727f0,0xe2c20e71),LL(0x236cbfea,0xc92f34a5),LL(0x56a02b8f,0xcc47dfd1)}},
   {{LL(0xe983ba13,0x5cdbda39),LL(0x6e96c8b2,0x20f3de57),LL(0x66b76faa,0x2ff05aa7),LL(0xd7f84b47,0xa876bc62)}, {LL(0x0d677d1f,0x962ef8a9),LL(0x801d3001,0xabc7bb1e),LL(0x7d13a23f,0xdb5f0b1a),LL(0x20b819e4,0x2664f3ab)}},
   {{LL(0xdc45375a,0x96be66c5),LL(0x4a6c24e8,0x780ee062),LL(0x013a13ee,0xc6fbfd1a),LL(0x21fc4f9c,0x6ce1496c)}, {LL(0x81f272c5,0x03130c09),LL(0xa26609cd,0x06e59457),LL(0xee5363b4,0xf4c5e564),LL(0x7df0775d,0x1cd19a11)}},
   {{LL(0xdfd6586e,0xcdfcfa67),LL(0x1ba23faa,0x358953e5),LL(0xaeec5d6f,0x0f467275),LL(0x5b0e6b2a,0xb815967a)}, {LL(0x012b89b4,0xb01bf133),LL(0x6839cc04,0xdd924bbc),LL(0x120dfd73,0xa5cd2180),LL(0x19bf8098,0x1abb11ef)}},
   {{LL(0x6a281d1d,0xd56c11ce),LL(0x70daeb19,0xfb01f455),LL(0x8f29fcc1,0xbb442a0d),LL(0xe9b2f829,0x9aa60157)}, {LL(0x90ae8113,0x1f3f6e61),LL(0x6c946c0d,0xc701a185),LL(0x52ba7caa,0xb4b89268),LL(0xb0a5c77f,0xd657c679)}},
   {{LL(0x0dd26330,0x0f14eb11),LL(0x9b036325,0xff622296),LL(0x186e735a,0xaf833fb8),LL(0xc7e710f5,0x7801b02f)}, {LL(0x5c948f43,0xa0bf821f),LL(0x86225c71,0x3be31aea),LL(0xe60b1c88,0xe98f4f7b),LL(0x73c5281b,0x6306588d)}},
   {{LL(0x83c9f829,0xd6178277),LL(0xc06675f1,0x67b20e6c),LL(0xcb2d5b9d,0x8a63fb89),LL(0x038c00fe,0xcb28128c)}, {LL(0x8c7c6c6e,0x070a5141),LL(0x8789c3b7,0xc05e18c3),LL(0xd5c67731,0x09fd03c2),LL(0xf54010ec,0xc59e2abb)}},
   {{LL(0xf1ef2232,0x03977889),LL(0x9c7409a5,0xbe2c82f1),LL(0x32004e03,0x35ac44f9),LL(0x856823a3,0x048bb359)}, {LL(0xec1cf253,0x2e108d6c),LL(0x703eb1d2,0xe98e74d7),LL(0x570ac34d,0xcaf64f60),LL(0x4d7797fa,0xff814e7d)}},
   {{LL(0x70818472,0x93b6abc3),LL(0x888357af,0x0e984be6),LL(0x3fe0c135,0x2a7ca1b0),LL(0x94a82d67,0x0c6c4a11)}, {LL(0xbb83ae74,0x0c90c359),LL(0x328b8af1,0x49b25e5e),LL(0x798ff0a6,0x26a36032),LL(0x1fc28ca3,0xbbf89c99)}},
   {{LL(0xe679eb71,0x4ce174e2),LL(0xd1c183bd,0x17c76176),LL(0x03a69f58,0x4bf67be8),LL(0xc0ee530c,0x937a391c)}, {LL(0x1f7daaff,0x2daa9d90),LL(0xc54f14d0,0xa47e99b2),LL(0xc57feeca,0x6be357e7),LL(0xcfdfd5dd,0x3753fad2)}},
   {{LL(0x48f90174,0x74e1457a),LL(0x9b4734da,0xb8092642),LL(0x5800ea72,0x291e75ba),LL(0xc72c28f7,0x25a21b38)}, {LL(0x505aa4d2,0x2193e0c9),LL(0xada9d3f8,0x2f6829e3),LL(0x92900e29,0x66cd5a1d),LL(0x7414dc1d,0x1360d287)}},
   {{LL(0x0d1b96aa,0x5deeb2eb),LL(0x3e52ccf1,0x25783ce3),LL(0x29d534ef,0xe4e251f4),LL(0x55797dad,0x9fe9693d)}, {LL(0xc6ab9935,0x6a173d69),LL(0x7721ca8e,0x23991318),LL(0xc393eb26,0x38cbcd30),LL(0x71d95a9e,0xe3627ab9)}},
   {{LL(0x7f6fe2d4,0xdf1218be),LL(0xfabd8326,0x850c8598),LL(0xb0f7cf49,0x1214d3d7),LL(0x1805345b,0xeaf60d31)}, {LL(0xbfee2c5f,0xc5caf65b),LL(0x45e23043,0x70127979),LL(0x500fbad2,0xda36e794),LL(0x4156e3a6,0x38fa60b0)}},
   {{LL(0x3cbab88a,0x45934bdd),LL(0x1b19dce4,0x72821e74),LL(0x8661e32d,0x532f706d),LL(0x73a9930e,0x3dbfc225)}, {LL(0x3cbeb0f1,0x72d1cb2a),LL(0xe20f5613,0x795b0696),LL(0x9fc88717,0x6e3469e8),LL(0x483864d2,0xf4bf0531)}},
   {{LL(0xfa19ddd9,0xc92e6a8c),LL(0x3528e628,0x7db7e2ee),LL(0xf321fc88,0x997a00eb),LL(0xacdf205f,0x7605a2c9)}, {LL(0xea9c3ed0,0x9fca58cf),LL(0x56ff0e98,0x833078cb),LL(0x662a1695,0x75159b8f),LL(0x1919f51f,0x186560b7)}},
   {{LL(0x5ef764b4,0xe9b60e57),LL(0xbe73c698,0x61ad29fc),LL(0xdd2be1ee,0x18489017),LL(0x7febda71,0xac95e3b6)}, {LL(0x5ac96092,0xa6985346),LL(0xbfc83566,0x906265c3),LL(0x5972fa4a,0x1692c812),LL(0x00773540,0x4e76f879)}},
   {{LL(0x542b137e,0xba9a6268),LL(0x4c7926e2,0x43a52b90),LL(0xfeae554e,0x28785bf5),LL(0x0ab61576,0xc023b688)}, {LL(0x10933a55,0xb3ec8181),LL(0x6331678e,0x75634459),LL(0x17c50b5d,0xe0dfa141),LL(0xe2151f25,0x4cbe7fda)}},
   {{LL(0xce81fbaf,0x3f3072ac),LL(0x0ff56a32,0xa387bb20),LL(0x99865968,0x2b08a811),LL(0x084cb1f2,0x7279f913)}, {LL(0xdad70f5a,0x78cca6c9),LL(0xff47647d,0x72469f6a),LL(0xe358b597,0x2505c7ff),LL(0x998ff0dc,0x7c5268a8)}},
   {{LL(0x99d5b1c1,0x32d70129),LL(0x24a90c34,0x72727c15),LL(0x715662b1,0x57dad21c),LL(0x132f3294,0x76b4b6ec)}, {LL(0x267d246e,0xd03b46b1),LL(0x29b47516,0xc7c848ec),LL(0x1660af51,0x5eab3dbc),LL(0x04c66383,0x818894c4)}},
   {{LL(0xa7b82f5c,0x26a45c3e),LL(0xea98adfb,0x494694de),LL(0x134b182c,0x44a06ec3),LL(0x75b82b41,0x5570befa)}, {LL(0x129f952e,0x819f6808),LL(0x914f87c4,0xa6fad25f),LL(0x7c7be905,0x831e5668),LL(0xe623a5c2,0x93107f38)}},
   {{LL(0x5e40c5f4,0xa9d88469),LL(0xaa5993c5,0x4314d233),LL(0x9c19bbc5,0x5eab88d0),LL(0xb385d3cd,0x795e1c21)}, {LL(0xce163fbe,0x532a1871),LL(0xb867aea4,0x727cb126),LL(0xf7f30476,0xfc7047eb),LL(0xfcc4fe35,0x18031271)}},
   {{LL(0x884a4c6d,0x4b84fa4a),LL(0x55c830ab,0x82cb9aee),LL(0x0cc927cc,0xd4cfdf04),LL(0xa16bef30,0x787efdde)}, {LL(0x32e3c763,0xd1fb2dd6),LL(0x16737272,0x8739566f),LL(0x03a1055a,0xf9ae4f46),LL(0xf9a7472b,0x199970cd)}},
   {{LL(0x1d33ac50,0xf9893cfb),LL(0x1e5ff4e5,0x74cf7dd4),LL(0xf7165331,0x72ec32e5),LL(0xbb4679cf,0xa082c59a)}, {LL(0x5c75461a,0x3cd0a467),LL(0x40f06206,0xd2872d68),LL(0xb5122795,0x08271eef),LL(0x1475e22b,0x7006d350)}},
   {{LL(0x89e35108,0xf7cd1bcc),LL(0x93f1cbaf,0x924efa43),LL(0xf35b13ac,0xe3716559),LL(0x60370a1d,0xa0a88e87)}, {LL(0x8c286ea3,0x1203be0a),LL(0x6ebd50c7,0x97fc5ab6),LL(0x74284d08,0x2b5b3602),LL(0x694a20e0,0x3055716f)}},
   {{LL(0x193520c0,0x793c8a89),LL(0x655f5ff2,0x356402d0),LL(0x1af5ddd6,0x0cf889ee),LL(0xb3f149b2,0x3eb7eb35)}, {LL(0x68e07e0e,0x5254b57c),LL(0x9c5bbfa7,0xb1971de2),LL(0x0ad81e7e,0xcc85a53a),LL(0xed3cbb10,0xbaaa4d2b)}},
   {{LL(0x8f3a7eec,0xbdf9941c),LL(0xa1f26666,0x6e1b7dab),LL(0x2d79a58f,0xe7a0dfa4),LL(0x1f2b9667,0x25e0ddb5)}, {LL(0x5fd96351,0x4b3b5105),LL(0x8874abd1,0x12325832),LL(0x795d91a5,0x56e90310),LL(0x2c32eec8,0x376a79d2)}},
   {{LL(0xa8a16445,0xd9dd8817),LL(0x0e00fa92,0xd61f6aec),LL(0x17d95f07,0x594a620d),LL(0xf4b15001,0xa1534fda)}, {LL(0x0974f4a3,0xe9402601),LL(0x8f671f13,0x4c3fc130),LL(0xc5f35bfb,0x8eaab35a),LL(0x4626baca,0x13b98472)}},
   {{LL(0xcdee6f8d,0xf48703ad),LL(0x12d39694,0xf1ba0309),LL(0x9fcda52c,0xeb2d4d92),LL(0x5507b401,0x984f5111)}, {LL(0xf6bab9ec,0xe3aa26ae),LL(0x6b2e077d,0x557b5e3f),LL(0x2f9c5f35,0x7185ab4f),LL(0x1680bcbc,0x96f21a33)}},
   {{LL(0x2ddb1173,0x2e7f6e07),LL(0x816ffc8d,0xa704416e),LL(0x52e26265,0x55acfaa3),LL(0x8b758c94,0x9c244253)}, {LL(0x0479936d,0x4012e0a6),LL(0x6d8541d8,0x12749e93),LL(0xce56a2a1,0x374f420d),LL(0xc79da17f,0x6a8e3614)}},
   {{LL(0x157cc9e1,0x3602ad09),LL(0x13603196,0xf3c4a54c),LL(0x8473ae27,0x354fc6ed),LL(0x651b8003,0xb4cf4251)}, {LL(0x3415b5f0,0x456b1b9b),LL(0xc4c71507,0xe078a858),LL(0x9a0a11fb,0xf4210409),LL(0xf930ec45,0x76de42a0)}},
   {{LL(0xcfa869a1,0x82ecd0af),LL(0xdccf2e47,0xa637938a),LL(0xc041648a,0x1d2858f2),LL(0xc0dfacd2,0xcf9abfe8)}, {LL(0xbdddebe1,0x3af77e19),LL(0x180b535f,0x15f6b0bb),LL(0x549d34c1,0x497de678),LL(0x3dba7d6f,0x31495c9e)}},
   {{LL(0x393ab61c,0x47b9368b),LL(0x27441f92,0xdb8ee3a8),LL(0xfb0b0117,0x214a6a5f),LL(0x43759430,0x429e61ad)}, {LL(0x77316c0e,0x78116e88),LL(0xa6a29e98,0x59c82616),LL(0xaaef4619,0xbfed454f),LL(0x35926335,0x673327c4)}},
   {{LL(0x18c616a7,0xaa66d8c5),LL(0x6d28fb98,0xa93946a6),LL(0x133336dd,0x4fc30de5),LL(0x7f0c805e,0x7e000a34)}, {LL(0xcf7eab23,0xa82dcf54),LL(0x8dc24371,0x679e67a8),LL(0x1af582d8,0x26b2dffc),LL(0xd3fe2155,0x4c1d692a)}},
   {{LL(0x2d024923,0x2475b110),LL(0xc303c1e7,0x0cc9245d),LL(0x03667a7a,0x290b7a77),LL(0xd87dbd9c,0x2ab8eb6d)}, {LL(0xc098719e,0x7089e481),LL(0x12c022c8,0x17dd6d74),LL(0x8b7aca14,0x90efa01f),LL(0xf55fbe83,0x8b601fba)}},
   {{LL(0x415aa7a5,0xf800bd76),LL(0x3aa74349,0x015573d3),LL(0xaf5ec789,0xd5143779),LL(0xd76dd2dd,0x32330b4b)}, {LL(0x82331ef1,0xec6caa4d),LL(0x25ad1965,0x92cc8865),LL(0x134370b0,0xf8209a40),LL(0xb883cf95,0x320a37b9)}},
   {{LL(0x5d39731d,0x94237ba2),LL(0x8532a968,0x6359d195),LL(0xf86b8b75,0x8bca94c9),LL(0xde316226,0xdb5c6a9c)}, {LL(0xa2fa0c26,0xdf300c59),LL(0x48af4357,0x6dbf6082),LL(0x06535fc9,0x25066c6c),LL(0xa29b2477,0xba2e774e)}},
   {{LL(0x1931299b,0x5157e93d),LL(0x3a8035a4,0xd6249c10),LL(0x81245740,0xcb18fcf1),LL(0xad5ebe1f,0xb4d84c9d)}, {LL(0x8df0d59d,0x95050bec),LL(0xac2a2e0c,0x190a4860),LL(0xfd1bbb11,0x29029e0f),LL(0xb075b432,0x341f73de)}},
   {{LL(0x74836028,0xa825c3c5),LL(0x8f55343a,0xec4fd74b),LL(0x60a683b3,0x009bcab5),LL(0xcd3adea6,0x29877303)}, {LL(0x684a33ac,0x9f264bf2),LL(0x84b3c379,0xc8bf19e6),LL(0xa1215470,0x8ac35fb8),LL(0x405386d6,0x2919d9da)}},
   {{LL(0x19780b2a,0xb4e4aa30),LL(0x356ddd4e,0x639b8fcb),LL(0x9322c245,0x6ed7b10c),LL(0x57f39c76,0x84ec0bc6)}, {LL(0x879176fb,0x6a1be66c),LL(0xe10e0f77,0x4cab3151),LL(0xe2ae0777,0x01c6321f),LL(0x65e57ff1,0x04d6a04c)}},
   {{LL(0x142596dc,0x8c1725ed),LL(0xb2d413a6,0xd321d49a),LL(0x2b5e8ae7,0x19c25fc3),LL(0xbd3c7dc6,0xfc32cbeb)}, {LL(0x57b98ff5,0xf3ec98b8),LL(0xf75b1a00,0x52e5f1ad),LL(0x8f8ad463,0x16812bb4),LL(0xa274f0c3,0x9d67cb11)}},
   {{LL(0x80064047,0xdec72055),LL(0x4319f87b,0x3f828c01),LL(0xff4d6c4a,0xffcad5c3),LL(0x67a6e030,0xee6659b2)}, {LL(0x0478715f,0x9cb5c729),LL(0x5a1c926e,0xc63fc281),LL(0xdeb11155,0x1b8788ca),LL(0x4f0c00b2,0xbe2eebf1)}},
   {{LL(0xa6af09d1,0x9b72ffd0),LL(0xa9a459f3,0xcbac42bd),LL(0xf560dc93,0x150806c0),LL(0xc57787ee,0x71c636e4)}, {LL(0x2a951b0d,0xe4f3acb8),LL(0x3b18d294,0x510dc771),LL(0xb060e782,0xfbb3fb53),LL(0x0358210e,0x0659cadd)}},
   {{LL(0xecde1629,0x23638245),LL(0xcc09daa5,0xee6e9a65),LL(0xee18f9cf,0xf440bb81),LL(0x955200e0,0x99e4d6e8)}, {LL(0x93e69354,0x34555b58),LL(0xfb29b19b,0xa6080e13),LL(0x5100ab06,0x3bfa4796),LL(0x22eec8fc,0xf5db4b13)}},
   {{LL(0xe5aaa980,0x2c1a229e),LL(0xd29eb83e,0x446cd46b),LL(0x7f5aa015,0xe0c044da),LL(0xa18f23f7,0xa55d5f23)}, {LL(0xb6b70f51,0xd71e58c1),LL(0xb5862c51,0x77c72d10),LL(0xfce94ceb,0x01db731a),LL(0x13a47553,0x877703a8)}},
   {{LL(0x3b75e6d9,0x4878b0b1),LL(0xfe60f98a,0xbe8421f0),LL(0xc69d6820,0x6312821b),LL(0x00d2db20,0x4c019374)}, {LL(0x2a1d8b74,0xb1bd5175),LL(0xef7fdad6,0xa0a24ad2),LL(0x929fc488,0xf29fd27d),LL(0x162a02de,0x8e28b4ed)}},
   {{LL(0xc166afcf,0x434cbdb3),LL(0xbf663e65,0x7b79e808),LL(0xa3c3b159,0xd445f1b0),LL(0xb35b2be9,0xdf9f92b7)}, {LL(0x788a9bbc,0x815b57f3),LL(0xabbba2e0,0x9e03e357),LL(0x91a658d8,0x3fc574d5),LL(0xadf4e250,0x83b35d8a)}},
   {{LL(0xacd1e4f0,0xa0e15175),LL(0x868b4e04,0xeca899a4),LL(0x782b7ee7,0x713b4e9e),LL(0xb7d58c1a,0xed177e1e)}, {LL(0xac8364b2,0x4778df76),LL(0x2e8f7ef7,0x6898fb31),LL(0xa03975b0,0xfccf4c53),LL(0x8597436f,0x0f908d14)}},
   {{LL(0x96671c53,0xbeaf1a16),LL(0x6bc4cbbf,0x9be64329),LL(0x80017bf3,0xc8f66f63),LL(0x8836ff35,0x92d700f2)}, {LL(0x13a4daf1,0x9ddd7a81),LL(0x9b72664b,0xb3c42723),LL(0x81babf43,0x3d96f79a),LL(0x7ce5830d,0xa46c9c0c)}},
   {{LL(0xf5999a71,0x54dfec97),LL(0xe586a578,0xdb5be461),LL(0xcfb4e7ad,0xf9bc3f04),LL(0xb11f07aa,0x6e5448a9)}, {LL(0x70132d5a,0x29ef7791),LL(0x28ba851a,0x4fe486c3),LL(0xb62f7f8a,0x6743fecd),LL(0x44d24d01,0xeb264387)}},
   {{LL(0x72ebb976,0xf93c05bc),LL(0xaaae099e,0xe65b30c0),LL(0xc8427104,0x4194721a),LL(0x3025853e,0x3af3487f)}, {LL(0xdbf48435,0xb33a2893),LL(0x2d79334e,0x2c5ac639),LL(0xb16b05a6,0x8fc199f8),LL(0x7661a77b,0xc145358e)}},
   {{LL(0x1841f719,0x15b580b6),LL(0xd7045528,0x24f5fadb),LL(0x98df2c22,0xe443c257),LL(0xd7eed58b,0x48acf5a7)}, {LL(0xedeb9e4b,0xe24e6571),LL(0x562533fd,0xcd047b81),LL(0xd1566e36,0x618ddd86),LL(0xdba1eced,0x09a77b70)}},
   {{LL(0x6968ddb9,0x0e9de410),LL(0x12985b32,0x10f0f429),LL(0x8ca7d2fa,0xbe21b10f),LL(0xc610ae2b,0x0844d8e8)}, {LL(0xd57107d5,0x58a08694),LL(0xc34441f3,0x45f44bd5),LL(0x79a54186,0xe8b3b3df),LL(0xb8b5f26b,0x6496d668)}},
   {{LL(0x192503be,0xd69cefb8),LL(0x40f928fc,0xb692128d),LL(0xa7ed8c47,0x13b11dfd),LL(0x5a196756,0x90bd279f)}, {LL(0x17ff33eb,0x78f2e8c6),LL(0xaaf6c182,0xa7b233d4),LL(0x76a31f46,0x63d62503),LL(0x1449dc97,0x53143dc3)}},
   {{LL(0x94922922,0x5bb86802),LL(0x89181334,0x2f45a7dc),LL(0x1ec5cce4,0xf7c466d5),LL(0xbb3bd5f3,0x52d15eed)}, {LL(0xe6eacf86,0x150bd5f6),LL(0x0ba66988,0x7fecaf3a),LL(0x11f6f528,0xcdd7fadb),LL(0x55042faf,0x60f64c68)}},
   {{LL(0x961ddf76,0x1615919d),LL(0x53f2f514,0xdba095cb),LL(0x1e6c076c,0xf04960ba),LL(0x4c9f17d5,0xe5276722)}, {LL(0x61c186a9,0x93ff80f9),LL(0x3c58ab92,0xd17378b0),LL(0x769492e8,0xc67f9ae1),LL(0x0c3d023b,0xaccfc868)}},
   {{LL(0x5b99708d,0x7d67a63d),LL(0x4b80189a,0xfb29bef7),LL(0x241c402e,0x3cb7eeaa),LL(0x2c5c2767,0x328cb6de)}, {LL(0x9cec231d,0x0d24a7b4),LL(0x0e2e6a7f,0x725955fc),LL(0xb7f17b13,0xa2040cfa),LL(0xa25c71cf,0x215eff8d)}},
},
/* digit=11 base_pwr=2^77 */
{
   {{LL(0xc0d41a94,0xe4d9cab1),LL(0x9e60f7d4,0xc38b202a),LL(0x9336116c,0x2bbf6b17),LL(0x2e068f13,0x2f9aa877)}, {LL(0xa4bac9fd,0xf8820627),LL(0x8a593cb4,0x2209cb9e),LL(0xc7417931,0xaa78ec63),LL(0x7cfccfbf,0x42d21251)}},
   {{LL(0x3e611940,0x40cee5ae),LL(0x0aa090ec,0x4e2d9ea9),LL(0x1b926e42,0x73d167ef),LL(0x5d5112a3,0x7fff36df)}, {LL(0xcaffa3fb,0x25587745),LL(0xc5a83504,0x224f7f4e),LL(0x3b47bf2a,0x5ceff018),LL(0xecfab5c6,0xed9bfa73)}},
   {{LL(0xd9b429c9,0xf3d570b8),LL(0xc5ad81cb,0x69460116),LL(0x2f6825bd,0x30833a08),LL(0x7c99f36a,0xa297122a)}, {LL(0x05c3abdf,0x6fc9b848),LL(0x5f2e24b2,0xefe95298),LL(0x8915d922,0xf045275a),LL(0x298a9726,0x79146aab)}},
   {{LL(0x4f831d0b,0x0c7d5905),LL(0x2d47d4fe,0xfaaaa26c),LL(0x85042e12,0x5ac28599),LL(0x7796977d,0x7eda370b)}, {LL(0x95c0be63,0x9f0bd71d),LL(0x8e821005,0x7c4601bc),LL(0x4c2ffae9,0xf1ecbc60),LL(0x9b688173,0x7e3efc57)}},
   {{LL(0xf4ea7641,0x868c672b),LL(0xd94243dc,0x4fa90a82),LL(0xf5eab489,0xbd82913e),LL(0xe966b39d,0xceebc159)}, {LL(0x35c51e2b,0x31fe4f5f),LL(0xc79c1725,0x2fea6ab1),LL(0x83afad40,0x5856cd85),LL(0x4ca89b88,0x7f960988)}},
   {{LL(0x1ed8fed0,0x9d237c2d),LL(0x4e0c6f10,0x69b4ec80),LL(0xe4648395,0x11f83283),LL(0x306e152a,0x6f4636a7)}, {LL(0x804539b3,0xf70fd23a),LL(0xb3cdb981,0x4db0903a),LL(0x6691eb18,0xe506ae6f),LL(0x1d8e9d9d,0xaa69c7aa)}},
   {{LL(0x3e4e2088,0x540af950),LL(0x95f04e57,0x8fab01d5),LL(0xa8c51a67,0x51bf909a),LL(0xfd19beb7,0x01299f5e)}, {LL(0xb8f15aeb,0xdf703400),LL(0x2d890394,0x19c70987),LL(0x203d2d99,0xf5fcc675),LL(0xc3d4ddea,0xabbf3f21)}},
   {{LL(0x587feffa,0x8348ca15),LL(0x7d69e4ad,0x585d0740),LL(0x885a0745,0x6fbe5619),LL(0xb10b24dd,0x04ee9eba)}, {LL(0x0f4c12d7,0x5c27075c),LL(0x3c51c605,0xacf4acdc),LL(0xfce336d0,0x782fa52b),LL(0x483621d2,0x6e1d078f)}},
   {{LL(0xd4dc3277,0xa2253bfb),LL(0x4691bc12,0x3a014307),LL(0xebdef616,0x415aa5b2),LL(0x16fe065b,0x1008a44a)}, {LL(0x16dfa94d,0x4004a90a),LL(0x8464785b,0x0e24f541),LL(0x88d30ea8,0xd2715c89),LL(0x1f05a312,0xaf81a9ff)}},
   {{LL(0x3e8d5eef,0x958da470),LL(0xd3c80414,0x09561898),LL(0x8bb74691,0xba6b731c),LL(0x577f2ef9,0x6b7355cd)}, {LL(0xb8a98efa,0xd1f416ed),LL(0x11590271,0xd4a1293a),LL(0x4658e9eb,0x2c4d5934),LL(0xd1f15d39,0x51803651)}},
   {{LL(0x3c95fffe,0x9b9f0c05),LL(0x31acd6ca,0x8f021451),LL(0xf9dba549,0x5fee2961),LL(0x17ea0456,0x05217975)}, {LL(0xd13a6a4a,0xc0591906),LL(0x90daf370,0xa7f5ed02),LL(0xfc4c304d,0x1f8b7158),LL(0x1b7f0246,0x77016c29)}},
   {{LL(0x2ea265d2,0xc27d1847),LL(0x2862781e,0xec0789c6),LL(0x5d86a60e,0x0a79ac1f),LL(0x130670a5,0xe325b563)}, {LL(0x6d33bfee,0xf4794460),LL(0xec25bb10,0x126e703e),LL(0xa7bf902f,0xeae22fd3),LL(0x28eef62e,0x8b2fb282)}},
   {{LL(0x059138b4,0xb68de35b),LL(0xd46e68e3,0xfc44bf56),LL(0xff11f76a,0x71567daa),LL(0x6b17cd2a,0x9110e849)}, {LL(0x69573b93,0x7c4027e3),LL(0x1eb9bf01,0x84ee945a),LL(0x28c26cdb,0xa3fadc6d),LL(0x575dfc1b,0x7037a74d)}},
   {{LL(0x9b2223dd,0x58c96a91),LL(0x51461b94,0x912fc795),LL(0x2df3329a,0xc18ced63),LL(0x88a002d0,0x79d6f75f)}, {LL(0xf44d3d84,0x73d7a089),LL(0x8c058073,0x98c78aa7),LL(0x333ae8ff,0x0ab8b3c7),LL(0xebd4e536,0xf5a8f5ec)}},
   {{LL(0x83a5f52b,0x2c7df9fd),LL(0xcc062097,0x314fc7c3),LL(0xc5a3113c,0x6c3195f8),LL(0x2c25a629,0xf130cef9)}, {LL(0x70c8dd70,0x10c8cc5b),LL(0x01cd40d3,0xecb7a35f),LL(0x6fe21c93,0xfbee538f),LL(0x2ba12ee8,0x57ec1959)}},
   {{LL(0xb2b806a8,0x74387a1b),LL(0xbad5d0f4,0x14efa300),LL(0x23a0e9e4,0xee7e4421),LL(0x3b6abdec,0x504ae428)}, {LL(0x927b1aac,0xb8c06fcb),LL(0x323b71d3,0x55e1c85c),LL(0x48d6dae1,0xf47e180f),LL(0xa84cb0b8,0x6d80dd63)}},
   {{LL(0xd75d7785,0xf8e07d53),LL(0x37614325,0x3edf19b7),LL(0x357946ed,0xf03709b0),LL(0xd12105e7,0x567d8c0d)}, {LL(0xa9383b49,0xecf6abc0),LL(0xcab329a7,0xfe9c760b),LL(0x43928852,0x425e62fa),LL(0x6194b51f,0x27945ae0)}},
   {{LL(0x3ee4f7cd,0x756f0f54),LL(0x26e4c865,0x4337ac7b),LL(0x35b9c3ae,0xf94596c3),LL(0x4d6aa5d2,0x066fd3da)}, {LL(0x43c8d94c,0xce1a5b7e),LL(0x614c0fc2,0xaed496a8),LL(0x2a6d5481,0x355e16f5),LL(0xa09d3afe,0x8f934639)}},
   {{LL(0x2bf2a099,0xd170daef),LL(0xae6ee507,0x3920d87a),LL(0x248158e3,0xbdac1c8e),LL(0x05c54e69,0x99033a9a)}, {LL(0x41872197,0x4513bdf0),LL(0xd3f0f889,0x15634020),LL(0x05d42aa8,0x76c1855a),LL(0xe8ba47cc,0x23079179)}},
   {{LL(0x728429e6,0xf80b787b),LL(0x3dd8c4f8,0x896b6ea5),LL(0xc7d9fe06,0x711eef39),LL(0xebced829,0xfff28d03)}, {LL(0x4ad40c88,0x5d7823b8),LL(0x3b112bd4,0x40a5a166),LL(0x63bce926,0x84230bfa),LL(0xbe17e7cd,0x39d2e6bd)}},
   {{LL(0xef03ee6c,0xa772e242),LL(0xfa009e67,0x888bc969),LL(0x4893e1f0,0x0f06ee83),LL(0x6b89e124,0xf28f0d3c)}, {LL(0x71f5cbc5,0xb3e70ef8),LL(0x6cad191e,0xff0f7e62),LL(0xf028d276,0x990697be),LL(0x4ad8f830,0x144c101c)}},
   {{LL(0x3556d74f,0xbcaafb45),LL(0xeb4c7ea0,0xbc216224),LL(0x234a62c8,0x73ad1043),LL(0x2d95ff46,0xa644eb6a)}, {LL(0x0a3373f8,0xd545b60a),LL(0xd4edaa10,0xf7a0572c),LL(0xa97a85b4,0xa7177049),LL(0x7d3ec769,0x529dbadd)}},
   {{LL(0x7822dd37,0xc45610f6),LL(0x98258666,0xfad07fab),LL(0x87259c1b,0xac27001f),LL(0x1b849745,0xa9bdc6a9)}, {LL(0x04c862b1,0xc7ee7216),LL(0x0012f591,0x32c1541e),LL(0x5a366d79,0x8198aadd),LL(0x68866e1b,0x03cd312e)}},
   {{LL(0x9ec64698,0xa0dbc381),LL(0x1ef36dd2,0x770e4d11),LL(0x8d56bdfd,0x01d17915),LL(0x75eb9947,0xb48153cd)}, {LL(0xfde98390,0xc1d17a54),LL(0x0fe2d6fc,0x0817eaf7),LL(0xa6a704f1,0x44a63591),LL(0x7f554182,0x9423465f)}},
   {{LL(0x13e0574e,0xc7c23cbd),LL(0x439941b6,0x6e06e2cb),LL(0xafa39c79,0xa8aebd2c),LL(0xedede735,0x1b859e2b)}, {LL(0x6b4f5465,0x2f485781),LL(0x624c81e8,0xec3093f0),LL(0xc282644c,0xc1f027c1),LL(0x2b74ab51,0x2f6e990b)}},
   {{LL(0xed2ea3df,0x9a988d1c),LL(0xff39d3df,0xa3f50efd),LL(0xec1d7614,0x418a3627),LL(0x3d4fa3e8,0xafc1520c)}, {LL(0x891a9c69,0x741305af),LL(0x5d6f8296,0xe87064d4),LL(0x12307b05,0x47c9394c),LL(0xc35f0f40,0x6b038acb)}},
   {{LL(0x1ccca008,0xa6e77647),LL(0x9dd71746,0x58e4cfb6),LL(0xc1fe84ae,0xdf649c98),LL(0x90db4388,0x2e308ddc)}, {LL(0xe9362400,0xc2641332),LL(0x42d265e5,0x92dd9842),LL(0x31eb91bd,0xe0e4ed9b),LL(0x145535c5,0x62ec7dd1)}},
   {{LL(0xf810812f,0x1ff29a09),LL(0x15e9b102,0x56b64acb),LL(0x5b353184,0xb5f6d4d4),LL(0x1c593774,0xc3c9292c)}, {LL(0x2c700292,0x16781036),LL(0x3ae2f0c6,0xf0948fc9),LL(0x4da778ea,0x40e353cc),LL(0xa34df03a,0x07febf09)}},
   {{LL(0x9ec397ff,0x349812ae),LL(0x330f02d0,0x7c78812f),LL(0x7d241ea2,0xf956700b),LL(0xebed08be,0x864b1809)}, {LL(0xb9eb1635,0xe4cec3df),LL(0xab49fb60,0x7dd65ad6),LL(0x86788a28,0x06551163),LL(0x11fb4214,0xda8792d5)}},
   {{LL(0xcec09c3e,0x82140df3),LL(0x539db03f,0xcd34ca30),LL(0xe7dd0e09,0xf07cf030),LL(0x56ae3487,0x7b08a242)}, {LL(0xbf5a6549,0x9c0fd607),LL(0xd189d68e,0x0b1fc745),LL(0x9cf52022,0x0d91be74),LL(0x43ff7fc3,0x6013f31f)}},
   {{LL(0xb5654233,0x3bf90bd5),LL(0x202bf534,0xd0a17969),LL(0xc97e02ba,0xff373b8b),LL(0xd31dba07,0x4606de54)}, {LL(0x8114562a,0xb045c50a),LL(0x7b8d8915,0xc123acac),LL(0xb60aa324,0xa8f7cd87),LL(0xabc48200,0x077cab67)}},
   {{LL(0x0d7fff59,0x88a68643),LL(0x67bfe743,0x82b92193),LL(0xc2ce06f9,0x1a8b86cf),LL(0xf9ad6134,0xa38414a0)}, {LL(0x28e2c39f,0x7f102610),LL(0x480856a0,0x34805c20),LL(0x18c3034d,0x1b3f9302),LL(0x574c0c9d,0x1713f457)}},
   {{LL(0x690ce2a5,0xd84fa26f),LL(0xe01a4637,0xd4cfa19f),LL(0xcc6fad47,0x4807fb82),LL(0xf933226a,0xc9d84b48)}, {LL(0x7cd2c9dd,0x9b7c530e),LL(0xf44818e3,0x6436a001),LL(0xdfb00461,0xbae0ceed),LL(0x51c8c8a3,0xed6a7c5f)}},
   {{LL(0x0463ac73,0xa6e7fa54),LL(0xc77b19e5,0xa0deed89),LL(0xff218725,0x4e0a3396),LL(0x2edf2704,0x7cfbbd57)}, {LL(0x4e8608c5,0x8114d0ca),LL(0x38c935b7,0xceae65b9),LL(0x330af8fd,0x052b1407),LL(0x723c422b,0x02e189a1)}},
   {{LL(0x657560c8,0xf1cd6216),LL(0xe5068361,0x099eec2f),LL(0x3de78037,0x68ef58fb),LL(0xf3e399e9,0x83e0d34e)}, {LL(0xf9a17095,0x3a2a14c8),LL(0xaaf9f08a,0xc7a360be),LL(0x30e99527,0x6420155f),LL(0x9f180405,0x8f610960)}},
   {{LL(0x02bc97fe,0x871a832f),LL(0x8dc7f1f2,0xa14b3326),LL(0x87f78ad1,0xc9bd8b41),LL(0x0b59b9c5,0xd378d02a)}, {LL(0x35c0dc14,0x418a32a5),LL(0xf53d85af,0x4c96979d),LL(0x08eb4b26,0xb6f40e97),LL(0xcaa6252f,0xa21349ca)}},
   {{LL(0x5de38e2d,0xb13d8062),LL(0x9b43c5d6,0x54ea3684),LL(0xb1d6521d,0xc0ad58d7),LL(0x22085236,0x182f8823)}, {LL(0x2a08449e,0x9d50cecc),LL(0x17ab0b68,0xeb85e785),LL(0x8d74e26b,0xb8a22ab7),LL(0x77d03987,0x77515524)}},
   {{LL(0x77ad71de,0x117a63f2),LL(0xc94c8c82,0x1cca30d0),LL(0x2f05382d,0xe5fefba9),LL(0x9b4b42f1,0xcc9e8916)}, {LL(0x9fe716c1,0xbe939e13),LL(0x95e38cc2,0xbf2b9c80),LL(0x37adde62,0xf60c4491),LL(0xf4df75a3,0x3eb3338a)}},
   {{LL(0xfe4d84df,0x16398af3),LL(0xfaf3e5f2,0xed752cf8),LL(0xb4cf0e1c,0x746a4339),LL(0x39fb6018,0xb8bd479a)}, {LL(0x57dffed3,0x3a9a045b),LL(0xa5ae3c78,0x2b46ea98),LL(0xde6b0399,0x74b5163f),LL(0x80e511c5,0x069628a0)}},
   {{LL(0x1b96672b,0x19cfc882),LL(0x379d1f57,0x2759c76b),LL(0x2269a514,0xa6cc7a98),LL(0x683b1413,0x1bc2015b)}, {LL(0x1bf4be62,0xc43b1178),LL(0x7bf2b0be,0xd2941975),LL(0xc4591cfd,0x1eac3587),LL(0x0e66d495,0x283169e6)}},
   {{LL(0x052352e1,0xd39bedb7),LL(0xd719cd43,0xb04af7f2),LL(0xe92956d7,0x702143d4),LL(0xa0e5b847,0x53498722)}, {LL(0x574885fb,0xf0e8edc5),LL(0x8b5187c6,0x4d9418ac),LL(0xd2a00019,0x70e99cb3),LL(0xe7f8a61b,0xf0da5be4)}},
   {{LL(0x7dd34fde,0x52704cbe),LL(0x2926bb6a,0x0fb7224a),LL(0xf2b0af92,0x0d58bddd),LL(0x0e9cad36,0x2f986a07)}, {LL(0x80e3a6f9,0xc85549d4),LL(0x322cb14c,0xa013e913),LL(0xf25ac201,0x8a19cf30),LL(0xffb8f2e4,0x130e4ce0)}},
   {{LL(0x0ce56c13,0x21ad2c8c),LL(0xb15f6a2f,0x13ed2106),LL(0x9453ce96,0xa217b5f6),LL(0x64e0bf9c,0x93b1cdc7)}, {LL(0xc4fe8e72,0x753d894d),LL(0xf3a3916a,0x46c6cea3),LL(0x383dd581,0xc1fb78e1),LL(0x17376a3e,0x1b7ba1a9)}},
   {{LL(0x5df66852,0xa1411287),LL(0xa30445d3,0x4e9d333c),LL(0x917568a9,0xb5a26c14),LL(0xe857a6ac,0x885f1857)}, {LL(0x84b1f8cf,0x05fbd3ee),LL(0x1e81e4e1,0x5c1f4097),LL(0x011f30e6,0x43999be4),LL(0xa890719d,0xa8aab3bd)}},
   {{LL(0xc7088eb2,0x49d598ce),LL(0xe341047c,0x7a892468),LL(0x07cb6075,0x8e69b5c4),LL(0x8c37dc04,0x83d066fd)}, {LL(0x6ffff7ac,0x4fcc6d02),LL(0x7edfb968,0x1afaaf74),LL(0x70d529de,0x2972b753),LL(0x08218b2e,0xf65bff0d)}},
   {{LL(0x4182b9fc,0x119b3c4b),LL(0x27b5e101,0xcab66591),LL(0x2ab87a02,0xfff2e939),LL(0xeec5949b,0x1c10c40d)}, {LL(0x30aa1242,0x98366224),LL(0xf225a4e7,0x833e9dee),LL(0x992e8475,0x07f1cfec),LL(0x1ef83a8a,0x377a9d79)}},
   {{LL(0xc6715544,0xaf1d0863),LL(0x1fd71505,0x34dd65c1),LL(0x04fed293,0x74d55c22),LL(0x86d2f3be,0x31b1e50e)}, {LL(0xc09594ac,0x876148b9),LL(0x8900b56e,0x73aace3b),LL(0xa2cf4c37,0x4617258a),LL(0xc6f38a92,0x554e8f16)}},
   {{LL(0xda0723bc,0xd8594800),LL(0xf3c8381d,0x524452df),LL(0x138ca980,0x846dfa02),LL(0xe2d32e56,0xaa77a80c)}, {LL(0x419c86b5,0x27573fbc),LL(0xb70216c3,0xe7486807),LL(0xc72036e6,0x8b7a685a),LL(0x15fae3d8,0xa1764627)}},
   {{LL(0x815f379c,0x0a1f2361),LL(0x01ab64d2,0x9811607e),LL(0xff2c75cd,0x31841038),LL(0x474982aa,0x8751674e)}, {LL(0x52a2523f,0x2f32b55b),LL(0xe85f2025,0x6ff8d2a7),LL(0x707b2dcb,0xd2ec31ee),LL(0x6e277971,0xdac81e59)}},
   {{LL(0x0e78191b,0x5445e3a2),LL(0x8c80db2f,0x134dba0b),LL(0x94002b55,0xe9925a87),LL(0x4293c71d,0xe56fa2be)}, {LL(0xa9d009c2,0x72aca4d2),LL(0x02fb0741,0x0c1219dd),LL(0x208fd227,0x689fbc66),LL(0xe4bb09d8,0x8266f2f7)}},
   {{LL(0x2a61b8bb,0x1a791f9b),LL(0x3eff4f21,0xb29b31b7),LL(0xab7812db,0x2f97803a),LL(0x880ceb4c,0xdbf27bae)}, {LL(0x45e9db5b,0xecb84887),LL(0x5cb7d0ec,0x3dfd84e1),LL(0x77c0b1e0,0xc89f61c2),LL(0xb7656544,0x7ada1d37)}},
   {{LL(0x910a966c,0x0bca9585),LL(0x6f12c20c,0x80385b47),LL(0xa4b30374,0xf63a1605),LL(0x104b4783,0x2f91b24c)}, {LL(0xb3ab423f,0x9210f5b9),LL(0x2fd424a6,0xb9aa656d),LL(0xf7e8d640,0x63c615d5),LL(0xbb59cfec,0xd567ff98)}},
   {{LL(0xf7692947,0x78121697),LL(0xbd9f5ed5,0xb9166739),LL(0xb64b20e2,0x58d9a4f4),LL(0xc9fcc93c,0x291898d9)}, {LL(0xd6c6065a,0xbce6509e),LL(0xb84834a4,0x39af658f),LL(0x94b49185,0x0f919d44),LL(0x5dbe7308,0x3b80fc51)}},
   {{LL(0xe321c228,0xb9fd8ae4),LL(0x360692ba,0x4a46bd2d),LL(0xd05b84b0,0x91d5396e),LL(0xd6b058d0,0x266e89fd)}, {LL(0xb2c42e38,0x6fb142d7),LL(0x994ebc2f,0x93c9fe18),LL(0x104b04a3,0x90e17885),LL(0x654eb6ac,0x6a5fa420)}},
   {{LL(0x3f349b26,0x26c8a9b4),LL(0xb4e528ae,0x39387f7e),LL(0x5eb46726,0xa74bea43),LL(0x9150b043,0x0b3e82dc)}, {LL(0xe2fc799f,0xc69ffac9),LL(0x48921338,0xd0479697),LL(0x0a4e061b,0x91a68264),LL(0x3f410bcc,0x93a6c41e)}},
   {{LL(0x6b1fb104,0xaea8d055),LL(0x31fe146f,0x2ff339a4),LL(0xcf63c413,0x3d7ef85b),LL(0x289a097b,0x1f0f57c5)}, {LL(0x5bda1160,0x82f2f83b),LL(0x6fea66e8,0x433eea4d),LL(0xcae11651,0x1f3fff4f),LL(0x6b1c243b,0xfa71c3fd)}},
   {{LL(0x674832a4,0x59f36add),LL(0x2891e4e6,0x7b6d3802),LL(0x084fa3c6,0x47b313bc),LL(0x6584c9c0,0x90003ac6)}, {LL(0xbc802849,0x9718c2dd),LL(0x2870ca08,0x9a5a2698),LL(0xcf68f352,0xb5cfe625),LL(0x6e6b0caa,0x90d0e2ed)}},
   {{LL(0xba64d50b,0xb30780c3),LL(0x7acb4fca,0x16328345),LL(0x84b258de,0xf64e01fd),LL(0x35dcd2f1,0x2a25873e)}, {LL(0xce4b39da,0x36606813),LL(0xa69a93e3,0x5285c91e),LL(0xdcb501d6,0x4da13aaa),LL(0x52e3dc24,0xb90d0a52)}},
   {{LL(0x60a57d0f,0x6882d15e),LL(0x167612fe,0x52142caf),LL(0x463d39cc,0x532ccfb1),LL(0xe5a969f3,0xcdecde85)}, {LL(0xd1bc4480,0xa89c1d1d),LL(0x83f32199,0x9373f362),LL(0x6d653c44,0x42f3493d),LL(0x6c80e27e,0xa867e4db)}},
   {{LL(0x5cb7623d,0x954fbd83),LL(0x0b83d55c,0xba8b3007),LL(0xe2b23256,0x71946b92),LL(0xfaf95492,0xe0a2a7bf)}, {LL(0x4e0c81ef,0x32ed3d91),LL(0x46f058d6,0xb8c8b14c),LL(0x67221924,0xc76c917f),LL(0x2ddf3cd4,0xd26c1d51)}},
   {{LL(0x4fc9b14a,0x184e1395),LL(0xc1969b8b,0x651a0c29),LL(0xc9d5bf9c,0x05687179),LL(0xebcd85b6,0xb2f18ed1)}, {LL(0xe446f1ef,0x8b662764),LL(0x71699f5a,0x6c0e051e),LL(0x27d93da8,0xf94a1151),LL(0xa05fe7a4,0x751235c6)}},
   {{LL(0x624e9ae2,0x40aaf88f),LL(0xf5f6e5c5,0x6499b3f5),LL(0x98157f61,0x01fb0b8e),LL(0x33827220,0x070438f3)}, {LL(0x50ab0b43,0x7409012f),LL(0x63c50e65,0xdbbba563),LL(0xc0d084ad,0x6b572ca3),LL(0x7b76cd6c,0xf10f6684)}},
   {{LL(0x0c34363b,0x32bcca97),LL(0xb40e8157,0x7a9cef10),LL(0x6eaec234,0x3d5ffc51),LL(0x5f23d481,0x7d7b41a5)}, {LL(0xeecdfe73,0xe5276c22),LL(0x8ac8c30d,0xa9b2725b),LL(0xed0c743b,0xee449588),LL(0x48df73b7,0x6d3b82a3)}},
   {{LL(0x023cb0df,0xcb52edc2),LL(0xd5a24591,0x08773a4d),LL(0xe12a9072,0x0d9a6aaa),LL(0x5bf5586e,0x4261f56f)}, {LL(0x60a08106,0x184b0402),LL(0xb09cfa61,0x1b398053),LL(0xd5dae483,0xdf7f55b1),LL(0x86ef2cde,0x9554210e)}},
},
/* digit=12 base_pwr=2^84 */
{
   {{LL(0x9204db30,0x564d6e85),LL(0x2aa84cdf,0x139bb928),LL(0x88476456,0x9413d7ea),LL(0x5a1ffa66,0x5c554483)}, {LL(0x2ed18080,0x7b863089),LL(0xd14e5daf,0x589aaf20),LL(0x7b5f81ca,0xeee4f96f),LL(0x1bb0b415,0x88d47007)}},
   {{LL(0x55c9bd11,0x1bb400d3),LL(0x06fc2851,0x8402465c),LL(0x65063b3e,0xa81ba22d),LL(0x6e1aa0c6,0xbab2dcbc)}, {LL(0xbe645e25,0xe5f43f1a),LL(0x4df84be1,0x62320533),LL(0x21a2eaf4,0x14ac7080),LL(0x58beb26f,0x3f946464)}},
   {{LL(0x7a82d20f,0x5f2a3e9a),LL(0x191011f2,0x399e015c),LL(0x886ac8e6,0xfbec312a),LL(0xeda47f96,0x0dd5140a)}, {LL(0x26b47318,0x0d4df313),LL(0xe6685ec8,0xe2c9ec78),LL(0xcd8442cd,0x4df119ae),LL(0x7b32a1cf,0xdb1ca955)}},
   {{LL(0x126506cc,0x7e2c5852),LL(0x08b3567d,0xba94aac7),LL(0xc05a3f24,0x6905cdf4),LL(0x3547f8b3,0xbf5f559b)}, {LL(0xaade7a1d,0x9e4b4e62),LL(0x1fda3088,0x56b8b9d6),LL(0x4c43d89f,0xea3eb4c6),LL(0x9c69e047,0xfb7e537c)}},
   {{LL(0xdfe5f6ab,0xc23d9491),LL(0xc1a9c0af,0x42fc362d),LL(0x127d2b35,0x04170b01),LL(0x04116aeb,0x4f0f17bc)}, {LL(0xc9184cf6,0x716c01df),LL(0x895ceae7,0x914dc877),LL(0x390bff2e,0x696b2ae8),LL(0xf88af5db,0xf6ccd628)}},
   {{LL(0x0f88095a,0xdada9bb9),LL(0x919ce305,0x7155c28f),LL(0x6d78b266,0x32a01e47),LL(0xb652c4f8,0x6da94459)}, {LL(0x827ea8ef,0xa31783a6),LL(0xbdb1af2b,0x4d69b7c6),LL(0xaf31dab9,0x2874eb38),LL(0xafd9bace,0xa0ed9910)}},
   {{LL(0x4037f17e,0x7d892e3a),LL(0x5f91a4fa,0x81fa9841),LL(0x961cf02f,0x17c7292d),LL(0x388bcc75,0x35af0c0e)}, {LL(0x127a29b0,0x340bec90),LL(0x3d087445,0x955714a4),LL(0xa587c273,0xfd430880),LL(0xd24dfda2,0x715ecd50)}},
   {{LL(0xaafd6cef,0x4ade066d),LL(0xf8c1decc,0xce59c8de),LL(0x77b96ece,0x3e12a24a),LL(0x44cc710c,0xee7c32fc)}, {LL(0x240e9bb7,0x70700e4f),LL(0x6a63b06e,0x837ada54),LL(0xd19644ee,0xa58ce980),LL(0x27e7451c,0xcaa5d14d)}},
   {{LL(0x387272fc,0x8e78d2ed),LL(0xfd8a0f13,0x9163a377),LL(0x635c55f0,0x858436bd),LL(0x5ba5b0eb,0x0a414f9b)}, {LL(0x7d7383b1,0x2b58373a),LL(0x6030a376,0x5e7b9d36),LL(0x543514ef,0x9c69af86),LL(0x26080ff3,0x044698cc)}},
   {{LL(0xa2e23074,0x76f54954),LL(0x17526081,0x90393264),LL(0xf3b78a50,0x0d095055),LL(0x69d8b26d,0x1f3a3776)}, {LL(0xf5e7c8fb,0x0575e3bb),LL(0xee40b0c5,0xee7dd406),LL(0x55dab556,0xe6522e5d),LL(0xb61cd918,0x2d1b5709)}},
   {{LL(0x01400b8d,0x0ea9278e),LL(0x6464f584,0x9552e745),LL(0x12fc094f,0x67f5645b),LL(0xde303128,0x77c40f3c)}, {LL(0x0e3f3381,0x16d7e9a5),LL(0x59947693,0x017795ab),LL(0x9222eaf5,0xb69b5708),LL(0x1b77f122,0x61b213e0)}},
   {{LL(0xdc8db00e,0xa7cc8bbf),LL(0x3aa7fc1f,0x1c51f5e4),LL(0xb4ac2d0c,0xb85b782e),LL(0x0468e5ea,0x32fde94b)}, {LL(0x7f7ff0a9,0x8ad5b9a2),LL(0x8fdbb3f9,0xcd26f418),LL(0x6ebf89db,0x853bc95d),LL(0xa066b849,0x1da0a323)}},
   {{LL(0x4bce0fa7,0xc4cc7aab),LL(0x6bc940f1,0xd4a05b69),LL(0x392dbd11,0xc77300e6),LL(0x21f70aae,0x0dc2eac6)}, {LL(0x4b2ad7e0,0x9d4b513b),LL(0xa6daee1d,0x19822167),LL(0x69b98eee,0x7d71d202),LL(0x35f3f150,0xdfd435dc)}},
   {{LL(0xddfd45ed,0x66d46ad3),LL(0xe50a2f01,0xf0325189),LL(0x3ec5683d,0xe19b9500),LL(0x91dd97e9,0xc46ab0a2)}, {LL(0xed682c4a,0x74c971d7),LL(0xa14da289,0xafedac2d),LL(0xe39ba740,0xd17838fe),LL(0x053536bc,0xeb497bca)}},
   {{LL(0xde6d4c38,0x551ba4ca),LL(0x4f52298b,0xa67be247),LL(0x9a5b40a8,0x98413188),LL(0xbb0acfb5,0x083a26aa)}, {LL(0x11d16ebb,0x4929ff5e),LL(0xa942ae7e,0x91f08b63),LL(0x876663ec,0xaa428ef3),LL(0x1e97cbb2,0xfaabd309)}},
   {{LL(0xf1edd62f,0xca0ed50c),LL(0xd29f48d9,0xc3c7ae6f),LL(0x8a72ae88,0xff47bf28),LL(0x348c6666,0x584ddfe5)}, {LL(0x36731fdf,0x271137e9),LL(0x88d98bc8,0x714bc7db),LL(0x0da6be30,0xcea912c1),LL(0xbe62d6a5,0x91cb844d)}},
   {{LL(0xec027bfa,0xe16ca42a),LL(0x17603e76,0x0c88f701),LL(0x63d5a31a,0x799418e3),LL(0xebb063f6,0x033bb53b)}, {LL(0x625d3909,0xbcd05461),LL(0x85f23129,0x2d7b7868),LL(0x95090997,0x23b07887),LL(0x18d2c218,0x216c08ae)}},
   {{LL(0xeebdbcf9,0xe1ccb6c1),LL(0xe873842e,0x89ca4552),LL(0x3c2fcdd5,0x4837f137),LL(0x108a8c0a,0x805874e8)}, {LL(0x3d442fa7,0xe7e524f4),LL(0xf8131f8a,0x580d82be),LL(0x93d3d50f,0x6dcb7d27),LL(0xb5b39168,0x51207d3e)}},
   {{LL(0x09110fe9,0x9a3ce117),LL(0x48721d93,0x8f3c6e4f),LL(0x87bdfa61,0x60a62b48),LL(0x7c01d84a,0x086dac65)}, {LL(0x53841493,0x4af7878c),LL(0xb3bd5aa1,0x3b1a8935),LL(0x902e5686,0x65c8445b),LL(0x2e3b1822,0xde16cfa5)}},
   {{LL(0x0a3e3684,0x19879e78),LL(0xee249180,0xec553912),LL(0xf8f4c1ee,0x8eb73fae),LL(0xb81fd20d,0xdee59877)}, {LL(0x20b5ece3,0x2452e63f),LL(0xb632dddb,0x17be9422),LL(0x94311e6d,0x01f89220),LL(0xa332f84f,0x8f0fe052)}},
   {{LL(0x1b9784d5,0x59657aab),LL(0xd8a7f347,0x6f2ce032),LL(0x6b95e6e9,0x84247793),LL(0x4395b044,0x34301cf4)}, {LL(0xf7fb5401,0x98ebfd98),LL(0xfcdb31a4,0x14fd494b),LL(0xf90e0481,0x042f89d8),LL(0x4134ab52,0x6b90a008)}},
   {{LL(0x7fe2ffec,0x8fa22555),LL(0xa778448f,0xc6dc3d32),LL(0x85f45aad,0x4886fedb),LL(0x51704d0c,0x5bdef90e)}, {LL(0xe2d1fdaf,0x46ad596d),LL(0x04126f0d,0x914e0090),LL(0xaef960a6,0x71aaeb18),LL(0xac77472c,0x8f4601e5)}},
   {{LL(0xd8d9768c,0x42e5a186),LL(0x00f6004f,0x8cbf3a6c),LL(0xc1ddebdc,0x9d4bf5ac),LL(0xa9c066fb,0x13354792)}, {LL(0x923fe808,0x72e0b81c),LL(0xc526d6e4,0x1e73b868),LL(0xa81f1e24,0x3f7bedc6),LL(0xe920ba24,0xed1ff363)}},
   {{LL(0x659604c5,0x58234c89),LL(0xce4b0872,0xa6a421ad),LL(0xcc19578f,0x5dc8848a),LL(0x4f28bdfc,0xfcb418d0)}, {LL(0x8d6442f5,0xf2e74820),LL(0x4dcf6378,0x0c481d85),LL(0x4556438b,0x4987d1a6),LL(0x3157c6be,0x76359363)}},
   {{LL(0x1c1dceef,0x29bbf3b7),LL(0x576f1dbd,0x0995c340),LL(0x8fa61304,0x0405db3d),LL(0xcc7d345e,0x63438f3d)}, {LL(0x942120e5,0x688174dd),LL(0xcd70c93c,0xc7dd05bd),LL(0x5e871ae0,0xdc8a32dc),LL(0x6178647a,0x1a7896b9)}},
   {{LL(0x59c437e3,0x1fc3f7a2),LL(0x24235e5e,0x737de2e3),LL(0x7a5eaabd,0x589a56e3),LL(0xcca140f3,0x5a79da8e)}, {LL(0xa12463fa,0x3d8b0d82),LL(0x0875daf5,0x63fc83d8),LL(0xbd9211f7,0x42a30803),LL(0x32d3935f,0x62f6167f)}},
   {{LL(0x6f269922,0x70cd6467),LL(0x96163b47,0xf694ca21),LL(0x5f5ba669,0xf3bafb2d),LL(0xb8ed8333,0xcf7cf341)}, {LL(0x9997edc2,0x34b2022d),LL(0x309c6508,0x57e6f4b5),LL(0x64841008,0xf6fbf864),LL(0xed075d44,0xbc9821f5)}},
   {{LL(0xf37cc6b7,0x78c80f73),LL(0x6ab88fc2,0x41d28626),LL(0x58ca26fc,0x2126981c),LL(0xbe3dbf87,0x7a956c64)}, {LL(0xce0ce9f3,0x2f41e27d),LL(0xf4c98e5b,0x0cb49ae0),LL(0xcace473e,0xba6224a6),LL(0x393e092f,0x25dddbc0)}},
   {{LL(0xa4fb974d,0x747daf46),LL(0xc76dbe2e,0xfb775fe7),LL(0x9670c22e,0xb7b3ad6d),LL(0x10a380bc,0xc6580b23)}, {LL(0x92087c3d,0x4ea226f5),LL(0xb53aa3c7,0xe67c379f),LL(0x991c3c9b,0x4133f831),LL(0x4fa0dd18,0x80f9e5bd)}},
   {{LL(0xc6f80fb4,0x0094e7c6),LL(0x351bebd3,0x16e99ebc),LL(0xaae16a6f,0xc555ed44),LL(0x2f6367eb,0xe9d2846f)}, {LL(0x83d46d0f,0xb34c93d0),LL(0x894fadc6,0xc0cb137a),LL(0xab31f937,0x21e289f8),LL(0x1bc72a35,0xac5e0516)}},
   {{LL(0xf3d4db0d,0x6221871b),LL(0xa039826c,0x72d1fdce),LL(0x668c8022,0x69d9cc8b),LL(0xfee064ff,0x0bf359ce)}, {LL(0xe8d16f19,0xb8e636b7),LL(0x443160ac,0xde88f403),LL(0x032836ee,0x4228177a),LL(0xe9801d86,0xee8fac37)}},
   {{LL(0x4626e343,0x496c9363),LL(0xf4e4c8fa,0xf6999578),LL(0xb8648a06,0xce7306f6),LL(0xae7996e5,0xe2775c8c)}, {LL(0xbf09d221,0x7b47e678),LL(0x515c2ace,0xf5251e1e),LL(0x77b48b41,0x087f9121),LL(0xeb38d74b,0xc40e7725)}},
   {{LL(0xce95134a,0x1d559f4a),LL(0x320c8bc6,0x1048a1bc),LL(0xe3085f1b,0xad2ddaf8),LL(0x0ad35636,0xf1cfc4cb)}, {LL(0x57db1e96,0x2bd8d4fb),LL(0xe1976ab7,0xd1813026),LL(0x15867022,0xa80e501c),LL(0x01f68017,0xecaf1497)}},
   {{LL(0x48ab68b7,0xd82c5e79),LL(0x204d2962,0xa0f117e4),LL(0x7dedbf05,0x99b3bda1),LL(0x52786ecd,0xb872dbff)}, {LL(0x57592d3c,0x56253c32),LL(0x4d570c07,0x495fbb05),LL(0xfaecad3e,0x073c49cb),LL(0xb46bad46,0xec8c1f57)}},
   {{LL(0xce3b07c7,0x13800a76),LL(0x0ffaec55,0x9bbf87d7),LL(0xaf2426c3,0xf69a9ee3),LL(0x2fd70c22,0x2d0c201f)}, {LL(0xc42bb661,0x957e5be1),LL(0x1dc771df,0x3e6ae19d),LL(0xe3cfafa7,0x60af970d),LL(0x5ebd1883,0x721ce869)}},
   {{LL(0xb87d0ede,0xab0a80a5),LL(0x2954a3e3,0x33576f02),LL(0xc413fc00,0xcc2fe8c0),LL(0xeb86a18b,0x5ae762bd)}, {LL(0x3fe6c6dc,0xbc309dde),LL(0xbf0d1eb5,0xb4f9d001),LL(0xd4fa748c,0xf3f3c5b9),LL(0x2ca78fdd,0x78e8867f)}},
   {{LL(0xcdf1624b,0x8f85f872),LL(0xa7db0248,0xfdce003b),LL(0x1ad9866b,0x0ad33ef7),LL(0x296248a4,0x27d12937)}, {LL(0xc99c656a,0x23bf35eb),LL(0x17753ace,0xcfb64da2),LL(0x6fbf7969,0x8bc0e741),LL(0xe719cff9,0x131018ef)}},
   {{LL(0xd1c02b67,0x98f4ef66),LL(0x1f81f578,0xe8aa6cdb),LL(0x159469de,0xa6f97fb3),LL(0xe3906d9e,0xf8e834cd)}, {LL(0x71bbd3d1,0x33ccda6d),LL(0xf952c038,0xeac76a4a),LL(0xe5b53383,0x2891eaa0),LL(0xedcf6de7,0xd629dbdd)}},
   {{LL(0xa3fb0fa1,0x4af093cd),LL(0x0d1ea294,0x130fd057),LL(0xb57747bf,0xb553cb13),LL(0x024e856b,0x107c0f0e)}, {LL(0xbd631fef,0xfd63a2ff),LL(0x12c01222,0x8df62ec2),LL(0xc0af11a9,0xacbce197),LL(0x5c4922b5,0x35fa3e80)}},
   {{LL(0xc3de57ba,0xbc257ccf),LL(0x293ad2df,0xb481ca1c),LL(0x2058e222,0xb123f3bb),LL(0xefe46989,0x219cde82)}, {LL(0xe9a316da,0x58ac87b8),LL(0xd4d25c91,0xa8294237),LL(0x62d14158,0xb54dad88),LL(0xb3da2a84,0x9250885f)}},
   {{LL(0xd54776bd,0xb4e3bedf),LL(0x78043ee5,0x81a4c582),LL(0x4eb87133,0x279a0963),LL(0xf2bfdb52,0x827d333c)}, {LL(0xed71e119,0x3601c6d1),LL(0x0d64df1d,0x3d9b1772),LL(0x3fa3c40e,0x2f5bcc09),LL(0x8e26aef5,0x74b7b30d)}},
   {{LL(0x3d3ac848,0x98fd949b),LL(0x92e259f1,0xd99e99d0),LL(0x8d353c77,0x34404265),LL(0x4d8dfb1f,0xffc05a7d)}, {LL(0x4e9d92c9,0xbaf2f471),LL(0x5ea9cef3,0xf354f8b2),LL(0xb8b2c8a0,0xf2be0fea),LL(0xfbce308f,0xa392d3e3)}},
   {{LL(0x02619258,0x58cd793d),LL(0xfea6eacc,0x16a8c9e7),LL(0xb90f9cb5,0x3fcae1ed),LL(0xd59bc4ce,0x1df76d07)}, {LL(0x8574a3ce,0x39248217),LL(0x03b6e82e,0x9d0df2b7),LL(0x33206733,0x64227c0f),LL(0xb342da7d,0xb909614f)}},
   {{LL(0xb8e15a20,0xe46e977f),LL(0x744eaa18,0xdf2aa89d),LL(0x7ff12f89,0xa40b36b7),LL(0x86b0e7d4,0xbf7ed788)}, {LL(0x9e044a5b,0x35930c5c),LL(0x4ac6b8a0,0x599cfa2b),LL(0xa8f06839,0x68f5f40d),LL(0xe8a1b9d5,0xe838649b)}},
   {{LL(0xdd45feda,0x2e3c91a9),LL(0x58de0625,0x5f73aa38),LL(0x7535cddc,0xcc2b2397),LL(0xca7825fa,0x60e69d0b)}, {LL(0x62424bd7,0x8f1a95c4),LL(0xf6f21e23,0x5e175a13),LL(0x4fa48b20,0x594e5b82),LL(0x9b14fed3,0x2bfed204)}},
   {{LL(0x74484bc3,0x87c925fc),LL(0x5639abc5,0x052b634f),LL(0x290426dc,0x169549b6),LL(0xdaaefd38,0xfe515a22)}, {LL(0xb4d87ccb,0x8a63a39c),LL(0x4034acdc,0x3dec5f62),LL(0x61090db0,0x59969d81),LL(0xf157248d,0xb089b8f7)}},
   {{LL(0x9d59a29f,0x42b0ca54),LL(0x9be7ee82,0x522b3e3e),LL(0xac166a7e,0x894aade2),LL(0x9184ec33,0x57aaf19a)}, {LL(0x5e50711a,0x84406a11),LL(0x1614f8d3,0x0cafd148),LL(0x3f7d19f8,0xc6174fdc),LL(0xff4958be,0xca5bed9a)}},
   {{LL(0xe4fdd396,0x8dc18aaa),LL(0xd371c3f4,0xf6e8a9ee),LL(0xa5dfefde,0xc6b58042),LL(0xfc4f3577,0xccc3bbb6)}, {LL(0xdedfdd55,0x9f583e4a),LL(0xb48c5fb2,0x9ea45133),LL(0x232d61e0,0xca2b3229),LL(0xb0b5cb38,0x642101a8)}},
   {{LL(0xa9ebda1a,0x0cfac5fc),LL(0xd2dc9c7c,0x02398bd6),LL(0x80591234,0xd95511d9),LL(0xe8230901,0x0e5cc99c)}, {LL(0x140eaba1,0x943350f6),LL(0xe0623c93,0x9fe19108),LL(0xd74e189b,0x052bf5d9),LL(0x40cd7173,0x3e341bff)}},
   {{LL(0xcb7d384d,0x89b5b355),LL(0x50b76f18,0xedee32da),LL(0x5804d9df,0x6a9cfb19),LL(0x376fc2d8,0xccf638f8)}, {LL(0xe14de014,0xebdce7a5),LL(0x7f606fa5,0x0135085f),LL(0x69b58c3b,0xf8a3de5f),LL(0x59ca19d1,0xbaa80445)}},
   {{LL(0x0ce7238d,0x3252147d),LL(0xd57bc36f,0xd446960b),LL(0xb275f5ca,0x9b1743ce),LL(0x27629de8,0xda048c48)}, {LL(0xd3bbac67,0x005354db),LL(0x1ba1facc,0x62c392fb),LL(0xa18da892,0xb066bfae),LL(0x367a8320,0xdb090711)}},
   {{LL(0x6f90249a,0xbb7092e2),LL(0xe22da86b,0x6de14155),LL(0xb38d4ad8,0xe16136d3),LL(0xd0fbb940,0x9deaa5c9)}, {LL(0xaacf50e3,0x54a54ba3),LL(0xb9ba4570,0x66e5645a),LL(0x48cb742a,0x77e28d94),LL(0xed98a2c9,0xc795b138)}},
   {{LL(0x1daa17ee,0x899331f6),LL(0x4a77734f,0xac950653),LL(0x71f3e3b6,0xd7f6304f),LL(0x65fc119c,0xe7256955)}, {LL(0xbe527794,0x3e60a04c),LL(0x7c578fb0,0xdaf53be4),LL(0xebc0754b,0xf785a4f8),LL(0xde1b78b4,0x8b21b116)}},
   {{LL(0x62fb1c56,0xfe47e04f),LL(0x229f1017,0x8a92f9e6),LL(0x68b7842c,0x2d73dd23),LL(0xa56dbc4f,0x3b43f7dc)}, {LL(0xd0f3f4ca,0x9435defe),LL(0x500594e3,0xdabfb1ba),LL(0x428f5ead,0x70e929e8),LL(0xbdc7716e,0x44adf585)}},
   {{LL(0x02204867,0x7b7ff077),LL(0x0c78476c,0xf2f306be),LL(0x7e783793,0x48849fd5),LL(0xaf77e3c7,0xc2dc3c7d)}, {LL(0xa980cdf6,0x5eb2b691),LL(0x204e25df,0x7ca7b7a4),LL(0xc5070eab,0x1e7c2f82),LL(0x4eb7cd3b,0x32ca4b36)}},
   {{LL(0xf94ad1ab,0x38ffde8f),LL(0x59921b25,0xb4757ae1),LL(0xb4d2f624,0x856cd3f3),LL(0x1eb40708,0x90593929)}, {LL(0x1193b3e4,0xffc4b89a),LL(0xbd2f804f,0x6afba7a8),LL(0x69dc21ed,0x72aabbaa),LL(0xe7fb6de1,0x5d1da32e)}},
   {{LL(0x98d1e26b,0x56c0f440),LL(0xf7cc7d6c,0x9456a6c3),LL(0x14f2f24d,0x9eb0aebb),LL(0x7dd788a5,0x51d7c699)}, {LL(0x46a22e97,0x053b8098),LL(0x8c025be8,0x27d8ea2a),LL(0x10d5afaa,0xe0bd464a),LL(0xe7cf120c,0x137c452d)}},
   {{LL(0xd091397b,0xd06bd227),LL(0x21bc796f,0x4b307bf3),LL(0x7f5a37b0,0x701eaf3a),LL(0xac7d4718,0x8d5a0f61)}, {LL(0xed8b1a37,0x0cf9eea3),LL(0x2aa9061c,0x10854f10),LL(0xa30eb4e6,0x0aaf430c),LL(0x2a050dfb,0xb74342f5)}},
   {{LL(0x20e1899f,0x2feee9d7),LL(0xf2a1dbfc,0x49464a8e),LL(0x5762d68e,0x4d7cf25e),LL(0x7bf43462,0xe7b6e759)}, {LL(0x79daf6e0,0x71fce284),LL(0x03858705,0x2d3ff71f),LL(0xbc4af4e6,0x07d8d288),LL(0x18f1c7d4,0x6777d197)}},
   {{LL(0x0e85f036,0xb5770041),LL(0x4c8d9768,0xe1bb263e),LL(0xe3917798,0x4fcc1d44),LL(0x07abcde4,0x274d1d90)}, {LL(0xb7a10472,0xc9b8ae9f),LL(0x8d10e8ec,0x6632e3be),LL(0x50f3a172,0xb6876fb0),LL(0xb4cf4867,0x753692d4)}},
   {{LL(0x58e598f7,0xfe3624e6),LL(0x6d81fb40,0x15f90418),LL(0x9bea3649,0xae762f7b),LL(0x161e85cb,0xc48d2262)}, {LL(0xcf5a21f0,0x8e8726a1),LL(0xa1f6653b,0x536c441f),LL(0x67ec5b86,0x0716bad0),LL(0xb2147d1f,0xa423a957)}},
   {{LL(0xdca2e393,0x8eec96c8),LL(0x2843ef12,0x3619e36d),LL(0x2ef695e1,0xdc16fe2d),LL(0xffea8124,0x04ed2cad)}, {LL(0x180ce636,0x5018a0ce),LL(0xdce7b2f8,0xc34b0bbf),LL(0x0c54fc30,0x645a02a9),LL(0xf3f819d9,0x6ee6772b)}},
   {{LL(0x7cecded6,0xe2bbbdcd),LL(0x3f038851,0x9ae4fd55),LL(0xa2f316c7,0xc30664ab),LL(0x63ffb50a,0x3cccf4a1)}, {LL(0xd00fb8f2,0xc37ee6ca),LL(0xad906eb1,0x593db6d5),LL(0x4aa84505,0x8f75b594),LL(0x9e5939f0,0xeff39d82)}},
   {{LL(0xc064f530,0x4b7fab3c),LL(0xde175892,0x731153ae),LL(0x3d4c4e60,0x335e6503),LL(0x776ce13a,0xb0876a8a)}, {LL(0x22241ecd,0xa8a566ee),LL(0x011e861c,0xb7456b3e),LL(0x177dd490,0xa9aff4eb),LL(0xc8f77c40,0x189b1ed9)}},
},
/* digit=13 base_pwr=2^91 */
{
   {{LL(0x2857a1fc,0x624de687),LL(0x2ff8f505,0xbd0a0d9c),LL(0xc381bc9a,0xeecb4fad),LL(0xfa94e41b,0x72386292)}, {LL(0xe75fc753,0x354d3f83),LL(0xa7a5a6bf,0x06afc753),LL(0xb2f568dc,0x1ce792ee),LL(0xbd2f9647,0xc5faaee3)}},
   {{LL(0xf912b74f,0x175fbeb0),LL(0x6e0ceedd,0x45fbe8e1),LL(0xd9233ee7,0xf0e1aa68),LL(0x406a626e,0xe55fc1ce)}, {LL(0xe08712e7,0x20efa1b9),LL(0xbcfd6360,0x5fd108b5),LL(0xeec1edac,0xea431df6),LL(0x940803f1,0xae1c0521)}},
   {{LL(0x15407ffe,0x584a16d0),LL(0x08a82a69,0xa977f702),LL(0x67f8a198,0x52eefecf),LL(0x19f7a7e0,0xec213738)}, {LL(0x35987b9a,0x6795cfef),LL(0x97028480,0xb243403b),LL(0x9c1b9124,0xac24b12b),LL(0xa90f8aeb,0x1f379501)}},
   {{LL(0x64bc0f09,0xa8e97fb6),LL(0xc953cd08,0x0b913991),LL(0x7fc3bf00,0x8385a1b3),LL(0xb09ccd8f,0xb6e74dec)}, {LL(0xec473ea7,0x6e1df026),LL(0x530766bd,0xf2f7fbbe),LL(0x3292052b,0xf18cb47a),LL(0x9114866a,0x7f8d4592)}},
   {{LL(0x8bfa2c22,0xf0a1c565),LL(0x2b326c0e,0xc28518c3),LL(0xec107d66,0xabafc6f0),LL(0x8478907a,0xbc7a6abf)}, {LL(0xa2920288,0x8c1c8f6a),LL(0x930c043e,0x6c87579d),LL(0xb309696d,0x25ee808d),LL(0xb7a71041,0x433bbbda)}},
   {{LL(0xb3086691,0x48d6d957),LL(0x26640916,0x9946a29b),LL(0x43db59a9,0x932ca93c),LL(0xe4fe91ba,0xaa61a0c5)}, {LL(0x815bf003,0x9e22e112),LL(0xc86ba8d3,0xa9ed1b18),LL(0x1069f434,0x1b5d3c14),LL(0x1cc01754,0x3cd2ebd0)}},
   {{LL(0x3350f670,0x5c06b244),LL(0xf6f9c751,0x7557dc9d),LL(0xde66fd97,0xa7ebd3b8),LL(0x2befe6fe,0xc126dbaa)}, {LL(0x396f434a,0x312f4897),LL(0x61a4124d,0xe05cfcd6),LL(0x1525c05e,0xc83b8688),LL(0x11899f64,0x4646dbf2)}},
   {{LL(0x8e419e08,0x2b7507cb),LL(0xaf855eec,0x785328d7),LL(0x7b8683a5,0x875db0c7),LL(0x90a597e9,0x3d1bc968)}, {LL(0x47eeeab4,0x7d4afa10),LL(0xd680ca71,0x2668dd43),LL(0x17365023,0xc3210d1f),LL(0x17fb31cc,0xd5bb2ee4)}},
   {{LL(0x08e9ba09,0xbefb6a4f),LL(0xb0c1b6e1,0xc6beedb8),LL(0x3510ef35,0x59daf057),LL(0xdbbabc65,0x604047cf)}, {LL(0xa06b7340,0xfabc80a8),LL(0xdf765977,0x7433dee7),LL(0xfd807cfb,0x149a2c4a),LL(0x3480a086,0x14e8ad3b)}},
   {{LL(0xb22c5f89,0xb0c3156f),LL(0xbf78675a,0xd10ece4a),LL(0x80b8ad9f,0xe270e317),LL(0xb0c2b420,0xfe7a6210)}, {LL(0x125ef635,0xf091d738),LL(0xc1a6f202,0xf1f277d6),LL(0x3587d9bb,0xe2727e7b),LL(0xb3e2b84b,0x83b209a9)}},
   {{LL(0x7a13effa,0xc9eb445d),LL(0x0d697480,0x89b856f1),LL(0x25c03cb7,0x834bbae2),LL(0xe0b4a7b2,0x0d8adb85)}, {LL(0xc7fbc240,0x7b6884af),LL(0xaa4f9097,0x6b485409),LL(0x290c106f,0x4d0a367f),LL(0x3f0efdfd,0xab87d218)}},
   {{LL(0x50f2b65b,0x15b9bab7),LL(0x5e5d53e4,0xa7403d4b),LL(0x28529212,0x2e23e376),LL(0x6e050767,0x6fe903a2)}, {LL(0x6cf570fb,0x4c5291a1),LL(0x7a30b326,0x4bfb8607),LL(0x27c572a9,0xec4905f8),LL(0x0f381c31,0x72eeb8c9)}},
   {{LL(0x460adca0,0x33346cec),LL(0x7b34756a,0xd4d5bba8),LL(0xeac84add,0x02b2e2d4),LL(0xdc1053b5,0xa129845b)}, {LL(0xdca6f9ce,0x53f067e0),LL(0x3526aba6,0x6e9998ed),LL(0x1c0982da,0xa4aef9e2),LL(0x93f5d96f,0xfe5b606e)}},
   {{LL(0x9c14b699,0x26b8502e),LL(0x0948a291,0xf1bcdca6),LL(0x2aefd415,0x73e43a32),LL(0xd1e2cfb5,0x7f523357)}, {LL(0x97d3fa94,0xa60151c0),LL(0x72129630,0x820c0d58),LL(0x5854acf5,0xb8f2e1ed),LL(0x3c656ac3,0x86d6646c)}},
   {{LL(0xbef1d0da,0x2284a612),LL(0xa8c8faba,0x2e7c5f4e),LL(0x70303ea3,0xfd441ae7),LL(0x5161cf82,0x9613f329)}, {LL(0x2e19531f,0x65a3cc65),LL(0x34281f69,0x177a2775),LL(0x7c82e094,0x0cc692a4),LL(0xb6f377f0,0x9d62a55b)}},
   {{LL(0xf96ec2b8,0xa24cf6ac),LL(0xa961cc16,0xd06747c3),LL(0xbd17f0a2,0x57c7001c),LL(0x34afe2d6,0x5f298db0)}, {LL(0xdf12f671,0x51b01ef2),LL(0x5ce712fe,0xc01c5066),LL(0x92a74776,0xac0f4034),LL(0x08d696bd,0xa3e9934f)}},
   {{LL(0xe7daaff8,0xafb6981a),LL(0x73bdcafc,0x5f8998d9),LL(0xbaf9906c,0x23ec39e1),LL(0xc999c9c0,0x5e248410)}, {LL(0x17dad895,0xd14c7a89),LL(0xcbb3f6b9,0xfde9d01a),LL(0x5f698f1b,0x1d6b26ef),LL(0xf0baff97,0xc6495cd1)}},
   {{LL(0x587674ec,0x5a72dc07),LL(0xdb09cd65,0x100f9ff0),LL(0xb30cf6e6,0xec0fb71f),LL(0x81066143,0xf54cb597)}, {LL(0x633857c4,0x0090e997),LL(0xda92c5d2,0x7326ed15),LL(0x47c56e86,0x794cd8af),LL(0xf89214c9,0xb272112f)}},
   {{LL(0x3445879d,0x37960861),LL(0xf2fcfc55,0xc5e496b0),LL(0x6559e153,0xfe74e95f),LL(0x54a772af,0x1e18b2b5)}, {LL(0x157c157c,0xd146980c),LL(0xa11d77b5,0x31ee3f25),LL(0x5707db6d,0x7762a07d),LL(0xbd2022b8,0x00804bcb)}},
   {{LL(0xd571c59e,0xdf3f4658),LL(0xcf45c1ee,0xc49e7a34),LL(0x43972cff,0xf401ba3d),LL(0xe509c2b6,0x146d989c)}, {LL(0xeb72392f,0x7c68d6c8),LL(0x0658b8e6,0xdd048de5),LL(0x9a0aeb72,0xc9dc61b7),LL(0xb741d644,0x98b080e0)}},
   {{LL(0xb1c5982a,0xa6ec0eed),LL(0x5ebbc26f,0x58d28317),LL(0x33e5b7dc,0xac8f1e1e),LL(0x9d8f9fed,0x31e4f65e)}, {LL(0x904ad76d,0x6c9af383),LL(0x9bdb0517,0xfc38c53c),LL(0x0e69f85e,0x9ae278ee),LL(0xefd9d887,0x18b362b7)}},
   {{LL(0x5bbbd3ac,0x65a5f74b),LL(0x077bfb4f,0x41eb4593),LL(0x83b38100,0xb642934b),LL(0xac1a99bb,0x643ceed7)}, {LL(0xee7cd5f7,0x9c27e66d),LL(0x6ddbaa6b,0x2ccf87d5),LL(0x447b1192,0xd51ca739),LL(0x95f5f142,0x78471053)}},
   {{LL(0x3a650829,0x915f50cd),LL(0x898a6a1c,0xe032bdc5),LL(0x2d15959f,0xde8fb4f1),LL(0xbad56140,0x1fc5fc73)}, {LL(0x8e60c3c3,0xdafbf206),LL(0xe428adb5,0x4963dc95),LL(0xd49584fb,0x1538e081),LL(0xbc0e5fa9,0xb34add66)}},
   {{LL(0xa7f28b2f,0x404ecf12),LL(0x7fa9c435,0x6ddc3ce1),LL(0x61ee755e,0xda887e3f),LL(0x8f71280a,0x4b5da661)}, {LL(0xdc79a4cd,0xee5a86df),LL(0x99be4d36,0xd8514b8a),LL(0xcc82c751,0x674793ea),LL(0x437aedcd,0xf3a2123a)}},
   {{LL(0xfcd6f027,0xf825ff37),LL(0xa681a001,0x60a056d8),LL(0xaa92c135,0x92a39248),LL(0xdcd190a7,0x61884e23)}, {LL(0x24cc911c,0xec0d1420),LL(0x5aa16ad7,0xbdb0baae),LL(0x8a1694d7,0xf12726b5),LL(0xc93673f9,0x8c7cf113)}},
   {{LL(0x7f2edc38,0x02fb6c69),LL(0x2fbe8690,0xcc4d4304),LL(0xe89c80d5,0x405b2491),LL(0x3d938bc1,0xdef46c76)}, {LL(0x2520e3b0,0xd92ec0fa),LL(0x1fe2dfda,0x2501cfa3),LL(0x1d5c8474,0xe7c5753d),LL(0xe6226dcf,0xc059abc0)}},
   {{LL(0x55a9011d,0x2dceefe6),LL(0xbbbbef00,0x8799064a),LL(0x0b49b5ef,0x7fe944c2),LL(0x225b21dc,0x722bbef0)}, {LL(0xd2bb14af,0x84687cbb),LL(0x9b6f6caf,0xfc4ab4f0),LL(0x2c146a52,0xb7b7bb59),LL(0x1dfea10b,0xb90d67f2)}},
   {{LL(0x713e1d30,0xca4ca8c8),LL(0xf8a13db8,0x50cbb994),LL(0xa5b0e3e5,0x2bee12b2),LL(0xe71e19fb,0xa7c5d6d1)}, {LL(0x9e0314cd,0x28442423),LL(0x66cda5c0,0xc95c2746),LL(0x1c5ffd19,0xfe52a79a),LL(0x38a0339f,0xb93875cc)}},
   {{LL(0xb49fb049,0x6a94196c),LL(0xcc12a38d,0xbeb1eb4b),LL(0xf5e8d167,0xbc136771),LL(0xd5f8ae87,0xa91e1232)}, {LL(0x95172460,0xb2e812c7),LL(0xb8f81870,0xc699d376),LL(0xa8900827,0x9e4a3b70),LL(0x506c0b29,0xe0d4b2f4)}},
   {{LL(0x7246fd96,0x13b4d1c7),LL(0x33965581,0x84ea2158),LL(0x2e53c024,0x9b9f071b),LL(0x864a1b78,0xcb005908)}, {LL(0x3f742c2f,0x03daddf5),LL(0xdf595911,0xd29230e5),LL(0xca0406a1,0x3f7d4e6b),LL(0xb1db7e47,0xeb646f66)}},
   {{LL(0x590e3107,0xb896003e),LL(0xf754ac01,0x7a0dc361),LL(0xe63ab0ac,0xe877a6f3),LL(0xdf60d307,0xd43b54f3)}, {LL(0x59cf0add,0x65ef91ba),LL(0x18990eb4,0x35e99393),LL(0x8e46fbf6,0xc186ab16),LL(0x8c1eaa91,0x4c0eb22f)}},
   {{LL(0x1abd31f0,0x4599b894),LL(0x9a1da7d3,0xdb34198d),LL(0xa0f0217d,0xa8b89523),LL(0xe56b884e,0x2014cc43)}, {LL(0x49efd4ee,0x6fb94f88),LL(0x287f4ae0,0xf1b81710),LL(0x99fd2deb,0x89d38a9a),LL(0x72b67a53,0x8179277a)}},
   {{LL(0x1a03755b,0x0ef6ce56),LL(0xfcdb3469,0x8dc768f2),LL(0xa21d959d,0x0be58a91),LL(0x9b103cd0,0xea44861a)}, {LL(0x808b8a46,0x332e86e7),LL(0x8772c3f8,0x9882015c),LL(0x9f4b5d29,0xe6b272fe),LL(0xa29b023b,0x0e183a28)}},
   {{LL(0x2286ebf3,0xf2fab88f),LL(0xfce83e6f,0xb7532ced),LL(0xe0cde4fc,0x17999d7c),LL(0xc1b7668a,0x7230fd85)}, {LL(0xef588309,0x97a57d39),LL(0xf906f6e7,0x7e175f28),LL(0x72b70bfe,0x51f67413),LL(0x2f82218c,0x2132f595)}},
   {{LL(0x9d8727cb,0x9cc0746e),LL(0xbba1ec8e,0xa2af77fb),LL(0x31a67cc9,0xc75aee60),LL(0x57408325,0xaeab9e0f)}, {LL(0xec34bb89,0xf24de697),LL(0x5d958bdf,0x06b90039),LL(0x0603d6cc,0x6f55222e),LL(0x2eb0b239,0x496537b5)}},
   {{LL(0x8be08323,0x083e5889),LL(0xf8dc0a78,0xc573596e),LL(0xe8901eca,0xc3e988fa),LL(0x6e350257,0x7f7b48f6)}, {LL(0xa216e329,0xed820567),LL(0x8ce989c1,0x55f46737),LL(0xeeab9441,0x7f48c5f1),LL(0x86fe0831,0x1d3cac11)}},
   {{LL(0x408a0306,0xe0364bae),LL(0x7a4eb2cb,0xe8d8aba0),LL(0x1fd7d5da,0xe548725e),LL(0xed5ed958,0x8de04491)}, {LL(0x61d73977,0x3e75eba2),LL(0x55420386,0x4f580400),LL(0xd859a690,0x54642fa4),LL(0x296e336e,0x2c905f7e)}},
   {{LL(0x22e260bc,0x4e287e66),LL(0x4a28d5bd,0x71a2ec99),LL(0xa7c5c3e3,0x5528da21),LL(0xa49471e0,0xae9f6856)}, {LL(0x587cd94f,0xdcd8e66b),LL(0x6c7b7db8,0x91afbd79),LL(0x067e3cdd,0xdf2e6625),LL(0xa6375f59,0x15b5a329)}},
   {{LL(0xb6586c5f,0x3b8b3b1d),LL(0xd34f10fb,0xe4d50a77),LL(0x7c3c01f7,0x26cb86f5),LL(0x8c57e6f7,0x36e9d3cc)}, {LL(0x62c6dbae,0xaa8e7ce1),LL(0x60d7fae5,0x7f6b7689),LL(0xc797ee16,0x519a7659),LL(0xb36a6b1b,0xa1c7b30e)}},
   {{LL(0x74dff201,0x8da05ba6),LL(0x40d0a835,0xd2eac07f),LL(0x610a7d6f,0x2701eb31),LL(0xbf893c4f,0x5c17a91e)}, {LL(0x6bc8b161,0x68b92e88),LL(0xf52e6ec0,0xa312fd5b),LL(0x6b7952cf,0xf7daf460),LL(0x18aeb57a,0x847f0cf3)}},
   {{LL(0xb0146708,0x27b178ed),LL(0x54ca2aa5,0x85a23554),LL(0x395a7b16,0x80dd123c),LL(0x0058bfce,0x64a9337b)}, {LL(0xf4addc4a,0xf6ae9380),LL(0x464536f1,0x0f84385a),LL(0x16534f6c,0x41fc2270),LL(0xb8795ec3,0x13d8976f)}},
   {{LL(0x8e12c560,0x2e90b3e4),LL(0x239b2c58,0x242a86ec),LL(0x0768275c,0x6fb42ecc),LL(0xbd96de9e,0xee341cd0)}, {LL(0x84355d11,0xfd1833ac),LL(0x5f55ec6c,0xf062392c),LL(0xfee33fba,0x6ee7b59b),LL(0xabf86e0f,0x8113f0ca)}},
   {{LL(0xcc68033c,0x2285aaaf),LL(0x78430646,0x850b732b),LL(0x2b3fa03d,0x50fa4b61),LL(0x3caf050d,0x4d893ecc)}, {LL(0x988df419,0x454764e6),LL(0xfb61f1a4,0x055d8a4b),LL(0x8475e07a,0x3b7c5f4b),LL(0xa6a228e4,0xf93a198b)}},
   {{LL(0xec8d566a,0xe0a8ce61),LL(0xc55f4bd6,0xe41397d6),LL(0x654bdf55,0x4cc18d48),LL(0x9325ac25,0xe1b49f9e)}, {LL(0x72c68871,0x79840752),LL(0x6d806fe8,0x8930d8b5),LL(0x0bd5f65e,0x11c8b5a8),LL(0xbf37d7a8,0xe931c025)}},
   {{LL(0xae8af083,0x25b17fd9),LL(0xde4215ed,0xd589fd8b),LL(0x4b3f61fd,0x56378f04),LL(0x6bfb4f9a,0xf0f39213)}, {LL(0xe906cc6a,0x6b0f9596),LL(0x096f8296,0x441f13da),LL(0x1e4940e0,0x08f933d4),LL(0x5a53e7ee,0x6c35391c)}},
   {{LL(0x19c3d18e,0x5f41c4d3),LL(0x1d389d95,0xc0804e09),LL(0x18a5a3f2,0x7323a9ab),LL(0x410a6381,0x7b7c2475)}, {LL(0xb02cfe18,0xd362eb9a),LL(0x553b2970,0x79ef3d0a),LL(0x3d2acdab,0x371f7760),LL(0x7f241dfd,0x6cd37890)}},
   {{LL(0xdf4a28e4,0x592a242e),LL(0x1bb45217,0x1e623cdc),LL(0x494074d3,0x5a9633a6),LL(0xd52fbfd8,0x81b74307)}, {LL(0xdec4c5ff,0x98292651),LL(0x3e0f6edf,0xe1b7bc86),LL(0x6bb8fb31,0x3d5fd86a),LL(0x1cf29f19,0xa830e9a2)}},
   {{LL(0xcf69c747,0xfffc5482),LL(0xa83549fd,0x7748a0f4),LL(0xe7ccf4a6,0xba1c8a0d),LL(0xa2ede6b7,0x6cd1399a)}, {LL(0x87bb90d9,0x8fb634e6),LL(0xc59a5304,0xfa8e659b),LL(0xa9122d95,0xcd6bfc75),LL(0xdfa6d75a,0xdb107def)}},
   {{LL(0xcc27760a,0xb0ec4cfc),LL(0xbed3a1a1,0xf24c1e22),LL(0x819bffc7,0x4f8522a1),LL(0xa93d97e1,0x263c7b5b)}, {LL(0xa4b4de49,0xab1d31e0),LL(0xebbfe8f5,0x374e968b),LL(0x51ca0d08,0xe82e9756),LL(0x7df3f2df,0xc05715a2)}},
   {{LL(0x038004ad,0x941f02c5),LL(0xa0fd46d4,0xc136a2a5),LL(0x3d31d26c,0x85db7d24),LL(0xbfefeecc,0x05bba6af)}, {LL(0x5a60aebf,0xf803b539),LL(0x813d0e6d,0x9bb8a479),LL(0x066abdfb,0xb689c813),LL(0x0072e530,0xd93b3f4b)}},
   {{LL(0x987446ad,0x242140a4),LL(0x06a02f0c,0x40b3f709),LL(0xa0fd6018,0x33f9bf20),LL(0xf21abfdc,0x58517c18)}, {LL(0xc1f80f3f,0xa33dc5db),LL(0x7ec91c80,0xbb7dfe27),LL(0x8ca97dd8,0xd2cf9338),LL(0x32e43d44,0x5761f871)}},
   {{LL(0xe513ea90,0x3c8ffb0e),LL(0x79bcdecc,0x91ecda36),LL(0x9b1a5514,0xdad3fdd5),LL(0x640d3af0,0x8fb630f9)}, {LL(0xf9d2e0be,0x82949b09),LL(0xeba23663,0x079ca0ff),LL(0x135238d4,0x51e62c53),LL(0xc446bd67,0xf5fa0c61)}},
   {{LL(0xe8da43d6,0x19dcdd2f),LL(0x95f29b5b,0x46fbf7ea),LL(0x635e8388,0x7f3eaa05),LL(0x5369750b,0x5ef817c3)}, {LL(0xc860c4aa,0x06025893),LL(0x5551c9ef,0xa2f6684d),LL(0xfbc0c667,0xd6af67dc),LL(0xcd2fe44b,0xfd8d7394)}},
   {{LL(0x302a17cc,0x011968ae),LL(0xc3e5a5cb,0x2206ff24),LL(0xa20dbfb7,0x4c7f0df3),LL(0xa395dd6f,0x59566376)}, {LL(0x373ea76b,0x68ff3d9f),LL(0xf6cf8ada,0x2394f93a),LL(0xe7514a94,0x3acc5dba),LL(0x5ddfa11b,0x0340da7a)}},
   {{LL(0x1a05155d,0xc3f03022),LL(0x4f7656c0,0x6cbbdc6b),LL(0x0b0875f5,0x6e30dbdd),LL(0x3471b0d5,0x5e7c2883)}, {LL(0x408b4bc8,0x49cfd71c),LL(0xf01c002b,0xd29a184e),LL(0xff415b0f,0x308be85c),LL(0x01a8fe7d,0x1b4176f0)}},
   {{LL(0x0c33bed3,0xb850acc7),LL(0x23af7af0,0x76aac640),LL(0x21d5853f,0x049187ee),LL(0x6620195c,0x44fbf6e5)}, {LL(0x36158178,0xf0abf14b),LL(0x90e419c2,0x9866ffca),LL(0x9e8523a8,0x7522e277),LL(0x08e90f1d,0x2f2590f3)}},
   {{LL(0x66d3f75b,0xde1c0c52),LL(0x6c299b57,0x47dc9ceb),LL(0x51f7f2b5,0x4ad12847),LL(0x452b07a5,0xeedf9d8d)}, {LL(0x3dad76c6,0x207b0627),LL(0x5d4c0280,0xccbb5201),LL(0xb019ae8d,0x0bdca05b),LL(0xf2da7eb4,0xb5f8d088)}},
   {{LL(0x4626c00c,0x4e79a0be),LL(0x5af82c0f,0xf6fdd64f),LL(0x5f7cba0f,0x7a828224),LL(0xb0765493,0xc2214834)}, {LL(0xabd53ccf,0x5b0d0d1a),LL(0xf4a1b517,0x3b03a22d),LL(0x6ece453e,0xb235c862),LL(0xf66471c7,0xf43ac344)}},
   {{LL(0xa1552fa5,0xeaff93dd),LL(0xcf3ae702,0xef1b40dc),LL(0x9ca613a4,0x35ced3fd),LL(0xa2f33a0d,0x90e350ab)}, {LL(0x002b5738,0x47bb89aa),LL(0x032b8b08,0xafc01bba),LL(0x61588b4b,0x688ae119),LL(0xcf66ef14,0xdb7d820a)}},
   {{LL(0x084910bf,0x83b654db),LL(0x60ea75a1,0xbb581f8b),LL(0x1cdae7c4,0x6917f282),LL(0xa40a07c3,0xb719f931)}, {LL(0x31297b3b,0xf8efb8b9),LL(0x4d6fc5c6,0x74246ded),LL(0xd2c61503,0x5a111754),LL(0x71654764,0xf64d2b88)}},
   {{LL(0x6ad8f168,0x4b452692),LL(0x0cc6fc91,0x907beb21),LL(0xbf13c18b,0xe876d523),LL(0x4cf37ca1,0x4d28e457)}, {LL(0x6d3d1172,0x4c0dc22d),LL(0x5a753137,0x7935a8d2),LL(0xda44d652,0x03484e3d),LL(0xc50025a9,0x05a3d80f)}},
   {{LL(0xff477c6d,0x6d43c503),LL(0x1ccd416a,0x35f4c4cf),LL(0xd5088349,0x7070f471),LL(0x281d30c8,0x678460ca)}, {LL(0xc6defb33,0x8923cd9a),LL(0xe2557cab,0x44930f56),LL(0xad156c4a,0x33b020bb),LL(0xbcaf4805,0xfdab31e3)}},
   {{LL(0x864b5564,0xffe79bec),LL(0x10c60d52,0x0510e352),LL(0x328a652e,0x66203aaf),LL(0x54fea042,0x9d5403bf)}, {LL(0x6e5e5c7c,0xb3fe6743),LL(0xecc66e02,0x6deef667),LL(0x44eacacb,0x199ee152),LL(0xd8803fd9,0x9f49fcd4)}},
   {{LL(0x2d3a6e28,0xdd5fee9e),LL(0x686d8ca3,0x8eed56d4),LL(0x78083491,0x36889a27),LL(0xbef20457,0xea1a6555)}, {LL(0xa501e2a3,0xe7e6b609),LL(0xfb23de2f,0x1ea0ae29),LL(0x632c9a6b,0x5f537d07),LL(0x9a3db961,0x61770d1f)}},
},
/* digit=14 base_pwr=2^98 */
{
   {{LL(0x7497e8a5,0x325c60db),LL(0x8c6949a9,0x05d8eab8),LL(0xc7bd5898,0x3169e466),LL(0x192d8e3f,0xadc06264)}, {LL(0xd55959fe,0x1ff468f4),LL(0x202dba19,0x97b33ee0),LL(0x21cf84bb,0xaa0c3fe2),LL(0x04a8d176,0x48cdc0af)}},
   {{LL(0x9c0d4009,0x53d8c448),LL(0x2e24dbaa,0xd3714617),LL(0x9b62e5f1,0xdd92e730),LL(0x9922cc8a,0x97b344d7)}, {LL(0x0bfe3e8f,0x416b009b),LL(0xf3c82269,0x56873834),LL(0xe6623555,0xf82a980f),LL(0x5ce68e54,0xb027ecaa)}},
   {{LL(0xfe87680d,0x005a4b24),LL(0x4cf6ee32,0xd92532dc),LL(0x4bd3e01f,0xfcd27c8c),LL(0xe1b59ffa,0xda7d9949)}, {LL(0x2735d373,0xe3d5f31f),LL(0x288e71fb,0x7e139ca5),LL(0x93979cb4,0xe474bc80),LL(0xf6fcc665,0x7f4f6017)}},
   {{LL(0xfbd613c3,0xe6982c86),LL(0xc366f17a,0xf4af69d5),LL(0xb85c2343,0x8683eed6),LL(0x24bc2116,0xf5bb244a)}, {LL(0xc9fc77d4,0x997a74bc),LL(0x6f44b54b,0xe202eb91),LL(0xa6997e76,0x77886412),LL(0x02c8837e,0x6996c8fb)}},
   {{LL(0x2c61382c,0x0986df8a),LL(0x667c8ee9,0x90607b92),LL(0x084eacde,0x051fcbf7),LL(0x6e685877,0x84e3dba4)}, {LL(0xe458da50,0x35861a82),LL(0xcf392b51,0xd036823f),LL(0x3dd86e74,0x43181481),LL(0x3741a385,0x8dcfe17d)}},
   {{LL(0x40f56786,0x8e1a77cf),LL(0x6d4b7774,0xc5bca7f6),LL(0xc81ec077,0x86b588a0),LL(0x9206354f,0x88952a01)}, {LL(0x43a8519a,0x5444a989),LL(0x2857b210,0xe29dd68c),LL(0x9a144624,0x36658903),LL(0xe423e899,0x8c4dedb0)}},
   {{LL(0xea886e5c,0x482040c5),LL(0x1cdd50f7,0x42fe5a56),LL(0x453b6e7f,0xf034f132),LL(0xb3c69762,0xba3fa97d)}, {LL(0xcadb598e,0x34262560),LL(0x07afe0a8,0x7ed74b51),LL(0xebe0e8bb,0x2261d849),LL(0x608cea1c,0x23747e55)}},
   {{LL(0x902c343d,0xb2e9371c),LL(0xda4fdba1,0xf57b2de8),LL(0xb67703a1,0x43f9afa4),LL(0xf79fe203,0xeafafb41)}, {LL(0xf649a494,0xfec99dc9),LL(0xfe378232,0x14799ef9),LL(0x7184e31e,0xba3f8114),LL(0xdc0e987f,0x0abbb815)}},
   {{LL(0x6fb5d02c,0x1dc7221f),LL(0x40f1607c,0xa0dfb113),LL(0xdbe8db86,0xd217f238),LL(0xe91f2859,0xc02547eb)}, {LL(0xa98d0875,0x41df6bcb),LL(0x78f6be54,0xf51a8077),LL(0xba66bef5,0x3ebdf28d),LL(0x9175ec20,0xc65c7b38)}},
   {{LL(0x475589d5,0x2302981d),LL(0x199d8ad4,0xbd479708),LL(0x8db46795,0xcb508be9),LL(0x6b224eb4,0xba1e066e)}, {LL(0x1326ab8a,0xae909633),LL(0xf5bbe5e6,0xac989728),LL(0xb1729b41,0x1de663f5),LL(0x00c3d293,0x2018f58c)}},
   {{LL(0x71efe25b,0xe647c0e6),LL(0x6104ebd6,0xaa30f871),LL(0x3c166de6,0x1003eebe),LL(0x1cbc42d3,0x0c308538)}, {LL(0x56cba120,0x98b6c312),LL(0x860d2916,0x9065ae66),LL(0xa7dabb6b,0x2162062b),LL(0x8e011036,0x1422386b)}},
   {{LL(0xe069de1c,0xd2589ad6),LL(0x7ee09300,0x3fe8e67a),LL(0x2f8ae49b,0x5b881860),LL(0x263a6a90,0x186a1482)}, {LL(0x9b35b33a,0xc6079b39),LL(0x53d3411d,0x6c38d789),LL(0x720b2f99,0x743c4462),LL(0x29e14d08,0x4d903dd7)}},
   {{LL(0x7b963913,0x625dfce0),LL(0xcc65e41f,0x60a83daf),LL(0x93e185a2,0x9e88c26c),LL(0x13707ac3,0x6d950e92)}, {LL(0xa0a23dd0,0xbd7df2dc),LL(0xc2116cc8,0x5fad27f2),LL(0xda4430be,0x0703b868),LL(0x83cc41df,0x5ebf0e2f)}},
   {{LL(0xf9e83fc8,0xebb91900),LL(0x6d60bb8a,0xf9539591),LL(0xb604935a,0x3bdd7a8b),LL(0x64e5eec0,0x2cae8c67)}, {LL(0x60aaf21d,0x30cf58bb),LL(0x359f63cc,0x5e0f6f5d),LL(0x0547e03a,0xda245055),LL(0xb9e143ce,0xa83fd8bb)}},
   {{LL(0x1b0215aa,0xa564435a),LL(0x354ba769,0xecffccec),LL(0xbdbbd594,0xd846149a),LL(0x65dd1597,0x9137df36)}, {LL(0xa9f3ac33,0xc4f39f37),LL(0x961d7e8d,0xf594bb74),LL(0x835befbc,0x41fa4b58),LL(0xed79139e,0xa983eae9)}},
   {{LL(0xb4e31aa4,0xd4085efc),LL(0x18b26adf,0xc760aec0),LL(0x76a7400d,0x14c1f78e),LL(0x317fe128,0x87b8aced)}, {LL(0xcbd85bb4,0x4433582e),LL(0x86adc041,0x58f01426),LL(0x8f0d5781,0x3596dd50),LL(0x1a31a82f,0x2e7f3b80)}},
   {{LL(0x2d1ede3e,0xcac7ccc8),LL(0xe89573db,0xc9b9a8f3),LL(0x54b40df9,0xbf744f69),LL(0xa85ecb47,0x88eb2281)}, {LL(0x426ec49d,0x6b115026),LL(0xc8c41110,0xebda4660),LL(0xdf392aae,0x0a4a32ac),LL(0x37cb7080,0x2a28f9b3)}},
   {{LL(0x6bd9414d,0xfed99d1a),LL(0x5715620e,0x15f59c41),LL(0xac431919,0x93edd9fc),LL(0xe1ccc47f,0xed1d43ae)}, {LL(0x556d1ab5,0xafed2acd),LL(0x02e039c0,0x817b967d),LL(0xe02a68bb,0x335d15da),LL(0x67df767c,0x4fa75ea0)}},
   {{LL(0x44833943,0x384704b3),LL(0x4084ef35,0x7809ed3c),LL(0x79c7ff41,0x2abab3c4),LL(0xa833fb73,0x2b7ef5b2)}, {LL(0x4901a4ff,0x12b0335b),LL(0x44d58617,0x3eea607b),LL(0xd7f57746,0x7161b669),LL(0xb1e93442,0xee17e43f)}},
   {{LL(0xd6d7878c,0x95c9bd80),LL(0x34ff7c75,0xe1ef20ee),LL(0xd2ccd599,0x3fab197a),LL(0x952ef4f9,0x9e480593)}, {LL(0x06ea3410,0x69777fd2),LL(0x74fa7dd5,0xb0280454),LL(0xc43bb5fc,0x641b6860),LL(0xecd7b8a8,0x9f359d5b)}},
   {{LL(0x70be68cd,0x4431d4ed),LL(0x08b55f8f,0x712117f4),LL(0x23d0b6ca,0x449e1319),LL(0xfdee5357,0x658323cc)}, {LL(0x62879a95,0xa1ef8114),LL(0x69963eeb,0xc21257e5),LL(0xc5bbee13,0x1016ab74),LL(0x43d81a86,0x99bb310a)}},
   {{LL(0x1d33a15a,0xdef03c44),LL(0x49127148,0x3e78cf18),LL(0x30b0cc00,0xe8d93368),LL(0xbd7ccd85,0xb05458fb)}, {LL(0xdbaa76b0,0x8c2896dd),LL(0x79e4cacb,0x0d826600),LL(0xff730ed0,0x50a45b23),LL(0xeba9030e,0x4a0e079c)}},
   {{LL(0xe3129aa0,0x3ead3fdc),LL(0x48aac890,0xa93b39f3),LL(0xf362465e,0x0fd73860),LL(0xf8df2764,0x69177f2c)}, {LL(0x824ebddf,0x4cd58c50),LL(0x2fcef01d,0x1478981f),LL(0x980524b3,0x511bd380),LL(0x4d23e8e9,0xc95252b1)}},
   {{LL(0x9ce08452,0x7ff12c37),LL(0xa3a87024,0x3dd8dd09),LL(0xb849dcb6,0x61ff0d39),LL(0xfefad6de,0x3f5eab86)}, {LL(0x251523f9,0xb6146886),LL(0x5be2135b,0x45ac1d52),LL(0x41d2c5d4,0x63507995),LL(0xb3064e72,0x7f19f799)}},
   {{LL(0xa2efb9be,0x7280ad9b),LL(0xba9659de,0xbc8fbb60),LL(0x875a8062,0xa4861c12),LL(0x2f2387ca,0x78c920c8)}, {LL(0x03b7da99,0xfe0a6ea7),LL(0x48e3afa3,0x936a44b8),LL(0x89fd127b,0x618a0ecf),LL(0x06f24bc5,0xa35614cd)}},
   {{LL(0x8d49c05f,0x21a1002e),LL(0x2fd7989d,0xceeacfd1),LL(0x8f5f4ea5,0x8c058f4b),LL(0x563e214b,0xf31bd38e)}, {LL(0x245c7585,0xbe47bd98),LL(0x6bc2a404,0x70d1a05c),LL(0xc68257f7,0x830d7f30),LL(0x8136c932,0x1abbbfbb)}},
   {{LL(0x6e03597b,0x6fbc43b7),LL(0x5fca14a2,0xc18fddb6),LL(0xe8f506e2,0xd3c4ca78),LL(0xa47e27de,0x6b8711dd)}, {LL(0x16382b5b,0xc308c189),LL(0xcf96bd7a,0xe9d3145b),LL(0xafb26629,0x5c290ec1),LL(0x209438ff,0xb4eb8130)}},
   {{LL(0xe1118539,0x7335044a),LL(0x5192a007,0xed6d43fc),LL(0xa43a2bd4,0x1a8bf622),LL(0xefa9f3a5,0xefec3fb6)}, {LL(0x6d834bde,0x6d224bbc),LL(0xb0fbc744,0xaaebfcb8),LL(0xc4ea1652,0x383b2bfc),LL(0x751ae816,0x9cd26d90)}},
   {{LL(0xf47f0f9a,0xae614386),LL(0x73c6ecd0,0x99873787),LL(0xe5414fc7,0x0b56c1ad),LL(0xfd40286c,0x9b85e6b5)}, {LL(0xd64687dc,0x7117aacd),LL(0xad8a8c4d,0x85d148e7),LL(0x7962655c,0xf62f8eb5),LL(0x7f0c6a2c,0x8386b37e)}},
   {{LL(0x90c47d7f,0x6b4715a3),LL(0x458a54e4,0x1fc4ced1),LL(0x0ed97b0a,0x01853943),LL(0x5b370e0e,0x58a280be)}, {LL(0x344f3960,0x8d488cb6),LL(0x0741604b,0x9050c599),LL(0x07954771,0x0878fb1b),LL(0xdbb3c82b,0xd927ea8c)}},
   {{LL(0x384f01fa,0x2fe71d59),LL(0x238bb66b,0x66d2b790),LL(0xb8fd803b,0xceaec11f),LL(0x6dd09c0c,0xbb919914)}, {LL(0x2ccb2f67,0xab5992e6),LL(0xcca50326,0x2421878f),LL(0x9ee6dc73,0x363d933d),LL(0x084b1fa3,0xa374ab0b)}},
   {{LL(0x161f6475,0x2d832a29),LL(0xfc8797eb,0x435b8d78),LL(0xd71b609c,0x66bc156d),LL(0xfe0c2004,0xb3dca798)}, {LL(0x02fd92d7,0x445d47bf),LL(0xc8b03083,0x1d1c9798),LL(0x079a7c51,0xca46d98d),LL(0x1afeb89a,0xb93f286c)}},
   {{LL(0xeeb6665d,0x1c174510),LL(0x7479a932,0x65874b6a),LL(0x2335e3b8,0x28d16a85),LL(0xc747eae6,0x5e22bd3b)}, {LL(0x04be16b4,0xa770e0a7),LL(0x40b3ff57,0x9f5f9ca9),LL(0x845ec39f,0x3f39e529),LL(0xebe697ce,0x5d5f4d60)}},
   {{LL(0xbd90d4f4,0xea2a262f),LL(0xfe1b6494,0xa1ce74ac),LL(0xfa0fc472,0x4e419004),LL(0x4e691060,0xdef0e440)}, {LL(0xa9f4baf3,0x57195a3a),LL(0x5e758c53,0xf14e108b),LL(0x920895e0,0x10a34f9d),LL(0xfeb57a63,0xc3f18af9)}},
   {{LL(0xda1bef0d,0x4b1c988c),LL(0x3df6915b,0x8b328cd9),LL(0xf45586d5,0x5ddc5ecc),LL(0x040322fc,0x426468b9)}, {LL(0xbe981558,0xf7f44765),LL(0x1855504a,0x25093991),LL(0xf7d6df43,0x72c51f2e),LL(0x849c99e5,0x858637fb)}},
   {{LL(0x0ee9f78b,0x68b84dfd),LL(0xf2ee390e,0xff42fc9b),LL(0x531e1dcf,0xaca71e10),LL(0x7feaedfa,0x391620e2)}, {LL(0xacf3e5da,0x7b2d6a02),LL(0xd20a16d3,0x261823d2),LL(0xbb00cd30,0xf9afa5d6),LL(0x1084580d,0xba151f4a)}},
   {{LL(0x26a3fcc3,0xb5f4b3a9),LL(0x6729f4da,0x5d84a823),LL(0xfc35f732,0x51540c41),LL(0xa6ae5bf7,0x81b0cb58)}, {LL(0xbd81bd27,0x91c7ae07),LL(0x1d56ff5d,0x0868980e),LL(0x65224df6,0xaef85a31),LL(0x17a69e35,0x112eba3b)}},
   {{LL(0xc3a9d932,0x07c34677),LL(0x8ac45e37,0x3b6b7cce),LL(0x31b6248a,0x5e0e2b6e),LL(0x453d9685,0x14ee5b66)}, {LL(0xbd4d807a,0x4c5e2be7),LL(0xc121fea8,0xc03f37f8),LL(0x8df7b5e7,0xcf34911e),LL(0x5f754191,0x00e7f18e)}},
   {{LL(0x2dcea4aa,0x89a8c9e1),LL(0x50f6db55,0xcc1cc31a),LL(0x9046955f,0x4a6f542c),LL(0xda2485d4,0x85fed580)}, {LL(0x9ac53748,0xa70f62d1),LL(0x655870a7,0xc2fbb850),LL(0x8c859aef,0xaeb2d438),LL(0xcc9ff51e,0xe3cc5ae5)}},
   {{LL(0x0a3ebbfc,0xf8d8c55d),LL(0xed48f61a,0xdcd838d5),LL(0xd4cba7ab,0x032f91ea),LL(0x2f70235e,0xeb0ed88d)}, {LL(0x000ef325,0xd4498170),LL(0x4b923c4a,0xfd34e07f),LL(0xb19b84cb,0xf71c07a9),LL(0xed9690a3,0x000a669c)}},
   {{LL(0xb5705e16,0xf45eb0ef),LL(0x7d9ce007,0x8cfd6a62),LL(0xd9e52885,0x76ba9a5f),LL(0x8aa9ffd6,0x13f72881)}, {LL(0x85528e4a,0x0a11e8dd),LL(0xcee8d663,0x58f1993d),LL(0xa1c81fd3,0xb49d750b),LL(0xe7de2e6b,0xaae29861)}},
   {{LL(0x5dd7de70,0x9a40644e),LL(0x937a5546,0x67fbae1c),LL(0x956c2fa8,0xb3e02907),LL(0x21b4aede,0xaf785374)}, {LL(0x9c0a8bfe,0xf42a1e96),LL(0x78957181,0x3f6690e6),LL(0xa6c5e0a7,0x1b1c9575),LL(0xf9cfb9bd,0x6def8124)}},
   {{LL(0x72faa1b0,0xde552cf9),LL(0x9b5ebbbc,0xfac2f4ad),LL(0x8ef89ba1,0x4b60a5a5),LL(0x8012f3b1,0xb6d9be57)}, {LL(0x9b2b083d,0x3992a6f7),LL(0xac640f77,0xe79ec527),LL(0x3f1e632e,0xf6cca775),LL(0x8fb80790,0x5dae8413)}},
   {{LL(0xb572407c,0xf0d4146c),LL(0x3f84cc39,0x829cfb38),LL(0xe31f007e,0xd7c9fed4),LL(0x09e68ce9,0x93b2a5bc)}, {LL(0xd01482b9,0x073fb24a),LL(0xb8d44e62,0xfe494244),LL(0x3dc49858,0xe59a1649),LL(0xf005b31f,0x071776f7)}},
   {{LL(0x285439af,0xaa368f59),LL(0xe27e783d,0xb0821574),LL(0x098746ba,0xe16903f6),LL(0x69dc00d0,0x436a95ae)}, {LL(0x877bfcec,0x9ae5a91a),LL(0x1416ad3a,0x5d8758d9),LL(0x25fd9a9a,0xa420ce40),LL(0x39b0fbfa,0x99610fdf)}},
   {{LL(0x41cc1e9b,0x782af222),LL(0x86ff4cf6,0x346a4a92),LL(0xabc900c1,0x9dc4c6cd),LL(0xe7fade51,0x7ed2476c)}, {LL(0x4f1dd24b,0x68a72185),LL(0xb08260cb,0xcefcf0fd),LL(0x8a6f61b6,0x0199a147),LL(0xcb5769c5,0x176b6079)}},
   {{LL(0x347d5a63,0x6dbcceb0),LL(0x9c4dc505,0x26433ebc),LL(0x05d5e74d,0x52570318),LL(0x057ca464,0x692f1d81)}, {LL(0x477f424e,0xa09554a0),LL(0x176a695d,0xbd3f9bbd),LL(0x5972db27,0x8c7f52f3),LL(0xf28b2aa4,0xacee8234)}},
   {{LL(0x3a4a0217,0xb853465a),LL(0x66f3a4d5,0x74a2534f),LL(0xeff0423b,0xae1a7ff3),LL(0xbb126028,0xd2a01a09)}, {LL(0x4963e855,0xff84c6f0),LL(0x6bc18d50,0x63db264c),LL(0xcc6a5e25,0x39792dca),LL(0xedb37a25,0xf20cdf3e)}},
   {{LL(0x8730f2c4,0x6a460f3d),LL(0x6a0ab6bb,0xe9b786c4),LL(0x084015c2,0xa9720a6b),LL(0x0dbe6f0f,0x28add2e2)}, {LL(0x26be7de7,0x90fb0ba7),LL(0xe40f15fd,0xfae8b5d4),LL(0xceb9c856,0x007363a1),LL(0x586b770e,0x6d8bfe14)}},
   {{LL(0xb7bcc0e0,0x63e7d78e),LL(0x3ed4ab62,0x56c569f8),LL(0x9e103abb,0x76c6a5bb),LL(0x638fc44d,0xeb24afeb)}, {LL(0xf0be16f1,0x15e00239),LL(0x8778f084,0x7db92f67),LL(0x63de2bef,0x5198680e),LL(0xe0510421,0x69031d0e)}},
   {{LL(0xb94753c1,0x8058f8aa),LL(0x454bf609,0x412d4c97),LL(0x95f9fd19,0xb8dbfe8a),LL(0x68b43233,0x6cd3221a)}, {LL(0xa5adaaff,0x384a9f15),LL(0x42b2ef95,0x60c70f90),LL(0x2c7da919,0x085b2f3b),LL(0xbc8407e1,0x1e5d23d1)}},
   {{LL(0xadb45b3e,0x9ea95bc9),LL(0xed06ec67,0xb5a28fec),LL(0x62a3c143,0xd678df46),LL(0x6793284a,0x80f0bc9d)}, {LL(0x07d4afc7,0xeb7865a9),LL(0xc1301d87,0x0fc5eafe),LL(0x4823349b,0x50a8e7f5),LL(0x2d019e96,0x97800fa2)}},
   {{LL(0x1bdd1d9c,0xfeff2579),LL(0x23886156,0x4d938c5d),LL(0x6979b9f6,0x25e3a806),LL(0x37bb6199,0xeeef8fa0)}, {LL(0xd7d308b8,0x4d917977),LL(0x4ae672cc,0x60ca7ff9),LL(0x2a68db6e,0xb24ec154),LL(0x9e9942f5,0x7b0604ec)}},
   {{LL(0xca4fad9e,0xfdf4794f),LL(0x3df5de22,0x086020f8),LL(0xd601533c,0x653da6ca),LL(0x735709f3,0xf97c1865)}, {LL(0x7cbd6ab6,0x2457ffd0),LL(0xd003a502,0xce05a482),LL(0x33ee2740,0x5c0c0cba),LL(0xf37174aa,0x8146ca00)}},
   {{LL(0x587262bb,0xec118827),LL(0x7e2a402c,0x8468c029),LL(0xbd6eb2aa,0xe1c3a9e3),LL(0xb496bde8,0x77d09b4d)}, {LL(0x854693bf,0x454e767a),LL(0xa4de85bc,0x6bbcc22e),LL(0xb180f206,0x66452f10),LL(0x0f5b1744,0x110c5a05)}},
   {{LL(0xa6b3f6e4,0xb7be75d8),LL(0x2c0d2e1d,0xf64bb3fd),LL(0x935ae640,0xad17a039),LL(0x8f243748,0x7304ad63)}, {LL(0xd278caa9,0x04316bb4),LL(0x1e84f91d,0x19b89c62),LL(0x6e7a2511,0xdf4a47e9),LL(0x998b6bc0,0xdef32df9)}},
   {{LL(0xcee95a1f,0xf1253ce0),LL(0x0ae96e31,0xbacf5206),LL(0x0b343e63,0x4ba2e24a),LL(0x79929dc6,0xca64d07f)}, {LL(0x9424ce75,0xf2823ac8),LL(0x0207ee9f,0x2d4add37),LL(0x387cde5f,0x44d9ecd0),LL(0xe180a21f,0xa5095ccb)}},
   {{LL(0x7c0cedf8,0x901cec8a),LL(0x3b8c759c,0xf291dc78),LL(0x49401234,0x98b8efdc),LL(0x058e9d9e,0x8f2b16e3)}, {LL(0x27dba00f,0x16ce8007),LL(0x6d66d2f1,0x5bb8fca9),LL(0x7b85a96b,0x092eda98),LL(0x973eae20,0xec53f4bc)}},
   {{LL(0x1b93a60f,0xe6e0df59),LL(0x2f6b0abf,0x65e06ecf),LL(0x569a9e1d,0xb8c2ec3e),LL(0xaa8c1cc3,0x27f9fe72)}, {LL(0xccd4d5e2,0x9cf3908f),LL(0x725c8528,0x5a40e0a9),LL(0xd470b0b0,0x27b15a1e),LL(0x33d81bff,0x50a09ec1)}},
   {{LL(0xda99fcf5,0xba976a58),LL(0xc3536b7c,0x3881ef1e),LL(0xfbc931b1,0xec65a069),LL(0xfc929a0e,0xab7f57b4)}, {LL(0xbc61f452,0xc7c63491),LL(0xc1750dbc,0x5c1aa935),LL(0x9ff0465c,0x35b8789b),LL(0x727647b7,0x8ff8589b)}},
   {{LL(0xa95022b6,0x2b56fe50),LL(0x2adbdbbd,0x24220569),LL(0xd2b80ea8,0x2370d94f),LL(0x71d9e6f7,0xe5d47b73)}, {LL(0x8d858032,0x2fe84b72),LL(0x4ffd7cfe,0x0411034b),LL(0x7c2c84e0,0x0819a9f3),LL(0x30228f9e,0xf8248dfd)}},
   {{LL(0x4fdf6c79,0x75c77f0a),LL(0x15579cc7,0x195b4b59),LL(0xf8d3705c,0x274c7c81),LL(0x70ee9be1,0x45a2209f)}, {LL(0x0b1e818b,0x4a4fce69),LL(0xbb9926b1,0x74d05e5f),LL(0x3f2a796b,0xb64639ce),LL(0xae957d59,0x2d168d5b)}},
   {{LL(0x16171846,0x067c2270),LL(0x1a03f8d1,0x7bb71d15),LL(0x495a09a1,0x2badd196),LL(0x51b990c4,0x024db163)}, {LL(0xe79dcaac,0xc19edc61),LL(0x60df81e9,0xf17f54bd),LL(0x9ae347e4,0x4560262e),LL(0x59eb711d,0x1d2c2542)}},
   {{LL(0x4919affe,0x40372d2d),LL(0xa009bd16,0x2d4a2ea3),LL(0xc1a62eb1,0x48f1e7f8),LL(0x587a1124,0xce083725)}, {LL(0xe7ebadd3,0xe874261b),LL(0x057b93e9,0x7ca5c156),LL(0xfe39e6ff,0xe8b381e5),LL(0xd30815c3,0x2d16b32f)}},
},
/* digit=15 base_pwr=2^105 */
{
   {{LL(0xad2e996a,0xdfd30b28),LL(0x9df0fc37,0x64d4eeec),LL(0xddc39763,0x8e049e3d),LL(0x37ea2e8e,0x9f55e0ce)}, {LL(0x4bf01984,0xf3fcba4e),LL(0x9d55bc4b,0x764d5c3d),LL(0xcfda895f,0x98cb92a1),LL(0x5c7bca69,0x27dfe795)}},
   {{LL(0x23a86e2c,0x86dfdecf),LL(0x3387f712,0x02ac466b),LL(0xd63509d1,0xc30a1ac2),LL(0x63aebbd1,0xd3913422)}, {LL(0x068ae34f,0xdc7b789f),LL(0x487dcd10,0x499f2d01),LL(0xa3e8a4b4,0x68e2a3bc),LL(0x14d2a26c,0xdf87ba71)}},
   {{LL(0x34479e7b,0x9e3ab999),LL(0xb026e780,0x9d5f2dc3),LL(0x4f1bd429,0x131374fd),LL(0x9be1379a,0x92e2e738)}, {LL(0xd13bc111,0x6cc32f80),LL(0x86f81c92,0x6fbfc350),LL(0x9263913a,0x12ca1b30),LL(0xe06ab75c,0x6f7da1ff)}},
   {{LL(0x19301b16,0x4780f12a),LL(0xbc368a20,0x233bc231),LL(0xcbadb344,0xd9650892),LL(0xad9425a1,0x38a0d964)}, {LL(0x4a8d4d7e,0x277abcf2),LL(0xb806ce9e,0x4ccd16b1),LL(0x7570d473,0x82ff40f0),LL(0xdf130780,0x57491414)}},
   {{LL(0xa96ef42b,0x9f1f39f2),LL(0x107f1321,0x1fd967ed),LL(0x29d4767e,0x93832497),LL(0x3fa9e796,0x7761a38b)}, {LL(0x66318df2,0x3e408c59),LL(0x41745f03,0x9283ca44),LL(0x2cce1a86,0xfedf8fa3),LL(0xb44600b4,0x8363860d)}},
   {{LL(0x45f45a89,0x3dbfde55),LL(0x26ce399c,0x8800c860),LL(0xc25e9479,0xfb25e8da),LL(0xf7d367a2,0x6ff0d6cb)}, {LL(0xa93f9890,0x70b0ba36),LL(0x49bd5a80,0xc07ca403),LL(0xed54d1aa,0x5f4feda6),LL(0x671ad0b4,0xfa1e2efd)}},
   {{LL(0x8c56e7aa,0xda465467),LL(0x25e45bc0,0x39d04cdc),LL(0xaf21c637,0x26661bd6),LL(0xb55ddfa5,0xf757ff5c)}, {LL(0x4394eb20,0x07318fd1),LL(0x2bcf3ad1,0xe010b19d),LL(0x8e5c7e7f,0x71e2031c),LL(0xedbfda69,0xee35f346)}},
   {{LL(0x5d8f6fab,0x8d8d4120),LL(0xabed09c8,0x5e420839),LL(0x4aacbb1e,0x5120e079),LL(0xec1bc996,0xf7e840f8)}, {LL(0x9707c3bb,0xd921d63b),LL(0xae891a71,0x3ab4b8b8),LL(0x73bb2278,0xbe465756),LL(0x553ee91c,0x776ce040)}},
   {{LL(0x88222e0a,0x86f07c2e),LL(0xdf97816e,0x3f3688f1),LL(0xa6df9d3a,0x20d2d944),LL(0xb2cb925a,0xff399057)}, {LL(0x27f1de19,0x010c747e),LL(0x7fc1fbc3,0xbe940669),LL(0x877ef709,0x3ab94600),LL(0x8f4c8a8e,0x9af40a6b)}},
   {{LL(0x713f023a,0xf7c0f202),LL(0xd0a8dc63,0xbe0bf3db),LL(0x64a41337,0x09789926),LL(0x2c2823cf,0xbb4a8964)}, {LL(0xb90e45c6,0x9279cb27),LL(0x382564ac,0x80283fd3),LL(0xfa5f0bcd,0x3a746b01),LL(0x0afaf10a,0x28667a8e)}},
   {{LL(0x015d5295,0xeccfd0ee),LL(0x73ce8e21,0xbd6678a5),LL(0x336ecb65,0x132a87f2),LL(0xbe7dc659,0x09de4efc)}, {LL(0x6f69b3bd,0xbedadd10),LL(0xe8303650,0x357c61e2),LL(0xa45777e6,0x6b3c613a),LL(0x08d41179,0x51dd30ad)}},
   {{LL(0xf98feac3,0xa7b53f3f),LL(0xf9497319,0xe8e00328),LL(0x0ca20b40,0x1f3b080d),LL(0xe19b088e,0x06dc5456)}, {LL(0x0dce02cb,0x1f3f18d7),LL(0x6fd1ade9,0x2d215544),LL(0x055550d0,0x4c6de38c),LL(0xf33f4973,0x5d535e6f)}},
   {{LL(0x744c033d,0xdcfd4163),LL(0x76fc420a,0x0706a490),LL(0x394f5b6d,0xdc9db55c),LL(0xe040a093,0x1a050a62)}, {LL(0xd0ab26f3,0x60367ebb),LL(0xc3d91820,0xb9c07239),LL(0xa3e32db2,0x14a8bed5),LL(0xd49f37dd,0x23b19cd3)}},
   {{LL(0xea6c85ce,0xd048c64f),LL(0xd5716cfe,0x8aefff19),LL(0xab85bdcf,0x199fddb1),LL(0xbaea49d0,0x3f592e7e)}, {LL(0xf1d58ff6,0x8c8decb6),LL(0x7fe8f425,0x02598a99),LL(0x64a93fd6,0xe4c73ae2),LL(0x878db06b,0x208a0f61)}},
   {{LL(0x27c4a266,0xff0272fe),LL(0x69e7c6a2,0xccfc96ae),LL(0x8e78d74a,0xbd7e715f),LL(0x32ed35bc,0xd04ae9a4)}, {LL(0x05614c5f,0x302af413),LL(0x33943b5a,0x98176637),LL(0x8a58cfd8,0xa4d1c9b2),LL(0xac2f3f79,0xf0ebd5f4)}},
   {{LL(0xfdc7945f,0xebadb010),LL(0x03750a4b,0x62c9cff0),LL(0xa0c85b9d,0x75192383),LL(0x16eb0bf9,0x2aba7b53)}, {LL(0xa4c06f9a,0x93d4adaa),LL(0x5f8e482c,0x573a86a2),LL(0xe791a79f,0x109c6fdc),LL(0xb37eeec2,0xd9ed02ce)}},
   {{LL(0x7dd63c8b,0x7b1fb4b4),LL(0x22493b49,0xae6e2767),LL(0x6a411bc9,0x5ef6beb7),LL(0xe1bf4242,0x4d88918d)}, {LL(0x02a97fbc,0x7ba26f8c),LL(0x7f41c518,0xf45b2a50),LL(0x83249e23,0x6c35fb69),LL(0x7a5778cc,0xc4a95134)}},
   {{LL(0x08287cf7,0x6173f860),LL(0xfac3a444,0xdcfc71d9),LL(0x079ce3c9,0x894f3b33),LL(0x916b661c,0x842bf7da)}, {LL(0xa758a243,0x94299d6f),LL(0xb242389a,0x0e23f984),LL(0xc03a7aa2,0x653050f0),LL(0x9559ad9c,0x2ec3041b)}},
   {{LL(0x97cf6e9b,0xa61dd499),LL(0x448fa6c6,0xfd090f38),LL(0x39b126bf,0x4f1b75ac),LL(0x1ef1a280,0xb48d0372)}, {LL(0xf2b2941f,0xe40c310d),LL(0x8d9d5ace,0x5b9a7369),LL(0x7ad9ad95,0xbe0415c1),LL(0xa8510622,0xffd864b6)}},
   {{LL(0x898f28e8,0x2aceddcd),LL(0xba89a51f,0xa0cfc30a),LL(0xe105e448,0xd87db983),LL(0x5e5ea6fb,0x16ba795e)}, {LL(0x352ad094,0x5f48e45a),LL(0x55fd75e7,0x1971a10f),LL(0xfd1c8d68,0xfb0f1767),LL(0x86df0665,0x904229d9)}},
   {{LL(0xe87ab22b,0xc2c88671),LL(0x33611a50,0xcbe384a1),LL(0xad2eb708,0x38eec58e),LL(0x86d7bdee,0xaa19b174)}, {LL(0x51832b61,0xa9f416d7),LL(0x355e7b6d,0x10b751ff),LL(0x4ff07a5a,0x6dd52063),LL(0x4e3505c1,0x6a6090c1)}},
   {{LL(0xd4c80f29,0x53addd1c),LL(0x0d3d793e,0xe76d668b),LL(0x191eedd9,0xbdcdc4c9),LL(0x0f8e4877,0x80675320)}, {LL(0x32f5661b,0xc41426dd),LL(0x06199185,0x9fcbe1ac),LL(0x404a1207,0x62fa2198),LL(0x33c8190e,0xc742fdc2)}},
   {{LL(0x778ee765,0x16ec1b96),LL(0xe29d992d,0xda656f58),LL(0xb4369e7f,0x5393775b),LL(0x0674fc45,0x512f5c7b)}, {LL(0x60efa8e0,0x55b8bd38),LL(0x155b57ab,0x1ab6d2df),LL(0xe26ad439,0xed0aa777),LL(0xd9b04376,0x5b16333e)}},
   {{LL(0x8fc7ea72,0x321119d3),LL(0x7211ef45,0x390b4ef5),LL(0x1feda1a2,0x289f3945),LL(0x605c8596,0xcee9cb33)}, {LL(0x971b6897,0x91109e96),LL(0x19701ea7,0xdf70c173),LL(0xa92c6b2b,0xa07d0ecd),LL(0xa9eab766,0xf8eb97fe)}},
   {{LL(0x0e3cf9e8,0xbb2cf63b),LL(0xdda02b26,0xffa6c503),LL(0x9cb18efd,0x42c1ec1a),LL(0xc45d5b22,0x13cdda9c)}, {LL(0xc820cdf5,0x6b3baf33),LL(0x610c8bfc,0xa337bc5e),LL(0x06a9ca6b,0x88186818),LL(0xa0c455af,0x382a34ee)}},
   {{LL(0xe8fc59df,0x725006c9),LL(0xf929c670,0x0d46b697),LL(0x893a9f6e,0x7bd6eceb),LL(0x1cd04e5a,0xf25aa637)}, {LL(0xf6612d4a,0xf1563d79),LL(0x5efc49d8,0x8c9372cf),LL(0x96c5bdc5,0x09cc0513),LL(0xd3cc34d8,0x70f19d46)}},
   {{LL(0xfdfbe16c,0xe62f6891),LL(0xdc60110b,0x8b7db2fd),LL(0xf7868674,0x3557bff8),LL(0x95a638d6,0x2b414c8e)}, {LL(0x6d19ac65,0x3b6610ac),LL(0xd641b0ea,0x20864204),LL(0x892161fb,0xee372a46),LL(0x4125868a,0xc7c5bd87)}},
   {{LL(0xa61ee957,0x5edc6afc),LL(0xd37faed4,0xa4374ba1),LL(0x1e52b0ab,0xf0684f46),LL(0x2a007b66,0x0812cbca)}, {LL(0xe68c4893,0xf3442afd),LL(0x2d9dd9a2,0xb02ac6df),LL(0x068c08bb,0xa4fe98dc),LL(0x7795e2e9,0xfcd5dc05)}},
   {{LL(0xa0f55cd8,0x28983aee),LL(0xe96cf557,0xb390daf4),LL(0x3f119098,0xbfd2f5ab),LL(0x6386c0ad,0xed114587)}, {LL(0x69539392,0x578995b9),LL(0x36b880ab,0xc8a77948),LL(0xe4c8b560,0x7e03cfb6),LL(0x06cdcbe0,0x018cd4db)}},
   {{LL(0xb006f8d5,0xbd7f2e3a),LL(0xd6d9f30e,0xf25d1718),LL(0x1b22ae3d,0x9ef6e2ee),LL(0x416154ab,0x89a2695d)}, {LL(0xda261e39,0x1975e0c8),LL(0xda2c2031,0x8fe66aed),LL(0x97e1e00c,0x6f93f839),LL(0xa742b328,0xcba9a773)}},
   {{LL(0x3417df8b,0x9529fec1),LL(0x54e314b1,0x37a27cd9),LL(0x65f94a63,0x93feb0f8),LL(0xb5e029c1,0x65978b84)}, {LL(0x81b705f1,0x576fd830),LL(0x688e8c52,0x37d07a37),LL(0x332838dc,0x3b91d360),LL(0xb0b14a94,0xcca9cbf8)}},
   {{LL(0x8b63b669,0x86f18c44),LL(0x6972d2d1,0x53c6eba6),LL(0x8cff59bc,0x2a522d8c),LL(0x6ed25ce5,0xbc181d15)}, {LL(0x5feb0eca,0x913f173d),LL(0xa207bd71,0x34273f8d),LL(0xfa1715ae,0x41b7572e),LL(0x7f16f4ae,0x8a8ffea2)}},
   {{LL(0xf95bdf78,0x2b852908),LL(0xa26328b9,0xa75adbb3),LL(0xdae21d25,0x992ac179),LL(0x78e14467,0x8c99927a)}, {LL(0x0c16e0c2,0x23fb2fee),LL(0xdbcb5f4e,0x58e797bb),LL(0xa07bd7de,0x33d6956e),LL(0x7172d36a,0xc921fdb3)}},
   {{LL(0x158267b5,0x035f1744),LL(0xa291374d,0xc7495f33),LL(0x4a41a6a0,0xe07db2f5),LL(0xd1238792,0xfdb2987e)}, {LL(0x49741ce6,0x616df624),LL(0x8a693880,0x90ecd21b),LL(0x341fe21b,0x447c729d),LL(0xc012a8ab,0x06ad3c90)}},
   {{LL(0xddfd6b5a,0x13dc4fa9),LL(0x64cfc0f3,0x238a1add),LL(0xc34a2b1e,0x874a3c2f),LL(0x0deb1dd4,0x427b6e3c)}, {LL(0x876f2621,0x78a1ad1d),LL(0x252f6837,0x34f9207c),LL(0x047d667b,0x1c812fbb),LL(0x3ee03ba7,0xc3512ea3)}},
   {{LL(0x527a1b55,0x762de5f0),LL(0xae3221af,0x7873d692),LL(0xb112d35f,0xa8ac73c7),LL(0x815414f6,0x1d118c30)}, {LL(0x865ab6f6,0xbc686118),LL(0xecf8c02d,0x01e75348),LL(0xe6220bf9,0x9b131840),LL(0xa67512b5,0x3d72dac4)}},
   {{LL(0xaf95e16e,0xd9e49ecc),LL(0x1e2aa105,0x1297c560),LL(0xa1a5e8c2,0x925301ac),LL(0x990ab503,0x3b0ea863)}, {LL(0x15f258c4,0x7860b660),LL(0x397253e4,0xa4497040),LL(0x14a4774d,0x88562ed0),LL(0x7adbd093,0x325d8b0d)}},
   {{LL(0x41e24010,0xd4df8df2),LL(0x580031be,0xe7cb8663),LL(0xd7fc7e5f,0xd653593a),LL(0xe9f1fe9d,0x429a8863)}, {LL(0x63e71ced,0x18a0a709),LL(0xa498140e,0x39d9316d),LL(0x40f40117,0x44466cff),LL(0x64602832,0x58d27cd6)}},
   {{LL(0x86018a70,0xf4a4c22b),LL(0x6d703f04,0x7a4d4102),LL(0xb424b0fb,0x4f5a0037),LL(0x322b1876,0xfb591cfd)}, {LL(0x632e0f56,0xb45798b0),LL(0xfdcbcf20,0x83ef9feb),LL(0x115749ac,0x0a23b09c),LL(0x946248f8,0x3b950273)}},
   {{LL(0x1e13eaee,0x5ed75e68),LL(0x09dafdb9,0xbebd4744),LL(0x69941fc4,0x8b46621f),LL(0x91129bc0,0x1fd3c13f)}, {LL(0xb7b9da22,0x371448d0),LL(0xd87a4742,0xd452ccea),LL(0xf91c38b9,0xe2f04778),LL(0xbd516bfe,0xfb443a5d)}},
   {{LL(0x044d666b,0xd7bd4056),LL(0x2a9b17c4,0xb4546ffd),LL(0x818fe55a,0xf66765ae),LL(0x1b5dc7a0,0xc375126c)}, {LL(0xc1a81c63,0xe9a7ed20),LL(0xf8cf06b9,0xaef2e23d),LL(0x3e67b95b,0xf4536865),LL(0x24309537,0x25cbb5a6)}},
   {{LL(0x3256c020,0x8a230e68),LL(0x215dca7b,0x4a33e462),LL(0x2935b6d1,0xefef4936),LL(0x852c39f4,0xb383df4e)}, {LL(0x57c21e90,0x197ddd77),LL(0x2164127f,0x236f98dd),LL(0xbbd61847,0x464b86ec),LL(0xfb89d515,0x107a387c)}},
   {{LL(0x400d66d5,0xe01e50b7),LL(0x5f864426,0x4377af2b),LL(0xf8fe847a,0xde21c49a),LL(0x887c0674,0xc133e58f)}, {LL(0xd2fda17c,0xda5b4c3b),LL(0xfed8fe68,0x24157f13),LL(0x8b6bb6bf,0x1c4483f3),LL(0xcf1bed73,0x940fab9e)}},
   {{LL(0x3c15c7e5,0xce3fca79),LL(0x066de109,0xb83fce10),LL(0x0cd5a04a,0xbd42ed01),LL(0x407fcb03,0xba5446b8)}, {LL(0xe5d35bda,0x4a8cb929),LL(0xbff8631f,0x6338fd7e),LL(0x4656a8cd,0xc85d4ee4),LL(0x92c99820,0x83b1f39a)}},
   {{LL(0x7e90c823,0x153fa4d4),LL(0x15326196,0xc15809ba),LL(0x6eb4b379,0x320b8fe9),LL(0x58704d9e,0x27cc07c0)}, {LL(0xf13d6ee7,0x301840b2),LL(0xc6d8c930,0xf1f6530f),LL(0x96a22034,0x3e9254ea),LL(0xaf5b8a2e,0xf8c0ee68)}},
   {{LL(0xb8207fde,0x88e9e44e),LL(0x29bc1310,0xdea11cbd),LL(0x9c9d7df7,0xa20c2f17),LL(0xbceac495,0x2a954927)}, {LL(0x2a58d6ba,0x3f405f5c),LL(0x2ac9aeaa,0x64df5364),LL(0xe8aa74a5,0xb618f6db),LL(0x74ef61e3,0x22dadc7f)}},
   {{LL(0x9cfdc4cd,0x306ee832),LL(0x40698a5b,0xaff53321),LL(0x89567341,0x98893443),LL(0xac7c68ec,0xdfefbdd4)}, {LL(0x3261a582,0xd3da7094),LL(0x62ce96e7,0xd23e3fa5),LL(0xd773337e,0x62c060c0),LL(0xa041f910,0x5cb2beca)}},
   {{LL(0x7e8215d8,0xe21ab479),LL(0x923b4b27,0x84384686),LL(0x6a3d21ef,0xa93c08fe),LL(0x2fa6de1c,0x7bd96248)}, {LL(0xca6de3e0,0xb858ecd7),LL(0x47c9fced,0x466a48c8),LL(0xc708239e,0x23ca9b75),LL(0xb5bbe833,0x860d553d)}},
   {{LL(0x9e76e71d,0x45804f1a),LL(0x51e59be2,0x9fdb8b8d),LL(0x3bbc2a19,0xa76db4b7),LL(0x96f82cd0,0xaebb47ee)}, {LL(0x97b1dd04,0x7a9b95b5),LL(0x5f0b1d7c,0xcc149a8d),LL(0x47a50eec,0xbba40e4d),LL(0x56b72539,0x4e635d70)}},
   {{LL(0xb18659c4,0x31c40e90),LL(0x6f694b35,0x08026224),LL(0xe0cd6e15,0x8ed3d7b8),LL(0x9293cb36,0x157e2a9d)}, {LL(0x96e54583,0x7b457bb1),LL(0x2609c44d,0x75647498),LL(0x970a8cf2,0x54667671),LL(0x3af72572,0x3b7da9c8)}},
   {{LL(0x4d63473a,0x8fbba977),LL(0x23001221,0x7af5af43),LL(0xea29d97e,0x99093197),LL(0xfa564089,0x4409f6a9)}, {LL(0x2f70e06f,0x64fd1cda),LL(0x2e9d55ea,0x8b7c83a6),LL(0x6385fcef,0x0dffbe4b),LL(0x7a6fe72d,0x4513f584)}},
   {{LL(0xba1de4ae,0x6a64f166),LL(0x89069fbd,0x1f8550a9),LL(0xda7ef7fc,0x72b411fd),LL(0x829ea712,0xa185d2c3)}, {LL(0xccc1868d,0x82f5ffb8),LL(0xff9fafa9,0xb484473a),LL(0x089132ed,0xe1857f3c),LL(0x08ef378f,0xdad885a9)}},
   {{LL(0x7af9e2aa,0xbdbdfc0e),LL(0x95486202,0x366c07bb),LL(0xfc9d979f,0x37040d45),LL(0xa0f80630,0xf279ed10)}, {LL(0x8f31b39c,0x27855261),LL(0xf26f91cb,0xea0c3b1c),LL(0xb38c050f,0x90b4e8c2),LL(0x1623ab47,0x7acb26b1)}},
   {{LL(0xa4a064d2,0xb6cc3cd9),LL(0x97245482,0xa410880c),LL(0x3a6e6a1f,0xfb470e11),LL(0x93f347e4,0xf19080b1)}, {LL(0xb75a53d9,0x23253dcf),LL(0x9c95d567,0x55047049),LL(0x7b20202a,0x8c75631e),LL(0x834b8380,0x58fccf32)}},
   {{LL(0x243dddde,0xaf6bdbd8),LL(0xcf38f705,0xa3ca3e2c),LL(0xca38c9a2,0xa2357b4b),LL(0x6eba095b,0x8bf0d270)}, {LL(0x9d998070,0xe4a43b7c),LL(0x8469214c,0xdf412faa),LL(0x000f1802,0xd2578cc4),LL(0xf8515863,0x2feb563f)}},
   {{LL(0x5022112c,0xe66ce02a),LL(0x1461b1c6,0x8415df81),LL(0xad247c38,0xc2546e6a),LL(0x9a9c74d6,0x4b9788e6)}, {LL(0xa22be3e8,0x9d0cb2e0),LL(0x295f76ad,0x15db086c),LL(0xa2337670,0x9758f99b),LL(0x9ab57f54,0x61ae09bb)}},
   {{LL(0x93926a37,0x7af4d4aa),LL(0xf25cadb4,0xa895f386),LL(0xc6f4f514,0x90e13821),LL(0x4eef97ab,0x46738d95)}, {LL(0xf0b479a2,0x66f559f3),LL(0x3262fb2b,0x9ea62dcd),LL(0x6a857025,0x91a39a5e),LL(0xbb3e6e11,0x11bdd19a)}},
   {{LL(0x9353cc19,0xfa411fd6),LL(0x94cd2276,0x275d094c),LL(0x25243d1b,0xe2069b22),LL(0x630c8cbc,0x9f02f1e2)}, {LL(0x379b6b63,0x85c942fd),LL(0xbdcc9307,0x293dcf87),LL(0xdc212ca8,0x56856d8e),LL(0x23a0c56d,0x1927e931)}},
   {{LL(0x9c2f8b66,0xacfed288),LL(0x386ad1e3,0x20f6b94e),LL(0xdcbeff77,0x0e622304),LL(0x5978f2f6,0x67e895fa)}, {LL(0x20029bfe,0x12a63bde),LL(0x8d968b8c,0x0ed75b6c),LL(0x57cec33e,0x611739ee),LL(0x42b9fc44,0xeffaae72)}},
   {{LL(0x971a90a9,0xa7de79ce),LL(0x4fead0d5,0x529bd8a2),LL(0x9a1a43ef,0x78434c30),LL(0x4f3c848c,0x904d1db2)}, {LL(0x6d4234ad,0xedb3c11e),LL(0x5975e9a0,0x4e0c9ec4),LL(0xff16ec74,0xc51236db),LL(0x36d09231,0xde652cff)}},
   {{LL(0xe60a0bc5,0x0d7c18b7),LL(0x9d7df012,0xaf041383),LL(0x47d4cfd0,0x9c0ff3f4),LL(0x3d97bac7,0x64c7e6c2)}, {LL(0x8cb44d50,0x239d2500),LL(0xbba96026,0x47189764),LL(0x31ddca37,0x22449328),LL(0xfb7c29ef,0xa53a1886)}},
   {{LL(0x97a3789b,0x2515b665),LL(0x540ea258,0x825c5031),LL(0x09a5b24b,0x76680656),LL(0x60fb8bcc,0x833c240f)}, {LL(0x01e55cc7,0x758e0b10),LL(0x8d260572,0x8b799c48),LL(0x6c5dd0cd,0x0981a802),LL(0x1b9c6cc9,0x6f6da55d)}},
   {{LL(0x8c4d503a,0x3457b685),LL(0x009a7a94,0xc537730f),LL(0xd01dfdff,0x334d46b4),LL(0x15e20bc7,0x3e5dc6a8)}, {LL(0x6ce8b8ab,0x1833b0bf),LL(0xc57a4292,0xe21b99ae),LL(0x1713ba15,0x743cb595),LL(0xe0bb44a7,0x550e41bc)}},
   {{LL(0x2f8ebcf5,0xf356917a),LL(0x953f056c,0x6f2c400b),LL(0xdd84bb48,0x09d9ac41),LL(0xa61e98e3,0x50dc7a8e)}, {LL(0x3d3a6776,0x1179a9d3),LL(0x16de8b3e,0xdda312e7),LL(0xce6e2bea,0x62a8b7c3),LL(0x645e4ca0,0x2b00036c)}},
},
/* digit=16 base_pwr=2^112 */
{
   {{LL(0xab6cf0b4,0x9ad2cbd7),LL(0xf13d1ddf,0x7a1e67f4),LL(0x746003ba,0xa58f0c73),LL(0xa64a8fcc,0x8263e888)}, {LL(0xbe2452f7,0x535cbe37),LL(0x6ae81a76,0x93125766),LL(0x3a553701,0x7d2ed0ab),LL(0xb0717d78,0x93d7e7df)}},
   {{LL(0xf9cf03ba,0x61bc013c),LL(0xfeee3a78,0x36223b88),LL(0x3d7e4c95,0x86efc923),LL(0x965625e4,0xaf3801be)}, {LL(0x3f32fd9d,0xa7bba141),LL(0x4e564acb,0x70724dec),LL(0xb7edcac1,0x52587f32),LL(0xb3969985,0x0b1cd94c)}},
   {{LL(0x661fbdab,0x9f08407a),LL(0x58e52151,0xc4d7c536),LL(0x63dfe954,0xa3765bce),LL(0xac2dddd1,0xc6829bfb)}, {LL(0xdc6e4487,0x97393f65),LL(0xbf04c930,0x9ba29422),LL(0x18838c0a,0x493c6914),LL(0x4b2f35cd,0x41b137ff)}},
   {{LL(0x4e1c5e60,0xc9e4aa24),LL(0x13253d51,0x54bb5289),LL(0x9bbabe53,0xf4a86ab3),LL(0x6ac0aa64,0xd561feae)}, {LL(0x1911bad7,0x27a896ab),LL(0x64337303,0x9cb22b98),LL(0x161f6928,0xf14262fb),LL(0x59ba6c9f,0x1661885c)}},
   {{LL(0xb82574db,0x4666ebd3),LL(0x8d8af3f6,0xc5e86672),LL(0x209319bf,0xcc645205),LL(0x3834d1a8,0xc183c12e)}, {LL(0x49eb0f40,0x533d73da),LL(0x6aca649e,0x3bcab0bc),LL(0xe39e8361,0xa02f4c41),LL(0xa89bdc85,0x2391e7ae)}},
   {{LL(0x608cbe2f,0x88067c5e),LL(0xf16c22f2,0xcdec82f6),LL(0xf1faf9dc,0x80aa719a),LL(0x2340185c,0x261fe951)}, {LL(0x9713e72e,0xcb411366),LL(0x6d8a2a46,0xdb1e405e),LL(0x11106ce0,0xed174757),LL(0xf71c0e69,0x6d70cf6e)}},
   {{LL(0xcf707c76,0xb5310322),LL(0x40b4b7d7,0x3a1eb2ff),LL(0xb1a2d26d,0xb83259fb),LL(0x799720c0,0xa83ffb0f)}, {LL(0x0bedb326,0xeecb1928),LL(0xe9271715,0x4473e820),LL(0x2f2d71a9,0x506e6d20),LL(0x4319756d,0xe7b253b1)}},
   {{LL(0xf576cb3c,0x27e19335),LL(0xdfb3b78f,0xe16e4573),LL(0x63da0001,0xaf96d785),LL(0xf7444c5e,0xb38deafd)}, {LL(0xc0eb0e28,0xaca6b38c),LL(0x7fe41b98,0xa6ca3541),LL(0x18965b31,0xfeb37b47),LL(0x597d5bc6,0x739cc322)}},
   {{LL(0x4cb1fbc3,0x82776041),LL(0x8e2a3ad1,0xcdaa873d),LL(0xc01318bf,0xb5f87b34),LL(0x229cb81e,0x0c692268)}, {LL(0xa53089f5,0xb0724016),LL(0x05415313,0xb2976a53),LL(0xcee8fdf2,0x436eab75),LL(0xd82b13e5,0x8794e1a6)}},
   {{LL(0xd5516e3d,0x0d51a072),LL(0x4b2fdb3c,0x7bae61ce),LL(0x550728ee,0x0d987e89),LL(0xee6778db,0xfd5a656e)}, {LL(0xbb4d9d7a,0x942ee623),LL(0x2dc1baf8,0xfc06d64b),LL(0x47c3dc8e,0x5244fcd8),LL(0x8568653e,0x5e37e156)}},
   {{LL(0x048c8355,0xe5c2c6ff),LL(0xa0474046,0x480872ec),LL(0x7ff92484,0x67e3089d),LL(0x29971b3e,0xdc071893)}, {LL(0x5a63e8ad,0x3857db2b),LL(0x5f2e0a09,0xf617d94e),LL(0x5844de79,0x267e9818),LL(0x861f5f92,0xfdb103b2)}},
   {{LL(0x1570d6e9,0xb969bd3c),LL(0xe0fb517e,0x7350b9db),LL(0x55aa543b,0x083b142e),LL(0xaa2bc581,0x424757be)}, {LL(0x69ea3302,0x4bd50f64),LL(0xed4c8288,0x053dcf83),LL(0xc118ac52,0xac2b3074),LL(0xe76ca88b,0x57f066a8)}},
   {{LL(0x0a3207cb,0xb9ac28fd),LL(0x205829b0,0x0ec2ad13),LL(0x4f403b64,0x76216e56),LL(0x6621dd5c,0x7eaef662)}, {LL(0x4b06ac54,0x9e0cc836),LL(0x9648523c,0x16ac3c6c),LL(0x08eec1d8,0xe1208a1a),LL(0xbe054a54,0x1e90b3a2)}},
   {{LL(0x23a836cb,0xdfab7f92),LL(0x6f7674c8,0x624d6bd2),LL(0xea06210a,0xc34a914c),LL(0xf26c4f23,0xba5314ec)}, {LL(0xa33a11cd,0xd440b196),LL(0x75eb951f,0xf81ab63e),LL(0x39350f0c,0x05ebb91a),LL(0x92e9528f,0x3f3c08ec)}},
   {{LL(0x4fe6f4e6,0x54ff8844),LL(0x79b7ba49,0x217c0e02),LL(0xbf3a4179,0x13c4372b),LL(0xa1434264,0x6e5ad10b)}, {LL(0x62bd8ff2,0xc8426540),LL(0x85fe6ef1,0x7c3dd284),LL(0x05da0e84,0x2c59b300),LL(0x17468e18,0xf303ed94)}},
   {{LL(0xa211ffd3,0xe19fac99),LL(0x3f1f6bca,0x408f9424),LL(0xa5648244,0x1f5b76d1),LL(0x95b2bd67,0xef3942e8)}, {LL(0xb538f1d7,0x1b9dee7f),LL(0x444b8f85,0x1cb78620),LL(0xcb8ea6a3,0x9f8ecd63),LL(0xb9d3b71f,0xca111b2e)}},
   {{LL(0xbdc4e8e2,0xff83d71e),LL(0x3f76a9d5,0x43745ddb),LL(0xa25856ee,0x72db22a9),LL(0x5e9a9ff7,0xf34d5aa2)}, {LL(0xbc529902,0x01f6b5f3),LL(0x086f4867,0xadf5d31e),LL(0xca556b56,0xbd88674f),LL(0xfdc81625,0xfd00120d)}},
   {{LL(0xfdde77f6,0x90fbaba0),LL(0x559ec6e7,0x266d3bfe),LL(0xc8094357,0x372acf54),LL(0x6c61bb78,0x772bd8e4)}, {LL(0x1af9aefc,0xcb2ac592),LL(0x5b178386,0xacc3dc9b),LL(0x23438463,0x09963084),LL(0x8359f1e6,0xae84f973)}},
   {{LL(0xa4cee642,0xc3b19aa0),LL(0xb19a918f,0xcd5ca5c8),LL(0xe67cb207,0x46ac0d2e),LL(0x73ffebf2,0x2ae45e10)}, {LL(0x10ef065c,0xf84aad8e),LL(0x32a7e903,0xa0af57fa),LL(0x43d346dc,0x42295904),LL(0x7f170965,0x8d6f711d)}},
   {{LL(0xb110cffe,0x11aa7070),LL(0x9755605d,0x091a100d),LL(0xd27d86a6,0xc5a0c654),LL(0xdb30feaa,0x1031a244)}, {LL(0xc02228d9,0x36804045),LL(0x8b746039,0x1877fc67),LL(0xe09bb238,0xba554015),LL(0x1de9b40d,0xc50d8f92)}},
   {{LL(0x032f7a2c,0x29e40d88),LL(0x1d315ec2,0x514b56dd),LL(0x61778f04,0x9052652e),LL(0xe1a1c195,0x0d2bc606)}, {LL(0xb05043ae,0x375fd7ec),LL(0x6eba4d1b,0x03b82577),LL(0xc349b39a,0x8e61b567),LL(0xb35fc43b,0xa670973a)}},
   {{LL(0x5571b4a7,0x80c05ca7),LL(0x8a4af0ba,0x95d14f49),LL(0x67bf4290,0x96767cdb),LL(0x46870ef1,0xb293372f)}, {LL(0x1afe1027,0xc6494405),LL(0x9019c4c2,0x9f027a1c),LL(0x188a593a,0xa392ac59),LL(0xfcb6e1ca,0x68acca2f)}},
   {{LL(0x68ed921b,0xd8f86cbe),LL(0x712d2c07,0x24679ac2),LL(0x4e672cd9,0x18fbdb21),LL(0x51d1f8e1,0x401bb718)}, {LL(0xaa8da4a1,0x688792e1),LL(0x3ca93d06,0xedf9266f),LL(0xaed484df,0x5ddba14a),LL(0xb0ea10a5,0xa5dab102)}},
   {{LL(0x833396ea,0xd397edcd),LL(0xed5e6747,0x78a75693),LL(0x1a5f8260,0xf2c844ba),LL(0x5fb9fec5,0xbcafe59d)}, {LL(0xd3147e7e,0xa2413d5f),LL(0xafdf26cd,0x130dd9e3),LL(0x9ad1abde,0x44be87ec),LL(0x6e77fbe8,0xe925c495)}},
   {{LL(0xf26ced16,0x07ce8d96),LL(0x86ef7306,0x36c85643),LL(0xc7e3d409,0x62545902),LL(0x2ed4d38e,0x1747bf4a)}, {LL(0x55adc839,0x6fe6fc3d),LL(0x8eaf64a8,0x20a3cc09),LL(0x622887b1,0xc1e9b766),LL(0xc41ac659,0x7b9d2f96)}},
   {{LL(0xf2a65e45,0xfdb897ce),LL(0x97889eb8,0x0c09c905),LL(0xe4becf5b,0xa15df10f),LL(0xccef7e40,0x14a3d4fe)}, {LL(0xa8fc67bd,0xedaa11f6),LL(0x5d185b42,0x7bf6fe9b),LL(0x6f9cb5c9,0x7bb9f1f5),LL(0xf97ea9cd,0x1b4ab74e)}},
   {{LL(0x07638d62,0xe9ebf11d),LL(0xa78cf538,0x413a4a87),LL(0x570dd371,0x93785f86),LL(0xfb48063c,0xba431a91)}, {LL(0x4ed4e5fa,0xf1f2ea5b),LL(0x5308325a,0x91a72c47),LL(0xc9ea6acb,0x4e6518e7),LL(0x208f67e3,0xfeaf4c3c)}},
   {{LL(0x2c16bb1a,0x98c5d768),LL(0xee31dc08,0xbf91b62d),LL(0x33a54959,0xe9ad408d),LL(0x38b7170b,0x9d754a64)}, {LL(0xd9d6da2b,0x106db7bc),LL(0xadd533af,0xf556cbb4),LL(0xf16d3b58,0x62db0de0),LL(0x1fa579ba,0x78a1b0be)}},
   {{LL(0x7b552058,0xda96740b),LL(0x626c4d93,0x0c689cc6),LL(0xaf68e53b,0xee3dd5c9),LL(0x134d763b,0x78653a9f)}, {LL(0x3ca5aa67,0xec9c1b72),LL(0x7d56992e,0x67471dac),LL(0xad1d8580,0x0a97dffe),LL(0x0063c040,0x11c7d93d)}},
   {{LL(0x6e90b157,0xb79e355c),LL(0xd9c894c4,0x2c06edcb),LL(0x71a75ed7,0x9b931897),LL(0x8e95ad91,0xd7f00247)}, {LL(0xb85bf054,0xfce1b489),LL(0x503b38bf,0xa3ffb8fd),LL(0xe0fe2ec9,0xe7ea3ad4),LL(0x0049865d,0x0f32f520)}},
   {{LL(0xcff21b51,0x33afa324),LL(0x62a1cd24,0x3d6847d7),LL(0x0b06ce2f,0xf534e159),LL(0xae7cdae0,0x24d27b3d)}, {LL(0x4ad68ab5,0xb727db29),LL(0xb63c0bc9,0x7e8e47e3),LL(0x02389039,0xe81f3122),LL(0x88e6a17c,0x0c281f52)}},
   {{LL(0x091e1c4c,0x3cc00e21),LL(0x867ccc75,0xd500db44),LL(0xf5ebbbe4,0xa8e2e84b),LL(0xc4279ac0,0xc3266881)}, {LL(0x7a170658,0x2e8fb4de),LL(0x51da4a2e,0x219c5ec1),LL(0xeeacee19,0xda69a3fd),LL(0x30462243,0x9d4c6fbd)}},
   {{LL(0xa978c29e,0x43673fe8),LL(0x5861bbc1,0x6e825c95),LL(0xdba8e7ba,0xb41d1435),LL(0xb885d004,0x0f286f78)}, {LL(0xee57436f,0xea42b7fd),LL(0xef7e29c7,0xcdae14bc),LL(0x24251056,0x50cff3f0),LL(0x6f6e8cb1,0xf60809fe)}},
   {{LL(0x12932e53,0xee9f1d15),LL(0x167d5097,0xa6e55cd6),LL(0x9d926359,0x5df8816d),LL(0x797b7eca,0x108e2117)}, {LL(0x91921587,0x7ba20319),LL(0xad23652e,0x304138e4),LL(0x51ebc32f,0x73d0ed57),LL(0xf01d0fc3,0xe0c10a38)}},
   {{LL(0x78c49b19,0x14c6b72f),LL(0x3b7c7418,0x4f952b7e),LL(0xa2d019bf,0x3fe75d21),LL(0x7ca33e87,0x4837bfd2)}, {LL(0x3946e7ea,0x4597729b),LL(0x4c37ea10,0xbe22c14a),LL(0xd7909474,0x91106c7c),LL(0xbf5551a1,0xfbf1e7db)}},
   {{LL(0x55ffea13,0x8e06336c),LL(0x0a1f99f5,0x0deaeda0),LL(0xfda98fc9,0x9b738c4b),LL(0xa59c98ba,0x061cc613)}, {LL(0xb477692c,0x5ceb5b83),LL(0x1fcc473b,0x5db77559),LL(0x83df476c,0x77214b62),LL(0x427ea01d,0x2ffac971)}},
   {{LL(0x49fd0ba7,0xf29f600b),LL(0x7233ef79,0x1106f8b2),LL(0xe8a0ca35,0x706bc171),LL(0xacbff08b,0x4da7a9e6)}, {LL(0x7725f97c,0x17c2fa4e),LL(0xe84a5095,0xab459314),LL(0x6b24d47e,0x01556f14),LL(0xb016dc1a,0x01399059)}},
   {{LL(0x28eca6c6,0x154b84c7),LL(0xd9084e68,0x88ed8612),LL(0x00bf9b5b,0x4dfd5080),LL(0xba9a0cca,0x853cd8ab)}, {LL(0x8af0e94b,0x8cbf9bd8),LL(0x83035d16,0x50782b73),LL(0x4f44533c,0x694d3e65),LL(0xa6e534eb,0x155d4bf4)}},
   {{LL(0xee908b6b,0x9028e2ab),LL(0xa6743060,0x36e5aac0),LL(0x3c37d6f1,0xd26f5a51),LL(0x33729b9e,0x8483703e)}, {LL(0x2e5f82a5,0xf27a6603),LL(0xca55d187,0x33bf2bdc),LL(0x7724a85f,0x894c415c),LL(0xa2ea178d,0x9255d416)}},
   {{LL(0x0a6dc5b9,0x35ecb4e2),LL(0x51c54ed2,0x8b3fc2c8),LL(0x9eede13d,0x059f86eb),LL(0x791dd5eb,0xa796c750)}, {LL(0xea88555b,0xb2f33680),LL(0x1245348c,0x92730950),LL(0xd1e63bfb,0x1a5545f8),LL(0xbebb7858,0xfebc9a14)}},
   {{LL(0x6bdf0c84,0x13cce767),LL(0xa1d341fa,0x1aa9dc07),LL(0x1ee6fa43,0xd46e8ff6),LL(0x4b1dda64,0x4198e5d6)}, {LL(0x2782abd1,0xe82a8134),LL(0xb6d64830,0xe6f7b1b4),LL(0x7f0fb374,0xabe00274),LL(0x7494d7d3,0xf1a8e2b7)}},
   {{LL(0xf274f296,0xd16b0d9e),LL(0x65246ee9,0x3c94a7ac),LL(0x91ec2262,0xd32c32da),LL(0x83116ec1,0x04c7bb90)}, {LL(0x78524a06,0x70fa0406),LL(0x07df8064,0x8d2d5176),LL(0xe2c8d48a,0x13e589f2),LL(0x122aed4e,0x3b110ac4)}},
   {{LL(0x34e972cf,0xe8e0eb52),LL(0xfb3a77fe,0xc082944a),LL(0x6a32c23b,0xcdaff7a3),LL(0xc37b4a2c,0x88cc568d)}, {LL(0xe27b2552,0xc9979a9c),LL(0xd6ef51f9,0x8612ae7d),LL(0xef4e8f85,0x7bf0f937),LL(0x3f12d45c,0x2f360a58)}},
   {{LL(0x9b336663,0x3ec9d0e3),LL(0xb1438d2b,0x5ac2df38),LL(0xff93fde4,0x7f2de910),LL(0xd92534ba,0xbbc460da)}, {LL(0x59a94ab9,0x74de3159),LL(0xc45b80dc,0xd51cfd32),LL(0x6e5b2397,0x9f1f349c),LL(0x995f7271,0xbdbd16ed)}},
   {{LL(0xf59709a6,0x4a7efc1c),LL(0x4b3db99a,0x74e2760d),LL(0x7726a2e1,0xa7238e9e),LL(0x1a376013,0x47764208)}, {LL(0x7e33ebc0,0xbc96f396),LL(0xc9e4ec0d,0x31e159e6),LL(0x6a2ab9f6,0x26a5aef2),LL(0x320eeea7,0x23add07c)}},
   {{LL(0x833b45b6,0xa79a97c9),LL(0xc51885e6,0xb08da907),LL(0xae8d5659,0x23f5e651),LL(0x1faff2f2,0x1eb0be48)}, {LL(0xa1e090da,0xe414ee3d),LL(0x7fcb664f,0x16e4f8fa),LL(0x98c36865,0x7a15a7e4),LL(0xaf89dacf,0xea60e8fd)}},
   {{LL(0x86c1a4b4,0x4e009f45),LL(0x769644ad,0x78c1bebf),LL(0x0b4b3501,0xa41b480f),LL(0x57f0a0e9,0x98be5037)}, {LL(0x3af24657,0x06934887),LL(0x2b6260f9,0xe2503ddb),LL(0xd1d0caaa,0x37c936c2),LL(0x16431f50,0xd371e991)}},
   {{LL(0x6087c5e5,0xd9621d16),LL(0xc53a8bc5,0xae49c2ce),LL(0xcad314cd,0xd7868882),LL(0xaa57af18,0xfde10dc7)}, {LL(0x3800f397,0x3fa8a60d),LL(0x388b333c,0xcec8ae7b),LL(0x85fa8129,0xefd8d692),LL(0x90776c32,0x33d5685b)}},
   {{LL(0x65971a6e,0x47ecec0a),LL(0xad7c5755,0xe8a20bbe),LL(0x87372016,0xbeed0a4d),LL(0x1d9b8dc0,0xd0d499bb)}, {LL(0x2800907d,0xf4ce27cd),LL(0x8622bbb7,0x07a30b77),LL(0x77e264db,0x7532f435),LL(0xd64f0710,0xfdd1a9c3)}},
   {{LL(0xa6543720,0x92ca210f),LL(0x5bb6985a,0x2f19ed66),LL(0xf9399b43,0x08a4ac61),LL(0x757f241d,0x0b7bac5d)}, {LL(0x3763c20d,0x93ef27cc),LL(0xa86b1f05,0xda3b206e),LL(0xd62467c0,0x8f19c74a),LL(0x6a3ad263,0x3ec5ef6d)}},
   {{LL(0x2bc8b9f0,0x249aa636),LL(0x7f70990c,0x0fca7318),LL(0x6f038497,0x6d4aabc5),LL(0x5afaaaef,0x4647f72b)}, {LL(0x7bf90444,0xc7cbe03a),LL(0xcac2efb0,0x6beb69ac),LL(0xbb3c6dc0,0x58544eba),LL(0x96aefc2f,0x569cdcd6)}},
   {{LL(0x5e4c950d,0x2e8a4c57),LL(0x4dd32002,0x6f5341d7),LL(0x6efa5908,0xd0345db6),LL(0xf1d2bbe6,0x4b043197)}, {LL(0xe8730bae,0xde8a4cb6),LL(0xa485cb62,0x9a89231f),LL(0xfcd9bcd4,0xb24ace89),LL(0x3ed5801d,0x01892cc0)}},
   {{LL(0x413408f9,0x80ce2f30),LL(0xf8773b6a,0xaf67343a),LL(0xdd6ade17,0x91acc6d2),LL(0xe5eb3def,0x9d2ffeca)}, {LL(0x50b029fd,0x72f8af06),LL(0x79c0aabd,0x339c802c),LL(0xafc0a6ad,0x46161fff),LL(0xbac9a2d4,0x1046d9f8)}},
   {{LL(0xab920e51,0x2f12eb06),LL(0x2892e552,0xfc004900),LL(0x13e61154,0x9aadf93e),LL(0xabcfd127,0x4468da94)}, {LL(0x152f220e,0x6a5d3ffe),LL(0x4737fe79,0xe6260c23),LL(0x5e6b4251,0x8b5dd1c9),LL(0x9af02b98,0x044f0656)}},
   {{LL(0xa97ff366,0x434d475c),LL(0x2c4bcc46,0xbae8db8e),LL(0xf9777251,0x2ba43a8b),LL(0xdd019f1a,0x7ff430a5)}, {LL(0x630064ba,0x65e9f290),LL(0x7e871c0e,0xfc57a54e),LL(0x5791ae48,0x54107bbb),LL(0x5c334de0,0xdfce137f)}},
   {{LL(0xaed5be73,0xab65c8f6),LL(0x174bf00b,0x838c3958),LL(0xf1c7e594,0x27c57199),LL(0x0d02fae6,0x62643d81)}, {LL(0x5f4431fa,0xc1e70c12),LL(0xb2b41f7e,0xfac86124),LL(0xf0108e3c,0x062ac35a),LL(0xa43d3a28,0xd7d34dfc)}},
   {{LL(0x3143de4d,0xc40fb44a),LL(0xd2e0f9ae,0x06eac4ed),LL(0x95d9a69a,0x998f3211),LL(0xe950bd2e,0xb268f6a0)}, {LL(0x1e9d4e40,0xadfab210),LL(0x73ce9324,0xc453a68c),LL(0x80881171,0x5e2f2544),LL(0xee7e543e,0xe4f596db)}},
   {{LL(0x76b6698e,0x77f48e4e),LL(0x227365c6,0x47b5165f),LL(0x14ef39e6,0xf56ec8d4),LL(0x74ce46fa,0x1638d647)}, {LL(0x08aa8b9a,0x244d0fac),LL(0x298326c9,0x98ccc4d0),LL(0xce0d2983,0x492d7661),LL(0x73158cda,0x728b3e3f)}},
   {{LL(0xc4948011,0x7959ca67),LL(0x08425a38,0x32044ae9),LL(0xb1070c2b,0x05a7a6a7),LL(0xcc129ba5,0x34ed541f)}, {LL(0xb2f1c3e2,0x4b6bf65c),LL(0xd0d8aec8,0x6f090ce6),LL(0xd4fe39c1,0x11ade23a),LL(0xa5a35528,0x50510c08)}},
   {{LL(0xad6fd7c6,0xb7e2a5de),LL(0x2745dca8,0x9d691939),LL(0xad053184,0xff669c38),LL(0xecd0703e,0x394ca6b7)}, {LL(0x60b15e75,0x59e32e80),LL(0x13c14864,0x82dde889),LL(0x64d4f086,0x0fd1624c),LL(0xc9893d7f,0x7fb502a7)}},
   {{LL(0x711479a1,0x59b86bcf),LL(0xc40b44bc,0xfd4bc4d8),LL(0x988100c3,0x2fae18f5),LL(0x615867d7,0xe4af2f4f)}, {LL(0xbe479e28,0x7d45e1e8),LL(0xa04798a5,0x547accbd),LL(0x1c458b5e,0xe88a85b1),LL(0x6032f0cc,0xe144f7f2)}},
   {{LL(0x3f807365,0xad5276d3),LL(0xb318a6ea,0x5b6519e7),LL(0x2d0fcf50,0x5889cbb5),LL(0x2bdab4e0,0xdce91cab)}, {LL(0x41b78954,0x17b6249f),LL(0x6f10449b,0xc9320b65),LL(0xf264ae8f,0xe38a7cc0),LL(0x52b85829,0xaab8803e)}},
   {{LL(0xdd97973c,0x63668621),LL(0xd04138c7,0x5aaedce7),LL(0x1762874c,0x8e8e6614),LL(0x163fc862,0xd0cefcf4)}, {LL(0xffed1ace,0x0ebe0048),LL(0x7a8c2673,0x070c3348),LL(0x9b0d3fd7,0xb801d159),LL(0x922d4842,0xf1d55911)}},
   {{LL(0x680dcbf9,0xf0acf768),LL(0x4f0a51df,0x5072b825),LL(0xd88df9c5,0x3a74751c),LL(0x1cc1a332,0x9d20f989)}, {LL(0x6926c34a,0x4e90042b),LL(0x00766880,0x5c728b1e),LL(0xf76e9dcb,0xf2e3bfe8),LL(0x15a125ae,0xd9822f0a)}},
},
/* digit=17 base_pwr=2^119 */
{
   {{LL(0xf51b14b0,0xbf84db58),LL(0xa39a79f0,0xdf73ccf5),LL(0x2b5a1f11,0x0ce1e584),LL(0x185fc400,0x841fa6a3)}, {LL(0x2455c32a,0x94b09c68),LL(0xbfa71cc3,0x383c9bde),LL(0x1e797929,0xb6381486),LL(0x623d0a5d,0x33036faf)}},
   {{LL(0x90f17cba,0x41b6cf7c),LL(0x30c7c5f4,0x5d655ff4),LL(0x4ccc7f38,0xc64f29d5),LL(0x6124a79e,0xf28e8531)}, {LL(0x67bf1e98,0x1efa8d51),LL(0x5d7a33b0,0x8610027f),LL(0xcb9a40a4,0x35fe2bb2),LL(0x43d50a0b,0xc5cc1bf1)}},
   {{LL(0x46e33870,0x84dbc605),LL(0x843c4e1e,0x23d8d2e5),LL(0x4cf8b569,0x69964b5e),LL(0xe0c546a5,0x2a5228e8)}, {LL(0x96d6e111,0x4c0467ed),LL(0xa12bd298,0x25764cdf),LL(0xfbaaad46,0x92a3e7fa),LL(0xd12fa469,0x08ac1d36)}},
   {{LL(0xa32106c2,0x60ae2bbf),LL(0x3e917750,0xef155b2a),LL(0x13853a30,0x5567c3c7),LL(0xeddb305b,0xa6be8290)}, {LL(0xade26eec,0x2db58c21),LL(0x003c17ed,0xfa3c895c),LL(0x6293f8a2,0x96ab0de1),LL(0xac3710c6,0xbd2365ec)}},
   {{LL(0x6aa24f73,0x93ea8553),LL(0xe0410c40,0xf75140d0),LL(0xaff0f7f2,0x760cfa2f),LL(0x3e580d68,0xc6dfb3c7)}, {LL(0xc16d39e2,0x25fc2091),LL(0x19e1d5e2,0xa05b0c81),LL(0x62bbec7a,0xd4d223d8),LL(0xf293f34a,0x11a93775)}},
   {{LL(0xe194c642,0x9ab03e73),LL(0x789e3c85,0x607b7106),LL(0x4bdacd55,0x952aab02),LL(0x21cc6084,0x31ca3ee2)}, {LL(0x1c6b93f9,0xd3149b2b),LL(0xead930f8,0xcbc5ef3b),LL(0x22872630,0xed04984f),LL(0x6c4b6fe2,0xef5d052d)}},
   {{LL(0x6010ffa2,0x808ae6c0),LL(0x1143166a,0x88b6fcd8),LL(0x5ab945ec,0xa2780263),LL(0x36db5012,0x4777b4aa)}, {LL(0x059aa3c7,0x2156bf83),LL(0x2a660260,0xcbef6fb7),LL(0x8b366ce5,0x189fa221),LL(0x08290914,0xd6f5bdaa)}},
   {{LL(0x57028677,0xd2028d05),LL(0xce9aabdf,0x90eebeeb),LL(0x06d4e5d0,0xab977aee),LL(0xf9361873,0x7a98c527)}, {LL(0xb7c2474d,0xe49b1251),LL(0x5f3e7b02,0xcdaf2a36),LL(0x6fe54af1,0x638bcaf4),LL(0x1dac06b7,0xfec42624)}},
   {{LL(0x3741a88b,0x422be225),LL(0x5304f139,0x1f3b82c3),LL(0x181c2fc2,0x101eab8e),LL(0x5898163c,0x8522738e)}, {LL(0x2d2bac1b,0x0372d05f),LL(0x1d1f3c42,0xb65c786c),LL(0x64e2f5b3,0xbd59694b),LL(0x24c5671b,0xf1c424bf)}},
   {{LL(0x1eafe37b,0xda738cf5),LL(0x30dd7c2b,0x503eac24),LL(0x11dd77af,0xf9b7b7a5),LL(0xe9dcfe7c,0x0ade03af)}, {LL(0xf479e3b5,0x489bd34a),LL(0x030a33f3,0x993ab403),LL(0x9fb64068,0xaef322bf),LL(0x0e27f110,0xa47cc71b)}},
   {{LL(0xefab99c8,0x1c445554),LL(0xa7f10e58,0x7c3c51e7),LL(0x78a87474,0xaa8b43ee),LL(0x2418475a,0x037d6397)}, {LL(0x10324911,0xc9c751fe),LL(0x3e0797d4,0x3d65d9e0),LL(0x7dea2a63,0x98b68d2b),LL(0xf4afca19,0xa211ed3b)}},
   {{LL(0xc63b9e75,0xe19ff8f8),LL(0x0d7d130a,0x43f6b4fc),LL(0x7c8490b7,0xdba3385d),LL(0x0252c14a,0x97889df7)}, {LL(0xb2f18b9f,0xfccfca86),LL(0xc3a87422,0xf092ff9e),LL(0x67474838,0xf96dd675),LL(0x5bad2e9f,0x039e8287)}},
   {{LL(0x52e041f6,0x7ed85e70),LL(0xcfdeb19f,0x3d6ef1e7),LL(0x0d9ac66e,0x9f9fe399),LL(0x16cb8554,0x5825e7bf)}, {LL(0xd954a4d5,0xecffdf90),LL(0x20678fc5,0x8617ffdd),LL(0x666df77b,0x3e974054),LL(0xb5d92788,0x748379d1)}},
   {{LL(0x2da32c0a,0x46a60911),LL(0xb2676ca3,0xb2e1ac32),LL(0x17614dc6,0xfb74093f),LL(0x3f27f965,0xf44939e4)}, {LL(0xc922422b,0x4656a402),LL(0x3ff5c56f,0xd60a55ba),LL(0xab9aa62e,0x0d061b41),LL(0xaca3acd2,0xc9ceacfe)}},
   {{LL(0xd946003b,0x056d5718),LL(0x2c7815f3,0xf8530d6d),LL(0x706536b8,0xbae14342),LL(0x2b901188,0x45c96dda)}, {LL(0xc64ed946,0x386d88b6),LL(0x6c00f1c2,0xb7017022),LL(0xec8988df,0x28519811),LL(0x5a05cffc,0x3b011fe2)}},
   {{LL(0x515f954c,0x4f581d47),LL(0x7f470a40,0x145f925b),LL(0x736feaaf,0xfee6b6b0),LL(0x2ea5743b,0xf90744af)}, {LL(0xa2f36f56,0x4d8e8cea),LL(0xe3ed4007,0x4239a6ce),LL(0xd515e6db,0x0906b5bd),LL(0x8ac973d1,0x53622990)}},
   {{LL(0xeb2fe229,0x472ceb94),LL(0x6a121363,0x0775ed41),LL(0x761ddb38,0xc0492e07),LL(0xaef9be2f,0x80c24d51)}, {LL(0xdcba73a1,0xa2a3982b),LL(0x4e26d062,0xe0d83978),LL(0xcd41c930,0x794959a8),LL(0x70131161,0x7d2a88d7)}},
   {{LL(0xf4f966da,0x48f93fc3),LL(0xed5b6487,0xf92691a0),LL(0xada2c1fc,0xc5a93e5d),LL(0x4b7d9243,0x4a7aca52)}, {LL(0xd7c5598b,0x810aba93),LL(0x25109638,0x98f9ead2),LL(0xa388481a,0xe8c6e893),LL(0xe91ce74b,0x56e96b9b)}},
   {{LL(0xd935f591,0xfa1e5dc3),LL(0x555eb816,0x985bb06c),LL(0xc4d14e69,0x6478c518),LL(0xc7f47349,0x48afbdbc)}, {LL(0x26fed96c,0xbde90933),LL(0xcd468186,0xf9b96f41),LL(0x730e8518,0x22de6a29),LL(0x915db842,0x7a3dc912)}},
   {{LL(0xfc1f9449,0x8d13b665),LL(0xdd4bba75,0x6e9932a9),LL(0x564083da,0xa90ce8e5),LL(0xbbf7989d,0x8a7cf362)}, {LL(0x1b07ee2f,0x652eccb7),LL(0x6857a141,0x0c0dcf1a),LL(0xb7bfb43e,0xa87ec410),LL(0x82b8d179,0xaebdb7e7)}},
   {{LL(0x625a24dd,0xeb3bc468),LL(0x463b1f89,0x7e45e47b),LL(0x00c75a48,0xc3013535),LL(0x13778807,0xafea920d)}, {LL(0x22dcef16,0x0d1e9277),LL(0x86cecfd6,0xa2a10f67),LL(0xd7160bf2,0xad40e29c),LL(0xeac1265e,0xe78e6589)}},
   {{LL(0x0c62c041,0xd3a24310),LL(0x6c03c747,0x4d27344a),LL(0x7d3ee9d1,0x0b19e4a6),LL(0xcd90de33,0x9cf2eccd)}, {LL(0xfda636a9,0x673a9d1f),LL(0xa86ee501,0xb7349981),LL(0xe43766ed,0x11ca1e49),LL(0xe3ff3b08,0x0806af6f)}},
   {{LL(0x8a01f119,0x21304338),LL(0xf3cb268f,0x58a6d3be),LL(0xe37d7851,0x40ceacca),LL(0xef5b81e8,0x18694595)}, {LL(0x84bad32a,0x35678ed7),LL(0xd1624256,0x4f280f92),LL(0xfb28709c,0xdecb1f1e),LL(0x164911d7,0x2a7f3048)}},
   {{LL(0x579d8a41,0x32551d31),LL(0x60a5ee33,0x754c7c24),LL(0x6a88f85f,0x2c53fbff),LL(0x2c7a36a0,0x6ad0bda7)}, {LL(0x15724d6c,0x8b3674f8),LL(0xb9b7b34a,0x614727ce),LL(0x82ca9cd7,0x384fba98),LL(0x0c566025,0x8ef4343c)}},
   {{LL(0x64886c98,0x5645fefb),LL(0x0f5c29e8,0x702befb3),LL(0x46de1523,0x6d74a7e0),LL(0xb1302163,0xcb2bcdb9)}, {LL(0xab4ca69b,0xe65cff39),LL(0xf2d4f6ec,0xeacb7103),LL(0x1770d3ef,0x15826c2d),LL(0x3f710600,0x38b008f1)}},
   {{LL(0x4bc7dccb,0xc68450cb),LL(0x9e5f2400,0xb5f11b26),LL(0x9c3a833b,0x2af58e9e),LL(0xa058abaa,0xb60e2672)}, {LL(0x75b080c0,0xe891bf8c),LL(0x2434bf38,0x5b09b276),LL(0x700b8739,0x0d90a040),LL(0xe69f4a0b,0x995cb042)}},
   {{LL(0x44a56b84,0xe30df0a1),LL(0x1ead5a62,0xbaf92d16),LL(0x6e0193a4,0xe214a062),LL(0xe9758b9e,0xd41de5bc)}, {LL(0x732d82d5,0xcf214213),LL(0xf949f07b,0xaa1421f6),LL(0xf7fb101c,0x5f38c91e),LL(0x2a3e41e4,0x47ce2ec2)}},
   {{LL(0x240c7897,0x6bb34768),LL(0x7b45473e,0x80ff54ea),LL(0x82fe5aac,0x16acd40f),LL(0x4350449f,0xa3e76f52)}, {LL(0xacacbeb9,0xf7a3579e),LL(0x7bc40732,0x9791e0e0),LL(0xbc58cb9d,0xb26da7b5),LL(0x987e18f4,0x11d9fc80)}},
   {{LL(0x1d8e0d34,0xc3c189a8),LL(0x2d42e0b5,0x3011097c),LL(0x94ab9371,0x4e385932),LL(0x0c237147,0x79e0c2ce)}, {LL(0x7172e6ce,0xc9f17122),LL(0x9b89a008,0xf8d73b1d),LL(0xa945995d,0x91690c6b),LL(0xc15306c6,0x089eb941)}},
   {{LL(0x12ac8df5,0xee5f526d),LL(0x3bf94c65,0xf1dd23f7),LL(0x81a3cb0e,0x594ceaac),LL(0x9757cc8b,0x07d41d3b)}, {LL(0xfc5603d5,0x9eb0081d),LL(0x57bd230c,0xfb5d3298),LL(0xcde3f845,0xf2c0402e),LL(0x41e8aba6,0xa2defd67)}},
   {{LL(0x2dd9573d,0xb300802a),LL(0x60c1ded3,0x64e401a5),LL(0x8ab1d3d8,0x19d4a677),LL(0xcca04f74,0x3c2092f2)}, {LL(0xac40056a,0xf4827ba5),LL(0x9c09ddc2,0x49d4cf22),LL(0xdbf20277,0xb2b00f6b),LL(0x5b281e9b,0xc9ac48d4)}},
   {{LL(0x32efbbce,0x648d6674),LL(0xe9639719,0x64a6c2b3),LL(0x30662e7d,0x38c04657),LL(0x352c9184,0x15d1d7ca)}, {LL(0xcc3020cc,0x70e8630c),LL(0xb09f038f,0xe4b56c9c),LL(0xfe76a744,0xdb9cb5ed),LL(0x6947b988,0x4c85f020)}},
   {{LL(0x29d8add4,0x7e500126),LL(0xbfaf6d7e,0xdbcfd295),LL(0x38df80be,0xc1a1c228),LL(0xf606ce3d,0xcfa6272a)}, {LL(0x8e0af540,0xbf2a5720),LL(0x5b599ab0,0xb9c544fd),LL(0xd0a22c9a,0xd6dc994d),LL(0xd23e4c0e,0xa8a12acf)}},
   {{LL(0xba588a5e,0x41f7ac85),LL(0xccdb9687,0x5425fa00),LL(0xec9398f0,0x12fbce76),LL(0x4f550b9b,0x2ad69251)}, {LL(0xbb318636,0x120ff0f2),LL(0x01ecd90b,0x9378346c),LL(0xd0ba149b,0x1b98fe99),LL(0xc9c481c8,0xd69d5462)}},
   {{LL(0x959e428e,0x11c79184),LL(0xcff227cc,0x9de61a8d),LL(0x1e09b860,0x144dfdcd),LL(0xf8ebe350,0x110c3a47)}, {LL(0xfadf86b0,0x59e574dc),LL(0xcf3b8d30,0xe6ff6e12),LL(0x19c77143,0xe2d512fc),LL(0x60279af1,0x63461543)}},
   {{LL(0x32b4d371,0xff65189c),LL(0x0faf5ba7,0x022fecca),LL(0x414707b4,0xd08fe9bf),LL(0x722d5fd2,0x0ef8af2b)}, {LL(0x4e6fa14a,0xbef06063),LL(0xcca29177,0x1c41f752),LL(0x65091fe1,0x17dc7e18),LL(0x23f13c18,0x693d72d2)}},
   {{LL(0xce8e2d30,0xce88eb02),LL(0xe972faca,0x7071f98a),LL(0x549c38ee,0xb7388d61),LL(0x0b788b8c,0x7cfccee2)}, {LL(0xcb93b5e8,0xdc470705),LL(0xab96d485,0xea053c18),LL(0xd634c9b3,0x70e96456),LL(0xd5999cf2,0x2c58c20b)}},
   {{LL(0xa77c1767,0xcd392b3c),LL(0x7c4e6bd9,0x14471fab),LL(0x75c905ff,0x312e1547),LL(0xace47038,0x45764654)}, {LL(0x8fc3d697,0xa34a0b0e),LL(0xd309ed3a,0x5d9ad31a),LL(0x0918f957,0xbba340c0),LL(0x31fd72a1,0x768e06e8)}},
   {{LL(0x3e1a4a54,0x77e5dd92),LL(0x3fdbc1e1,0x0970719f),LL(0xb0371fe2,0xd4f1da6f),LL(0xfd7f895a,0x3635f433)}, {LL(0x411c8e6f,0x0e8e40e6),LL(0xec18103c,0x31d126bd),LL(0xc299d7cc,0x415a0cc1),LL(0x3a8e97f1,0xdf528e7b)}},
   {{LL(0xeed95e91,0x4551a8c7),LL(0x32bcfb03,0x8de89888),LL(0x2eac5c3a,0x25da4f5f),LL(0x5f88d63f,0x6d0b2e25)}, {LL(0x575d6145,0x8d158d14),LL(0x345f62b0,0xe5601a6b),LL(0x113c6895,0x6f951699),LL(0xb87e50ef,0x79e29fd5)}},
   {{LL(0xd5fa51ff,0xf1ab215c),LL(0xaf2c3094,0x4fc5c4ea),LL(0x2c006042,0x1baeda40),LL(0x3e30e75f,0xcdfcc37c)}, {LL(0x467f57eb,0xdd64e5dd),LL(0x22902d21,0xa5b13731),LL(0x1c52cb7b,0x856866dd),LL(0x16a08caa,0x05cf0f7a)}},
   {{LL(0x533b4d09,0xa46e8a55),LL(0x4e073af1,0xfc803998),LL(0xe0d589c3,0x8e3825c8),LL(0x4c1daef3,0x505e8e5d)}, {LL(0xc5f3db12,0x9f8363b1),LL(0x74f569e2,0xe7d46700),LL(0x4d68461a,0x551fd2ed),LL(0xa8bbe83d,0x26248da5)}},
   {{LL(0x65681dbd,0x8d90c47f),LL(0x2200ba6b,0xe726d25e),LL(0x65a3bc9b,0xa2fe408f),LL(0x9c443b57,0x94a80457)}, {LL(0x07364677,0x95f7f024),LL(0xdaf0fb34,0xe9d9bc87),LL(0x5588e979,0xe9082548),LL(0xa0e61ff2,0xede1f94d)}},
   {{LL(0x45e1c230,0xcb89a1e8),LL(0x50a15304,0xee014c23),LL(0x2bab57e1,0xf25d8ffa),LL(0x26223c6e,0x8a920680)}, {LL(0xaadf7e6a,0xc5abb7af),LL(0x9e7d8da5,0xcb57c893),LL(0x7d589a91,0x839bcda0),LL(0x77e82027,0x1fa774c0)}},
   {{LL(0xba6504d7,0xeca669cf),LL(0x6845e47d,0x7bf09544),LL(0x607b3641,0x5eb6c33e),LL(0x64bab450,0xf445556e)}, {LL(0x86096fde,0xed0b1c02),LL(0x8ea41693,0x2c5ba666),LL(0x37ec248d,0xe578b315),LL(0xf64ed28f,0x97ef44fe)}},
   {{LL(0xce419462,0xfa5a6c46),LL(0x9cce80e9,0x29336dc9),LL(0xeee7617f,0x9e9054b9),LL(0xf3d51cba,0xcea9a100)}, {LL(0x13267ec6,0xc3cce5e8),LL(0xa4e807e7,0x650c8383),LL(0x9b2726dc,0x1338e22e),LL(0xbf79b47a,0x220c50b2)}},
   {{LL(0xa0e0962a,0xe160d496),LL(0xe1ed5cdc,0xe1a26460),LL(0x31427c62,0x9a1ed8c3),LL(0xe99a096a,0x65ef5300)}, {LL(0x4e3ad558,0x38abea5f),LL(0x0880ba0c,0x03bb15e9),LL(0x0141b036,0x1e6dda7e),LL(0x5bf37669,0xd31b08bf)}},
   {{LL(0x68da20d2,0x948e0366),LL(0x4108fe36,0x36998a24),LL(0xf9d6563b,0x7606e6ed),LL(0xe42172ba,0xcf7cbdd3)}, {LL(0xa1265b99,0x2335a9a4),LL(0x30ac3f69,0x64776cdc),LL(0xa59b595e,0x04040362),LL(0x2cbc03cd,0x82df96b9)}},
   {{LL(0x6cea2796,0xe9d18c7f),LL(0xe1ea7e35,0x3112c4f6),LL(0x5f8a786d,0xf9cbc205),LL(0x2097da0d,0x36cc6d42)}, {LL(0x2153e665,0x54093350),LL(0xce937bb9,0xebe9db0f),LL(0xd95942f8,0x9d1a5972),LL(0xd4bd5c74,0x81c1f94a)}},
   {{LL(0xaa04152e,0x61dc7318),LL(0x95e5ec9f,0xdf636db1),LL(0x48090850,0x64a80d46),LL(0xce890a30,0x2b37679e)}, {LL(0xff6025e3,0x9f46d5b9),LL(0xf24534dd,0x6eed5a44),LL(0xf740a84b,0xc56b5cb1),LL(0x228cc826,0xb4641c28)}},
   {{LL(0xaf62b943,0x676289be),LL(0x1eae5409,0xe3f3810c),LL(0x04b5be78,0x73613f32),LL(0x398b556c,0xe6359179)}, {LL(0xc0263f77,0x6a342b12),LL(0xc10a6db5,0x6b093bbd),LL(0x29832fb9,0x8f3fc90d),LL(0xff03b2ff,0xb3f2d8fc)}},
   {{LL(0x64457331,0x1de7bd1c),LL(0x43bb1380,0x0a03a06b),LL(0x8bf17419,0x6720cc3d),LL(0x33938d5a,0x2627c7da)}, {LL(0x8d98f34c,0x204da058),LL(0x51cbd284,0x80e29f46),LL(0xa46f93d5,0x11b22dd4),LL(0xe971a61a,0xd7341655)}},
   {{LL(0xee69f782,0x36a748b7),LL(0x94f08ac0,0xa3740020),LL(0xc36099f3,0x383fb245),LL(0x00137fdc,0xa7cb0ef9)}, {LL(0x6e1dd2e5,0x5371052f),LL(0x7976a1d3,0xed3ab7b5),LL(0x9df822e6,0xb0119c0d),LL(0x358685d1,0xafd2a477)}},
   {{LL(0x4ae1103c,0x82879cb0),LL(0x94385be6,0x61cd6ca8),LL(0xd85d4a62,0x7c551809),LL(0xb318d965,0x9632ac5f)}, {LL(0xe1340605,0x67baad2c),LL(0xac6ed4f7,0x39c2c8c7),LL(0x71211c2f,0x42c4a7b1),LL(0x9bf758f6,0x43c778bb)}},
   {{LL(0xf519acb2,0x2dc8fc39),LL(0x08eff177,0xd3c30a6d),LL(0x5144484b,0xf889c021),LL(0xca376af3,0x01b82327)}, {LL(0xd3e71253,0x168a0b2f),LL(0x3f9ff89d,0x5e92c6ba),LL(0x5b4c6844,0x8c74c132),LL(0x33de6858,0x707c4a40)}},
   {{LL(0x9c36dd9e,0xb13f6abd),LL(0x9b3aa9f5,0x4baaef52),LL(0xcd156392,0x0a4fa929),LL(0x6506f42f,0xde0f1956)}, {LL(0x150d4ee7,0xe43dd7f0),LL(0x7247f552,0xf7afe7db),LL(0x9072b450,0x9abc8d1c),LL(0xc9a8e159,0x5d0645d5)}},
   {{LL(0x01c6f17a,0x863d3e8f),LL(0xdf080690,0x3a0575ac),LL(0x2b0fb150,0xcad62d87),LL(0x625c35c6,0xa1f54744)}, {LL(0x41fe59ec,0x7d3bcec3),LL(0x169f1e04,0x0fd3e40e),LL(0x2ed9aa4b,0xbde8c827),LL(0x13046c6e,0x71562ee6)}},
   {{LL(0xe9acac7a,0xaf049c5c),LL(0x261dd422,0x7efec06c),LL(0x095564c4,0xa530fbfd),LL(0x2a57af79,0x000c0c82)}, {LL(0x2ce1315c,0x9f79868f),LL(0x1b5d575e,0x0dd99453),LL(0x1e534cfd,0xf1a49419),LL(0xed7e8b39,0xc7de8756)}},
   {{LL(0x3ed2ccb2,0xef61f5c8),LL(0x34af2a15,0x032ee766),LL(0x9f69ae9d,0xe0692ed5),LL(0xf64900df,0xd34fc2d5)}, {LL(0xaca6d51b,0x1c51c950),LL(0xa7717dfb,0x10ae0fb2),LL(0xa7ec7ca8,0x9fa305f7),LL(0xb5728214,0xb215a8ab)}},
   {{LL(0x8819505b,0x62628fdf),LL(0x004ba54e,0x3cefd86c),LL(0xc571da3d,0xa17bed74),LL(0x93a5faa5,0x362dfef6)}, {LL(0xf8aeea05,0x1bee6899),LL(0x16f18b7a,0xd7bf7e31),LL(0x1cb7685c,0x3f3cf39d),LL(0xe2e57c8e,0x1df41f23)}},
   {{LL(0xe2fd94f1,0x8f62ecb8),LL(0x4c30a178,0x652099c9),LL(0x4262e9e6,0xaa2454e1),LL(0x2015d4a9,0x7f0d440f)}, {LL(0xbb5b39fa,0xa2c76313),LL(0x1ab47bb3,0x46e57ab2),LL(0x8697e682,0xd181f444),LL(0x33273dfe,0x55db129e)}},
   {{LL(0xe71d029f,0xda188361),LL(0xb5def631,0x3e3e19da),LL(0x087ad30b,0x7431f513),LL(0x9f27c84e,0x2537887e)}, {LL(0xac9df89d,0x0c228c62),LL(0x10031289,0xdcd2c5e9),LL(0x0321d1b6,0x5cc76782),LL(0x6cb3d600,0x4e460bdf)}},
   {{LL(0x9a870166,0x6f356aab),LL(0x497d4ac0,0x21aecb3b),LL(0xf0495ef1,0xd981a4b0),LL(0x0fb7704b,0x615e8bff)}, {LL(0x8478bf12,0xc148e8ea),LL(0x364eee52,0x7011ec5b),LL(0xf692bc12,0xd9075965),LL(0xe622ad51,0x3019c824)}},
   {{LL(0xec83c953,0x349e4873),LL(0x3a21ef0a,0xb4f59fb3),LL(0x40f7d93e,0x3872d314),LL(0xc2568c82,0x479e1d02)}, {LL(0x65d43d22,0xd7e4dc9a),LL(0xe775efa8,0xcc068e81),LL(0x326defa6,0xb78ccae9),LL(0x2da64956,0x8f92b296)}},
},
/* digit=18 base_pwr=2^126 */
{
   {{LL(0xdea227ee,0xb721f8d5),LL(0x3dda8ba0,0xf48c766c),LL(0xe43e3520,0x0583d94b),LL(0xe1d898b6,0xebda36c9)}, {LL(0x6627adaa,0x1808286a),LL(0x9938368e,0x19c4c620),LL(0xf750949f,0xe0dbd707),LL(0x0cf356d9,0xcadf4bea)}},
   {{LL(0x2dc890a7,0xf5de2126),LL(0x95aa75a3,0x76b7b675),LL(0x2a070b32,0x475fc143),LL(0x8e31d68f,0x7429a646)}, {LL(0x09be3dca,0xec3a9aaa),LL(0xaf780ed7,0x07e119a9),LL(0x64fd96c4,0x62125625),LL(0xe8e80577,0xb571494f)}},
   {{LL(0x5228d735,0x955ee349),LL(0x8fc5d4b6,0xa04ef2bb),LL(0x3600814f,0x0c532891),LL(0x59f85bd4,0x41f1f637)}, {LL(0xe3dcdfb4,0x72f1d731),LL(0x3aa5edb3,0x28a4ddb9),LL(0xf702dcdb,0x116a68e1),LL(0x3bde657e,0x1975bc42)}},
   {{LL(0x8a914b50,0x7b9f561a),LL(0x9154d377,0x2bf7130e),LL(0x519b4c35,0x6800f696),LL(0x568b4c56,0xc9e65040)}, {LL(0x6d98a331,0x30706e00),LL(0xe211ce1e,0x781a12f6),LL(0x40562e5f,0x1fff9e3d),LL(0x8c166747,0x6356cf46)}},
   {{LL(0x429945a7,0x80e87329),LL(0xb7ab06ad,0xc619fe17),LL(0x6fd86b17,0x9116bc2e),LL(0xb9116aac,0x64a41877)}, {LL(0x32ba4f3b,0xe3ed867e),LL(0x68b4ebe6,0x013e263b),LL(0xe779e4ec,0x305ebfe7),LL(0x50178251,0x5536d45d)}},
   {{LL(0x8873a93d,0x5abb939f),LL(0x8c4c9cb1,0x0263ba48),LL(0x6b78a4b5,0x36764b8d),LL(0x28bebc1e,0x205bb45d)}, {LL(0xae89dcd5,0x16df4bb0),LL(0x316fadb7,0x85994670),LL(0x3af3c724,0x71f75664),LL(0xe8520c9c,0x43e30313)}},
   {{LL(0x29e91921,0x3ab9ec54),LL(0xe3299f47,0xd931436e),LL(0xb89cd49f,0xb56da7bf),LL(0xcff7f637,0x90623412)}, {LL(0x714022de,0x751e7944),LL(0x2c338262,0x86bcc342),LL(0x314c23bb,0x85f6a9bc),LL(0x1f0a3991,0xedbe8e74)}},
   {{LL(0x003b40dd,0x7a748d63),LL(0x3951b7ae,0x8a682402),LL(0x704a91b0,0x41e92dd9),LL(0x858cd3ea,0x2dfb3eb9)}, {LL(0xf5094667,0xc3c2af35),LL(0x7435aa2d,0xffa287dc),LL(0x7462714f,0xd03f3979),LL(0x203e5b0e,0xdb550f67)}},
   {{LL(0xe241ed0c,0x6df7801b),LL(0xb642fd3a,0xb52c0a3f),LL(0x1977a29d,0xdd35e1cf),LL(0xa661404c,0x8e793d60)}, {LL(0x6b9442ae,0x393e2b87),LL(0x2aa6b707,0x123b893a),LL(0xdb8d306a,0xeec88682),LL(0xce847879,0x92c2d93d)}},
   {{LL(0x80ec63b4,0x725f1e7d),LL(0x74113de0,0xcb8f53d9),LL(0xb819f730,0x2132a072),LL(0xb4c61f06,0xfabf3c47)}, {LL(0x2cb243d8,0x79c1bc86),LL(0x757e3600,0x442833c5),LL(0x4e918b8a,0xfa4f69ad),LL(0x73bc193e,0x5816f3f3)}},
   {{LL(0x30f40e93,0xc671c7a4),LL(0x5c51cfa4,0x6041aa03),LL(0x2fac25d7,0x3a713549),LL(0x24a7df01,0xf5053237)}, {LL(0xd29f4ec5,0x99efb34a),LL(0x71d2cb1b,0x74810523),LL(0xf3a029ab,0xacefaf8f),LL(0x069d9545,0xc82e4f5a)}},
   {{LL(0xd3341d80,0xd759549d),LL(0x31a2a0a4,0x079e9fa7),LL(0x2a164f75,0x75da56c7),LL(0xbeefc182,0x9313ef5a)}, {LL(0xbde130ad,0x0aa365b6),LL(0x98411180,0x44265977),LL(0xaa26466a,0xa65373f7),LL(0x2e2cf404,0x1a43bee6)}},
   {{LL(0xb37a9390,0xe029ed6d),LL(0x34970065,0x5c2351ca),LL(0x1c46d72c,0x7c4f3c30),LL(0x7262ce20,0x09ce770a)}, {LL(0xdd58a9f8,0x0cfeefad),LL(0x408addaa,0x06797d79),LL(0x05aed325,0x76a87c06),LL(0x8a46d0c6,0xe002b672)}},
   {{LL(0x05b6e1a4,0xcf77ea31),LL(0xa5d92b00,0x3bf900bc),LL(0xdccfe144,0x05996d8c),LL(0x951a602c,0x73d4dfd7)}, {LL(0x0ed8885d,0x033f3959),LL(0x36400817,0x8332dc73),LL(0x2d8ebda7,0x96372295),LL(0xb5da0c67,0x3fb32cf6)}},
   {{LL(0x3e36defc,0xcb521d65),LL(0xa67f00f0,0xc293d170),LL(0xfb35bd06,0x6a3a2fd4),LL(0x0bd490a5,0x537937dd)}, {LL(0xc274ee5a,0x898d94bc),LL(0x7515b5e7,0xdc70f9bd),LL(0x3749900f,0xa94673db),LL(0x49ad3b04,0x3e6e2af0)}},
   {{LL(0x207eecd9,0xb9dae1b8),LL(0xec07b47c,0xd3f50d63),LL(0x364292da,0x02b4d909),LL(0xfc35975b,0x919a6df3)}, {LL(0xb616452e,0xb41ed4aa),LL(0x5cfc6abb,0xe58689cd),LL(0xf389b025,0xeac325d9),LL(0x8f255de5,0x45ceb1e6)}},
   {{LL(0x5e46cdff,0xda4a0715),LL(0x0f6c761c,0x8a860a55),LL(0x5fe1eef1,0xe1395274),LL(0xf7bc535f,0x256e296a)}, {LL(0x2755dd27,0xf3d4b06c),LL(0xbb530c26,0x3ced6ee5),LL(0x96ba599d,0x73249ad7),LL(0xe8a66027,0x5de8dab3)}},
   {{LL(0xc2f97e01,0xa4892840),LL(0x427945be,0xbe0dbe49),LL(0xa57d4e4f,0x6fd86a7b),LL(0x04a2e778,0x7f56c3e0)}, {LL(0xffc13d49,0x734708cc),LL(0x788d31fe,0x3c1d9413),LL(0x8d3e4c36,0xfe85545b),LL(0x8815129c,0xcca441fc)}},
   {{LL(0x15e3d172,0x2e2095e2),LL(0x64b43e81,0xc0c8d3c4),LL(0xc68e802e,0x084557ab),LL(0x30d239b9,0xa6b73590)}, {LL(0xb67b0548,0x61ec00a9),LL(0xb8ab138d,0x630059de),LL(0x36ca9888,0x800abf01),LL(0x9517149e,0xe26d644a)}},
   {{LL(0x58bf21d9,0x775d5a98),LL(0xdbeab706,0x00eb6846),LL(0x8232d477,0x9d714c9f),LL(0xb70f91c2,0x7cde2c3e)}, {LL(0xe9871f0c,0xe6d0a8ce),LL(0x19e8444a,0x902bc60b),LL(0xff0cd43a,0x8651ed57),LL(0xd480d222,0x4418cc07)}},
   {{LL(0xf3cbe01d,0xb5e0c7e3),LL(0xe43adcdf,0xbf4a899f),LL(0x78f8f79d,0xb89b022c),LL(0xf42c797c,0x79cbbf97)}, {LL(0x59d53cc1,0x46d73cc5),LL(0x4ffca67c,0x99f683e6),LL(0x98865e5b,0x527c16ec),LL(0xf68f8ee0,0xc844b70f)}},
   {{LL(0xc9854994,0xcffcccc0),LL(0x74926d5d,0x4aafcc15),LL(0x835aea59,0xeb084832),LL(0x20df21cf,0xcb317b5f)}, {LL(0xe43d1853,0x3c45b084),LL(0xb93b9167,0xd12c9411),LL(0x19316bdf,0xb0901982),LL(0xd11ab5e2,0x76bfa2ac)}},
   {{LL(0x4e84d3e9,0x22bf23cb),LL(0xd1572d4a,0x96ec9f8e),LL(0x080ba39a,0x31594ae4),LL(0xadc6bae4,0x105b5677)}, {LL(0xa644e230,0x501e45dd),LL(0x64573126,0xeb571f27),LL(0xa36ac1ef,0x1fc3d478),LL(0x327c7da7,0xbd5fcee8)}},
   {{LL(0x34a70bfe,0x1b2b1885),LL(0xa36345c5,0xcfa421f7),LL(0x6f322ae9,0x2f33f4cc),LL(0x4dabb7a0,0xdac0bb75)}, {LL(0x923cea0a,0xfba35536),LL(0x6d9cb50c,0xc16f73e5),LL(0x25812c96,0x23216dc6),LL(0x3d7ab424,0x82945e67)}},
   {{LL(0x0796605d,0x829577b2),LL(0x5026907f,0x47fa0978),LL(0x2d0f68b2,0x99701169),LL(0xbc1e46db,0xa0d24be4)}, {LL(0x2eb2ac98,0xcf409c2e),LL(0x97f3ff5c,0x7b59c3c5),LL(0x81ed7f02,0x2f4576bd),LL(0x10399c22,0xe41339e5)}},
   {{LL(0x2ecce0e6,0x562d7744),LL(0x9a1656c2,0x1afc3869),LL(0x86200621,0x5714820e),LL(0x566da805,0xee36f7b6)}, {LL(0x6e5a2a06,0xe6694104),LL(0x8caabaab,0xd4390b74),LL(0x93b0d142,0x9db20998),LL(0x7926baf3,0xe1811b81)}},
   {{LL(0x08bc1965,0xd578f2ed),LL(0x35f00d5d,0x9a7e31e2),LL(0xc9007327,0x3725b65c),LL(0x29c36f38,0x157cfe95)}, {LL(0x23a521d7,0xb1c3d0f1),LL(0xb8a9ae08,0x3e65fb7c),LL(0x690b8f78,0xed48bcf9),LL(0x90d5dfde,0xe5f46b2c)}},
   {{LL(0x0b6da2b6,0x14aebb35),LL(0x7b65ee55,0x91fef336),LL(0x1a0a004d,0xdb77b57b),LL(0x23aef1f7,0x1c59b628)}, {LL(0x3ec88d18,0xa79c8c89),LL(0x4fde31f1,0x52cca38a),LL(0xcf4e30b6,0xe2f64a94),LL(0x37ff1cbb,0x2b4cdbd7)}},
   {{LL(0x0b566632,0xcb542f68),LL(0x676fae9f,0xedab69a6),LL(0xc45cb6f0,0xc4531e0b),LL(0xb88fe4a5,0xf967ec6e)}, {LL(0x2919785d,0x4ab4e645),LL(0x7a17b20f,0x2dcaefca),LL(0xda7afaa0,0x65c89c05),LL(0x4dafc6a2,0x59ea00e9)}},
   {{LL(0x8eb43733,0xa6362bf8),LL(0x12011803,0xae2dddc1),LL(0x0bb2aaf8,0xbbf516b1),LL(0xd8de21a3,0x9f2627e9)}, {LL(0x43a20b74,0xaf30439a),LL(0x4ce86408,0xac7e07b0),LL(0x7c10566b,0xc54cdff2),LL(0x6279af0a,0xe3ee0622)}},
   {{LL(0xf7770f95,0x57d09708),LL(0x123e020b,0x6f0ba223),LL(0x6cd41559,0x6c123fb9),LL(0x6fb30f58,0xc54f5c65)}, {LL(0xbbf7101c,0x5e168af3),LL(0xce974455,0xf6d6dbdb),LL(0x88313516,0xa001f3b9),LL(0xdfb4ac20,0xe6e4a26d)}},
   {{LL(0x506f7dcd,0x74e7b7fc),LL(0x5d037d69,0x985e8546),LL(0x1ec8d374,0xff00a4da),LL(0x505b4180,0x8c339ae3)}, {LL(0x3a5f71c4,0x78bcd4f2),LL(0x67ac3e9f,0x2fb4d99f),LL(0xee65dad1,0x7dd25aa6),LL(0xb62c34dc,0x2fd63fc2)}},
   {{LL(0xf7700192,0xdee42663),LL(0x2c3248e9,0x9925a206),LL(0x2ea9f052,0x4a55a55d),LL(0x16ac67fe,0xe1d6efcd)}, {LL(0x9bb02415,0x7f82246d),LL(0x72cd7a6c,0x2fadbb9b),LL(0x712004dc,0xe977a037),LL(0xb3c9f4b9,0xe8c449b2)}},
   {{LL(0x861ea138,0xa2cb838a),LL(0x356ae003,0xfcbe219a),LL(0x1838504f,0x15c02496),LL(0x0769d5dc,0x58cef52c)}, {LL(0xb3fef999,0x7e94ff7d),LL(0x04e4fc87,0xf55501e0),LL(0xc05430dc,0xcdb5fd36),LL(0x778c5cd4,0x49872453)}},
   {{LL(0x1b5e7ace,0x4c4855ff),LL(0xb159fe74,0x89fc6309),LL(0x3c9ebbe2,0xaca00404),LL(0x866bf867,0x4c030591)}, {LL(0x9b18a535,0xa7e8f599),LL(0x5c0a0a44,0x9203ebfc),LL(0x463207c9,0xbf1b30cc),LL(0x9d135aeb,0x90b59001)}},
   {{LL(0x794cb3ab,0xedc44d04),LL(0x0ad7be70,0xb3baa475),LL(0x6c09fc91,0xb7d8c7c5),LL(0xf45a5bd6,0x2a362d71)}, {LL(0x8cf3e5a6,0x36e308c3),LL(0x0a649c31,0x4caf2cd1),LL(0xb3c501c7,0xbae328f5),LL(0x83a0eeb3,0x2efeca03)}},
   {{LL(0x7086093a,0xc3a27585),LL(0x6d686d83,0x78e86515),LL(0xedf0def6,0x18cf3ac1),LL(0x5a1d6cf4,0x2f6a56da)}, {LL(0x30084873,0x350c822e),LL(0x65843610,0x82d48087),LL(0xf393ecd1,0xa4e752c1),LL(0xeeb74f25,0xe3034d6d)}},
   {{LL(0xb8b0c5c7,0x1793727c),LL(0x7ec9ce37,0xde561ca6),LL(0x6190f612,0xd9eddc50),LL(0xca89a191,0xb52dc77c)}, {LL(0x4bf1e87a,0x990010b2),LL(0x15b91691,0x073136b2),LL(0x15546011,0x50111261),LL(0x0196cb8d,0x17d48864)}},
   {{LL(0xfd61d824,0x7ec44104),LL(0xf088d3db,0x213550ef),LL(0xacbbb608,0x5e8d321f),LL(0x39312b64,0xc317c1f8)}, {LL(0x27de4329,0x7a4a1cd0),LL(0xf9b135e4,0xbfb33f07),LL(0x59b94480,0xcf82b639),LL(0x70b118e6,0xca62d957)}},
   {{LL(0x2b1d45fb,0x95b2ff03),LL(0x2570686c,0x472dd56c),LL(0xd3d50e32,0x4fbae8a0),LL(0x65796a08,0xa31c65dd)}, {LL(0x037ce5bb,0xe261f6f8),LL(0xd042073b,0x3b7816bf),LL(0xbfba45f8,0x6d0ebbee),LL(0xc9522e36,0xf2d99979)}},
   {{LL(0x77cb5b0c,0x707f2a18),LL(0xdfc02b82,0x954b5a91),LL(0xc20ae04b,0x246b9a55),LL(0x9dd79f93,0xa1486775)}, {LL(0xc11f6d8e,0xd4092830),LL(0x267a4dab,0x74ca482f),LL(0x9c58706f,0xe3c80bb6),LL(0x099154c1,0x245f04b7)}},
   {{LL(0xf149259f,0x3a4b25b5),LL(0x65ccbe91,0xeac735f8),LL(0x572788a4,0x260e749f),LL(0xe34d40cb,0x30b9c736)}, {LL(0xf524a17f,0x65981d50),LL(0xcddbbefc,0x6c462f5d),LL(0xa1e57312,0x245bfa18),LL(0x46dc8ae0,0x3b4b003c)}},
   {{LL(0x5d262a35,0xb1958797),LL(0xffafd8c5,0x83f6e604),LL(0xbc2e0801,0x60843f9c),LL(0xc783ad3d,0x11d85ac1)}, {LL(0x2e016e43,0x1ce477dd),LL(0xfb4a0201,0x2b628f06),LL(0xbf4f77d7,0x897b7f62),LL(0x10277d8a,0x52e04f22)}},
   {{LL(0x5f3f0d6a,0x17132351),LL(0x59a96c4d,0x13c9e064),LL(0x86f05ae8,0xc73892b0),LL(0x4212ad65,0x94545c8a)}, {LL(0x3dc4984c,0x0591b091),LL(0xf2ec1ca9,0x06845698),LL(0xb3ac894b,0xb0e1e1d0),LL(0xa7c915cd,0x962ca1da)}},
   {{LL(0x95331bd5,0xb0640de8),LL(0x478c1b6d,0x2544348a),LL(0x5647a67e,0x3c3bd415),LL(0x5b20e5fb,0xd7970ef8)}, {LL(0xe06b4fa6,0xd6e6f6be),LL(0x871390ff,0x5ae29e5e),LL(0x7256daa1,0xc7924188),LL(0x59f61750,0xfae5e501)}},
   {{LL(0xd1ef1d2b,0xfac83ece),LL(0x554736da,0xa567060c),LL(0x1dba8bc7,0x697571f4),LL(0x553fbcfc,0xd3fc5aeb)}, {LL(0x9755fab0,0xe665970a),LL(0xb5537da8,0x30fbe8d9),LL(0x97c2b5f0,0x7a7d0013),LL(0x1b700a02,0x9fea5c9c)}},
   {{LL(0xe9a377da,0xcfc0166e),LL(0xac502375,0xcc78f3d8),LL(0xba64c3b7,0x803fbbda),LL(0x4d70cc42,0xe53c7d6b)}, {LL(0x5189b7da,0x6b927bba),LL(0x8b05322a,0x2c86253b),LL(0xf3869873,0x333e7491),LL(0x4b492879,0x9308348a)}},
   {{LL(0xb9ab0a36,0x39bfa2a8),LL(0x18f71ac7,0x560f80a6),LL(0x45e24412,0xca9b265a),LL(0x8e2ddac3,0x6796bece)}, {LL(0x17bfcabb,0x87f1eee5),LL(0x195c9bb2,0x624db4d9),LL(0x2b4db6d2,0xf7110fcf),LL(0xb432d19d,0x41d3fb0d)}},
   {{LL(0x73554a3c,0x3344ea7d),LL(0x830a3881,0x4c968dad),LL(0x687f71ec,0x5df71ad2),LL(0x259cbc07,0x4c4df41f)}, {LL(0xeb541d72,0x8d12d2e0),LL(0xa20fb162,0x94c0dab6),LL(0x1eda0516,0x9bbc2524),LL(0xdd7871ff,0x696c924e)}},
   {{LL(0x1db84dc1,0x97efb495),LL(0x03cbfbf8,0x7d293ce5),LL(0xbc48d007,0x79e25d3e),LL(0x8591a1ea,0xc900a580)}, {LL(0xd37508c3,0xf0277a09),LL(0xe84557bf,0xbf583aa4),LL(0xd8228591,0x2e258d60),LL(0x117da3a9,0xb037e17c)}},
   {{LL(0x243d588d,0x4b35355e),LL(0xcce2539e,0xbe6dfa36),LL(0x4843c9da,0xa57d5823),LL(0xf59348fa,0xe3d91511)}, {LL(0x2791c08f,0xb5d1395c),LL(0xf6fdcc93,0x04129e5d),LL(0x0f53087b,0x635a63ba),LL(0xf237612e,0x66da6bec)}},
   {{LL(0x22755420,0xc3d052e5),LL(0xd7a1bd35,0xc37a9b47),LL(0x9b347e02,0xf19613f3),LL(0xbbda7ae0,0xee84dbac)}, {LL(0x3a85f2e5,0x603be21d),LL(0xff679451,0x5f0927c2),LL(0x8674f8d7,0x799013ad),LL(0x00f465e5,0x17b248d3)}},
   {{LL(0x96ca19de,0x2a29135f),LL(0x957d1844,0xc8e56e32),LL(0xa11a4349,0x935e7eaf),LL(0x741b73d3,0x717308e1)}, {LL(0x7233a9df,0x40477acb),LL(0xd2c83b72,0x7a78dac2),LL(0x2c5d79d2,0xfb882461),LL(0x76f44fa0,0x984505fb)}},
   {{LL(0xdfdc4a9d,0x5cdded16),LL(0x3f0ff597,0x4cbea135),LL(0x8a79078e,0x38daf27a),LL(0xce1bbf0e,0xb4b0085d)}, {LL(0x86f19fd0,0xb6b0d8d7),LL(0x1778ca6a,0xe0fdcdae),LL(0x0b26b9b5,0x257c7df9),LL(0x141dcafc,0x4b82422c)}},
   {{LL(0x4d3cf148,0xcf8a2dad),LL(0x5f17e914,0xf1a4e292),LL(0x60de8f64,0xc40755bb),LL(0x8718f49d,0x412449f8)}, {LL(0x8737b6cb,0xdabb9968),LL(0x6236ea05,0xdd94ae81),LL(0x05c5aca2,0xb5223cd0),LL(0x762210ed,0x6b81bd33)}},
   {{LL(0x5d4164df,0x1f0921db),LL(0x8d4a35df,0xf6fdb08f),LL(0xc602d4d8,0x1efcf3c7),LL(0x057f3aa0,0xa2ecd9e6)}, {LL(0xeb4fcba2,0x13a6c576),LL(0x13130559,0x16425bd4),LL(0x416b4968,0xa9eac848),LL(0x2119600e,0x617c32a9)}},
   {{LL(0x0bb49e40,0x1a84eca5),LL(0xbc2310b3,0x2ed98d25),LL(0x5edbc719,0xad191f88),LL(0x0376ae08,0xd8d667d5)}, {LL(0xf0b4fe29,0xb855a8ee),LL(0xe75354f7,0xc3fe79fb),LL(0x403b651e,0x1ee9b9e6),LL(0x2baa2c6e,0x99ddbb3c)}},
   {{LL(0xeccce37d,0xc6a84c47),LL(0x038c9821,0x71a05a24),LL(0x9a6353d8,0x8d32194c),LL(0xcf0a1462,0x14cd3ea6)}, {LL(0x7bdbe521,0x40d70aa2),LL(0x95c80cd8,0x200f0b21),LL(0x3efdf656,0x4c79dab9),LL(0xa981d8b5,0xafa44e4c)}},
   {{LL(0xa7111315,0x811b9387),LL(0x7590c35d,0x0255a234),LL(0xf1af875c,0xb18e87c0),LL(0xced5cc1f,0x0a930b41)}, {LL(0x96094a55,0x6ff4fca4),LL(0x6a9dc519,0x74095b88),LL(0xafa4894a,0x44492273),LL(0xa2e6f56e,0x54f16f88)}},
   {{LL(0x34485e31,0xd613fbb4),LL(0xd2464242,0xc716c370),LL(0x1644f2e1,0x21535837),LL(0xbe417c3a,0x7719474b)}, {LL(0x2045d2be,0x31bfb158),LL(0xf50e6828,0x10855524),LL(0x98a67af1,0xdb9490ad),LL(0x1c281ff3,0x41a34aa6)}},
   {{LL(0xa8bf2580,0x87109ba8),LL(0x2d7eb52d,0x70c2e936),LL(0xfb3fc109,0xefe9fe2c),LL(0x780526bf,0xfd3f4d7b)}, {LL(0x9ed0c3bc,0x6f9a48d8),LL(0x5d8205b2,0x0aec850f),LL(0x1c6a13ef,0xa378f8c6),LL(0x9d10e11b,0xac02f367)}},
   {{LL(0x3b9bbf54,0x79c6b396),LL(0x42779c58,0xfb586d71),LL(0x889eecb3,0x5d975728),LL(0x434537d8,0xda2ec867)}, {LL(0x62f31813,0x15a3c9c3),LL(0x3c30433e,0xc4b357c8),LL(0xc464e972,0xf26d281f),LL(0x4512ffcf,0x99fa49e7)}},
   {{LL(0x725b9753,0x456db1b2),LL(0xb42941c5,0xec501760),LL(0x7d6d406f,0xd822a9d5),LL(0x7bbcd4d6,0x4bb7a820)}, {LL(0xcc96a5b7,0x079b1fe0),LL(0x24aa4901,0xf83e5755),LL(0x20da7fcb,0x317cdd1d),LL(0x93b04a81,0x487fd706)}},
   {{LL(0xe43332ef,0x43e0671f),LL(0x441c2218,0x71c5dd5b),LL(0xe922ba18,0x4c1d2c1f),LL(0xd619cb67,0x558e9c2f)}, {LL(0x1ec51255,0xd04acde0),LL(0xaf824507,0x824b3740),LL(0x744c6afe,0x62d1b9de),LL(0xab0d52e3,0xb99616db)}},
},
/* digit=19 base_pwr=2^133 */
{
   {{LL(0x7f6a1cda,0x5ec9c084),LL(0x823d6350,0x68839c14),LL(0x03bad007,0xcbbb678b),LL(0x4788854e,0x6a727255)}, {LL(0xef5c7294,0xc747fea2),LL(0x4875e775,0x74852778),LL(0xaa61a893,0xad7b8e8b),LL(0x40da98b1,0x18ff3335)}},
   {{LL(0x5529ec80,0xa51e9f4f),LL(0x6fd146d1,0x0420274a),LL(0x8e300c2c,0xbbf1ab66),LL(0x41653fea,0x2d0b3a9d)}, {LL(0x23a495b9,0x2be2180f),LL(0x5415d73b,0x6ef3c374),LL(0xc67ae4fc,0x1d3e1ec8),LL(0x98d31f5f,0xa5839e9c)}},
   {{LL(0x37d77c01,0xf54114d6),LL(0x41023c87,0xc2e18a4b),LL(0x9e6e1221,0x6fa6c3d3),LL(0x410e48f9,0x9a6cf4e2)}, {LL(0xb181828f,0xe0881140),LL(0x78cb7833,0x17c6df29),LL(0xa7cd2367,0xc1eb8df1),LL(0xca89f922,0xb78f1c8d)}},
   {{LL(0xd0d42887,0xf25d4777),LL(0x2b7a2707,0x4b489218),LL(0x2d3966fe,0x1b4dbf9b),LL(0x41ae2bec,0x4bac5f48)}, {LL(0x1733964e,0x68db2733),LL(0x6a814a69,0xa10c5dff),LL(0xa9898348,0x84ebdaf0),LL(0xa74da3f4,0x60e46823)}},
   {{LL(0x93420649,0x452b6b1d),LL(0x6ed5d7f6,0x9dd6452b),LL(0xe687b577,0x4a9b8fa1),LL(0x854c49d7,0x1e203166)}, {LL(0xa45feba8,0xf523667e),LL(0x5f9f4a56,0x9ecb4d44),LL(0x7fb1c641,0xb8655a5f),LL(0x87c26201,0x5516401a)}},
   {{LL(0x0d2face6,0x24677754),LL(0xa8ade59c,0xd9f7da7f),LL(0x7fa7df06,0x27e3ad77),LL(0xf60395ad,0x35a4caf0)}, {LL(0xe4e701ac,0xfaef231c),LL(0x23755489,0x9e135976),LL(0x43554ad3,0x7caa73ab),LL(0x94f0d878,0x9d8554d9)}},
   {{LL(0xa85b81d5,0xe42040ce),LL(0x40fa9631,0x4d28aca7),LL(0x7e04b755,0x076fed3d),LL(0x1129ce4c,0xdde3d347)}, {LL(0x1179af95,0x77f785d7),LL(0xf74e0672,0x4782f842),LL(0x0b4597cb,0xbd068cc1),LL(0x8f4c65b7,0x3d6d4b2a)}},
   {{LL(0xf9066d73,0xe0642d18),LL(0xa098b3bf,0xbe1d2ec3),LL(0x21b4954c,0xefee860c),LL(0x27b629bb,0x4d7c4e6d)}, {LL(0x8e8b81b0,0xcd8f1e03),LL(0x7fe77eb0,0x4a80168e),LL(0xce247c73,0x4d977591),LL(0x857e0356,0x9b30c9f2)}},
   {{LL(0x2940e9de,0xc02495ba),LL(0xb6d2b72c,0x357299f5),LL(0x06a9c2e4,0x132b4c63),LL(0x084d8c67,0xe90a90c5)}, {LL(0xace1b471,0x0f0c9e94),LL(0xf1e3d8f6,0x769457e1),LL(0xd71118c6,0xc4c30ce3),LL(0x6b652a3d,0xdb5fd8d6)}},
   {{LL(0x4def5978,0x090df107),LL(0x2d8a5f3a,0x1abcfa32),LL(0xa34b70db,0x2976b012),LL(0xfa5e75b9,0x90f541d4)}, {LL(0x37a6e9a0,0x50c991a9),LL(0x903bffda,0xf51e8693),LL(0x8d344776,0xa2697ab4),LL(0xe34a7850,0x77134fe8)}},
   {{LL(0xa72597ac,0x723e5d3d),LL(0x4269aff7,0x4a49847a),LL(0x443b8db6,0x75ad9088),LL(0xa51d80a1,0x9b7d00d5)}, {LL(0xe5e04ac2,0xce1c7049),LL(0x2a792bde,0xb8c2793c),LL(0xe410e175,0xde9220a0),LL(0x9401bc2a,0x4b3a9b85)}},
   {{LL(0xf037d15f,0xc7eaf2c5),LL(0xc7afbf8b,0x410b627e),LL(0xd7bedf50,0x243cdb79),LL(0xbe6512d0,0x04813b51)}, {LL(0x26beca2f,0x2fb77cab),LL(0x7baa3099,0xbb601975),LL(0x40bda4d0,0x8c327e59),LL(0x13c23444,0x85b9c764)}},
   {{LL(0x08ed59d8,0x26960d9c),LL(0x4a72854d,0x9b76dced),LL(0xfdc3b7f5,0xca2f579a),LL(0x6cae8b4f,0xac27028a)}, {LL(0x42326aa5,0x48fd1a49),LL(0x5759c63f,0xb95fdb4f),LL(0xe0a96abf,0x27655358),LL(0x36ed53b0,0x26d38b64)}},
   {{LL(0xfc6d1f3e,0x03cfdd49),LL(0x15adaba0,0x20af5886),LL(0x754dd268,0x74c6c943),LL(0x7977717e,0xe7d52cdf)}, {LL(0x3b414dd2,0x9a81d440),LL(0xd790a4c7,0x697c7b4a),LL(0xedbce1f2,0xb1b7735f),LL(0xbefa7210,0xbd90e63f)}},
   {{LL(0x7ab207d1,0x2e2b0dad),LL(0x9b373211,0x89abbd83),LL(0x8e35e2bb,0x45d34ebc),LL(0x064856f6,0x67ba3ac5)}, {LL(0xa52c7676,0xb5527dbe),LL(0x71294012,0x906fb217),LL(0xab305260,0x65fca552),LL(0x14ee193b,0x89ac52a3)}},
   {{LL(0x88c06b1c,0x673aead4),LL(0x49d9d4e8,0xea8af420),LL(0xcb9e86bf,0xa7b4409a),LL(0x5414aa56,0x49f76f71)}, {LL(0x8c13857a,0x6603c801),LL(0xce742384,0x7c26f1c2),LL(0x2a665719,0x042fb224),LL(0xe175b0c6,0x2619f254)}},
   {{LL(0x7c092397,0x5b3b71ea),LL(0xf18c29ae,0xd9087023),LL(0x2008841d,0x48dbecbd),LL(0x22622bba,0x658c998e)}, {LL(0x578e463f,0x38a2cc6d),LL(0xcbb88796,0x7002380f),LL(0x71695101,0xc545baff),LL(0xce65b49c,0x064a0500)}},
   {{LL(0xb1ae0398,0x3651d926),LL(0x4ace0e86,0x33c9ea8f),LL(0x1a6debd7,0x481fab1b),LL(0x4d365031,0x65b58a79)}, {LL(0x811e8f01,0xb73ec84b),LL(0x51342ef2,0xb6aa3955),LL(0x9efcdbcc,0xdbce3d9f),LL(0xcfbf2a4f,0x5791b35f)}},
   {{LL(0x6eaad1f0,0x67024158),LL(0x0063ae25,0xe8dbaa88),LL(0x9fedc144,0x6d2051cc),LL(0x18b5e86d,0x136c2ab1)}, {LL(0xc89241d4,0x3b2d3d63),LL(0x4a82dec6,0x843cfa3d),LL(0xf0a5f163,0x64fa5860),LL(0x1ae3be83,0x2d9b6095)}},
   {{LL(0xb01a91e5,0x75f97753),LL(0xcd0d8cac,0xd374dfa2),LL(0x8eb72ba0,0xe5dbffef),LL(0xd7b8a624,0x61049807)}, {LL(0xa39277d3,0x9c8b5e93),LL(0x3b1cc635,0x6e5ba593),LL(0x21cde059,0x8bd0a69e),LL(0x071ec0c8,0xd0a19b53)}},
   {{LL(0xd1bb088d,0x8c87785a),LL(0x7e249c39,0xd801d5a6),LL(0x8688914f,0x002ee598),LL(0x6b68413d,0x52b014fc)}, {LL(0x507946df,0xaf1d7e88),LL(0x84ccebf1,0xa38e436f),LL(0xaa86a4b6,0x37d9b946),LL(0xc506a394,0x55da0db6)}},
   {{LL(0x02b900bd,0x856928c3),LL(0x7bc6a67b,0x9eb926a3),LL(0xd0f39446,0x2f4d392d),LL(0x01c49daa,0xb12f2761)}, {LL(0x13874ac7,0x07b8d23f),LL(0x1efaa157,0xa473ef4c),LL(0xdf8cf2ab,0x550765f6),LL(0xd23d3750,0xeba88504)}},
   {{LL(0x2434fa2e,0xf05791d4),LL(0x4e2a05ea,0x8c0899c3),LL(0x898bc9b0,0x40a53bdd),LL(0x40c8bf7c,0x6c255f6f)}, {LL(0xe164b910,0x203db8c5),LL(0xc1c4de69,0x070baaee),LL(0x5df5c0a7,0x89660629),LL(0xdb364b99,0x0b9c2f4b)}},
   {{LL(0x44bb5a79,0x012c6994),LL(0x9bd1fdc0,0xf5928e0c),LL(0x3ce49191,0xd30b8a97),LL(0xe3a05dd3,0x52792b85)}, {LL(0x1d3d69c3,0x0da08916),LL(0xed59a28d,0x931759e8),LL(0x6ca05485,0x412148d9),LL(0x3d6e9964,0xb1517aa0)}},
   {{LL(0xde75812d,0x15204ba9),LL(0x5698b03f,0x49e377e0),LL(0x05c9072e,0xe7790d41),LL(0xdba28e80,0xf79adbed)}, {LL(0x4644840d,0x6aad9f96),LL(0x2e0a695b,0xc3f3d032),LL(0xaa4aa737,0x3eb739d2),LL(0x37d8d520,0x45c6b665)}},
   {{LL(0x9917cb85,0xc3ba2408),LL(0xd7bf6304,0x1c729ffb),LL(0xcc160245,0x56b9935e),LL(0xe03cb227,0x42379567)}, {LL(0xb66bfc5d,0x2dc20028),LL(0x95de8ed3,0xfaf7d224),LL(0x3214024e,0xa7541158),LL(0x50aabdb6,0x2f7755d8)}},
   {{LL(0x7ea9b93a,0xb74ac27b),LL(0xa2e0516c,0xc1c5a8fe),LL(0x6b64f56f,0xe9f4f222),LL(0x8fbc4a64,0xf3c0c7fb)}, {LL(0xa16edc23,0x43ac0ac2),LL(0x6d086e9e,0x0e26e4ad),LL(0x5bc0961f,0x5b8ef949),LL(0xd2b77c29,0xa0d16d39)}},
   {{LL(0x78845d09,0x50b43efa),LL(0xcb3acdd9,0x3899e1be),LL(0x18d4ec31,0xa93a28e3),LL(0x0a66fe47,0x18a4eeed)}, {LL(0x87333831,0xd7a7bf46),LL(0xdbe14699,0xbbf5c1a8),LL(0x80b9c9d0,0xf2a3da73),LL(0x82bceb4e,0x133c138a)}},
   {{LL(0x335a923a,0xcfd4b885),LL(0x8fc82f3b,0xf9b69b3f),LL(0x8784c35c,0x08908b60),LL(0xd843b66e,0x76bf1082)}, {LL(0xbb57a641,0x1ba730bf),LL(0x34e9f095,0x3bb4a8d7),LL(0xc28d5414,0x0342d32b),LL(0xcfd99e1a,0x8fb13cbf)}},
   {{LL(0x1d02f47c,0x3845e507),LL(0x14ef0b26,0x4d77af89),LL(0x5ef578d9,0x93454480),LL(0xbdc408ec,0x23138c57)}, {LL(0x47cf528a,0xdac833ed),LL(0x29d7cf20,0xd18e9865),LL(0xcdc8e55a,0x93208743),LL(0x724025a0,0xbfe570c8)}},
   {{LL(0x3aee838e,0xb75c3de0),LL(0xe0f21f23,0x29304886),LL(0x82791daf,0xe46792ab),LL(0x3f124394,0x3d798d92)}, {LL(0x29a6fb5e,0x2446dc81),LL(0xbd68c23a,0x2446e5b3),LL(0x689b1176,0xe1b5c76d),LL(0x9a852082,0x3fb66661)}},
   {{LL(0xd9b45206,0x8d6fbcc7),LL(0xeabc4640,0x00ab735d),LL(0x810e37d1,0x428c7017),LL(0x27af5718,0xa4365872)}, {LL(0x0a910146,0x8f195823),LL(0x0ff76704,0xc13ccdd7),LL(0x44d6f1c8,0x59d34ad6),LL(0x795b61b4,0xd3dfa6b2)}},
   {{LL(0x12eea439,0x1ec08010),LL(0x7b2cd52a,0xafbbea32),LL(0x68cfe98b,0x99428f9a),LL(0x95628fe7,0x4ff9a5bc)}, {LL(0x7ac41e9a,0x212baeb7),LL(0x29206e86,0x595cf03f),LL(0x733f37c4,0x4b62a429),LL(0x4d3cb6a6,0xa1fac4ae)}},
   {{LL(0x1aed3c45,0x2d6cb0e6),LL(0x4e6da48d,0xf6703493),LL(0x2d13f9c1,0xa0036fb4),LL(0x7fe3ea2e,0x7db5078a)}, {LL(0xd5992163,0x152a1fc0),LL(0x744b44ff,0xd63270e9),LL(0xf177c075,0x56730292),LL(0x17c3e08c,0x470f5e72)}},
   {{LL(0xecb927f5,0xbf53d223),LL(0x629e8aa1,0xc80fbc1b),LL(0x24d72477,0xed59f186),LL(0x38811583,0xc266f5a6)}, {LL(0x7c404560,0xc6f37bc1),LL(0x0c5b68e9,0xd58c10e5),LL(0x916e8f3c,0x696de793),LL(0x56a7781f,0x7298af8e)}},
   {{LL(0xb16679d5,0xaf063553),LL(0x4316ed7e,0xa509f449),LL(0xb53cc0e2,0xe3d6ec43),LL(0x16ba34cd,0x9e957ce0)}, {LL(0x7857d80d,0x2b0c7fbc),LL(0x3daffbf3,0xc2c671fe),LL(0x0d251d41,0xebcbf012),LL(0xffef45f5,0xedcfe7f7)}},
   {{LL(0x334a1734,0xf5b66555),LL(0xe505f4bb,0x4354ccfa),LL(0x52a59260,0x6ee0b5b9),LL(0x5a699a93,0xb7bb64c1)}, {LL(0x6de84422,0x85e34c0e),LL(0x8bbe0560,0xca9bacfe),LL(0x952a52d2,0xa08c780f),LL(0x3919176b,0x0e794b05)}},
   {{LL(0x154d282d,0x8a496598),LL(0xdc34508c,0xb2999dc4),LL(0x9db4410a,0xfc304fe3),LL(0xe1bc07c8,0xbc09aee4)}, {LL(0xef6d497d,0x1d2f0147),LL(0x96488fc1,0x3b9e06e0),LL(0x34cb97a7,0x37635d04),LL(0x8757f955,0x9a294b89)}},
   {{LL(0x59508819,0x38c568ac),LL(0x46e15b82,0x854370fc),LL(0xee57f0b4,0x9f676404),LL(0x8f45319c,0x268854cc)}, {LL(0x63746274,0x4256d25c),LL(0x0496cf9c,0x0a553821),LL(0x15e2fc17,0xb6bf27de),LL(0x99bd538a,0x6848f83a)}},
   {{LL(0x1685e460,0x00e15d0a),LL(0x155d00b6,0x6fae8b37),LL(0xdc561456,0x277126d8),LL(0x6bf70c63,0x331c02e5)}, {LL(0x515f39b7,0xc9b7da4e),LL(0x966c2060,0xb7e0d135),LL(0xc401f926,0x9a801457),LL(0xffb0137e,0xcc560825)}},
   {{LL(0x5c7e38fc,0xbcfac5f8),LL(0x174e97ba,0xd542c1a4),LL(0x0bb507b8,0xbea67b1e),LL(0x3b782fd8,0xf008cc2c)}, {LL(0x0aa329bc,0x865834da),LL(0x2b6db70a,0x0fd746f2),LL(0x65fbe439,0x8e72e5f7),LL(0x005295ee,0xac23881d)}},
   {{LL(0xad9d013c,0xc2c45fef),LL(0x71c311f9,0x0df74277),LL(0x6bb32b66,0x69caf967),LL(0xb8e4a3e5,0x9fbd32ff)}, {LL(0x78c0c439,0x39d94e31),LL(0xffa4b625,0x7489a8f0),LL(0x8aac717c,0x59af0ec3),LL(0xa12d996f,0xdd3b470e)}},
   {{LL(0x8da3fef0,0x6d60cb97),LL(0x044d64fc,0x5164d722),LL(0xfc21305b,0xefe06ead),LL(0xceed89c1,0x72b4c45e)}, {LL(0x8cabf0df,0x072cf1dc),LL(0xa5371d3e,0x0a0d7c0c),LL(0x2ae831d5,0xb13ba707),LL(0x269f189e,0x7702c3c5)}},
   {{LL(0xc8239fe7,0xfb8e903e),LL(0x524f213c,0x5805c2ef),LL(0x70645f7f,0xdf056e45),LL(0x454c4577,0xfe10ecfb)}, {LL(0x990dc567,0x422126da),LL(0xbf55cd81,0x95a5d753),LL(0x8c2688ed,0x2705a00c),LL(0x2f8f91af,0xd079ecb4)}},
   {{LL(0x2b69a2c8,0x8cd13fa0),LL(0x36b657b8,0x7b0f310a),LL(0x251c595b,0xa7247cfd),LL(0x5a36e4b1,0xda352dc8)}, {LL(0xf43312de,0x588d2e88),LL(0xdb9f6dec,0xef80a48f),LL(0x3fb2d6e3,0x39583634),LL(0x5a46bc46,0x0fbfa769)}},
   {{LL(0xfe701598,0x3570a3f2),LL(0xac815fbb,0xd1d0d091),LL(0xd7f2b1b2,0x4d7bfadd),LL(0x66496326,0x509298d4)}, {LL(0xcad9fb50,0xb7accafc),LL(0x9c592dee,0xcdbcb762),LL(0x6888482a,0xfe47a3b1),LL(0xe8b8c133,0x312be210)}},
   {{LL(0x00167f93,0xc474b07f),LL(0xa637f35e,0x19457000),LL(0x5005d8a1,0x3eafa14e),LL(0xadf25f29,0x2a84723a)}, {LL(0xa741cf9e,0x2c9d7ebb),LL(0xc3913acf,0x94024dc2),LL(0x97b98f1f,0xac2db91d),LL(0x46a7bf92,0xfb9a0502)}},
   {{LL(0x6487a5d4,0x8874ffb5),LL(0x2f53e25f,0xc02a12b5),LL(0x416ba8fc,0x38654a57),LL(0x0c0b25d6,0x226356f2)}, {LL(0x6030f2ac,0x34f2eaa6),LL(0x9cea9176,0xb788baa1),LL(0x4e912104,0x66fbe9f7),LL(0x39a69e3d,0x982ef71d)}},
   {{LL(0xbbe5733a,0x9f361d17),LL(0x1988f31e,0xc79569a0),LL(0x9e0f52fe,0xf2b96ecb),LL(0x80235136,0xc78e44dc)}, {LL(0x8462ef4f,0x96053ab5),LL(0x81506701,0xf83c1f6d),LL(0xa65c09e9,0xc7313eb1),LL(0x4efcf558,0xf5dfaa4a)}},
   {{LL(0xe65ede91,0x8b4819e4),LL(0x6dc0a533,0x5a5824ba),LL(0xb4c930f8,0x89d18b20),LL(0xfcefa378,0xaad7a5d8)}, {LL(0x298dba63,0x2ef790c2),LL(0xe90c322f,0x3e4b31b6),LL(0x52ce2ee4,0xa257bb81),LL(0xd39c36bb,0xb8c2966e)}},
   {{LL(0x487719c7,0x13954df8),LL(0x791b00e7,0xcb0f7ae5),LL(0xc8d21faf,0x367a1cad),LL(0x3fbd8a7c,0x44dd204d)}, {LL(0x5f67ec30,0x778fdb56),LL(0x5de5caeb,0xfb288790),LL(0xca53300c,0x310b4d56),LL(0x325c54b1,0x37dbb7c4)}},
   {{LL(0xfe771ef7,0xc80c83a4),LL(0x1c1c1b92,0xe212050f),LL(0xf09c666f,0x0f12bb88),LL(0x10a2eca2,0x8ec5f396)}, {LL(0x90a22eb7,0xdaf96996),LL(0x450de941,0xeb77eee5),LL(0x58fb0165,0x13823c58),LL(0x31272111,0x2157ba6e)}},
   {{LL(0x2b4f9e7e,0x110ee33e),LL(0xf682d48f,0x7e1b550b),LL(0x3e17cb9b,0x8fd8c6c1),LL(0xe1843894,0x91cfbcf7)}, {LL(0x2917b1c7,0x5fc64346),LL(0xba86d14a,0x06f56d0f),LL(0xaf219f21,0xb8874d88),LL(0x11ab8b0b,0xf8803b37)}},
   {{LL(0xbe12841e,0x7e63cf63),LL(0xbc90765a,0x9c9cc421),LL(0x1084fa84,0x0264a597),LL(0x252a9bbe,0xce260a60)}, {LL(0x2fefa4f2,0xfaff225c),LL(0x05bd09b0,0x02b900ad),LL(0x11b1b81c,0x631e5cfb),LL(0x0a193140,0x4d93de46)}},
   {{LL(0xe3173750,0xd92a710a),LL(0x671a3833,0xd712d3a1),LL(0x4116e26b,0xbc9caad1),LL(0xa72fbd71,0xeb24f658)}, {LL(0x9055f802,0x3986a207),LL(0xe2707793,0x212446f8),LL(0x1721b395,0x602541d6),LL(0xb07160c2,0x4099a2e6)}},
   {{LL(0x2369ce91,0x765390f6),LL(0x5754d219,0x2dc37639),LL(0x7c018afb,0xbc552369),LL(0x35bf6b66,0xca835077)}, {LL(0x61d4b0a6,0x61b83e43),LL(0x27cf66c5,0x8f87f597),LL(0x9357cbf2,0xace57840),LL(0xabe47fb7,0x24834481)}},
   {{LL(0xdb3c6e47,0xa434c950),LL(0xaa1da775,0x1f479519),LL(0xf14f9d5e,0x338c9cd2),LL(0x1e75f72e,0x4666ce7e)}, {LL(0xe56564e5,0x4fce4d95),LL(0x89e0ff6f,0x0db55ed5),LL(0x00190b73,0x88796e85),LL(0x454e31d0,0xfdf6492a)}},
   {{LL(0xb9ed6e3b,0x30cb3fbe),LL(0xde8f0544,0x5c796282),LL(0xb6af89bc,0xe11b36bd),LL(0xec439d95,0x0a91cf73)}, {LL(0x0a93fe1c,0xbbe74a5e),LL(0xa5d75083,0xcf1f376f),LL(0xf7725460,0x6718bce5),LL(0xa316d17f,0x6654d7b1)}},
   {{LL(0x0393aa3b,0xdaa925e5),LL(0x9446cdbd,0x81217e18),LL(0xa7afc408,0x07708483),LL(0x44709dfe,0xa4c76c4f)}, {LL(0x3a1c412b,0x72557d71),LL(0xb49b0e1c,0xeb4c2648),LL(0xe4d6c002,0xcdd24b77),LL(0x77113e0d,0x3384ea5b)}},
   {{LL(0x6a10a9ea,0x906fb748),LL(0x0a3b0e89,0x46cda42e),LL(0x7ae4ad43,0x10b9096d),LL(0x3bf2afea,0xe1f23996)}, {LL(0x0dd82d19,0xcb50b941),LL(0x832d93a0,0x008e593c),LL(0xd86a71e0,0x0b1fb0e6),LL(0xb1730860,0x75f2aa6b)}},
   {{LL(0x7efc480a,0xed5d4d7c),LL(0xc76c64de,0x500b9d8c),LL(0xec4fc026,0x28904003),LL(0xdec8b315,0xe41b3f23)}, {LL(0x70c06860,0xa9b5caff),LL(0x28343b2d,0x5cb9a4d1),LL(0x9986a0c3,0xec157abd),LL(0xb5fc67e9,0xbcad3bc6)}},
   {{LL(0x13aa9c17,0x6e64dd26),LL(0x271aef54,0xa347c4a2),LL(0x883d90bf,0x47b26cb9),LL(0xe1c412c9,0xe84d9c6a)}, {LL(0x1c67439f,0xd2eacc10),LL(0xc61b2b5d,0xd7797bb3),LL(0x8ebdb4be,0x0deda652),LL(0xac3fc2f4,0x9e04455d)}},
   {{LL(0x27c86688,0xbbfc6e69),LL(0xa1715a33,0xf7cf2947),LL(0x47bc6409,0xe047a3e3),LL(0x6f2a5b28,0xefeb573a)}, {LL(0xd105ba3b,0xbf3ea3af),LL(0x426c6482,0x5f01b4c2),LL(0x968390b1,0x778a5240),LL(0x72bcf6a6,0xc9c71625)}},
   {{LL(0x6fd8b309,0x698ec2c9),LL(0xa055809f,0x512ea17a),LL(0x8822943c,0x28cb44e7),LL(0xdeb7d3e6,0x434dc709)}, {LL(0x1be76434,0xb8b324d1),LL(0x382ff0b1,0x7cf24ed3),LL(0x2905e726,0xda8265fe),LL(0xee6c3abc,0xd57b3915)}},
},
/* digit=20 base_pwr=2^140 */
{
   {{LL(0xd2a819b5,0xf9010095),LL(0x48f2f653,0x5291aaf9),LL(0xf0afe366,0xfa533907),LL(0x8e279e27,0x88a58ecf)}, {LL(0xfae130bc,0x0f077127),LL(0xf8a54c75,0xee9ccf1a),LL(0xbed82b6a,0x38a6783e),LL(0xed414524,0x9a1acb3d)}},
   {{LL(0xd9c12e2a,0xe4e53cee),LL(0x7fc1308f,0x11983fc1),LL(0x892c2d0f,0x3eb4d84d),LL(0x74499723,0xa0bfc1ca)}, {LL(0x0145176b,0x708344d9),LL(0x6f12e75b,0xbb2988e0),LL(0xada67545,0xdf73cead),LL(0x2bb8f989,0xf37069d1)}},
   {{LL(0x9cc17f65,0xa24a35e6),LL(0x89d9abe0,0xc49b3e9a),LL(0x2fc09ae3,0x82f40303),LL(0x002cc587,0xbffe7d4d)}, {LL(0x424ef713,0x5511f4e6),LL(0xa658f660,0xb86bf654),LL(0x1c8baea2,0x623388d9),LL(0x33656759,0x60664a71)}},
   {{LL(0xd8447e16,0x18996198),LL(0x662171dd,0x17195d76),LL(0xf448b8e6,0x28cfe6a1),LL(0x0658c923,0x8a3c2823)}, {LL(0x9c35e852,0x0c548d89),LL(0x2b378157,0xadf1cd2f),LL(0xf30113b6,0x999e41af),LL(0x9cf4696f,0xf87515a5)}},
   {{LL(0x9778aa8e,0x6c332c55),LL(0xd1b8d8b2,0x290ae3ea),LL(0xbf533883,0x3e2bfa0b),LL(0x1a523ee2,0xe48e3747)}, {LL(0x50fde3ed,0x4d40f1d5),LL(0x48710433,0xb57e695b),LL(0x1b241f3f,0xa4101258),LL(0x3042cabc,0xa0cabf7b)}},
   {{LL(0xedfea522,0x68eb19c7),LL(0xa400db7b,0x68e028b8),LL(0xa8f03a08,0x6cd97bf7),LL(0xf442fe36,0x09f4d266)}, {LL(0x5d713a1f,0x1ac77f91),LL(0x2f58ccb5,0x356e3a35),LL(0xf8ddc47d,0x31d657f1),LL(0x90092115,0xfea7aede)}},
   {{LL(0x4ad49f66,0x2aeba1d2),LL(0x9d40861b,0x16ff2bad),LL(0x7da225f5,0x25464f37),LL(0x1ffc3f0b,0xa2fe66cc)}, {LL(0xac757f41,0x74074d7f),LL(0xcd0a2c20,0x5c85d3d1),LL(0xc974b266,0xccda2a05),LL(0xcc10a04f,0x5c2e2511)}},
   {{LL(0x510d515f,0x01ea2535),LL(0x489e7856,0xc861c54c),LL(0x680d17bc,0x9bc8485b),LL(0x819ccc86,0x71472c11)}, {LL(0x0e9b5d8b,0xa7ef9485),LL(0xd029720d,0x698c9fe8),LL(0x61f50161,0x6ce987d1),LL(0x9d240bf6,0x035f6f32)}},
   {{LL(0x44ec2bed,0xe7c03c9d),LL(0x76cf95c5,0x0bc4f4a2),LL(0x88f014ee,0x0722d57c),LL(0x76fa941b,0xae406348)}, {LL(0x23ee068d,0x046424df),LL(0xe8c130c5,0xd30b6530),LL(0x554f149d,0x17b69098),LL(0x92f95b71,0x887e04f7)}},
   {{LL(0x941c1244,0x414e7371),LL(0x94f1da50,0x1d48fe53),LL(0x6519802a,0xc18bcd89),LL(0x48925019,0xfae7c2d8)}, {LL(0xf2ece2af,0x0f311ddf),LL(0x0a779f79,0x7e8e0e08),LL(0xb6207944,0x47daa5f9),LL(0xefd08d6e,0xf29dc331)}},
   {{LL(0x23e48f60,0x9c096e19),LL(0x8dd36f0c,0xbcc6fe53),LL(0xbb86a9ca,0x452e60f9),LL(0xed16cf06,0xad35f732)}, {LL(0x2bf445f7,0xcbdd01a2),LL(0xf60ce893,0xb7848e94),LL(0x2939a977,0x5e65e8ca),LL(0x63cfa5e4,0x304ebedc)}},
   {{LL(0x252cc655,0x79bae721),LL(0xc4743792,0xa3b9a4e5),LL(0x36fdba1c,0xf32dcfeb),LL(0x7ac19885,0xadbd0c1f)}, {LL(0xdc42a2cd,0xefb4fb68),LL(0x2289a71f,0x78b1ca37),LL(0x87fc6df4,0x7e1f70fe),LL(0x90a9faec,0x8d024301)}},
   {{LL(0x46cd4141,0x37c08672),LL(0x1a60d8e5,0x3c0fed17),LL(0x0f56fea1,0xab18bf06),LL(0x372e757b,0x879ee748)}, {LL(0x1d280206,0x84b19b80),LL(0xd96ac240,0xa40d7ce3),LL(0xfea42ebc,0x5d493fb1),LL(0x40d79bbd,0x9a5fdafd)}},
   {{LL(0x383b371c,0x790c0b30),LL(0x676f8168,0x6dae5df9),LL(0x4c584948,0x101bb4fe),LL(0x55faafeb,0xe3d7e99f)}, {LL(0x134c2e3b,0xd2c9aefa),LL(0x79e27788,0x0aa2a71f),LL(0x7ed0a606,0x4082f7a6),LL(0x6a1be308,0x843c12bb)}},
   {{LL(0x56e9e473,0xae72ee74),LL(0x743e16ee,0xcecde6c1),LL(0x7c48ca04,0x9a06f105),LL(0x5f822a31,0x79179cd2)}, {LL(0xe3530605,0x570d3eeb),LL(0x4c7b03b0,0xbacb30c3),LL(0x0eea0cb4,0x0a8fe254),LL(0x2cdf203a,0xa052a555)}},
   {{LL(0x9c34971b,0xee031587),LL(0xe76545cf,0x5829eb07),LL(0x33a81bb9,0xb7a3a6ae),LL(0x49c9f710,0xff42daff)}, {LL(0xbffb951b,0x894eae85),LL(0xce70f324,0x815fe3e2),LL(0x428b1f12,0x636564cb),LL(0xa029b0bd,0x722e0050)}},
   {{LL(0xd373a65b,0xf45cb816),LL(0x6078d95e,0xf2210e00),LL(0x20d2924a,0xf767d7a6),LL(0x25b66987,0x06d6b552)}, {LL(0x790563a1,0x5c4a3999),LL(0x3c85510c,0xcea00a91),LL(0xd2db6297,0x7e37da9c),LL(0xf67303e8,0xfca4735f)}},
   {{LL(0xaf76f475,0x324ca06e),LL(0x76391adb,0x3367845e),LL(0xa26fe169,0x222aa1ce),LL(0x7ede94c7,0xb15a8665)}, {LL(0x6b6a1f33,0x5b736342),LL(0x8562f392,0x25db61e1),LL(0xf2066206,0xfd4d720d),LL(0x82c555c3,0x26ef773f)}},
   {{LL(0xfde6caa3,0xb6e35b3a),LL(0x87fabf4a,0x34eb5e13),LL(0x86236a62,0x4ad68635),LL(0x28510f8f,0x2651d3e6)}, {LL(0xe0873ba6,0x88073e34),LL(0x22f63746,0x3becce70),LL(0x7c08dac6,0xff8f9b61),LL(0x8c28aa65,0xc8b45a9c)}},
   {{LL(0xd87b59dc,0xe09c063b),LL(0xcbbdd4ec,0xf3e4b9ef),LL(0x4855a43e,0x1b6b1793),LL(0x4ada74ac,0x594d5565)}, {LL(0xb410c5ef,0x10ee400f),LL(0x35695fe9,0xfc118113),LL(0x8f75d723,0x766cfe48),LL(0xff63aa76,0xc72023eb)}},
   {{LL(0x9df9a165,0xc503a858),LL(0x851acc4b,0x9b1099ef),LL(0x66202ca0,0x9246c61a),LL(0x81390ccd,0xaba97788)}, {LL(0xba9e2785,0x3309fa65),LL(0x2220f385,0xbc0388be),LL(0x00ddc8ba,0x94c01a9e),LL(0xbccfdec8,0xaa54aed9)}},
   {{LL(0x059fc0d6,0x1a959c58),LL(0xf518e1c1,0xd0f34c38),LL(0xb53be8fe,0x38aa2b1d),LL(0xacdc872f,0xd95a2a19)}, {LL(0xb4140bd6,0x97bde382),LL(0x4cfd5718,0x4084ba9d),LL(0xfd22450c,0xed016bfa),LL(0xa5d1f5bc,0xf00cdccf)}},
   {{LL(0x111696ea,0x905114cc),LL(0x3a46e782,0x1f58a4d3),LL(0xa5e57fa0,0x899d1856),LL(0x68c45c2f,0x25186954)}, {LL(0xfa6e3eec,0x806deb4a),LL(0x65a063a6,0x3c358d48),LL(0x3feacdcd,0xce28ed1f),LL(0xaaa8e601,0xef9ee31c)}},
   {{LL(0x87c1c66b,0xddd4fe3d),LL(0xb3dbfac5,0xfc2b063e),LL(0x20c81dc5,0x52d37cd0),LL(0x083b5f53,0xb628f163)}, {LL(0x7e42860b,0xd9295094),LL(0x307316ab,0xb877a744),LL(0x6b8d99b6,0xadec0d2d),LL(0x190bc675,0xa75183bd)}},
   {{LL(0x4ad6bd44,0x2b1e0215),LL(0x9e020c66,0xda01ad52),LL(0x0c2913d9,0x65afd73d),LL(0xf0035373,0x67024b45)}, {LL(0x4d308206,0xf501bb4c),LL(0x77e2e936,0xfa020c88),LL(0x936476a3,0x662b72bd),LL(0xbae57d17,0x07f76845)}},
   {{LL(0xf34ca404,0x77a43055),LL(0x4eebc853,0x8e403294),LL(0x402fde89,0xe19ee69a),LL(0xfe00df56,0x9092acd0)}, {LL(0xfb225f92,0x640c035c),LL(0xdce3aa84,0x92d94246),LL(0x971e9886,0x7fe8d3f9),LL(0x014b2a74,0xc569905e)}},
   {{LL(0x7b7c299b,0xbafb8c4d),LL(0xd534cd21,0x3d289c60),LL(0xd311dee4,0x95e7032b),LL(0x6e8892a4,0xac0c46dd)}, {LL(0xe5bd6486,0x9fedef00),LL(0x99f703aa,0x3f4d8daa),LL(0xf0c0ecd2,0x78e47925),LL(0xfdac73de,0x8f143c2b)}},
   {{LL(0xc14bd094,0x1f88f5a9),LL(0x99d9532b,0x6cc19e43),LL(0x639ba66c,0x6e474499),LL(0xf5d06b03,0x5d9a283b)}, {LL(0xaa25dbb5,0xc7e8164f),LL(0xf03faec8,0x7ab42a48),LL(0x647a0d72,0x4135765b),LL(0xe196b571,0x9562a676)}},
   {{LL(0xa720cc20,0x62cc4c05),LL(0x13fa1ad2,0x9ed3f637),LL(0x7f59bac9,0xe5816f51),LL(0xb6884359,0x738e1544)}, {LL(0x34d0fb02,0x83bb2666),LL(0x8014c57b,0x0e582c6b),LL(0x145e2bff,0xbb4069ae),LL(0x4f5f1d7d,0xd1965cdf)}},
   {{LL(0x3cead86d,0xaf77f98b),LL(0x0e51cbd8,0x0ba278bd),LL(0xac2ebb7c,0xf11f20f6),LL(0xc9992b55,0xafd2333f)}, {LL(0x322472b6,0x425dd0e4),LL(0x0958215e,0x0027a74a),LL(0x4cf7e0e4,0xddb301e7),LL(0xcbb70c2e,0xd0656ed5)}},
   {{LL(0x0e7662e8,0xa40f629f),LL(0xe399a5ca,0xdaa85755),LL(0x7297010c,0x4c119aba),LL(0xe5df7140,0x4a4a6a43)}, {LL(0x474f7873,0x6d90d303),LL(0xd1f8f867,0xc5b0e19c),LL(0x4f6dc217,0x188bcae6),LL(0x6777357f,0x51ce999a)}},
   {{LL(0x41aeb39f,0xdfc9578b),LL(0x7dd55c1f,0xeeda86fe),LL(0xfb814075,0xd4b8fc54),LL(0x33a1317c,0x12e32a78)}, {LL(0x2fd217d1,0xeb79cd2b),LL(0xdbd07638,0x5f5f20c0),LL(0x53dc7d8b,0xfc57643a),LL(0xf08350e2,0x65126014)}},
   {{LL(0x871b0d3d,0x737ef5b4),LL(0xae3143a5,0x6b7e04ce),LL(0xb7ae12b9,0x0e5ab52e),LL(0xdb66ee0e,0x1a956daa)}, {LL(0xeaa7042e,0x59657e47),LL(0xbf84a2cf,0xbbc35318),LL(0x78679b8b,0xef55429c),LL(0x60cb7678,0xef92df9d)}},
   {{LL(0x1dd267d6,0x17655580),LL(0xeb0bc1fa,0x00a3ec71),LL(0x50514840,0xafa0a256),LL(0xf161c685,0x68c28d0c)}, {LL(0xb1c766dc,0x069f7862),LL(0xd5ad4568,0x6868a463),LL(0x70e46d7d,0xf9c3d670),LL(0x6c875260,0xd2432cc9)}},
   {{LL(0x088cecd9,0x534c3425),LL(0xb4e34c6c,0x3f1818e6),LL(0x028f153b,0x3aedf0a8),LL(0x50d9433a,0xe0a1cb9d)}, {LL(0xe523b764,0x9b4e225f),LL(0xe5f8542c,0xcba6cba9),LL(0xa8f6b81e,0x59c307e4),LL(0x01bb44fc,0x36abf4b7)}},
   {{LL(0xdd62528c,0xf7692c14),LL(0xdf57773e,0x0d4a8341),LL(0xc9b4f968,0xece6957d),LL(0x52779586,0x82eda200)}, {LL(0x2f06ec6b,0xb902c488),LL(0x91a876f0,0x127dd7ba),LL(0x33ad0c13,0x06eb96d8),LL(0xfc5985ce,0xd7394080)}},
   {{LL(0x661aaa4d,0x624c8f61),LL(0x6717a3e1,0x6fe10a11),LL(0x53168ad0,0x6c288c53),LL(0x8b52d037,0x91b8779b)}, {LL(0x1b5b0ab9,0x89e664d4),LL(0xf30d47d3,0x9f69b44f),LL(0x03176019,0xfe67cad5),LL(0xb346a205,0xb83efd48)}},
   {{LL(0xaeea0c91,0x63fc4863),LL(0xdb56004b,0xbabf9042),LL(0xa9917def,0xdb19f2ee),LL(0x54c3fae1,0x1d12f2dc)}, {LL(0x55e36d40,0x7bb496af),LL(0x6be63b27,0x1f6c11f8),LL(0xcaf9a5b9,0x96d79804),LL(0x0648051c,0x03a085c4)}},
   {{LL(0xb56baf4c,0x3b54c223),LL(0x559c1fc1,0x04af8c4c),LL(0xabd3cebb,0x05d55266),LL(0xf865e262,0xd2d3ae9b)}, {LL(0xedfedc54,0x3bd3ca3a),LL(0x922776c4,0x30a6ff1c),LL(0x1616a6f2,0xfecd8845),LL(0x94948d8c,0x4e7bc2e8)}},
   {{LL(0xedca784b,0x16e0d824),LL(0x67ea1eea,0x84584f98),LL(0x8625626b,0xeceb1418),LL(0xc34fc1f3,0xa487cf9f)}, {LL(0xa57cec36,0x4ecfedd2),LL(0xd24a0709,0x08624865),LL(0x6a48f3ee,0x47bb1909),LL(0xc69bc041,0x54c5dd0c)}},
   {{LL(0x7527166e,0x15a291e6),LL(0x4a9a8315,0x8a92370d),LL(0xda584bd6,0xe9fe705d),LL(0x3625a669,0xed441dc3)}, {LL(0x3063f2de,0xa57929ce),LL(0x6348cc31,0x2809fe4b),LL(0x04cc19c0,0x92041d54),LL(0xb62c1f94,0xd7c227fc)}},
   {{LL(0xcd0d497c,0xecfeee53),LL(0x128818a8,0x8d1ea9fd),LL(0x2ae4725d,0xb5cf2c28),LL(0xc6abad3a,0x7de9f967)}, {LL(0xb14a183c,0xc64a11fc),LL(0xd5777d77,0x7f14d0fe),LL(0x12957444,0xbe79846c),LL(0x0e3257ca,0x4cf23abf)}},
   {{LL(0x6e01b48e,0x8da0fd8d),LL(0x5ee87ca4,0x63a7ff16),LL(0x5cc96b94,0x90dff4d3),LL(0x406fc905,0xff1b41c3)}, {LL(0x3ac71c41,0xdd932925),LL(0xcf65e59d,0xec57f1b8),LL(0x3ce0512b,0xa3116d6f),LL(0xa2e28316,0x3b46fd3c)}},
   {{LL(0x60156a5e,0x5a6c0314),LL(0x55d46fd0,0xfab3afe3),LL(0x9846f0db,0x4617926c),LL(0x121ef237,0xc2d5a447)}, {LL(0xf1cda3b1,0x789498d1),LL(0xccd64aac,0xa195cf03),LL(0x9440be2b,0xe8d1a162),LL(0x3ad5373e,0x7399890a)}},
   {{LL(0x4fbf1899,0x65dcea2f),LL(0x44ee1a5a,0x44d9c8ab),LL(0x406880fb,0x2e94f8c8),LL(0x51faab7d,0x70732bad)}, {LL(0xf1e92b52,0xd69309dd),LL(0x3c7685d0,0x25f9a675),LL(0x1dbfbaa0,0x3604f30b),LL(0x2ff28c22,0x5ac0001b)}},
   {{LL(0x52762d8d,0x648ec4fa),LL(0x0cef95f8,0x4fc248c6),LL(0xe5fbb57d,0xfc0f7030),LL(0x5ff2db51,0x2e644729)}, {LL(0x3775471c,0xd85877ec),LL(0x6076a271,0xe2580058),LL(0x09cb3873,0x58a4a24f),LL(0xb142da8c,0xb412928b)}},
   {{LL(0x5997987a,0x1da7964b),LL(0x1825d97b,0x69765ff0),LL(0x4c97095c,0xde8ae407),LL(0xbb59316c,0xb257a968)}, {LL(0xcf2dfbd2,0x80e5240d),LL(0x1fccd0e7,0x2b3b04b0),LL(0x8ff1093e,0x949f1234),LL(0x65f62273,0xa4df3290)}},
   {{LL(0xd0058ffa,0x305b7479),LL(0x180b0de6,0xe459ecff),LL(0x8ca0585f,0xfbe00c08),LL(0xc3dd4fa0,0xc169e23a)}, {LL(0x44026f6e,0x65d9009a),LL(0x1d96fe58,0xbbc258c3),LL(0x328ed1e0,0xd7ed379c),LL(0x23970884,0xe8b45744)}},
   {{LL(0xed1095b5,0x2f44c8ce),LL(0xc59404aa,0x25725b0d),LL(0xa273e09f,0xea926278),LL(0x529143d5,0x102b120b)}, {LL(0x81174d10,0xbd2c012d),LL(0x4e8333ad,0x0bf5cf89),LL(0xb2f60645,0x6b93e3b0),LL(0xf38df3ce,0x040298b8)}},
   {{LL(0xb89e18fd,0x6433b6fc),LL(0x6bd6af88,0x48d6584a),LL(0x0e1747a5,0x46c8a061),LL(0x1ed79faa,0xe225d3cd)}, {LL(0x5c28a48b,0x6c579abb),LL(0xda3889d6,0xc7a5ff4d),LL(0x008febdf,0x037ebc4b),LL(0xa7c5cbc9,0x03b60593)}},
   {{LL(0xdcaa3319,0x0840804d),LL(0xb7481f45,0x38b0f084),LL(0x112e794e,0xfdb059c9),LL(0xe1afb614,0xb62bce05)}, {LL(0x53be7c02,0xc15035b6),LL(0x153ee8e5,0x66fc7106),LL(0x8258727f,0x27fd2ea4),LL(0xb470105e,0x8e012416)}},
   {{LL(0xd7711aee,0x3d24685b),LL(0x021bab69,0x66a83c3b),LL(0x5501d69d,0x6e1112a7),LL(0x068b0504,0x2219fe0b)}, {LL(0xa926ab1b,0xaaa553c1),LL(0x56d6ca58,0x1c81af95),LL(0x221ef447,0x0a997380),LL(0xf5f55487,0x881a62fa)}},
   {{LL(0x4f1b618a,0xf88fa0bd),LL(0xcbac98e8,0xb20e161c),LL(0x3bc6a7ad,0x443352b5),LL(0x0fd5748a,0xbc8e03ff)}, {LL(0x0ca427fc,0x64006aff),LL(0x7cbbda99,0x1a477593),LL(0x1a347c47,0x21ef1afd),LL(0xdee162b6,0xfe056287)}},
   {{LL(0x797a0b14,0x9d4eb7da),LL(0x951c4bd0,0xe4e01a46),LL(0x7fe354a6,0xaf8fa17f),LL(0xc430b12b,0xd71f160c)}, {LL(0x83d46be0,0x5bb68437),LL(0x619bba86,0x99d10d82),LL(0xf4327042,0x95c2219d),LL(0x9c19ab57,0xdace2322)}},
   {{LL(0xc8750fe7,0x88abbc67),LL(0xd3abe7d2,0x81ab300a),LL(0x45aa8948,0x62a6d8d5),LL(0x5d4ce8c3,0x76175bbd)}, {LL(0x8ea70976,0x095cb181),LL(0xf7e62a19,0x785de3fc),LL(0xed11a7fe,0xc478bce8),LL(0x1528aee2,0xb7e5993b)}},
   {{LL(0x76c32e4b,0xb9ec58d7),LL(0x2dbc9a61,0xef815613),LL(0x5e07410d,0x372c3856),LL(0x033276d0,0xa1b16510)}, {LL(0x82640d26,0xd8589581),LL(0x9053fff0,0x1cb98180),LL(0xc1ff11f1,0x41519ce7),LL(0x666431e4,0x2f21a48f)}},
   {{LL(0xe83ee840,0x2c223ed1),LL(0x1e7cf4dc,0x833ae708),LL(0xacd13385,0xec8853d8),LL(0x6a7a8cb1,0x559115ab)}, {LL(0xeb184e71,0xe2f4ab2a),LL(0xc10194df,0x679abbce),LL(0x3aca0828,0x10199c84),LL(0x978cc1d6,0x7474e113)}},
   {{LL(0x44e8eb9a,0xa4460ef1),LL(0x828aa4bf,0x4cde5260),LL(0x249bff50,0xd3d23790),LL(0x6bc7fbbc,0x2e6660da)}, {LL(0x61494df0,0x3e3cc146),LL(0x0bcda8ba,0x6e9a1571),LL(0x096e00b7,0x68ce233e),LL(0x5106c85b,0x247a5c49)}},
   {{LL(0x55fc5757,0xe6f0cb5c),LL(0xb7585799,0x452682b9),LL(0x869e747e,0x84509dfe),LL(0x8d23be04,0x904366e5)}, {LL(0xb0f72c6d,0x7324a14d),LL(0x1913a9ff,0x9fbe3116),LL(0x428a6b5d,0x2f9fa62a),LL(0xf8a34d9e,0x8039270f)}},
   {{LL(0x407aec78,0x0e3ce7ae),LL(0x826458cd,0x4d935d3d),LL(0xfc6f46d4,0xf75db7db),LL(0x88586874,0x3ab7ba68)}, {LL(0x9a198b50,0xec92749f),LL(0xefc51cdb,0x0ffc7681),LL(0xe17bc0e3,0x951406d5),LL(0xc898a018,0x39cd2d07)}},
   {{LL(0xf2f79af6,0x9dc3803c),LL(0x0a56cd68,0x292f3164),LL(0xf6fbdbdf,0xdcac21f9),LL(0x23e9e959,0x6f9ce2a4)}, {LL(0x970f6c34,0x2011d221),LL(0x9e2decfb,0xd2e63711),LL(0x118ff327,0x19c7a489),LL(0xbb6e534e,0xe19d7e83)}},
   {{LL(0xbd1a426b,0xc685389a),LL(0x8c679952,0x432ff7b0),LL(0xc5e2687f,0x516cbdfa),LL(0xba1eac8f,0x8242405d)}, {LL(0x0b09854e,0x63af3152),LL(0x231ec979,0xcecd0faa),LL(0x7273f0b2,0x4746733f),LL(0x8f001365,0x69b28d87)}},
   {{LL(0xc6f2623c,0x0d87d506),LL(0x86c40ed2,0xd209a9c6),LL(0x0fa20f3b,0xa5b7fde2),LL(0x4f5b2129,0x54550dc5)}, {LL(0xfeddaa1b,0x36884047),LL(0xd899a29f,0x51398fa0),LL(0xcdf11867,0x14a416be),LL(0x3e466b62,0x86351ac6)}},
   {{LL(0x39bb481a,0xd63e91e1),LL(0x99984155,0xdcdc072a),LL(0xd2d8e622,0x9fce6e38),LL(0x8e8c8034,0xbf6978b6)}, {LL(0x8c37990a,0xaa1ae874),LL(0x0e749b86,0xd1f0e068),LL(0xcbdc7c12,0x5aa303b1),LL(0xc9130211,0x9a78baaf)}},
},
/* digit=21 base_pwr=2^147 */
{
   {{LL(0xb3e2087b,0x5eeba910),LL(0x44a61a33,0xbd016dd3),LL(0x48cd5c0f,0xffd1f081),LL(0x2e6a8e94,0x041c6aa0)}, {LL(0xc4ac3d91,0xe09c35c5),LL(0x634767a4,0x58298105),LL(0x1040c2b5,0x6120e7cb),LL(0xa713589f,0x32a661ef)}},
   {{LL(0xbd74c70e,0x5a056a90),LL(0x8af672f3,0x44f7d00d),LL(0xef4e9a48,0xdc25ab68),LL(0xfdfb1069,0xadd15cc3)}, {LL(0x9f3033bf,0xb1f4fd28),LL(0xebb8b8a7,0x088a49bb),LL(0xa8d2861c,0xea485869),LL(0x6b977fb5,0x46dbfdaf)}},
   {{LL(0xd88ae888,0x04e6461e),LL(0x9be2d7ad,0x112d204a),LL(0xdb558427,0x952dc813),LL(0x27e35200,0x39b01652)}, {LL(0xff02cdbb,0x8b89bfce),LL(0x3e854e4c,0x1381a99b),LL(0x999efd2b,0x68185218),LL(0x86dc62e1,0xeb41e1bb)}},
   {{LL(0x02d0aaff,0xa264ef40),LL(0x6e679fe2,0xa678c07d),LL(0xfd88bdce,0xcff13be7),LL(0x617badb8,0x9a8efe8d)}, {LL(0xad5a22f4,0x1388a815),LL(0xfec398b3,0x8f821400),LL(0xff4fc2da,0x85a6a565),LL(0x858dd4f3,0x681f0181)}},
   {{LL(0x91ee75bc,0xdc870745),LL(0x64f20e50,0xbadbf940),LL(0x49d3d417,0xf3ea437f),LL(0xc02109d0,0x7bebd868)}, {LL(0xd16bb240,0xe6c8d93c),LL(0xfdab9bbd,0x2087141a),LL(0x20a3b470,0x8dba80ff),LL(0x29d3a8d7,0x960a0c7b)}},
   {{LL(0xf8ec1151,0xae61b637),LL(0xaadc8545,0x415dd36b),LL(0x69d0985a,0xed21d176),LL(0x97893f50,0xc4d062af)}, {LL(0x337b81f9,0x4d93ba1a),LL(0xb7c163a2,0xb995fe9e),LL(0x5416e4ed,0x447eff3b),LL(0x0bf4a8e7,0xd7660300)}},
   {{LL(0x56d9e00f,0x9e14c6a2),LL(0xfa1f60e0,0xa228491c),LL(0x8782a9be,0xd540713e),LL(0xdcd55e21,0x5d3fcce8)}, {LL(0x35c87b90,0xa176c34e),LL(0xf9508f35,0xc1d80aa1),LL(0x92302d47,0x14f7e7fc),LL(0x2b076e72,0x459372ba)}},
   {{LL(0x4e933b19,0x44168fbc),LL(0xe54ea969,0xaf2db74c),LL(0xaeacbb56,0x36fade13),LL(0x84e6cd1d,0x29708665)}, {LL(0xb692df97,0x6f7ff1e1),LL(0x6ae66307,0x5a68c1a2),LL(0xe7685f20,0x85bc544c),LL(0x0f65eeec,0xb3f42e6d)}},
   {{LL(0x5b91b644,0xef209f44),LL(0x50cb02b6,0x808b930a),LL(0x099f684f,0xc5da5e86),LL(0x4330c2d8,0xd8f1dbcc)}, {LL(0xd8608776,0x52e8cab5),LL(0x16e56f5d,0x13c89771),LL(0xb135282b,0x7d7d802a),LL(0xe9be8a20,0x69762c39)}},
   {{LL(0x2a08a1dc,0x13f6bbad),LL(0x7f2dba7a,0xa7131e4a),LL(0x7864f5a3,0x751dce48),LL(0x08135109,0xc5af1b45)}, {LL(0x4f08636e,0x3c4d75f7),LL(0x2e251e48,0x9949b2f5),LL(0x9bd98853,0xd0497977),LL(0x909b0e84,0x09d8e627)}},
   {{LL(0x4ceff1c9,0x505753ee),LL(0x460710ca,0x03ca4571),LL(0x5480abc0,0x0cf72dee),LL(0xc19c8ef3,0x55d5a30e)}, {LL(0x86541f6f,0x9e47641b),LL(0x10c9d6fb,0x89b2e48f),LL(0x0860915c,0x9d16382b),LL(0xf54b337f,0x770ac417)}},
   {{LL(0x77ef7f67,0x366d0781),LL(0x2b6340c9,0xfefec947),LL(0x097acf63,0x7ce3a056),LL(0xaf306dec,0x26538c7c)}, {LL(0x287dc8d1,0x8a8bb55e),LL(0x448823ae,0x9431095b),LL(0x7970fc51,0x8358087a),LL(0x413509ac,0x95299959)}},
   {{LL(0x8b0911d4,0xb165f92a),LL(0xbfb37459,0xdde855ee),LL(0x3d9ce36e,0xe672eed4),LL(0x74bae261,0xf7da91e5)}, {LL(0xb741c8e6,0x1f67323c),LL(0xc92c91ce,0x8efd4661),LL(0x78e73c42,0x556f0b1a),LL(0x16101f96,0x7d326b7f)}},
   {{LL(0xfb48bd3a,0x8814ef0b),LL(0xc508309e,0x1bbbe13e),LL(0x67709c10,0x7ddaf061),LL(0x6436f655,0x82b67847)}, {LL(0x03712e64,0x2a5601c6),LL(0x3e3f9b2e,0xac1f0362),LL(0x09184b5d,0xcc7e6a09),LL(0xb4625149,0x1258b265)}},
   {{LL(0x384a6b54,0xd9f21461),LL(0x6cfe9311,0xde483161),LL(0x889f43cc,0x593dae45),LL(0xedee221b,0x8454335b)}, {LL(0x3a2cbced,0x90f3fb43),LL(0xcc8dcb75,0x895ed692),LL(0x14233aa7,0x9857d719),LL(0x48166d5f,0x91b1a2ab)}},
   {{LL(0x675b47a0,0xfbf7033a),LL(0x6542378f,0xcb3669c4),LL(0x125ec248,0x96abb0f7),LL(0x795fc346,0x6d5d2047)}, {LL(0x8f5cffb6,0xa6c6c9e8),LL(0xbea5ee09,0xb968f2c7),LL(0x844ffd6f,0x2f2ce735),LL(0x27e40ac8,0x7931b877)}},
   {{LL(0x2b63d538,0xe1f62dcf),LL(0xf44d7bf9,0x395681dd),LL(0x54aec359,0xf02eedf7),LL(0xa0ad5eb7,0xc64b6233)}, {LL(0x346b086a,0xc65093c7),LL(0xe957b243,0xfcf8ecc9),LL(0x1ca48020,0xe1accffa),LL(0x4047bbeb,0xe1f29792)}},
   {{LL(0x9fc6be25,0xb1097d40),LL(0x923eb7b4,0x02d33d19),LL(0x1f58f545,0x9e25200c),LL(0xda51efcb,0x2ffae306)}, {LL(0xc0b011f2,0x7e6d76c1),LL(0xf680676b,0xedbd8398),LL(0x8d7fc8c2,0x38517fc2),LL(0x5c3ab05c,0x55a7fcf9)}},
   {{LL(0x8e6dd02d,0x047e2422),LL(0x1f290d6a,0x7b3bf0e6),LL(0x6a99a6d0,0xbcf326fc),LL(0x2eef8232,0x1e3f38fa)}, {LL(0x15bac529,0x9890780e),LL(0x9f143ba0,0x94202e0e),LL(0x885e4ed5,0xbd574712),LL(0x396f938d,0x2404c223)}},
   {{LL(0xacde8286,0xd54d401c),LL(0xe7af01fd,0x80397794),LL(0xf615a8eb,0x94457d07),LL(0xd22d9ef7,0x34680480)}, {LL(0x2c489ccf,0x04d4b302),LL(0xc3673dae,0x11dea4bd),LL(0x58cdfe41,0x8fbb4df5),LL(0x0f10a70e,0x49425184)}},
   {{LL(0x077a59ce,0x831b977a),LL(0x894627f3,0xee08fb0c),LL(0x2f8553f0,0x21360507),LL(0x0487171b,0xca77ccd1)}, {LL(0x07e11059,0xc17d20c2),LL(0xbe613256,0xcf74be6b),LL(0xa5fe18c1,0x06f185e6),LL(0x2b57ce3e,0x8d2cf4f5)}},
   {{LL(0xc9c983e7,0x7179273c),LL(0x153f58d8,0xc7d27357),LL(0x4f400bd4,0xc0273069),LL(0x26262553,0x23309c7f)}, {LL(0x712d0314,0xf26b6e11),LL(0xf96ee39a,0xb925cebf),LL(0x73944251,0x6df57108),LL(0x589d90aa,0x95419b24)}},
   {{LL(0x796a8ee2,0x57a1bcc5),LL(0x2acee09d,0x22a22530),LL(0x66fa2911,0xa4c2cc03),LL(0xd85f13dc,0x9cc2b7fa)}, {LL(0xce152790,0xf2498b8a),LL(0x1caf39d1,0xd8406007),LL(0x84c0822f,0x7ff50064),LL(0x155f1280,0xaf14ca4b)}},
   {{LL(0x89b781c2,0x113f094b),LL(0x013833a5,0x996bf893),LL(0xc0b9cf6d,0x26bc6210),LL(0x6a88f1cf,0x18e2d3ac)}, {LL(0xa21a2d35,0xc0ff2b3c),LL(0xa79e918e,0x409c2598),LL(0xb6917e22,0xffcf65a0),LL(0xbb4f8f3c,0x8036897f)}},
   {{LL(0x9ec27fd7,0xac660365),LL(0x0c56cbb1,0x3c5ca1a9),LL(0xbe9e9ec7,0x01c5dce1),LL(0x386edb4a,0xdc21b01a)}, {LL(0x4b1dde01,0x47e02a92),LL(0x44af3e0b,0x0613b7ca),LL(0x1c445b6f,0x644ac708),LL(0x87243e2a,0xb5566f0f)}},
   {{LL(0xba9f354a,0x5b244172),LL(0xeb653a5e,0xaca4e9d3),LL(0x514809f2,0x6ff6904a),LL(0x96595230,0xf87a329b)}, {LL(0x8d4bd051,0x39ebe6eb),LL(0x07d17d59,0x66f05f5c),LL(0xe0f81731,0xfa1ee673),LL(0xd12804a9,0xf41c1042)}},
   {{LL(0xacd14cf5,0x1c4a655a),LL(0xef47548f,0xdc72f5bc),LL(0x0b3ee6c7,0xab07ceff),LL(0xbb501a28,0xcfa88319)}, {LL(0xd8f03f7c,0xcec9c2e2),LL(0xe0c98d62,0x3098d752),LL(0x0a8681b3,0xa41a0794),LL(0x23587932,0x0e588076)}},
   {{LL(0x5ef86f7c,0x4617dc66),LL(0xcedb5377,0x51de8430),LL(0x737381b6,0x0dda704a),LL(0x008f4671,0x83a44653)}, {LL(0x38401c11,0x71bbb73e),LL(0x4894d378,0x879fe49c),LL(0xeab186a2,0x8875eef9),LL(0x12a382a9,0xedefe278)}},
   {{LL(0x13b897fd,0x95ef41b8),LL(0x2a98ddd9,0xfefd495f),LL(0x680b12e8,0x09cbccfc),LL(0x167da5dc,0xc1888a21)}, {LL(0x3bb290b4,0x2a204912),LL(0xd9613190,0xdcac95fc),LL(0x709c76f7,0x4df94f62),LL(0xa5cea926,0xc8c3a8ab)}},
   {{LL(0x37a2b813,0x15c876b2),LL(0x9c3c821c,0x9b52803e),LL(0xcb3d6ff1,0x40f2268c),LL(0xa1573601,0x689f1696)}, {LL(0x8e921f56,0x8d7566dd),LL(0xd992335a,0x5d8a990c),LL(0x20dc4f4b,0x6339153a),LL(0xdc5d06ab,0x0b07419c)}},
   {{LL(0x97c201f9,0xe9cc014d),LL(0xa635f472,0xec04a52e),LL(0xa538a84f,0x6aac504a),LL(0x5762fe7c,0x4d0288e3)}, {LL(0x34cbd09a,0xaa8539f0),LL(0x2619bcf7,0x6f7e0e94),LL(0x0dd338d0,0x178303dd),LL(0x8326f40e,0x6b58c2b1)}},
   {{LL(0xfe73e699,0x98bb15ec),LL(0x47526489,0x7533abdc),LL(0x491dcc6d,0x4b269607),LL(0x77187363,0x325ec2a0)}, {LL(0x7e9ab865,0x766faa19),LL(0xc25a9448,0x1c105b4a),LL(0x0531b5ba,0x0b6b8963),LL(0x2db1a579,0x32691f11)}},
   {{LL(0x643e479c,0x24d90a57),LL(0xb98578df,0x048b27cb),LL(0xe53bed53,0x0600f93f),LL(0x0aac585f,0x1fd57dfc)}, {LL(0x71d0e4e0,0xc3d72121),LL(0xf612fc4e,0x5ff10dfb),LL(0xb5a7ec79,0x9edf4b23),LL(0xd87706ab,0x975165c7)}},
   {{LL(0x7095c3c4,0x8b99db04),LL(0x897faf50,0x65196441),LL(0xdd5b64cb,0x5d23d7d9),LL(0xe95fe601,0xec734b06)}, {LL(0x0b5fcde9,0x03a5f53f),LL(0xebe35310,0x0186ad22),LL(0x84846603,0xe9a65eef),LL(0xa7c6e5be,0xe99e5188)}},
   {{LL(0xf0887da6,0xa917327d),LL(0xe3f9fa70,0x49965f78),LL(0x4f10b31d,0x02ed227d),LL(0xb6120be5,0x535b4386)}, {LL(0xcc1bf98a,0xdff21a8a),LL(0xeb1634bc,0x5b52a09a),LL(0xa3f61fa2,0x60f8690d),LL(0xb863c790,0x58a02566)}},
   {{LL(0x5c6b2929,0xf9b90a9e),LL(0x22fca36e,0xd552e84c),LL(0x9eabcb58,0x6b23da4f),LL(0x5d4136dc,0x01111d07)}, {LL(0xb3642a09,0xfaa80059),LL(0x5f49d533,0x1de667f4),LL(0x17525176,0xb3268776),LL(0xda729fde,0x75b0b102)}},
   {{LL(0x6e9fe6ed,0x4ec7f678),LL(0x2717f8b0,0x28d29578),LL(0xd4cc149f,0x6a713c37),LL(0x7dfdf8c6,0x4804e04f)}, {LL(0x5c931aa6,0xe7c6daab),LL(0xa0394f29,0x793e411d),LL(0x79ed9819,0xc0741c0d),LL(0x24d5d992,0x3f2ba70b)}},
   {{LL(0xca9c987a,0xa61dc03f),LL(0xe64b50c1,0x78201cb8),LL(0x1861f4e4,0x45a23c25),LL(0xc4ee5d82,0x10f19f4f)}, {LL(0xf3f055f4,0xf1520547),LL(0x006ccf49,0x69ae26b3),LL(0x33d8d4ad,0xe96eec0b),LL(0x48a4fc2c,0x00765f0c)}},
   {{LL(0xa3976c07,0xad47e14e),LL(0xd905b6b4,0x82b1f882),LL(0x91382bac,0x7a1b9d73),LL(0x18178290,0xcc84a820)}, {LL(0xb4e845ab,0x1123c6f6),LL(0xb92e3b08,0x63216635),LL(0x183879fb,0x748be745),LL(0xa73e9ada,0x7f20e1f0)}},
   {{LL(0x9224c152,0x05de3e11),LL(0xea8fda4e,0x2fa9a474),LL(0xf48055ec,0xf5c8df05),LL(0x9e23a599,0x48bbf43a)}, {LL(0x148086db,0xf593f034),LL(0xef0a2b62,0x0173a87a),LL(0x3fbabb6f,0x90ef0132),LL(0x21ade107,0x56ced09a)}},
   {{LL(0x73f1d3e0,0xcf1ce892),LL(0x22424580,0x765236c6),LL(0xd939d063,0x0d223937),LL(0x7cb2fe2f,0x9a21beda)}, {LL(0x1ce3a7fc,0xa559a571),LL(0x1b060dd5,0x7fd6b255),LL(0xc5afdf1a,0x4dfbd210),LL(0x1239368a,0xa74751ce)}},
   {{LL(0x6d9a3eec,0x93acdd06),LL(0x9832dcfd,0x7d97f794),LL(0x0cc645ca,0xdafa9a44),LL(0xcfee0817,0x1da27ddf)}, {LL(0x01b8dd49,0x0c1e6319),LL(0xd91aeace,0x8267e508),LL(0x87f43f20,0x86a2cedc),LL(0x07db2f24,0x7dd0e670)}},
   {{LL(0x9db25177,0x8ea1e973),LL(0xab8802df,0x659cccb8),LL(0x004940ab,0x2bd39c65),LL(0xd9419db5,0x57676876)}, {LL(0x6d6f237c,0xd52058a3),LL(0xfb4a9a7b,0xd9812dcd),LL(0x53bec56e,0x879890d1),LL(0x7ac5d9d9,0x17b7f151)}},
   {{LL(0x1ec6db6c,0x2db3f5df),LL(0xafdf29b0,0xe9a3b18f),LL(0xe63a9ae9,0xda633d62),LL(0x38d13c8c,0x0922b16d)}, {LL(0x7462c8e6,0xaeb7e270),LL(0xa01b6984,0xa98c96d8),LL(0x297c242f,0x5586e0d3),LL(0xff587596,0xbeddd1ab)}},
   {{LL(0xc02ea084,0x79ac33ce),LL(0x8e02ae2f,0xe7d06753),LL(0x94d526b8,0x05fffd7d),LL(0x5ebc46d0,0x4590d655)}, {LL(0x855f85e6,0xfb79c066),LL(0x7400ed08,0xbb3f0a6d),LL(0x67fb3683,0x46f4c3cd),LL(0xd19804cf,0x62fc1931)}},
   {{LL(0xd1b6f356,0x0480e772),LL(0x56320242,0xa5810f25),LL(0x64073c03,0x6cf6c9c3),LL(0x46a6bfbc,0x7dfe137b)}, {LL(0xba59baf8,0xa5633fa0),LL(0x5fd4929a,0xb703e5db),LL(0xd7515518,0x09eef835),LL(0xa0e3b067,0x2e596aa8)}},
   {{LL(0x8649bb99,0x793831fc),LL(0x5ba4c1b6,0x91cb0057),LL(0x270ec9e8,0x44e93dbd),LL(0xd139d219,0xbf2ed01a)}, {LL(0xc9d68198,0x39697e05),LL(0xde2b6894,0xf04439cf),LL(0x1e6b8e6d,0x65b7a04a),LL(0xce35ae6d,0xce3e9425)}},
   {{LL(0x9f102fb3,0x041e0aff),LL(0x106ae748,0x91b3a87c),LL(0xc426fa5d,0xfd969804),LL(0x28f95b76,0xe624f1cd)}, {LL(0x34f2ea56,0x6fe28cce),LL(0xd230f37c,0xdea55947),LL(0xf92f2742,0xd5e336f2),LL(0x1899c751,0x86852e3c)}},
   {{LL(0xa5d1bd04,0x5ef2a63b),LL(0xb6ca2b79,0x5f4721a2),LL(0x9f484f78,0xbdb27b7c),LL(0xb085b4ed,0x2b07bf5b)}, {LL(0x501b62ab,0x96b8ae73),LL(0x3ba64e23,0x0b1e003a),LL(0x93024347,0x43f9ec00),LL(0xae180a03,0x3c8c0c7e)}},
   {{LL(0x8c0b21d4,0x58c72237),LL(0x2b15a1fa,0x9d51a996),LL(0xec755eda,0xf5201743),LL(0x933800fb,0x0c845fa3)}, {LL(0x0e82418e,0xb6b66cdb),LL(0x3ae3eeb7,0x875258e5),LL(0x1a8f2b3e,0xf2c30b1e),LL(0x250f3328,0xa10b3843)}},
   {{LL(0xc47c53f8,0x9f449967),LL(0x8775e16e,0x5dfe8c76),LL(0x336f2194,0xb02813a1),LL(0x5636498b,0x90ad3ad5)}, {LL(0xc7c647e0,0x095acf96),LL(0x1f57c069,0xc90ef12b),LL(0x1fb85cc1,0x52f51878),LL(0x25a125ad,0x582cfd67)}},
   {{LL(0x0d43ffad,0x53b4bfc7),LL(0x04dcf047,0x143b0b48),LL(0xd4500bf4,0x65d16216),LL(0x9ab1e4cd,0x960c7910)}, {LL(0xe1d08c70,0x38b7ef7e),LL(0x9806e01e,0x64ae69e1),LL(0x6796b923,0x07468184),LL(0x70af1e64,0x6480887a)}},
   {{LL(0x02384b34,0x4eb2d6fb),LL(0x05be47f3,0xb29337a8),LL(0x6b744f9d,0xfec96fc0),LL(0xc8c9afc3,0xc3de2fb0)}, {LL(0xcc6dd0a5,0xe8ccc3eb),LL(0x71d7de7a,0x0329a9b9),LL(0xe357c4f9,0x459fbc8c),LL(0x025fdc97,0x80287f50)}},
   {{LL(0xa089583d,0xedf1b0aa),LL(0xfb08add3,0xb1ad1a57),LL(0xe1ae76c1,0xd6826d03),LL(0x541462c8,0x3070cd2e)}, {LL(0x83e6f4da,0x7b03c859),LL(0x24bdb487,0x5b39a809),LL(0x453bebb7,0x70017570),LL(0xb8ebbfc6,0xfe4e6206)}},
   {{LL(0x106defe3,0xbb8a1899),LL(0x8683287a,0x6f23dc7a),LL(0x65d96aed,0x2cf01995),LL(0xdda4ea18,0x4e4cf7e9)}, {LL(0xd2d0316b,0x72ad201f),LL(0x61de6cd4,0xd6451150),LL(0xc84856be,0x12432dbf),LL(0xd2a8378a,0xdd4dca98)}},
   {{LL(0xbf881f9e,0xe70af958),LL(0xc4e58ec4,0xd4cd35ad),LL(0x5a531924,0x3889d3d9),LL(0xb4ce15ec,0xac657424)}, {LL(0xf41e1344,0xdbe384ca),LL(0x5ab8bb08,0x9a1aed23),LL(0x8561df1d,0x375a041f),LL(0xb7685c1c,0x19f7a238)}},
   {{LL(0xa4ba6317,0x8ba59933),LL(0x271f4aa0,0x0c44b6df),LL(0xbd64e922,0x51f4e88f),LL(0x9095769f,0x7279df94)}, {LL(0xeaf8c8d3,0x098c17b6),LL(0x1aa841d1,0xe602ff2c),LL(0x8b63ce81,0xbe4e4926),LL(0xfcc79573,0x85de277a)}},
   {{LL(0x5b8304db,0x38253d40),LL(0xe422af76,0x58c50c3b),LL(0xbf95c27a,0x7f7ec0d1),LL(0x6041df33,0xcb7c3a8c)}, {LL(0x35364c89,0xc55595c0),LL(0x2a6eb1e6,0xd1a72aa7),LL(0xdeb98a3d,0x1fa941de),LL(0xeff46690,0x1e9607ab)}},
   {{LL(0xad46a05a,0x6633e398),LL(0xb585e241,0xb99e5784),LL(0xea558424,0xd63106a4),LL(0x5df0e501,0xf0a5f939)}, {LL(0x59dacce3,0xba17aaef),LL(0xe907c457,0x03dc5a07),LL(0xa9800bc3,0xa59f6d63),LL(0x364e1ef7,0x294a3827)}},
   {{LL(0xdd191356,0x741bbab9),LL(0xc43954a4,0xe8fe9161),LL(0x65341d90,0x6a711fa9),LL(0xadef2d82,0x09bd0faa)}, {LL(0x21ffc303,0x2112f27e),LL(0x395b69e2,0xcd2214dd),LL(0x8670b06f,0xe4b503c9),LL(0xc4e13ef4,0x219a678a)}},
   {{LL(0x4a993816,0xc4020eff),LL(0x1bac14d5,0x00a9f5de),LL(0xeba7c3a4,0xd00fce1f),LL(0x14b537ab,0x2c6d4993)}, {LL(0x6b898739,0xe9b2b540),LL(0x29dbf826,0xae53e6e3),LL(0xc8438b2c,0x634606c7),LL(0xfabfd429,0x268a9ee4)}},
   {{LL(0x173b5583,0xb0486aae),LL(0xbf222673,0xf88a2f80),LL(0x0b3178c8,0x49c56f76),LL(0xd77d1406,0xeab47059)}, {LL(0x993b1a7a,0x95035846),LL(0xa9b83efa,0xd6446e94),LL(0xc4424fa8,0x1d1a71ce),LL(0x3d08b8d2,0x8d814c4d)}},
   {{LL(0x7b9374ac,0xbc3ed8d2),LL(0x77a3c020,0x8dd2d56d),LL(0x97efca8d,0x93ada735),LL(0x37974cd3,0x072bb2d0)}, {LL(0x7bd74e40,0xa7c86e7e),LL(0x5b52e0ed,0x7bff5613),LL(0x053af1f1,0xc8d0bb30),LL(0x840bcb7d,0xc5bdb8f9)}},
   {{LL(0x41690d1c,0xabdf5f73),LL(0xf0edac8c,0x0e857a78),LL(0x8238cfb0,0x59f40fcf),LL(0x511d41d2,0xdcb54f67)}, {LL(0x0e645117,0x3f036ac8),LL(0x7af5fdcc,0xdc4e833e),LL(0x3d7bab2c,0x67d859b2),LL(0x5f8b32bd,0x92489b23)}},
},
/* digit=22 base_pwr=2^154 */
{
   {{LL(0xb9e2f9f8,0xe412aaf7),LL(0x0ff419ac,0x0484c1aa),LL(0x417bcb90,0x9d944989),LL(0xfe7318ca,0x2b73dbe3)}, {LL(0x52dd7b0a,0xb91b71e5),LL(0x3954afeb,0xd61f8eea),LL(0xa07e3958,0xaaeab13c),LL(0x02a1ff49,0xde442032)}},
   {{LL(0x0b054a0f,0x8292d96d),LL(0x74b9077a,0xa978af88),LL(0xfff1d49f,0x70bd185b),LL(0x0279eab1,0xbe6d0844)}, {LL(0xb8ed07e9,0xa8fffe45),LL(0xcb920e55,0x714824a1),LL(0xaf1bb143,0xcd5c628a),LL(0x7637dbb7,0xd151afcd)}},
   {{LL(0x62d7ee7c,0x83fb0f37),LL(0x9a3bcb7e,0x58c2282f),LL(0xeac2ca5a,0x79f77476),LL(0x579a262b,0x7e80c351)}, {LL(0xedb4f0fc,0x19e67272),LL(0x1fbbe9fe,0xe142bb31),LL(0x95ea6cc1,0x5c7d7cce),LL(0xabfdcf7f,0x6465a380)}},
   {{LL(0x5a26a1d4,0xa433bd2e),LL(0xd1c2d78c,0x1148bb1d),LL(0x64102515,0x4aae419e),LL(0x66489384,0xd03b9939)}, {LL(0xd61a9919,0xe21d58b1),LL(0x4a0ef3d4,0x17618c36),LL(0x6fe8c0dd,0x2519020d),LL(0x00b87a75,0x48d837d6)}},
   {{LL(0x426c1aa6,0xe6e067ab),LL(0xb11d1280,0x431579d2),LL(0x2ead6552,0xb926943f),LL(0x057fed1f,0x8fd692bf)}, {LL(0xe9a98fae,0xed11c0ed),LL(0x9bcb2abd,0xe2bc967a),LL(0x68729b8e,0x1b388d66),LL(0x6f74563d,0x2144d67c)}},
   {{LL(0x03fcd3bf,0xbe519757),LL(0xc8c7b62f,0x3f9dbd8d),LL(0xfc476e0e,0xce91fce6),LL(0x2715393a,0x2f140c73)}, {LL(0xf11da35b,0x8a149a94),LL(0x5367030d,0xf6a2be5e),LL(0x0269def3,0xb68c0d82),LL(0x8eecb775,0x32d58819)}},
   {{LL(0x32845ab0,0xccea6f53),LL(0x2541c834,0x792bc041),LL(0xb1336aa7,0xd726425f),LL(0x3ddd6256,0x85b1d21e)}, {LL(0xd9b1ba0b,0xd575bfa8),LL(0xb778b77a,0xd23084e2),LL(0x44bb1010,0xd44e7399),LL(0xa91623fc,0x3d665388)}},
   {{LL(0x3f11fc00,0x5cfd3a69),LL(0x8bc8eade,0x1f2b5d01),LL(0xa6b9f7ae,0x5160359b),LL(0xfa696463,0x1e2601dc)}, {LL(0x915f6084,0x7f5ac6d2),LL(0x679176d5,0x6e387789),LL(0xae26abee,0x7fb99f4b),LL(0xaa409d22,0x4798a2fc)}},
   {{LL(0x5965615c,0x582164f7),LL(0x0472cbeb,0x2c9dfb60),LL(0x2266724f,0x36eacc3f),LL(0x5fcb8868,0x253eb08c)}, {LL(0x760c15b3,0x749a0577),LL(0x5686b036,0x71e4ce1e),LL(0xb710196a,0x47893a8f),LL(0xdf51c4e8,0xe27dfbac)}},
   {{LL(0xffb3df08,0xc9536d6b),LL(0x6dde9e09,0xc95169ce),LL(0xcc085766,0xcb050de7),LL(0x0df088dc,0x92fce77e)}, {LL(0x88781592,0x10c124ca),LL(0xc81030f3,0x6429d30b),LL(0x09e20c50,0x2e37721d),LL(0xf3e3d604,0x43e7f9ca)}},
   {{LL(0x5b095e01,0xa277a87e),LL(0x83a51a95,0x968bc951),LL(0x53aff355,0x3b375d45),LL(0xe1ebac06,0xb79d7cce)}, {LL(0x022995b4,0xd929e1a6),LL(0x28164ff7,0x228cf7f4),LL(0x5d3e4608,0x7bd12900),LL(0x2f97ebd8,0xc57ac873)}},
   {{LL(0x86b383b5,0xc192342d),LL(0x706b01d5,0xe85f303f),LL(0x88cdcb89,0x19e19213),LL(0x2ce0de2a,0xe88f1943)}, {LL(0xe453aecc,0xf6fcf8cf),LL(0x9a67b49f,0x0dcd10b8),LL(0xafece961,0xb93d5b4d),LL(0xc39d0b53,0xe232f34a)}},
   {{LL(0x30735384,0x1b8f6cc3),LL(0xe4f26c08,0xc35c5a82),LL(0xba98f626,0x9e0c933b),LL(0x4c70aed7,0x49868100)}, {LL(0xb7f26c66,0x711a3aad),LL(0x7dac506b,0x786ea28d),LL(0x43935657,0xd3a7ab1e),LL(0xd1b69e9e,0xda7f5c1f)}},
   {{LL(0x0e6c8579,0xc08c85e5),LL(0x8d991759,0x29d04ad4),LL(0x3a8ccd69,0xbae8f163),LL(0x1790a49c,0xade66539)}, {LL(0x45915cc1,0xf9f5bc8c),LL(0x4f2b18c3,0x63461cf0),LL(0xd236e848,0xceb75a9c),LL(0x847ce6c2,0xac653e3b)}},
   {{LL(0xdb088764,0xb93b3032),LL(0xa78e5943,0x567fe1c3),LL(0xe359cb34,0xba7a7acf),LL(0xe2c3827e,0x38f4fbfd)}, {LL(0xc90abad2,0x761c36d4),LL(0x75027c02,0xac1af4e7),LL(0xd4715572,0x95e6d01c),LL(0xd621145d,0x5b06cf39)}},
   {{LL(0x64ca2efc,0x799acd7c),LL(0x4e0bcb6c,0x3397a15b),LL(0x0358a26c,0xb9b10ced),LL(0x4b8ddfaa,0x0a30dbbe)}, {LL(0xe20f6fac,0xa70e9712),LL(0xd11451b0,0x87c7f732),LL(0xd5eece8f,0xf0c967b1),LL(0xab370e2d,0xbc62882a)}},
   {{LL(0x59ddb7cb,0x134fb08e),LL(0x3ae8f816,0xe937c663),LL(0x802ed184,0x083f73a7),LL(0x8cd69f8d,0xd4badd85)}, {LL(0x987f389e,0x2d8bfaf5),LL(0x4454b1f2,0x5338c056),LL(0x2f104468,0xdce38439),LL(0x83c5278b,0xffd94d27)}},
   {{LL(0x5628ad08,0x8740af50),LL(0x8b1284e8,0x30a233db),LL(0x57acc8cd,0xb3982d73),LL(0x37c5ff03,0x211d53d3)}, {LL(0xf6578d40,0xb6371f1b),LL(0xa80dec53,0x7f749bea),LL(0xa9816ec0,0xe6b3f730),LL(0xf5423ec6,0xd26832fd)}},
   {{LL(0x63e27b64,0x80127368),LL(0xd2d21879,0x17b7a4b2),LL(0x43cf40d4,0x7dcced37),LL(0x97cf7c4c,0x999bbb80)}, {LL(0x6bafa0b0,0x191c84e5),LL(0x917f6b17,0x1d08c049),LL(0xf4715c99,0x02e5fe53),LL(0x0658f1de,0xa92c6085)}},
   {{LL(0x16a010bc,0xe9c0ba85),LL(0xea4f3e8e,0x2fd90fba),LL(0x4570a1e5,0x8af18371),LL(0x7cca9004,0xe869e8f7)}, {LL(0x2dd83019,0xe2c8afb7),LL(0xfd99b386,0xb877995d),LL(0xf5adab87,0x1e3efc16),LL(0xaa3b191a,0x93105fe4)}},
   {{LL(0xae504c31,0x21690dca),LL(0x698f629d,0x2d51ead4),LL(0x724c9cbf,0x2af3eef1),LL(0x81a0d4aa,0xa6181e60)}, {LL(0xa94f6b05,0x580982c7),LL(0x48653ad7,0xe8bea903),LL(0xa608598d,0x0270614c),LL(0x3d0d5360,0xa7cae0f0)}},
   {{LL(0x96067f64,0x81407687),LL(0x77a62d7d,0xab2c2706),LL(0xae19786b,0xbe9c1edf),LL(0x887814eb,0xa313f2b2)}, {LL(0x08fd3c04,0xe2bc4c1f),LL(0xe5a9d032,0x25387129),LL(0x8fbc5030,0x7b3ced22),LL(0xadbf1bdc,0xc22bea3b)}},
   {{LL(0x7b1308da,0x4f6b6b6d),LL(0xd0e33069,0x0f2faaaf),LL(0x0d3677c4,0xb461990f),LL(0x0e6a876b,0x55c9df43)}, {LL(0x316d252a,0x5ce9aaa4),LL(0x0e8e5097,0x7d98a813),LL(0x9aa3343e,0x047ecd13),LL(0x939277e1,0x15cc7072)}},
   {{LL(0x0a020be7,0x305165d1),LL(0xf66eaf8b,0x48560411),LL(0xffd2380e,0x5ff898dd),LL(0x784b4b11,0x7da35f08)}, {LL(0x38fd05c7,0x50f53e2c),LL(0x47ada3a5,0x64b3ee82),LL(0x678995de,0x672ae316),LL(0xdfe96605,0x74707460)}},
   {{LL(0x441e7150,0xb346dc71),LL(0x55fd483c,0xd9505e7a),LL(0x94302331,0xca96e59f),LL(0x801930cf,0xcfde701c)}, {LL(0x73c31e5d,0x02fc9936),LL(0x8cda0b51,0x4ef53a55),LL(0xa269a1f3,0xa934e268),LL(0x7cca8542,0x7ba4e5e0)}},
   {{LL(0xa2ae339c,0x4c6408f9),LL(0x5a70ba33,0xf9ea4cb2),LL(0x5cac2af4,0x3eaa9364),LL(0x95eaea09,0x62686d46)}, {LL(0x3e771722,0x5196e88f),LL(0x7108b198,0x749518e8),LL(0x29b25254,0x394107c4),LL(0x3a315aad,0xf9945ac1)}},
   {{LL(0xaab9dbe5,0xce15c84d),LL(0x3940eb15,0xebb54d52),LL(0xa2fdd11d,0x69b649c7),LL(0x3f6ade80,0x4e2d1782)}, {LL(0x2327f7d8,0x0f53ac9c),LL(0xc79eb564,0xf6158d6e),LL(0x4536f5c1,0x2903bfc0),LL(0xfb9e9e07,0x0a25518b)}},
   {{LL(0x62a0b0ed,0x70cbce8b),LL(0x0abbc9be,0x92f5dc33),LL(0xf369c2d6,0xbb92b7d3),LL(0x79ef83e1,0x70dd90c8)}, {LL(0x7937ab45,0xe0b33153),LL(0xc054af6d,0x3a8d1f74),LL(0xb05ebfc4,0x35cf7380),LL(0x58c2cd0c,0xefb8dac2)}},
   {{LL(0x7d665d26,0xe7316f99),LL(0x800fba6f,0x59a7ead9),LL(0x08a2cb88,0xfa4d2a2a),LL(0xb441995d,0x2e7d3bab)}, {LL(0x93046f2b,0x390988c9),LL(0x08869cf5,0xfd95b86e),LL(0x9a76537b,0x0185b6be),LL(0xb6cd3d59,0xa89563bd)}},
   {{LL(0xecb1ad25,0xe79a4f63),LL(0x6948504d,0x1857cec7),LL(0xa497922f,0x03b7b3ad),LL(0x38930f36,0x9df2f2e4)}, {LL(0x4bb5927c,0x355e4a7a),LL(0x636ec349,0x5ad3fd47),LL(0xc41b19ca,0x5400730d),LL(0x555afa93,0xbfeabac1)}},
   {{LL(0x6cca58b7,0xb6232083),LL(0x76d0c53e,0x55faae6b),LL(0x40a8eb5a,0x64ef60e2),LL(0xe8f22c94,0xc68bc678)}, {LL(0x10a0416e,0x5156dc1c),LL(0x5c2037e4,0xac779644),LL(0xc7162aaa,0xd2e30925),LL(0x2cf21e2f,0x7bb5275f)}},
   {{LL(0x0c11e65a,0x7722cb40),LL(0x68ff2be5,0xc94a7f52),LL(0x8d9f9352,0x420085cc),LL(0xca4b2544,0x4addb986)}, {LL(0x06264a47,0x3c6ceac0),LL(0xe2b48ccc,0xebc01a03),LL(0xea94fef2,0xc430e7ab),LL(0xbd94aa8a,0x973bb6f0)}},
   {{LL(0x3225b585,0xd60e5feb),LL(0x01b56ad1,0x6cbab39c),LL(0x37d6d1b4,0xcb55a9cc),LL(0xfbce1d89,0xd7288c1e)}, {LL(0x162d4b46,0xcb516843),LL(0x15edb910,0xf0aca3a6),LL(0x08a6685a,0xdb998b55),LL(0x07811873,0x16b442e6)}},
   {{LL(0xa1a7e0c2,0xa9badd09),LL(0x9f813289,0x0a9a339b),LL(0xd4cda45b,0xabf1793f),LL(0xc7378a84,0xa9830a12)}, {LL(0xd28165b1,0x1ae11c32),LL(0xf71bca14,0xbfd49ace),LL(0xfc035476,0x9a3990df),LL(0x6c32b72a,0x0fd2b153)}},
   {{LL(0x3541b5ae,0xceece835),LL(0x8256c750,0x2f7429f5),LL(0x88104f8c,0x456c3478),LL(0x8b23da06,0x8a435588)}, {LL(0xd817ce6a,0x6b6c14f2),LL(0xf35ab86a,0x83bf0acb),LL(0x364b83fd,0xdadb89ba),LL(0x5cfecaf3,0x2c8fcf90)}},
   {{LL(0x20d12c92,0xa90f77ca),LL(0x69d1739c,0x2e278e0e),LL(0x5c1f9e82,0x29d24b44),LL(0x647c59b1,0xbf4fb4cb)}, {LL(0x90ffd733,0x9c8ea39d),LL(0xf14db3fc,0xe37a1352),LL(0x8f3e1dca,0x3c9164a2),LL(0xaec86440,0x515c16f2)}},
   {{LL(0x5c483906,0x736fee4c),LL(0xa3f651c7,0x2325cabb),LL(0x35b94e45,0x582324df),LL(0x45598c64,0xeacedb3a)}, {LL(0xde9ea8cd,0x674e1740),LL(0x89d2b975,0x30f2f423),LL(0x9c8abe45,0x330bd76d),LL(0x5371c0c4,0xb97e89f6)}},
   {{LL(0xb7569543,0xb1769248),LL(0xd85f4d72,0xd29cc9d2),LL(0x10a5b6dd,0x89e1fd0c),LL(0xa693a796,0x501be0ae)}, {LL(0xe490e600,0xc70965b6),LL(0x1bb6c5cd,0xf518f8af),LL(0x76f6daa2,0xf51d40bb),LL(0x5ec7849c,0x83a83b67)}},
   {{LL(0x6d8aa314,0x0fe0d975),LL(0xea664a8c,0x9bf9aed5),LL(0x96fad9aa,0xef8bb989),LL(0x04a0e441,0xd07dce35)}, {LL(0xb3c5eb81,0x53bd2a16),LL(0xaf178b66,0x49e29fe2),LL(0x24dced32,0x62cf7a62),LL(0x0f541e36,0xcc111fba)}},
   {{LL(0xda9dd111,0xc93cd7c1),LL(0x28c9c1b4,0x56b625ab),LL(0x3769f3a2,0xeff436ae),LL(0xcbd31a17,0xa0d8d46b)}, {LL(0xc80dc873,0x241693fa),LL(0x3cd579ab,0x56083f64),LL(0x33fbd431,0x12ee753b),LL(0xd66c283a,0x1bde60ad)}},
   {{LL(0x0243cd83,0x0db508dd),LL(0x1349307c,0x3b12c134),LL(0x61d86bda,0x8296aa6d),LL(0x630adc96,0x1d5c8a4f)}, {LL(0xa30a8ae6,0x9d01dc28),LL(0x1dab8168,0xc555a743),LL(0x7abe577a,0x61fe0d14),LL(0xc8c93bb7,0xe26aa4d8)}},
   {{LL(0xda2bab5b,0xfb4b03bf),LL(0x79b4e6c0,0xfbd49089),LL(0x86806aa4,0xda1a0108),LL(0xdc078112,0x281f76ae)}, {LL(0xe0fbd693,0x9f662594),LL(0x49ec4ee0,0x1da897b0),LL(0xfc7d1578,0x20d52a97),LL(0x6b1f4ab4,0xdbf8d157)}},
   {{LL(0x3b97d1e3,0xfc0a5936),LL(0x1aa091b6,0x00f0f283),LL(0x13aadeb0,0x505e183e),LL(0xa55b3f8a,0xe28041ad)}, {LL(0x086c2d23,0x2e0f76da),LL(0xf2c5eceb,0x815b147d),LL(0x673ba5f2,0x02066c02),LL(0xce043d4d,0xb85d6a8a)}},
   {{LL(0x113890f6,0xd5f023a3),LL(0xa9d2491b,0xaa4f9058),LL(0x16d175a3,0x6d82393e),LL(0x671e2aed,0x1d1e00b2)}, {LL(0x40018bae,0xd47c4f28),LL(0x7b30838f,0xd08eac83),LL(0x5dfe910d,0xa0fde631),LL(0x5c66d5c6,0xfc16adf7)}},
   {{LL(0x18d8c6b1,0x0ed2a8a2),LL(0x632b5b07,0x67ee6037),LL(0x21a89b8d,0x7eed42e5),LL(0x33e6da02,0xd99942cf)}, {LL(0x39971405,0x759ec79e),LL(0x174dca4c,0x669a92c7),LL(0x9d1e7c55,0x85935ed7),LL(0xa82055c0,0x5f3f9e68)}},
   {{LL(0x56aa5af3,0xab0507c8),LL(0x1bd2726f,0x354cac5d),LL(0xb864816f,0x46e85e16),LL(0xd1840add,0xef2548f6)}, {LL(0xc3842f43,0xe494ea07),LL(0xedf6c13a,0xa6169c4a),LL(0xa460e30b,0x65d9cca3),LL(0x31e7dfc3,0xa6848e4f)}},
   {{LL(0x5c8109dd,0x4309f315),LL(0xc5799833,0x7a4ec14e),LL(0xa8132b78,0xcb768a63),LL(0xb416c77c,0x229106d1)}, {LL(0xded02f41,0x1ca71df6),LL(0xc1a1fc66,0xb6365d3e),LL(0x1431d1fa,0xf7c432a1),LL(0xa5654387,0x30364500)}},
   {{LL(0xd5b13b2e,0xc9ed0cf8),LL(0xd18d5a28,0xdbd541bb),LL(0x754de9d2,0x6b78c887),LL(0x54651568,0x7d32fedb)}, {LL(0x0d37c339,0x7f319680),LL(0x37d70b76,0x22304d1f),LL(0x6fb5e555,0x01b2709e),LL(0xfd5d1708,0x978b0d3e)}},
   {{LL(0x96bc118d,0x83206b9d),LL(0xec7bfc1c,0xb1a4d7bf),LL(0xb6b41502,0x753f98a6),LL(0x4c5187ce,0x41139110)}, {LL(0x587a8213,0x56e9e218),LL(0xad9aefd0,0x3b39955b),LL(0xb9947ceb,0x7428b03f),LL(0xbe8bda29,0xbbe82668)}},
   {{LL(0x5c4b4c63,0x5142e8ba),LL(0xe92ee291,0x90c3e2e3),LL(0x8f6a076d,0x6947a55a),LL(0x61964435,0x9acdeec1)}, {LL(0x181dac35,0x56bc8e4c),LL(0x7a824372,0x4f4f2c0a),LL(0xc1033f6b,0xd1958b99),LL(0xc83ecf98,0xeeaa6604)}},
   {{LL(0xaca52cb3,0xe43c0b44),LL(0x75443f14,0x12446426),LL(0xddcc00b4,0x0d14e885),LL(0x6cfe5734,0xb0f5f11d)}, {LL(0x1013afcb,0x0e160164),LL(0xed9f4535,0x4f570ca9),LL(0x73a307ad,0xe5162a12),LL(0x3321ae54,0x6a431695)}},
   {{LL(0x5ae301b4,0xa6c7b0c5),LL(0xbd2d3f1d,0x6f5d42b1),LL(0x15c0c94b,0x4eb12c09),LL(0x28618c41,0xf1c40386)}, {LL(0xc0f55c25,0x30302333),LL(0xbd1c19f0,0xa5e41426),LL(0xcfcc66f8,0xd5d4d4d7),LL(0x449253c5,0xcfdf3039)}},
   {{LL(0xb30ec0ff,0x17b0eb72),LL(0x5e6424f9,0xbce593e2),LL(0x2a73184e,0xa5d82937),LL(0xebe58773,0x23d2857a)}, {LL(0x067e1eac,0xe3f0f676),LL(0x50509d7f,0x073ded2d),LL(0xca405a7e,0xc22af8f0),LL(0x6df6a46c,0x7a4ef592)}},
   {{LL(0x97067006,0xf9cb0178),LL(0x489d2a39,0x9ae132af),LL(0x6a2da1c1,0xc7c46b35),LL(0xd95850c9,0x0993353b)}, {LL(0xa25d52ef,0x6c313a57),LL(0x93c852c3,0xa6bdb2b2),LL(0x7e9e296d,0x27ed916b),LL(0xc7aeb09b,0x10b58337)}},
   {{LL(0xecebe36e,0x78800c35),LL(0x2234ce8a,0xd93e2423),LL(0xfa95019f,0xe4cf5cee),LL(0x71e13748,0x21396d37)}, {LL(0x0c32fdad,0xeb028350),LL(0x61f1652b,0x31645697),LL(0xf6677491,0x9e1c6e0b),LL(0x74176c12,0x4d18f2e5)}},
   {{LL(0x3832d713,0x78d559bf),LL(0xb6e00e15,0x04f0b57b),LL(0xe80add3a,0xd6c9cb16),LL(0x5c7b1d70,0xeabfabc5)}, {LL(0x98a62cc3,0x40570866),LL(0x4abb2b1a,0x39ef8ff1),LL(0x0c19959c,0xadb40548),LL(0x388b1f7c,0xd61632d7)}},
   {{LL(0xd73b7d50,0xd1f9b736),LL(0x560bf0aa,0x652ed78e),LL(0x50e3fc4f,0x58e71e33),LL(0x55df1ad1,0xbfaf5f44)}, {LL(0x9106744f,0xefe8893b),LL(0x356d1fe9,0xabfbd51e),LL(0x9eb1cbaf,0xab03570b),LL(0x3919012c,0x92cfe2e4)}},
   {{LL(0xb6f7c64d,0x7671e5fb),LL(0x6e0a44b7,0xf040c039),LL(0x62b36088,0xf430f593),LL(0x94c7c0ac,0xa85b4bc9)}, {LL(0x16b54fff,0x07d5c40c),LL(0xc53a3788,0x47aa73ee),LL(0x7000d61e,0xa63c5b36),LL(0x91b9f09f,0x04e8f53d)}},
   {{LL(0x87dc6a3d,0x7e48021d),LL(0x28ae0f74,0xa2b5516b),LL(0x705132e2,0x84412003),LL(0xe243d1fa,0xc55f69cf)}, {LL(0x6a8f80bd,0x758c0f71),LL(0xd09b719d,0x69ecf887),LL(0xa9b45194,0x51b100f0),LL(0x90ae0769,0x1fb9ef66)}},
   {{LL(0x30fcdfd2,0xfee82fab),LL(0x36a6990b,0xf36185be),LL(0x3d33027b,0x88f343f6),LL(0x38ae16c6,0xb775dcbb)}, {LL(0x85a82e45,0xa107b9f0),LL(0xde6b9806,0xaff8b0ae),LL(0x0392fad0,0x3cd3980f),LL(0xf3cf7650,0xdd829fc6)}},
   {{LL(0x0dc8d031,0x177190cc),LL(0x7fc491eb,0x3e21cd25),LL(0x0d929039,0xea0cc90e),LL(0x1dfc37b3,0x5f7e6292)}, {LL(0xe23bdd04,0x66dd6dde),LL(0x64fa490a,0x70e7a317),LL(0x10a03dd8,0x59c90f81),LL(0x96d58314,0x425ee6ce)}},
   {{LL(0x5f896ed1,0x868001eb),LL(0x91dad4fd,0xc4c003f5),LL(0xd9ef80b4,0xfb4782b2),LL(0x323e4fc5,0xb9edb975)}, {LL(0x53ef4ccc,0xa2ec9b6c),LL(0xa77922b6,0x4af8b2ca),LL(0x6697874b,0x73850e89),LL(0x3568523f,0x76e0fd72)}},
   {{LL(0xe9c400a6,0x64799f46),LL(0xa9c245de,0x6c5176e7),LL(0x93503700,0xbd97c80c),LL(0xffbe539f,0xa92d9ee5)}, {LL(0x8376bb3b,0x76003d14),LL(0xac564679,0x2e75cc77),LL(0x3a333970,0x126af6c7),LL(0x6b6604bd,0xdbfd0133)}},
   {{LL(0x24424a48,0x11cf4c2e),LL(0x37d4471c,0x843c73ee),LL(0x617a488b,0xb3047fc5),LL(0xe3cf861c,0xf2a91709)}, {LL(0x1c3a60f7,0x84444421),LL(0x26679148,0x74787a36),LL(0x53d9404b,0x115fbd06),LL(0x6244cef0,0x70fd3365)}},
},
/* digit=23 base_pwr=2^161 */
{
   {{LL(0x2b574b7f,0x76695c9b),LL(0xc369b6be,0xcca80405),LL(0xe3108ded,0x1f4bae99),LL(0xea133fce,0x9e715ce2)}, {LL(0x54c2ee1c,0x60d52055),LL(0x1680742e,0x56bab301),LL(0x3fe438b9,0xa409b5f6),LL(0x8036f7ce,0xe3a8e4d0)}},
   {{LL(0x247fdfdf,0xe1d7ec0f),LL(0x4a23d1dc,0xfb9d90e7),LL(0x190fdc41,0x7012eb2c),LL(0xddced48c,0x5c2bbff6)}, {LL(0x68cd7feb,0x8a93426a),LL(0x6b4854e1,0xb5963962),LL(0xe772bbd8,0x8ac72b8e),LL(0xa6b3040a,0xc10d24d2)}},
   {{LL(0x94d5f347,0x8fdfef16),LL(0x2b04af0a,0xf3189490),LL(0x6d2ca633,0x30e3da7a),LL(0x4803814a,0x8d002aea)}, {LL(0x95a0bfe9,0xc15e311f),LL(0x4b4cc50c,0x2891ec7e),LL(0x8834df25,0x0936fed8),LL(0x78e00289,0x7e5d7dbf)}},
   {{LL(0xfbfcf1b5,0xb9a92d78),LL(0xe8427d74,0x17ce4fab),LL(0xac66e74e,0xbae98ffd),LL(0x145bb5e5,0x6d548304)}, {LL(0x0992abe1,0xbf3dc603),LL(0xbefdc5c5,0x318cfbda),LL(0x59f8efb8,0xbb5fa37d),LL(0x4ef5bef8,0x347874a0)}},
   {{LL(0xbf68688b,0xdf552b01),LL(0x8f96a57a,0x2fc542cb),LL(0x4edb340e,0x5a731b61),LL(0x181cf578,0x5143d103)}, {LL(0x2cc936b6,0x749ab511),LL(0x0dd355c2,0xbc94c053),LL(0xa3900fa2,0xa825eff5),LL(0xc1dc2b31,0x60a909a3)}},
   {{LL(0xaf5bcab5,0x59b33c78),LL(0x496fbcdf,0x0053d789),LL(0xd7883bc1,0x5a5afe02),LL(0xfa66951d,0xec9afe78)}, {LL(0x728e56a6,0x38f28b83),LL(0x78cafb9d,0x21d0b6ac),LL(0x7042e327,0xd43996bc),LL(0x7c31c145,0x60686637)}},
   {{LL(0x3d919304,0xe1f8d2e6),LL(0x456be82a,0x09cf437c),LL(0xf0c21973,0x6a01dae8),LL(0x246d9ef8,0x8bffcda8)}, {LL(0x5d853975,0x7e03a0d4),LL(0x32533ba3,0xc3800ca8),LL(0xf02ce43c,0xd77152cc),LL(0x6392089a,0xb8bc17a6)}},
   {{LL(0x4b4558fb,0x6f5fcb61),LL(0x1f2545aa,0x9602597b),LL(0xabe5e469,0xfd89ab3f),LL(0xfb2e16bc,0xf1daeea2)}, {LL(0x3a12940f,0xe699acd7),LL(0x4d7c7311,0x24980f6c),LL(0x336c8ec6,0x4a5cf975),LL(0x8c27d3dc,0x8e180e32)}},
   {{LL(0xd36cb503,0xafb66269),LL(0x754fdd67,0xe98b07d2),LL(0x5a1fe9bf,0x1e0b425b),LL(0x24fc4a85,0xb4ac13e9)}, {LL(0xc05a9c3f,0xef693781),LL(0x5c0124dc,0x266c1216),LL(0x64ee22e2,0x7f3184c4),LL(0xcdb5f1a9,0x3f985fb3)}},
   {{LL(0xfc01efaa,0xb258cd5f),LL(0x0775588e,0x861688b1),LL(0xfa46eae0,0x72184b18),LL(0x5003404a,0xd17c9dea)}, {LL(0x92e7bf9e,0xa8791966),LL(0x7891ac50,0x049c63cb),LL(0x5d46b33d,0x2ed32928),LL(0x0623595a,0x49d1bfbf)}},
   {{LL(0x36c8e3e9,0x9f871470),LL(0xb20d610d,0xdec7eb98),LL(0x7b151f4e,0x15b9326f),LL(0x04005d02,0xa624c23e)}, {LL(0xd9cacded,0x89fc2a8e),LL(0x9a2c3a00,0x9eb8defa),LL(0xe8d7eab7,0x7c5dc2d6),LL(0xeb0a77cf,0x48fa5403)}},
   {{LL(0xbf033733,0xcc4c31d0),LL(0xef211c17,0xf37d0072),LL(0xae35b246,0x8967fe49),LL(0x5cb1aa9b,0x8c4cbd66)}, {LL(0x04840da3,0xab0097db),LL(0x5828733e,0x3946faec),LL(0x87d64045,0x96c6531e),LL(0x83bc0d0e,0x893d3780)}},
   {{LL(0x53bec0dc,0xf833e355),LL(0x2803a655,0xc9ff7280),LL(0x42b99b53,0x300ff7aa),LL(0x6a7c3f2c,0x3b48a8db)}, {LL(0xf617f8aa,0xf78c21d9),LL(0xcbe4d565,0x23684cb7),LL(0x7514e9a0,0xf64ae9c8),LL(0x8429d8ba,0x4ff5483c)}},
   {{LL(0x5cb18391,0xdedab351),LL(0x769ae948,0xd3126ffc),LL(0xd3546ad9,0x6c2f9ba8),LL(0x69aabfb7,0x4567e48a)}, {LL(0xaa284747,0x6fbe29b0),LL(0x98af4f2f,0x3185f0db),LL(0x5b4c14e3,0xf2a958a2),LL(0x27d04855,0x106150c5)}},
   {{LL(0x68a19ca9,0x60a3b4fb),LL(0xfac47c70,0x65c5719a),LL(0x973e4cfd,0xe228e088),LL(0xcb63c89f,0x122a2429)}, {LL(0xbaea08f7,0x39fda97e),LL(0x621c12cb,0xe7da5324),LL(0xff9b3c84,0x569c8a51),LL(0x4c3b8d54,0x5ab8bb6d)}},
   {{LL(0x00e25a95,0x4f02ece4),LL(0x7ac1732e,0xef947402),LL(0x51149260,0xecdb65ac),LL(0xa9180d51,0x6043aa29)}, {LL(0x852deca0,0x07fc92bd),LL(0x15237c8d,0xf3338297),LL(0xe84b3f38,0xecfb0e76),LL(0x6b89af17,0x21f2f5c5)}},
   {{LL(0x9659963f,0xf7aec268),LL(0xa0cb213c,0x67fb5260),LL(0x66d931b7,0x5daa0fef),LL(0x34d309ff,0x95457a7e)}, {LL(0xc21285b6,0xe7cf1a56),LL(0x244e11b4,0xcbff9b08),LL(0xc0ecce3d,0xd79ee62d),LL(0x8267c254,0xe3f20739)}},
   {{LL(0x037ef2d3,0xee06dd39),LL(0xd522f762,0x790d1b0f),LL(0xf30c47d0,0xf0659106),LL(0xb5fdc6b5,0xcd83214b)}, {LL(0x6593b717,0xc8621660),LL(0xfe3fa381,0xb10a6d99),LL(0xab254244,0xa5c3224c),LL(0x5854b18e,0xd15287e6)}},
   {{LL(0x225806ae,0x6bf9594c),LL(0x57e554f2,0x75a97e21),LL(0x82b00b16,0x0ea199f3),LL(0x5389c90f,0xde81a726)}, {LL(0x86922afe,0x8503609e),LL(0x254b75c3,0x6778ad88),LL(0xf3e660ba,0x6bc2ac1b),LL(0x209c04a4,0x7efc1550)}},
   {{LL(0x2528ec51,0x6e90b6a5),LL(0x0548389e,0x9196a7c9),LL(0x7b5b5dde,0xf7e285c1),LL(0x223d4837,0x6335a624)}, {LL(0x412d19c4,0x8acef5af),LL(0x9783256b,0xb22808a5),LL(0xf53e4b62,0x6ea3daaa),LL(0xfa7bada4,0x7ca4c51b)}},
   {{LL(0xe4d3115e,0x3e40461e),LL(0x3646fc40,0x24889b50),LL(0xfa26ccf7,0x39e0eb1e),LL(0xa82af350,0xfcad5d47)}, {LL(0x4862b1fd,0x90037503),LL(0x1a79283c,0x88e937e8),LL(0x9a0127fb,0x16dd07c0),LL(0x39fca31a,0xac62a168)}},
   {{LL(0xa294dac9,0x26542e2a),LL(0x2a5dcfe8,0xefab45af),LL(0xe642bbe8,0x6166857d),LL(0xff6290a8,0x3f3ad480)}, {LL(0x5f50633f,0x435d4c2b),LL(0x84451c8b,0x36da60a7),LL(0x261612e4,0x00f5e2e4),LL(0x2d04786a,0xe4318273)}},
   {{LL(0x2c175edb,0x192bcda5),LL(0x59a6f637,0x74681e0a),LL(0x2d244985,0x696df08b),LL(0xfcf577c6,0xde61a87c)}, {LL(0xf2c9de81,0xcbd2ceab),LL(0xd36162e8,0x878f06ce),LL(0xb3d22955,0xc4f312e0),LL(0xe903efee,0x736ed43f)}},
   {{LL(0xca9bf60f,0x2c687134),LL(0xbc7da3a5,0x2473ea8f),LL(0xb45fb57e,0xf54ef685),LL(0x3383cadb,0x594e8445)}, {LL(0x4a7df4bb,0xe1edd3fb),LL(0xc17c2c92,0xa783d987),LL(0xcf8fcba8,0x0d498637),LL(0x3acd6e4c,0xdebd801d)}},
   {{LL(0x34d3761e,0x2ade8a7c),LL(0xd825cd19,0xc591c889),LL(0x39b42dec,0x3ffd60ba),LL(0xfd9674dc,0x136d4902)}, {LL(0xda4842c4,0x373a70f8),LL(0x3f078bfd,0x3208c485),LL(0xef608639,0x3587f871),LL(0xf04e46ed,0xf990ab0f)}},
   {{LL(0xa83a8450,0x39d542ab),LL(0xdacb7c65,0x634b9198),LL(0x82486a05,0x680cef78),LL(0x16eaf88b,0xab1d4d77)}, {LL(0x699c7aa5,0x5e605279),LL(0x3c40a07f,0x7e37906f),LL(0xfb6926e3,0x4ae84ad8),LL(0xe2ebc73b,0x236b5f07)}},
   {{LL(0x9e0939a5,0xa94e50ab),LL(0x2d9e10e2,0xabeed810),LL(0x4e6423d3,0xea8190fb),LL(0x17acb62c,0xc739d209)}, {LL(0x6fdbe8dc,0xae38106e),LL(0x63204138,0x1c6532d7),LL(0xbb7d0510,0x03af879d),LL(0x8cd2b1a4,0x1d76faf0)}},
   {{LL(0xd77386cc,0x2fcdaf9b),LL(0xe32d2633,0x30f9f5a4),LL(0x382e7298,0xa4fc8189),LL(0x588af205,0x946923a1)}, {LL(0x114f2beb,0x2c527a79),LL(0x077762eb,0xa2ca55d3),LL(0xcc85e41e,0xe4b2eb7c),LL(0x89346ada,0x4b5938d2)}},
   {{LL(0x4c2084cf,0x8e94e741),LL(0xa839ecb4,0x4ef32d29),LL(0x802f0897,0xc5371755),LL(0xc49ae8a1,0xb0274ff1)}, {LL(0x417bff62,0xf7716d1c),LL(0x918f9555,0x6efb0748),LL(0x7aeb1e8d,0x7d3bb9c8),LL(0x20d51e18,0xee9bd5e1)}},
   {{LL(0xd52033b1,0xfaf0a1a5),LL(0xb8626432,0x7967d3f4),LL(0x5574dc0e,0xe837ca4b),LL(0x2c11d8ff,0xf7eae237)}, {LL(0x87dc4007,0xc0f2f1fa),LL(0x8dfb51f7,0xf5f1f153),LL(0x5bd9ac7f,0xa64b10ae),LL(0xa2198841,0xb3c2ba37)}},
   {{LL(0x66c1ee7b,0x5a7ebac5),LL(0xdba62ea8,0x59e06f4c),LL(0x30944ef3,0xa2ea165e),LL(0x3e21385b,0xfd5c7dfa)}, {LL(0xe3bb110d,0x4a012c73),LL(0x4fb2fdf3,0x16d85219),LL(0x7cad0594,0x1aac7f11),LL(0x4b098d9f,0xea7f7dbf)}},
   {{LL(0x7fd181e7,0x88abaa5c),LL(0xca3ad1eb,0x136a0c9f),LL(0xf394aab5,0xe6e5e6c2),LL(0x9349e4a5,0x84d697d4)}, {LL(0xf76f4b3b,0x8215578b),LL(0x12feeb5f,0x81a1cec6),LL(0x3e876bc3,0x5d336eb7),LL(0x071892ca,0xe8afdcb5)}},
   {{LL(0x3da8d94c,0x22f16f6b),LL(0x2d150069,0x28b276c5),LL(0x643d3e58,0x49d20441),LL(0x3da3a7fb,0x3450c84a)}, {LL(0x442ca3e3,0x8f5bf388),LL(0x9e615382,0xca31411c),LL(0x7798675f,0xbe63e34d),LL(0xd1ea01e1,0x551eb64d)}},
   {{LL(0x34a00e27,0x1738a83b),LL(0xbf58ce70,0xe7591d15),LL(0x57d806d8,0xde2ace5a),LL(0xd0338020,0xe89e8110)}, {LL(0x4e25756c,0x935ed5de),LL(0x46d0f00b,0x07ef8c2f),LL(0xa659592a,0xa28e5fb4),LL(0x7fa4986a,0xcb45c622)}},
   {{LL(0x74de493c,0x6b7df066),LL(0x79aa5258,0x4d6bdaef),LL(0xe2b255ed,0xe9709c34),LL(0x7d0e7443,0xdba2653a)}, {LL(0xa00eb3e4,0xeb8da5c8),LL(0x7ab0e45c,0xe978228e),LL(0x9d551a51,0x3a31bafd),LL(0x403352f5,0x1de0e9cf)}},
   {{LL(0x23ddd51c,0xb94d5478),LL(0x130e78e3,0x7c215c91),LL(0xed547bce,0x556b92e0),LL(0x909f5c6f,0x0072da6b)}, {LL(0xf0dc846b,0x4ec71b11),LL(0xbf7baaa1,0xd0f3b3b4),LL(0x47770705,0x896391c5),LL(0x66732587,0x41fe5736)}},
   {{LL(0x4acd3c51,0x02a7e3e3),LL(0xd30407b3,0x217df736),LL(0xe47c33cb,0x503a31ae),LL(0x4912bbb0,0xe3186392)}, {LL(0x75a5df9a,0x2491a08a),LL(0xc09294ad,0x2882f937),LL(0x979ad9f9,0xe2576b69),LL(0x26dc1ffc,0xf44ddc15)}},
   {{LL(0x968268ae,0x7dad21d4),LL(0xbe9c6fc0,0x07378e90),LL(0x2b329579,0x9406a872),LL(0x761f10ae,0xb27b5c51)}, {LL(0xd04cf60b,0xf5dad2f9),LL(0xdf950997,0x3154dff5),LL(0xd8534a9a,0xaaec9d30),LL(0xac43f212,0x4ac722f5)}},
   {{LL(0x46464c70,0x722882f4),LL(0x6c3c702e,0x9b9b5226),LL(0x8325964e,0x4e3974bb),LL(0xaa0c5227,0xd3ceff9d)}, {LL(0x9534dba5,0xd530c8f9),LL(0xbc751878,0xd26e547b),LL(0xea79b19a,0x184a3527),LL(0x74f1cdc4,0x8dab9214)}},
   {{LL(0xc051e9f6,0x708abc8c),LL(0x4be2d9ca,0x75194e9f),LL(0xd6ab5348,0x031d69c1),LL(0x78b0e490,0x1785990e)}, {LL(0xf6c41f8e,0xd825f125),LL(0x0fbf2fe6,0x429924ea),LL(0xfb87161e,0x53c044be),LL(0x0651d153,0xa3bbdf1b)}},
   {{LL(0xec6ecb9c,0xda660697),LL(0xddb8c619,0x51b4a5fd),LL(0x230fbffb,0x80b87520),LL(0x8848da9d,0xa0587430)}, {LL(0x864c2502,0x98715939),LL(0xaf973396,0x2b10cbfb),LL(0x09572b5f,0x28675184),LL(0x39adf777,0x0a40cdef)}},
   {{LL(0x3ead6eef,0x2efa3bb4),LL(0xd1b9fe65,0xbd76b425),LL(0x5e527201,0x95f006cd),LL(0x38a7dc3f,0x00890f3b)}, {LL(0x3a7ce6be,0x84ffa014),LL(0x89541c2e,0x3406aaa0),LL(0x9559d989,0x430542b6),LL(0xb53bddd8,0x9b427b08)}},
   {{LL(0x49639170,0x2182bd91),LL(0x3299ae83,0xb9fb2b42),LL(0x423b7ea2,0xbc993d59),LL(0xc110039e,0x03e416ac)}, {LL(0x3ffe24aa,0x90c2269a),LL(0x1c322c49,0x421ea02d),LL(0x0ef8fa01,0x40677b1c),LL(0xc59407d4,0xa1acd239)}},
   {{LL(0x8f14decc,0xb8cd4f40),LL(0x69e16a6b,0x95e90d87),LL(0xc3c38fd3,0x85dcf163),LL(0x0c01f90a,0xf4fb87ba)}, {LL(0xdcd0f994,0x8274f825),LL(0x2e6bf7d8,0x4c685fa5),LL(0x3d928011,0xc87d8847),LL(0xf9efa96a,0x9add0105)}},
   {{LL(0x50db3113,0xed39152b),LL(0xb794e6b4,0x6b523496),LL(0x84630b17,0x6bb241b6),LL(0x1de3ae08,0x6e9f8ae0)}, {LL(0xd94ce4fe,0x97bd7c09),LL(0x9e61057a,0xe887b02c),LL(0xc62c27fa,0x853e8feb),LL(0x01600ed6,0x3f9d951a)}},
   {{LL(0xb57b9742,0x3e957b36),LL(0x82b72110,0x92bfd61e),LL(0xfdce7ec4,0x108b450b),LL(0xcc29c494,0xd8af107a)}, {LL(0x47688c92,0x8d67ff70),LL(0x28b9b681,0x57f42933),LL(0xaaf8a48d,0xbbc98ef3),LL(0xe2d549b6,0x14113b1a)}},
   {{LL(0x0b412b3c,0x1172b259),LL(0x1d42a63e,0xaf86ca6f),LL(0x83660d24,0x5f893135),LL(0x5a21a441,0xe7bfe9a8)}, {LL(0x4ee5122e,0xecd0aa5b),LL(0x5e4df46e,0xbb68654c),LL(0x5e243845,0x0c3e820b),LL(0x5c46bfa5,0x042b1895)}},
   {{LL(0x894f7f16,0x791b2085),LL(0xb5c353fb,0x42eb80f2),LL(0xdf8db0d4,0x377777f7),LL(0x34c42ef2,0x023c0963)}, {LL(0xa34cb6d0,0xba05eb5e),LL(0x55cd1242,0xffb8b01e),LL(0x87cd9f24,0xeab6ff7d),LL(0xab3c09fc,0x175e94c9)}},
   {{LL(0x7075fd9d,0x6dc68140),LL(0x4b203c44,0x63851566),LL(0x871d1be7,0x3071e924),LL(0x85ee9cd9,0xe6285b56)}, {LL(0x4bcf8edc,0x738dd629),LL(0x4ace75f5,0xf3a36813),LL(0x3cf6feb4,0x37a09e34),LL(0x2cd0c8af,0x4c2eaef7)}},
   {{LL(0x16205f2a,0xd945a28b),LL(0xabadde7a,0xfe9112a7),LL(0x2bbf97c2,0x7db6c5ee),LL(0xb5b54833,0x3eb84a8f)}, {LL(0x273007d9,0x9732a49f),LL(0xc6a2e3ef,0xe61431c0),LL(0x10a101da,0x88aa1a06),LL(0xb972cc61,0x64b94de3)}},
   {{LL(0xf8402027,0xe79eb6aa),LL(0xea6e7157,0xbb1fa5e3),LL(0x4ebdbe4b,0x457f33a2),LL(0x7a61b393,0xf4e955e0)}, {LL(0x698d37cf,0x578e2e64),LL(0x82ecbb69,0xbb139e23),LL(0xcfe8d05f,0x268d0291),LL(0x625fa854,0x7dcfef41)}},
   {{LL(0x9c4da5e3,0xe21d5b8f),LL(0x10bf3df1,0xb5e22209),LL(0x437bf2c6,0xb04dd106),LL(0x1d055404,0x807c5d04)}, {LL(0x2c06fd15,0x6e983206),LL(0xed63ea25,0x773450af),LL(0x95c8dca3,0xc2dae106),LL(0xd82229e8,0x5323f6ba)}},
   {{LL(0x57c062bb,0x647fabee),LL(0xcd5210ac,0xcd6adee7),LL(0x181f674f,0x11b4df3b),LL(0xf2a92b48,0x4e23bf4e)}, {LL(0x84a83d6f,0xeea34e2e),LL(0x9cb197e5,0xeaa09d51),LL(0x845e5008,0x7f36a278),LL(0x1581c0ab,0x41fa9b52)}},
   {{LL(0x23d1206a,0x58917f67),LL(0x11062b8d,0xc04601ce),LL(0xf31f7326,0xdcc60fb6),LL(0x4b071708,0xc5aeef46)}, {LL(0xdc6939eb,0x5364069e),LL(0x034a1052,0x44bd15a2),LL(0x62a307fe,0x8177eeb1),LL(0x1907ad16,0x451ae4e7)}},
   {{LL(0x27eb3193,0x80e49544),LL(0xaf88f4c9,0xd788e57a),LL(0xd944e00a,0xf062c60f),LL(0xeb4a609f,0x504463e6)}, {LL(0x74f13c8b,0x3593ad20),LL(0xc50bce88,0xdc7c5a35),LL(0xb657d1f9,0xa6336115),LL(0x591425ef,0x18d14e5d)}},
   {{LL(0x1454f76e,0x73896725),LL(0x425c87a9,0x52772de4),LL(0xc6efb7d6,0xe59e4516),LL(0xd76bbc11,0xdddb8bf3)}, {LL(0xc6fd2066,0x1acbebd9),LL(0x1d7082ea,0x88c3b525),LL(0x6d69cea3,0x6a3b3d62),LL(0x8d065405,0xdbf73dfa)}},
   {{LL(0x4a7bd06e,0xd659c8d6),LL(0x7bd10bb0,0x67867520),LL(0x97838647,0x7c4e3be5),LL(0xc5891864,0x545c7144)}, {LL(0xfa78d62c,0xf64e1031),LL(0xfa71692b,0x1f046593),LL(0x71310c47,0xd35a9cb7),LL(0x0ea84922,0x10911b96)}},
   {{LL(0x93a9f5ac,0x5647310d),LL(0x6c05eedb,0xa6785861),LL(0x43950b68,0x2f5aa7c8),LL(0xa9d03b3a,0x57580907)}, {LL(0x42e15fe3,0xd581049b),LL(0x916c4e88,0x55dcf9d2),LL(0x27d1a183,0x87ebfd13),LL(0xf5aaa51e,0x13aee909)}},
   {{LL(0x3b9fc03e,0xa651959d),LL(0x98997a74,0x05c28772),LL(0xae2e4a65,0x73e047f4),LL(0x783aa072,0x359e6c45)}, {LL(0x7a04b710,0x1124e9f0),LL(0x6d2053f2,0xd35094de),LL(0x2575dab0,0x0d57d976),LL(0x69171229,0x822256fc)}},
   {{LL(0x3d19de1c,0xbd46937a),LL(0x6f0be84d,0x71feede4),LL(0x7c4dc4b3,0xca205366),LL(0xe3e851cb,0xfbb97d0d)}, {LL(0x2066e9a4,0x0270b5ea),LL(0x42ae150b,0xeade87ff),LL(0x8eb1bafa,0x9a7f9e81),LL(0x0eb5f68e,0xcb374aaf)}},
   {{LL(0xd5525ab2,0xa5841c9a),LL(0x03e02cd0,0x3eed9ba8),LL(0x279fca98,0x29449bca),LL(0x3f450c92,0x4990ec0f)}, {LL(0xbecbba58,0xa241a8e3),LL(0x2eb47817,0xd0e2487c),LL(0x8300837d,0x6db7d420),LL(0x2d7f59ef,0x78872895)}},
   {{LL(0x1314fc73,0x1b3d5033),LL(0xe710aded,0x2cf4cd42),LL(0x6f4026b7,0x9159bc5d),LL(0x2e62cc45,0x403f947b)}, {LL(0x47d97843,0x18d6ac70),LL(0x0694f7eb,0x69d5faaa),LL(0x6932e0f0,0x7711535c),LL(0x6ebd1488,0xc85c9616)}},
   {{LL(0xd3542212,0x558e3750),LL(0x02921066,0x21fe02d7),LL(0x46b90554,0x1636a1a2),LL(0x0108cc04,0x8acf01ed)}, {LL(0xb4d60d37,0x57a2b16a),LL(0x91f4fdb4,0x3301a33b),LL(0x8e09b548,0x70dc3d3a),LL(0x079c0c2f,0x35ae7d07)}},
   {{LL(0x978f92cc,0x95792f06),LL(0x23196752,0xb11574d3),LL(0xb8cfcac1,0xc3249711),LL(0xcf93af67,0x2061c767)}, {LL(0x2f63dbe7,0xeff09a1b),LL(0x48091edd,0x527776b6),LL(0x19bba5a9,0xf0fa985e),LL(0x66ae3221,0xc54f89f3)}},
},
/* digit=24 base_pwr=2^168 */
{
   {{LL(0x6a436476,0xbc5a6284),LL(0x35dbb9cb,0x6fcc2313),LL(0x5012ffbf,0xa77d2d9f),LL(0x4ae4bd14,0xcc25e9f4)}, {LL(0x1a5e40c6,0xd17fcfc4),LL(0xff085322,0x7d716a5f),LL(0xee3077c4,0x9dcbc50b),LL(0xdb4a6e61,0xebfe953c)}},
   {{LL(0xd3d777d7,0xe7e66f2f),LL(0xcf1a6b09,0x3519dc64),LL(0xdbf88dcf,0x0df07beb),LL(0xacd4e105,0x17b09654)}, {LL(0x4e70c783,0xcbd7acd0),LL(0x96b9d577,0xda66e747),LL(0xe3e52f8a,0x6d0488a1),LL(0x6ff71c1b,0x3ec0fd11)}},
   {{LL(0xbe4f2782,0x75474cb6),LL(0x41c2c0cd,0x10ef5e6b),LL(0x6a65e29c,0x592c6b06),LL(0xd12d0608,0x4d424662)}, {LL(0xb1a714fe,0xf5280949),LL(0x1199f802,0x52697bcc),LL(0xe6a4ff3a,0xc68ba4f8),LL(0x351849ce,0x25a5380f)}},
   {{LL(0x573ec6f5,0x33207f69),LL(0x67bd2e8b,0x7ecc4bbe),LL(0x8ffe2420,0xa07acd34),LL(0xa13f9cdd,0x0a957eb8)}, {LL(0x9ec9c0c5,0x0bc7f95b),LL(0x6a8578cd,0xd82147cc),LL(0x9e61923c,0x07a2e7c5),LL(0x32e83f25,0x591eb066)}},
   {{LL(0x957c94fa,0xaaa61588),LL(0x364911fb,0x6a2bc707),LL(0xc4907b19,0x09771450),LL(0x9694ccc4,0x4cc48773)}, {LL(0x50c878ac,0x9db6216e),LL(0x6f3031f1,0x6e89210c),LL(0xced0d41e,0xb711dcbf),LL(0x0fbf9751,0xe39bfe3e)}},
   {{LL(0x764636b5,0x18fd7a45),LL(0xb75d48f3,0xe437ee86),LL(0x60a80177,0xe323bb18),LL(0xbc94c0ea,0xedc3c8f3)}, {LL(0xec8cb0cf,0xd8351164),LL(0x2472936d,0xccdd8829),LL(0x58059756,0xa8db1b85),LL(0xd55c184a,0x4eda8cf8)}},
   {{LL(0x2923b8cb,0xdfb5727d),LL(0xe6773d5e,0x6e793e5c),LL(0xa0641165,0x8ecc901b),LL(0xd6da5095,0x6077ab26)}, {LL(0x6b127d9d,0x00669b0c),LL(0xd63e2e1f,0x8140e4e0),LL(0x9641b6a2,0x1ad5b03c),LL(0x9baed7b0,0x44299f88)}},
   {{LL(0x1ea4a056,0x1736296d),LL(0xd77811ba,0x6f74702c),LL(0x432dd74b,0x5c927548),LL(0xe7a194ab,0x9cc73271)}, {LL(0xd6328dca,0x0f035ede),LL(0x28db755e,0x5292aa39),LL(0xa0192a4a,0xb5488385),LL(0xdfc6895c,0x6e7d2fa8)}},
   {{LL(0x5d8bbec9,0xfa912a0a),LL(0x0087edb3,0x7051140a),LL(0x64865e5b,0x5293672b),LL(0xc82c48d5,0x6e8448c9)}, {LL(0xa2c437b3,0xeece41cb),LL(0x21ce1ef4,0x148967d2),LL(0x6b05c2a5,0xf14391fa),LL(0x8fed2f1f,0x15ff5fc9)}},
   {{LL(0x4557b49f,0x18ae5e74),LL(0x3db266b2,0xe33760c6),LL(0xb1b249b5,0xd5d830c7),LL(0xc5fff531,0x24c665b9)}, {LL(0xc57df7c0,0x6b304406),LL(0xc3958e89,0x59706667),LL(0x790a5483,0xbf590ff2),LL(0x5ce77aaa,0xbcaea5a5)}},
   {{LL(0x80ceb559,0x8578a002),LL(0xd8d61946,0x3639aadf),LL(0xadd3bb00,0x3fd52d94),LL(0xe09a8ce3,0x16c27846)}, {LL(0x294c7967,0x75cfd6c6),LL(0x59195034,0xfb9b7f37),LL(0xaa972a86,0xae687a99),LL(0xebd2394e,0x04bdefdb)}},
   {{LL(0x2f96144d,0x8e245a19),LL(0x3b61e5ab,0xc740d348),LL(0x293ddb25,0x8703710e),LL(0x2bbf8f63,0xf4bb6ac0)}, {LL(0xde3b5805,0x86396457),LL(0x65d29e63,0x607022db),LL(0xcc930fe3,0xad0a0cdc),LL(0x1626abf6,0xd9997ebb)}},
   {{LL(0x2a510565,0x2d872d17),LL(0x0357ba07,0x3e682079),LL(0xebfaf203,0x49edd962),LL(0xf81eda20,0x3a13edfb)}, {LL(0x7a75f2d5,0x87b5b5e1),LL(0xddfd9511,0xf04de2b8),LL(0xcfc5c5ff,0xf29a1569),LL(0x07160ed3,0xa3995532)}},
   {{LL(0xcb2b061b,0xb6247469),LL(0x2f10fe1e,0xe75c5351),LL(0xd20e1bf7,0xbaf44963),LL(0x2d93babf,0x216cb6ab)}, {LL(0xf5109e45,0x7e0b655c),LL(0x6657450d,0xdcc712fc),LL(0xd51fc733,0xe06c408e),LL(0xed9c0912,0x85b11f96)}},
   {{LL(0x37365c9b,0x954cb91c),LL(0xb2f74fe7,0xe0eaa047),LL(0x15716541,0x9af74b86),LL(0xf73dc7bd,0x4da06207)}, {LL(0xe07890a1,0xdb0d089e),LL(0x73902f91,0x5bf09681),LL(0xa897f0fe,0x14e1710c),LL(0x3605b1c2,0x191ec9a1)}},
   {{LL(0x0133903b,0x271b2e2a),LL(0xe495ee32,0x5b3686f2),LL(0x0c991f28,0x89bcc974),LL(0x34f93b8a,0xadd20cce)}, {LL(0x680b65b6,0x5f5a1768),LL(0xaad41c40,0x0c453ab8),LL(0xa7fb4269,0xd479630f),LL(0x52c4e929,0x60039d01)}},
   {{LL(0xff860883,0x0d8d112c),LL(0x723c6e29,0xe1dce5c9),LL(0x191ad70e,0xc19eadae),LL(0x62ce0e64,0x4af8194d)}, {LL(0xcc81415c,0xf207bfb0),LL(0x008495c8,0x3ab92f3b),LL(0xfdb9534b,0xe7250e17),LL(0x6c0c1d1c,0xba67e9b8)}},
   {{LL(0x072c793f,0x117ae3ff),LL(0x9fb3091e,0x5243e6ea),LL(0x31a59e39,0xf93ad514),LL(0xc93c1891,0x8ce9cfb0)}, {LL(0x1ed08b0e,0xbfcbf901),LL(0xb53d687d,0x4d13cf2a),LL(0x5d81e4ad,0x25aa82db),LL(0x63c3cb41,0xd12f01f5)}},
   {{LL(0xf8d1333a,0x1e799084),LL(0x653bcd0a,0x30c96c55),LL(0x44b5195c,0x9cf130fd),LL(0x13c77763,0x4cffc531)}, {LL(0x9430619f,0x082287f8),LL(0xb08ce0d9,0x78bb037d),LL(0x3affe8e8,0x2e69d512),LL(0xba9ec693,0xe9dbb263)}},
   {{LL(0x62f132b5,0x67b66ad8),LL(0xbeb47184,0x70318d2b),LL(0xf50a0e98,0x46c429ea),LL(0xe2b3542c,0xd7e32eba)}, {LL(0xe096b4b7,0x625c1ce9),LL(0x389fd4dd,0x09221351),LL(0xfb0ee85a,0x08dc02d2),LL(0x853cd901,0x98c0ba7d)}},
   {{LL(0x0deb1d99,0x88a0cd6d),LL(0x79a6b90c,0x989e4962),LL(0x24dd89d5,0xf5d19b95),LL(0xb37cf19e,0x189e5230)}, {LL(0xb0c5fefa,0x84a607b8),LL(0xd8c7fbd1,0xe48450c9),LL(0x46479ad7,0x178f9b56),LL(0xcbcd2ae5,0x7d6a36c6)}},
   {{LL(0x71ae6516,0x95a4d51f),LL(0x566e2171,0x0363349f),LL(0xed1f2fc7,0x4d4bb4b0),LL(0xf10fa10c,0xde435aaf)}, {LL(0xb76e3b6e,0x711258a9),LL(0x2792e0b3,0x9a640eeb),LL(0x5fab8617,0x7953ead8),LL(0xdd64702a,0xd4b6d248)}},
   {{LL(0x2d672209,0x95bbe528),LL(0xb6926b8a,0xfcc53cfc),LL(0x57659f87,0x05814190),LL(0x08d25069,0x4836e93b)}, {LL(0x6a5ad81e,0xd1eb2006),LL(0xaf0d37f8,0x4bee145a),LL(0xd31ce6cb,0xd44362ad),LL(0x936c1060,0xdc03e581)}},
   {{LL(0x16fcb889,0x13cffce9),LL(0xac7e709a,0xed7e6683),LL(0x5896e541,0xb655d098),LL(0xb92a6204,0x07124356)}, {LL(0xa8f50043,0xa2ae43c8),LL(0x68731891,0xeb39255c),LL(0x3d9c408b,0xe07be0ad),LL(0x0b4f5c3a,0x0db7904f)}},
   {{LL(0x4d70bb81,0x7ddc0235),LL(0x5347797a,0xe3b323c3),LL(0x3536deee,0x3536cd9d),LL(0x001bfd25,0x579b6894)}, {LL(0xebe2922e,0x58ad5301),LL(0x92a88d43,0xe0aa2cae),LL(0x4409e205,0x24567a3b),LL(0x2258f0cb,0x3cece61a)}},
   {{LL(0x3babf4f6,0x8da5cf46),LL(0x81fff8e6,0xb37428d9),LL(0x48495d23,0xcda1ff77),LL(0x34f392ad,0x98f9208f)}, {LL(0x5bc88514,0x931f5b37),LL(0xcb375921,0xd49971be),LL(0xb5c01fab,0x9dcd4986),LL(0xc1ab1c94,0xcc26ec02)}},
   {{LL(0xb4b874d6,0x34e8087d),LL(0x9d0a3761,0x224fc277),LL(0x3f7e5159,0xacc1f258),LL(0x8966d593,0xc82d71ec)}, {LL(0x7dcd691a,0x5b1f9f40),LL(0xba28f416,0xd8fafdae),LL(0x43b6d90f,0xe8622ae6),LL(0x9ec71d5b,0xec13fce7)}},
   {{LL(0xfd2e8214,0x07b6aeb8),LL(0x4cbc297a,0x813e718e),LL(0x81fd6931,0xfac0dfab),LL(0x3c48ffd7,0xa1fe8821)}, {LL(0x85e03c08,0xd2715c18),LL(0x977c57f0,0xb6e4418a),LL(0x73418cde,0xfaa79ea4),LL(0x171e2a89,0x6ab8c25b)}},
   {{LL(0x4ec7cf05,0x2800445c),LL(0xb66c6200,0x8e74a7b0),LL(0x481db950,0x081b1177),LL(0xb89f7c02,0x526d051c)}, {LL(0x5c29c905,0x3c830942),LL(0x44c15ce5,0xbfbd9e3e),LL(0xa29472e6,0x6055c949),LL(0xa37c4912,0xab0010c7)}},
   {{LL(0x5b7d3647,0xeb8492be),LL(0x1ee31caf,0x0b4cfd7b),LL(0x4b46304b,0x81cfcde2),LL(0xc554a5bc,0x968df75d)}, {LL(0x8d0e043c,0x7ce78806),LL(0x345ea27c,0x1e896819),LL(0x6e287603,0xe040c19c),LL(0x138e8ece,0xa581856f)}},
   {{LL(0xc354a9d6,0xe49f6558),LL(0xc0cfb2d3,0xc4ad763a),LL(0x1b76b8f3,0x4be2143b),LL(0xd0ad0247,0xa8caae14)}, {LL(0x928b0ae5,0xcfe96bd5),LL(0x7724f8e4,0xcf5051f7),LL(0xec4af64a,0x9128916f),LL(0xcb437bfb,0xc211ff4b)}},
   {{LL(0xbce59c0f,0xee6e8134),LL(0xd59f7f86,0x3d068b4c),LL(0x96283457,0xafa2753c),LL(0x1aedcbf0,0x453fe33c)}, {LL(0x483c0b1a,0x781294c8),LL(0x5c2ad1ee,0x9e6f5133),LL(0x69383e0b,0x2a77b6ce),LL(0xfa9f0142,0xcb5a83ab)}},
   {{LL(0x3b0e027f,0x2318aa98),LL(0xc2c68dd5,0xdea716a3),LL(0x9f548eb3,0x3f75c46d),LL(0x96120de9,0x71642513)}, {LL(0xdbee488e,0xf733614c),LL(0xaad077f4,0xdf940026),LL(0x94a840cb,0xeda9c098),LL(0x393be3b9,0x5108bf0b)}},
   {{LL(0x39980cee,0x137c08b0),LL(0x0839112b,0x2e31bba0),LL(0xba614ea3,0x9ec73de2),LL(0xd17822c0,0xd0bca8d4)}, {LL(0x50b7805d,0x5d9f7482),LL(0x298becf6,0x16035a80),LL(0xd7c318e7,0x46571500),LL(0xd0ee6956,0x6bd30919)}},
   {{LL(0xb2e13320,0x5c0ad747),LL(0xda47666d,0xe7f7f71e),LL(0x318a8e8e,0xce322037),LL(0xe9f84dd6,0xf15232ae)}, {LL(0x915a03b7,0xc59709c5),LL(0x9a3040b4,0x2e2000f7),LL(0x8398a5a9,0x41955f77),LL(0x7086b69e,0xa8e6620e)}},
   {{LL(0x8344224b,0x63acd70e),LL(0xc3145159,0x966efefc),LL(0xf5e0f955,0x406619ec),LL(0xec6de618,0xedd0efc9)}, {LL(0xb2580ed4,0x6fe3e34e),LL(0x4139b95e,0x9d8875b5),LL(0x8e5be187,0x85baf0c1),LL(0x09553886,0x549cefca)}},
   {{LL(0xae9ef2cc,0xc965b2a2),LL(0x15afee63,0xd43079fb),LL(0x076cdb05,0x02b8794a),LL(0xa0d1a953,0xd0ae7321)}, {LL(0x2ac5fff0,0x5a8b5281),LL(0xcdda362d,0x73437d67),LL(0x1a95ff87,0x1866b2b9),LL(0x0420b3e1,0x5ff11398)}},
   {{LL(0x92284adf,0x0d43b92c),LL(0x4da4c4a7,0x81425367),LL(0xdf17641a,0xc8093c56),LL(0xb5ccd14d,0xc418f19d)}, {LL(0x506762ed,0xaad98608),LL(0xddb2c829,0xb6f45297),LL(0xd395692a,0xd0e49176),LL(0x3b1073d3,0xc05b4e27)}},
   {{LL(0xe5808e53,0xe8ca133b),LL(0x06a64b56,0x6105cd0e),LL(0x53cf6d7e,0x89a64669),LL(0x1bebfea5,0xe281ca2d)}, {LL(0x324b25d8,0x98ee67ac),LL(0xdca154ec,0x2227631f),LL(0x4406e8ba,0xa242c5a1),LL(0x49250026,0xced39f05)}},
   {{LL(0xdd77d731,0xd256dd83),LL(0x7414d0c0,0x2faa6a0e),LL(0x3b90f004,0xa2e0f928),LL(0x8719bfd4,0x019bb3ef)}, {LL(0xe2d515c2,0x3f4f6109),LL(0xbf88d7a6,0xb50a9907),LL(0x015ac4de,0x8e5fbc2d),LL(0xe78a2117,0x96992421)}},
   {{LL(0x26e53df3,0x321e6086),LL(0xf42b2508,0x07eb1d15),LL(0x0ef22bc2,0x7b552108),LL(0x00f3e571,0x9eedb828)}, {LL(0x6f0e883c,0x556abbaf),LL(0x40473ead,0x8025770b),LL(0x6ece1cc8,0x2fdab965),LL(0x00ec1adc,0xba07cf89)}},
   {{LL(0x4be5ad18,0xefec4deb),LL(0xd59fa732,0x16625be8),LL(0x6808cdf7,0xffee542e),LL(0xd7a9f29b,0x85c19ef3)}, {LL(0x82dc1ae3,0xca4ac1f9),LL(0xca5e8f58,0xa6c726d1),LL(0x66960edd,0x0bcc3d58),LL(0x56f94ea8,0x8e8445d0)}},
   {{LL(0x938e64c9,0xd4d0177b),LL(0xf9a0288f,0x8d0199f1),LL(0x14a226c0,0x9176d559),LL(0xa00aea02,0x13b373ee)}, {LL(0x6b629fea,0xc63b2d79),LL(0xa7e0cc42,0x36df7c09),LL(0x40bdbc8e,0x4628ba4f),LL(0x0de296f2,0x7f5b0228)}},
   {{LL(0x3c63d73f,0xb0598130),LL(0x0431550e,0x55e59f61),LL(0x6693eb8c,0x6f2e109d),LL(0x470b10fe,0x3602ba82)}, {LL(0x5ec7f357,0x3acd0af4),LL(0xb071c758,0xfa7479f4),LL(0xe13652c9,0xbf47caa0),LL(0xf5f5eca9,0x6fa139bb)}},
   {{LL(0x8c0e197e,0xfa149b84),LL(0x60ae7755,0xca31714c),LL(0x8ccc4241,0x934ed1af),LL(0x781a024e,0x39772769)}, {LL(0xbe24eb34,0x9f07dfb1),LL(0x0a3dac06,0xfa8a9c60),LL(0x8e410ce7,0x08fbbe21),LL(0x396a9702,0xea55fb96)}},
   {{LL(0xf18882bb,0x4422bc58),LL(0x0ddd0dd7,0x1ccb7b47),LL(0xf40ea941,0x828580a8),LL(0x0db78350,0xf9ec9728)}, {LL(0x1612f28a,0x2823b4fd),LL(0x82b26487,0x96dc3e29),LL(0x2497420a,0x1740fdae),LL(0x322f1c6f,0x3bb39dfa)}},
   {{LL(0x4cb19878,0xf32a21e6),LL(0x9277c80b,0xeac04097),LL(0x13380801,0x67178d8f),LL(0x34bf8872,0xfe5e2694)}, {LL(0x327129d3,0x8278bad4),LL(0x941c4e5c,0xb42a3f9b),LL(0x39de36f0,0x04eefb7d),LL(0x8d967703,0xed2aab7f)}},
   {{LL(0x72aa1c89,0xa3283a2c),LL(0x2a4d513e,0x1969613e),LL(0xddd5ea18,0x0d4c0347),LL(0x43cee5fe,0xbbad9ce4)}, {LL(0x57313b49,0xe8c050a8),LL(0xff09bf31,0x3b91c3cc),LL(0x610395cb,0xe6e5ab6d),LL(0xdeb31bef,0xfc36cde0)}},
   {{LL(0x5d43c8a7,0x76f806f2),LL(0x63b7c746,0x08a64cb2),LL(0x45256a2a,0xb6cdcdab),LL(0x9bebae90,0x640f67ea)}, {LL(0xcf016971,0x682eeeb6),LL(0x50f59261,0x4d16d566),LL(0xf41db99d,0xdaca66bb),LL(0xf8f04d96,0xccdb3da0)}},
   {{LL(0xcf41b164,0x7c228cae),LL(0xedbefa7c,0x40bef27f),LL(0xecb43685,0x4efdd6c2),LL(0xa834a50b,0x4d0fa367)}, {LL(0xb87f7ec7,0x2ec9c445),LL(0x23170d0f,0xc3663ced),LL(0xc5b47b29,0x189872e4),LL(0x746d6a13,0xf8047387)}},
   {{LL(0xb75ac898,0x753837d3),LL(0x91959a78,0xaee88a60),LL(0xe6f59621,0xf46b0f6e),LL(0x10d981c8,0x0e92e271)}, {LL(0x8d578b6d,0x610d0f80),LL(0xb4d9b9de,0x962bd7bb),LL(0x84a0c394,0xbe26960d),LL(0x3b5bd996,0x142a0c75)}},
   {{LL(0x0be95497,0x442bb39a),LL(0x0f33c9de,0xce5d2c60),LL(0x283dc751,0x1ce0d08c),LL(0x79b3c1a8,0x106ed588)}, {LL(0x7f8ee4d7,0x4b2e29c6),LL(0x08bbd660,0x7d61e3bb),LL(0x1e964a3e,0x11504dc5),LL(0xc77a957a,0x31544a52)}},
   {{LL(0xcd7d0dac,0x1fc4161e),LL(0x370c15c9,0x83733f27),LL(0x853758cc,0x224976de),LL(0x47c1ab78,0x1bbb7730)}, {LL(0x19c89029,0x94a3b697),LL(0x37dfc44f,0x031432f0),LL(0xd88090cb,0xf84593ac),LL(0x65bcfee8,0x381b51bc)}},
   {{LL(0x10b412b7,0x38dac75b),LL(0xc7e06d08,0x6df5c9a1),LL(0x0e08c41c,0x9c6d8068),LL(0xc3600f4f,0x1544e3c5)}, {LL(0x9c83e0a1,0xf827a48d),LL(0x06bcb3c4,0xd8539228),LL(0x6268cf12,0x149862b3),LL(0x6ec4e354,0x4829ee56)}},
   {{LL(0xb712a1f9,0x44b2c3bf),LL(0xc90852af,0xe556b78a),LL(0x906a13b6,0x50f6de2e),LL(0x568a1293,0x1744efd5)}, {LL(0x2b5745a1,0x942ad99e),LL(0xca362913,0x0f100bd9),LL(0x91e96cde,0xd9b6ad51),LL(0x5a2f88e9,0x4aa440bc)}},
   {{LL(0x57a10070,0x53c4c956),LL(0xae6e4872,0x7d1be72e),LL(0xd427eda4,0xb704009c),LL(0x5f39b7d8,0x3e0aa93f)}, {LL(0x3153a757,0xdea1ab48),LL(0x9ee60ead,0x10a070e7),LL(0xe6c916bf,0xd6a6e92d),LL(0xbd7bb216,0x02b1e0e6)}},
   {{LL(0xb49138a3,0x6efb5f1b),LL(0xe88d2db0,0x11f7a9be),LL(0x3233df5b,0x0b9a2b11),LL(0x1824fcc5,0x0688afda)}, {LL(0x5ff97f9a,0xcf1ea2a5),LL(0x4998e602,0xe8ad7b15),LL(0xa455aad1,0xdb4ae67e),LL(0x74a27ff3,0x823ac090)}},
   {{LL(0x2573443f,0x5c431060),LL(0x94258714,0x92f9f9ab),LL(0xb1283d2e,0x1548fe21),LL(0x5c5be5f9,0xf86fe50b)}, {LL(0x520c5fc6,0xd20dfc8a),LL(0x53b5e7c5,0x6e721dd9),LL(0x8f2a8969,0x8ef7eee5),LL(0x62d07bdf,0xe894859f)}},
   {{LL(0x1cfc6627,0xaf279176),LL(0x483755e9,0x94b8cff4),LL(0x0fda4bcb,0xa5916f70),LL(0x47ba65f3,0x9c5318d0)}, {LL(0x636cd7e3,0x9e9c8e54),LL(0x54c49da3,0x5c64a261),LL(0x690e932c,0x04d7ff61),LL(0xc84b0b78,0x92a357b3)}},
   {{LL(0xc6f3bd8d,0x47f6144c),LL(0x71c19265,0xdf7b1ee4),LL(0x3fd5c30f,0xa7ea37f1),LL(0x79fa08cf,0xdc2d890b)}, {LL(0x2fd40236,0x9813bced),LL(0x432dde17,0xa8a1055f),LL(0x7772c727,0x70011f47),LL(0x2e2e802f,0x965c130a)}},
   {{LL(0xf5bd4ac5,0x31a6aca7),LL(0xd825db6f,0x83995bde),LL(0xfe521833,0xcbf20325),LL(0x0278f4a0,0x8dcd25a1)}, {LL(0x5f2293ea,0xf1e83d97),LL(0x52317ad3,0x1717876b),LL(0x14181928,0x0df62167),LL(0x2fe203ce,0x24adfd6e)}},
   {{LL(0x797f25ff,0x1d264af0),LL(0xd22e3da1,0x2cb7cc17),LL(0xe0016a19,0x10c4b51a),LL(0xd82b2a86,0x5956ce8f)}, {LL(0xa3d4780e,0xdef0fefc),LL(0x6e92b93a,0x97e693ab),LL(0x20bcc98f,0x8fa3f4fa),LL(0xf9182867,0x4fc004f1)}},
   {{LL(0x93e73961,0x1a206da3),LL(0x1e7db32c,0x37d75a90),LL(0x0455b815,0xa39f0db1),LL(0xb69ee770,0x841646e0)}, {LL(0x0939f065,0xadb0aaaa),LL(0x0b147d7a,0x5608613b),LL(0x470f6875,0x84ce1a4c),LL(0x7983050e,0x501de5fe)}},
   {{LL(0xc3d8ed98,0x19915b26),LL(0x9a66a6e5,0xf451e57a),LL(0x30dab6a3,0x29843607),LL(0x3d1a1ebb,0x1710267c)}, {LL(0xe11d88c0,0xce4ecfd4),LL(0x11ce026a,0x12fc2787),LL(0x691227de,0x9801cecd),LL(0x76ce6dae,0x517a92f3)}},
},
/* digit=25 base_pwr=2^175 */
{
   {{LL(0x648c48e5,0x821b0fdf),LL(0x9f45a433,0x689e6d56),LL(0x2e572855,0xa5a9dca8),LL(0x8adfb547,0xb0f07eb7)}, {LL(0x552c8d55,0x48ecb166),LL(0xce854171,0xfe3fc268),LL(0xeeee9bc0,0x323af5eb),LL(0x41ae1c80,0x0666a2a3)}},
   {{LL(0x9ff262fb,0xa06d20bc),LL(0xd075868b,0xcba032fd),LL(0x943fd973,0x70376026),LL(0xe35c5e02,0x81c57cba)}, {LL(0xba871f1b,0x1964e700),LL(0x6b265f57,0xf03a8c04),LL(0x0b950259,0xc8ebc912),LL(0xad32ca8b,0xd2b0ee30)}},
   {{LL(0x89c8e719,0xe01bf0c2),LL(0xb2f4cdb0,0xbce1e20f),LL(0xa1053ca5,0x8c38eeaf),LL(0x7cd633a5,0x8c2d85ef)}, {LL(0x9b53cdb1,0x75695364),LL(0x447eb1a5,0x5e999741),LL(0xdbd88626,0x6d6b2d88),LL(0x21876357,0x87eaf045)}},
   {{LL(0xdeec441e,0x2c88f1ff),LL(0xd01b2157,0xab52096b),LL(0x6c45cf5c,0x37eee275),LL(0x0520ecaa,0xa070d24e)}, {LL(0x546b9fd3,0x61d15bd1),LL(0x2c96db1c,0x3276fb74),LL(0xb95b29b7,0xc5c1b041),LL(0xbd7d3254,0xe18008db)}},
   {{LL(0x98dfb69a,0xd56ae44f),LL(0x609d941c,0xd5f66b0b),LL(0xb91b5245,0xca6b6d35),LL(0x7b3f98a6,0x98e3a4e3)}, {LL(0xf358c56a,0x0715dfa6),LL(0x36a66c64,0x3b02ff21),LL(0xcb22cbd3,0x737b1401),LL(0x6b8e9624,0x9dd15f5b)}},
   {{LL(0xd360d017,0x25f5a71d),LL(0x29b0ed73,0x4c0779b5),LL(0x9825a018,0xc662fedc),LL(0x61d4add0,0xeee89125)}, {LL(0x92163d14,0x1543814d),LL(0x27370d3c,0x79f2376f),LL(0xcbe1af7a,0xf80c6963),LL(0xeb9e41f7,0xf2d521bc)}},
   {{LL(0xc1805864,0xe241619f),LL(0xb2de204a,0x6f1d6166),LL(0x50e68d0b,0x13c3f912),LL(0xc4a24f5a,0x32eb021d)}, {LL(0x0e78c588,0x3f1452f5),LL(0xc9971e98,0xa267bf19),LL(0xe801c021,0x77a231a7),LL(0xc2666e80,0xf363c9b3)}},
   {{LL(0xae309a0a,0xb8eb0bf0),LL(0x375b8fbc,0xa9f52f58),LL(0x1a4993b7,0xb8e4f948),LL(0x8f73c292,0x50ce578e)}, {LL(0x02e503d6,0x2437a4a6),LL(0xe4c68ea3,0x20cdfc50),LL(0x3616f348,0xfec5993b),LL(0xc0c420df,0x5d96b4c5)}},
   {{LL(0xcca0c48a,0x6b71cdf0),LL(0xc1aea033,0x00e71b54),LL(0x69f6d107,0x247f5baa),LL(0x050c3704,0x4e3ec504)}, {LL(0x7a438e93,0xf2b2be8a),LL(0xa331e01e,0x240b632d),LL(0x91150675,0x61e66557),LL(0x95a99a28,0x32364134)}},
   {{LL(0xd3399e1e,0x5e5de136),LL(0xfe2f8b75,0xe38bab00),LL(0x3a77db29,0x736126de),LL(0xf2aa121e,0x7b0d1865)}, {LL(0xdecf9cde,0x5545e45e),LL(0x2318be70,0x9608ebce),LL(0xfa55b0e5,0xe6596006),LL(0xbc4b6ca0,0x0c8c2f41)}},
   {{LL(0x92025945,0xda1c5c7a),LL(0x5d3b0775,0xb114ba22),LL(0xcedb69a0,0x11cc6888),LL(0x0f83c772,0x4365bea8)}, {LL(0xbda8dbe3,0x006fe80b),LL(0xc2d3d266,0x334adcb6),LL(0x1521de1c,0x8c92c084),LL(0x78d8f72c,0x57873ef9)}},
   {{LL(0x3b64dcd7,0xcfb0a7d0),LL(0x558c9d55,0xf4c2f1fc),LL(0xa0fbc656,0x110c2db2),LL(0xef5b6bea,0x3cad85ca)}, {LL(0x4e0b1230,0x7099dd0e),LL(0x098a2fcd,0xc769b937),LL(0x1e1e7407,0x9209f550),LL(0x1ba7cb47,0x1b47255d)}},
   {{LL(0x2c01b596,0xd8aed0cd),LL(0x30efcda3,0x1a1a2e11),LL(0x36b1a5b5,0xf771f93b),LL(0x14fcd251,0x2ea34e3d)}, {LL(0xfd893979,0x6895cb31),LL(0x14f556b4,0x10b1d2c9),LL(0x6430bfa8,0x835fdf7e),LL(0x24bf4ba5,0x1f4bbef5)}},
   {{LL(0xd562b5f1,0xbc805aa5),LL(0x35dac658,0x7101b9da),LL(0xddc28e5a,0x5b7f211d),LL(0xd3d1cd0a,0xea89f24c)}, {LL(0x7567c80d,0xbaaa9ef5),LL(0x9a60c5ee,0xe0d1f26d),LL(0xab36cd64,0xc88a044c),LL(0x1b125de6,0xb8e03d02)}},
   {{LL(0x3a707a66,0xda0c1047),LL(0x0c692d44,0x76ddb98f),LL(0xb15b7254,0xeccae586),LL(0xe7e82423,0xeadc1b51)}, {LL(0x7c3cb698,0xd6779ff2),LL(0xdf6e7da6,0x0e271cb4),LL(0x45900023,0xeacf34c3),LL(0x03da2ba5,0xafd017ad)}},
   {{LL(0x27c7e6eb,0x49266998),LL(0x6625bc7f,0x84ffa372),LL(0x05c9cb15,0xedec9247),LL(0x8075b84e,0xcfad0b90)}, {LL(0xbc0898d3,0x94bed316),LL(0x11f92525,0x02481eec),LL(0x0d7e59d7,0x19896e1b),LL(0xf2bb3129,0xa06adb6c)}},
   {{LL(0x62a0a690,0x1539228e),LL(0x8ae02bda,0x98ac50b9),LL(0xe5cf21b9,0xaf233c85),LL(0xd6a9f599,0x943348d3)}, {LL(0xdb363eaa,0xf5a2f2d1),LL(0x7a8ea96b,0xe917e2c5),LL(0xbf5c8293,0xc80b56c8),LL(0xcdbb5c4f,0xcfc1c24f)}},
   {{LL(0xfbddf07b,0x7812dce2),LL(0x0186013a,0xd4af2f9b),LL(0x6fe8d485,0x1fadcd16),LL(0xc968f0b7,0xc3c2cd95)}, {LL(0x778bff58,0xdbdd2ef0),LL(0x8706da34,0x67369204),LL(0xb8e70e35,0x31cf3a66),LL(0xd333801f,0x0b9e5cc5)}},
   {{LL(0xf7177c4a,0x1212a811),LL(0x2d379e12,0x9e266ec3),LL(0x2e8bbbf7,0xc7382848),LL(0xa973be5f,0x3f3f1dc1)}, {LL(0x786e727e,0x534d04de),LL(0x225497dc,0xfd7a5fbb),LL(0xb63b6320,0x3c03a7fd),LL(0x5dc76e05,0xe77f8985)}},
   {{LL(0x265f8b8f,0xe8d14f32),LL(0xb90549c9,0xfeaab021),LL(0x081ccea6,0x7cd36751),LL(0x1f1e8f7a,0x7a001441)}, {LL(0x1fdfd622,0x2e87b8a2),LL(0x8bb4f812,0xe76138ce),LL(0x71e03be4,0x9a5e8722),LL(0x153e0a5f,0x795e60f3)}},
   {{LL(0xd0eb7d4c,0x11d28438),LL(0x4254a845,0x147884e1),LL(0x2a8693fb,0x6795f20f),LL(0xee01bd1a,0x5a10d535)}, {LL(0x218c0513,0xe39982c9),LL(0x1d4e6ab5,0x6c23e5be),LL(0x0f424e7c,0x20a8c27f),LL(0x3bbb486f,0x74ae9998)}},
   {{LL(0xb90ce3a1,0x3fae61be),LL(0x571c968b,0xf0f5a1e4),LL(0x7780d91b,0x6b9dded8),LL(0x7597e866,0x10f60ce2)}, {LL(0xf1eb7d1c,0xf268ed02),LL(0x6030bf9b,0xa49b5a46),LL(0x251f8676,0xc939c4e7),LL(0xe2b9928f,0xbdfe5036)}},
   {{LL(0xbccf7f43,0x5abfbcc2),LL(0x28c560af,0xb22067b6),LL(0x04c6a2da,0xecf07771),LL(0x8c4ae7dc,0xa2bf88db)}, {LL(0x616675e8,0x172396f2),LL(0x8bfcfbc2,0x9abbb19c),LL(0xe85edd21,0x52e26c06),LL(0xa65de52f,0xfca4c4e0)}},
   {{LL(0x281d58be,0x255e2d10),LL(0x3614ed6c,0x93ec2934),LL(0x6dc71abe,0x36d6cc15),LL(0xf592ae78,0xaa2ad1ef)}, {LL(0xcc9291fb,0x39a82740),LL(0x102600d8,0x6812b80f),LL(0x50c511d5,0x64f4843c),LL(0x03445793,0x28f5795e)}},
   {{LL(0x29f20b0c,0x2c566372),LL(0x9e24474c,0xb168ca7a),LL(0xabe37949,0xfadd8f80),LL(0x4cd69856,0xafa1bea2)}, {LL(0x46713b88,0x5ce6ed80),LL(0x4b3bb84d,0xaf8b5fb3),LL(0x29d53349,0x134e5120),LL(0xcdcedefa,0x1214f4f0)}},
   {{LL(0x4bb405b9,0xc346821b),LL(0xddd624d6,0x753afa86),LL(0xc7c014e3,0x15fe543c),LL(0x43d08964,0x6b3c0c5d)}, {LL(0x745221aa,0xc77c8fb0),LL(0x152995c9,0x3e144fce),LL(0xa61b96bc,0x57089907),LL(0x5e05c1ee,0xd19a912c)}},
   {{LL(0xa6ddd20b,0x7bcdc697),LL(0x2d5090f3,0xcb07e229),LL(0xf089607e,0x76afc803),LL(0x637dae27,0x9f7a88b9)}, {LL(0x3bd20d78,0x1352d8bd),LL(0x5ea79d4c,0xede1a780),LL(0xf389e31d,0x59a8222b),LL(0x5c09f3d4,0xed066aa6)}},
   {{LL(0x684529d3,0xade16197),LL(0x96a2a159,0x97bed904),LL(0x1b695d68,0xdd3da765),LL(0x02fecb9e,0xb8fa37e8)}, {LL(0xbc0f7b99,0x1af4311c),LL(0x2a492a7e,0x600bdd46),LL(0x45dc9d16,0x6aa9cb30),LL(0xc0b93019,0xaa494815)}},
   {{LL(0xba052dd8,0x1211834b),LL(0x86822bf1,0xcdc0208e),LL(0x8c8362a0,0x515eebd4),LL(0x9b90cf96,0x9ea7b9f5)}, {LL(0x3a0a5a48,0x8418fe34),LL(0x331a2db1,0x654d3c32),LL(0xafde743c,0x22362ddf),LL(0x6f6ee3ba,0x617a89e8)}},
   {{LL(0xb7deb988,0xed5f3d04),LL(0xbbc8a6b2,0x31c2c9e6),LL(0x81a3f184,0x8faa80e1),LL(0x51ecc548,0xa7183488)}, {LL(0xa3780d0a,0xe67512d0),LL(0x822db54d,0x9f868036),LL(0xe555beab,0x6c74490a),LL(0xd989d6be,0xe747e666)}},
   {{LL(0xdf8cd308,0xf8346dd6),LL(0x4745cd8e,0xe7ca105f),LL(0x31055db8,0xee059c58),LL(0x18b38aa0,0x90f4053a)}, {LL(0x41081a21,0xbb2e7fc3),LL(0x45b33a71,0x3602525e),LL(0x2b411945,0xff21f2aa),LL(0x064ccb11,0xbeaadbd3)}},
   {{LL(0xfe94629d,0xc35f6950),LL(0x9f860b15,0x1cbaa935),LL(0xf24f8f15,0x29b4bcd3),LL(0xd29c8458,0x0ae5b06a)}, {LL(0x1b6c2df1,0xa645c31d),LL(0xd804facc,0x640b0990),LL(0x122b33e6,0x7a4a7f59),LL(0x7479b828,0x94bb0b2b)}},
   {{LL(0xc4cd4887,0x0567272a),LL(0xfc8e4b0b,0x676d6962),LL(0x8661c0c2,0xa712b020),LL(0x279454a7,0x660e6aff)}, {LL(0x1cd25bfd,0xe1295106),LL(0x077496a8,0x7096885c),LL(0x3006ab7b,0xdbc47c92),LL(0x509205f3,0x498761fa)}},
   {{LL(0xe85ecfee,0x5d1eaeca),LL(0x534f38f5,0x9fcddeed),LL(0x8af32f70,0x4d34ec80),LL(0x24b3b4e3,0x476dffc9)}, {LL(0x8bbcda9f,0xb45cd869),LL(0xdf684c2d,0x3b0079e7),LL(0x765cd483,0xcaf3eeb5),LL(0x63945b62,0x0b9e25e6)}},
   {{LL(0x06492e0a,0xfd73833e),LL(0x9507ea57,0x4d2937e1),LL(0xcf562953,0x3e532c2e),LL(0x81ca80c3,0xe4baa2d4)}, {LL(0x28d22139,0x4699e5c4),LL(0x6b1c665a,0x69aab900),LL(0x641016ac,0xf6a37330),LL(0x5f3b7c71,0x335f14cb)}},
   {{LL(0xfacd904f,0x94a6c868),LL(0x2ec2bf99,0xb1127cc4),LL(0xa4b72d69,0x0ccfceb3),LL(0x55172f5b,0x16b786a3)}, {LL(0xe093a729,0x51ebe029),LL(0xc40c4487,0xf57f4a1e),LL(0xa8ed5a3d,0x8aaf0dd6),LL(0x811f35d6,0x617c51f7)}},
   {{LL(0x11e98d29,0x18c7ac62),LL(0x2c904ea8,0x71c578c4),LL(0x3c4ef8bd,0x4626b0ad),LL(0xa390be8b,0x121eb676)}, {LL(0x154e297d,0xcb7249f5),LL(0xc2288ba0,0x678ad966),LL(0x57cc9cbc,0x3c2ab066),LL(0x80c8fbda,0xe32c1d45)}},
   {{LL(0xf0b35526,0xd2f152cb),LL(0x13877dfb,0xc7f75fd4),LL(0xe83ca4a2,0x8603feff),LL(0xcd680589,0x6be89bb3)}, {LL(0x45e1f141,0x5650549e),LL(0xa55ffadc,0x7dab03b8),LL(0x2dc5d31f,0x342edda4),LL(0x9af8105a,0xa97451ac)}},
   {{LL(0x705b8fd7,0x796e1fe3),LL(0x02d131b8,0x6252a7ba),LL(0x086c3099,0x3db2ab14),LL(0x9db0ce72,0xeb763df5)}, {LL(0x52b62fa5,0xe7b57bab),LL(0x88b820bd,0x6076d449),LL(0x1b660123,0xc43e1f61),LL(0x189eace5,0xc802d40b)}},
   {{LL(0x341309a1,0x1f2a2a91),LL(0x414db96b,0x8680be67),LL(0xc846e288,0x65dd0396),LL(0xb0bbea85,0x8a1d871e)}, {LL(0x8ff931c6,0x623e2408),LL(0xe14c5941,0x4933ffda),LL(0xb2cbff67,0x72688986),LL(0x8cf79450,0xe51504d8)}},
   {{LL(0xfeba1168,0x50cd0a3f),LL(0xcd833df8,0x08d2e0fe),LL(0x0a4370ed,0xdbd60827),LL(0x66f4f58d,0x010cf800)}, {LL(0x144e9656,0xffa29252),LL(0x9d1e9d61,0x90b896a2),LL(0x81f7b4d3,0x1802257c),LL(0x595612a5,0xd7758e8b)}},
   {{LL(0x241b4dd2,0x751882d8),LL(0xfe177abe,0x7dae3003),LL(0xee6fe1cd,0x8f4d5dc4),LL(0xb08f625d,0x93a9cd5b)}, {LL(0xf91cc442,0xa4d6ee1a),LL(0x594d172f,0xe05976cd),LL(0x6e762b2e,0xfb4064c6),LL(0x51a0156d,0xb2068ad9)}},
   {{LL(0x24f06e82,0x0d2d5b26),LL(0x8c85a9a9,0xad70f276),LL(0x0ed413cb,0x00ede5d5),LL(0x927064d2,0x245be28b)}, {LL(0x2af70d77,0x06eb2825),LL(0x52b0592b,0x472af630),LL(0xd881d50e,0x493afd98),LL(0x1189c989,0x56fa76a8)}},
   {{LL(0xfaa974f7,0x775665d2),LL(0xc3f54eff,0xe395ccdb),LL(0x3fc83a7f,0xf0a40e4e),LL(0x4c00087a,0xc3b11d22)}, {LL(0xddb50678,0xef8d2f06),LL(0x6cd5f681,0x6e41f315),LL(0xa1b97891,0x7c9d7a3d),LL(0xa0a41260,0x8b297d75)}},
   {{LL(0x3806a30a,0xca44b65c),LL(0x61a6206e,0x125c5702),LL(0x87003e1e,0x311842a2),LL(0x4513d726,0xe049a7d0)}, {LL(0x7b123469,0x8022c2d0),LL(0x86294393,0x76533934),LL(0x892e7bc4,0x6a6e84e7),LL(0xdb2007fe,0x7daf8b11)}},
   {{LL(0x923e185c,0x092d1914),LL(0x3def87c2,0x5ec11237),LL(0x18742a51,0x38019e96),LL(0x4808ca10,0xe05ea79e)}, {LL(0x134cbf9b,0x1fc8ae26),LL(0x01b4c1c4,0x14054672),LL(0x64051972,0x32abf912),LL(0x1af62fca,0x0edaa925)}},
   {{LL(0x3c47d01a,0x58fa82e2),LL(0x780762c6,0xdb12a452),LL(0xfdbf4683,0x16d5a733),LL(0x2f798deb,0x1d7e8507)}, {LL(0x9eab12f7,0x259aa9b9),LL(0x91261397,0xb13e6e41),LL(0x32602f2e,0x564706fa),LL(0x9c2283ef,0x50daef9c)}},
   {{LL(0xf07a196e,0x9275f219),LL(0xbb8fcd35,0xfc58ebea),LL(0xbad69c11,0x5d1025f1),LL(0x1605c11b,0xcf364154)}, {LL(0x2992276c,0x427bd117),LL(0x6a73cbb3,0x5545bec5),LL(0x133f3266,0x86855c2f),LL(0x67d9e5b2,0xb3d753d1)}},
   {{LL(0xc9fb343b,0x2134b384),LL(0xb0e12b7a,0xb572f5d6),LL(0x392d24fe,0x7ee5852f),LL(0xc4f285f9,0x73559fae)}, {LL(0x7711c525,0x269cb9e7),LL(0xf00d5606,0x4badfc52),LL(0xb465df15,0xef66d726),LL(0xaa4a301b,0x83eb59a3)}},
   {{LL(0xed329b12,0xdb406469),LL(0xd933eb45,0x6eb95cc9),LL(0x6b638bdd,0xe2dabfa4),LL(0x031df114,0x7a5d0098)}, {LL(0x38dbfaaf,0xe22d8f3e),LL(0xd79d1ce1,0x2306fd54),LL(0x7acb7cce,0xda324535),LL(0x88f61a1e,0xde6fcc16)}},
   {{LL(0xb730fe5c,0xaf3e4894),LL(0x28adf897,0x7a3e4a7d),LL(0xb160ae0e,0x352c0069),LL(0xee52c58a,0x225cfb67)}, {LL(0x96b2505f,0x12391b71),LL(0x3758141d,0x8d811bee),LL(0x8cd82e11,0xc941524d),LL(0xbb83a251,0x0feb26a5)}},
   {{LL(0x76da8411,0x60ad0665),LL(0x88d4a968,0xe3c033d9),LL(0xde120078,0x767b3c05),LL(0x9f31e1e8,0xab7da95a)}, {LL(0xb93e7cb7,0x7ad9b263),LL(0xfd626ea3,0x280f6bc3),LL(0x62713cad,0x746c3945),LL(0x3a4edce8,0xfa2b4599)}},
   {{LL(0x10ab7f93,0x8792d0cb),LL(0xc25a2a86,0xfa38d031),LL(0x08b028e8,0x6914db0b),LL(0x383cab40,0x75a98aa0)}, {LL(0x6da884bf,0x462e6b6d),LL(0xd3aa74b1,0x2b0f682b),LL(0x5cee0a83,0xb3b7995e),LL(0x3cce609a,0xe99fca2b)}},
   {{LL(0x45451744,0x342c41c9),LL(0xc81be29f,0xb00d3c24),LL(0x022e8d27,0xd1e64d86),LL(0xbcf67326,0x404550be)}, {LL(0xc8aab829,0xc7c510f0),LL(0xa90c605d,0xb61ae647),LL(0x02db8602,0x582ad9c9),LL(0x71cb4397,0x732b19ed)}},
   {{LL(0x265e5369,0xea097c35),LL(0x9d5ea687,0xea7c368f),LL(0x8fcae7f1,0x7fc3b213),LL(0x49c54942,0x641daa3f)}, {LL(0x404c39a6,0x0696372b),LL(0x87b4b181,0x56815716),LL(0xfca24eb8,0xa6e156b3),LL(0xd078a39c,0xf278eeae)}},
   {{LL(0xac762dbe,0x046566d1),LL(0x625ed2e8,0x662ef0f6),LL(0x650e4764,0x15499e72),LL(0x84edf50e,0x361ccef0)}, {LL(0x1f4a2200,0x2441f6f4),LL(0xdb730d58,0xf36fff06),LL(0x3c01edc7,0xcc18624d),LL(0x8a77e5bb,0x4889078f)}},
   {{LL(0x75f8dd8a,0x02294e3b),LL(0xfc4113c5,0x5f6f6057),LL(0x6f699f18,0xb5300e0d),LL(0x639dc977,0x52cce358)}, {LL(0x328fd50b,0x5dbe59b8),LL(0x39d73c2f,0x81500be6),LL(0x96ae313d,0x409ac4d7),LL(0x5b16c316,0x15205b7b)}},
   {{LL(0xfc688c09,0xe272300c),LL(0xbdf71f2d,0xb412cf39),LL(0xf85b23d4,0xe3ab9c44),LL(0x7b662694,0x5c14085c)}, {LL(0x9956d07b,0x24b0b385),LL(0x30b2c82d,0xfa8ea968),LL(0x6d403b58,0xd443b2aa),LL(0xe7fc8d57,0x6da53ecb)}},
   {{LL(0xdfdf488f,0x29655314),LL(0xadc57e2b,0xb418943d),LL(0x6395a287,0xbaf090f1),LL(0xd62f5b38,0x8fdb4fc8)}, {LL(0x371c9db1,0x115653c0),LL(0x96463359,0x6f5e1f39),LL(0x825e6273,0x106aaf1e),LL(0x398cbe1b,0xba22b7db)}},
   {{LL(0x62b6bf7e,0x3b545300),LL(0x5bb6f993,0x495d7d27),LL(0x3f00290b,0xf558fc5d),LL(0x2cfc2381,0xdddbeb3e)}, {LL(0x65c79326,0xca402179),LL(0x33b1973c,0x376ce4ac),LL(0x9b30749a,0xd6e65ae4),LL(0x5f54bf5a,0xdf68ee04)}},
   {{LL(0x4cfdb41a,0xa982948d),LL(0xf2a7b4b3,0xddd1d3ba),LL(0x581aaba1,0xf52a6b33),LL(0x894ebf68,0xc3258264)}, {LL(0x84665ac1,0x26c32c27),LL(0x20119b0e,0xda0190eb),LL(0xb86745c1,0x4a599db7),LL(0x58964b41,0xf9570f50)}},
   {{LL(0xe0648365,0xb34d039b),LL(0x5c5f61e1,0x2cd7fde0),LL(0xbc6b08cc,0x76f514a3),LL(0x18a3cabf,0xc957b50d)}, {LL(0x2334cd1f,0x775fc56a),LL(0x67ec91c6,0x7bfe3864),LL(0x35ad3a9a,0x99037daa),LL(0xb7ca5657,0x17ffe391)}},
   {{LL(0xfef04aef,0x19f6d369),LL(0xd1876f8c,0x8030b467),LL(0x3cd7878f,0xa014be02),LL(0x3358c943,0x03c22a58)}, {LL(0x2a257094,0x3c77f083),LL(0xd962a04f,0x47386957),LL(0x82da3329,0x768da40c),LL(0x458219cf,0x1507350d)}},
   {{LL(0x4397ee7c,0xf460aed3),LL(0x36fbc7fe,0xf249e4cc),LL(0xaab03dfe,0xc42d9da8),LL(0x435ab9f3,0xa3d73ce3)}, {LL(0x3813a3f3,0x86dddbc3),LL(0xb79c32a3,0x86d5779a),LL(0x028a2c3f,0x7c3d9aff),LL(0xb1add2bf,0xc687e71b)}},
},
/* digit=26 base_pwr=2^182 */
{
   {{LL(0x09b3fed3,0x4612a562),LL(0x3579966a,0xf296c17a),LL(0x07960971,0xa124a36f),LL(0x380c4a05,0x6d03b214)}, {LL(0x70f1f268,0xcb0746e2),LL(0x9341aea4,0xcc9b47ff),LL(0x6d2f59cc,0x1b3662d5),LL(0xd4b1a15d,0xa6c65b2d)}},
   {{LL(0xcccb0a4b,0xf96c113a),LL(0x3615f016,0x24c26bba),LL(0xeead2f5b,0x52fe115a),LL(0x0d7aaabb,0x85623d26)}, {LL(0x31a2564f,0x50791fd0),LL(0xcd0d59a4,0x3659974d),LL(0x7a8b7466,0x2cffdb74),LL(0x514787b0,0xcf6b36e5)}},
   {{LL(0x4ab1ccd2,0x8afccd36),LL(0x1c03ab29,0x67314635),LL(0xd7ff3a32,0x458f36bf),LL(0xfcf71648,0x70e9e789)}, {LL(0xa6e904cf,0xf3764534),LL(0xf4bdd378,0x2d6130b1),LL(0x1ca5ce34,0xc61c98fb),LL(0xa4a684f5,0xda11f502)}},
   {{LL(0xb6223f04,0x8d9daa41),LL(0x841c3fab,0x803c9c0e),LL(0xc50b42cf,0x60eee3f9),LL(0x77298193,0xaf4a7a5a)}, {LL(0xbf809ad6,0xd379c2e1),LL(0xf67c0ff2,0x903ab4b1),LL(0x90f8e637,0xc779d7ed),LL(0x2cf3d363,0x968b0cc0)}},
   {{LL(0xacf51940,0xaadfa857),LL(0x50156581,0x0c789d1e),LL(0x62cff8f4,0x5e79cef7),LL(0x65eb0d49,0x54cdaba9)}, {LL(0x3321c57e,0xdf7a5828),LL(0xa21a51a6,0x8571e6e2),LL(0xc3726e69,0x0b9b482b),LL(0x1d92b657,0x3bc201e3)}},
   {{LL(0x8a3b4cf8,0x271c58bb),LL(0x717eb539,0x269fc422),LL(0x5b381fe1,0xe82644e9),LL(0xcb62982f,0x27fb0822)}, {LL(0x5b5ec726,0xb0acd51c),LL(0xea4eff73,0xfd01053d),LL(0x00b11c80,0x465311dd),LL(0x2ed8460c,0xe50a8615)}},
   {{LL(0x7b2243a0,0x3eade5eb),LL(0x77424d11,0xa59ec933),LL(0xf5c7c3b7,0x65a8e1aa),LL(0x0c1db536,0x008399fa)}, {LL(0xfb194a74,0x80b20e97),LL(0x43be90dc,0x2316fb9b),LL(0x0da4d913,0xb2773b23),LL(0xce973d27,0x945d0671)}},
   {{LL(0xb79f82af,0x64ca871c),LL(0x2dab52f6,0x31304b02),LL(0x928239a7,0x1825ab54),LL(0x8e4ad736,0x740413b2)}, {LL(0x44071d19,0xc5c5d3fa),LL(0x3f0b2da8,0x83e438f1),LL(0xc70a1981,0xfd759448),LL(0x565ebae3,0x13e0c7ee)}},
   {{LL(0x26bd7c0a,0x31b74b0a),LL(0xd280cb56,0x66e0e8e8),LL(0x3d1c83d2,0x086795e6),LL(0x396ecf25,0x59e678da)}, {LL(0xf015a36e,0xab3c8d74),LL(0xadc03171,0x0d19aed3),LL(0x5a263686,0xc83b787f),LL(0x9057ed63,0x46b94ad0)}},
   {{LL(0x90979da2,0xfbf783a7),LL(0xa335c784,0xf04dd6a0),LL(0x87d93c4c,0x6e3c2554),LL(0x47994eb3,0xe3e6b289)}, {LL(0x1b74ba16,0x473c0684),LL(0xabe84e1c,0x4e959eb4),LL(0x7c4a67b7,0xdc3bfd51),LL(0x5095bd6e,0xb4e3cb85)}},
   {{LL(0x3229fb05,0x96fc11f0),LL(0x4b36c83a,0x598227e4),LL(0xd46fca66,0xdc69ad06),LL(0x703ad6be,0x14cc98e5)}, {LL(0x6b22cd50,0xf0fdd142),LL(0xf89c1a5d,0x9b821fe2),LL(0x829f9a74,0xa3762dca),LL(0xf0c320cc,0xf65a584a)}},
   {{LL(0x5568f242,0x58f4eaba),LL(0x029afc1f,0x83b0c37b),LL(0x994d7dfc,0x93de2d27),LL(0xb1679532,0x0d9a6edb)}, {LL(0x95f085b0,0x3b834279),LL(0xa299355f,0x46ebac98),LL(0x044427f8,0x0212e489),LL(0xa2f37d0e,0xf9e4ce34)}},
   {{LL(0xfdc9e233,0x0fa328d6),LL(0x51ae732f,0xd5c8afab),LL(0x83c00cee,0x85e59553),LL(0x87505db4,0x9fce31f9)}, {LL(0x7069d066,0x33ea5eb6),LL(0xf01c0ce9,0x10e32a39),LL(0x0c4f1a2e,0xf170233c),LL(0xbd4cb366,0x8a907760)}},
   {{LL(0x79bf05e3,0xda593421),LL(0xce49a5bd,0x7730907a),LL(0x09be5c7b,0x0dfb8a97),LL(0x23eb936e,0x6f50c692)}, {LL(0xcb18ff1a,0xc6160551),LL(0x661cc384,0xfa1d23fd),LL(0x7ddec262,0xfef12307),LL(0xb15af580,0xd1aca960)}},
   {{LL(0x2ee50fbd,0x7eab8a59),LL(0xbe1e7a42,0xe7f71845),LL(0x6121e573,0x5f46a511),LL(0x38ff7eba,0xa25dacbf)}, {LL(0xf58f047c,0xe8aefcc7),LL(0xb538aace,0xc343aaa7),LL(0x3c340b1f,0x3e58cdda),LL(0x1fb98ccb,0xb0e9867c)}},
   {{LL(0xef7750f0,0x034dd314),LL(0x22da84d7,0x2ceaa705),LL(0xfc0d647b,0x4561a254),LL(0xbbe01157,0x81cf0915)}, {LL(0x34b798eb,0x547a3d4e),LL(0x8b1c7544,0xbb5dd625),LL(0xc8194168,0x94fee01b),LL(0xfaeb010b,0xdb4c25ff)}},
   {{LL(0xe4d4a176,0x1ff217fa),LL(0xaf87f4f5,0x8b46e6c2),LL(0x89734273,0xcf658775),LL(0x52746de9,0x73c4390d)}, {LL(0xb01c7a06,0xb5c84899),LL(0xedd9ef63,0xfa5ffe4d),LL(0xa1a8b2d4,0x28a313c5),LL(0xdaf5a33f,0xadd45f47)}},
   {{LL(0x1fdb8694,0xc2dc9d13),LL(0x9a90b4d1,0xaa5e026c),LL(0x208cbfa5,0x5edffd39),LL(0x72a4d6cc,0xf095b3fb)}, {LL(0x6645fcc1,0xbfca4e10),LL(0x92408c6a,0x14b872ac),LL(0xd0b82d2d,0x3d9261e1),LL(0x0294e75b,0x13e4ecb6)}},
   {{LL(0x3ac2ab9d,0xabd4541c),LL(0x4d5d1531,0x025355b2),LL(0xfb726ab8,0x3d85f67c),LL(0x6d6fc6bc,0x56e26c82)}, {LL(0x495e05a0,0xb24608bb),LL(0xe5afdc5d,0x840e0978),LL(0x248727e2,0x2cc543b5),LL(0x3bc8c021,0xe48146da)}},
   {{LL(0x530c98b7,0xa1b36baf),LL(0x5acf023b,0x04503d7b),LL(0x21de1503,0x96bc4449),LL(0xd2a9c89f,0xbb8a122f)}, {LL(0xd5d4b133,0x66df99df),LL(0xc97d3e52,0x1bb4a13b),LL(0x79b318d6,0xdab370f3),LL(0x9f18552e,0xfa6c823e)}},
   {{LL(0x6388a194,0xe5b27e78),LL(0xc88ba441,0x13270523),LL(0x4fecfef5,0x9f309fbf),LL(0x10afee60,0x72cd374b)}, {LL(0x93dfe3af,0x16bd0e2e),LL(0x24bc7e8e,0x7e92096a),LL(0xfec7f0bb,0x144fdf82),LL(0xe1f765f7,0x5d1d4598)}},
   {{LL(0x72c67697,0xb6b91efc),LL(0xb2487905,0xc7a2ceaf),LL(0x7fb24d99,0x4a4c9e63),LL(0x4d742089,0x7ed373ac)}, {LL(0x9149ac54,0x3f9e6ae1),LL(0x0611efc8,0x64fd7fef),LL(0x3d779af6,0x1c38da32),LL(0x0a1681f5,0x6893691b)}},
   {{LL(0x50a0fa72,0xbac29978),LL(0xba55c665,0x98d5c82e),LL(0x2d4b76bc,0xf3e5b58e),LL(0x90615b32,0xfae27d9a)}, {LL(0xd49b2500,0xb93bc327),LL(0xbbc213cd,0x7d9d4bff),LL(0xd1ee81c4,0xf985fe72),LL(0x381f9e48,0x6e2a94d1)}},
   {{LL(0x14fb9624,0x1f09b225),LL(0xca4229d7,0x2eba4ff8),LL(0x21dc8c19,0x5b159dd1),LL(0xb1aa553e,0x1e1f968f)}, {LL(0xc7674d52,0x6ea25976),LL(0x7b283501,0x98e73adc),LL(0xd39468c2,0x7cfce0e1),LL(0x08438a62,0x7aad0af9)}},
   {{LL(0xb2a3dde2,0x2291cdd0),LL(0xf77a0aa4,0x3a625d50),LL(0x5fbc5a0a,0x3be0fba2),LL(0xe794bf46,0x67b7598a)}, {LL(0x531ad772,0x3673d805),LL(0x03e8479f,0xf9a9b392),LL(0x2e16a126,0x142d264c),LL(0x5a2f6f2c,0xc20409ac)}},
   {{LL(0xcd43f273,0xd9d84438),LL(0xbda7be72,0xfecc561d),LL(0x241b7ec2,0xc4b113c6),LL(0x40dba9e3,0xfc5bc32b)}, {LL(0xd56bca47,0x70853d39),LL(0xa5306217,0x2b9a902d),LL(0x2017bfd0,0x2bb1836d),LL(0xcd1c2768,0x829ce116)}},
   {{LL(0x697097f5,0x42d5fcf8),LL(0x1e936db5,0xc1fe7be6),LL(0xcb6a81d4,0xcbc5cdcc),LL(0xafef5ffa,0xab1e4ecb)}, {LL(0xb933c216,0x3cbbdf76),LL(0x503607e2,0xdb5808da),LL(0x6bc96246,0x5bdaab7c),LL(0x68274629,0x91e5d17c)}},
   {{LL(0x2eb1db21,0xa3cd09f6),LL(0x92c3e3e1,0xbe370485),LL(0x6aa43da5,0xeb51fa29),LL(0xd726625e,0x2c7fa809)}, {LL(0xf0ec0e99,0x90c6786f),LL(0x08135cbf,0xd315af33),LL(0x1504751b,0xc1b60172),LL(0x0e28781a,0x88674e2d)}},
   {{LL(0xed74e633,0x6aa74055),LL(0x7d06ce02,0xc44e740f),LL(0xa33b8d5e,0x8b40bc5e),LL(0x20f00f14,0x42d3539f)}, {LL(0x3307ef15,0xd9f1f5cd),LL(0xc8599bcc,0xa9fe4dfb),LL(0xefa80b8d,0x31cb6703),LL(0x53bb73fe,0x4172b46d)}},
   {{LL(0x20e4c321,0x85a70280),LL(0x5ac075f3,0x999a0d07),LL(0x7bdb478c,0x59a62b62),LL(0x573c403b,0x9aeb710a)}, {LL(0x950bb8fc,0x1c099614),LL(0x5dc09741,0xc1efafab),LL(0x7296a74b,0x0de58ca5),LL(0xf5be2ec4,0x657116a4)}},
   {{LL(0xcb199b77,0x0ce52f0f),LL(0xbcd11438,0xdcdc5cb9),LL(0x4777327b,0x587a68ff),LL(0x1cc6fbb3,0x55d9abb7)}, {LL(0x9eeb28a9,0xf1970b82),LL(0x4ceef00f,0xe1ab4e14),LL(0xf7462893,0x184d3fb6),LL(0xc8ea54fd,0x9942a008)}},
   {{LL(0x1e33b2a3,0x1fee0f77),LL(0x9f789870,0xd4bed81f),LL(0x6ef05b7e,0x6396feea),LL(0x2640b62a,0x9c5d6a01)}, {LL(0x6834bea4,0x170cfec9),LL(0xe131feca,0x68d16728),LL(0x00affb4d,0x4be9c5d6),LL(0x99a6f256,0xe34a423c)}},
   {{LL(0x09b9ed61,0x1a254e4a),LL(0x30b10207,0x902bc066),LL(0x62121f53,0xd2d5ed01),LL(0x30f1b518,0x0ba86811)}, {LL(0xabe139c9,0x7916c132),LL(0x62c4f337,0xb3a30fe0),LL(0xaa5693be,0x85d0a769),LL(0xe3c7687b,0x2d414379)}},
   {{LL(0x94958719,0x92b0cb3c),LL(0x4ec6575d,0xb78aa37b),LL(0x4f1bf26a,0xd035aae1),LL(0xd31d5108,0x1383992d)}, {LL(0x92bdd6f5,0x53ecc535),LL(0x08c622ca,0xa9925ff6),LL(0x916d890c,0xcaa3146e),LL(0xb9c10220,0x8cd0f12e)}},
   {{LL(0x7e12a730,0xcb6ad82b),LL(0xac9485db,0x3f89047c),LL(0xfea2d970,0x6f926117),LL(0x46a19ecb,0x87b0cd9d)}, {LL(0x01e45bf6,0x98bb5b02),LL(0x2ed7716d,0xfc814620),LL(0x4f5caa95,0x8d6808cf),LL(0x082f799e,0x3b57df03)}},
   {{LL(0x2df84ca2,0x469e1854),LL(0x64aac069,0x00dd62eb),LL(0x88d9efff,0x7d3ee9ce),LL(0xbb830ffc,0x9faed6a2)}, {LL(0xd2d74f58,0xd073aac1),LL(0x2d44199e,0xf69e96b4),LL(0x83ed62ca,0x6cb3a3b1),LL(0xd799acf8,0x472489fd)}},
   {{LL(0xb63a36cc,0x5f84382d),LL(0x92d5b175,0x6ba1de87),LL(0x516101b7,0x25aab130),LL(0x5f039793,0x6f69c3fc)}, {LL(0x89e3da4f,0xd28439ee),LL(0x5e6b2b61,0x8cb40a0e),LL(0xe3d6650d,0xdfa55805),LL(0x0be59fd2,0x2651f6c7)}},
   {{LL(0x140d01c8,0x290e0044),LL(0x62ea488f,0x78afa0a4),LL(0x91eaa932,0xc4e39971),LL(0xfe2e79dc,0x8a9ef3a2)}, {LL(0x50705b7e,0xdcfae315),LL(0xd4be3d75,0x73324dca),LL(0x03a91415,0x900bdd43),LL(0xedfdc74d,0xc3ed02ed)}},
   {{LL(0xf22b4a66,0x509bd1d6),LL(0xb78d264b,0xfd8ed371),LL(0xa419134f,0x562b2d3a),LL(0x7a57a51e,0x80a2c274)}, {LL(0x8c662472,0xebba5317),LL(0xa0be71fb,0xebafedf2),LL(0xb77899c8,0x0c5b9c1c),LL(0xc4888cb5,0x82448008)}},
   {{LL(0x78401c3b,0xb795ea00),LL(0xa85ab971,0x86776990),LL(0x185739da,0xdd81393b),LL(0x58136c97,0x76d0031f)}, {LL(0x641d39d1,0x6aceaa56),LL(0x39be7ca8,0x918844c7),LL(0xe18efc54,0xa63993f7),LL(0x4af0f30a,0xb5369150)}},
   {{LL(0x3d04af4f,0x9bc2068c),LL(0xa7796ed2,0xf309dff9),LL(0x4e15b6a2,0x46e9a59d),LL(0xc22ef488,0x617aaeba)}, {LL(0xa15cf0cb,0xd91a8f90),LL(0xc30fb779,0xc6ce12a4),LL(0xb9d0a7ff,0xf3b80254),LL(0x6e9b6fa1,0x32a63bf9)}},
   {{LL(0x546fe4a8,0x3e1ac837),LL(0x1279c7ef,0x91ed89a5),LL(0xc73e9dea,0x8eb7b88e),LL(0x18238af0,0x96d07205)}, {LL(0xe96abf80,0x56ebf306),LL(0x52c4b10f,0x5088ce24),LL(0xc979137f,0x65293176),LL(0x228d300a,0x824642fb)}},
   {{LL(0x7836aea5,0x968963a8),LL(0xfabbfac1,0x2d4c556c),LL(0xd3f9977a,0xa4c389bb),LL(0x99b4ccb6,0x2e8b2818)}, {LL(0x6cb67df6,0xc1fd8565),LL(0xa72d2be8,0x0ac57d2a),LL(0xb8224ead,0xa51ce6b8),LL(0xf417d744,0x33f7b468)}},
   {{LL(0xf9f0bdf4,0xcf8c80af),LL(0xd3222dd6,0x0728f880),LL(0x653afc12,0x436188a3),LL(0x3c7940bb,0x0f8bf160)}, {LL(0x424dcd2a,0xdc18c13f),LL(0x20d3cd1f,0x038c1842),LL(0x7b12fd42,0xed7f86a5),LL(0x7aaf1881,0xa75ab77b)}},
   {{LL(0xdf0574e2,0x5c3d7612),LL(0x719414ce,0x2eeeeb6f),LL(0x90349fc4,0x797c5771),LL(0x2232eb33,0x0d850f73)}, {LL(0x2638c051,0x0a0744f3),LL(0xb6e7dbfa,0x739e6278),LL(0x659fc5f5,0xa77f286d),LL(0x9654b0eb,0xb18b7cf1)}},
   {{LL(0x6062e78e,0x5a2089ac),LL(0xdfa6fb44,0x152f1804),LL(0xb61e6faa,0xe8a404b4),LL(0x08d06ea8,0x4774d30f)}, {LL(0x3c359648,0xd7119b91),LL(0x09473ff7,0x850b02bd),LL(0x936b7868,0x4db6f9a0),LL(0xae38c3c5,0x84064dd5)}},
   {{LL(0xfe448461,0x294d6831),LL(0x42cd2105,0xc3c6f446),LL(0x3a2fdcae,0xa4412eb0),LL(0x3d5a9181,0x394c3774)}, {LL(0x5ca87c4b,0x58f19024),LL(0x89ad5685,0xba1879db),LL(0x803c2589,0x43c55c6a),LL(0xa8249c65,0xae1fad20)}},
   {{LL(0xe0aff809,0x4929e89f),LL(0x1769a00a,0x19755ec2),LL(0xc242f335,0x3b6a207b),LL(0x090edab0,0xeca054ef)}, {LL(0xcd9e1c26,0x217e9c8b),LL(0x35d4ac57,0x917c2ecd),LL(0xad33911d,0xdc869d5d),LL(0x2e828bd7,0x22d9d860)}},
   {{LL(0xf38dfaa1,0x89262252),LL(0xeb9cd8d7,0x155c96ce),LL(0xed5ebcc4,0xb0082b5d),LL(0x17182086,0x7b6f9203)}, {LL(0xee92aa6d,0xaefe28aa),LL(0x9aaaa0eb,0xbe67090c),LL(0x2f8ef18d,0x88c5fbf1),LL(0xdd1fd65f,0xbdc8bef1)}},
   {{LL(0xa9c7b483,0xfb7052f5),LL(0xbd6c8a99,0x49634258),LL(0xc9f424f8,0x1410a747),LL(0xe9805723,0xfda0a304)}, {LL(0x0879bd30,0x1a438bd3),LL(0x7f6903cb,0xed09a9d3),LL(0x57e53497,0x920878f8),LL(0xa7fca0ed,0x87a12968)}},
   {{LL(0x38590ca1,0x7c8207cb),LL(0xfae885c2,0x4cf52db1),LL(0xe8dc711f,0x6cf384c4),LL(0x221dc698,0x6fea20ff)}, {LL(0xa07bb79f,0x6af56700),LL(0x33ca79c6,0xc7da3b52),LL(0xd05eb519,0x3a214691),LL(0x93d4f089,0xea94c4f1)}},
   {{LL(0xba51f002,0x734039d0),LL(0xce206406,0xc45e2042),LL(0x4b3c3d53,0xc0e75dbb),LL(0x55b1b97c,0x3a701272)}, {LL(0xd6addb6c,0xec856e95),LL(0xf283aae1,0xb63fe8c6),LL(0x405788d1,0x148fb239),LL(0xe0181912,0x42e7148b)}},
   {{LL(0x7de07978,0x00bddcdd),LL(0x3c2e0a27,0xac79b657),LL(0xdf1dd3dd,0x94024ba6),LL(0x0bac41ad,0xcddeb357)}, {LL(0x500c4f4b,0x51ec3dd7),LL(0xd31c8fbe,0xf00d594f),LL(0x373a3e93,0x6b8c6f43),LL(0xfc2b6be9,0x891ba3a5)}},
   {{LL(0xddd72e36,0x3928225a),LL(0xcee362c1,0x1e6a63bf),LL(0xc5eb404c,0x317b78f4),LL(0x67c5e6b3,0xb882736b)}, {LL(0x1f2f07aa,0xb1da56ce),LL(0xff83b633,0xab3c4fbe),LL(0x0ceeab99,0x9cc32f1c),LL(0x1062070e,0xf1dead0d)}},
   {{LL(0x8a3e79c4,0x49ea0d9b),LL(0xec9f16d1,0x4e7abe3f),LL(0x5549ade0,0x19bda1c6),LL(0xe5885734,0xaae756a5)}, {LL(0xcc2a1aaf,0xb3cff8ce),LL(0xf896ca47,0x812eebff),LL(0x9b2e1123,0x0951b2bb),LL(0xdef6d6a9,0x7f245699)}},
   {{LL(0x1be7ef41,0xa1331e95),LL(0x9fa1be62,0xd1f0c3c3),LL(0x4383e451,0xb1d8295e),LL(0x9f08bc14,0x658d8a84)}, {LL(0x3ba4b85b,0xb0587aef),LL(0x481cbb27,0xb519c587),LL(0x040d8f06,0x2b975db6),LL(0x1691d809,0x399f6417)}},
   {{LL(0x7c6204fb,0x207a0e46),LL(0x62c3e9d7,0xe30f1420),LL(0x792f8208,0x6127b782),LL(0xb0d3fca9,0x38f806ab)}, {LL(0x2ff46c64,0x38248542),LL(0x926ec166,0xc18ffe85),LL(0xc0c133fa,0xfd268866),LL(0xb93770e6,0xb7f63f5a)}},
   {{LL(0xb13afb71,0xd8f1db26),LL(0x32a790de,0x5c5627eb),LL(0xdf50b6f8,0x7f41bc1d),LL(0x92d4c803,0x49d4ef17)}, {LL(0xe8530065,0x577f909f),LL(0xe630ff2d,0x482cdede),LL(0x14f54de8,0x682c8c6a),LL(0xb4079343,0xe6b5a504)}},
   {{LL(0xe58bde6b,0x00d927fc),LL(0xf34841f4,0x65d85f03),LL(0x2ac001d8,0x265aec02),LL(0x2dfe518d,0x1b785666)}, {LL(0xc01e6e47,0x76142488),LL(0xdd5648dc,0x8e8b2044),LL(0xb3a340b3,0x2c422006),LL(0x3dd67b22,0xa5392113)}},
   {{LL(0xa1567aaa,0xbd08d05b),LL(0x02acbec6,0x84a55e43),LL(0x5d898af0,0x744ffd21),LL(0x6682e78a,0x38067622)}, {LL(0xffd08522,0xf3696ff2),LL(0x2bf02466,0x49dd0060),LL(0x59c3e65d,0xc9e0d1a5),LL(0x0a37fc25,0x29537f56)}},
   {{LL(0xa5f6b17a,0x6f6cb9eb),LL(0x9c55857e,0xc1854320),LL(0x45dacc6e,0x959585c6),LL(0xe5f4e930,0xf4e97c94)}, {LL(0x57d2a492,0x966deb5f),LL(0x55d2df12,0x98256831),LL(0xaa457eca,0xfdd65534),LL(0x03551474,0x76dbb021)}},
   {{LL(0x09d9b4aa,0x0aeefee9),LL(0x784ca675,0x30fea11a),LL(0xff1d381a,0x56b4b509),LL(0x9fce40da,0xd1b26fea)}, {LL(0x48d22911,0x4835b911),LL(0x8bbe57e8,0x6aaac57a),LL(0x19d02037,0xc8882792),LL(0x3ee49afa,0x301e0aa6)}},
   {{LL(0x00e6b020,0x1641ce6b),LL(0xeac7cad8,0x846b97de),LL(0x61aa6886,0x9b74bfd8),LL(0xb0fa37ac,0xdd95e765)}, {LL(0xf848a83b,0xda0cde52),LL(0x355b3528,0xd2cc831d),LL(0x5e22238f,0xc7fd2e03),LL(0xab9a6c34,0x6d5373fa)}},
   {{LL(0xd8247f13,0x5dfc2874),LL(0xe3c11f56,0xc211a7a1),LL(0xa2503b97,0x7512563f),LL(0x5c007c82,0x124cd984)}, {LL(0x491cd249,0x4f6eb682),LL(0xa683359d,0xaf4f70a3),LL(0xcc302b62,0x2f1dfe71),LL(0xe57fbf56,0x83c474bb)}},
},
/* digit=27 base_pwr=2^189 */
{
   {{LL(0x916a8016,0x43af7ab7),LL(0x532bfb9c,0xf93d487f),LL(0xe2174971,0xa5f9af3c),LL(0x2d59b4d4,0xd1b9cf1f)}, {LL(0x44f4eb91,0x4a779418),LL(0xc226edc5,0x6a131fac),LL(0x80d4bb33,0x472ab897),LL(0x2f6ca1fe,0xb69687a5)}},
   {{LL(0xfabd066a,0xffa73ca2),LL(0xf9c78bfd,0x494e03a8),LL(0xff55cfef,0xe585a878),LL(0xd7053784,0x00770b1f)}, {LL(0x056fe70b,0xdec4da4a),LL(0x57bd444f,0xe37395d8),LL(0x685df668,0x666250d4),LL(0xbe6cc583,0x0549569e)}},
   {{LL(0xab11639e,0x87629830),LL(0xa4488d53,0x869dd3ba),LL(0xbaf06eb6,0x10fe1c0b),LL(0x1687ac37,0x99034839)}, {LL(0x7f1ffe7b,0x38418377),LL(0x25bd7c57,0x3334a74c),LL(0x7008ba67,0xc57cb7ed),LL(0xc1e4e12d,0x384c12d0)}},
   {{LL(0xdb4bdb35,0xf48cdca6),LL(0x74d913a7,0x6bc23aec),LL(0x12ed94d5,0x8f0ccd9d),LL(0x86db09e7,0xe4aabd12)}, {LL(0x1e948326,0x0cbff31a),LL(0x17a479a2,0xcf68c47c),LL(0xca7686f1,0x3cced8e2),LL(0x4eb62669,0x15ed1e99)}},
   {{LL(0xbdb0c561,0xc373ab4b),LL(0x6a9066a7,0x15082022),LL(0x62d31801,0x330a60c3),LL(0xe35bea57,0x53d97f09)}, {LL(0x9c5dbb92,0xf204e006),LL(0xf831262a,0xfb9a8219),LL(0x42136174,0x3468ae41),LL(0x0e03218e,0x0f8fb5bc)}},
   {{LL(0x4ad8bba6,0x90337499),LL(0xe3ecb618,0xdb71e1fb),LL(0x3cf2a8ad,0x6955e874),LL(0xed691fee,0x594501f5)}, {LL(0xd29bd364,0x7e2baef3),LL(0x6f766759,0x5cbd91ac),LL(0xb2201a96,0xaba54aaa),LL(0xcfa392ab,0x2cfea457)}},
   {{LL(0x86f8f7da,0xa4da4162),LL(0xcbc0b934,0x88d70b86),LL(0xacff4f7b,0x9df02795),LL(0xc65ef81b,0x0fc80219)}, {LL(0xa299ca0f,0x32d457de),LL(0x0896427b,0x97081b35),LL(0x41bab6b4,0x92d6c309),LL(0x73b8d601,0x5d5e56f3)}},
   {{LL(0x202bde39,0xfb3992a4),LL(0x3d6bab98,0x2549f564),LL(0x87712512,0x0b564642),LL(0x7fde7e50,0xd52442b4)}, {LL(0xa3d3e16e,0xa6cefd08),LL(0xc83b29bd,0x5b194f0a),LL(0x906dec8c,0x6db0edd8),LL(0x02570c1e,0x7a090959)}},
   {{LL(0x4c41eb53,0xf6f74fcc),LL(0x5b944a6c,0xd07678a9),LL(0xb80677ea,0xf53bf11d),LL(0xbc5900f8,0x569a5761)}, {LL(0xd3d4de91,0x34e5bba8),LL(0x8361f73e,0xc5774804),LL(0x59abdbd5,0xd637d3dd),LL(0x8772b615,0x64a81bf9)}},
   {{LL(0x7f3d83ab,0x78bb12ea),LL(0x573f9b99,0xca22c31c),LL(0x2aed4c39,0x4283c173),LL(0x39f32bdb,0xda054c1d)}, {LL(0x1da2cbd7,0x2ead717e),LL(0x62390b41,0x747d67cd),LL(0x6b9666a6,0x43739d9c),LL(0x8c827b12,0xb84e2f22)}},
   {{LL(0xc0312773,0x0e4ac2b1),LL(0xe53f068e,0x571cfc75),LL(0x42bfe41e,0x6c44df85),LL(0x627e30bb,0xe7d2edb9)}, {LL(0x0dd5cedc,0x9c2e4fd6),LL(0x0f7d22d7,0xe2d885ef),LL(0x1329bcfd,0x44b0b5db),LL(0xba1c96f6,0x006e872f)}},
   {{LL(0x7e952317,0xdbadab5d),LL(0xc2a5bcaa,0xab849ed4),LL(0x1e72dbb1,0xe3acbb74),LL(0x5d4b7cb7,0xbf42c3d3)}, {LL(0x3d748639,0xebe967b5),LL(0xc03af7a1,0x1fe93db5),LL(0xa944ea06,0x2ab14596),LL(0x76655c09,0xfb05a759)}},
   {{LL(0x6f8a532b,0x5117890c),LL(0x59430c5b,0x2f57781f),LL(0x79e07b84,0xe70968b3),LL(0xe86d7223,0x05df2305)}, {LL(0x31e32933,0x57af0dc5),LL(0x84afc419,0x5473e34a),LL(0x03d5feb4,0xa7337a42),LL(0x1b1c6bd8,0xd85c8602)}},
   {{LL(0x753008e6,0x25ca1891),LL(0x5f0ff93a,0x4338ec98),LL(0xddd30a7c,0xd2ba8557),LL(0x09c51794,0xb4b65361)}, {LL(0xd1cbc66e,0xfbb51399),LL(0xe53bca50,0x28853781),LL(0xfd5a9aaa,0x5b797232),LL(0x5b88c4f3,0x6249afd7)}},
   {{LL(0xba6918a0,0xcc5ab6cb),LL(0x8fb65c7d,0x9f824ec1),LL(0x56b18754,0x4796d80b),LL(0x67721520,0x4c83d371)}, {LL(0x63b03348,0xd77c373c),LL(0x54f27457,0x91930e5e),LL(0xaf40c03f,0x83f97370),LL(0x34eea661,0x65b55872)}},
   {{LL(0xeb10175e,0x310695d0),LL(0xcd236aa1,0x79aaa6ea),LL(0x3edfff40,0xf78539ff),LL(0x02cd6063,0x2369c517)}, {LL(0x5c8631ff,0x81e43ae5),LL(0x216a60bd,0x065e8212),LL(0xe761a5f9,0x225cb473),LL(0xab6de6fa,0x695ef860)}},
   {{LL(0x7d7d98d4,0x03536a46),LL(0x18413673,0xa17d3a69),LL(0x295ae303,0xa6ddcd46),LL(0x61beae2b,0x86de0bbd)}, {LL(0x7699458e,0xdd73dfcc),LL(0xb53f88dd,0x827deba5),LL(0x42a9a11c,0x213c376b),LL(0x12c73975,0xc854fd72)}},
   {{LL(0x15ac27ff,0x1fa96547),LL(0xf49b6c9a,0xcb0dc17b),LL(0x709dd202,0xa3e44853),LL(0xcfe2bbea,0xd3905c5f)}, {LL(0x6c35ce9c,0xb01e5799),LL(0x900ef600,0x0063e7ac),LL(0xfffa5cc0,0x8c70b87e),LL(0x74230b0c,0xebd76d34)}},
   {{LL(0xed5f8529,0x914eec9e),LL(0xe8edf477,0x7a65ffd3),LL(0x70c74bee,0xf0cb206d),LL(0xd1b02e01,0x03445ff1)}, {LL(0xe5dbf791,0x664ca356),LL(0x254e69c4,0xd678d4ae),LL(0x8617386b,0x370c9f0f),LL(0xfdcd985d,0x42af7a0c)}},
   {{LL(0x83c3da54,0x8c4b5009),LL(0x4c8a87c8,0x086a7ec5),LL(0xaa166c4c,0x9ba0b368),LL(0xa658ac1c,0xa279670f)}, {LL(0x5d0544da,0xc49f49bd),LL(0x15cb0b41,0x28c22323),LL(0xa4834d71,0x86293dfa),LL(0xd1e1d63b,0x283e191d)}},
   {{LL(0xca188555,0x0cad6519),LL(0x0cbd0c5c,0x323ce5da),LL(0x38560254,0x6b7d2be1),LL(0x1696b9b9,0xb05ed385)}, {LL(0x9ae59f92,0x8ce4b5a7),LL(0x4f7e61a3,0xabe5ff33),LL(0xdbfeb302,0xae15a3cc),LL(0x837fde82,0x691b1129)}},
   {{LL(0x2e6d116b,0xb60b31f3),LL(0xecab5aa9,0xd49e9d11),LL(0x6787f23d,0x3e95f844),LL(0xa12f4846,0x2ab8834f)}, {LL(0x5b6359cc,0xe70e2ab1),LL(0x9322a047,0x7a6349e9),LL(0x6c1e483a,0xc01e424c),LL(0x92bd5d1b,0x424b2027)}},
   {{LL(0x254e49a3,0x8a6e6766),LL(0x97e70d58,0xb8d85d42),LL(0xb51b3abc,0xa859082f),LL(0xe7bb828a,0x2850573b)}, {LL(0x7bfe8021,0x47cc95b2),LL(0x5853f12c,0x7c28fe9e),LL(0x10c0f389,0xe5fb0558),LL(0xdaf0a7e7,0xb99a639f)}},
   {{LL(0xf60ee3e5,0xa6b9e6c9),LL(0xa236377f,0xb397af7f),LL(0x7da4af71,0xb7a318ac),LL(0x0a9d39fb,0xae64b613)}, {LL(0x902b3411,0x66ce6c74),LL(0x5a199e53,0xea256a70),LL(0x550fb76f,0x8dcddd89),LL(0x03e70f9c,0x9443b477)}},
   {{LL(0x142113a6,0x1787b8a5),LL(0x180aec95,0xa58b6c61),LL(0x947ff26d,0xcc381420),LL(0x3d8b8c36,0x22520e8f)}, {LL(0xef2cc6ef,0x192ee945),LL(0xe9ca0c7a,0xea52afee),LL(0xe6e30d93,0x5970d794),LL(0x57c786ac,0x0a325e42)}},
   {{LL(0x33ca1226,0x5e2dddf8),LL(0x588cb1e3,0x18e624b9),LL(0x21809265,0xf3ba597a),LL(0x5d575728,0x90247702)}, {LL(0xc1f918db,0x48a5bf7b),LL(0xd6840541,0x17d1efaf),LL(0x3e2e754d,0x13dfe6fe),LL(0x707a531f,0xc471e16a)}},
   {{LL(0x97d34b48,0x79085bbd),LL(0xc2e9bea9,0xfa5ba99d),LL(0x6c5a6dc2,0x70b9c9fc),LL(0x4e94c5db,0x4e042213)}, {LL(0x25ebb95f,0x4a37b41f),LL(0x055d79fb,0x24691010),LL(0x3f572a8f,0xdaff9352),LL(0xf327ec2a,0xe63d55b0)}},
   {{LL(0xdebd5116,0xc5a86d3c),LL(0xa2ddef2a,0xd547fe08),LL(0x6a149f12,0xbabb617f),LL(0x8a766128,0x14f69a1b)}, {LL(0x48236f77,0xb83a1477),LL(0x35711279,0xd0d81be1),LL(0x5eab1c3a,0x706f9067),LL(0x16a1ffaf,0x8c4823f1)}},
   {{LL(0xaff5ea89,0xd845c68b),LL(0x6b75eadb,0xa276eaeb),LL(0xcc230ec1,0x2d0fc819),LL(0xedaaf1f2,0xdfad96e8)}, {LL(0x40868254,0x0f25dcbf),LL(0x5babd7f9,0x53bbe31e),LL(0xcf804a8d,0x7f8afc48),LL(0x5f9b9a0d,0x7f4922ef)}},
   {{LL(0xd7422804,0x703cbf6d),LL(0x83349bdd,0xe5df61f3),LL(0x77d285ad,0x0fa3d8cd),LL(0x2e88e15e,0xe990f9e5)}, {LL(0x8561d8a6,0x40ec61f7),LL(0x16650305,0x7fc498a6),LL(0x8e5beabf,0xa3bf5cb4),LL(0x76ae0350,0xfaa52008)}},
   {{LL(0xe4fc3e72,0x99e24318),LL(0x2079c616,0x9241c8ab),LL(0x9584a258,0xefa5bf38),LL(0x1eebb098,0xd7b770b5)}, {LL(0xe1fc18a7,0x28b714a3),LL(0x5b83dd9a,0xf0426bd2),LL(0x291b28ee,0x956d8972),LL(0x6eb553ff,0x8bb8cbde)}},
   {{LL(0x95298003,0x396cfe2d),LL(0xad8412fc,0xcaa66550),LL(0x83997dd3,0xf41d0215),LL(0x45534587,0x7066e356)}, {LL(0x5b6de0d7,0x0d5b5c3e),LL(0xcecd5f26,0x8ead45d0),LL(0xd252ae50,0xe2f24e2c),LL(0x815150bf,0xf71e5d4f)}},
   {{LL(0x54527ce5,0x3872685d),LL(0x91fd99ee,0x59b343ae),LL(0x3462cc0c,0xd621d027),LL(0x8dbfbcf4,0xfa42481f)}, {LL(0xaf7ae918,0xda481a9e),LL(0x7c909a18,0xfd5fd37c),LL(0x805fb7b7,0xa5ebb7bf),LL(0x165200b1,0xeac65687)}},
   {{LL(0x7cef9b97,0x56302866),LL(0xae3ddb64,0x8f662dd6),LL(0x60c1aa98,0x90cb4e87),LL(0x986fb3bc,0x33f9fc60)}, {LL(0x974593cd,0x76f41ecc),LL(0x6e0f01e8,0xb19501f9),LL(0x25760dd5,0x587d9035),LL(0x9391032e,0xa31c971c)}},
   {{LL(0x95c9a84f,0x7650e3b1),LL(0x78c66087,0xbb83ea93),LL(0xdfcf1365,0xda08a04c),LL(0xca0b84a4,0xd23daeba)}, {LL(0x2ca3bd2b,0xf89d395d),LL(0x6e53fc93,0x779e2aaf),LL(0x34216082,0xc0fc7dc8),LL(0x42a66691,0x6cd8bdf6)}},
   {{LL(0x0fe014cf,0x836a2cf3),LL(0x0c08373d,0xdde5fc22),LL(0xcb3b2b54,0xc4fa2387),LL(0xe2aa434a,0x96253732)}, {LL(0x1d502ce8,0x4c4f5979),LL(0xb6df40c4,0xf046f5a9),LL(0xac6b75b5,0xc7d05765),LL(0xb69f3c34,0xae1cd887)}},
   {{LL(0x49b86680,0xafed4be6),LL(0x14f70463,0x17b75fa5),LL(0x90b7c07f,0xb12e8051),LL(0x39a8e99d,0xe2c8cf2b)}, {LL(0xd5fdb65b,0x984744c4),LL(0xa28942e4,0xd8b1c012),LL(0x46693fb2,0x295d0199),LL(0xa0a3b4fa,0x5ab3a305)}},
   {{LL(0x26755b3f,0x3c191023),LL(0xb6c084de,0x75f35df1),LL(0x63e6286b,0x30640e66),LL(0xd2c6c059,0x3b3720ec)}, {LL(0x6ea837be,0x2c821a0f),LL(0x84f23bd0,0x238c4d93),LL(0x390ea4f5,0xbdc40703),LL(0xae68a2db,0xcb207d0a)}},
   {{LL(0xe25f098f,0x487d18bd),LL(0x9ab84e10,0x39018023),LL(0x8b7ab4a2,0xaa19aa62),LL(0x89f08fbd,0xcb9cdebe)}, {LL(0x2ca57917,0x26a4c9eb),LL(0xda92ce1b,0xaadfd472),LL(0xdaa907db,0x32b592d8),LL(0x7d6f995e,0x9bbebacc)}},
   {{LL(0xe1d88c25,0xa27a4735),LL(0x9bd66b67,0x339905e1),LL(0x62af9942,0xa9bfa0ed),LL(0x2e2cb83c,0xd94dd9e0)}, {LL(0xab28e268,0x279d8fda),LL(0x51a97518,0xf28ab69b),LL(0x9691f33e,0xce9bd2ea),LL(0x74be3d64,0xb9e8b2fe)}},
   {{LL(0xabefa07d,0x35072fab),LL(0x7b51ba8e,0x1c2ba05c),LL(0xd32d6bf5,0x3bb1ec56),LL(0x5d7bd7dc,0x326bdfdc)}, {LL(0xd95bdcb1,0x33f4f4f6),LL(0x453ef338,0x781bfd34),LL(0x1ef61a66,0x4d210cad),LL(0x2799bcc7,0x6ae7bb14)}},
   {{LL(0x194f4d6a,0xb105e5ec),LL(0x52b77f08,0x204a5480),LL(0xa328ab98,0x13f4e022),LL(0xb7f9a240,0xa56863c4)}, {LL(0xce4cf7bd,0x2780c9a7),LL(0xc497fdf3,0xf98e3f58),LL(0xf52738fc,0x635b8bc6),LL(0x58937500,0xc5fd89b8)}},
   {{LL(0x75e98a64,0x57070428),LL(0x946f094b,0x66aabaae),LL(0x06d37944,0x7d2376e8),LL(0x09319f13,0x9b316827)}, {LL(0xa77eb79b,0xbbde59a8),LL(0xf451fde0,0xb7f35bbb),LL(0x64aa79fd,0xb2f7914e),LL(0x9f68a486,0x4d2944b3)}},
   {{LL(0xc1a7e537,0xbd8a92de),LL(0x4fc930a3,0x76695e9a),LL(0xbcb4153b,0x1b75f9eb),LL(0xf6485404,0xf5a34d2d)}, {LL(0x26853a8e,0xe09ee965),LL(0x9dbb319a,0x63de8595),LL(0xda079d6d,0xbbbc1b07),LL(0xdfa71b9d,0x5956bb3d)}},
   {{LL(0x209cbcc3,0x69709306),LL(0xe3360429,0xbe2a08d0),LL(0x92a58752,0xd377a9fe),LL(0x997bc322,0x37e175ea)}, {LL(0x042ff2e4,0xfe355d4d),LL(0x4c4babd3,0x4332ef31),LL(0x2314b1af,0x634429c2),LL(0x91a7d5e5,0xae6e8275)}},
   {{LL(0x250a1476,0x134a39c7),LL(0x08994f0c,0xec9bb642),LL(0xd38704cd,0x2a9e0ac0),LL(0x536a4ad0,0x16490507)}, {LL(0x7c8dbfeb,0xc7f747d2),LL(0xc0bb24ac,0x91e67dd2),LL(0x959eca45,0x2dfc6c8a),LL(0xc54fefe8,0x78bafaf0)}},
   {{LL(0x5da056f3,0xf3eb2d1b),LL(0x3b89c967,0xda14b631),LL(0xcb51f621,0x80923b1c),LL(0x6609791c,0xc3d5fd1f)}, {LL(0x817b1063,0x68ad7bef),LL(0xa1f0b00c,0x3775b686),LL(0x6c7f0dc1,0xb207c9a5),LL(0xa9b91fff,0xb7c30a7d)}},
   {{LL(0x8b9f8e8c,0x9274c090),LL(0x24e94ce1,0xa3757610),LL(0x4f0f3ec1,0x8f2b1f2c),LL(0x3938d26f,0x02e53bb2)}, {LL(0x701e5ae8,0x90a3da2c),LL(0xa271dcca,0x60c2eaca),LL(0x31fb2372,0xc9c08e39),LL(0xb838908a,0xcaa3245e)}},
   {{LL(0xa6a6a275,0x2e168b0b),LL(0x0030ef6b,0x986a30a3),LL(0x170ab683,0x79f768f9),LL(0xff70981e,0x7296fd6f)}, {LL(0x13a323cd,0xbab6fedf),LL(0x186e9813,0xa86ec0dd),LL(0xcd56e7d5,0xd9f8db04),LL(0xaa8b1c96,0x47b20676)}},
   {{LL(0xf1fb3b03,0xdff4574e),LL(0x1051f9fc,0x41a1f765),LL(0x7f289a4e,0x35779aee),LL(0x11c96180,0x93bd54c9)}, {LL(0x37b95c67,0x1485074a),LL(0x0865b2f0,0x0b01af95),LL(0x90ce0b41,0x43033ffe),LL(0x71504d6f,0xffd6e34c)}},
   {{LL(0x1aa946c8,0xb380cd60),LL(0x806b2e19,0x7f7cc73b),LL(0x2db88e6d,0xc17df7d8),LL(0x072e5416,0x7b767ca2)}, {LL(0x0ad6134b,0xbb42d3ed),LL(0x640df8af,0x5444339f),LL(0x5bc73112,0x7e7c7e7b),LL(0xf899dba4,0xe8f139b4)}},
   {{LL(0x43a06bf3,0xd13b436d),LL(0x773e4863,0xe43f8567),LL(0x56b814d7,0x35555cd5),LL(0xd429ccc8,0x54af8e53)}, {LL(0x82ae0477,0xc346718f),LL(0xbe02c7a8,0x301fb382),LL(0xd2a70595,0xcd65b3b2),LL(0x5aad01d6,0xcfcff499)}},
   {{LL(0x589feca8,0xd0fcc076),LL(0x7c603ed8,0x7b2b93c7),LL(0x6ddfc3b8,0x2dda7a8c),LL(0x74723d99,0x678d66e9)}, {LL(0x6db60b07,0x0f7e4215),LL(0xc0bfa2f9,0x40666848),LL(0x8e116caf,0x70b46b5c),LL(0xfba46d90,0xbd753511)}},
   {{LL(0x019d8279,0xe48374cd),LL(0x309b0fc6,0x7d40e6e2),LL(0x9dec7a42,0x226435ee),LL(0x4243e7d0,0x818e79cb)}, {LL(0x54658634,0x3d7376d7),LL(0x9f8727ac,0xa40cafeb),LL(0x81f556bc,0xdc1d09f0),LL(0x63223573,0x32ca7367)}},
   {{LL(0x5810a27d,0x92e10f91),LL(0x1fdf969f,0x6fb34bad),LL(0x657a067e,0xe5c2b2ff),LL(0x382ba37a,0x173c0900)}, {LL(0x86d87c1e,0xdd5113c8),LL(0xcaf77294,0x56a2ca9d),LL(0x666a9821,0x9f956881),LL(0xa3b18c0f,0xc4bcafc7)}},
   {{LL(0x2b02578b,0xb100f338),LL(0x64b2c607,0x4716339e),LL(0x5b161179,0x92c923ae),LL(0x0df442a0,0xada2e4da)}, {LL(0x47f85474,0x4d4b90c5),LL(0x824e3195,0xa378bf79),LL(0x2478a0d4,0x4469339d),LL(0x0c1e69e2,0x0972e788)}},
   {{LL(0x72edc384,0x1aedd761),LL(0x9898d672,0xcabcd976),LL(0xba814ca2,0xd370aa7a),LL(0xe88eca9c,0x20fa58db)}, {LL(0x45a7ab8d,0x1540ada9),LL(0xbdca94fc,0x8dcf9860),LL(0xaa9df4f4,0xf0187e2c),LL(0x54a49576,0x9a197dc3)}},
   {{LL(0xb4a1f341,0xb54f5cb2),LL(0xfe32827b,0x1439caf0),LL(0xd36783f5,0x3c1a356d),LL(0xc56a6e47,0x284e2f15)}, {LL(0x4dcfaddf,0xc6abad59),LL(0x082bb2b4,0xe82993f7),LL(0x23655955,0x3cb46972),LL(0x992551e3,0x8ab06385)}},
   {{LL(0xdaa13ab3,0xcbd6cb99),LL(0x2dc1333d,0x01375bbd),LL(0x972c4440,0x638a7f20),LL(0x24dcb1cc,0x150665c6)}, {LL(0x1ea989c6,0x4044e12f),LL(0x61012ea3,0x204c4eba),LL(0xac2719c1,0x78b8edaa),LL(0x2ab50d99,0x6772643d)}},
   {{LL(0x606d63de,0x94604146),LL(0x693aadc8,0xa876d9b0),LL(0x667044ff,0xf7401ffb),LL(0xb599ecb4,0xab98d73e)}, {LL(0xda5cbee3,0xe2b2048f),LL(0xa2b3da50,0x526e3aa1),LL(0xb4ad2073,0x4d0885e3),LL(0x644a1a19,0x916ce3d2)}},
   {{LL(0x96930e8d,0x952b5747),LL(0xb0cf7f5f,0x2a489fd6),LL(0xa8b3b538,0xbff4b59b),LL(0x6aff1cbe,0xba0e03ff)}, {LL(0xd56b2285,0xfa614adc),LL(0x50d58e62,0x2305edd4),LL(0xe36877e9,0xb349fdce),LL(0x43a6103b,0x5f808fc2)}},
   {{LL(0x86493abe,0x66d8404b),LL(0x9b08ff7d,0x18c92d3d),LL(0x89754762,0x6a60ab6b),LL(0x8233dee7,0xec69fd4c)}, {LL(0x06beadfa,0x32449242),LL(0xe0df7084,0x421caf1e),LL(0xd7969339,0x6f89693b),LL(0xfa30a7a9,0xb9a53713)}},
   {{LL(0x11556d9a,0xf89d9bf5),LL(0xee8cf993,0xe4e9c5f0),LL(0x17ed9a7e,0xe5b2a323),LL(0x93e80c9e,0xd4db3920)}, {LL(0x1fda3726,0xae857864),LL(0xa3e88485,0xe5cb36a3),LL(0xf495b9a8,0xa6b85205),LL(0x38f3b180,0xc1be0108)}},
   {{LL(0x36a1f3a8,0x79d0585b),LL(0x913ba5f2,0xa3d8f17f),LL(0x225acf11,0x1eaee5d6),LL(0x0d32de79,0xd4dfd0a2)}, {LL(0x6b3ceff3,0x0cec324b),LL(0xab447870,0x3acc6dec),LL(0xb9c759ac,0xabbf7e6d),LL(0xa5196938,0x0d5c1f47)}},
},
/* digit=28 base_pwr=2^196 */
{
   {{LL(0x45e7ea91,0x781a2157),LL(0xacadfc40,0x4da3f86d),LL(0x162cd393,0xc81d6c7d),LL(0xad6e60fc,0x2c38a2a1)}, {LL(0xf753479d,0x575b25d6),LL(0xbdec6025,0xc914e08e),LL(0x492d5547,0xf81cea34),LL(0xfb1b6969,0x6bbb8bb1)}},
   {{LL(0x1279504c,0x1ee8082c),LL(0x2c92ffb7,0xa466abb2),LL(0x3e81c7e2,0x4118b26a),LL(0xfc60e33a,0x1a76cc50)}, {LL(0x5736d7ae,0x34998bc2),LL(0xbd1ef993,0x20b39558),LL(0x5fbf2525,0xd669e2ae),LL(0x01cc7626,0xbf956ec6)}},
   {{LL(0xb0ccbaa5,0xce817029),LL(0x279b78a6,0x57ef5bd2),LL(0x4df45d89,0xc9283747),LL(0x2ec4bfd3,0xe86b91a8)}, {LL(0xfe565915,0xe5ab4c6d),LL(0x7c58a042,0xe6574716),LL(0x6301c4bc,0xe141deda),LL(0x8084513a,0x2f95d561)}},
   {{LL(0xeecede3d,0xdc424508),LL(0x386440d0,0x11889b35),LL(0x98de0d77,0x7b229f93),LL(0x300a7447,0x73fced8a)}, {LL(0xe31c8f88,0xf75e1c79),LL(0xbb277e4f,0x8db20bdd),LL(0x2b87c02c,0x8ded0a70),LL(0x4d164c1a,0x166281b5)}},
   {{LL(0xeedd8e0c,0x887356cf),LL(0xe44c012b,0x8afab37f),LL(0xe4aa3eb6,0x0795935f),LL(0xda6dfa57,0x9b9efc0c)}, {LL(0xa8ab0840,0x0ff0f8aa),LL(0xc8561605,0x0f3a4b63),LL(0xd5db9315,0x2ca911ef),LL(0xc8ded9f8,0xef70e5ba)}},
   {{LL(0xa6aae58b,0x443d9209),LL(0x274edda3,0x3d0798e8),LL(0xc2be3c9a,0x5c2d462c),LL(0x439882dc,0xb5488239)}, {LL(0x977d4de4,0x6391bb41),LL(0x1e8245c4,0x7fd91040),LL(0x3b093dc2,0x1a6d3c71),LL(0x7b22fe12,0x423a4e3a)}},
   {{LL(0x3a9a04a3,0xe3156f40),LL(0x297d9aff,0x9b32c4e5),LL(0x62a89850,0x7e0b401e),LL(0xa84ef082,0xffbf542e)}, {LL(0xf990caf5,0x377cc0e0),LL(0xec88ea9b,0x02704343),LL(0x63f96a51,0x846fd46c),LL(0xe9855c47,0x37f5cebf)}},
   {{LL(0xe6ad29dc,0xbd140bd8),LL(0x6a04da28,0x7dca4b10),LL(0xade05b33,0xa84feafc),LL(0x7630aacf,0x44d031f8)}, {LL(0xcdee269c,0x18af2fa6),LL(0x8697a40b,0x1e40571b),LL(0xf0e5f826,0xf71d44ad),LL(0xa434cfe6,0x2a47ddf9)}},
   {{LL(0xad17cc54,0x22b97078),LL(0xf2c105b2,0x223732dc),LL(0xe284fae8,0x25168336),LL(0xb356407b,0x41b1bb94)}, {LL(0x89933a11,0x299e7d7a),LL(0xff7dd9f6,0x19e13d3c),LL(0xf23d7ca7,0x9517bd16),LL(0x1eb978a4,0x9e5e9e34)}},
   {{LL(0x5fa3f92f,0x4c222dae),LL(0xed489ca7,0xd5e38e84),LL(0x70ea613d,0x3d81aca4),LL(0xbe4e88f6,0xc7bed301)}, {LL(0x0757d8db,0x6fd5a7bf),LL(0x7a9181b0,0x1472441d),LL(0x5a90b66f,0x78b78753),LL(0xabdae620,0xe3fd5e91)}},
   {{LL(0x84535653,0xea6a77d8),LL(0x81d7e667,0x8d241deb),LL(0xfaf4ef1b,0x1af73798),LL(0x3e0dae27,0x5e1ae728)}, {LL(0x2f7450b5,0x6a67088c),LL(0xda9cb3c6,0x7bccbe06),LL(0x5b808e05,0x520fabab),LL(0x702b247b,0x84222f68)}},
   {{LL(0xe0bd7ef2,0x2471546a),LL(0x656a62a3,0x27d310dc),LL(0xad35da30,0xb8014eca),LL(0x7f35cd7a,0xbdfdcd82)}, {LL(0x040ae645,0xf1e4d51f),LL(0xf42a4d9b,0x672ffadf),LL(0x2d0be1c0,0x9d874370),LL(0xc6e55471,0xcc3671c6)}},
   {{LL(0xbb9c9667,0x39aa705c),LL(0xc51f661d,0x8c3e584a),LL(0xe570769c,0xe5645b1d),LL(0xbc97abf4,0x81923fda)}, {LL(0x0caac97c,0x51d64f64),LL(0xff847f4a,0x45c17651),LL(0x8cbfa2c7,0xc7a6eaf9),LL(0xba8ab893,0x6c2ab9f7)}},
   {{LL(0xf435624e,0xbdaa2c7b),LL(0x1d961058,0xc113e971),LL(0xa2021a1c,0xb230f1b0),LL(0x521a4816,0x6b34e1ff)}, {LL(0x9b874f4d,0x159dc24d),LL(0xbeaab169,0xeaa0f951),LL(0xb56f4916,0x4f38733f),LL(0xdc9d3ac7,0x4ee689db)}},
   {{LL(0x7bf8d03b,0x720254bb),LL(0xd31d7679,0x78b0e6d6),LL(0xf130d7b0,0x848fb878),LL(0xd3ba625a,0xe8e478ec)}, {LL(0x100dfefb,0xb0ce9168),LL(0xe5098aa8,0xfe1463ab),LL(0xa11ec558,0xf780ac38),LL(0x8e474b9f,0x92f15c52)}},
   {{LL(0x46410cb1,0x3b3892d3),LL(0x03a5a136,0x72097f22),LL(0x98de068d,0xdb3a1b80),LL(0x4b1a3890,0xfb7438e4)}, {LL(0x3839d3d9,0x8a10d5ea),LL(0xf4bd8126,0xd9ad034d),LL(0xd4800261,0x07d108ef),LL(0x9c5d6c52,0x978d98ba)}},
   {{LL(0xecb9ce1c,0x63ae69e1),LL(0x51b28f39,0x70d2b437),LL(0x77f848a2,0xc15696b6),LL(0xd8ab4d76,0x6b6e60f4)}, {LL(0x030bf112,0x33a581a4),LL(0xc5e74a76,0x9cdb1a6e),LL(0x7a950053,0x6c6f6ec4),LL(0xb04ebcff,0xd47dc472)}},
   {{LL(0x5e0970df,0xe85ca76a),LL(0x78988af2,0x74448d98),LL(0x81620019,0x5227649b),LL(0xaabc2027,0x47e2ac62)}, {LL(0xfbffedf2,0xfea15155),LL(0x3b4cb501,0xa565c484),LL(0xd830cece,0x4c523be5),LL(0xc321a440,0x2e2de6bc)}},
   {{LL(0x8d69887d,0xa7d62771),LL(0x4e138de4,0xf9d8ac67),LL(0x9fcb0a09,0xad3fbc08),LL(0xbfc3bc9a,0xcaabb0b0)}, {LL(0xb1c20604,0x84646bc3),LL(0xd1574260,0xf1059ac4),LL(0xeefff298,0x5c15c6a2),LL(0xf3b0a393,0x7975ede6)}},
   {{LL(0x27c2343a,0x0ea9d355),LL(0x4b32e339,0xe21c75e4),LL(0xa7fc353a,0x1438785e),LL(0xe9a1dd56,0x0b8d64ba)}, {LL(0x75347c02,0xcacf9b64),LL(0xcaad57aa,0xf788c83e),LL(0x36ecf2e0,0x90df1ab8),LL(0xf45070ac,0x4db604a3)}},
   {{LL(0xb4c4ed93,0xbc76e168),LL(0x85b65a6c,0x07177b5e),LL(0x00d21309,0x41e3c275),LL(0x76a48f42,0xcc9678e4)}, {LL(0xb1c6256f,0x3a04d197),LL(0xb2cc7330,0x940920a9),LL(0x2523d52f,0x990e4da8),LL(0x5a59d733,0x34709b24)}},
   {{LL(0x8e745028,0x2f0da81c),LL(0xcd5668ab,0x32b5f384),LL(0xee538e7e,0x82744a5a),LL(0xf3eb2516,0x1b019bab)}, {LL(0xd79d735f,0xccbd28fb),LL(0x85f90aa2,0x0bb54a6e),LL(0x9a69ecaf,0xacf5552f),LL(0xd1f1e30b,0xbc51ee85)}},
   {{LL(0xfa25193d,0x12bf8b0b),LL(0x5ba4b3c8,0x3f0f51b0),LL(0x66181f23,0xc1b65deb),LL(0xc0156b03,0xfeb037f9)}, {LL(0xa9dc59ed,0xdd7a0a8c),LL(0x7b57e018,0x20b5c8ea),LL(0xefaadad5,0x0c3ebc94),LL(0x18758eba,0x146d05b6)}},
   {{LL(0x09c7b43c,0xcb952e41),LL(0x1c1b8fb6,0x7f7a0ae3),LL(0x331dfb05,0xbca8a9cf),LL(0xe0db9d7d,0x4a1db4a1)}, {LL(0xe5b9c892,0x988d36a3),LL(0x010ad00e,0x64640e55),LL(0xc011bffd,0x4c33c7e8),LL(0xa0ad217b,0x5d7cf370)}},
   {{LL(0x71f3df52,0xbaf8b74a),LL(0xa5995b20,0x300963bc),LL(0xd6c27636,0x695cf7ee),LL(0x03ac244e,0x74d4d3a1)}, {LL(0xc898e5bb,0xddba3bd6),LL(0xfe3499f7,0x27573a89),LL(0x4b0a6c98,0x666b4415),LL(0xaa4ccfaa,0xf4f3e8c5)}},
   {{LL(0x1a5b919b,0x5f136875),LL(0x670d4b04,0xed8eb5db),LL(0x0d0d73bf,0x4cd83d19),LL(0xbdf22579,0xd2a5c62a)}, {LL(0xc2d04c2b,0x8c41be16),LL(0xbf9ad465,0x5aa33bc4),LL(0x5e00d922,0x36e20646),LL(0x00b70e17,0x9df21e7c)}},
   {{LL(0xd440af4d,0x0601e630),LL(0x963e87dc,0x4aab0d33),LL(0x36d39000,0x2712abdb),LL(0xf9147e2a,0x856d7e3b)}, {LL(0xc8e5d2f4,0xadc4a96a),LL(0x2e70c206,0xac3e5336),LL(0x6f6f3d0e,0x1ee7d838),LL(0xead72426,0x4674ef20)}},
   {{LL(0x6af5f580,0x3a804dd8),LL(0xd74ea5ce,0x724a756b),LL(0x0432d854,0x0c2968d0),LL(0xa4f262fe,0xe3be03f3)}, {LL(0xc84c22bb,0xe446692a),LL(0x9647650d,0x156b3168),LL(0xb5d3d62a,0x4e06bc39),LL(0x80eea174,0xf99d4fec)}},
   {{LL(0xc08f1144,0x3a2b7ae8),LL(0x12dae8d6,0x35e65bf9),LL(0xae3892b5,0xfa0af1cf),LL(0xac408112,0xa57062dc)}, {LL(0x24bf1af9,0xef6a9ec3),LL(0x5decd8bc,0xdda3b476),LL(0x7bed3775,0x9314a36c),LL(0x60aa296e,0x9e254b0e)}},
   {{LL(0x65b9cf2c,0x8be2de80),LL(0xcb3b96cf,0x1b110df6),LL(0x18957e21,0x0f647a12),LL(0x4f907766,0xa1e11238)}, {LL(0xc5528977,0x751a0d82),LL(0x9a4b1260,0x958d8738),LL(0x773658ee,0x99087543),LL(0xf19f74cf,0x18148bbe)}},
   {{LL(0x0a19a374,0x5f50ef19),LL(0x6bdd3392,0xc5bc4160),LL(0xb80ad74c,0x1bdf5e4b),LL(0xed7e68c8,0xc40ec2f7)}, {LL(0xdecef5b8,0xedd7dd6a),LL(0x896c95a3,0x3d29a1cb),LL(0x70ad41d4,0xfa84c325),LL(0xc398c177,0x6a577072)}},
   {{LL(0x7375f2de,0x4f942d01),LL(0x8aa1523a,0x968a7608),LL(0x377e5c4c,0x55dc7da6),LL(0x282b540e,0xb75fff53)}, {LL(0xfd4b6951,0xfee35c15),LL(0xf04ddfae,0x6d1d64f6),LL(0xaf7c8714,0x320f1769),LL(0x482ba6fd,0x2b5f86a4)}},
   {{LL(0x1ab9986b,0xcf691cb7),LL(0x7377ba6b,0x42913d71),LL(0x1e47bf08,0x120b4601),LL(0x764b2661,0xfb514e52)}, {LL(0xa140ae04,0x371c0155),LL(0x2e186763,0x94e65b70),LL(0xd5764306,0x5e440f7b),LL(0x7b8a5eeb,0x3411dadf)}},
   {{LL(0xf0e3e158,0x6c25e519),LL(0xe8898c80,0x46ee66d6),LL(0xec4f9b03,0xa0e9d4b1),LL(0x126c1f31,0xba48d97c)}, {LL(0xbdbf0928,0xb9f96818),LL(0x7f51cb48,0x293ce87d),LL(0x76f09d38,0x077a7420),LL(0xedea4c81,0xc71cb875)}},
   {{LL(0x9ddd1485,0xfeda750d),LL(0x51c10504,0x987876dc),LL(0x75ec7922,0x4d0253f8),LL(0xc676b336,0xbcc15e39)}, {LL(0xb9370740,0x33d533d8),LL(0xcb8c88ab,0xc5c482db),LL(0xc7c08a75,0x1ff3f223),LL(0x401b00fd,0xccfaf564)}},
   {{LL(0x6ac9757c,0x6ba93d3a),LL(0xec2c92a0,0xff09b546),LL(0xc5960be8,0x95d3436c),LL(0x69029082,0x90b7e8cb)}, {LL(0xdb6b32e5,0xbdd1e2b9),LL(0xfd47ad85,0xf4d2e43b),LL(0xcb005dbe,0x8923251e),LL(0x662912e7,0xc21368a0)}},
   {{LL(0x062d205c,0xc7ce2963),LL(0x9542b831,0x1e8f812f),LL(0x818c322d,0x4f8a7915),LL(0xfb678809,0x50073cba)}, {LL(0x0cb91b3e,0xed7b5237),LL(0x60d3fe6b,0x22d1fa41),LL(0x2d690f75,0x3de39063),LL(0xf164ec1f,0x12b2e39e)}},
   {{LL(0x332f408f,0xa28a0d83),LL(0x6a054db1,0xe6d9406c),LL(0x5ddd64e0,0x67369b76),LL(0x02b21c2d,0x6d671707)}, {LL(0xac42170f,0xb9ad3368),LL(0xe5802ffa,0x5e8f5277),LL(0xd9b4a0a9,0x1b4468fb),LL(0x96c24331,0x0daf8269)}},
   {{LL(0xc09ad26d,0x976c2f23),LL(0x19c68d38,0xd47afe88),LL(0xd3d8d84f,0x0e96c03b),LL(0x0932b2fe,0xe05b5fd8)}, {LL(0x347fbbbd,0x13931043),LL(0xb0ccc752,0xe0fa842f),LL(0xc75bf745,0x7295ee0f),LL(0xb0aa9d61,0xebaae0dc)}},
   {{LL(0x6355564c,0xb392d49b),LL(0x887c5a18,0x57e2f166),LL(0x230a278a,0x88b3a014),LL(0x4c194449,0x088e4908)}, {LL(0x43d6af69,0xc6cd309f),LL(0x589a7f7e,0x394445e3),LL(0x031e7c08,0x0610077a),LL(0xa3519f78,0xd05547cc)}},
   {{LL(0x926e26ed,0x0123b543),LL(0x62d06da6,0xcd430b80),LL(0x0dcd6db5,0xddb182d0),LL(0x8eb6e010,0x724c9bce)}, {LL(0x50a4a597,0x985a2f0f),LL(0x900f2a49,0x35f2427f),LL(0x13cbf187,0xce6136fe),LL(0x1086c2aa,0xc893bdee)}},
   {{LL(0x07eca624,0xe2410ccb),LL(0xddf9afb0,0xeda92913),LL(0x5bb66033,0x8fc0cfd0),LL(0x0509ffc8,0x0ab7d29b)}, {LL(0xb3d4f10a,0xc063b004),LL(0xeb8cf642,0xed94a955),LL(0xa272ac4d,0xacfb2f14),LL(0xc4ebbf0b,0x10f2c91a)}},
   {{LL(0x06ea04eb,0x73f6e02e),LL(0x8b97ea93,0xb969e8f8),LL(0x0cd48657,0xa9b27472),LL(0x99264937,0xe1a874ec)}, {LL(0xf794332d,0x34e677a4),LL(0x5ee2daea,0x5e67865e),LL(0xe6886879,0x3fe02b91),LL(0x0f9776ad,0xe113432f)}},
   {{LL(0x6a2c47d1,0x37567306),LL(0x62feb54a,0xf66df9b8),LL(0x3e07ce56,0xf734ee37),LL(0x659809fd,0x50c4982d)}, {LL(0x9daf8faa,0xe2fa768f),LL(0x8b9fd7c3,0x66088ddc),LL(0x333683c6,0xb8265661),LL(0xdff2a0a7,0xe7dacf81)}},
   {{LL(0x5e3da06c,0x1e99d6bd),LL(0xbae05a25,0xbd50e15c),LL(0x802d7b40,0x47a0d997),LL(0x193ef621,0x0a25b51b)}, {LL(0x148ee5a3,0x24d3d4f4),LL(0x022a9df0,0x7012618f),LL(0xf68e648f,0xb3777339),LL(0xd7544352,0xcdfb557f)}},
   {{LL(0x1f912c5f,0x4b0b2d46),LL(0x957515d9,0xddaf929a),LL(0x0ae46856,0x29e4bf1f),LL(0x44e32ab0,0x158b4c85)}, {LL(0x7c48d2d2,0x17935398),LL(0x6f2430bc,0xe4ab6300),LL(0x0d8b24d4,0x71dd7284),LL(0xfc21d7e4,0xd9303af1)}},
   {{LL(0x450f7f6d,0x816c616f),LL(0x3306df19,0x17875d8e),LL(0x087e86e0,0x7ce8d4a5),LL(0x36251f01,0xa53970ac)}, {LL(0xfc98edaf,0x2037f12c),LL(0xabf72b6f,0xc359a382),LL(0x85130fa6,0x06acf1a6),LL(0xadfe4344,0x08f45064)}},
   {{LL(0xdd857b31,0xc01e1f3b),LL(0xc9513734,0x92c2263a),LL(0x589327b8,0x562652d5),LL(0x96a1c164,0xa8edd065)}, {LL(0x79f8df8d,0x2cbf8f98),LL(0x40847dde,0x3d5cf771),LL(0x597c0622,0x69b08ee4),LL(0x8a868f2a,0xfff18c4d)}},
   {{LL(0xf5ad0686,0x28bca3d2),LL(0x2d4eef7b,0xf7992f89),LL(0x3977e15d,0xab10b9cc),LL(0x2db8ef03,0x47912ca1)}, {LL(0xdf27884b,0x1f3e70e6),LL(0x0e9d8efa,0xdd9bb81e),LL(0x279e27f6,0x97a83b6f),LL(0x24daf922,0x47e259fb)}},
   {{LL(0x124e71be,0x49eb72bc),LL(0x3a6778ba,0x01dba001),LL(0x3be03050,0x8d02baec),LL(0x6dd6c8ef,0xe3571b3c)}, {LL(0x2cc11ffe,0x6e1ffbac),LL(0xf4e2e6f0,0x6d725c75),LL(0xf2b53a58,0x96c31b45),LL(0x97f1634b,0xa0e38dd7)}},
   {{LL(0x143ce001,0xe8507959),LL(0x8b49cc63,0xad9a9f52),LL(0x8438b0fa,0x950fd33d),LL(0xbe0cbdf6,0x2b294c00)}, {LL(0x937b00ce,0xb2076b47),LL(0x78041498,0x026153a1),LL(0xe9e53d27,0xe958f12d),LL(0xe8f8fad9,0xf49e1124)}},
   {{LL(0x2bca0ae1,0xb78a5b74),LL(0xcccdc3d3,0x35180dec),LL(0xa97e519f,0x15e4fba5),LL(0xf5b8340e,0xe49dac9d)}, {LL(0xe5978024,0xdbd8ed3a),LL(0xeb105505,0xd181f26a),LL(0x29f57098,0x38364818),LL(0x3900171a,0xd674fe1f)}},
   {{LL(0xf1bd5803,0x5a2ff729),LL(0xeda23387,0x53de7261),LL(0xf0dc417c,0x7f1d84c8),LL(0x5360fa80,0xa65694a7)}, {LL(0x96ed36e6,0x356e4518),LL(0x406bfd36,0x127a52de),LL(0xde925d04,0xb575a98e),LL(0xc0627c4f,0x35fb44be)}},
   {{LL(0x471e745a,0xc85f2c69),LL(0x6213d79e,0x1c01e1ea),LL(0x2f5081f0,0x95ea99a1),LL(0xc3367864,0xdb38bd3e)}, {LL(0xd8880436,0x0e8cafec),LL(0xf9c63d09,0x1d59fd74),LL(0x7f875dbb,0xe57b0b4f),LL(0x77084bd7,0xe266c939)}},
   {{LL(0x2fc1f909,0x0e289c5a),LL(0x86c4fc40,0xece9d225),LL(0x5d79b4b3,0xe0a56fbe),LL(0xd4489041,0x2b96fae7)}, {LL(0xe23c85e7,0x0f66316b),LL(0xadfef0c2,0x2d1a3c78),LL(0x9fbce9cd,0x1aece4ad),LL(0x6c32d32d,0xccd0f334)}},
   {{LL(0xfb9ba6dd,0x958d7a5c),LL(0xe673275d,0xa0052032),LL(0x7f978d07,0x514ffd9d),LL(0x450b76e1,0x544bbce9)}, {LL(0x6b5201b6,0xeaa25d74),LL(0x74d082a5,0x7528a4ea),LL(0x66609e27,0xa08c8d31),LL(0xda7c6fd9,0x5150d1be)}},
   {{LL(0x39930618,0x864f5b4c),LL(0xcebb516e,0xe71e7f1a),LL(0xebf1f8ac,0xaeee7fa5),LL(0x0ea827c6,0x6efcad4a)}, {LL(0x74e21dd8,0x6e0f4ecb),LL(0xf33a7939,0xc5311600),LL(0xa4d93fc4,0xdf62f3c3),LL(0x9a18476d,0xd3b62727)}},
   {{LL(0xc0e1256b,0x0b54f5e6),LL(0x97ba9afa,0xe8987efb),LL(0x41d11c15,0x4b6ea064),LL(0x79b79f0f,0xfed7017e)}, {LL(0x5bd04e40,0x5a6bcf9e),LL(0x8fd3b4bd,0xf3090153),LL(0x82240648,0xa23b5acb),LL(0xb16cf033,0x61d9a8b1)}},
   {{LL(0xc9fbee1e,0x2feb1706),LL(0xd7e07918,0xfaa4cd69),LL(0x447cba7a,0x28562c58),LL(0xa61a1064,0x727926c4)}, {LL(0x97ac7eff,0x1b32db7f),LL(0x452253ce,0xfd968b22),LL(0x5cdd3217,0x69d1842f),LL(0x26f4007d,0xba7689da)}},
   {{LL(0x141c8b35,0x16445a64),LL(0x73c61779,0xc11c3101),LL(0x485300ee,0xa5aa0d18),LL(0x1cc02bf0,0x531b6de1)}, {LL(0xc4efeb2c,0xf8b94155),LL(0xd015a9c8,0x83632663),LL(0xcba18b7f,0xc369b3ce),LL(0xc29e0f9b,0xe11b3ef6)}},
   {{LL(0x903ca95b,0x1a5e9bf2),LL(0xa50cb332,0x2d9aefc6),LL(0xb29ce693,0xb5670264),LL(0xab1d7b7e,0x806d08ac)}, {LL(0xc9107eac,0xcbdfdf28),LL(0x6cdf12ac,0xa8086243),LL(0x903d5999,0xe7d9c315),LL(0xc079d951,0x4379820b)}},
   {{LL(0xbf1edf4c,0xe789ecad),LL(0x47bc7752,0xec086811),LL(0xc2fc8798,0xeea2eeb8),LL(0xe031a96b,0x763183e0)}, {LL(0xf9a6bfaf,0xc7daf0b2),LL(0x4b957cf7,0x1a2a7ffb),LL(0xbf2d2e7d,0xa483c7c8),LL(0x58ff7f9c,0xf96921fc)}},
   {{LL(0x574ee010,0x41386185),LL(0x2780c649,0x62e6a1d8),LL(0x60f2516e,0xdec553af),LL(0x7a04eb11,0x5b091537)}, {LL(0x67eb90c5,0x1b53e9dd),LL(0xddfda333,0xc390a23a),LL(0x480568aa,0xdd4e7c6d),LL(0x59ccbe61,0xd6c1e8a8)}},
   {{LL(0x9107901b,0x1c6fd7a9),LL(0xa211d116,0x5dc4a41e),LL(0xaf1b78a8,0x597e94e7),LL(0x53afcb6a,0xe72da34d)}, {LL(0x74512c24,0xbc364db7),LL(0xb2811e91,0xc26a8fb9),LL(0x290469b1,0xfdd39d7f),LL(0x20612535,0x84515392)}},
},
/* digit=29 base_pwr=2^203 */
{
   {{LL(0x1539cf31,0x7fe996a0),LL(0x0ded7c6e,0x4a3f729a),LL(0x3016f614,0x86f1f299),LL(0x86cb9163,0xc3d44e18)}, {LL(0x558fa36c,0x96984531),LL(0x369c89d6,0x58e8bf05),LL(0xf9ee923f,0x287da114),LL(0xec271fbc,0x2032e984)}},
   {{LL(0xd39207ad,0x91b8579d),LL(0x0b1fe916,0x6f62c725),LL(0xd89e01bf,0x0f1599ac),LL(0x4d1e5843,0x8d9bb86d)}, {LL(0x726e38d1,0x348b90d4),LL(0x52a8c6b9,0xb824a1ca),LL(0xb1d2f6f4,0x984d9309),LL(0x431ec12e,0xefa485b7)}},
   {{LL(0x9d616a5c,0x24cafa66),LL(0x4c9d0ea8,0xc1c7445f),LL(0x90bee7b6,0xf733e085),LL(0xd251d2ba,0xa2f3ece3)}, {LL(0x66aeba6c,0x6e422a45),LL(0x37c1337f,0x35e99b16),LL(0x6d4f8d03,0x52d0fdf7),LL(0x79c92672,0xa043420c)}},
   {{LL(0x76ac1925,0x99725607),LL(0x3442fc58,0x086449db),LL(0x2e311e74,0x8dbab920),LL(0x7ea25561,0x29dee69b)}, {LL(0x19a7cd6c,0x5a62b6ee),LL(0x0d0dd5a0,0xba38cc4c),LL(0x166d0ff1,0x779279e5),LL(0xf48b3dae,0x0eef53cc)}},
   {{LL(0x0f82c6a6,0x0463dcaa),LL(0x11d7d6d8,0x75dfc96d),LL(0x6c100d92,0x61f05e7b),LL(0xe13eabb4,0xa118e548)}, {LL(0xdcdf06b5,0xcc77e3c8),LL(0x6ac25960,0x902d37d6),LL(0x347d7116,0x967d9993),LL(0x9ae33561,0xd2828650)}},
   {{LL(0x8c7c6d89,0x955b7840),LL(0x42c2555f,0xbfa78bc8),LL(0xa69c3165,0x8c56ae3d),LL(0xe33bb1bc,0x72b20e72)}, {LL(0xd1aa6416,0x686870b4),LL(0x7db03cdd,0xf000040b),LL(0xb0a0fd40,0xd25b16a9),LL(0x36815f1f,0xeb89e932)}},
   {{LL(0x349b549c,0xb3e5e912),LL(0x086c4e74,0x801f788e),LL(0x0082ae92,0xafb9ea4f),LL(0x0d740026,0x4e8f27a4)}, {LL(0x05f4a6ac,0xc4f8285a),LL(0x0adcd58c,0xefea5297),LL(0xc52f8c21,0x9d5b6ae5),LL(0xe97af4dd,0x92622a7a)}},
   {{LL(0x79f34a8a,0x39b43935),LL(0x30046435,0x61acf55f),LL(0x3f05fdb1,0xf0a23fe6),LL(0x0c4fa7ff,0x7d6baee1)}, {LL(0xe2daf735,0x253f62b6),LL(0x2948637b,0xe370ead8),LL(0xd84e6206,0xda57c16a),LL(0x0dd22ad3,0xf19ffe09)}},
   {{LL(0x95bf2003,0x701acab2),LL(0x9dff6efc,0x50e4e10a),LL(0x43b95430,0xe637bcf0),LL(0x85050cbc,0xac45cd3e)}, {LL(0x80639e4d,0xc2ebff64),LL(0x3056f603,0xe5af1fb5),LL(0x08b17132,0x302791d5),LL(0xed517904,0x87775ac4)}},
   {{LL(0x4263a566,0xfe64ed1e),LL(0x1d5e8f55,0x735bbee4),LL(0x8294342c,0x9ac61915),LL(0xd4904935,0x0f522e5a)}, {LL(0x7c1e11f4,0x2ee883b5),LL(0xf0c237f4,0x0a2ce30f),LL(0x8d955086,0xf4a7157b),LL(0x022dc2db,0x7ec0462e)}},
   {{LL(0xca391b0b,0x562fb75b),LL(0x3bb1efd6,0x13030aac),LL(0x347f44fe,0x305d9831),LL(0x94b2615b,0x9f70c1ad)}, {LL(0x4162ff22,0xaaf935f4),LL(0xa68d160e,0x2b20f047),LL(0x39222d1b,0x30d52a97),LL(0x6551642f,0x051223b1)}},
   {{LL(0x39620daa,0xae65a5c7),LL(0x6f7c078f,0x8ef6f93f),LL(0xb0177db8,0xb06d52bc),LL(0x68fdf535,0x915cdd08)}, {LL(0xc5183222,0x0070d150),LL(0x7817a2ae,0x2b6495cd),LL(0x0b194f0b,0x3ce47614),LL(0x513bfdfb,0x2eec6acf)}},
   {{LL(0xf3dbd34d,0x725dbede),LL(0x3621fc75,0x01c4412a),LL(0x3c07f048,0x17bd68de),LL(0x62e735eb,0x117df57e)}, {LL(0xb249c407,0xb1596c6d),LL(0xd46c55c4,0xa878f56a),LL(0xb8aa0cb4,0x33385670),LL(0x800ec887,0xc7faa80e)}},
   {{LL(0xd7daf836,0x2cd2814f),LL(0x0d616922,0x877b72b7),LL(0xdb066012,0xea73ca1b),LL(0xb0d4159d,0xbe336c7b)}, {LL(0x0f8fcd76,0xb993b07f),LL(0x8a593562,0x5fdceaba),LL(0xf691ec19,0x716595fb),LL(0x8e68e3c0,0x51a77f61)}},
   {{LL(0x7a7c18db,0xe9e4cdfe),LL(0x7b4f69b7,0x967d3575),LL(0xa9a434c1,0x6dd350a1),LL(0x00c79ba7,0xb92cdef9)}, {LL(0xa6bb0f93,0x7a762493),LL(0x8158ad36,0x6c18cdc2),LL(0xc529ecfd,0xa0bd83e3),LL(0x3962f96d,0x98363c59)}},
   {{LL(0x1d63aa7f,0xd80f45a5),LL(0xb3b32da2,0x8d5eba75),LL(0xa4708858,0x0ef233df),LL(0x52161c61,0x74c3f4f7)}, {LL(0xe6420de4,0xfa9bfe6b),LL(0x97dd86d5,0x96c0c501),LL(0xcfce233b,0x28e6827b),LL(0x58e74d63,0x035cc9a9)}},
   {{LL(0x7948782d,0x9ba64bf4),LL(0x80d9ce1a,0x5e5b7c72),LL(0xf51df862,0x7b9435db),LL(0xb4dd2421,0xe74ab6e8)}, {LL(0x60954c75,0xb0d704db),LL(0x0b59ae5b,0xd31c5145),LL(0xd99ba307,0xe0ff4660),LL(0x986bd82b,0x1a3800fd)}},
   {{LL(0x509a0a50,0xe7e06ab7),LL(0xe4539885,0xbdf63778),LL(0x16ddb433,0xf888320f),LL(0x18e18998,0x0f108304)}, {LL(0xfa834b14,0x27e7ffd6),LL(0xc68b9434,0x16de9a71),LL(0x4d360436,0x53a12e2c),LL(0x5e110b02,0x5ad2c986)}},
   {{LL(0x3cf53e0c,0x3439443c),LL(0x1d65a7a3,0xfeae29b0),LL(0x78ad7d78,0x1e7774f6),LL(0x6fee368c,0x0c79fb01)}, {LL(0xe4faf3ee,0xbec71de1),LL(0x263868e8,0x1a88f3e5),LL(0x90e91a8c,0x975d8381),LL(0x0f999c60,0x69c5a65d)}},
   {{LL(0x8b884857,0xbd3360d8),LL(0x31b7c579,0x04190413),LL(0x142cc145,0x40dd9229),LL(0xdad0b2df,0xb5faab94)}, {LL(0x52df4831,0x3e7d7921),LL(0x228bf67d,0xcf5bd1ed),LL(0xc4980085,0xd8669635),LL(0x1c71903d,0x094b8973)}},
   {{LL(0xc4abb028,0x493a7a3d),LL(0xb4ab8e35,0x0e1a8fac),LL(0x017aa5f2,0x26094ca2),LL(0x021476cb,0x94fcb8b1)}, {LL(0x4abf3bcd,0x57f101f9),LL(0x2d7f12a8,0x1ac2c125),LL(0x2e42789c,0x575259d9),LL(0x22471eb3,0xa64a4a4b)}},
   {{LL(0xe1c00144,0xcc02304d),LL(0x754734b2,0x6269dfb9),LL(0xf14fbc81,0x72e7a183),LL(0x2a05caa2,0xd92a5b1c)}, {LL(0x15efc2fb,0xd593492e),LL(0xd8dd458b,0x1ace7dca),LL(0xaef2ae81,0x576b4bc8),LL(0x351b397e,0x6de6a2db)}},
   {{LL(0x656cf9ab,0x73f13b48),LL(0xaee7e01d,0xc18df1c9),LL(0x560355e7,0x30fb5155),LL(0x9ad059d5,0xd2c9a0ee)}, {LL(0x5e5e0c7c,0xd9f89936),LL(0xf0a6c9d7,0x5d0a2cbd),LL(0x58fa9be9,0x3c2c497d),LL(0x6ac61a2f,0xe6c6fcf2)}},
   {{LL(0x35607bc4,0xf7ec89e3),LL(0x9946bf52,0x17ca00ca),LL(0x180c8bd8,0xee46be5b),LL(0xb2873893,0xd29d5eb0)}, {LL(0x97487b3c,0x348ac939),LL(0xfeef78ce,0xc18f0ceb),LL(0xf797cce5,0xfc648dca),LL(0x442148d7,0xe2467e0c)}},
   {{LL(0xda6dbaf6,0x8e201ee7),LL(0xabd0490c,0xc1a93ee4),LL(0x4de7c210,0xfd0007bf),LL(0x083ffce0,0x02eccb8b)}, {LL(0x97844c8d,0xbba82bbb),LL(0x2747a430,0xb7ff954c),LL(0x18fb5f29,0xb05f0584),LL(0xee7c263f,0x35a29cf5)}},
   {{LL(0x25282da8,0x5fd84efd),LL(0x7682db7c,0xc1fc84c1),LL(0x4bb291e6,0x8422b56a),LL(0x66a56078,0xce379feb)}, {LL(0xe7f045a0,0x1c3c581e),LL(0x6b3f89ac,0x8f42985d),LL(0xa6b2ba59,0x112839e9),LL(0xc2a7b29a,0x3f0c7269)}},
   {{LL(0x79bd3046,0xecdadb6f),LL(0x9617ff6e,0x669f559b),LL(0x68928ca9,0x242bb14d),LL(0xc19cafcc,0x28e2b0cb)}, {LL(0x81330593,0xfb7d8954),LL(0x9fbf8665,0x5c3741fd),LL(0xc3b93e86,0xaf49e83a),LL(0x32822548,0xd60ecf7d)}},
   {{LL(0x7cfb37cd,0x4bf36759),LL(0xdb7af2ed,0x94b0c7f0),LL(0xebf8b461,0x2f1bebf6),LL(0x073e5c18,0x90ebc9c7)}, {LL(0x3774e905,0xe431c793),LL(0xe03265bb,0xb8a4bc2f),LL(0xfabb629e,0x0bee0825),LL(0x84557170,0xbd1481dc)}},
   {{LL(0xe1a010a0,0xa2257b58),LL(0x4688bb89,0x43f5451c),LL(0x91b96371,0xb87a5ff0),LL(0xde3c7b25,0x445dd02f)}, {LL(0x51a0964c,0x44c0c080),LL(0x9afdcefa,0xb0c3190e),LL(0x0044d258,0x14cc65ad),LL(0x374fdd44,0x8c500b3e)}},
   {{LL(0x51b07812,0xed8d840f),LL(0x76530691,0xd414a4a2),LL(0x1468ef8d,0x9db9d138),LL(0x292b3870,0xfc6b7434)}, {LL(0xc9d7ad96,0x80b66797),LL(0x2a9c1e99,0x81e74eb6),LL(0x9e92f64b,0x48657d9a),LL(0x4c851ddd,0xf5c60075)}},
   {{LL(0xd99d5cfe,0x08fa89be),LL(0x4db4addf,0x78b1f26e),LL(0x3523ead9,0x03237177),LL(0x6a281494,0x0147af5c)}, {LL(0x916836b0,0x8db3952a),LL(0xfd365650,0x0632b102),LL(0xccb3f2f1,0x3854a8e9),LL(0x586ad427,0x5048486c)}},
   {{LL(0x17a86e18,0x22de9979),LL(0xbe029111,0xe2ac2321),LL(0x35cc5a17,0xbfd34397),LL(0x525e13cf,0x7a93461f)}, {LL(0x5122d6f1,0xd433542c),LL(0x833982c7,0x41d2d9de),LL(0x8ec24d27,0xe9f1f29a),LL(0xf3b99d58,0x4ae251f3)}},
   {{LL(0x10adb458,0x7234dd24),LL(0x88379ef5,0x0e4b6567),LL(0x748dba5d,0x3007df15),LL(0x35103772,0x1485ef01)}, {LL(0x29c2382f,0xe21a9dc9),LL(0x6b6c1c8d,0xcf7e0c24),LL(0x30550c0a,0xf8a71820),LL(0xb797de2e,0xb30e5c0f)}},
   {{LL(0x03705145,0xbe136119),LL(0xf94aadc7,0xe6d1f720),LL(0x255f5297,0x38ce1872),LL(0xc3143f58,0xbbba4793)}, {LL(0x0984e265,0xda5345fe),LL(0xd895e0d7,0xe93989d6),LL(0xcaab40a3,0xb7392b18),LL(0x65e754fc,0x4a586963)}},
   {{LL(0xa3afd381,0xb3e88445),LL(0x693ad961,0xa6cbab0a),LL(0x257d56dc,0x64d51359),LL(0xbbde137a,0xf9e70fcc)}, {LL(0xadd016b1,0xa33872fa),LL(0x7344f234,0xd1d263d2),LL(0x24ba41b2,0xc2d51210),LL(0xe4ab65cb,0x8c9c830c)}},
   {{LL(0x175b4039,0x3b47563c),LL(0x116b2025,0x53521dfd),LL(0x1a9f1cca,0xe4f3aa89),LL(0xe7cb1d2b,0xcc73485c)}, {LL(0xbf58fe30,0xa6ca61ef),LL(0x531a2b6a,0x5d50e15d),LL(0x22611c31,0x71cfdb41),LL(0x61e3d46a,0x0dc15533)}},
   {{LL(0x479074ba,0xb363c60b),LL(0x24cb405d,0x2a4d8f4c),LL(0x646b7f0a,0x3d3bee13),LL(0x5571af63,0xdfa9194c)}, {LL(0xee76521c,0x951a61a7),LL(0x65eda1f1,0x67466ba5),LL(0x69ebc7ea,0xe41d33b8),LL(0xd4f4848f,0x8b6c992d)}},
   {{LL(0xa5003eaa,0x3572faac),LL(0xabf54df1,0x01e36500),LL(0xac6f3af7,0x6622f12f),LL(0x0a8bb122,0xb5d7c17f)}, {LL(0x240286b1,0xd1fc1b99),LL(0xad968ede,0x519d52ce),LL(0xece28bb4,0xcd1e7d0c),LL(0x93f0b097,0x64ffc699)}},
   {{LL(0x25af7d81,0xb18d51f8),LL(0x19820fb2,0x8d0bb08f),LL(0xaa344723,0xe9f45919),LL(0x5f9e0501,0x558f18ea)}, {LL(0x07cc79dc,0x56eff076),LL(0xd5fa9783,0xf171e880),LL(0x8be7f1fe,0xd5fb41f3),LL(0xd6fe9afc,0x19a34620)}},
   {{LL(0x7d8a042a,0x74c2696b),LL(0x54230ded,0xcf4908c3),LL(0xdb32943b,0x98a870d8),LL(0x52f33e07,0x181cbe5c)}, {LL(0x93709d8b,0x4d9d1172),LL(0x2b2b7939,0xb800c292),LL(0xa8920f60,0xd5a1fb7a),LL(0xbf7df622,0x8d0a7725)}},
   {{LL(0xe9415cf4,0x83a370cf),LL(0x62a4ff4d,0x9f24d1e1),LL(0x0a6ff7cb,0xca33914b),LL(0xda1d1aaf,0x2576f2d8)}, {LL(0xb4793101,0xbb2668bd),LL(0xca990b4f,0xb576672c),LL(0xff1d3988,0xfa165b5f),LL(0xba8c9e2c,0x273d5b18)}},
   {{LL(0x720a5b3f,0x13594ae5),LL(0x2e65454c,0x38c4e04a),LL(0x55d0d903,0xc5b55d98),LL(0xf72db301,0xfabeb890)}, {LL(0x2f3deaa2,0xe471f1d5),LL(0xc5ade08d,0x3f8d39f5),LL(0x6baf9d1a,0xe08486a4),LL(0xe5140d3b,0x77c6d30e)}},
   {{LL(0xd23d4e09,0x828db96a),LL(0x84dcac15,0x2de4856d),LL(0xa6ac856d,0x313c7f8b),LL(0xfe7dea98,0x7c9f671c)}, {LL(0xfebe9e72,0x4d6c14a5),LL(0x6014be55,0x185ac4e6),LL(0x5ed64e23,0x42809988),LL(0xbd6440fe,0xdc9395a1)}},
   {{LL(0x24108055,0x10169080),LL(0x38961403,0xfe0d9cb0),LL(0xceb87d29,0xf88e6a48),LL(0x69618944,0x0365ca2f)}, {LL(0x9fb59ec9,0x1504647e),LL(0xa4aadbb7,0xb6486b3b),LL(0x22ef3214,0xfe8701af),LL(0x5f56d985,0x4c895bc1)}},
   {{LL(0x2e1e68d2,0x6fdc6cb6),LL(0x13f07350,0x0689b22b),LL(0x6d62eb1f,0xba94416b),LL(0x98996d72,0x5a2fcbba)}, {LL(0x04b2afed,0x2ca2a2e9),LL(0x0bf61008,0x5b62c764),LL(0x37f4d486,0x30074e57),LL(0x31865287,0x4e02be2a)}},
   {{LL(0x6842ab16,0x401cfb89),LL(0x5b2eb304,0x440fb52d),LL(0xd22eaa61,0x3245fd38),LL(0x373f201e,0x252120e8)}, {LL(0xb2e724c9,0x4d253f5c),LL(0x27e5b5e4,0x9428d6be),LL(0x6785ee9c,0x00d4c598),LL(0x56208d4b,0x0b7fc5f9)}},
   {{LL(0x92310137,0x4426665d),LL(0xfee8da95,0x75b96cd3),LL(0xb561c6d8,0xaaaac6c3),LL(0x3f09e1d9,0x0784a3c5)}, {LL(0xdcac620b,0xac78c064),LL(0x119b8d90,0x49dd5f02),LL(0x57e5caf4,0xf1f5ebf2),LL(0x0eb825e2,0xd8a9fa2d)}},
   {{LL(0xbfb6a2fd,0x3553633a),LL(0xa0c9ce9a,0x06533515),LL(0x04c71728,0x6decd9e8),LL(0x980b29bd,0xcbc0df55)}, {LL(0x31826d15,0x17b1b559),LL(0x1c5cae17,0xc96ed7d7),LL(0x88cda83e,0x24f58740),LL(0x0c69f40f,0x9e2ee1bc)}},
   {{LL(0x9e5604ed,0x138ebf0f),LL(0xf229f097,0x0577f4c2),LL(0x9762825f,0x0a44f975),LL(0xdd085e55,0x113b8639)}, {LL(0x73acc59e,0x4be02fee),LL(0xada7a17d,0x7829f288),LL(0x84fb30d4,0x086bd736),LL(0xe5338eca,0xb2f120ee)}},
   {{LL(0xfb778d2f,0x21701393),LL(0x6441fd75,0xd46bc61e),LL(0x135b55bc,0x466671de),LL(0x51c0f145,0xee1d9cbb)}, {LL(0x6d9ce27c,0x7a7bce67),LL(0x26d82b1d,0xa8c9b1e0),LL(0x4c87bd6e,0x250bee03),LL(0xd3829702,0xd6b02f71)}},
   {{LL(0x0e555d98,0xf14b3748),LL(0x6f775e78,0xf795e62a),LL(0x8f46de18,0xe9a4e4ac),LL(0xc5ab76ef,0x773bd32a)}, {LL(0x5264cae9,0x4f2dcc68),LL(0xc63a6419,0x453b627e),LL(0xe441c6d9,0xc3aeddd1),LL(0x3027b8f3,0x669500b7)}},
   {{LL(0x88d601e5,0x06b93413),LL(0x76c4483c,0x0e996e87),LL(0x00eb0c21,0xe2ff3a6d),LL(0xf4690674,0x86ec3a73)}, {LL(0xe9f82ca7,0x673d123e),LL(0xbf611d0c,0x952c2765),LL(0x3676497d,0x26ed9a68),LL(0x9d29cefc,0x2c9c0049)}},
   {{LL(0x2ae73af6,0x0b8c12fe),LL(0x6455c8e1,0x555b0ab3),LL(0x4804b006,0xd2f49f03),LL(0x02e829a0,0x408a22bc)}, {LL(0xf2832c53,0xde2a59cf),LL(0x96a54082,0x01332439),LL(0xbbd38f9f,0x11dc4ab6),LL(0xa24764b5,0x0248fd93)}},
   {{LL(0x1374972a,0xc7774c12),LL(0x92764b41,0xc7355966),LL(0x23143092,0x31c10ea5),LL(0x9070137f,0xe89d9f88)}, {LL(0xa504d91e,0x7d074406),LL(0x70b7aa8c,0xc5210379),LL(0xea799dd8,0xa67904f8),LL(0xe7b02c04,0x81e6516b)}},
   {{LL(0xb08cc6fe,0x18daf05f),LL(0xf59d49f6,0xfbbd3061),LL(0x93a78581,0x5429b117),LL(0x1906df65,0x795a4465)}, {LL(0xd51a7866,0x643c37e3),LL(0x663a17b1,0x69b8118d),LL(0x3e8a2c53,0x5e385989),LL(0x50f007d2,0xbc18c2ea)}},
   {{LL(0xb616aa15,0x4adec20a),LL(0xea085548,0x99f77e49),LL(0xc01b9a33,0x9108c205),LL(0x6ef3bcef,0x298fbeb1)}, {LL(0xefd8ba0e,0xdf1a8d2e),LL(0xe9756e7b,0xf0ec9492),LL(0x7ff5fbc3,0x4fd33389),LL(0x03ac8371,0x122a6bfb)}},
   {{LL(0x90d66732,0x7d053c8c),LL(0xf9b2909f,0x83f26571),LL(0x66cba4b6,0x350dd6d0),LL(0x40d0d47d,0x8c71c17a)}, {LL(0x4d0be34a,0x3bf85053),LL(0xe11bd49f,0x91ae4f59),LL(0xa22c648f,0xf8a38b41),LL(0x58abaaea,0xcb96620e)}},
   {{LL(0xa7fabcf5,0xa55cee46),LL(0x79c8fbce,0xd16a8b92),LL(0xcbf048bf,0x26ad700b),LL(0x47bb5f1d,0x83b3ce11)}, {LL(0x6b310318,0x31a48f46),LL(0x00612ef3,0x13a7f781),LL(0xa18db234,0xcd840f2a),LL(0x30611c74,0x3be2a7a8)}},
   {{LL(0x2b1c0447,0xbdf37cb2),LL(0xfe71722d,0x7f572382),LL(0x25535e86,0x085b3566),LL(0x3f5b9cc2,0xb5b43063)}, {LL(0xdee66228,0x7c7cff51),LL(0xb676fd6e,0xe29236ae),LL(0xab0cdb1a,0xf0c0105e),LL(0x06b52812,0x0adc9d6e)}},
   {{LL(0x483baf0f,0xc9e6ca97),LL(0xf9bf5055,0x09b063bf),LL(0xfc5a407f,0x8c4c6b2a),LL(0x99a6f540,0xe29cb487)}, {LL(0xcb9a2758,0x18b72239),LL(0x8a5ed308,0xa0ae9f10),LL(0x6e402657,0x2a2cb603),LL(0xaf6f4d14,0x9c7f52cf)}},
   {{LL(0x70ca9046,0x0ed032e7),LL(0x59cac9e9,0xe4b0b1d3),LL(0x18462dfd,0xd12c87b0),LL(0xbb8860dd,0xa25a23ee)}, {LL(0x507fa3d9,0x6c9a8676),LL(0x218f325f,0xc6bb96c4),LL(0x2386b7b2,0xe141bbb8),LL(0xd4183c77,0xf86a72d0)}},
   {{LL(0xaece96b6,0x35537f86),LL(0x63f7e1fa,0x83aa1df9),LL(0x7ac4aaf2,0xa39ab4aa),LL(0x8a202375,0xb8d0ffa6)}, {LL(0x86514cd8,0xd916df09),LL(0x11902747,0x71f905b3),LL(0x8c744f32,0x6388c2ee),LL(0xa5431096,0x6282e1f5)}},
   {{LL(0x7c388413,0x14bfa765),LL(0x7b4437aa,0x95dd04d9),LL(0x3c39d7c3,0xdf6ca849),LL(0x0c2ddf38,0x85cb1123)}, {LL(0xc401529c,0xf7766d86),LL(0x99a4d031,0xe33416a8),LL(0xb874ace4,0x5c507c3f),LL(0xdad6fcb0,0x0e3a42b6)}},
   {{LL(0x47920742,0x402da460),LL(0xb45f94cc,0xb142d6ef),LL(0x76608dd4,0xc2d613e8),LL(0x5d75d4b5,0xa2c06cdd)}, {LL(0x3c564ff4,0xa1951bc5),LL(0xad1d5ecd,0xe60f126b),LL(0x702135ad,0xa634e765),LL(0x8df44695,0xa5a56a6e)}},
},
/* digit=30 base_pwr=2^210 */
{
   {{LL(0x9e2207b4,0x234b8c7a),LL(0xf7ee9f62,0x1f724f30),LL(0xc58e21b6,0xfa908ca2),LL(0xa74296ae,0x55587744)}, {LL(0x02911ae1,0x7dbe9130),LL(0x9d3af02e,0xc2075433),LL(0x0f3955a1,0x505b724b),LL(0xcaeced66,0x480e1a92)}},
   {{LL(0x446d9f66,0xb20f6128),LL(0xc354b5a1,0xd6e06b14),LL(0x63558aac,0xa72d287d),LL(0xae68a8fd,0x4819be29)}, {LL(0x205fbdf2,0xb024c324),LL(0x210927f9,0x2fca94e7),LL(0xbe658f80,0x74798be7),LL(0xef07c046,0x618e07f1)}},
   {{LL(0xb35a8c3d,0xfba715fc),LL(0xed1beba8,0xc2548193),LL(0x2ceb663c,0xb956c6dd),LL(0xaacafe85,0x13d4ddbe)}, {LL(0x30a29cc3,0x2f8275b5),LL(0xf51b39ef,0x10432e15),LL(0x2509b2d0,0xd6c9277c),LL(0x849b946c,0x4ee0d4c3)}},
   {{LL(0x54b01bbc,0x547ba946),LL(0x055d4821,0x7c56c36d),LL(0x05575f20,0x8e933620),LL(0x3a621cf4,0xaec65be9)}, {LL(0x46287937,0x820b96df),LL(0x733c67e7,0x35cea883),LL(0x58cf3e05,0x30366a3a),LL(0x2da39773,0x2580d565)}},
   {{LL(0xba4417ed,0x7717c42f),LL(0x654c1086,0xb2d66fc7),LL(0x57503cd8,0x07fe918e),LL(0x3cacf74f,0xf9385159)}, {LL(0x63063029,0x157d9081),LL(0x659034cf,0x79c84c08),LL(0xa8048cb9,0x02976610),LL(0x03e81417,0xef822006)}},
   {{LL(0x22e489c6,0x5fb5dd4d),LL(0x81e167e9,0x9a06d9c2),LL(0x6b974c90,0x83fc248f),LL(0x7110dca6,0xb78cab72)}, {LL(0x370ff66a,0x73f8f311),LL(0x3b61d20f,0x8c5049eb),LL(0xc8516e05,0xaac47edb),LL(0x53f0201b,0x2ceba50d)}},
   {{LL(0x0b93fbc7,0x6679dc5e),LL(0xa560bd27,0xf4457919),LL(0xb1acadc9,0x2561bfca),LL(0x46708164,0x338fbb6d)}, {LL(0x8b9cfd27,0x9f407621),LL(0xd3123732,0xe806c1e6),LL(0x7f24a161,0xaa1eafc4),LL(0x68e6650b,0xbee3f4a1)}},
   {{LL(0x5832cd6c,0x453b6181),LL(0x985e90ba,0xc002e337),LL(0x6414f686,0x4b33afde),LL(0x8511fd45,0xf9ab29e9)}, {LL(0x6fb9a688,0x067f0972),LL(0x7202a1b3,0x7db6e14c),LL(0x73a881ab,0x0c15b6e9),LL(0xfad10660,0xc8c324e0)}},
   {{LL(0x87d9f927,0xa997a6d2),LL(0xacd2f107,0x62307f24),LL(0x9c80a742,0xed7b48a5),LL(0xa7c81e7e,0xecd33ae5)}, {LL(0xefa00a94,0xcf05c763),LL(0xd9ee5aa7,0x38844b0d),LL(0x214b293f,0x02e0b05d),LL(0x8a8a510e,0x732e776b)}},
   {{LL(0x6c929e53,0x784cd909),LL(0x56a33da2,0xe436e294),LL(0xce9e06d2,0x68eeb727),LL(0xfce7e2f0,0x637890b3)}, {LL(0xc3fde38c,0xc0681a1c),LL(0x76dda134,0x9cb729d9),LL(0x5c333ece,0xaa69eb97),LL(0x48eed8a6,0xe138a680)}},
   {{LL(0x505dc701,0xd53cbd01),LL(0x6a64c3d1,0x413de346),LL(0x3170a5bf,0x91f6cde9),LL(0x8489b5fa,0x58ffdfd9)}, {LL(0x5c584a48,0xcc0b89d7),LL(0x167f28de,0x74f8ceed),LL(0x8c878c91,0x250fa9f7),LL(0x630adfdb,0xeb960a79)}},
   {{LL(0xc9489dcb,0xe43ed412),LL(0x112d084b,0xcec053a5),LL(0x664c7cd2,0x0fd4fe42),LL(0x82a917f7,0x48ee06f4)}, {LL(0x83cd65f5,0xc5de1970),LL(0x5569c42f,0x3a176578),LL(0x4f876110,0xf24508f3),LL(0x0a415bc4,0xf350374b)}},
   {{LL(0xc63aa8ad,0x9c2b11c2),LL(0x7a51c0cc,0x6ac1ae12),LL(0xe1db428f,0x75acd0d7),LL(0x19800684,0x9e391227)}, {LL(0xb1050426,0x4f89e9c6),LL(0xdaf99eee,0x099d97cc),LL(0x1ffce97f,0x27a19ad0),LL(0x3c038d77,0x05fad057)}},
   {{LL(0x8597d09c,0x96d6c678),LL(0x1e1d8b57,0x38f6336c),LL(0x6330ace7,0x1f945bef),LL(0x613f9faf,0x9d627bbb)}, {LL(0x19176cb7,0xc0e7f21b),LL(0xbb9db710,0xf0e09be3),LL(0xc650133c,0x16b06011),LL(0xf3673d7c,0x8629b975)}},
   {{LL(0x67115575,0x8c8230b2),LL(0x9c9f30c0,0x9bfc55b1),LL(0xce93fd71,0x132d0e07),LL(0x511c5947,0x08e4736e)}, {LL(0xfe881630,0xd54a098e),LL(0x98efa501,0x8ec67a85),LL(0x2267db00,0x72975dc7),LL(0xa338290d,0x3d6fc706)}},
   {{LL(0x55ade88b,0x0c891082),LL(0x4b212e85,0x0525b501),LL(0xb61362fa,0x9ede010b),LL(0x881eecac,0x52f3d088)}, {LL(0xbc6f0ae4,0x49957b6e),LL(0x1659701d,0x25fe7263),LL(0x07b76f45,0x41e9b7f5),LL(0xbda77d42,0x5f2ad664)}},
   {{LL(0xa9c418c4,0x5bdcb490),LL(0xe500a527,0xd0e2c38d),LL(0xca83fada,0x0af29f6b),LL(0x62273db6,0x1f75b1f2)}, {LL(0x9e857e57,0x8454f751),LL(0xb9e79612,0x3fb816d1),LL(0x6412b5f3,0xbe3040ae),LL(0x843ca984,0x99535136)}},
   {{LL(0x3257f19d,0xb26ec8a8),LL(0xe54dd785,0xd32dc622),LL(0xf8698ab5,0x0c52e874),LL(0x79183316,0xf9a60af5)}, {LL(0xf7f4496a,0x38575d53),LL(0x3d5cd0de,0x33adfd1e),LL(0x7f502017,0x2133f4a1),LL(0x3e8676f8,0x46c09393)}},
   {{LL(0x3fb4c7fe,0xca8a5a58),LL(0x328ff257,0x2ad58826),LL(0x13b8d08d,0xd9264875),LL(0xdc5a845a,0x661ae2b2)}, {LL(0x49a408d3,0xd2dcaa06),LL(0x85c21e84,0x9ef164f8),LL(0xb7819b61,0x55efaf85),LL(0xf504c32a,0x9488bb1c)}},
   {{LL(0x9bb777fc,0xb571518c),LL(0x82415187,0xf369c391),LL(0x2720284b,0x2d7c5dd9),LL(0x4eec41cc,0x6feab634)}, {LL(0x24ecd0be,0x2522d5db),LL(0x00338736,0x1fca0d97),LL(0x20145279,0x74416105),LL(0x0496e925,0xf4492e1e)}},
   {{LL(0xbc33b547,0xa62e09fc),LL(0xae063835,0x9434475a),LL(0xa139b769,0x51edd69f),LL(0xe5166a9d,0x17bbe224)}, {LL(0x1b4b6c84,0x6ecb0a02),LL(0x97968c70,0x16439490),LL(0xbc8aa671,0x75af0456),LL(0x3b4411ff,0xaef056ab)}},
   {{LL(0x05cef121,0x686b7714),LL(0x078f4500,0x5ad6bdf3),LL(0x072e70eb,0x56df858c),LL(0x254c0917,0xa0fc5e6f)}, {LL(0xc15bf9cf,0x1a99de09),LL(0xd008aacb,0x8aeb587a),LL(0xb900d652,0xba2d8c53),LL(0xad0f69b6,0x60eb5d0c)}},
   {{LL(0x10b27762,0x27098ff8),LL(0x13264ed3,0x33329ca9),LL(0x887e3f40,0xffceaf40),LL(0x930df9ef,0x854b8284)}, {LL(0x7d5627be,0xdda913a8),LL(0x35e9823b,0x8eb94d64),LL(0x2eb9e9bd,0x94c527fd),LL(0xf57b9f74,0x18335b1b)}},
   {{LL(0xd193a526,0x3c44dac0),LL(0xe2b2d54f,0xd0717099),LL(0xc4c67d11,0x65624fb4),LL(0xbccedad8,0x04aa7033)}, {LL(0x31470c52,0x0c522fac),LL(0x33b05d54,0x08eb33ca),LL(0xb4565e57,0x940e0693),LL(0x7be56370,0x7e2fd553)}},
   {{LL(0x12e206ff,0xf722793c),LL(0x155d3d02,0xb57e1f23),LL(0x94fc6ce3,0xd68be193),LL(0x22d4815b,0xb0f3606c)}, {LL(0xb62d5741,0xdaf995a9),LL(0x3e8f5df5,0xa7d19980),LL(0xe4631583,0x7bcdb661),LL(0x5a97dc7e,0x013193e3)}},
   {{LL(0xfc73494b,0x8210be46),LL(0x57e10efc,0x508817ee),LL(0x126f2466,0x7b6b8da2),LL(0x177bee35,0x2f3b0ec6)}, {LL(0x48db1eef,0x5ceb71e0),LL(0xdc62b6bc,0xd989d9c3),LL(0xf78fac92,0x2cc38cb9),LL(0x955ba5f4,0xcd2a009f)}},
   {{LL(0x453b668e,0x65a74191),LL(0xc081b447,0x40e9dc38),LL(0x8c3fdf2c,0x48eb63bf),LL(0x5763071a,0x7845cf66)}, {LL(0x787754ca,0x30d9b771),LL(0x8783a05a,0x10b3729f),LL(0x6ab570d9,0xf8090b3b),LL(0x502b3558,0xc1dfbde1)}},
   {{LL(0xbe4d36ec,0xa568f5d0),LL(0x28e952fe,0x1e9e5393),LL(0xeaad6b24,0x768113f9),LL(0xa8fbede9,0x2bc798fc)}, {LL(0xaaa9010d,0x2c787f9b),LL(0x761834ca,0x32cbc77a),LL(0x032c25d8,0x449e55f7),LL(0xca6216dd,0xe824a5bf)}},
   {{LL(0x17ac1130,0x3beaec52),LL(0x5e9d1aa8,0xcc28c64b),LL(0x3af8c635,0x355d68bf),LL(0xd3d1adaa,0xcd12e443)}, {LL(0x6c2030d8,0xa456daca),LL(0x62427eff,0x0dfe5bbb),LL(0xf69e2484,0xae45325d),LL(0x7245c979,0xfc2a90ab)}},
   {{LL(0xa008b446,0xc34f38e9),LL(0xaac355e0,0x5e86163d),LL(0xd586a2fa,0x61432646),LL(0x3d92e0c3,0xc68c7c8e)}, {LL(0x020c1dd6,0xbfa8c268),LL(0x8bbcc223,0x25788741),LL(0xef62642c,0xbaf8c9a3),LL(0xa8c496d2,0x6d2f1ae5)}},
   {{LL(0x44e57ce4,0x92d1c805),LL(0xaacd2100,0x34cdf4a3),LL(0x31e9c649,0xd5b89e4d),LL(0x232cfc85,0x558a6e26)}, {LL(0xcea31a31,0xb40f3e4a),LL(0x35c5c924,0x346c1e07),LL(0x3fcede81,0x8ffedd8e),LL(0x3b55d143,0x35269ba3)}},
   {{LL(0x848bdc53,0x0366065a),LL(0x078554dd,0xba2af074),LL(0x19ff3b4d,0x3c755fba),LL(0x35a22cbb,0x5ea93372)}, {LL(0x1eb3e23b,0x0e55fe02),LL(0x765dede4,0x2626ecca),LL(0x81f445da,0x187bf094),LL(0x9df30578,0xba011017)}},
   {{LL(0xd72507f2,0x81148037),LL(0x5db072d0,0x3a5841fc),LL(0xbd910aa1,0xfd631862),LL(0x23881c60,0x17b22d68)}, {LL(0xfcc13359,0x6fa799cb),LL(0x2d39fc5a,0x55c40219),LL(0x9f1f6515,0xd50bfff6),LL(0x2e30fa1a,0x575090b5)}},
   {{LL(0x9ba20c27,0x70343a0b),LL(0x749306a5,0xef34db86),LL(0x4ba85f8d,0xd7ad61d2),LL(0x7e091a33,0xe4d24ad8)}, {LL(0xfc348493,0xbd6b49b9),LL(0x299c325d,0x4f11b543),LL(0x55036e93,0x4574a254),LL(0x676b75d9,0x534a623e)}},
   {{LL(0x7d05afbd,0x54aa3d8a),LL(0x7f3f2c90,0x13349009),LL(0xa94b18aa,0xcd03de0f),LL(0x855c050d,0x13123c18)}, {LL(0xf598b5f8,0x747c3cda),LL(0x8644df20,0x0f7ed9b6),LL(0x08d73585,0x45e010fd),LL(0xf8cec4af,0x3b0ff430)}},
   {{LL(0xb028a832,0x3745a41f),LL(0xc4106172,0xcd6d2468),LL(0x1eceac6f,0x56c5a9b0),LL(0xe1e6e980,0x769c1285)}, {LL(0xfa113196,0xbd163a36),LL(0x5840c242,0x206ffc36),LL(0x9c57ef67,0x12de1147),LL(0x03bea885,0x50260273)}},
   {{LL(0x9684d63f,0x3099c21b),LL(0xc7c66691,0x06adb196),LL(0x3d63b3be,0x8464492c),LL(0x0bd38c15,0x86024ef4)}, {LL(0x10565cbf,0x226022a2),LL(0xc9899033,0x2ae6b298),LL(0x262ffa14,0x5564856b),LL(0x7e038b55,0x9472d0e1)}},
   {{LL(0xb1dd268f,0x1b17ea07),LL(0x4899352e,0x8340b9d5),LL(0x75242992,0x4f159400),LL(0xe6727e9d,0xe4392a31)}, {LL(0xaef59026,0x4df1ef86),LL(0x950cfee6,0xe40671ff),LL(0xde4dd990,0x7b36d1cd),LL(0x3366ff4b,0x25df10a6)}},
   {{LL(0x584ef631,0x83fb7e59),LL(0x91af7b6a,0xf12dd400),LL(0xe26f11c7,0x4a5ae41e),LL(0x96d90445,0xeb86d5dd)}, {LL(0xa355d0a3,0x028ae37e),LL(0x99260127,0x3c118ef4),LL(0x76f51bd7,0xb8c7538c),LL(0x5fbadc4d,0x66b90aae)}},
   {{LL(0x17dfd0cf,0x078de9df),LL(0xfe44b17c,0x938df6da),LL(0xc40bc950,0x4a09616b),LL(0xbc969aac,0x0b507845)}, {LL(0x35f7fb82,0x23bae091),LL(0x9ad29b83,0xebc04d37),LL(0x26a3c5fa,0x9fa48a5b),LL(0xf08f3d8c,0xf67c661c)}},
   {{LL(0xb2e9c3a1,0x21825747),LL(0x46c3eb43,0x3520a82e),LL(0x9646183d,0xe309ae9f),LL(0x26dac288,0xa19c31be)}, {LL(0xc76623de,0x3c37018d),LL(0x64b51087,0x59b9a0fa),LL(0xf1954f4e,0xa32f8357),LL(0x1f6da143,0x24251df4)}},
   {{LL(0x01fb07d0,0x825c61cf),LL(0xf2f0243c,0x69ae93fd),LL(0x0943f7bd,0xd507c99e),LL(0x463ee64a,0x5e6dfb09)}, {LL(0x29afd00b,0x10a3c32a),LL(0xbe8cbada,0x92179b5f),LL(0x7d89f9a6,0x7bebad0a),LL(0xd13b3073,0x7ba9bbf6)}},
   {{LL(0x6481cc75,0xfc23c215),LL(0xfeca7532,0x6a654801),LL(0x0fed841f,0x20785ec0),LL(0x25269b21,0xcb612be9)}, {LL(0x414a0fab,0xe9a9c09b),LL(0x6b4fa0dd,0x5404c7a7),LL(0xcb418588,0xde62dae8),LL(0xc594334e,0x2d80afd4)}},
   {{LL(0x95366a44,0xfe454df1),LL(0x755cf8b2,0xda7626c3),LL(0x41397051,0x4446f0ab),LL(0x70eb8b23,0xd1788064)}, {LL(0x977b040a,0xbc7737f1),LL(0x57590edb,0xbfb39418),LL(0x343a7333,0xb094b4a8),LL(0xeb91372f,0xb15912ce)}},
   {{LL(0x0b56002c,0x584e1d5e),LL(0xaa0cb90f,0x1460ce24),LL(0x8f7ffcb6,0x58f0c144),LL(0xbe0d802e,0x56e39f33)}, {LL(0x529458d0,0xb02a6edb),LL(0xa730f9d5,0xa0fbae74),LL(0x1bf69928,0xd98c0ac5),LL(0x796f12e9,0x5c9f888f)}},
   {{LL(0x22e065c8,0x749207b0),LL(0x88c61143,0x6e0232a4),LL(0x837adda6,0x27f30808),LL(0xd0c215d5,0x0b748853)}, {LL(0xbf076ba7,0x97bc4008),LL(0xf157f4d2,0xadae0275),LL(0x8bcba88b,0x394e5d7a),LL(0x23ef50ad,0xf995ec14)}},
   {{LL(0xaa9b60a9,0x6b207f9c),LL(0x52f9979f,0xcd7509c1),LL(0x0834e0ad,0xe3e8f6dc),LL(0xcd5b1314,0x6e2a4165)}, {LL(0xfd60d975,0x073a2db3),LL(0x2c053b7a,0x5ad92ca4),LL(0xfba97ae8,0xceb10220),LL(0xd265e913,0xab82f6a0)}},
   {{LL(0xaa68a13d,0xc7a755ad),LL(0x10dd277a,0x56c13360),LL(0xdef56183,0xbbcf6411),LL(0xb863a4e1,0xebffe360)}, {LL(0x814e8aec,0x67ff26e5),LL(0xa0804732,0x90553561),LL(0xad5fe672,0x407396ac),LL(0x11ad53af,0x053a068b)}},
   {{LL(0xc652cb95,0xb518dd04),LL(0x57e2b99f,0xc818563a),LL(0xb96432a7,0x217cf87d),LL(0x8cc2fdcd,0x7fdc35be)}, {LL(0x4effaebf,0x8c2ef271),LL(0xf427c7c2,0x21879369),LL(0x5197ba08,0xd80ebbea),LL(0xfc6f4c66,0x1b00db45)}},
   {{LL(0x3828a5c0,0x99b9d2be),LL(0xd24a69e8,0xa8855350),LL(0x3ceaa821,0x2f919695),LL(0x04296439,0x89390c91)}, {LL(0xc5cf8a0c,0xc12b3852),LL(0x73afc431,0xdcf234ea),LL(0xf4adf01b,0xdddf5568),LL(0x1b426097,0x2d04fc76)}},
   {{LL(0xcb27c5e0,0x02a21dd5),LL(0xa6b2f9b3,0x01b842c3),LL(0xd8bd7a07,0xefbd483e),LL(0xd13a781e,0x0220825e)}, {LL(0xde802c17,0x8aa029a0),LL(0x3a3f0fbf,0xb62fcd6e),LL(0xe9635f48,0x80558aff),LL(0x65dbeb2d,0xbdc6851d)}},
   {{LL(0xb5412271,0x99f48286),LL(0xe242a8a3,0xa53ef798),LL(0xc0371086,0x41d18606),LL(0xfcb6d1aa,0x5df3379c)}, {LL(0x00a7a13e,0x53f2f5a1),LL(0x2bf51e2a,0x3565a6eb),LL(0x930c5a71,0xa2832b52),LL(0xee2abfcb,0x66071ec7)}},
   {{LL(0x3496a532,0x75ce0a56),LL(0xbe0d69b7,0xa316dfbb),LL(0x353e94fc,0x35438d6a),LL(0x1e0ce775,0xf53433c1)}, {LL(0x22ff3a1d,0x47ea3e8f),LL(0xcd7ccdb6,0x60ebfba8),LL(0x33c475d0,0x47c6b6e2),LL(0xb7959fd5,0xd18637e7)}},
   {{LL(0xa1ae3404,0x8d7a35ce),LL(0x75b832bf,0xf15c71d6),LL(0x98b9d24b,0x65047419),LL(0x0dcf73f4,0x28625a55)}, {LL(0xc7c99478,0x5aa9dce8),LL(0x0bde8d53,0x752d1625),LL(0x93e99ee4,0x7255ecfa),LL(0x22706f48,0x1c53bf71)}},
   {{LL(0x4d478014,0x28335451),LL(0x3f392c0a,0xd64b05ff),LL(0x4fba1661,0x1d9ac89d),LL(0x34828cd8,0x281a0ffb)}, {LL(0x577ed419,0x07abacdd),LL(0x3cfb2c56,0xa66949f5),LL(0x847ebe65,0x38e69105),LL(0x44d6236d,0x8fbbba5a)}},
   {{LL(0x725ef820,0x0c85bd64),LL(0x25a152a7,0x4ef91524),LL(0xb019cebf,0x5237ef0e),LL(0xc9a7724f,0x48203f41)}, {LL(0xc55fc0d4,0x1f61885c),LL(0xbcb3daeb,0x2c4dd07a),LL(0x4b7dafc5,0x9855d5e7),LL(0x5f3769af,0xd76e6fdf)}},
   {{LL(0xbb547be6,0xb834df1a),LL(0x3e7a9586,0x43056b12),LL(0x4375fc7e,0x7459e0bb),LL(0x9c85fc6b,0x5f5f295a)}, {LL(0xbb23b709,0x3f2d2bb7),LL(0x955983ce,0x0b53bd8c),LL(0xfaf68dcc,0x5aee93df),LL(0x509f09dc,0x5575770c)}},
   {{LL(0x40b1b744,0x2bf23c0c),LL(0x9a5bb913,0x524154bb),LL(0x296bdb2e,0xb0e93d76),LL(0xda0b2925,0xb3c71f5b)}, {LL(0x0c617939,0x8e77ae7a),LL(0x1aca9b0a,0x2bfea97e),LL(0x7897c5a8,0x8e3317c9),LL(0x4cee2716,0x850ddefb)}},
   {{LL(0x107d9186,0x684ceee9),LL(0x0082c312,0x48708423),LL(0x6c556897,0x5300137c),LL(0x7e99bc76,0x6d464401)}, {LL(0x8808ca2b,0x6b11e1e3),LL(0xcca6433e,0xedd30eee),LL(0xaa072ff0,0xa9099f60),LL(0x5830f69d,0x774662ec)}},
   {{LL(0xf94547d9,0x0770355c),LL(0x42967865,0xb5041edb),LL(0xe585a93b,0x169a6274),LL(0xd04d6a81,0x06cebf5e)}, {LL(0x2ebc1791,0x0a59450f),LL(0x765ac18a,0x69fd006e),LL(0xa54f7e7a,0x4711ec9c),LL(0x819c6af9,0xd72c8d58)}},
   {{LL(0xb5418e15,0x89c97c6c),LL(0xa558a854,0xd520b03c),LL(0x0d76773a,0xe3c24aca),LL(0x67e5110e,0xc4deb5ce)}, {LL(0xcbb04ba4,0x5bb40152),LL(0xde1b628a,0x672563b6),LL(0x2e8d9e54,0xaec916aa),LL(0x3c60ac70,0xa4e8cb47)}},
   {{LL(0x452d5064,0x54a03e39),LL(0x1e7bb355,0x1e405c2d),LL(0x3bbd3ab0,0x2ab2d5df),LL(0xdbc9fbd8,0x0808410e)}, {LL(0x27f23f6f,0x4180ceea),LL(0xba1d6beb,0x2b965b35),LL(0xf66d6546,0x14f1f010),LL(0xf85cfb4a,0xefdca6a8)}},
   {{LL(0x6614c1c0,0x69e6188e),LL(0xe07cb7f8,0x00bd1531),LL(0x4bb7ee68,0x1b90f515),LL(0x51abb1f3,0x8afdf466)}, {LL(0xb5f34316,0xf59a7327),LL(0x64c7bf22,0x43c3c19d),LL(0xcdb00a2b,0xb275733f),LL(0x602915ba,0x0160df79)}},
   {{LL(0x1baea574,0x1ae4ee9a),LL(0x03ae5168,0x0d922f28),LL(0xca691124,0x07df28fd),LL(0x8dd457c4,0x5aad2f31)}, {LL(0x137384ca,0xe0d4f443),LL(0x6620ea8c,0xd93d424a),LL(0x5067797a,0x21d544d3),LL(0x9d8a15bc,0xc8a8cc99)}},
},
/* digit=31 base_pwr=2^217 */
{
   {{LL(0x941d80a3,0x610f0e26),LL(0xad36514e,0x30927879),LL(0x98f22601,0xaa2dfd48),LL(0x88c8b0f6,0xbc5b31b7)}, {LL(0x6c841cc8,0xb1281f37),LL(0x5a412b84,0xdae16719),LL(0x828f210d,0x9ec1f6c8),LL(0xe8d92901,0x1935d576)}},
   {{LL(0xaf3840f2,0x47247921),LL(0xdf3fcdfc,0x348325d2),LL(0xc43961bd,0xef578508),LL(0x1bd98c29,0x7d5e8ccd)}, {LL(0xf8a30164,0x59cdba10),LL(0x7cb8c653,0x0757408f),LL(0xc3056ef4,0xcd7ed73f),LL(0xfb99cd1b,0xd28e7cc1)}},
   {{LL(0x6bb62286,0xa8a67494),LL(0xe7d87467,0x8d6ef816),LL(0xf673b6d5,0x3529f938),LL(0x8bbf101b,0xf5c0ee76)}, {LL(0x49fdc949,0x3768ed61),LL(0xcf405ee0,0x53b6999e),LL(0x1718e51a,0xbf0108a0),LL(0x5181ebd7,0x38e609ec)}},
   {{LL(0xd8595159,0x82051640),LL(0x579a3876,0x30e1c706),LL(0x0298a67c,0x091154c6),LL(0x51132d27,0x76d9a9c9)}, {LL(0x5c661a4d,0xe41de8b7),LL(0x2a6dcbaf,0xf24b5e96),LL(0xd714e045,0x1ed4cb0c),LL(0x0c187c70,0x605926a4)}},
   {{LL(0xf34b1c93,0x97672559),LL(0xe0b43506,0xa256be98),LL(0xe1a98eb3,0x7fcdd412),LL(0x295a061b,0x7fcfcd84)}, {LL(0xcc2386cf,0xddbac22b),LL(0x928c2556,0x7741adb7),LL(0x4e6e1288,0x3a2bb869),LL(0x025bb4a1,0x53ed11da)}},
   {{LL(0x4108c8e2,0xb114bd67),LL(0x3deb8e23,0x85948c6b),LL(0xd0e9434c,0x6a9e05d9),LL(0xb9c4fd70,0x395060b7)}, {LL(0x18893751,0xa0ccfd16),LL(0xbbf65477,0xa941ff60),LL(0x59423e35,0x34ada238),LL(0x7e570058,0x4ba7a7d1)}},
   {{LL(0x869ae596,0x3d05d455),LL(0x93368325,0x92a15400),LL(0x62eb7ffa,0xbd64ae0a),LL(0xd34b2c59,0x3f263a7a)}, {LL(0xa22244e1,0xe95eece3),LL(0x706fc500,0x39ccef58),LL(0x22f1d2e6,0x39a88f93),LL(0x6d47c670,0x1ec796b3)}},
   {{LL(0xaa1ff4af,0x1558e0f2),LL(0x390503a4,0x61f43487),LL(0x3c4f76f1,0x66164732),LL(0x9e13432e,0x50d0706e)}, {LL(0xf5eba0fb,0x5f1a87ca),LL(0x80bda2c5,0x28a95c0f),LL(0x12ae6462,0x10d693e0),LL(0xf45e6ba6,0x79871340)}},
   {{LL(0x887051c0,0x8abf0cad),LL(0xb3c6b540,0xd09f571d),LL(0xe30ab25d,0x3fb2e16b),LL(0x539e8bc4,0x12e057a7)}, {LL(0x733c2597,0x6efe71ce),LL(0xe71058ac,0x72fa0df5),LL(0xccc037bf,0x49f14d06),LL(0xceb3eb6c,0x9a3ceb03)}},
   {{LL(0x4195370a,0x9895e908),LL(0x927345e7,0xa75007e5),LL(0xd3850711,0x00b4c212),LL(0xdfee8b34,0x76e4060d)}, {LL(0x184c1d07,0x15801862),LL(0x302f5711,0x234e39c0),LL(0x6aa74204,0x4c0bd387),LL(0x6f496836,0x0515eddc)}},
   {{LL(0xc849afbe,0xa9fd0cb5),LL(0xcad5c0aa,0x041df5ba),LL(0xddff259e,0x9a54af37),LL(0x9b550a8e,0xa3f156bf)}, {LL(0x7e3298d4,0x4df2d33d),LL(0x65ff0e1a,0x0957a0a0),LL(0x1e2b3a45,0xff7fb43d),LL(0x1a73285a,0xb86d386a)}},
   {{LL(0x28b18e93,0x6e283c47),LL(0x4b4132ed,0x5458b92f),LL(0xba674332,0x7026345e),LL(0x5c9fc32d,0xc8e38151)}, {LL(0x58e7b4fe,0xd6aaf8e1),LL(0x4267253a,0x3e77a5c9),LL(0xa0949607,0x6441cba2),LL(0xdee20b2e,0xfa205185)}},
   {{LL(0xf5b44600,0x64f3d576),LL(0xef5c8446,0xf12125db),LL(0x487a0116,0x1467539b),LL(0xf380229a,0x3aa0fa49)}, {LL(0xd79f36dc,0xcc6586f1),LL(0x1b9a0b42,0xebcf653d),LL(0x9c1df729,0x68af006d),LL(0xa7903ee6,0x507e055a)}},
   {{LL(0xafd8ac9b,0xd1b48ef8),LL(0x3b5f6b51,0xd8920af7),LL(0x983339c8,0x9629e789),LL(0xfa9248d3,0xbfd2d787)}, {LL(0x9a078882,0xb50ca302),LL(0xf5cb0529,0x1c34f848),LL(0x1646a9f8,0xb9b01519),LL(0x80d53f9d,0xc9e3679e)}},
   {{LL(0x1af3e5f2,0x5a63e822),LL(0xff58e3d1,0x05480ad8),LL(0xd6b3626b,0x2d241643),LL(0xc1eda15f,0x33b15660)}, {LL(0x8528e5d6,0x3e74f855),LL(0xf63188f4,0xafb6dc9a),LL(0xaeeb1d32,0x0cac44cb),LL(0xa627eff8,0x50661046)}},
   {{LL(0x64b49667,0xadc4b012),LL(0x1e05f760,0xa4bdafa7),LL(0xf185d27a,0x171b28b3),LL(0x33425747,0x987e5163)}, {LL(0xc3864a65,0x7c42ac4e),LL(0xbf449c12,0x2dae1bb8),LL(0x06965832,0x680d9743),LL(0x7e31d9f4,0x6ac1ef01)}},
   {{LL(0x579d6ae4,0xdef57433),LL(0xd5240bf9,0xe055b087),LL(0x90a5e069,0xe4dbbe60),LL(0xddb3dc15,0x2161d5fe)}, {LL(0xda297b71,0x7d303012),LL(0xd468046c,0x04481034),LL(0x0ac93c6c,0xaa94d5bb),LL(0xd8d8f53a,0x95bacd45)}},
   {{LL(0x3e03e895,0x790a5d6f),LL(0x44fa5a81,0x27efd502),LL(0xe5998b32,0xd9d35230),LL(0xf22ade19,0xb36a0c07)}, {LL(0xf979a2fe,0x46ec8691),LL(0xced8cb94,0xa67ba933),LL(0x2f856ab3,0x00d07245),LL(0x3c925dae,0xadc9ff42)}},
   {{LL(0x563038a5,0x0e4eaa25),LL(0x8a8f6483,0xfef7e89c),LL(0xace61af8,0x50433633),LL(0x2e1a3515,0x8a1589e0)}, {LL(0x5fdcb1ac,0x99f21e29),LL(0xc9a466da,0x8fd2d411),LL(0xf56b7f13,0x55323c6f),LL(0x5cff438c,0xa016d64a)}},
   {{LL(0xdc05b5cc,0x3e3dfcbc),LL(0xfc3c70ec,0xc1956ca8),LL(0xe63f02df,0x7dbbd169),LL(0x240b87c8,0x95206689)}, {LL(0x1aa6d48a,0x7bacda5e),LL(0x39280f78,0x51dcf19f),LL(0x660abac2,0x1511ae04),LL(0xd905af53,0x3a95adc9)}},
   {{LL(0xea464737,0x0c8c4330),LL(0x34fc4b51,0x989c09c4),LL(0xe2cf3376,0x1852120d),LL(0x25c04de3,0x5a1cb8a8)}, {LL(0x75fe7596,0x50486f98),LL(0x223030b1,0x8cd78d2e),LL(0xcfa1ab11,0x524cb8f8),LL(0x5a15b0b9,0xa716ea3f)}},
   {{LL(0xb902d114,0x7618e95e),LL(0x084ebf5d,0x0a1a4146),LL(0xe3f87683,0xdfb909e9),LL(0x4107410c,0xa0b7eee1)}, {LL(0xf02b0e12,0xa45a551c),LL(0x9efccb9f,0xceabbfd2),LL(0x740f4e3a,0xb0d1b6bc),LL(0x4cbfd0de,0xfc737250)}},
   {{LL(0x32452b0e,0x3fad2d9e),LL(0xf523527d,0xb4e659fe),LL(0x6c0ff353,0xf0dcd701),LL(0xd67b6f98,0x28f06e2a)}, {LL(0x82a789b4,0x2d0c36ce),LL(0x49c0d54e,0x20e577da),LL(0xae38dd0e,0x8d1d5c7f),LL(0x894d9143,0x72e01399)}},
   {{LL(0x3958e055,0xf7806856),LL(0x5df44aee,0xac35ee40),LL(0x97c18b8d,0x2b478913),LL(0xfa2586cd,0x5396824e)}, {LL(0x1b23f8c4,0x22b37b25),LL(0xcdecdefa,0xf9ced36e),LL(0xc2fc39c0,0x28c3bee5),LL(0x6d9db32b,0xa1731fae)}},
   {{LL(0xbc3e2c91,0xa0445fa7),LL(0x75a4aa72,0xa1ab6955),LL(0xbbe0a1c7,0xf0cd61c6),LL(0x0123bc52,0x923c3b69)}, {LL(0xafd7c4bc,0x818ad28c),LL(0x28b15b05,0x7c229224),LL(0x1f78a4f4,0xecde7efb),LL(0x03ef3ab3,0x550d68e7)}},
   {{LL(0xfc5f8c35,0x0371021d),LL(0x0ed2b06e,0x4440aa1e),LL(0x9ba7247d,0x70c8ede9),LL(0x84f23fde,0x0d2b6ed3)}, {LL(0x5ff4478c,0xd0119d95),LL(0xf79c69d5,0x66152d27),LL(0x02afd13b,0x56d8bea4),LL(0x15bb938a,0x035efe5f)}},
   {{LL(0x2ccaa425,0xc5ca7d08),LL(0xeeee9376,0xc8c69ea6),LL(0x493a2051,0xb22cfe59),LL(0xdc7b90fb,0xcb50e618)}, {LL(0xe05a8705,0x0f6fdf2b),LL(0x4814df31,0x081f3fe7),LL(0xeb1e3e76,0x6fefe18a),LL(0x03e06a50,0x81910050)}},
   {{LL(0xdb45bfea,0x8a801df1),LL(0x7a828cf6,0x8c7fe1fd),LL(0x8d173cfd,0x1c1868b5),LL(0x0dbde1c8,0xe18f0a36)}, {LL(0x9ac345b6,0x3b29ed64),LL(0x9dcd07a5,0xd56d5956),LL(0xc6813a88,0xf4191570),LL(0xeda3af42,0x39033ebc)}},
   {{LL(0xad5d215d,0xdee5591b),LL(0xafbe5a28,0x9cfa11c6),LL(0x1823c28f,0x73d0f1e2),LL(0xafab1f67,0x75d49925)}, {LL(0x7c521448,0x61c81e2c),LL(0x4a96edb5,0xc547be6f),LL(0x4ca368b3,0xccb9fc59),LL(0x04fc3303,0x175ebe48)}},
   {{LL(0xfce42989,0x507620cf),LL(0x9abfadb2,0xf236e043),LL(0xab36ab58,0x381c50c3),LL(0xae22c6a3,0xed4cb73e)}, {LL(0x2158dc4c,0xa68a2827),LL(0xe9fa53ff,0x1715ac43),LL(0xfa266797,0xb02fdf73),LL(0x7eefb203,0x3079f3c7)}},
   {{LL(0x7f7545bd,0x0a41fb94),LL(0xcb923ace,0x6b9dd022),LL(0x3bea2541,0x582c7ff5),LL(0x5ecdbe2d,0x992f2379)}, {LL(0xfe17bdca,0x821f1670),LL(0x2626bdde,0x521c06f2),LL(0x1864ca0b,0x6292748c),LL(0x1bc74d8b,0x554d4ece)}},
   {{LL(0xea3d4446,0x745d4f74),LL(0x40ad1c7f,0xa439f178),LL(0x51374e92,0xc95d9510),LL(0x90229008,0x75870e9f)}, {LL(0xc54e7e81,0x3fec98c2),LL(0x94b3860b,0xef537ee9),LL(0x40bfc8f6,0x139dd834),LL(0x0f114403,0x20b51364)}},
   {{LL(0x30b4b4db,0x4752a49f),LL(0x8c3c90e0,0xdfbb8b17),LL(0x70f0b16a,0x60c8915b),LL(0x40528319,0x5e395000)}, {LL(0xa641f2e3,0x8a1624c7),LL(0xbb4ca0dc,0x3c9925c6),LL(0x2c3152b5,0x2aae6edb),LL(0x08b896ff,0x8dbac580)}},
   {{LL(0xe5a36bc8,0xe0516205),LL(0x3295707b,0xd7714332),LL(0x51c3513f,0x61db6804),LL(0xab552df8,0xf2ee6e20)}, {LL(0x353c17f0,0x5ddcfa99),LL(0x046d5fd4,0x65687a2f),LL(0xfd1ccad4,0xef567e9f),LL(0xa0238b70,0x7cd5f7dd)}},
   {{LL(0x92c01197,0x96fba79e),LL(0x83b50e70,0x46a9f2de),LL(0xfe287169,0x7efcbbb2),LL(0x4528d67d,0xe30d60cb)}, {LL(0x6cb04d3a,0x88fed0cc),LL(0x0221ceb8,0x63eb9d0d),LL(0x748b5813,0xc6954e9f),LL(0x5c96e897,0xceef2bd8)}},
   {{LL(0x85648f04,0x99503ae2),LL(0x923e87d7,0xeee51f99),LL(0xb6560ceb,0x90908fca),LL(0x80e0f6b3,0xafad5926)}, {LL(0xaea32cf9,0xa50f31f3),LL(0xa74ae92d,0x7ea17064),LL(0xcda71d1a,0x0675ccc1),LL(0x1e0a464a,0xd1e3b630)}},
   {{LL(0x2442872d,0xa361f2b7),LL(0x46e52c97,0xb21bcd39),LL(0x85574630,0x1405f89c),LL(0x8e0a96ab,0x0da7bfbd)}, {LL(0x4220f57b,0x48af06c2),LL(0x6a333e4f,0x772a9b12),LL(0x6f712eb8,0x3afc661e),LL(0x2eba8817,0x29deff6c)}},
   {{LL(0xd8c69e5a,0xbab680de),LL(0xe93daf10,0xf8615abb),LL(0xcef6fae6,0x7455ea1d),LL(0x868455fd,0xac0a30ea)}, {LL(0xe47d628a,0xae967b17),LL(0x65f1f482,0xa6d703e2),LL(0x0bfcc371,0x2723a965),LL(0x6db4a042,0x9b06cc14)}},
   {{LL(0xa77c8b21,0xa973d738),LL(0xc008f2ed,0x9a981f80),LL(0xaf27cdb3,0xecc7bbcb),LL(0xb5cb693a,0x514db964)}, {LL(0xe75c93d1,0x24125414),LL(0x1f00d53c,0xd9308c0e),LL(0x831eba6d,0xdb56d155),LL(0x672b86f1,0x29eefc2c)}},
   {{LL(0xdd13b3c9,0x332f6ab6),LL(0xe371f873,0x70e052f6),LL(0x125712ab,0x05740742),LL(0xb3512100,0x4239152d)}, {LL(0x80b22915,0x98355eaa),LL(0xb896f6fa,0xd0e263ec),LL(0x442b4c8f,0x9378a8a6),LL(0xf65795bb,0x40c2b546)}},
   {{LL(0xd572ead8,0x0cfa46ed),LL(0x78361300,0xb9b4abdb),LL(0x8c102020,0x5fe63ef1),LL(0x785a4b54,0x1805c84e)}, {LL(0x805cb642,0x147cf487),LL(0x487e581f,0x87cf50aa),LL(0x9eaebcd0,0xe942fa5b),LL(0xd1af71f2,0x06d4fa96)}},
   {{LL(0xc4fc3823,0x20c1a770),LL(0x85140885,0xcdffd09e),LL(0x6b3592e9,0x27ce78ab),LL(0x8ba82008,0xb8e8c15e)}, {LL(0xfef74187,0x5fe8f3f0),LL(0x77ce808d,0x8e85a3a5),LL(0xc7395f64,0x8447dc69),LL(0x1181b854,0xae90769f)}},
   {{LL(0x456114c8,0x54adc101),LL(0x9ca6a9c1,0xe7962b76),LL(0x909410a4,0x3f0e77fb),LL(0x9e2e44f9,0xe18151cd)}, {LL(0x2cf6e29e,0x5e510a0a),LL(0xb1836b07,0x136896ab),LL(0x0fe11010,0x3ad4fdec),LL(0xdbddf038,0x35b36790)}},
   {{LL(0x75903df9,0x7c4f5a68),LL(0x2f5b7193,0x3e9cb056),LL(0x591a4524,0x745e9452),LL(0x1a056e15,0xc406ad44)}, {LL(0xa69e11ef,0x2e93edf2),LL(0x73a1cb88,0xa28b82fd),LL(0x1225c3d5,0xdc1c9cda),LL(0xa5569794,0x86e9a994)}},
   {{LL(0x5b092dde,0xd698506e),LL(0xd1ca8b06,0x076a4c82),LL(0x2ef2bc6f,0x4516033b),LL(0xd78fa65f,0x0574c792)}, {LL(0x735bb362,0xa3b1c3d8),LL(0x0da54317,0x22fca7a4),LL(0x60aaebb6,0x3e7ae709),LL(0x937638c1,0x42417d54)}},
   {{LL(0x1dfe8b0e,0x32f00a5d),LL(0x8dcdbdbc,0x8ea5e8e1),LL(0x6b30ea52,0x38df57cb),LL(0xe94c30ca,0xd325aa1c)}, {LL(0xdce4d256,0x7aa04a9d),LL(0x74c7db6b,0x78e98cd3),LL(0x443d5c9f,0x631475a8),LL(0x7adfbceb,0x34e5c73a)}},
   {{LL(0x9f1e8828,0x7fb69bab),LL(0xc84149e3,0xcadc78be),LL(0x1fe86af8,0xe9424ecc),LL(0xbc504ea8,0x13160cc8)}, {LL(0x4c96a680,0xcb380078),LL(0x845faae5,0x006fb9d8),LL(0x1e0e66d1,0xc6a64277),LL(0x428f526d,0x13f77d6e)}},
   {{LL(0x28474530,0x9f80fe8c),LL(0xdb7fec00,0x5649a173),LL(0xd9cb05ca,0xdeed5bf4),LL(0xd7077c41,0x14b1a3a9)}, {LL(0x096883ec,0x4c2ed239),LL(0x44ae671d,0xd550edfe),LL(0xf7b7362a,0xb233e5dc),LL(0x4fd464f2,0x32c15820)}},
   {{LL(0x68880bf9,0x0ecb18f7),LL(0xaf230a34,0x53468bed),LL(0x370cd6ef,0xe3ba97b9),LL(0x3516d77e,0xf5cdabf4)}, {LL(0x11462032,0x08d78a56),LL(0xd583ccc5,0x1393fa93),LL(0x0c1b3514,0x52af7f5d),LL(0x188ca043,0xf48cac66)}},
   {{LL(0x5461a1d1,0x2524c8dd),LL(0x91b6e707,0x6eee8101),LL(0xca2fe87e,0x209fece6),LL(0x9ac56706,0x50b35727)}, {LL(0xec373bb2,0x651a6701),LL(0x1a4c2e84,0x881de85b),LL(0xcfdb47d5,0x4892861d),LL(0x5cdc4424,0x5ae2e653)}},
   {{LL(0xa1f90dd9,0xc58f4f59),LL(0xfcf158a4,0xa5584f85),LL(0xab072a7a,0xbde86fb0),LL(0x268bae62,0x7c69e25a)}, {LL(0x44fc7b3e,0xee3478f3),LL(0x6b7d3647,0xec148394),LL(0xe1c8c0ca,0x2a542ebf),LL(0x161dc0c1,0x63d1d635)}},
   {{LL(0x57ab9282,0x769acdbe),LL(0x2a119cb9,0x9c338971),LL(0x125e5b4c,0x049e366f),LL(0xf0c8fde4,0x3aec68e0)}, {LL(0x324cefda,0x9d95b6e5),LL(0x704014b5,0x844cce33),LL(0x6a6bb216,0x03920a61),LL(0xf379db8e,0xd69d17e3)}},
   {{LL(0xc5e386e5,0x1924ac16),LL(0xd64953c3,0x62373a48),LL(0x47f4e4a4,0x5b1f7d64),LL(0xffa115fd,0xc043b5b5)}, {LL(0x87fb16b0,0xb2a2656e),LL(0xd8cd79a6,0xcac56b9b),LL(0xcc19d3af,0x544971f6),LL(0x0fd63db9,0xf539244b)}},
   {{LL(0xfbf4d232,0x0f052d3c),LL(0x7a2a7280,0x6b3c8366),LL(0x48079b9f,0xaa6579db),LL(0xa4d9edcf,0xc5beb93d)}, {LL(0x0f1599a3,0x8ad58825),LL(0x5f3f640b,0x3f3a2634),LL(0x9032fd7c,0xda15393a),LL(0xac0e7136,0x97c10230)}},
   {{LL(0x599785ee,0xfa32ef9f),LL(0x6b4c7a65,0xe1ed3b28),LL(0x2da1dcdd,0xcee1af27),LL(0x6861e2c2,0x4e480c11)}, {LL(0x9c8ad8c3,0x35b5ec42),LL(0x3fc55f23,0xfd07f6a4),LL(0xea76d444,0xab18ead2),LL(0x22ba099a,0xcb8bde14)}},
   {{LL(0xc61ae967,0x252e6a81),LL(0x72a2e1e6,0xaf11042c),LL(0x1a211ef8,0xb353902a),LL(0xc99a25fc,0x644d16e0)}, {LL(0x5b67e48a,0x637fd606),LL(0x51a0b665,0xfa570963),LL(0x7ee072b8,0xaa661c73),LL(0xf2e0a727,0xde1eb4fe)}},
   {{LL(0x22ed7ee6,0x56096a0c),LL(0x5825908b,0x31aaf403),LL(0xbfa02db6,0xfd5f6ba7),LL(0xff798900,0x85f4f9a9)}, {LL(0x4a0cd878,0xa0997d56),LL(0xb1b6982e,0xdd76909c),LL(0xeccf338e,0x874fab15),LL(0x4ce82bb1,0x5e072b3c)}},
   {{LL(0x6dd0d997,0x5dbe883f),LL(0x41765fb6,0xa32117f2),LL(0x7d87fc5e,0x59ca4da3),LL(0xb95ec918,0xc91002cd)}, {LL(0x6548248f,0xd53bc123),LL(0x6c6d1e0e,0xef10a373),LL(0x99d9893f,0xafb2d760),LL(0xce0ba0ca,0xb77c1f1b)}},
   {{LL(0xcfb9f6b2,0xabce362c),LL(0x35f9be91,0xe6e108d2),LL(0x7187fa9d,0xb2331290),LL(0xfc7ddce6,0xdcd1f4fd)}, {LL(0x9086eb29,0x3a129991),LL(0x53a56d57,0xb0730520),LL(0xabd421bd,0x9fcdf4cf),LL(0x08f3e8e0,0x96271270)}},
   {{LL(0x401e0217,0x951ea7e2),LL(0x733f637b,0xa4d1d708),LL(0x4f4cd676,0xc75170f4),LL(0x832f0b4d,0x568279ba)}, {LL(0x25c17ab7,0xda4c01f7),LL(0xfa30e1b9,0xfcc13028),LL(0xacba57ec,0x4d1d8f71),LL(0xef6b3913,0x0c7971cf)}},
   {{LL(0xc014f166,0xdf16e73d),LL(0xf96f2c30,0xd5796183),LL(0x3f70dd7c,0xd13ee9f7),LL(0xdac738c5,0x3f9aa0dd)}, {LL(0xad021e28,0xa200c7e4),LL(0x08414fd0,0x982abae3),LL(0xc3779882,0x76d16a8c),LL(0xe70a6ff5,0x41563d33)}},
   {{LL(0x4b553a17,0xdbb9656e),LL(0xd9c87aa1,0x96af21a0),LL(0x7bd9a625,0x2de13a03),LL(0xfeb1fec2,0x29f8c49b)}, {LL(0x1a4ce44a,0x84e2df47),LL(0x548b39ee,0x83bb2965),LL(0x94d996eb,0x38b91cce),LL(0x9441ae0b,0x41e0a3cd)}},
   {{LL(0xdaa92f34,0x720d30d8),LL(0x06f30fbb,0xba587579),LL(0x4c96ad59,0x24f74676),LL(0x0d33bd5f,0xf40493f7)}, {LL(0x126a7267,0x9068c3e9),LL(0x18927490,0xa51099df),LL(0xa9cfe02f,0x27452423),LL(0xb8749653,0xcfd035be)}},
   {{LL(0xfda6a4a9,0x0dd9bc2a),LL(0x0106ae0e,0xdba0178a),LL(0x4969a4bb,0x3820c9f5),LL(0x99fbc715,0x5031e9fd)}, {LL(0xc193d942,0x642a030a),LL(0x454cbb39,0xdc3d6ab7),LL(0x1c8fa77c,0x507c17b9),LL(0xe3642a95,0x8465bcc8)}},
},
/* digit=32 base_pwr=2^224 */
{
   {{LL(0xc25dfad3,0xe74e265b),LL(0x493f44b6,0xd03630b9),LL(0xbfd6d473,0xb3270892),LL(0x1c5ee992,0x5b2d9543)}, {LL(0xa36f7c5f,0xeeb94537),LL(0x8ab0b81d,0x9befc01d),LL(0x188b45e5,0x483cdb08),LL(0x01e4648b,0x44c753b7)}},
   {{LL(0xb2411618,0xee43bc87),LL(0xf07924c4,0x08754bd2),LL(0x4ac92557,0xef205033),LL(0xee0387f4,0x6e7e4fe6)}, {LL(0x76961d0e,0x51f3e2e2),LL(0x37eac10f,0x2b69d417),LL(0x73757a88,0x36d0f45f),LL(0x2b0c7d35,0x38b967e5)}},
   {{LL(0xb31fa779,0x94ba8fc4),LL(0x0f13036e,0x8024dc85),LL(0x82d754b7,0xfda2af63),LL(0xae9ea9ae,0x4a784242)}, {LL(0xf9887947,0x67dd14ab),LL(0xcd555a0a,0x7f2ecfc4),LL(0xf63a46aa,0xb37c4244),LL(0xff71b4b5,0xd032cfc1)}},
   {{LL(0x6b8a6a97,0x0aef84c1),LL(0x0b2bca36,0xd2e7f3de),LL(0x5b174d43,0x721c6c09),LL(0xd52ccc5b,0x5719cf31)}, {LL(0x3adf9517,0x6c7361f0),LL(0xabe20ff5,0x1e264169),LL(0x69eacc0e,0x01f9d997),LL(0xc2e635d2,0x721eba63)}},
   {{LL(0x25df8bb5,0x4225e9c8),LL(0xb5752d7e,0x931f721e),LL(0x0a3b281d,0x3c4ed475),LL(0x4a4668be,0xcf927682)}, {LL(0x75b7e90c,0x1b7f358e),LL(0xb7a29b9a,0x06e5c24d),LL(0xa167f2c8,0x0058967a),LL(0xa4ee62d3,0x9f1a6fb9)}},
   {{LL(0x278291f1,0xca899c4f),LL(0xf4e64c1d,0x69a90324),LL(0x8d62916e,0x46cc5d42),LL(0xec1007cc,0x3c802e65)}, {LL(0x6219cfbb,0xdadcf2aa),LL(0xd10258b2,0x942870dc),LL(0xa5e142af,0x77264e68),LL(0x089cc7a3,0xf25675e2)}},
   {{LL(0x7336aa16,0x177e8a3b),LL(0xbc5c622c,0x5a92cc2d),LL(0x1789e029,0x33a35a2c),LL(0x4e4d5573,0x6f91306e)}, {LL(0xda0a46f5,0xe5a2a581),LL(0x42640bba,0xfb532bed),LL(0x4a7b3ae4,0x88ff0f11),LL(0xb8ff7a71,0x2223e7b6)}},
   {{LL(0x5d21d261,0x75933133),LL(0xcabb1fbd,0xa336289a),LL(0x631b3b61,0x797db2f3),LL(0xd7e6a511,0xc2cedb25)}, {LL(0x10355332,0xb8806f34),LL(0x5d0ae37f,0xe5f1fb4a),LL(0x5d17c5c7,0x57cf26a5),LL(0x68c43ec3,0x82e8df47)}},
   {{LL(0xf86bd784,0x70fa23eb),LL(0x51b0ce75,0x711a9dbb),LL(0x82170008,0x83bb4a90),LL(0x630602dc,0x8f096ee9)}, {LL(0x7f15e77a,0x7d275fc9),LL(0xfe727ec7,0x63516a6a),LL(0x1dce9d38,0x6b06827a),LL(0x023b31c2,0xa01a5382)}},
   {{LL(0x886209b8,0x12537433),LL(0xc5a11b32,0xb7875fa8),LL(0xbd61176d,0xfa63cb99),LL(0x33378ebb,0xebb204ea)}, {LL(0x70c135f6,0xf29a29a0),LL(0xfa29d69f,0xf53941e9),LL(0x9917da42,0xab97b39a),LL(0x45947ae4,0x4677cfea)}},
   {{LL(0x0f6dd908,0xd4668cff),LL(0x987e0769,0x48bb09ed),LL(0x8d64b6fd,0x794ed298),LL(0xfac845da,0xaf89d530)}, {LL(0x2d06e70d,0x57445649),LL(0x079e70a7,0xe2a1a8c2),LL(0xf524fc03,0xd2ef1779),LL(0xb137bb1b,0xeaccaacc)}},
   {{LL(0x5d279f42,0x34d8ed87),LL(0x1cd68904,0x4dd5344c),LL(0x24abd550,0xb558b71d),LL(0x40df135e,0x3728e850)}, {LL(0xcfe86519,0x9329e2b2),LL(0xac74cde2,0x48ad17fb),LL(0x30b388b5,0x2ad61b22),LL(0xfaea71e1,0xebcbc1ad)}},
   {{LL(0x35990d9d,0x50d7b19e),LL(0x6eb70243,0xb17138e5),LL(0xaa8ae7e6,0xb61618f6),LL(0xabce12c6,0xedee15b0)}, {LL(0xcc7205fc,0xa99ce250),LL(0x69e0d75c,0xe438efc9),LL(0x5084b390,0x1feb6a10),LL(0x9c80d42d,0x7b348954)}},
   {{LL(0xe4b68140,0x67ac89d5),LL(0xc9b092af,0x34afd23b),LL(0xfe1ad661,0xad65cae9),LL(0xe02d884c,0x4f402947)}, {LL(0x6b1c40c1,0xd0a48fcc),LL(0x8961487b,0xf950c9f7),LL(0x206d1752,0xdb1cd811),LL(0xe99fd891,0x863b0ded)}},
   {{LL(0xbb2a4515,0xd3aad8c2),LL(0x797e223c,0xc93c8cb8),LL(0x12a77d46,0x0f471e49),LL(0x600872b6,0xa2ac9434)}, {LL(0x915f730b,0x6fb1c7ef),LL(0xd254d363,0x9fb72401),LL(0x6b44796a,0xf521e33a),LL(0x97c9fafb,0xb7ed2e8d)}},
   {{LL(0xffb5e7ce,0x60d41128),LL(0xaecb96c2,0xdbd8b542),LL(0x0b5ca788,0x029ab3dd),LL(0x190eb38c,0x8b1148a2)}, {LL(0x69fb1924,0x59048db8),LL(0xb18391a8,0xcd2149f0),LL(0xfed311b9,0x6bece5b6),LL(0x9ffd29b9,0x5edbe9b9)}},
   {{LL(0x1156ded2,0x53810556),LL(0x721f3e68,0xf812ce5d),LL(0x7ccdc8cb,0x50504d40),LL(0xc60fa4fc,0xb559ba08)}, {LL(0x1d6bd879,0x862a83d9),LL(0x836e26ba,0x2f8f653b),LL(0xeb26ca11,0x8587e6df),LL(0x8c8aaf7b,0x127bd905)}},
   {{LL(0xd67d9902,0xe26e690d),LL(0xb58e7e78,0x1a6061f4),LL(0x480dd4d1,0x960ef741),LL(0x75589610,0x7fd09736)}, {LL(0x855a8b2b,0x5a20a1a2),LL(0x355b4e0f,0x3ed68662),LL(0xe76595b4,0xd3786f45),LL(0x0bdedcfb,0x72a6999d)}},
   {{LL(0x6a175646,0x4e48e843),LL(0xe05dc02d,0xde53c427),LL(0x97d31bc6,0x9728a4c5),LL(0x5bb3bd37,0x01a07129)}, {LL(0xa74a0fcc,0x83c08a98),LL(0xbc345df8,0x233e400f),LL(0xcc3e0edb,0x9578c5f2),LL(0xf144a31f,0x0fe89df2)}},
   {{LL(0x14c5a2cc,0x308098a0),LL(0xba40c0bc,0xeda5a59d),LL(0xb718a5ae,0x0b10f7e0),LL(0x5b8ad9ba,0xdaf7da8c)}, {LL(0x87394cde,0xddc71285),LL(0xe43458d3,0x9bdb27cd),LL(0x4bd7c11c,0xc698d972),LL(0x3540be14,0x2ee97fbc)}},
   {{LL(0x72f98422,0x2c704995),LL(0xef8661c5,0xfc71fee2),LL(0xce08043e,0x6574e022),LL(0x5143733c,0x3d17162e)}, {LL(0x730e5b80,0x3bf0b448),LL(0x7cf94b5e,0x56de346a),LL(0x6c797797,0xfa87a53e),LL(0x6487d014,0xe8b9edfa)}},
   {{LL(0x7be60b03,0x09e74387),LL(0xec8750db,0x2277ebc3),LL(0x7aeaa545,0xf1e9d594),LL(0x44c03394,0x41564562)}, {LL(0x4de9f7ec,0x57943adc),LL(0x2a220cd5,0x09dd58f9),LL(0x06973808,0xdf848ec8),LL(0xd3950024,0xf1d5def1)}},
   {{LL(0x5a8707e5,0xd089eba5),LL(0xb0b90ebe,0x914046cb),LL(0x63fe6bc2,0xb01180b2),LL(0x7ede9d83,0x1ffbc968)}, {LL(0x3c52c09f,0xe16d336f),LL(0xdf40338d,0x32270ecb),LL(0x7eec7039,0xb55ff5c6),LL(0x38a63fab,0xb5ffb314)}},
   {{LL(0xee18ffec,0x3e9f284f),LL(0x1d1b4e80,0x702d97f5),LL(0x214c4da1,0x2005ee57),LL(0x2f5ea2f4,0x1c210413)}, {LL(0xa4149949,0xd24a486c),LL(0x23c8e201,0x3869a339),LL(0x0149992e,0x00f6e410),LL(0xf0a367dd,0x54e97b46)}},
   {{LL(0xe169de5c,0xd967726c),LL(0x6a0fcd03,0xa3e81f93),LL(0xdb1b1733,0x171faa9d),LL(0x3828e41b,0x0bbb5e91)}, {LL(0xf0828387,0x789a7b2e),LL(0xfca60b9b,0x9465cc16),LL(0xab630d23,0xcb58e90a),LL(0xa339d4b4,0xe7d30293)}},
   {{LL(0x18e75428,0x0bcac958),LL(0x9a95900a,0xd2f1554a),LL(0x03846833,0xc63c2afb),LL(0x6d1e8753,0x703d0220)}, {LL(0x04a357a2,0x47f5fe57),LL(0xcdc17255,0xaafba53e),LL(0x821af8d5,0x8f94c8eb),LL(0x35e37920,0x4d9918bc)}},
   {{LL(0xe32dd067,0xc029bd84),LL(0xf77f8962,0x25982357),LL(0x510b7cfb,0x7af256ca),LL(0x446925d7,0xca397f37)}, {LL(0xe0614e1e,0xb3dc7be5),LL(0xbbc4cc93,0x3b64cd27),LL(0xfb99bbc9,0xbd762df5),LL(0x04d7177f,0xc1ef0d4d)}},
   {{LL(0x65e75ed6,0x77b6d3d6),LL(0x53053b45,0xbe59c8da),LL(0x54fe92cc,0x054d089f),LL(0x6fa4758f,0x2f2327e0)}, {LL(0x5d2b5a01,0x948cf03f),LL(0x9c23b83e,0x47544c4c),LL(0x6c128d69,0x338590fa),LL(0x76842160,0x5724008d)}},
   {{LL(0xc4f063e6,0x4cbeb18e),LL(0x9c2f826c,0x507ba094),LL(0x6f4e49f3,0x0e877a6e),LL(0x34f56868,0x050c2040)}, {LL(0x0f119e25,0x8fd667c4),LL(0x13b47d3f,0x881dd34d),LL(0xca8e9a6a,0x2a6b636d),LL(0x107ea18c,0x67b081fb)}},
   {{LL(0x84af55d9,0xd3a46367),LL(0xd7626b67,0x0e709a00),LL(0x4c6dfc8e,0x135725fa),LL(0x133a6e20,0xbf6328d9)}, {LL(0xa87031f2,0xa4b743b4),LL(0x13825d07,0x62e90e67),LL(0xb85f3c45,0x7004028e),LL(0x11751be0,0x0465c502)}},
   {{LL(0x8a5ab076,0x4d5d467f),LL(0xf4fb8a45,0x9aa3f414),LL(0x5dc1fa84,0x9fa0422e),LL(0xd02cfd2b,0x3205c05f)}, {LL(0x078836b6,0x3eac28fa),LL(0xfc3ff573,0x53bc0189),LL(0x00b02100,0x2c45ef09),LL(0x34360ef7,0x61bc02ae)}},
   {{LL(0x532e8d6a,0xeb5593e7),LL(0xf02a1ee4,0x94092904),LL(0x200496c0,0x379b32e8),LL(0x360a27c8,0x46fb6e9e)}, {LL(0x62005158,0x8a3377ba),LL(0x0de3f191,0x1a3266c1),LL(0xc94d2127,0xe60fad96),LL(0x646302f3,0x41553dd1)}},
   {{LL(0x377e0766,0x88bf0bfa),LL(0xe75bf57b,0x870a24db),LL(0x79e77976,0xc133cb49),LL(0xf43b6f18,0x2f14924d)}, {LL(0xaa94cd73,0xe561dc90),LL(0xd6eb695d,0x8c420eb2),LL(0x2f04ef79,0x99e41ba8),LL(0x71e6d054,0x7f427bdf)}},
   {{LL(0x10bde8ce,0x7304bb25),LL(0xe48b16f8,0x5dbc4325),LL(0x8796db7b,0x47d17ab2),LL(0x94c77832,0x83426817)}, {LL(0x9878ace2,0x6781850e),LL(0x019e97aa,0x7f747b90),LL(0x949f9b08,0xa0545c85),LL(0x244bc083,0xe0a0bbf8)}},
   {{LL(0xa1f38ea6,0x8cb53666),LL(0x4989a568,0x9be29ff6),LL(0x083a7fcd,0xbc5a7f87),LL(0x44ca10f6,0x90d0129c)}, {LL(0xd724b7e2,0x1ad274bb),LL(0xcad5f069,0xa5290cbd),LL(0x86a4e0a9,0x886b1a7c),LL(0x8d8fb13f,0xd2481b5a)}},
   {{LL(0x4f606ac5,0x80075fb2),LL(0xbfc10e7f,0xf984b5a2),LL(0xf056142f,0xd3d91aea),LL(0x4afdc017,0x770bee0b)}, {LL(0x6c49c827,0x3c42ca88),LL(0x0aaa3139,0xb620c4e8),LL(0xad87890c,0xac6b512d),LL(0x0eb61f92,0xaee62df7)}},
   {{LL(0x21dad9ec,0xcf0f37fc),LL(0xc52e24c1,0xd780d315),LL(0x23a2b699,0x0263bcab),LL(0x9714b577,0xdc8dcd2f)}, {LL(0x55622b11,0xeb16eca8),LL(0x01118edf,0x94b3d649),LL(0xec66879d,0x6bafea64),LL(0xc4ab9f48,0xc35739c0)}},
   {{LL(0xf3232370,0x082ccf53),LL(0x71407825,0x01b55dd3),LL(0x4f7f4038,0x86e0fe94),LL(0xb04159e9,0x1a623675)}, {LL(0xbc4df337,0xf908ca59),LL(0x816162ce,0x1b4f1ffe),LL(0x2d60e05b,0xb5128952),LL(0xd38cbdf7,0xb47ca0eb)}},
   {{LL(0x8ee38219,0xdccba22f),LL(0x9fbb36ff,0xc9436453),LL(0x8ac12c9d,0x83cecbf5),LL(0xf4cb1ebf,0x591191b5)}, {LL(0xf03c1632,0x693cf383),LL(0xcb6abaca,0xaebd3f9b),LL(0x0fa26e7a,0x1427c154),LL(0x01bf37af,0x4f0de894)}},
   {{LL(0xd88da2a6,0x4e497acf),LL(0xe5c86482,0x8014a215),LL(0xcf94ee40,0xa627d78f),LL(0x9ca36aa3,0x7647708d)}, {LL(0x87e108c8,0x3d4e8bb1),LL(0x516f8b28,0xacdc3223),LL(0x1d955038,0x74e4d436),LL(0xedd68905,0x7e4a198c)}},
   {{LL(0xc4bfbad1,0x41dc4bdb),LL(0x85964942,0xfd1121b1),LL(0x0c0d85ff,0xe462eb9c),LL(0x2b012d88,0xade1ccb3)}, {LL(0x318f2aa3,0x2eec3503),LL(0xdec8a13e,0x656b44da),LL(0xcda13a8c,0x90b7aac8),LL(0x214a530f,0xe7f3a5ff)}},
   {{LL(0x159060b9,0xa0c8062a),LL(0xd9305873,0xc19f2608),LL(0xc9609521,0x0d6213c4),LL(0x9aec4539,0xde2b9134)}, {LL(0xaeddf0a6,0x4a2a6476),LL(0x5cf2e85d,0x89222dff),LL(0x084a0245,0xad92a1d3),LL(0x2138063f,0x29644a60)}},
   {{LL(0xb8164832,0x5b57a05b),LL(0x885ce4d1,0xecf6620f),LL(0x045d3b88,0xde28ed2f),LL(0xb4502308,0x3eb11759)}, {LL(0x4d9f94b8,0xe97f1db2),LL(0xeb83f016,0xfa248583),LL(0xcda53ba0,0x63a273b4),LL(0xa228b7b9,0x9692973a)}},
   {{LL(0xb6707cbd,0x5968cb12),LL(0x5c1a2b4d,0x1895ccb4),LL(0x37f0b240,0xff309157),LL(0xb90875c2,0x374d983e)}, {LL(0xc4e815e7,0x22fc40c6),LL(0x98d18506,0xf2db27be),LL(0xaa9ae62e,0x2854a948),LL(0x79e39df1,0xd614a722)}},
   {{LL(0xb3501c19,0xebeec551),LL(0xd89cefca,0xe2b878eb),LL(0x6b4cd6bc,0xa0a34757),LL(0x70bfdf88,0x0159129c)}, {LL(0x489502ca,0x26fa3e53),LL(0x285be050,0x7932084f),LL(0x71912b0c,0xfe4998f4),LL(0xc60b88b7,0x3dce0a87)}},
   {{LL(0xc718ee5c,0x5b93edb0),LL(0xbb075836,0xb93a225f),LL(0x7aa0595c,0x87a08c94),LL(0xc31e3249,0x401d2620)}, {LL(0xdae2cdb9,0xbe6928b4),LL(0x5a226630,0x4b68e106),LL(0xc9d32e4f,0xdc38c2fc),LL(0x26542f89,0xc51a6245)}},
   {{LL(0x1757f3c4,0x5fb37c1b),LL(0x89128aa4,0xa27d6c02),LL(0x5e629309,0x3b74f56f),LL(0x2f7aeef2,0x24b5ad84)}, {LL(0xbdc89596,0x54a962cc),LL(0xcc2f3d5d,0x6e8bccf8),LL(0x312e9241,0x4c1df22c),LL(0xfc30f0dc,0x8ffe6b0d)}},
   {{LL(0xa6ec0fe4,0x670431a2),LL(0x35964572,0x49da0b42),LL(0x9dda5c39,0xbb12d1b0),LL(0x72d3de51,0x64170fe1)}, {LL(0xa4a2f5d9,0xea8b2b16),LL(0xe590be92,0xde1bad64),LL(0xf0b9b0b5,0xb7f93581),LL(0xe115d67e,0xb007f4dd)}},
   {{LL(0x415732d4,0xab9d7523),LL(0x905ec0fe,0x2951149d),LL(0x74350478,0x94bb2c63),LL(0xe9b1ada4,0xe6b63bfd)}, {LL(0x13e8528f,0xd09b4d4b),LL(0x685bf239,0x6bed3d25),LL(0x1a14b7ce,0x83023ad9),LL(0xd0505d6b,0x4bffff63)}},
   {{LL(0x8bb1cfa0,0x2ccc180a),LL(0x3a09c912,0x70c18508),LL(0x5878201c,0x318c41c2),LL(0x64c01149,0xb9f207b1)}, {LL(0xc58287d9,0x89fdd9ee),LL(0x05c43da0,0xdb6fa8db),LL(0x311a34a1,0xc31354f0),LL(0xf1521976,0xccfbaddb)}},
   {{LL(0x0c5e8388,0xf4175f75),LL(0xa81226cf,0x7e090ce8),LL(0xeda6a1ab,0x5209107e),LL(0x7e5ccefd,0xf0a263cb)}, {LL(0x10c704a1,0x9fe05a36),LL(0xcf133aee,0x3784d7ca),LL(0x37af86c8,0x066c3116),LL(0xd7ebeb8a,0xbf32ca04)}},
   {{LL(0xa9b5bab4,0x0447a950),LL(0x41bb1f1c,0x3b2f98bd),LL(0x59c62f08,0xd89bbdd7),LL(0x3ded069b,0x26bab370)}, {LL(0x69ea63ae,0xb0db4ca5),LL(0x9133df68,0x57b32f32),LL(0x5a17efbe,0xc79a22d0),LL(0xf8ae3c2d,0x576976a3)}},
   {{LL(0x9d02d212,0x5754198b),LL(0x230d0213,0x9cc9e61e),LL(0x92889e33,0x76772179),LL(0xf5df6cba,0xb1246608)}, {LL(0x8d491280,0x821766bc),LL(0x96bd3df5,0xe148f470),LL(0xed753b73,0xc1e9fc70),LL(0xd6cecfc5,0x840e40ed)}},
   {{LL(0x93e2f3a0,0x03874679),LL(0x5b646b64,0x462e5abf),LL(0xd7ae0e67,0x6fb19eda),LL(0xc3d2dddf,0x01e8a27f)}, {LL(0x4bacfe2d,0xc9e69639),LL(0x712e8afb,0xbc3a134e),LL(0x8af6d30f,0x5d943a86),LL(0x443c942c,0x65eb5f99)}},
   {{LL(0x2339e348,0xf5000308),LL(0xeb0d80e6,0xd69b7693),LL(0x5b9d220b,0x7b00b43b),LL(0x497bbcf9,0xde0dfc80)}, {LL(0x0c2e851f,0xcfe2e3f3),LL(0x7e91d378,0xef7793d1),LL(0x9d266a5b,0x9e43eeac),LL(0x1766c5c1,0x9c81d68b)}},
   {{LL(0xf6a4d560,0x121db320),LL(0x073582a7,0xcd0a4f03),LL(0x6e841041,0xbf469f9a),LL(0x5eb2d753,0x4314f0f6)}, {LL(0x8c498414,0x09021001),LL(0x859275b7,0xf63d00ee),LL(0xf1c0385a,0x228fa809),LL(0x694c3318,0x44259d51)}},
   {{LL(0xa2ad4eac,0xb0a5d5fe),LL(0xabdedf11,0xbb950010),LL(0x6264c601,0x081024ce),LL(0xaefb4115,0x6cc7faf2)}, {LL(0x8692a7a4,0x90511289),LL(0x60596012,0x2bef4135),LL(0x0f0becb6,0xfec5209a),LL(0xd1ceb42e,0xad0907a6)}},
   {{LL(0xf517372c,0x6cde3f21),LL(0x26131459,0x9acd4f09),LL(0x491f59d7,0xf3629a43),LL(0x41a037dd,0xe944561a)}, {LL(0x826d1b5c,0x07beeabe),LL(0x3a1476cd,0x0728a907),LL(0x7d4a1abf,0xa91f41a0),LL(0xa7a119c4,0xdf58ed06)}},
   {{LL(0xba701704,0x19990669),LL(0x47b67175,0x8aa3f76b),LL(0xdd0a6e9a,0x8bccff3e),LL(0x24f49029,0x4173fcda)}, {LL(0x61c18233,0x2a688911),LL(0x78b9fa8f,0xdf54b239),LL(0x37596f40,0x714cf627),LL(0x24e6a879,0x2c73ddba)}},
   {{LL(0xf2547f19,0x1538fd36),LL(0x0e7e84ee,0xd85c4730),LL(0x306f5fc0,0x00799e9f),LL(0x49ce114c,0xfccc6a37)}, {LL(0x3fe08120,0xf9cff5e8),LL(0xc2be9f27,0xdf876a1f),LL(0x6939fdb9,0xe817c72e),LL(0xd34d0e43,0x244a1800)}},
   {{LL(0x78fa7f11,0x41e83eef),LL(0xba6367e5,0xecaa250c),LL(0x8def6ae6,0x9c420347),LL(0x250b9e58,0x99efb3b1)}, {LL(0x79b2298c,0xdaf311ee),LL(0x69b6dff3,0xb49200cf),LL(0x559e51f5,0x5c7f17bb),LL(0x424be7e9,0x117d0cbe)}},
   {{LL(0x36e3af54,0x290a35c4),LL(0xe3a643b1,0xd2326cd8),LL(0x3580f9ee,0xc208b2b3),LL(0x4464a9e0,0x2419c661)}, {LL(0xbccb2759,0x87123d3a),LL(0x1a77d469,0x5d36fcf3),LL(0x49b07e5a,0x5aafd58a),LL(0x6b71e237,0xf534595b)}},
   {{LL(0x6705039f,0x0f0d3161),LL(0xca701676,0x7282b08c),LL(0x13796941,0xb05e8c3e),LL(0xfca06e08,0x5250012e)}, {LL(0x980c5ea3,0x7eb2341a),LL(0xa41defb2,0x92f5aeb1),LL(0x0e62453f,0x203244e0),LL(0x96181756,0x74341218)}},
   {{LL(0xe12a94ab,0x3b0cd36f),LL(0xb5ad7c48,0xf364b3b9),LL(0x8e768469,0x96a7a2a7),LL(0x1bbc7cc5,0xccc31c7e)}, {LL(0x080dbb92,0xe70ad5d0),LL(0x56fb0f1f,0xfb201e92),LL(0x29d99f57,0xdfce7a1e),LL(0x06457da5,0xc12a02b0)}},
},
/* digit=33 base_pwr=2^231 */
{
   {{LL(0x2a80f39c,0xdea72ba6),LL(0x68cbe088,0xcb2174b1),LL(0xd6bd1cc1,0x9c73ec69),LL(0xf20dcce6,0x6a2dbe20)}, {LL(0xeeaae9da,0x20a5150b),LL(0x9df630da,0xc935e85d),LL(0xa1634cd8,0x2147144f),LL(0x44f3af02,0x5eccb56c)}},
   {{LL(0xc0e2b70a,0xf77a79cf),LL(0xee8cbae7,0x2569c8bc),LL(0xfadb18fc,0x392a5dbe),LL(0x3ce6a0ff,0x59bc96b4)}, {LL(0x8b551005,0x287f04f4),LL(0xa44b2bd8,0x7efe3aa5),LL(0x6ac447d7,0x0e9cb8ed),LL(0x7783bdd0,0x9b4eb10a)}},
   {{LL(0xb981d96f,0x793c4880),LL(0xd56fb2a6,0xf719d828),LL(0x8149057e,0x9fcc236f),LL(0xb4d65c2b,0x318c63ec)}, {LL(0x5f95c5de,0x5269c1d7),LL(0xd0efd6bc,0x33b3745f),LL(0x4ae87c64,0xace344d5),LL(0xdd30ba2c,0x238809d6)}},
   {{LL(0x71192348,0x3cc32acc),LL(0x3f17ef60,0x545401bf),LL(0xcde25b0e,0xe209a493),LL(0x663abab9,0x5c11886b)}, {LL(0x28ec7c90,0xe61a81b1),LL(0x75b57f5c,0x18b125a6),LL(0xfad91696,0x86d1b45a),LL(0xa4c3f7ff,0xb31a786d)}},
   {{LL(0xf45831d8,0x2fd4cd72),LL(0x8be40d9f,0x85978fa6),LL(0xa9301111,0x38106329),LL(0x9e5979ee,0x1527e462)}, {LL(0x76c5fc8a,0x97f71c7e),LL(0xf1f33056,0x205fa473),LL(0xa6546a05,0x7bb9d24e),LL(0xf84c4d35,0x0e282a5c)}},
   {{LL(0x59471f1f,0x59d21896),LL(0xf6303573,0x2e613dec),LL(0x78bf5a4b,0xa7995794),LL(0xbf19fbe5,0x20adf6b5)}, {LL(0x1574d34d,0x3a48c95f),LL(0x09323ceb,0x95488f09),LL(0x552df9cf,0x450aee7f),LL(0x53557500,0xdf016f7a)}},
   {{LL(0x2da8a2a6,0xf2acedc6),LL(0x2f4a0632,0x03fc8cf8),LL(0x5b82f03a,0xe7ff136b),LL(0x9e88c421,0xd5841c4d)}, {LL(0x7eef63f0,0x75a4d66f),LL(0x2865c14b,0x92091ade),LL(0x64fe7ba3,0x7060474c),LL(0xfe30cb3e,0x4056724c)}},
   {{LL(0x8d9fceb6,0x38cf4c6f),LL(0xab528f38,0x11e85f78),LL(0x52303b2b,0xe2896d25),LL(0xed68c605,0xf929675a)}, {LL(0x10c708a9,0xfbd22374),LL(0x40d7e5a7,0x4682ca17),LL(0x9041047f,0x4242b5c5),LL(0x0f9c6840,0xaf571053)}},
   {{LL(0xa56af433,0x713b2bbb),LL(0x5e82f947,0x45aaf2ce),LL(0x106283c7,0x9882571a),LL(0x9b9c3c3c,0x37de12ca)}, {LL(0xbef10529,0xcb463af2),LL(0xd771236c,0xe18d763d),LL(0x62935de3,0xb47a69ca),LL(0x9a41e09a,0x4798e91f)}},
   {{LL(0x8e93edd5,0x89696697),LL(0xb7ea4f45,0x35cdb8e1),LL(0xfed33d87,0x36f8305d),LL(0x625642d6,0x57623440)}, {LL(0x945dd7d6,0xdfd9b580),LL(0x731739bc,0x965ffcb5),LL(0x637bf127,0x34588e1f),LL(0x539d21c7,0x936c0ba0)}},
   {{LL(0x71640eed,0x70832099),LL(0x3ff407e3,0x916b1952),LL(0x88440bc0,0x4cd58881),LL(0xc280e977,0xd9fcb83d)}, {LL(0xdf6cda83,0x0d3df9db),LL(0x3d55047e,0xc629414e),LL(0xc16b84c9,0xe05738a8),LL(0xe8783570,0xf4bdc724)}},
   {{LL(0x9a93a5c9,0x7d876a59),LL(0xa8d12f61,0x026be75c),LL(0xe9b2aa43,0xe49900ed),LL(0xb3a68dad,0x44d6dc80)}, {LL(0x7d23e11b,0xf96b116b),LL(0xb6814209,0x12791212),LL(0x6cc65956,0x3e2807cf),LL(0xf405ffae,0xcc606ca7)}},
   {{LL(0x5df47226,0x5484b2c5),LL(0x8802da81,0xfbaf9042),LL(0x087adadc,0x84146315),LL(0x58d593b3,0x6adbcbc1)}, {LL(0x68b97533,0xc1fb3896),LL(0x954cc1b7,0xa6919aac),LL(0x27a4ddd0,0xf301b2e4),LL(0xdf1a07b1,0xa15c16eb)}},
   {{LL(0xc145a6c7,0xb36c017d),LL(0x968798da,0xcca64382),LL(0x8d0eff5d,0xd13b6376),LL(0x2206e681,0x06e39e2d)}, {LL(0x3add517a,0x1d9dffa4),LL(0x10d95fef,0xe670e6c8),LL(0xf1c7c7a8,0x0ecb51ab),LL(0x74945305,0xf17dff51)}},
   {{LL(0xb00d9c0f,0xf71b09b1),LL(0x9c72c80a,0xc719cf62),LL(0xe00a49a1,0x310c5aeb),LL(0x1b33c5e6,0xd01285e5)}, {LL(0x05aa6eb7,0x7b23e7c7),LL(0x6bc88677,0xf84188b1),LL(0x64be321a,0x7e034cb5),LL(0xe884323f,0x270df734)}},
   {{LL(0xe5137d20,0x218e68f9),LL(0x0f7e70ad,0x79588cba),LL(0x58b86b0a,0xb6d37f52),LL(0x7cc591fe,0xcb281c98)}, {LL(0x8fe40e06,0x30e03fed),LL(0xed9ca793,0x394ded95),LL(0xbcd3a3da,0xf1d22cdd),LL(0x0c591130,0xcb88cb27)}},
   {{LL(0x8ff0cbf4,0x67786ba3),LL(0x0565d337,0x85738a44),LL(0xaf9a667b,0x9d3b35ec),LL(0x8f5f540a,0x45a17512)}, {LL(0xade5a5ba,0xf1ae5171),LL(0x39869be4,0x720e2823),LL(0x5352d84b,0x6893f14a),LL(0xc784dc20,0x919a4f15)}},
   {{LL(0xf7ae5a67,0x36136612),LL(0xeaff0924,0x11f43d1c),LL(0x39449b96,0xcfee088c),LL(0x70c42ff6,0x3dc48359)}, {LL(0xbf208b67,0x4072683a),LL(0xbe15d18f,0x35a7c2be),LL(0xe2c3391b,0xe61d2748),LL(0x39921d95,0x0a4109b1)}},
   {{LL(0x05545cfc,0xe2cd131b),LL(0x3ae20d7f,0xa898da2c),LL(0x50dc4c61,0x501cd849),LL(0x3374e7f0,0x10e287d4)}, {LL(0x38aea879,0x90d6326f),LL(0xef43fa08,0xc48d9af7),LL(0x6c20f168,0xf8f4886a),LL(0x23ccac4b,0xc5d34a86)}},
   {{LL(0xb3d7004e,0x72357752),LL(0x817bd377,0x167db0ed),LL(0xdfb64d05,0x5d45b3da),LL(0xf0b414ac,0xed4b7fc4)}, {LL(0x0bf1dd64,0xc178941b),LL(0x8fe835a5,0x43feac17),LL(0x6a014609,0xe1c23a17),LL(0xd5e23bd5,0x63255991)}},
   {{LL(0xd7dfec55,0xefc76468),LL(0xc0831696,0xb1bc3fee),LL(0x5f52433b,0x0996811b),LL(0x799649fc,0x6b8b6daa)}, {LL(0xab518b64,0x6e9f7cb6),LL(0x38a3a2ab,0x6a67a009),LL(0x928209e2,0xe55de954),LL(0x3da81142,0x98b6d0a7)}},
   {{LL(0xe3f832e8,0xdec30331),LL(0xe50fa9e3,0xa9b77f3b),LL(0x5167c6a6,0x20febc21),LL(0x76fb0f13,0x0ce07d1a)}, {LL(0xe796f8a3,0x9745dead),LL(0xd95deba6,0x2cb4eb1f),LL(0x4caf2afe,0x062e7cac),LL(0x16ace879,0xf50ce065)}},
   {{LL(0x1d99d3e7,0xdec8954b),LL(0xa48262c7,0x5287e954),LL(0xcc3530de,0x1c6fbd17),LL(0x53af4211,0x6bcbea50)}, {LL(0x4dce058d,0xe3533bca),LL(0xfc9cdf00,0x6fe62e64),LL(0xe8ec4cf9,0xee29fdec),LL(0xc8d52f80,0x7361797d)}},
   {{LL(0xf4e36023,0xb1d858da),LL(0x73e6dee1,0x4a1282ce),LL(0xce1d71cc,0x6ba8f8ba),LL(0xcbbd8eb9,0xf5b7d6b4)}, {LL(0x5aed382b,0x60f8bd50),LL(0x3f3a46b1,0x47b40519),LL(0x8a997d93,0xaed13bb9),LL(0x4dc6e35d,0x6cc2260e)}},
   {{LL(0xccf915d9,0x173bfcdd),LL(0xc2d46f6e,0xad4525e1),LL(0xcdd2382b,0xb7ecec0b),LL(0xd2b76c84,0x01ae8291)}, {LL(0xbec6b980,0x2d1e2a91),LL(0x7008a00c,0x1b0040be),LL(0x7d555120,0x6ac708d7),LL(0x0d745eef,0xa6017568)}},
   {{LL(0x1ed38ef4,0x735e3511),LL(0xcebe5a8c,0x7c97f284),LL(0x24fecbac,0xd4059313),LL(0xde18c42c,0xf874ca4b)}, {LL(0xdbb829b6,0x9ab736a8),LL(0x82ff128d,0xe914bdde),LL(0xfd0f362b,0x6e949bab),LL(0xffea2e79,0x275824cf)}},
   {{LL(0x8cc52417,0x81f57245),LL(0x2f274090,0xed0a9079),LL(0xfdd0ba2f,0x98c3372e),LL(0x3ae99478,0x49820f41)}, {LL(0xce373d3e,0x1c47e09f),LL(0x6dd12846,0x875d7920),LL(0x15d5bbb1,0x7a9e7973),LL(0x6d227962,0x48512656)}},
   {{LL(0x199241e3,0x2c167c88),LL(0x7209ca37,0x98c1df6a),LL(0xde89e412,0x09a1583f),LL(0xc792de48,0xc19ed5b9)}, {LL(0x74dc0834,0xb8dd1b1d),LL(0xa04456e9,0x9d458529),LL(0xad0ad39d,0x66ef5463),LL(0xe641edc5,0x8d7df4a1)}},
   {{LL(0x6bd322e4,0x97815de2),LL(0xc1f77fb3,0x0bf6fc83),LL(0x8b4f7152,0x49378167),LL(0xbf0a1712,0xfdd476ef)}, {LL(0xf2f9883a,0xe87e1977),LL(0x9ad2d034,0xdbb2fcbf),LL(0x64e1a4c6,0x5afdd161),LL(0x3e435191,0x0e43f911)}},
   {{LL(0x42253804,0xde2d1ee6),LL(0xdaf13e57,0x6def6ceb),LL(0x05423bab,0xae09fd4f),LL(0xad46e409,0x6f6c17b8)}, {LL(0x6c345cd3,0x966fa09d),LL(0x47abc35d,0x6c8aa1e9),LL(0xe015a566,0x02991686),LL(0xcd2f925d,0x39b3aeee)}},
   {{LL(0x9119c117,0xf9cda92a),LL(0xf4f833e1,0x7b21ce82),LL(0xe4f99193,0x87517bf5),LL(0x4eb412f9,0x1b7ddec9)}, {LL(0x6b077498,0x7a30dd57),LL(0x0ec44230,0xe060625f),LL(0xb0e5446b,0x0f355dc4),LL(0xbbd2df28,0xdf324d65)}},
   {{LL(0x649966a5,0x28c7eb34),LL(0x26639e19,0x97587f4f),LL(0x0bce0f38,0x0724cc00),LL(0x4ae6280a,0x63578add)}, {LL(0xc7fd6a1a,0xf1beaa57),LL(0x7b017e35,0x83b1a533),LL(0xefdf2ed1,0x01c027e3),LL(0xd2d31852,0xf373d4ea)}},
   {{LL(0x65b8f5d7,0xe568acb6),LL(0x1240a545,0xea8ce1b8),LL(0x555fac44,0xb95b0db2),LL(0x768333dc,0x01d18170)}, {LL(0xab1798ad,0xf938b55e),LL(0x999a7e5b,0x73a0d9dd),LL(0x57fd9b51,0xd2359bb5),LL(0x77fb4e5e,0x20f1d4fd)}},
   {{LL(0x932dc388,0xf5efd71f),LL(0x21a37385,0x40f86819),LL(0xff935ef3,0x05395fb2),LL(0x1b615e8e,0xc2ee43ac)}, {LL(0xe82d509a,0xa3bb6518),LL(0x30b93347,0x3a87d5a2),LL(0x5b130bcc,0xac0a5ad0),LL(0x9154d73a,0x91fe8fdd)}},
   {{LL(0xdeb203a2,0x677d7d48),LL(0x8b0168e1,0x4d4108fe),LL(0xddc3d24c,0x16be4ad1),LL(0x9865df69,0x9b0ea387)}, {LL(0x4c50ec70,0x16daf932),LL(0x478c96a3,0xa4799bda),LL(0x7114d3bb,0x4ef24d3f),LL(0x46e6bbdc,0x30a31509)}},
   {{LL(0x97f3cb4b,0x60137187),LL(0x0a29d865,0xf2b66d8f),LL(0x60064a5c,0x93a4a37a),LL(0xe8c3cf47,0x7dee9bed)}, {LL(0x0b7ee8b8,0x748833ce),LL(0x56f89483,0xc07f2f6d),LL(0xd24b406f,0xd71a40d8),LL(0xebbb7655,0xbe3b2e8f)}},
   {{LL(0x4a536998,0xa23c2b05),LL(0xa9be2e14,0xdcaf45b3),LL(0xfe346ad1,0x515ad15f),LL(0xb7847640,0xb9c06a18)}, {LL(0xf35bff4a,0x8552eb06),LL(0x2b7a29f5,0x4fb792e7),LL(0xa41a38b4,0x1cce2af5),LL(0x02b42414,0xde68bd0d)}},
   {{LL(0x7cd66728,0x8124d6e2),LL(0x55efbadd,0x5906d1b4),LL(0x827f2741,0x7e17855a),LL(0x12c6966c,0xab525dfb)}, {LL(0x758e0cd3,0x065ae99a),LL(0x517318a9,0x0dcb8f5d),LL(0x42441f5e,0x48756645),LL(0xd79d535e,0x03859154)}},
   {{LL(0x8217e4bf,0x99bb28cd),LL(0x8291e54d,0xd6aed2e5),LL(0x1c92a65e,0x8f9067e3),LL(0x1540b9b5,0x120890ea)}, {LL(0xec60a215,0x227d7c86),LL(0x556d8c65,0xb6609e85),LL(0x47f8c8a3,0xa6a26c37),LL(0xf1204bdc,0x4c850fe3)}},
   {{LL(0x42db4eb8,0x25f7e61a),LL(0x3d62869d,0xfdf05575),LL(0x52b31c23,0x8b36a744),LL(0x1a5e8d4c,0x83b83c89)}, {LL(0x5d9208bf,0x72d38dd3),LL(0x8cf7b6f4,0xbeb8873b),LL(0xcf90bcb6,0xa3ec5c36),LL(0x9a6d5fe7,0x35adda6f)}},
   {{LL(0xf61c68d9,0x7312423d),LL(0x20bcaf77,0xb1c4e10f),LL(0xf168ee57,0x4df2850d),LL(0x180985e1,0xed07a4de)}, {LL(0x2fba1f23,0xcb353d6b),LL(0x778cc15e,0x00ea9094),LL(0x20857170,0x4967faaa),LL(0xd7153bc4,0x9ff70dbe)}},
   {{LL(0x59f62fc6,0x49eb7994),LL(0x3c91862d,0x5f459faf),LL(0x46d8f2e0,0x1c10f621),LL(0x252954e7,0x7e669c9a)}, {LL(0xa83f6c57,0x4ccf659a),LL(0xec0a0a03,0xdc2b77eb),LL(0x2cc6b8a2,0xcf09ad07),LL(0xa97aa2d0,0x231960fc)}},
   {{LL(0xde227de8,0xc0609130),LL(0xf1d7ddf1,0x40d2691c),LL(0xf9a73890,0x60349cf4),LL(0xf9968132,0x3f50005d)}, {LL(0xf16f44b9,0xb4be853e),LL(0x799caac5,0x48bf4804),LL(0x3c562849,0xe6a64876),LL(0x854f283f,0x2f4d487f)}},
   {{LL(0x159484c4,0x64b77e39),LL(0x523e1529,0xd419d4bd),LL(0x143dcf7d,0x1bf9510c),LL(0xed5cb4e1,0xa85bea71)}, {LL(0xec715818,0x73a4cfd2),LL(0x67f475f5,0x88b11d0e),LL(0x4d12361c,0xbfe170d8),LL(0x00a0f979,0x9fc48e64)}},
   {{LL(0x65682105,0x6a8bb2dd),LL(0x00bd952a,0xc1362a9c),LL(0xa6013753,0xef5b3d89),LL(0x8fdfa22a,0xc87bbacb)}, {LL(0x31bb19e4,0x74fbdfc0),LL(0x32bfe260,0x7d058029),LL(0xe53da990,0x54a4cce4),LL(0x822da672,0x01acdff6)}},
   {{LL(0x95597766,0xd2a2d484),LL(0xd43dc7fd,0x5960ac1f),LL(0x8d6db685,0xcf095b6f),LL(0xa85618f3,0x87232088)}, {LL(0x34753c7c,0x91497a48),LL(0xd6353024,0xf682e372),LL(0x0c9b271c,0x7889ceda),LL(0x18340951,0x7126504e)}},
   {{LL(0x967c8a60,0xf786b821),LL(0xc17f3d99,0xfce01b37),LL(0x1f2a8575,0xe23c00a1),LL(0xab6ff8a0,0x7f56aa1b)}, {LL(0xd193dfcb,0xdb73869d),LL(0xd644733e,0xbec02c94),LL(0xf7b43261,0x283117bc),LL(0xb4108e39,0x920acf5d)}},
   {{LL(0xe49aebb8,0x33f1ef5e),LL(0x0fcea2c1,0x9ead51e4),LL(0xf8503f28,0x1f800a68),LL(0x34a75f67,0x78818531)}, {LL(0xb70ffb27,0x1aeb3760),LL(0xcb6309e9,0x1cca590a),LL(0x7170d241,0x8d09f360),LL(0xa0e0d0f8,0xbc970b5b)}},
   {{LL(0x31d038a3,0x2ec93eea),LL(0x5153f8a2,0x3482a0d7),LL(0x6641b5d8,0xedcbe914),LL(0xe516e387,0xc086e61b)}, {LL(0x9b875513,0x03814266),LL(0x37340a4f,0x6d37fee3),LL(0xe5d17ab7,0xcf78515e),LL(0x4119a759,0x0c7cd830)}},
   {{LL(0x54924618,0xbd49062b),LL(0x41e7e7a3,0x34c44f45),LL(0x706bd0af,0x0039f3d2),LL(0x0be9a628,0x146cadc6)}, {LL(0x57d48746,0x6d5d5020),LL(0x82caf4b0,0x0ea43f7b),LL(0x8a064d34,0x11a08927),LL(0x95638fa2,0x30c0ef40)}},
   {{LL(0x602a871b,0x4b950c04),LL(0x6a731537,0xf50cb1ef),LL(0xcbe5e4ef,0xb87a1cd3),LL(0x3dd1c601,0xb1fc4894)}, {LL(0xa516617a,0xdf402711),LL(0xaaf63c65,0x5bdd1d67),LL(0x6e559bd9,0x020d1062),LL(0x81ec09fc,0x4dec26d0)}},
   {{LL(0xeeeeb2bc,0x7b91fafd),LL(0x33aaf2c4,0x56290f98),LL(0x79c7bf9e,0x57abbd27),LL(0x2b1e1ecf,0x568bdee6)}, {LL(0x470f1d01,0x58f8c80c),LL(0x1b9cb76b,0xeecfe398),LL(0x311a0634,0xc0ffa4de),LL(0x0ae99877,0x425fcd13)}},
   {{LL(0xf7bd0748,0x1964c681),LL(0x9d920471,0xebcca16f),LL(0xab4aa03e,0xa72b40cb),LL(0xa05624fc,0x4397d6af)}, {LL(0xa94fca0a,0x372d522c),LL(0x3035b9fc,0xe1010d60),LL(0x4f788d44,0x9f1f00cc),LL(0x6a88b672,0xfd00ec75)}},
   {{LL(0x2983aef7,0x53706702),LL(0x9b458edb,0xa5f67b0b),LL(0x7db93ca8,0x10789b90),LL(0xfd27cd55,0x885346f0)}, {LL(0x2ebb5f15,0x3af5b0c8),LL(0x2a36b2a7,0x282e4c4a),LL(0xa6d88bd4,0x2f9d5d8b),LL(0x9856b7aa,0x6f016bda)}},
   {{LL(0xa8198c1d,0x990ae53e),LL(0xa07e7ac5,0x295beceb),LL(0x48c2d246,0x576f790f),LL(0xe3ea9864,0xe99ab2ae)}, {LL(0x43e2d400,0xcf4959f2),LL(0x7a39dfea,0xdd1d8fad),LL(0xfcd7fda0,0xdd6ff9c2),LL(0xb6ace55e,0x61c25b3e)}},
   {{LL(0xb4dcddad,0xf94742af),LL(0x44601959,0xc49cfa21),LL(0x30c18470,0x07b3f1d1),LL(0x6e6afc82,0x2736cb99)}, {LL(0xe24a8785,0x401fb234),LL(0x074f51ea,0x9af8ba40),LL(0xa9faed0c,0xe1acc646),LL(0xc9667008,0xd5a5f789)}},
   {{LL(0x68c3ab8f,0xc6436514),LL(0xfe8d6b46,0x6fa0d734),LL(0xaf7f49c7,0xe5fccbfc),LL(0xbebcc58c,0x42c88c53)}, {LL(0xe2a0f7f2,0x7d2e2fed),LL(0x36a18b26,0x694eb76c),LL(0x6b0f657b,0xf0e6ae43),LL(0x48f1ece7,0x8a0f6255)}},
   {{LL(0x8674bfee,0xd594c168),LL(0xac7d5ebd,0xe59ad38d),LL(0x21645a1e,0x080a6b97),LL(0xf221b37b,0xb900f0e1)}, {LL(0x04cab97d,0x562dabce),LL(0x6f472462,0x5c308741),LL(0xc7c4cba8,0xa5d87e23),LL(0x9b061062,0x5237fe16)}},
   {{LL(0x222021c1,0xeddfbeb4),LL(0x4e7a2a8e,0xa4fe57d0),LL(0x2de56c47,0x0fbf6bdb),LL(0x6fcebc6c,0x819588e7)}, {LL(0xdf041e3a,0x14196961),LL(0x40cd4f23,0x76a31437),LL(0x8e1a877d,0x44acd14d),LL(0x37d7b7de,0x227a35c6)}},
   {{LL(0x842a9534,0xe1934f1d),LL(0x53ed73e2,0x7a2ed2c1),LL(0x3903924b,0xcffedd58),LL(0xb306431d,0x7c9dbf55)}, {LL(0x56e06ab5,0x61a72f10),LL(0x616bc5cb,0xb46cf5cc),LL(0xf7c22216,0xecf07e10),LL(0xd9364b3a,0xa4bddad9)}},
   {{LL(0xda8b1c93,0x548b95b2),LL(0xa1e1c0cb,0xc09a9598),LL(0x21d80851,0xedd80ef1),LL(0xc283f047,0x4684c439)}, {LL(0x87333da3,0x07ca41f3),LL(0xca79a8f4,0x173ec4de),LL(0xb4aec6eb,0x89ce06f2),LL(0x15aaf7f0,0xfe6b0e92)}},
   {{LL(0x7c1b9ed3,0xdab8876d),LL(0xa2606f83,0x88aba90f),LL(0xbebaf9f6,0xcd21a408),LL(0x0042a162,0x09da6696)}, {LL(0x2d66ccf6,0x4a9b8b21),LL(0x44d5a648,0x34c74904),LL(0x3b0e9564,0xf3fe98e9),LL(0x221aa4a5,0xe4a8a352)}},
   {{LL(0x26c2b53e,0x6278b4b5),LL(0x1b1708ea,0x4ddf26ce),LL(0x6eb0d845,0x704207af),LL(0x0f5862ef,0x60533de3)}, {LL(0xe54393c0,0x2b5945dd),LL(0x145ea298,0x55941df2),LL(0xc240f654,0xe2b500b6),LL(0xcf9f6934,0x5a49d8f1)}},
   {{LL(0x27502203,0xfe8d5468),LL(0x58ade316,0x985039d4),LL(0x0a687415,0xefd373f1),LL(0x43526774,0xefccb791)}, {LL(0x0f4497d9,0xeef8d46e),LL(0x1601ab9a,0x4152df71),LL(0xe47b2ad1,0x4250cd2f),LL(0xfb048180,0xa2b63fa5)}},
},
/* digit=34 base_pwr=2^238 */
{
   {{LL(0x787d1f1c,0xd8a6cb6f),LL(0x3d219a66,0x427bac94),LL(0x383146b0,0x51d7d49f),LL(0x7863d781,0x8164b77f)}, {LL(0x2f9631b8,0x1646b084),LL(0x849388df,0xef5b3aa8),LL(0xe58cd383,0x60536422),LL(0xf43ea3a0,0xb079d911)}},
   {{LL(0xcb73887e,0x504ac041),LL(0xc3ce3a33,0xf878b618),LL(0x56393e75,0x57ef73d5),LL(0xd276c08c,0xe4372d2e)}, {LL(0x0924cf58,0xfd9bc894),LL(0xaaa317e2,0xfa2a4deb),LL(0x79608da5,0xe51edccc),LL(0x8cd4b960,0xadcc68fa)}},
   {{LL(0xf8e156c7,0xaa66c201),LL(0x1ab2e3fe,0x7c7cf22e),LL(0x0a677d85,0xe479c393),LL(0xb87c412b,0xc0cd340f)}, {LL(0xf95ff321,0x2b2bcef4),LL(0xb8409952,0x65da11c9),LL(0xeb67eb9c,0x143a2218),LL(0xe53508e4,0x8919ff25)}},
   {{LL(0xa9e0eeae,0x6f154f09),LL(0xab05a657,0x2246e6fe),LL(0x1045b85d,0x4d7c1c81),LL(0xd3bb7432,0xde99ea37)}, {LL(0x63184ff4,0x058f8187),LL(0xd134bfc3,0x2a223421),LL(0x23120320,0x1560dbed),LL(0x76a3de9c,0x37243c95)}},
   {{LL(0xd36a81b1,0xb8f3851a),LL(0xbdad7ad9,0xfbc62bfc),LL(0x561e0f8c,0xf68215c7),LL(0x1bcf765b,0x894131d1)}, {LL(0x45c5d736,0x8da01f9e),LL(0x7484e0c1,0x025de05c),LL(0x6858b504,0x62f4c66c),LL(0xd6dc5f93,0x754b85d6)}},
   {{LL(0x822a3de0,0x5b37cecc),LL(0xa98a37c2,0x422e49b1),LL(0xbe41e927,0x3ef53d89),LL(0xf4d5bffa,0x0994dd11)}, {LL(0xf7eacca3,0xa62ea556),LL(0x7c746025,0x37b4e230),LL(0xa8e14253,0xb4291e37),LL(0x2a2b666c,0x2bfc9eaa)}},
   {{LL(0xc26e5588,0xf604604a),LL(0xa7ec3971,0xf75816ff),LL(0x26a30a6d,0x859e9ec7),LL(0xa1a5b815,0x2ce57b66)}, {LL(0xd65e8ec2,0xc7aa4df4),LL(0xa5d82ede,0xbab6b3bb),LL(0x7b088314,0x7a11b25d),LL(0xc2c636ac,0x501a3891)}},
   {{LL(0xe256b02b,0x9f116c8f),LL(0xfa5946e0,0x71495693),LL(0xc335452a,0xeb9696ff),LL(0x4971162e,0x01ca5929)}, {LL(0xc0f28e72,0xee0a1f50),LL(0x70d8df1a,0x2baac62c),LL(0xf49110f8,0xcf65d297),LL(0x9a45e16a,0x041dbb01)}},
   {{LL(0x5e1410c0,0x8db69426),LL(0xa70d0268,0xb21f3c6a),LL(0xbac0ddac,0x64a3c30e),LL(0x66a2d33a,0xdcebdedc)}, {LL(0xa519de21,0xc5dcd769),LL(0x19322c69,0xa692b6a0),LL(0x154fca13,0x454add5b),LL(0x4935eba2,0xd2281cf0)}},
   {{LL(0xf2602323,0xb5f44fe7),LL(0x5d68a3db,0x772fb6a6),LL(0x76eec37a,0xf519c5d4),LL(0xada6c3f4,0xbc8e9a15)}, {LL(0xf18effee,0x9cd2a2f2),LL(0x1808ab42,0x9601c142),LL(0x0480ad18,0x05d110f7),LL(0xef6a7f33,0x5f7e0721)}},
   {{LL(0xe6409e21,0x1afbeaec),LL(0x317f7967,0xf6714651),LL(0x80124751,0x34cd21ff),LL(0x931d9639,0xf85c70ec)}, {LL(0x1ca19094,0x4e26bef6),LL(0x0b841b9a,0xc513f66b),LL(0xb9b41001,0xe25507bd),LL(0x94f49f7c,0xd77fee94)}},
   {{LL(0xd39e1ee4,0x20b19223),LL(0x4e3e6c2c,0xc8832a2c),LL(0xa3a45c34,0x64a8f43d),LL(0x21fb291b,0x52a05eef)}, {LL(0xe4b68e38,0x10d3e24a),LL(0xee2d8a40,0x5289120a),LL(0x425b7da8,0x33836b98),LL(0xb00c64e1,0x5bd418f3)}},
   {{LL(0xd511c3f8,0x10e92e5a),LL(0x18b62b7d,0x17f1301d),LL(0x97f0fcae,0xf710b02d),LL(0xbd394477,0x8b1030f6)}, {LL(0xe5aab897,0x49040009),LL(0xce75b4d3,0xfdb23ac1),LL(0xf2b70e1b,0x7a43d904),LL(0xf94fa56f,0xdc09e995)}},
   {{LL(0xd075dd65,0x9f314e85),LL(0xb9e26b8d,0xc0d39ce0),LL(0xfdc3b678,0xd3f6778e),LL(0xfc8497df,0xce6573e9)}, {LL(0x67abaf7a,0x1f4535f8),LL(0xa47dd948,0x80706dab),LL(0xc059242b,0x670ae5bc),LL(0xcf5f9308,0x3a29bc73)}},
   {{LL(0x8af2bf74,0xd2f8e297),LL(0x6c48bbec,0x98dbb443),LL(0xeb448447,0x211a3a96),LL(0x5af4a2c0,0x88ffb240)}, {LL(0x9cdf9e20,0x1959dd34),LL(0xf34627e0,0xa4d0d839),LL(0xf00057cc,0xa5cecad3),LL(0xc5d97b18,0x22f32cce)}},
   {{LL(0xcedc2c97,0x31a02241),LL(0x2b632641,0xf0f5489b),LL(0xcbfb588f,0xb09091dd),LL(0x5d9478e7,0x5ffd0f38)}, {LL(0xdae35eda,0x13f141a1),LL(0x62f0b26c,0xd25563a6),LL(0x1b9dde18,0x80014b17),LL(0x7da49b36,0x9fcf8f81)}},
   {{LL(0x93519f31,0x68aac84a),LL(0xc0b3660a,0xe14c35c1),LL(0x08195bc5,0x29f447dd),LL(0x10bba62f,0xc61fbbe6)}, {LL(0x4ed8621a,0xc471624c),LL(0x0950a4c6,0x8005f67f),LL(0x93a2a33e,0xdfc3b3e5),LL(0x3c1d0e42,0x9c3c815e)}},
   {{LL(0x93904766,0x1ed1aedb),LL(0x376fd0bc,0xcd5e0cf6),LL(0x90d03033,0xdd4c3378),LL(0xd85dca30,0xde39db01)}, {LL(0xe6fab58b,0x49d01dc2),LL(0x6d449f51,0xd16d7940),LL(0xd20a95e6,0x3fb6cf8e),LL(0xbbeeccb1,0x0b10a596)}},
   {{LL(0x0005acd3,0x06ceaa62),LL(0x47555053,0x09db6b28),LL(0x7e7d18d7,0x45d84a85),LL(0x229ad33e,0x33c28b02)}, {LL(0x72e4f34c,0x1e5a6b52),LL(0xb64fa831,0x81eefbe6),LL(0x2aa209aa,0x4983b84a),LL(0x38d6a8d6,0x20777198)}},
   {{LL(0x40096f25,0xbe993805),LL(0xec820131,0x900d4bdd),LL(0x2a993f9c,0x2170cfd3),LL(0x2dfe1007,0xa0e3d894)}, {LL(0x0e7df109,0x600d0b5a),LL(0x47fde3dd,0xc904985a),LL(0xcb89816a,0x15597a84),LL(0xb9dfeb9e,0x8ac8b027)}},
   {{LL(0x6450a179,0x5c9211bc),LL(0xf6333f95,0xd448a70a),LL(0x824e1d7f,0xe9c9a964),LL(0x15750ae4,0xc47d3f3c)}, {LL(0xadcf9482,0x959f776b),LL(0xe741ceb3,0x00557ffe),LL(0x353d7898,0x8b69d3f6),LL(0x45cfa492,0x6b4d80d3)}},
   {{LL(0x30c313da,0xc33ead78),LL(0x67eee139,0x86f96c3a),LL(0x08611b15,0x0c6675c7),LL(0x60620c27,0xf9ee695d)}, {LL(0xd70c9258,0xb35d438c),LL(0xa5e7a4b1,0x1bc2b1e7),LL(0xef92f629,0x38d257f8),LL(0x79fd1eb0,0x090af73a)}},
   {{LL(0xf59342e5,0x96ebd1f0),LL(0x4d053375,0xd4869362),LL(0x5fab54aa,0x7db504e2),LL(0x6e8e43fb,0x17c0801e)}, {LL(0x136b1941,0xd3904d62),LL(0x28a43bd1,0x5932b753),LL(0xacb35572,0x551d895e),LL(0x1a6fdfbe,0x3f7a8a46)}},
   {{LL(0xf7a2df83,0x9e3ea4fd),LL(0x64524d44,0x8b68b26b),LL(0x126aee21,0x74caeeab),LL(0x915d9e1c,0x590a00a5)}, {LL(0x49b90eff,0x5ae2a6ab),LL(0x2df4fe51,0x74b4cb1e),LL(0x07fcb6ed,0x0306ed11),LL(0x502f5b30,0x564ebe2e)}},
   {{LL(0x0c89e9ba,0x5a09a32e),LL(0x84f2466d,0x967f9dfb),LL(0x26a1a8a0,0x8b27416c),LL(0xc3158a18,0x1c21ef95)}, {LL(0x2be23ae9,0xa7ee1ad8),LL(0x4daa1fcf,0x1f312d04),LL(0x44f9c7d7,0x6782f843),LL(0xe19e2681,0xb12ea2bf)}},
   {{LL(0xd20578af,0xd2e43cbf),LL(0xbb5819b4,0x5566460a),LL(0xb658c03c,0x86f6c860),LL(0x62d42d82,0xc8d90309)}, {LL(0xcb883ceb,0x7975a5f3),LL(0xdcded5a1,0xf6f5cf0f),LL(0xd3eb810b,0x25554fb1),LL(0xa596c7c6,0x3df7536b)}},
   {{LL(0x83de31cd,0x255ca136),LL(0x7795eb96,0x7ac532ee),LL(0xb12bc235,0xfa9d83a9),LL(0x4b287a83,0x7df5d231)}, {LL(0xb4f19fce,0xb2eaaaf6),LL(0x1a045f6a,0x7caabfb0),LL(0xb1449e6a,0x6d1b7f40),LL(0x12d22c82,0x24ae41da)}},
   {{LL(0xc0a9d128,0xb0f7a0c3),LL(0xaed0d3bd,0x2adc34d3),LL(0x13e778e6,0x4ebf5778),LL(0xbb8476ba,0xd3b89bd0)}, {LL(0x37413953,0xe09eb528),LL(0xd8ba3471,0x952b705c),LL(0x86a79c09,0xcaa81ade),LL(0x7e0e7b2a,0xc08eed3d)}},
   {{LL(0xc80b4196,0x313fb103),LL(0x88c7ac81,0x25449ece),LL(0x24f16fa2,0xa6cb9ad3),LL(0x728a0c4b,0x4602c441)}, {LL(0x5a000a9c,0xc3366f88),LL(0xef8778bd,0x146623e3),LL(0xf0773fdc,0x184ba0b1),LL(0xaecd8d63,0xe1d115ef)}},
   {{LL(0xae165e6c,0x420d5473),LL(0x108d8575,0xefe137cd),LL(0x6fcff7d9,0x15378c57),LL(0xc1f5b601,0x49c48099)}, {LL(0xf68473f1,0x1b0e3aee),LL(0xd320720d,0xa78329bb),LL(0xdca54cad,0x385e521b),LL(0x5c5b8d60,0x478ce06f)}},
   {{LL(0xca7c4d4b,0x215d7d33),LL(0xa095366c,0x773f3ab3),LL(0x7afeeaa1,0x668e0d11),LL(0x4878d8a3,0x410cd205)}, {LL(0xb2800646,0x2748fd98),LL(0x73a1dbef,0xf1183786),LL(0x7567ed3a,0xecc31bd2),LL(0x0ceb3873,0x775210df)}},
   {{LL(0x9a8f42ff,0x2ea0c2bc),LL(0x4a1c029a,0x62974240),LL(0x9e4dd41b,0x5ee5f5f6),LL(0x2e110249,0x5b1bba80)}, {LL(0x78da8016,0x5ac4eadb),LL(0x1809f79f,0xc2978780),LL(0xe3f8c0de,0x39d2dbce),LL(0x064d3ba9,0x7fb4b5fc)}},
   {{LL(0xcd481ab4,0x038a736d),LL(0x396888eb,0xc4b15c5c),LL(0xd421f36f,0x459a896d),LL(0x47f54663,0x6058b90f)}, {LL(0xafefebe5,0x0a5f6771),LL(0xa5b7c2ca,0x45c97ab2),LL(0x85139ca4,0x6d547af9),LL(0xa29d71cb,0x6db218de)}},
   {{LL(0xfb956184,0xca6e0e7b),LL(0x1f660ac6,0x6682b6e3),LL(0x8b21bceb,0x3959e396),LL(0x632cf9c8,0x0459fd46)}, {LL(0x74f296bb,0xc741250c),LL(0x990dbefa,0x29b9cacf),LL(0xfc35bdf7,0x5065d818),LL(0xa551dc04,0xeb8e9e1b)}},
   {{LL(0x11befe9e,0x4f7d6f7d),LL(0x7478fdee,0xa88f1fce),LL(0xafa688e3,0x39b1e053),LL(0xe16847e0,0x562a0d6e)}, {LL(0x34c26d14,0xf6044e4b),LL(0x7df61b90,0x5ebe8727),LL(0x6b5e5a39,0xa82a4de4),LL(0xfb9d296c,0xc916b0ba)}},
   {{LL(0x2e1dc01e,0x029f1cb2),LL(0xfc429483,0x7699d92e),LL(0x154560f0,0xee0e425a),LL(0x787b6641,0x3f5cdfe6)}, {LL(0xe5f6794e,0x726d87bb),LL(0x23aecad2,0x97d73588),LL(0x09ca351c,0x47f4f5b9),LL(0x57dc5e3b,0xd742ef4b)}},
   {{LL(0x71411a86,0xccd2209d),LL(0x2223e6ce,0x94d57663),LL(0x66c7950c,0x228a7400),LL(0x54dd4e37,0x2d00ef6e)}, {LL(0xd60f66be,0x9ea5daf3),LL(0x8aca724f,0x743c58a5),LL(0x44e38446,0x1f638406),LL(0x92ef6bb0,0x06314bb0)}},
   {{LL(0xbb05589d,0xa7459e7f),LL(0xbfa913d7,0xc3a39592),LL(0xdf07b190,0x27dbabee),LL(0xd2ee62ff,0x1c8a2f33)}, {LL(0xe31e8d63,0x60c8bcb8),LL(0xce410061,0xea1604d1),LL(0x3d7f7a98,0x55e8cfee),LL(0xebc64102,0x49efc316)}},
   {{LL(0x41934914,0x04c86d8e),LL(0xab7facd4,0x26667c76),LL(0xa71a8916,0x319158db),LL(0x114fff43,0xb802942d)}, {LL(0x8ce544f5,0x5efdef7b),LL(0x70e870c1,0xf531c718),LL(0x4d92071d,0x4b9a5f1b),LL(0x60cc48b6,0xbe767cf2)}},
   {{LL(0x717381ea,0xbf389d37),LL(0x06bc5bcb,0xefd9e984),LL(0x67ff1110,0xcc8bc960),LL(0xb05612e4,0xd3414c0b)}, {LL(0x927fad1a,0x084e5f05),LL(0x438e241f,0x999bd581),LL(0xfaa4fab8,0x0c917602),LL(0x95080657,0xda0520d2)}},
   {{LL(0xce2f1af2,0x3160f928),LL(0x364f56e4,0x61186d84),LL(0x25fa68f0,0xe36a5fc0),LL(0x774c584b,0x9e6f66bd)}, {LL(0x9ecb449a,0x2611bba4),LL(0xec5a0667,0xb1e0b341),LL(0x6cddb6c3,0x336de76d),LL(0x65a18f95,0x9668b5b3)}},
   {{LL(0x7c3ec388,0x1ff6c81f),LL(0x40a8e2d0,0x53545b05),LL(0x14ae31d6,0x990a3cc5),LL(0x063a2995,0x769b4c26)}, {LL(0x039e279f,0xcea238f4),LL(0x732fb38e,0xbfc5cfb9),LL(0x82fa05d8,0x99f5a33c),LL(0x69c42686,0x274dc741)}},
   {{LL(0x76af2af7,0x193338ee),LL(0x6914ae66,0x0488c19f),LL(0x5fc58bf4,0x8d197f4e),LL(0xf0e61d4b,0x23de54df)}, {LL(0x44a165e1,0x547dd09f),LL(0x1c2d5471,0x99878065),LL(0x39b311db,0xb2cabfad),LL(0x4b61a7eb,0x0aed63d9)}},
   {{LL(0xbe8110ef,0x03713ac5),LL(0x50f989d3,0xaab1917d),LL(0x358fe8b0,0x0d80fe98),LL(0xa7a1f8e3,0xf6e874c5)}, {LL(0xdeb42398,0x05650fd8),LL(0x1c44de73,0xbad3e085),LL(0x1c27f3c2,0x5369135f),LL(0xa7fc74ac,0x14bc04f8)}},
   {{LL(0xb5dae291,0x18cbf622),LL(0x9356b88c,0xce290570),LL(0x39eba4e6,0x61bbb446),LL(0x980fee37,0xa79c291b)}, {LL(0x19960cc6,0xd9f18006),LL(0x0ce98896,0xb0823f41),LL(0x1377df6f,0xf2bc612e),LL(0xc0b0e71c,0x1c07bded)}},
   {{LL(0x37211fd5,0xffbf9841),LL(0x04a81410,0xbd704a6b),LL(0x6abf1c67,0x653cd2ee),LL(0x40681621,0x73ab8aa1)}, {LL(0x271ada5c,0xc0bae4fd),LL(0xc46f189d,0xf567cae8),LL(0xa5535334,0xd183cb27),LL(0xe53c530c,0xcbf133f7)}},
   {{LL(0xedd6a17e,0x32e53f78),LL(0xa2194fae,0x6ce6da9a),LL(0x58cd3586,0xa89b8054),LL(0x43b520a5,0x0037febb)}, {LL(0x653e2c0b,0xbe67a2cf),LL(0x50301f52,0xc07a1ed1),LL(0xf5ea954f,0xf98b2b60),LL(0x7af6c499,0xfa6da95d)}},
   {{LL(0xe3889cb1,0x44892091),LL(0xd45ae338,0x123fc555),LL(0x02a63128,0x2bc4a9ef),LL(0xa1dbb436,0xb72012c9)}, {LL(0x556a0b46,0x8c75f7b3),LL(0x5b7608a4,0xe4c6f46c),LL(0x38fce20f,0xb36abf68),LL(0xbf6c21e1,0xb5a8e657)}},
   {{LL(0xcecd5963,0x9ceaeefe),LL(0x6105fc29,0xe84f200d),LL(0x8591e346,0xc28981a9),LL(0x207001f1,0x0be4e931)}, {LL(0x88616b18,0x31399d9d),LL(0x238c626e,0x3dac9f55),LL(0x65574274,0x0213fca7),LL(0x827aa853,0xa3560244)}},
   {{LL(0x1ca99134,0x3ffbfeeb),LL(0xd07a2dac,0x0a4b56f6),LL(0x75417a6b,0x01795eca),LL(0x18a5fb22,0xe2a6dd9c)}, {LL(0x8aca0cd8,0x13c97586),LL(0x7c323c52,0x3c2bb26e),LL(0xe38319bf,0xa3688cae),LL(0x4c88726a,0xe04b44b4)}},
   {{LL(0xb0a88a4c,0xfed864d0),LL(0x6b1fa305,0x3e6cf152),LL(0x00e18e4a,0x8416b5f0),LL(0xfa4cd8f2,0x3a7603ca)}, {LL(0x7ec750ef,0x8b04d5ff),LL(0x1c1ced05,0xe1867f9b),LL(0xdac2f8c1,0x87ffd0fb),LL(0x08b3cdca,0xc9ebe420)}},
   {{LL(0x029c247e,0x5028a4fd),LL(0xa5ae3e76,0xd6225a43),LL(0xf970817e,0xfb3fa71c),LL(0x9ab4aef1,0x74216809)}, {LL(0xa39c2f13,0xca81ee99),LL(0x86a97827,0xa8336e42),LL(0xb6489555,0xb75aff99),LL(0xe565435c,0x005b2338)}},
   {{LL(0x524bdb34,0xbaee27bb),LL(0x82e47e71,0xbf22e1c9),LL(0x97491139,0x6ab1d712),LL(0x2cf4cbff,0xf420ce06)}, {LL(0xb2b0c86a,0x9f96a2fc),LL(0xabeb7335,0x42765dd9),LL(0x45b7e044,0x7c223bb7),LL(0x1794e144,0xce3f9245)}},
   {{LL(0xf3ee5c4e,0xa0a15b27),LL(0x54622215,0x1721c5bf),LL(0xada5a99c,0x0003fd16),LL(0xdbdccc7b,0x8e96dd56)}, {LL(0x43f83748,0xd1abdc0b),LL(0x0f5ce74b,0x71cac4a6),LL(0xd46717ad,0xb8539aff),LL(0xb99661d9,0xeb65c589)}},
   {{LL(0x85e89e17,0x66b4df3b),LL(0x6fc30672,0xc94fad66),LL(0x81d90df8,0xfd2aa80f),LL(0xbd8234c1,0xed73a163)}, {LL(0x72eb45dd,0xe1a2c0b9),LL(0x902b5ca9,0x417e355d),LL(0xd5128812,0xa8f55aaa),LL(0x3c14cb79,0x4826b934)}},
   {{LL(0x394d7a92,0xeae495e3),LL(0xb90faec0,0xcba23153),LL(0xf6d9d80c,0xd687c821),LL(0x8bff3082,0x951dada2)}, {LL(0x701708ad,0x4e74f1f0),LL(0xdd2134f2,0xa6333cd1),LL(0x04665f7c,0xeea276cf),LL(0xae74c17a,0x527257fc)}},
   {{LL(0xe51b53bd,0xeb3fd493),LL(0x69ec567e,0xae7807db),LL(0x5de15fd0,0xa50124aa),LL(0x1336f055,0x781bfe70)}, {LL(0xd70a0dfc,0xb5729a74),LL(0x9f50c1a3,0x89da37f3),LL(0xcd8e6c1c,0x6e063297),LL(0x181d0271,0x17eb6ec1)}},
   {{LL(0xe4e52a8c,0x36e7251a),LL(0x94705324,0x3acfe89b),LL(0xbc130c3b,0xaa94f06e),LL(0x309ae25a,0x01b5e44c)}, {LL(0x0f61b55b,0xb72160f2),LL(0xe7bbc3f2,0xbef61953),LL(0x1bf138a1,0x96734d7a),LL(0x08c25135,0xdaa6186c)}},
   {{LL(0xf34534a2,0xa3b031b2),LL(0xde46f581,0x44136619),LL(0x6d379647,0x4d0ed04b),LL(0xbb2b6735,0x4879d90d)}, {LL(0x590156e0,0x8f7e031a),LL(0xf42bbc53,0x28428354),LL(0x5c5b791e,0x1cbed33c),LL(0x4cfc5562,0x17571645)}},
   {{LL(0x7f76462f,0x8392350a),LL(0x0c216ccb,0x659ce7db),LL(0x047e35d5,0xe87a78b7),LL(0x6e0862d6,0x307c4861)}, {LL(0xe70741bd,0xd444fb86),LL(0xfea1abe2,0x1138a886),LL(0x62b79c4f,0x4695397d),LL(0x003130ee,0x11aaf588)}},
   {{LL(0x3a11712b,0x53bdda6d),LL(0x40fba3d2,0x30c384bd),LL(0x50ea38be,0x63039585),LL(0x3da9738a,0x7f110eca)}, {LL(0x5b68c01e,0xbd701fc6),LL(0xcc48f38d,0xd23f3e8f),LL(0xf8b9bb65,0x6e2557eb),LL(0xa3dafc8f,0x29ceb4b6)}},
   {{LL(0x4b6b7705,0x24659686),LL(0x4aca2b43,0x04143a8a),LL(0x975e06d8,0x3baed256),LL(0x3e834249,0x846fb3c9)}, {LL(0x75f6770a,0x7890761e),LL(0x203c78fd,0x1187920e),LL(0x6b26281f,0x9b06c3a9),LL(0xa98215e1,0x3fe3dccd)}},
   {{LL(0x4f33655e,0x099d7d7a),LL(0x662fb55a,0x1ba571e6),LL(0xcbc497f0,0x1a0d0147),LL(0x2862ff38,0xa94218ae)}, {LL(0x5ce08047,0x1b0f919b),LL(0x2baf46cd,0x9a3ac37a),LL(0x8964cc68,0x76b7a03a),LL(0x4d3e1a40,0x5aed8c6d)}},
   {{LL(0x7f034ff4,0x6607179c),LL(0x3781eac2,0xefb8fcd9),LL(0x7b022138,0xfa57f8a9),LL(0x56ab6039,0xc5bb9f1d)}, {LL(0xe4d2ab7f,0xf9027e24),LL(0x77a9e364,0x3d67ad71),LL(0x1f7f487d,0xc579e70c),LL(0x2a7e6bd0,0x7fefc894)}},
},
/* digit=35 base_pwr=2^245 */
{
   {{LL(0xa45cfd31,0x2cb91662),LL(0x16f65cfe,0x09dd56d3),LL(0x14f3de51,0x983e005d),LL(0x210f64fc,0xb9dc05b0)}, {LL(0x885eafe5,0x22790afd),LL(0x7444bdec,0xbd5213d3),LL(0x8987300a,0x289dca92),LL(0xb3960b76,0x69fb2ac2)}},
   {{LL(0x9ae7540f,0xe3274886),LL(0x6131e921,0xd7386631),LL(0xf2a360c8,0x2e3d4fd8),LL(0x3d9d41e0,0xb20a59b6)}, {LL(0x99082a34,0x72b67eae),LL(0x51819470,0xfad6aa7d),LL(0xa2d1d007,0x7c299b9a),LL(0x8100bed0,0xc1f841e0)}},
   {{LL(0x43e90ebd,0x2c1f7d4c),LL(0x1fc72b07,0x58b78107),LL(0xaf94f402,0xda8503e1),LL(0x59f244b0,0xfbb724b7)}, {LL(0xfcd8755a,0x2fcd2186),LL(0x868482b7,0x7407cdee),LL(0x349be3d5,0x4d62f578),LL(0xdcc6202c,0x4a012544)}},
   {{LL(0x151ffc08,0xb8a414d2),LL(0x740d6b55,0xaa79acf0),LL(0xcdf472ab,0xeeab0104),LL(0xa3aa5f1d,0x5014a8c1)}, {LL(0x33f13425,0x8c743405),LL(0x57eb54d4,0x2b776b49),LL(0x548a723b,0x3a0cc4ac),LL(0xc79fe63a,0x65aae6f3)}},
   {{LL(0xee5e579c,0xe8b388f2),LL(0x991c03d4,0x31cc9373),LL(0x567bfa7c,0x53eed518),LL(0x67f985ed,0x267e282d)}, {LL(0xb4763ea0,0xc435fd22),LL(0xe39b7703,0xead83837),LL(0x094ba5b4,0x806daad5),LL(0x45842672,0xa738a847)}},
   {{LL(0x9984c4c2,0x99421b42),LL(0xd35c7bbd,0x1a3bce27),LL(0x3563b09e,0xe51ae6f6),LL(0x8d9c9fbf,0x8e67853b)}, {LL(0x6b2100b5,0xca8784da),LL(0x98879bba,0xe89a24f7),LL(0xe286b039,0xe901b45c),LL(0xf50384bd,0x23dedbb8)}},
   {{LL(0x5cbf7df2,0x4728cbdb),LL(0x6764329e,0xed274fdf),LL(0x642d199a,0xc2af1a07),LL(0x17a50e7e,0x5d665659)}, {LL(0xfaa5eb82,0x7babf4bc),LL(0x99fe4026,0xd3bcfc67),LL(0x607d9f41,0xaa5d2648),LL(0x967efac1,0x7405c071)}},
   {{LL(0xdfa782a4,0x79447ef9),LL(0x74cd9500,0x6dadc8e1),LL(0xdc38f762,0x0574020e),LL(0xe2ee7a14,0x17596d7e)}, {LL(0x9e1f8adc,0x9ef75af7),LL(0xa4791da0,0x5ac5f216),LL(0x7b7b5d80,0x1583226b),LL(0xa21c26cc,0x59f3f053)}},
   {{LL(0xf95e30d9,0xd80e7fdc),LL(0x0a3a3616,0xecf5041c),LL(0x03043fa6,0x50b93b8b),LL(0xae262ad6,0xa31a2aa4)}, {LL(0xd63cd98d,0x1468b370),LL(0xdc07a367,0xfb89cc65),LL(0x4d47b59f,0x6cf1df6b),LL(0x1b6350fe,0xab451a99)}},
   {{LL(0x8c124dff,0xeb74554d),LL(0x21be0be0,0x781a8c4d),LL(0xe3510068,0xfaacc154),LL(0xd6238265,0x16655d65)}, {LL(0x0466134a,0xba46d27b),LL(0x3101e283,0x1a3f51b9),LL(0x096ec237,0xc08298a9),LL(0xc69cfb5b,0x46248627)}},
   {{LL(0x81a0500f,0xf9e7a5a4),LL(0xbd2e03e7,0x92db27d5),LL(0x82899e3c,0x3dcce4f6),LL(0xf39a39c7,0x861f1797)}, {LL(0x69dc8961,0x175b2430),LL(0xdc67953e,0x93d2a88e),LL(0x92d794d6,0xa40f3704),LL(0x3526eeaf,0x607019f0)}},
   {{LL(0x22f37d65,0xf20e163b),LL(0x32cf180a,0x70fd00c8),LL(0x0b17244e,0xff1a97d2),LL(0xacedb33a,0x9a5a141b)}, {LL(0xcc16bbb4,0xf03dd868),LL(0xa40e44e9,0x9b15372d),LL(0x15ac397f,0xd5ba6436),LL(0xc821f6b7,0xb1a886d4)}},
   {{LL(0x4b7b4e21,0xbe3aacda),LL(0x66b56729,0xad9829fe),LL(0xd541cc1a,0x78965cef),LL(0x7d17631e,0x414bfea7)}, {LL(0xc64dd621,0xf55835d9),LL(0xef644d93,0xa0ebf68b),LL(0xc8a40021,0x01d15340),LL(0x42b99aa0,0x00ae640d)}},
   {{LL(0x6881e64f,0x92321ee2),LL(0x5267bdd3,0xaccab5c8),LL(0x5927647b,0x0c1f87ea),LL(0x162c6d86,0x0093d07e)}, {LL(0x389c711d,0x49868df4),LL(0xc11e1f37,0xe68d69ae),LL(0xb4207246,0xa40e7aa8),LL(0xce832395,0x4ae8d126)}},
   {{LL(0x86450cc0,0x5f9b128a),LL(0xc8ec07e6,0x88f76293),LL(0x179702b8,0x0762f293),LL(0x4910006d,0xb5696102)}, {LL(0x35fe0505,0x3951291b),LL(0xce8d7056,0x70f75a5c),LL(0x2eb13369,0x4541beaf),LL(0xa643316c,0x7060a749)}},
   {{LL(0x49c879a5,0xee16abd0),LL(0xa47ac42e,0x844874a7),LL(0x3c9c2326,0xee3f8a20),LL(0xdeaed33b,0x99a12054)}, {LL(0x63b333ae,0x4814a15b),LL(0x9d923fa0,0xee9f28a5),LL(0x33b1b1ef,0x5b0cd250),LL(0x8346d928,0x3ccc39b9)}},
   {{LL(0x002bec95,0xf5c1000e),LL(0xf63528c2,0x2ba2f18c),LL(0xcdcec15a,0x8102f6c8),LL(0xbb13d14a,0xab7effcd)}, {LL(0xfcd3787c,0x183e0ba2),LL(0x2f4a7fc0,0xae70635e),LL(0x760bbc96,0x473ed37f),LL(0x8a8efb39,0xf0ea0acf)}},
   {{LL(0x29b526a9,0x63cea36c),LL(0x9d03f3db,0xcdb31613),LL(0xd57cca8e,0xa3891096),LL(0xa14a8ffc,0x646418a9)}, {LL(0x8075291f,0x10f8906b),LL(0x2c618bf6,0x8221d941),LL(0x8a5da4df,0x1dc1ae7a),LL(0x8a8cc8bc,0xb66b96e3)}},
   {{LL(0xfe3551b4,0xe4da7e48),LL(0xad9b3827,0xe6891cc9),LL(0x6b37b99f,0xb266733f),LL(0xfd5d1219,0xfccce911)}, {LL(0x7262b8cc,0xe5a47c4b),LL(0xe656af49,0x5d349caf),LL(0x7e04e48e,0x7a3a4a28),LL(0x80ea7d03,0x7c39a68e)}},
   {{LL(0xbee0d434,0xf35d5e32),LL(0x0b536161,0x9651f3d9),LL(0x72cb370c,0x42634cc9),LL(0x406b3457,0xa7764026)}, {LL(0x65d53d80,0xec7525bd),LL(0xadcc8b05,0xf44a1bca),LL(0xda112ddc,0x12ef8427),LL(0x20a0f78f,0x796a65b3)}},
   {{LL(0x6bd5b0ab,0x12726e24),LL(0x8242fe07,0x9e441467),LL(0xde2bea52,0x4b52e276),LL(0x10422c2c,0x3a76b6b4)}, {LL(0xb4e496b9,0x71f14945),LL(0xf36dce4f,0xd20f04b0),LL(0x2b310e90,0xa0e57d8d),LL(0x32ec8f38,0x59eb7737)}},
   {{LL(0xaaf6619e,0x20a19834),LL(0x691a538e,0x633b94e8),LL(0x92cdf395,0xea1a8985),LL(0x4349b971,0xa3a01c57)}, {LL(0x30822c59,0x0d65192a),LL(0xa522ae8c,0x93a5152d),LL(0x0e1aa4bc,0x5b09a7a3),LL(0x8d3b97a9,0xdd2588f3)}},
   {{LL(0x5b62a3a5,0xafa1f62a),LL(0xa9ace9c5,0xbded10e6),LL(0xbf6e7fb2,0x9d03e061),LL(0x4b87177a,0x60c564a8)}, {LL(0xc22c14c1,0x36be1308),LL(0x9425b6bb,0xeeda57e8),LL(0x36af38c2,0x5ddaae14),LL(0xecdc2911,0x1695464b)}},
   {{LL(0x161e13e7,0x4b795e1d),LL(0x907e7069,0x0f9abc20),LL(0x54522fe7,0xfb3be618),LL(0x1938134e,0x9e2d0f37)}, {LL(0xd292c6b0,0xb8dc7c36),LL(0xc1e77895,0xbafbf59c),LL(0x1b6c55f3,0x7d0132cd),LL(0xf609f087,0xefa02ed9)}},
   {{LL(0x03519f9f,0x4bfe6aeb),LL(0xdab4c075,0x248e19a0),LL(0x69429f29,0x83ee803d),LL(0x8190ce56,0xdbbe31e2)}, {LL(0x6b787a5d,0x3ba735d2),LL(0x1818070c,0xfa021185),LL(0xa3683cee,0x9b653000),LL(0xe9517ba2,0xfc3c7907)}},
   {{LL(0x88d94f69,0x6521c92f),LL(0x7b52da8d,0x3950e9e8),LL(0x8ee64a88,0xadb81700),LL(0xf73994fe,0x8ccbfa3c)}, {LL(0xb048e71e,0xb780ab12),LL(0xe2aeb607,0x52879e7b),LL(0x3237554f,0xef04b1ed),LL(0xe1d5a5ef,0xaeba6a96)}},
   {{LL(0x266f7e93,0xedb58542),LL(0x5ea02852,0x9a1b8057),LL(0x5beb3fbd,0x1963c6f2),LL(0x1ad52473,0xf4183355)}, {LL(0x6faed2f4,0xca772e9e),LL(0x3cf8fd1f,0x937eddd0),LL(0xc1d49dac,0xb3255376),LL(0xe670d3cc,0x549c2119)}},
   {{LL(0x3b6cd199,0x10edbf39),LL(0x75644d6a,0xe9479223),LL(0xd6e8cc36,0x36cfba92),LL(0xfe00d80a,0xa37b1d91)}, {LL(0xdeb5ef4a,0x3aadf918),LL(0xd3893df2,0x5bb2ca4d),LL(0x6569ab8b,0xa776000e),LL(0x1cf64905,0x4fb2308f)}},
   {{LL(0x273731c2,0x04da4d09),LL(0x23c05518,0x1d4a61fe),LL(0x0d5206e5,0x201098a3),LL(0x06430c82,0xd9a7ad4e)}, {LL(0x36f7f462,0x56df0d06),LL(0x44c81042,0x2c067f3d),LL(0xc453d28e,0x01193bc9),LL(0x45ce6e64,0xcdf5af5d)}},
   {{LL(0x0f7d8d12,0x9992ce1a),LL(0x0e5e4542,0xa7c46a61),LL(0x057802ba,0x3fcc0647),LL(0xc7dccbe2,0xa78f73d8)}, {LL(0xf138dc6d,0x67f37b94),LL(0x650a9002,0x89330885),LL(0x68aa24c7,0xf661caf2),LL(0xbf73c435,0x47732bcd)}},
   {{LL(0x3b04622e,0xb9ba5f91),LL(0x477d420a,0x24265f73),LL(0x0d44cb89,0x5da6ddb0),LL(0x151fc40b,0x9f8cb8b6)}, {LL(0x9b9f2613,0x81b6956b),LL(0xebb917df,0x37419206),LL(0x2bb7a709,0xdb9cfc16),LL(0xbacd3fb7,0x7a800aa3)}},
   {{LL(0xd93f6e1a,0xf8ea9359),LL(0x3d41c118,0x729005d4),LL(0x7cb641af,0x4c293410),LL(0x895e8e78,0x6b2b4671)}, {LL(0x5958fad3,0x2a1251d0),LL(0x78619fe4,0xb69bc2be),LL(0xd911d318,0xd74df34c),LL(0x15102704,0x5def8378)}},
   {{LL(0x08268482,0xb19ea17a),LL(0x1c37e5d1,0x14591196),LL(0x7640df9c,0xe0e12d2e),LL(0x8c941274,0x8fd6bd4d)}, {LL(0xdcd142b1,0xc3f9f120),LL(0x78dfe6b0,0x106c49ac),LL(0xcfd9b542,0x243c8e93),LL(0x0a2c5fe6,0x6758062d)}},
   {{LL(0x15f2f638,0xee5a99e8),LL(0x13979ab6,0xb95b008d),LL(0xacfcca6a,0x7fd03105),LL(0xe4ced1b3,0x6af914a4)}, {LL(0xa25f8627,0x8bef3d0f),LL(0xf9b2f931,0x21bae309),LL(0x2195a839,0xe85dee2b),LL(0xa3baeb25,0x46ad0ad9)}},
   {{LL(0x022b62a9,0x6d8e40f8),LL(0x90b5cd33,0x4a6bbabf),LL(0xffa89bb2,0x53481e6b),LL(0x22003cc2,0xd674b3b3)}, {LL(0x004a2aa6,0xc71a0a85),LL(0xb5404657,0x86df9697),LL(0xc74e80cc,0x407727f4),LL(0x950a7b08,0x39c13926)}},
   {{LL(0xd74472a4,0x26bee75a),LL(0x2eb6f0d6,0xbf7c4ea0),LL(0x608bea48,0x689a5de5),LL(0x29d513f8,0x5b383892)}, {LL(0xda457cf9,0x49fee2c2),LL(0x62d523d3,0x7fc0aee7),LL(0xb636a46e,0x5bf447de),LL(0x8194766e,0xda3efd98)}},
   {{LL(0xd382756d,0xa77c3ad2),LL(0x0fa55315,0xc0eaa9de),LL(0xb1df90e3,0xe68d0a51),LL(0x01d1d8a7,0x0994e8c7)}, {LL(0xa91bfed0,0x4f898bc3),LL(0xab6025df,0x1c2a3e46),LL(0x8b269271,0x37bd5c37),LL(0x8b97f0af,0x4e07f5ca)}},
   {{LL(0x97923d14,0xe346b5aa),LL(0x9e0bd9c4,0xa8549f61),LL(0x40113a60,0x78e59d6b),LL(0xed8a0fc6,0xe3015fb2)}, {LL(0x8b43288a,0xfc56a08f),LL(0xcae6c18a,0xcbdb8cae),LL(0x5f4423db,0xcb147c44),LL(0x10f361c1,0xa6aaa6c9)}},
   {{LL(0x7caf702a,0x6be86c0c),LL(0x736f6dac,0x2188e83c),LL(0x59ba2af9,0x40b5ed25),LL(0xab8018c3,0x76211894)}, {LL(0xf5b0b048,0x0c1c532f),LL(0xe3200521,0x7503aca9),LL(0xdfa7eb2d,0xb9325d85),LL(0x2edbb68f,0xe6c25a00)}},
   {{LL(0x8c673c89,0xf9ff5867),LL(0x099c7bae,0x4925a046),LL(0xdbb1e1b6,0x0b3bf59a),LL(0x30ae084f,0xc7e9d9f2)}, {LL(0x0fa1776f,0x70982396),LL(0x624c897b,0xb2e1b98f),LL(0x6c3534d5,0xa9a6407d),LL(0xa4dc3f63,0x5e22319b)}},
   {{LL(0x2431dc01,0xc2f0bf3f),LL(0xc2cfb315,0x478f797d),LL(0x3b3ae0c5,0x6559f59c),LL(0xe18e74a8,0x7e95aa62)}, {LL(0xd3fce912,0xf2a94006),LL(0xe1bd96ce,0x7f1b57a2),LL(0xa3d7b828,0x55028ad0),LL(0x4f09fe92,0xadae7e92)}},
   {{LL(0x757b418f,0x2174c736),LL(0x3661d54d,0xd904ba43),LL(0x63218ecb,0x0281f912),LL(0xc50c8eb6,0x5fd03ba0)}, {LL(0x896a5aea,0x29626906),LL(0xe55ee73f,0xab4d3f27),LL(0xedfd1991,0x3db1c43d),LL(0xa3f9c340,0x498cc31a)}},
   {{LL(0x4fe75d33,0xa43bdec1),LL(0x66ae5d4f,0x5b067dfb),LL(0x464c8239,0x84581814),LL(0x503a52ea,0x2f10557f)}, {LL(0xa10fbb90,0x21c4c180),LL(0xf79d5e02,0x33b191ee),LL(0xb499478e,0x6dee3881),LL(0xbfbd56fa,0x27dfef0b)}},
   {{LL(0x28be2d62,0x671a3dd7),LL(0x050897ff,0x06f2f4c2),LL(0xb7c4587d,0xd92bdab6),LL(0xfd8d5160,0xd2253a16)}, {LL(0xf1c098b1,0x64f6e4ae),LL(0x11ea7255,0x005a3939),LL(0xdab542e5,0x2ed4eb92),LL(0x50c5e874,0x26920bc1)}},
   {{LL(0x5d0bc87c,0x93e8f58a),LL(0xb2b29b4b,0xaa4d313e),LL(0x01b2096f,0x3e175dec),LL(0x1cf31783,0x6c560972)}, {LL(0x73b76f6b,0x9d41aca2),LL(0x5f1d4b12,0xa2454cf5),LL(0x65b35eea,0xa5615196),LL(0x70af4fde,0xf241e516)}},
   {{LL(0x65061472,0x5255e91b),LL(0x5bdbb257,0x6ef98d2d),LL(0xc74c7b2c,0x0d1d1ab1),LL(0x2e9febde,0x9ffb9fdf)}, {LL(0x6c50bf24,0x853f3b9f),LL(0x6fbd22bd,0x3d369594),LL(0xbcdad9a9,0x4d281126),LL(0xdc46ddc1,0x99eb62b6)}},
   {{LL(0x4b10c402,0x5aa8c8b2),LL(0x473af61d,0x2e79f595),LL(0xce360f37,0x96153360),LL(0x66bc29dd,0x16dffe22)}, {LL(0x1137f9c3,0x35536eb1),LL(0xe2a6a47a,0xd636ecad),LL(0xb499f840,0x83cdf214),LL(0xd247f18c,0x3642c77c)}},
   {{LL(0x916ef527,0x4d906a2e),LL(0x293dc085,0xadeb94d0),LL(0x1491da3e,0x03a07801),LL(0x0b84d2eb,0x177dceae)}, {LL(0x7b691e0c,0x61e5a3c1),LL(0xd172cea3,0x47d40bd7),LL(0x8ca76bce,0x7d0646ad),LL(0xc64d635f,0x90b030a9)}},
   {{LL(0x97118df2,0x71eca8e7),LL(0x3ac9536b,0x2cd48f70),LL(0x89fb4d72,0x9ffd991d),LL(0xebf781fb,0xd49006bc)}, {LL(0xd3e60da1,0x688af07f),LL(0x619063b7,0x5f74aa46),LL(0xa40d313f,0x44fcbeb3),LL(0x326faaa4,0x0ed5908b)}},
   {{LL(0xf41ec05d,0xe836d537),LL(0x221b0c32,0x01eaf207),LL(0x72f8c904,0x1d6a0bb6),LL(0xdfd74641,0xa6ef58b2)}, {LL(0x811bd6cb,0xbb855ceb),LL(0x05408eab,0x7b1c8b71),LL(0x4187fb7f,0xd24d709e),LL(0x8b30a9be,0x283d647d)}},
   {{LL(0xf9f0d6e6,0x6d9d3793),LL(0xb1c06b19,0x02fc3ddb),LL(0x94d9abec,0x8ff86793),LL(0x24705873,0x1f20bba2)}, {LL(0x0021b290,0x74eebc12),LL(0x35b6c157,0xd859521e),LL(0x431aea79,0x2201aa41),LL(0x90da1a75,0x79c1caaf)}},
   {{LL(0x6e412a6a,0xcd6abab7),LL(0xb4c58b05,0x82399139),LL(0xa3b55538,0xdf416966),LL(0x679882d3,0x2b2d546f)}, {LL(0xf9145657,0x17425cbc),LL(0xe1b8247e,0x3cc6735f),LL(0x57edd04c,0x13e50c56),LL(0x1b85b7cb,0xc8723137)}},
   {{LL(0xdc0ab9d5,0x907b5b02),LL(0x4ab23b78,0x5617fb7f),LL(0xe8f449cd,0x7ae8ff03),LL(0x174e0e22,0x86d3ff17)}, {LL(0xbf1e9f8d,0x22cb7f69),LL(0x0b0c62f0,0x12f0abbe),LL(0x537f658c,0xc8315981),LL(0xc467f2b4,0x43da2770)}},
   {{LL(0x5b9e88ef,0x3ef9bb81),LL(0x3a8e51f2,0xb8526318),LL(0xf8d744ac,0x2e47cb7f),LL(0x510aaa7c,0x63d6dc16)}, {LL(0xb40ccc41,0x54da7cdb),LL(0x402b2ad9,0xdecbe5fd),LL(0x34c8f225,0x14c6f15c),LL(0xc6559496,0x6d8b2342)}},
   {{LL(0x66fea635,0xa4b72281),LL(0x22f248a8,0x55f5c27f),LL(0x0959cd01,0x3ced1483),LL(0xb53bdf42,0xcc6469db)}, {LL(0x1e460645,0x2bb2346f),LL(0x9d7535e7,0x4d8573c6),LL(0x49cd2d68,0x988cddd5),LL(0xb9835538,0x785c4a70)}},
   {{LL(0x1f6e396c,0xb08337b3),LL(0x49a782f3,0x6166b21e),LL(0x8ec9b974,0x1ac6198b),LL(0x0bb44d3d,0xee2e3446)}, {LL(0x35039dd9,0xdb283740),LL(0x29f5c692,0x7c708f95),LL(0x98ddb466,0x8914cce0),LL(0xd446f3cf,0x8bb1b9f1)}},
   {{LL(0xee0933a3,0xa9dea222),LL(0x4b26049e,0x2538bd43),LL(0xbdcafae2,0x18741aca),LL(0x16b0f4bb,0xe0f830f7)}, {LL(0x902caefa,0x0479ec95),LL(0xdcda9e64,0x1f858937),LL(0x515c4089,0xe75b4f7b),LL(0x2eb91b51,0xb78afde4)}},
   {{LL(0x18949935,0x1eebe3e9),LL(0xba092037,0xde8deaa9),LL(0x93609411,0xd43cf4ef),LL(0xc2d7b76e,0xe0fdb1e4)}, {LL(0x4e34b4bd,0x1d3191a5),LL(0x9ccc4c26,0x106d92f1),LL(0x29a2a6d1,0x1a404ef6),LL(0xc598f481,0x3338bc9c)}},
   {{LL(0xe3fcbf71,0x3945e39d),LL(0x9c89ab61,0x123b082c),LL(0x0f9f3c37,0xc7477f77),LL(0x7dbcc077,0x408c0c7a)}, {LL(0x3654f98c,0x6c4d99f5),LL(0x05299a1a,0x276a007a),LL(0x23e2d7d0,0xabd4b8ea),LL(0x86017545,0xe05a5f3a)}},
   {{LL(0xa11b03cd,0xde3b885c),LL(0x8df5d64e,0x46ef0755),LL(0xbf3f085d,0x112a49d6),LL(0x198ff32f,0xf6ebf441)}, {LL(0x7feae481,0x581c00d8),LL(0xcfde5b2f,0xf2b43827),LL(0x9b7358f2,0x3ceb7f8f),LL(0x55fe7117,0x95761fbd)}},
   {{LL(0xdc04773c,0x305836fa),LL(0xb3c4873c,0x66324504),LL(0x55b130de,0x5d878c1f),LL(0x8ad49a9b,0x96e9b28c)}, {LL(0x76d70429,0xd1a707b8),LL(0xaa402e90,0xaff33f93),LL(0xedbfb28f,0x733d6256),LL(0xa75d2342,0x9e421a7c)}},
   {{LL(0xc02e49c1,0xdf86b254),LL(0xb56d308a,0x6bb53867),LL(0x73f29561,0x771dde4b),LL(0x8bf28e5f,0x96eaf73e)}, {LL(0x06fbb550,0x9b1ee6be),LL(0x97d4a4e8,0xe09fec77),LL(0xd5aa84fd,0x93bdcd60),LL(0xd457ab9c,0x3fa8d3a0)}},
   {{LL(0xa0a2e52c,0x315b32b1),LL(0x3bbcb61d,0xe7e50b2d),LL(0x5e5c6e11,0x8a55cc0e),LL(0x961295ef,0xc2bfa998)}, {LL(0x66e996d1,0x4a5ab3bb),LL(0x4886a421,0x22c42e4f),LL(0x4850e0a4,0xa0cdd364),LL(0xc64ed713,0x7682d38d)}},
   {{LL(0xa2c539e4,0xe31575c2),LL(0xa756daf9,0x0bac5dcd),LL(0x91f55a12,0xe917cecf),LL(0xe96f6299,0x1e96433b)}, {LL(0x3700d8fb,0xeec7c71c),LL(0xdc9b4444,0x9a1d2965),LL(0xcf74f19c,0x3d2c6970),LL(0xac5e0d6b,0x3b444c48)}},
},
/* digit=36 base_pwr=2^252 */
{
   {{LL(0x8ccb854c,0xe563cefd),LL(0x65b0c45a,0xf5452cdb),LL(0x9c37f743,0xb3c78769),LL(0x95d444ab,0x34e9d192)}, {LL(0x52ff26b7,0x29347946),LL(0x9b94d642,0x70d6ecfa),LL(0xfdaffb8f,0x7d201858),LL(0x45dcdc71,0xc288719d)}},
   {{LL(0x0728a2eb,0xc695469d),LL(0xc433d11c,0x7b46244e),LL(0xf106c08e,0x4a8b99ba),LL(0x63422083,0x7989794f)}, {LL(0xd4fc5696,0x82218867),LL(0x3c79cdb8,0x6b021f28),LL(0xb26d5049,0x5ff7bbea),LL(0xa7261628,0xb78611ca)}},
   {{LL(0x531313d7,0x5a75f961),LL(0x66dcdc9e,0x85a1f4db),LL(0x6460e991,0xae3026b9),LL(0x17ecf7cc,0x7d467bef)}, {LL(0x05118708,0x8a0dbf67),LL(0xf3b2f1c9,0x54bfa368),LL(0xf2c0e4e0,0xa9fc9d5c),LL(0x5e93611b,0xa8c2ad11)}},
   {{LL(0xaa1256bd,0x3ef1faf0),LL(0x9e4631de,0x0f224545),LL(0xde9c2676,0x69cb9800),LL(0x95782b24,0x26019816)}, {LL(0xa66c0ccd,0x945c172c),LL(0xb440719a,0x6c25f635),LL(0xa49f681a,0x917d5dba),LL(0xb2dc5df4,0xc0cad047)}},
   {{LL(0x5960ef1c,0xd45bcf4c),LL(0x8c6979d5,0xbabcb16d),LL(0xae9090d6,0x8e3be750),LL(0xac0eb728,0x9481d261)}, {LL(0x0d6a7d46,0x46b436cd),LL(0x1f976501,0x6eb1a6a3),LL(0xdbe1064f,0x5984ffa2),LL(0xf809dc20,0xe6575fb1)}},
   {{LL(0x4d974a81,0xf0426d80),LL(0x97a74be6,0x61304f0f),LL(0xa9486531,0x2346ff98),LL(0xf53d781a,0xa1242cca)}, {LL(0x97355f15,0x482f03df),LL(0xbd6058cf,0xc607ed33),LL(0x68aefe28,0x03bc8cd4),LL(0x851307e4,0xa6e7de5a)}},
   {{LL(0xc6af7d44,0x2c07df0f),LL(0xb15a9188,0x310b251f),LL(0xd3e15c2f,0xd42661ce),LL(0x1b4d8313,0x5198fd90)}, {LL(0xda8368a1,0x7a6062cd),LL(0x5e9c2542,0x1a905d11),LL(0xdae37cee,0x1d752b70),LL(0x16bf84ca,0x3ed8c1a5)}},
   {{LL(0xeecc2f22,0x5190fb0f),LL(0x698d8e60,0x3df210f3),LL(0xf5f3ce72,0xcce57d3a),LL(0x312b8fc6,0xb2fb6223)}, {LL(0x71867c84,0x79947005),LL(0xbe139ebe,0x141cd92c),LL(0x5de7944e,0x415efc9e),LL(0x45821058,0xae9ee919)}},
   {{LL(0x5bf363dc,0xd696e1d9),LL(0x8251449c,0x6a1bcfc0),LL(0xa5fa53e9,0xa1b82dff),LL(0xeef05378,0x6c56b5be)}, {LL(0xc0e74dc3,0xaf9efe4c),LL(0xe5c1f1a0,0x3d9a7ae9),LL(0x2823c3e5,0x34b38577),LL(0x41fbabac,0x69f297dc)}},
   {{LL(0xd74c5a65,0xf01aff98),LL(0x1951a915,0x97993104),LL(0x723096a6,0x8b211915),LL(0xa769ef1f,0xf85910c4)}, {LL(0x8ddc0eb4,0x30cefb9e),LL(0xbb09607b,0xd5957eef),LL(0x2355b499,0x2e139c9c),LL(0xc1789013,0x5749531d)}},
   {{LL(0x5475f2d2,0x1603ca64),LL(0x0a336508,0x57190e0e),LL(0xcea7d558,0x2203b703),LL(0xfb5503e3,0xf16eba4d)}, {LL(0xb7344a98,0x62e2ce3d),LL(0x9a4efa7a,0xebf5b243),LL(0x1c914064,0x96212455),LL(0xbe5bbc07,0xd2c5e31c)}},
   {{LL(0x06c30b28,0x2b5f2f77),LL(0xbc9823d3,0x0931841d),LL(0xadfa6fdb,0xd16fb08b),LL(0xd6fd441e,0x8892bae2)}, {LL(0x2e576983,0x3fc64630),LL(0x07b05529,0x08c60fb6),LL(0x7afc1d4d,0x32b283b1),LL(0xa2f0e37f,0xc9c56965)}},
   {{LL(0x4644e173,0x8e719178),LL(0xf88b43ff,0x4c2a11ec),LL(0x7d3ddbb3,0xb13644e6),LL(0xc3d8703c,0xd4746056)}, {LL(0x55dca667,0x6611395f),LL(0x27c91d73,0x63596712),LL(0xea2ff489,0x4ca68a87),LL(0x337adc1d,0x2864a816)}},
   {{LL(0x224d4f21,0x8aa830ae),LL(0x9f7845dc,0xda6c122e),LL(0xfb240793,0xb0c61ffc),LL(0xce8580e9,0xf4df6842)}, {LL(0x0a990dc7,0x94372aaa),LL(0x5ce1aa24,0x42968cd3),LL(0x4df363a5,0x177c5ff0),LL(0x68c4546f,0xa8c3f737)}},
   {{LL(0xbd21c524,0xc69750d5),LL(0x22a6c4ae,0xbf3b4857),LL(0xe2883a1d,0xcefcbb98),LL(0xae13f22b,0x6ffef743)}, {LL(0x5defea01,0x6316ba60),LL(0x4ba63810,0x0a89e6a7),LL(0x15ab0e11,0x7f9af1de),LL(0x385911c9,0x6247ca15)}},
   {{LL(0x32f9eaf5,0x6f7b1a6a),LL(0xacfc13dc,0x2c440f94),LL(0x66b18adf,0x2cf39bc5),LL(0x9f01533f,0xb9939fe8)}, {LL(0x383a6450,0x031c4553),LL(0xf0102087,0x16d96ad3),LL(0x501f1360,0xcbd6fa95),LL(0x65f96c08,0x667d3ea0)}},
   {{LL(0x68a1a680,0xa5a7cbfa),LL(0x42041db7,0xf131d779),LL(0xd85d377f,0xbefee3ac),LL(0x3b62dfa2,0x6d0ed6b7)}, {LL(0x1baacfbd,0xef683f0f),LL(0xc976cebd,0xc586c4f2),LL(0x3b163339,0x3a4120dc),LL(0xc79e5c1f,0x9ac9b950)}},
   {{LL(0xe294926a,0xaf1ff35f),LL(0xa2103168,0x2703bab8),LL(0x658a52bf,0xc645560a),LL(0xe466fd97,0x5ff3ccd9)}, {LL(0x54867f14,0xe62fdc01),LL(0x9cdba39e,0x435ef950),LL(0x92828acc,0x2a7bbffd),LL(0xfe763981,0xe7538fdb)}},
   {{LL(0x8bfe9773,0xedf45173),LL(0x471b8b9c,0xd187fa01),LL(0x78fa54df,0x34506c35),LL(0xc2767589,0x73cab9fd)}, {LL(0x6726f305,0xf8f76c65),LL(0x8de332b2,0xea45012d),LL(0x87970e03,0xb746f40d),LL(0x1ba8fbd6,0xb2b2279a)}},
   {{LL(0x21147dbc,0x79cdc610),LL(0x9939a3cc,0x738ef680),LL(0x8101bd8b,0xd66d6ce6),LL(0x09323caa,0x65612acb)}, {LL(0x10310a29,0x6874b372),LL(0x5ee9ecfa,0x3cf30f0a),LL(0x8cfe1df8,0x4e1026ad),LL(0xd5989af5,0x75a153f7)}},
   {{LL(0x8b8e0c49,0xc362ccee),LL(0xb533f3dd,0x8adfc0d2),LL(0xa109572e,0xe02ab03b),LL(0xfd3066ec,0x06efacdc)}, {LL(0x3fa28700,0xf136a8ba),LL(0x308cceb9,0x48a1e987),LL(0x68253387,0xe8ee7c03),LL(0xc2b463c7,0x47febbe8)}},
   {{LL(0x39a5c4d3,0x485195f2),LL(0xa26241ec,0xf42e000e),LL(0xcd05368d,0x08c64f90),LL(0x857cdbdb,0x46fbd381)}, {LL(0x4c7e16ae,0xf711df8b),LL(0xe4edea42,0x95637e46),LL(0xad171465,0x2df8b206),LL(0x4bccedce,0xa31ea895)}},
   {{LL(0x50743bb6,0x28dbcb77),LL(0xf9cf84b9,0x13d12f8e),LL(0xc8f7d408,0x39e3d3af),LL(0xeba591d4,0x5824117f)}, {LL(0x1bead2d6,0xd8ef7c9a),LL(0xcaf53dd7,0x9003a559),LL(0x174cb9a9,0x33b2365c),LL(0xadb33afb,0x1149d080)}},
   {{LL(0xaea9bd3c,0x55231d00),LL(0xfdf3f015,0x07e107c9),LL(0xec9d8fce,0xf535947d),LL(0xbba349a7,0x8b64ed8a)}, {LL(0x049301df,0xdd5881fd),LL(0xe6490fd0,0xefac9c43),LL(0x73740a78,0xd9902852),LL(0x942c326c,0x6eef3724)}},
   {{LL(0x5cfb3c8c,0x5671a6e9),LL(0x0ea29721,0x040aabd2),LL(0xeac8da18,0x24e92ca6),LL(0xa31170c3,0xc34d3d79)}, {LL(0xb061e416,0xf81dd15f),LL(0x85f80af0,0xff7be70e),LL(0xade45cd4,0xa9faba4b),LL(0x505fddd4,0x42a6ab05)}},
   {{LL(0x0a793534,0x17d5821d),LL(0xce0ade43,0x9e094e54),LL(0xc42cb4d2,0xa127fb6d),LL(0xdb12dc99,0x43865428)}, {LL(0x59e3bfc1,0xb6b1b347),LL(0x1ec5b810,0x0b0076a9),LL(0xa6864982,0xbf2dd17a),LL(0x9d523c87,0x0c45947f)}},
   {{LL(0x4c5dd59e,0x9f53372f),LL(0xca5ce09f,0x3d0ceaea),LL(0x7c0337fb,0xf3ff88e8),LL(0xfaa022c7,0xb4fa4593)}, {LL(0xd65ea54d,0x575240a7),LL(0xadb92fb0,0xa4ec0a39),LL(0x79429eb1,0xc20e737c),LL(0x69addec4,0xcea931d1)}},
   {{LL(0x3e09f46a,0x7a29011f),LL(0x0e578a5b,0x9c36865e),LL(0x71d805f4,0x8746ea51),LL(0xe12d3024,0xf024de85)}, {LL(0x15a7f6be,0xc397b46c),LL(0x1b0580d7,0x612db6fb),LL(0xf736d087,0xe5342f76),LL(0x8c1e844c,0x65276853)}},
   {{LL(0x113841a5,0xedf48adc),LL(0xe5c482f0,0xc21b67e1),LL(0x684a540b,0xe43b0138),LL(0xa5d4b266,0xc4f2782b)}, {LL(0x397f3664,0x184e240c),LL(0x0d8788f8,0x968e89e7),LL(0x377e18bf,0xec3eba1a),LL(0x36002652,0x4d03cbbc)}},
   {{LL(0x1005a953,0x21eedee7),LL(0x75ba987e,0xc178ddf1),LL(0xc4ba43f6,0xd0d577f6),LL(0x4d6f24fd,0x9486f46c)}, {LL(0xc5421895,0x3d33c574),LL(0x842e52ab,0x5be6cb4c),LL(0x9debc9ff,0x3809690d),LL(0xa84a5b6f,0xe4b1c692)}},
   {{LL(0xd7e18b57,0x58b966ad),LL(0x77c94715,0x7ff0b61e),LL(0xf06add82,0x0e295883),LL(0x65c7f5a4,0x7c3c04fd)}, {LL(0x60223be5,0x4ea92660),LL(0x89262bfd,0x5d843a57),LL(0x36da11c0,0x35bf4aef),LL(0xaf859eb1,0xa6692f14)}},
   {{LL(0xa12fdf41,0xca1fc13b),LL(0x8224f5d2,0xd798c04b),LL(0x1dd5872b,0x22f4594e),LL(0x1bddfda8,0xdee12df5)}, {LL(0xed83420a,0x96473ff0),LL(0x8daa27f4,0xf41cf1c7),LL(0xaecefd8a,0x2772cd56),LL(0x4902b47f,0xd5ddaf18)}},
   {{LL(0xc0798101,0xff77551f),LL(0x26946bda,0x8baa01d6),LL(0x100525f2,0xd0087e47),LL(0x4c0de308,0x521d6254)}, {LL(0x9bbce049,0x4a0f45eb),LL(0xa6c6b96e,0x5ee33cbe),LL(0xd6a22345,0x9a6af4b7),LL(0x38b1b406,0x0d0d35e7)}},
   {{LL(0xbbedc29b,0x9e71252d),LL(0xcad1455e,0x3aa70bb6),LL(0x42a1778c,0xa406fb7a),LL(0xf0897613,0xd94f9646)}, {LL(0xf57f66c8,0x5256370f),LL(0x4432f679,0x95891e35),LL(0xbcb6b3d3,0x75d6423a),LL(0x2367483f,0x79d9ea01)}},
   {{LL(0x9efb0473,0x1e36ccc6),LL(0xdfdc0cec,0x3e64b034),LL(0x028bb238,0x13bfd326),LL(0x209edd95,0x171e9d96)}, {LL(0x07b22424,0xda258380),LL(0xd41b8023,0xe31e97f6),LL(0x7269cecd,0xdd4ed390),LL(0x12d5cec6,0x810fb3c8)}},
   {{LL(0xbabeec88,0x2f956519),LL(0x455baf52,0xb0350c52),LL(0x48d5abf1,0xa7fb548a),LL(0xca5e2d9f,0xcb81bd0c)}, {LL(0xa6d17b19,0xda5ecd39),LL(0x508e5149,0xd2588bab),LL(0xc3e23cfd,0x1a30cff5),LL(0xf89f8712,0x2dd398b4)}},
   {{LL(0x5b304005,0x2a911800),LL(0xd9dece69,0xd091be7a),LL(0xf6cabc89,0x147e93da),LL(0x44935824,0x7eac2018)}, {LL(0x32f5de9b,0xd4aaf2be),LL(0xd9396cd1,0xe302bc41),LL(0x2c069d1a,0x3c2794cf),LL(0xa9d433ae,0xf9197eaa)}},
   {{LL(0x4445e8c2,0x98f822ef),LL(0x1383ece8,0xc578360e),LL(0x01869457,0xa5372c12),LL(0x787d6644,0x1c6ed00d)}, {LL(0x86531814,0x77fb08cd),LL(0x63a70db8,0xeff6ee26),LL(0x80976119,0x980be153),LL(0xd69d60c5,0x534a09bd)}},
   {{LL(0x759dba20,0x71a58b0c),LL(0x679c0b40,0x34d5f06c),LL(0xceed2f9f,0xdc0e7e5f),LL(0x48808edb,0xaaa5996e)}, {LL(0xbcdd88e5,0x8ca96ff0),LL(0xc37c2b46,0x91b02d67),LL(0x95526319,0xbe4f3948),LL(0x89be56d1,0x4315c7f2)}},
   {{LL(0xdc85bba1,0xa312a3c0),LL(0x431ca797,0x3328fa8e),LL(0x68fd219a,0x5438bf1c),LL(0x85837d74,0x98812c6f)}, {LL(0xf8c96d49,0xe88c4913),LL(0xc2442aca,0xcc62e79c),LL(0x046655f1,0x4ef3c7d4),LL(0xdadab1ea,0x04a362ed)}},
   {{LL(0x30a199cf,0x975e2f3c),LL(0x014a165a,0x831e04a9),LL(0xaa126719,0x1e1d3c53),LL(0x1bf707a0,0xc42661e0)}, {LL(0xaa2da264,0x295b0738),LL(0x65d4ba34,0xb45f5ed8),LL(0x9f3938fa,0x27fb5a12),LL(0xcb26f86c,0x25fba614)}},
   {{LL(0xcf3c1c4d,0x6bd41981),LL(0xa0dedafd,0xd6f9239c),LL(0xae55b97f,0x46882526),LL(0x81b628d4,0x8e6fa994)}, {LL(0xdc0aa158,0xbdb314dd),LL(0x12ba2a17,0x35343678),LL(0x32e2e431,0xac018e83),LL(0xe65cc63e,0x43a64e35)}},
   {{LL(0x0b6603ea,0x887f3a2a),LL(0x76b2673f,0xe015426c),LL(0x27edfe8a,0x59dc5530),LL(0x68d9ebf3,0xea9eacf3)}, {LL(0xcc3e07ca,0x40301c8e),LL(0x0f57a2e6,0xd8cb9b5b),LL(0x60ec5864,0x542e6b52),LL(0x17f6affe,0xb8791dd6)}},
   {{LL(0x798d9993,0x6735bd1c),LL(0xd5da393c,0x006d8b25),LL(0x49e6d0d2,0x1d675bdb),LL(0x8607f99e,0x331d9a10)}, {LL(0x9dc4cd07,0x4ff8ab74),LL(0x64ea3192,0xa87d4ae1),LL(0x41196b5b,0xdde0d92e),LL(0xb2e010eb,0xa15ad47b)}},
   {{LL(0xa5522a75,0x23e6003f),LL(0x84afa749,0xc6ef3f1b),LL(0x146d10a3,0x9a723f75),LL(0x119106b0,0x5fa99480)}, {LL(0xc0013dba,0x01d500db),LL(0x548edbe0,0x10b30ada),LL(0xb04ffc6b,0xb2eb046e),LL(0x64f25ee2,0xa57088f3)}},
   {{LL(0x83a068a3,0xc0c919c3),LL(0xfbde282f,0x8139559d),LL(0x9fec9a99,0x4e2b5d13),LL(0xfbefa7e6,0x53bad712)}, {LL(0x2860bd4f,0xa6befe0d),LL(0x0011bd15,0x6ea0ae15),LL(0x2bce3779,0xc1ef3463),LL(0x5d742dbb,0xc09ecb30)}},
   {{LL(0xf73db19d,0x29526afd),LL(0x6a029a40,0x7c02c905),LL(0xde5a48ba,0xa778460f),LL(0x77c105f6,0xda05993e)}, {LL(0xc9ddece9,0xb6d599f9),LL(0x0cfc33ca,0x9f9df668),LL(0xa0aa67a8,0xdcd8ef4f),LL(0xc3f4d178,0x31277019)}},
   {{LL(0x53e86ae1,0x98e05abf),LL(0x3850830d,0xc1dc4d90),LL(0xe06bc33c,0xbd7fd806),LL(0xacf1286f,0x1ac330d6)}, {LL(0xe1588c1e,0x28ce2303),LL(0x1b7e9c19,0xdc25e54b),LL(0x4b7149f2,0x11e51e49),LL(0xb5c7fa25,0x551b8391)}},
   {{LL(0x1bf69873,0xa2fc251c),LL(0x2aec1574,0x099b7b53),LL(0x7c53296a,0x9ff98156),LL(0xa2dc60de,0xaf3f8d08)}, {LL(0x59b72d6c,0x18dd295c),LL(0xe75f44fc,0x165c9063),LL(0x427a0c55,0x9046ee7c),LL(0xc79ffdb3,0x317ea24d)}},
   {{LL(0x0ef0447d,0x6835a315),LL(0x8068e7c7,0xb2b9c786),LL(0x0e646af5,0xe6352714),LL(0x442baaa0,0xc5554a91)}, {LL(0x6d0ba1ea,0x671febc5),LL(0x0cf649ed,0x44f9ef7b),LL(0x0c1dac6b,0x4aa0cd61),LL(0x6e393e68,0x865f3c23)}},
   {{LL(0xa71dee29,0xf6886bcd),LL(0xda44ffae,0x934b0455),LL(0x016d6039,0xda7621c4),LL(0x3ad35493,0xf36c41bf)}, {LL(0xe5f6ab8d,0x9063135e),LL(0x47bdc0a8,0xb0e8eaba),LL(0x4c737cf3,0x62530616),LL(0x64f6b6cb,0x8046423e)}},
   {{LL(0x7958e3dc,0x11e50ad7),LL(0xb57234ab,0x4dab4e16),LL(0xe0916210,0x6ccfe2c6),LL(0x80f49d40,0x4d5dbc3b)}, {LL(0xef1b2b1b,0x2b8ff368),LL(0x752fea2a,0xf2afb326),LL(0x0246e36b,0xffa48ea7),LL(0x589b7444,0x3a4bae9b)}},
   {{LL(0x6ff3efcf,0x80ff984a),LL(0x56b77b47,0x7af53f30),LL(0x9320cae6,0x1f1c33b0),LL(0x26fc4ad4,0xce1f1c48)}, {LL(0xad350ee5,0x9cac662b),LL(0xe27a7dbd,0xf4c72fff),LL(0x703184e5,0xd766f986),LL(0x7c5b241e,0x36d3efd5)}},
   {{LL(0xf7ff5804,0xd4d6e358),LL(0x8f5e0bf6,0xa832b302),LL(0x453d9a22,0x4b3d73f7),LL(0xdf938705,0xb4dae072)}, {LL(0x92401620,0x6bff7b2e),LL(0x9bfa61cd,0x96b8494e),LL(0xb74dc1e5,0x4bcda341),LL(0xc19c393d,0x383fe3d2)}},
   {{LL(0x077e8821,0xa375fb70),LL(0xc17eb9bc,0xea35e04b),LL(0x7c4dd076,0x941d21ba),LL(0x3d0c3d8a,0x916c0a59)}, {LL(0x15b2cf47,0x2c1304e3),LL(0xd0c955c0,0x9233ebf3),LL(0x2b2fc935,0x77acdd07),LL(0xc04276bf,0xd71b6a7a)}},
   {{LL(0xd2ee8d68,0x789ea49b),LL(0x0a84a920,0x89552b46),LL(0x1a4ea456,0xe629d5de),LL(0x22ddd405,0xddfefe87)}, {LL(0x1cdb9e7b,0x3d56f697),LL(0xa8bf715b,0x95a861b0),LL(0x7896c993,0xb01248d6),LL(0x01a3085c,0x3c4e3d98)}},
   {{LL(0x9085b360,0x674939e1),LL(0xb589a287,0xae67dea9),LL(0x2bfdcfc9,0xc563856f),LL(0x313b685d,0x62fa9a80)}, {LL(0x7ad501d9,0x36ff33d9),LL(0x730ab349,0xf8bab4dd),LL(0xc46ba69d,0x18fd59f3),LL(0xe65278e9,0x81e08665)}},
   {{LL(0xeb8a1e84,0x5a5e803f),LL(0x47243604,0x5b4eef35),LL(0x393d6cdf,0x0ee71ee0),LL(0xc3a9c6db,0xde4d9dea)}, {LL(0x64466b53,0x0c14c376),LL(0x89e3b45e,0xc2ce9642),LL(0x54a2de21,0x6aa8012f),LL(0x29b6bc2c,0x519759c1)}},
   {{LL(0xe4667322,0x17768527),LL(0xac83b2ea,0x09fdfe4d),LL(0x04a0d5f5,0xd422125d),LL(0x2b86b310,0x02e8ff96)}, {LL(0xd7ee97af,0xf033628d),LL(0x7d72e0e6,0x778a846c),LL(0x882f63d6,0x06fde613),LL(0x8d434f14,0x9e258b0d)}},
   {{LL(0xccdcd600,0x5cdda529),LL(0x033c4535,0x37038b38),LL(0x391c1d7d,0xd6a1d639),LL(0x31d4ce6b,0x4f6489e4)}, {LL(0x5754e08c,0xd1b82f17),LL(0x75db7bd6,0x7df268ee),LL(0xad14dcfa,0x1e4a1202),LL(0xccfb9b77,0x7ab92ce2)}},
   {{LL(0x23aef997,0x61388e03),LL(0x06440ce3,0x9981f5bf),LL(0xc67d0edd,0x8d7631da),LL(0xc0a93516,0xc6ea593f)}, {LL(0xee841b38,0x064a06e0),LL(0x521ce83f,0x0d1d4f57),LL(0x70df2613,0xf7a0e0c3),LL(0x84c071ab,0x1506cccb)}},
   {{LL(0x328565e9,0x42a138ec),LL(0xb8130d16,0xe16b4578),LL(0x45ba251a,0x0628ff22),LL(0x210e22e8,0x016a84ca)}, {LL(0x94592d43,0x8ba14bb4),LL(0x785274a5,0xffee4308),LL(0x01354b75,0x01fc21ab),LL(0x7e424674,0xc37ce45f)}},
   {{LL(0xa7fe2735,0x71e153af),LL(0xc307721f,0x000fcee9),LL(0x805b56e3,0x3b189004),LL(0x7f504d9b,0x2f1435aa)}, {LL(0xa083bd72,0xd9aa1eba),LL(0x720ccf3d,0xf1145036),LL(0x4084fa32,0x95b29e27),LL(0x01f94f0c,0x8862d213)}},
   {{LL(0x1510a406,0x23fc5ddf),LL(0xc9f0e98d,0x475a78f4),LL(0xe72843a0,0xb6d681c4),LL(0x4a00c5a6,0xa90af2a4)}, {LL(0xa34f4412,0x95fc6d45),LL(0xe7f5d703,0x60f9c0e2),LL(0xad110925,0x2bc0642b),LL(0xbe24a4d5,0x79abfc10)}},
}
};
#endif /* _DISABLE_ECP_SM2_HARDCODED_BP_TBL_ */
#endif  /* _IPP_DATA */


IPP_OWN_DEFN (const cpPrecompAP*, gfpec_precom_sm2_fun, (void))
{
   static cpPrecompAP t = {
      /* w */                  7,
      /* select function */    p256r1_select_ap_w7,
      /* precomputed data */   (BNU_CHUNK_T*)precomputed_ec_sm2
   };
   return &t;
};
