﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

#nullable disable

using System;
using System.Linq;
using Microsoft.AspNetCore.Razor.Language.Intermediate;
using Xunit;
using static Microsoft.AspNetCore.Razor.Language.Intermediate.IntermediateNodeAssert;

namespace Microsoft.AspNetCore.Razor.Language;

public class DirectiveRemovalOptimizationPassTest
{
    [Fact]
    public void Execute_Custom_RemovesDirectiveNodeFromDocument()
    {
        // Arrange
        var content = "@custom \"Hello\"";
        var sourceDocument = TestRazorSourceDocument.Create(content);
        var codeDocument = RazorCodeDocument.Create(sourceDocument);
        var defaultEngine = RazorProjectEngine.Create(b =>
        {
            b.AddDirective(DirectiveDescriptor.CreateDirective("custom", DirectiveKind.SingleLine, d => d.AddStringToken()));
        }).Engine;
        var documentNode = Lower(codeDocument, defaultEngine);
        var pass = new DirectiveRemovalOptimizationPass()
        {
            Engine = defaultEngine,
        };

        // Act
        pass.Execute(codeDocument, documentNode);

        // Assert
        Children(documentNode,
            node => Assert.IsType<NamespaceDeclarationIntermediateNode>(node));
        var @namespace = documentNode.Children[0];
        Children(@namespace,
            node => Assert.IsType<ClassDeclarationIntermediateNode>(node));
        var @class = @namespace.Children[0];
        var method = SingleChild<MethodDeclarationIntermediateNode>(@class);
        Assert.Empty(method.Children);
    }

    [Fact]
    public void Execute_MultipleCustomDirectives_RemovesDirectiveNodesFromDocument()
    {
        // Arrange
        var content = """
            @custom "Hello"
            @custom "World"
            """;
        var sourceDocument = TestRazorSourceDocument.Create(content);
        var codeDocument = RazorCodeDocument.Create(sourceDocument);
        var defaultEngine = RazorProjectEngine.Create(b =>
        {
            b.AddDirective(DirectiveDescriptor.CreateDirective("custom", DirectiveKind.SingleLine, d => d.AddStringToken()));
        }).Engine;
        var documentNode = Lower(codeDocument, defaultEngine);
        var pass = new DirectiveRemovalOptimizationPass()
        {
            Engine = defaultEngine,
        };

        // Act
        pass.Execute(codeDocument, documentNode);

        // Assert
        Children(documentNode,
            node => Assert.IsType<NamespaceDeclarationIntermediateNode>(node));
        var @namespace = documentNode.Children[0];
        Children(@namespace,
            node => Assert.IsType<ClassDeclarationIntermediateNode>(node));
        var @class = @namespace.Children[0];
        var method = SingleChild<MethodDeclarationIntermediateNode>(@class);
        Assert.Empty(method.Children);
    }

    [Fact]
    public void Execute_DirectiveWithError_PreservesDiagnosticsAndRemovesDirectiveNodeFromDocument()
    {
        // Arrange
        var content = "@custom \"Hello\"";
        var expectedDiagnostic = RazorDiagnostic.Create(new RazorDiagnosticDescriptor("RZ9999", "Some diagnostic message.", RazorDiagnosticSeverity.Error));
        var sourceDocument = TestRazorSourceDocument.Create(content);
        var codeDocument = RazorCodeDocument.Create(sourceDocument);
        var defaultEngine = RazorProjectEngine.Create(b =>
        {
            b.AddDirective(DirectiveDescriptor.CreateDirective("custom", DirectiveKind.SingleLine, d => d.AddStringToken()));
        }).Engine;
        var documentNode = Lower(codeDocument, defaultEngine);

        // Add the diagnostic to the directive node.
        var directiveNode = documentNode.FindDescendantNodes<DirectiveIntermediateNode>().Single();
        directiveNode.Diagnostics.Add(expectedDiagnostic);

        var pass = new DirectiveRemovalOptimizationPass()
        {
            Engine = defaultEngine,
        };

        // Act
        pass.Execute(codeDocument, documentNode);

        // Assert
        var diagnostic = Assert.Single(documentNode.Diagnostics);
        Assert.Equal(expectedDiagnostic, diagnostic);

        Children(documentNode,
            node => Assert.IsType<NamespaceDeclarationIntermediateNode>(node));
        var @namespace = documentNode.Children[0];
        Children(@namespace,
            node => Assert.IsType<ClassDeclarationIntermediateNode>(node));
        var @class = @namespace.Children[0];
        var method = SingleChild<MethodDeclarationIntermediateNode>(@class);
        Assert.Empty(method.Children);
    }

    private static DocumentIntermediateNode Lower(RazorCodeDocument codeDocument, RazorEngine engine)
    {
        foreach (var phase in engine.Phases)
        {
            phase.Execute(codeDocument);

            if (phase is IRazorDirectiveClassifierPhase)
            {
                break;
            }
        }

        var documentNode = codeDocument.GetDocumentIntermediateNode();
        Assert.NotNull(documentNode);

        return documentNode;
    }
}
