/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "timeline.h"

TimeLine::TimeLine(int timelineType,
                   PumpController *pumpController,
                   QWidget *parent) :  QWidget(parent)
{
    this->timelineType = timelineType;
    this->pController = pumpController;

    // Simulated data for demo posts
    QVariantMap demoLocationData;
    demoLocationData.insert("displayName", "Demoville");

    QVariantMap demoAuthorData;
    demoAuthorData.insert("displayName", "Demo User");
    demoAuthorData.insert("id",          "demo@somepump.example");
    demoAuthorData.insert("location",    demoLocationData);
    demoAuthorData.insert("summary",     "I am not a real user");

    QVariantMap demoGeneratorData;
    demoGeneratorData.insert("displayName", "Dianara");

    QVariantMap demoObjectData;
    demoObjectData.insert("published",   "2013-05-01T00:00:00Z");


    QSettings settings; // FIXME: kinda tmp, until posts have "unread" status, etc.
    settings.beginGroup("TimelineStates");

    // Demo post content depends on timeline type
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        demoObjectData.insert("displayName", tr("Welcome to Dianara"));
        demoObjectData.insert("content",     tr("Dianara is a <b>pump.io</b> client.")
                                             + "<br>"

                                      + tr("If you don't have a Pump account yet, you can get one "
                                           "at the following address:")
                                      + "<br>"
                                        "<a href=\"http://pump.io/tryit.html\">http://pump.io/tryit.html</a>"
                                        "<br><br>"

                                      + tr("First, configure your account from the "
                                           "<b>Settings - Account</b> menu.")
                                      + "<br>"
                                      + tr("After the process is done, your profile "
                                           "and timelines should update automatically.")
                                      + "<br><br>"

                                      + tr("Take a moment to look around the menus and "
                                           "the Configuration window.")
                                      + "<br><br>"

                                      + tr("You can also set your profile data and picture from "
                                           "the <b>Settings - Edit Profile</b> menu.")
                                      + "<br><br>"

                                      + tr("There are tooltips everywhere, so if you hover "
                                           "a button or a text field with your mouse, you'll "
                                           "probably see some extra information.")
                                      + "<br><br>"

                                      + "<a href=\"http://jancoding.wordpress.com/dianara\">"
                                      + tr("Dianara's blog") + "</a><br><br>"

                                        "<a href=\"https://github.com/e14n/pump.io/wiki/FAQ\">"
                                      + tr("Frequently asked questions about pump.io")
                                      + "</a>"
                                      + "<br><br>");

        this->previousNewestPostId = settings.value("previousNewestPostIdMain", "").toString();
        break;


    case TimelineTypeDirect:
        demoObjectData.insert("displayName",  tr("Direct Messages Timeline"));
        demoObjectData.insert("content",      tr("Here, you'll see posts specifically directed to you.")
                                              + "<br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdDirect", "").toString();
        break;


    case TimelineTypeActivity:
        demoObjectData.insert("displayName", tr("Activity Timeline"));
        demoObjectData.insert("content",     tr("You'll see your own posts here.")
                                             + "<br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdActivity", "").toString();
        break;


    case TimelineTypeFavorites:
        demoObjectData.insert("displayName", tr("Favorites Timeline"));
        demoObjectData.insert("content",     tr("Posts and comments you've liked.")
                                             + "<br><br>");
        this->previousNewestPostId = settings.value("previousNewestPostIdFavorites", "").toString();
        break;



    default:
        demoObjectData.insert("content", "<h2>Empty timeline</h2>");

    }
    settings.endGroup();


    QVariantMap demoPostData;
    demoPostData.insert("actor",          demoAuthorData);
    demoPostData.insert("generator",      demoGeneratorData);
    demoPostData.insert("object",         demoObjectData);


    this->unreadPostsCount = 0;
    this->timelineOffset = 0;

    this->minMaxHeightForPosts = 400;

    firstPageButton = new QPushButton(QIcon::fromTheme("go-first"), tr("F&irst Page"));
    connect(firstPageButton, SIGNAL(clicked()),
            this, SLOT(goToFirstPage()));

    currentPageLabel = new QLabel();
    currentPageLabel->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
    currentPageLabel->setSizePolicy(QSizePolicy::Maximum,
                                    QSizePolicy::Maximum);

    previousPageButton = new QPushButton(QIcon::fromTheme("go-previous"), tr("&Previous Page"));
    connect(previousPageButton, SIGNAL(clicked()),
            this, SLOT(goToPreviousPage()));
    nextPageButton = new QPushButton(QIcon::fromTheme("go-next"), tr("&Next Page"));
    connect(nextPageButton, SIGNAL(clicked()),
            this, SLOT(goToNextPage()));


    ///// Layout
    mainLayout = new QVBoxLayout();

    postsLayout = new QVBoxLayout();
    postsLayout->setAlignment(Qt::AlignTop);
    mainLayout->addLayout(postsLayout, 1);

    mainLayout->addSpacing(2); // 2 pixel separation

    bottomLayout = new QHBoxLayout();
    bottomLayout->addWidget(firstPageButton);
    bottomLayout->addSpacing(8);
    bottomLayout->addWidget(currentPageLabel);
    bottomLayout->addSpacing(8);
    bottomLayout->addWidget(previousPageButton);
    bottomLayout->addWidget(nextPageButton);
    mainLayout->addLayout(bottomLayout, 0);

    this->setLayout(mainLayout);


    // Add the default "demo" post
    ASActivity *demoActivity = new ASActivity(demoPostData, this);
    Post *demoPost = new Post(pController, demoActivity, false, this);
    demoPost->setMinMaxHeight(this->minMaxHeightForPosts);

    postsInTimeline.append(demoPost);
    postsLayout->addWidget(demoPost);

    qDebug() << "TimeLine created";
}


/*
 * Destructor stores timeline states in the settings
 *
 */
TimeLine::~TimeLine()
{
    QSettings settings;
    settings.beginGroup("TimelineStates");

    switch (timelineType)
    {
    case TimelineTypeMain:
        settings.setValue("previousNewestPostIdMain", this->previousNewestPostId);
        break;

    case TimelineTypeDirect:
        settings.setValue("previousNewestPostIdDirect", this->previousNewestPostId);
        break;

    case TimelineTypeActivity:
        settings.setValue("previousNewestPostIdActivity", this->previousNewestPostId);
        break;

    case TimelineTypeFavorites:
        settings.setValue("previousNewestPostIdFavorites", this->previousNewestPostId);
        break;
    }
    settings.endGroup();


    qDebug() << "TimeLine destroyed";
}



/*
 * Remove all widgets (Post *) from the timeline
 *
 */
void TimeLine::clearTimeLineContents()
{
    foreach (Post *oldPost, postsInTimeline)
    {
        this->mainLayout->removeWidget(oldPost);
        delete oldPost;
    }
    this->postsInTimeline.clear();

    qApp->processEvents(); // So GUI gets updated
}



void TimeLine::requestTimelinePage()
{
    switch (this->timelineType)
    {
    case TimelineTypeMain:
        this->pController->getMainTimeline(this->timelineOffset);
        break;

    case TimelineTypeDirect:
        this->pController->getDirectTimeline(this->timelineOffset);
        break;

    case TimelineTypeActivity:
        this->pController->getActivityTimeline(this->timelineOffset);
        break;

    case TimelineTypeFavorites:
        this->pController->getFavoritesTimeline(this->timelineOffset);
        break;
    }

}


/*
 *  Update the label at the bottom of the page, indicating current "page"
 *
 */
void TimeLine::updateCurrentPageNumber()
{
    QString currentPageString;
    currentPageString = QString("%1").arg((this->timelineOffset / this->postsPerPage) + 1);

    this->currentPageLabel->setText(currentPageString);
}

void TimeLine::setMinMaxHeightForPosts(int newMinMaxHeightForPosts)
{
    this->minMaxHeightForPosts = newMinMaxHeightForPosts;
}


/*
 * Resize all posts in timeline
 *
 */
void TimeLine::resizePosts()
{
    foreach (Post *post, postsInTimeline)
    {
        post->setMinMaxHeight(this->minMaxHeightForPosts);

        // Force a Post() resizeEvent, which will call
        // setPostContents() and setPostHeight()
        post->resize(post->width() - 1,
                     post->height() - 1);
    }
}

void TimeLine::markPostsAsRead()
{
    foreach (Post *post, postsInTimeline)
    {
        post->setPostAsRead();
    }
}



/*********************************************************/
/*********************************************************/
/************************ SLOTS **************************/
/*********************************************************/
/*********************************************************/



void TimeLine::setTimeLineContents(QVariantList postList, int postsPerPage,
                                   QString previousLink, QString nextLink)
{
    qDebug() << "TimeLine::setTimeLineContents()";


    // Remove all previous posts in timeline
    qDebug() << "Removing previous posts from timeline";
    this->clearTimeLineContents();

    emit scrollToTop(); // ask mainWindow to scroll the QScrollArea containing the timeline to the top

    // Hide the widget while it reloads; helps performance a lot
    this->hide();

    this->postsPerPage = postsPerPage;


    int newPostCount = 0;
    bool allNewPostsCounted = false;
    QString newestPostId; // Here we'll store the postID for the first (newest) post in the timeline
                          // With it, we can know how many new posts (if any) we receive next time

    // Fill timeline with new contents
    foreach (QVariant singlePost, postList)
    {
        if (singlePost.type() == QVariant::Map)
        {
            this->previousPageLink = previousLink;   // Useless at the moment
            this->nextPageLink = nextLink;
            qDebug() << "Prev/Next links:";
            qDebug() << previousPageLink;
            qDebug() << nextPageLink;

            this->postIsNew = false;

            QVariantMap activityMap;

            // Since "favorites" is a collection of objects, not activities,
            // we need to put "Favorites" posts into fake activities
            if (this->timelineType != TimelineTypeFavorites)
            {
                // Data is already an activity
                activityMap = singlePost.toMap();
            }
            else
            {
                // Put object into the empty/fake VariantMap for the activity
                activityMap.insert("object", singlePost.toMap());
                activityMap.insert("actor",  singlePost.toMap().value("author").toMap());
                activityMap.insert("id",     singlePost.toMap().value("id").toString());
            }

            ASActivity *activity = new ASActivity(activityMap, this);
            QString postDeletedTime = activity->object()->getDeletedTime();


            if (newestPostId.isEmpty()) // only first time, for newest post
            {
                if (this->timelineOffset == 0)
                {
                    newestPostId = activity->getId();
                }
                else
                {
                    newestPostId = this->previousNewestPostId;
                    allNewPostsCounted = true;
                }
            }


            if (!allNewPostsCounted)
            {
                if (activity->getId() == this->previousNewestPostId)
                {
                    allNewPostsCounted = true;
                }
                else
                {
                    // If post is NOT deleted, and not ours, add it to the count
                    if (postDeletedTime.isEmpty()
                        && activity->getAuthorId() != pController->currentUserId())
                    {
                        ++newPostCount;

                        // Mark current post as new
                        postIsNew = true;
                    }
                }
            }



            if (postDeletedTime.isEmpty())  // if post was NOT deleted
            {
                Post *newPost = new Post(pController,
                                         activity,
                                         false,  // NOT standalone
                                         this);
                if (postIsNew)
                {
                    newPost->setPostAsNew();
                    connect(newPost, SIGNAL(postRead()),
                            this, SLOT(decreaseUnreadPostsCount()));
                }
                this->postsInTimeline.append(newPost);
                this->postsLayout->addWidget(newPost);

                connect(newPost, SIGNAL(postEditRequested(QString,QString,QString)),
                        this, SIGNAL(postEditRequested(QString,QString,QString)));
                connect(newPost, SIGNAL(commentingOnPost(QWidget*)),
                        this, SIGNAL(commentingOnPost(QWidget*)));
            }
            else
            {
                // If there's a "deleted" key, ignore this post
                qDebug() << "This post was deleted on" << postDeletedTime << " / Not adding.";

                delete activity;
            }
        }
        else  // singlePost.type() is not a QVariant::Map
        {
            qDebug() << "Expected a Map, got something else";
            qDebug() << postList;
        }

        qApp->processEvents(); // Avoid GUI freeze

    } // end foreach



    this->previousNewestPostId = newestPostId;
    this->unreadPostsCount = newPostCount;
    qDebug() << "New posts:" << newPostCount << "; Newest post ID:" << previousNewestPostId;

    this->updateCurrentPageNumber();
    this->resizePosts();

    emit timelineRendered(this->timelineType, unreadPostsCount);

    // Show timeline again, since everything is added and drawn
    this->show();

    qDebug() << "setTimeLineContents() /END";
}





/*
 * Add the full list of likes to a post
 *
 */
void TimeLine::setLikesInPost(QVariantList likesList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setLikesInPost()";

    QString originatingPostCleanUrl = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanUrl;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->likesURL() << "==" << originatingPostCleanUrl;

        if (post->likesURL() == originatingPostCleanUrl)
        {
            qDebug() << "Found originating Post; setting likes on it...";
            post->setLikes(likesList);

            break;
        }
    }
}


/*
 * Add the full list of comments to a post
 *
 */
void TimeLine::setCommentsInPost(QVariantList commentsList, QString originatingPostURL)
{
    qDebug() << "TimeLine::setCommentsInPost()";

    QString originatingPostCleanURL = originatingPostURL.split("?").at(0);
    qDebug() << "Originating post URL:" << originatingPostCleanURL;


    // Look for the originating Post() object
    qDebug() << "Looking for the originating Post() object";
    foreach (Post *post, postsInTimeline)
    {
        qDebug() << "Checking if" << post->commentsURL() << "==" << originatingPostCleanURL;

        if (post->commentsURL() == originatingPostCleanURL)
        {
            qDebug() << "Found originating Post; setting comments on it...";
            post->setComments(commentsList);

            // break;
            /* Don't break, so comments get set in copies of the post too,
               like if JohnDoe posted something and JaneDoe shared it soon
               after, so both the original post and its shared copy are visible
               in the timeline. */
        }
    }
}





void TimeLine::goToFirstPage()
{
    qDebug() << "TimeLine::goToFirstPage()";

    this->timelineOffset = 0;

    this->requestTimelinePage();
}



void TimeLine::goToPreviousPage()
{
    qDebug() << "TimeLine::goToPreviousPage()";

    this->timelineOffset -= this->postsPerPage;
    if (timelineOffset < 0)
    {
        timelineOffset = 0;
    }

    this->requestTimelinePage();
}



void TimeLine::goToNextPage()
{
    qDebug() << "TimeLine::goToNextPage()";

    this->timelineOffset += this->postsPerPage;

    this->requestTimelinePage();
}



/*
 * Decrease internal counter of unread posts (by 1), and inform
 * the parent window, so it can update its tab titles
 *
 */
void TimeLine::decreaseUnreadPostsCount()
{
    --unreadPostsCount;

    emit unreadPostsCountChanged(this->timelineType, this->unreadPostsCount);
}
