/**
   @author Shin'ichiro Nakaoka
*/

#include "LinkTreeWidget.h"
#include <cnoid/Link>
#include <cnoid/LinkGroup>
#include <cnoid/Archive>
#include <cnoid/MessageView>
#include <cnoid/MenuManager>
#include <QHeaderView>
#include <QCheckBox>
#include <QRadioButton>
#include <QBoxLayout>
#include <QEvent>
#include <QApplication>
#include <boost/bind.hpp>
#include <set>
#include <map>
#include <deque>
#include <iostream>
#include <cassert>
#include "gettext.h"

using namespace std;
using namespace boost;
using namespace cnoid;

namespace {
    const bool TRACE_FUNCTIONS = false;
}

namespace cnoid {

    class LinkTreeWidgetImpl
    {
    public:
        LinkTreeWidgetImpl(LinkTreeWidget* self);
        ~LinkTreeWidgetImpl();

        LinkTreeWidget* self;

        struct ColumnInfo {
            LinkTreeWidget::ColumnDataFunction dataFunction;
            LinkTreeWidget::ColumnSetDataFunction setDataFunction;
            LinkTreeWidget::ColumnWidgetFunction widgetFunction;
        };
        vector<ColumnInfo> columnInfos;

        QTreeWidgetItem* headerItem;
        int nameColumn;
        int jointIdColumn;
        int rowIndexCounter;
        int itemWidgetWidthAdjustment;
        vector<LinkTreeItem*> customRows;
        LinkTreeWidget::ListingMode listingMode;
        ComboBox listingModeCombo;
        Menu popupMenu; // This must be defined before popupMenuManager
        MenuManager popupMenuManager;

        boost::signal<void(bool isInitialCreation)> sigUpdateRequest;
        boost::signal<void(LinkTreeItem* item, int column)> sigItemChanged;
        boost::signal<void()> sigSelectionChanged;

        int defaultExpansionLevel;
        bool isCacheEnabled;
        bool isArchiveOfCurrentBodyItemEnabled;

        class BodyItemInfo : public Referenced {
        public:
            bool isRestoringTreeStateNeeded;

            dynamic_bitset<> selection;
            vector<int> selectedLinkIndices;
            boost::signal<void()> sigSelectionChanged;

            bool needTreeExpansionUpdate;
            dynamic_bitset<> linkExpansions;
            std::set<string> expandedParts;
                
            signals::connection detachedFromRootConnection;

            BodyItemInfo() {
                isRestoringTreeStateNeeded = true;
            }
            ~BodyItemInfo() {
                detachedFromRootConnection.disconnect();
            }
            void setNumLinks(int n, bool doInitialize) {
                if(doInitialize){
                    selection.reset();
                    linkExpansions.set();
                    expandedParts.clear();
                }
                selection.resize(n, false);
                linkExpansions.resize(n, true);
            }
        };
        typedef intrusive_ptr<BodyItemInfo> BodyItemInfoPtr;

        typedef map<BodyItemPtr, BodyItemInfoPtr> BodyItemInfoMap;
        BodyItemInfoMap bodyItemInfoCache;

        vector<LinkTreeItem*> linkIndexToItemMap;

        BodyItemPtr currentBodyItem;
        BodyItemInfoPtr currentBodyItemInfo;

        boost::signal<void()> dummySigSelectionChanged; // never emitted
        vector<int> emptyLinkIndices;
        dynamic_bitset<> emptySelection;

        void initialize();
        void enableCache(bool on);
        BodyItemInfoPtr getBodyItemInfo(BodyItemPtr bodyItem);
        void onBodyItemDetachedFromRoot(BodyItem* bodyItem);
        void clearTreeItems();
        void setCurrentBodyItem(BodyItemPtr bodyItem, bool forceTreeUpdate);
        void restoreTreeState();
        void restoreSubTreeState(QTreeWidgetItem* item);
        void restoreTreeStateSub(QTreeWidgetItem* parentItem);
        void addChild(LinkTreeItem* parentItem, LinkTreeItem* item);
        void addChild(LinkTreeItem* item);
        void setLinkTree(Link* link, bool onlyJoints);
        void setLinkTreeSub(Link* link, Link* parentLink, LinkTreeItem* parentItem, bool onlyJoints);
        void setJointList(const BodyPtr& body);
        void setLinkList(const BodyPtr& body);
        void setLinkGroupTree(const BodyItemPtr bodyItem);
        void setLinkGroupTreeSub(LinkTreeItem* parentItem, const LinkGroupPtr& linkGroup, const BodyPtr& body);
        void addCustomRows();
        void onListingModeChanged(int index);
        void onSelectionChanged();
        boost::signal<void()>& sigSelectionChangedOf(BodyItemPtr bodyItem);
        const std::vector<int>& getSelectedLinkIndices(BodyItemPtr bodyItem);
        const boost::dynamic_bitset<>& getLinkSelection(BodyItemPtr bodyItem);
        void onCustomContextMenuRequested(const QPoint& pos);
        void setExpansionState(const LinkTreeItem* item, bool on);
        void onItemExpanded(QTreeWidgetItem* treeWidgetItem);
        void onItemCollapsed(QTreeWidgetItem* treeWidgetItem);
        bool makeSingleSelection(BodyItemPtr& bodyItem, int linkIndex);
        bool storeState(Archive& archive);
        bool restoreState(const Archive& archive);
    };
}


namespace {

    void jointIdData(const LinkTreeItem* item, int role, QVariant& out_value)
    {
        const Link* link = item->link();
        if(role == Qt::DisplayRole && link && (link->jointId >= 0)){
            out_value = link->jointId;
        } else if(role == Qt::TextAlignmentRole){
            out_value = Qt::AlignHCenter;
        }
        
    }

    void nameData(const LinkTreeItem* item, int role, QVariant& out_value)
    {
        if(role == Qt::DisplayRole){
            out_value = item->name().c_str();
        }
    }
}


LinkTreeItem::LinkTreeItem(const std::string& name)
    : name_(name)
{
    rowIndex_ = -1;
    link_ = 0;
    isLinkGroup_ = true;
}


LinkTreeItem::LinkTreeItem(Link* link, LinkTreeWidgetImpl* treeImpl)
    : name_(link->name())
{
    rowIndex_ = -1;
    link_ = link;
    isLinkGroup_ = false;
    treeImpl->linkIndexToItemMap[link->index] = this;
}


LinkTreeItem::LinkTreeItem(LinkGroup* linkGroup, LinkTreeWidgetImpl* treeImpl)
    : name_(linkGroup->name())
{
    rowIndex_ = -1;
    link_ = 0;
    isLinkGroup_ = true;
}


QVariant LinkTreeItem::data(int column, int role) const
{
    QVariant value;
    LinkTreeWidget* tree = static_cast<LinkTreeWidget*>(treeWidget());
    LinkTreeWidget::ColumnDataFunction& func = tree->impl->columnInfos[column].dataFunction;
    if(!func.empty()){
        func(this, role, value);
    }
    if(value.isValid()){
        return value;
    }
    return QTreeWidgetItem::data(column, role);
}


void LinkTreeItem::setData(int column, int role, const QVariant& value)
{
    LinkTreeWidget* tree = static_cast<LinkTreeWidget*>(treeWidget());
    LinkTreeWidget::ColumnSetDataFunction& func = tree->impl->columnInfos[column].setDataFunction;
    if(!func.empty()){
        func(this, role, value);
    }
    return QTreeWidgetItem::setData(column, role, value);
}


LinkTreeWidget::LinkTreeWidget(QWidget* parent)
    : TreeWidget(parent)
{
    impl = new LinkTreeWidgetImpl(this);
    impl->initialize();
}


LinkTreeWidgetImpl::LinkTreeWidgetImpl(LinkTreeWidget* self)
    : self(self),
      popupMenuManager(&popupMenu)
{

}


void LinkTreeWidgetImpl::initialize()
{
    rowIndexCounter = 0;
    defaultExpansionLevel = std::numeric_limits<int>::max();
    isCacheEnabled = false;
    isArchiveOfCurrentBodyItemEnabled = false;
    itemWidgetWidthAdjustment = 0;
    
    headerItem = new QTreeWidgetItem;
    QHeaderView* header = self->header();
    header->setStretchLastSection(false);
    QObject::connect(header, SIGNAL(sectionResized(int, int, int)),
                     self, SLOT(onHeaderSectionResized()));
    self->setHeaderItem(headerItem);
    self->setSelectionMode(QAbstractItemView::ExtendedSelection);
    self->setIndentation(12);
    self->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    self->setVerticalScrollMode(QAbstractItemView::ScrollPerPixel);
    self->setHorizontalScrollMode(QAbstractItemView::ScrollPerPixel);

    nameColumn = self->addColumn(_("link"));
    header->setResizeMode(nameColumn, QHeaderView::Stretch);
    self->setColumnDataFunction(nameColumn, &nameData);

    jointIdColumn = self->addColumn(_("id"));
    self->setColumnDataFunction(jointIdColumn, &jointIdData);
    header->setResizeMode(jointIdColumn, QHeaderView::ResizeToContents);
    self->moveVisualColumnIndex(jointIdColumn, 0);

    QObject::connect(self, SIGNAL(itemChanged(QTreeWidgetItem*, int)),
                     self, SLOT(onItemChanged(QTreeWidgetItem*, int)));

    QObject::connect(self, SIGNAL(itemExpanded(QTreeWidgetItem*)), self,
                     SLOT(onItemExpanded(QTreeWidgetItem*)));

    QObject::connect(self, SIGNAL(itemCollapsed(QTreeWidgetItem*)),
                     self, SLOT(onItemCollapsed(QTreeWidgetItem*)));

    QObject::connect(self, SIGNAL(itemSelectionChanged()),
                     self, SLOT(onSelectionChanged()));

    // The order of the following labes must correspond to that of ListingMode
    listingModeCombo.enableI18n(CNOID_GETTEXT_DOMAIN_NAME);
    listingModeCombo.addI18nItem(N_("link list"));
    listingModeCombo.addI18nItem(N_("link tree"));
    listingModeCombo.addI18nItem(N_("joint list"));
    listingModeCombo.addI18nItem(N_("joint tree"));
    listingModeCombo.addI18nItem(N_("part tree"));

    listingMode = LinkTreeWidget::LINK_LIST;
    listingModeCombo.setCurrentIndex(listingMode);
    listingModeCombo.sigCurrentIndexChanged().connect(
        bind(&LinkTreeWidgetImpl::onListingModeChanged, this, _1));
}


LinkTreeWidget::~LinkTreeWidget()
{
    delete impl;
}


LinkTreeWidgetImpl::~LinkTreeWidgetImpl()
{
    for(size_t i=0; i < customRows.size(); ++i){
        delete customRows[i];
    }
}


void LinkTreeWidget::setDefaultExpansionLevel(int level)
{
    impl->defaultExpansionLevel = level;
}


void LinkTreeWidget::enableCache(bool on)
{
    impl->enableCache(on);
}


void LinkTreeWidgetImpl::enableCache(bool on)
{
    isCacheEnabled = on;

    if(!isCacheEnabled){
        bodyItemInfoCache.clear();
    }
}


int LinkTreeWidget::nameColumn()
{
    return impl->nameColumn;
}


int LinkTreeWidget::jointIdColumn()
{
    return impl->jointIdColumn;
}


int LinkTreeWidget::addColumn()
{
    int column = impl->columnInfos.size();
    impl->columnInfos.push_back(LinkTreeWidgetImpl::ColumnInfo());
    setColumnCount(impl->columnInfos.size());
    return column;
}


int LinkTreeWidget::addColumn(const QString& headerText)
{
    int column = addColumn();
    impl->headerItem->setText(column, headerText);
    return column;
}

                    
void LinkTreeWidget::moveVisualColumnIndex(int column, int visualIndex)
{
    header()->moveSection(header()->visualIndex(column), visualIndex);
}


void LinkTreeWidget::setColumnDataFunction(int column, ColumnDataFunction func)
{
    impl->columnInfos[column].dataFunction = func;
}


void LinkTreeWidget::setColumnSetDataFunction(int column, ColumnSetDataFunction func)
{
    impl->columnInfos[column].setDataFunction = func;
}


void LinkTreeWidget::setColumnWidgetFunction(int column, ColumnWidgetFunction func)
{
    impl->columnInfos[column].widgetFunction = func;
}


void LinkTreeWidget::changeEvent(QEvent* event)
{
    QTreeWidget::changeEvent(event);
    if(event->type() == QEvent::StyleChange){
        //impl->setCurrentBodyItem(impl->currentBodyItem, true);
    }
}


void LinkTreeWidget::setAlignedItemWidget
(LinkTreeItem* item, int column, QWidget* widget, Qt::Alignment alignment)
{
    QHBoxLayout* box = new QHBoxLayout();
    box->setContentsMargins(0, 0, 0, 0);
    if(impl->itemWidgetWidthAdjustment > 0){
        widget->setMinimumWidth(widget->sizeHint().width() + impl->itemWidgetWidthAdjustment);
    }
    box->addWidget(widget, 0, alignment);
    QWidget* base = new QWidget();
    base->setLayout(box);
    setItemWidget(item, column, base);
}


QWidget* LinkTreeWidget::alignedItemWidget(LinkTreeItem* item, int column)
{
    QWidget* base = itemWidget(item, column);
    if(base){
        QLayout* layout = base->layout();
        if(layout){
            QLayoutItem* layoutItem = layout->itemAt(0);
            if(layoutItem){
                QWidget* widget = layoutItem->widget();
                if(widget){
                    return widget;
                }
            }
        }
    }
    return 0;
}


void LinkTreeWidget::addCustomRow(LinkTreeItem* treeItem)
{
    impl->customRows.push_back(treeItem);
}


int LinkTreeWidget::numLinkTreeItems()
{
    return impl->rowIndexCounter;
}


SignalProxy< boost::signal<void(bool isInitialCreation)> > LinkTreeWidget::sigUpdateRequest()
{
    return impl->sigUpdateRequest;
}


ComboBox* LinkTreeWidget::listingModeCombo()
{
    return &impl->listingModeCombo;
}


void LinkTreeWidget::setListingMode(ListingMode mode)
{
    impl->listingMode = mode;
    impl->listingModeCombo.setCurrentIndex(mode);
}


void LinkTreeWidget::fixListingMode(bool on)
{
    impl->listingModeCombo.setEnabled(on);
}


LinkTreeWidgetImpl::BodyItemInfoPtr LinkTreeWidgetImpl::getBodyItemInfo(BodyItemPtr bodyItem)
{
    BodyItemInfoPtr info;

    if(bodyItem){
        if(bodyItem == currentBodyItem){
            info = currentBodyItemInfo;
        } else if(isCacheEnabled){
            BodyItemInfoMap::iterator p = bodyItemInfoCache.find(bodyItem);
            if(p != bodyItemInfoCache.end()){
                info = p->second;
            }
        }

        if(!info && bodyItem->findRootItem()){
            if(!isCacheEnabled){
                bodyItemInfoCache.clear();
            }
            info = new BodyItemInfo();
            info->detachedFromRootConnection = bodyItem->sigDetachedFromRoot().connect(
                bind(&LinkTreeWidgetImpl::onBodyItemDetachedFromRoot, this, bodyItem.get()));
            bodyItemInfoCache[bodyItem] = info;
        }

        if(info){
            info->setNumLinks(bodyItem->body()->numLinks(), false);
        }
    }

    return info;
}


void LinkTreeWidgetImpl::onBodyItemDetachedFromRoot(BodyItem* bodyItem)
{
    if(currentBodyItem.get() == bodyItem){
        setCurrentBodyItem(0, false);
    }

    BodyItemInfoMap::iterator p = bodyItemInfoCache.find(bodyItem);
    if(p != bodyItemInfoCache.end()){
        p->second->detachedFromRootConnection.disconnect();
        bodyItemInfoCache.erase(p);
    }
}

void LinkTreeWidget::setBodyItem(BodyItemPtr bodyItem)
{
    impl->setCurrentBodyItem(bodyItem, false);
}


BodyItem* LinkTreeWidget::bodyItem()
{
    return impl->currentBodyItem.get();
}


LinkTreeItem* LinkTreeWidget::itemOfLink(int linkIndex)
{
    if(linkIndex < (int)impl->linkIndexToItemMap.size()){
        return impl->linkIndexToItemMap[linkIndex];
    }
    return 0;
}


void LinkTreeWidgetImpl::clearTreeItems()
{
    // Take custom row items before calling clear() to prevent the items from being deleted.
    for(size_t i=0; i < customRows.size(); ++i){
        LinkTreeItem* item = customRows[i];
        if(item->treeWidget()){
            self->takeTopLevelItem(self->indexOfTopLevelItem(item));
        }
    }
    self->clear();
}


void LinkTreeWidgetImpl::setCurrentBodyItem(BodyItemPtr bodyItem, bool forceTreeUpdate)
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::setCurrentBodyItem()" << endl;
    }
    
    if(forceTreeUpdate || bodyItem != currentBodyItem){

        self->blockSignals(true);

        clearTreeItems();
        rowIndexCounter = 0;
        linkIndexToItemMap.clear();

        if(QApplication::style()->objectName() == "cleanlooks"){
            itemWidgetWidthAdjustment = 2;
        } else {
            itemWidgetWidthAdjustment = 0;
        }
        
        self->blockSignals(false);

        currentBodyItemInfo = getBodyItemInfo(bodyItem);
        currentBodyItem = bodyItem;

        if(bodyItem){

            BodyPtr body = bodyItem->body();
            linkIndexToItemMap.resize(body->numLinks(), 0);

            switch(listingMode){
            case LinkTreeWidget::LINK_LIST:
                self->setRootIsDecorated(false);
                setLinkList(body);
                break;
            case LinkTreeWidget::LINK_TREE:
                self->setRootIsDecorated(true);
                setLinkTree(body->rootLink(), false);
                break;
            case LinkTreeWidget::JOINT_LIST:
                self->setRootIsDecorated(false);
                setJointList(body);
                break;
            case LinkTreeWidget::JOINT_TREE:
                self->setRootIsDecorated(true);
                setLinkTree(body->rootLink(), true);
                break;
            case LinkTreeWidget::PART_TREE:
                self->setRootIsDecorated(true);
                setLinkGroupTree(bodyItem);
                break;
            default:
                break;
            }

            addCustomRows();

            sigUpdateRequest(true);
        }
    }
}


void LinkTreeWidgetImpl::restoreTreeState()
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::restoreTreeState()" << endl;
    }
    
    if(currentBodyItemInfo){
        restoreSubTreeState(self->invisibleRootItem());
        currentBodyItemInfo->isRestoringTreeStateNeeded = false;
    }
}


void LinkTreeWidgetImpl::restoreSubTreeState(QTreeWidgetItem* item)
{
    self->blockSignals(true);
    restoreTreeStateSub(item);
    self->blockSignals(false);
}


void LinkTreeWidgetImpl::restoreTreeStateSub(QTreeWidgetItem* parentItem)
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::restoreTreeStateSub()" << endl;
    }

    int n = parentItem->childCount();
    for(int i=0; i < n; ++i){
        LinkTreeItem* item = dynamic_cast<LinkTreeItem*>(parentItem->child(i));
        if(item){
            const Link* link = item->link();
            if(link){
                const dynamic_bitset<>& selection = currentBodyItemInfo->selection;
                item->setSelected(selection[link->index]);
            }
        
            if(item->childCount() > 0){ // Tree
            
                bool expanded = item->isExpanded();
        
                if(listingMode == LinkTreeWidget::PART_TREE){
                    if(!link){
                        const std::set<string>& parts = currentBodyItemInfo->expandedParts;                        
                        expanded = (parts.find(item->name()) != parts.end());
                        item->setExpanded(expanded);
                    }
                } else if(link){
                    expanded = currentBodyItemInfo->linkExpansions[link->index];
                    item->setExpanded(expanded);
                }
                if(expanded){
                    restoreTreeStateSub(item);
                }
            }
        }
    }
}


void LinkTreeWidgetImpl::addChild(LinkTreeItem* parentItem, LinkTreeItem* item)
{
    if(parentItem){
        parentItem->addChild(item);
    } else {
        self->addTopLevelItem(item);
    }
    item->rowIndex_ = rowIndexCounter++;

    for(size_t col=0; col < columnInfos.size(); ++col){
        LinkTreeWidget::ColumnWidgetFunction& func = columnInfos[col].widgetFunction;
        if(!func.empty()){
            QWidget* widget = func(item);
            if(widget){
                self->setItemWidget(item, col, widget);
            }
        }
    }
}


void LinkTreeWidgetImpl::addChild(LinkTreeItem* item)
{
    addChild(0, item);
}


void LinkTreeWidgetImpl::setLinkTree(Link* link, bool onlyJoints)
{
    self->blockSignals(true);
    setLinkTreeSub(link, 0, 0, onlyJoints);
    self->blockSignals(false);
}


void LinkTreeWidgetImpl::setLinkTreeSub(Link* link, Link* parentLink, LinkTreeItem* parentItem, bool onlyJoints)
{
    LinkTreeItem* item = 0;

    if(onlyJoints && link->jointId < 0){
        item = parentItem;
    } else {
        item = new LinkTreeItem(link, this);
        addChild(parentItem, item);
        item->setExpanded(true);
    }

    if(link->child){
        for(Link* child = link->child; child; child = child->sibling){
            setLinkTreeSub(child, link, item, onlyJoints);
        }
    }
}


void LinkTreeWidgetImpl::setJointList(const BodyPtr& body)
{
    for(int i=0; i < body->numJoints(); ++i){
        Link* joint = body->joint(i);
        if(joint->isValid()){
            addChild(new LinkTreeItem(joint, this));
        }
    }
}


void LinkTreeWidgetImpl::setLinkList(const BodyPtr& body)
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::setLinkList(" << body->name() << ")" << endl;
    }
    
    for(int i=0; i < body->numLinks(); ++i){
        addChild(new LinkTreeItem(body->link(i), this));
    }
}


void LinkTreeWidgetImpl::setLinkGroupTree(const BodyItemPtr bodyItem)
{
    BodyPtr body = bodyItem->body();
    LinkGroupPtr linkGroup = body->linkGroup();
    if(linkGroup){
        self->blockSignals(true);
        setLinkGroupTreeSub(0, linkGroup, body);
        self->blockSignals(false);
    }
}


void LinkTreeWidgetImpl::setLinkGroupTreeSub(LinkTreeItem* parentItem, const LinkGroupPtr& linkGroup, const BodyPtr& body)
{
    LinkTreeItem* item = new LinkTreeItem(linkGroup.get(), this);
    addChild(parentItem, item);

    int numChildGroups = 0;
    int n = linkGroup->numElements();
    for(int i=0; i < n; ++i){
        if(linkGroup->isSubGroup(i)){
            setLinkGroupTreeSub(item, linkGroup->subGroup(i), body);
            ++numChildGroups;
        } else if(linkGroup->isLinkIndex(i)){
            Link* link = body->link(linkGroup->linkIndex(i));
            if(link){
                addChild(item, new LinkTreeItem(link, this));
            }
        }
    }

    item->setExpanded(numChildGroups > 0);
}


void LinkTreeWidgetImpl::addCustomRows()
{
    for(size_t i=0; i < customRows.size(); ++i){
        addChild(customRows[i]);
    }
}


void LinkTreeWidgetImpl::onListingModeChanged(int index)
{
    LinkTreeWidget::ListingMode newListingMode = static_cast<LinkTreeWidget::ListingMode>(index);
    if(newListingMode != listingMode){
        listingMode = newListingMode;
        if(currentBodyItem){
            setCurrentBodyItem(currentBodyItem, true);
        }
    }
}


void LinkTreeWidget::onSelectionChanged()
{
    impl->onSelectionChanged();
}


void LinkTreeWidgetImpl::onSelectionChanged()
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::onSelectionChanged()" << endl;
    }

    if(currentBodyItem){
        currentBodyItemInfo->selection.reset();

        QList<QTreeWidgetItem*> selected = self->selectedItems();
        for(int i=0; i < selected.size(); ++i){
            LinkTreeItem* item = dynamic_cast<LinkTreeItem*>(selected[i]);
            if(item && item->link()){
                currentBodyItemInfo->selection[item->link()->index] = true;
            }
        }
        currentBodyItemInfo->sigSelectionChanged();
        sigSelectionChanged();
    }
}


SignalProxy< boost::signal<void(LinkTreeItem* item, int column)> > LinkTreeWidget::sigItemChanged()
{
    return impl->sigItemChanged;
}


void LinkTreeWidget::onItemChanged(QTreeWidgetItem* item, int column)
{
    LinkTreeItem* linkTreeItem = dynamic_cast<LinkTreeItem*>(item);
    if(linkTreeItem){
        impl->sigItemChanged(linkTreeItem, column);
    }
}


SignalProxy< boost::signal<void()> > LinkTreeWidget::sigSelectionChanged()
{
    return impl->sigSelectionChanged;
}


SignalProxy< boost::signal<void()> > LinkTreeWidget::sigSelectionChanged(BodyItemPtr bodyItem)
{
    return impl->sigSelectionChangedOf(bodyItem);
}


boost::signal<void()>& LinkTreeWidgetImpl::sigSelectionChangedOf(BodyItemPtr bodyItem)
{
    BodyItemInfoPtr info = getBodyItemInfo(bodyItem);
    if(info){
        return info->sigSelectionChanged;
    }
    return dummySigSelectionChanged;
}


const std::vector<int>& LinkTreeWidget::getSelectedLinkIndices()
{
    return impl->getSelectedLinkIndices(impl->currentBodyItem);
}


const std::vector<int>& LinkTreeWidget::getSelectedLinkIndices(BodyItemPtr bodyItem)
{
    return impl->getSelectedLinkIndices(bodyItem);
}


const std::vector<int>& LinkTreeWidgetImpl::getSelectedLinkIndices(BodyItemPtr bodyItem)
{
    BodyItemInfoPtr info = getBodyItemInfo(bodyItem);
    if(info){
        info->selectedLinkIndices.clear();
        const dynamic_bitset<>& selection = info->selection;
        for(size_t i=0; i < selection.size(); ++i){
            if(selection[i]){
                info->selectedLinkIndices.push_back(i);
            }
        }
        return info->selectedLinkIndices;
    }
    return emptyLinkIndices;
}


const boost::dynamic_bitset<>& LinkTreeWidget::getLinkSelection()
{
    return impl->getLinkSelection(impl->currentBodyItem);
}


const boost::dynamic_bitset<>& LinkTreeWidget::getLinkSelection(BodyItemPtr bodyItem)
{
    return impl->getLinkSelection(bodyItem);
}


const boost::dynamic_bitset<>& LinkTreeWidgetImpl::getLinkSelection(BodyItemPtr bodyItem)
{
    BodyItemInfoPtr info = getBodyItemInfo(bodyItem);
    if(info){
        return info->selection;
    }
    return emptySelection;
}


void LinkTreeWidget::onCustomContextMenuRequested(const QPoint& pos)
{
    impl->onCustomContextMenuRequested(pos);
}


void LinkTreeWidgetImpl::onCustomContextMenuRequested(const QPoint& pos)
{
    popupMenu.popup(pos);
}


void LinkTreeWidgetImpl::setExpansionState(const LinkTreeItem* item, bool on)
{
    if(listingMode == LinkTreeWidget::LINK_TREE || listingMode == LinkTreeWidget::JOINT_TREE){
        if(item->link()){
            currentBodyItemInfo->linkExpansions[item->link()->index] = on;
        }

    } else if(listingMode == LinkTreeWidget::PART_TREE){
        if(on){
            currentBodyItemInfo->expandedParts.insert(item->name());
        } else {
            currentBodyItemInfo->expandedParts.erase(item->name());
        }
    }
}


void LinkTreeWidget::onItemExpanded(QTreeWidgetItem* treeWidgetItem)
{
    impl->onItemExpanded(treeWidgetItem);
}


void LinkTreeWidgetImpl::onItemExpanded(QTreeWidgetItem* treeWidgetItem)
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::onItemExpanded()" << endl;
    }
    LinkTreeItem* item = dynamic_cast<LinkTreeItem*>(treeWidgetItem);
    if(item){
        setExpansionState(item, true);
        restoreSubTreeState(item);
    }
}


void LinkTreeWidget::onItemCollapsed(QTreeWidgetItem* treeWidgetItem)
{
    impl->onItemCollapsed(treeWidgetItem);
}


void LinkTreeWidgetImpl::onItemCollapsed(QTreeWidgetItem* treeWidgetItem)
{
    if(TRACE_FUNCTIONS){
        cout << "LinkTreeWidgetImpl::onItemCollapsed()" << endl;
    }
    LinkTreeItem* item = dynamic_cast<LinkTreeItem*>(treeWidgetItem);
    if(item){
        setExpansionState(item, false);
    }
}


void LinkTreeWidget::onHeaderSectionResized()
{
    updateGeometry();
}


void LinkTreeWidget::enableArchiveOfCurrentBodyItem(bool on)
{
    impl->isArchiveOfCurrentBodyItemEnabled = on;
}


bool LinkTreeWidget::makeSingleSelection(BodyItemPtr bodyItem, int linkIndex)
{
    return impl->makeSingleSelection(bodyItem, linkIndex);
}


bool LinkTreeWidgetImpl::makeSingleSelection(BodyItemPtr& bodyItem, int linkIndex)
{
    BodyItemInfoPtr info = getBodyItemInfo(bodyItem);

    if(!info){
        return false;
    }
    
    if(static_cast<size_t>(linkIndex) < info->selection.size()){
        if(!info->selection[linkIndex] || info->selection.count() > 1){
            info->selection.reset();
            info->selection.set(linkIndex);
            
            if(bodyItem == currentBodyItem){
                restoreTreeState();

                LinkTreeItem* item = linkIndexToItemMap[linkIndex];
                if(item){
                    self->scrollToItem(item);
                }
                
                currentBodyItemInfo->sigSelectionChanged();
                sigSelectionChanged();
            } else {
                info->sigSelectionChanged();
            }
        }
    }
    return true;
}


MenuManager& LinkTreeWidget::popupMenuManager()
{
    return impl->popupMenuManager;
}


bool LinkTreeWidget::storeState(Archive& archive)
{
    return impl->storeState(archive);
}


bool LinkTreeWidgetImpl::storeState(Archive& archive)
{
    if(listingModeCombo.isEnabled()){
        archive.write("listingMode", listingModeCombo.currentOrgText().toStdString(), YAML_DOUBLE_QUOTED);
    }

    if(isArchiveOfCurrentBodyItemEnabled){
        archive.writeItemId("currentBodyItem", currentBodyItem);
    }

    if(isCacheEnabled && !bodyItemInfoCache.empty()){

        YamlSequencePtr bodyItemNodes = new YamlSequence();

        for(BodyItemInfoMap::iterator it = bodyItemInfoCache.begin(); it != bodyItemInfoCache.end(); ++it){

            BodyItemPtr bodyItem = it->first;
            BodyItemInfo& info = *it->second;
            YamlMappingPtr bodyItemNode = new YamlMapping();
            bool isEmpty = true;

            bodyItemNode->write("id", archive.getItemId(bodyItem));
            
            const vector<int>& indices = getSelectedLinkIndices(bodyItem);
            if(!indices.empty()){
                YamlSequence& selected = *bodyItemNode->createFlowStyleSequence("selectedLinks");
                int n = indices.size();
                for(int i=0; i < n; ++i){
                    selected.append(indices[i], 20, n);
                }
                isEmpty = false;
            }

            int n = info.linkExpansions.size();
            int m = n - info.linkExpansions.count();
            if(m > 0){
                YamlSequence& nonExpanded = *bodyItemNode->createFlowStyleSequence("nonExpandedLinks");
                for(int i=0; i < n; ++i){
                    if(!info.linkExpansions[i]){
                        nonExpanded.append(i, 20, m);
                    }
                }
                isEmpty = false;
            }

            n = info.expandedParts.size();
            if(n > 0){
                YamlSequence& expanded = *bodyItemNode->createFlowStyleSequence("expandedParts");
                for(std::set<string>::iterator p = info.expandedParts.begin(); p != info.expandedParts.end(); ++p){
                    expanded.append(*p, 10, n, YAML_DOUBLE_QUOTED);
                }
                isEmpty = false;
            }

            if(!isEmpty){
                bodyItemNodes->append(bodyItemNode);
            }
        }

        if(!bodyItemNodes->empty()){
            archive.insert("bodyItems", bodyItemNodes);
        }
    }

    return true;
}


bool LinkTreeWidget::restoreState(const Archive& archive)
{
    return impl->restoreState(archive);
}


bool LinkTreeWidgetImpl::restoreState(const Archive& archive)
{
    if(isCacheEnabled){
        const YamlSequence& nodes = *archive.findSequence("bodyItems");
        if(nodes.isValid() && !nodes.empty()){
            for(int i=0; i < nodes.size(); ++i){
                const YamlMapping& node = *nodes[i].toMapping();
                int id = node.get("id", -1);
                if(id >= 0){
                    BodyItem* bodyItem = archive.findItem<BodyItem>(id);
                    if(bodyItem){
                        int numLinks = bodyItem->body()->numLinks();
                        BodyItemInfoPtr info = getBodyItemInfo(bodyItem);
                        const YamlSequence& selected = *node.findSequence("selectedLinks");
                        if(selected.isValid()){
                            info->setNumLinks(numLinks, true);
                            for(int i=0; i < selected.size(); ++i){
                                int index = selected[i].toInt();
                                if(index < numLinks){
                                    info->selection[index] = true;
                                }
                            }
                        }
                        const YamlSequence& expanded = *node.findSequence("expandedParts");
                        for(int i=0; i < expanded.size(); ++i){
                                info->expandedParts.insert(expanded[i]);
                        }
                    }
                }
            }
        }
    }

    bool updateListingMode = false;
    string listingModeString;
    if(archive.read("listingMode", listingModeString)){

        if(listingModeString != listingModeCombo.currentOrgText().toStdString()){
            listingModeCombo.blockSignals(true);
            if(listingModeCombo.findOrgText(listingModeString, true) >= 0){
                updateListingMode = true;
            }
            listingModeCombo.blockSignals(false);
        }
    }

    if(isArchiveOfCurrentBodyItemEnabled){
        setCurrentBodyItem(archive.findItem<BodyItem>("currentBodyItem"), updateListingMode);
    } else if(updateListingMode){
        setCurrentBodyItem(currentBodyItem, true);
    } else {
        restoreTreeState();
    }

    return true;
}
