var count = (function toPrecision(BigNumber) {
    var start = +new Date(),
        log,
        error,
        undefined,
        passed = 0,
        total = 0;

    if (typeof window === 'undefined') {
        log = console.log;
        error = console.error;
    } else {
        log = function (str) { document.body.innerHTML += str.replace('\n', '<br>') };
        error = function (str) { document.body.innerHTML += '<div style="color: red">' +
          str.replace('\n', '<br>') + '</div>' };
    }

    if (!BigNumber && typeof require === 'function') BigNumber = require('../bignumber');

    function assert(expected, actual) {
        total++;
        if (expected !== actual) {
           error('\n Test number: ' + total + ' failed');
           error(' Expected: ' + expected);
           error(' Actual:   ' + actual);
           //process.exit();
        }
        else {
            passed++;
            //log('\n Expected and actual: ' + actual);
        }
    }

    function assertException(func, message) {
        var actual;
        total++;
        try {
            func();
        } catch (e) {
            actual = e;
        }
        if (actual && actual.name == 'BigNumber Error') {
            passed++;
            //log('\n Expected and actual: ' + actual);
        } else {
            error('\n Test number: ' + total + ' failed');
            error('\n Expected: ' + message + ' to raise a BigNumber Error.');
            error(' Actual:   ' + (actual || 'no exception'));
            //process.exit();
        }
    }

    function T(expected, value, precision){
        assert(String(expected), new BigNumber(value).toPrecision(precision));
    }

    log('\n Testing toPrecision...');

    BigNumber.config({
        DECIMAL_PLACES : 20,
        ROUNDING_MODE : 4,
        ERRORS : true,
        RANGE : 1E9,
        EXPONENTIAL_AT : [-7, 40]
    });

    // ---------------------------------------------------------------- v8 start

    T('1e+27', '1.2345e+27', 1);
    T('1.2e+27', '1.2345e+27', 2);
    T('1.23e+27', '1.2345e+27', 3);
    T('1.235e+27', '1.2345e+27', 4);
    T('1.2345e+27', '1.2345e+27', 5);
    T('1.23450e+27', '1.2345e+27', 6);
    T('1.234500e+27', '1.2345e+27', 7);

    T('-1e+27', '-1.2345e+27', 1);
    T('-1.2e+27', '-1.2345e+27', 2);
    T('-1.23e+27', '-1.2345e+27', 3);
    T('-1.235e+27', '-1.2345e+27', 4);
    T('-1.2345e+27', '-1.2345e+27', 5);
    T('-1.23450e+27', '-1.2345e+27', 6);
    T('-1.234500e+27', '-1.2345e+27', 7);

    T('7', 7, 1);
    T('7.0', 7, 2);
    T('7.00', 7, 3);

    T('-7', -7, 1);
    T('-7.0', -7, 2);
    T('-7.00', -7, 3);

    T('9e+1', 91, 1);
    T('91', 91, 2);
    T('91.0', 91, 3);
    T('91.00', 91, 4);

    T('-9e+1', -91, 1);
    T('-91', -91, 2);
    T('-91.0', -91, 3);
    T('-91.00', -91, 4);

    T('9e+1', 91.1234, 1);
    T('91', 91.1234, 2);
    T('91.1', 91.1234, 3);
    T('91.12', 91.1234, 4);
    T('91.123', 91.1234, 5);
    T('91.1234', 91.1234, 6);
    T('91.12340', 91.1234, 7);
    T('91.123400', 91.1234, 8);
    T('-9e+1', -91.1234, 1);
    T('-91', -91.1234, 2);
    T('-91.1', -91.1234, 3);
    T('-91.12', -91.1234, 4);
    T('-91.123', -91.1234, 5);
    T('-91.1234', -91.1234, 6);
    T('-91.12340', -91.1234, 7);
    T('-91.123400', -91.1234, 8);

    T('NaN', NaN, 1);
    T('Infinity', Infinity, 2);
    T('-Infinity', -Infinity, 2);

    T('5.55000000000000e-7', 0.000000555, 15);
    T('-5.55000000000000e-7', -0.000000555, 15);
    T('-1.2e-9', -.0000000012345, 2);
    T('-1.2e-8', -.000000012345, 2);
    T('-1.2e-7', -.00000012345, 2);
    T('1e+8', 123456789, 1);
    T('123456789', 123456789, 9);
    T('1.2345679e+8', 123456789, 8);
    T('1.234568e+8', 123456789, 7);
    T('-1.234568e+8', -123456789, 7);

    T('-0.0000012', -.0000012345, 2);
    T('-0.000012', -.000012345, 2);
    T('-0.00012', -.00012345, 2);
    T('-0.0012', -.0012345, 2);
    T('-0.012', -.012345, 2);
    T('-0.12', -.12345, 2);
    T('-1.2', -1.2345, 2);
    T('-12', -12.345, 2);
    T('-1.2e+2', -123.45, 2);
    T('-1.2e+3', -1234.5, 2);
    T('-1.2e+4', -12345, 2);
    T('-1.235e+4', -12345.67, 4);
    T('-1.234e+4', -12344.67, 4);

    T('1.3', 1.25, 2);
    T('1.4', 1.35, 2);

    T('1e+4', 9631.01, 1);
    T('1.0e+7', 9950095.87, 2);
    T('1e+1', '9.856839969', 1);
    T('1e+2', '97.504', 1);
    T('1e+5', 97802.6, 1);
    T('1e+1', 9.9617, 1);
    T('1e+3', 989.2, 1);
    T('1.0e+5', 99576, 2);
    T('1e+8', '96236483.87', 1);

    // ------------------------------------------------------------------ v8 end

    BigNumber.config({ROUNDING_MODE : 0});

    T('-0.000090000000', '-0.00009', 8);
    T('-7e-7', '-0.0000007', 1);
    T('68.9316834061848', '68.931683406184761912218250317', 15);
    T('7.8601018089704732e+27', '7860101808970473167417935916.60087069', 17);
    T('3.21445885399803244067719798337437062000000e-11', '0.0000000000321445885399803244067719798337437062', 42);
    T('-8171786349835057630612358814.162756978', '-8171786349835057630612358814.162756977984', 37);
    T('3340.9039701', '3340.903970019817086594869184429527413533291595472085', 11);
    T('-7269097658095414435895.9161181115739745427300313060', '-7269097658095414435895.916118111573974542730031306', 50);
    T('0.00000632207', '0.00000632206077863', 6);
    T('6e+2', '573', 1);
    T('7.4e-7', '0.000000738', 2);
    T('-5.031561e-7', '-0.0000005031560306227217140253964236911907612837', 7);
    T('-4.291e+11', '-429050053964', 4);
    T('8.514e+7', '85131637', 4);
    T('-3.4e-9', '-0.000000003326783057540398442677461', 2);
    T('6.9404295962722512e-20', '0.00000000000000000006940429596272251146200868514973032594273', 17);
    T('-828376248340605120247.15155295014', '-828376248340605120247.15155295013990774586360178257303370779', 32);
    T('-7.9828e+6', '-7982750.6677764682946015520272838914918899297118139169410659', 5);
    T('0.00712610393722542527880200', '0.007126103937225425278801997738', 24);
    T('-5.7e+4', '-56242', 2);
    T('-8928855203945443164.755136735230293537', '-8928855203945443164.755136735230293536124112124', 37);
    T('5218572327.99', '5218572327.98424443372003772604597054153304', 12);
    T('71707870535238750871516796339.60', '71707870535238750871516796339.59678962573869890935', 31);
    T('88817462.7137982220652429', '88817462.71379822206524285939115943006583441400005007918', 24);
    T('3.00000e-9', '0.000000003', 6);
    T('-6.053', '-6.05291095813493573191', 4);
    T('6.51630828677e+19', '65163082867698740076', 12);
    T('2483202135696501.60187899', '2483202135696501.60187898870193199949004966876115645', 24);
    T('1.0766e-10', '0.000000000107650515680635692286894826641576642261', 5);
    T('642724503819056076.659397077514269963295025', '642724503819056076.659397077514269963295024012414', 42);
    T('-7.1192e+21', '-7119169102619893823635.32141854354', 5);
    T('-6.717481255640638829101946114674e-8', '-0.000000067174812556406388291019461146732616998258', 31);
    T('-12.41976452', '-12.4197645179995365323309894', 10);
    T('-6.529258780126449116249954644017839921024112900e-16', '-0.00000000000000065292587801264491162499546440178399210241129', 46);
    T('-441838.0', '-441838', 7);
    T('1.128285293592950e-8', '0.000000011282852935929493101783925259749957192', 16);
    T('-8.654857e+7', '-86548567', 7);
    T('3.8883293855303995e-7', '0.00000038883293855303994672627854769926811949', 17);
    T('3.25870000e-13', '0.00000000000032587', 9);
    T('3.702e+6', '3701031.59037494113', 4);
    T('-3580077435.93682917449675702508371047', '-3580077435.93682917449675702508371046631533', 36);
    T('-7.400', '-7.4', 4);
    T('109519523263844229810.068', '109519523263844229810.067657779734413280795410968892638', 24);
    T('-509247322311590671954830.86847660619', '-509247322311590671954830.8684766061855', 35);
    T('7.5518638430980800496570562671727890e-10', '0.00000000075518638430980800496570562671727889997', 35);
    T('-5056721600639122835615986051.468831942818200', '-5056721600639122835615986051.4688319428182', 43);
    T('-1.796146861125551785886171829251460000000000e-16', '-0.000000000000000179614686112555178588617182925146', 43);
    T('6.0e+2', '599', 2);
    T('7.619930e-16', '0.00000000000000076199293', 7);
    T('834668.2370121038159610193', '834668.237012103815961019258574789273273342', 25);
    T('-3.92251395952329649490768e+26', '-392251395952329649490767912.240768552138247705202732', 24);
    T('-47504099413385554632166.5098', '-47504099413385554632166.50972492550706', 27);
    T('-763912347.2814762', '-763912347.28147614121123622213255703', 16);
    T('62.06092655083887409325613', '62.06092655083887409325612694639', 25);
    T('-5262696454512614512.606481226453660', '-5262696454512614512.6064812264536594032183632269343356197', 34);
    T('-324.4757687696223789483683661674', '-324.475768769622378948368366167382', 31);
    T('41034172.92', '41034172.91600339960206', 10);
    T('78.8705822927994376887853', '78.870582292799437688785229493004059423117558', 24);
    T('13.2', '13.12562822628823049', 3);
    T('-47510172284493547923917836.573231120531506713', '-47510172284493547923917836.573231120531506712946048319', 44);
    T('-762632827', '-762632826.1', 9);
    T('4e+13', '33953600811490.5124040357996528537249966', 1);
    T('-8.1071720769966e+25', '-81071720769965824452477185.9774686', 14);
    T('5.680e+22', '56797151043432713156004.54588148769825716', 4);
    T('-32.861964853600294215914162138', '-32.8619648536002942159141621375696711453', 29);
    T('-30816296472.656223819', '-30816296472.656223818627686674207740641739447', 20);
    T('9.085158071966474886768332e+30', '9085158071966474886768331250478.08286703003069431737582', 25);
    T('-504664.230671', '-504664.230670963', 12);
    T('-1.013105775e+16', '-10131057742551895.1862632947', 10);
    T('-542318552011993.7986', '-542318552011993.798599548369674038472319697531161473933214', 19);
    T('-9.0310e+16', '-90309463572405956', 5);
    T('-6.771931816', '-6.77193181559601521980141', 10);
    T('4703514776786483.3', '4703514776786483.2838035091610781996968798', 17);
    T('-8.43684044711e+12', '-8436840447101.126789480845', 12);
    T('-343.5602326850284', '-343.56023268502830337554680474463898876525434', 16);
    T('-3.17649252e+24', '-3176492517226717196752073', 9);
    T('-2.3912888503759090e+28', '-23912888503759089673174904075', 17);
    T('6.8853846820341808e+23', '688538468203418073592188', 17);
    T('4e+17', '343455415908256944', 1);
    T('-1.4e+9', '-1336106841', 2);
    T('-2244450.2134814273335263', '-2244450.2134814273335262397290334104071203538487453309626146', 23);
    T('8.74e+29', '873625255363763952428129881990.679929486040461455296118489', 3);
    T('-1.85453549733179613185923288786', '-1.8545354973317961318592328878502252820666161607740183', 30);
    T('431.7150651927', '431.71506519265522010949747887049', 13);
    T('-8606297211156287.52520023752564', '-8606297211156287.5252002375256362382564355963505470716151', 30);
    T('-8.4634889709e+24', '-8463488970828351722405003.220603', 11);

    BigNumber.config({ROUNDING_MODE : 1});

    T('-844789036.5239726', '-844789036.52397268892', 16);
    T('-5056.20629012767878749185273209679064306054', '-5056.206290127678787491852732096790643060542', 42);
    T('-0.3287519131314873763501859870298952500', '-0.32875191313148737635018598702989525', 37);
    T('-60729764', '-60729764', 8);
    T('-7.622e-14', '-0.00000000000007622481594531380999826456196664586', 4);
    T('-4686402261639729535.736324492474', '-4686402261639729535.7363244924747488', 31);
    T('-2.0', '-2', 2);
    T('-13801188035233586637950193108.13592574381473451125649500', '-13801188035233586637950193108.135925743814734511256495', 55);
    T('0.0000807327587149839799300000', '0.00008073275871498397993', 24);
    T('-6.000000e-8', '-0.00000006', 7);
    T('-3.83574993e+11', '-383574993535', 9);
    T('7.6987000000000000e-14', '0.000000000000076987', 17);
    T('80928866804.6112050947427973', '80928866804.6112050947427973864826014844575374353', 27);
    T('-0.00730140', '-0.0073014067221009206110062377503733', 6);
    T('2.72104773884160491036088486e+30', '2721047738841604910360884862459.4086993273252009015', 27);
    T('3.008780781917733594e+25', '30087807819177335941398228.1424107931203', 19);
    T('-1.31528920779613669158250146972297797867760000000000000000000e-19', '-0.00000000000000000013152892077961366915825014697229779786776', 60);
    T('-8.5e+11', '-858982311008.257025719798657844609315293821', 2);
    T('-3.6312e-12', '-0.0000000000036312827608449878', 5);
    T('-0.0060000', '-0.006', 5);
    T('-1e+1', '-12', 1);
    T('5.779447e+14', '577944759667712', 7);
    T('-8.753124714248104872487955947563035887800000000000e-13', '-0.00000000000087531247142481048724879559475630358878', 49);
    T('0.000736948830704113912', '0.000736948830704113912970821957479', 18);
    T('-4.65727e+23', '-465727983501322687372765', 6);
    T('-0.00000332331666628036603', '-0.000003323316666280366035430077076052', 18);
    T('3.533702e-8', '0.00000003533702791135712510338001418872124', 7);
    T('-0.04340', '-0.0434', 4);
    T('-597340.278566069086858587852236235470', '-597340.2785660690868585878522362354706741', 36);
    T('6.000e-8', '0.00000006', 4);
    T('-3.624323359112776296e-19', '-0.00000000000000000036243233591127762966338166', 19);
    T('-3731378568692042924197.154', '-3731378568692042924197.15400334142251496795634388', 25);
    T('-68249040894032065692.62', '-68249040894032065692.62771690318493', 22);
    T('8786096722661914.89732851', '8786096722661914.89732851188880184891692993684242690315', 24);
    T('-1.8413321536281347264486372900000000000e-12', '-0.00000000000184133215362813472644863729', 38);
    T('4.0e-9', '0.0000000040395827543504045', 2);
    T('-2.9427e+16', '-29427119846374896', 5);
    T('-917760614.4', '-917760614.45404359204911454', 10);
    T('8e+4', '89427', 1);
    T('0.00000920323988134356953828667260', '0.0000092032398813435695382866726', 27);
    T('8.2e+16', '82068995955708118', 2);
    T('3.35195944828e+26', '335195944828445911672446409.3379497158141', 12);
    T('-3.89774891030e-9', '-0.00000000389774891030223957363124620581272897758735065471', 12);
    T('-4', '-4', 1);
    T('8', '8', 1);
    T('1.41172955693912934219137966000000e-10', '0.000000000141172955693912934219137966', 33);
    T('9.21481e+13', '92148111958857', 6);
    T('-5.859975978432853e-18', '-0.0000000000000000058599759784328539', 16);
    T('-72.0', '-72', 3);
    T('3785098751297.8929911950994079707157472', '3785098751297.89299119509940797071574729867819252140059', 38);
    T('4.38e+16', '43893416753778361.297703358127215475077814', 3);
    T('-33110.29096', '-33110.2909623520267070846514', 10);
    T('-74.38305251784882707720486436292121914036495', '-74.3830525178488270772048643629212191403649548392158614', 43);
    T('-4.31091381814e+27', '-4310913818147299779611829988.1707181186375975966133328', 12);
    T('-1e+7', '-19238355', 1);
    T('-6996635475270055814687.6', '-6996635475270055814687.6250552375470211825551', 23);
    T('-8.203834974e+12', '-8203834974826.23347025', 10);
    T('-7.4775e+5', '-747754.16564979702874976822', 5);
    T('-9.291256959320e+23', '-929125695932058727753757.0232350927089256760451379', 13);
    T('8.5e+11', '853985704471', 2);
    T('-6.6560212', '-6.65602121044617863313449309597493831', 8);
    T('1785977942777.20398797', '1785977942777.2039879764361236566223563439', 21);
    T('6.1333504356e+23', '613335043569565749922342.8859983523919141148812213832', 11);
    T('-5.6e+8', '-565718507', 2);
    T('87732918932081', '87732918932081.5225691355449629111825', 14);
    T('34510.55200915393645123', '34510.55200915393645123649', 22);
    T('80406604570281847.64813851700344044652354', '80406604570281847.648138517003440446523542379', 40);
    T('4350.66340515', '4350.66340515436550356256', 12);
    T('-1.795651762606996e+19', '-17956517626069967584.285356976401607845756322546530214497', 16);
    T('9.162e+24', '9162436195089050810891391.493612', 4);
    T('-7.82552e+6', '-7825522.1080200627404337', 6);
    T('-358162040.1796393759838430', '-358162040.17963937598384303781972517649539', 25);
    T('-20732451778.4', '-20732451778.464877395794562570976729066571095229', 12);
    T('-239748.58739', '-239748.5873964402372997371903319', 11);
    T('-6.106537e+9', '-6106537070.58700935776016694', 7);
    T('4e+23', '405561947729011104089456.7617832102516', 1);
    T('-1.7252987e+10', '-17252987633.58674364430598655792', 8);
    T('61.38960691398015334867512513960', '61.3896069139801533486751251396015198659145775291764', 31);
    T('-70493899102410752290043364.4667507415385', '-70493899102410752290043364.466750741538512', 39);
    T('-891.3505', '-891.35058685025619', 7);
    T('1.5e+8', '153705028.906', 2);
    T('5.80e+18', '5805164734299168659.6173113885173384955443', 3);
    T('-1.719875889271327', '-1.719875889271327133154458155573493605566221534', 16);
    T('113.672129563', '113.672129563441659725876055771857758675550104070419635029', 12);
    T('-77950052814622081084397.9', '-77950052814622081084397.91853869253589242574', 24);
    T('4.53106985e+27', '4531069852787151785292512309.2901993579425172826443679877', 9);
    T('45285.246089613169416440797840714', '45285.2460896131694164407978407142422013937', 32);
    T('307760226411464.7333268079863299', '307760226411464.73332680798632996332324381779707', 31);

    BigNumber.config({ROUNDING_MODE : 2});

    T('-0.0300', '-0.0300921721159558', 3);
    T('65317841202.20949859371772273480125', '65317841202.2094985937177227348012464402154', 34);
    T('-8.9231575495202e+29', '-892315754952021994731329589682.1894180393920044085713', 14);
    T('-2.8075679202e-8', '-0.0000000280756792028583066', 11);
    T('9.71456e+9', '9714558552', 6);
    T('2.9514099281e-10', '0.00000000029514099281', 11);
    T('-1.24459e+14', '-124459985101107', 6);
    T('0.0000734657394154607815562372000000', '0.0000734657394154607815562372', 30);
    T('1.78719530353972e+15', '1787195303539715', 15);
    T('-2.8e+9', '-2861102528', 2);
    T('-8.74480375581000e-9', '-0.00000000874480375581', 15);
    T('-1792404726015427380.248150830448457643618022', '-1792404726015427380.248150830448457643618022', 43);
    T('-678437320202616518.2220157912209286', '-678437320202616518.22201579122092864', 34);
    T('-1.937304915215780220809799809655893674619672771e-8', '-0.000000019373049152157802208097998096558936746196727718', 46);
    T('824172.15863347130174103087', '824172.15863347130174103086069960571', 26);
    T('1.90040714061724000e-9', '0.00000000190040714061724', 18);
    T('-1634488249956745498.58311', '-1634488249956745498.58311123049258868631623840423306', 24);
    T('0.0000019600923098540334001755857361187871270117098000', '0.0000019600923098540334001755857361187871270117098', 47);
    T('8.383e+4', '83829', 4);
    T('2.843306120337864064e+23', '284330612033786406376718', 19);
    T('1.86235e+15', '1862340943682995.08270612464203237562317928642459', 6);
    T('-2.31e+13', '-23195312138083', 3);
    T('5.450237e+21', '5450236028274773541895.65198933808968167192289601277', 7);
    T('-0.008976419749408075453861117865459', '-0.00897641974940807545386111786545972434475187220274239581167', 31);
    T('-761181660548661030.25', '-761181660548661030.25539542029', 20);
    T('-1844205.93619958', '-1844205.936199580689273072905714475263817', 15);
    T('4842.77906784902805070438222238898372327093', '4842.77906784902805070438222238898372327092242428134814721', 42);
    T('-4.161198953445629503503971e+26', '-416119895344562950350397179', 25);
    T('1.084e+4', '10836', 4);
    T('8.71081704218174598654542083000e-8', '0.0000000871081704218174598654542083', 30);
    T('7.9139683e+36', '7913968291641940848703040206324645237.8515176490912667096', 8);
    T('-0.000008', '-0.000008', 1);
    T('8.3660085625e+34', '83660085624983922907621996804192921.3992927', 11);
    T('0.000006980263008', '0.000006980263007423150706324065130475391', 10);
    T('-31348084528321454060964445534333629317.69561497283830023', '-31348084528321454060964445534333629317.69561497283830023', 55);
    T('-2417953792643886.3485495754363678888681996409674308643', '-2417953792643886.3485495754363678888681996409674308643', 53);
    T('4.0e+6', '3982592', 2);
    T('-2092315.015029722200', '-2092315.0150297222', 19);
    T('-364992136844916.9092238', '-364992136844916.909223894931280218350055327754935', 22);
    T('8.34e+24', '8333642861002789136219873', 3);
    T('7.6008837179413e+14', '760088371794122.3380234188299740029832128019574765416', 14);
    T('-6655726127.0', '-6655726127', 11);
    T('-8.2157109e-9', '-0.000000008215710991605913786700324', 8);
    T('-0.00007003302912717900', '-0.000070033029127179', 16);
    T('-919151201.84874', '-919151201.8487431', 14);
    T('-7.7284e+34', '-77284694095619151624570282373349775.20332', 5);
    T('-0.01348565', '-0.013485650787487', 7);
    T('4793.07921563762902275733457926767780', '4793.0792156376290227573345792676778', 36);
    T('-29428.0', '-29428', 6);
    T('-5.031066774187e+17', '-503106677418717710.020194320886816967824316089135', 13);
    T('8.5822119333e+30', '8582211933222895201417193603829.362', 11);
    T('3.69818e+29', '369817665788648417491163239098.45906115246177782675574', 6);
    T('-16637318966.7921513256', '-16637318966.79215132564236', 21);
    T('-3.511414e+7', '-35114143.07750577', 7);
    T('-4.00583795e+15', '-4005837953660576.377392671047611906101', 9);
    T('2.857013789e+29', '285701378835628725742568343419.93', 10);
    T('3.784446708460892550157924e+30', '3784446708460892550157923126965.213', 25);
    T('-8.07e+11', '-807835898139.8273423102575232570378422434793962915', 3);
    T('7.2e+12', '7166828666682', 2);
    T('-2.7e+15', '-2759498523697862.0885969105603319015115245', 2);
    T('3.93e+3', '3920.77076847274147345709652305252825254482870430341848100141', 3);
    T('-6e+12', '-6791423282682', 1);
    T('-8.6204e+14', '-862048518070094.31', 5);
    T('124280692175695486153.808', '124280692175695486153.80744660510519294193', 24);
    T('-460557721667773.3587267520', '-460557721667773.3587267520989', 25);
    T('-6268536499825359064300.23', '-6268536499825359064300.2317858', 24);
    T('292408901.64362273508249026852286', '292408901.64362273508249026852285294673307', 32);
    T('-649622345434955387029125.11357971191', '-649622345434955387029125.113579711917604812061404975326264229', 35);
    T('-4.287461556179478781e+27', '-4287461556179478781817700851.131100167', 19);
    T('-5.891552271022619e+29', '-589155227102261925251047170629.30784624401', 16);
    T('1.88e+5', '187009.128', 3);
    T('4299388.1132142278863818606739416640', '4299388.1132142278863818606739416639837103457725931818979', 35);
    T('-7.8e+8', '-788088836.225886207482064192607002511282756502400977', 2);
    T('-56025768755085222.404269', '-56025768755085222.404269295514', 23);
    T('-8376.71149693765842', '-8376.71149693765842060199698996606139145426', 18);
    T('-1.7218673528e+29', '-172186735288586033321621121024.11240623', 11);
    T('-3.31e+28', '-33197729862068219255677464974', 3);
    T('-4.835191326e+29', '-483519132605694848658321267839.23575134378118945659616358', 10);
    T('7.3', '7.24882150443803', 2);
    T('-89186640077683569.407061427673', '-89186640077683569.4070614276736450982125609', 29);
    T('-49379651041268.5', '-49379651041268.548293', 15);
    T('-7685054.17489171951660', '-7685054.17489171951660508194254495141726065698575306365447451', 21);

    BigNumber.config({ROUNDING_MODE : 3});

    T('-39449414270333.925852213835', '-39449414270333.925852213834759031494508489474', 26);
    T('-7.50437989976', '-7.50437989975503711836768', 12);
    T('-0.000004303975760000000', '-0.00000430397576', 16);
    T('-16040233916257241895.97650633973989', '-16040233916257241895.9765063397398857', 34);
    T('-7438.9287248601393819', '-7438.9287248601393818639176907606', 20);
    T('9.857465584298e-7', '0.000000985746558429876825600458537705318327799', 13);
    T('532637.9095983547284850466577958315920', '532637.90959835472848504665779583159203905641996', 37);
    T('-1.40416695292e+30', '-1404166952915258058306475434520.7856110230505157', 12);
    T('60346876.6670832429026869255506808488', '60346876.6670832429026869255506808488', 36);
    T('-2.52466133e+23', '-252466132238128405832984', 9);
    T('55', '55', 2);
    T('8', '8', 1);
    T('-63075151.962465776516325792253177939493172', '-63075151.9624657765163257922531779394931714', 41);
    T('7.411461e+17', '741146113404361548.543142388', 7);
    T('-58835755359067474972692072494278983.7', '-58835755359067474972692072494278983.6314961114191480012916', 36);
    T('-3.5408424427810e+21', '-3540842442780946836975', 14);
    T('-8.6985e+22', '-86984550895486812167623.3816747460029582321093085895', 5);
    T('-4.4625326e+20', '-446253250722400223428', 8);
    T('-79301328.93777304419247399162092400', '-79301328.937773044192473991620924', 34);
    T('-1.6065669647394805e+28', '-16065669647394804383207152895.0285044537455', 17);
    T('-333', '-333', 3);
    T('7', '7', 1);
    T('7.24707e+13', '72470760481059', 6);
    T('39232618.1513515442233995765535454389', '39232618.151351544223399576553545438981252', 36);
    T('-4e+5', '-357994', 1);
    T('-1.90e+4', '-18904.11335233460016293296574557643545512393801643609213933', 3);
    T('-6585152111956929.924309477123328984876184272828762900', '-6585152111956929.9243094771233289848761842728287629', 52);
    T('4.505e-7', '0.0000004505328', 4);
    T('-2.4125965461846e+19', '-24125965461845662271', 14);
    T('4.82673137e+33', '4826731373891127996812671510065700.871947701', 9);
    T('-6621278.2', '-6621278.1120573461544975284970826524341806671316100080257485', 8);
    T('-1.8015392869565386634525164264799463344376205007391000000e-7', '-0.00000018015392869565386634525164264799463344376205007391', 56);
    T('-0.00026465463574439280006655492609887593', '-0.00026465463574439280006655492609887592672292239588307259', 35);
    T('4.87815228988300090', '4.8781522898830009076096556452567', 18);
    T('-5.1107117199524082779077801201617e+35', '-511071171995240827790778012016163902', 32);
    T('1.4734242515706890557e+20', '147342425157068905574.390834406', 20);
    T('-4019325091848890817268596991.815200', '-4019325091848890817268596991.8152', 34);
    T('3.8e+14', '384715413967421', 2);
    T('7483444.49', '7483444.498791364040133403947480439118040376737700653', 9);
    T('-594538312.6255', '-594538312.625485172379', 13);
    T('0.00753000', '0.00753', 6);
    T('8.1440148247e+13', '81440148247675.27449603492606125135884', 11);
    T('8.444003009300e+21', '8444003009300239495556', 13);
    T('2308.1529840912558574923966042774800185916972327325289', '2308.1529840912558574923966042774800185916972327325289261', 53);
    T('2.67e+3', '2674.698673623', 3);
    T('-2.82819136180287470854625537e+30', '-2828191361802874708546255368471.80800005766', 27);
    T('518250411', '518250411', 9);
    T('3.2e+4', '32661.9135347256259375001777960775509', 2);
    T('29.15347602216416991973', '29.153476022164169919735054013077734177', 22);
    T('-4.611285536613066108e+30', '-4611285536613066107912600830385', 19);
    T('-51774110.0705144989023975360207167071143094356321', '-51774110.070514498902397536020716707114309435632036586', 48);
    T('-11969053.91', '-11969053.9052', 10);
    T('3102686944.558209725206279080384565972890930884678', '3102686944.5582097252062790803845659728909308846780130141', 49);
    T('-3601967476456.11863985450841401751857855', '-3601967476456.1186398545084140175185785472952682624279698', 39);
    T('-5e+15', '-4873768150955988', 1);
    T('-352.0819', '-352.08189544801640267067628', 7);
    T('-2.58805959847e+29', '-258805959846025073839294200101', 12);
    T('-66245829859.35391480', '-66245829859.353914791938511206971693', 19);
    T('1.54e+9', '1544806884.11336335261587391', 3);
    T('-27.7997003414813453645099801', '-27.79970034148134536450998001339677019', 27);
    T('14062458038542559389162.9204850167', '14062458038542559389162.9204850167680814', 33);
    T('1.558308e+23', '155830857739562225455438.36', 7);
    T('-191388637226531701343.3', '-191388637226531701343.25549694555307', 22);
    T('5551.7364563066033013381', '5551.73645630660330133811512206', 23);
    T('-374.187067', '-374.187066872511899560500516595762548924654039141', 9);
    T('5608.7939', '5608.79395345957', 8);
    T('-7.46461560634688e+16', '-74646156063468781.44597747432564', 15);
    T('6.282e+14', '628222207265224.793350069927452126508488621324740335935808339', 4);
    T('739267731.33076658725535583758', '739267731.3307665872553558375867276395038136046', 29);
    T('-7.243744595180e+19', '-72437445951792218018.4147098155', 13);
    T('148197.230592476071658991268667398', '148197.23059247607165899126866739893696346154456779371449089', 33);
    T('-7326871.99257009310974109937661882759811033', '-7326871.9925700931097410993766188275981103204155306', 42);
    T('-5.2007521e+21', '-5200752087996702875406.6925', 8);
    T('9.00107829e+18', '9001078299504900356', 9);
    T('229140061917', '229140061917.91723092039513459551808768805307572856707938', 12);
    T('-6868103.8726464561656824818722569258791476905', '-6868103.872646456165682481872256925879147690458928033592856', 44);
    T('-220947971933643883580237.50', '-220947971933643883580237.49534341528328', 26);
    T('544164102001101766247312.91529628700', '544164102001101766247312.915296287008639054933', 35);
    T('1.70e+23', '170271631736408409477543.35894', 3);
    T('-5735975666.6511674981929172446', '-5735975666.65116749819291724455000274115296', 29);
    T('-67513065.4797', '-67513065.4796695356', 12);
    T('-9e+19', '-82164590986048729101.278942224271247884118371796531523', 1);
    T('687378946204028.408158998985701', '687378946204028.408158998985701430935094', 30);
    T('42.452', '42.4523909443358871476552683504968536100051', 5);
    T('-22771061110217019663705702.44170142085172', '-22771061110217019663705702.44170142085171219649140996', 40);
    T('-1470.640309974016167512235698629586', '-1470.6403099740161675122356986295857257144815364', 34);
    T('-1.110228e+27', '-1110227398804733429555663947.06619', 7);
    T('-6.4898237111e+26', '-648982371105405071851661301', 11);
    T('-4641197449469148.658850361201903', '-4641197449469148.658850361201902222', 31);

    BigNumber.config({ROUNDING_MODE : 4});

    T('7.905300379788e+16', '79053003797878062.6454954', 13);
    T('-6.83490000000e-13', '-0.00000000000068349', 12);
    T('-62760641815.69084973661201201', '-62760641815.690849736612012010742308663', 28);
    T('0.000704', '0.000704496313', 3);
    T('82926865286287.8852357368342860830310721063079299643', '82926865286287.88523573683428608303107210630792996432', 51);
    T('-0.00032388272393900301214220090249', '-0.00032388272393900301214220090248744799603424908', 29);
    T('8.6e+12', '8621641486938.4837308885005093571508566552428700982454', 2);
    T('2', '2', 1);
    T('1.4641440117052559075e+20', '146414401170525590746.047955203899370771105088', 20);
    T('3511.925583', '3511.925583', 10);
    T('2861824.253079699095728', '2861824.253079699095727765750377038689', 22);
    T('-3.940097756e+10', '-39400977564.548924098664431671700066962', 10);
    T('-888', '-888', 3);
    T('-0.000302106125213724988141721256104', '-0.00030210612521372498814172125610432438685', 30);
    T('6943.4804552555315615809650428503', '6943.480455255531561580965042850266831249032130818358478956', 32);
    T('3365678', '3365678.3397481381125085749', 7);
    T('-5.3943374314e+19', '-53943374313769567458.386865325', 11);
    T('-6.67880509225510150542252852147049489938254298497979', '-6.6788050922551015054225285214704948993825429849797925563674', 51);
    T('1.36424e+18', '1364240644139816224.60228356028', 6);
    T('1.410236477950416725e+23', '141023647795041672538410.84935693266374259666015274447', 19);
    T('-802.817765', '-802.81776500697712984253334522', 9);
    T('-5.276210722424690668896260075355037218851', '-5.27621072242469066889626007535503721885096', 40);
    T('-0.000874209568970788', '-0.0008742095689707877849902027926289294748756775668387', 15);
    T('0.092053833162002', '0.09205383316200189249855864903410820435666385119723209239', 14);
    T('7.0656298318128209e-14', '0.0000000000000706562983181282092835675843980510112', 17);
    T('-8.66511516852116659e+18', '-8665115168521166587', 18);
    T('3.3490648464e+22', '33490648463534229842937.79268276945692333064632966129475', 11);
    T('-39041587174692569176.82740706154183894', '-39041587174692569176.827407061541838942655371389185', 37);
    T('-3834.0', '-3834', 5);
    T('-0.008912382644814418776268630', '-0.00891238264481441877626863', 25);
    T('-2.1e+5', '-206119', 2);
    T('4.83340000000e-8', '0.000000048334', 12);
    T('3.185196533675230520000000000000e-19', '0.000000000000000000318519653367523052', 31);
    T('6.0431217298488095562718496137220939447806000000000000000e-17', '0.000000000000000060431217298488095562718496137220939447806', 56);
    T('196.519569070149034', '196.51956907014903416531531', 18);
    T('0.0000046405006597117307566000', '0.0000046405006597117307566', 23);
    T('9.10e+16', '90974867783311624.1073050261392195984211985571898902', 3);
    T('0.0009', '0.0009', 1);
    T('-784.344', '-784.3442317667756502522526185951859933319162', 6);
    T('4.407336482399797058693e+28', '44073364823997970586929155979.43263841350505', 22);
    T('-3.0000000000e-13', '-0.0000000000003', 11);
    T('0.800', '0.8', 3);
    T('0.04643398170143261158595951942031', '0.046433981701432611585959519420314960367263', 31);
    T('-8e+26', '-786589693451258754942279859.3834', 1);
    T('-26.0', '-26', 3);
    T('-8.462226728e+11', '-846222672789.2087639320702375427266333530942524245', 10);
    T('-4e-7', '-0.0000004019666978288041783154210868', 1);
    T('-315609.775843992', '-315609.775843992', 15);
    T('-3.319e+9', '-3318880945', 4);
    T('-6', '-6.2847', 1);
    T('7.754663772705e+20', '775466377270546647581.033426922028458904663', 13);
    T('-72577466365074249372160551.716563', '-72577466365074249372160551.71656300408', 32);
    T('-7.8e+14', '-775743793612078', 2);
    T('132441.1194131940273344', '132441.119413194027334448511114274180643744', 22);
    T('-2e+8', '-175718250.88225246544054572629398592939731158738360059', 1);
    T('8603217351572193.39188696', '8603217351572193.391886964766947146712574336', 24);
    T('-9.1544942231978215224e+22', '-91544942231978215224182.9277714', 20);
    T('2.67483212861962e+22', '26748321286196185405759.132664', 15);
    T('-5812371.3', '-5812371.311809024582418495005304074', 8);
    T('-4.56681272e+10', '-45668127184.1622', 9);
    T('-6.833879652430027734e+28', '-68338796524300277341620461049.174596381', 19);
    T('3.5253e+11', '352531868532', 5);
    T('6.18754e+9', '6187538472.1814915517411034136013806202710623466754380762318', 6);
    T('-49119914201836431396827151123.9982195', '-49119914201836431396827151123.99821949990542', 36);
    T('-2.50e+18', '-2498994955335714645.22910610209', 3);
    T('112714.50647532453078481574527706184222476885', '112714.50647532453078481574527706184222476884905812', 44);
    T('1.3e+10', '13358297773', 2);
    T('3.85346866600e+27', '3853468666000315958109987025.078941', 12);
    T('-6.849e+16', '-68490080550892289', 4);
    T('9.095', '9.094726073939375', 4);
    T('4.6722099483e+12', '4672209948311.8638324115985415208264055834', 11);
    T('-75494281.3585391383', '-75494281.3585391382541907932608754414663476859104837422712', 18);
    T('7.9e+2', '787.7709059965548561711769118765', 2);
    T('6103081090513.979878497219802', '6103081090513.9798784972198017843', 28);
    T('-6207456599626114.392919', '-6207456599626114.39291886624528055513014220851925', 22);
    T('844941600554602638837.461606663208684075561936', '844941600554602638837.461606663208684075561935576396', 45);
    T('159438905444627555.28986', '159438905444627555.28985729196359392', 23);
    T('-3688253681705278.414841830526919796661181971979', '-3688253681705278.4148418305269197966611819719792068915', 46);
    T('-63', '-63.164640732796214571844119', 2);
    T('2.8e+11', '276059026705.36069', 2);
    T('357378.987253867425946425403370727230144', '357378.9872538674259464254033707272301441754336', 39);
    T('1597.52674152596523825479', '1597.526741525965238254790848976407269408999607', 24);
    T('4.63310587686706257280646279e+30', '4633105876867062572806462788592.801009', 27);
    T('-6.21108762339449e+20', '-621108762339448671355.1393522133', 15);
    T('8380435.063269894549337249', '8380435.063269894549337248813357930541546715547', 25);

    BigNumber.config({ROUNDING_MODE : 5});

    T('-1408003897645960.648499616456', '-1408003897645960.648499616456', 28);
    T('-7719307749101742537.6299396338672184', '-7719307749101742537.6299396338672184334306', 35);
    T('-1.0', '-1', 2);
    T('-8.28e+14', '-827860423543649', 3);
    T('0.00054398953021585321711560388890', '0.00054398953021585321711560388889590290139888', 29);
    T('-4.409e-9', '-0.000000004408792', 4);
    T('4.0000e-10', '0.0000000004', 5);
    T('3.40e+16', '34001779327925905', 3);
    T('-9.03e+34', '-90332622851356543193546536340366547', 3);
    T('-4.5320e+16', '-45320100856429143.39155209710530673318222777', 5);
    T('3.618e+30', '3618328715720583671291544414202', 4);
    T('-1003.61140', '-1003.61139687804673322250551', 9);
    T('-8139415035028632370.38737', '-8139415035028632370.38736602659835', 24);
    T('8e+7', '83198058', 1);
    T('-7.99492e+14', '-799491603856548', 6);
    T('-3.351956508998634059456001730355207230e-9', '-0.000000003351956508998634059456001730355207229966', 37);
    T('-14.69863659940007820467946969441090', '-14.698636599400078204679469694410899305', 34);
    T('-8.1805185086529e+32', '-818051850865289066860294784032304.6373757407', 14);
    T('8.21371840206651626757e+29', '821371840206651626756943367010.81915938727', 21);
    T('444', '444', 3);
    T('0.00000613258266938', '0.0000061325826693823067791292255878336353793864046451956723', 12);
    T('-554696279951718746537611.26040', '-554696279951718746537611.26040029508470430208572833137315', 29);
    T('446', '446.189185820662709163412845035853873', 3);
    T('22873128187827109553471831451.06623850867', '22873128187827109553471831451.06623850866672688842662473', 40);
    T('9e+5', '880389', 1);
    T('-6.7516118890844e+16', '-67516118890844443.625641', 14);
    T('-0.36107158435820', '-0.36107158435820101656696353075596201902674001080619510849', 14);
    T('8.958386374640407365', '8.958386374640407364828679985365339921820421370157246', 19);
    T('3e+2', '257', 1);
    T('-1.904659739878e+18', '-1904659739878060478.113131137688927604413210083841', 13);
    T('-0.0000627142', '-0.00006271421732891589577305487292334', 6);
    T('3.310541e+8', '331054103', 7);
    T('-1.793886e+23', '-179388600781592577147651.2641684828762234473', 7);
    T('0.0004600', '0.00046', 4);
    T('-2.9e+21', '-2906505321975413509885', 2);
    T('86415.94739506', '86415.9473950557683374', 13);
    T('6.730414', '6.7304135909152', 7);
    T('-5.032367e+14', '-503236749968584', 7);
    T('-5.0241682013868216287718e+32', '-502416820138682162877178622610283', 23);
    T('-0.0552606118984074172116684879479087', '-0.0552606118984074172116684879479087', 33);
    T('91017414629852252476380368766.471', '91017414629852252476380368766.47117955844005', 32);
    T('28586.32124747000', '28586.32124747000107561236523943', 16);
    T('0.000001935665545322534195131', '0.0000019356655453225341951305040536808235510260170838860718', 22);
    T('7.8', '7.803563246406851025', 2);
    T('-4.89914223627882382434323e+26', '-489914223627882382434323457.50920109688497974624541155867073', 24);
    T('384718796891211107', '384718796891211107', 18);
    T('42510.74002309897971230194', '42510.7400230989797123019438399853496258', 25);
    T('-7.388e+11', '-738804895894', 4);
    T('-5.0000000e-7', '-0.0000005', 8);
    T('8.364583286198657757274487081e+29', '836458328619865775727448708084.5405786', 28);
    T('-6.24e+26', '-624168076184333836471774105.20710913228879473008695839055', 3);
    T('19804.9', '19804.875536771730958444190952514101502', 6);
    T('-74106.212623408289', '-74106.2126234082888281', 17);
    T('7432670190286.34100080472041', '7432670190286.341000804720411465540223412277267472', 27);
    T('5.3250588635e+21', '5325058863479337983860.6152606488098384817869174221885211', 11);
    T('6.865e+9', '6865129903.657345356274690732979469003170132760589', 4);
    T('5795370.885430786885', '5795370.8854307868847728814464165810658237393757773', 19);
    T('-29172007010418365641.7578738', '-29172007010418365641.7578738219989133084908106406123747833195', 27);
    T('-62322.86188017646654', '-62322.8618801764665355127105700053481622040465444574371', 19);
    T('-6374', '-6373.604850300043463878', 4);
    T('5.846101e+26', '584610089745513547435367965.045755404292155403517947658', 7);
    T('-4.9589e+12', '-4958880864611.79783789828786433416628187354312472853462765', 5);
    T('-976.708', '-976.7080061576', 6);
    T('-6.265387e+7', '-62653867.768253566156', 7);
    T('5.943e+14', '594338013726832.675613519', 4);
    T('5.0e+27', '5018407166428602036582808244', 2);
    T('9e+16', '86282182181939888.936', 1);
    T('-5319867042361146027.570343834017247178243381990233', '-5319867042361146027.5703438340172471782433819902325543283', 49);
    T('-280.611828072', '-280.611828072476084775', 12);
    T('497125.5349688434079217115738652', '497125.5349688434079217115738651759109278602', 31);
    T('-8.74679213554e+15', '-8746792135535203.818773729011249091171163901426235584485964', 12);
    T('2750816434321727711.90126468620', '2750816434321727711.901264686199491277747822638', 30);
    T('-804111355.871490666462196181', '-804111355.8714906664621961811894645876', 27);
    T('5592072638309750852858746183.6506680977', '5592072638309750852858746183.6506680977126', 38);
    T('-4.0e+20', '-400904317147714566361', 2);
    T('-3.9e+26', '-390267222260748697649150634.14444', 2);
    T('43.2', '43.2482', 3);
    T('42334337596496149636254', '42334337596496149636254.4926162509306406461', 23);
    T('-7e+9', '-7246374971.34279698356', 1);
    T('71516263932998764871838469072', '71516263932998764871838469072.280115355524', 29);
    T('71257489.5995227415169007618702182092', '71257489.59952274151690076187021820922744', 36);
    T('268492835', '268492834.77041', 9);
    T('50325.551277778107847798802', '50325.551277778107847798801525', 26);
    T('-5.289303987e+29', '-528930398665449048343281311623.69686', 10);

    BigNumber.config({ROUNDING_MODE : 6});

    T('0.08000', '0.08', 4);
    T('-4.5132e+21', '-4513243388120382069815.8508153058993058875', 5);
    T('-73549', '-73549.2594630551663822238', 5);
    T('1.275868004728922895890883e+29', '127586800472892289589088296800.6', 25);
    T('-0.0003715444034899460421534099962225699000', '-0.0003715444034899460421534099962225699', 37);
    T('-6.9625565265e+24', '-6962556526511822306135536', 11);
    T('1.67583703641e+13', '16758370364138.915293525076269061228714877', 12);
    T('-173594.95064085553515176707313947534918109631092170', '-173594.950640855535151767073139475349181096310921699', 50);
    T('-6.9503965525e+19', '-69503965525000308384.151383', 11);
    T('4.411225e+20', '441122486054080817112', 7);
    T('2.467044064783596937642371770e+31', '24670440647835969376423717700462.39', 28);
    T('3.9711897549481645654e+24', '3971189754948164565361634.8039734590476326224193520402091769', 20);
    T('-1.4757613208690e+21', '-1475761320868963235919.64499841336073105746686372924161', 14);
    T('91683083887068.6191146', '91683083887068.61911461351134520171343337804061135', 21);
    T('-7923074181102822.578', '-7923074181102822.5778', 19);
    T('-6.800e-8', '-0.000000068', 4);
    T('-2.57954671081460000000e-10', '-0.00000000025795467108146', 21);
    T('5.5352911972e-9', '0.000000005535291197169667611325365189624523452', 11);
    T('6.0488358e+8', '604883577', 8);
    T('3', '3', 1);
    T('-4.072637936805672015603149446630136089530560102165', '-4.0726379368056720156031494466301360895305601021653459970194', 49);
    T('-7.2e+10', '-71689970391', 2);
    T('655754242958.1563938760094919', '655754242958.15639387600949190369', 28);
    T('-7.575535014e-9', '-0.00000000757553501363609536678641245355', 10);
    T('7.547067960578900230644488e-10', '0.00000000075470679605789002306444877998602723', 25);
    T('-3.64561456763e+12', '-3645614567625.4', 12);
    T('9.0e-7', '0.0000009', 2);
    T('7e+2', '687', 1);
    T('517277827334839.8174848543680868', '517277827334839.8174848543680868015165926618', 31);
    T('7e+2', '655.46270361324473194', 1);
    T('1632131488313153.49737424823493573157', '1632131488313153.497374248234935731568', 36);
    T('274068317992.5998880719845028748169734442', '274068317992.5998880719845028748169734442394151076', 40);
    T('-7.060e-9', '-0.00000000706025531009734073', 4);
    T('0.004444', '0.0044439457493', 4);
    T('72482770689153111154104782082.023', '72482770689153111154104782082.022764082943227214833851', 32);
    T('5.9130694036072794206e+24', '5913069403607279420613864.152', 20);
    T('843384561300245347961437.966', '843384561300245347961437.96592523791', 27);
    T('0.0000035198821282510000000', '0.000003519882128251', 20);
    T('-1.00371560130267706870097e-9', '-0.00000000100371560130267706870096885251', 24);
    T('17504218.4970302', '17504218.49703016415913667026121376499', 15);
    T('-5e-9', '-0.000000005169058703', 1);
    T('6.922803246e+10', '69228032455', 10);
    T('-16', '-16', 2);
    T('-1.355147513468192707127939151e+40', '-13551475134681927071279391508441439066206.58705380600075', 28);
    T('81670324.1197758695', '81670324.1197758695212865075629796973196504241126', 18);
    T('0.00005', '0.00004797485174640366805332660647', 1);
    T('-4.864397594e-10', '-0.0000000004864397594461335282648538530108953965681345', 10);
    T('47694105.2312532', '47694105.23125322528167211284521303', 15);
    T('-4.962106181e+26', '-496210618135432953927871636.779236', 10);
    T('1.2800030559497062236642e+37', '12800030559497062236641930592334626609.7332', 23);
    T('-574830783.7', '-574830783.6689168903917696583746514637433390929', 10);
    T('5969.431086199057470', '5969.43108619905746956015212970904111744101', 19);
    T('-4.8e+3', '-4814.32904953003285', 2);
    T('4.297e+16', '42973001760252134', 4);
    T('-5.7628e+6', '-5762846.590152347665179652381407653797146356303622218259885', 5);
    T('904864662232032.160612401810317927291657403142932', '904864662232032.16061240181031792729165740314293194205879163', 48);
    T('7.9892e+20', '798923115068265241915.537619430376605', 5);
    T('-8.97759349384000643', '-8.97759349384000643427096282979', 18);
    T('841598023200278219780', '841598023200278219780.04764720909930685', 21);
    T('7.294115e+17', '729411462980818078', 7);
    T('-493854.469231', '-493854.46923056217873', 12);
    T('1.16760483177812e+16', '11676048317781198.761924013', 15);
    T('4.91431629960536e+17', '491431629960536053.49611060493021241774', 15);
    T('-391357204564683265466220678.5248961969115394056441165', '-391357204564683265466220678.524896196911539405644116478', 52);
    T('-1138622.4269179222525707405725062065185867', '-1138622.42691792225257074057250620651858665807616', 41);
    T('7762491414507273289587174.60526', '7762491414507273289587174.60526424654003', 30);
    T('-8.34305e+12', '-8343051798787.85784573983', 6);
    T('-448090139696.5', '-448090139696.540044682', 13);
    T('-249554483941810.04760758280384259798256931579', '-249554483941810.0476075828038425979825693157901967215767', 44);
    T('-4937249656843391.056849', '-4937249656843391.056849458', 22);
    T('-4.90029240789e+24', '-4900292407887576632220011.4', 12);
    T('884134', '884134.30546381722', 6);
    T('-67686.285431006', '-67686.2854310057290328136776917246126204655', 14);
    T('5.1454907927786956678e+21', '5145490792778695667848.5080878826658832100351133', 20);
    T('-3.75540093e+9', '-3755400930.115945946791361377756114557824815082', 9);
    T('790548.1', '790548.055405', 7);
    T('21.9776441681934305611827', '21.9776441681934305611826542081066055', 24);
    T('-8.62915591e+12', '-8629155908036.5010483', 9);
    T('-62521191175', '-62521191175.03721539877599449', 11);
    T('-63947010170235145618893.55048', '-63947010170235145618893.55048264587643', 28);
    T('-4.4791e+5', '-447912.9929543492037', 5);
    T('876897.06887720787797443065', '876897.0688772078779744306464727', 26);
    T('-609834676.749497163216150672711104329822616519', '-609834676.749497163216150672711104329822616518762', 45);
    T('-2.9407315435e+18', '-2940731543474095094.56340709357589521', 11);
    T('243028.94040290384317164750687246', '243028.940402903843171647506872458168411478', 32);
    T('5313610990.737', '5313610990.7373810218', 13);
    T('-3.56e+4', '-35566.4678487', 3);

    T('123', '12.345e1', BigNumber(3));
    T('123.45', '12.345e1', null);
    T('123.45', '12.345e1', undefined);

    assertException(function () {new BigNumber(1.23).toPrecision(NaN)}, "(1.23).toPrecision(NaN)");
    assertException(function () {new BigNumber(1.23).toPrecision('NaN')}, "(1.23).toPrecision('NaN')");
    assertException(function () {new BigNumber(1.23).toPrecision([])}, "(1.23).toPrecision([])");
    assertException(function () {new BigNumber(1.23).toPrecision({})}, "(1.23).toPrecision({})");
    assertException(function () {new BigNumber(1.23).toPrecision('')}, "(1.23).toPrecision('')");
    assertException(function () {new BigNumber(1.23).toPrecision(' ')}, "(1.23).toPrecision(' ')");
    assertException(function () {new BigNumber(1.23).toPrecision('hello')}, "(1.23).toPrecision('hello')");
    assertException(function () {new BigNumber(1.23).toPrecision('\t')}, "(1.23).toPrecision('\t')");
    assertException(function () {new BigNumber(1.23).toPrecision(new Date)}, "(1.23).toPrecision(new Date)");
    assertException(function () {new BigNumber(1.23).toPrecision(new RegExp)}, "(1.23).toPrecision(new RegExp)");
    assertException(function () {new BigNumber(1.23).toPrecision(2.01)}, "(1.23).toPrecision(2.01)");
    assertException(function () {new BigNumber(1.23).toPrecision(10.5)}, "(1.23).toPrecision(10.5)");
    assertException(function () {new BigNumber(1.23).toPrecision('1.1e1')}, "(1.23).toPrecision('1.1e1')");
    assertException(function () {new BigNumber(1.23).toPrecision(true)}, "(1.23).toPrecision(true)");
    assertException(function () {new BigNumber(1.23).toPrecision(false)}, "(1.23).toPrecision(false)");
    assertException(function () {new BigNumber(1.23).toPrecision(function (){})}, "(1.23).toPrecision(function (){})");

    assertException(function () {new BigNumber('12.345e1').toPrecision('-1')}, ".toPrecision('-1')");
    assertException(function () {new BigNumber('12.345e1').toPrecision(-23)}, ".toPrecision(-23)");
    assertException(function () {new BigNumber('12.345e1').toPrecision(1e9 + 1)}, ".toPrecision(1e9 + 1)");
    assertException(function () {new BigNumber('12.345e1').toPrecision(1e9 + 0.1)}, ".toPrecision(1e9 + 0.1)");
    assertException(function () {new BigNumber('12.345e1').toPrecision(0)}, ".toPrecision(0)");
    assertException(function () {new BigNumber('12.345e1').toPrecision('-0')}, ".toPrecision('-0')");
    assertException(function () {new BigNumber('12.345e1').toPrecision(0.9)}, ".toPrecision(0.9)");
    assertException(function () {new BigNumber('12.345e1').toPrecision('-1e-1')}, ".toPrecision('-1e-1')");
    assertException(function () {new BigNumber('12.345e1').toPrecision(Infinity)}, ".toPrecision(Infinity)");
    assertException(function () {new BigNumber('12.345e1').toPrecision('-Infinity')}, ".toPrecision('-Infinity')");

    BigNumber.config({ERRORS : false});

    T('123', '12.345e1', BigNumber(3));
    T('123.45', '12.345e1', null);
    T('123.45', '12.345e1', undefined);
    T('123.45', '12.345e1', NaN);
    T('123.45', '12.345e1', 'NaN');
    T('123.45', '12.345e1', []);
    T('123.45', '12.345e1', {});
    T('123.45', '12.345e1', '');
    T('123.45', '12.345e1', ' ');
    T('123.45', '12.345e1', 'hello');
    T('123.45', '12.345e1', '\t');
    T('123.45', '12.345e1', ' ');
    T('123.45', '12.345e1', new Date);
    T('123.45', '12.345e1', new RegExp);
    T('123.4500', '12.345e1', 7.5);
    T('123.45000000', '12.345e1', '1.1e1');

    T('1e+2', '12.345e1', 1);
    T('123.45', '12.345e1', '-1');
    T('123.45', '12.345e1', -23);
    T('123.45', '12.345e1', 1e9 + 1);
    T('123.45', '12.345e1', 1e9 + 0.1);
    T('123.45', '12.345e1', 0);
    T('123.45', '12.345e1', '-0');
    T('123.45', '12.345e1', 0.9);
    T('123.45', '12.345e1', '-1e-1');
    T('123.45', '12.345e1', Infinity);
    T('123.45', '12.345e1', '-Infinity');

    log('\n ' + passed + ' of ' + total + ' tests passed in ' + (+new Date() - start) + ' ms \n');
    return [passed, total];;
})(this.BigNumber);
if (typeof module !== 'undefined' && module.exports) module.exports = count;