unit unit_star_database;
{HNSKY reads star databases type .290 and 1476}

{Copyright (C) 2017, 2023 by Han Kleijn, www.hnsky.org
 email: han.k.. at...hnsky.org

This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this
file, You can obtain one at https://mozilla.org/MPL/2.0/.   }

{$mode delphi}

interface

uses
  Classes, SysUtils, math;

var
  cos_telescope_dec : double;{store here the cos(telescope_dec) value before and read series}
  database2         : array[0..(11*10)] of ansichar;{info star database, length 110 char equals 10x11 bytes}

// telescope_ra, telescope_dec [radians], contains to center position of the field of interest
// field_diameter [radians], FOV diameter of field of interest. This is ignored in searchmode=T}
// ra, dec [radians],   reported star position
// mag2 [magnitude*10]  reported star magnitude
// result [true/false]  if reported true more stars available. If false no more stars available
// extra outputs:
//          naam2,  string containing the star Tycho/UCAC4 designation for record size above 7
//          database2  : array[0..(11*10)] of ansichar;{text info star database used}
// preconditions:
//   cos_telescope_dec, double variable should contains the cos(telescope_dec) to detect if star read is within the FOV diameter}

function select_star_database(database:string;fov:double): boolean; {select a star database, report false if none is found}
procedure find_areas(ra1,dec1,fov :double; out area1,area2,area3,area4 :integer; out frac1,frac2,frac3,frac4:double);{find up to four star database areas for the square image.  Maximum size is a little lesse the one database field 9.5x9.5 degrees for .290 files and 5.14 x 5.14 degrees for .1476 files}
function readdatabase290(telescope_ra,telescope_dec, field_diameter:double; out ra2,dec2, mag2,Bp_Rp : double): boolean; inline;{star 290 file database search}
procedure close_star_database;{Close the tfilestream}
function open_database(telescope_dec: double; area290: integer): boolean; {open database file}

// The format of the 290 star databases is described in the HNSKY help file
//
// The sky is divided in 290 areas of equal surface except for the poles which are half of that size.
// The stars are stored in these 290 separate files and sorted from bright to faint. Each file starts with a header of 110 bytes of which the first part contains
// a textual description and the last byte contains the record size 5, 6, 7, 10 or 11 bytes.  The source of the utility program to make star databases is provided.
//
// The 290 area's:
// The areas are based on an mathematical method described in a paper of the PHILLIPS LABORATORY called "THE DIVISION OF A CIRCLE OR SPHERICAL SURFACE INTO EQUAL-AREA CELLS OR PIXELS"
// by Irving I. Gringorten Penelope J. Yepez on 30 June 1992
// First circles of constant declination are assumed. The first sphere segment defined by circle with number 1 has a height h1 from the pole and a surface of pi*sqr(h1).
// If the second circle of constant declination has a sphere segment with a height of 3*h1 then the surface area of the second sphere segment is nine times higher equal pi*sqr(3*h1).
// If the area between circle 1 en 2 is divided in 8 segments then these eight have the same area as the area of the first segment.
// The same is possible for the third circle by diving it in 16 segments, then in 24, 32, 40, 48, 56, 64 segments.
// The area of the third segment is pi*sqr(5*h1) so 25 times larger, where 25 equals 1+8+16. So the sphere segments have a height of h1, 3*h1, 5*h1, 7*h1.
// The height of h1=1-sin(declination). All areas are equal area but rectangle.
// In HNSKY all area's are a combination of two except for the polar areas to have a more square shape especially around the equator.
// The south pole is stored in file 0101.290 Area A2 and A3 are stored in file 02_01.290, area A4 and A5 are stored in file 0202.290.
// The distances between the circles is pretty constant and around 10 to 12 degrees. The distance between the area centres is around 15 degrees maximum.
// The declinations are calculated by arcsin (1-1/289), arcsin(1-(1+8)/289), arcsin (1-(1+8+16)/289), arcsin(1-(1+8+16+24)/289)...
//
//     	Ring 	declination 	declination     Areas 	HNSKY
//              minimum         maximum         equal   area's
//                                              size
//   	0-1 	-90 	        -85.23224404 	1 	1          {arcsin(1-1/289)}
//     	1-2 	-85.23224404 	-75.66348756 	8 	4          {arcsin(1-(1+8)/289)}
//	2-3 	-75.66348756 	-65.99286637 	16 	8          {arcsin (1-(1+8+16)/289)}
//	4-5 	-65.99286637 	-56.14497387 	24 	12
//	6-7 	-56.14497387 	-46.03163067 	32 	16
//	7-8 	-46.03163067 	-35.54307745 	40 	20
//	8-9 	-35.54307745 	-24.53348115 	48 	24
//	7-8 	-24.53348115 	-12.79440589 	56 	28
//	8-9 	-12.79440589 	0 	        64 	32
//	9-10 	0 	        12.79440589 	64 	32
//	10-11 	12.79440589 	24.53348115 	56 	28
//	11-12 	24.53348115 	35.54307745 	48 	24
//	12-13 	35.54307745 	46.03163067 	40 	20
//	13-14 	46.03163067 	56.14497387 	32 	16
//	14-15 	56.14497387 	65.99286637 	24 	12
//	15-16 	65.99286637 	75.66348756 	16 	8
//	16-17 	75.66348756 	85.23224404 	8 	4
//	17-18 	85.23224404 	90 	        1 	1
//    ----------------------------------------------------
//                              Total   	578 	290



// The new 1476 format
// The sky is devided in 1476 areas of around 5 degrees (90/17.5) using ring of constant declination=DEC.
// Each ring is divided by lines of constant RA such that the minimum width in RA is about 5 degrees (deltaRA*cos(DEC).


//	Declination  	Declination 	Ring	RA 	RA step	        RA step 	DEC step                        Files
//      minimum         maximum                 cells   distance        distance        distance
//                                                      north[degr]     south[degr]
//	-90.00000000	-87.42857143	0-1	1                                                                       0101.1476
//	-87.42857143	-82.28571429	1-2	3	5.38377964	16.10799190	-2.57142857  {=90/(2*17.5)}     0201.1476,  0202.1476, 0203.1476
//	-82.28571429	-77.14285714	2-3	9	5.36933063	8.90083736	-5.14285714  {=90/17.5}         0301.1476, . . . . . , 0309.1476
//	-77.14285714	-72.00000000	3-4	15	5.34050241	7.41640786	-5.14285714  {=90/17.5}         0401.1476, . . . . . , 0415.1476
//	-72.00000000	-66.85714286	4-5	21	5.29743419	6.73757197	-5.14285714                     0501.1476, . . . . . , 0521.1476
//	-66.85714286	-61.71428571	5-6	27	5.24033376	6.31824883	-5.14285714                     06 . . .
//	-61.71428571	-56.57142857	6-7	33	5.16947632	6.00978525	-5.14285714                     07 . . .
//	-56.57142857	-51.42857143	7-8	38	5.21902403	5.90674549	-5.14285714                     08 . . .
//	-51.42857143	-46.28571429	8-9	43	5.21991462	5.78564078	-5.14285714                     09 . . .
//	-46.28571429	-41.14285714	9-10	48	5.18296987	5.64803600	-5.14285714                     10 . . .
//	-41.14285714	-36.00000000	10-11	52	5.21357169	5.60088688	-5.14285714                     11 . . .
//	-36.00000000	-30.85714286	11-12	56	5.20082354	5.51859939	-5.14285714                     12 . . .
//	-30.85714286	-25.71428571	12-13	60	5.15069276	5.40581321	-5.14285714                     13 . . .
//	-25.71428571	-20.57142857	13-14	63	5.14839353	5.34991355	-5.14285714                     14 . . .
//	-20.57142857	-15.42857143	14-15	65	5.18530082	5.33887123	-5.14285714                     15 . . .
//	-15.42857143	-10.28571429	15-16	67	5.17950194	5.28678585	-5.14285714                     16 . . .
//	-10.28571429	-5.14285714	16-17	68	5.20903900	5.27280509	-5.14285714                     17 . . .
//	-5.14285714	0.00000000	17-18	69	5.19638762	5.21739130	-5.14285714                     18 . . .
//
//	0.00000000	5.14285714	18-19	69	5.21739130	5.19638762	-5.14285714                     19 . . .
//	5.14285714	10.28571429	19-20	68	5.27280509	5.20903900	-5.14285714                     20 . . .
//	10.28571429	15.42857143	20-21	67	5.28678585	5.17950194	-5.14285714                     21 . . .
//	15.42857143	20.57142857	21-22	65	5.33887123	5.18530082	-5.14285714                     22 . . .
//	20.57142857	25.71428571	22-23	63	5.34991355	5.14839353	-5.14285714                     23 . . .
//	25.71428571	30.85714286	23-24	60	5.40581321	5.15069276	-5.14285714                     24 . . .
//	30.85714286	36.00000000	24-25	56	5.51859939	5.20082354	-5.14285714                     25 . . .
//	36.00000000	41.14285714	25-26	52	5.60088688	5.21357169	-5.14285714                     26 . . .
//	41.14285714	46.28571429	26-27	48	5.64803600	5.18296987	-5.14285714                     27 . . .
//	46.28571429	51.42857143	27-28	43	5.78564078	5.21991462	-5.14285714                     28 . . .
//	51.42857143	56.57142857	28-29	38	5.90674549	5.21902403	-5.14285714                     29 . . .
//	56.57142857	61.71428571	29-30	33	6.00978525	5.16947632	-5.14285714                     30 . . .
//	61.71428571	66.85714286	30-31	27	6.31824883	5.24033376	-5.14285714                     31 . . .
//	66.85714286	72.00000000	31-32	21	6.73757197	5.29743419	-5.14285714                     32 . . .
//	72.00000000	77.14285714	32-33	15	7.41640786	5.34050241	-5.14285714                     33 . . .
//	77.14285714	82.28571429	33-34	9	8.90083736	5.36933063	-5.14285714                     34 . . .
//	82.28571429	87.42857143	34-35	3	16.10799190	5.38377964	-5.14285714                     3501.1476, 3502.1476, 3503.1476
//	87.42857143	90.00000000	36-37	1                                                                       3601.1476
//
//
//			                Total  	1476 cells/areas

//The 1476 star records are the same as for the 290 file format.


{Magnitude: The stars are sorted with an accuracy of 0.1 magnitude. Prior to each group a special record is written where RA is $FFFFFF and DEC contains the magnitude}

type
  hnskyhdr1476_6 = packed record  {G16 for storing Rp-Bp. This format is the same as 290_5 but Gaia color information added in an extra shortint}
             ra7 : byte; {The RA is stored as a 3 bytes word. The DEC position is stored as a two's complement (=standard), three bytes integer. The resolution of this three byte storage will be for RA: 360*60*60/((256*256*256)-1) = 0.077 arc seconds. For the DEC value it will be: 90*60*60/((128*256*256)-1) = 0.039 arc seconds.}
             ra8 : byte;
             ra9 : byte;
             dec7: byte;
             dec8: byte;
             B_R: shortint;{Gaia Bp-Rp}
   end;
  hnskyhdr1476_5 = packed record  {Most compact format, used for Gaia}
              ra7 : byte;
              ra8 : byte;
              ra9 : byte;
              dec7: byte;
              dec8: byte;
  end;

// The .290 format divides the sky in 290 area's and 290 corresponding files with the extension .290.  The 290 areas of similar area size except for the poles which are half of that size.  Each star is stored in a record of 5 bytes. The files start 110 byte header with textual description and the record size binary stored in byte 110.
// The .1476 format divides the sky in 1476 area's and 1476 corresponding files with the extension .1476.  The 1476 areas of similar area size except for the poles which are half of that size.  Each star is stored in a record of 5 bytes. The files start 110 byte header with textual description and the record size binary stored in byte 110.
//
// The RA is stored as a 3 bytes word. The DEC position is stored as a two's complement (=standard), three bytes integer. The resolution of this three byte storage will be for RA: 360*60*60/((256*256*256)-1) = 0.077 arc seconds. For the DEC value it will be: 90*60*60/((128*256*256)-1) = 0.039 arc seconds.  The starts are sorted on magnitude and the magnitude is stored in a byte of the special preceding header with an offset to make all values positive.
//
// Example of star Sirius RA and DEC position:
// The RA position is stored as C3 06 48 equals: (195+6*256+72*256*256)*24/((256*256*256)-1)=6.75247662 hours equals: 6:45:8.9
// The DEC position is stored as D7 39 E8, equals: 215 57 -24. The DEC is then (215+57*256-24*256*256)*90/((128*256*256)-1)=-16.7161401 degrees equals -16d 42 58
//
// Stars are sorted from bright to faint in the "0.1" magnitude steps. Within the magnitude range, the stars are additional sorted in DEC. For a series of stars with the same DEC9 value, a header-record
// is preceding containing the DEC9 value stored at location DEC7. Since the stars are already sorted in 290 areas, the number of DEC9 values is already limited by a factor 18.
// 1476-5 header-record example: FF FF FF 20 06 This indicates the following records have a DEC9 value of 20 -128 offset and a magnitude of (06 - 16)/10 equals -1.0 (new method, +16 offset).
//
//The shorter records methods become only space efficient for very large star collection of a few million stars. In these large collections many stars can be found with the same magnitude and DEC9 shortint. The Gaia database is only issued in the 1476-5 format of 5 bytes per star. or in an older format 290-6 (G05) or 290-5. The 290-6 has one more byte for the colour information. This is documented in the HNSKY planetarium program help file.

//So the record sequence will be as follows:
//
//header-record {new section will start with a different magnitude and dec9}
//record
//record
//record
//record
//header-record  {new section will start with a different magnitude and dec9}
//record
//record



var
  name_database         : string;{name star database}
  cache_valid_pos       : integer;
  database_type         : integer;

var {################# initialised variables #########################}
  file_open: boolean=false;{file is not open}
  area2    : double=1*pi/180; {search area}
  old_area  : integer=9999999;
  cache_position : integer=0;

implementation

uses forms,astap_main, unit_stack {for memo2_message};

var {################# initialised variables #########################}

filenames290 : array[1..290] of string= {}
(('0101.290'),

 ('0201.290'),  {combined cells from 8}
 ('0202.290'),
 ('0203.290'),
 ('0204.290'),

 ('0301.290'),
 ('0302.290'),
 ('0303.290'),
 ('0304.290'),
 ('0305.290'),
 ('0306.290'),
 ('0307.290'),
 ('0308.290'),

 ('0401.290'),
 ('0402.290'),
 ('0403.290'),
 ('0404.290'),
 ('0405.290'),
 ('0406.290'),
 ('0407.290'),
 ('0408.290'),
 ('0409.290'),
 ('0410.290'),
 ('0411.290'),
 ('0412.290'),

 ('0501.290'),
 ('0502.290'),
 ('0503.290'),
 ('0504.290'),
 ('0505.290'),
 ('0506.290'),
 ('0507.290'),
 ('0508.290'),
 ('0509.290'),
 ('0510.290'),
 ('0511.290'),
 ('0512.290'),
 ('0513.290'),
 ('0514.290'),
 ('0515.290'),
 ('0516.290'),

 ('0601.290'),
 ('0602.290'),
 ('0603.290'),
 ('0604.290'),
 ('0605.290'),
 ('0606.290'),
 ('0607.290'),
 ('0608.290'),
 ('0609.290'),
 ('0610.290'),
 ('0611.290'),
 ('0612.290'),
 ('0613.290'),
 ('0614.290'),
 ('0615.290'),
 ('0616.290'),
 ('0617.290'),
 ('0618.290'),
 ('0619.290'),
 ('0620.290'),

 ('0701.290'),
 ('0702.290'),
 ('0703.290'),
 ('0704.290'),
 ('0705.290'),
 ('0706.290'),
 ('0707.290'),
 ('0708.290'),
 ('0709.290'),
 ('0710.290'),
 ('0711.290'),
 ('0712.290'),
 ('0713.290'),
 ('0714.290'),
 ('0715.290'),
 ('0716.290'),
 ('0717.290'),
 ('0718.290'),
 ('0719.290'),
 ('0720.290'),
 ('0721.290'),
 ('0722.290'),
 ('0723.290'),
 ('0724.290'),

 ('0801.290'),
 ('0802.290'),
 ('0803.290'),
 ('0804.290'),
 ('0805.290'),
 ('0806.290'),
 ('0807.290'),
 ('0808.290'),
 ('0809.290'),
 ('0810.290'),
 ('0811.290'),
 ('0812.290'),
 ('0813.290'),
 ('0814.290'),
 ('0815.290'),
 ('0816.290'),
 ('0817.290'),
 ('0818.290'),
 ('0819.290'),
 ('0820.290'),
 ('0821.290'),
 ('0822.290'),
 ('0823.290'),
 ('0824.290'),
 ('0825.290'),
 ('0826.290'),
 ('0827.290'),
 ('0828.290'),

 ('0901.290'),
 ('0902.290'),
 ('0903.290'),
 ('0904.290'),
 ('0905.290'),
 ('0906.290'),
 ('0907.290'),
 ('0908.290'),
 ('0909.290'),
 ('0910.290'),
 ('0911.290'),
 ('0912.290'),
 ('0913.290'),
 ('0914.290'),
 ('0915.290'),
 ('0916.290'),
 ('0917.290'),
 ('0918.290'),
 ('0919.290'),
 ('0920.290'),
 ('0921.290'),
 ('0922.290'),
 ('0923.290'),
 ('0924.290'),
 ('0925.290'),
 ('0926.290'),
 ('0927.290'),
 ('0928.290'),
 ('0929.290'),
 ('0930.290'),
 ('0931.290'),
 ('0932.290'),

 ('1001.290'),
 ('1002.290'),
 ('1003.290'),
 ('1004.290'),
 ('1005.290'),
 ('1006.290'),
 ('1007.290'),
 ('1008.290'),
 ('1009.290'),
 ('1010.290'),
 ('1011.290'),
 ('1012.290'),
 ('1013.290'),
 ('1014.290'),
 ('1015.290'),
 ('1016.290'),
 ('1017.290'),
 ('1018.290'),
 ('1019.290'),
 ('1020.290'),
 ('1021.290'),
 ('1022.290'),
 ('1023.290'),
 ('1024.290'),
 ('1025.290'),
 ('1026.290'),
 ('1027.290'),
 ('1028.290'),
 ('1029.290'),
 ('1030.290'),
 ('1031.290'),
 ('1032.290'),

 ('1101.290'),
 ('1102.290'),
 ('1103.290'),
 ('1104.290'),
 ('1105.290'),
 ('1106.290'),
 ('1107.290'),
 ('1108.290'),
 ('1109.290'),
 ('1110.290'),
 ('1111.290'),
 ('1112.290'),
 ('1113.290'),
 ('1114.290'),
 ('1115.290'),
 ('1116.290'),
 ('1117.290'),
 ('1118.290'),
 ('1119.290'),
 ('1120.290'),
 ('1121.290'),
 ('1122.290'),
 ('1123.290'),
 ('1124.290'),
 ('1125.290'),
 ('1126.290'),
 ('1127.290'),
 ('1128.290'),

 ('1201.290'),
 ('1202.290'),
 ('1203.290'),
 ('1204.290'),
 ('1205.290'),
 ('1206.290'),
 ('1207.290'),
 ('1208.290'),
 ('1209.290'),
 ('1210.290'),
 ('1211.290'),
 ('1212.290'),
 ('1213.290'),
 ('1214.290'),
 ('1215.290'),
 ('1216.290'),
 ('1217.290'),
 ('1218.290'),
 ('1219.290'),
 ('1220.290'),
 ('1221.290'),
 ('1222.290'),
 ('1223.290'),
 ('1224.290'),

 ('1301.290'),
 ('1302.290'),
 ('1303.290'),
 ('1304.290'),
 ('1305.290'),
 ('1306.290'),
 ('1307.290'),
 ('1308.290'),
 ('1309.290'),
 ('1310.290'),
 ('1311.290'),
 ('1312.290'),
 ('1313.290'),
 ('1314.290'),
 ('1315.290'),
 ('1316.290'),
 ('1317.290'),
 ('1318.290'),
 ('1319.290'),
 ('1320.290'),

 ('1401.290'),
 ('1402.290'),
 ('1403.290'),
 ('1404.290'),
 ('1405.290'),
 ('1406.290'),
 ('1407.290'),
 ('1408.290'),
 ('1409.290'),
 ('1410.290'),
 ('1411.290'),
 ('1412.290'),
 ('1413.290'),
 ('1414.290'),
 ('1415.290'),
 ('1416.290'),

 ('1501.290'),
 ('1502.290'),
 ('1503.290'),
 ('1504.290'),
 ('1505.290'),
 ('1506.290'),
 ('1507.290'),
 ('1508.290'),
 ('1509.290'),
 ('1510.290'),
 ('1511.290'),
 ('1512.290'),

 ('1601.290'),
 ('1602.290'),
 ('1603.290'),
 ('1604.290'),
 ('1605.290'),
 ('1606.290'),
 ('1607.290'),
 ('1608.290'),


 ('1701.290'),
 ('1702.290'),
 ('1703.290'),
 ('1704.290'),

 ('1801.290'));

 dec_boundaries : array[0..18] of double=
    ((-90         * pi/180),
     (-85.23224404* pi/180), {arcsin(1-1/289)}
     (-75.66348756* pi/180), {arcsin(1-(1+8)/289)}
     (-65.99286637* pi/180), {arcsin(1-(1+8+16)/289)}
     (-56.14497387* pi/180), {arcsin(1-(1+8+16+24)/289)}
     (-46.03163067* pi/180),
     (-35.54307745* pi/180),
     (-24.53348115* pi/180),
     (-12.79440589* pi/180),
     (0),
     (12.79440589* pi/180),
     (24.53348115* pi/180),
     (35.54307745* pi/180),
     (46.03163067* pi/180),
     (56.14497387* pi/180),
     (65.99286637* pi/180),
     (75.66348756* pi/180),
     (85.23224404* pi/180),
     (90         * pi/180) );



 filenames1476 : array[1..1476] of string= {}
    (('0101.1476'),

     ('0201.1476'),  {combined cells from 8}
     ('0202.1476'),
     ('0203.1476'),

     ('0301.1476'),
     ('0302.1476'),
     ('0303.1476'),
     ('0304.1476'),
     ('0305.1476'),
     ('0306.1476'),
     ('0307.1476'),
     ('0308.1476'),
     ('0309.1476'),

     ('0401.1476'),
     ('0402.1476'),
     ('0403.1476'),
     ('0404.1476'),
     ('0405.1476'),
     ('0406.1476'),
     ('0407.1476'),
     ('0408.1476'),
     ('0409.1476'),
     ('0410.1476'),
     ('0411.1476'),
     ('0412.1476'),
     ('0413.1476'),
     ('0414.1476'),
     ('0415.1476'),

     ('0501.1476'),
     ('0502.1476'),
     ('0503.1476'),
     ('0504.1476'),
     ('0505.1476'),
     ('0506.1476'),
     ('0507.1476'),
     ('0508.1476'),
     ('0509.1476'),
     ('0510.1476'),
     ('0511.1476'),
     ('0512.1476'),
     ('0513.1476'),
     ('0514.1476'),
     ('0515.1476'),
     ('0516.1476'),
     ('0517.1476'),
     ('0518.1476'),
     ('0519.1476'),
     ('0520.1476'),
     ('0521.1476'),


     ('0601.1476'),
     ('0602.1476'),
     ('0603.1476'),
     ('0604.1476'),
     ('0605.1476'),
     ('0606.1476'),
     ('0607.1476'),
     ('0608.1476'),
     ('0609.1476'),
     ('0610.1476'),
     ('0611.1476'),
     ('0612.1476'),
     ('0613.1476'),
     ('0614.1476'),
     ('0615.1476'),
     ('0616.1476'),
     ('0617.1476'),
     ('0618.1476'),
     ('0619.1476'),
     ('0620.1476'),
     ('0621.1476'),
     ('0622.1476'),
     ('0623.1476'),
     ('0624.1476'),
     ('0625.1476'),
     ('0626.1476'),
     ('0627.1476'),


     ('0701.1476'),
     ('0702.1476'),
     ('0703.1476'),
     ('0704.1476'),
     ('0705.1476'),
     ('0706.1476'),
     ('0707.1476'),
     ('0708.1476'),
     ('0709.1476'),
     ('0710.1476'),
     ('0711.1476'),
     ('0712.1476'),
     ('0713.1476'),
     ('0714.1476'),
     ('0715.1476'),
     ('0716.1476'),
     ('0717.1476'),
     ('0718.1476'),
     ('0719.1476'),
     ('0720.1476'),
     ('0721.1476'),
     ('0722.1476'),
     ('0723.1476'),
     ('0724.1476'),
     ('0725.1476'),
     ('0726.1476'),
     ('0727.1476'),
     ('0728.1476'),
     ('0729.1476'),
     ('0730.1476'),
     ('0731.1476'),
     ('0732.1476'),
     ('0733.1476'),

     ('0801.1476'),
     ('0802.1476'),
     ('0803.1476'),
     ('0804.1476'),
     ('0805.1476'),
     ('0806.1476'),
     ('0807.1476'),
     ('0808.1476'),
     ('0809.1476'),
     ('0810.1476'),
     ('0811.1476'),
     ('0812.1476'),
     ('0813.1476'),
     ('0814.1476'),
     ('0815.1476'),
     ('0816.1476'),
     ('0817.1476'),
     ('0818.1476'),
     ('0819.1476'),
     ('0820.1476'),
     ('0821.1476'),
     ('0822.1476'),
     ('0823.1476'),
     ('0824.1476'),
     ('0825.1476'),
     ('0826.1476'),
     ('0827.1476'),
     ('0828.1476'),
     ('0829.1476'),
     ('0830.1476'),
     ('0831.1476'),
     ('0832.1476'),
     ('0833.1476'),
     ('0834.1476'),
     ('0835.1476'),
     ('0836.1476'),
     ('0837.1476'),
     ('0838.1476'),


     ('0901.1476'),
     ('0902.1476'),
     ('0903.1476'),
     ('0904.1476'),
     ('0905.1476'),
     ('0906.1476'),
     ('0907.1476'),
     ('0908.1476'),
     ('0909.1476'),
     ('0910.1476'),
     ('0911.1476'),
     ('0912.1476'),
     ('0913.1476'),
     ('0914.1476'),
     ('0915.1476'),
     ('0916.1476'),
     ('0917.1476'),
     ('0918.1476'),
     ('0919.1476'),
     ('0920.1476'),
     ('0921.1476'),
     ('0922.1476'),
     ('0923.1476'),
     ('0924.1476'),
     ('0925.1476'),
     ('0926.1476'),
     ('0927.1476'),
     ('0928.1476'),
     ('0929.1476'),
     ('0930.1476'),
     ('0931.1476'),
     ('0932.1476'),
     ('0933.1476'),
     ('0934.1476'),
     ('0935.1476'),
     ('0936.1476'),
     ('0937.1476'),
     ('0938.1476'),
     ('0939.1476'),
     ('0940.1476'),
     ('0941.1476'),
     ('0942.1476'),
     ('0943.1476'),


     ('1001.1476'),
     ('1002.1476'),
     ('1003.1476'),
     ('1004.1476'),
     ('1005.1476'),
     ('1006.1476'),
     ('1007.1476'),
     ('1008.1476'),
     ('1009.1476'),
     ('1010.1476'),
     ('1011.1476'),
     ('1012.1476'),
     ('1013.1476'),
     ('1014.1476'),
     ('1015.1476'),
     ('1016.1476'),
     ('1017.1476'),
     ('1018.1476'),
     ('1019.1476'),
     ('1020.1476'),
     ('1021.1476'),
     ('1022.1476'),
     ('1023.1476'),
     ('1024.1476'),
     ('1025.1476'),
     ('1026.1476'),
     ('1027.1476'),
     ('1028.1476'),
     ('1029.1476'),
     ('1030.1476'),
     ('1031.1476'),
     ('1032.1476'),
     ('1033.1476'),
     ('1034.1476'),
     ('1035.1476'),
     ('1036.1476'),
     ('1037.1476'),
     ('1038.1476'),
     ('1039.1476'),
     ('1040.1476'),
     ('1041.1476'),
     ('1042.1476'),
     ('1043.1476'),
     ('1044.1476'),
     ('1045.1476'),
     ('1046.1476'),
     ('1047.1476'),
     ('1048.1476'),


     ('1101.1476'),
     ('1102.1476'),
     ('1103.1476'),
     ('1104.1476'),
     ('1105.1476'),
     ('1106.1476'),
     ('1107.1476'),
     ('1108.1476'),
     ('1109.1476'),
     ('1110.1476'),
     ('1111.1476'),
     ('1112.1476'),
     ('1113.1476'),
     ('1114.1476'),
     ('1115.1476'),
     ('1116.1476'),
     ('1117.1476'),
     ('1118.1476'),
     ('1119.1476'),
     ('1120.1476'),
     ('1121.1476'),
     ('1122.1476'),
     ('1123.1476'),
     ('1124.1476'),
     ('1125.1476'),
     ('1126.1476'),
     ('1127.1476'),
     ('1128.1476'),
     ('1129.1476'),
     ('1130.1476'),
     ('1131.1476'),
     ('1132.1476'),
     ('1133.1476'),
     ('1134.1476'),
     ('1135.1476'),
     ('1136.1476'),
     ('1137.1476'),
     ('1138.1476'),
     ('1139.1476'),
     ('1140.1476'),
     ('1141.1476'),
     ('1142.1476'),
     ('1143.1476'),
     ('1144.1476'),
     ('1145.1476'),
     ('1146.1476'),
     ('1147.1476'),
     ('1148.1476'),
     ('1149.1476'),
     ('1150.1476'),
     ('1151.1476'),
     ('1152.1476'),

     ('1201.1476'),
     ('1202.1476'),
     ('1203.1476'),
     ('1204.1476'),
     ('1205.1476'),
     ('1206.1476'),
     ('1207.1476'),
     ('1208.1476'),
     ('1209.1476'),
     ('1210.1476'),
     ('1211.1476'),
     ('1212.1476'),
     ('1213.1476'),
     ('1214.1476'),
     ('1215.1476'),
     ('1216.1476'),
     ('1217.1476'),
     ('1218.1476'),
     ('1219.1476'),
     ('1220.1476'),
     ('1221.1476'),
     ('1222.1476'),
     ('1223.1476'),
     ('1224.1476'),
     ('1225.1476'),
     ('1226.1476'),
     ('1227.1476'),
     ('1228.1476'),
     ('1229.1476'),
     ('1230.1476'),
     ('1231.1476'),
     ('1232.1476'),
     ('1233.1476'),
     ('1234.1476'),
     ('1235.1476'),
     ('1236.1476'),
     ('1237.1476'),
     ('1238.1476'),
     ('1239.1476'),
     ('1240.1476'),
     ('1241.1476'),
     ('1242.1476'),
     ('1243.1476'),
     ('1244.1476'),
     ('1245.1476'),
     ('1246.1476'),
     ('1247.1476'),
     ('1248.1476'),
     ('1249.1476'),
     ('1250.1476'),
     ('1251.1476'),
     ('1252.1476'),
     ('1253.1476'),
     ('1254.1476'),
     ('1255.1476'),
     ('1256.1476'),


     ('1301.1476'),
     ('1302.1476'),
     ('1303.1476'),
     ('1304.1476'),
     ('1305.1476'),
     ('1306.1476'),
     ('1307.1476'),
     ('1308.1476'),
     ('1309.1476'),
     ('1310.1476'),
     ('1311.1476'),
     ('1312.1476'),
     ('1313.1476'),
     ('1314.1476'),
     ('1315.1476'),
     ('1316.1476'),
     ('1317.1476'),
     ('1318.1476'),
     ('1319.1476'),
     ('1320.1476'),
     ('1321.1476'),
     ('1322.1476'),
     ('1323.1476'),
     ('1324.1476'),
     ('1325.1476'),
     ('1326.1476'),
     ('1327.1476'),
     ('1328.1476'),
     ('1329.1476'),
     ('1330.1476'),
     ('1331.1476'),
     ('1332.1476'),
     ('1333.1476'),
     ('1334.1476'),
     ('1335.1476'),
     ('1336.1476'),
     ('1337.1476'),
     ('1338.1476'),
     ('1339.1476'),
     ('1340.1476'),
     ('1341.1476'),
     ('1342.1476'),
     ('1343.1476'),
     ('1344.1476'),
     ('1345.1476'),
     ('1346.1476'),
     ('1347.1476'),
     ('1348.1476'),
     ('1349.1476'),
     ('1350.1476'),
     ('1351.1476'),
     ('1352.1476'),
     ('1353.1476'),
     ('1354.1476'),
     ('1355.1476'),
     ('1356.1476'),
     ('1357.1476'),
     ('1358.1476'),
     ('1359.1476'),
     ('1360.1476'),


     ('1401.1476'),
     ('1402.1476'),
     ('1403.1476'),
     ('1404.1476'),
     ('1405.1476'),
     ('1406.1476'),
     ('1407.1476'),
     ('1408.1476'),
     ('1409.1476'),
     ('1410.1476'),
     ('1411.1476'),
     ('1412.1476'),
     ('1413.1476'),
     ('1414.1476'),
     ('1415.1476'),
     ('1416.1476'),
     ('1417.1476'),
     ('1418.1476'),
     ('1419.1476'),
     ('1420.1476'),
     ('1421.1476'),
     ('1422.1476'),
     ('1423.1476'),
     ('1424.1476'),
     ('1425.1476'),
     ('1426.1476'),
     ('1427.1476'),
     ('1428.1476'),
     ('1429.1476'),
     ('1430.1476'),
     ('1431.1476'),
     ('1432.1476'),
     ('1433.1476'),
     ('1434.1476'),
     ('1435.1476'),
     ('1436.1476'),
     ('1437.1476'),
     ('1438.1476'),
     ('1439.1476'),
     ('1440.1476'),
     ('1441.1476'),
     ('1442.1476'),
     ('1443.1476'),
     ('1444.1476'),
     ('1445.1476'),
     ('1446.1476'),
     ('1447.1476'),
     ('1448.1476'),
     ('1449.1476'),
     ('1450.1476'),
     ('1451.1476'),
     ('1452.1476'),
     ('1453.1476'),
     ('1454.1476'),
     ('1455.1476'),
     ('1456.1476'),
     ('1457.1476'),
     ('1458.1476'),
     ('1459.1476'),
     ('1460.1476'),
     ('1461.1476'),
     ('1462.1476'),
     ('1463.1476'),


     ('1501.1476'),
     ('1502.1476'),
     ('1503.1476'),
     ('1504.1476'),
     ('1505.1476'),
     ('1506.1476'),
     ('1507.1476'),
     ('1508.1476'),
     ('1509.1476'),
     ('1510.1476'),
     ('1511.1476'),
     ('1512.1476'),
     ('1513.1476'),
     ('1514.1476'),
     ('1515.1476'),
     ('1516.1476'),
     ('1517.1476'),
     ('1518.1476'),
     ('1519.1476'),
     ('1520.1476'),
     ('1521.1476'),
     ('1522.1476'),
     ('1523.1476'),
     ('1524.1476'),
     ('1525.1476'),
     ('1526.1476'),
     ('1527.1476'),
     ('1528.1476'),
     ('1529.1476'),
     ('1530.1476'),
     ('1531.1476'),
     ('1532.1476'),
     ('1533.1476'),
     ('1534.1476'),
     ('1535.1476'),
     ('1536.1476'),
     ('1537.1476'),
     ('1538.1476'),
     ('1539.1476'),
     ('1540.1476'),
     ('1541.1476'),
     ('1542.1476'),
     ('1543.1476'),
     ('1544.1476'),
     ('1545.1476'),
     ('1546.1476'),
     ('1547.1476'),
     ('1548.1476'),
     ('1549.1476'),
     ('1550.1476'),
     ('1551.1476'),
     ('1552.1476'),
     ('1553.1476'),
     ('1554.1476'),
     ('1555.1476'),
     ('1556.1476'),
     ('1557.1476'),
     ('1558.1476'),
     ('1559.1476'),
     ('1560.1476'),
     ('1561.1476'),
     ('1562.1476'),
     ('1563.1476'),
     ('1564.1476'),
     ('1565.1476'),


     ('1601.1476'),
     ('1602.1476'),
     ('1603.1476'),
     ('1604.1476'),
     ('1605.1476'),
     ('1606.1476'),
     ('1607.1476'),
     ('1608.1476'),
     ('1609.1476'),
     ('1610.1476'),
     ('1611.1476'),
     ('1612.1476'),
     ('1613.1476'),
     ('1614.1476'),
     ('1615.1476'),
     ('1616.1476'),
     ('1617.1476'),
     ('1618.1476'),
     ('1619.1476'),
     ('1620.1476'),
     ('1621.1476'),
     ('1622.1476'),
     ('1623.1476'),
     ('1624.1476'),
     ('1625.1476'),
     ('1626.1476'),
     ('1627.1476'),
     ('1628.1476'),
     ('1629.1476'),
     ('1630.1476'),
     ('1631.1476'),
     ('1632.1476'),
     ('1633.1476'),
     ('1634.1476'),
     ('1635.1476'),
     ('1636.1476'),
     ('1637.1476'),
     ('1638.1476'),
     ('1639.1476'),
     ('1640.1476'),
     ('1641.1476'),
     ('1642.1476'),
     ('1643.1476'),
     ('1644.1476'),
     ('1645.1476'),
     ('1646.1476'),
     ('1647.1476'),
     ('1648.1476'),
     ('1649.1476'),
     ('1650.1476'),
     ('1651.1476'),
     ('1652.1476'),
     ('1653.1476'),
     ('1654.1476'),
     ('1655.1476'),
     ('1656.1476'),
     ('1657.1476'),
     ('1658.1476'),
     ('1659.1476'),
     ('1660.1476'),
     ('1661.1476'),
     ('1662.1476'),
     ('1663.1476'),
     ('1664.1476'),
     ('1665.1476'),
     ('1666.1476'),
     ('1667.1476'),


     ('1701.1476'),
     ('1702.1476'),
     ('1703.1476'),
     ('1704.1476'),
     ('1705.1476'),
     ('1706.1476'),
     ('1707.1476'),
     ('1708.1476'),
     ('1709.1476'),
     ('1710.1476'),
     ('1711.1476'),
     ('1712.1476'),
     ('1713.1476'),
     ('1714.1476'),
     ('1715.1476'),
     ('1716.1476'),
     ('1717.1476'),
     ('1718.1476'),
     ('1719.1476'),
     ('1720.1476'),
     ('1721.1476'),
     ('1722.1476'),
     ('1723.1476'),
     ('1724.1476'),
     ('1725.1476'),
     ('1726.1476'),
     ('1727.1476'),
     ('1728.1476'),
     ('1729.1476'),
     ('1730.1476'),
     ('1731.1476'),
     ('1732.1476'),
     ('1733.1476'),
     ('1734.1476'),
     ('1735.1476'),
     ('1736.1476'),
     ('1737.1476'),
     ('1738.1476'),
     ('1739.1476'),
     ('1740.1476'),
     ('1741.1476'),
     ('1742.1476'),
     ('1743.1476'),
     ('1744.1476'),
     ('1745.1476'),
     ('1746.1476'),
     ('1747.1476'),
     ('1748.1476'),
     ('1749.1476'),
     ('1750.1476'),
     ('1751.1476'),
     ('1752.1476'),
     ('1753.1476'),
     ('1754.1476'),
     ('1755.1476'),
     ('1756.1476'),
     ('1757.1476'),
     ('1758.1476'),
     ('1759.1476'),
     ('1760.1476'),
     ('1761.1476'),
     ('1762.1476'),
     ('1763.1476'),
     ('1764.1476'),
     ('1765.1476'),
     ('1766.1476'),
     ('1767.1476'),
     ('1768.1476'),


     ('1801.1476'),
     ('1802.1476'),
     ('1803.1476'),
     ('1804.1476'),
     ('1805.1476'),
     ('1806.1476'),
     ('1807.1476'),
     ('1808.1476'),
     ('1809.1476'),
     ('1810.1476'),
     ('1811.1476'),
     ('1812.1476'),
     ('1813.1476'),
     ('1814.1476'),
     ('1815.1476'),
     ('1816.1476'),
     ('1817.1476'),
     ('1818.1476'),
     ('1819.1476'),
     ('1820.1476'),
     ('1821.1476'),
     ('1822.1476'),
     ('1823.1476'),
     ('1824.1476'),
     ('1825.1476'),
     ('1826.1476'),
     ('1827.1476'),
     ('1828.1476'),
     ('1829.1476'),
     ('1830.1476'),
     ('1831.1476'),
     ('1832.1476'),
     ('1833.1476'),
     ('1834.1476'),
     ('1835.1476'),
     ('1836.1476'),
     ('1837.1476'),
     ('1838.1476'),
     ('1839.1476'),
     ('1840.1476'),
     ('1841.1476'),
     ('1842.1476'),
     ('1843.1476'),
     ('1844.1476'),
     ('1845.1476'),
     ('1846.1476'),
     ('1847.1476'),
     ('1848.1476'),
     ('1849.1476'),
     ('1850.1476'),
     ('1851.1476'),
     ('1852.1476'),
     ('1853.1476'),
     ('1854.1476'),
     ('1855.1476'),
     ('1856.1476'),
     ('1857.1476'),
     ('1858.1476'),
     ('1859.1476'),
     ('1860.1476'),
     ('1861.1476'),
     ('1862.1476'),
     ('1863.1476'),
     ('1864.1476'),
     ('1865.1476'),
     ('1866.1476'),
     ('1867.1476'),
     ('1868.1476'),
     ('1869.1476'),

     ('1901.1476'),
     ('1902.1476'),
     ('1903.1476'),
     ('1904.1476'),
     ('1905.1476'),
     ('1906.1476'),
     ('1907.1476'),
     ('1908.1476'),
     ('1909.1476'),
     ('1910.1476'),
     ('1911.1476'),
     ('1912.1476'),
     ('1913.1476'),
     ('1914.1476'),
     ('1915.1476'),
     ('1916.1476'),
     ('1917.1476'),
     ('1918.1476'),
     ('1919.1476'),
     ('1920.1476'),
     ('1921.1476'),
     ('1922.1476'),
     ('1923.1476'),
     ('1924.1476'),
     ('1925.1476'),
     ('1926.1476'),
     ('1927.1476'),
     ('1928.1476'),
     ('1929.1476'),
     ('1930.1476'),
     ('1931.1476'),
     ('1932.1476'),
     ('1933.1476'),
     ('1934.1476'),
     ('1935.1476'),
     ('1936.1476'),
     ('1937.1476'),
     ('1938.1476'),
     ('1939.1476'),
     ('1940.1476'),
     ('1941.1476'),
     ('1942.1476'),
     ('1943.1476'),
     ('1944.1476'),
     ('1945.1476'),
     ('1946.1476'),
     ('1947.1476'),
     ('1948.1476'),
     ('1949.1476'),
     ('1950.1476'),
     ('1951.1476'),
     ('1952.1476'),
     ('1953.1476'),
     ('1954.1476'),
     ('1955.1476'),
     ('1956.1476'),
     ('1957.1476'),
     ('1958.1476'),
     ('1959.1476'),
     ('1960.1476'),
     ('1961.1476'),
     ('1962.1476'),
     ('1963.1476'),
     ('1964.1476'),
     ('1965.1476'),
     ('1966.1476'),
     ('1967.1476'),
     ('1968.1476'),
     ('1969.1476'),


     ('2001.1476'),
     ('2002.1476'),
     ('2003.1476'),
     ('2004.1476'),
     ('2005.1476'),
     ('2006.1476'),
     ('2007.1476'),
     ('2008.1476'),
     ('2009.1476'),
     ('2010.1476'),
     ('2011.1476'),
     ('2012.1476'),
     ('2013.1476'),
     ('2014.1476'),
     ('2015.1476'),
     ('2016.1476'),
     ('2017.1476'),
     ('2018.1476'),
     ('2019.1476'),
     ('2020.1476'),
     ('2021.1476'),
     ('2022.1476'),
     ('2023.1476'),
     ('2024.1476'),
     ('2025.1476'),
     ('2026.1476'),
     ('2027.1476'),
     ('2028.1476'),
     ('2029.1476'),
     ('2030.1476'),
     ('2031.1476'),
     ('2032.1476'),
     ('2033.1476'),
     ('2034.1476'),
     ('2035.1476'),
     ('2036.1476'),
     ('2037.1476'),
     ('2038.1476'),
     ('2039.1476'),
     ('2040.1476'),
     ('2041.1476'),
     ('2042.1476'),
     ('2043.1476'),
     ('2044.1476'),
     ('2045.1476'),
     ('2046.1476'),
     ('2047.1476'),
     ('2048.1476'),
     ('2049.1476'),
     ('2050.1476'),
     ('2051.1476'),
     ('2052.1476'),
     ('2053.1476'),
     ('2054.1476'),
     ('2055.1476'),
     ('2056.1476'),
     ('2057.1476'),
     ('2058.1476'),
     ('2059.1476'),
     ('2060.1476'),
     ('2061.1476'),
     ('2062.1476'),
     ('2063.1476'),
     ('2064.1476'),
     ('2065.1476'),
     ('2066.1476'),
     ('2067.1476'),
     ('2068.1476'),


     ('2101.1476'),
     ('2102.1476'),
     ('2103.1476'),
     ('2104.1476'),
     ('2105.1476'),
     ('2106.1476'),
     ('2107.1476'),
     ('2108.1476'),
     ('2109.1476'),
     ('2110.1476'),
     ('2111.1476'),
     ('2112.1476'),
     ('2113.1476'),
     ('2114.1476'),
     ('2115.1476'),
     ('2116.1476'),
     ('2117.1476'),
     ('2118.1476'),
     ('2119.1476'),
     ('2120.1476'),
     ('2121.1476'),
     ('2122.1476'),
     ('2123.1476'),
     ('2124.1476'),
     ('2125.1476'),
     ('2126.1476'),
     ('2127.1476'),
     ('2128.1476'),
     ('2129.1476'),
     ('2130.1476'),
     ('2131.1476'),
     ('2132.1476'),
     ('2133.1476'),
     ('2134.1476'),
     ('2135.1476'),
     ('2136.1476'),
     ('2137.1476'),
     ('2138.1476'),
     ('2139.1476'),
     ('2140.1476'),
     ('2141.1476'),
     ('2142.1476'),
     ('2143.1476'),
     ('2144.1476'),
     ('2145.1476'),
     ('2146.1476'),
     ('2147.1476'),
     ('2148.1476'),
     ('2149.1476'),
     ('2150.1476'),
     ('2151.1476'),
     ('2152.1476'),
     ('2153.1476'),
     ('2154.1476'),
     ('2155.1476'),
     ('2156.1476'),
     ('2157.1476'),
     ('2158.1476'),
     ('2159.1476'),
     ('2160.1476'),
     ('2161.1476'),
     ('2162.1476'),
     ('2163.1476'),
     ('2164.1476'),
     ('2165.1476'),
     ('2166.1476'),
     ('2167.1476'),


     ('2201.1476'),
     ('2202.1476'),
     ('2203.1476'),
     ('2204.1476'),
     ('2205.1476'),
     ('2206.1476'),
     ('2207.1476'),
     ('2208.1476'),
     ('2209.1476'),
     ('2210.1476'),
     ('2211.1476'),
     ('2212.1476'),
     ('2213.1476'),
     ('2214.1476'),
     ('2215.1476'),
     ('2216.1476'),
     ('2217.1476'),
     ('2218.1476'),
     ('2219.1476'),
     ('2220.1476'),
     ('2221.1476'),
     ('2222.1476'),
     ('2223.1476'),
     ('2224.1476'),
     ('2225.1476'),
     ('2226.1476'),
     ('2227.1476'),
     ('2228.1476'),
     ('2229.1476'),
     ('2230.1476'),
     ('2231.1476'),
     ('2232.1476'),
     ('2233.1476'),
     ('2234.1476'),
     ('2235.1476'),
     ('2236.1476'),
     ('2237.1476'),
     ('2238.1476'),
     ('2239.1476'),
     ('2240.1476'),
     ('2241.1476'),
     ('2242.1476'),
     ('2243.1476'),
     ('2244.1476'),
     ('2245.1476'),
     ('2246.1476'),
     ('2247.1476'),
     ('2248.1476'),
     ('2249.1476'),
     ('2250.1476'),
     ('2251.1476'),
     ('2252.1476'),
     ('2253.1476'),
     ('2254.1476'),
     ('2255.1476'),
     ('2256.1476'),
     ('2257.1476'),
     ('2258.1476'),
     ('2259.1476'),
     ('2260.1476'),
     ('2261.1476'),
     ('2262.1476'),
     ('2263.1476'),
     ('2264.1476'),
     ('2265.1476'),


     ('2301.1476'),
     ('2302.1476'),
     ('2303.1476'),
     ('2304.1476'),
     ('2305.1476'),
     ('2306.1476'),
     ('2307.1476'),
     ('2308.1476'),
     ('2309.1476'),
     ('2310.1476'),
     ('2311.1476'),
     ('2312.1476'),
     ('2313.1476'),
     ('2314.1476'),
     ('2315.1476'),
     ('2316.1476'),
     ('2317.1476'),
     ('2318.1476'),
     ('2319.1476'),
     ('2320.1476'),
     ('2321.1476'),
     ('2322.1476'),
     ('2323.1476'),
     ('2324.1476'),
     ('2325.1476'),
     ('2326.1476'),
     ('2327.1476'),
     ('2328.1476'),
     ('2329.1476'),
     ('2330.1476'),
     ('2331.1476'),
     ('2332.1476'),
     ('2333.1476'),
     ('2334.1476'),
     ('2335.1476'),
     ('2336.1476'),
     ('2337.1476'),
     ('2338.1476'),
     ('2339.1476'),
     ('2340.1476'),
     ('2341.1476'),
     ('2342.1476'),
     ('2343.1476'),
     ('2344.1476'),
     ('2345.1476'),
     ('2346.1476'),
     ('2347.1476'),
     ('2348.1476'),
     ('2349.1476'),
     ('2350.1476'),
     ('2351.1476'),
     ('2352.1476'),
     ('2353.1476'),
     ('2354.1476'),
     ('2355.1476'),
     ('2356.1476'),
     ('2357.1476'),
     ('2358.1476'),
     ('2359.1476'),
     ('2360.1476'),
     ('2361.1476'),
     ('2362.1476'),
     ('2363.1476'),


     ('2401.1476'),
     ('2402.1476'),
     ('2403.1476'),
     ('2404.1476'),
     ('2405.1476'),
     ('2406.1476'),
     ('2407.1476'),
     ('2408.1476'),
     ('2409.1476'),
     ('2410.1476'),
     ('2411.1476'),
     ('2412.1476'),
     ('2413.1476'),
     ('2414.1476'),
     ('2415.1476'),
     ('2416.1476'),
     ('2417.1476'),
     ('2418.1476'),
     ('2419.1476'),
     ('2420.1476'),
     ('2421.1476'),
     ('2422.1476'),
     ('2423.1476'),
     ('2424.1476'),
     ('2425.1476'),
     ('2426.1476'),
     ('2427.1476'),
     ('2428.1476'),
     ('2429.1476'),
     ('2430.1476'),
     ('2431.1476'),
     ('2432.1476'),
     ('2433.1476'),
     ('2434.1476'),
     ('2435.1476'),
     ('2436.1476'),
     ('2437.1476'),
     ('2438.1476'),
     ('2439.1476'),
     ('2440.1476'),
     ('2441.1476'),
     ('2442.1476'),
     ('2443.1476'),
     ('2444.1476'),
     ('2445.1476'),
     ('2446.1476'),
     ('2447.1476'),
     ('2448.1476'),
     ('2449.1476'),
     ('2450.1476'),
     ('2451.1476'),
     ('2452.1476'),
     ('2453.1476'),
     ('2454.1476'),
     ('2455.1476'),
     ('2456.1476'),
     ('2457.1476'),
     ('2458.1476'),
     ('2459.1476'),
     ('2460.1476'),


     ('2501.1476'),
     ('2502.1476'),
     ('2503.1476'),
     ('2504.1476'),
     ('2505.1476'),
     ('2506.1476'),
     ('2507.1476'),
     ('2508.1476'),
     ('2509.1476'),
     ('2510.1476'),
     ('2511.1476'),
     ('2512.1476'),
     ('2513.1476'),
     ('2514.1476'),
     ('2515.1476'),
     ('2516.1476'),
     ('2517.1476'),
     ('2518.1476'),
     ('2519.1476'),
     ('2520.1476'),
     ('2521.1476'),
     ('2522.1476'),
     ('2523.1476'),
     ('2524.1476'),
     ('2525.1476'),
     ('2526.1476'),
     ('2527.1476'),
     ('2528.1476'),
     ('2529.1476'),
     ('2530.1476'),
     ('2531.1476'),
     ('2532.1476'),
     ('2533.1476'),
     ('2534.1476'),
     ('2535.1476'),
     ('2536.1476'),
     ('2537.1476'),
     ('2538.1476'),
     ('2539.1476'),
     ('2540.1476'),
     ('2541.1476'),
     ('2542.1476'),
     ('2543.1476'),
     ('2544.1476'),
     ('2545.1476'),
     ('2546.1476'),
     ('2547.1476'),
     ('2548.1476'),
     ('2549.1476'),
     ('2550.1476'),
     ('2551.1476'),
     ('2552.1476'),
     ('2553.1476'),
     ('2554.1476'),
     ('2555.1476'),
     ('2556.1476'),


     ('2601.1476'),
     ('2602.1476'),
     ('2603.1476'),
     ('2604.1476'),
     ('2605.1476'),
     ('2606.1476'),
     ('2607.1476'),
     ('2608.1476'),
     ('2609.1476'),
     ('2610.1476'),
     ('2611.1476'),
     ('2612.1476'),
     ('2613.1476'),
     ('2614.1476'),
     ('2615.1476'),
     ('2616.1476'),
     ('2617.1476'),
     ('2618.1476'),
     ('2619.1476'),
     ('2620.1476'),
     ('2621.1476'),
     ('2622.1476'),
     ('2623.1476'),
     ('2624.1476'),
     ('2625.1476'),
     ('2626.1476'),
     ('2627.1476'),
     ('2628.1476'),
     ('2629.1476'),
     ('2630.1476'),
     ('2631.1476'),
     ('2632.1476'),
     ('2633.1476'),
     ('2634.1476'),
     ('2635.1476'),
     ('2636.1476'),
     ('2637.1476'),
     ('2638.1476'),
     ('2639.1476'),
     ('2640.1476'),
     ('2641.1476'),
     ('2642.1476'),
     ('2643.1476'),
     ('2644.1476'),
     ('2645.1476'),
     ('2646.1476'),
     ('2647.1476'),
     ('2648.1476'),
     ('2649.1476'),
     ('2650.1476'),
     ('2651.1476'),
     ('2652.1476'),


     ('2701.1476'),
     ('2702.1476'),
     ('2703.1476'),
     ('2704.1476'),
     ('2705.1476'),
     ('2706.1476'),
     ('2707.1476'),
     ('2708.1476'),
     ('2709.1476'),
     ('2710.1476'),
     ('2711.1476'),
     ('2712.1476'),
     ('2713.1476'),
     ('2714.1476'),
     ('2715.1476'),
     ('2716.1476'),
     ('2717.1476'),
     ('2718.1476'),
     ('2719.1476'),
     ('2720.1476'),
     ('2721.1476'),
     ('2722.1476'),
     ('2723.1476'),
     ('2724.1476'),
     ('2725.1476'),
     ('2726.1476'),
     ('2727.1476'),
     ('2728.1476'),
     ('2729.1476'),
     ('2730.1476'),
     ('2731.1476'),
     ('2732.1476'),
     ('2733.1476'),
     ('2734.1476'),
     ('2735.1476'),
     ('2736.1476'),
     ('2737.1476'),
     ('2738.1476'),
     ('2739.1476'),
     ('2740.1476'),
     ('2741.1476'),
     ('2742.1476'),
     ('2743.1476'),
     ('2744.1476'),
     ('2745.1476'),
     ('2746.1476'),
     ('2747.1476'),
     ('2748.1476'),


     ('2801.1476'),
     ('2802.1476'),
     ('2803.1476'),
     ('2804.1476'),
     ('2805.1476'),
     ('2806.1476'),
     ('2807.1476'),
     ('2808.1476'),
     ('2809.1476'),
     ('2810.1476'),
     ('2811.1476'),
     ('2812.1476'),
     ('2813.1476'),
     ('2814.1476'),
     ('2815.1476'),
     ('2816.1476'),
     ('2817.1476'),
     ('2818.1476'),
     ('2819.1476'),
     ('2820.1476'),
     ('2821.1476'),
     ('2822.1476'),
     ('2823.1476'),
     ('2824.1476'),
     ('2825.1476'),
     ('2826.1476'),
     ('2827.1476'),
     ('2828.1476'),
     ('2829.1476'),
     ('2830.1476'),
     ('2831.1476'),
     ('2832.1476'),
     ('2833.1476'),
     ('2834.1476'),
     ('2835.1476'),
     ('2836.1476'),
     ('2837.1476'),
     ('2838.1476'),
     ('2839.1476'),
     ('2840.1476'),
     ('2841.1476'),
     ('2842.1476'),
     ('2843.1476'),


     ('2901.1476'),
     ('2902.1476'),
     ('2903.1476'),
     ('2904.1476'),
     ('2905.1476'),
     ('2906.1476'),
     ('2907.1476'),
     ('2908.1476'),
     ('2909.1476'),
     ('2910.1476'),
     ('2911.1476'),
     ('2912.1476'),
     ('2913.1476'),
     ('2914.1476'),
     ('2915.1476'),
     ('2916.1476'),
     ('2917.1476'),
     ('2918.1476'),
     ('2919.1476'),
     ('2920.1476'),
     ('2921.1476'),
     ('2922.1476'),
     ('2923.1476'),
     ('2924.1476'),
     ('2925.1476'),
     ('2926.1476'),
     ('2927.1476'),
     ('2928.1476'),
     ('2929.1476'),
     ('2930.1476'),
     ('2931.1476'),
     ('2932.1476'),
     ('2933.1476'),
     ('2934.1476'),
     ('2935.1476'),
     ('2936.1476'),
     ('2937.1476'),
     ('2938.1476'),

     ('3001.1476'),
     ('3002.1476'),
     ('3003.1476'),
     ('3004.1476'),
     ('3005.1476'),
     ('3006.1476'),
     ('3007.1476'),
     ('3008.1476'),
     ('3009.1476'),
     ('3010.1476'),
     ('3011.1476'),
     ('3012.1476'),
     ('3013.1476'),
     ('3014.1476'),
     ('3015.1476'),
     ('3016.1476'),
     ('3017.1476'),
     ('3018.1476'),
     ('3019.1476'),
     ('3020.1476'),
     ('3021.1476'),
     ('3022.1476'),
     ('3023.1476'),
     ('3024.1476'),
     ('3025.1476'),
     ('3026.1476'),
     ('3027.1476'),
     ('3028.1476'),
     ('3029.1476'),
     ('3030.1476'),
     ('3031.1476'),
     ('3032.1476'),
     ('3033.1476'),

     ('3101.1476'),
     ('3102.1476'),
     ('3103.1476'),
     ('3104.1476'),
     ('3105.1476'),
     ('3106.1476'),
     ('3107.1476'),
     ('3108.1476'),
     ('3109.1476'),
     ('3110.1476'),
     ('3111.1476'),
     ('3112.1476'),
     ('3113.1476'),
     ('3114.1476'),
     ('3115.1476'),
     ('3116.1476'),
     ('3117.1476'),
     ('3118.1476'),
     ('3119.1476'),
     ('3120.1476'),
     ('3121.1476'),
     ('3122.1476'),
     ('3123.1476'),
     ('3124.1476'),
     ('3125.1476'),
     ('3126.1476'),
     ('3127.1476'),


     ('3201.1476'),
     ('3202.1476'),
     ('3203.1476'),
     ('3204.1476'),
     ('3205.1476'),
     ('3206.1476'),
     ('3207.1476'),
     ('3208.1476'),
     ('3209.1476'),
     ('3210.1476'),
     ('3211.1476'),
     ('3212.1476'),
     ('3213.1476'),
     ('3214.1476'),
     ('3215.1476'),
     ('3216.1476'),
     ('3217.1476'),
     ('3218.1476'),
     ('3219.1476'),
     ('3220.1476'),
     ('3221.1476'),


     ('3301.1476'),
     ('3302.1476'),
     ('3303.1476'),
     ('3304.1476'),
     ('3305.1476'),
     ('3306.1476'),
     ('3307.1476'),
     ('3308.1476'),
     ('3309.1476'),
     ('3310.1476'),
     ('3311.1476'),
     ('3312.1476'),
     ('3313.1476'),
     ('3314.1476'),
     ('3315.1476'),


     ('3401.1476'),
     ('3402.1476'),
     ('3403.1476'),
     ('3404.1476'),
     ('3405.1476'),
     ('3406.1476'),
     ('3407.1476'),
     ('3408.1476'),
     ('3409.1476'),

     ('3501.1476'),
     ('3502.1476'),
     ('3503.1476'),

     ('3601.1476')

    );



const
   {1476 boundaries}
    Stepsize=90/17.5;{5.142857143 degrees}
var {################# initialised variables #########################}
    dec_boundaries1476 : array[0..36] of double=
    (-90*pi/180,
     -87.42857143*pi/180,
     -82.28571429*pi/180,
     -77.14285714*pi/180,
     -72*pi/180,
     -66.85714286*pi/180,
     -61.71428571*pi/180,
     -56.57142857*pi/180,
     -51.42857143*pi/180,
     -46.28571429*pi/180,
     -41.14285714*pi/180,
     -36*pi/180,
     -30.85714286*pi/180,
     -25.71428571*pi/180,
     -20.57142857*pi/180,
     -15.42857143*pi/180,
     -10.28571429*pi/180,
     -5.142857143*pi/180,

     0*pi/180,

     5.142857143*pi/180,
     10.28571429*pi/180,
     15.42857143*pi/180,
     20.57142857*pi/180,
     25.71428571*pi/180,
     30.85714286*pi/180,
     36*pi/180,
     41.14285714*pi/180,
     46.28571429*pi/180,
     51.42857143*pi/180,
     56.57142857*pi/180,
     61.71428571*pi/180,
     66.85714286*pi/180,
     72*pi/180,
     77.14285714*pi/180,
     82.28571429*pi/180,
     87.42857143*pi/180,
     90*pi/180);


var {################# initialised variables #########################}
   record_size:integer=11;{default}
var
  p6        : ^hnskyhdr1476_6;       { pointer to hnsky record }
  p5        : ^hnskyhdr1476_5;       { pointer to hnsky record }
  dec9_storage: shortint;

  buf2: array[1..11] of byte;  {read buffer stars}
  thefile_stars      : tfilestream;
  cache_array       : array of byte;{Maximum 53 mbyte for largest G18 file}
  cache_size        : integer;


procedure area_and_boundaries(ra1,dec1 :double; var area_nr: integer; var spaceE,spaceW,spaceN,spaceS: double); {For a ra, dec position find the star database area number and the corresponding boundary distances N, E, W, S}
var
  rot,cos_dec1 :double;
begin
  cos_dec1:=cos(dec1);
  if dec1>dec_boundaries[17] then
  begin
   area_nr:=290;   {celestial north pole area}
   spaceS:=dec1-dec_boundaries[17];
   spaceN:=dec_boundaries[18]{90}-dec_boundaries[17];{minimum, could go beyond the celestical pole so above +90 degrees}
   spaceW:=pi*2;
   spaceE:=pi*2;
  end
  else
  if dec1>dec_boundaries[16] then {4x RA}
  begin
    rot:=ra1*4/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+24+20+16+12+8+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[16];
    spaceN:=dec_boundaries[17]-dec1;
    cos_dec1:=cos(dec1);
    spaceW:=(pi*2/4) * frac(rot)*cos_dec1; {ra decrease in direction west}
    spaceE:=(pi*2/4) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[15] then {8x RA}
  begin
    rot:=ra1*8/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+24+20+16+12+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[15];
    spaceN:=dec_boundaries[16]-dec1;
    cos_dec1:=cos(dec1);
    spaceW:=(pi*2/8) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/8) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[14] then {12x RA}
  begin
    rot:=ra1*12/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+24+20+16+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[14];
    spaceN:=dec_boundaries[15]-dec1;
    spaceW:=(pi*2/12) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/12) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[13] then {16x RA}
  begin
   rot:=ra1*16/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+24+20+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[13];
    spaceN:=dec_boundaries[14]-dec1;
    spaceW:=(pi*2/16) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/16) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[12] then {20x RA}
  begin
    rot:=ra1*20/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+24+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[12];
    spaceN:=dec_boundaries[13]-dec1;
    spaceW:=(pi*2/20) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/20) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[11] then {24x RA}
  begin
    rot:=ra1*24/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+28+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[11];
    spaceN:=dec_boundaries[12]-dec1;
    spaceW:=(pi*2/24) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/24) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[10] then {28x RA}
  begin
    rot:=ra1*28/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+32+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[10];
    spaceN:=dec_boundaries[11]-dec1;
    spaceW:=(pi*2/28) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/28) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[9] then {32x RA}
  begin
    rot:=ra1*32/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+32+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[9];
    spaceN:=dec_boundaries[10]-dec1;
    spaceW:=(pi*2/32) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/32) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[8] then {32x RA}
  begin
    rot:=ra1*32/(2*pi);
    area_nr:=1+4+8+12+16+20+24+28+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[8];
    spaceN:=dec_boundaries[9]-dec1;
    spaceW:=(pi*2/32) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/32) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[7] then {28x RA}
  begin
    rot:=ra1*28/(2*pi);
    area_nr:=1+4+8+12+16+20+24+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[7];
    spaceN:=dec_boundaries[8]-dec1;
    spaceW:=(pi*2/28) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/28) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[6] then {24x RA}
  begin
    rot:=ra1*24/(2*pi);
    area_nr:=1+4+8+12+16+20+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[6];
    spaceN:=dec_boundaries[7]-dec1;
    spaceW:=(pi*2/24) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/24) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[5] then {20x RA}
  begin
     rot:=ra1*20/(2*pi);
     area_nr:=1+4+8+12+16+1+trunc(rot);
     spaceS:=dec1-dec_boundaries[5];
     spaceN:=dec_boundaries[6]-dec1;
     spaceW:=(pi*2/20) * frac(rot)*cos_dec1;
     spaceE:=(pi*2/20) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[4] then  {16x RA}
  begin
     rot:=ra1*16/(2*pi);
     area_nr:=1+4+8+12+1+trunc(rot);
     spaceS:=dec1-dec_boundaries[4];
     spaceN:=dec_boundaries[5]-dec1;
     spaceW:=(pi*2/16) * frac(rot)*cos_dec1;
     spaceE:=(pi*2/16) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[3] then  {12*RA}
  begin
    rot:=ra1*12/(2*pi);
    area_nr:=1+4+8+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[3];
    spaceN:=dec_boundaries[4]-dec1;
    spaceW:=(pi*2/12) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/12) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[2] then  {8x RA}
  begin
    rot:=ra1*8/(2*pi);
    area_nr:=1+4+1+trunc(rot);
    spaceS:=dec1-dec_boundaries[2];
    spaceN:=dec_boundaries[3]-dec1;
    spaceW:=(pi*2/8) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/8) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries[1] then  {4x RA}
  begin
    rot:=ra1*4/(2*pi);
    area_nr:=1+ 1+trunc(rot);
    spaceS:=dec1-dec_boundaries[1];
    spaceN:=dec_boundaries[2]-dec1;
    spaceW:=(pi*2/4) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/4) * (1-frac(rot))*cos_dec1;
  end
  else
  begin
    area_nr:=1;   {celestial south pole area}
    spaceS:=dec_boundaries[1]-dec_boundaries[0];{minimum, could go beyond the celestical pole so below -90 degrees}
    spaceN:=dec_boundaries[1]-dec1;
    spaceW:=pi*2;
    spaceE:=pi*2;

  end;
end;



procedure area_and_boundaries1476(ra1,dec1 :double; out area_nr: integer; out spaceE,spaceW,spaceN,spaceS: double); {For a ra, dec position find the star database area number and the corresponding boundary distances N, E, W, S}
var
  rot,cos_dec1 :double;
begin
  cos_dec1:=cos(dec1);
  if dec1>dec_boundaries1476[35] then
  begin
   area_nr:=1476;   {celestial north pole area}
   spaceS:=dec1-dec_boundaries1476[35];
   spaceN:=dec_boundaries1476[36]{90}-dec_boundaries1476[35];{minimum, could go beyond the celestical pole so above +90 degrees}
   spaceW:=pi*2;
   spaceE:=pi*2;
  end
  else
  if dec1>dec_boundaries1476[34] then {3x RA}
  begin
    rot:=ra1*3/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+33+27+21+15+9+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[34];
    spaceN:=dec_boundaries1476[35]-dec1;
    cos_dec1:=cos(dec1);
    spaceW:=(pi*2/3) * frac(rot)*cos_dec1; {ra decrease in direction west}
    spaceE:=(pi*2/3) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[33] then {9x RA}
  begin
    rot:=ra1*9/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+33+27+21+15+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[33];
    spaceN:=dec_boundaries1476[34]-dec1;
    cos_dec1:=cos(dec1);
    spaceW:=(pi*2/9) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/9) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[32] then {15x RA}
  begin
    rot:=ra1*15/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+33+27+21+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[32];
    spaceN:=dec_boundaries1476[33]-dec1;
    spaceW:=(pi*2/15) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/15) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[31] then {21x RA}
  begin
    rot:=ra1*21/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+33+27+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[31];
    spaceN:=dec_boundaries1476[32]-dec1;
    spaceW:=(pi*2/21) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/21) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[30] then {27x RA}
  begin
    rot:=ra1*27/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+33+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[30];
    spaceN:=dec_boundaries1476[31]-dec1;
    spaceW:=(pi*2/27) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/27) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[29] then {33x RA}
  begin
    rot:=ra1*33/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+38+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[29];
    spaceN:=dec_boundaries1476[30]-dec1;
    spaceW:=(pi*2/33) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/33) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[28] then {38x RA}
  begin
    rot:=ra1*38/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+43+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[28];
    spaceN:=dec_boundaries1476[29]-dec1;
    spaceW:=(pi*2/38) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/38) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[27] then {43x RA}
  begin
    rot:=ra1*43/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+48+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[27];
    spaceN:=dec_boundaries1476[28]-dec1;
    spaceW:=(pi*2/43) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/43) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[26] then {48x RA}
  begin
    rot:=ra1*48/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+52+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[26];
    spaceN:=dec_boundaries1476[27]-dec1;
    spaceW:=(pi*2/48) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/48) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[25] then {52x RA}
  begin
    rot:=ra1*52/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+56+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[25];
    spaceN:=dec_boundaries1476[26]-dec1;
    spaceW:=(pi*2/52) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/52) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[24] then {56x RA}
  begin
    rot:=ra1*56/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+60+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[24];
    spaceN:=dec_boundaries1476[25]-dec1;
    spaceW:=(pi*2/56) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/56) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[23] then {60x RA}
  begin
    rot:=ra1*60/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+63+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[23];
    spaceN:=dec_boundaries1476[24]-dec1;
    spaceW:=(pi*2/60) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/60) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[22] then {63x RA}
  begin
    rot:=ra1*63/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+65+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[22];
    spaceN:=dec_boundaries1476[23]-dec1;
    spaceW:=(pi*2/63) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/63) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[21] then {65x RA}
  begin
    rot:=ra1*65/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+67+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[21];
    spaceN:=dec_boundaries1476[22]-dec1;
    spaceW:=(pi*2/65) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/65) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[20] then {67x RA}
  begin
    rot:=ra1*67/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+68+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[20];
    spaceN:=dec_boundaries1476[21]-dec1;
    spaceW:=(pi*2/67) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/67) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[19] then {68x RA}
  begin
    rot:=ra1*68/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+69+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[19];
    spaceN:=dec_boundaries1476[20]-dec1;
    spaceW:=(pi*2/68) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/68) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[18] then {69x RA}
  begin
    rot:=ra1*69/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+69+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[18];
    spaceN:=dec_boundaries1476[19]-dec1;
    spaceW:=(pi*2/69) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/69) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[17] then {69x RA}
  begin
    rot:=ra1*69/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+68+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[17];
    spaceN:=dec_boundaries1476[19]-dec1;
    spaceW:=(pi*2/69) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/69) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[16] then {68x RA}
  begin
    rot:=ra1*68/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+67+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[16];
    spaceN:=dec_boundaries1476[17]-dec1;
    spaceW:=(pi*2/68) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/68) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[15] then {67x RA}
  begin
    rot:=ra1*67/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+65+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[15];
    spaceN:=dec_boundaries1476[16]-dec1;
    spaceW:=(pi*2/67) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/67) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[14] then {65x RA}
  begin
    rot:=ra1*65/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+63+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[14];
    spaceN:=dec_boundaries1476[15]-dec1;
    spaceW:=(pi*2/65) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/65) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[13] then {63x RA}
  begin
    rot:=ra1*63/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+60+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[13];
    spaceN:=dec_boundaries1476[14]-dec1;
    spaceW:=(pi*2/63) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/63) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[12] then {60x RA}
  begin
    rot:=ra1*60/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+56+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[12];
    spaceN:=dec_boundaries1476[13]-dec1;
    spaceW:=(pi*2/60) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/60) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[11] then {56x RA}
  begin
    rot:=ra1*56/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+52+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[11];
    spaceN:=dec_boundaries1476[12]-dec1;
    spaceW:=(pi*2/56) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/56) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[10] then {52x RA}
  begin
    rot:=ra1*52/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+48+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[10];
    spaceN:=dec_boundaries1476[11]-dec1;
    spaceW:=(pi*2/52) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/52) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[9] then {48x RA}
  begin
    rot:=ra1*48/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+43+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[9];
    spaceN:=dec_boundaries1476[10]-dec1;
    spaceW:=(pi*2/48) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/48) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[8] then {43x RA}
  begin
    rot:=ra1*43/(2*pi);
    area_nr:=1+3+9+15+21+27+33+38+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[8];
    spaceN:=dec_boundaries1476[9]-dec1;
    spaceW:=(pi*2/43) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/43) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[7] then {38x RA}
  begin
    rot:=ra1*38/(2*pi);
    area_nr:=1+3+9+15+21+27+33+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[7];
    spaceN:=dec_boundaries1476[8]-dec1;
    spaceW:=(pi*2/38) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/38) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[6] then {33x RA}
  begin
    rot:=ra1*33/(2*pi);
    area_nr:=1+3+9+15+21+27+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[6];
    spaceN:=dec_boundaries1476[7]-dec1;
    spaceW:=(pi*2/33) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/33) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[5] then {27x RA}
  begin
    rot:=ra1*27/(2*pi);
    area_nr:=1+3+9+15+21+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[5];
    spaceN:=dec_boundaries1476[6]-dec1;
    spaceW:=(pi*2/27) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/27) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[4] then {21x RA}
  begin
    rot:=ra1*21/(2*pi);
    area_nr:=1+3+9+15+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[4];
    spaceN:=dec_boundaries1476[5]-dec1;
    spaceW:=(pi*2/21) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/21) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[3] then {15x RA}
  begin
    rot:=ra1*15/(2*pi);
    area_nr:=1+3+9+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[3];
    spaceN:=dec_boundaries1476[4]-dec1;
    spaceW:=(pi*2/15) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/15) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[2] then {9x RA}
  begin
    rot:=ra1*9/(2*pi);
    area_nr:=1+3+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[2];
    spaceN:=dec_boundaries1476[3]-dec1;
    spaceW:=(pi*2/9) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/9) * (1-frac(rot))*cos_dec1;
  end
  else
  if dec1>dec_boundaries1476[1] then {3x RA}
  begin
    rot:=ra1*3/(2*pi);
    area_nr:=1+1+trunc(rot);
    spaceS:=dec1-dec_boundaries1476[1];
    spaceN:=dec_boundaries1476[2]-dec1;
    spaceW:=(pi*2/3) * frac(rot)*cos_dec1;
    spaceE:=(pi*2/3) * (1-frac(rot))*cos_dec1;
  end
  else
  begin
    area_nr:=1;   {celestial south pole area}
    spaceS:=dec_boundaries1476[1]-dec_boundaries1476[0];{minimum, could go beyond the celestical pole so below -90 degrees}
    spaceN:=dec_boundaries1476[1]-dec1;
    spaceW:=pi*2;
    spaceE:=pi*2;
  end;
end;


procedure find_areas(ra1,dec1,fov :double; out area1,area2,area3,area4 :integer; out frac1,frac2,frac3,frac4:double);{find up to four star database areas for the square image. Maximum size is a little lesse the one database field 9.5x9.5 degrees for .290 files and 5.14 x 5.14 degrees for .1476 files}
var
  ra_cornerWN,ra_cornerEN,ra_cornerWS,ra_cornerES,
  dec_cornerN,dec_cornerS,fov_half,
  spaceE,spaceW,spaceN,spaceS                                           : double;
begin
  fov_half:=fov/2; {warning FOV should be less the database tile dimensions, so <=9.53 degrees for .290 files and <=5.14 for .1476 files. Otherwise a tile beyond next tile could be selected}

  dec_cornerN:=dec1+fov_half; {above +pi/2 doesn't matter since it is all area 290}
  dec_cornerS:=dec1-fov_half; {above -pi/2 doesn't matter since it is all area 1}
  ra_cornerWN:=ra1-fov_half/cos(dec_cornerN); if ra_cornerWN<0     then ra_cornerWN:=ra_cornerWN+2*pi;{For direction west the RA decreases}
  ra_cornerEN:=ra1+fov_half/cos(dec_cornerN); if ra_cornerEN>=2*pi then ra_cornerEN:=ra_cornerEN-2*pi;
  ra_cornerWS:=ra1-fov_half/cos(dec_cornerS); if ra_cornerWS<0     then ra_cornerWS:=ra_cornerWS+2*pi;
  ra_cornerES:=ra1+fov_half/cos(dec_cornerS); if ra_cornerES>=2*pi then ra_cornerES:=ra_cornerES-2*pi;

  if database_type=290 then
  begin
    {corner 1}
    area_and_boundaries(ra_cornerEN,dec_cornerN, area1, spaceE,spaceW,spaceN,spaceS);
    frac1:=min(spaceW,fov)*min(spaceS,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 2}
    area_and_boundaries(ra_cornerWN,dec_cornerN, area2, spaceE,spaceW,spaceN,spaceS);
    frac2:=min(spaceE,fov)*min(spaceS,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 3}
    area_and_boundaries(ra_cornerES,dec_cornerS, area3, spaceE,spaceW,spaceN,spaceS);
    frac3:=min(spaceW,fov)*min(spaceN,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 4}
    area_and_boundaries(ra_cornerWS,dec_cornerS, area4, spaceE,spaceW,spaceN,spaceS);
    frac4:=min(spaceE,fov)*min(spaceN,fov)/(fov*fov);{fraction of the image requiring this database area}
  end
  else
  begin
    {corner 1}
    area_and_boundaries1476(ra_cornerEN,dec_cornerN, area1, spaceE,spaceW,spaceN,spaceS);
    frac1:=min(spaceW,fov)*min(spaceS,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 2}
    area_and_boundaries1476(ra_cornerWN,dec_cornerN, area2, spaceE,spaceW,spaceN,spaceS);
    frac2:=min(spaceE,fov)*min(spaceS,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 3}
    area_and_boundaries1476(ra_cornerES,dec_cornerS, area3, spaceE,spaceW,spaceN,spaceS);
    frac3:=min(spaceW,fov)*min(spaceN,fov)/(fov*fov);{fraction of the image requiring this database area}

    {corner 4}
    area_and_boundaries1476(ra_cornerWS,dec_cornerS, area4, spaceE,spaceW,spaceN,spaceS);
    frac4:=min(spaceE,fov)*min(spaceN,fov)/(fov*fov);{fraction of the image requiring this database area}
  end;



  if area2=area1 then begin area2:=0;frac2:=0;end; {area2 and area1 equivalent}
  if area3=area1 then begin area3:=0;frac3:=0;end; {area3 and area1 equivalent}
  if area4=area1 then begin area4:=0;frac4:=0;end; {area4 and area1 equivalent}

  if area3=area2 then begin area3:=0;frac3:=0;end; {area3 and area2 equivalent}
  if area4=area2 then begin area4:=0;frac4:=0;end;

  if area4=area3 then begin area4:=0;frac4:=0;end;

  if frac1<0.01 then begin area1:=0;frac1:=0;end;{too small, ignore}
  if frac2<0.01 then begin area2:=0;frac2:=0;end;{too small, ignore}
  if frac3<0.01 then begin area3:=0;frac3:=0;end;{too small, ignore}
  if frac4<0.01 then begin area4:=0;frac4:=0;end;{too small, ignore}

end;


function select_star_database(database:string;fov:double): boolean; {select a star database, report false if none is found}
var
  typ : ansichar;
  warning : boolean;
begin
  result:=true;
  warning:=false; //warning for old database
  database_type:=1476;{type .1476 database}
  database:=lowercase(database);
  if length(database)>0 then typ:=database[1] else typ:='a';

  if typ<>'a' then {manual setting}
  begin
    if typ='w' then
    begin
      if fileexists( database_path+database+'_0101.001') then begin name_database:=database; {try preference}database_type:=001;exit; end
    end
    else
    if typ in ['d','v','h'] then //d80,v50, h18
    begin
      if fileexists( database_path+database+'_0101.1476') then begin name_database:=database; {try preference}  exit; end;
    end; // no else since there is a v50 and v17!!
    if typ in ['v','g'] then//v17, g18
    begin
      if fileexists( database_path+database+'_0101.290') then begin name_database:=database; {try preference}database_type:=290;exit; end
    end;
  end;{auto}

  if fov>20 then
  begin
    if fileexists( database_path+'w08_0101.001') then begin name_database:='w08';database_type:=001; exit; end
    else
    memo2_message('Could not find w08 star database. Will try with an other database.');
  end;

  if ((fov>6) and (fileexists( database_path+'g05_0101.290'))) then begin name_database:='g05'; database_type:=290; end //preference for G05 for large FOV
  else
  if ((fov>6) and (fileexists( database_path+'v05_0101.290'))) then begin name_database:='v05'; database_type:=290; end
  else
  if ((fov>6) and (fileexists( database_path+'v17_0101.290'))) then begin name_database:='v17'; database_type:=290; warning:=true; end //preference for V17 for large FOV
  else
  if fileexists( database_path+'d80_0101.1476') then begin name_database:='d80'; end //for tiny field of view
  else
  if fileexists( database_path+'d50_0101.1476') then begin name_database:='d50'; end
  else
  if fileexists( database_path+'v50_0101.1476') then begin name_database:='v50'; end //photometry database
  else
  if fileexists( database_path+'d20_0101.1476') then begin name_database:='d20'; end
  else
  if fileexists( database_path+'d05_0101.1476') then begin name_database:='d05'; end
  else
  if fileexists( database_path+'g05_0101.290') then begin name_database:='g05'; database_type:=290; end
  else
  if fileexists( database_path+'v05_0101.290') then begin name_database:='v05'; database_type:=290; end
  else
  if fileexists( database_path+'h18_0101.1476') then begin name_database:='h18';warning:=true; end //old database sorted on magnitude
  else
  if fileexists( database_path+'g18_0101.290') then begin name_database:='g18';warning:=true; database_type:=290; end //old database sorted on magnitude
  else
  if fileexists( database_path+'h17_0101.1476') then begin name_database:='h17'; warning:=true;end // old database sorted on magnitude
  else
  if fileexists( database_path+'v17_0101.290') then begin name_database:='v17'; database_type:=290;warning:=true; end //old database sorted on magnitude
  else
  if fileexists( database_path+'g17_0101.290') then begin name_database:='g17'; database_type:=290;warning:=true; end //old database sorted on magnitude
  else
  begin
    application.messagebox(pchar('No star database found!'+#13+'Download and install one star database.'), pchar('ASTAP error:'),0);
    result:=false;
  end;

  if warning then warning_str:='Old database!'; //first potential warning. No add required
end;


procedure close_star_database;{Close the tfilestream}
begin
  if file_open then
  begin
    thefile_stars.free;
    file_open:=false;
  end;
end;


function open_database(telescope_dec: double; area290: integer): boolean; {open database file}
var
  namefile : string;
begin
  result:=true;{assume succes}
  cos_telescope_dec:=cos(telescope_dec);{here to save CPU time}

  if ((area290<>old_area) or (file_open=false)) then
  begin
    close_star_database;{close the reader if open}

    if  database_type=290 then
      namefile:=name_database+'_'+filenames290[area290] {g17_0101.290}
    else
      namefile:=name_database+'_'+filenames1476[area290]; {g17_0101.1476}


    try
      thefile_stars:=tfilestream.Create( database_path+namefile, fmOpenRead or fmShareDenyWrite); {read but do not lock file}
    except
       result:=false;
       exit;
    end;
    file_open:=true; {file is open in tfilestream}

    cache_valid_pos:=0;{new file name}
    thefile_stars.read(database2,110); {read header info, 10x11 is 110 bytes}
    if database2[109]=' ' then record_size:=11 {default}
    else
    record_size:=ord(database2[109]);{5,6,7,9,10 or 11 bytes record}

    cache_size:=thefile_stars.size-110;

    if cache_size>length(cache_array) then  {Only resize when required. Resizing takes time}
    begin
      cache_array:=nil;{prevent it is copied to the new resized array}
      setlength(cache_array,cache_size);{ increase cache to star database file size}
    end;

    old_area:=area290;
  end;
  {else  use old data in cache}

  {This cache works  about 35 % faster then Treader for files with FOV of 0.5 degrees. This due to re-use cache. No difference for FOV 1.3 degrees. At FOV=0.25 the improvement is 40%}
  //reading database take time:
  //fov=0.23 degrees, 73% of total time
  //fov=0.5  degrees, 60% of total time
  //fov=4.6  degrees,  8% of total time

  cache_position:=0;
end;


// This readdatabase is a stripped version for record sizes 5 and 6 only. See HNSKY source files for reading other record size files.
//
// telescope_ra, telescope_dec [radians], contains to center position of the field of interest
// field_diameter [radians], FOV diameter of field of interest. This is ignored in searchmode=T}
// ra, dec [radians],   reported star position
// mag2 [magnitude*10]  reported star magnitude
// Bp-Rp, Gaia color information, not used in ASTAP for solving}
// result [true/false]  if reported true then more stars are available. If false no more stars available.
// extra outputs:
//          database2  : array[0..(11*10)] of ansichar;{text info star database used}
// preconditions:
//   area290 should be set at 290+1 before any read series
//   cos_telescope_dec, double variable should contains the cos(telescope_dec) to detect if star read is within the FOV diameter}
//
{$INLINE ON}
function readdatabase290(telescope_ra,telescope_dec, field_diameter:double; out ra2,dec2, mag2,Bp_Rp : double): boolean; inline;{star 290 file database search}
            {searchmode=S screen update }
            {searchmode=M mouse click  search}
            {searchmode=T text search}
  var
    ra_raw,block_to_read         : integer;
    delta_ra                     : double;
    header_record                : boolean;
const
   blocksize=5*6*4*1024; {a multiply of record sizes 5, 6}
begin
   {$I-}
  readdatabase290:=true;
  repeat
    repeat
      if cache_position>=cache_size then {should be end of file}
      begin
        readdatabase290:=false; {no more data in this file}
        exit;
      end;

      if cache_position>=cache_valid_pos then {add more data to cache. This cache works  about 35 % faster then Treader for files with FOV of 0.5 degrees. This due to re-use cache. No difference for FOV 1.3 degrees. At FOV=0.25 the improvement is 40%}
      begin
        block_to_read:=min(cache_size, blocksize);{for small files, don't read more than cache size=file size!}
        block_to_read:=min(block_to_read,cache_size - cache_valid_pos);{don't read more than file size!}
        thefile_stars.read(cache_array[cache_valid_pos],block_to_read); {fill cache more. In most cases it can be re-used. Especially for small field of view}
        cache_valid_pos:=cache_valid_pos+block_to_read;{increase postion where cache buffer is valid.}
      end;
      move(cache_array[cache_position],buf2,record_size);{move one record for reading}
      cache_position:=cache_position + record_size;{update cache position}

      header_record:=false;

      case record_size of
      5: begin {record size 5}
           with p5^ do
           begin
             ra_raw:=(ra7 + ra8 shl 8 +ra9 shl 16);{always required, fasted method}
             if ra_raw=$FFFFFF  then  {special magnitude record is found}
             begin
               mag2:=dec8-16;{new magn shifted 16 to make sirius and other positive}
               {magnitude is stored in mag2 till new magnitude record is found}
               dec9_storage:=dec7-128;{recover dec9 shortint and put it in storage}
               header_record:=true;
             end
             else
             begin {normal record without magnitude}
               ra2:= ra_raw*(pi*2  /((256*256*256)-1));
               dec2:=((dec9_storage shl 16)+(dec8 shl 8)+dec7)*(pi*0.5/((128*256*256)-1));// dec2:=(dec7+(dec8 shl 8)+(dec9 shl 16))*(pi*0.5/((128*256*256)-1)); {FPC compiler makes mistake, but dec7 behind}
               {The RA is stored as a 3 bytes word. The DEC position is stored as a two's complement (=standard), three bytes integer. The resolution of this three byte storage will be for RA: 360*60*60/((256*256*256)-1) = 0.077 arc seconds. For the DEC value it will be: 90*60*60/((128*256*256)-1) = 0.039 arc seconds.}
             end;
           end;
         end;{record size 5}
      6: begin {record size 6}
            with p6^ do
            begin
              ra_raw:=(ra7 + ra8 shl 8 +ra9 shl 16);{always required, fasted method}
              if ra_raw=$FFFFFF  then  {special magnitude record is found}
              begin
                mag2:=dec8-16;{new magn shifted 16 to make sirius and other positive}
                {magnitude is stored in mag2 till new magnitude record is found}
                dec9_storage:=dec7-128;{recover dec9 shortint and put it in storage}
                header_record:=true;
              end
              else
              begin {normal record without magnitude}
                ra2:= ra_raw*(pi*2  /((256*256*256)-1));
                dec2:=((dec9_storage shl 16)+(dec8 shl 8)+dec7)*(pi*0.5/((128*256*256)-1));// dec2:=(dec7+(dec8 shl 8)+(dec9 shl 16))*(pi*0.5/((128*256*256)-1)); {FPC compiler makes mistake, put dec7 behind}
                Bp_Rp:=b_r;{gaia (Bp-Rp)*10}    {color information}
              end;
            end;
          end;{record size 6}
      end;{case}
    until header_record=false;
    delta_ra:=abs(ra2-telescope_ra); if delta_ra>pi then delta_ra:=pi*2-delta_ra;{Here because ra2 is not defined in case header_record}
  until  (  (delta_ra*cos_telescope_dec<field_diameter/2) and (abs(dec2-telescope_dec)<field_diameter/2)  );
                           {calculate distance and skip when too far from center screen, {if false then outside screen,go to next line}
end;




begin
  p6:= @buf2[1];	{ set pointer }
  p5:= @buf2[1];	{ set pointer }
end.

