; -*- fill-column: 79 -*-
;
; acl2-doc.lisp - Documentation for the ACL2 Theorem Prover
;
; ACL2 Version 8.4 -- A Computational Logic for Applicative Common Lisp
; Copyright (C) 2021, Regents of the University of Texas
;
; This documentation was derived from the ACL2 system in October 2013, which
; was a descendant of ACL2 Version 1.9, Copyright (C) 1997 Computational Logic,
; Inc.  See the documentation topic NOTE-2-0.
;
; This program is free software; you can redistribute it and/or modify it under
; the terms of the LICENSE file distributed with ACL2.
;
; This program is distributed in the hope that it will be useful, but WITHOUT
; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
; FOR A PARTICULAR PURPOSE.  See the LICENSE file distributed with ACL2 for
; more details.
;
; Here are the original authors of this book.
;
; Written by:  Matt Kaufmann               and J Strother Moore
; email:       Kaufmann@cs.utexas.edu      and Moore@cs.utexas.edu
; Department of Computer Science
; University of Texas at Austin
; Austin, TX 78712 U.S.A.
;
; The ACL2 community is invited to contribute to this book through the github
; project https://github.com/acl2/acl2/.  If you edit this book, then please
; conform to its style, for example restricting to 79 columns, avoiding tab
; characters, and keeping topics in alphabetical order.  Please add additional
; instructions here as appropriate.  For general information about writing
; documentation, see the ACL2+Books Manual (see :DOC documentation), in
; particular the topic DEFXDOC and the related topics XDOC::MARKUP and
; XDOC::PREPROCESSOR.

; Note that :long is optional; if its value is "", feel free to omit it.

; Many topics are nothing but links, as generated by calls of defpointer,
; below.

(in-package "ACL2")
(include-book "xdoc/top" :dir :system)

(defconst *acl2-broken-links-alist*

; To update the value of this constant, especially when preparing for an ACL2
; release, see file acl2-doc-broken-links.lsp in this directory.

; The value of this constant is an alist whose keys are the XDOC topics
; referenced in the ACL2-only manual that are not in the ACL2-only manual
; (i.e. that are in the Books part of the ACL2+Books manual).  Each key in this
; alist is associated with the name of the file where the referenced topic is
; defined.  Each file name is a string starting with [books]/ (see below).  So
; these links are "broken" in the ACL2-only manual.

; So in practice, if you link to an ACL2+Books topic that is not in the
; ACL2-only manual, just add an entry following the pattern below.  It would be
; best if you keep it sorted by the SYMBOL-NAME of the CAR.

  '((*ACL2-SYSTEM-EXPORTS* "[books]/system/acl2-system-exports.lisp")
    (<< "[books]/misc/total-order.lisp")
    (ADD-IO-PAIRS "[books]/std/util/add-io-pairs.lisp")
    (APPEND-WITHOUT-GUARD "[books]/std/lists/flatten.lisp")
    (OSLIB::ARGV "[books]/oslib/argv-logic.lisp")
    (ARITH-EQUIVS "[books]/std/basic/arith-equiv-defs.lisp")
    (ARITHMETIC "[books]/doc/more-topics.lisp")
    (ARITHMETIC-1 "[books]/arithmetic/top.lisp")
    (ARITHMETIC/NATP-POSP "[books]/arithmetic/natp-posp.lisp")
    (ARITY+ "[books]/kestrel/std/system/arity-plus.lisp")
    (ASSERT! "[books]/std/testing/assert-bang.lisp")
    (ASSERT!-STOBJ "[books]/std/testing/assert-bang-stobj.lisp")
    (B* "[books]/std/util/bstar.lisp")
    (BRIDGE "[books]/centaur/bridge/top.lisp")
    (BUILD::CERT.PL "[books]/build/doc.lisp")
    (BUILD::CERT_PARAM "[books]/build/doc.lisp")
    (CGEN "[books]/acl2s/cgen/top.lisp")
    (CONSIDERATION "[books]/hints/consider-hint.lisp")
    (BUILD::CUSTOM-CERTIFY-BOOK-COMMANDS "[books]/build/doc.lisp")
    (STD::DEFAGGREGATE "[books]/std/util/defaggregate.lisp")
    (DEFCONSTS "[books]/std/util/defconsts.lisp")
    (DEFDATA "[books]/acl2s/defdata/top.lisp")
    (DEFINE "[books]/std/util/define.lisp")
    (DEFMAC "[books]/misc/defmac.lisp")
    (DEFMACROQ "[books]/kestrel/utilities/defmacroq.lisp")
    (FTY::DEFPROD "[books]/centaur/fty/top.lisp")
    (DEFPUN "[books]/misc/defpun.lisp")
    (DEFTHM<W "[books]/kestrel/utilities/auto-instance.lisp")
    (DEFTHMG "[books]/tools/defthmg.lisp")
    (ACL2S::DEFUNC "[books]/acl2s/defunc.lisp")
    (DEFXDOC "[books]/xdoc/topics.lisp")
    (GETOPT-DEMO::DEMO2 "[books]/centaur/getopt/demo2.lisp")
    (DEVELOPERS-GUIDE "[books]/system/doc/developers-guide.lisp")
    (DEVELOPERS-GUIDE-UTILITIES "[books]/system/doc/developers-guide.lisp")
    (DO-NOT-HINT "[books]/tools/do-not.lisp")
    (EASY-SIMPLIFY-TERM "[books]/tools/easy-simplify.lisp")
    (ER-SOFT+ "[books]/kestrel/utilities/er-soft-plus.lisp")
    (ER-SOFT-LOGIC "[books]/tools/er-soft-logic.lisp")
    (FINAL-CDR "[books]/std/lists/final-cdr.lisp")
    (FTY "[books]/centaur/fty/top.lisp")
    (GETOPT "[books]/centaur/getopt/top.lisp")
    (GL "[books]/centaur/gl/doc.lisp")
    (HACKER "[books]/hacking/hacking-xdoc.lisp")
    (IHS "[books]/ihs/ihs-doc-topic.lisp")
    (INCLUDE-RAW "[books]/tools/include-raw.lisp")
    (INSTALL-NOT-NORMALIZED "[books]/misc/install-not-normalized.lisp")
    (LIST-EQUIV "[books]/std/lists/equiv.lisp")
    (LIST-FIX "[books]/std/lists/list-fix.lisp")
    (LOGBITP-REASONING "[books]/centaur/bitops/equal-by-logbitp.lisp")
    (MAKE-FLAG "[books]/tools/flag.lisp")
    (MAKE-TERMINATION-THEOREM
     "[books]/kestrel/utilities/make-termination-theorem.lisp")
    (MEMOIZED-PROVER-FNS "[books]/tools/memoize-prover-fns.lisp")
    (STR::NATSTR "[books]/std/strings/decimal.lisp")
    (NON-PARALLEL-BOOK "[books]/std/system/non-parallel-book.lisp")
    (NOTE-6-4-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-6-5-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-7-0-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-7-1-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-7-2-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-8-0-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-8-1-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-8-2-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-8-3-BOOKS "[books]/doc/relnotes.lisp")
    (NOTE-8-4-BOOKS "[books]/doc/relnotes.lisp")
    (STR::NUMBERS "[books]/std/strings/top.lisp")
    (OPEN-TRACE-FILE! "[books]/tools/open-trace-file-bang.lisp")
    (ORACLE-TIMELIMIT "[books]/tools/oracle-timelimit.lisp")
    (OSLIB "[books]/oslib/top-logic.lisp")
    (PATBIND-THE "[books]/std/util/bstar.lisp")
    (BUILD::PRE-CERTIFY-BOOK-COMMANDS "[books]/build/doc.lisp")
    (XDOC::PREPROCESSOR "[books]/xdoc/topics.lisp")
    (STR::PRETTY "[books]/std/strings/pretty.lisp")
    (STR::PRETTY-PRINTING "[books]/std/strings/pretty.lisp")
    (PROFILE-ACL2 "[books]/centaur/memoize/old/profile.lisp")
    (PROFILE-ALL "[books]/centaur/memoize/old/profile.lisp")
    (QUICKLISP "[books]/quicklisp/top.lisp")
    (RELEASE-NOTES-BOOKS "[books]/doc/relnotes.lisp")
    (REMOVABLE-RUNES "[books]/tools/removable-runes.lisp")
    (REMOVE-HYPS "[books]/tools/remove-hyps.lisp")
    (REWRITE$ "[books]/tools/rewrite-dollar.lisp")
    (REWRITE-EQUIV-HINT "[books]/coi/util/rewrite-equiv.lisp")
    (RUN-SCRIPT "[books]/tools/run-script.lisp")
    (SATLINK::SAT-SOLVER-OPTIONS "[books]/centaur/satlink/top.lisp")
    (SATLINK "[books]/centaur/satlink/top.lisp")
    (XDOC::SAVE "[books]/xdoc/topics.lisp")
    (XDOC::SAVE-RENDERED "[books]/xdoc/topics.lisp")
    (XDOC::SAVE-RENDERED-EVENT "[books]/xdoc/topics.lisp")
    (SET-MAX-MEM "[books]/centaur/misc/memory-mgmt-logic.lisp")
    (SPACEWALK "[books]/centaur/misc/spacewalk.lisp")
    (STD "[books]/std/top.lisp")
    (STD/IO "[books]/std/io/top.lisp")
    (STD/STRINGS "[books]/std/strings/top.lisp")
    (STD/UTIL "[books]/std/util/top.lisp")
    (STD::STRICT-LIST-RECOGNIZERS "[books]/std/util/deflist-base.lisp")
    (SUBSEQ-LIST "[books]/std/lists/subseq.lisp")
    (TRANS-EVAL-ERROR-TRIPLE "[books]/kestrel/utilities/trans-eval-error-triple.lisp")
    (TRANS-EVAL-STATE "[books]/kestrel/utilities/trans-eval-error-triple.lisp")
    (UNSOUND-READ "[books]/std/io/unsound-read.lisp")
    (UNTRANSLATE-PATTERNS "[books]/misc/untranslate-patterns.lisp")
    (USE-TRIVIAL-ANCESTORS-CHECK "[books]/tools/trivial-ancestors-check.lisp")
    (BUILD::USING-EXTENDED-ACL2-IMAGES "[books]/build/doc.lisp")
    (WITH-RAW-MODE "[books]/hacking/hacking-xdoc.lisp")
    (WITH-REDEF-ALLOWED "[books]/hacking/hacking-xdoc.lisp")
    (WITH-TIMEOUT "[books]/acl2s/cgen/with-timeout.lisp")
    (WITHOUT-SUBSUMPTION "[books]/tools/without-subsumption.lisp")
    (WORKING-WITH-PACKAGES "[books]/doc/practices.lisp")
    (WRITE-LIST "[books]/misc/file-io.lisp")
    (XDOC "[books]/xdoc/topics.lisp")))

(defconst *acl2-url*

; Warning: This event appears identically in acl2-doc.lisp and
; acl2-doc-wrap.lisp.  If you change one, then change the other the same way!

; We use the specific release rather than "current" in the URL, so that those
; who are looking at an older version of ACL2 will see the corresponding
; ACL2+Books Manual at this link.

  "http://www.cs.utexas.edu/users/moore/acl2/v8-4/")

(defconst *installation-url*

; Warning: This event appears identically in acl2-doc.lisp and
; acl2-doc-wrap.lisp.  If you change one, then change the other the same way!

  (concatenate 'string *acl2-url* "HTML/installation/installation.html"))

(defun combined-manual-ref ()

; Warning: This event appears identically in acl2-doc.lisp and
; acl2-doc-wrap.lisp.  If you change one, then change the other the same way!

  (concatenate
   'string
   "<a href='"
   (concatenate 'string *acl2-url* "combined-manual/index.html") ; url
   "'>"
   "ACL2+Books Manual"
   "</a>"))

(defun clhs (url title)

; Warning: This event appears identically in acl2-doc.lisp and
; acl2-doc-wrap.lisp.  If you change one, then change the other the same way!

  (declare (xargs :guard (and (or (not url)
                                  (stringp url))
                              (stringp title))))
  (concatenate 'string
               "<a href='http://www.lispworks.com/documentation/HyperSpec/"
               (or url "")
               "'>" title "</a>"))

(defxdoc *
  :parents (numbers acl2-built-ins)
  :short "Multiplication macro"
  :long "<p>@('*') is really a macro that expands to calls of the function
 @(tsee binary-*).  So for example</p>

 @({
  (* x y 4 z)
 })

 <p>represents the same term as</p>

 @({
  (binary-* x (binary-* y (binary-* 4 z))).
 })

 <p>See @(see binary-*).</p>

 <p>@('*') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>")

(defxdoc *acl2-exports*
   :parents (packages acl2-built-ins)
   :short "Symbols that are often imported into new @(see packages) to provide
 easy access to ACL2 functionality."

   :long "<p>When you define a new package for your own work with @(see defpkg),
 you will usually want to import many symbols from the @('\"ACL2\"') package;
 for instance you will usually want to be able to use symbols like @(see
 defthm), @(see in-theory), @(see xargs), @(see state), etc., without an
 @('acl2::') prefix.</p>

 <p>The constant @('*acl2-exports*') lists @(`(len *acl2-exports*)`) symbols,
 including most documented ACL2 system constants, functions, and macros.  You
 will typically also want to import many symbols from Common Lisp; see @(see
 *common-lisp-symbols-from-main-lisp-package*).</p>

 <p>Those who write code using built-in ACL2 functions (see @(see
 acl2-built-ins)) may wish to import symbols into their package from the large
 list @(tsee *acl2-system-exports*).</p>

 @(`(:code *acl2-exports*)`)")

 (defxdoc *common-lisp-symbols-from-main-lisp-package*
   :parents (packages acl2-built-ins)
   :short "Symbols that are often imported into new packages to provide easy
 access to Common Lisp functionality."

   :long "<p>When you define a new package for your own work with @(see defpkg),
 you will usually want to import many symbols from the @('\"COMMON-LISP\"')
 package so that you can access them without a @('common-lisp::') or @('acl2::')
 prefix.</p>

 <p>The constant @('*common-lisp-symbols-from-main-lisp-package*') lists the
 @(`(len *common-lisp-symbols-from-main-lisp-package*)`) symbols of the
 @('COMMON-LISP') package found in <a
 href='http://dx.doi.org/10.1145/147135.147249'>dpAns</a>.  You will typically
 also want to import many symbols from ACL2; see @(see *acl2-exports*).</p>

 @(`(:code *common-lisp-symbols-from-main-lisp-package*)`)")

(defxdoc *standard-ci*
  :parents (io acl2-built-ins)
  :short "An ACL2 character-based analogue of CLTL's @('*standard-input*')"
  :long "<p>The value of the ACL2 constant @('*standard-ci*') is an open
 character input channel that is synonymous to Common Lisp's
 @('*standard-input*').</p>

 <p>ACL2 character input from @('*standard-ci*') is actually obtained by
 reading @(see characters) from the stream named by Common Lisp's
 @('*standard-input*').  That is, by changing the setting of
 @('*standard-input*') in raw Common Lisp you can change the source from which
 ACL2 reads on the channel @('*standard-ci*').  See @(see *standard-co*).</p>")

(defxdoc *standard-co*
  :parents (io acl2-built-ins)
  :short "The ACL2 analogue of CLTL's @('*standard-output*')"
  :long "<p>The value of the ACL2 constant @('*standard-co*') is an open
 character output channel that is synonymous to Common Lisp's
 @('*standard-output*').</p>

 <p>ACL2 character output to @('*standard-co*') will go to the stream named by
 Common Lisp's @('*standard-output*').  That is, by changing the setting of
 @('*standard-output*') in raw Common Lisp you can change the actual
 destination of ACL2 output on the channel named by @('*standard-co*').
 Observe that this happens without changing the logical value of
 @('*standard-co*') (which is some channel symbol).  Changing the setting of
 @('*standard-output*') in raw Common Lisp essentially just changes the map
 that relates ACL2 to the physical world of terminals, files, etc.</p>

 <p>To see the value of this observation, consider the following.  Suppose you
 write an ACL2 function which does character output to the constant channel
 @('*standard-co*').  During testing you see that the output actually goes to
 your terminal.  Can you use the function to output to a file?  Yes, if you are
 willing to do a little work in raw Common Lisp: open a stream to the file in
 question, set @('*standard-output*') to that stream, call your ACL2 function,
 and then close the stream and restore @('*standard-output*') to its nominal
 value.  Similar observations can be made about the two ACL2 input channels,
 @(tsee *standard-oi*) and @(tsee *standard-ci*), which are analogues of
 @('*standard-input*').</p>

 <p>Another reason you might have for wanting to change the actual streams
 associated with @(tsee *standard-oi*) and @('*standard-co*') is to drive the
 ACL2 top-level loop, @(tsee ld), on alternative input and output streams.
 This end can be accomplished easily within ACL2 by either calling @(tsee ld)
 on the desired channels or file names or by resetting the ACL2 @(tsee state)
 global variables @(''')@(tsee standard-oi) and @(''')@(tsee standard-co) which
 are used by @(tsee ld).  See @(see standard-oi) and see @(see
 standard-co).</p>")

(defxdoc *standard-oi*
  :parents (io acl2-built-ins)
  :short "An ACL2 object-based analogue of CLTL's @('*standard-input*')"
  :long "<p>The value of the ACL2 constant @('*standard-oi*') is an open object
 input channel that is synonymous to Common Lisp's @('*standard-input*').</p>

 <p>ACL2 object input from @('*standard-oi*') is actually obtained by reading
 from the stream named by Common Lisp's @('*standard-input*').  That is, by
 changing the setting of @('*standard-input*') in raw Common Lisp you can
 change the source from which ACL2 reads on the channel @('*standard-oi*').
 See @(see *standard-co*).</p>")

(defxdoc +
  :parents (numbers acl2-built-ins)
  :short "Addition macro"
  :long "<p>@('+') is really a macro that expands to calls of the function
 @(tsee binary-+).  So for example</p>

 @({
  (+ x y 4 z)
 })

 <p>represents the same term as</p>

 @({
  (binary-+ x (binary-+ y (binary-+ 4 z))).
 })

 <p>See @(see binary-+).</p>

 @(def +)")

(defxdoc -
  :parents (numbers acl2-built-ins)
  :short "Macro for subtraction and negation"
  :long "<p>See @(see binary-+) for addition and see @(see unary--) for
 negation.</p>

 <p>Note that @('-') represents subtraction as follows:</p>

 @({
  (- x y)
 })

 <p>represents the same term as</p>

 @({
  (+ x (- y))
 })

 <p>which is really</p>

 @({
  (binary-+ x (unary-- y)).
 })

 <p>Also note that @('-') represents arithmetic negation as follows:</p>

 @({
  (- x)
 })

 <p>expands to</p>

 @({
  (unary-- x).
 })

 @(def -)")

(defxdoc /
  :parents (numbers acl2-built-ins)
  :short "Macro for division and reciprocal"
  :long "<p>See @(see binary-*) for multiplication and see @(see unary-/) for
 reciprocal.</p>

 <p>Note that @('/') represents division as follows:</p>

 @({
  (/ x y)
 })

 <p>represents the same term as</p>

 @({
  (* x (/ y))
 })

 <p>which is really</p>

 @({
  (binary-* x (unary-/ y)).
 })

 <p>Also note that @('/') represents reciprocal as follows:</p>

 @({
  (/ x)
 })

 <p>expands to</p>

 @({
  (unary-/ x).
 })

 <p>@('/') is a Common Lisp macro.  See any Common Lisp documentation for more
 information.</p>

 @(def /)")

(defxdoc /=
  :parents (numbers acl2-built-ins)
  :short "Test inequality of two numbers"
  :long "<p>@('(/= x y)') is logically equivalent to @('(not (equal x
  y))').</p>

 <p>Unlike @(tsee equal), @('/=') has a @(see guard) requiring both of its
 arguments to be numbers.  Generally, @('/=') is executed more efficiently than
 a combination of @(tsee not) and @(tsee equal).</p>

 <p>For a discussion of the various ways to test against 0, See @(see
 zero-test-idioms).</p>

 <p>@('/=') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def /=)")

(defxdoc 1+
  :parents (numbers acl2-built-ins)
  :short "Increment by 1"
  :long "<p>@('(1+ x)') is the same as @('(+ 1 x)').  See @(see +).</p>

 <p>@('1+') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def 1+)")

(defxdoc 1-
  :parents (numbers acl2-built-ins)
  :short "Decrement by 1"
  :long "<p>@('(1- x)') is the same as @('(- x 1)').  See @(see -).</p>

 <p>@('1-') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def 1-)")

(defxdoc <
  :parents (numbers acl2-built-ins)
  :short "Less-than"
  :long "<p>Completion Axiom (@('completion-of-<')):</p>

 @({
  (equal (< x y)
         (if (and (rationalp x)
                  (rationalp y))
             (< x y)
           (let ((x1 (if (acl2-numberp x) x 0))
                 (y1 (if (acl2-numberp y) y 0)))
             (or (< (realpart x1) (realpart y1))
                 (and (equal (realpart x1) (realpart y1))
                      (< (imagpart x1) (imagpart y1)))))))
 })

 <p>@(see Guard) for @('(< x y)'):</p>

 @({
  (and (rationalp x) (rationalp y))
 })

 <p>Notice that like all arithmetic functions, @('<') treats non-numeric inputs
 as @('0'). Thus, the following are theorems.</p>

 @({
  (thm (equal (< (fix x) y) (< x y)))
  (thm (equal (< x (fix y)) (< x y)))
 })

 <p>This function has the usual meaning on the rational numbers, but is
 extended to the complex rational numbers using the lexicographic order: first
 the real parts are compared, and if they are equal, then the imaginary parts
 are compared.</p>")

(defxdoc <=
  :parents (numbers acl2-built-ins)
  :short "Less-than-or-equal test"
  :long "<p>@('<=') is a macro, and @('(<= x y)') expands to the same thing as
 @('(not (< y x))').  See @(see <).</p>

 <p>@('<=') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def <=)")

(defxdoc =
  :parents (numbers equal equality-variants acl2-built-ins)
  :short "Test equality of two numbers"
  :long "<p>@('(= x y)') is logically equivalent to @('(equal x y)').</p>

 <p>Unlike @(tsee equal), @('=') has a @(see guard) requiring both of its
 arguments to be numbers.  Generally, @('=') is executed more efficiently than
 @(tsee equal).</p>

 <p>For a discussion of the various ways to test against 0, See @(see
 zero-test-idioms).</p>

 <p>@('=') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def =)")

(defxdoc >
  :parents (numbers acl2-built-ins)
  :short "Greater-than test"
  :long "<p>@('>') is a macro, and @('(> x y)') expands to the same thing as
 @('(< y x)').  See @(see <).</p>

 <p>@('>') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def >)")

(defxdoc >=
  :parents (numbers acl2-built-ins)
  :short "Greater-than-or-equal test"
  :long "<p>@('>=') is a macro, and @('(>= x y)') expands to the same thing as
 @('(not (< x y))').  See @(see <).</p>

 <p>@('>=') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def >=)")

(defxdoc @
  :parents (programming-with-state acl2-built-ins)
  :short "Get the value of a global variable in @(tsee state)"
  :long "@({
  Examples:
  (+ (@ y) 1)
  (assign a (aset1 'ascii-map-array (@ a) 66 'Upper-case-B))

  General Form:
  (@ symbol)
 })

 <p>where @('symbol') is any symbol to which you have @(tsee assign)ed a global
 value.  This macro expands into @('(f-get-global 'symbol state)'), which
 retrieves the stored value of the symbol.</p>

 <p>The macro @('f-get-global') is closely related to @(tsee @): @('(@ var)')
 macroexpands to @('(f-get-global 'var state)').</p>

 <p>The macro @(tsee assign) makes it convenient to set the value of a symbol.
 The @(':')@(tsee ubt) operation has no effect on the @('global-table') of
 @(tsee state).  Thus, you may use these globals to hang onto useful data
 structures even though you may undo back past where you computed and saved
 them.</p>")

(defxdoc |A Flying Tour of ACL2|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Flying Tour of ACL2"
  :long "<p><see topic='@(url |About the ACL2 Home Page|)'><img
 src='res/tours/large-flying.gif'></img></see></p>

 <p>On this tour you will learn a little about what ACL2 is for rather than how
 ACL2 works.  At the top and bottom of the ``page'' there are ``flying tour''
 icons.  Click on either icon to go to the next page of the tour.</p>

 <p>The tour visits the following topics sequentially.  But on your first
 reading, don't navigate through the tour by clicking on these links; they are
 shown as live links only so that later you can determine what you've visited.
 Instead, just use the flying tour icons.</p>

 <code>
 <b>The Flight Plan</b>
 * <see topic='@(url |About the ACL2 Home Page|)'>This Documentation</see>
 * <see topic='@(url |What Is ACL2(Q)|)'>What is ACL2?</see>
 * <see topic='@(url |What is a Mathematical Logic(Q)|)'>Mathematical Logic</see>
 * <see topic='@(url |What is a Mechanical Theorem Prover(Q)|)'>Mechanical Theorem Proving</see>
 * <see topic='@(url |About Models|)'>Mathematical Models in General</see>
 * <see topic='@(url |Models of Computer Hardware and Software|)'>Mathematical Models of Computing Machines</see>
      <see topic='@(url |A Typical State|)'>Formalizing Models</see>
      <see topic='@(url |Running Models|)'>Running Models</see>
      <see topic='@(url |Symbolic Execution of Models|)'>Symbolic Execution of Models</see>
      <see topic='@(url |Proving Theorems about Models|)'>Proving Theorems about Models</see>
 * Requirements of ACL2
      <see topic='@(url |What is Required of the User(Q)|)'>The User's Skills</see>
      <see topic='@(url |How Long Does It Take to Become an Effective User(Q)|)'>Training</see>
      <see topic='@(url |Other Requirements|)'>Host System</see>
 </code>

 <p>On your first reading, don't explore other links you see in the tour.  Some
 of them lead to the Walking Tour, which you can take coherently when you
 finish this tour.  Others lead into the extensive hypertext documentation and
 you are liable to get lost there unless you're trying to answer a specific
 question.  We intend the tour to take about 10 minutes of your time.</p>

 <p><see topic='@(url |About the ACL2 Home Page|)'><img
 src='res/tours/flying.gif'></img></see></p>")

(defxdoc |A Sketch of How the Rewriter Works|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Sketch of How the Rewriter Works"
  :long "<p>Below we show the first target term, extracted from the current
 conjecture.  Below it we show the associativity rule.</p>

 <p><img src='res/tours/uaa-rewrite.gif'></img></p>

 <p>The variables of the rewrite rule are <b>instantiated</b> so that the
 <b>left-hand side</b> of the rule matches the target:</p>

 @({
       variable          term from target
         a                     x1
         b                     x2
         c                     (app x3 x4)
 })

 <p>Then the target is <b>replaced</b> by the instantiated <b>right-hand
 side</b> of the rule.</p>

 <p>Sometimes rules have <b>hypotheses</b>.  To make a long story short, if the
 rule has hypotheses, then after matching the left-hand side, the rewriter
 instantiates the hypotheses and rewrites them recursively.  This is called
 <b>backchaining</b>.  If they all rewrite to true, then the target is replaced
 as above.</p>

 <p>We discuss the rewriter in more detail in the extended introduction to how
 to use the theorem prover, see @(see introduction-to-the-theorem-prover),
 which we will recommend you work through <b>after</b> you have finished the
 two tours.</p>")

(defxdoc |A Tiny Warning Sign|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Tiny Warning Sign"
  :long "<p><img src='res/tours/warning.gif'></img></p>

 <p>This warning sign, which usually appears as ``<icon src='res/tours/twarning.gif'/>'',
 indicates that the link it marks takes you into ACL2's online
 documentation.</p>

 <p>The documentation is a vast graph of documented topics intended to help the
 <i>user</i> of ACL2 rather than the <i>potential user</i>.  If you are
 exploring ACL2's home page to learn about the system, perhaps you should go
 back rather than follow the link marked with this sign.  But you are welcome
 to explore the online documentation as well.  Good luck.</p>")

(defxdoc |A Trivial Proof|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Trivial Proof"
  :long "<p><img src='res/tours/concrete-proof.gif'></img></p>")

(defxdoc |A Typical State|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Typical State"
  :long "<p><see topic='@(url |Functions for Manipulating these Objects|)'><img
 src='res/tours/flying.gif'></img></see></p>

 <p><img src='res/tours/state-object.gif'></img></p>

 <p>Observe that the states in typical models talk about</p>

 <code>
 <b>booleans</b>    <b>integers</b>   <b>vectors</b>     <b>records</b>   <b>caches</b>
 <b>bits</b>        <b>symbols</b>    <b>arrays</b>      <b>stacks</b>    <b>files</b>
 <b>characters</b>  <b>strings</b>    <b>sequences</b>   <b>tables</b>    <b>directories</b>
 </code>

 <p>These objects are <b>discrete</b> rather than <b>continuous</b>;
 furthermore they are built incrementally or <b>inductively</b> by repeatedly
 using primitive operations to put together smaller pieces.</p>

 <p>The functions we need to manipulate these objects do things like
 <b>concatenate</b>, <b>reverse</b>, <b>sort</b>, <b>search</b>, <b>count</b>,
 etc.</p>

 <p><see topic='@(url |Functions for Manipulating these Objects|)'><img
 src='res/tours/flying.gif'></img></see></p>")

(defxdoc |A Walking Tour of ACL2|
  :parents (|Pages Written Especially for the Tours|)
  :short "A Walking Tour of ACL2"
  :long "<p><see topic='@(url |Common Lisp|)'><img
 src='res/tours/large-walking.gif'></img></see></p>

 <p>On this tour you will learn a little more about the ACL2 logic, the theorem
 prover, and the user interface.</p>

 <p>This time we will stick with really simple things, such as the
 associativity of list concatenation.</p>

 <p>We assume you have taken the Flying Tour but that you did not necessarily
 follow all the ``off-tour'' links because we encouraged you not to.  With the
 Walking Tour we encourage you to visit off-tour links &mdash; provided they
 are not marked with the tiny warning sign (<see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>).
 But they are ``branches'' in the tour that lead to ``dead ends.''  When you
 reach a dead end, remember to use your browser's Back Button to return to the
 Walking Tour to continue.</p>

 <p>When you get to the end of the tour we'll give you a chance to repeat
 quickly both the Flying and the Walking Tours to visit any off-tour links
 still of interest.</p>

 <p><see topic='@(url |Common Lisp|)'><img src='res/tours/walking.gif'></img></see></p>")

(defxdoc a!
  :parents (ld)
  :short "To return to the top-level of ACL2's command loop"
  :long "<p>When @('(a!)') is evaluated inside of ACL2's command loop, the
 current computation is aborted and control returns to the top of the command
 loop, exactly as though the user had interrupted and aborted the current
 computation.  (Note: Versions of ACL2 up to Version_3.4 provided `@('#.')' for
 this purpose, but no longer; see @(see sharp-dot-reader).)</p>

 <p>If you are at an ACL2 prompt (as opposed to a raw Lisp break), then you may
 type @(':a!') in place of @('(a!)'); see @(see keyword-commands).</p>

 <p>For a related feature that only pops up one level, see @(see p!).</p>

 <p>Logically speaking, @('(a!) = nil').  But imagine that it is defined in
 such a way that it causes a stack overflow or other resource exhaustion when
 called.</p>")

(defxdoc abort!
  :parents (ld)
  :short "To return to the top-level of ACL2's command loop"
  :long "<p>This is an alias for @('a!'); see @(see a!).  For a related feature
 that only pops up one level, see @(see p!).</p>")

(defxdoc about-acl2
  :parents (acl2)
  :short "General information About ACL2"
  :long "<p>This is @(`(:raw (@ acl2-version))`), @(see copyright) (C) 2021,
 Regents of the University of Texas, authored by Matt Kaufmann and J Strother
 Moore.</p>

 <p>See the <a href='http://www.cs.utexas.edu/users/moore/acl2/'>ACL2 home
 page</a> for additional information including tutorials, installation
 instructions, mailing lists, related publications, ACL2 workshops and
 seminars, acknowledgments, and other ACL2 releases.</p>

 <p>See @(see documentation) for how to access the ACL2 User's Manual.</p>

 <p>For statistics on ACL2 code size, see file @('doc/acl2-code-size.txt').</p>")

(defxdoc abs
  :parents (numbers acl2-built-ins)
  :short "The absolute value of a real number"
  :long "<p>@('(Abs x)') is @('-x') if @('x') is negative and is @('x')
 otherwise.</p>

 <p>The @(see guard) for @('abs') requires its argument to be a rational (@(see
 real), in ACL2(r)) number.</p>

 <p>@('Abs') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 <p>From ``Common Lisp the Language'' page 205, we must not allow complex
 @('x') as an argument to @('abs') in ACL2, because if we did we would have to
 return a number that might be a floating point number and hence not an ACL2
 object.</p>

 @(def abs)")

(defxdoc access
  :parents (defrec acl2-built-ins)
  :short "Accessor macro for @(see defrec) structures."
  :long "<p>The @('access') macro is built into ACL2, and allows you to access
 particular fields from structures that have been introduced with @(see
 defrec).  For instance:</p>

 @({
    (access employee x :name)
 })

 <p>would return the @('name') field from the employee @('x').  See @(see
 defrec) for more information.</p>")

(defxdoc accumulated-persistence
  :parents (debugging)
  :short "To get statistics on which @(see rune)s are being tried"
  :long "@({
  Useful Forms:
  (accumulated-persistence t)              ; Activate statistics gathering.
  (accumulated-persistence :all)           ; As above, ``enhanced'' (see below)

  (show-accumulated-persistence :frames)   ; Display statistics ordered by
  (show-accumulated-persistence :tries)    ; frames built, times tried,
  (show-accumulated-persistence :ratio)    ; or their ratio.
  (show-accumulated-persistence)           ; Same as supplying :frames argument.

  (accumulated-persistence nil)            ; Deactivate.

  (accumulated-persistence-oops)           ; Undo the clearing effect of
                                           ; (accumulated-persistence t).

  Advanced forms:
  (show-accumulated-persistence :frames-s) ; The `s', `f', and `a' suffixes
  (show-accumulated-persistence :frames-f) ; stand for `success' (`useful'),
  (show-accumulated-persistence :frames-a) ; `failure' (`useless'), and `all',
  (show-accumulated-persistence :tries-s)  ; respectively.  The only effect of
  (show-accumulated-persistence :tries-f)  ; the `s' and `f' versions is to
  (show-accumulated-persistence :tries-a)  ; sort first by useful or useless
                                           ; applications, respectively (see
                                           ; below).  The `a' versions avoid
                                           ; showing the useful/useless
                                           ; breakdown.

  (show-accumulated-persistence :runes)    ; Just show runes alphabetically.
  (show-accumulated-persistence :useless)  ; Just show useless runes.
  (show-accumulated-persistence :useless :list)
                                           ; Just show useless runes as a list.
 })

 <p>In summary, @('(accumulated-persistence t)') turns on fresh statistics
 gathering for rules, @('(accumulated-persistence nil)') turns it off,
 @('(show-accumulated-persistence)') displays the statistics that were
 gathered, and @('(accumulated-persistence-oops)') restores the statistics, if
 any, that were cleared by @('(accumulated-persistence t)') or
 @('(accumulated-persistence :all)').</p>

 <p>In general, if the optional second argument of
 @('show-accumulated-persistence') is supplied as @(':list'), then instead of
 the result being displayed a ``pretty'' human-readable format, the result will
 be displayed as a corresponding list of entries of the form @('(frames tries
 xrune)').</p>

 <p>Note: @('set-accumulated-persistence') is equivalent to
 @('accumulated-persistence').</p>

 <p>See the end of this item for a discussion of ``enhanced statistics
 gathering,'' which can be useful for more fine-grained proof debugging.</p>

 <p>Generally speaking, the more ACL2 knows, the slower it runs.  That is
 because the search space grows with the number of alternative rules.  Often,
 the system tries to apply rules that you have forgotten were even there, if
 you knew about them in the first place!  ``Accumulated-persistence'' is a
 statistic (originally developed for Nqthm) that helps you identify the rules
 that are causing ACL2's search space to explode.</p>

 <p>For other proof debugging utilities, see @(see break-rewrite) and see @(see
 dmr).</p>

 <p>Accumulated persistence tracking can be turned on or off.  It is generally
 off.  When on, proofs may take a little more time than otherwise.  (We
 measured approximately 11% more time in a so-called ``everything'' regression
 run in May 2020.)  But some useful numbers are collected.  When it is turned
 on, by</p>

 @({
  ACL2 !>(accumulated-persistence t)
 })

 <p>an accumulation site is initialized and henceforth data about which rules
 are being tried is accumulated into that site.  That accumulated data can be
 displayed with @('show-accumulated-persistence'), as described in detail
 below.  When accumulated persistence is turned off, with
 @('(accumulated-persistence nil)'), the accumulation site is wiped out and the
 data in it is lost.</p>

 <p>The ``accumulated persistence'' of a @(see rune) is the number of @(see
 rune)s the system has attempted to apply (since accumulated persistence was
 last activated) while the given @(see rune) was being tried.</p>

 <p>Consider a @(':')@(tsee rewrite) rule named @(tsee rune).  For simplicity,
 let us imagine that @(tsee rune) is tried only once in the period during which
 accumulated persistence is being monitored.  Recall that to apply a rewrite
 rule we must match the left-hand side of the conclusion to some term we are
 trying to rewrite, establish the hypotheses of @(tsee rune) by rewriting, and,
 if successful, then rewrite the right-hand side of the conclusion.  We say
 @(tsee rune) is ``being tried'' from the time we have matched its left-hand
 side to the time we have either abandoned the attempt or finished rewriting
 its right-hand side.  (By ``match'' we mean to include any loop-stopper
 requirement; see @(see loop-stopper).)  During that period of time other rules
 might be tried, e.g., to establish the hypotheses.  The rules tried while
 @(tsee rune) is being tried are ``billed'' to @(tsee rune) in the sense that
 they are being considered here only because of the demands of @(tsee rune).
 Thus, if no other rules are tried during that period, the accumulated
 persistence of @(tsee rune) is @('1') &mdash; we ``bill'' @(tsee rune) once
 for its own application attempt.  If, on the other hand, we tried @('10')
 rules on behalf of that application of @(tsee rune), then @(tsee rune)'s
 accumulated persistence would be @('11').</p>

 <p>One way to envision accumulated persistence is to imagine that every time a
 @(see rune) is tried it is pushed onto a stack.  The rules tried on behalf of
 a given application of a @(see rune) are thus pushed and popped on the stack
 above that @(see rune).  A lot of work might be done on its behalf &mdash; the
 stack above the @(see rune) grows and shrinks repeatedly as the search
 continues for a way to use the @(see rune).  All the while, the @(see rune)
 itself ``persists'' in the stack, until we finish with the attempt to apply
 it, at which time we pop it off.  The accumulated persistence of a @(see rune)
 application is thus the number of stack frames built while that @(see rune)
 was on the stack.</p>

 <p>Note that accumulated persistence is tallied whether or not the attempt to
 apply a @(see rune) is successful.  Each of the rules tried on its behalf
 might have failed and the attempt to apply the @(see rune) might have also
 failed.  The ACL2 proof script would make no mention of the @(see rune) or the
 rules tried on its behalf because they did not contribute to the proof.  But
 time was spent pursuing the possible application of the @(see rune) and
 accumulated persistence is a measure of that time.</p>

 <p>A high accumulated persistence might come about in two extreme ways.  One
 is that the rule causes a great deal of work every time it is tried.  The
 other is that the rule is ``cheap'' but is tried very often.  We therefore
 keep track of the number of times each rule is tried as well as its
 persistence.  The ratio between the two is the average amount of work done on
 behalf of the rule each time it is tried.</p>

 <p>We do not claim that tracking of runes for accumulated-persistence is
 perfect.  In practice, we believe it is quite reliable with the exception of
 @(see congruence) runes and, in some cases @(see executable-counterpart)
 runes. (For the latter, details are in a comment in the ACL2 source definition
 of function @('print-useless-runes').)</p>

 <p>When the accumulated persistence totals are displayed by the function
 @('show-accumulated-persistence') we sort them so that the most expensive
 @(see rune)s are shown first.  We can sort according to one of three basic
 keys:</p>

 @({
  :frames - the number of frames built on behalf of the rune
  :tries  - the number of times the rune was tried
  :ratio  - frames built per try
 })

 <p>The key simply determines the order in which the information is presented.
 If no argument is supplied to @('show-accumulated-persistence'), @(':frames')
 is used.</p>

 <p>The display breaks each total into ``useful'' and ``useless'' subtotals.  A
 ``useful'' rule try is one that is viewed as contributing to the progress of
 the proof, and the rest are ``useless'' rule applications.  For example, if a
 @(':')@(tsee rewrite) rule is tried but its hypotheses are not successfully
 relieved, then that rule application and all work done on behalf of those
 hypotheses is ``useless'' work.  In general, an attempt to apply a @(see rune)
 is viewed as ``useful'' unless the attempt fails or the attempt is on the
 stack (as described above) for a @(see rune) application that ultimately
 fails.  A large number of ``useless'' @(':frames') or @(':tries') along with
 correspondingly small ``useful'' counts may suggest @(see rune)s to consider
 disabling (see @(see disable) and see @(see in-theory)).  Thus, here is a more
 complete list of the arguments that may be supplied to
 @('show-accumulated-persistence').  Suffixes ``s'', ``f'', and ``a'' are
 intended to suggest ``success'' (``useful''), ``failure'' (``useless''), and
 ``all''.</p>

 @({
  :frames     - sort by the number of frames built on behalf of the rune
     :frames-s -   as above, but sort by useful applications
     :frames-f -   as above, but sort by useless applications
     :frames-a -   as above, but inhibit display of ``useful'' and
                   ``useless'' subtotals
  :tries      - sort by the number of times the rune was tried
     :tries-s  -   as above, but sort by useful applications
     :tries-f  -   as above, but sort by useless applications
     :tries-a  -   as above, but inhibit display of ``useful'' and
                   ``useless'' subtotals
  :ratio      - sort by frames built per try
  :useless    - show only the runes tried whose tries were all ``useless''
 })

 <p>For a given line of the report, every frame credited to a ``useful''
 (respectively, ``useless'') rule application is considered ``useful''
 (respectively, ``useless'').  We illustrate with the following example.</p>

 @({
  (progn
    (defstub hyp (x) t)
    (defstub concl (x) t)
    (defstub bad (x) t)
    (defstub good (x) t)
    (defaxiom good-ax
      (implies (good x) (hyp x)))
    (defaxiom bad-ax
      (implies (bad x) (hyp x)))
    (defaxiom hyp-implies-concl
      (implies (hyp x) (concl x)))
    )
  (accumulated-persistence t)
  (thm (implies (good x) (concl x)))
  (show-accumulated-persistence)
 })

 <p>To prove the @(tsee thm) form, ACL2 attempts to rewrite @('(concl x)') to
 true by applying rule @('hyp-implies-concl').  It then attempts to establish
 @('(hyp x)') first by trying rule @('bad-ax'), which fails, and second by
 trying rule @('good-ax'), which succeeds.  As expected, the report labels as
 ``useless'' the failure of the attempt to establish the hypothesis, @('(bad
 x)').</p>

 @({
     --------------------------------
           1        1 (    1.00) (:REWRITE BAD-AX)
           0        0    [useful]
           1        1    [useless]
     --------------------------------
 })

 <p>Now consider the top-level application of rule @('hyp-implies-concl').
 Even though the above report shows the application of @('bad-ax') as
 ``useless'', note that this rule was applied on behalf of the successful
 (``useful'') application of @('hyp-implies-concl'), and hence is incorporated
 into the ``useful'' line for @('hyp-implies-concl'), as follows.</p>

 @({
     --------------------------------
           3        1 (    3.00) (:REWRITE HYP-IMPLIES-CONCL)
           3        1    [useful]
           0        0    [useless]
     --------------------------------
 })

 <p>In summary: categorization of @(':frames') as ``useful'' or ``useless'' is
 based on whether they support ``useful'' or ``useless'' @(':tries').</p>

 <p>See @(see useless-runes) for a way to speed up proofs by automatically
 turning off useless rules.</p>

 <p>Note that a @(see rune) with high accumulated persistence may not actually
 be the ``culprit.''  For example, suppose @('rune1') is reported to have a
 @(':ratio') of @('101'), meaning that on the average a hundred and one frames
 were built each time @('rune1') was tried.  Suppose @('rune2') has a
 @(':ratio') of @('100').  It could be that the attempt to apply @('rune1')
 resulted in the attempted application of @('rune2') and no other @(see rune).
 Thus, in some sense, @('rune1') is ``cheap'' and @('rune2') is the ``culprit''
 even though it is reported as costing less than @('rune1').</p>

 <p>If a proof is aborted, then in general, @(tsee
 show-accumulated-persistence) will only display totals for runes whose
 attempted application is complete: that is, if the rewriter was in the process
 of relieving hypotheses for a rule, then information for that rule will not be
 included in the tally.  We say ``in general'' because, as indicated near the
 top of the output from @(tsee show-accumulated-persistence) when such
 incomplete information is omitted, you can get this information by using
 argument @(':frames-a') or @(':tries-a').</p>

 <p>There are other subtleties in how rune applications are tallied, documented
 elsewhere: see @(see accumulated-persistence-subtleties).</p>

 <p>We conclude with a discussion of ``enhanced'' statistics gathering, which
 is enabled by supplying @('accumulated-persistence') the argument
 @(':ALL'):</p>

 @({
  (accumulated-persistence :all)
 })

 <p>At some additional performance expense (but probably well under a factor of
 2 altogether), ACL2 then gathers additional statistics for individual
 hypotheses of rules as well as their conclusions.  To understand how this
 works, suppose @('rn') is a @(see rune).  Then we prepend the keyword
 @(':CONC') to @('rn') to form what we call its ``conclusion xrune'', and for
 its @('I')-th hypothesis we prepend @(':HYP I') to @('rn') to form its
 @('I')-th ``hypothesis xrune.''  Here, ``xrune'' is pronounced ``ex rune'',
 and is mnemonic for ``extended rune.''  For example, if @('(REWRITE FOO)') is
 a @(see rune) then @('(:CONC REWRITE FOO)') is its conclusion xrune, and
 @('(:HYP 2 REWRITE FOO)') is a hypothesis xrune corresponding to the second
 hypothesis of the corresponding rewrite rule.</p>

 <p>With @('(accumulated-persistence :all)'), we instruct ACL2 to track not
 only runes but also xrunes.  Then, @('(show-accumulated-persistence)') will
 display information for all xrunes in a format that we consider to be ``raw'',
 in the sense that data for xrunes are displayed just as for runes.  But a
 ``merged'' format is also available.  Here is a summary of display commands,
 followed below by further discussion.</p>

 @({
    (show-accumulated-persistence :frames t) ; t is optional, i.e., the default
       ; Display enhanced statistics sorted by frames, in a ``raw'' format.
    (show-accumulated-persistence :frames :merge)
       ; Display enhanced statistics sorted by frames, in a ``merged'' format.
    (show-accumulated-persistence :frames nil)
       ; Display regular statistics sorted by frames
       ; (runes only, that is, without the enhancements).
    (show-accumulated-persistence :frames :merge)
       ; Display a list of entries (frames tries xrune), sorted by frames

    ; More generally, the descriptions just above apply for any legal first
    ; argument:

    (show-accumulated-persistence KEY t)
    (show-accumulated-persistence KEY :merge)
    (show-accumulated-persistence KEY nil)
    (show-accumulated-persistence KEY :list)

    ; Note also these alternate forms, equivalent to the first of the two forms
    ; just above, i.e., the form with second argument of t:
    (show-accumulated-persistence KEY :raw)
    (show-accumulated-persistence KEY)
 })

 <p>There is a significant difference between how runes are tracked and how
 ACL2 tracks hypothesis and conclusion xrunes: unlike regular runes, these
 xrunes do not contribute to the accumulated @(':frames') counts.  Rather, they
 serve as accumulation sites without contributing their @(':tries') to any
 accumulation.  Consider for example the snippet below, taken from a report
 created with the @(':merge') option (to be discussed further below), i.e., by
 evaluating the form @('(show-accumulated-persistence :frames :merge)').</p>

 @({
     :frames   :tries    :ratio  rune
     --------------------------------
         462      211 (    2.18) (:REWRITE PERM-MEM)
          13        6    [useful]
         449      205    [useless]
        .............................
         251       47 (    5.34) (:HYP 2 :REWRITE PERM-MEM)
           6        6    [useful]
         245       41    [useless]
        .............................
           0      211 (    0.00) (:HYP 1 :REWRITE PERM-MEM)
           0        6    [useful]
           0      205    [useless]
        .............................
           0        7 (    0.00) (:CONC :REWRITE PERM-MEM)
           0        6    [useful]
           0        1    [useless]
     --------------------------------
 })

 <p>Notice that while @(':tries') are recorded for the xrune @('(:HYP 1
 :REWRITE PERM-MEM)'), no @(':frames') are recorded.  This is because no stack
 frames were built for runes while this xrune was on the stack &mdash; only for
 the xrune itself, which as we explained above is not accumulated into the
 total @(':frames') counts.  As it turns out, this lack of stack frames is
 explained by the fact that the rewrite rule @('PERM-MEM') has a free variable
 in the first hypothesis.</p>

 @({
  ACL2 !>:pe perm-mem
           18  (DEFTHM PERM-MEM
                       (IMPLIES (AND (PERM X Y) (MEM A X))
                                (MEM A Y))
                       :RULE-CLASSES ((:REWRITE :MATCH-FREE :ONCE)))
  ACL2 !>
 })

 <p>The second hypothesis, however, does cause additional rewriting in order to
 rewrite it to true, resulting in 251 stack frames for runes.  We see that the
 conclusion does not lead to creation of any rune stack frames, which might
 seem to suggest that only 251 stack frames for runes were created on behalf of
 this rule application &mdash; yet, we see that 462 frames were actually
 created.  The difference is the 211 frames created for the rewrite rule
 itself.  Even if the total had been a bit more than 462, one need not be
 surprised, as there could be some work recorded during application of the
 rewrite rule, such as type-prescription reasoning, that is not done during
 rewriting of a hypothesis or the conclusion.</p>

 <p>Now suppose we have executed @('(accumulated-persistence :all)') and
 attempted some proofs, and now we are ready to see statistics.  The form
 @('(show-accumulated-persistence)') displays statistics exactly as described
 above, treating these extra xrunes just as though they are runes; similarly
 for the form @('(show-accumulated-persistence KEY)'), for any legal @('KEY').
 A second optional argument may however be supplied to
 @('show-accumulated-persistence').  The default for that second argument is
 @('t'), and a second argument of @(':raw') is treated the same as @('t');
 thus, these arguments provide the behavior just described, where data for
 xrunes are displayed just as for runes.  You may restrict output to runes,
 ignoring hypothesis and conclusion xrunes, by giving a second argument of
 @('nil').  (This gives the same behavior as if we had started with the command
 @('(accumulated-persistence t)') instead of the command
 @('(accumulated-persistence :all)').)  You may give a second argument
 of @(':merge'), in which case output will be sorted and displayed as though
 only runes were tracked (not the extra xrunes), but each data item for a
 non-rune xrune will be merged so that it is displayed in suitable order just
 below its corresponding rune, as in the @('PERM-MEM') example displayed
 above.  Finally, you may give a second argument of @(':list'), which is
 equivalent to the default second argument of @('t'), except that the results
 are printed as a list of entries @('(frames tries xrune)').</p>

 <p>We close by mentioning two aspects of enhanced statistics display for
 @(':CONC') xrunes that have potential to be confusing.  First consider the
 following example.</p>

 @({
       :frames   :tries    :ratio  rune
     --------------------------------
          14        4 (    3.50) (:REWRITE DEFAULT-+-2)
           0        0    [useful]
          14        4    [useless]
        .............................
          10        4 (    2.50) (:HYP 1 :REWRITE DEFAULT-+-2)
           0        0    [useful]
          10        4    [useless]
     --------------------------------
 })

 <p>It may be surprising that no data is displayed for the corresponding
 @(':CONC') xrune.  The explanation, however, is simple: the hypothesis never
 rewrote to true, so the conclusion was never rewritten.  This is consistent
 with the marking as ``useless'' of all @(':frames') and @(':tries') for the
 rune and the hypothesis xrune.  Note by the way, once again, that the
 hypothesis xrune does not contribute to any @(':frames') count.</p>

 <p>Another reason not to see data displayed for a @(':CONC') xrune is that if
 a rule has no hypotheses, then no such data is collected.  This decision was
 made because in the case of no hypotheses, we expect it to be very rare that
 information for the @(':CONC') xrune will add any useful insight.</p>

 <p>On a final note: @('(show-accumulated-persistence :runes)') may be used
 simply to see a list of all @(see rune)s (or xrunes) displayed
 alphabetically.</p>

 <p>Users are encouraged to think about other meters we could install in ACL2
 to help diagnose performance problems.</p>")

(defxdoc accumulated-persistence-subtleties
  :parents (accumulated-persistence)
  :short "Some subtle aspects of the counting done by @(tsee accumulated-persistence)"
  :long "<p>In this topic we cover the overcounting of ``useful'' @(see rune)
 application attempts, and we describe how ``useless'' rune application
 attempts can actually be critical for a proof's success.  We conclude with a
 few words about counting frames when there are nested (recursive) applications
 of a rune.</p>

 <p><i>Overcounting of ``useful'' rune application attempts.</i> Not every
 @(see rune) application may be necessary for a proof's success.  Consider for
 example:</p>

 @({
  (thm (equal (car (cons a (cdr (cons b x))))
              a))
 })

 <p>Then @('show-accumulated-persistence') will tell us that @(':')@(tsee
 rewrite) rules @('car-cons') and @('cdr-cons') each had one useful
 application.  However, the rule @('cdr-cons') is used to simplify @('(cdr
 (cons b x))') to @('x'), and this simplification is unnecessary for the proof.
 Indeed, the proof succeeds even when preceded by the event: @('(in-theory
 (disable cdr-cons))').  We thus see that a @(see rune) application labeled as
 ``useful'' may be simplifying a term that is not relevant to the proof.</p>

 <p>As of this writing, we consider every @(':')@(tsee forward-chaining) rule
 application to be ``useful'', for simplicity of the implementation.  Moreover,
 our counting of these rules is such that a single rule may be counted more
 than once.</p>

 <p><i>How ``useless'' attempts can be critical for a proof's success.</i> The
 command @('(accumulated-persistence :useless')) will list rules that did not
 contribute directly to the proof (see @(see accumulated-persistence), in
 particular the discussion of ``useless'' there).  However, a ``useless'' rule
 can on rare occasions be critical to the success of a proof.  In the following
 example, we have a ``bad'' rule that can take the proof in the wrong
 direction, but a ``useless'' rule does a rewrite that prevents the successful
 relieving of a hypothesis of the ``bad'' rule.  In summary:</p>

 @({
  ; Assume p0.  We want to prove p1.

  ; Key rule:
  p0 -> p1 = t

  ; Bad rule that could ruin the proof:
  p3 -> p1 = p2

  ; But unfortunately, we know p3:
  p0 -> p3

  ; Important ``useless'' rule, preventing ``bad rule'' above from firing:
  p3 = p4
 })

 <p>The following event captures the rules described above.</p>

 @({
  (encapsulate
   ((p0 (x) t)
    (p1 (x) t)
    (p2 (x) t)
    (p3 (x) t)
    (p4 (x) t))
   (local (defun p0 (x) x))
   (local (defun p1 (x) x))
   (local (defun p2 (x) x))
   (local (defun p3 (x) x))
   (local (defun p4 (x) x))

  ; Key rule:
   (defthm p0-implies-p1
     (implies (p0 x)
              (p1 x)))

  ; Bad rule that could ruin the proof:
   (defthm p3-implies-p1-is-p2
     (implies (p3 x)
              (equal (p1 x) (p2 x))))

  ; But unfortunately, we know p3:
   (defthm p0-implies-p3
     (implies (p0 x)
              (p3 x)))

  ; Important ``useless'' rule, preventing p3-implies-p1-is-p2 from firing:
   (defthm p3-is-p4
     (equal (p3 x) (p4 x))))
 })

 <p>Now we can see that @('p3-is-p4') is labeled as ``useless'', by evaluating
 these commands.</p>

 @({
  (accumulated-persistence t)
  (thm (implies (p0 x) (p1 x)))
  (show-accumulated-persistence)
 })

 <p>If instead we first evaluate @('(in-theory (disable p3-is-p4))') before the
 @('thm') above, then the proof fails, even though @('p3-is-p4') was labeled as
 ``useless''!</p>

 <p>Nevertheless, in general it is probably safe to disable rules reported as
 ``useless'' by @('(show-accumulated-persistence :useless)'), and doing so may
 speed up a proof considerably.</p>

 <p>Remark. The example above suggests a surprising fact: on rare occasions, a
 proof may fail when you give an @(':')@(tsee in-theory) hint consisting of
 exactly the @(see rune)s reported in a proof that succeeds.  For, imagine a
 rule R that is needed in part of the proof but is ``bad'' in a second part,
 and that some other, ``useless'' rule prevents the application of R in that
 second part.  The example above suggests that disabling this ``useless'' rule
 can allow the second application of R, thus preventing the proof.</p>

 <p>Finally we discuss accumulation into frame counts in the case of a nested
 (recursive) application of a rule: that is, the case that during the
 application of a rule, the rule is applied again &mdash; in particular, while
 relieving a hypothesis or rewriting the right-hand side from the original rule
 application.  Recall that the implementation of @(see accumulated-persistence)
 keeps a stack of @(see rune)s currently being applied; thus, we are
 considering here the case that a rune is pushed onto a stack on which it
 already resides.  In that case, we count tries as usual but we avoid
 accumulating until we reach the outermost (topmost) application of that rune.

 Consider the following example.</p>

 @({
  (defun mem (a x)
    (if (atom x)
        nil
      (or (equal a (car x)) (mem a (cdr x)))))
 })

 <p>Now suppose we consider the theorem @('(mem a (list 1 2 3 a))').  Each time
 the definition of @('mem') is applied, a new stack frame is pushed.  We avoid
 accumulating into the @(':frames') count for that stack frame unless it is the
 topmost stack frame for that definition.  Otherwise the final @(':frames')
 count would be the sum of the counts for those individual frames, which form a
 linear sequence whose sum would therefore be quadratic in the number of
 applications of the definition of @('mem').</p>")

(defxdoc acknowledgments
  :parents (about-acl2)
  :short "Some contributors to the well-being of ACL2"
  :long "<p>The development of ACL2 was initially made possible by funding from
 the U. S. Department of Defense, including ARPA and ONR.  We thank all the
 organizations that have contributed support, including the following (in
 alphabetical order).</p>

 <ul>

 <li>AMD, for providing significant time over several years for Matt Kaufmann
 to carry out ACL2 research, support, and development</li>

 <li>Computational Logic, Inc. and its president, Don Good, where the first
 eight years of ACL2 development occurred</li>

 <li>Centaur Technology</li>

 <li>DARPA</li>

 <li>Digital Equipment Corporation</li>

 <li>EDS, which provided some time for Matt Kaufmann's ACL2 work 1998-1999</li>

 <li>ForrestHunt and, more generally, Warren A. Hunt, Jr. (see below)</li>

 <li>IBM</li>

 <li>Kestrel Institute</li>

 <li>Kestrel Technology</li>

 <li>NSF</li>

 <li>ONR</li>

 <li>Rockwell Collins</li>

 <li>SRC</li>

 <li>Sun Microsystems</li>

 <li>University of Texas at Austin (in particular support to J Moore through the
 Admiral B. R.  Inman Chair of Computing Theory)</li>

 </ul>

 <p>We are especially grateful to Warren A. Hunt, Jr. for his unrivaled efforts
 in securing support for the entire ACL2 research group at both Computational
 Logic, Inc., and the University of Texas at Austin.  Without his efforts, we
 would have spent less time working on the system and fewer students would have
 been funded to apply it.</p>

 <p>ACL2 was started in August, 1989 by Boyer and Moore working together.  They
 co-authored the first versions of axioms.lisp and basis.lisp, with Boyer
 taking the lead in the formalization of ``@(see state)'' and the most
 primitive @(see io) functions.  Boyer also had a significant hand in the
 development of the early versions of the files interface-raw.lisp and
 translate.lisp.  For several years, Moore alone was responsible for developing
 the ACL2 system code, though he consulted often with both Boyer and Kaufmann.
 In August, 1993, Kaufmann became jointly responsible with Moore for developing
 the system.  Boyer has continued to provide valuable consulting on an informal
 basis.</p>

 <p>Bishop Brock was the heaviest early user of ACL2, and provided many
 suggestions for improvements.  In particular, the @(':cases') and
 @(':restrict') @(see hints) were his idea; he developed an early version of
 congruence-based reasoning for Nqthm; and he helped in the development of some
 early @(see books) about arithmetic.  In a demonstration of his courage and
 faith in us, he pushed for Computational Logic, Inc., to agree to the Motorola
 CAP contract &mdash; which required formalizing a commercial DSP in the
 untested ACL2 &mdash; and moved to Scottsdale, AZ, to do the work with the
 Motorola design team.  His demonstration of ACL2's utility was an inspiration,
 even to those of us designing ACL2.</p>

 <p>John Cowles also helped in the development of some early @(see books) about
 arithmetic, and also provided valuable feedback and bug reports.</p>

 <p>Other early users of ACL2 at Computational Logic, Inc. helped influence its
 development.  In particular, Warren Hunt helped with the port to Macintosh
 Common Lisp, and Art Flatau and Mike Smith provided useful general
 feedback.</p>

 <p>Mike Smith helped develop the Emacs portion of the implementation of proof
 trees.</p>

 <p>ACL2 depends on the availability of robust Common Lisp implementations, so
 we are grateful to the developers of those implementations.  Early in ACL2's
 history, Bill Schelter made some enhancements to AKCL (now GCL) that helped to
 enhance ACL2 performance in that Common Lisp implementation, and more
 generally, responded helpfully to our bug reports.  Camm Maguire has since
 provided wonderful GCL support, and has created a Debian package for ACL2
 built on GCL.  Gary Byers and R. Matthew Emerson have continually improved
 Clozure Common Lisp (CCL), often based on feedback from the ACL2 community.
 We are also grateful to developers of other Common Lisp implementations.</p>

 <p>Kent Pitman helped in our interaction with the ANSI Common Lisp
 standardization committee, X3J13.</p>

 <p>John Cowles helped with the port to Windows (98) by answering questions and
 running tests.</p>

 <p>Ruben Gamboa created a modification of ACL2 to allow reasoning about the
 real numbers using non-standard analysis.  His work has been incorporated into
 the ACL2 distribution; see @(see real).</p>

 <p>Rob Sumners has made numerous useful suggestions.  In particular, he has
 designed and implemented improvements for @(see stobj)s and been key in our
 development of locally-bound stobjs; see @(see note-2-6).</p>

 <p>Robert Krug has designed and implemented many changes in the vicinity of
 the linear arithmetic package and its connection to type-set and rewrite.  He
 was also instrumental in the development of @(see extended-metafunctions).</p>

 <p>Pete Manolios has made numerous useful suggestions.  In particular, Pete
 helped us to organize the first workshop and was a wonderful equal partner
 with the two of us (Kaufmann and Moore) in producing the books that arose from
 that workshop.  Pete and his student, Daron Vroon, provided the current
 implementation of @(see ordinals).</p>

 <p>Jared Davis, Sol Swords, and David Rager have our gratitude for starting
 the <a href='https://github.com/acl2/acl2/'>ACL2+Books repository</a>.</p>

 <p>We thank David L. Rager for contributing an initial version of the support
 for @(see parallelism) in an experimental extension of ACL2.</p>

 <p>Bob Boyer and Warren A. Hunt, Jr. developed a canonical representation for
 ACL2 data objects, applicative hash tables, and a function memoization
 mechanism to facilitate reuse of previously computed results.  Subsequently,
 Jared Davis and Sol Swords made further contributions.  We thank them all for
 this work, most of which has been incorporated into ACL2; see @(see
 hons-and-memoization).</p>

<p>Other contributions to the ACL2 system continue to be made by members of the
 ACL2 community.  In particular, following the first Developers Workshop in
 May, 2017 through the time of this writing in November, 2019, such
 contributors include Alessandro Coglio, Keshav Kini, Mihir Mehta, Pete
 Manolios, and especially Sol Swords, while Eric Smith and many others have
 suggested changes that we have implemented, often by providing helpful
 examples.  The @(see release-notes) detail such contributions as well as many
 suggestions from the community for improvements that we ultimately
 implemented.</p>

 <p>We also thank the contributors to the ACL2 workshops for some suggested
 improvements and for the extensive collection of publicly distributed
 benchmark problems.  And we thank participants at the ACL2 seminar at the
 University of Texas for useful feedback.  More generally, we thank the ACL2
 community for feedback, contributed @(see books) (see @(see community-books)),
 and their interest in the ACL2 project.</p>

 <p><i>Regarding the documentation:</i></p>

 <blockquote><p>Bill Young wrote significant portions of the original
 @('acl2-tutorial') section of the ACL2 documentation, including what is now
 called @(see alternative-introduction).  This was an especially important task
 in the early years when there was no guide for how to use ACL2 and we are very
 grateful.  He, Bishop Brock, Rich Cohen, and Noah Friedman read over
 considerable amounts of the documentation, and made many useful comments.
 Others, particularly Bill Bevier and John Cowles, have also made useful
 comments on the @(see documentation).</p>

 <p>Art Flatau helped develop the ACL2 markup language in which ACL2 @(see
 documentation) was originally developed, along with translators from that
 language to Texinfo and HTML.  Michael ``Bogo'' Bogomolny created a search
 engine, beginning with Version 2.6, and for that purpose modified the HTML
 translator to create one file per topic (a good idea in any case).</p>

 <p>Laura Lawless provided many hours of help in marking up appropriate parts
 of the @(see documentation) in typewriter font.</p>

 <p>Noah Friedman developed an Emacs tool that helped us insert ``invisible
 links'' into the @(see documentation), which improve the usability of that
 documentation under HTML readers such as Mosaic.</p>

 <p>Richard Stallman contributed a texinfo patch, to be found in the file
 @('doc/texinfo.tex').</p>

 <p>Jared Davis created the @(see xdoc) system that is now the basis not only
 for the ACL2 system @(see documentation) (file
 @('books/system/doc/acl2-doc.lisp')), but also for the @(see community-books)
 documentation.</p>

 </blockquote>

 <p>We thank Blake Grugett for designing the current version of the ACL2 logo
 (which for example appears on the ACL2 home page), based on an original design
 created in the 1990s by Computational Logic, Inc.</p>

 ")

(defxdoc acl2
  :parents (top)
  :short "ACL2 documentation (system only, not including the community books)"
  :long "<p>This is the ACL2 documentation.  For the ACL2+Books Manual, which
  that includes both the ACL2 documentation and the ACL2 @(see
  community-books), see the @(`(:raw (combined-manual-ref))`).</p>")

(defxdoc |ACL2 Characters|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 Characters"
  :long "<p>ACL2 accepts 256 distinct characters, which are the characters
 obtained by applying the function @(tsee code-char) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 to each integer from 0 to 255.  Among these, Common Lisp designates certain
 ones as @('*standard-characters*'), namely those of the form @('(code-char
 n)') where n is from 33 to 126, together with @('#\\Newline') and
 @('#\\Space').  The actual standard characters may be viewed by evaluating the
 constant expression @('*standard-chars*').</p>

 <p>The standard character constants are written by writing a hash mark
 followed by a backslash (#\\) followed by the character.</p>

 <p>The function @(tsee characterp) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 recognizes characters.  For more details, See @(see characters) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.</p>")

(defxdoc |ACL2 Conses or Ordered Pairs|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 Conses or Ordered Pairs"
  :long "<p>The function @(tsee cons) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 creates an ordered pair.  @(tsee Car) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see> and
 @(tsee cdr) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> return the first and second components,
 respectively, of an ordered pair.  The function @(tsee consp) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 recognizes ordered pairs.</p>

 <p>Ordered pairs are used to represent lists and trees.  See any Common Lisp
 documentation for a discussion of how list constants are written and for the
 many list processing functions available.  Also, see @(see programming) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 where we list all the ACL2 primitive functions.</p>

 <p>Here are some examples of list constants to suggest their syntax.</p>

 @({
  '(a . b)                ; a pair whose car is 'a and cdr is 'b
  '(a . nil)              ; a pair whose car is 'a and cdr is nil
  '(a)                    ; another way to write the same thing
  '(a b)                  ; a pair whose car is 'a and cdr is '(b)
  '(a b c)                ; a pair whose car is 'a and cdr is '(b c)
                          ;  i.e., a list of three symbols, a, b, and c.
  '((a . 1) (b . 2))      ; a list of two pairs
 })

 <p>It is useful to distinguish ``proper'' conses from ``improper'' ones, the
 former being those cons trees whose right-most branch terminates with
 @('nil').  A ``true list'' (see @(see true-listp) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>) is
 either @('nil') or a proper cons.  @('(A b c . 7)') is an improper cons and
 hence not a true list.</p>")

(defxdoc |ACL2 Strings|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 Strings"
  :long "<p>Strings of ACL2 <see topic='@(url
 |ACL2 Characters|)'>characters</see> are written as sequences of characters
 delimited by ``double quotation marks''
 (\").  To put a double quotation mark in a string (or, any other character
 such as backslash or newline that seems to cause problems), escape it by
 preceding it with a backslash (\\).</p>

 <p>The function @(tsee stringp) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 recognizes strings and @(tsee char) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 will fetch the nth character of a string.  There are many other primitives for
 handling strings, such as @(tsee string<) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see> for
 comparing two strings lexicographically.  We suggest you See @(see
 programming) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> where we list all of the primitive ACL2 functions.
 Alternatively, see any Common Lisp language documentation.</p>")

(defxdoc |ACL2 Symbols|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 Symbols"
  :long "<p>Common Lisp's symbols are a data type representing words.  They are
 frequently regarded as atomic objects in the sense that they are not
 frequently broken down into their constituents.  Often the only important
 properties of symbols is that they are not numbers, characters, strings, or
 lists and that two symbols are not equal if they look different (!).  Examples
 of symbols include @('PLUS') and @('SMITH::ABC').  All function and variable
 names in ACL2 are symbols.  When symbols are used as constants they must be
 quoted, as in @(''PLUS').</p>

 <p>The symbol @('T') is commonly used as the Boolean ``true.''  The symbol
 @('NIL') is commonly used both as the Boolean ``false'' and as the ``empty
 list.''  Despite sometimes being called the ``empty list'' @('NIL') is a
 <b>symbol</b> not an ``empty cons.''  Unlike other symbols, @('T') and
 @('NIL') may be used as constants without quoting them.</p>

 <p>Usually, symbols are written as sequences of alphanumeric characters other
 than those denoting numbers.  Thus, @('A12'), @('+1A') and @('1+') are symbols
 but @('+12') is a number.  Roughly speaking, when symbols are read lower case
 characters are converted to upper case, so we frequently do not distinguish
 @('ABC') from @('Abc') or @('abc').  Click <see topic='@(url
 |Conversion|)'>here</see> for information about case conversion when symbols
 are read.  However, any character can be used in a symbol, but some characters
 must be ``escaped'' to allow the Lisp reader to parse the sequence as a
 symbol.  For example, @('|Abc|') is a symbol whose first character is
 capitalized and whose remaining characters are in lower case.  @('|An odd
 duck|') is a symbol containing two #\\Space characters.  See any Common Lisp
 documentation for the syntactic rules for symbols.</p>

 <p>Technically, a symbol is a special kind of pair consisting of a package
 name (which is a string) and a symbol name (which is also a string).  (See
 @(see symbol-package-name) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see> and
 see @(see symbol-name) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.)  The symbol SMITH::ABC is said to be in package
 \"SMITH\" and to have the symbol name \"ABC\".  The symbol @('ABC') in package
 \"SMITH\" is generally not equal to the symbol @('ABC') in package \"JONES\".
 However, it is possible to ``import'' symbols from one package into another
 one, but in ACL2 this can only be done when the package is created.  (See
 @(see defpkg) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.)  If the @(tsee current-package) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see> is
 \"SMITH\" then @('SMITH::ABC') may be more briefly written as just @('ABC').
 @(tsee Intern) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> ``creates'' a symbol of a given name in a given
 package.</p>")

(defxdoc |ACL2 System Architecture|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 System Architecture"
  :long "<p><see topic='@(url
 |Rewrite Rules are Generated from DEFTHM Events|)'><img
 src='res/tours/walking.gif'></img></see></p>

 <p><img src='res/tours/acl2-system-architecture.gif'></img></p>

 <p>The user interacts with the theorem prover by giving it definitions,
 theorems and advice.  Most often the advice is about how to store each proved
 theorem as a rule.  Sometimes the advice is about how to prove a specific
 theorem.</p>

 <p>The database consists of all the rules ACL2 ``knows.''  It is possible to
 include in the database all of the rules in some certified file of other
 events.  Such certified files are called @(see books) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.</p>

 <p>Interesting proofs are usually built on top of many books, some of which
 are written especially for that problem domain and others of which are about
 oft-used domains, like arithmetic or list processing.  ACL2's distribution
 includes many books written by users.  See the ``books'' link under the
 <b>Lemma Libraries and Utilities</b> <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 link of the ACL2 home page.</p>

 <p><see topic='@(url |Rewrite Rules are Generated from DEFTHM Events|)'><img
 src='res/tours/walking.gif'></img></see></p>")

(defxdoc |ACL2 as an Interactive Theorem Prover|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 as an Interactive Theorem Prover"
  :long "<p>The ACL2 theorem prover finds proofs in the ACL2 logic.  It can be
 automatic.  But most often the user must help it.</p>

 <p><img src='res/tours/interactive-theorem-prover.gif'></img></p>

 <p>The user usually guides ACL2 by suggesting that it first prove key
 <b>lemmas</b>.  Lemmas are just theorems used in the proofs of other
 theorems.</p>")

(defxdoc |ACL2 as an Interactive Theorem Prover (cont)|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 as an Interactive Theorem Prover (cont)"
  :long "<p><see topic='@(url |ACL2 System Architecture|)'><img
 src='res/tours/walking.gif'></img></see></p>

 <p>When ACL2 proves a lemma, it is converted into one or more <b>rules</b> and
 stored in a <b>database</b>.  The theorem prover is <b>rule-driven</b>.  By
 proving lemmas you can configure ACL2 to behave in certain ways when it is
 trying to prove formulas in a certain problem domain.  The expert user can
 make ACL2 do amazingly ``smart'' looking things.</p>

 <p>But it would be wrong to think that ACL2 <i>knows</i> the mathematical
 content of a formula just because it has proved it.  What ACL2 knows &mdash;
 all ACL2 knows &mdash; is what is encoded in its rules.  There are many types
 of rules (see @(see rule-classes) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>).</p>

 <p>Many formulas can be effectively coded as rules.  But by the same token, it
 is possible to encode a formula as a rule that is so ineffective it cannot
 even prove itself!</p>

 <p>The way a formula is stored as a rule is entirely up to the user.  That is,
 <b>you</b> determine how ACL2 should use each formula that it proves.</p>

 <p>The most common kind of rule is the <b>rewrite rule</b>.  It is so common
 that if you don't tell ACL2 how to store a formula, it stores it as a rewrite
 rule.</p>

 <p><see topic='@(url |ACL2 System Architecture|)'><img
 src='res/tours/walking.gif'></img></see></p>")

(defxdoc |ACL2 is an Untyped Language|
  :parents (|Pages Written Especially for the Tours|)
  :short "ACL2 is an Untyped Language"
  :long "<p>The example</p>

 <code>
 ACL2 !&gt;<b>(app '(a b c) 27)</b>
 (A B C . 27)
 </code>

 <p>illustrates the fact that ACL2's logic is untyped (click <see topic='@(url
 |About Types|)'>here</see> for a brief discussion of the typed versus untyped
 nature of the logic).</p>

 <p>The definition of @('app') makes no restriction of the arguments to lists.
 The definition says that if the first argument satisfies @(tsee endp) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 then return the second argument.  In this example, when @('app') has recursed
 three times down the @('cdr') of its first argument, @(''(a b c)'), it reaches
 the final @('nil'), which satisfies @('endp'), and so 27 is returned.  It is
 naturally consed into the emerging list as the function returns from
 successive recursive calls (since @('cons') does not require its arguments to
 be lists, either).  The result is an ``improper'' list, @('(a b c . 27)').</p>

 <p>You can think of @('(app x y)') as building a binary tree by replacing the
 right-most tip of the tree @('x') with the tree @('y').</p>")

(defxdoc acl2-as-standalone-program
  :parents (acl2-tutorial)
  :short "Calling ACL2 from another program"
  :long "<p>ACL2 is intended for interactive use.  It is generally unrealistic
 to expect it to prove theorems fully automatically; see @(see the-method), and
 see @(see introduction-to-the-theorem-prover) for a more detailed
 tutorial.</p>

 <p>Nevertheless, here we describe an approach for how to call the ACL2 theorem
 prover noninteractively.  These steps can of course be modified according to
 your needs.  Here, we illustrate how to call ACL2 from another Lisp program
 (or an arbitrary program) to attempt to prove an arithmetic theorem.</p>

 <p>See also @(see interfacing-tools).  In particular, if you want to make a
 command-line tool to ACL2 with options, you may be interested in @(see
 oslib::argv), @(see getopt::getopt), and especially @(see getopt-demo::demo2).
 Alternately, if you want to develop a server application on top of ACL2, you
 might consider @(see bridge::bridge).</p>

 <h3>Step 1</h3>

 <p>Build a suitable ACL2 image by starting ACL2 and then executing the
 following forms.  In particular, these define a macro, @('try-thm'), that
 causes ACL2 to exit with an exit status indicating success or failure of a
 proof attempt.</p>

 @({
  (include-book \"arithmetic-5/top\" :dir :system)
  (defmacro try-thm (&rest args)
    `(mv-let (erp val state)
             (with-prover-time-limit 3 (thm ,@args))
             (declare (ignore val))
             (prog2$ (if erp (exit 1) (exit 0)) state))))
  (reset-prehistory) ; optional
  :q
  (save-exec \"arith-acl2\" \"Included arithmetic-4/top\")
 })

 <p>If you prefer, above you can replace 3 by some other number of seconds as a
 time limit for the prover.  Also, you can replace</p>

 @({
  (with-prover-time-limit 3 (thm ,@args))
 })

 <p>by</p>

 @({
  (with-output :off :all (with-prover-time-limit 3 (thm ,@args)))
 })

 <p>if you want to turn off output.  It may be best to leave the output on,
 instead eliminating it in the calling program (see Step 3 below).</p>

 <h3>Step 2</h3>

 <p>Try a little test.  In that same directory try this:</p>

 @({
  echo '(try-thm (equal x x))' | ./arith-acl2
  echo $?
 })

 <p>The exit status should be 0, indicating success.  Now try this:</p>

 @({
  echo '(try-thm (not (equal x x)))' | ./arith-acl2
  echo $?
 })

 <p>The exit status should be 1, indicating failure.</p>

 <h3>Step 3</h3>

 <p>Create a shell script that automates Step 2, for example:</p>

 @({
  #!/bin/sh
  (echo \"(try-thm $1)\" | ./arith-acl2) >& /dev/null
  exit $?
 })

 <h3>Step 4</h3>

 <p>Try your script from a Lisp program, if you like.  Here is how you can do
 it in SBCL, for example.  (Different Lisps have different ways to do this, as
 summarized in function @('system-call') in ACL2 source file
 @('acl2-init.lisp').)</p>

 @({
  (defun provable? (x)
    (let ((status
           (process-exit-code
            (sb-ext:run-program \"./try-thm.sh\" (list (format nil \"~s\" x))
                                :output t :search t))))
      (eql status 0)))
 })

 <p>Then here is a log:</p>

 @({
    * (provable? '(equal x y))

    NIL
    * (provable? '(equal x x))

    T
    *
 })

 <p>Certainly refinements are possible &mdash; for example the above doesn't
 distinguish between unprovable and ill-formed input.  But it's a start.</p>

 ")

(defxdoc acl2-built-ins
  :parents (programming)
  :short "''Catch-all'' topic for built-in ACL2 functions"
  :long "<p>This @(see documentation) topic is a parent topic under which we
 include documentation for built-in functions, macros, and special forms that
 are typically used in @(see programming).  For others, including those
 typically used as top-level commands or those that create @(see events)
 (@(tsee defun), @(tsee defmacro), @(tsee defthm), and so on), documentation
 may be found as a subtopic of some other parent topic.  We do not document
 some of the more obscure functions provided by ACL2 that do not correspond to
 functions of Common Lisp.</p>

 <p>If you are already familiar with Common Lisp (or even some other Lisp
 variant), then you may find it helpful to start with the topic, @(see
 introduction-to-programming-in-acl2-for-those-who-know-lisp).</p>

 <p>See any documentation for Common Lisp for more details on many of these
 functions.</p>")

(defxdoc acl2-count
  :parents (basics acl2-built-ins)
  :short "A commonly used measure for justifying recursion"
  :long "<p>@('(Acl2-count x)') returns a nonnegative integer that indicates
 the ``size'' of its argument @('x').</p>

 <p>All @(see characters) and symbols have @('acl2-count 0').  The
 @('acl2-count') of a string is the number of @(see characters) in it, i.e.,
 its length.  The @('acl2-count') of a @(tsee cons) is one greater than the sum
 of the @('acl2-count')s of the @(tsee car) and @(tsee cdr).  The
 @('acl2-count') of an integer is its absolute value.  The @('acl2-count') of a
 rational is the sum of the @('acl2-count')s of the numerator and denominator.
 The @('acl2-count') of a complex rational is one greater than the sum of the
 @('acl2-count')s of the real and imaginary parts.</p>

 @(def acl2-count)")

(defxdoc acl2-customization
  :parents (miscellaneous)
  :short "File of initial commands for ACL2 to run at @(see startup)"
  :long "<p>ACL2 provides a mechanism to load automatically a so-called ``ACL2
 customization file,'' via @(tsee ld), the first time @(tsee lp) is called in
 an ACL2 session.  ACL2 looks for this file as follows.</p>

 <ol>

 <li>If the host Lisp reads a non-empty value for the system's
 environment variable @('ACL2_CUSTOMIZATION'), then that string value is used
 for the customization file name.  In this case, if the file does not exist or
 if the string is \"NONE\" then there is no customization file.  Notes:

     <ul>

     <li>If the customization file name is a relative pathname (see @(see
     pathname)), then the pathname is considered relative to the connected book
     directory (see @(see cbd)).</li>

     <li>If this variable is not already defined, then its value is set to
     @('NONE') when books are certified using @(see BUILD::cert.pl) or other,
     legacy Make-based certification tools.</li>

     </ul></li>

 <li>Otherwise (empty environment variable value), file
 @('\"acl2-customization.lsp\"') or @('\"acl2-customization.lisp\"') on the
 connected book directory (see @(see cbd)), generally the current directory, is
 the customization file (in that order) if either exists.</li>

 <li>Otherwise file @('\"acl2-customization.lsp\"') or
 @('\"acl2-customization.lisp\"') on your home directory is the customization
 file (in that order), if either exists (except, this case is skipped on
 Windows operating systems.</li>

 </ol>

 <p>Except for the fact that this @(tsee ld) command is not typed explicitly by
 you, it is a standard @(tsee ld) command except that any settings of @(tsee
 ld) specials are remembered once this call of @(tsee ld) has completed other
 than @(tsee ld-error-action), which will always be @(':command-conventions')
 after that call of @('ld') completes.  For example, suppose that you start
 your customization file with @('(set-ld-skip-proofsp t state)'), so that
 proofs are skipped as it is loaded with @(tsee ld).  Then the @(tsee ld)
 special @(tsee ld-skip-proofsp) will remain @('t') after the @(tsee ld) has
 completed, causing proofs to be skipped in your ACL2 session, unless your
 customization file sets this variable back to @('nil'), say with
 @('(set-ld-skip-proofsp nil state)').</p>

 <p>If the customization file exists, it is loaded with @(tsee ld) using the
 usual default values for the @(tsee ld) specials (see @(see ld)) except that
 @(':ld-error-action') is @(':error').  If an error is encountered, then no
 subsequent forms in the file will be evaluated and ACL2 will quit
 immediately.</p>

 <p>To create a customization file it is recommended that you first give it a
 name other than @('\"acl2-customization.lsp\"') or
 @('\"acl2-customization.lisp\"') so that ACL2 does not try to include it
 prematurely when you next enter @(tsee lp).  Then, while in the uncustomized
 @(tsee lp), explicitly invoke @(tsee ld) on your evolving (but renamed)
 customization file until all forms are successfully evaluated.  The same
 procedure is recommended if for some reason ACL2 cannot successfully evaluate
 all forms in your customization file: temporarily rename your customization
 file so that ACL2 does not try to @(tsee ld) it automatically and then debug
 the new file by explicit calls to @(tsee ld).</p>

 <p>WARNING!  If you certify a book after the (automatic) loading of a
 customization file, the forms in that file will be part of the @(see
 portcullis) of the @(see books) you certify!  That is, the forms in your
 customization file at certification time will be loaded whenever anybody uses
 the @(see books) you are certifying.  Since customization files generally
 contain idiosyncratic @(see command)s, you may not want yours to be part of
 the @(see books) you create for others.  Thus, if you have a customization
 file then you may want to invoke @(':')@(tsee ubt)@(' 1') before certifying
 any @(see books); alternatively, see @(see certify-book!) for automatic
 invocation of @(tsee ubt).</p>

 <p>On the other hand, if you wish to prevent undoing commands from the
 customization file, see @(see reset-prehistory).</p>

 <p>Note that except on Windows-based systems, if there is a file
 @('acl2-init.lsp') in your home directory, then it will be loaded into raw
 Lisp when ACL2 is invoked.</p>

 <h3>Silent loading of ACL2 customization files</h3>

 <p>When the environment variable @('ACL2_CUSTOMIZATION_QUIET') is set and not
 @('\"\"'), there will generally be no output from ACL2 customization.  A
 special value of @('\"all\"') for this variable will cause continued minimal
 output after startup, as explained in the following remark.</p>

 <p>Technical Remark.  For quiet loading of acl2-customization files, @(tsee
 ld) specials are bound to the following values.</p>

 @({
 ld-verbose = nil
 ld-pre-eval-print = :never
 ld-post-eval-print = nil
 ld-prompt = nil
 })

 <p>These @('ld') specials are returned to their normal values after loading an
 ACL2 customization file, with one exception: if @('ACL2_CUSTOMIZATION_QUIET')
 has value @('\"ALL\"') (or @('\"all\"'); the case is irrelevant), then those
 values are retained in the ACL2 loop even after customization completes.</p>")

(defxdoc acl2-defaults-table
  :parents (table)
  :short "A @(see table) specifying certain defaults, e.g., the default @(see defun-mode)"
  :long "@({
  Example Forms:
  (table acl2-defaults-table :defun-mode) ; current default defun-mode
  (table acl2-defaults-table :defun-mode :program)
             ; set default defun-mode to :program
 })

 <p>See @(see table) for a discussion of tables in general.  The legal keys for
 this @(see table) are shown below.  They may be accessed and changed via the
 general mechanisms provided by @(see table)s.  However, there are often more
 convenient ways to access and/or change the defaults.  (See also the note
 below.)</p>

 @({
  :user
 })

 <p>The @(':user') key is for ACL2 users; the system does not consult this key
 or set it (other than as part of general @('acl2-defaults-table ')
 maintenance).  Its value is required to be an association list (@(tsee
 alistp)), so that different users can read and write their ``own'' keys.  For
 example, suppose user Joe uses key @(':j') while user Mary uses key @(':m');
 then we could see a value for @(':user') of @('((:j . 3) (:m . 4))') after Joe
 sets his value to 3 and Mary sets her value to 4.</p>

 @({
  :defun-mode
 })

 <p>the default @(see defun-mode), which must be @(':')@(tsee program) or
 @(':')@(tsee logic).  See @(see defun-mode) for a general discussion of @(see
 defun-mode)s.  The @(':')@(tsee defun-mode) key may be conveniently set by
 keyword commands naming the new @(see defun-mode), @(':')@(tsee program) and
 @(':')@(tsee logic).  See @(see program) and see @(see logic).</p>

 @({
  :enforce-redundancy
 })

 <p>if @('t'), cause ACL2 to insist that most events are redundant (see @(see
 redundant-events)); if @(':warn'), cause a warning instead of an error for
 such non-redundant events; else, @('nil').  See @(see
 set-enforce-redundancy).</p>

 @({
  :verify-guards-eagerness
 })

 <p>an integer between 0 and 2 indicating how eager the system is to verify the
 @(see guard)s of a @(see defun) event.  See @(see
 set-verify-guards-eagerness).</p>

 @({
  :compile-fns
 })

 <p>When this key's value is @('t'), functions are compiled when they are
 @(tsee defun)'d; otherwise, the value is @('nil').  (Except, this key's value
 is ignored when explicit compilation is suppressed; see @(see compilation).)
 To set the flag, see @(see set-compile-fns).</p>

 @({
  :measure-function
 })

 <p>the default measure function used by @(tsee defun) when no @(':measure') is
 supplied in @(tsee xargs).  The default measure function must be a function
 symbol of one argument. Let @('mfn') be the default measure function and
 suppose no @(':measure') is supplied with some recursive function definition.
 Then @(tsee defun) finds the first formal, @('var'), that is tested along
 every branch and changed in each recursive call.  The system then ``guesses''
 that @('(mfn var)') is the @(':measure') for that @(tsee defun).</p>

 @({
  :well-founded-relation
 })

 <p>the default well-founded relation used by @(tsee defun) when no
 @(':well-founded-relation') is supplied in @(tsee xargs).  The default
 well-founded relation must be a function symbol, @('rel'), of two arguments
 about which a @(':well-founded-relation') rule has been proved.  See @(see
 well-founded-relation-rule).</p>

 @({
  :bogus-defun-hints-ok
 })

 <p>When this key's value is @('t'), ACL2 allows @(':hints') and also
 @(':measure') for nonrecursive function definitions.  Otherwise, the value is
 @('nil') (the default) or @(':warn') (which makes the check but merely warns
 when the check fails).  See @(see set-bogus-defun-hints-ok) and @(see
 set-bogus-measure-ok).</p>

 @({
  :bogus-mutual-recursion-ok
 })

 <p>When this key's value is @('t'), ACL2 skips the check that every function
 in a @(tsee mutual-recursion) (or @(tsee defuns)) ``clique'' calls at least
 one other function in that ``clique.''  Otherwise, the value is @('nil') (the
 default) or @(':warn') (which makes the check but merely warns when the check
 fails).  See @(see set-bogus-mutual-recursion-ok).</p>

 @({
  :irrelevant-formals-ok
 })

 <p>When this key's value is @('t'), the check for irrelevant formals is
 bypassed; otherwise, the value is the keyword @('nil') (the default) or
 @(':warn') (which makes the check but merely warns when the check fails).  See
 @(see irrelevant-formals) and see @(see set-irrelevant-formals-ok).</p>

 @({
  :ignore-ok
 })

 <p>When this key's value is @('t'), the check for ignored variables is
 bypassed; otherwise, the value is the keyword @('nil') (the default) or
 @(':warn') (which makes the check but merely warns when the check fails).  See
 @(see set-ignore-ok).</p>

 @({
  :bdd-constructors
 })

 <p>This key's value is a list of function symbols used to define the notion of
 ``BDD normal form.''  See @(see bdd-algorithm) and see @(see hints).</p>

 @({
  :ttag
 })

 <p>This key's value, when non-@('nil'), allows certain operations that extend
 the trusted code base beyond what is provided by ACL2.  See @(see defttag).
 See @(see defttag).</p>

 @({
  :state-ok
 })

 <p>This key's value is either @('t') or @('nil') and indicates whether the
 user is aware of the syntactic restrictions on the variable symbol @('STATE').
 See @(see set-state-ok).</p>

 @({
  :backchain-limit
 })

 <p>This key's value is a list of two ``numbers.''  Either ``number'' may
 optionally be @('nil'), which is treated like positive infinity.  The numbers
 control backchaining through hypotheses during type-set reasoning and
 rewriting.  See @(see backchain-limit).</p>

 @({
  :default-backchain-limit
 })

 <p>This key's value is a list of two ``numbers.''  Either ``number'' may
 optionally be @('nil'), which is treated like positive infinity.  The numbers
 are used respectively to set the backchain limit of a rule if one has not been
 specified. See @(see backchain-limit).</p>

 @({
  :step-limit
 })

 <p>This key's value is either @('nil') or a natural number not exceeding the
 value of @('*default-step-limit*').  If the value is @('nil') or the value of
 @('*default-step-limit*'), there is no limit on the number of ``steps'' that
 ACL2 counts during a proof: currently, the number of top-level rewriting
 calls.  Otherwise, the value is the maximum number of such calls allowed
 during evaluation of any event.  See @(see set-prover-step-limit).</p>

 @({
  :rewrite-stack-limit
 })

 <p>This key's value is a nonnegative integer less than @('(expt 2 28)').  It
 is used to limit the depth of calls of ACL2 rewriter functions.  See @(see
 rewrite-stack-limit).</p>

 @({
  :let*-abstractionp
 })

 <p>This key affects how the system displays subgoals.  The value is either
 @('t') or @('nil').  When t, let* expressions are introduced before printing
 to eliminate common subexpressions.  The actual goal being worked on is
 unchanged.</p>

 @({
  :case-split-limitations
 })

 <p>This key's value is a list of two ``numbers.''  Either ``number'' may
 optionally be @('nil'), which is treated like positive infinity.  The numbers
 control how the system handles case splits in the simplifier.  See @(see
 set-case-split-limitations).</p>

 @({
  :include-book-dir-alist
 })

 <p>This key's value is used by @(tsee include-book)'s @(':DIR') argument to
 associate a directory with a keyword.  An exception is the keyword
 @(':SYSTEM') for the @('books/') directory; see @(see include-book), in
 particular the section on ``Books Directory.''  Also see @(tsee
 add-include-book-dir) and @(tsee add-include-book-dir!).</p>

 @({
  :match-free-default
 })

 <p>This key's value is either @(':all'), @(':once'), or @('nil').  See @(see
 set-match-free-default).</p>

 @({
  :match-free-override
 })

 <p>This key's value is a list of runes.  See @(see
 add-match-free-override).</p>

 @({
  :match-free-override-nume
 })

 <p>This key's value is an integer used in the implementation of @(see
 add-match-free-override), so that only existing runes are affected by that
 event.</p>

 @({
  :non-linearp
 })

 <p>This key's value is either @('t') or @('nil') and indicates whether the
 user wishes ACL2 to extend the linear arithmetic decision procedure to include
 non-linear reasoning.  See @(see non-linear-arithmetic).</p>

 @({
  :tau-auto-modep
 })

 <p>This key's value is either @('t') or @('nil') and indicates whether the
 user wishes ACL2 to look for opportunities to create @(':')@(tsee tau-system)
 rules from all suitable @('defun')s and from all suitable @('defthm')s (with
 non-@('nil') @(':')@(tsee rule-classes)).  See @(see set-tau-auto-mode).</p>

 @({
  :ruler-extenders
 })

 <p>This key's value may be a list of symbols, indicating those function
 symbols that are not to block the collection of rulers; see @(see defun).
 Otherwise the value is @(':all') to indicate all function symbols, i.e., so
 that no function symbol blocks the collection of rulers.  If a list is
 specified (rather than @(':all')), then it may contain the keyword
 @(':lambdas'), which has the special role of specifying all @('lambda')
 applications.  No other keyword is permitted in the list.  See @(see
 rulers).</p>

 @({
  :memoize-ideal-okp
 })

 <p>This key's value must be either @('t'), @('nil'), or @(':warn').  If the
 value is @('nil') or not present, then it is illegal by default to @(see
 memoize) a @(':')@(tsee logic) mode function that has not been @(see
 guard)-verified (see @(see verify-guards)), sometimes called an ``ideal-mode''
 function.  This illegality is the default because such calls of such functions
 in the ACL2 loop are generally evaluated in the logic (using executable
 counterpart definitions; see @(see evaluation)), rather than directly by
 executing calls of the corresponding (memoized) raw Lisp function.  However,
 such a raw Lisp call can be made when the function is called by a @(':')@(tsee
 program) mode function, so we allow you to override the default behavior by
 associating the value @('t') or @(':warn') with the key
 @(':memoize-ideal-okp'), where with @(':warn') you get a suitable warning.
 Note that you can also allow memoization of ideal-mode functions by supplying
 argument @(':ideal-okp') to your memoization event (see @(see memoize)), in
 which case the value of @(':memoize-ideal-okp') in the
 @('acl2-defaults-table') is irrelevant.</p>

 @({
  :check-invariant-risk
 })

 <p>For an explanation of this key, see @(see set-check-invariant-risk).</p>

 @({
  :register-invariant-risk
 })

 <p>For an explanation of this key, see @(see set-register-invariant-risk).</p>

 @({
  :in-theory-redundant-okp
 })

 <p>When this key's value is @('t'), an @(tsee in-theory) event may be
 redundant.  See @(tsee set-in-theory-redundant-okp).</p>

 <p>Note: Unlike all other @(see table)s, @('acl2-defaults-table') can affect
 the soundness of the system.  The @(see table) mechanism therefore enforces on
 it a restriction not imposed on other @(see table)s: when @(tsee table) is
 used to update the @('acl2-defaults-table'), the key and value must be
 variable-free forms.  Thus, while</p>

 @({
  (table acl2-defaults-table :defun-mode :program),

  (table acl2-defaults-table :defun-mode ':program), and

  (table acl2-defaults-table :defun-mode (compute-mode *my-data*))
 })

 <p>are all examples of legal @(see events) (assuming @('compute-mode') is a
 function of one non-@(tsee state) argument that produces a @(see defun-mode)
 as its single value),</p>

 @({
  (table acl2-defaults-table :defun-mode (compute-mode (w state)))
 })

 <p>is not legal because the value form is @(tsee state)-sensitive.</p>

 <p>Consider for example the following three @(see events) which one might make
 into the text of a book.</p>

 @({
  (in-package \"ACL2\")

  (table acl2-defaults-table
    :defun-mode
    (if (ld-skip-proofsp state) :logic :program))

  (defun crash-and-burn (x) (car x))
 })

 <p>The second event is illegal because its value form is @(tsee
 state)-sensitive.  If it were not illegal, then it would set the @(':')@(tsee
 defun-mode) to @(':')@(tsee program) when the book was being certified but
 would set the @(see defun-mode) to @(':')@(tsee logic) when the book was being
 loaded by @(tsee include-book).  That is because during certification, @(tsee
 ld-skip-proofsp) is @('nil') (proof obligations are generated and proved), but
 during book inclusion @(tsee ld-skip-proofsp) is non-@('nil') (those
 obligations are assumed to have been satisfied.)  Thus, the above book, when
 loaded, would create a function in @(':')@(tsee logic) mode that does not
 actually meet the conditions for such status.</p>

 <p>For similar reasons, @(tsee table) @(see events) affecting
 @('acl2-defaults-table') are illegal within the scope of @(tsee local) forms.
 That is, the text</p>

 @({
  (in-package \"ACL2\")

  (local (table acl2-defaults-table :defun-mode :program))

  (defun crash-and-burn (x) (car x))
 })

 <p>is illegal because @('acl2-defaults-table') is changed locally.  If this
 text were acceptable as a book, then when the book was certified,
 @('crash-and-burn') would be processed in @(':')@(tsee program) mode, but when
 the certified book was included later, @('crash-and-burn') would have
 @(':')@(tsee logic) mode because the @(tsee local) event would be skipped.</p>

 <p>The text</p>

 @({
  (in-package \"ACL2\")

  (program) ;which is (table acl2-defaults-table :defun-mode :program)

  (defun crash-and-burn (x) (car x))
 })

 <p>is acceptable and defines @('crash-and-burn') in @(':')@(tsee program)
 mode, both during certification and subsequent inclusion.</p>

 <p>We conclude with important observations about the interaction of the
 @('acl2-defaults-table') with @(tsee include-book), @(tsee certify-book), and
 @(tsee encapsulate).  If the @('acl2-defaults-table') has value @('V') and you
 evaluate a call of @(tsee include-book), @(tsee certify-book), or @(tsee
 encapsulate), then the @('acl2-defaults-table') has value @('V') when that
 call returns.  Thus, if you want to set the @('acl2-defaults-table') in a way
 that persists, you need to do so using @(see command)s that are not inside
 @(see books).  It may be useful to set your favorite defaults in your @(tsee
 acl2-customization) file; see @(see acl2-customization).</p>

 <p>ACL2 disallows (for logical reasons) setting of the
 @('acl2-defaults-table') in the context of @('LOCAL').  Often it is easy
 simply to avoid wrapping an @('acl2-defaults-table') event, or a macro that
 generates such an event, inside a @('LOCAL'); after all, a local context is
 not useful when setting the @('acl2-defaults-table'), since that table is
 restored as discussed above upon completion of @(tsee include-book), @(tsee
 certify-book), and @(tsee encapsulate) forms.  Occasionally a bit more thought
 is required to work around this restriction, but usually it's not difficult to
 do so.  Suppose for example that you attempt to put following event form into
 a book.</p>

 @({
 (local (progn (defttag t) (defun foo (x) (sys-call x nil))))
 })

 <p>ACL2 responds with the following error message.</p>

 @({
 ACL2 Error in ( PROGN (DEFTTAG T) ...):  The form (DEFTTAG T)
 is not an embedded event form in the context of LOCAL
 because it implicitly sets the acl2-defaults-table in
 a local context; see :DOC acl2-defaults-table, in particular
 the explanation about this error message.  See :DOC
 embedded-event-form.
 })

 <p>A solution is to use @(tsee encapsulate) instead of (or in addition to)
 @(tsee progn), because @(tsee encapsulate) establishes a new context that is
 not considered within a surrounding @('LOCAL').  For example, the following
 replacement for the form above is legal in a book.</p>

 @({
 (local (encapsulate () (defttag t) (defun foo (x) (sys-call x nil))))
 })

 <p>To see that this works, try creating a file @('\"foo.lisp\"') whose first
 form is @('(in-package \"ACL2\")') and whose only other form is the one
 displayed just above.  Then the command @('(certify-book \"foo\" 0 t :ttags
 :all)') will successfully certify that book.</p>")

(defxdoc acl2-doc

; Warning: Keep this in sync with acl2-doc-summary.

  :parents (documentation)
  :short "A custom Emacs browser for reading ACL2 @(see documentation)"
  :long "<p>As discussed elsewhere (see @(see documentation)), the web-based
 @(`(:raw (combined-manual-ref))`) provides a way to browse the combined
 documentation for the ACL2 system and community books.  Such documentation can
 also be read at the terminal using the @(':')@(tsee doc) command, though
 documentation for @(see books) will only be included for those books that have
 been included in the session.  In this topic we describe how to browse the
 documentation using ACL2-Doc, a browser for reading
 documentation inside Emacs.  It supports reading the combined documentation
 (ACL2 plus books), but it also supports reading the ACL2-only manual as well
 as custom manuals.  We assume some familiarity with Emacs, for example, the
 notion of a ``prefix argument'': a numeric value given first with the
 @('meta') key (or, probably the @('control') key), for example, @('meta-0
 control-t g') or @('control-3 control-t /').</p>

 <p>While ACL2-Doc is much like Emacs Info, it is a separate system that
 provides some additional functionality.  ACL2-Doc is text-based.  Any word
 that names a topic is a link: you can hit the @('<Return>') key while standing
 on that topic to go to the page of documentation for that topic.  However,
 many links are shown explicitly, inside square brackets.  For example, here is
 a link that will take you to the BOOKS topic; it should show up surrounded by
 square brackets if you are now reading at the terminal or in ACL2-Doc, but
 there are no square brackets for this link if you are reading on the web.</p>

 <blockquote>

 <p>@(see books)</p>

 </blockquote>

 <p>It should be very rare for square brackets to be intended simply as square
 brackets, not as link indicators.</p>

 <p>In order to use ACL2-Doc, load the distributed file
 <tt>emacs/acl2-doc.el</tt> into Emacs.  This will happen automatically if you
 load <tt>emacs/emacs-acl2.el</tt>, which will happen automatically if you put
 the following form in your <tt>~/.emacs</tt> file, replacing @('DIR') by a
 path to your ACL2 installation.</p>

 @({
 (load \"DIR/emacs/emacs-acl2.el\")
 })

 <p>Then to start the browser at the top-level topic, either execute the Emacs
 command</p>

 @({
 meta-x acl2-doc
 })

 <p>or else type:</p>

 @({
 Control-t g
 })

 <p>Thus, you can put the following form in your @('.emacs') file if you want
 @('acl2-doc') to run automatically when Emacs starts up.</p>

 @({
 (acl2-doc)
 })

 <p>By default you will browse the ACL2+Books Manual, though if you are using a
 git version between ACL2 releases then you may be queried; more on that below.
 Or, see below for how to set a variable in your @('.emacs') file,
 @('*acl2-doc-manual-name*'), so that you will browse a custom manual.  You can
 enter the ACL2-Doc browser at a specific documentation topic as follows (in
 analogy to Emacs command @('Meta-.')):</p>

 @({
 Control-t .
 })

 <p>In each of the cases above, you will now be in a buffer called
 \"<tt>acl2-doc</tt>\", which will be displaying the top-level ACL2 topic in a
 special mode, the ACL2-Doc major mode.  That mode provides the following key
 bindings; you can also see these by typing <tt>Control-h m</tt> while in that
 buffer.</p>

 @({
  <Return>      acl2-doc-go!
  g             acl2-doc-go
  h             acl2-doc-help
  ?             acl2-doc-summary
  i             acl2-doc-index
  ,             acl2-doc-index-next
  <             acl2-doc-index-previous
  l             acl2-doc-last
  n             acl2-doc-search-next
  p             acl2-doc-search-previous
  q             acl2-doc-quit
  r             acl2-doc-return
  s             acl2-doc-search
  S             acl2-doc-re-search
  t             acl2-doc-top
  u             acl2-doc-up
  w             acl2-doc-where
  SPC           scroll-up
  TAB           acl2-doc-tab
  Control-TAB or <backtab> (which often is Shift-TAB):
                acl2-doc-tab-back
  D             acl2-doc-rendered-combined-download
  H             acl2-doc-history
  I             acl2-doc-initialize
  /             acl2-doc-definition
  Control-t /   acl2-doc-definition
  W             acl2-doc-where-definition
 })

 <p>You can see the documentation for each of these in the usual way, using
 <tt>Control-h k {key}</tt> or <tt>Control-h f {command}</tt>.  Here is what
 you will find in each case if you do that.</p>

 @({
  <Return>      acl2-doc-go!
     Go to the topic occurring at the cursor position.  In the case
     of <NAME>, instead go to the source code definition of NAME for
     the current manual (as for `/', but without a minibuffer query).

  g             acl2-doc-go
     Go to the specified topic; performs completion.

  h             acl2-doc-help
     Go to the ACL2-DOC topic to read about how to use the ACL2-Doc browser.

  ?             acl2-doc-summary
     Go to the ACL2-Doc-summary topic for one-line summaries of ACL2-Doc
     browser commands.

  i             acl2-doc-index
     Go to the specified topic or else one containing it as a substring;
     performs completion.  If the empty string is supplied, then go to the
     index buffer.  Otherwise, with prefix argument, consider only descendants
     of the topic supplied in response to a prompt.  Note that the index buffer
     is in ACL2-Doc mode; thus, in particular, you can type <RETURN> while
     standing on a topic in order to go directly to that topic.

  ,             acl2-doc-index-next
     Find the next topic containing, as a substring, the topic of the most
     recent i command.  Note: if this is the first \",\" or \"<\" after an
     exact match from \"i\", then start the topic search alphabetically from
     the beginning, but avoid a second hit on the original topic.

  <             acl2-doc-index-previous
     Find the previous topic containing, as a substring, the topic of the
     most recent i command.  Note: if this is the first \",\" or \"<\" after an
     exact match from \"i\", then start the topic search alphabetically
     (backwards) from that exact match.

  l             acl2-doc-last
     Go to the last topic visited.

  n             acl2-doc-search-next
     Find the next occurrence for the most recent search or regular expression
     search.

  p             acl2-doc-search-previous
     Find the previous occurrence for the most recent search or regular
     expression search.  Note: as for \"n\", the cursor will end up at the end
     of the match.

  q             acl2-doc-quit
     Quit the ACL2-Doc browser.

  r             acl2-doc-return
     Return to the last topic visited, popping the stack of such topics.

  s             acl2-doc-search
     Search forward from the top of the manual for the input string.  If the
     search succeeds, then go to that topic with the cursor put immediately
     after the found text, with the topic name displayed in the minibuffer.
     With prefix argument, consider (also for subsequent \"n\" and \"p\"
     commands) only descendants of the topic supplied in response to a prompt.

  S             acl2-doc-re-search
     Perform a regular expression search, forward from the top of the manual,
     for the input string.  If the search succeeds, then go to that topic with
     the cursor put immediately after the found text, with the topic name
     displayed in the minibuffer.  With prefix argument, consider (also for
     subsequent \"n\" and \"p\" commands) only descendants of the topic
     supplied in response to a prompt.

  t             acl2-doc-top
     Go to the top topic.

  u             acl2-doc-up
     Go to the parent of the current topic.

  w             acl2-doc-where
     Display the topic name in the minibuffer, together with the manual name
     (ACL2+Books Manual or ACL2 User's Manual)

  SPC           scroll-up
     Scroll up (same as Control-v)

  TAB           acl2-doc-tab
     Visit the next link after the cursor on the current page, searching from
     the top if no link is below the cursor.

  Control-TAB or <backtab> (which often is Shift-TAB):
                acl2-doc-tab-back
     Visit the previous link before the cursor on the current page, searching
     from the bottom if no link is below the cursor.

  D
     Download the ``bleeding edge'' ACL2+Books Manual from the web; then
     restart the ACL2-Doc browser to view that manual.

  H             acl2-doc-history
     Visit a buffer that displays the names of all visited topics in order,
     newest at the bottom.  That buffer is in acl2-doc mode; thus the usual
     acl2-doc commands may be used.  In particular, you can visit a displayed
     topic name by putting your cursor on it and typing <RETURN>.

  I             acl2-doc-initialize
     Restart the ACL2-Doc browser, clearing its state.  With a prefix argument,
     a query asks you to select the name of an available manual, using
     completion.  See the section on \"Selecting a Manual\", below,
     for more information.

  /             acl2-doc-definition
  (also control-t /)
     Find an ACL2 definition (in analogy to built-in Emacs command meta-.).
     With numeric prefix argument, find the next matching definition;
     otherwise, the user is prompted, where the default is the name at
     the cursor, obtained after stripping off any enclosing square
     brackets (@('[..]')), angle brackets (@('<..>')) as from srclink tags, and
     package prefixes.  With @('control-u') prefix argument, search only
     ACL2 source definitions; otherwise, books are searched as well.
     As with built-in Emacs command @('meta-.') , exact matches are given
     priority.  For more information, see the Section on \"Selecting a
     Manual\" in the acl2-doc online XDOC-based documentation.

  W             acl2-doc-where-definition
     Find an ACL2 definition.  This is the same as
     acl2-doc-definition (the acl2-doc `/' command, as well as
     control-t /), except that the default comes from the name of the
     current page's topic instead of the cursor position.  Searches
     are continued identically when control-t / is given a numeric
     prefix argument, regardless of whether the first command was /,
     control-t /, or W; thus, a search started with W can be continued
     with, for example, meta-3 control-t /.

 })

 <h4>Selecting a Manual, Tags Files, and Custom Manuals</h4>

 <p>ACL2-Doc can display the ACL2 User's Manual, which includes documentation
 for the ACL2 system but not for the @(see community-books).  But by default,
 ACL2-Doc will display the ACL2+Books Manual, which includes documentation for
 those books as well.  To change which of these two manuals you display, just
 give a prefix argument to the `@('I')' command, as described briefly
 above.</p>

 <p>The acl2-doc browser makes a query when first loading a manual if there is
 a newer web-based manual (specifically, comparing the write date of the
 acl2-doc manual, which is typically in
 @('books/system/doc/rendered-doc-combined.lsp'), to the write-date of the file
 @('books/doc/manual/index.html') when that file exists).  If you decline, then
 you will be given the opportunity to download that acl2-doc manual from the
 web.  If you prefer, you can rebuild the acl2-doc manual yourself when buiding
 the web-based manual, for example as follows.</p>

 @({
 cd <your_acl2_directory>/books
 make manual ACL2_DOC_GENERATE_SUPPORTING_FILES=t
 })

 <p>Note that the ``@('make')'' target, ``@('regression-everything')'',
 automatically sets @('ACL2_DOC_GENERATE_SUPPORTING_FILES'); or you can set it
 as an environment variable.  Any non-empty value other than (case-insensitive)
 @('SKIP') will cause the acl2-doc manual to be built when building the
 web-based manual.</p>

 <p>For the `@('/')' and `@('W')' commands, you will need tags table files.
 These come with the ACL2 gzipped tarfile distribution, but if you obtain ACL2
 from github then you will need to build them.  The file @('\"TAGS\"') is used
 when these commands are given a prefix argument (to search only the ACL2
 sources), and is generated when building the @('saved_acl2') executable with
 `@('make')'.  Without a prefix argument the file @('\"TAGS-acl2-doc\"') is
 used for searching both the ACL2 sources and the books, and is created
 automatically if you build the manual by certifying community book
 @('books/doc/top.lisp').  You can also build @('\"TAGS-acl2-doc\"') by running
 the command @('bin/make-tags-acl2-doc.sh'), or by building the ACL2 executable
 after setting variable @('TAGS_ACL2_DOC') to a non-empty value other than
 @('SKIP') either on the command line with `@('make')' or, for example, by
 putting one of the following forms in your @('~/.cshrc') or @('~/.bashrc'),
 respectively.</p>

 @({
 setenv TAGS_ACL2_DOC t

 export TAGS_ACL2_DOC=t
 })

 <p>If you are using a git version of ACL2 and the books, between releases,
 then you may need to download an extra file in order to browse the ACL2+Books
 Manual.  Most likely you will just answer @('y') when queried about
 downloading the file when first using ACL2-Doc.  If you want more details, see
 the last of the notes in the ``Notes'' section below.</p>

 <p>As mentioned above, you can give the `@('I')' command a prefix argument in
 order to select a specific manual.  You will be asked for a name, which by
 default will be the most recently selected such name, if any.  As noted above,
 the only two manuals initially known to acl2-doc are the ACL2+Books Manual and
 the ACL2 User's Manual.  These have the names `@('combined')' and
 `@('acl2-only')', respectively.  You can also tell acl2-doc about a custom
 manual, by evaluating (in Emacs) the following form, e.g., by adding it to
 your @('~/.emacs') file before starting Emacs.  Here, @('filename') is the
 pathname of a file typically created by calling @(see
 xdoc::save-rendered).</p>

 @({
 (extend-acl2-doc-manual-alist
  'name          ; the name of the manual
  filename       ; documentation source, typically of the form *doc*.lsp
  'top           ; the top node name, typically TOP
  printname      ; optional print name (user-level name) of manual
  url            ; optional URL of gzipped file to download into filename
  tags-file-name ; optional tags filename (from output of etags)
  acl2-tags-file-name ; as above, but without files from books/ directory
  )
 })

 <p>For example, acl2-doc is initially built with the following two forms,
 which is why you can respond to the query mentioned above with `@('combined')'
 or `@('acl2-only')'.</p>

 @({
 (extend-acl2-doc-manual-alist
  'combined
  (concat *acl2-sources-dir*
          \"books/system/doc/rendered-doc-combined.lsp\")
  'TOP
  \"ACL2+Books Manual\"
  \"http://www.cs.utexas.edu/users/moore/acl2/manuals/current/rendered-doc-combined.lsp.gz\"
  (concat *acl2-sources-dir* \"TAGS-acl2-doc\")
  (concat *acl2-sources-dir* \"TAGS\"))

 (extend-acl2-doc-manual-alist
  'acl2-only
  (concat *acl2-sources-dir* \"doc.lisp\")
  'ACL2
  \"ACL2 User's Manual\"
  nil
  nil
  (concat *acl2-sources-dir* \"TAGS\"))
 })

 <p>Note that the first of these forms specifies the location of the
 @('\"TAGS-acl2-doc\"') and @('\"TAGS\"') files mentioned above, but the second
 only specifies @('\"TAGS\"') since the second form is for an ACL2-only manual
 (no books).</p>

 <p>If you want a specific manual to come up when you first run acl2-doc in an
 Emacs session, you can put the following into your @('.emacs') file, where
 @(''name') is the name a manual for which you have included a form
 @('(extend-acl2-doc-manual-alist 'name ...)') in your @('.emacs') file.</p>

 @({
 (setq *acl2-doc-manual-name* 'name)
 })

 <h4>Color</h4>

 <p>By default, links (indeed, any text) in square brackets will be shown in
 blue.  You can customize this behavior by setting (e.g., in your @('.emacs')
 file) the Emacs variable @('*acl2-doc-link-color*') to the desired link color,
 or to @('nil') if you don't want the links to be in color.  For example:</p>

 @({
 (setq *acl2-doc-link-color* \"#FF0000\") ; red
 (setq *acl2-doc-link-color* \"Green\")   ; green
 (setq *acl2-doc-link-color* nil)         ; no special color for links
 })

 <h4>Notes</h4>

 <ul>

 <li>You might find that when you attempt to follow @(see some-broken-link),
 you find yourself at the @(see broken-link) topic.  If you are using the ACL2
 User's Manual rather than the ACL2+Books Manual, the reason might be that
 @('some-broken-link') is documented in a book, not in the ACL2 system.  In
 that case, the @('broken-link') page will show you where to find that book;
 but if you want to read the documentation for @('some-broken-link') in the
 ACL2-Doc browser, you can do so by switching to the ACL2+Books Manual.  See
 the @('I') command, documented above.<p/></li>

 <li>Files with names ending in <tt>.acl2-doc</tt> will come up in ACL2-Doc
 mode.  Thus, you may wish to save a file with that extension, for example
 <tt>bookmarks.acl2-doc</tt>, that contains your favorite bookmarks.  You may
 wish to use the history command (<tt>H</tt>) to obtain a list of names of
 visited topics, in order to create an initial such file.<p/></li>

 <li>Many commands offer defaults, and many offer completion.  The default is
 determined by cursor position: if the cursor is sitting on a letter of a
 documentation topic name, or on a space character immediately after it, then
 that name will be offered as the default.  Completion tips:<p/>

 <ul>

 <li>Completion is carried out with the usual emacs ``@('completing-read')'';
 thus, for example, the character `@('?')' is a help key, so if you want that
 character as part of your topic name, prefix it with @('control-q').  For
 example, after the `@('g')' command you can go to the topic @(tsee mv?) by
 typing the character sequence @('<m,v,control-q ?>').</li>

 <li>To find completions that have package prefixes, type a colon (:) in the
 front, and completion will show matching topics.  For example, @('\"g\"')
 followed by @('\":rew\"') and then two tabs will show, at least in recent
 versions of Emacs, a list of topics that includes
 @('\"ACL2-PC::REWRITE\"').</li>

 </ul><p/></li>

 <li>Square brackets typically indicate documentation topic names, for example:
 <tt>[acl2-doc]</tt>.  (As mentioned above, there are occasional exceptions,
 where square brackets are simply part of the intended documentation text.)
 The square brackets are really there, for example when you are searching using
 \"s\", \"S\", or \"n\".  However, for purposes of determining the default
 name (see above), the only effect of the enclosing square brackets is to
 extend the region in which the default is offered.  For example, consider the
 string \"<tt>[acl2-doc]</tt>\": the default name of \"<tt>acl2-doc</tt>\" is
 offered if the cursor is on either square bracket.  But links have some
 idiosyncrasies.<p/>

 <ol>

   <li>Topic names, including links `<tt>[..]</tt>' to topic names, are printed
   relative to the ACL2 package.  Especially in the case of the ACL2+Books
   Manual, you may therefore see links that include package prefixes.  Here,
   for example, is a sentence from the documentation for @(see gl) in the
   ACL2+Books Manual.

   <code>
    We call these structures [gl::symbolic-objects].
   </code>

   The \"<tt>gl</tt>\" package prefix allows commands to pick up
   \"<tt>gl::symbolic-objects</tt>\" as the name to use as a default, so that
   for example, hitting @('<Return>') will take you to that topic.  But when
   reading the sentence, for best results you should ignore package prefixes.
   So for example, you would read the sentence above as follows.

   <code>
    We call these structures symbolic-objects.
   </code><p/></li>

   <li>Inside ACL2-Doc, topic names that originally contained spaces now have
   underscores in place of the spaces.  So for example, the topic @(see
   ACL2-TUTORIAL) in the ACL2+Books Manual contains a link to the topic
   originally named as follows.<br/>

   @('|Pages Written Especially for the Tours|')<br/>

   This link shows up in the ACL2-Doc browser as:<br/>

   @('[Pages_Written_Especially_for_the_Tours]').</li>

 </ol>

 Of course, the web-based browser avoids these idiosyncrasies (see @(see
 xdoc::save)); in particular, that browser is best for ``Tours'' pages like the
 one shown above and its subtopics, in part because that way you may see
 images.  Hence the web-based browser may be more appropriate for some topics,
 and for those who have no particular preference for using Emacs to browse the
 documentation.<p/></li>

 <li>Searching using the \"s\" or \"S\" command is carried out by searching
 top-to-bottom in a hidden Emacs buffer that contains all of the
 documentation.  The topics are listed in the following order according to
 topic name:<p/>

   <ol>

   <li>All topics whose names reside in the \"<tt>ACL2</tt>\" package;</li>

   <li>All topics whose names reside in the \"<tt>ACL2-PC</tt>\" package; and,
   for the ACL2+Books Manual,</li>

   <li>All other topics, sorted by @(tsee symbol-name) and then by @(tsee
   symbol-package-name).</li>

   </ol><p/></li>

 <li>You may be queried, regarding whether you want to browse the ACL2+Books
 Manual, which is preferred, or the ACL2 User's Manual, which omits
 documentation for the books.  Both of these manuals are based on files that
 you will have if you are using a released version of ACL2 (after Version 6.3).
 But if you are using a <a href='https://github.com/acl2/acl2/'>git
 version</a>, then to use the ACL2+Books Manual you will need an extra file.
 You can build this file yourself, as described below but you may prefer to
 download it: for example, when you start ACL2-Doc, you may be given the option
 of downloading <a
 href='http://www.cs.utexas.edu/users/moore/acl2/manuals/current/rendered-doc-combined.lsp.gz'>a
 tarball for the latest ``bleeding edge'' copy</a> and extracting into
 directory @('system/doc/') of your community books directory.  Indeed, the
 system will do all this for you if you answer @('y') to that query.
 Alternatively, you can insist on a download of a ``bleeding edge'' version by
 using the `<tt>D</tt>' command.  However, if you prefer to browse the ACL2
 User's Manual (without the books), you can put the following form into your
 <tt>~/.emacs</tt> file, above the form that loads the code for ACL2-Doc (see
 above).<p/>

 @({
 (defvar *acl2-doc-manual-name* 'acl2-only)
 })

 If you prefer to build @('rendered-doc-combined.lsp') yourself, you can do so
 as follows.<p/>

 <ol>

   <li>Build ACL2:
   @({
   make
   })</li>

   <li>Build the file @('books/system/doc/rendered-doc-combined.lsp') while
   standing in the @('books/') directory, as follows.  If \"<tt>acl2</tt>\"
   invokes the ACL2 executable that you just built, then you may omit
   \"<tt>ACL2=acl2</tt>\" below; otherwise replace \"<tt>acl2</tt>\" by a
   suitable executable.  On a multi-core machine you may wish to use @('-j'),
   e.g., @('make -j 4 ...').
   @({
   cd books
   make doc/top.cert USE_QUICKLISP=1 ACL2=acl2
   })</li>

 </ol></li></ul>")

(defxdoc acl2-doc-summary

; Warning: Keep this in sync with acl2-doc.

  :parents (documentation)
  :short "Summary of @(see acl2-doc) commands"
  :long "<p>See @(see acl2-doc) for information about the custom Emacs browser
  for viewing ACL2 @(see documentation).  In the present topic we list the
  commands with extremely abbreviation documentation: only a single line for
  each.  For even briefer summaries, you can use the standard Emacs command,
  @('Control-h m').</p>

 @({
    <Return>      acl2-doc-go!
       Go to the topic occurring at the cursor position.
    g             acl2-doc-go
       Go to the specified topic; performs completion.
    h             acl2-doc-help
       Go to the ACL2-Doc topic to read about how to use the ACL2-Doc browser.
    ?             acl2-doc-summary
       Go to the ACL2-Doc-summary topic for one-line summaries of commands.
    i             acl2-doc-index
       Go to the specified topic or else one containing it as a substring.
    ,             acl2-doc-index-next
       Continue to the next topic for the most recent i command.
    <             acl2-doc-index-previous
       Return to the preceding topic for the most recent i command.
    l             acl2-doc-last
       Go to the last topic visited.
    n             acl2-doc-search-next
       Find the next occurrence for the most recent search.
    p             acl2-doc-search-previous
       Find the previous occurrence for the most recent search.
    q             acl2-doc-quit
       Quit the ACL2-Doc browser.
    r             acl2-doc-return
       Return to the last topic visited, popping the stack of such topics.
    s             acl2-doc-search
       Search for the input string (with prefix arg: under a given topic).
    S             acl2-doc-re-search
       Regular-expression search (with prefix arg: under a given topic).
    t             acl2-doc-top
       Go to the top topic.
    u             acl2-doc-up
       Go to the parent of the current topic.
    w             acl2-doc-where
       Display the topic and manual name in the minibuffer.
    SPC           scroll-up
       Scroll up (same as Control-v)
    TAB           acl2-doc-tab
       Visit the next link on the current page.
    Control-TAB or <backtab> (which often is Shift-TAB): acl2-doc-tab-back
       Visit the previous link on the current page.
    D
       Download the manual from the web; then restart ACL2-Doc.
    H             acl2-doc-history
       Visit the History buffer, with names of all visited topics in order.
    I             acl2-doc-initialize
       Restart ACL2-Doc.  With a prefix argument, choose which manual.
    /             acl2-doc-definition
       Find an ACL2 definition (in analogy to built-in Emacs command meta-.).
    W             acl2-doc-where-definition
       Find an ACL2 definition, with default from current page's topic.
 })")

(defxdoc acl2-help
  :parents (about-acl2)
  :short "The acl2-help mailing list"
  :long "<p>You can email questions about ACL2 usage to the acl2-help mailing
 list: @('acl2-help@utlists.utexas.edu').  If you have more general questions
 about ACL2, for example, about projects completed using ACL2, you may prefer
 the acl2 mailing list, @('acl2@utlists.utexas.edu'), which tends to have wider
 distribution.</p>")

(defxdoc acl2-number-listp
  :parents (numbers lists acl2-built-ins)
  :short "Recognizer for a true list of numbers"
  :long "<p>The predicate @('acl2-number-listp') tests whether its argument is
 a true list of numbers.</p>

 @(def acl2-number-listp)")

(defxdoc acl2-numberp
  :parents (numbers acl2-built-ins)
  :short "Recognizer for numbers"
  :long "<p>@('(acl2-numberp x)') is true if and only if @('x') is a number,
 i.e., a rational or complex rational number.</p>")

(defxdoc acl2-sedan
  :parents (acl2-tutorial)
  :short "ACL2 Sedan interface"
  :long "<p>Many successful ACL2 users run in an shell under Emacs; see @(see
 emacs).  However, those not familiar with Emacs may prefer to start with an
 Eclipse-based interface initially developed by Peter Dillinger and Pete
 Manolios called the <a href='http://acl2s.ccs.neu.edu/acl2s/doc/'>ACL2
 Sedan</a> or ``ACL2s''.</p>

 <p>ACL2 sessions in the ACL2 Sedan can utilize non-standard extensions and
 enhancements, especially geared toward new users, termination reasoning, and
 attaching rich user interfaces.  These extensions are <a
 href='http://acl2s.ccs.neu.edu/acl2s/src/acl2-extensions'>generally
 available</a> as certifiable ACL2 books.
 (Some code originating from this project has been migrated to the ACL2
 community books, but only after it was quite stable.)  Thanks to Peter
 Dillinger, Pete Manolios, Daron Vroon, and Harsh Raju Chamarthi for their work
 on the ACL2 Sedan and for making their books available to ACL2 users.</p>")

(defxdoc acl2-tutorial

; The following comment was near the top of ACL2 source file tutorial.lisp
; before that file was eliminated.

; This document currently has the following form:
;
; :doc ACL2-tutorial
;   introduction
;     OVERVIEW
;     ABOUT THIS TUTORIAL:
;     GETTING STARTED:
;     INTERACTING WITH ACL2:
;   :doc examples
;     EXAMPLE: TOWERS OF HANOI
;     EXAMPLE: EIGHTS PROBLEM
;     A LARGER EXAMPLE: A PHONEBOOK SPECIFICATION
;     DEFUN-SK-EXAMPLE:: example of quantified notions
;     :doc miscellaneous-examples
;       * FILE-READING-EXAMPLE:: example of reading files in ACL2
;       * MUTUAL-RECURSION-PROOF-EXAMPLE:: a small proof about mutually
;            recursive functions
;       * GUARD-EXAMPLE    a brief transcript illustrating guards in ACL2
;   STARTUP
;   TIDBITS
;   TIPS

  :parents (acl2)
  :short "Tutorial introduction to ACL2"
  :long "<p>To learn about ACL2, read at least the following two links.</p>

 <ul>

 <li><see topic='@(url INTERESTING-APPLICATIONS)'>Industrial Applications of
 ACL2</see> (10 minutes) to help you understand what sophisticated users can
 do;</li>

 <li><see topic='@(url |A Flying Tour of ACL2|)'>A Flying Tour</see> (10
 minutes) to get an overview of the system and what skills the user must
 have.</li>

 </ul>

 <p>Alternatively, or in addition, there are @(see talks) that you can peruse,
 many of them introductory in nature.</p>

 <p>If you want to learn <i>how to use</i> ACL2, we recommend that you read a
 selection of the materials referenced below, depending on your learning style,
 and do suggested exercises.</p>

 <ul>

 <li><see topic='@(url |A Walking Tour of ACL2|)'>A Walking
 Tour</see> (1 hour) provides an overview of the theorem prover.</li>

 <li>The <a href='http://tryacl2.org'>Try ACL2</a> web site provides
 interactive lessons to get you started using ACL2.</li>

 <li>See @(see introduction-to-the-theorem-prover) (10-40 hours) for
 instruction on how to interact with the system.  Unlike the three documents
 above, this document expects you to <i>think</i>!  It cites the necessary
 background pages on programming in ACL2 and on the logic and then instructs
 you in @(see the-method), which is how expert users use ACL2.  It concludes
 with some challenge problems for the ACL2 beginner (including solutions) and
 an FAQ.  Most users will spend several hours a day for several days working
 through this material.</li>

 <li>The book <a
 href='http://www.cs.utexas.edu/users/moore/publications/acl2-books/car/index.html'>Computer-Aided
 Reasoning: An Approach</a> is worth a careful read, as you work exercises and
 learn @(see the-method).</li>

 <li><see topic='@(url ANNOTATED-ACL2-SCRIPTS)'>Annotated ACL2 Scripts and
 Demos</see> contains relatively elementary proof scripts that have been
 annotated to help train the newcomer.</li>

 <li>Many files (``books'') in the ACL2 community books (see @(see
 community-books)) are extensively annotated.</li>

 <li>An <see topic='@(url ALTERNATIVE-INTRODUCTION)'>Alternative
 Introduction</see> document, while largely subsumed by the <see topic='@(url
 INTRODUCTION-TO-THE-THEOREM-PROVER)'>Introduction to the Theorem Prover</see>
 mentioned above, still might be useful because it covers much of the tutorial
 material in a different way.</li>

 </ul>

 <p>At this point you are probably ready to use ACL2 on your own <i>small</i>
 projects.  A common mistake for beginners is to browse the documentation and
 then try to do something that is too big!  Think of a very small project and
 then simplify it!</p>

 <p>Note that ACL2 has a very supportive user network.  See the link to
 ``Mailing Lists'' on the <a
 href='http://www.cs.utexas.edu/users/moore/acl2'>ACL2 home page</a>.</p>

 <p>The topics listed below are a hodge podge, developed over time.  Although
 some of these are not mentioned above, you might find some to be useful as
 well.</p>")

(defxdoc acl2-user
  :parents (packages)
  :short "A package the ACL2 user may prefer"
  :long "<p>This package imports the standard Common Lisp symbols that ACL2
 supports and also a few symbols from package @('\"ACL2\"') that are commonly
 used when interacting with ACL2.  You may prefer to select this as your
 current package so as to avoid colliding with ACL2 system names.</p>

 <p>This package imports the symbols listed in
 @('*common-lisp-symbols-from-main-lisp-package*'), which contains hundreds of
 CLTL function and macro names including those supported by ACL2 such as @(tsee
 cons), @(tsee car), and @(tsee cdr).  It also imports the symbols in
 @('*acl2-exports*'), which contains a few symbols that are frequently used
 while interacting with the ACL2 system, such as @(tsee implies), @(tsee
 defthm), and @(tsee rewrite).  It imports nothing else.</p>

 <p>Thus, names such as @(tsee alistp), @(tsee member-equal), and @(tsee
 type-set), which are defined in the @('\"ACL2\"') package are not present
 here.  If you find yourself frequently colliding with names that are defined
 in @('\"ACL2\"') you might consider selecting @('\"ACL2-USER\"') as your
 current package (see @(see in-package)).  If you select @('\"ACL2-USER\"') as
 the current package, you may then simply type @(tsee member-equal) to refer to
 @('acl2-user::member-equal'), which you may define as you see fit.  Of course,
 should you desire to refer to the @('\"ACL2\"') version of @(tsee
 member-equal), you will have to use the @('\"ACL2::\"') prefix, e.g.,
 @('acl2::member-equal').</p>

 <p>If, while using @('\"ACL2-USER\"') as the current package, you find that
 there are symbols from @('\"ACL2\"') that you wish we had imported into it
 (because they are frequently used in interaction), please bring those symbols
 to our attention.  For example, should @(tsee union-theories) and @(tsee
 universal-theory) be imported?  Except for stabilizing on the ``frequently
 used'' names from @('\"ACL2\"'), we intend never to define a symbol whose
 @(tsee symbol-package-name) is @('\"ACL2-USER\"').</p>")

(defxdoc acl2p-key-checkpoints
  :parents (parallel-proof)
  :short "Key checkpoints in ACL2(p)"
  :long "<p>This @(see documentation) topic relates to the experimental
 extension of ACL2 supporting parallel execution and proof; see @(see
 parallelism).</p>

 <p>For printing output, the parallel version of the waterfall follows the
 precedent of @(tsee gag-mode).  The idea behind gag mode is to print only the
 subgoals most relevant to debugging a failed proof attempt.  These subgoals
 are called 'key checkpoints' (see @(see set-gag-mode) for the definition of
 ``key'' and ``checkpoint''), and we restrict the default output mode for the
 parallel version of the waterfall to printing checkpoints similar to these key
 checkpoints.</p>

 <p>As of this writing, we are aware of exactly one discrepancy between gag
 mode's key checkpoints and the parallel version of the waterfall's
 checkpoints.  This discrepancy occurs when using ``by'' hints (see @(see
 hints)).  As an example, take the following form, which attempts to prove a
 non-theorem:</p>

 @({
  (thm (equal (append x y z) (append z (append y x)))
       :hints ((\"Subgoal *1/2'''\" :by nil)))
 })

 <p>With waterfall parallelism enabled, @('Subgoal *1/2''') will be printed as
 a key checkpoint.  This is different from using @(tsee gag-mode) while running
 the serial version of the waterfall, which skips printing the subgoal as a
 checkpoint.</p>

 <p>For those familiar with the ACL2 waterfall, we note that the parallel
 version of the waterfall prints key checkpoints that are unproved in the
 following sense: a subgoal is a key checkpoint if it leads, in the current
 call of the waterfall, to a goal that is pushed for induction.</p>")

(defxdoc acons
  :parents (alists acl2-built-ins)
  :short "Constructor for association lists"
  :long "<p>@('(Acons key datum alist)') equals the result of consing the pair
 @('(cons key datum)') to the front of the association list @('alist').</p>

 <p>@('(Acons key datum alist)') has a @(see guard) of @('(alistp alist)').
 @('Acons') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def acons)")

(defxdoc active-runep
  :parents (theories)
  :short "Check that a @(see rune) exists and is @(see enable)d"
  :long "@({
  Example:
  (active-runep '(:rewrite left-to-right))

  General Form:
  (active-runep rune &optional strict)
 })

 <p>where @('rune') has the shape of a @(see rune).  This macro expands to an
 expression using the variables @('ens') and @('state'), and returns
 non-@('nil') when the given rune exists and is @(see enable)d (according to
 the given ``enabled structure,'' @('ens'), and the current logical @(see
 world) of the given @(tsee state)).  See @(see theory-invariant) for how this
 macro can be of use.</p>

 <p>When the optional argument is @('nil') or is omitted, then although the
 argument is required to have the shape of a @(see rune), it need not be a
 rune.  For example, if there is no rewrite rule named @('left-to-right'), then
 @('(active-runep '(:rewrite left-to-right))') will simply return @('nil').  If
 instead you'd like this call to cause an error, use a non-nil optional
 argument or, equivalently, use @(tsee active-or-non-runep).</p>")

(defxdoc active-or-non-runep
  :parents (theories)
  :short "Require a @(see rune) to exist, and check that it is @(see enable)d"
  :long "<p>This variant of @('active-runep') causes an error if its argument
  is not a @(see rune).  See @(see active-runep).</p>")

(defxdoc add-binop
  :parents (macros)
  :short "Associate a function name with a macro name"
  :long "<p>The form @('(add-binop macro macro-fn)') is an abbreviation for the
 form @('(add-macro-fn macro macro-fn t)').  See @(see add-macro-fn).</p>")

(defxdoc add-custom-keyword-hint
  :parents (events)
  :short "Add a new custom keyword hint"
  :long "@({
  Examples:
  (add-custom-keyword-hint :my-hint (my-hint-fn val ...))

  (add-custom-keyword-hint :my-hint
                           (my-hint-fn val ...)
                           :checker (my-hint-checker-fn val ...))
 })

 @({
  General Form:
  (add-custom-keyword-hint :key term1 :checker term2)
 })

 <p>where @(':key') is a @(tsee keywordp) not among the primitive keyword hints
 listed in @('*hint-keywords*'), the @(':checker') argument is optional, and
 @('term1') and (if supplied) @('term2') are terms with certain free-variable
 and signature restrictions described below.  Henceforth, @(':key') is treated
 as a custom keyword hint, e.g., the user can employ @(':key') in hints to
 @(tsee defthm), such as:</p>

 @({
  (defthm name ...
    :hints ((\"Subgoal *1/1'\" ... :key val ...))).
 })

 <p>Custom keyword hints are complicated.  To use them you must understand
 @(tsee state), multiple values (e.g., @(tsee mv) and @(tsee mv-let)), ACL2's
 notion of error triples (see @(see programming-with-state)), how to generate
 ``soft'' errors with @(tsee er), how to use @(tsee fmt)-strings to control
 output, how to use computed hints (see @(see computed-hints)) and some aspects
 of ACL2's internal event processing.  Furthermore, it is possible to implement
 a custom keyword hint that can make an event non-reproducible!  So we
 recommend that these hints be developed by ACL2 experts.  Basically the custom
 keyword feature allows the implementors and other experts to extend the hint
 facility without modifying the ACL2 sources.</p>

 <p>@('Term1') is called the ``generator'' term and @('term2') is called the
 ``checker'' term of the custom keyword hint @(':key').  Together they specify
 the semantics of the new custom keyword hint @(':key').  Roughly speaking,
 when a custom keyword hint is supplied by the user, as in</p>

 @({
  (defthm name ...
    :hints ((\"Subgoal *1/1'\" ... :my-hint val ...))).
 })

 <p>the checker term is evaluated on @('val') to check that @('val') is of the
 expected shape.  Provided @('val') passes the check, the generator term is
 used to compute a standard hint.  Like computed hints, the generator of a
 custom keyword hint is allowed to inspect the actual clause on which it is
 being fired.  Indeed, it is allowed to inspect the entire list of hints
 (standard and custom) supplied for that clause.  Thus, in the most general
 case, a custom keyword hint is just a very special kind of computed hint.</p>

 <p>The generator, @('term1'), must have no free variables other than:</p>

 @({
  (val keyword-alist
   id clause world stable-under-simplificationp
   hist pspv ctx state).
 })

 <p>Moreover, either @('term1') must evaluate to a single non-@(see stobj)
 value, or else it must be single-threaded in @('state') and have the standard
 @(see error-triple) output signature, @('(mv * * state)').</p>

 <p>The restrictions on the checker, @('term2'), are that it be single-threaded
 in @('state'), have the standard @(see error-triple) output signature, @('(mv * *
 state)'), and have no free variables other than:</p>

 @({
  (val world ctx state).
 })

 <p>For examples, see the community books directory @('books/hints/'), in
 particular @('basic-tests.lisp').</p>

 <p>To delete a previously added custom keyword hint, see @(see
 remove-custom-keyword-hint).</p>

 <p>The community book @('hints/merge-hint.lisp') can be useful in writing
 custom keyword hints.  See the examples near the of the file.</p>

 <p>Note: This is an event!  It does not print the usual event @(see summary)
 but nevertheless changes the ACL2 logical @(see world) and is so
 recorded.</p>")

(defxdoc add-default-hints
  :parents (default-hints)
  :short "Add to the default hints"
  :long "@({
  Examples:
  (add-default-hints '((computed-hint-1 clause)
                       (computed-hint-2 clause
                                        stable-under-simplificationp)))
  (add-default-hints '((computed-hint-3 id clause world))
                     :at-end t)
 })

 <p>Note: This is an event!  It does not print the usual event @(see summary)
 but nevertheless changes the ACL2 logical @(see world) and is so recorded.  It
 is @(tsee local) to the book or @(tsee encapsulate) form in which it
 occurs (see @(see add-default-hints!) for a corresponding non-@(tsee local)
 event).</p>

 @({
  General Forms:
  (add-default-hints lst)
  (add-default-hints lst :at-end flg)
 })

 <p>where @('lst') is a list.  Generally speaking, the elements of @('lst')
 should be suitable for use as @(tsee computed-hints).</p>

 <p>This event is completely analogous to @(tsee set-default-hints), the
 difference being that @('add-default-hints') appends the indicated hints to
 the front of the list of default hints, so that they are tried first &mdash;
 or, if @('flg') is supplied and evaluates to other than @('nil'), at the end
 of the list, so that they are tried last &mdash; rather than <b>replacing</b>
 the default hints with the indicated hints.  Each new hint is thus considered
 after each existing hints when both are applied to the same goal.  Also See
 @(see set-default-hints), see @(see remove-default-hints), and see @(see
 default-hints).</p>

 <p>Finally, note that the effects of @('set-default-hints'), @(tsee
 add-default-hints), and @(tsee remove-default-hints) are @(tsee local) to the
 book in which they appear.  Thus, users who include a book with such forms
 will not have their default hints affected by such forms.  In order to export
 the effect of setting the default hints, use @(tsee set-default-hints!),
 @(tsee add-default-hints!), or @(tsee remove-default-hints!).</p>

 <p>For a related feature, which however is only for advanced system builders,
 see @(see override-hints).</p>")

(defxdoc add-default-hints!
  :parents (default-hints)
  :short "Add to the default hints non-@(tsee local)ly"
  :long "<p>Please see @(see add-default-hints), which is the same as
 @('add-default-hints!')  except that the latter is not @(tsee local) to the
 @(tsee encapsulate) or the book in which it occurs.  Probably @(see
 add-default-hints) is to be preferred unless you have a good reason for
 wanting to export the effect of this event outside the enclosing @(tsee
 encapsulate) or book.</p>")

(defxdoc add-dive-into-macro
  :parents (dive-into-macros-table)
  :short "Associate @(see proof-builder) diving function with macro name"
  :long "@({
  Examples:
  (add-dive-into-macro cat expand-address-cat)
 })

 <p>This feature is used so that the interactive @(see proof-builder)'s @('DV')
 command and numeric diving commands (e.g., @('3')) will dive properly into
 subterms.  Please see @(see dive-into-macros-table).</p>")

(defxdoc add-include-book-dir
  :parents (books-reference)
  :short "Link keyword for @(':dir') argument of @(tsee ld) and @(tsee
include-book)"
  :long "@({
 Example Forms:

 ; For (include-book \"foo\" :dir :smith), prepend \"/u/smith/\" to \"foo\".
 (add-include-book-dir :smith \"/u/smith/\")

 ; For (include-book \"bar\" :dir :util), prepend absolute directory pathname
 ; corresponding to the relative pathname, \"utilities/\".
 (add-include-book-dir :util \"utilities\")
 })

 <p>Note: This is an event!  It does not print the usual event @(see summary)
 but nevertheless changes the ACL2 logical @(see world) and is so recorded.  It
 is @(tsee local) to the book or @(tsee encapsulate) form in which it occurs.
 See @(tsee add-include-book-dir!) for a corresponding non-@(tsee local)
 event.</p>

 @({
  General Form:
  (add-include-book-dir kwd dir)
 })

 <p>where @('kwd') is a @(tsee keywordp) and @('dir') is a relative or absolute
 @(see pathname) for a directory, optionally using the syntax @('(:system
 . filename)') described in @(see full-book-name).  If the final '@('/')' is
 missing for the resulting directory, ACL2 will add it for you.  The effect of
 this event is to modify the meaning of the @(':dir') keyword argument of
 @(tsee include-book) or @(tsee ld) as indicated by the examples above, that
 is, by associating the indicated directory with the indicated keyword for
 purposes of the @(':dir') argument.  By the ``indicated directory'' we mean,
 in the case that the pathname is a relative pathname, the directory relative
 to the current connected book directory; see @(see cbd).  See @(see
 delete-include-book-dir) for how to undo this effect.</p>

 <p>For a keyword already associated with a directory string by a previous
 invocation of @('add-include-book-dir') or @(tsee add-include-book-dir!), it
 is illegal to associate a different directory string until removing the
 existing association; see @(see delete-include-book-dir) (and see @(see
 delete-include-book-dir!) if the existing association was made by @(tsee
 add-include-book-dir!).  If however the new directory string is identical with
 the existing one, which was already assigned by @('add-include-book-dir'),
 then the new call of @('add-include-book-dir') will be redundant (see @(see
 redundant-events)).</p>

 <p>The keyword @(':system') can never be redefined.  It will always point to
 the absolute pathname of the system books directory, which by default is
 immediately under the directory where the ACL2 executable was originally built
 (see @(see include-book), in particular the discussion there of ``books
 directory'').</p>

 <p>This macro generates a @(tsee table) event that updates the table
 @('include-book-dir!-table'), which associates keywords with absolute
 pathnames.  However, as with @(tsee add-include-book-dir), direct table
 updates are disallowed; you must use @('add-include-book-dir!') to add to the
 table and @(tsee delete-include-book-dir!) to remove from the table.</p>

 <p>It is illegal to call @('add-include-book-dir!') in a @(tsee local)
 context.  (If you are tempted to do that, consider using @(tsee
 add-include-book-dir) instead.)  To understand this restriction, imagine a
 book that contains the following sequence of @(see events).</p>

 @({
 (add-include-book-dir! :my-dir \"path/to/BAD/dir\")
 (local (delete-include-book-dir! :my-dir))
 (local (add-include-book-dir! :my-dir \"path/to/GOOD/dir\"))
 (include-book \"foo\" :dir :my-dir)
 (defthm f-def
   (equal (f x) x))
 })

 <p>During the first (proof) pass of @(tsee certify-book), the book
 @('path/to/GOOD/dir/foo.lisp') will be included.  But on the second pass, the
 book @('path/to/BAD/dir/foo.lisp') will be included.  Now imagine that the
 ``good'' version contains the event @('(defun f (x) x)') but the ``bad''
 version instead contains the event @('(defun f (x) (not x))').  Then we can
 easily prove @('nil') from the theorem @('f-def')!  Although it is likely that
 @(see book-hash) values could catch this error at @(tsee include-book) time,
 we prefer not to rely on these for soundness.</p>")

(defxdoc add-include-book-dir!
  :parents (books-reference)
  :short "Non-@(tsee local)ly link keyword for @(':dir') argument of @(tsee ld)
and @(tsee include-book)"
  :long "<p>Please see @(see add-include-book-dir), which has completely
 analogous syntax and semantics, except that @('add-include-book-dir!') is not
 @(tsee local) to the @(tsee encapsulate) or the book in which it occurs.
 Probably @(tsee add-include-book-dir) is to be preferred unless you have a
 good reason for wanting to export the effect of this event outside the
 enclosing @(tsee encapsulate) or book.</p>

 <p>Note: This is an event!  It does not print the usual event @(see summary)
 but nevertheless changes the ACL2 logical @(see world) and is so recorded.</p>

 <p>This macro is essentially a @(tsee table) event that updates the table
 @('include-book-dir!-table'), which associates keywords with absolute
 pathnames.  However, as with @(tsee add-include-book-dir), direct table
 updates are disallowed; you must use @('add-include-book-dir!') to add to the
 table and @(tsee delete-include-book-dir!) to remove from the table.</p>

 <p>It is illegal to call @('add-include-book-dir!') in a @(tsee local)
 context.  (If you are tempted to do that, consider using @(tsee
 add-include-book-dir) instead.)  To understand this restriction, imagine a
 book that contains the following sequence of @(see events).</p>

 @({
 (add-include-book-dir! :my-dir \"path/to/BAD/dir\")
 (local (delete-include-book-dir! :my-dir))
 (local (add-include-book-dir! :my-dir \"path/to/GOOD/dir\"))
 (include-book \"foo\" :dir :my-dir)
 (defthm f-def
   (equal (f x) x))
 })

 <p>During the first (proof) pass of @(tsee certify-book), the book
 @('path/to/GOOD/dir/foo.lisp') will be included.  But on the second pass, the
 book @('path/to/BAD/dir/foo.lisp') will be included.  Now imagine that the
 ``good'' version contains the event @('(defun f (x) x)') but the ``bad''
 version instead contains the event @('(defun f (x) (not x))').  Then we can
 easily prove @('nil') from the theorem @('f-def')!  Although it is likely that
 @(see book-hash) values could catch this error at @(tsee include-book) time,
 we prefer not to rely on these for soundness.</p>")

(defxdoc add-invisible-fns
  :parents (loop-stopper)
  :short "Make some unary functions invisible to the @(see loop-stopper)
 algorithm"
  :long "@({
  Examples:
  (add-invisible-fns binary-+ unary-- foo)
  (add-invisible-fns + unary-- foo)
 })

 <p>Each of the @(see events) above makes unary functions @(tsee unary--) and
 @('foo') ``invisible'' for the purposes of applying permutative @(':')@(tsee
 rewrite) rules to @(tsee binary-+) trees.  Thus, @('arg') and @('(unary--
 arg)') will be given the same weight and will be permuted so as to be
 adjacent.</p>

 @({
  General Form:
  (add-invisible-fns top-fn unary-fn1 ... unary-fnk)
 })

 <p>where @('top-fn') is a function symbol and the @('unary-fni') are unary
 function symbols, or more generally, these are all macro aliases for such
 function symbols (see @(see macro-aliases-table)).</p>

 <p>For more information see @(see invisible-fns-table).  Also see @(see
 set-invisible-fns-table), which explains how to set the entire table in a
 single event, and see @(see remove-invisible-fns).</p>")

(defxdoc add-macro-alias
  :parents (macros)
  :short "Associate a function name with a macro name"
  :long "@({
  Example:
  (add-macro-alias append binary-append)
 })

 <p>This example associates the function symbol @(tsee binary-append) with the
 macro name @(tsee append).  As a result, the name @(tsee append) may be used
 as a runic designator (see @(see theories)) by the various theory functions.
 See @(see macro-aliases-table) for more details.  Also see @(see add-macro-fn)
 for an extension of this utility that also affects printing.</p>

 @({
  General Form:
  (add-macro-alias macro-name function-name)
 })

 <p>This is a convenient way to add an entry to @(tsee macro-aliases-table).
 See @(see macro-aliases-table) and also see @(see remove-macro-alias).</p>")

(defxdoc add-macro-fn
  :parents (macros)
  :short "Associate a function name with a macro name"
  :long "@({
  Examples:
  (add-macro-fn append binary-append)
  (add-macro-fn append binary-append t)
 })

 <p>These examples each associate the function symbol @(tsee binary-append)
 with the macro name @(tsee append).  As a result, theory functions will
 understand that @('append') refers to @('binary-append') &mdash; see @(see
 add-macro-alias) &mdash; and moreover, proof output will be printed using
 @('append') rather than @('binary-append').  In the first case, @('(append x
 (append y z))') is printed rather than @('(append x y z)').  In the second
 case, right-associated arguments are printed flat: @('(append x y z)').  Such
 right-association is considered only for binary function symbols; otherwise
 the optional third argument is ignored.</p>

 @({
  General Forms:
  (add-macro-fn macro-name function-name)
  (add-macro-fn macro-name function-name nil) ; same as above
  (add-macro-fn macro-name function-name t)
 })

 <p>This is a convenient way to add an entry to @(tsee macro-aliases-table) and
 at the same time extend the @(tsee untrans-table).  As suggested by the
 example above, calls of a function in this table will be printed as
 corresponding calls of macros, with right-associated arguments printed flat in
 the case of a binary function symbol if the optional third argument is t.  In
 that case, for a binary function symbol @('fn') associated with macro name
 @('mac'), then a call @('(fn arg1 (fn arg2 (... (fn argk arg))))') will be
 displayed to the user as though the ``term'' were @('(mac arg1 arg2 ... argk
 arg)').  For a call @('(f a1 ... ak)') of a function symbol that is not
 binary, or the optional argument is not supplied as @('t'), then the effect is
 simply to replace @('f') by the corresponding macro symbol.  See @(see
 add-macro-alias), which is invoked on the first two arguments.  Also see @(see
 remove-macro-alias), see @(see untrans-table), and see @(see
 remove-macro-fn).</p>")

(defxdoc add-match-free-override
  :parents (free-variables)
  :short "Set @(':match-free') value to @(':once') or @(':all') in existing
  rules"
  :long "@({
  Example Forms:
  (add-match-free-override :once t)
      ; Try only the first binding of free variables when relieving hypotheses
      ; of any rule of class :rewrite, :linear, or :forward-chaining.
  (add-match-free-override :all (:rewrite foo) (:rewrite bar))
      ; For rewrite rules foo and bar, try all bindings of free variables when
      ; relieving hypotheses.
  (add-match-free-override :clear)
      ; Restore :match-free to what was originally stored for each rule (either
      ; :all or :once).
 })

 <p>As described elsewhere (see @(see free-variables)), a @(see rewrite), @(see
 linear), or @(see forward-chaining) rule may have free variables in its
 hypotheses, and ACL2 can be directed either to try all bindings
 (``@(':all')'') or just the first (``@(':once')'') when relieving a
 hypothesis, as a basis for relieving subsequent hypotheses.  This direction is
 generally provided by specifying either @(':match-free :once') or
 @(':match-free :all') in the @(':')@(tsee rule-classes) of the rule, or by
 using the most recent @(tsee set-match-free-default) event.  Also see @(see
 rule-classes).</p>

 <p>However, if a proof is going slowly, you may want to modify the behavior of
 some such rules so that they use only the first match for free variables in a
 hypothesis when relieving subsequent hypotheses, rather than backtracking and
 trying additional matches as necessary.  (But note:
 @('add-match-free-override') is not relevant for @(see type-prescription)
 rules.)  The event @('(add-match-free-override :once t)') has that effect.  Or
 at the other extreme, perhaps you want to specify all rules as @(':all') rules
 except for a some specific exceptions.  Then you can execute
 @('(add-match-free-override :all t)') followed by, say,
 @('(add-match-free-override :once (:rewrite foo) (:linear bar))').</p>

 @({
  General Forms:
  (add-match-free-override :clear)
  (add-match-free-override flg t)
  (add-match-free-override flg rune1 rune2 ... runek)
 })

 <p>where @('flg') is @(':once') or @(':all') and the @('runei') are @(tsee
 rune)s.  If @(':clear') is specified then all rules will have the
 @(':all')/@(':once') behavior from when they were first stored.  The second
 general form causes all @(see rewrite) @(see linear), and @(see
 forward-chaining) rules to have the behavior specified by @('flg') (@(':all')
 or @(':once')).  Finally, the last of these, where runes are specified, is
 additive in the sense that only the indicated rules are affected; all others
 keep the behavior they had just before this event was executed (possible
 because of earlier @('add-match-free-override') events).</p>

 <p>At the conclusion of this event, ACL2 prints out the list of all
 @(':')@(tsee linear), @(':')@(tsee rewrite), and @(':')@(tsee
 forward-chaining) runes whose rules contain free variables in hypotheses that
 are to be bound @(':once'), except that if there are no overrides (value
 @(':clear') was used), then @(':clear') is printed.</p>

 <p>This event only affects rules that exist at the time it is executed.
 Future rules are not affected by the override.</p>

 <p>Note: This is an event!  It does not print the usual event @(see summary)
 but nevertheless changes the ACL2 logical @(see world) and is so recorded.  It
 uses the @(tsee acl2-defaults-table), and hence its effect is @(tsee local) to
 the book or @(tsee encapsulate) form in which it occurs.</p>

 <p><i>Remarks</i></p>

 <p>Lists of the @(':')@(tsee rewrite), @(':')@(tsee linear), and @(':')@(tsee
 forward-chaining) @(see rune)s whose behavior was originally @(':once') or
 @(':all') are returned by the following forms, respectively.</p>

 @({
  (free-var-runes :once (w state))
  (free-var-runes :all  (w state))
 })

 <p>The form</p>

 @({
  (match-free-override (w state))
 })

 <p>evaluates to a pair, whose @(tsee car) is a number used by ACL2 to
 determine whether a @(see rune) is sufficiently old to be affected by the
 override, and whose @(tsee cdr) is the list of @(see rune)s whose behavior is
 specified as @(':once') by @('add-match-free-override'); except, if no runes
 have been overridden, then the keyword @(':clear') is returned.</p>")

(defxdoc add-nth-alias
  :parents (nth-aliases-table)
  :short "Associate one symbol with another for printing of @(tsee nth)/@(tsee update-nth) terms"
  :long "@({
  Example:
  (add-nth-alias st0 st)
 })

 <p>This example associates the symbol @('st0') with the symbol @('st') for
 purposes of printing certain terms of the form @('(nth n st0)') and
 @('(update-nth n val st0)').</p>

 @({
  General Form:
  (add-nth-alias alias-name name)
 })

 <p>This is a convenient way to add an entry to @(tsee nth-aliases-table).  See
 @(see nth-aliases-table) and also see @(see remove-nth-alias).</p>")

(defxdoc add-override-hints
  :parents (override-hints)
  :short "Add to the @(see override-hints)"
  :long "<p>See @(see override-hints) for a discussion of override-hints.  Here
 we describe how to extend the list of override-hints.  Note that the effects
 of @('add-override-hints') @(see events) are @(see local) to the @(see books)
 or @('encapsulate') @(see events) in which they reside; see @(see
 add-override-hints!) to avoid that restriction.  Also see @(see
 set-override-hints) to set a new list of override-hints to it, ignoring the
 present list rather than adding to it.</p>

 @({
  General Forms:
  (add-override-hints form)
  (add-override-hints form :at-end t)
  (add-override-hints form :at-end nil) ; default for :at-end
 })

 <p>where @('form') evaluates to a list of computed hint forms.  The effect of
 this event is to extend the current list of @(see override-hints) by appending
 the result of that evaluation.  The default is to append the evaluation result
 to the front of the current list of override-hints, but if @(':at-end t') is
 specified, then the evaluation result is appended to the end of the current
 list.</p>")

(defxdoc add-override-hints!
  :parents (override-hints)
  :short "Add non-@(see local)ly to the @(see override-hints)"
  :long "<p>@('Add-override-hints!') is the same as @(tsee add-override-hints),
 except that the former is not @(see local) to @(see books) or @(tsee
 encapsulate) @(see events) in which it occurs.  See @(see add-override-hints);
 also see @(see set-override-hints).</p>")

(defxdoc add-raw-arity
  :parents (set-raw-mode)
  :short "Add arity information for raw mode"
  :long "<p>Technical note: This macro is a no-op, and is not necessary, when
 ACL2 is built with #-acl2-mv-as-values.</p>

 <p>Users of raw mode (see @(see set-raw-mode)) can use arbitrary raw Lisp
 functions that are not known inside the usual ACL2 loop.  In such cases, ACL2
 may not know how to display a multiple value returned by ACL2's @(tsee mv)
 macro.  The following example should make this clear.</p>

 @({
  ACL2 P>(defun foo (x y) (mv y x))
  FOO
  ACL2 P>(foo 3 4)

  Note: Unable to compute number of values returned by this evaluation
  because function FOO is not known in the ACL2 logical world.  Presumably
  it was defined in raw Lisp or in raw mode.  Returning the first (perhaps
  only) value for calls of FOO.
  4
  ACL2 P>(add-raw-arity foo 2)
   RAW-ARITY-ALIST
  ACL2 P>(foo 3 4)
  (4 3)
  ACL2 P>
 })

 <p>The first argument of @('add-raw-arity') should be a symbol, representing
 the name of a function, macro, or special form, and the second argument should
 either be a non-negative integer (denoting the number of values returned by
 ACL2) or else the symbol @(':LAST'), meaning that the number of values
 returned by the call is the number of values returned by the last
 argument.</p>

 <p>The current arity assignments can be seen by evaluating @('(@
 raw-arity-alist)').  See @(see remove-raw-arity) for how to undo a call of
 @('add-raw-arity').</p>")

(defxdoc add-to-set
  :parents (lists symbols acl2-built-ins)
  :short "Add a symbol to a list"
  :long "@({
  General Forms:
  (add-to-set x lst)
  (add-to-set x lst :test 'eql)   ; same as above (eql as equality test)
  (add-to-set x lst :test 'eq)    ; same, but eq is equality test
  (add-to-set x lst :test 'equal) ; same, but equal is equality test
 })

 <p>For a symbol @('x') and an object @('lst'), @('(add-to-set-eq x lst)') is
 the result of @(tsee cons)ing @('x') on to the front of @('lst'), unless
 @('x') is already a @(tsee member) of @('lst'), in which case the result is
 @('lst'). The optional keyword, @(':TEST'), has no effect logically, but
 provides the test (default @(tsee eql)) used for comparing @('x') with
 successive elements of @('lst').</p>

 <p>The @(see guard) for a call of @('add-to-set') depends on the test.  In all
 cases, the second argument must satisfy @(tsee true-listp).  If the test is
 @(tsee eql), then either the first argument must be suitable for @(tsee eql)
 (see @(see eqlablep)) or the second argument must satisfy @(tsee
 eqlable-listp).  If the test is @(tsee eq), then either the first argument
 must be a symbol or the second argument must satisfy @(tsee symbol-listp).</p>

 <p>See @(see equality-variants) for a discussion of the relation between
 @('add-to-set') and its variants:</p>

 <blockquote><p>@('(add-to-set-eq x lst)') is equivalent to @('(add-to-set x
 lst :test 'eq)');</p>

 <p>@('(add-to-set-equal x lst)') is equivalent to @('(add-to-set x lst :test
 'equal)').</p></blockquote>

 <p>In particular, reasoning about any of these primitives reduces to reasoning
 about the function @('add-to-set-equal').</p>")

(defxdoc advanced-features
  :parents (acl2-tutorial programming)
  :short "Some advanced features of ACL2"
  :long "<p>Maybe you've been using ACL2 for awhile, and you wonder if there
 are lesser-known features that you might find useful.  Then this topic is for
 you.  We present below a ``laundry list'' of some such features, with brief
 descriptions and links to @(see documentation) topics.</p>

 <p>Although the list below is long, it is not intended to be complete, and
 indeed some topics have been deliberately excluded.  Some have fallen out of
 use, perhaps for good reason, such as @(see OBDD).  Others are already likely
 to be discovered when needed, such as @(tsee GETENV$) and perhaps @(tsee
 DOUBLE-REWRITE).  Some topics are referenced by documentation for others in
 the list, such as @(tsee MBT), which is referenced by @(tsee MBE).  Some
 utilities such as @(tsee PSTACK) and @(tsee VERBOSE-PSTACK) seem too low-level
 to be worthy of inclusion below.</p>

 <p>For an extensive introduction to using the prover, which may include some
 aspects new to you, see @(see INTRODUCTION-TO-THE-THEOREM-PROVER).  A shorter
 topic contains highlights for efficient prover usage: see @(see TIPS).  Also
 see @(see ACL2-SEDAN) for an extension of ACL2 (written by others), ACL2s,
 that includes an Eclipse-based interface, more powerful and automatic
 termination reasoning, and other features.</p>

 <p>We now move on to the list.</p>

 <h3>Top-level commands and utilities:</h3>

 <ul>
 <li>See @(see A!) and see @(see P!) to abort or pop.</li>

 <li>See @(see ACL2-CUSTOMIZATION) for initial commands to run at startup.</li>

 <li>See @(see KEYWORD-COMMANDS) for how keyword commands are processed.</li>

 <li>See @(see LD) for many ways to control the top-level loop.</li>

 <li>See @(see COMPILATION) for a discussion of @('set-compiler-enabled') and
 other compiler-related utilities.</li>

 <li>For useful reader macros `@('#!')', `@('#.')', and `@('#u')', see @(see
 SHARP-BANG-READER), see @(see SHARP-DOT-READER), and see @(see
 SHARP-U-READER).</li>

 <li>To save and use an ACL2 executable, see @(see ACL2-AS-STANDALONE-PROGRAM)
 and see @(see SAVE-EXEC).</li>

 <li>For utilities related to timing, see @(see TIME$), see @(see
 WITH-PROVER-TIME-LIMIT), see @(see WITH-PROVER-STEP-LIMIT), and see @(see
 SET-PROVER-STEP-LIMIT).</li>

 <li>To query and manage the database, see @(see HISTORY) (which discusses
 many useful utilities, such as @(':')@(tsee PBT) and @(':')@(tsee PL)), and
 see @(see DEAD-EVENTS).</li>

 <li>See @(see ADD-INCLUDE-BOOK-DIR) for linking keyword for @(':dir')
 argument of @(tsee LD) and @(tsee INCLUDE-BOOK).</li>

 <li>See @(see REBUILD) for a fast way to load a file without waiting for
 proofs.</li>

 <li>For parallel certification, see @(see BOOKS-CERTIFICATION) for use of the
 @('-j') option of `make'; also see @(see
 PROVISIONAL-CERTIFICATION).</li>

 </ul>

 <h3>Some relatively less common events</h3>

 <ul>

 <li>See @(see RESET-PREHISTORY) to reset the prehistory.</li>

 <li>See @(see ASSERT-EVENT) to assert that a given form returns a non-nil
 value.</li>

 <li>See @(see DEFATTACH) to execute constrained functions using corresponding
 attached functions.</li>

 <li>See @(see DEFUN-SK) to define a function whose body has an outermost
 quantifier.</li>

 <li>See @(see DEFCHOOSE) to define a Skolem (witnessing) function.</li>

 <li>See @(see SET-VERIFY-GUARDS-EAGERNESS) to specify when @(see guard)
 verification is tried by default.</li>

 </ul>


 <h3>Output and its control (see @(see IO) for additional information)</h3>

 <ul>

 <li>See @(see WITH-OUTPUT) to suppress or turn on specified
 output for an event.</li>

 <li>See @(see EVISC-TABLE) for support for abbreviated output.</li>

 <li>See @(see NTH-ALIASES-TABLE) for a table used to associate names for
 @(tsee NTH)/@(tsee UPDATE-NTH) printing.</li>

 <li>See @(see OUTPUT-TO-FILE) to redirect output to a file.</li>

 <li>See @(see PRINT-CONTROL) to control ACL2 printing.</li>

 <li>See @(see SET-EVISC-TUPLE) to control suppression of details when
 printing.</li>

 <li>See @(see SET-INHIBIT-OUTPUT-LST) to control output by type.</li>

 <li>See @(see SET-IPRINT) to allow abbreviated output to be read back in.</li>

 <li>See @(see SET-PRINT-BASE-RADIX) (also @(see SET-PRINT-BASE) and @(see
 SET-PRINT-RADIX)) to control the radix in which numbers are printed.</li>

 <li>See @(see SET-PRINT-CASE) to control whether symbols are printed in upper
 case or in lower case.</li>

 </ul>

 <h3>On proving termination for definitions:</h3>

 <ul>

 <li>See @(see ORDINALS) for a discussion of ordinals in ACL2.</li>

 <li>See @(see RULER-EXTENDERS) for a control on ACL2's termination and
 induction analyses.</li>

 <li>See @(see SET-WELL-FOUNDED-RELATION) to set the default well-founded
 relation for termination analysis.</li>

 <li>See @(see ACL2-SEDAN) for a related tool that provides extra automation
 for termination proofs.</li>

 </ul>

 <h3>Proof debugging and output control:</h3>

 <ul>

 <li>See @(see ACCUMULATED-PERSISTENCE) to get statistics on which
 runes are being tried.</li>

 <li>See @(see ADD-MACRO-FN) and see @(see ADD-MACRO-ALIAS) to associate a
 function name with a macro name.</li>

 <li>See @(see BREAK-REWRITE) for how to monitor rewrite rules.</li>

 <li>See @(see DMR) for dynamic monitoring of rewriting and other prover
 activity.</li>

 <li>See @(see FORWARD-CHAINING-REPORTS) to see reports about the forward
 chaining process.</li>

 <li>See @(see GUARD-DEBUG) and @(see MEASURE-DEBUG) to generate markers to
 indicate sources of @(see guard) and termination proof obligations.</li>

 <li>See @(see PROOF-BUILDER) for support for low-level interaction.</li>

 <li>See @(see REDO-FLAT) for redo on failure of a @(tsee PROGN), @(tsee
 ENCAPSULATE), or @(tsee CERTIFY-BOOK).</li>

 <li>See @(see SET-GAG-MODE) and see @(see PSO) to abbreviate or restore proof
 output.</li>

 <li>See @(see SET-INHIBIT-OUTPUT-LST), see @(see SET-INHIBIT-WARNINGS), and
 see @(see SET-INHIBITED-SUMMARY-TYPES) to inhibit various types of output.</li>

 <li>See @(see SET-RAW-PROOF-FORMAT) to make proof output display lists of
 @(see rune)s.</li>

 <li>See @(see SET-RAW-WARNING-FORMAT) to make some warnings display in a
 ``raw'' s-expression format.</li>

 <li>See @(see SKIP-PROOFS) to skip proofs for a given form.</li>

 </ul>

 <h3>Program debugging:</h3>

 <ul>

 <li>See @(see BREAK$) to cause an immediate Lisp break.</li>

 <li>See @(see BREAK-ON-ERROR) to break when encountering a hard or soft error
 caused by ACL2.</li>

 <li>See @(see DISASSEMBLE$) to disassemble a function.</li>

 <li>See @(see PRINT-GV) to print a form whose evaluation caused a guard
 violation.</li>

 <li>See @(see PROFILE) to turn on profiling for one function.</li>

 <li>See @(see TRACE$) and see @(see OPEN-TRACE-FILE) to @(see trace) function
 evaluations, possibly sending trace output to a file.</li>

 <li>See @(see WET) to evaluate a form and print a subsequent error
 trace.</li>

 </ul>

 <h3>Programming and evaluation idioms, support, utilities</h3>

 <p>(also see @(see PROGRAMMING) for more utilities, e.g., @(tsee RANDOM$)).</p>

 <ul>

 <li>See @(see ARRAYS) and See @(see DEFSTOBJ) for introductions
 to ACL2 arrays and single-threaded objects (stobjs), respectively, each of
 which provides efficient destructive operations in an applicative setting.
 Also see @(see WITH-LOCAL-STOBJ) for a way to create local stobjs.</li>

 <li>See @(see ASSERT$) to cause a hard error if the given test is false.</li>

 <li>See @(see CANONICAL-PATHNAME) to obtain the true absolute filename, with
 soft links resolved.</li>

 <li>See @(see CASE-MATCH) for a utility providing pattern matching and
 destructuring.</li>

 <li>See @(see DEFPUN) to define a tail-recursive function symbol.</li>

 <li>See @(see EC-CALL) to execute a call in the ACL2 logic instead of raw
 Lisp.</li>

 <li>See @(see ER) to print an error message and ``cause an error''.</li>

 <li>See @(see FLET) to provide local binding of function symbols.</li>

 <li>See @(see GC$) to invoke the garbage collector.</li>

 <li>See @(see MBE) to attach code for execution.</li>

 <li>See @(see MV-LIST) to convert a multiple-valued result to a single-valued
 list.</li>

 <li>See @(see MV?) to return one or more values.</li>

 <li>For non-executable code, see @(see DEFUN-NX) and see @(see NON-EXEC).</li>

 <li>See @(see PROG2$) and see @(see PROGN$) to execute two or more forms and
 return the value of the last one.</li>

 <li>See @(see PROGRAMMING-WITH-STATE) for how to program using the von
 Neumannesque ACL2 @(see state) object.</li>

 <li>See @(see TOP-LEVEL) to evaluate a top-level form as a function body.</li>

 <li>See @(see WITH-GUARD-CHECKING) to suppress or enable guard-checking for a
 form.</li>

 <li>For ways to fake access to the state see @(see WORMHOLE), see @(see
 WITH-LOCAL-STATE), see @(see CW), see @(see CW!), see @(see
 PRINTING-TO-STRINGS), see @(see OBSERVATION-CW), and (dangerous!) see @(see
 WITH-LIVE-STATE).</li>

 </ul>

 <h3>Connecting with the underlying host Lisp, and doing other evil:</h3>

 <ul>

 <li>See @(see DEFTTAG) to introduce a trust tag (ttag).</li>

 <li>See @(see DEFMACRO-LAST) to define a macro that returns its last
 argument, but with side effects.</li>

 <li>See @(see PROGN!) to evaluate forms that are not necessarily @(see
 events).</li>

 <li>See @(see RETURN-LAST) to return the last argument, perhaps with side
 effects.</li>

 <li>See @(see SET-RAW-MODE) to enter or exit ``raw mode,'' a raw Lisp
 environment.</li>

 <li>See @(see SYS-CALL), @(see SYS-CALL+), and @(see SYS-CALL*) to make a
 system call to the host operating system.</li>

 </ul>

 <h3>Macros and related utilities:</h3>

 <ul>

 <li>See @(see DEFABBREV) for a convenient form of macro
 definition for simple expansions.</li>

 <li>See @(see MACRO-ARGS) for the formals list of a macro definition (see
 @(see DEFMACRO)).</li>

 <li>See @(see MAKE-EVENT) for a sort of extension of @(tsee DEFMACRO) that
 allows access to the @(see state), by evaluating (expanding) a given form and
 then evaluate the result of that expansion.</li>

 <li>See @(see TRANS), see @(see TRANS!), and see @(see TRANS1) to print the
 macroexpansion of a form.</li>

 </ul>

 <h3>Additional capabilities:</h3>

 <ul>

 <li>See @(see HONS-AND-MEMOIZATION) for a discussion of the @(see
 hons-enabled) features providing hash cons, function memoization, and
 applicative hash tables.  In particular, see @(see MEMOIZE) for efficient
 function memoization and see @(see PROFILE) for profiling.</li>

 <li>See @(see REAL) for ACL2(r), which supports the real numbers.</li>

 <li>See @(see PARALLELISM) for ACL2(p), which supports parallel evaluation
 and proof.</li>

 </ul>

 <h3>Database control and query:</h3>

 <ul>

 <li>See @(see DISABLEDP) to determine whether a given name or
 rune is disabled.</li>

 <li>For redefinition support see @(see REDEF), see @(see REDEF!), see @(see
 REDEF+), see @(see REDEF-), and see @(see REDEFINED-NAMES).</li>

 <li>See @(see TABLE) for user-managed tables.</li>

 <li>See @(see GUARD-FORMULA-UTILITIES) for how to view a guard proof
 obligation without doing the proof.</li>

 </ul>

 <h3>Prover control</h3>

 <ul>

 <li>For congruence-based reasoning see @(see DEFCONG), see @(see
 CONGRUENCE), see @(see EQUIVALENCE), see @(see DEFEQUIV), and see @(see
 DEFREFINEMENT).</li>

 <li>For meta rules and clause processors see @(see META), see @(see
 DEFEVALUATOR), see @(see CLAUSE-PROCESSOR), see @(see
 DEFINE-TRUSTED-CLAUSE-PROCESSOR) (for connecting with external tools, such as
 SAT solvers), and See @(see EXTENDED-METAFUNCTIONS) (for @(see state) and
 context-sensitive metafunctions).</li>

 <li>For theory control, see @(see THEORIES) for detailed information, but in
 particular see @(see DEFTHEORY), see @(see THEORY-FUNCTIONS), see @(see
 IN-ARITHMETIC-THEORY) (and see @(see NON-LINEAR-ARITHMETIC)), and see @(see
 THEORY-INVARIANT).</li>

 <li>See @(see HINTS) for a complete list of prover hints, including some of
 the more obscure ones such as @(':restrict'), @(':')@(tsee clause-processor),
 @(':nonlinearp'), @(':backchain-limit-rw'), @(':reorder'), and
 @(':backtrack').  Also see @(see HINTS-AND-THE-WATERFALL) for an explanation
 of how hints interact with the ACL2 proof process.  For other topics related
 to hints, see @(see OVERRIDE-HINTS), see @(see ADD-CUSTOM-KEYWORD-HINT), see
 @(see DEFAULT-HINTS), and see @(see COMPUTED-HINTS) and @(see
 USING-COMPUTED-HINTS).</li>

 <li>See @(see BIND-FREE) to bind @(see free-variables) of a @(see rewrite) or
 @(see linear) rule.</li>

 <li>See @(see CASE-SPLIT) for a utility like @(tsee FORCE) that immediately
 splits the top-level goal on the indicated hypothesis.</li>

 <li>See @(see CASE-SPLIT-LIMITATIONS) for a way to the number of cases
 produced at once</li>

 <li>See @(see DEFAULT-BACKCHAIN-LIMIT) to specify the backchain limit for a
 rule.</li>

 <li>See @(see FORCE) for an identity function used to force a hypothesis.</li>

 <li>See @(see OTF-FLG) for a way to push more than one initial subgoal for
 induction.</li>

 <li>See @(see RULE-CLASSES) to add various kinds of rules to the database,
 including more unusual sorts such as @(':')@(tsee built-in-clause) rules and
 @(':')@(tsee induction) rules.</li>

 <li>See @(see SET-BACKCHAIN-LIMIT) to set the backchain-limit used by the
 type-set and rewriting mechanisms.</li>

 <li>See @(see SET-BODY) to set an alternate definition body for @(':expand')
 @(see hints).</li>

 <li>See @(see SET-REWRITE-STACK-LIMIT) to set the @(see rewrite) stack depth
 used by the rewriter.</li>

 <li>See @(see SYNTAXP) to attach a heuristic filter on a @(':')@(tsee
 rewrite), @(':')@(tsee meta), or @(':')@(tsee linear) rule.</li>

 </ul>

 ")

(defxdoc alist-keys-subsetp
  :parents (alists acl2-built-ins)
  :short "Check that all keys of the alist belong to a given set"
  :long "<p>The call @('(alist-keys-subsetp alist keys)') returns @('t') when
 each key of the given alist belongs to the given list of keys; else it returns
 @('nil').  This is Boolean-equivalent to @('(subsetp-eq (strip-cars alist)
 keys)'), but it avoids consing up the keys of @('alist').</p>")

(defxdoc alist-to-doublets
  :parents (alists acl2-built-ins)
  :short "Convert an alist to a list of two-element lists"
  :long "<p>The call @(call alist-to-doublets) returns the result of replacing
 each pair @('(x . y)') in the given alist by the two-element list @('(x y)').
 The order is preserved, i.e., the following is a theorem.</p>

 @({
 (implies (and (natp i) (< i (len alist)))
          (equal (nth i (alist-to-doublets alist))
                 (let ((pair (nth i alist)))
                   (list (car pair) (cdr pair)))))
 })")

(defxdoc alistp
  :parents (alists acl2-built-ins)
  :short "Recognizer for association lists"
  :long "<p>@('(alistp x)') is true if and only if @('x') is a list of @(tsee
 cons) pairs.</p>

 <p>@('(alistp x)') has a @(see guard) of @('t').</p>

 @(def alistp)")

(defxdoc alists
  :parents (programming)
  :short "Operations on association lists, which bind keys to values.")

(defxdoc allocate-fixnum-range
  :parents (numbers acl2-built-ins)
  :short "Set aside fixnums in GCL"
  :long "<p>@('(Allocate-fixnum-range fixnum-lo fixnum-hi)') causes Gnu Common
 Lisp
 (GCL) to create a persistent table for the integers between @('fixnum-lo') and
 @('fixnum-hi') (both bounds inclusive). This table is referenced first when
 any integer is boxed and the existing box in the table is used if the integer
 is in bounds.  This can speed up GCL considerably by avoiding wasteful fixnum
 boxing.  Here, @('fixnum-lo') and @('fixnum-hi') should be fixnums.  On 32-bit
 machines it would be good for them to be of type @('(signed-byte 30)'), with
 @('fixnum-lo <= fixnum-hi').</p>

 <p>When this function is executed in a Lisp implementation other than GCL, it
 has no side effect.  This function always returns @('nil').</p>")

(defxdoc alpha-char-p
  :parents (characters acl2-built-ins)
  :short "Recognizer for alphabetic characters"
  :long "<p>@('(Alpha-char-p x)') is true for a standard character @('x') if
 and only if @('x') is alphabetic, i.e., one of the @(see characters)
 @('#\\a'), @('#\\b'), ..., @('#\\z'), @('#\\A'), @('#\\B'), ...,
 @('#\\Z').</p>

 <p>The @(see guard) for @('alpha-char-p') requires its argument to be a
 standard character (see @(see standard-char-p)).</p>

 <p>@('Alpha-char-p') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def alpha-char-p)")

(defxdoc alphorder
  :parents (<< acl2-built-ins)
  :short "Total order on atoms"
  :long "<p>@('Alphorder') is a non-strict total order, a ``less than or
 equal,'' on atoms.  By ``non-strict total order'' we mean a function that
 always returns @('t') or @('nil') and satisfies the following properties.</p>

 <ul>

 <li>Antisymmetry: @('XrY & YrX -> X=Y')</li>

 <li>Transitivity: @('XrY & YrZ -> XrZ')</li>

 <li>Trichotomy: @('XrY v YrX')</li>

 </ul>

 <p>Also see @(see lexorder), which extends @('alphorder') to all objects.</p>

 <p>@('(Alphorder x y)') has a guard of @('(and (atom x) (atom y))').</p>

 <p>Within a single type: rationals are compared arithmetically, complex
 rationals are compared lexicographically, characters are compared via their
 char-codes, and strings and symbols are compared with alphabetic ordering.
 Across types, rationals come before complexes, complexes come before
 characters, characters before strings, and strings before symbols.  We also
 allow for ``bad atoms,'' i.e., atoms that are not legal Lisp objects but make
 sense in the ACL2 logic; these come at the end, after symbols.</p>

 @(def alphorder)")

(defxdoc alternative-introduction
  :parents (acl2-tutorial)
  :short "Introduction to ACL2"
  :long "<p>This section contains introductory material on ACL2 including what
 ACL2 is, how to get started using the system, how to read the output, and
 other introductory topics.  It was written almost entirely by Bill Young of
 Computational Logic, Inc.</p>

 <p>You might also find CLI Technical Report 101 helpful, especially if you are
 familiar with Nqthm.  If you would like more familiarity with Nqthm, we
 suggest CLI Technical Report 100.</p>

 <p><i>OVERVIEW</i></p>

 <p>ACL2 is an automated reasoning system developed (for the first 9 years) at
 Computational Logic, Inc. and (from January, 1997) at the University of Texas
 at Austin.  It is the successor to the Nqthm (or Boyer-Moore) logic and proof
 system and its Pc-Nqthm interactive enhancement.  The acronym ACL2 actually
 stands for ``A Computational Logic for Applicative Common Lisp''.  This title
 suggests several distinct but related aspects of ACL2.</p>

 <p>We assume that readers of the ACL2 @(see documentation) have at least a
 very slight familiarity with some Lisp-like language.  We will address the
 issue of prerequisites further, in ``ABOUT THIS TUTORIAL'' below.</p>

 <p>As a <b>logic</b>, ACL2 is a formal system with rigorously defined syntax
 and semantics.  In mathematical parlance, the ACL2 logic is a first-order
 logic of total recursive functions providing mathematical induction on the
 ordinals up to epsilon-0 and two extension principles: one for recursive
 definition and one for constrained introduction of new function symbols, here
 called encapsulation.  The syntax of ACL2 is that of Common Lisp; ACL2
 specifications are ``also'' Common Lisp programs in a way that we will make
 clear later.  In less formal language, the ACL2 logic is an integrated
 collection of rules for defining (or axiomatizing) recursive functions,
 stating properties of those functions, and rigorously establishing those
 properties.  Each of these activities is mechanically supported.</p>

 <p>As a <b>specification language</b>, ACL2 supports modeling of systems of
 various kinds.  An ACL2 function can equally be used to express purely formal
 relationships among mathematical entities, to describe algorithms, or to
 capture the intended behavior of digital systems.  For digital systems, an
 ACL2 specification is a mathematical <b>model</b> that is intended to
 formalize relevant aspects of system behavior.  Just as physics allows us to
 model the behavior of continuous physical systems, ACL2 allows us to model
 digital systems, including many with physical realizations such as computer
 hardware.  As early as the 1930's Church, Kleene, Turing and others
 established that recursive functions provide an expressive formalism for
 modeling digital computation.  Digital computation should be understood in a
 broad sense, covering a wide variety of activities including almost any
 systematic or algorithmic activity, or activity that can be reasonably
 approximated in that way.  This ranges from the behavior of a digital circuit
 to the behavior of a programming language compiler to the behavior of a
 controller for a physical system (as long as the system can be adequately
 modeled discretely).  All of these have been modeled using ACL2 or its
 predecessor Nqthm.</p>

 <p>ACL2 is a <b>computational</b> logic in at least three distinct senses.
 First, the theory of recursive functions is often considered the mathematics
 of computation.  Church conjectured that any ``effective computation'' can be
 modeled as a recursive function.  Thus, ACL2 provides an expressive language
 for modeling digital systems.  Second, many ACL2 specifications are
 executable.  In fact, recursive functions written in ACL2 <b>are</b> Common
 Lisp functions that can be submitted to any compliant Common Lisp compiler and
 executed (in an environment where suitable ACL2-specific macros and functions
 are defined).  Third, ACL2 is computational in the sense that calculation is
 heavily integrated into the reasoning process.  Thus, an expression with
 explicit constant values but no free variables can be simplified by
 calculation rather than by complex logical manipulations.</p>

 <p>ACL2 is a powerful, automated <b>theorem prover</b> or proof checker.  This
 means that a competent user can utilize the ACL2 system to discover proofs of
 theorems stated in the ACL2 logic or to check previously discovered proofs.
 The basic deductive steps in an ACL2-checked proof are often quite large, due
 to the sophisticated combination of decision procedures, conditional
 rewriting, mathematical and structural induction, propositional
 simplification, and complex heuristics to orchestrate the interactions of
 these capabilities.  Unlike some automated proof systems, ACL2 does not
 produce a formal proof.  However, we believe that if ACL2 certifies the
 ``theoremhood'' of a given conjecture, then such a formal proof exists and,
 therefore, the theorem is valid.  The ultimate result of an ACL2 proof session
 is a collection of ``@(see events),'' possibly grouped into ``@(see books),''
 that can be replayed in ACL2.  Therefore, a proof can be independently
 validated by any ACL2 user.</p>

 <p>ACL2 may be used in purely automated mode in the shallow sense that
 conjectures are submitted to the prover and the user does not interact with
 the proof attempt (except possibly to stop it) until the proof succeeds or
 fails.  However, any non-trivial proof attempt is actually interactive, since
 successful proof ``@(see events)'' influence the subsequent behavior of the
 prover.  For example, proving a lemma may introduce a rule that subsequently
 is used automatically by the prover.  Thus, any realistic proof attempt, even
 in ``automatic'' mode, is really an interactive dialogue with the prover to
 craft a sequence of @(see events) building an appropriate theory and proof
 rules leading up to the proof of the desired result.  Also, ACL2 supports
 annotating a theorem with ``@(see hints)'' designed to guide the proof
 attempt.  By supplying appropriate @(see hints), the user can suggest proof
 strategies that the prover would not discover automatically.  There is a
 ``@(see proof-tree)'' facility (see @(see proof-tree)) that allows the user to
 @(see monitor) the progress and structure of a proof attempt in real-time.
 Exploring failed proof attempts is actually where heavy-duty ACL2 users spend
 most of their time.</p>

 <p>ACL2 can also be used in a more explicitly interactive mode.  The
 interactive @(see proof-builder) subsystem of ACL2 allows exploration of a
 proof on a fairly low level including expanding calls of selected function
 symbols, invoking specific @(see rewrite) rules, and selectively navigating
 around the proof.  This facility can be used to gain sufficient insight into
 the proof to construct an automatic version, or to generate a detailed
 interactive-style proof that can be replayed in batch mode.</p>

 <p>Because ACL2 is all of these things &mdash; computational logic,
 specification language, @(see programming) system, and theorem prover &mdash;
 it is more than the sum of its parts.  The careful integration of these
 diverse aspects has produced a versatile automated reasoning system suitable
 for building highly reliable digital systems.  In the remainder of this
 tutorial, we will illustrate some simple uses of this automated reasoning
 system.</p>

 <p><i>ABOUT THIS TUTORIAL</i></p>

 <p>ACL2 is a complex system with a vast array of features, bells and whistles.
 However, it is possible to perform productive work with the system using only
 a small portion of the available functionality.  The goals of this tutorial
 are to:</p>

 <blockquote>

 <p>familiarize the new user with the most basic features of and modes of
 interaction with ACL2;</p>

 <p>familiarize her with the form of output of the system; and</p>

 <p>work through a graduated series of examples.</p></blockquote>

 <p>The more knowledge the user brings to this system, the easier it will be to
 become proficient.  On one extreme: the <b>ideal</b> user of ACL2 is an expert
 Common Lisp programmer, has deep understanding of automated reasoning, and is
 intimately familiar with the earlier Nqthm system.  Such ideal users are
 unlikely to need this tutorial.  However, without some background knowledge,
 the beginning user is likely to become extremely confused and frustrated by
 this system.  We suggest that a new user of ACL2 should:</p>

 <blockquote>

 <p>(a) have a little familiarity with Lisp, including basic Lisp programming
 and prefix notation (a Lisp reference manual such as Guy Steele's ``Common
 Lisp: The Language'' is also helpful);</p>

 <p>(b) be convinced of the utility of formal modeling; and</p>

 <p>(c) be willing to gain familiarity with basic automated theorem proving
 topics such as rewriting and algebraic simplification.</p></blockquote>

 <p>We will not assume any deep familiarity with Nqthm (the so-called
 ``Boyer-Moore Theorem Prover''), though the book ``A Computational Logic
 Handbook'' by Boyer and Moore (Academic Press, 1988) is an extremely useful
 reference for many of the topics required to become a competent ACL2 user.
 We'll refer to it as ACLH below.</p>

 <p>As we said in the introduction, ACL2 has various facets.  For example, it
 can be used as a Common Lisp @(see programming) system to construct
 application programs.  In fact, the ACL2 system itself is a large Common Lisp
 program constructed almost entirely within ACL2.  Another use of ACL2 is as a
 specification and modeling tool.  That is the aspect we will concentrate on in
 the remainder of this tutorial.</p>

 <p><i>GETTING STARTED</i></p>

 <p>This section is an abridged version of what's available elsewhere; feel
 free to see @(see startup) for more details.</p>

 <p>How you start ACL2 will be system dependent, but you'll probably type
 something like ``acl2'' at your operating system prompt.  Consult your system
 administrator for details.</p>

 <p>When you start up ACL2, you'll probably find yourself inside the ACL2 @(see
 command) loop, as indicated by the following @(see prompt).</p>

 @({
    ACL2 !>
 })

 <p>If not, you should type @('(LP)').  See @(see lp), which has a lot more
 information about the ACL2 @(see command) loop.</p>

 <p>There are two ``modes'' for using ACL2, @(':')@(tsee logic) and
 @(':')@(tsee program).  When you begin ACL2, you will ordinarily be in the
 @(':')@(tsee logic) mode.  This means that any new function defined is not
 only executable but also is axiomatically defined in the ACL2 logic.  (See
 @(see defun-mode) and see @(see default-defun-mode).)  Roughly speaking,
 @(':')@(tsee program) mode is available for using ACL2 as a @(see programming)
 language without some of the logical burdens necessary for formal reasoning.
 In this tutorial we will assume that we always remain in @(':')@(tsee logic)
 mode and that our purpose is to write formal models of digital systems and to
 reason about them.</p>

 <p>Now, within the ACL2 @(see command) loop you can carry out various kinds of
 activities, including the following.  (We'll see examples later of many of
 these.)</p>

 <blockquote>

 <p>define new functions (see @(see defun));</p>

 <p>execute functions on concrete data;</p>

 <p>pose and attempt to prove conjectures about previously defined functions
 (see @(see defthm));</p>

 <p>query the ACL2 ``@(see world)'' or database (e.g., see @(see pe)); and</p>

 <p>numerous other things.</p></blockquote>

 <p>In addition, there is extensive on-line @(see documentation), of which this
 tutorial introduction is a part.</p>

 <p><i>INTERACTING WITH ACL2</i></p>

 <p>The standard means of interacting with ACL2 is to submit a sequence of
 forms for processing by the ACL2 system.  These forms are checked for
 syntactic and semantic acceptability and appropriately processed by the
 system.  These forms can be typed directly at the ACL2 @(see prompt).
 However, most successful ACL2 users prefer to do their work using the Emacs
 text editor, maintaining an Emacs ``working'' buffer in which forms are
 edited.  Those forms are then copied to the ACL2 interaction buffer, which is
 often the @('\"*shell*\"') buffer.</p>

 <p>In some cases, processing succeeds and makes some change to the ACL2
 ``logical @(see world),'' which affects the processing of subsequent forms.
 How can this processing fail?  For example, a proposed theorem will be
 rejected unless all function symbols mentioned have been previously defined.
 Also the ability of ACL2 to discover the proof of a theorem may depend on the
 user previously having proved other theorems.  Thus, the order in which forms
 are submitted to ACL2 is quite important.  Maintaining forms in an appropriate
 order in your working buffer will be helpful for re-playing the proof
 later.</p>

 <p>One of the most common @(see events) in constructing a model is introducing
 new functions.  New functions are usually introduced using the @(tsee defun)
 form; we'll encounter some exceptions later.  Proposed function definitions
 are checked to make sure that they are syntactically and semantically
 acceptable (e.g., that all mentioned functions have been previously defined)
 and, for recursive functions, that their recursive calls <b>terminate</b>.  A
 recursive function definition is guaranteed to terminate if there is some some
 ``measure'' of the arguments and a ``well-founded'' ordering such that the
 arguments to the function get smaller in each recursive call.  See @(see
 well-founded-relation-rule).</p>

 <p>For example, suppose that we need a function that will append two lists
 together.  (We already have one in the ACL2 @(tsee append) function; but
 suppose perversely that we decide to define our own.)  Suppose we submit the
 following definition (you should do so as well and study the system
 output):</p>

 @({
    (defun my-app (x y)
      (if (atom x)
          y
        (cons (car x) (my-app x y))))
 })

 <p>The system responds with the following message:</p>

 @({
    ACL2 Error in ( DEFUN MY-APP ...):  No :MEASURE was supplied with
    the definition of MY-APP.  Our heuristics for guessing one have not
    made any suggestions.  No argument of the function is tested along
    every branch and occurs as a proper subterm at the same argument
    position in every recursive call.  You must specify a :MEASURE.  See
    :DOC defun.
 })

 <p>This means that the system could not find an expression involving the
 formal parameters @('x') and @('y') that decreases under some well-founded
 order in every recursive call (there is only one such call).  It should be
 clear that there is no such measure in this case because the only recursive
 call doesn't change the arguments at all.  The definition is obviously flawed;
 if it were accepted and executed it would loop forever.  Notice that a
 definition that is rejected is not stored in the system database; there is no
 need to take any action to have it ``thrown away.''  Let's try again with the
 correct definition.  The interaction now looks like (we're also putting in the
 ACL2 @(see prompt); you don't type that):</p>

 @({
    ACL2 !>(defun my-app (x y)
             (if (atom x)
                 y
               (cons (car x) (my-app (cdr x) y))))

    The admission of MY-APP is trivial, using the relation O<
    (which is known to be well-founded on the domain recognized by
    O-P) and the measure (ACL2-COUNT X).  We observe that the
    type of MY-APP is described by the theorem
    (OR (CONSP (MY-APP X Y)) (EQUAL (MY-APP X Y) Y)).
    We used primitive type reasoning.

    Summary
    Form:  ( DEFUN MY-APP ...)
    Rules: ((:FAKE-RUNE-FOR-TYPE-SET NIL))
    Warnings:  None
    Time:  0.07 seconds (prove: 0.00, print: 0.00, other: 0.07)
    MY-APP
 })

 <p>Notice that this time the function definition was accepted.  We didn't have
 to supply a measure explicitly; the system inferred one from the form of the
 definition.  On complex functions it may be necessary to supply a measure
 explicitly.  (See @(see xargs).)</p>

 <p>The system output provides several pieces of information.</p>

 <blockquote>

 <p>The revised definition is acceptable.  The system realized that there is a
 particular measure (namely, @('(acl2-count x)')) and a well-founded relation
 (@('o<')) under which the arguments of @('my-app') get smaller in recursion.
 Actually, the theorem prover proved several theorems to admit @('my-app').
 The main one was that when @('(atom x)') is false the @('acl2-count') of
 @('(cdr x)') is less than (in the @('o<') sense) the @('acl2-count') of
 @('x').  @(tsee Acl2-count) is the most commonly used measure of the ``size``
 of an ACL2 object.  @(tsee o<) is the ordering relation on ordinals less than
 epsilon-0.  On the natural numbers it is just ordinary ``&lt;''.</p>

 <p>The observation printed about ``the type of MY-APP'' means that calls of
 the function @('my-app') will always return a value that is either a @(see
 cons) pair or is equal to the second parameter.</p>

 <p>The @(see summary) provides information about which previously introduced
 definitions and lemmas were used in this proof, about some notable things to
 watch out for (the Warnings), and about how long this event took to
 process.</p></blockquote>

 <p>Usually, it's not important to read this information.  However, it is a
 good habit to scan it briefly to see if the type information is surprising to
 you or if there are Warnings.  We'll see an example of them later.</p>

 <p>After a function is accepted, it is stored in the database and available
 for use in other function definitions or lemmas.  To see the definition of any
 function use the @(':')@(tsee pe) command (see @(see pe)).  For example,</p>

 @({
    ACL2 !>:pe my-app
     L       73:x(DEFUN MY-APP (X Y)
                        (IF (ATOM X)
                            Y (CONS (CAR X) (MY-APP (CDR X) Y))))
 })

 <p>This displays the definition along with some other relevant information.
 In this case, we know that this definition was processed in @(':')@(tsee
 logic) mode (the ``@('L')'') and was the 73rd @(see command) processed in the
 current session.</p>

 <p>We can also try out our newly defined function on some sample data.  To do
 that, just submit a form to be evaluated to ACL2.  For example,</p>

 @({
    ACL2 !>(my-app '(0 1 2) '(3 4 5))
    (0 1 2 3 4 5)
    ACL2 !>(my-app nil nil)
    NIL
    ACL2 !>
 })

 <p>Now suppose we want to prove something about the function just introduced.
 We conjecture, for example, that the length of the @(see append) of two lists
 is the sum of their lengths.  We can formulate this conjecture in the form of
 the following ACL2 @(tsee defthm) form.</p>

 @({
    (defthm my-app-length
      (equal (len (my-app x y))
             (+ (len x) (len y))))
 })

 <p>First of all, how did we know about the functions @('len') and @(tsee +),
 etc.?  The answer to that is somewhat unsatisfying &mdash; we know them from
 our past experience in using Common Lisp and ACL2.  It's hard to know that a
 function such as @('len') exists without first knowing some Common Lisp.  If
 we'd guessed that the appropriate function was called @(tsee length) (say,
 from our knowledge of Lisp) and tried @(':pe length'), we would have seen that
 @(tsee length) is defined in terms of @('len'), and we could have explored
 from there.  Luckily, you can write a lot of ACL2 functions without knowing
 too many of the primitive functions.</p>

 <p>Secondly, why don't we need some ``type'' hypotheses?  Does it make sense
 to append things that are not lists?  Well, yes.  ACL2 and Lisp are both quite
 weakly typed.  For example, inspection of the definition of @('my-app') shows
 that if @('x') is not a @(see cons) pair, then @('(my-app x y)') always
 returns @('y'), no matter what @('y') is.</p>

 <p>Thirdly, would it matter if we rewrote the lemma with the equality
 reversed, as follows?</p>

 @({
    (defthm my-app-length2
      (equal (+ (len x) (len y))
             (len (my-app x y)))).
 })

 <p>The two are <b>logically</b> equivalent, but...yes, it would make a big
 difference.  Recall our remark that a lemma is not only a ``fact'' to be
 proved; it also is used by the system to prove other later lemmas.  The
 current lemma would be stored as a @(see rewrite) rule.  (See @(see
 rule-classes).)  For a @(see rewrite) rule, a conclusion of the form @('(EQUAL
 LHS RHS)') means to replace instances of the @('LHS') by the appropriate
 instance of the @('RHS').  Presumably, it's better to @(see rewrite) @('(len
 (my-app x y))') to @('(+ (len x) (len y))') than the other way around.  The
 reason is that the system ``knows'' more about @(tsee +) than it does about
 the new function symbol @('my-app').</p>

 <p>So let's see if we can prove this lemma.  Submitting our preferred @(tsee
 defthm) to ACL2 (do it!), we get the following interaction:</p>

 @({
            --------------------------------------------------
  ACL2 !>(defthm my-app-length
    (equal (len (my-app x y))
           (+ (len x) (len y))))

  Name the formula above *1.

  Perhaps we can prove *1 by induction.  Three induction schemes are
  suggested by this conjecture.  These merge into two derived
  induction schemes.  However, one of these is flawed and so we are
  left with one viable candidate.

  We will induct according to a scheme suggested by (LEN X), but
  modified to accommodate (MY-APP X Y).  If we let (:P X Y) denote *1
  above then the induction scheme we'll use is
  (AND (IMPLIES (NOT (CONSP X)) (:P X Y))
       (IMPLIES (AND (CONSP X) (:P (CDR X) Y))
                (:P X Y))).
  This induction is justified by the same argument used to admit LEN,
  namely, the measure (ACL2-COUNT X) is decreasing according to the
  relation O< (which is known to be well-founded on the domain
  recognized by O-P).  When applied to the goal at hand the
  above induction scheme produces the following two nontautological
  subgoals.

  Subgoal *1/2
  (IMPLIES (NOT (CONSP X))
           (EQUAL (LEN (MY-APP X Y))
                  (+ (LEN X) (LEN Y)))).

  But simplification reduces this to T, using the :definitions of FIX,
  LEN and MY-APP, the :type-prescription rule LEN, the :rewrite rule
  UNICITY-OF-0 and primitive type reasoning.

  Subgoal *1/1
  (IMPLIES (AND (CONSP X)
                (EQUAL (LEN (MY-APP (CDR X) Y))
                       (+ (LEN (CDR X)) (LEN Y))))
           (EQUAL (LEN (MY-APP X Y))
                  (+ (LEN X) (LEN Y)))).

  This simplifies, using the :definitions of LEN and MY-APP, primitive
  type reasoning and the :rewrite rules COMMUTATIVITY-OF-+ and
  CDR-CONS, to

  Subgoal *1/1'
  (IMPLIES (AND (CONSP X)
                (EQUAL (LEN (MY-APP (CDR X) Y))
                       (+ (LEN Y) (LEN (CDR X)))))
           (EQUAL (+ 1 (LEN (MY-APP (CDR X) Y)))
                  (+ (LEN Y) 1 (LEN (CDR X))))).

  But simplification reduces this to T, using linear arithmetic,
  primitive type reasoning and the :type-prescription rule LEN.

  That completes the proof of *1.

  Q.E.D.

  Summary
  Form:  ( DEFTHM MY-APP-LENGTH ...)
  Rules: ((:REWRITE UNICITY-OF-0)
          (:DEFINITION FIX)
          (:REWRITE COMMUTATIVITY-OF-+)
          (:DEFINITION LEN)
          (:REWRITE CDR-CONS)
          (:DEFINITION MY-APP)
          (:TYPE-PRESCRIPTION LEN)
          (:FAKE-RUNE-FOR-TYPE-SET NIL)
          (:FAKE-RUNE-FOR-LINEAR NIL))
  Warnings:  None
  Time:  0.30 seconds (prove: 0.13, print: 0.05, other: 0.12)
   MY-APP-LENGTH
            --------------------------------------------------
 })

 <p>Wow, it worked!  In brief, the system first tried to @(see rewrite) and
 simplify as much as possible.  Nothing changed; we know that because it said
 ``Name the formula above *1.''  Whenever the system decides to name a formula
 in this way, we know that it has run out of techniques to use other than proof
 by induction.</p>

 <p>The induction performed by ACL2 is structural or ``Noetherian'' induction.
 You don't need to know much about that except that it is induction based on
 the structure of some object.  The heuristics infer the structure of the
 object from the way the object is recursively decomposed by the functions used
 in the conjecture.  The heuristics of ACL2 are reasonably good at selecting an
 induction scheme in simple cases.  It is possible to override the heuristic
 choice by providing an @(':induction') hint (see @(see hints)).  In the case
 of the theorem above, the system inducts on the structure of @('x') as
 suggested by the decomposition of @('x') in both @('(my-app x y)') and @('(len
 x)').  In the base case, we assume that @('x') is not a @(tsee consp).  In the
 inductive case, we assume that it is a @(tsee consp) and assume that the
 conjecture holds for @('(cdr x)').</p>

 <p>There is a close connection between the analysis that goes on when a
 function like @('my-app') is accepted and when we try to prove something
 inductively about it.  That connection is spelled out well in Boyer and
 Moore's book ``A Computational Logic,'' if you'd like to look it up.  But it's
 pretty intuitive.  We accepted @('my-app') because the ``size'' of the first
 argument @('x') decreases in the recursive call.  That tells us that when we
 need to prove something inductively about @('my-app'), it's a good idea to try
 an induction on the size of the first argument.  Of course, when you have a
 theorem involving several functions, it may be necessary to concoct a more
 complicated @(see induction) schema, taking several of them into account.
 That's what's meant by ``merging'' the induction schemas.</p>

 <p>The proof involves two cases: the base case, and the inductive case.
 You'll notice that the subgoal numbers go <b>down</b> rather than up, so you
 always know how many subgoals are left to process.  The base case (@('Subgoal
 *1/2')) is handled by opening up the function definitions, simplifying, doing
 a little rewriting, and performing some reasoning based on the types of the
 arguments.  You'll often encounter references to system defined lemmas (like
 @('unicity-of-0')).  You can always look at those with @(':')@(tsee pe); but,
 in general, assume that there's a lot of simplification power under the hood
 that's not too important to understand fully.</p>

 <p>The inductive case (@('Subgoal *1/1')) is also dispatched pretty easily.
 Here we assume the conjecture true for the @(tsee cdr) of the list and try to
 prove it for the entire list.  Notice that the prover does some simplification
 and then prints out an updated version of the goal (@('Subgoal *1/1'')).
 Examining these gives you a pretty good idea of what's going on in the
 proof.</p>

 <p>Sometimes one goal is split into a number of subgoals, as happened with the
 induction above.  Sometimes after some initial processing the prover decides
 it needs to prove a subgoal by induction; this subgoal is given a name and
 pushed onto a stack of goals.  Some steps, like generalization (see ACLH), are
 not necessarily validity preserving; that is, the system may adopt a false
 subgoal while trying to prove a true one.  (Note that this is ok in the sense
 that it is not ``unsound.''  The system will fail in its attempt to establish
 the false subgoal and the main proof attempt will fail.)  As you gain facility
 with using the prover, you'll get pretty good at recognizing what to look for
 when reading a proof script.  The prover's @(see proof-tree) utility helps
 with monitoring an ongoing proof and jumping to designated locations in the
 proof (see @(see proof-tree)).  See @(see tips) for a number of useful
 pointers on using the theorem prover effectively.</p>

 <p>When the prover has successfully proved all subgoals, the proof is
 finished.  As with a @(tsee defun), a @(see summary) of the proof is printed.
 This was an extremely simple proof, needing no additional guidance.  More
 realistic examples typically require the user to look carefully at the failed
 proof log to find ways to influence the prover to do better on its next
 attempt.  This means either: proving some rules that will then be available to
 the prover, changing the global state in ways that will affect the proof, or
 providing some @(see hints) locally that will influence the prover's behavior.
 Proving this lemma (@('my-app-length')) is an example of the first.  Since
 this is a @(see rewrite) rule, whenever in a later proof an instance of the
 form @('(LEN (MY-APP X Y))') is encountered, it will be rewritten to the
 corresponding instance of @('(+ (LEN X) (LEN Y))').  Disabling the rule by
 executing the @(see command)</p>

 @({
    (in-theory (disable my-app-length)),
 })

 <p>is an example of a global change to the behavior of the prover since this
 @(see rewrite) will not be performed subsequently (unless the rule is again
 @(see enable)d).  Finally, we can add a (local) @(see disable) ``hint'' to a
 @(tsee defthm), meaning to @(see disable) the lemma only in the proof of one
 or more subgoals.  For example:</p>

 @({
    (defthm my-app-length-commutativity
      (equal (len (my-app x y))
             (len (my-app y x)))
      :hints ((\"Goal\" :in-theory (disable my-app-length))))
 })

 <p>In this case, the hint supplied is a bad idea since the proof is much
 harder with the hint than without it.  Try it both ways.</p>

 <p>By the way, to undo the previous event use @(':u') (see @(see u)).  To undo
 back to some earlier event use @(':ubt') (see @(see ubt)).  To view the
 current event use @(':pe :here').  To list several @(see events) use @(':pbt')
 (see @(see pbt)).</p>

 <p>Notice the form of the hint in the previous example (see @(see hints)).  It
 specifies a goal to which the hint applies.  @('\"Goal\"') refers to the
 top-level goal of the theorem.  Subgoals are given unique names as they are
 generated.  It may be useful to suggest that a function symbol be @(see
 disable)d only for Subgoal 1.3.9, say, and a different function @(see enable)d
 only on Subgoal 5.2.8.  Overuse of such @(see hints) often suggests a poor
 global proof strategy.</p>

 <p>We now recommend that you visit @(see documentation) on additional
 examples.  See @(see annotated-acl2-scripts).</p>")

(defxdoc and
  :parents (basics acl2-built-ins)
  :short "Conjunction"
  :long "<p>@('And') is the macro for conjunctions.  @('And') takes any number
 of arguments.  @('And') returns @('nil') if one of the arguments is @('nil'),
 but otherwise returns the last argument.  If there are no arguments, @('and')
 returns @('t').</p>

 <p>@('And') is a Common Lisp macro.  See any Common Lisp documentation for
 more information.</p>

 @(def and)
 @(def and-macro)")

(defxdoc annotated-acl2-scripts
  :parents (acl2-tutorial)
  :short "Examples of ACL2 scripts"
  :long "<p>Beginning users may find these annotated scripts useful.  We
 suggest that you read these in the following order:</p>

 <code>
 @(see Tutorial1-Towers-of-Hanoi)
 @(see Tutorial2-Eights-Problem)
 @(see Tutorial3-Phonebook-Example)
 @(see Tutorial4-Defun-Sk-Example)
 @(see Tutorial5-Miscellaneous-Examples)
 </code>

 <p>You can also find useful demos in the @(see community-books) directory,
 @('books/demos/'), and its subdirectories.</p>

 <p>The web page <a
 href='http://www.cs.utexas.edu/users/moore/publications/tutorial/rev3.html'>Brief
 ACL2 Tutorial</a> contains a script that illustrates how it feels to use The
 Method to prove an unusual list reverse function correct.  The screen shots of
 ACL2's proof output are outdated &mdash; in the version shown, ACL2 does not
 print Key Checkpoints, but the concept of key checkpoint is clear in the
 discussion and the behavior of the user.</p>

 <p>See <a
 href='http://www.cs.utexas.edu/users/moore/acl2/contrib/POLISHING-PROOFS-TUTORIAL.html'>Polishing
 Proofs Tutorial</a> for a tutorial on becoming successful at approaching a
 formalization and proof problem in ACL2.  That tutorial, written by Shilpi
 Goel and Sandip Ray, has two parts: it illustrates how to guide the theorem
 prover to a successful proof, and it shows how to clean up the proof in order
 to facilitate maintenance and extension of the resulting book (see @(see
 books)).</p>

 <p>The <a
 href='http://www.cs.utexas.edu/users/moore/publications/tutorial/kaufmann-TPHOLs08/index.html'>ACL2
 Demo Given at TPHOLs 2008</a> by Matt Kaufmann includes scripts and a gzipped
 tar file containing the entire contents of the demos.</p>

 <p>The <a
 href='http://www.cs.utexas.edu/users/moore/publications/tutorial/sort-equivalence'>sort
 equivalence demo</a> is a collection of scripts illustrating both high-level
 strategy and lower-level tactics dealing with the functional equivalence of
 various list sorting algorithms.  Start with the @('README') on that
 directory.  There is also a <a
 href='http://www.cs.utexas.edu/users/moore/publications/tutorial/sort-equivalence.tgz'>gzipped
 tar file</a> with all of these scripts.</p>

 <p>When you feel you have read enough examples, you might want to try the
 following very simple example on your own. (See @(see
 solution-to-simple-example) for a solution, after you work on this example.)
 First define the notion of the ``fringe'' of a tree, where we identify trees
 simply as @(see cons) structures, with @(see atom)s at the leaves.  For
 example:</p>

 @({
    ACL2 !>(fringe '((a . b) c . d))
    (A B C D)
 })

 <p>Next, define the notion of a ``leaf'' of a tree, i.e., a predicate
 @('leaf-p') that is true of an atom if and only if that atom appears at the
 tip of the tree.  Define this notion without referencing the function
 @('fringe').  Finally, prove the following theorem, whose proof may well be
 automatic (i.e., not require any lemmas).</p>

 @({
    (defthm leaf-p-iff-member-fringe
      (iff (leaf-p atm x)
           (member-equal atm (fringe x))))
 })")

(defxdoc append
  :parents (lists acl2-built-ins)
  :short "@(see concatenate) zero or more lists"
  :long "<p>@('Append'), which takes zero or more arguments, expects all the
 arguments except perhaps the last to be true (@('nil')-terminated) lists.  It
 returns the result of concatenating all the elements of all the given lists
 into a single list.  Actually, in ACL2 @('append') is a macro that expands
 into calls of the binary function @(tsee binary-append) if there are at least
 two arguments; if there is just one argument then the expansion is that
 argument; and finally, @('(append)') expands to @('nil').</p>

 <p>@('Append') is a Common Lisp function.  See any Common Lisp documentation
 for more information.  See @(see append-without-guard) for a version of
 @('append') that has a guard of @('t').</p>

 @(def append)
 @(def binary-append)")

(defxdoc apply$
  :parents (acl2-built-ins programming)
  :short "Apply a badged function or tame lambda to arguments"

  :long "<p>We recommend that you read the paper <a
  href='http://www.cs.utexas.edu/users/kaufmann/papers/apply/index.html'>``Limited
  Second-Order Functionality in a First-Order Setting''</a> by Matt Kaufmann
  and J Strother Moore for both motivation and foundational details.  You might
  also read @(see introduction-to-apply$)!</p>

  <p>This documentation starts with a <b>glossary</b> of terms.  Then we
  provide some <b>examples</b> and present the <b>specification</b> of
  @('apply$').  Next, we deal with issues related to @('apply$') in
  <b>definitions</b>, <b>stating and proving theorems</b>, <b>guards and guard
  verification</b>, and <b>top-level evaluation</b>.  Finally we exhibit the
  <b>formal definitions</b> @('apply$') and some related concepts.  We have
  sprinkled in a little tutorial material for readability but have not provided
  much motivation for some design decisions.</p>

  <h3>Glossary</h3>

  <p>Here is a brief glossary of terms used in the semantics of @('apply$').
  While we provide links to the documentation of the concepts, we urge you not
  to follow those links until you've understood the big picture!</p>

  <ul>

  <li>@('apply$') -- the ACL2 function that takes two arguments, one
      representing a function and the other listing actuals to be fed to that
      function.  Under certain conditions, @('apply$') applies the function to
      the arguments and returns the result.  @('Apply$') is mutually recursive
      with @(tsee apply$-lambda), @(tsee ev$), and @(tsee ev$-list).
      @('Apply$')'s ``badge'' (see below) is @('(APPLY$-BADGE 2 1 :FN NIL)')
      its arity is 2, its ``out arity'' is 1 (i.e., it returns 1 result), its
      first argument has ``ilk'' @(':FN') and is thus treated as a
      ``function;'' its second argument has ilk @('NIL') and is thus treated as
      an ordinary object.  Initially, @('apply$') is the only symbol in ACL2
      with an ilk of @(':FN').</li>

  <li>@(tsee badge) -- an object associated with some function symbols
      indicating that @('apply$') can ``handle'' them and under what
      conditions.  The badge of a function symbol specifies its arity, its
      ``out arity,'' (i.e., the number of results the function returns), and
      the @(see ilk) of each argument position telling @('apply$') how each
      argument is treated.  The ilks are @(':FN'), @(':EXPR') and @('NIL').
      The association between a non-primitive function symbol and its badge is
      manged by @(tsee warrant)s.  In proofs, @('apply$') must have a warrant
      for every non-primitive function symbol to be applied.  Those warrants
      are provided as hypotheses to the theorem being proved.  Symbols without
      badges cannot be @('apply$')d.  Badges are generated, when possible, by
      @(tsee defwarrant).  (Badges can be generated for @(':')@('program') mode
      functions by @(tsee defbadge), allowing @('apply$') to handle such
      functions in top level evaluation not not in proofs.)  Not every function
      symbol can have a badge.</li>

  <li>compiled @('LAMBDA') cache (or simply <i>cache</i> in this context) -- a
      cache in the raw Lisp under ACL2 that supports the application of
      @('apply$') on well-formed, guard verified @('LAMBDA') objects.  Later in
      this Glossary we define ``lambda expression,'' ``@('LAMBDA') object,''
      and ``lambda$ expression'' -- three similar looking phrases with very
      different meanings.  See @(tsee print-cl-cache) for some details of the
      cache.</li>

  <li>evaluation theory -- the logical theory in which expressions submitted at
      the top level of the ACL2 read-eval-print loop are evaluated.  The
      evaluation theory is a consistent extension of the proof theory, the
      latter being the logical theory in which the ACL2 theorem prover
      operates.  The evaluation theory is not new to @('apply$'); it was
      introduced when @(tsee defattach) was added.  But the evaluation theory
      changed with the introduction of @('apply$').  All @(tsee warrant)s
      introduced by @('defwarrant') are assumed true in the evaluation theory
      but not in the proof theory.  This means ACL2 can execute calls of
      @('apply$') that arise in the evaluation of top-level input, but ACL2
      cannot evaluate all calls of @('apply$') that arise in proofs unless the
      appropriate warrants are available as hypotheses.</li>

  <li>lambda expression -- an integral part of ACL2's formal term syntax,
      lambda expressions are the way @('let') expressions and other
      variable-binding idioms are translated into formal terms.  Lambda
      expressions have nothing to do with @('apply$')!  See @(see lambda) for a
      discussion of three confusingly similar but different concepts: lambda
      expressions, @('LAMBDA') objects, and @('lambda$') expressions.  Read
      carefully anytime you see the word ``lambda!''</li>

  <li>@('LAMBDA') object -- an ACL2 list constant, typically of the form
      @('(LAMBDA vars body)') or @('(LAMBDA vars dcl body)') that may be used
      as a ``function'' by @('apply$').  @('Apply$') treats any @(tsee consp)
      object in its first argument position as though it were a @('LAMBDA')
      object.  But it only gives sensible meanings to @(see tame) @('LAMBDA')
      objects.  And only well-formed @('LAMBDA') objects are executed
      efficiently.  But well-formed @('LAMBDA') objects are hard to type by
      hand -- there are many constraints to keep in mind to guarantee
      well-formedness.  See @(tsee well-formed-lambda-objectp) if you really
      want to see all the rules.  But that is generally unnecessary.  We
      <i>strongly</i> recommend not entering @('LAMBDA') objects as quoted
      constants, e.g., @(''(LAMBDA (X) (+ 1 X))') -- which is actually
      ill-formed!  Instead, use @(tsee lambda$), as in @('(lambda$ (x) (+ 1
      x))').  See also @(see lambda) for some clarifications.</li>

  <li>@(tsee lambda$) expression -- an ACL2 macro that allows you to enter
      quoted well-formed @('LAMBDA') objects into your terms by typing
      untranslated expressions that resemble lambda expressions.  The
      @('lambda$') expression @('(lambda$ (x) (+ 1 x))') essentially translates
      into the quoted @('LAMBDA') object @(''(LAMBDA (X) (BINARY-+ '1 X))').
      We say ``essentially'' because @('lambda$') always inserts a
      @('(DECLARE (IGNORABLE v1 ... vn))') listing every formal and tags the
      body with a @(tsee RETURN-LAST) form that indicates it came from a
      translated @('lambda$'). See also @(see lambda) for some
      clarifications.</li>

  <li>@(tsee scion) -- a function that is ancestrally dependent on @('apply$').
      In the early days of @('apply$') we called scions ``mapping function''
      but in the Lisp community that implies iteration over a list and scions
      are more general.  Of course, a function that iterates over a list
      @('apply$')ing a ``function'' to each element and collecting the results
      is an example of a scion.  But so is function that takes a ``function''
      and applies it in one special situation, e.g., as a test or base case.
      Any function ancestrally dependent on @('apply$') is a scion whether or
      not it takes a ``function'' as an argument or maps over a domain.</li>

  <li>@(see tame) -- the class of functions that @('apply$') knows about; we
      actually talk about ``tame functions,'' ``tame @('LAMBDA') objects,'' and
      ``tame expressions.''  The last are expressions that are evaluable by an
      interpreter named @(tsee ev$) that is mutually-recursive with @('apply$').
      @('Apply$') cannot handle all defineable functions: ACL2 is first order
      and if @('apply$') were able to ``handle'' certain functions the logic
      would be inconsistent.</li>

  <li>@(tsee warrant) -- a 0-ary predicate associated with some user-defined
      function symbols that must be a hypothesis of any theorem whose proof
      involves ``expanding'' @('apply$') on such symbols; the warrant gives
      @('apply$') ``permission'' to expand if the arguments to which the
      function is applied are appropriately @(see tame).  The warrant for a
      function specifies the function's @(tsee badge) and how @('apply$')
      behaves on the function symbol.  Warrants (and badges) are computed and
      introduced by the @(tsee defwarrant) event.  Not all function symbols can
      be warranted.</li>

  </ul>

  <p>You will get a much better understanding of these concepts if you read the
  paper cited above.</p>

  <h3>Examples</h3>

  <p>To illustrate @('apply$') and some related concepts we need some
  user-defined functions.  We therefore imagine that the following events
  have been successfully admitted.</p>

  <p><b>We strongly recommend that you include the following book in any
  session in which you intend to use or reason about @('apply$').</b></p>

  @({
  (include-book \"projects/apply/top\" :dir :system)

  (defun$ sq (x) (* x x))

  (defun$ collect$ (fn lst)
    (if (endp lst)
        nil
        (cons (apply$ fn (list (car lst)))
              (collect$ fn (cdr lst)))))

  (defun$ foldr (lst fn init)
    (if (endp lst)
        init
        (apply$ fn
                (list (car lst)
                      (foldr (cdr lst) fn init)))))

  (defun$ russell (fn x)
    (not (apply$ fn (list x x))))
  })

  <p>Note: @('Collect$') is pre-defined in ACL2 because it is part of the
  support for the @(tsee loop$) statement.</p>

  <p>@('Collect$') and @('foldr') might informally be called ``mapping
  functions'' because they map a given function over some domain and accumulate
  the answers somehow.  They are useful examples of what we call <i>scions
  of</i> @('apply$') or simply <i>scions</i>: functions in which @('apply$') is
  ancestral, i.e., functions that call @('apply$') or call functions that call
  @('apply$'), etc.  @('Russell') is also a scion.  See @(see scion) for
  more.</p>

  <p>Here are some evaluations carried out at the top-level of the ACL2 loop
  after the events above.  Top-level evaluations take place in ACL2's
  evaluation theory (see the discussion of the semantics of @(tsee defattach)),
  which is an extension of the theory in which proofs are conducted.  Put more
  bluntly, the following evaluations won't be carried out in proofs unless you
  have the right hypotheses!</p>

  @({
  ACL2 !>(apply$ 'sq '(5))
  25

  ACL2 !>(collect$ 'sq '(1 2 3 4 5))
  (1 4 9 16 25)

  ACL2 !>(collect$ (lambda$ (x) (* x x)) '(1 2 3 4 5))
  (1 4 9 16 25)

  ACL2 !>(foldr '(1 2 3) 'cons '(4 5 6))
  (1 2 3 4 5 6)

  ACL2 !>(foldr '(1 2 3 4 5)
                (lambda$ (x y)
                  (cons (sq x) y))
                nil)
  (1 4 9 16 25)

  ACL2 !>(foldr '(1 2 3 4)
                (lambda$ (x y) (foldr y 'cons (list x)))
                nil)
  (4 3 2 1)

  ACL2 !>(russell 'natp 3)
  NIL

  ACL2 !>(russell 'consp 3)
  T
  })

  <p>@('Apply$') doesn't always work the way you might want!</p>

  @({
  ACL2 !>(let ((x 'russell))(russell x x))

  ACL2 Error in TOP-LEVEL:  The value of APPLY$-USERFN is not specified when
  the first argument, fn, is RUSSELL, and the second argument, args,
  is (RUSSELL RUSSELL).  Fn has badge (APPLY$-BADGE 2 1 :FN NIL) and
  args is not known to satisfy the tameness requirement of that badge.
  })

  <p> @(tsee Apply$-userfn) is the undefined function called by @('apply$')
  when it is asked to apply a user-defined function symbol instead of a builtin
  function symbol.  The @(tsee warrant) for @('russell') actually specifies the
  value of @('(apply$-userfn 'russell ...)')  under the @(tsee tame)ness
  requirements, and those requirements are violated above.  This is necessary
  to preserve the consistency of the logic.  Otherwise:</p>

  <code>
  (russell 'russell 'russell)
  = {by defun of russell}
  (not (apply$ 'russell (list 'russell 'russell)))
  =  {by the naive expectation that apply$ always ``works''}
  (not (russell 'russell 'russell))
  Contradiction!
  </code>

  <p>Top-level evaluation of @('apply$') expressions raises problems not seen
  anywhere else in ACL2's execution model: While executing syntactically legal
  terms the evaluator can encounter undefined functions or weirdly ill-formed
  terms not caught by the usual ACL2 translation mechanism.  The ACL2
  translation mechanism checks the well-formedness of @(tsee lambda$)
  expressions (and user-typed quoted @('LAMBDA') objects) that occur in
  positions of ilk @(':FN') and are therefore destined for @('apply$').  But
  the translation checks can be defeated.  The @('LAMBDA') object below
  contains a call of the undefined function @('foo') but the error is not
  caught at translation time; it is caught only when the form executed.</p>

  @({
  ACL2 !>(apply$ `(lambda (x) (foo x)) '(5))

  ACL2 Error in TOP-LEVEL:  The value of BADGE-USERFN is not specified
  on FOO because FOO is not a known function symbol.
  })

  <p>Note the <i>backquote</i> on the @('LAMBDA') object.  This defeats the
  check of well-formedness because the @('LAMBDA') object is not @('quote')d.
  We could have equally written</p>

  @({
  ACL2 !>(apply$ (list 'lambda '(x) (cons 'foo '(x))) '(5))
  })

  <p>with the same result.  There is nothing unsound about this.  @('Apply$')
  can take any objects as arguments.  But it won't always ``behave'' as you
  might expect.  One way to explore the edge cases of @('apply$') is to execute
  it on ill-formed input.  In addition, some theorems may require consing up a
  @('LAMBDA') object in terms of objects used elsewhere in the theorem.  See
  example theorem @('[3]') below.</p>

  <p>A peculiar aspect of @('LAMBDA') objects is that they can be written as
  legal ACL2 constants <i>before</i> they are well-formed @('LAMBDA') objects,
  e.g., by referring to undefined functions, @(':program') mode functions,
  unbadged functions, etc.  They are, after all, just arbitrary quoted objects
  and any value in ACL2 can be quoted.  But an ill-formed object can
  <i>become</i> well-formed if the world is appropriately extended, e.g., the
  appropriate @('defun')s or @('defwarrant')s are made.  Perhaps worse, they
  can be well-formed and then <i>become</i> ill-formed by an undo.  So at
  runtime @('apply$') has to check that the function symbol or @('LAMBDA')
  object is appropriate.  There is a sophisticated cache behind the execution
  machinery for @('LAMBDA') objects in the evaluation theory.</p>

  <p>Here are some theorems that can be proved about these concepts.  The last
  of the theorems shown below requires two lemmas, named
  @('weird-little-lemma1') and @('weird-little-lemma2'), shown in
  @('books/projects/apply/report.lisp').</p>

  @({

  ; [1] SQ squares, if you have the warrant for sq!  Imagine for a moment that
  ; we could prove (equal (apply$ 'SQ (list i)) (* i i)) without the warrant
  ; hypothesis shown below.  And imagine that we did so in an encapsulated
  ; environment in which sq was locally defined to be (* x x).  Then imagine we
  ; exported the simpler theorem out of that encapsulate and defined sq to be
  ; (+ 1 (* x x)).  Then ACL2 would be unsound.  Exporting a theorem requires
  ; that the theorem be ancestrally independent of every locally defined
  ; function and the simpler hypothetical theorem is, because the symbol 'SQ is
  ; not ancestrally dependent on sq.  But ACL2 cannot prove the simpler
  ; theorem!  It cannot ``open'' apply$ on 'SQ without the warrant for sq and
  ; the warrant for sq is ancestrally dependent on sq.  So the theorem below
  ; cannot be exported from an environment in which sq is locally defined.
  ; Thus warrants solve the so-called ``LOCAL problem.''

  (thm (implies (warrant sq)
                (equal (apply$ 'SQ (list i))
                       (* i i))))

  ; [2] Collect$ distributes over append for any fn.

  (thm (equal (collect$ fn (append a b))
              (append (collect$ fn a)
                      (collect$ fn b))))

  ; [3] Foldr can be used to collect$, but the collection must
  ; be with an ``ok'' function (a tame function of one
  ; argument).  Note the backquote on the LAMBDA.  This is
  ; a theorem that requires us to cons up a LAMBDA object.

  (thm (implies (ok-fnp fn)
                (equal (foldr lst
                              `(LAMBDA (X Y) (CONS (,fn X) Y))
                              nil)
                       (collect$ fn lst))))
  })

  <h3>Specification of APPLY$</h3>

  <p><b>We strongly recommend that you include the following book in any
  session in which you intend to use or reason about @('apply$').</b></p>

  @({
  (include-book \"projects/apply/top\" :dir :system)
  })

  @({
  General Form:
  (apply$ fn args)
  })

  <p>where @('fn') is some function symbol or @('LAMBDA') object and @('args')
  is a true list.  Informally, @('apply$') applies the function named by the
  first argument to the appropriate number of elements taken from the second
  argument.  We might express this as:</p>

  <code>
  <b>Naive Specification (for a single-valued function):</b>
  (apply$ 'fn args) = (fn (nth 0 args) ... (nth (- n 1) args))

  <b>Naive Specification (for a multi-valued function):</b>
  (apply$ 'fn args) = (mv-list k (fn (nth 0 args) ... (nth (- n 1) args)))
  </code>

  <p>where @('fn') is of arity @('n') and @('k') is the ``out arity'' of
  @('fn') (the number of returned values).  <b>However</b>, these naive
  specifications are guaranteed only if either (i) @('fn') is a function symbol
  that has a @(tsee badge) and @('args') satisfies the @(see tame)ness
  requirements of the badge, or (ii) @('fn') is a well-formed @('LAMBDA')
  object that returns 1 result.  The tameness requirement is that if an element
  of @('args') is in an argument position of @('fn') with ilk @(':FN') then the
  element must satisfy @('tamep-functionp') and if the element is in an
  argument position of ilk @(':EXPR') it must satisfy @('tamep').  See @(tsee
  badge) for further discussion of condition (i).  As for (ii), rather than
  explain ``well-formed @('LAMBDA') object'' here we encourage you to write
  @(tsee lambda$) expressions when you want to @('apply$') a @('LAMBDA')
  object.</p>

  <p>The @(see ilk)s of @('apply$') are @(':FN') and @('NIL') respectively,
  telling us that @('apply$') treats its first argument as a ``function'' and
  its second as an ordinary object (never as a function).  Initially
  @('apply$') is the only symbol in ACL2 with an ilk of @(':FN').  However as
  @(tsee defwarrant) is used successfully on @(see scion)s -- functions that
  somehow call @('apply$') -- other symbols can have ilk @(':FN') too.</p>

  <p>@('Apply$') has a guard, namely @('(apply$-guard fn args)').  This is an
  exceptionally weak guard, requiring only that @('args') be a true-list and,
  if @('fn') is a cons -- which is automatically treated as a @('LAMBDA')
  object -- the length of @('args') be the length of the second element of
  @('fn').  We discuss <b>guards and guard verification</b> in a subsequent
  section.</p>

  <p><b>Note for Experts</b>: Technically, @('apply$') treats any @('consp')
  object as a @('LAMBDA') object.  But the results are as you'd naively expect
  only if the object is a @(see tame) @('LAMBDA') object.  However, we
  frequently write as though the object must be <i>well-formed</i>, which is
  different from but implies tameness.  What's going on?  The reason for this
  and related discrepancies in the documentation is that there is a tension
  between the logical definition of @('apply$') and the practical business of
  executing it.  The former involves the existence of a model, soundness, and
  the difficulty of proving theorems about @('apply$').  The latter involves
  the Common Lisp compiler.  We want the logical foundations to be simple to
  make it easier to reason about @('apply$'), but the compiler imposes
  unavoidable and complicated restrictions.  The upshot is that the logical
  foundations assign meaning to @('LAMBDA') objects that cannot be compiled.
  Applying merely ``tame'' @('LAMBDA')s is slower than applying ``well-formed''
  ones.  In a sense, by acting like ``tame @('LAMBDA') objects'' and
  ``well-formed @('LAMBDA') objects'' are synonymous we're trying to trick you!
  If you ever have occasion to formally express the restrictions on @('apply$')
  in some theorem, use @('tamep-functionp').  But when you write concrete
  @('LAMBDA') constants, try to keep them well-formed.  We encourage this by
  providing @(tsee lambda$) and by enforcing full blown well-formedness checks
  -- not just tameness checks -- in translate on every quoted @('LAMBDA')
  object entered in a @(':FN') slot.  And we give you ways to circumvent these
  checks -- see @(see gratuitous-lambda-object-restrictions) -- if you really
  mean to supply ill-formed @('LAMBDA') objects to @(':FN') slots.</p>

  <p>Badges are assigned by @(tsee defwarrant), and also by @(tsee defbadge).
  See @(tsee badge) for documentation about how to find out whether a function
  has a badge and how to interpret a badge.  The terms ``out arity'' and
  ``tameness requirements,'' used above, are explained there too.</p>

  <p>Intuitively, the badge of @('fn') tells @('apply$') how each formal of
  @('fn') is used in the definition of @('fn') and there are only three
  ``ilks'' of use.  Ilk @(':FN') means the formal is used exclusively as a
  function, meaning the formal can be passed into @(':FN') slots of other
  functions and eventually reaches @('apply$'), but it is never touched by
  other ACL2 functions.  Ilk @(':EXPR') means the formal is used exclusively as
  an expression, meaning the formal may be passed into @(':EXPR') slots of
  other functions and eventually reaches @(tsee ev$), but is never otherwise
  touched.  Finally, ilk @('NIL') means the formal is treated as an ordinary
  ACL2 object and, in particular, never used as either a function or an
  expression.  The ``tameness requirement'' on each actual is determined by the
  ilk of the corresponding formal: actuals in @(':FN') slots must satisfy
  @('tamep-functionp'), actuals in @(':EXPR') slots must satisfy @('tamep'),
  and there are no requirements on actuals in ilk @('NIL') slots.  For
  discussions of @('tamep-functionp') and @('tamep') see the topic @(see
  tame).</p>

  <p>Generally speaking, if you want to be able to @('apply$') a function you
  should introduce it with @(tsee defun) and then call @(tsee defwarrant) on
  the function name, or use @(tsee defun$), which is a convenient abbreviation
  for the above sequence of events.  But @('defun$') only works for @(':logic')
  mode functions because @('defwarrant') enforces that restriction.  If you
  want to @('apply$') a @(':program') mode function you should define it with
  @('defun') and then call @('defbadge') on its name.</p>

  <p>We summarize specification of @('apply$') with an example.  Consider</p>

  @({  (apply$ 'foldr
          '((1 2 3)     ; actual 1
            cons        ; actual 2
            (4 5 6)))   ; actual 3
  })

  <p>The badge of @('foldr'), computed by @('(badge 'foldr)'), is
  @('(APPLY$-BADGE 3 1 NIL :FN NIL)').  The arity is @('3'), the out arity is
  1 (@('foldr') is single-valued) and the ilks list is @('(NIL :FN NIL)').
  Thus the first and third formals have ilk @('NIL') and are treated as
  ordinary objects; the second formal has ilk @(':FN') and is treated as a
  function.  Thus, the tameness requirement is that the second actual to a call
  of @('foldr') must satisfy @('tamep-functionp').  Referring to the
  specification above, we see that the @('apply$') term has the ``naive
  specification'' since @('foldr') has a badge, its out arity is 1, and its
  second actual, @('cons'), satisfies @('tamep-functionp'). That is,</p>

  @({  (apply$ 'foldr
          '((1 2 3)     ; actual 1
            cons        ; actual 2
            (4 5 6)))   ; actual 3
  =
  (foldr '(1 2 3) 'cons '(4 5 6))
  =
  '(1 2 3 4 5 6)})

  <p>The first equation above is just the naive specification of @('apply$')
  and the second equation is just the definition of @('foldr').</p>

  <p>Formals are classified by @(tsee defwarrant) when it tries to compute the
  badge of a function.  What are the rules that lead to a formal being assigned
  ilk @(':FN'), for example?  What does ilk @(':FN') actually signify?</p>

  <p>Let <i>v</i> be the <i>i </i>th formal parameter of a badged function
  <i>fn</i>.  If the badge says that <i>v</i> has ilk @(':FN') then we know
  that <i>v </i> is ``used as a function'' in the definition of <i>fn </i>,
  i.e., the value of <i>v </i> eventually makes its way into the first argument
  of @('apply$').  Furthermore, <i>v </i> is never used any other way: every
  place <i>v </i> occurs in the body it is treated as a function.  And finally,
  in every recursive call of <i>fn </i> <i>v </i> is passed identically in the <i>i
  </i>th argument position of every recursive call.</p>

  <p>If the badge says that formal variable <i>v </i> has ilk @(':EXPR') then
  it signifies analogous conditions except that instead of eventually getting
  into the first argument of @('apply$') it eventually gets into the first
  argument of @('ev$').  We say such formals are ``used as expressions.''
  @(tsee Ev$) is the natural notion of evaluation in this context: look up the
  values of variables in the alist argument to @('ev$'), return quoted
  constants, and otherwise @('apply$') function symbols and @('lambda') objects
  to the recursively obtained list of values returned by evaluating the
  actuals.  However, @('ev$') first checks that the expression is @(tsee
  tamep).</p>

  <p>If the badge says a formal <i>v </i> has ilk @('NIL') in the definition of
  <i>fn </i> then <i>v </i> is <i>never used </i> as a function or as an
  expression in the definition.</p>

  <p>It is the job of @(tsee defwarrant) to analyze a definition and assign
  ilks, if possible.  But it may not be possible!  For example,</p>

  @({(defun foo (x) (apply$ x (list x)))})

  <p>is such a definition.  The formal @('x') is used as a function in its first
  occurrence but is not used as a function in its second.  Thus</p>

  @({(defwarrant foo)})

  <p>will fail.</p>

  <p>When successful, @(tsee defwarrant) also defines the @(tsee warrant)
  function for the function it analyzed.  Warrants are crucial to stating and
  proving theorems about function symbols being applied with @('apply$').  We
  illustrated warrants in the ``Examples'' section above and discuss them
  further in the secion on ``Theorems Involving @('Apply$')'' below.  See also
  @(tsee warrant).</p>

  <p>@('Apply$') is a defined function in the ACL2 source code.  We exhibit its
  definition at the end of this documentation but you may also see its
  definition by doing</p>

  @({
  ACL2 !>:pe apply$
  })

  <p>The definition is mutually recursive with</p>

  <ul>
  <li>@(tsee apply$-lambda): used by @('apply$') to handle the case when
  the first argument to @('apply$') is a @('LAMBDA') object.</li>

  <li>@(tsee ev$): used by @('apply$-lambda') to evaluate the body of a
  @('LAMBDA') object in an environment binding the object's formal variables to
  the actuals.</li>

  <li>@(tsee ev$-list): used by @('ev$') to evaluate a list of expressions in
  an environment binding formals to actuals.</li>

  </ul>

  <p>@('Apply$') calls three undefined functions:</p>

  <ul>

  <li>@(tsee apply$-userfn): used by @('apply$') when it is asked to apply
  anything other than a @('LAMBDA') object or a built-in function symbol.  In
  the evaluation theory, we attach a function to @('apply$-userfn') that
  explicitly enforces the tameness requirements for each user-defined
  @(':logic') mode function symbol that has had a badge computed by @(tsee
  defwarrant) and, if those requirements are met, applies the corresponding
  function.  Magically, that attachment to @('apply$-userfn') can also evaluate
  @(':program') mode functions with badges created by @(tsee defbadge).  We say
  ``magically'' because there are no axioms that explain this behavior, just as
  there are no axioms that explain how you can evaluate ordinary calls of
  @(':program') mode functions in the evaluation theory.  But in the proof
  theory @('apply$-userfn') remains undefined.  The value of @('(apply$-userfn
  'fn ...)'), and thus of @('(apply$ 'fn ...)'), is specified by a special
  hypothesis, called the ``warrant for @('fn').''  You can't prove anything
  interesting about the behavior of @('apply$') on a user-defined function
  symbol @('fn') unless the warrant for @('fn') is a governing hypothesis.  We
  discuss warrants in @(tsee warrant).  See also @(tsee defwarrant).</li>

  <li>@('untame-apply$'): used by @('apply$') when it is asked to deal with a
  situation in which tameness is violated.</li>

  <li>@('untame-ev$'): used by @('ev$') when it is asked to deal with a
  situation in which tameness is violated.</li>

  </ul>

  <h3>Definitions Involving on @('Apply$')</h3>

  <p>In one sense, @('apply$') is just an ordinary ACL2 function that takes two
  arguments and returns one result.  Like all ACL2 functions, @('apply$') is
  untyped.  You can supply any two objects as arguments and the axioms tell you
  what the result is -- though sometimes the result is delivered by an
  undefined function.</p>

  <p>But in a deeper sense, if you want @('apply$') to ``behave,'' and in
  particular if you want functions that use @('apply$') to ``behave,'' you have
  to follow certain rules.  For example, ACL2 must be able to determine whether
  a formal parameter is ``used as a function'' in a given definition.
  Basically, you will want every @(':logic') mode function that you define to
  be processed by @('defwarrant') so that it gets a badge and warrant if at all
  possible and at least has a chance of being applied as expected by
  @('apply$').</p>

  <p>The macro @('defun$') is just an abbreviation for a @('defun') followed by
  a @('defwarrant') and it is easy to imagine the other ACL2 definitional
  idioms introduced in the ACL2 Community Books eventually being extended to
  include a subsequent @('defwarrant').</p>

  <p>So the question becomes ``What rules must a @('defun') obey in order to be
  processed successfully by @('defwarrant')?''  The full answer is given in the
  documentation for @(tsee defwarrant).  But here are some guidelines to follow:
  <ul>
  <li>use @(':logic') mode,</li>
  <li>don't use @(tsee state) or @(tsee stobj)s in the signature,</li>
  <li>use a measure that either returns a natural number or a
      lexicographic combination of natural numbers as defined by the @('llist')
      function in the Community Books at @('books/ordinals/'),</li>
  <li>make sure every function used in the definition has a badge,</li>
  <li>ensure that every @(':FN') slot in the body is occupied either by
      a formal parameter or a quoted, badged function symbol or @(tsee lambda$)
      expression, and</li>
  <li>ensure that no parameter occupying a @(':FN') slot is ever used in a
      slot of any other ilk, and</li>
  <li>ensure that every parameter passed into a @(':FN') slot is passed into
      the same argument position in any recursive calls of the function being
      defined.</li>
  </ul></p>

  <p>You can certainly violate these rules and still get an admissible
  definition.  For example @('(defun rus (x) (not (apply$ x (list x))))') is
  admissible and you can run it on some arguments, e.g., @('(rus 'consp)')
  evaluates to @('T').  You can even prove @('(equal (rus 'consp) t)').  But
  @('(defwarrant rus)') fails because @('rus') violates the rules.  So you will not
  be able to @('apply$') @(''rus').</p>

  <h3>Theorems Involving @('Apply$')</h3>

  <p>Because @('apply$') is undefined on user-defined function symbols and
  warrant hypotheses specify the tameness requirements and value of @('apply$') on
  such symbols, you can't prove much about the application of particular user-defined
  symbols unless you provide the corresponding warrants as hypotheses.</p>

  <p>To emphasize this point, suppose @('sq') has been introduced with
  @('defun$') as shown above, then the following top-level evaluation is
  possible:</p>

  @({
  ACL2 !>(apply$ 'sq '(5))
  25
  })

  <p>You might expect to be able to prove the obvious little theorem</p>

  @({(thm (equal (apply$ 'sq '(5)) 25))})

  <p>However, you would be wrong!  While ACL2's evaluation theory assumes all
  warrants, the proof theory does not.  (If it did we could suffer the
  @('LOCAL') problem mentioned in example theorem @('[1]') above and in @(see
  introduction-to-apply$).)  Logically, there is no connection between the
  symbol @(''SQ') and the user-defined function @('sq').  That connection is
  established by warrant.  All the necessary warrants must be explicitly
  provided as hypotheses by the user.</p>

  <p>The warranted version of the little theorem above is easily proved.</p>

  @({(thm (implies (warrant sq) (equal (apply$ 'sq '(5)) 25)))})

  <p>Here @('(warrant sq)') is just an abbreviation for a call of the 0-ary
  function symbol @('apply$-warrant-sq') which is the name of the warrant for
  @('sq').  @('Apply$-warrant-sq') is introduced when @('(defwarrant sq)')
  completes successfully.  In particular, the following is a theorem:</p>

  @({
      (warrant sq)
    <-->
      (apply$-warrant-sq)
    <-->
      (((badge 'SQ) = '(APPLY$-BADGE 1 1 . T))
       &
       ((apply$ 'SQ args) = (sq (car args))))
   })

  <p>Thus, the warrant for @('sq') specifies the value of @('(badge 'sq)') and
  of @('(apply$ 'sq ...)').</p>

  <p>If you try to prove the unwarranted version of the little theorem about
  @(''sq') it fails in a forcing round with</p>

  @({
  [1]Goal
  (APPLY$-WARRANT-SQ)
  })

  <p>This is a clear indication that you forgot to provide the warrant.</p>

  <p>You might worry that theorems burdened by warrants are vacuously valid
  because it might be impossible to satisfy all the warrant hypotheses.  You
  needn't worry about this.  <i>There is a model of @('apply$') and all of its
  scions that makes every warrant issued by @('defwarrant') valid.</i> The
  proof of this is sketched in <a
  href='http://www.cs.utexas.edu/users/kaufmann/papers/apply/index.html'>``Limited
  Second-Order Functionality in a First-Order Setting''</a> by Matt Kaufmann
  and J Strother Moore and fully fleshed out in the comment titled <tt>Essay on
  Admitting a Model for Apply$ and the Functions that Use It</tt> in the ACL2
  source file @('apply-raw.lisp').</p>

  <p>So there are four lessons here:</p>

  <p><b>Lesson 1:</b> When stating theorems involving @('apply$') or scions on
  concrete user-defined functions, provide as additional hypotheses the
  warrants for all user-defined functions that @('apply$') will encounter
  during the proof.  This generally means you should add the hypothesis
  <tt>(warrant </tt><i>fn1 fn2 ... fnk</i><tt>)</tt> typically listing every
  function symbol that appears inside a quoted constant destined for
  @('apply$') or @('ev$') in your conjecture.  In particular, you should
  include every quoted function symbol appearing in a @(':FN') slot of
  @('apply$') or any scion, including every function symbol appearing in the
  body of any @('LAMBDA') object or @('lambda$') term or any @('until'),
  @('when'), or body expressions of @('loop$')s.  (Macro expansion in
  @('lambda$')s and @('loop$')s may introduce function symbols not evident in
  the untranslated forms.  See @(':')@(tsee translam) and @(':')@(tsee
  trans).)</p>

  <p><b>Lesson 2:</b> You need not worry that adding warrant hypotheses makes
  your theorems vacuously valid!  There is a model of @('apply$') and all your
  scions in which all warrants are valid.</p>

  <p><b>Lesson 3:</b> If a proof involving @('apply$') or a scion fails in a
  forcing round with a checkpoint whose conclusion is the warrant for some
  function, you should remember Lesson 1 and include that function symbol in
  the warrant for your conjecture!  That is, if you forget to supply a warrant
  but your conjecture is otherwise provable, ACL2's checkpoints will remind
  you.</p>

  <p><b>Lesson 4:</b> If a proof involving @('apply$') or a scion fails here
  are some things to think about.  The basic question is whether something is
  ``wrong'' with one or more function symbols supposedly handled by
  @('apply$').  You have to identify which quoted function symbols are not
  being simplified or expanded.  Typically you'll see a checkpoint with a term
  like @('(apply$ 'fn ...)') or @('(ev$ '(fn ...)  ...)') that you expect would
  be expanded into an actual call of @('fn').  In that case, @('fn') is of
  interest.  Here are some questions you should ask yourself about @('fn').</p>

  <ul>

  <li>Is @('fn') defined and in @(':logic') mode?  If @('fn') is in
  @(':program') mode it is treated by the prover as an undefined symbol.  You
  should try to convert it @(':logic') mode with @(tsee
  verify-termination).</li>

  <li>Is @('fn') warranted?  If not, see @(tsee defwarrant).  If @('fn') is
  warranted then it is possible @('fn') is not the problem.  Maybe the warrant
  for @('fn') was not provided as a hypothesis?  Normally, missing warrant
  hypotheses are forced, but the proof might have failed for other reasons
  before the warrant for @('fn') was forced.  But you should ask whether
  forcing is disabled; see @(tsee force).</li>

  <li>If you see @('(apply$ 'fn ...)') then perhaps the rewrite rule
  @('APPLY$-fn') is disabled.  That rule is the one that forces the warrant for
  @('fn') and it was proved when @('fn') was warranted.</li>

  </ul>

  <p>These issues are discussed further in the documentation for @(tsee
  warrant).</p>

  <p>An unfortunate implication of the need for warrants is highlighted during
  the proofs of measure conjectures while admitting new definitions.
  Consider</p>

  @({
  (defun$ my-cdr (x) (cdr x))

  (defun$ my-len (x)
    (if (endp x)
        0
        (+ 1 (my-len (apply$ 'my-cdr (list x))))))
  })

  <p>The definition of @('my-len') fails!  The reason is that without the
  warrant for @('my-cdr') we cannot prove that the measure decreases in the
  recursion above.  Unfortunately, there is no way to provide a warrant in a
  definition.  At the moment we advise users to avoid the use of @('apply$') --
  and functions that use @('apply$') -- in ``termination-critical'' roles.  By
  that we mean do not use @('apply$') if its properties are important to proofs
  of your measure conjectures.  This is easy advice to implement in the case of
  @('my-len'), i.e., replace the recursive call above by @('(my-len (my-cdr
  x))').  However, in more sophisticated definitions, e.g., where a @(tsee
  loop$) is being used in recursive calls and the @('loop$') calls user-defined
  functions in its body, following this advice means replacing that @('loop$')s
  by a recursive function.  That is unfortunate since the whole point of
  @('loop$') is to avoid the introduction of such functions!  We hope to
  address this limitation in the future, e.g., by making the definition of
  @('my-len') above be conditional on the warrant for @('my-cdr').</p>

  <h3>Guards and Guard Verification</h3>

  <p>As noted, @('apply$') has a guard of @('(apply$-guard fn args)') and is
  itself guard verified.  The guard is weak, basically requiring that @('fn')
  either be a symbol or a @('LAMBDA') object, that @('args') be a true-list,
  and, when @('fn') is a @('LAMBDA') object, the length of the list of formals
  is equal to the length of @('args').  To verify the guards of a scion you
  must make sure these properties hold of every application of anything in a
  @(':FN') slot.  Mainly you must make sure that every time a function object
  is @('apply$')d, it is applied to a list of the right length.</p>

  <p>Note also that @(see mixed-mode-functions), i.e., @(':logic') mode
  functions that use @(':program') mode functions in slots of @(see ilk)
  @(':FN') or @(':EXPR'), cannot be guard verified.</p>

  <p>But guards arise in another way in connection with @('apply$').  How does
  @('(apply$ fn args)') behave when @('fn') has guards?  The short answer is:
  logically speaking, @('apply$') completely ignores guards.  Guards in ACL2 are
  ``extra-logical.''</p>

  <p>Let's define and warrant a well-guarded version of ``square'',</p>

  @({
  (defun$ squ (n) (declare (xargs :guard (natp n))) (* n n))
  })

  <p>@('Squ') is guard verified.  Now let's consider the little conjecture:</p>

  @({
  (thm (implies (warrant squ) (equal (apply$ 'SQU (list x)) (* x x))))
  })

  <p>Do we need need to require @('(natp x)')?  We would if the logical
  definition of @('apply$') checked the guard of @('fn') before interpretting
  it.  But it does not check.  It just behaves as specified above.  So,
  regardless of whether the guard is satisfied or not, @('(apply$ 'squ (list
  x))') naively expands (under the warrant) to @('(squ x)'), from which the
  rest of the proof follows.</p>

  <p>However, now let's do a top-level evaluation of this @('apply$') term:</p>

  @({
  ACL2 !>(apply$ 'SQU (list 'NAN))

  ACL2 Error in TOP-LEVEL:  The guard for the function call
  (SQU N), which is (NATP N), is violated by the arguments
  in the call (SQU 'NAN).
  })

  <p>(Remember that ACL2's evaluation theory effectively assumes all warrants.)
  What happened?  @('Apply$') expanded to @('(SQU 'NAN)') and that caused the
  usual guard violation, given the default configuration of @(tsee
  set-guard-checking).</p>

  <p>A similar guard violation error is signalled if a guarded @('LAMBDA')
  object is @('apply$')ed to something violating its guard.</p>

  <p>But now consider</p>

  @({
  (defun$ strange (x)
    (declare (xargs :guard t))
    (apply$ 'SQU (list x)))
  })

  <p>This succeeds and @('strange') is now a guard verified, warranted
  function, with a guard of @('T').  This might be surprising since (a) the
  guard of @('strange') tells us nothing about @('x'), (b) @('SQU') is applied
  to @('x'), and (c) we know the guard of @('SQU') requires its argument to be
  a @('natp').  Guard verification ignores the guards of a quoted function
  symbol being applied by a scion.  This may be particularly offensive to one's
  intuitions when the scion is @('apply$') itself, since the appropriate
  information is available.  But consider a call of an arbitrary user-defined
  scion, e.g., @('(my-scion 'SQU x)').  To what arguments will @('my-scion')
  @('apply$') @('SQU')?  And how can the definition of @('my-scion') even
  specify what functional objects are acceptable in its first argument?  This
  is a limitation suffered by ACL2 that a suitably expressive type system would
  not.  Our way of coping with it is to ignore the guard here and make sure
  that when @('apply$') applies the function symbol executes it checks the
  guard of the symbol.</p>

  <p>Guard verification does not ignore guards of a quoted lambda object being
  @('apply$')ed.  Thus, for example, while @('strange') can be guard
  verified,</p>

  @({
  (defun$ stranger (x)
    (declare (xargs :guard t))
    (apply$ (lambda$ (e) (SQU e)) (list x)))
  })

  <p>cannot be guard verified, because guard verification tries to verify the
  guards of every @('lambda') object in a @(':FN') slot so that the @('lambda')
  object can be marked as guard verified in the compiled @('lambda') cache
  (see @(tsee print-cl-cache)).  But guards of @('lambda') objects must be
  verified independently of the context in which they are used.  To be specific,
  even</p>

  @({
  (defun$ stranger (x)
    (declare (xargs :guard (natp x)))
    (apply$ (lambda$ (e) (SQU e)) (list x)))
  })

  <p>cannot be guard verified because the @('lambda') object's guards are
  verified independently of the context.  The @('lambda') object must carry
  its own guard, as in</p>

  @({
  (defun$ stranger (x)
    (declare (xargs :guard (natp x)))
    (apply$ (lambda$ (e)
              (declare (xargs :guard (natp e)))
              (SQU e))
            (list x)))
  })

  <p>The last definition of stranger can be guard verified, the @('lambda')
  object is so marked in the cache and compiled, and if that @('lambda') object
  is used in any other context it is recognized as being guard verified.  The
  guard for that @('lambda') is checked when the object is @('apply$')ed but if
  the check approves then the body of the @('lambda') is executed as compiled
  code without further guard checking.</p>

  <p>While @('apply$') is not evident in a @(tsee loop$) statement like</p>

  @({
  (loop$ for e in lst collect (SQU e))
  })

  <p>similar treatment is given.  In particular, the @('loop$') above cannot
  be guard verified but</p>

  @({
  (loop$ for e in lst collect :guard (natp e) (SQU e))
  })

  <p>can be and is compiled into a Common Lisp @('loop').  Recall also from
  the @(tsee loop$) documentation that the formal semantics of the above
  statement is essentially</p>

  @({
  (collect$ (lambda$ (e)
              (declare (xargs :guard (natp e)))
              (SQU e))
            lst)
  })

  <p>so guard verification of the @('loop$') also compiles and marks that
  @('lambda') expression as guard verified.</p>

  <p>So now let's return to consideration of</p>

  @({
  (defun$ strange (x)
    (declare (xargs :guard t))
    (apply$ 'SQU (list x)))
  })

  <p>which we've seen is guard verified despite the fact that @('SQU') expects
  a natural number and will not necessarily be given one.  What happens when we
  call @('strange') on a non-natural?</p>

  @({
  ACL2 !>(strange 'NAN)

  ACL2 Error in TOP-LEVEL:  The guard for the function call (SQU N),
  which is (NATP N), is violated by the arguments in the call (SQU 'NAN).

  ACL2 !>:q

  Exiting the ACL2 read-eval-print loop.  To re-enter, execute (LP).
  ? (strange 'nan)

  ACL2 Error in ACL2-INTERFACE:  The guard for the function call (SQU N),
  which is (NATP N), is violated by the arguments in the call (SQU 'NAN).
  })

  <p>We see that we can provoke a guard violation with @('strange')
  even though it is guard verified with a guard of @('T').  Furthermore,
  we get the error both in the ACL2 read-eval-print loop and in the raw
  Lisp under ACL2.</p>

  <p>This might at first violate your understanding of the link between ACL2
  and Common Lisp.  Naively, a guard verified ACL2 function with a guard of
  @('T') never causes a runtime error in Common Lisp.  But that's not quite
  what the guarantee is.  Such a function will never cause a hard Lisp error,
  other than possibly resource errors like running out of memory or stack
  space.  Neither of the errors above were signalled by Common Lisp.  They were
  ``soft'' ACL2 errors.  In particular, when @('apply$') calls @('squ') above,
  even when running in raw Lisp, it actually calls the executable counterpart
  of @('squ'), which checks guards at runtime and executes properly under the
  ACL2 axioms.</p>

  @({
  ACL2 !>(set-guard-checking :none)

  Turning off guard checking entirely.

  ACL2 >(strange 'nan)
  0
  })

  <p>The last evaluation can be explained by the fact that ACL2 multiplication
  defaults non-numbers to 0.</p>

  <p>We discuss the evaluation of ground @('apply$') terms in the evaluation
  theory further below.</p>

  <p>When the guard conjectures of a function are proved all necessary warrants
  are assumed.  This is unlike what happens when the measure conjectures are
  proved.  The reason we can assume warrants during guard verification is that
  guard verification is relevant in the evaluation theory, where attachments
  are allowed and all warrants have true attachments.</p>

  <h3>Top-Level Evaluation of Apply$</h3>

  <p>As noted, ACL2's evaluation theory implicitly assumes all warrants
  produced by @(tsee defwarrant).  Since top-level evaluation in ACL2 is
  conducted in the evaluation theory, ground calls of @('apply$') -- whether
  literally in top-level input to the ACL2 read-eval-print loop or hidden
  inside scions called from the top-level -- can be evaluated on quoted
  warranted function symbols and @(tsee lambda$) expressions -- provided the
  @(see tame)ness restrictions are met.  This is in contrast to opportunities
  for evaluation of ground @('apply$') expressions arising in proofs, where
  warrants must be explicit.</p>

  <p>In this section we focus on calls of @('apply$') arising in the evaluation
  theory.  We start with a discussion of the use of @('apply$') with badged
  @(':logic') mode functions.  We then describe how @('apply$') handles badged
  @(':program') mode functions.</p>

  <p>Evaluation of @('apply$') terms in the evaluation theory respects guards
  on quoted function symbols and @(tsee lambda$) expressions (which is to say,
  on the quoted well-formed @('LAMBDA') objects that @(tsee lambda$) produces).
  So consider a call of @('apply$') on @('fn') and @('args') in the evaluation
  theory, where @('fn') is a badged function symbol or a well-formed (and thus
  tame) @('LAMBDA') object.  Here's what happens.  Except where noted, the
  description below applies both to badged @(':logic') and badged @(':program')
  mode functions.</p>

  <p>@('Apply$') determines whether @('fn')'s tameness restrictions are met by
  @('args').  Tameness is a syntactic property and so can be checked.  If the
  function's tameness restrictions are not met, an error is caused.</p>

  <p>If the tameness restrictions are met, @('apply$') determines whether
  @('fn') has been guard verified.  In the case of function symbols this is a
  simple lookup on the property list of @('fn').  (Of course, this check fails
  for @(':program') mode functions.)  In the case of @('LAMBDA') objects it is
  a cache query and if the query reveals that we have not yet tried to verify
  the guards of this @('LAMBDA') object, @('apply$') uses tau reasoning
  alone (see @(see introduction-to-the-tau-system)) to verify the guard
  conjectures.</p>

  <p><b>Note:</b>An important distinction between the runtime handling of
  function symbols versus @('LAMBDA') objects by @('apply$') is that function
  symbols can only be guard verified by prior events, e.g., the introductory
  @(tsee defun) or a subsequent @('verify-guards'), but @('apply$') tries to
  verify the guards of @('LAMBDA') objects <i>on the fly</i>!  The reason for
  this distinction is that we anticipate that many @('LAMBDA') objects will not
  be associated with any event.  For example, an ACL2 macro might generate a
  call of a scion on a never-before-seen @('LAMBDA') object and that
  @('LAMBDA') object may only be seen by the top-level evaluator.  We discuss
  this further in @(tsee verify-guards)</p>

  <p>If @('fn') is guard verified, @('apply$') next checks whether @('fn')'s
  guard holds of the actuals in @('args').  This is done by evaluation of the
  compiled code for the guard on @('args').</p>

  <p>If the guard check of @('args') succeeds, a compiled version of @('fn') is
  applied to @('args').  If the check fails, a guard violation is signalled or
  else the application of @('fn') to @('args') is interpreted under the
  definitional axioms of @('apply$') and @('ev$'), depending on how @(tsee
  set-guard-checking) has been configured.</p>

  <p>Finally, if @('fn') is not guard verified, the application of @('fn') to @('args')
  is interpreted under the definitional axioms of @('apply$') and @('ev$').</p>

  <p>We discuss the cache that supports @('LAMBDA') application in @(tsee
  print-cl-cache).  See also the discussion of guard verification in @(tsee
  lambda$).  It should be noted that a @('LAMBDA') object can also be guard
  verified using the @(tsee verify-guards) event.</p>

  <p>Users are accustomed to executing @(':program') mode functions at the
  top-level of the ACL2 read-eval-print loop.  Indeed, the prover itself and
  the various event commands are mostly written in @(':program') mode.
  Furthermore, the evaluation theory is described as an extension of the proof
  theory, i.e., as an axiomatic theory.  And yet no part of that axiomatization
  explains how @(':program') mode functions are run!  It simply isn't
  important.  The implementation supports it and no questions are asked.  We,
  the implementors of ACL2, view top-level evaluation of both @(':logic') mode
  and @(':program') mode functions as a convenience not affecting the
  consistency of the proof theory.  No inconsistency results from starting a
  non-terminating computation because you can never inspect the result, whereas
  if you added the corresponding definition as an axiom you might be able to
  prove something contradictory.  So we regard the seamless execution of
  @(':program') mode functions as a convenience to the user who might use them
  to inspect the ACL2 logical world, gather data, experiment with constrained
  formal models by attaching executable code to unspecified functions,
  prototype something to be formalized, etc.  In that spirit, we have arranged
  for @('apply$') to handle @(':program') mode functions <i>provided they have
  badges</i>.  Badges are critical because it means that execution of the
  functions won't ``go outside the sandbox.''  However, @('apply$') runs
  @(':program') mode functions in @(tsee safe-mode) to ensure the functional
  substitutivity of @('apply$'): identical calls must always yield identical
  results.</p>

  <h3>Logical Definitions</h3>

  <p>In the following definitions, @('apply$-userfn') is an undefined function
  that is constrained by warrants to describe the tameness requirement and
  behavior of @('apply$') on specific function symbols.  The functions
  @('untame-apply$') and @('untame-ev$') are simply undefined functions for
  giving unspecified values when untame objects are being used.</p>

  @(def apply$)

  @(def apply$-lambda)

  @(def apply$-lambda-logical)

  @(def ev$)

  @(def ev$-list)

  @(def apply$-guard)

  @(def apply$-lambda-guard)

  ")

(defxdoc apply$-guard
  :parents (apply$)
  :short "The guard on @('apply$')"
  :long "<p>The guard on @('(apply$ fn lst)') is @('(apply$-guard fn lst)') which
  is extraordinarily weak.</p>

  @(def apply$-guard)

  <p>where</p>

  @(def apply$-lambda-guard)

  <p>This guard is just strong enough to allow the definitions of the functions
  in the @('apply$') clique to be guard verified.  It does not guarantee that
  @('fn') is tame or well-formed or that @('args') satisfy the guard of
  @('fn').  The last condition is in fact impossible to state given the untyped
  nature of ACL2.  Thus, @('(apply$ fn args)') has to check tameness,
  well-formedness, guard verified, and that @('fn')'s guard is satisfied by
  @('args') when the @('apply$') is executed in the evaluation theory.</p>

  <p>The issue of guards and guard verification of definitions involving
  @('apply$') is further discussed in @(tsee apply$) and in @(tsee
  verify-guards).</p>")

(defxdoc apply$-lambda
  :parents (apply$)
  :short "Used by @('apply$') on @('LAMBDA') objects"
  :long "<p>When @('apply$') is given a @('consp') object as its first argument
  it treats it as a @('LAMBDA') expression and calls this function to apply it.
  This function evaluates the body of the object with @('ev$') under an alist
  binding the formals of the object to the actuals.  See @(tsee apply$) for
  details.</p>")

(defxdoc apply$-lambda-guard
  :parents (apply$)
  :short "The guard on @('apply$-lambda')"
  :long "<p>The guard on @('(apply$-lambda fn lst)') is @('(apply$-lambda-guard
  fn lst)') which is extraordinarily weak.</p>

  @(def apply$-lambda-guard)

  <p>This guard is just strong enough to allow the definitions of the functions
  in the @('apply$') clique to be guard verified.  It does not guarantee that
  @('fn') is @('tame') or well-formed or that @('args') satisfy the guard of
  @('fn').  The last condition is in fact impossible to state given the untyped
  nature of ACL2.  Thus, @('(apply$ fn args)') has to check tameness,
  well-formedness, guard verified and that @('fn')'s guard is satisfied by
  @('args') when the @('apply$') is executed in the evaluation theory.</p>

  <p>The issue of guards and guard verification of definitions involving
  @('apply$') is further discussed in @(tsee apply$) and in @(tsee
  verify-guards).</p>")

(defxdoc apply$-userfn
  :parents (apply$)
  :short "Undefined function used by @('apply$') on non-primitives"
  :long "<p>When @('apply$') is given a non-primitive function symbol it calls
   this function to determine the results of applying that symbol to the given
   arguments.  But this function is undefined.  In the proof theory, its value
   on a given function symbol @('fn') is specified, if at all, by the @(tsee
   warrant) for @('fn') which must be available as a hypothesis in the formula
   being proved.  In the evaluation theory, @('apply$-userfn') has an
   attachment that makes it behave as though all warrants are assumed.  See
   @(tsee apply$) for details.</p>")

(defxdoc architecture-of-the-prover
  :parents (introduction-to-the-theorem-prover)
  :short "A simple overview of how the prover works"
  :long "<p>Six built-in proof techniques are used by ACL2 to decompose the
 goal formula into subgoals.</p>

 <ul>

 <li><i>simplification</i> &mdash; decision procedures and rewriting with
 previously proved rules, but actually including a host of other techniques
 under your control.  Simplification is the only proof technique that can
 reduce a formula to 0 subgoals (i.e., prove it) rather than just transform it
 to other formulas.  The predominant activity in most proofs is simplification.
 There are many ways you can affect what the simplifier does to your formulas.
 Good users spend most of their time thinking about how to control the
 simplifier.</li>

 <li><i>destructor elimination</i> &mdash; getting rid of ``destructor terms''
 like @('(CAR X)') and @('(CDR X)') by replacing a variable, e.g., @('X'), by a
 ``constructor'' term, e.g., @('(CONS A B)').  But you can tell ACL2 about new
 destructor/constructor combinations.</li>

 <li><i>fertilization</i> &mdash; using an equivalence hypothesis by
  substituting one side for the other in the goal.  When under induction, ACL2
  may decide to restrict the substitution as follows, using its so-called
  <i>cross-fertilization</i> heuristic: substitute only into one side of the
  conclusion, thus using an inductive hypothesis in preparation for possible
  generalization in advance of another induction.  Note that
  cross-fertilization is used only when generalization is enabled: with the
  hint @(':do-not '(generalize)'), only full fertilization is applied.</li>

 <li><i>generalization</i> &mdash; replacing a term by a new variable and
 restricting the new variable to have some of the properties of the term.  You
 can control the restrictions imposed on the new variable.  This is a heuristic
 that prepares the goal for another induction.</li>

 <li><i>elimination of irrelevance</i> &mdash; throwing away unnecessary
 hypotheses.  This is a heuristic that prepares the goal for another
 induction.</li>

 <li><i>induction</i> &mdash; selecting an induction scheme to prove a formula.
 Inductions are ``suggested'' by the recursive functions appearing in the
 formula.  But you can control what inductions are suggested by terms.</li>

 </ul>

 <p>But you can add additional techniques, called <i>clause processors</i>.</p>

 <p>The various techniques are tried in turn, with simplification first and
 induction last.  Each technique reports one of three outcomes: it found
 nothing to change (i.e., the technique doesn't <i>apply</i> to that subgoal),
 it decided to abort the proof attempt (typically because there is reason to
 believe the proof is failing), or it decomposed the goal into <i>k</i>
 subgoals.</p>

 <p>The last outcome has a special case: if <i>k</i> is 0 then the technique
 proved the goal.  Whenever <i>k</i> is non-0, the process starts over again
 with simplification on each of the <i>k</i> subgoals.  However, it saves up
 all the subgoals for which induction is the only proof technique left to
 try. That way you see how it performs on every base case and induction step of
 one induction before it launches into another induction.</p>

 <p>It runs until you or one of the proof techniques aborts the proof attempt
 or until all subgoals have been proved.</p>

 <p>Note that if simplification produces a subgoal, that subgoal is
 re-simplified.  This process continues until the subgoal cannot be simplified
 further.  Only then is the next proof technique is tried.  Such subgoals are
 said to be <i>stable under simplification</i>.</p>

 <p>While this is happening, the prover prints an English narrative describing
 the process.  Basically, after each goal is printed, the system prints an
 English paragraph that names the next applicable proof technique, gives a
 brief description of what that technique does to the subgoal, and says how
 many new subgoals are produced.  Then each subgoal is dealt with in turn.</p>

 <p>If the proof is successful, you could read this log as a proof of the
 conjecture.  But output from successful proofs is generally never read because
 it is not important to The Method described in @(see
 introduction-to-the-theorem-prover).</p>

 <p>The output of an unsuccessful proof attempt concludes with some <i>key</i>
 <i>checkpoints</i> which usually bear looking at.</p>

 <p>For more information about how ACL2 orchestrates its proof techniques, see
 @(see hints-and-the-waterfall).</p>")

(defxdoc aref1
  :parents (arrays acl2-built-ins)
  :short "Access the elements of a 1-dimensional array"
  :long "@({
  Example Form:
  (aref1 'delta1 a (+ i k))

  General Form:
  (aref1 name alist index)
 })

 <p>where @('name') is a symbol, @('alist') is a 1-dimensional array and
 @('index') is a legal index into @('alist').  This function returns the value
 associated with @('index') in @('alist'), or else the default value of the
 array.  See @(see arrays) for details.</p>

 <p>This function executes in virtually constant time if @('alist') is in fact
 the ``semantic value'' associated with @('name') (see @(see arrays)).  When it
 is not, @('aref1') must do a linear search through @('alist').  In that case
 the correct answer is returned but a <b>slow array</b> comment is printed to
 the comment window.  See @(see slow-array-warning).</p>

 @(def aref1)")

(defxdoc aref2
  :parents (arrays acl2-built-ins)
  :short "Access the elements of a 2-dimensional array"
  :long "@({
  Example Form:
  (aref2 'delta1 a i j)

  General Form:
  (aref2 name alist i j)
 })

 <p>where @('name') is a symbol, @('alist') is a 2-dimensional array and @('i')
 and @('j') are legal indices into @('alist').  This function returns the value
 associated with @('(i . j)') in @('alist'), or else the default value of the
 array.  See @(see arrays) for details.</p>

 <p>This function executes in virtually constant time if @('alist') is in fact
 the ``semantic value'' associated with @('name') (see @(see arrays)).  When it
 is not, @('aref2') must do a linear search through @('alist').  In that case
 the correct answer is returned but a <b>slow array</b> comment is printed to
 the comment window.  See @(see slow-array-warning).</p>

 @(def aref2)")

(defxdoc args
  :parents (documentation)
  :short "@('args'), @(tsee guard), @('type'), @(tsee constraint), etc., of a
  function symbol"
  :long "@({
  Example:
  :args assoc-eq
 })

 <p>@('Args') takes one argument, a symbol which must be the name of a function
 or macro, and prints out some information about it including the formal
 parameters, the @(see guard) expression, the output @(see signature), the
 deduced type, the @(see constraint) (if any), and its @(tsee badge) and @(tsee
 warrant), if any.</p>")

(defxdoc arities-okp
  :parents (acl2-built-ins)
  :short "check the arities of given function symbols"
  :long "@({
  Example:
  (arities-okp '((IF . 3) (CAR . 1) (CONS . 2)) (w state))

  General Form:
  (arities-okp alist w)
 })

 <p>where @('alist') is a @(tsee symbol-alistp) and @('w') is an ACL2 logical
 @(see world).  The alist is presumed to pair @(':')@(tsee logic)-mode function
 symbols with arities.  The result is @('t') or @('nil') according to whether
 each symbol in the alist is a @(':logic')-mode function symbol with the
 associated arity as its @(tsee arity) in @('w').  See @(see
 well-formedness-guarantee).</p>

 @(def arities-okp)")

(defxdoc arity
  :parents (acl2-built-ins)
  :short "number of arguments of a function symbol"
  :long "@({
  Examples:
  (arity 'IF (w state))
  (arity '(LAMBDA (X) (CONS X X)) (w state))

  General Form:
  (arity fn w)
 })

 <p>where @('fn') is a function symbol or a lambda expression and @('w') is an
 ACL2 logical @(see world).  The result is the number of arguments the function
 or lambda expression takes, or @('nil') if the function symbol is not defined
 in @('w').</p>

 See @(see arity+) for a variant of @('arity') with a stronger @(see guard).")

(defxdoc array1p
  :parents (arrays acl2-built-ins)
  :short "Recognize a 1-dimensional array"
  :long "@({
  Example Form:
  (array1p 'delta1 a)

  General Form:
  (array1p name alist)
 })

 <p>where @('name') and @('alist') are arbitrary objects.  This function
 returns @('t') if @('alist') is a 1-dimensional ACL2 array.  Otherwise it
 returns @('nil').  The function operates in constant time if @('alist') is the
 semantic value of @('name').  See @(see arrays).</p>

 @(def array1p)")

(defxdoc array2p
  :parents (arrays acl2-built-ins)
  :short "Recognize a 2-dimensional array"
  :long "@({
  Example Form:
  (array2p 'delta1 a)

  General Form:
  (array2p name alist)
 })

 <p>where @('name') and @('alist') are arbitrary objects.  This function
 returns @('t') if @('alist') is a 2-dimensional ACL2 array.  Otherwise it
 returns @('nil').  The function operates in constant time if @('alist') is the
 semantic value of @('name').  See @(see arrays).</p>

 @(def array2p)")

(defxdoc arrays
  :parents (programming)
  :short "ACL2 arrays and operations on them"
  :long "<p>Below we begin a detailed presentation of ACL2 arrays.  ACL2's
 single-threaded objects (see @(see stobj)) provide a similar functionality
 that is generally more efficient when there are updates (writes), but is also
 more restrictive.</p>

 <p>See @(see arrays-example) for a brief introduction illustrating the use of
 ACL2 arrays.</p>

 <p>ACL2 provides relatively efficient 1- and 2-dimensional arrays.  Arrays are
 awkward to provide efficiently in an applicative language because the
 programmer rightly expects to be able to ``modify'' an array object with the
 effect of changing the behavior of the element accessing function on that
 object.  This, of course, does not make any sense in an applicative setting.
 The element accessing function is, after all, a function, and its behavior on
 a given object is immutable.  To ``modify'' an array object in an applicative
 setting we must actually produce a new array object.  Arranging for this to be
 done efficiently is a challenge to the implementors of the language.  In
 addition, the programmer accustomed to the von Neumann view of arrays must
 learn how to use immutable applicative arrays efficiently.</p>

 <p>In this note we explain 1-dimensional arrays.  In particular, we explain
 briefly how to create, access, and ``modify'' them, how they are implemented,
 and how to program with them.  2-dimensional arrays are dealt with by
 analogy.</p>

 <h3>The Logical Description of ACL2 Arrays</h3>

 <p>An ACL2 1-dimensional array is an object that associates arbitrary objects
 with certain integers, called ``indices.'' Every array has a dimension,
 @('dim'), which is a positive integer.  The indices of an array are the
 consecutive integers from @('0') through @('dim-1').  To obtain the object
 associated with the index @('i') in an array @('a'), one uses @('(aref1 name a
 i)').  @('Name') is a symbol that is irrelevant to the semantics of @(tsee
 aref1) but affects the speed with which it computes.  We will talk more about
 array ``names'' later.  To produce a new array object that is like @('a') but
 which associates @('val') with index @('i'), one uses @('(aset1 name a i
 val)').</p>

 <p>An ACL2 1-dimensional array is actually an alist.  There is no special ACL2
 function for creating arrays; they are generally built with the standard list
 processing functions @(tsee list) and @(tsee cons).  However, there is a
 special ACL2 function, called @(tsee compress1), for speeding up access to the
 elements of such an alist.  We discuss @(tsee compress1) later.</p>

 <p>One element of the alist must be the ``header'' of the array.  The @(see
 header) of a 1-dimensional array with dimension @('dim') is of the form:</p>

 @({
  (:HEADER :DIMENSIONS (dim)
           :MAXIMUM-LENGTH max
           :DEFAULT obj ; optional
           :NAME name   ; optional
           :ORDER order ; optional values are < (the default), >, or :none/nil
           ).
 })

 <p>@('Obj') may be any object and is called the ``default value'' of the
 array.  @(tsee Max) must be an integer greater than @('dim').  @('Name') must
 be a symbol.  The @(':')@(tsee default) and @(':name') entries are optional;
 if @(':')@(tsee default) is omitted, the default value is @('nil').  The
 function @(tsee header), when given a name and a 1- or 2-dimensional array,
 returns the @(see header) of the array.  The functions @(tsee dimensions),
 @(tsee maximum-length), and @(tsee default) are similar and return the
 corresponding fields of the @(see header) of the array.  The role of the
 @(':')@(tsee dimensions) field is obvious: it specifies the legal indices into
 the array.  The roles played by the @(':')@(tsee maximum-length) and
 @(':')@(tsee default) fields are described below.</p>

 <p>Aside from the @(see header), the other elements of the alist must each be
 of the form @('(i . val)'), where @('i') is an integer and @('0 <= i < dim'),
 and @('val') is an arbitrary object.</p>

 <p>The @(':order') field of the header is ignored for 2-dimensional arrays.
 For 1-dimensional arrays, it specifies the order of keys (@('i'), above) when
 the array is compressed as with @(tsee compress1), as described below.  An
 @(':order') of @(':none') or @('nil') specifies no reordering of the alist by
 @(tsee compress1), and an order of @('>') specifies reordering by @(tsee
 compress1) so that keys are in descending order.  Otherwise, the alist is
 reordered by @(tsee compress1) so that keys are in ascending order.</p>

 <p>@('(Aref1 name a i)') is @(see guard)ed so that @('name') must be a symbol,
 @('a') must be an array and @('i') must be an index into @('a').  The value of
 @('(aref1 name a i)') is either @('(cdr (assoc i a))') or else is the default
 value of @('a'), depending on whether there is a pair in @('a') whose @(tsee
 car) is @('i').  Note that @('name') is irrelevant to the value of an @(tsee
 aref1) expression.  You might @(':pe aref1') to see how simple the definition
 is.</p>

 <p>@('(Aset1 name a i val)') is @(see guard)ed analogously to the @(tsee
 aref1) expression.  The value of the @(tsee aset1) expression is essentially
 @('(cons (cons i val) a)').  Again, @('name') is irrelevant.  Note @('(aset1
 name a i val)') is an array, @('a''), with the property that @('(aref1 name a'
 i)') is @('val') and, except for index @('i'), all other indices into @('a'')
 produce the same value as in @('a').  Note also that if @('a') is viewed as an
 alist (which it is) the pair ``binding'' @('i') to its old value is in @('a'')
 but ``covered up'' by the new pair.  Thus, the length of an array grows by one
 when @(tsee aset1) is done.</p>

 <p>Because @(tsee aset1) covers old values with new ones, an array produced by
 a sequence of @(tsee aset1) calls may have many irrelevant pairs in it.  The
 function @(tsee compress1) can remove these irrelevant pairs.  Thus,
 @('(compress1 name a)') returns an array that is equivalent (vis-a-vis @(tsee
 aref1)) to @('a') but which may be shorter.  For technical reasons, the alist
 returned by @(tsee compress1) may also list the pairs in a different order
 than listed in @('a').</p>

 <p>To prevent arrays from growing excessively long due to repeated @(tsee
 aset1) operations, @(tsee aset1) essentially calls @(tsee compress1) on the
 new alist whenever the length of the new alist exceeds the @(':')@(tsee
 maximum-length) entry, @(tsee max), in the @(see header) of the array.  See
 the definition of @(tsee aset1) (for example by using @(':')@(tsee pe)).  This
 is primarily just a mechanism for freeing up @(tsee cons) space consumed while
 doing @(tsee aset1) operations.  Note however that this @(tsee compress1) call
 is replaced by a hard error if the header specifies an @(':order') of
 @(':none') or @('nil').</p>

 <p>This completes the logical description of 1-dimensional arrays.
 2-dimensional arrays are analogous.  The @(':')@(tsee dimensions) entry of the
 @(see header) of a 2-dimensional array should be @('(dim1 dim2)').  A pair of
 indices, @('i') and @('j'), is legal iff @('0 <= i < dim1') and @('0 <= j <
 dim2').  The @(':')@(tsee maximum-length) must be greater than @('dim1*dim2').
 @(tsee Aref2), @(tsee aset2), and @(tsee compress2) are like their
 counterparts but take an additional @('index') argument.  Finally, the pairs
 in a 2-dimensional array are of the form @('((i . j) . val)').</p>

 <h3>The Implementation of ACL2 Arrays</h3>

 <p>Very informally speaking, the function @(tsee compress1) ``creates'' an
 ACL2 array that provides fast access, while the function @(tsee aref1)
 ``maintains'' fast access.  We now describe this informal idea more
 carefully.</p>

 <p>@(tsee Aref1) is essentially @(tsee assoc).  If @(tsee aref1) were
 implemented naively the time taken to access an array element would be linear
 in the dimension of the array and the number of ``assignments'' to it (the
 number of @(tsee aset1) calls done to create the array from the initial
 alist).  This is intolerable; arrays are ``supposed'' to provide constant-time
 access and change.</p>

 <p>The apparently irrelevant names associated with ACL2 arrays allow us to
 provide constant-time access and change when arrays are used in
 ``conventional'' ways.  The implementation of arrays makes it clear what we
 mean by ``conventional.''</p>

 <p>Recall that array names are symbols.  Behind the scenes, ACL2 associates
 two objects with each ACL2 array name.  The first object is called the
 ``semantic value'' of the name and is an alist.  The second object is called
 the ``raw lisp array'' and is a Common Lisp array.</p>

 <p>When @('(compress1 name alist)') builds a new alist, @('a''), it sets the
 semantic value of @('name') to that new alist.  Furthermore, it writes into a
 Common Lisp array all of the index/value pairs of @('a''), initializing
 unassigned indices with the default value.  In general this is a new array,
 which becomes the raw lisp array of @('name').  However, if a raw lisp array
 is already associated with @('name') and is at least as long as the dimension
 specified in the @(see header), then that array is reused and all indices out
 of range are ignored.  (Such reuse can be avoided; see @(see flush-compress)
 for how to remove the existing association of a raw lisp array with a name.)
 Either way, @(tsee compress1) then returns @('a''), the semantic value, as its
 result, as required by the definition of @(tsee compress1).</p>

 <p>When @('(aref1 name a i)') is invoked, @(tsee aref1) first determines
 whether the semantic value of @('name') is @('a') (i.e., is @(tsee eq) to the
 alist @('a')).  If so, @(tsee aref1) can determine the @('i')th element of
 @('a') by invoking Common Lisp's @('aref') function on the raw lisp array
 associated with name.  Note that no linear search of the alist @('a') is
 required; the operation is done in constant time and involves retrieval of two
 global variables, an @(tsee eq) test and @('jump'), and a raw lisp array
 access.  In fact, an ACL2 array access of this sort is about 5 times slower
 than a C array access.  On the other hand, if @('name') has no semantic value
 or if it is different from @('a'), then @(tsee aref1) determines the answer by
 linear search of @('a') as suggested by the @('assoc-like') definition of
 @(tsee aref1).  Thus, @(tsee aref1) always returns the axiomatically specified
 result.  It returns in constant time if the array being accessed is the
 current semantic value of the name used.  The ramifications of this are
 discussed after we deal with @(tsee aset1).</p>

 <p>When @('(aset1 name a i val)') is invoked, @(tsee aset1) does two @(tsee
 cons)es to create the new array.  Call that array @('a'').  It will be
 returned as the answer.  (In this discussion we ignore the case in which
 @(tsee aset1) does a @(tsee compress1).)  However, before returning, @(tsee
 aset1) determines if @('name')'s semantic value is @('a').  If so, it makes
 the new semantic value of @('name') be @('a'') and it smashes the raw lisp
 array of @('name') with @('val') at index @('i'), before returning @('a'') as
 the result.  Thus, after doing an @(tsee aset1) and obtaining a new semantic
 value @('a''), all @(tsee aref1)s on that new array will be fast.  Any @(tsee
 aref1)s on the old semantic value, @('a'), will be slow.</p>

 <p>To understand the performance implications of this design, consider the
 chronological sequence in which ACL2 (Common Lisp) evaluates expressions:
 basically inner-most first, left-to-right, call-by-value.  An array use, such
 as @('(aref1 name a i)'), is ``fast'' (constant-time) if the alist supplied,
 @('a'), is the value returned by the most recently executed @(tsee compress1)
 or @(tsee aset1) on the name supplied.  In the functional expression of
 ``conventional'' array processing, all uses of an array are fast.</p>

 <p>The @(':name') field of the @(see header) of an array is completely
 irrelevant.  Our convention is to store in that field the symbol we mean to
 use as the name of the raw lisp array.  But no ACL2 function inspects
 @(':name') and its primary value is that it allows the user, by inspecting the
 semantic value of the array &mdash; the alist &mdash; to recall the name of
 the raw array that probably holds that value.  We say ``probably'' since there
 is no enforcement that the alist was compressed under the name in the @(see
 header) or that all @('aset')s used that name.  Such enforcement would be
 inefficient.</p>

 <h3>Some Programming Examples</h3>

 <p>In the following examples we will use ACL2 ``global variables'' to hold
 several arrays.  See @(see @), and see @(see assign).</p>

 <p>Let the @(tsee state) global variable @('a') be the 1-dimensional
 compressed array of dimension @('5') constructed below.</p>

 @({
  ACL2 !>(assign a (compress1 'demo
                              '((:header :dimensions (5)
                                         :maximum-length 15
                                         :default uninitialized
                                         :name demo)
                                (0 . zero))))
 })

 <p>Then @('(aref1 'demo (@ a) 0)') is @('zero') and @('(aref1 'demo (@ a) 1)')
 is @('uninitialized').</p>

 <p>Now execute</p>

 @({
  ACL2 !>(assign b (aset1 'demo (@ a) 1 'one))
 })

 <p>Then @('(aref1 'demo (@ b) 0)') is @('zero') and @('(aref1 'demo (@ b) 1)')
 is @('one').</p>

 <p>All of the @(tsee aref1)s done so far have been ``fast.''</p>

 <p>Note that we now have two array objects, one in the global variable @('a')
 and one in the global variable @('b').  @('B') was obtained by assigning to
 @('a').  That assignment does not affect the alist @('a') because this is an
 applicative language.  Thus, @('(aref1 'demo (@ a) 1)') must <b>still</b> be
 @('uninitialized').  And if you execute that expression in ACL2 you will see
 that indeed it is.  However, a rather ugly comment is printed, namely that
 this array access is ``slow.''  The reason it is slow is that the raw lisp
 array associated with the name @('demo') is the array we are calling @('b').
 To access the elements of @('a'), @(tsee aref1) must now do a linear search.
 Any reference to @('a') as an array is now ``unconventional;'' in a
 conventional language like Ada or Common Lisp it would simply be impossible to
 refer to the value of the array before the assignment that produced our
 @('b').</p>

 <p>Now let us define a function that counts how many times a given object,
 @('x'), occurs in an array.  For simplicity, we will pass in the name and
 highest index of the array:</p>

 @({
  ACL2 !>(defun cnt (name a i x)
           (declare (xargs :guard
                           (and (array1p name a)
                                (integerp i)
                                (>= i -1)
                                (< i (car (dimensions name a))))
                           :mode :logic
                           :measure (nfix (+ 1 i))))
           (cond ((zp (1+ i)) 0) ; return 0 if i is at most -1
                 ((equal x (aref1 name a i))
                  (1+ (cnt name a (1- i) x)))
                 (t (cnt name a (1- i) x))))
 })

 <p>To determine how many times @('zero') appears in @('(@ b)') we can
 execute:</p>

 @({
  ACL2 !>(cnt 'demo (@ b) 4 'zero)
 })

 <p>The answer is @('1').  How many times does @('uninitialized') appear in
 @('(@ b)')?</p>

 @({
  ACL2 !>(cnt 'demo (@ b) 4 'uninitialized)
 })

 <p>The answer is @('3'), because positions @('2'), @('3') and @('4') of the
 array contain that default value.</p>

 <p>Now imagine that we want to assign @(''two') to index @('2') and then count
 how many times the 2nd element of the array occurs in the array.  This
 specification is actually ambiguous.  In assigning to @('b') we produce a new
 array, which we might call @('c').  Do we mean to count the occurrences in
 @('c') of the 2nd element of @('b') or the 2nd element of @('c')?  That is, do
 we count the occurrences of @('uninitialized') or the occurrences of @('two')?
 If we mean the former the correct answer is @('2') (positions @('3') and
 @('4') are @('uninitialized') in @('c')); if we mean the latter, the correct
 answer is @('1') (there is only one occurrence of @('two') in @('c')).</p>

 <p>Below are ACL2 renderings of the two meanings, which we call @('[former]')
 and @('[latter]').  (Warning: Our description of these examples, and of an
 example @('[fast former]') that follows, assumes that only one of these three
 examples is actually executed; for example, they are not executed in sequence.
 See ``A Word of Warning'' below for more about this issue.)</p>

 @({
  (cnt 'demo (aset1 'demo (@ b) 2 'two) 4 (aref1 'demo (@ b) 2))  ; [former]

  (let ((c (aset1 'demo (@ b) 2 'two)))                           ; [latter]
    (cnt 'demo c 4 (aref1 'demo c 2)))
 })

 <p>Note that in @('[former]') we create @('c') in the second argument of the
 call to @('cnt') (although we do not give it a name) and then refer to @('b')
 in the fourth argument.  This is unconventional because the second reference
 to @('b') in @('[former]') is no longer the semantic value of @('demo').
 While ACL2 computes the correct answer, namely @('2'), the execution of the
 @(tsee aref1) expression in @('[former]') is done slowly.</p>

 <p>A conventional rendering with the same meaning is</p>

 @({
  (let ((x (aref1 'demo (@ b) 2)))                           ; [fast former]
    (cnt 'demo (aset1 'demo (@ b) 2 'two) 4 x))
 })

 <p>which fetches the 2nd element of @('b') before creating @('c') by
 assignment.  It is important to understand that @('[former]') and @('[fast
 former]') mean exactly the same thing: both count the number of occurrences of
 @('uninitialized') in @('c').  Both are legal ACL2 and both compute the same
 answer, @('2').  Indeed, we can symbolically transform @('[fast former]') into
 @('[former]') merely by substituting the binding of @('x') for @('x') in the
 body of the @(tsee let).  But @('[fast former]') can be evaluated faster than
 @('[former]') because all of the references to @('demo') use the then-current
 semantic value of @('demo'), which is @('b') in the first line and @('c')
 throughout the execution of the @('cnt') in the second line.  @('[Fast
 former]') is the preferred form, both because of its execution speed and its
 clarity.  If you were writing in a conventional language you would have to
 write something like @('[fast former]') because there is no way to refer to
 the 2nd element of the old value of @('b') after smashing @('b') unless it had
 been saved first.</p>

 <p>We turn now to @('[latter]').  It is both clear and efficient.  It creates
 @('c') by assignment to @('b') and then it fetches the 2nd element of @('c'),
 @('two'), and proceeds to count the number of occurrences in @('c').  The
 answer is @('1').  @('[Latter]') is a good example of typical ACL2 array
 manipulation: after the assignment to @('b') that creates @('c'), @('c') is
 used throughout.</p>

 <p>It takes a while to get used to this because most of us have grown
 accustomed to the peculiar semantics of arrays in conventional languages.  For
 example, in raw lisp we might have written something like the following,
 treating @('b') as a ``global variable'':</p>

 @({
  (cnt 'demo (aset 'demo b 2 'two) 4 (aref 'demo b 2))
 })

 <p>which sort of resembles @('[former]') but actually has the semantics of
 @('[latter]') because the @('b') from which @('aref') fetches the 2nd element
 is not the same @('b') used in the @('aset')!  The array @('b') is destroyed
 by the @('aset') and @('b') henceforth refers to the array produced by the
 @('aset'), as written more clearly in @('[latter]').</p>

 <p>A Word of Warning: Users must exercise care when experimenting with
 @('[former]'), @('[latter]') and @('[fast former]').  Suppose you have just
 created @('b') with the assignment shown above,</p>

 @({
  ACL2 !>(assign b (aset1 'demo (@ a) 1 'one))
 })

 <p>If you then evaluate @('[former]') in ACL2 it will complain that the @(tsee
 aref1) is slow and compute the answer, as discussed.  Then suppose you
 evaluate @('[latter]') in ACL2.  From our discussion you might expect it to
 execute fast &mdash; i.e., issue no complaint.  But in fact you will find that
 it complains repeatedly.  The problem is that the evaluation of @('[former]')
 changed the semantic value of @('demo') so that it is no longer @('b').  To
 try the experiment correctly you must make @('b') be the semantic value of
 @('demo') again before the next example is evaluated.  One way to do that is
 to execute</p>

 @({
  ACL2 !>(assign b (compress1 'demo (@ b)))
 })

 <p>before each expression.  Because of issues like this it is often hard to
 experiment with ACL2 arrays at the top-level.  We find it easier to write
 functions that use arrays correctly and efficiently than to so use them
 interactively.</p>

 <p>This last assignment also illustrates a very common use of @(tsee
 compress1).  While it was introduced as a means of removing irrelevant pairs
 from an array built up by repeated assignments, it is actually most useful as
 a way of insuring fast access to the elements of an array.</p>

 <p>Many array processing tasks can be divided into two parts.  During the
 first part the array is built.  During the second part the array is used
 extensively but not modified.  If your @(see programming) task can be so
 divided, it might be appropriate to construct the array entirely with list
 processing, thereby saving the cost of maintaining the semantic value of the
 name while few references are being made.  Once the alist has stabilized, it
 might be worthwhile to treat it as an array by calling @(tsee compress1),
 thereby gaining constant time access to it.</p>

 <p>ACL2's theorem prover uses this technique in connection with its
 implementation of the notion of whether a @(see rune) is @(see disable)d or
 not.  Associated with every @(see rune) is a unique integer @('index'), called
 its ``nume.''  When each rule is stored, the corresponding nume is stored as a
 component of the rule.  @(see Theories) are lists of @(see rune)s and
 membership in the ``current theory'' indicates that the corresponding rule is
 @(see enable)d.  But these lists are very long and membership is a linear-time
 operation.  So just before a proof begins we map the list of @(see rune)s in
 the current theory into an alist that pairs the corresponding numes with
 @('t').  Then we compress this alist into an array.  Thus, given a rule we can
 obtain its nume (because it is a component) and then determine in constant
 time whether it is @(see enable)d.  The array is never modified during the
 proof, i.e., @(tsee aset1) is never used in this example.  From the logical
 perspective this code looks quite odd: we have replaced a linear-time
 membership test with an apparently linear-time @(tsee assoc) after going to
 the trouble of mapping from a list of @(see rune)s to an alist of numes.  But
 because the alist of numes is an array, the ``apparently linear-time @(tsee
 assoc)'' is more apparent than real; the operation is constant-time.</p>")

(defxdoc arrays-example

; The transcript below was generated essentially after executing the following
; two forms:
; (set-fmt-soft-right-margin 55 state)
; (set-fmt-hard-right-margin 68 state)

  :parents (arrays)
  :short "An example illustrating ACL2 arrays"
  :long "<p>The example below illustrates the use of ACL2 arrays.  It is not,
 of course, a substitute for the detailed explanations provided elsewhere (see
 @(see arrays), including subtopics).</p>

 @({
  ACL2 !>(defun defarray (name size initial-element)
           (compress1 name
                      (cons (list :HEADER
                                  :DIMENSIONS (list size)
                                  :MAXIMUM-LENGTH (1+ size)
                                  :DEFAULT initial-element
                                  :NAME name)
                            nil)))

  Since DEFARRAY is non-recursive, its admission is trivial.  We observe
  that the type of DEFARRAY is described by the theorem
  (AND (CONSP (DEFARRAY NAME SIZE INITIAL-ELEMENT))
       (TRUE-LISTP (DEFARRAY NAME SIZE INITIAL-ELEMENT))).
  We used the :type-prescription rule COMPRESS1.

  Summary
  Form:  ( DEFUN DEFARRAY ...)
  Rules: ((:TYPE-PRESCRIPTION COMPRESS1))
  Warnings:  None
  Time:  0.02 seconds (prove: 0.00, print: 0.02, other: 0.00)
   DEFARRAY
  ACL2 !>(assign my-ar (defarray 'a1 5 17))
   ((:HEADER :DIMENSIONS (5)
             :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3)
  17
  ACL2 !>(aref1 'a1 (@ my-ar) 8)

  ACL2 Error in TOP-LEVEL:  The guard for the function symbol AREF1,
  which is
  (AND (ARRAY1P NAME L) (INTEGERP N) (>= N 0) (< N (CAR (DIMENSIONS NAME L)))),
  is violated by the arguments in the call (AREF1 'A1 '(#) 8).

  ACL2 !>(assign my-ar (aset1 'a1 (@ my-ar) 3 'xxx))
   ((3 . XXX)
    (:HEADER :DIMENSIONS (5)
             :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3)
  XXX
  ACL2 !>(aset1 'a1 (@ my-ar) 3 'yyy) ; BAD: (@ my-ar) now points to
                                      ;      an old copy of the array!
  ((3 . YYY)
   (3 . XXX)
   (:HEADER :DIMENSIONS (5)
            :MAXIMUM-LENGTH 6 :DEFAULT 17 :NAME A1))
  ACL2 !>(aref1 'a1 (@ my-ar) 3) ; Because of \"BAD\" above, the array
                                 ; access is done using assoc rather
                                 ; than Lisp aref, hence is slower;
                                 ; but the answer is still correct,
                                 ; reflecting the value in (@ my-ar),
                                 ; which was not changed above.

  **********************************************************
  Slow Array Access!  A call of AREF1 on an array named
  A1 is being executed slowly.  See :DOC slow-array-warning
  **********************************************************

  XXX
  ACL2 !>
 })")

(defxdoc aset1
  :parents (arrays acl2-built-ins)
  :short "Set the elements of a 1-dimensional array"
  :long "@({
  Example Form:
  (aset1 'delta1 a (+ i k) 27)

  General Form:
  (aset1 name alist index val)
 })

 <p>where @('name') is a symbol, @('alist') is a 1-dimensional array named
 @('name'), @('index') is a legal index into @('alist'), and @('val') is an
 arbitrary object.  See @(see arrays) for details.  Roughly speaking this
 function ``modifies'' @('alist') so that the value associated with @('index')
 is @('val').  More precisely, it returns a new array, @('alist''), of the same
 name and dimension as @('alist') that, under @(tsee aref1), is everywhere
 equal to @('alist') except at @('index') where the result is @('val').  That
 is, @('(aref1 name alist' i)') is @('(aref1 name alist i)') for all legal
 indices @('i') except @('index'), where @('(aref1 name alist' i)') is
 @('val').</p>

 <p>In order to ``modify'' @('alist'), @('aset1') @(tsee cons)es a new pair
 onto the front.  If the length of the resulting alist exceeds the @(':')@(tsee
 maximum-length) entry in the array @(see header), @('aset1') compresses the
 array as with @(tsee compress1).</p>

 <p>It is generally expected that the ``semantic value'' of @('name') will be
 @('alist') (see @(see arrays)).  This function operates in virtually constant
 time whether this condition is true or not (unless the @(tsee compress1)
 operation is required).  But the value returned by this function cannot be
 used efficiently by subsequent @('aset1') operations unless @('alist') is the
 semantic value of @('name') when @('aset1') is executed.  Thus, if the
 condition is not true, @('aset1') prints a <b>slow array</b> warning to the
 comment window.  See @(see slow-array-warning).</p>

 <p>Note that @(tsee aset1) is marked as having @(tsee invariant-risk), which
 can affect the execution of @(':')@(tsee program)-mode functions.  To get
 around this problem (but only with great care!), see @(see aset1-trusted).</p>

 @(def aset1)")

(defxdoc aset1-trusted
  :parents (arrays acl2-built-ins aset1)
  :short "Set the elements of a 1-dimensional array without @(see
 invariant-risk)"
  :long "@({
 Example Form:
 (aset1-trusted 'delta1 a (+ i k) 27)

 General Form:
 (aset1-trusted name alist index val)
 })

 <p>This utility is identical to @(tsee aset1); in fact, it has the same guard.
 The difference is that it does not carry @(see invariant-risk).  Because of
 that, functions that call @('aset1-trusted') may suffer from invariant-risk
 but not be noted by the system as carrying invariant-risk.  Therefore,
 @('aset1-trusted') it is @(see untouchable) and should be used with great
 care.  If your system consists of @(':')@(tsee logic)-mode functions, then
 there is no reason to use @('aset1-trusted'), because only @(':')@(tsee
 program)-mode functions truly carry invariant-risk.</p>

 @(def aset1-trusted)")

(defxdoc aset2
  :parents (arrays acl2-built-ins)
  :short "Set the elements of a 2-dimensional array"
  :long "@({
  Example Form:
  (aset2 'delta1 a i j 27)

  General Form:
  (aset2 name alist i j val)
 })

 <p>where @('name') is a symbol, @('alist') is a 2-dimensional array named
 @('name'), @('i') and @('j') are legal indices into @('alist'), and @('val')
 is an arbitrary object.  See @(see arrays) for details.  Roughly speaking this
 function ``modifies'' @('alist') so that the value associated with @('(i
 . j)') is @('val').  More precisely, it returns a new array, @('alist''), of
 the same name and dimension as @('alist') that, under @(tsee aref2), is
 everywhere equal to @('alist') except at @('(i . j)') where the result is
 @('val').  That is, @('(aref2 name alist' x y)') is @('(aref2 name alist x
 y)') for all legal indices @('x') @('y') except @('i') and @('j') where
 @('(aref2 name alist' i j)') is @('val').</p>

 <p>In order to ``modify'' @('alist'), @('aset2') @(tsee cons)es a new pair
 onto the front.  If the length of the resulting @('alist') exceeds the
 @(':')@(tsee maximum-length) entry in the array @(see header), @('aset2')
 compresses the array as with @(tsee compress2).</p>

 <p>It is generally expected that the ``semantic value'' of @('name') will be
 @('alist') (see @(see arrays)).  This function operates in virtually constant
 time whether this condition is true or not (unless the @(tsee compress2)
 operation is required).  But the value returned by this function cannot be
 used efficiently by subsequent @('aset2') operations unless @('alist') is the
 semantic value of @('name') when @('aset2') is executed.  Thus, if the
 condition is not true, @('aset2') prints a <b>slow array</b> warning to the
 comment window.  See @(see slow-array-warning).</p>

 @(def aset2)")

(defxdoc ash
  :parents (numbers acl2-built-ins)
  :short "Arithmetic shift operation"
  :long "<p>@('(ash i c)') is the result of taking the two's complement
 representation of the integer @('i') and shifting it by @('c') bits: shifting
 left and padding with @('c') @('0') bits if @('c') is positive, shifting right
 and dropping @('(abs c)') bits if @('c') is negative, and simply returning
 @('i') if @('c') is @('0').</p>

 <p>The @(see guard) for @('ash') requires that its arguments are integers.</p>

 <p>@('Ash') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def ash)")

(defxdoc assert$
  :parents (errors acl2-built-ins)
  :short "Cause a hard error if the given test is false"
  :long "@({
  General Form:
  (assert$ test form)
 })

 <p>where @('test') returns a single value and @('form') is arbitrary.
 Semantically, this call of @('assert$') is equivalent to @('form').  However,
 it causes a hard error if the value of @('test') is @('nil').  That hard error
 invokes the function @(tsee illegal), which has a @(see guard) that is equal
 to @('nil'); so if you use @('assert$') in code for which you verify guards,
 then a proof obligation will be that the occurrence of @('test') is never
 @('nil').</p>

 <p>See also @(tsee assert*).  Both @(tsee assert$) and @(tsee assert*) create
 a @(see guard) proof obligation (when used in a definition made in @(tsee
 logic)-mode).  However, @('assert$') checks the assertion at runtime, while
 @('assert*') does not.</p>

 <p>Also see @(see assert-event) for an assertion-checking utility that is an
 @(see event).</p>")

(defxdoc assert*
  :parents (errors acl2-built-ins)
  :short "Create a @(see guard) proof obligation that given test holds"
  :long "@({
  General Form:
  (assert* test form)
 })

 <p>where @('test') returns a single value and @('form') is arbitrary.
 Semantically, this call of @('assert*') is equivalent to @('form').  However,
 a @(see guard) proof obligation is created that @('test') holds, when used in
 a definition made in @(tsee logic)-mode.</p>

 <p>For a related utility, see @(see assert$).  Both @('assert$') and
 @('assert*') create a @(see guard) proof obligation (when used in a definition
 made in @(tsee logic)-mode).  However, @('assert$') checks the assertion at
 runtime, while @('assert*') does not.</p>

 <p>Also see @(see assert-event) for an assertion-checking utility that is an
 @(see event).</p>

 @(def assert*)")

(defxdoc assert-event
  :parents (events errors)
  :short "Assert that a given form returns a non-@('nil') value"
  :long "<p>@('Assert-event') provides a flexible way to check that evaluation
 of an expression returns a non-@('nil') value, causing an error otherwise.
 Calls of @('assert-event') are @(see event) forms; thus, they may occur in
 @(see books) as well as @(tsee encapsulate) and @(tsee progn) events.  See
 also @(see assert!) and @(see assert!-stobj) for simple interfaces to
 @('assert-event').  See @(see assert$) and @(see assert*) for
 assertion-checking utilities to use in programs.</p>

 <p>Basic calls of @('assert-event') will take just one argument, called an
 ``assertion'', which is a form that evaluates to a single value that is not a
 @(see stobj).  The following log shows a successful invocation &mdash; one
 where the assertion evaluates to a non-@('nil') value.</p>

 @({
 ACL2 !>(assert-event (equal (+ 3 4) 7))
  :PASSED
 ACL2 !>
 })

 <p>Such a use of @('assert-event') will probably suffice for most users, that
 is, where the form evaluates to a single non-stobj value and there are no
 keyword arguments.  The keyword arguments, which are optional and discussed
 below, extend that functionality, for example: multiple values are permitted
 by keyword @(':stobjs-out'), and keyword @(':on-skip-proofs') can override the
 default behavior of ignoring assertions when proofs are being skipped.</p>

 @({
 General Form:
 (assert-event assertion
               :event event           ; default nil
               ;; evaluated keyword arguments:
               :ctx                   ; default 'assert-event
               :msg msg               ; default t
               :on-skip-proofs sp     ; default nil
               :safe-mode safe-mode   ; default :same
               :stobjs-out stobjs-out ; default nil
               )
 })

 <p>where @('assertion') and @('event') are not evaluated but all the other
 arguments are evaluated, with the defaults shown above corresponding to values
 after evaluation.</p>

 <p>The following example illustrates all of the keyword arguments, which are
 documented below.</p>

 @({
 (assert-event (mv (equal (+ 3 4) 7) state)
               :event (defun f (x) (cons x x))
               :ctx '(assert-event . <some-mv>)
               :msg (msg \"Oops, I forgot what ~x0+~x1 is!\" 3 4)
               :on-skip-proofs t
               :safe-mode nil
               :stobjs-out '(nil state))
 })

 <p>@('Assert-event') is a macro whose expansion directly produces a call of
 the primitive event, @('value-triple'), where: if a call of @('assert-event')
 speifies @(':msg msg'), then the corresponding call of @('value-triple')
 specifies @(':check (or msg t)').  But unlike @('value-triple'),
 @('assert-event') can specify an event to evaluate when the assertion has
 non-@('nil') value, using the @(':event') keyword.  (You can get a sense of
 the @('value-triple') call generated from an @('assert-event') call by using
 @(':')@(tsee trans1) on the @('assert-event') form.)  The remaining keyword
 arguments of @('assert-event') are also arguments of @('value-triple').  Here
 is a brief summary of the keyword arguments, but <b>NOTE</b>: see @(see
 value-triple) for more detailed explanations of keywords other than
 @(':EVENT').</p>

 <p>@(':EVENT event') (default @('nil')): When @('event') is not @('nil'), it
 should be an @(see event), that is, a form that may be in a book or a call of
 @(tsee encapsulate) or @(tsee progn).  If the assertion evaluates to a
 non-@('nil') value (or to multiple values where the first value is not a stobj
 and is non-@('nil'); see @(':STOBJS-OUT') below), then @('event') is
 evaluated; otherwise the evaluation results in an error.</p>

 <p>@(':CTX ctx') (default: @(''assert-event')): context for error messages.</p>

 <p>@(':MSG msg') (default: @('t')): message to print when there is an
 error (equivalent to keyword argument @(':CHECK') of @(tsee
 value-triple)).</p>

 <p>@(':ON-SKIP-PROOFS sp') (default: @('nil')): supply @('t') to evaluate the
 assertion even when skipping proofs (i.e., during @(tsee include-book) or the
 second pass of an @(tsee encapsulate) event, or after invoking @(tsee
 set-ld-skip-proofsp) to skip proofs).</p>

 <p>@(':SAFE-MODE safe-mode') (default: @(':same')): provides backward
 compatibility, but is probably best ignored.</p>

 <p>@(':STOBJS-OUT stobjs-out') (default: @('nil')): specify @(':auto') to
 allow any return, even with multiple values provided the first return value is
 not a @(see stobj); or specify a list starting with @('nil'), corresponding to
 the multiple values returned, with stobjs in stobj positions and @('nil')
 elsewhere.  A @('stobjs-out') of @('nil') is treated as @('(nil)').  The first
 return value is the one checked to be non-@('nil') with one exception: when an
 @(see error-triple) @('(mv erp val state)') is returned, @('erp') must be
 @('nil') and it is @('val') that is checked to be non-@('nil').</p>")

(defxdoc assign
  :parents (programming-with-state acl2-built-ins)
  :short "Assign to a global variable in @(tsee state)"
  :long "@({
  Examples:
  (assign x (expt 2 10))
  (assign a (aset1 'ascii-map-array (@ a) 66 'Upper-case-B))

  General Form:
  (assign symbol term)
 })

 <p>where @('symbol') is any symbol (with certain enforced exclusions to avoid
 overwriting ACL2 system ``globals'') and @('term') is any ACL2 term that could
 be evaluated at the top-level.  @('Assign') evaluates the term, stores the
 result as the value of the given symbol in the @('global-table') of @(tsee
 state), and returns the result.  (Note: the actual implementation of the
 storage of this value is much more efficient than this discussion of the logic
 might suggest.)  @('Assign') is a macro that effectively expands to the more
 complicated but understandable:</p>

 @({
  (pprogn (f-put-global 'symbol term state)
          (mv nil (f-get-global 'symbol state) state)).
 })

 <p>The macro @('f-put-global') is closely related to @(tsee assign):
 @('(assign var val)') macroexpands to @('(f-put-global 'var val state)').</p>

 <p>The macro @(tsee @) gives convenient access to the value of such globals.
 The @(':')@(tsee ubt) operation has no effect on the @('global-table') of
 @(tsee state).  Thus, you may use these globals to hang onto useful data
 structures even though you may undo back past where you computed and saved
 them.</p>")

(defxdoc assoc
  :parents (alists acl2-built-ins)
  :short "Look up key in association list"
  :long "@({
  General Forms:
  (assoc x alist)
  (assoc x alist :test 'eql)   ; same as above (eql as equality test)
  (assoc x alist :test 'eq)    ; same, but eq is equality test
  (assoc x alist :test 'equal) ; same, but equal is equality test
 })

 <p>@('(Assoc x alist)') is the first member of @('alist') whose @(tsee car) is
 @('x'), or @('nil') if no such member exists.  The optional keyword,
 @(':TEST'), has no effect logically, but provides the test (default @(tsee
 eql)) used for comparing @('x') with the @(tsee car)s of successive elements
 of @('alist').</p>

 <p>The @(see guard) for a call of @('assoc') depends on the test.  In all
 cases, the second argument must satisfy @(tsee alistp).  If the test is @(tsee
 eql), then either the first argument must be suitable for @(tsee eql) (see
 @(see eqlablep)) or the second argument must satisfy @(tsee eqlable-alistp).
 If the test is @(tsee eq), then either the first argument must be a symbol or
 the second argument must satisfy @(tsee symbol-alistp).</p>

 <p>See @(see equality-variants) for a discussion of the relation between
 @('assoc') and its variants:</p>

 <blockquote><p>@('(assoc-eq x alist)') is equivalent to @('(assoc x alist
 :test 'eq)');</p>

 <p>@('(assoc-equal x alist)') is equivalent to @('(assoc x alist :test
 'equal)').</p></blockquote>

 <p>In particular, reasoning about any of these primitives reduces to reasoning
 about the function @('assoc-equal').</p>

 <p>@('Assoc') is defined by Common Lisp.  See any Common Lisp documentation
 for more information.</p>

 @(def assoc-equal)")

(defxdoc assoc-keyword
  :parents (keyword-value-listp acl2-built-ins)
  :short "Look up key in a @(tsee keyword-value-listp)"
  :long "<p>If @('l') is a list of even length of the form @('(k1 a1 k2 a2
 ... kn an)'), where each @('ki') is a keyword, then @('(assoc-keyword key l)')
 is the first tail of @('l') starting with @('key') if key is some @('ki'), and
 is @('nil') otherwise.</p>

 <p>The @(see guard) for @('(assoc-keyword key l)') is @('(keyword-value-listp
 l)').</p>

 @(def assoc-keyword)")

(defxdoc assoc-string-equal
  :parents (alists acl2-built-ins)
  :short "Look up key, a string, in association list"
  :long "<p>@('(Assoc-string-equal x alist)') is similar to @(tsee
 assoc-equal).  However, for string @('x') and alist @('alist'), the comparison
 of @('x') with successive keys in @('alist') is done using @(tsee
 string-equal) rather than @(tsee equal).</p>

 <p>The @(see guard) for @('assoc-string-equal') requires that @('x') is a
 string and @('alist') is an alist.</p>

 @(def assoc-string-equal)")

(defxdoc atom
  :parents (conses acl2-built-ins)
  :short "Recognizer for atoms"
  :long "<p>@('(atom x)') is true if and only if @('x') is an atom, i.e., not a
 @(tsee cons) pair.</p>

 <p>@('Atom') has a @(see guard) of @('t'), and is a Common Lisp function.  See
 any Common Lisp documentation for more information.</p>

 @(def atom)")

(defxdoc atom-listp
  :parents (atom lists acl2-built-ins)
  :short "Recognizer for a true list of @(see atom)s"
  :long "<p>The predicate @('atom-listp') tests whether its argument is a
 @(tsee true-listp) of @(see atom)s, i.e., of non-conses.</p>

 <p>Also see @(see good-atom-listp).</p>

 @(def atom-listp)")

(defxdoc |About Models|
  :parents (|Pages Written Especially for the Tours|)
  :short "About Models"
  :long "<p><see topic='@(url |Models of Computer Hardware and Software|)'><img
 src='res/tours/flying.gif'></img></see></p>

 <p>ACL2 is used to construct mathematical models of computer hardware and
 software (i.e., ``digital systems'').</p>

 <p><img src='res/tours/computing-machine.gif'></img></p>

 <p>A <b>mathematical model</b> is a set of mathematical formulas used to
 predict the behavior of some artifact.</p>

 <p>The use of mathematical models allows <b>faster</b> and <b>cheaper</b>
 delivery of <b>better</b> systems.</p>

 <p>Models need not be <b>complete</b> or <b>perfectly accurate</b> to be
 useful to the trained engineer.</p>

 <p>Click <see topic='@(url |Models in Engineering|)'>here</see> for more
 discussion of these assertions in an engineering context.</p>

 <p><see topic='@(url |Models of Computer Hardware and Software|)'><img
 src='res/tours/flying.gif'></img></see></p>")

(defxdoc |About Types|
  :parents (|Pages Written Especially for the Tours|)
  :short "About Types"
  :long "<p>The universe of ACL2 objects includes objects of many different
 types.  For example, @('t') is a ``symbol'' and 3 is an ``integer.''  Roughly
 speaking the objects of ACL2 can be partitioned into the following types:</p>

 <ul>

 <li><see topic='@(url |Numbers in ACL2|)'>Numbers</see> such as @('3, -22/7,
 #c(3 5/2)').</li>

 <li><see topic='@(url |ACL2 Characters|)'>Characters</see> such as @('#\\A,
 #\\a, #\\Space').</li>

 <li><see topic='@(url |ACL2 Strings|)'>Strings</see> such as @('\"This is a
 string.\"').</li>

 <li><see topic='@(url |ACL2 Symbols|)'>Symbols</see> such as @(''abc,
 'smith::abc').</li>

 <li><see topic='@(url |ACL2 Conses or Ordered Pairs|)'>Conses (or
 Ordered Pairs)</see> such as @(''((a . 1) (b . 2))').</li>

 </ul>

 <p>When proving theorems it is important to know the types of object returned
 by a term.  ACL2 uses a complicated heuristic algorithm, called @(tsee
 type-set) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>, to determine what types of objects a term may
 produce.  The user can more or less program the @('type-set') algorithm by
 proving @(tsee type-prescription) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 rules.</p>

 <p>ACL2 is an ``untyped'' logic in the sense that the syntax is not typed: It
 is legal to apply a function symbol of n arguments to any n terms, regardless
 of the types of the argument terms.  Thus, it is permitted to write such odd
 expressions as @('(+ t 3)') which sums the symbol @('t') and the integer 3.
 Common Lisp does not prohibit such expressions.  We like untyped languages
 because they are simple to describe, though proving theorems about them can be
 awkward because, unless one is careful in the way one defines or states
 things, unusual cases (like @('(+ t 3)')) can arise.</p>

 <p>To make theorem proving easier in ACL2, the axioms actually define a value
 for such terms.  The value of @('(+ t 3)') is 3; under the ACL2 axioms,
 non-numeric arguments to @('+') are treated as though they were 0.</p>

 <p>You might immediately wonder about our claim that ACL2 is Common Lisp,
 since @('(+ t 3)') is ``an error'' (and will sometimes even ``signal an
 error'') in Common Lisp.  It is to handle this problem that ACL2 has
 <b>guards</b>.  We will discuss guards later in the Walking Tour.  However,
 many new users simply ignore the issue of guards entirely and that is what we
 recommend for now.</p>

 <p>You should now return to <see topic='@(url
 |Revisiting the Admission of App|)'>the Walking Tour</see>.</p>")

(defxdoc |About the ACL2 Home Page|
  :parents (|Pages Written Especially for the Tours|)
  :short "About the ACL2 Home Page"
  :long "<p><see topic='@(url |What Is ACL2(Q)|)'><img
 src='res/tours/flying.gif'></img></see></p>

 <p>The ACL2 Home Page is integrated into the ACL2 online documentation.  Over
 4 megabytes of hypertext is available here.</p>

 <p>The vast majority of the text is user-level documentation.  For example, to
 find out about @(see rewrite) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 rules you could click on the link.  (If you do that, remember to use your
 browser's <b>Back Button</b> to come back here.)</p>

 <p>The tiny warning signs <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> mark links that lead out of the introductory-level
 material and into the user documentation.  We advise against following such
 links upon your first reading of the documentation.</p>

 <p>At the end of the tours you will have a chance to revisit them quickly to
 explore alternative paths more fully.</p>

 <p><see topic='@(url |What Is ACL2(Q)|)'><img
 src='res/tours/flying.gif'></img></see></p>")

(defxdoc |About the Admission of Recursive Definitions|
  :parents (|Pages Written Especially for the Tours|)
  :short "About the Admission of Recursive Definitions"
  :long "<p>You can't just add any formula as an axiom or definition and expect
 the logic to stay sound!  For example, if we were permitted to define @('(APP
 X Y)') so that it was equal to @('(NOT (APP X Y))') then we could prove
 anything.  The purported ``definition'' of @('APP') must have several
 properties to be admitted to the logic as a new axiom.</p>

 <p>The key property a recursive definition must have is that the recursion
 terminate.  This, along with some syntactic criteria, ensures us that there
 exists a function satisfying the definition.</p>

 <p>Termination must be proved before the definition is admitted.  This is done
 in general by finding a measure of the arguments of the function and a
 well-founded relation such that the arguments ``get smaller'' every time a
 recursive branch is taken.</p>

 <p>For @('app') the measure is the ``size'' of the first argument, @('x'), as
 determined by the primitive function @(tsee acl2-count) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>.
 The well-founded relation used in this example is @(tsee o-p) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>,
 which is the standard ordering on the ordinals less than ``epsilon naught.''
 These particular choices for @('app') were made ``automatically'' by ACL2.
 But they are in fact determined by various ``default'' settings.  The user of
 ACL2 can change the defaults or specify a ``hint'' to the @(tsee defun) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 command to specify the measure and relation.</p>

 <p>You should now return to <see topic='@(url
 |Revisiting the Admission of App|)'>the Walking Tour</see>.</p>")

(defxdoc |About the Prompt|
  :parents (|Pages Written Especially for the Tours|)
  :short "About the Prompt"
  :long "<p>The string ``@('ACL2 !>')'' is the ACL2 prompt.</p>

 <p>The prompt tells the user that an ACL2 @(see command) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> is expected.  In addition, the prompt
 tells us a little about the current state of the ACL2 command interpreter.  We
 explain the prompt briefly below.  But first we talk about the command
 interpreter.</p>

 <p>An ACL2 command is generally a Lisp expression to be evaluated.  There are
 some unusual commands (such as @(':')@(see q) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see> for
 <b>quitting</b> ACL2) which cause other behavior.  But most commands are read,
 evaluated, and then have their results printed.  Thus, we call the command
 interpreter a ``read-eval-print loop.''  The ACL2 command interpreter is named
 @(tsee LD) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> (after Lisp's ``load'').</p>

 <p>When a command is read, all the symbols in it are converted to uppercase.
 Thus, typing @('(defun app ...)') is the same as typing @('(DEFUN APP ...)')
 or @('(defun App ...)').  There are ways to force lowercase case characters
 into symbols but we won't discuss them here.  A consequence of Common Lisp's
 default uppercasing is that you'll see a general lack of concern over the case
 used when symbols are displayed in this documentation.</p>

 <p>In addition, symbols ``belong'' to ``packages'' which give the user a way
 to control namespaces.  The prompt tells us which package is the default one,
 namely @('\"ACL2\"').  That means when we call @('car'), for example, we are
 invoking the standard definition of that symbol.  If the packager were
 @('\"JONES\"') then @('car') would refer to the definition of that symbol in
 that package (which may or may not be different depending on what symbols were
 imported into that package.</p>

 <p>A command like <b>(defun app (x y) ...)</b> causes ACL2 to evaluate the
 @(see defun) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> function on <b>app</b>, <b>(x y)</b> and
 <b>...</b>.  When that command is evaluated it prints some information to the
 terminal explaining the processing of the proposed definition.  It returns the
 symbol @('APP') as its value, which is printed by the command interpreter.
 (Actually, @('defun') is not a function but a <see topic='@(url
 DEFMACRO)'>macro</see> <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> which expands to a form that involves @(tsee state)
 <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>, a necessary precondition to printing output to the
 terminal and to ``changing'' the set of axioms.  But we do not discuss this
 further here.)</p>

 <p>The @('defun') command is an example of a special kind of command called an
 ``event.''  @(see Events) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> are those commands that change the ``logical
 world'' by adding such things as axioms or theorems to ACL2's database.  See
 @(see world) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.  But not every command is an event command.</p>

 <p>A command like <b>(app '(1 2 3) '(4 5 6 7))</b> is an example of a
 non-event.  It is processed the same general way: the function <b>app</b> is
 applied to the indicated arguments and the result is printed.  The function
 <b>app</b> does not print anything and does not change the ``world.''</p>

 <p>A third kind of command is one that display information about the current
 logical world or that ``roll back'' to previous versions of the world.  Such
 commands are called ``@(see history)'' <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon src='res/tours/twarning.gif'/></see>
 commands.</p>

 <p>What does the ACL2 prompt tell us about the read-eval-print loop?  The
 prompt ``@('ACL2 !>')'' tells us that the command will be read with @(tsee
 current-package) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> set to @('\"ACL2\"'), that guard checking (see
 @(see set-guard-checking) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>) is on (``@('!')''), and that we are at the
 top-level (there is only one ``@('>')'').  For more about the prompt, see
 @(see default-print-prompt) <see
 topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see>.</p>

 <p>You should now return to <see topic='@(url
 |Revisiting the Admission of App|)'>the Walking Tour</see>.</p>")

(defxdoc |An Example Common Lisp Function Definition|
  :parents (|Pages Written Especially for the Tours|)
  :short "An Example Common Lisp Function Definition"
  :long "<p><see topic='@(url |An Example of ACL2 in Use|)'><img
 src='res/tours/walking.gif'></img></see></p>

 <p>Consider the binary trees @('x') and @('y') below.</p>

 <p><img src='res/tours/binary-trees-x-y.gif'></img></p>

 <p>In Lisp, @('x') is written as the list @(''(A B)') or, equivalently, as
 @(''(A B . NIL)').  Similarly, @('y') may be written @(''(C D E)').  Suppose
 we wish to replace the right-most tip of @('x') by the entire tree @('y').
 This is denoted @('(app x y)'), where @('app') stands for ``append''.</p>

 <p><img src='res/tours/binary-trees-app.gif'></img></p>

 <p>We can define @('app') with:</p>

 <code>
 <b>(defun app (x y)</b>                           <i>; Concatenate x and y.</i>
   <b>(declare (type (satisfies true-listp) x))</b><i>; We expect x to end in NIL.</i>
   <b>(cond ((endp x) y)</b>                       <i>; If x is empty, return y.</i>
         <b>(t (cons (car x)</b>                   <i>; Else, copy first node</i>
                  <b>(app (cdr x) y)))))</b>       <i>;  and recur into next.</i>
 </code>

 <p>If you defined this function in some Common Lisp, then to run @('app') on
 the @('x') and @('y') above you could then type</p>

 @({
  (app '(A B) '(C D E))
 })

 <p>and Common Lisp will print the result @('(A B C D E)').</p>

 <p><see topic='@(url |An Example of ACL2 in Use|)'><img
 src='res/tours/walking.gif'></img></see></p>")

(defxdoc |An Example of ACL2 in Use|
  :parents (|Pages Written Especially for the Tours|)
  :short "An Example of ACL2 in Use"
  :long "<p><see topic='@(url |How To Find Out about ACL2 Functions|)'><img
 src='res/tours/walking.gif'></img></see></p>

 <p>To introduce you to ACL2 we will consider the @('app') function discussed
 in the <see topic='@(url |Common Lisp|)'>Common Lisp</see> page, <b>except</b>
 we will omit for the moment the <b>declare</b> form, which in ACL2 is called a
 <b>guard</b>.</p>

 <p>Guards are arbitrary ACL2 terms that express the ``intended domain'' of
 functions.  In that sense, guards are akin to type signatures.  However,
 Common Lisp and ACL2 are untyped programming languages: while the language
 supports several different data types and the types of objects can be
 determined by predicates at runtime, any type of object may be passed to any
 function.  Thus, guards are ``extra-logical.''  Recognizing both the practical
 and intellectual value of knowing that your functions are applied to the kinds
 of objects you intend, ACL2 imposes guards on Common Lisp and provides a means
 of proving that functions are used as intended.  But the story is necessarily
 complicated and we do not recommend it to the new user.  Get used to the fact
 that any ACL2 function may be applied to any objects and program accordingly.
 Read about guards later.</p>

 <p>Here is the definition again</p>

 <code>
 <b>(defun app (x y)</b>
   <b>(cond ((endp x) y)</b>
         <b>(t (cons (car x) </b>
                  <b>(app (cdr x) y)))))</b>
 </code>

 <p>The next few stops along the Walking Tour will show you</p>

 <ul>

 <li>how to use the ACL2 documentation,</li>

 <li>what happens when the above definition is submitted to ACL2,</li>

 <li>what happens when you evaluate calls of @('app'),</li>

 <li>what one simple theorem about @('app') looks like,</li>

 <li>how ACL2 proves the theorem, and</li>

 <li>how that theorem can be used in another proof.</li>

 </ul>

 <p>Along the way we will talk about the <b>definitional principle</b>,
 <b>types</b>, the ACL2 <b>read-eval-print loop</b>, and how the <b>theorem
 prover</b> works.</p>

 <p>When we complete this part of the tour we will return briefly to the notion
 of <b>guards</b> and revisit several of the topics above in that context.</p>

 <p><see topic='@(url |How To Find Out about ACL2 Functions|)'><img
 src='res/tours/walking.gif'></img></see></p>")

(defxdoc |Analyzing Common Lisp Models|
  :parents (|Pages Written Especially for the Tours|)
  :short "Analyzing Common Lisp Models"
  :long "<p>To analyze a model you must be able to reason about the operations
 and relations involved.  Perhaps, for example, some aspect of the model
 depends upon the fact that the concatenation operation is associative.</p>

 <p>In any Common Lisp you can confirm that</p>

 @({
  (app '(A B) (app '(C D) '(E F)))
 })

 <p>and</p>

 @({
  (app (app '(A B) '(C D)) '(E F)))
 })

 <p>both evaluate to the same thing, @('(A B C D E F)').</p>

 <p>But what distinguishes ACL2 (the logic) from applicative Common Lisp (the
 language) is that in ACL2 you can <b>prove</b> that the concatenation function
 @('app') is associative when its arguments are true-lists, whereas in Common
 Lisp all you can do is test that proposition.</p>

 <p>That is, in ACL2 it makes sense to say that the following formula is a
 ``theorem.''</p>

 <code>
 <b>Theorem</b> Associativity of App
 (implies (and (true-listp a)
               (true-listp b))
          (equal (app (app a b) c)
                 (app a (app b c))))
 </code>

 <p>Theorems about the properties of models are proved by symbolically
 manipulating the operations and relations involved.  If the concatenation of
 sequences is involved in your model, then you may well need the theorem above
 in order to that your model has some particular property.</p>")

(defxdoc backchaining
  :parents (rule-classes)
  :short "Attempting to relieve the hypotheses of a rule"
  :long "<p>When the theorem prover attempts to apply a rule (e.g., a @(see
  rewrite) rule), it must relieve (prove) the hypotheses of that rule.  In the
  ACL2 community, this process of relieving hypotheses is called
  backchaining.</p>

  <p>There is no such thing as a backchaining or backward-chaining rule
  class (see @(see rule-classes)) in ACL2.</p>")

(defxdoc backchain-limit
  :parents (rewrite meta linear type-prescription)
  :short "Limiting the effort expended on relieving hypotheses"
  :long "<p>Before ACL2 can apply a rule with hypotheses, it must establish
 that the hypotheses are true.  (We ignore the relaxing of this requirement
 afforded by @(tsee case-split)s and @(tsee force)d hypotheses.)  ACL2
 typically establishes each hypothesis by backchaining &mdash; instantiating
 the hypothesis and then rewriting it recursively.  Here we describe how ACL2
 allows the user to limit backchaining.  At the end, below, we describe the
 function @(tsee backchain-limit).</p>

 <p>Each hypothesis of a @(tsee rewrite), @(tsee meta), @(tsee linear), or
 @(tsee type-prescription) rule is assigned a backchain-limit when the rule is
 stored.  By default, this limit is @('nil'), denoting infinity (no limit).
 However, the value used for the default may be set to a non-negative integer
 (or to @('nil')) by the user; see @(see set-default-backchain-limit).  The
 default is overridden when a @(':backchain-limit-lst') is supplied explicitly
 with the rule; see @(see rule-classes).  The number of recursive applications
 of backchaining starting with the hypothesis of a rule is limited to the
 backchain-limit associated with that hypothesis.</p>

 <p>Moreover, the user may set global backchain-limits that limit the total
 backchaining depth.  See @(see set-backchain-limit).  One limit is for the use
 of @(tsee rewrite), @(tsee meta), and @(tsee linear) rules, while the other
 limit is for so-called ``@(see type-set) reasoning'', which uses rules of
 class @(tsee type-prescription) rules.  The two limits operate independently.
 Below, we discuss the first kind of backchain limits, i.e., for other than
 @(tsee type-prescription) rules, except as otherwise indicated; but the
 mechanism for those rules is similar.</p>

 <p>Below we lay out the precise sense in which a global backchain-limit
 interacts with the backchain-limits of individual rules in order to limit
 backchaining.  But first we note that when further backchaining is disallowed,
 ACL2 can still prove a hypothesis in a given context by using that contextual
 information.  In fact, @(see type-set) reasoning may be used (except that a
 weaker version of it is used in the second case above, i.e., where we are
 already doing type-set reasoning).  Thus, the relieving of hypotheses may be
 limited to the use of contextual information (without backchaining, i.e.,
 without recursively rewriting hypotheses) by executing @(':set-backchain-limit
 0').</p>

 <p>Recall that there are two sorts of backchain limits: those applied to
 hypotheses of individual rules, as assigned by their @(':')@(tsee
 rule-classes) or else taken from the default (see @(see
 set-default-backchain-limit)); and the global limit, initially @('nil') (no
 limit) but settable with @(':')@(tsee set-backchain-limit).  Here is how these
 two types of limits interact to limit backchaining, i.e., recursive rewriting
 of hypotheses.  ACL2 maintains a current backchain limit, which is the limit
 on the depth of recursive calls to the rewriter, as well as a current
 backchain depth, which is initially 0 and is incremented each time ACL2
 backchains (and is decremented when a backchain completes).  When ACL2 begins
 to rewrite a literal (crudely, one of the ``top-level'' terms of the goal
 currently being worked on), it sets the current backchain-limit to the global
 value, which is initially @('nil') but can be set using @(':')@(tsee
 set-backchain-limit).  When ACL2 is preparing to relieve a hypothesis by
 backchaining (hence, after it has already tried type-set reasoning), it first
 makes sure that the current backchain limit is greater than the current
 backchain depth.  If not, then it refuses to relieve that hypothesis.
 Otherwise, it increments the current backchain depth and calculates a new
 current backchain-limit by taking the minimum of two values: the existing
 current backchain-limit, and the sum of the current backchain depth and the
 backchain-limit associated with the hypothesis.  Thus, ACL2 only modifies the
 current backchain-limit if it is necessary to decrease that limit in order to
 respect the backchain limit associated with the hypothesis.</p>

 <p>We illustrate with the following examples.</p>

 @({
  ; We stub out some functions so that we can reason about them.

  (defstub p0 (x) t)
  (defstub p1 (x) t)
  (defstub p2 (x) t)
  (defstub p3 (x) t)

  ; Initially, the default-backchain-limit is nil, or infinite.

  (defaxiom p2-implies-p1-limitless
    (implies (p2 x)
             (p1 x)))

  ; The following rule will have a backchain-limit of 0.

  (defaxiom p1-implies-p0-limit-0
    (implies (p1 x)
             (p0 x))
    :rule-classes ((:rewrite :backchain-limit-lst 0)))

  ; We have (p2 x) ==> (p1 x) ==> (p0 x).  We wish to establish that
  ; (p2 x) ==> (p0 x).  Normally, this would be no problem, but here
  ; we fail because ACL2 cannot establish (p0 x) by type-set reasoning
  ; alone.

  (thm
    (implies (p2 x)
             (p0 x)))

  ; We set the default-backchain-limit (for rewriting) to 1.

  :set-default-backchain-limit 1

  ; The following is more powerful than p1-implies-p0-limit-0
  ; because it can use rewrite rules to establish (p1 x).

  (defaxiom p1-implies-p0-limit-1
    (implies (p1 x)
             (p0 x)))

  ; This theorem will succeed:

  (thm
    (implies (p2 x)
             (p0 x)))

  ; We return the default-backchain-limit to its initial value.

  :set-default-backchain-limit nil

  ; Here is our last axiom.

  (defaxiom p3-implies-p2-limitless
    (implies (p3 x)
             (p2 x)))

  ; We now have (p3 x) ==> (p2 x) ==> (p1 x) ==> (p0 x).  However the
  ; rule p1-implies-p0-limit-1 has a backchain-limit of 1; hence we
  ; are not allowed to backchain far enough back to use
  ; p3-implies-p2-limitless.  We therefore lose.

  (defthm will-fail
    (implies (p3 x)
             (p0 x)))
 })

 <p>Finally, we remark that to see the current global backchain-limits, issue
 the following commands.</p>

 @({
  (backchain-limit wrld :ts) ; backchain limit for type-set reasoning
  (backchain-limit wrld :rewrite) ; backchain limit for rewriting
 })")

(defxdoc badge
  :parents (apply$)
  :short "Syntactic requirements on a function symbol to be used by @('apply$')"
  :long "<p>``Badge'' is both the name of an ACL2 function and the name of a
  concept key to the @(tsee apply$) machinery.  We discuss the function named
  @('badge') first.  The discussion also mentions the concept of warrants,
  which are easily confused with badges.  See the discussion of <b>Badges
  versus Warrants</b> at the top of @('defbadge').  But roughly put, badges
  extend the ACL2 syntax and warrants extend the proof theory.  You'll need a
  badge for @('fn') to allow the system to syntactically analyze @('(apply$ 'fn
  ...)').  You'll need a both a badge and a warrant for @('fn') if you wish to
  reason about that term with ACL2.</p>

  <p>General Form:</p>
  @({(badge fn)})

  <p>The argument, @('fn'), is expected to be a function symbol.  If @('fn') is
  one of about 800 ACL2 primitives (discussed below) or is a user-defined
  function successfully processed by either the event @(tsee defbadge) or the
  event @(tsee defwarrant), the result is an object, called the ``badge'' of
  @('fn'), which among other things specifies the @(see ilk) of each formal of
  @('fn').  Otherwise, an error is caused.  We explain below, where we define
  the concepts of the ``out arity,'' ``ilks,'' and ``tameness requirements'' of
  @('fn')'s badge.</p>

  <p>A function symbol must have a badge in order to @('apply$') the symbol and
  it is up to you, the user, to invoke an event that will assign a badge to
  your user-defined functions, if possible.  @('Defbadge') will assign a badge
  to a function symbol, if possible, and @('defwarrant') will assign both a
  badge (if the function symbol doesn't already have one) and a @(tsee
  warrant), if possible.  The macro @(tsee defun$) is just an abbreviation for
  a @('defun') followed by a @('defwarrant').  Almost all primitive system
  functions already have badges.</p>

  <p>The complete list of badged primitives can be seen by evaluating</p>

  @({
  (append '(BADGE TAMEP TAMEP-FUNCTIONP SUITABLY-TAMEP-LISTP
                  APPLY$ EV$)
          (strip-cars *badge-prim-falist*))
  })

  <p>@('Badge') is a defined function in ACL2.  You can inspect its definition
  with</p>

  @({ACL2 !>:pe badge
  })

  <p>and see that after handling the built-in symbols it defers to the
  undefined function @(tsee badge-userfn).  In the evaluation theory,
  @('badge-userfn') has an attachment that returns the badge computed by
  @('defbadge') or @('defwarrant').  But in the proof theory, @('badge-userfn')
  is undefined and the @(tsee warrant) for @('fn') specifies the badge of
  @('fn').  Thus, in the proof theory, you cannot reason about the application
  of a non-primitive function unless there is a warrant for the function
  available as a hypothesis.</p>

  <p>The rest of this documentation illustrates and explains what badges mean,
  starting with a few examples.</p>

  @({
  ACL2 !>(badge 'cons)
  (APPLY$-BADGE 2 1 . T)

  ACL2 !>(badge 'apply$)
  (APPLY$-BADGE 2 1 :FN NIL)

  ACL2 !>(badge 'foldr)
  (APPLY$-BADGE 3 1 NIL :FN NIL)
  })

  <p>The last example assumes that @('foldr') has been defined with</p>

  @({
  (defun$ foldr (lst fn init)
    (if (endp lst)
        init
        (apply$ fn
                (list (car lst)
                      (foldr (cdr lst) fn init)))))
  })

  <p>In general, badges have the form @('(APPLY$-BADGE n k . ilks)'), where
  @('n') is the arity of @('fn'), @('k') is the out arity (i.e., the number of
  results returned by @('fn')) and @('ilks') is either @('T') or a list of
  @('n') tokens.  Each token is either @(':FN'), @(':EXPR'), or @('NIL').</p>

  <p>The badge of @('fn'), if any, is computed when the event @('(defbadge
  fn)') or @('(defwarrant fn)') completes successfully.  See @(tsee defbadge)
  for a sketch of the algorithm used to compute badges.  Here though we are
  just concerned with how badges impact @('apply$').</p>

  <p>The @('ilks') of a function, @('fn'), determines the ``tameness
  requirements'' mentioned in the specification of @(tsee apply$).  When the
  @('ilks') component of @('fn')'s badge is a list, it has as many elements as
  there are formals to @('fn') and each successive element is called the
  <i>ilk</i> of the corresponding formal.  For example, given the definition of
  @('foldr') above and the badge shown for it, the first and third formals,
  @('lst') and @('init'), each have ilk @('NIL') and the second formal,
  @('fn'), has ilk @(':FN').  In the special case that @('ilks') is not a list
  it is @('T') and we just say each formal has <i>ilk</i> @('NIL') -- treating
  that @('T') as a suitably long list of @('NIL')s.</p>

  <p>Each non-@('NIL') ilk imposes a <i>tameness requirement</i> on @('(apply$
  fn args)').  If a formal has ilk @(':FN') the corresponding element of
  @('args') must satisfy @('tamep-functionp').  If a formal has ilk @(':EXPR')
  the corresponding element of @('args') must satisfy @('tamep').  Ilk @('NIL')
  imposes no requirement.  (Thus, if the @('ilks') of @('fn')'s badge is
  @('T'), as it is for @('cons') for example, there is no tameness requirement
  at all.)  See @(see tame) for a discussion of the various notions of
  tameness.</p>

  <p>Informally, if a formal's ilk is @(':FN'), the corresponding element of
  @('args') must be a tame function symbol or well-formed @('LAMBDA') object.
  If a formal's ilk is @(':EXPR'), the corresponding element of @('args') must
  be a tame expression.</p>

  <p>If a formal has ilk @(':FN') then you are allowed to put a @(tsee lambda$)
  expression in that slot.  Any quoted @('LAMBDA') object you explicitly write
  in such a slot must be well-formed (see @(see well-formed-lambda-objectp)).
  Well-formedness can be hard to achieve in quoted hand-written @('LAMBDA')
  objects; we recommend that you use @('lambda$')!  But the restrictions on
  what can occupy a @(':FN') slot are enforced when user input is translated
  into formal terms.  It is possible to circumvent these syntactic checks
  without endangering soundness: axiomatically @('apply$') puts no restrictions
  on its arguments, it just doesn't behave the way you might expect on
  ill-formed @('LAMBDA') objects.  See @(see
  gratuitous-lambda-object-restrictions).</p>

  <p><b>Clarification</b>: The careful reader will note that the formal
  requirement on a @(':FN') argument is that it must satisfy
  @('tamep-functionp').  Inspection of the definition of @('tamep-functionp')
  reveals that the argument must either be badged symbol with ilks @('T') or
  else be a tame @('LAMBDA') object.  But in the informal description above we
  said that it must be a ``tame function symbol or a <i>well-formed</i>
  @('LAMBDA') object.''  Well-formedness implies tameness but they are not the
  same.  What's going on?  The reason for this and related discrepancies in the
  documentation is that there is a tension between the logical definition of
  @('apply$') and the practical business of executing it.  The former involves
  the existence of a model, soundness, and the difficulty of proving theorems
  about @('apply$').  The latter involves the Common Lisp compiler.  We want
  the logical foundations to be simple so we -- and you -- can reason about
  @('apply$'), but the compiler imposes unavoidable and complicated
  restrictions.  The upshot is that the logical foundations assign meaning to
  @('LAMBDA') objects that cannot be compiled.  Applying merely ``tame''
  @('LAMBDA')s is slower than applying ``well-formed'' ones.  In a sense by
  acting like ``tame @('LAMBDA') objects'' and ``well-formed @('LAMBDA')
  objects'' are the same thing we're trying to trick you!  If you ever have
  occasion to formally express the restrictions on @('apply$') in some theorem,
  use @('tamep-functionp').  But when you write concrete @('LAMBDA') constants,
  try to keep them well-formed.  We try to encourage this by providing
  @(tsee lambda$), which guarantees well-formedness at translate-time, and by
  implementing full well-formedness checks -- not just tameness checks -- on
  quoted @('LAMBDA') objects in @(':FN') slots.  And we give you ways to
  circumvent these checks -- see @(see gratuitous-lambda-object-restrictions)
  -- if you really mean to.</p>")

(defxdoc badge-userfn
  :parents (apply$)
  :short "Undefined function used by @('badge') on non-primitives"
  :long "<p>When @(tsee badge) is given a non-primitive function symbol @('fn')
   it calls this function to determine the badge of @('fn').  But this function
   is undefined.  In the proof theory, its value on a given function symbol
   @('fn') is specified, if at all, by the @(tsee warrant) for @('fn') which
   must be available as a hypothesis in the formula being proved.  In the
   evaluation theory, @('badge-userfn') has an attachment that makes it behave
   as though all warrants are assumed.  See @(tsee badge) for details.</p>")

(defxdoc basics
  :parents (programming)
  :short "Basic control structures for @(see programming) like @(see if) and
 @(see cond), binding operators like @(see let) and @(see flet), multiple-value
 constructs like @(see mv), and so forth.")

(defxdoc bdd
  :parents (acl2)
  :short "Ordered binary decision diagrams with rewriting"
  :long "<p>Note.  The ACL2 bdd capability has been essentially superseded by
 GL; see @(see gl).</p>

 <p>Ordered binary decision diagrams (OBDDs, often simply called BDDs) are a
 technique, originally published by Randy Bryant, for the efficient
 simplification of Boolean expressions.  In ACL2 we combine this technique with
 rewriting to handle arbitrary ACL2 terms that can represent not only Boolean
 values, but non-Boolean values as well.  In particular, we provide a setting
 for deciding equality of bit vectors (lists of Boolean values).</p>

 <p>An introduction to BDDs for the automated reasoning community may be found
 in ``Introduction to the OBDD Algorithm for the ATP Community'' by J Moore,
 <i>Journal of Automated Reasoning</i> (1994), pp. 33&ndash;45.  (This paper
 also appears as Technical Report #84 from Computational Logic, Inc.)</p>

 <p>Further information about BDDs in ACL2 can be found in the subtopics of
 this @(see documentation) section.  In particular, see @(see bdd-introduction)
 for a good starting place that provides a number of examples.</p>

 <p>See @(see hints) for a description of @(':bdd') hints.  For quick
 reference, here is an example; but only the @(':vars') part of the hint is
 required, as explained in the documentation for @(see hints).  The values
 shown are the defaults.</p>

 @({
  (:vars nil :bdd-constructors (cons) :prove t :literal :all)
 })

 <p>We suggest that you next visit the documentation topic @(see
 BDD-INTRODUCTION).</p>")

(defxdoc bdd-algorithm
  :parents (bdd)
  :short "Summary of the BDD algorithm in ACL2"
  :long "<p>The BDD algorithm in ACL2 uses a combination of manipulation of
 @('IF') terms and unconditional rewriting.  In this discussion we begin with
 some relevant mathematical theory.  This is followed by a description of how
 ACL2 does BDDs, including concluding discussions of soundness, completeness,
 and efficiency.</p>

 <p>We recommend that you read the other documentation about BDDs in ACL2
 before reading the rather technical material that follows.  See @(see
 BDD).</p>

 <p>Here is an outline of our presentation.  Readers who want a user
 perspective, without undue mathematical theory, may wish to skip to Part (B),
 referring to Part (A) only on occasion if necessary.</p>

 <p>(A) <b>Mathematical Considerations</b></p>

 <blockquote>

 <p>(A1) BDD term order</p>

 <p>(A2) BDD-constructors and BDD terms, and their connection with aborting the
 BDD algorithm</p>

 <p>(A3) Canonical BDD terms</p>

 <p>(A4) A theorem stating the equivalence of provable and syntactic equality
 for canonical BDD terms</p>

 </blockquote>

 <p>(B) <b>Algorithmic Considerations</b></p>

 <blockquote>

 <p>(B1) BDD rules (rules used by the rewriting portion of the ACL2 BDD
 algorithm)</p>

 <p>(B2) Terms ``known to be Boolean''</p>

 <p>(B3) An ``IF-lifting'' operation used by the algorithm, as well as an
 iterative version of that operation</p>

 <p>(B4) The ACL2 BDD algorithm</p>

 <p>(B5) Soundness and Completeness of the ACL2 BDD algorithm</p>

 <p>(B6) Efficiency considerations</p></blockquote>

 <p>(A) <b>Mathematical Considerations</b></p>

 <p>(A1) <i>BDD term order</i></p>

 <p>Our BDD algorithm creates a total ``BDD term order'' on ACL2 terms, on the
 fly.  We use this order in our discussions below of IF-lifting and of
 canonical BDD terms, and in the algorithm's use of commutativity.  The
 particular order is unimportant, except that we guarantee (for purposes of
 commutative functions) that constants are smaller in this order than
 non-constants.</p>

 <p>(A2) <i>BDD-constructors</i> (assumed to be @(''(cons)')) and <i>BDD
 terms</i></p>

 <p>We take as given a list of function symbols that we call the
 ``BDD-constructors.''  By default, the only BDD-constructor is @(tsee cons),
 although it is legal to specify any list of function symbols as the
 BDD-constructors, either by using the @(see acl2-defaults-table) (see @(see
 acl2-defaults-table)) or by supplying a @(':BDD-CONSTRUCTORS') hint (see @(see
 hints)).  Warning: this capability is largely untested and may produce
 undesirable results.  Henceforth, except when explicitly stated to the
 contrary, we assume that BDD-constructors is @(''(cons)').</p>

 <p>Roughly speaking, a @(see BDD) term is the sort of @(see term) produced by
 our BDD algorithm, namely a tree with all @(tsee cons) nodes lying above all
 non-@('CONS') nodes.  More formally, a @(see term) is said to be a @(see BDD)
 term if it contains <b>no</b> subterm of either of the following forms, where
 @('f') is not @('CONS').</p>

 @({
  (f ... (CONS ...) ...)

  (f ... 'x ...)  ; where (consp x) = t
 })

 <p>We will see that whenever the BDD algorithm attempts to create a @(see
 term) that is not a @(see BDD) term, it aborts instead.  Thus, whenever the
 algorithm completes without aborting, it creates a @(see BDD) term.</p>

 <p>(A3) <i>Canonical BDD terms</i></p>

 <p>We can strengthen the notion of ``BDD term'' to a notion of ``canonical BDD
 term'' by imposing the following additional requirements, for every subterm of
 the form @('(IF x y z)'):</p>

 <blockquote><p>(a) @('x') is a variable, and it precedes (in the BDD term
 order) every variable occurring in @('y') or @('z');</p>

 <p>(b) @('y') and @('z') are syntactically distinct; and,</p>

 <p>(c) it is not the case that @('y') is @('t') and @('z') is
 @('nil').</p></blockquote>

 <p>We claim that it follows easily from our description of the BDD algorithm
 that every term it creates is a canonical BDD term, assuming that the
 variables occurring in all such terms are treated by the algorithm as being
 Boolean (see (B2) below) and that the terms contain no function symbols other
 than @('IF') and @('CONS').  Thus, under those assumptions the following
 theorem shows that the BDD algorithm never creates distinct terms that are
 provably equal, a property that is useful for completeness and efficiency (as
 we explain in (B5) and (B6) below).</p>

 <p>(A4) <i>Provably equal canonical BDD terms are identical</i></p>

 <p>We believe that the following theorem and proof are routine extensions of a
 standard result and proof to terms that allow calls of @('CONS').</p>

 <p><b>Theorem</b>.  Suppose that @('t1') and @('t2') are canonical BDD terms
 that contain no function symbols other than @('IF') and @('CONS').  Also
 suppose that @('(EQUAL t1 t2)') is a theorem.  Then @('t1') and @('t2') are
 syntactically identical.</p>

 <p>Proof of theorem: By induction on the total number of symbols occurring in
 these two terms.  First suppose that at least one term is a variable; without
 loss of generality let it be @('t1').  We must prove that @('t2') is
 syntactically the same as @('t1').  Now it is clearly consistent that
 @('(EQUAL t1 t2)') is false if @('t2') is a call of @('CONS') (to see this,
 simply let @('t1') be an value that is not a @('CONSP')).  Similarly, @('t2')
 cannot be a constant or a variable other than @('t1').  The remaining
 possibility to rule out is that @('t2') is of the form @('(IF t3 t4 t5)'),
 since by assumption its function symbol must be @('IF') or @('CONS') and we
 have already handled the latter case.  Since @('t2') is canonical, we know
 that @('t3') is a variable.  Since @('(EQUAL t1 t2)') is provable, i.e.,</p>

 @({
  (EQUAL t1 (if t3 t4 t5))
 })

 <p>is provable, it follows that we may substitute either @('t') or @('nil')
 for @('t3') into this equality to obtain two new provable equalities.  First,
 suppose that @('t1') and @('t3') are distinct variables.  Then these
 substitutions show that @('t1') is provably equal to both @('t4') and @('t5')
 (since @('t3') does not occur in @('t4') or @('t5') by property (a) above, as
 @('t2') is canonical), and hence @('t4') and @('t5') are provably equal to
 each other, which implies by the inductive hypothesis that they are the same
 term &mdash; and this contradicts the assumption that @('t2') is canonical
 (property (b)).  Therefore @('t1') and @('t3') are the same variable, i.e.,
 the equality displayed above is actually @('(EQUAL t1 (if t1 t4 t5))').
 Substituting @('t') and then @('nil') for @('t1') into this provable equality
 lets us prove @('(EQUAL t t4)') and @('(EQUAL nil t5)'), which by the
 inductive hypothesis implies that @('t4') is (syntactically) the term @('t')
 and @('t5') is @('nil').  That is, @('t2') is @('(IF t1 t nil)'), which
 contradicts the assumption that @('t2') is canonical (property (c)).</p>

 <p>Next, suppose that at least one term is a call of @('IF').  Our first
 observation is that the other term is also a call of @('IF').  For if the
 other is a call of @('CONS'), then they cannot be provably equal, because the
 former has no function symbols other than @('IF') and hence is Boolean when
 all its variables are assigned Boolean values.  Also, if the other is a
 constant, then both branches of the @('IF') term are provably equal to that
 constant and hence these branches are syntactically identical by the inductive
 hypothesis, contradicting property (b).  Hence, we may assume for this case
 that both terms are calls of @('IF'); let us write them as follows.</p>

 @({
  t0:  (IF t1 t2 t3)
  u0:  (IF u1 u2 u3)
 })

 <p>Note that @('t1') and @('u1') are variables, by property (a) of canonical
 BDD terms.  First we claim that @('t1') does not strictly precede @('u1') in
 the BDD term order.  For suppose @('t1') does strictly precede @('u1').  Then
 property (a) of canonical BDD terms guarantees that @('t1') does not occur in
 @('u0').  Hence, an argument much like one used above shows that @('u0') is
 provably equal to both @('t2') (substituting @('t') for @('t1')) and @('t3')
 (substituting @('nil') for @('t1')), and hence @('t2') and @('t3') are
 provably equal.  That implies that they are identical terms, by the inductive
 hypothesis, which then contradicts property (b) for @('t0').  Similarly,
 @('u1') does not strictly precede @('t1') in the BDD term order.  Therefore,
 @('t1') and @('u1') are the same variable.  By substituting @('t') for this
 variable we see that @('t2') and @('u2') are provably equal, and hence they
 are equal by the inductive hypothesis.  Similarly, by substituting @('nil')
 for @('t1') (and @('u1')) we see that @('t3') and @('u3') are provably, hence
 syntactically, equal.</p>

 <p>We have covered all cases in which at least one term is a variable or at
 least one term is a call of @('IF').  If both terms are constants, then
 provable and syntactic equality are clearly equivalent.  Finally, then, we may
 assume that one term is a call of @('CONS') and the other is a constant or a
 call of @('CONS').  The constant case is similar to the @('CONS') case if the
 constant is a @('CONSP'), so we omit it; while if the constant is not a
 @('CONSP') then it is not provably equal to a call of @('CONS'); in fact it is
 provably <i>not</i> equal!</p>

 <p>So, we are left with a final case, in which canonical BDD terms @('(CONS t1
 t2)') and @('(CONS u1 u2)') are provably equal, and we want to show that
 @('t1') and @('u1') are syntactically equal as are @('t2') and @('u2').  These
 conclusions are easy consequences of the inductive hypothesis, since the ACL2
 axiom @('CONS-EQUAL') (which you can inspect using @(':')@(tsee PE)) shows
 that equality of the given terms is equivalent to the conjunction of @('(EQUAL
 t1 t2)') and @('(EQUAL u1 u2)').  Q.E.D.</p>

 <p>(B) <b>Algorithmic Considerations</b></p>

 <p>(B1) <i>BDD rules</i></p>

 <p>A rule of class @(':')@(tsee rewrite) (see @(see rule-classes)) is said to
 be a ``@(see BDD) rewrite rule'' if and only if it satisfies the following
 criteria.  (1) The rule is @(see enable)d.  (2) Its @(see equivalence)
 relation is @(tsee equal).  (3) It has no hypotheses.  (4) Its @(':')@(tsee
 loop-stopper) field is @('nil'), i.e., it is not a permutative rule.  (5) All
 variables occurring in the rule occur in its left-hand side (i.e., there are
 no ``free variables''; see @(see rewrite)).  A rule of class @(':')@(tsee
 definition) (see @(see rule-classes)) is said to be a ``@(see BDD) definition
 rule'' if it satisfies all the criteria above (except (4), which does not
 apply), and moreover the top function symbol of the left-hand side was not
 recursively (or mutually recursively) defined.  Technical point: Note that
 this additional criterion is independent of whether or not the indicated
 function symbol actually occurs in the right-hand side of the rule.</p>

 <p>Both BDD rewrite rules and BDD definition rules are said to be ``BDD
 rules.''</p>

 <p>(B2) <i>Terms ''known to be Boolean''</i></p>

 <p>We apply the BDD algorithm in the context of a top-level goal to prove,
 namely, the goal at which the @(':BDD') hint is attached.  As we run the BDD
 algorithm, we allow ourselves to say that a set of @(see term)s is ``known to
 be Boolean'' if we can verify that the goal is provable from the assumption
 that at least one of the terms is not Boolean.  Equivalently, we allow
 ourselves to say that a set of terms is ``known to be Boolean'' if we can
 verify that the original goal is provably equivalent to the assertion that if
 all terms in the set are Boolean, then the goal holds.  The notion ``known to
 be Boolean'' is conservative in the sense that there are generally sets of
 terms for which the above equivalent criteria hold and yet the sets of terms
 are not noted as as being ``known to be Boolean.''  However, ACL2 uses a
 number of tricks, including @(see type-set) reasoning and analysis of the
 structure of the top-level goal, to attempt to establish that a sufficiently
 inclusive set of terms is known to be Boolean.</p>

 <p>From a practical standpoint, the algorithm determines a set of terms known
 to be Boolean; we allow ourselves to say that each term in this set is ``known
 to be Boolean.''  The algorithm assumes that these terms are indeed Boolean,
 and can make use of that assumption.  For example, if @('t1') is known to be
 Boolean then the algorithm simplifies @('(IF t1 t nil)') to @('t1'); see (iv)
 in the discussion immediately below.</p>

 <p>(B3) <i>IF-lifting</i> and the <i>IF-lifting-for-IF loop</i></p>

 <p>Suppose that one has a @(see term) of the form @('(f ... (IF test x y)
 ...)'), where @('f') is a function symbol other than @('CONS').  Then we say
 that ``IF-lifting'' @('test') ``from'' this term produces the following term,
 which is provably equal to the given term.</p>

 @({
  (if test
      (f ... x ...)  ; resulting true branch
      (f ... y ...)) ; resulting false branch
 })

 <p>Here, we replace each argument of @('f') of the form @('(IF test .. ..)'),
 for the same @('test'), in the same way.  In this case we say that
 ``IF-lifting applies to'' the given term, ``yielding the test'' @('test') and
 with the ``resulting two branches'' displayed above.  Whenever we apply
 IF-lifting, we do so for the available @('test') that is least in the BDD term
 order (see (A1) above).</p>

 <p>We consider arguments @('v') of @('f') that are ``known to be Boolean''
 (see above) to be replaced by @('(IF v t nil)') for the purposes of
 IF-lifting, i.e., before IF-lifting is applied.</p>

 <p>There is one special case, however, for IF-lifting.  Suppose that the given
 term is of the form @('(IF v y z)') where @('v') is a variable and is the test
 to be lifted out (i.e., it is least in the BDD term order among the potential
 tests).  Moreover, suppose that neither @('y') nor @('z') is of the form
 @('(IF v W1 W2)') for that same @('v').  Then IF-lifting does not apply to the
 given term.</p>

 <p>We may now describe the IF-lifting-for-IF loop, which applies to terms of
 the form @('(IF test tbr fbr)') where the algorithm has already produced
 @('test'), @('tbr'), and @('fbr').  First, if @('test') is @('nil') then we
 return @('fbr'), while if @('test') is a non-@('nil') constant or a call of
 @('CONS') then we return @('tbr').  Otherwise, we see if IF-lifting applies.
 If IF-lifting does not apply, then we return @('(IF test tbr fbr)').
 Otherwise, we apply IF-lifting to obtain a term of the form @('(IF x y z)'),
 by lifting out the appropriate test.  Now we recursively apply the
 IF-lifting-for-IF loop to the term @('(IF x y z)'), unless any of the
 following special cases apply.</p>

 <blockquote><p>(i) If @('y') and @('z') are the same term, then return
 @('y').</p>

 <p>(ii) Otherwise, if @('x') and @('z') are the same term, then replace @('z')
 by @('nil') before recursively applying IF-lifting-for-IF.</p>

 <p>(iii) Otherwise, if @('x') and @('y') are the same term and @('y') is known
 to be Boolean, then replace @('y') by @('t') before recursively applying
 IF-lifting-for-IF.</p>

 <p>(iv) If @('z') is @('nil') and either @('x') and @('y') are the same term
 or @('x') is ``known to be Boolean'' and @('y') is @('t'), then return
 @('x').</p></blockquote>

 <p>NOTE: When a variable @('x') is known to be Boolean, it is easy to see that
 the form @('(IF x t nil)') is always reduced to @('x') by this algorithm.</p>

 <p>(B4) <i>The ACL2 BDD algorithm</i></p>

 <p>We are now ready to present the BDD algorithm for ACL2.  It is given an
 ACL2 @(see term), @('x'), as well as an association list @('va') that maps
 variables to terms, including all variables occurring in @('x').  We maintain
 the invariant that whenever a variable is mapped by @('va') to a term, that
 term has already been constructed by the algorithm, except: initially @('va')
 maps every variable occurring in the top-level term to itself.  The algorithm
 proceeds as follows.  We implicitly ordain that whenever the BDD algorithm
 attempts to create a @(see term) that is not a @(see BDD) term (as defined
 above in (A2)), it aborts instead.  Thus, whenever the algorithm completes
 without aborting, it creates a @(see BDD) term.</p>

 <blockquote>

 <p>If @('x') is a variable, return the result of looking it up in @('va').</p>

 <p>If @('x') is a constant, return @('x').</p>

 <p>If @('x') is of the form @('(IF test tbr fbr)'), then first run the
 algorithm on @('test') with the given @('va') to obtain @('test'').  If
 @('test'') is @('nil'), then return the result @('fbr'') of running the
 algorithm on @('fbr') with the given @('va').  If @('test'') is a constant
 other than @('nil'), or is a call of @('CONS'), then return the result
 @('tbr'') of running the algorithm on @('tbr') with the given @('va').  If
 @('tbr') is identical to @('fbr'), return @('tbr').  Otherwise, return the
 result of applying the IF-lifting-for-IF loop (described above) to the term
 <tt>(IF test' tbr' fbr')</tt>.</p>

 <p>If @('x') is of the form @('(IF* test tbr fbr)'), then compute the result
 exactly as though @(tsee IF) were used rather than @(tsee IF*), except that if
 @('test'') is not a constant or a call of @('CONS') (see paragraph above),
 then abort the BDD computation.  Informally, the tests of @(tsee IF*) terms
 are expected to ``resolve.''  NOTE: This description shows how @(tsee IF*) can
 be used to implement conditional rewriting in the BDD algorithm.</p>

 <p>If @('x') is a @('LAMBDA') expression @('((LAMBDA vars body) . args)')
 (which often corresponds to a @(tsee LET) term; see @(see let)), then first
 form an alist @('va'') by binding each @('v') in @('vars') to the result of
 running the algorithm on the corresponding member of @('args'), with the
 current alist @('va').  Then, return the result of the algorithm on @('body')
 in the alist @('va'').</p>

 <p>Otherwise, @('x') is of the form @('(f x1 x2 ... xn)'), where @('f') is a
 function symbol other than @(tsee IF) or @(tsee IF*).  In that case, let
 @('xi'') be the result of running the algorithm on @('xi'), for @('i') from 1
 to @('n'), using the given alist @('va').  First there are a few special
 cases.  If @('f') is @(tsee EQUAL) then we return @('t') if @('x1'') is
 syntactically identical to @('x2'') (where this test is very fast; see (B6)
 below); we return @('x1'') if it is known to be Boolean and @('x2'') is
 @('t'); and similarly, we return @('x2'') if it is known to be Boolean and
 @('x1'') is @('t').  Next, if each @('xi'') is a constant and the @(':')@(tsee
 executable-counterpart) of @('f') is enabled, then the result is obtained by
 computation.  Next, if @('f') is @(tsee BOOLEANP) and @('x1'') is known to be
 Boolean, @('t') is returned.  Otherwise, we proceed as follows, first possibly
 swapping the arguments if they are out of (the BDD term) order and if @('f')
 is known to be commutative (see below).  If a BDD rewrite rule (as defined
 above) matches the term <tt>(f x1'... xn')</tt>, then the most recently stored
 such rule is applied.  If there is no such match and @('f') is a
 BDD-constructor, then we return <tt>(f x1'... xn')</tt>.  Otherwise, if a BDD
 definition rule matches this term, then the most recently stored such rule
 (which will usually be the original definition for most users) is applied.  If
 none of the above applies and neither does IF-lifting, then we return <tt>(f
 x1'... xn')</tt>.  Otherwise we apply IF-lifting to <tt>(f x1'... xn')</tt> to
 obtain a term @('(IF test tbr fbr)'); but we aren't done yet.  Rather, we run
 the BDD algorithm (using the same alist) on @('tbr') and @('fbr') to obtain
 terms @('tbr'') and @('fbr''), and we return <tt>(IF test tbr' fbr')</tt>
 unless @('tbr'') is syntactically identical to @('fbr''), in which case we
 return @('tbr'').</p></blockquote>

 <p>When is it the case that, as said above, ``@('f') is known to be
 commutative''?  This happens when an enabled rewrite rule is of the form
 @('(EQUAL (f X Y) (f Y X))').  Regarding swapping the arguments in that case:
 recall that we may assume very little about the BDD term order, essentially
 only that we swap the two arguments when the second is a constant and the
 first is not, for example, in @('(+ x 1)').  Other than that situation, one
 cannot expect to predict accurately when the arguments of commutative
 operators will be swapped.</p>

 <p>(B5) Soundness and Completeness of the ACL2 BDD algorithm</p>

 <p>Roughly speaking, ``soundness'' means that the BDD algorithm should give
 correct answers, and ``completeness'' means that it should be powerful enough
 to prove all true facts.  Let us make the soundness claim a little more
 precise, and then we'll address completeness under suitable hypotheses.</p>

 <p><b>Claim</b> (Soundness).  If the ACL2 BDD algorithm runs to completion on
 an input term @('t0'), then it produces a result that is provably equal to
 @('t0').</p>

 <p>We leave the proof of this claim to the reader.  The basic idea is simply
 to check that each step of the algorithm preserves the meaning of the term
 under the bindings in the given alist.</p>

 <p>Let us start our discussion of completeness by recalling the theorem proved
 above in (A4).</p>

 <p><b>Theorem</b>.  Suppose that @('t1') and @('t2') are canonical BDD terms
 that contain no function symbols other than @('IF') and @('CONS').  Also
 suppose that @('(EQUAL t1 t2)') is a theorem.  Then @('t1') and @('t2') are
 syntactically identical.</p>

 <p>Below we show how this theorem implies the following completeness property
 of the ACL2 BDD algorithm.  We continue to assume that @('CONS') is the only
 BDD-constructor.</p>

 <p><b>Claim</b> (Completeness).  Suppose that @('t1') and @('t2') are provably
 equal terms, under the assumption that all their variables are known to be
 Boolean.  Assume further that under this same assumption, top-level runs of
 the ACL2 BDD algorithm on these terms return terms that contain only the
 function symbols @('IF') and @('CONS').  Then the algorithm returns the same
 term for both @('t1') and @('t2'), and the algorithm reduces @('(EQUAL t1
 t2)') to @('t').</p>

 <p>Why is this claim true?  First, notice that the second part of the
 conclusion follows immediately from the first, by definition of the algorithm.
 Next, notice that the terms @('u1') and @('u2') obtained by running the
 algorithm on @('t1') and @('t2'), respectively, are provably equal to @('t1')
 and @('t2'), respectively, by the Soundness Claim.  It follows that @('u1')
 and @('u2') are provably equal to each other.  Since these terms contain no
 function symbols other than @('IF') or @('CONS'), by hypothesis, the Claim now
 follows from the Theorem above together with the following lemma.</p>

 <p><b>Lemma</b>.  Suppose that the result of running the ACL2 BDD algorithm on
 a top-level term @('t0') is a term @('u0') that contains only the function
 symbols @('IF') and @('CONS'), where all variables of @('t0') are known to be
 Boolean.  Then @('u0') is a canonical BDD term.</p>

 <p>Proof: left to the reader.  Simply follow the definition of the algorithm,
 with a separate argument for the IF-lifting-for-IF loop.</p>

 <p>Finally, let us remark on the assumptions of the Completeness Claim above.
 The assumption that all variables are known to be Boolean is often true; in
 fact, the system uses the forward-chaining rule @('boolean-listp-forward')
 (you can see it using @(':')@(tsee pe)) to try to establish this assumption,
 if your theorem has a form such as the following.</p>

 @({
  (let ((x (list x0 x1 ...))
        (y (list y0 y1 ...)))
    (implies (and (boolean-listp x)
                  (boolean-listp y))
             ...))
 })

 <p>Moreover, the @(':BDD') hint can be used to force the prover to abort if it
 cannot check that the indicated variables are known to be Boolean; see @(see
 hints).</p>

 <p>Finally, consider the effect in practice of the assumption that the terms
 resulting from application of the algorithm contain calls of @('IF') and
 @('CONS') only.  Typical use of BDDs in ACL2 takes place in a theory (see
 @(see theories)) in which all relevant non-recursive function symbols are
 enabled and all recursive function symbols possess enabled BDD rewrite rules
 that tell them how open up.  For example, such a rule may say how to expand on
 a given function call's argument that has the form @('(CONS a x)'), while
 another may say how to expand when that argument is @('nil')).  (See for
 example the rules @('append-cons') and @('append-nil') in the documentation
 for @(tsee IF*).)  We leave it to future work to formulate a theorem that
 guarantees that the BDD algorithm produces terms containing calls only of
 @('IF') and @('CONS') assuming a suitably ``complete'' collection of rewrite
 rules.</p>

 <p>(B6) <i>Efficiency considerations</i></p>

 <p>Following Bryant's algorithm, we use a graph representation of @(see term)s
 created by the BDD algorithm's computation.  This representation enjoys some
 important properties.</p>

 <blockquote>

 <p>(Time efficiency) The test for syntactic equality of BDD terms is very
 fast.</p>

 <p>(Space efficiency) Equal BDD data structures are stored identically in
 memory.</p></blockquote>

 <p><i>Implementation note.</i> The representation actually uses a sort of hash
 table for BDD terms that is implemented as an ACL2 1-dimensional array.  See
 @(see arrays).  In addition, we use a second such hash table to avoid
 recomputing the result of applying a function symbol to the result of running
 the algorithm on its arguments.  We believe that these uses of hash tables are
 standard.  They are also discussed in Moore's paper on BDDs; see @(see bdd)
 for the reference.</p>")

(defxdoc bdd-introduction
  :parents (bdd)
  :short "Examples illustrating the use of BDDs in ACL2"
  :long "<p>See @(see bdd) for a brief introduction to BDDs in ACL2 and for
 pointers to other documentation on BDDs in ACL2.  Here, we illustrate the use
 of BDDs in ACL2 by way of some examples.  For a further example, see @(see
 if*).</p>

 <p>Let us begin with a really simple example.  (We will explain the @(':bdd')
 hint @('(:vars nil)') below.)</p>

 @({
  ACL2 !>(thm (equal (if a b c) (if (not a) c b))
              :hints ((\"Goal\" :bdd (:vars nil)))) ; Prove with BDDs

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  But simplification with BDDs (7 nodes) reduces this to T, using the
  :definitions EQUAL and NOT.

  Q.E.D.

  Summary
  Form:  ( THM ...)
  Rules: ((:DEFINITION EQUAL) (:DEFINITION NOT))
  Warnings:  None
  Time:  0.18 seconds (prove: 0.05, print: 0.02, other: 0.12)

  Proof succeeded.
  ACL2 !>
 })

 <p>The @(':bdd') hint @('(:vars nil)') indicates that BDDs are to be used on
 the indicated goal, and that any so-called ``variable ordering'' may be used:
 ACL2 may use a convenient order that is far from optimal.  It is beyond the
 scope of the present documentation to address the issue of how the user may
 choose good variable orderings.  Someday our implementation of BDDs may be
 improved to include heuristically-chosen variable orderings rather than rather
 random ones.</p>

 <p>Here is a more interesting example.</p>

 @({
  (defun v-not (x)
  ; Complement every element of a list of Booleans.
    (if (consp x)
        (cons (not (car x)) (v-not (cdr x)))
      nil))

  ; Now we prove a rewrite rule that explains how to open up v-not on
  ; a consp.
  (defthm v-not-cons
    (equal (v-not (cons x y))
           (cons (not x) (v-not y))))

  ; Finally, we prove for 7-bit lists that v-not is self-inverting.
  (thm
   (let ((x (list x0 x1 x2 x3 x4 x5 x6)))
     (implies (boolean-listp x)
              (equal (v-not (v-not x)) x)))
   :hints ((\"Goal\" :bdd
                   ;; Note that this time we specify a variable order.
                   (:vars (x0 x1 x2 x3 x4 x5 x6)))))
 })

 <p>It turns out that the variable order doesn't seem to matter in this
 example; using several orders we found that 30 nodes were created, and the
 proof time was about 1/10 of a second on a (somewhat enhanced) Sparc 2.  The
 same proof took about a minute and a half without any @(':bdd') hint!  This
 observation is a bit misleading perhaps, since the theorem for arbitrary
 @('x'),</p>

 @({
  (thm
   (implies (boolean-listp x)
            (equal (v-not (v-not x)) x)))
 })

 <p>only takes about 1.5 times as long as the @(':bdd') proof for 7 bits,
 above!  Nevertheless, BDDs can be very useful in reducing proof time,
 especially when there is no regular structure to facilitate proof by
 induction, or when the induction scheme is so complicated to construct that
 significant user effort is required to get the proof by induction to go
 through.</p>

 <p>Finally, consider the preceding example, with a @(':bdd') hint of (say)
 @('(:vars nil)'), but with the rewrite rule @('v-not-cons') above disabled.
 In that case, the proof fails, as we see below.  That is because the BDD
 algorithm in ACL2 uses hypothesis-free @(':')@(see rewrite) rules,
 @(':')@(tsee executable-counterpart)@('s'), and nonrecursive definitions, but
 it does not use recursive definitions.</p>

 <p>Notice that when we issue the @('(show-bdd)') command, the system's
 response clearly shows that we need a rewrite rule for simplifying terms of
 the form @('(v-not (cons ...))').</p>

 @({
  ACL2 !>(thm
          (let ((x (list x0 x1 x2 x3 x4 x5 x6)))
            (implies (boolean-listp x)
                     (equal (v-not (v-not x)) x)))
          :hints ((\"Goal\" :bdd (:vars nil)
                   :in-theory (disable v-not-cons))))

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  ACL2 Error in ( THM ...):  Attempted to create V-NOT node during BDD
  processing with an argument that is a call of a bdd-constructor,
  which would produce a non-BDD term (as defined in :DOC
  bdd-algorithm).  See :DOC show-bdd.

  Summary
  Form:  ( THM ...)
  Rules: NIL
  Warnings:  None
  Time:  0.58 seconds (prove: 0.13, print: 0.00, other: 0.45)

  ******** FAILED ********  See :DOC failure  ******** FAILED ********
  ACL2 !>(show-bdd)

  BDD computation on Goal yielded 17 nodes.
  ------------------------------

  BDD computation was aborted on Goal, and hence there is no
  falsifying assignment that can be constructed.  Here is a backtrace
  of calls, starting with the top-level call and ending with the one
  that led to the abort.  See :DOC show-bdd.

  (LET ((X (LIST X0 X1 X2 X3 X4 X5 ...)))
       (IMPLIES (BOOLEAN-LISTP X)
                (EQUAL (V-NOT (V-NOT X)) X)))
    alist: ((X6 X6) (X5 X5) (X4 X4) (X3 X3) (X2 X2) (X1 X1) (X0 X0))

  (EQUAL (V-NOT (V-NOT X)) X)
    alist: ((X (LIST X0 X1 X2 X3 X4 X5 ...)))

  (V-NOT (V-NOT X))
    alist: ((X (LIST X0 X1 X2 X3 X4 X5 ...)))

  (V-NOT X)
    alist: ((X (LIST X0 X1 X2 X3 X4 X5 ...)))
  ACL2 !>
 })

 <p>The term that has caused the BDD algorithm to abort is thus @('(V-NOT X)'),
 where @('X') has the value @('(LIST X0 X1 X2 X3 X4 X5 ...)'), i.e., @('(CONS
 X0 (LIST X1 X2 X3 X4 X5 ...))').  Thus, we see the utility of introducing a
 rewrite rule to simplify terms of the form @('(V-NOT (CONS ...))').  The moral
 of this story is that if you get an error of the sort shown above, you may
 find it useful to execute the command @('(show-bdd)') and use the result as
 advice that suggests the left hand side of a rewrite rule.</p>

 <p>Here is another sort of failed proof.  In this version we have omitted the
 hypothesis that the input is a bit vector.  Below we use @('show-bdd') to see
 what went wrong, and use the resulting information to construct a
 counterexample.  This failed proof corresponds to a slightly modified input
 theorem, in which @('x') is bound to the 4-element list @('(list x0 x1 x2
 x3)').</p>

 @({
  ACL2 !>(thm
          (let ((x (list x0 x1 x2 x3)))
            (equal (v-not (v-not x)) x))
          :hints ((\"Goal\" :bdd
                   ;; This time we do not specify a variable order.
                   (:vars nil))))

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  ACL2 Error in ( THM ...):  The :BDD hint for the current goal has
  successfully simplified this goal, but has failed to prove it.
  Consider using (SHOW-BDD) to suggest a counterexample; see :DOC
  show-bdd.

  Summary
  Form:  ( THM ...)
  Rules: NIL
  Warnings:  None
  Time:  0.18 seconds (prove: 0.07, print: 0.00, other: 0.12)

  ******** FAILED ********  See :DOC failure  ******** FAILED ********
  ACL2 !>(show-bdd)

  BDD computation on Goal yielded 73 nodes.
  ------------------------------

  Falsifying constraints:
  ((X0 \"Some non-nil value\")
   (X1 \"Some non-nil value\")
   (X2 \"Some non-nil value\")
   (X3 \"Some non-nil value\")
   ((EQUAL 'T X0) T)
   ((EQUAL 'T X1) T)
   ((EQUAL 'T X2) T)
   ((EQUAL 'T X3) NIL))

  ------------------------------

  Term obtained from BDD computation on Goal:

  (IF X0
      (IF X1
          (IF X2 (IF X3 (IF # # #) (IF X3 # #))
              (IF X2 'NIL (IF X3 # #)))
          (IF X1 'NIL
              (IF X2 (IF X3 # #) (IF X2 # #))))
      (IF X0 'NIL
          (IF X1 (IF X2 (IF X3 # #) (IF X2 # #))
              (IF X1 'NIL (IF X2 # #)))))

  ACL2 Query (:SHOW-BDD):  Print the term in full?  (N, Y, W or ?):
  n ; I've seen enough.  The assignment shown above suggests
    ; (though not conclusively) that if we bind x3 to a non-nil
    ; value other than T, and bind x0, x1, and x2 to t, then we
    ; this may give us a counterexample.
  ACL2 !>(let ((x0 t) (x1 t) (x2 t) (x3 7))
           (let ((x (list x0 x1 x2 x3)))
             ;; Let's use LIST instead of EQUAL to see how the two
             ;; lists differ.
             (list (v-not (v-not x)) x)))
  ((T T T T) (T T T 7))
  ACL2 !>
 })

 <p>See @(see if*) for another example.</p>")

(defxdoc bibliography
  :parents (about-acl2)
  :short "Reports about ACL2"
  :long "<p>The ACL2 home page includes a <a
  href='http://www.cs.utexas.edu/users/moore/publications/acl2-papers.html'>list
  of notes and reports about ACL2</a>.</p>")

(defxdoc binary-*
  :parents (* acl2-built-ins)
  :short "Multiplication function"
  :long "<p>Completion Axiom (@('completion-of-*')):</p>

 @({
  (equal (binary-* x y)
         (if (acl2-numberp x)
             (if (acl2-numberp y)
                 (binary-* x y)
               0)
           0))
 })

 <p>@(see Guard) for @('(binary-* x y)'):</p>

 @({
  (and (acl2-numberp x) (acl2-numberp y))
 })

 <p>Notice that like all arithmetic functions, @('binary-*') treats non-numeric
 inputs as @('0').</p>

 <p>Calls of the macro @(tsee *) expand to calls of @('binary-*'); see @(see
 *).</p>")

(defxdoc binary-+
  :parents (+ acl2-built-ins)
  :short "Addition function"
  :long "<p>Completion Axiom (@('completion-of-+')):</p>

 @({
  (equal (binary-+ x y)
         (if (acl2-numberp x)
             (if (acl2-numberp y)
                 (binary-+ x y)
               x)
           (if (acl2-numberp y)
               y
             0)))
 })

 <p>@(see Guard) for @('(binary-+ x y)'):</p>

 @({
  (and (acl2-numberp x) (acl2-numberp y))
 })

 <p>Notice that like all arithmetic functions, @('binary-+') treats non-numeric
 inputs as @('0'). Thus, the following are theorems.</p>

 @({
  (thm (equal (+ (fix x) y) (+ x y)))
  (thm (equal (+ x (fix y)) (+ x y)))
 })

 <p>Calls of the macro @(tsee +) expand to calls of @('binary-+'); see @(see
 +).</p>")

(defxdoc binary-append
  :parents (append acl2-built-ins)
  :short "@(see concatenate) two lists"
  :long "<p>This binary function implements @(tsee append), which is a macro in
 ACL2.  See @(see append)</p>

 <p>The @(see guard) for @('binary-append') requires the first argument to be a
 @(tsee true-listp).</p>

 @(def binary-append)")

(defxdoc bind-free
  :parents (rewrite linear definition)
  :short "To bind free variables of a rewrite, definition, or linear rule"
  :long "@({
  Examples:
  (IMPLIES (AND (RATIONALP LHS)
                (RATIONALP RHS)
                (BIND-FREE (FIND-MATCH-IN-PLUS-NESTS LHS RHS) (X)))
           (EQUAL (EQUAL LHS RHS)
                  (EQUAL (+ (- X) LHS) (+ (- X) RHS))))

  (IMPLIES (AND (BIND-FREE
                  (FIND-RATIONAL-MATCH-IN-TIMES-NESTS LHS RHS MFC STATE)
                  (X))
                (RATIONALP X)
                (CASE-SPLIT (NOT (EQUAL X 0))))
           (EQUAL (< LHS RHS)
                  (IF (< 0 X)
                      (< (* (/ X) LHS) (* (/ X) RHS))
                     (< (* (/ X) RHS) (* (/ X) LHS)))))
 })

 <p>General Forms:</p>

 @({
  (BIND-FREE term var-list)
  (BIND-FREE term t)
  (BIND-FREE term)
 })

 <p>A rule which uses a @('bind-free') hypothesis has similarities to both a
 rule which uses a @(tsee syntaxp) hypothesis and to a @(':')@(tsee meta) rule.
 @('Bind-free') is like @(tsee syntaxp), in that it logically always returns
 @('t') but may affect the application of a @(':')@(tsee rewrite), @(':')@(tsee
 definition), or @(':')@(tsee linear) rule when it is called at the top-level
 of a hypothesis.  It is like a @(':')@(tsee meta) rule, in that it allows the
 user to perform transformations of terms under programmatic control.</p>

 <p>Note that a @('bind-free') hypothesis does not, in general, deal with the
 meaning or semantics or values of the terms, but rather with their syntactic
 forms.  Before attempting to write a rule which uses @('bind-free'), the user
 should be familiar with @(tsee syntaxp) and the internal form that ACL2 uses
 for terms.  This internal form is similar to what the user sees, but there are
 subtle and important differences.  @(tsee Trans) can be used to view this
 internal form.</p>

 <p>Just as for a @(tsee syntaxp) hypothesis, there are two basic types of
 @('bind-free') hypotheses.  The simpler type of @('bind-free') hypothesis may
 be used as the nth hypothesis in a @(':')@(tsee rewrite), @(':')@(tsee
 definition), or @(':')@(tsee linear) rule whose @(':')@(tsee corollary) is
 @('(implies (and hyp1 ... hypn ... hypk) (equiv lhs rhs))') provided @('term')
 is a term, @('term') contains at least one variable, and every variable
 occurring freely in @('term') occurs freely in @('lhs') or in some @('hypi'),
 @('i<n').  In addition, @('term') must not use any stobjs.  Later below we
 will describe the second type, an <i>extended</i> @('bind-free') hypothesis,
 which is similar except that it may use @(tsee state) and @(tsee mfc).
 Whether simple or extended, a @('bind-free') hypothesis may return an alist
 that binds free variables, as explained below, or it may return a list of such
 alists.  We focus on the first of these cases: return of a single binding
 alist.  We conclude our discussion with a section that covers the other case:
 return of a list of alists.</p>

 <p>We begin our description of @('bind-free') by examining the first example
 above in some detail.</p>

 <p>We wish to write a rule which will cancel ``like'' addends from both sides
 of an equality.  Clearly, one could write a series of rules such as</p>

 @({
  (DEFTHM THE-HARD-WAY-2-1
     (EQUAL (EQUAL (+ A X B)
                   (+ X C))
            (EQUAL (+ A B)
                   (FIX C))))
 })

 <p>with one rule for each combination of positions the matching addends might
 be found in (if one knew before-hand the maximum number of addends that would
 appear in a sum).  But there is a better way.  (In what follows, we assume the
 presence of an appropriate set of rules for simplifying sums.)</p>

 <p>Consider the following definitions and theorem:</p>

 @({
  (DEFUN INTERSECTION-EQUAL (X Y)
    (COND ((ENDP X)
           NIL)
          ((MEMBER-EQUAL (CAR X) Y)
           (CONS (CAR X) (INTERSECTION-EQUAL (CDR X) Y)))
          (T
           (INTERSECTION-EQUAL (CDR X) Y))))

  (DEFUN PLUS-LEAVES (TERM)
    (IF (EQ (FN-SYMB TERM) 'BINARY-+)
        (CONS (FARGN TERM 1)
              (PLUS-LEAVES (FARGN TERM 2)))
      (LIST TERM)))

  (DEFUN FIND-MATCH-IN-PLUS-NESTS (LHS RHS)
    (IF (AND (EQ (FN-SYMB LHS) 'BINARY-+)
             (EQ (FN-SYMB RHS) 'BINARY-+))
        (LET ((COMMON-ADDENDS (INTERSECTION-EQUAL (PLUS-LEAVES LHS)
                                                  (PLUS-LEAVES RHS))))
          (IF COMMON-ADDENDS
              (LIST (CONS 'X (CAR COMMON-ADDENDS)))
            NIL))
      NIL))

  (DEFTHM CANCEL-MATCHING-ADDENDS-EQUAL
    (IMPLIES (AND (RATIONALP LHS)
                  (RATIONALP RHS)
                  (BIND-FREE (FIND-MATCH-IN-PLUS-NESTS LHS RHS) (X)))
             (EQUAL (EQUAL LHS RHS)
                    (EQUAL (+ (- X) LHS) (+ (- X) RHS)))))
 })

 <p>How is this rule applied to the following term?</p>

 @({
  (equal (+ 3 (expt a n) (foo a c))
         (+ (bar b) (expt a n)))
 })

 <p>As mentioned above, the internal form of an ACL2 term is not always what
 one sees printed out by ACL2.  In this case, by using @(':')@(tsee trans) one
 can see that the term is stored internally as</p>

 @({
  (equal (binary-+ '3
                   (binary-+ (expt a n) (foo a c)))
         (binary-+ (bar b) (expt a n))).
 })

 <p>When ACL2 attempts to apply @('cancel-matching-addends-equal') to the term
 under discussion, it first forms a substitution that instantiates the
 left-hand side of the conclusion so that it is identical to the target term.
 This substitution is kept track of by the substitution alist:</p>

 @({
  ((LHS . (binary-+ '3
                     (binary-+ (expt a n) (foo a c))))
   (RHS . (binary-+ (bar b) (expt a n)))).
 })

 <p>ACL2 then attempts to relieve the hypotheses in the order they were given.
 Ordinarily this means that we instantiate each hypothesis with our
 substitution and then attempt to rewrite the resulting instance to true.
 Thus, in order to relieve the first hypothesis, we rewrite:</p>

 @({
  (RATIONALP (binary-+ '3
                        (binary-+ (expt a n) (foo a c)))).
 })

 <p>Let us assume that the first two hypotheses rewrite to @('t').  How do we
 relieve the @('bind-free') hypothesis?  Just as for a @(tsee syntaxp)
 hypothesis, ACL2 evaluates @('(find-match-in-plus-nests lhs rhs)') in an
 environment where @('lhs') and @('rhs') are instantiated as determined by the
 substitution.  In this case we evaluate</p>

 @({
  (FIND-MATCH-IN-PLUS-NESTS '(binary-+ '3
                                        (binary-+ (expt a n) (foo a c)))
                            '(binary-+ (bar b) (expt a n))).
 })

 <p>Observe that, just as in the case of a @(tsee syntaxp) hypothesis, we
 substitute the quotation of the variables bindings into the term to be
 evaluated.  See @(see syntaxp) for the reasons for this.  The result of this
 evaluation, @('((X . (EXPT A N)))'), is then used to extend the substitution
 alist:</p>

 @({
  ((X . (EXPT A N))
   (LHS . (binary-+ '3
                     (binary-+ (expt a n) (foo a c))))
   (RHS . (binary-+ (bar b) (expt a n)))),
 })

 <p>and this extended substitution determines
 @('cancel-matching-addends-equal')'s result:</p>

 @({
  (EQUAL (+ (- X) LHS) (+ (- X) RHS))
  ==>
  (EQUAL (+ (- (EXPT A N)) 3 (EXPT A N) (FOO A C))
         (+ (- (EXPT A N)) (BAR B) (EXPT A N))).
 })

 <p>Question: What is the internal form of this result?<br></br>

 Hint: Use @(':')@(tsee trans).</p>

 <p>When this rule fires, it adds the negation of a common term to both sides
 of the equality by selecting a binding for the otherwise-free variable @('x'),
 under programmatic control.  Note that other mechanisms such as the binding of
 @(see free-variables) may also extend the substitution alist.</p>

 <p>Just as for a @(tsee syntaxp) test, a @('bind-free') form signals failure
 by returning @('nil').  However, while a @(tsee syntaxp) test signals success
 by returning true, a @('bind-free') form signals success by returning an alist
 which is used to extend the current substitution alist.  Because of this use
 of the alist, there are several restrictions on it &mdash; in particular the
 alist must only bind variables, these variables must not be already bound by
 the substitution alist, and the variables must be bound to ACL2 terms.  If
 @('term') returns an alist and the alist meets these restrictions, we append
 the alist to the substitution alist and use the result as the new current
 substitution alist.  This new current substitution alist is then used when we
 attempt to relieve the next hypothesis or, if there are no more, instantiate
 the right hand side of the rule.</p>

 <p>There is also a second, optional, @('var-list') argument to a
 @('bind-free') hypothesis.  If provided, it must be either @('t'), @('nil'),
 or a non-empty list of variables.  If it is not provided, it defaults to
 @('t'); and it is also treated as @('t') if the value provided is @('nil').
 If it is a non-empty list of variables, this second argument is used to place
 a further restriction on the possible values of the alist to be returned by
 @('term'): any variables bound in the alist must be present in that list of
 variables.  We strongly recommend the use of this list of variables, as that
 list is considered to contribute to the list of variables in the hypotheses of
 a linear rule; see @(see linear), in particular condition (b) mentioned there
 regarding a requirement that maximal terms and hypotheses must suffice for
 instantiating all the variables in the conclusion.  If @('var-list') is @('t)
 (either expicitly or implicitly, as described above), then that condition is
 considered to be met trivially; this could prevent ACL2 from rejecting
 ineffective linear rules.</p>

 <p>An extended @('bind-free') hypothesis is similar to the simple type
 described above, but it uses two additional variables, @('mfc') and
 @('state'), which must not be bound by the left hand side or an earlier
 hypothesis of the rule.  They must be the last two variables mentioned by
 @('term'): first @('mfc'), then @('state').  These two variables give access
 to the functions @('mfc-')xxx; see @(see extended-metafunctions).  As
 described there, @('mfc') is bound to the so-called metafunction-context and
 @('state') to ACL2's @(tsee state).  See @(see bind-free-examples) for
 examples of the use of these extended @('bind-free') hypotheses.</p>

 <p><b>SECTION</b>: Returning a list of alists.</p>

 <p>As promised above, we conclude with a discussion of the case that
 evaluation of the @('bind-free') term produces a list of alists, @('x'),
 rather than a single alist.  In this case each member @('b') of @('x') is
 considered in turn, starting with the first and proceeding through the list.
 Each such @('b') is handled exactly as discussed above, as though it were the
 result of evaluating the @('bind-free') term.  Thus, each @('b') extends the
 current variable binding alist, and all remaining hypotheses are then
 relieved, as though @('b') had been the value obtained by evaluating the
 @('bind-free') term.  As soon as one such @('b') leads to successful relieving
 of all remaining hypotheses, the process of relieving hypotheses concludes, so
 no further members of @('x') are considered.</p>

 <p>We illustrate with a simple pedagogical example.  First introduce functions
 @('p1') and @('p2') such that a rewrite rule specifies that @('p2') implies
 @('p1'), but with a free variable.</p>

 @({
  (defstub p1 (x) t)
  (defstub p2 (x y) t)

  (defaxiom p2-implies-p1
    (implies (p2 x y)
             (p1 x)))
 })

 <p>If we add the following axiom, then @('(p1 x)') follows logically for all
 @('x').</p>

 @({
  (defaxiom p2-instance
    (p2 v (cons v 4)))
 })

 <p>Unfortunately, evaluation of @('(thm (p1 a))') fails, because ACL2 fails to
 bind the free variable @('y') in order to apply the rule @('p2-instance').</p>

 <p>Let's define a function that produces a list of alists, each binding the
 variable @('y').  Of course, we know that only the middle one below is
 necessary in this simple example.  In more complex examples, one might use
 heuristics to construct such a list of alists.</p>

 @({
  (defun my-alists (x)
    (list (list (cons 'y (fcons-term* 'cons x ''3)))
          (list (cons 'y (fcons-term* 'cons x ''4)))
          (list (cons 'y (fcons-term* 'cons x ''5)))))
 })

 <p>The following rewrite rule uses @('bind-free') to return a list of
 candidate alists binding @('y').</p>

 @({
  (defthm p2-implies-p1-better
    (implies (and (bind-free (my-alists x)
                             (y)) ; the second argument, (y), is optional
                  (p2 x y))
             (p1 x)))
 })

 <p>Now the proof succeeds for @('(thm (p1 a))').  Why?  When ACL2 applies the
 @('rewrite') rule @('p2-implies-p1-better'), it evaluates @('my-alists'), as
 we can see from the following @(see trace), to bind @('y') in three different
 alists.</p>

 @({
  ACL2 !>(thm (p1 a))
  1> (ACL2_*1*_ACL2::MY-ALISTS A)
  <1 (ACL2_*1*_ACL2::MY-ALISTS (((Y CONS A '3))
                                ((Y CONS A '4))
                                ((Y CONS A '5))))

  Q.E.D.
 })

 <p>The first alist, binding @('y') to @('(cons a '3)'), fails to allow the
 hypothesis @('(p2 x y)') to be proved.  But the next binding of @('y'), to
 @('(cons a '4)'), succeeds: then the current binding alist is @('((x . a) (y
 . (cons a '4)))'), for which the hypothesis @('(p2 x y)') rewrites to true
 using the rewrite rule @('p2-instance').</p>")

(defxdoc bind-free-examples
  :parents (bind-free)
  :short "Examples pertaining to @(tsee bind-free) hypotheses"
  :long "<p>See @(see bind-free) for a basic discussion of the use of
 @('bind-free') to control rewriting.</p>

 <p>Note that the examples below all illustrate the common case in which a
 @('bind-free') hypothesis generates a binding alist.  See @(see bind-free), in
 particular the final section, for a discussion of the case that instead a list
 of binding alists is generated.</p>

 <p>We give examples of the use of @(tsee bind-free) hypotheses from the
 perspective of a user interested in reasoning about arithmetic, but it should
 be clear that @(tsee bind-free) can be used for many other purposes also.</p>

 <p>EXAMPLE 1: Cancel a common factor.</p>

 @({
  (defun bind-divisor (a b)

  ; If a and b are polynomials with a common factor c, we return a
  ; binding for x.  We could imagine writing get-factor to compute the
  ; gcd, or simply to return a single non-invertible factor.

    (let ((c (get-factor a b)))
      (and c (list (cons 'x c)))))

  (defthm cancel-factor
    ;; We use case-split here to ensure that, once we have selected
    ;; a binding for x, the rest of the hypotheses will be relieved.
    (implies (and (acl2-numberp a)
                  (acl2-numberp b)
                  (bind-free (bind-divisor a b) (x))
                  (case-split (not (equal x 0)))
                  (case-split (acl2-numberp x)))
             (iff (equal a b)
                  (equal (/ a x) (/ b x)))))
 })

 <p>EXAMPLE 2: Pull integer summand out of floor.  Note: This example has an
 <i>extended</i> @(tsee bind-free) hypothesis, which uses the term
 @('(find-int-in-sum sum mfc state)').</p>

 @({
  (defun fl (x)
    ;; This function is defined, and used, in the IHS books.
    (floor x 1))

  (defun int-binding (term mfc state)
    ;; The call to mfc-ts returns the encoded type of term. ;
    ;; Thus, we are asking if term is known by type reasoning to ;
    ;; be an integer. ;
    (declare (xargs :stobjs (state) :mode :program))
    (if (ts-subsetp (mfc-ts term mfc state)
                    *ts-integer*)
        (list (cons 'int term))
      nil))

  (defun find-int-in-sum (sum mfc state)
    (declare (xargs :stobjs (state) :mode :program))
    (if (and (nvariablep sum)
             (not (fquotep sum))
             (eq (ffn-symb sum) 'binary-+))
        (or (int-binding (fargn sum 1) mfc state)
            (find-int-in-sum (fargn sum 2) mfc state))
      (int-binding sum mfc state)))

  ; Some additional work is required to prove the following.  So for
  ; purposes of illustration, we wrap skip-proofs around the defthm.

  (skip-proofs
   (defthm cancel-fl-int
    ;; The use of case-split is probably not needed, since we should
    ;; know that int is an integer by the way we selected it.  But this
    ;; is safer.
     (implies (and (acl2-numberp sum)
                   (bind-free (find-int-in-sum sum mfc state) (int))
                   (case-split (integerp int)))
              (equal (fl sum)
                     (+ int (fl (- sum int)))))
     :rule-classes ((:rewrite :match-free :all)))
  )

  ; Arithmetic libraries will have this sort of lemma.
  (defthm hack (equal (+ (- x) x y) (fix y)))

  (in-theory (disable fl))

  (thm (implies (and (integerp x) (acl2-numberp y))
                (equal (fl (+ x y)) (+ x (fl y)))))
 })

 <p>EXAMPLE 3: Simplify terms such as (equal (+ a (* a b)) 0)</p>

 @({
  (defun factors (product)
    ;; We return a list of all the factors of product.  We do not
    ;; require that product actually be a product.
    (if (eq (fn-symb product) 'BINARY-*)
        (cons (fargn product 1)
              (factors (fargn product 2)))
      (list product)))

  (defun make-product (factors)
    ;; Factors is assumed to be a list of ACL2 terms.  We return an
    ;; ACL2 term which is the product of all the elements of the
    ;; list factors.
    (cond ((atom factors)
           ''1)
          ((null (cdr factors))
           (car factors))
          ((null (cddr factors))
           (list 'BINARY-* (car factors) (cadr factors)))
          (t
           (list 'BINARY-* (car factors) (make-product (cdr factors))))))

  (defun quotient (common-factors sum)
    ;; Common-factors is a list of ACL2 terms.   Sum is an ACL2 term each
    ;; of whose addends have common-factors as factors.  We return
    ;; (/ sum (make-product common-factors)).
    (if (eq (fn-symb sum) 'BINARY-+)
        (let ((first (make-product (set-difference-equal (factors (fargn sum 1))
                                                         common-factors))))
          (list 'BINARY-+ first (quotient common-factors (fargn sum 2))))
      (make-product (set-difference-equal (factors sum)
                                          common-factors))))

  (defun intersection-equal (x y)
    (cond ((endp x)
           nil)
          ((member-equal (car x) y)
           (cons (car x) (intersection-equal (cdr x) y)))
          (t
           (intersection-equal (cdr x) y))))

  (defun common-factors (factors sum)
    ;; Factors is a list of the factors common to all of the addends
    ;; examined so far.  On entry, factors is a list of the factors in
    ;; the first addend of the original sum, and sum is the rest of the
    ;; addends.  We sweep through sum, trying to find a set of factors
    ;; common to all the addends of sum.
    (declare (xargs :measure (acl2-count sum)))
    (cond ((null factors)
           nil)
          ((eq (fn-symb sum) 'BINARY-+)
           (common-factors (intersection-equal factors (factors (fargn sum 1)))
                           (fargn sum 2)))
          (t
           (intersection-equal factors (factors sum)))))

  (defun simplify-terms-such-as-a+ab-rel-0-fn (sum)
    ;; If we can find a set of factors common to all the addends of sum,
    ;; we return an alist binding common to the product of these common
    ;; factors and binding quotient to (/ sum common).
    (if (eq (fn-symb sum) 'BINARY-+)
        (let ((common-factors (common-factors (factors (fargn sum 1))
                                              (fargn sum 2))))
          (if common-factors
              (let ((common (make-product common-factors))
                    (quotient (quotient common-factors sum)))
                (list (cons 'common common)
                      (cons 'quotient quotient)))
            nil))
      nil))

  (defthm simplify-terms-such-as-a+ab-=-0
    (implies (and (bind-free
                   (simplify-terms-such-as-a+ab-rel-0-fn sum)
                   (common quotient))
                  (case-split (acl2-numberp common))
                  (case-split (acl2-numberp quotient))
                  (case-split (equal sum
                                     (* common quotient))))
             (equal (equal sum 0)
                    (or (equal common 0)
                        (equal quotient 0)))))

  (thm (equal (equal (+ u (* u v)) 0)
        (or (equal u 0) (equal v -1))))
 })")

(defxdoc bitp
  :parents (numbers acl2-built-ins)
  :short "A recognizer for the set of bits, {0,1}"
  :long "<p>@('Bitp') returns @('t') if and only its argument is @('0') or
 @('1'), and @('nil') otherwise.</p>

 <p>Note that this is a predicate form of the @(see type-spec) declaration
 @('(TYPE BIT b)').</p>

 @(def bitp)")

(defxdoc book-compiled-file
  :parents (books-reference)
  :short "Creating and loading of compiled and expansion files for @(see books)"
  :long "<p>An effect of @(see compilation) is to speed up the execution of the
 functions defined in a book.  Compilation can also remove tail recursion, thus
 avoiding stack overflows.  The presence of compiled code for the functions in
 the book should not otherwise affect the performance of ACL2.  See @(see
 guard) for a discussion; also see @(see compilation).</p>

 <p>By default, the @(tsee certify-book) command compiles the book that it
 certifies.  see @(see certify-book) for how to control this behavior.</p>

 <p>By default, the @(tsee include-book) command loads the compiled file for
 the book.  The details of how this loading works are subtle, and do not need
 to be understood by most users.  The ACL2 source code contains an ``Essay on
 Hash Table Support for Compilation'' that explains such details for those
 interested.  All that users should generally need to know about this is that
 the compiled file is always the result of compiling a so-called ``expansion
 file'', which contains certain additional code besides the book itself.  The
 relevance to users of the expansion file is that it can be loaded if the
 compiled file is missing (except when @(':load-compiled-file t') is specified
 by the @(tsee include-book) form), and its existence is required in order for
 @(tsee include-book) to create a book's compiled file, as described below.</p>

 <p>Most users can skip the remainder of this documentation topic, which
 addresses the uncommon activity of using @(tsee include-book) to compile
 books.</p>

 <p>@('Include-book') can be made to compile a book by supplying its keyword
 argument @(':load-compiled-file') the value @(':comp').  However, a compiled
 file can only be produced if there is already an <i>expansion file</i> that is
 at least as recent as the book's @(see certificate).  Such a file, whose name
 happens to be the result of concatenating the string @('\"@expansion.lsp\"')
 to the book name (without the @('\".lisp\"') suffix), is created by @(tsee
 certify-book) when state global variable @(''save-expansion-file') has a
 non-@('nil') value.  That will be the case if ACL2 started up when environment
 variable @('ACL2_SAVE_EXPANSION') was @('t') (or any value that is not the
 empty string and whose @(tsee string-upcase) is not @('\"NIL\"')), until the
 time (if any) that @(''save-expansion-file') is assigned a different value by
 the user.  In most respects, the @(':comp') setting is treated exactly the
 same as @(':warn'); but after all events in the book are processed, the
 expansion file is compiled if a compiled file was not loaded, after which the
 resulting compiled file is loaded.</p>

 <p>One can thus, for example, compile books for several different host Lisps
 &mdash; useful when installing ACL2 executables at the same site that are
 built on different host Lisps.  A convenient way to do this in an environment
 that provides Gnu `make' is to certify the community books using the shell
 command ``@('make regression')'' in the @('acl2-sources/') directory, after
 setting environment variable @('ACL2_SAVE_EXPANSION') to @('t'), and then
 moving to the @('books') directory and executing the appropriate `make'
 commands to compile the books (targets @('fasl'), @('o'), and so on, according
 to the compiled file extension for the host Lisp).</p>

 <p>We conclude by saying more about the @(':load-compiled-file') argument of
 @(tsee include-book).  We assume that @(see state) global
 @(''compiler-enabled') has a non-@('nil') value; otherwise
 @(':load-compiled-file') is always treated as @('nil').</p>

 <p>We do not consider raw mode below (see @(see set-raw-mode)), which presents
 a special case: ACL2 will attempt to load the book itself whenever it would
 otherwise load the expansion or compiled file, but cannot (either because the
 @(':load-compiled-file') argument is @('nil'), or for each of the expansion
 and compiled files, either it does not exist or it is out of date with respect
 to the @('.cert') file).</p>

 <p>The @(':load-compiled-file') argument is not recursive: calls of
 @('include-book') that are inside the book supplied to @('include-book') use
 their own @(':load-compiled-file') arguments.  However, those subsidiary
 @('include-book') calls can nevertheless be sensitive to the
 @(':load-compiled-file') arguments of enclosing @('include-book') calls, as
 follows.  If @(':load-compiled-file') has value @('t'), then every subsidiary
 @('include-book') is required to load a compiled file.  Moreover, if a book's
 compiled file or expansion file is loaded in raw Lisp, then an attempt will be
 made to load the compiled file or expansion file for any @(tsee include-book)
 form encountered during that load.  If that attempt fails, then that load
 immediately aborts, as does its parent load, and so on up the chain.  If, when
 going up the chain, an @(tsee include-book) is aborted for which keyword
 argument @(':load-compiled-file') has value @('t'), then an error occurs.</p>

 <p>When loading a book's compiled file or expansion file, @('FILE'), it is
 possible to encounter an @(tsee include-book) form for a book that has no
 suitable compiled file or expansion file.  In that case, the load of @('FILE')
 is aborted at that point.  Similarly, the load of @('FILE') is aborted in the
 case that this @('include-book') form has a suitable compiled file or
 expansion file whose load is itself aborted.  Thus, whenever any
 @('include-book') aborts, so do all of its parent @('include-book')s, up the
 chain.  Such an abort causes an error when the @('include-book') form
 specifies a @(':load-compiled-file') value of @('t').</p>")

(defxdoc book-contents
  :parents (books-tour)
  :short "Restrictions on the forms inside @(see books)"
  :long "@({
  Example Book:

  ; This book defines my app function and the theorem that it is
  ; associative.  One irrelevant help lemma is proved first but
  ; it is local and so not seen by include-book.  I depend on the
  ; inferior book \"weird-list-primitives\" from which I get
  ; definitions of hd and tl.

  (in-package \"MY-PKG\")

  (include-book \"weird-list-primitives\")

  (defun app (x y) (if (consp x) (cons (hd x) (app (tl x) y)) y))

  (local
   (defthm help-lemma
     (implies (true-listp x) (equal (app x nil) x))))

  (defthm app-is-associative
    (equal (app (app a b) c) (app a (app b c))))
 })

 <p>The first form in a book must be @('(in-package \"pkg\")') where
 @('\"pkg\"') is some package name known to ACL2 whenever the book is
 certified.  The rest of the forms in a book are embedded event forms, i.e.,
 @(tsee defun)s, @(tsee defthm)s, etc., some of which may be marked @(tsee
 local).  See @(see embedded-event-form).  The usual Common Lisp commenting
 conventions are provided.  Note that since a book consists of embedded event
 forms, we can talk about the ``@(tsee local)'' and ``non-local'' @(see events)
 of a book.</p>

 <p>Because @(tsee in-package) is not an embedded event form, the only @(tsee
 in-package) in a book is the initial one.  Because @(tsee defpkg) is not an
 embedded event form, a book can never contain a @(tsee defpkg) form.  Because
 @(tsee include-book) is an embedded event form, @(see books) may contain
 references to other @(see books).  This makes @(see books) structured
 objects.</p>

 <p>When the forms in a book are read from the file, they are read with @(tsee
 current-package) set to the package named in the @(tsee in-package) form at
 the top of the file.  The effect of this is that all symbols are @(see
 intern)ed in that package, except those whose packages are given explicitly
 with the ``::'' notation.  For example, if a book begins with @('(in-package
 \"ACL2-X\")') and then contains the form</p>

 @({
    (defun fn (x)
      (acl2::list 'car x))
 })

 <p>then @(tsee defun), @('fn'), @('x'), and @(tsee car) are all @(see
 intern)ed in the @('\"ACL2-X\"') package.  I.e., it is as though the following
 form were read instead:</p>

 @({
    (acl2-x::defun acl2-x::fn (acl2-x::x)
        (acl2::list 'acl2-x::car acl2-x::x)).
 })

 <p>Of course, @('acl2-x::defun') would be the same symbol as @('acl2::defun')
 if the @('\"ACL2-X\"') package imported @('acl2::defun').</p>

 <p>If each book has its own unique package name and all the names defined
 within the book are in that package, then name clashes between @(see books)
 are completely avoided.  This permits the construction of useful logical @(see
 world)s by the successive inclusion of many @(see books).  Although it is
 often too much trouble to manage several packages, their judicious use is a
 way to minimize name clashes.  Often, a better way is to use @('local'); see
 @(see local).</p>

 <p>How does @(tsee include-book) know the definitions of the packages used in
 a book, since @(tsee defpkg)s cannot be among the forms?  More generally, how
 do we know that the forms in a book will be admissible in the host logical
 @(see world) of an @(tsee include-book)?  See @(see certificate) for answers
 to these questions.</p>")

(defxdoc book-example
  :parents (books-tour)
  :short "How to create, certify, and use a simple book"
  :long "<p>Suppose you have developed a sequence of admissible @(see events)
 which you want to turn into a book.  We call this ``publishing'' the book.
 This note explains how to do that.</p>

 <p>A key idea of @(see books) is that they are ``incremental'' in the sense
 that when you include a book in a host logical @(see world), the @(see world)
 is incrementally extended by the results established in that book.  This is
 allowed only if every name defined by the incoming book is either new or is
 already identically defined.  See @(see redundant-events).  This is exactly
 the same problem faced by a programmer who wishes to provide a utility to
 other people: how can he make sure he doesn't create name conflicts?  The
 solution, in Common Lisp, is also the same: use packages.  While @(see books)
 and packages have a very tenuous formal connection (every book must start with
 an @(tsee in-package)), the creation of a book is intimately concerned with
 the package issue.  Having motivated what would otherwise appear as an
 unnecessary fascination with packages below, we now proceed with a description
 of how to publish a book.</p>

 <p>Just to be concrete, let's suppose you have already gotten ACL2 to accept
 the following sequence of @(see command)s, starting in the ACL2 initial @(see
 state).</p>

 @({
     (defpkg \"ACL2-MY-BOOK\"
             (union-eq *common-lisp-symbols-from-main-lisp-package*
                       *acl2-exports*))
     (in-package \"ACL2-MY-BOOK\")
     (defun app (x y)
       (if (consp x) (cons (car x) (app (cdr x) y)) y))
     (defun rev (x)
       (if (consp x) (app (rev (cdr x)) (list (car x))) nil))
     (defthm rev-app-hack
       (equal (rev (app a (list x))) (cons x (rev a))))
     (defthm rev-rev
       (implies (acl2::true-listp x) (equal (rev (rev x)) x)))
 })

 <p>Observe that the first form above defines a package (which imports the
 symbols defined in CLTL such as @(tsee if) and @(tsee cons) and the symbols
 used to @(see command) ACL2 such as @(tsee defun) and @(tsee defthm)).  The
 second form selects that package as the current one.  All subsequent forms are
 read into that package.  The remaining forms are just event forms: @(tsee
 defun)s and @(tsee defthm)s in this case.</p>

 <p>Typically you would have created a file with Emacs containing these forms
 and you will have submitted each of them interactively to ACL2 to confirm that
 they are all admissible.  That interactive verification should start in ACL2's
 initial @(see world) &mdash; although you might, of course, start your
 sequence of @(see events) with some @(tsee include-book)s to build a more
 elaborate @(see world).</p>

 <p>The first step towards publishing a book containing the results above is to
 create a file that starts with the @(tsee in-package) and then contains the
 rest of the forms.  Let's call that file @('\"my-book.lisp\"').  The name is
 unimportant, except it must end with @('\".lisp\"').  If there are @(see
 events) that you do not wish to be available to the user of the book &mdash;
 e.g., lemmas you proved on your way toward proving the main ones &mdash; you
 may so mark them by enclosing them in @(tsee local) forms.  See @(see local).
 Let us suppose you wish to hide @('rev-app-hack') above.  You may also add
 standard Lisp comments to the file.  The final content of
 @('\"my-book.lisp\"') might be:</p>

 @({
   ; This book contains my app and rev functions and the theorem
   ; that rev is its own inverse.

     (in-package \"ACL2-MY-BOOK\")
     (defun app (x y)
       (if (consp x) (cons (car x) (app (cdr x) y)) y))
     (defun rev (x)
       (if (consp x) (app (rev (cdr x)) (list (car x))) nil))

   ; The following hack is not exported.
     (local (defthm rev-app-hack
       (equal (rev (app a (list x))) (cons x (rev a)))))

     (defthm rev-rev
       (implies (acl2::true-listp x) (equal (rev (rev x)) x)))
 })

 <p>The file shown above <b>is</b> the book.  By the time this note is done you
 will have seen how to certify that the book is correct, how to compile it, and
 how to use it in other host @(see world)s.  Observe that the @(tsee defpkg) is
 not in the book.  It cannot be: Common Lisp compilers disagree on how to treat
 new package definitions appearing in files to be compiled.</p>

 <p>Since a book is just a source file typed by the user, ACL2 provides a
 mechanism for checking that the @(see events) are all admissible and then
 marking the file as checked.  This is called certification.  To certify
 @('\"my-book.lisp\"') you should first get into ACL2 with an initial @(see
 world).  Then, define the package needed by the book, by typing the following
 @(tsee defpkg) to the ACL2 @(see prompt):</p>

 @({
  ACL2 !>(defpkg \"ACL2-MY-BOOK\"
                 (union-eq *common-lisp-symbols-from-main-lisp-package*
                           *acl2-exports*))
 })

 <p>Then execute the @(see command):</p>

 @({
  ACL2 !>(certify-book \"my-book\" 1 t) ; the `t' is in fact the default
 })

 <p>Observe that you do not type the @('\".lisp\"') part of the file name.  For
 purposes of @(see books), the book's name is @('\"my-book\"') and by the time
 all is said and done, there will be several extensions in addition to the
 @('\".lisp\"') extension associated with it.</p>

 <p>The @('1') tells @(tsee certify-book) that you acknowledge that there is
 one command in this ``certification @(see world)'' (namely the @(tsee
 defpkg)).  To use the book, any prospective host @(see world) must be extended
 by the addition of whatever @(see command)s occurred before certification.  It
 would be a pity to certify a book in a @(see world) containing junk because
 that junk will become the ``@(see portcullis)'' guarding entrance to the book.
 The @('t') above tells @(tsee certify-book) that you wish to compile
 @('\"my-book.lisp\"') also (but see @(see compilation) for an exception).
 @(tsee Certify-book) makes many checks but by far the most important and
 time-consuming one is that it ``proves'' every event in the file.</p>

 <p>When @(tsee certify-book) is done it will have created two new files.  The
 first will be called @('\"my-book.cert\"') and contains the ``@(see
 certificate)'' attesting to the admissibility of the @(see events) in
 @('\"my-book.lisp\"').  The @(see certificate) contains the @(tsee defpkg) and
 any other forms necessary to construct the certification @(see world).  It
 also contains so-called @(see book-hash) values that are used to help you keep
 track of which version of @('\"my-book.lisp\"') was certified.</p>

 <p>The second file that may be created by @(tsee certify-book) is the compiled
 version of @('\"my-book.lisp\"') and will have a name that is assigned by the
 host compiler (e.g., @('\"my-book.o\"') in GCL, @('\"my-book.fasl\"') in
 SBCL).  @(tsee Certify-book) will also load this object file.  When @(tsee
 certify-book) is done, you may throw away the logical @(see world) it created,
 for example by executing the @(see command) @(':u').</p>

 <p>To use the book later in any ACL2 session, just execute the event
 @('(include-book \"my-book\")').  This will do the necessary @(tsee defpkg),
 load the non-@(tsee local) @(see events) in @('\"my-book.lisp\"') and then may
 load the compiled code for the non-local functions defined in that file.
 Checks are made to ensure that the @(see certificate) file exists and
 describes the version of @('\"my-book.lisp\"') that is read.  The compiled
 code is loaded if and only if it exists and has a later write date than the
 source file (but see @(see compilation) for an exception).</p>

 <p>Since @(tsee include-book) is itself an event, you may put such forms into
 other @(see books).  Thus it is possible for the inclusion of a single book to
 lead to the inclusion of many others.  The @(see book-hash) information
 maintained in @(see certificate)s helps deal with the version control problem
 of the referenced @(see books).  I.e., if this version of @('\"my-book\"') is
 used during the certification of @('\"your-book\"'), then the @(see
 certificate) for @('\"your-book\"') includes the book-hash for this version of
 @('\"my-book\"').  If a later @('(include-book \"your-book\")') finds a
 version of @('\"my-book\"') with a different book-hash, an error is signaled.
 But book-hash values are not perfect and the insecurity of the host file
 system prevents ACL2 from guaranteeing the logical soundness of an @(tsee
 include-book) event, even for a book that appears to have a valid @(see
 certificate) (they can be forged, after all).  (See @(see certificate) for
 further discussion.)</p>

 <p>This concludes the example of how to create, certify and use a book.  If
 you wish, you could now review the @(see documentation) for book-related
 topics (see @(see books)) and browse through them.  They'll probably make
 sense in this context.  Alternatively, you could continue the ``guided tour''
 through the rest of the @(see documentation) of @(see books).  See @(see
 book-name), following the pointer given at the conclusion.</p>")

(defxdoc book-hash
  :parents (books)
  :short "Assigning ``often unique'' fingerprints to @(see books)"
  :long "<p>ACL2 provides a <i>certification</i> process for recording into a
 @(see certificate) file that a book is valid.  That process records certain
 ``fingerprint'' values for the book and the books that it includes, in order
 to give some confidence in the book's validity.  We call that value the
 ``book-hash'' for the book.  By default, a book-hash is an alist that records,
 for a given book (@('.lisp') file), its file size (in bytes) and its
 write-date.</p>

 <p>You can arrange for a book-hash to be a checksum instead of an alist, which
 gives a bit greater security, as illustrated in an example provided below.
 See @(see checksum).  Nevertheless, the (default) use of book-hash alists may
 be worthwhile, in spite of the decreased security, because of faster times for
 @(tsee certify-book) and @(tsee include-book) when using book-hash alists
 instead of checksums.  If you want to use checksums, however, there are these
 two ways to do so.</p>

 <ul>

 <li>Before starting ACL2, set environment variable @('ACL2_BOOK_HASH_ALISTP')
 to @('NIL') (or @('nil'); actually it suffices that @(tsee string-upcase)
 applied to the indicated string is @('\"NIL\"')).  A common way to get this
 effect is to supply the argument @('ACL2_BOOK_HASH_ALISTP=NIL') with your
 @('\"make\"') command.</li>

 <li>After starting ACL2, set @(see state) global variable
 @('book-hash-alistp') to @('nil'), e.g.: @('(assign book-hash-alistp
 nil)').  (Set this variable to @('t') to revert to the default behavior:
 @('(assign book-hash-alistp t)').)</li>

 </ul>

 <p>The simple example below illustrates the potential weakness of book-hash
 alists (as compared to checksums), by exploiting the fact that these alists do
 not hash information in the @(see certificate) itself.  (Book-hash alists also
 ignore @(see portcullis) @(see command)s and the corresponding @('.acl2x')
 file, if any (see @(see set-write-acl2x)); but we don't exploit these facts in
 this example.)  We start with the following book, @('\"foo.lisp\"').</p>

 @({
 (in-package \"ACL2\")
 (make-event '(defun foo (x) (cons x x)))
 })

 <p>Now start ACL2 and run these commands.</p>

 @({
 (assign book-hash-alistp t) ; a no-op, by default
 (certify-book \"foo\")
 (read-file \"foo.cert\" state)
 (quit)
 })

 <p>Next, replace the contents of @('\"foo.cert\"') by the values in the list
 printed by the @('read-file') form above (hence, without the outer
 parentheses).  Further edit that file by twice changing @('(cons x x)') to
 @('x').  Also, update the write-date on the compiled file, e.g. with @('touch
 foo.dx64fsl'), if you want to avoid a warning.  Then submit @('(include-book
 \"foo\")').  We get the following contradictory behavior, without any
 warnings.</p>

 @({
 ACL2 !>(foo 3)
 (3 . 3)
 ACL2 !>(thm (equal (foo x) x))

 Q.E.D.

 Summary
 Form:  ( THM ...)
 Rules: ((:DEFINITION FOO))
 Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)
 Prover steps counted:  5

 Proof succeeded.
 ACL2 !>
 })

 <p>For more information about connections between book-hash values and
 certification status, see @(see book-hash-mismatch).</p>")

(defxdoc book-hash-mismatch
  :parents (book-hash)
  :short "Deeming a book uncertified because of a @(see book-hash) mismatch"
  :long "<p>When @(tsee include-book) is invoked, the specified book may be
 considered uncertified by ACL2 because its @(see book-hash) &mdash; which by
 default is an alist specifying the book's size and write-date, but could be a
 checksum &mdash; does not match what was expected.  In that case you will
 typically see a warning such as the following, which instead will be an error
 if this occurs during certification.  In this topic, we explain how this
 warning can occur.</p>

 @({
 ACL2 Warning [Uncertified] in ( INCLUDE-BOOK \"foo\" ...):
 The certificate for \"/Users/kaufmann/temp/foo.lisp\" lists the book-
 hash of that book as ((:BOOK-LENGTH . 38) (:BOOK-WRITE-DATE . 3674050905)).
 But its book-hash is now computed to be
 ((:BOOK-LENGTH . 39) (:BOOK-WRITE-DATE . 3674050914)).
 See :DOC book-hash-mismatch.
 })

 <p>Probably the simplest way for such a mismatch to occur is if you change the
 book after certifying it, and then try to include the book.  In the example
 message displayed above, the book-hash for @('foo.lisp') stored into
 @('foo.cert') was 516310554.  But then @('foo.lisp') was edited (in this case,
 by adding a single newline character), and ACL2 complained because at
 @('include-book') time it computed a different book-hash for that book.</p>

 <p>The mismatch might occur on a book that is being included while including a
 superior book.  For example, suppose that the act of evaluating
 @('(include-book \"bk1\")') triggers evaluation of the event @('(include-book
 \"bk2\")') from @('bk1.lisp'), which similarly in turn causes evaluation of
 the event @('(include-book \"bk3\")') from @('bk2.lisp').  Now certify
 @('bk3.lisp'), then @('bk2.lisp'), and finally @('bk1.lisp').  So far, so
 good.  But now suppose we edit the lowest-level book, @('bk3.lisp').  We will
 likely get a warning (or error, if during book certification) such as the
 one displayed above when evaluating @('(include-book \"bk1\")'), complaining
 that the book-hash for @('bk3.lisp') has changed &mdash; followed by warnings
 complaining about inclusions of uncertified books by superior books.</p>

 <p>There are several ways for a book-hash in a @('.cert') file to become
 invalid, that is, to fail to match a recomputation of the book-hash when later
 including the book.  All such mismatches are due to a difference between the
 value of a book-hash at certification time and at later @('include-book')
 time; see @(see book-hash).</p>

 <p>Note that it is possible to mix alist and checksum book-hash values.  The
 key is to avoid conflicts between book-hash values stored for the same book in
 different @('.cert') files.  The following example works fine, because there
 is no such conflict: the book @('\"sub\"') is associated with a book-hash
 checksum in @('sub.cert'), which provides that same checksum as the book-hash
 value for @('\"sub\"') in @('top.cert'); and the book @('\"top\"') is
 associated with a book-hash alist in @('top.cert').</p>

 @({
 (certify-book \"sub\")
 (u)
 (assign book-hash-alistp t)
 (certify-book \"top\") ; contains the event (include-book \"sub\")
 (u)
 (include-book \"top\") ; no problem!
 })

 <p>However, if we instead evaluate the following forms in a fresh ACL2
 session, the last one causes an error, because a checksum serves as the
 book-hash for @('\"sub\"') in @('top.cert'), but at @('include-book') time the
 book-hash for @('\"sub\"') in @('sub.cert') is an alist, not a checksum.</p>

 @({
 (certify-book \"sub\")
 :u
 (certify-book \"top\") ; contains the event (include-book \"sub\")
 :u
 (assign book-hash-alistp t)
 (certify-book \"sub\") ; now the book-hash for sub is an alist
 :u
 ; Error: mismatch for book-hash values for \"sub\" in
 ;        new sub.cert and earlier top.cert.
 (include-book \"top\")
 })

 <p>That said, ACL2 does not care about the current value of state global
 @('book-hash-alistp') at @('include-book') time.  For example, the following
 works without error: ACL2 sees the checksum stored in @('sub.cert'), so at
 @('include-book') time it computes a checksum for the book @('\"sub\"'), not
 an alist, even though @(''book-hash-alistp') has value @('t').</p>

 @({
 (assign book-hash-alistp nil)
 (certify-book \"foo\")
 (assign book-hash-alistp t)
 (include-book \"foo\")
 })

 <p>We conclude by discussing a rather insidious book-hash mismatch that can
 occur when including a book using an ACL2 executable that differs from the
 executable that was used when certified the book.  Here is an example.
 Consider a book, @('\"foo.lisp\"'), containing the event @('(include-book
 \"tools/remove-hyps\" :dir :system)').  Suppose we have ACL2 executables
 @('acl2-cksum') and @('acl2-alist'), in different ACL2 directories.  Also
 suppose that we have certified (disjoint copies of) the @(see community-books)
 with each of these ACL2 executables, where we used
 @('ACL2_BOOK_HASH_ALISTP=NIL') in our @('\"make\"') command with
 @('acl2-cksum') but not with @('acl2-alist').  Now suppose we certify
 @('\"foo.lisp\"') using @('acl2-cksum') but then, in a session with
 @('acl2-alist'), we include that same book.  Then we will see a warning
 beginning as displayed below.  It complains that the book-hash required of
 @('books/tools/remove-hyps.cert') according to @('foo.cert') &mdash; which was
 created using @('acl2-cksum') &mdash; differs from the book-hash in the
 @('.cert') file of the copy of that book that is under the directory of the
 currently running ACL2, @('acl2-alist').</p>

 @({
 ACL2 Warning [Uncertified] in ( INCLUDE-BOOK \"foo\" ...):  After including
 the book \"/Users/smith/temp/foo.lisp\":

 -- its certificate requires the book \"/Users/smith/acl2/books/tools/remove-
 hyps.lisp\" with certificate annotations
   ((:SKIPPED-PROOFSP) (:AXIOMSP) (:TTAGS))
 and book hash 1958615726, but we have included a version of \"remove-hyps\"
 with certificate annotations
   ((:SKIPPED-PROOFSP) (:AXIOMSP) (:TTAGS))
 and book-hash ((:BOOK-LENGTH . 14650) (:BOOK-WRITE-DATE . 3656629263)),
 so book recertification is probably required;
 })")

(defxdoc book-name
  :parents (books-tour)
  :short "Conventions associated with book names"
  :long "@({
  Examples:
  \"list-processing\"
  \"/usr/home/smith/my-arith\"
 })

 <p>Book names are string constants that can be elaborated into file names.  We
 elaborate book names by concatenating the ``connected book directory'' (see
 @(see cbd)) string on the left and some ``extension,'' such as @('\".lisp\"'),
 on the right.  However, the connected book directory is not added if the book
 name itself already represents an absolute file name.  Furthermore, @(tsee
 include-book) and @(tsee certify-book) temporarily reset the connected book
 directory to be the directory of the book being processed.  This allows @(tsee
 include-book) forms to use file names without explicit mention of the
 enclosing book's directory.  This in turn allows @(see books) (together with
 those that they include, using @(tsee include-book)) to be moved between
 directories while maintaining their certification and utility.</p>

 <p>You may wish to read elsewhere for details of ACL2 file name conventions
 (see @(see pathname)), for a discussion of the filename that is the result of
 the elaboration described here (see @(see full-book-name)), and for details of
 the concept of the connected book directory (see @(see cbd)).  For details of
 how @(tsee include-book) (see @(see include-book)) and @(tsee certify-book)
 (see @(see certify-book)) use these concepts, see below.</p>

 <p>Often a book name is simply the familiar name of the file.  (See @(see
 full-book-name) for discussion of the notions of ``directory string,''
 ``familiar name,'' and ``extension''.  These concepts are not on the guided
 tour through @(see books) and you should read them separately.)  However, it
 is permitted for book names to include a directory or part of a directory
 name.  Book names never include the extension, since ACL2 must routinely tack
 several different extensions onto the name during @(tsee include-book).  For
 example, @(tsee include-book) uses the @('\".lisp\"'), @('\".cert\"') and
 possibly the @('\".o\"') or @('\".lbin\"') extensions of the book name.</p>

 <p>Book names are elaborated into full file names by @(tsee include-book) and
 @(tsee certify-book).  This elaboration is sensitive to the ``connected book
 directory.'' The connected book directory is an absolute filename string (see
 @(see pathname)) that is part of the ACL2 @(tsee state).  (You may wish to see
 @(see cbd) and to see @(see set-cbd) &mdash; note that these are not on the
 guided tour).  If a book name is an absolute filename string, ACL2 elaborates
 it simply by appending the desired extension to the right.  If a book name is
 a relative filename string, ACL2 appends the connected book directory on the
 left and the desired extension on the right.</p>

 <p>Note that it is possible that the book name includes some partial
 specification of the directory.  For example, if the connected book directory
 is @('\"/usr/home/smith/\"') then the book name @('\"project/task-1/arith\"')
 is a book name that will be elaborated to</p>

 @({
  \"/usr/home/smith/project/task-1/arith.lisp\".
 })

 <p>Observe that while the @(see events) in this @('\"arith\"') book are being
 processed the connected book directory will temporarily be set to</p>

 @({
  \"/usr/home/smith/project/task-1/\".
 })

 <p>Thus, if the book requires other @(see books), e.g.,</p>

 @({
  (include-book \"naturals\")
 })

 <p>then it is not necessary to specify the directory on which they reside
 provided that directory is the same as the superior book.</p>

 <p>This inheritance of the connected book directory and its use to elaborate
 the names of inferior @(see books) makes it possible to move @(see books) and
 their inferiors to new directories, provided they maintain the same relative
 relationship.  It is even possible to move with ease whole collections of
 @(see books) to different filesystems that use a different operating system
 than the one under which the original certification was performed.</p>

 <p>The @('\".cert\"') extension of a book, if it exists, is presumed to
 contain the most recent @(see certificate) for the book.  See @(see
 certificate) (or, if you are on the guided tour, wait until the tour gets
 there).</p>

 <p>See @(see book-contents) to continue the guided tour.</p>")

(defxdoc bookdata
  :parents (books)
  :short "An optional tool for writing out small files with meta-data about
 the books that are being certified."

  :long "<p>ACL2 provides a primitive capability for writing out a file of data
 associated with a book.  This information might be useful, for example, in
 building a database that allows you to search for name conflicts.  See @(see
 community-books) directory @('books/tools/book-conflicts/') for an application
 of this capability by Dave Greve.  If you use this capability and have ideas
 for enhancing it, please feel free to send them to the ACL2 developers.</p>

 <p>If the book has the name @('BK'), then the output file is named
 @('BK__bookdata.out').  That file is generated in the same directory as
 @('BK'), by certifying @('BK') when @(see state) global @(''write-bookdata')
 has a non-@('nil') value, for example as follows.</p>

 @({
  (assign write-bookdata t)
  (certify-book \"BK\" ...)
 })

 <p>The resulting file will contain a single form of the following shape,
 although not necessarily in the following order, according to the description
 that follows below.</p>

 @({
  (\"...BK.lisp\"
   :PKGS          pkgs-val
   :BOOKS         book-val
   :PORT-BOOKS    port-book-val
   :CONSTS        consts-val
   :PORT-CONSTS   port-consts-val
   :FNS           fns-val
   :PORT-FNS      port-fns-val
   :LABELS        labels-val
   :PORT-LABELS   port-labels-val
   :MACROS        macros-val
   :PORT-MACROS   port-macros-val
   :STOBJS        stobjs-val
   :PORT-STOBJS   port-stobjs-val
   :THEORIES      theories-val
   :PORT-THEORIES port-theories-val
   :THMS          thms-val
   :PORT-THMS     port-thms-val
 }))

 <p>The first entry in the form will always be the full book name (see @(see
 full-book-name)) of the certified book, @('BK').</p>

 <p>Subsequent values in the form are based on @(see events) introduced by
 including @('BK').  For various values of @('xxx') as described below,
 @('port-')<i>xxx</i>@('-val') is a list of values corresponding to @(see
 events) introduced in the certification @(see world) for @('BK') (see @(see
 portcullis)), and <i>xxx</i>@('-val') is a list of values corresponding to
 @(see events) introduced non-@(tsee local)ly by @('BK').  These lists include
 only ``top-level'' events, not those that are introduced by a book included
 either in @('BK') or its certification world.</p>

 <p>@('pkgs-val') is a list of names of packages introduced in the
 certification world (at the top level, not in an included book).  Note that no
 packages are introduced in a book itself, so no distinction is made between
 @('pkgs-val') and @('port-pkgs-val').  Both @('port-book-val') and
 @('book-val') are lists of full book names (see @(see full-book-name)) of
 included books.  The values associated with the other keywords are,
 themselves, association lists (see @(see alistp)) such that each key is a
 package name, which is associated with a list of @(see symbol-name)s for
 symbols in that package that are introduced for that keyword.  For example,
 @('fns-val') may be the alist</p>

 @({
  ((\"ACL2\" \"F1\" \"F2\")
   (\"MY-PKG\" \"G1\" \"G2\"))
 })

 <p>if the function symbols introduced in the book are @('F1') and @('F2') in
 the @('\"ACL2\"') package, as well as @('G1') and @('G2') in the
 @('\"MY-PKG\"') package.</p>

 <p>We next explain what kinds of symbols are introduced for each keyword
 @(':xxx').  Each such symbol would be associated with either the keyword
 @(':port-xxx') or the keyword @(':xxx') depending respectively on whether the
 symbol is introduced at the top level of the certification world for @('BK')
 or @('BK') itself.</p>

 <dl>

 <dt>@(':CONSTS')</dt>

 <dd>constant symbol introduced by @('defconst')</dd>

 <dt>@(':FNS')</dt>

 <dd>function symbol: introduced by @('defun'), @('defuns'), or @('defchoose'); or
 constrained (by an @(tsee encapsulate) event)</dd>

 <dt>@(':LABELS')</dt>

 <dd>symbol introduced by @('deflabel')</dd>

 <dt>@(':MACROS')</dt>

 <dd>macro name introduced by @('defmacro')</dd>

 <dt>@(':STOBJS')</dt>

 <dd>@('stobj') name introduced by @('defstobj') or @('defabsstobj')</dd>

 <dt>@(':THEORIES')</dt>

 <dd>theory name introduced by @('deftheory')</dd>

 <dt>@(':THMS')</dt>

 <dd>theorem name, which may be introduced by @('defthm') or a macro call
 expanding to a call of @('defthm'), such as see @(see defequiv) or
 @('defaxiom'); but may be introduced by @(tsee defpkg), for example, with
 name @('\"MYPKG-PACKAGE\"') if the package name is @('\"MYPKG\"')</dd>

 </dl>

 <p>Our hope is that people in the ACL2 community will generate and use this
 data to improve the ACL2 @(see community-books).  Here is an example
 illustrating how to generate bookdata files for those books as a byproduct of
 a regression run.  Below, we write @('{DIR}') as an abbreviation for the ACL2
 sources directory, and assume that this command is run from that directory.
 Of course, you may wish to use @('make') options like @('-j 8') and @('make')
 variable settings like @('ACL2={DIR}/my-saved_acl2'); see @(see
 books-certification) for details.</p>

 @({
   make regression-fresh \\
   ACL2_CUSTOMIZATION={DIR}/acl2-customization-files/bookdata.lisp
 })")

(defxdoc books
  :parents (acl2)
  :short "<i>Books</i> are files of ACL2 @(see events)&mdash;they are the main
way to split up large ACL2 developments into separate modules."

  :long "<p>This @(see documentation) topic is about ACL2 <a
 href='https://en.wikipedia.org/wiki/Source_code'>source code</a> files.
 However, there are also <a
 href='http://www.cs.utexas.edu/users/moore/publications/acl2-papers.html#Books'>traditional,
 paper books</a> published about ACL2 and its applications.</p>

 <p>You will almost surely want to organize your own ACL2 work into books.
 They facilitate reuse, allow you to reload proofs more quickly, allow you to
 rebuild parts of your proof in parallel, and so forth.  You will also want to
 be aware of the many <i>community books</i>, which provide useful tools and
 lemmas to build upon.  See @(see community-books) for more information,
 including how to contribute.</p>

 <h3>Introduction</h3>

 <p>A <b>book</b> is a file of ACL2 forms.  Books are prepared entirely by the
 user of the system, i.e., they are <i>source files</i> not <i>object
 files</i>.  Some of the forms in a book are marked @(tsee local) and the
 others are considered ``non-local.''</p>

 <p>@(tsee Include-book) lets you load a book into any ACL2 @(see world).  A
 successful @('include-book') extends the logic of the host @(see world) by
 adding just the non-local @(see events) in the book.  Ordinarily, you might
 include a variety of books to load all of their definitions and rules.</p>

 <p>Successful book inclusion is consistency preserving, provided that the book
 itself is consistent, as discussed later.  However, @(tsee include-book)
 assumes the @(see events) in a book are valid, so if you include a book that
 contains an inconsistency (e.g., an inadmissible definition) then the
 resulting theory is inconsistent!</p>

 <p>@(tsee Certify-book) lets you <i>certify</i> a book to guarantee that its
 successful inclusion is consistency preserving.  During certification, both
 the @(tsee local) and non-local forms are processed.  This lets you mark as
 @(tsee local) any @(see events) you need for certification, but that you want
 to hide from users of the book&mdash;e.g., the hacks, crocks, and kludges on
 the way to a good set of @(see rewrite) rules.</p>

 <p>Certification can also <see topic='@(url compilation)'>compile</see> a book
 to speed up the execution of the functions defined within it.  The desire to
 compile books is largely responsible for the restrictions we put on the forms
 allowed in books.</p>

 <p>Extensive @(see documentation) is available on the various aspects of
 books.  We recommend that you read it all before using books.  It has been
 written so as to make sense when read in a certain linear sequence, called the
 ``guided tour'', though in general you may browse through it randomly.  If you
 are on the guided tour, you should next read @(see book-example).</p>")

(defxdoc books-certification
  :parents (community-books)
  :short "Instructions for certifying the ACL2 @(see community-books)."

  :long "<p>The <see topic='@(url community-books)'>Community Books</see>
provides a @('make') system, which is recommended for certifying a specified
subset of those books from the @('books/') directory of your ACL2 distribution.
Alternate instructions are however available for certifying from the top-level
directory (see @(see books-certification-alt)).</p>

<p>Below are instructions for certifying various sets of books.  They all have
the following form in common.  Note: If there is a suitable @('\"acl2\"')
executable on your Unix @('PATH') &mdash; for example, if the @('bin')
subdirectory of the main ACL2 directory is on your @('PATH'), so that
@('bin/acl2') may be your executable &mdash; then you can omit
@('\"ACL2=...\"') below.</p>

@({
cd /path/to/acl2-sources/books
make ACL2=/path/to/acl2-sources/saved_acl2 ...
})

<p>For example, the section ``A Full Build'', below, says to do the following
if you would like to make a change to the @(see community-books) (where the
@('-j') argument is optional).</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make ACL2=/path/to/acl2-sources/saved_acl2 -j 2 all
})

<p>Success is indicated by a Linux exit status of 0.  Equivalently, there
should be no failures in the log.  Failures may be found by searching for
@('**'); for example, if output is redirected to a log file
@('make-regression.log'), then the following should provide no output.</p>

@({
fgrep -a '**' make-regression.log
})

<p>Unusual books that create output in the log file should not produce the
string @('**') except upon failure.</p>

<p>By default, @('make') commands for certifying books take advantage of files
@('*@useless-runes.lsp').  See @(see useless-runes).</p>

<h3>Prerequisites</h3>

<p>We assume that you have already downloaded and installed ACL2 as per the <a
href=\"@(`(:raw *installation-url*)`)\">ACL2 installation instructions</a> on
the ACL2 home page.</p>

<p>We assume you know the path to your ACL2 executable.  Typically this is a
script named @('saved_acl2') in your @('acl2-sources') directory.</p>

<p>We assume the ACL2 @(see community-books) are installed in the @('books/')
subdirectory of your ACL2 distribution, as is the case when you have followed
the ACL2 installation instructions above.</p>

<p>The instructions below are suitable for ACL2 and all of its experimental
extensions, e.g., ACL2(p) and ACL2(r).</p>

<p>It may be preferable to avoid being logged in as root, since developers do
not test as root and at least one community book
(@('books/oslib/tests/copy.lisp')) has failed to certify when logged in as
root.</p>

<h3>A Basic Build</h3>

<p>Before ACL2 Version 6.4, building the Community Books could take several
hours.  Now, the default @('make') target in @('books/GNUmakefile'), called
@('basic'), is much faster &mdash; it excludes many books and certifies only
books listed below, which tend to be widely used.  <b>WARNING</b>: the
@('basic') target of @('books/GNUmakefile') is insufficient for validating
changes that will go into the @(see community-books); for that, use the
@('all') target.</p>

<ul>
<li>arithmetic</li>
<li>arithmetic-3</li>
<li>arithmetic-5</li>
<li>@(see ihs)</li>
<li>misc</li>
<li>tools</li>
<li>@(see std)</li>
<li>@(see xdoc)</li>
<li>data-structures</li>
</ul>

<p>To certify these books, you should be able to run @('make') as follows.  The
@('-j 2') part of this command is suitable for a computer with two cores.  If
you have, e.g., a quad-core computer, you should probably use @('-j 4')
instead, and so on.</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make ACL2=/path/to/acl2-sources/saved_acl2 -j 2 basic
})

<p>If you configure your @('PATH') so that you can launch ACL2 by typing
@('acl2'), then you may omit the @('ACL2=...') part.</p>


<h3>Certifying Additional Books</h3>

<p>We expect that most ACL2 users will want to certify at least the @('basic')
books described above.  But what if you also need other books?  One option is
to do a full build (see below).  But it is usually <b>much faster</b> to simply
tell @('make') to build the books you actually want to use.</p>

<p>There are @('make') targets corresponding to most directory names.  For
instance, to build the books under @('coi') and @('rtl') and @('cgen'), you can
run:</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make ACL2=/path/to/acl2-sources/saved_acl2 coi rtl cgen -j 2
})

<p>For finer grained control, you can name individual books.  This works
particularly well for libraries that have @('top') books.  For instance, if you
want the @('rtl/rel9') library, you could run:</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make ACL2=/path/to/acl2-sources/saved_acl2 rtl/rel9/lib/top.cert -j 2
})


<h3>Books that Require ACL2 Extensions</h3>

<p>Some books require experimental extensions to ACL2, such as ACL2(p) (see
@(see parallelism)) or ACL2(r) (see @(see real)).  Other books require certain
additional software.</p>

<p>The build system will automatically determine which kind of ACL2 you are
running (ACL2, ACL2(p), or ACL2(r)) and, based on this, may prevent
incompatible books from being certified.  The output of @('make') should
explain which books are being excluded and why.</p>

<p>These kinds of book requirements are controlled by special @(see
build::cert_param) comments.</p>


<h3>Books that Require Quicklisp</h3>

<p>Some books, especially @(see interfacing-tools) like @(see oslib) and the
ACL2 @(see bridge), require certain Common Lisp libraries.</p>

<p>These libraries are now bundled with ACL2 via @(see quicklisp), so you
should not need to download anything extra to use them.  They are enabled by
default for all host Lisps except GCL, but you can avoid books that depend on
Quicklisp libraries by setting @('USE_QUICKLISP=0') in your
@('make') command.</p>

<p>Using Quicklisp should definitely work if the host Lisp is CCL or SBCL.
There is some chance it will work with Allegro CL, LispWorks, and CMUCL.  It
will almost certainly <b>not</b> work for GCL (at least as of 2018).</p>


<h3>Books that Require Additional Software</h3>

<p>Some other books based on @(see satlink) and @(see gl) require a SAT solver,
typically Glucose, to be installed; see @(see satlink::sat-solver-options) for
installation options.  The build system should automatically determine if
Glucose is installed on your system, and will avoid trying to certify these
books unless Glucose is present.</p>


<h3>Building the manual</h3>

<p>If you just want to get a copy of the ACL2+Books manual for local viewing,
you probably <b>don't need to build it yourself</b> because you can just <a
href='download/'>download</a> a copy.  If for some reason you do want to build
the manual yourself, you should be able to do so as follows, provided you have
installed glucose.  (That requirement might be eliminated in the future.)</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make manual -j 4
})

<p>Building the manual should work on at least CCL and SBCL on Linux and Mac OS
X.  It <b>may not work</b> for some other OS/Lisp combinations.  In particular,
building the manual requires some features from @(see oslib) and @(see
quicklisp) that may not be available on some other Lisps.</p>

<p>The resulting web-based manual may be found in:</p>

@({
    acl2-sources/books/doc/manual/index.html
})

<p>See also @(see acl2::acl2-doc) for details about how to build your own
Emacs-based manual, and @(see xdoc::save) for general information about how to
build and distribute custom XDOC manuals, e.g., manuals that additionally
include your own unreleased books.</p>


<h3>A Full Build</h3>

<p>Building all of the books can take hours and is <b>usually unnecessary</b>.
That said, it is easy to do: just run @('make all'), for example as follows.
(But as noted above, you may omit @('\"ACL2=...\"') if a suitable executable
named @('acl2') is on your Unix @('PATH'), such as @('bin/acl2').)</p>

@({
    $ cd /path/to/acl2-sources/books
    $ make ACL2=/path/to/acl2-sources/saved_acl2 -j 2 all
})

<p>This includes a few books that are quite slow to certify.  You can
exclude those by replacing ``@('all')'' by ``@('regression')'' in the
command above.</p>


<h3>Cleaning Up</h3>

<p>If you want to delete generated files, you can run @('make clean') to remove
certificates, compiled files, and build logs.</p>

<p>If you just want to remove the files in a particular subdirectory (and its
subdirectories), you can go into that directory and then run the
@('build/clean.pl') script.  This will delete, starting from your current
directory, recursively, all certificates, logs, compiled files, etc.</p>

<p>Note that @('make clean') doesn't remove some files, e.g., @(see xdoc)
manuals.  To remove everything, try @('make moreclean').</p>


<h3>Debugging Failed Certifications</h3>

<p>If a book fails to certify, you may want to try certifying it in an
interactive session.  The most reliable way to do this is to replicate the
environment and commands that the build system used.  This information can be
found at the top of the @('[bookname].cert.out') file.  For instance:</p>

@({
    ;; foo.cert.out
    -*- Mode: auto-revert -*-
    ...
    Environment variables:
    ACL2_CUSTOMIZATION=NONE                 ;; <-- first configure your
    ACL2_SYSTEM_BOOKS=/path/to/acl2/books   ;;     environment to match
    ACL2=/path/to/saved_acl2                ;;     these settings
    ...
    Temp lisp file:
    (acl2::value :q)                 ;; <--- then submit these commands to
    (acl2::in-package \"ACL2\")      ;;      $ACL2 to debug the failure
    ...                              ;;      interactively
    --- End temp lisp file ---
})

<p>Some other notes/tips:</p>

<ul>

<li>Make sure the ACL2 image you run is the same as the one listed as ACL2 in
those environment variables!</li>

<li>You may wish to set the environment variables for only the duration of your
ACL2 session by using the \"env\" command.</li>

<li>You may wish to edit some of the commands for better debugging purposes;
e.g. you may modify the @(see set-inhibit-output-lst) command, or insert a
@(see set-debugger-enable) command, etc.</li>

<li>If you don't want your session to exit after a successful certification,
replace the last form @('(er-progn (time$ (certify-book ...') with just the
@('(time$ (certify-book ...))') part.</li>

</ul>


<h3>Further Resources</h3>

<p>The build system is largely based on @(see BUILD::cert.pl).  There is
considerable documentation about @('cert.pl'), and we highly recommend using it
to manage your own ACL2 projects.</p>

<p>The main build script is @('books/GNUmakefile').  There are many comments at
the start of this file, and you can also inspect it to see what targets are
available.</p>

<p>Please feel absolutely free to contact the @(see acl2-help) mailing list
with any questions about building the community books.</p>")

(defxdoc books-certification-alt
  :parents (books-certification)
  :short "Alternate instructions for certifying the @(see community-books),
 from the perspective of the @('acl2-sources') directory."

  :long "<p>WARNING: Parts of this documentation are probably obsolete, but parts
 are still relevant.  See @(see books-certification) as the primary source of
 information on how to certify the @(see community-books).</p>

 <p>For background on the ACL2 community books, see @(see community-books).
 Here we explain how to certify those books, or some of those books, with ACL2.
 We thank Bishop Brock, Jared Davis, and Sol Swords for their substantial
 contributions to this methodology.  See @('books/GNUmakefile'), in the
 community books, for more about ``Credits and History'', and for additional
 technical details not covered in this topic.</p>

 <p>For more information about installing ACL2, see the <a href=\"@(`(:raw
 *installation-url*)`)\">ACL2 installation instructions</a>.  For information
 about so-called ``classic ACL2 `make'-based certification'', which provides
 support for certifying directories of books but may disappear in a future ACL2
 release, see @(see books-certification-classic).</p>

 <p><b>The Basics</b></p>

 <p>We make the following assumptions.</p>

 <ul>
  <li>Gnu `make' is available on your system via the `make' command
 (rather than some other flavor of `make').  (Execute `@('make --version') to
 verify this.)</li>

  <li>You have built or obtained an ACL2 executable.</li>

  <li>The ACL2 @(see community-books) are installed in the @('books/')
 subdirectory of your ACL2 distribution, as is the case when you have followed
 the standard installation instructions.</li> </ul>

 <box><p>Note: All commands shown below are issued in the top-level (ACL2
 sources) directory of your ACL2 distribution.</p></box>

 <p>By default the ACL2 executable is file @('saved_acl2') in your ACL2
 sources directory, and you can issue the following command to the shell in
 order to do a ``regression run'' that certifies all of the community books
 using that executable.</p>

 @({
  make regression
 })

 <p>Better yet, save a log file in case there are problems, for example as
 follows.</p>

 @({
  (make regression) >& make-regression.log
 })

 <p>or perhaps better yet:</p>

 @({
  (time nice make regression) >& make-regression.log
 })

 <p>For the sake of brevity, below we'll skip mentioning any of `@('time')',
 `@('nice')', or `@('>& make-regression.log')'.  But saving a log file, in
 particular, is useful in case you encounter problems to report.</p>

 <p>If you fetched the community books using git, then you will have a
 directories such @('books/workshops/') that is not necessary for certifying
 the most widely-included books.  You can certify just such books as
 follows.</p>

 @({
  (time nice make basic) >& make-basic.log
 })

 <p>Whether you use target `@('regression')' or target `@('basic')', then for
 each book @('foo.lisp') whose certification is attempted, a file
 @('foo.cert.out') in the same directory will contain the output from the
 book's certification attempt.</p>

 <p>A regression run may take a few hours, but if you have a multiprocessing
 computer, you can speed it up by certifying some books in parallel, by
 providing a value for `make' option @('-j').  For example, if you have 8
 hardware threads then you might want to issue the following command.</p>

 @({
  make regression -j 8
 })

 <p><b>Specifying the ACL2 Executable</b></p>

 <p>If your ACL2 executable is not file @('saved_acl2') in the ACL2 sources
 directory, then you will need to specify that executable.  You can do that by
 setting variable @('ACL2'), either as an environment variable or, as displayed
 below, as a `make' variable.  Either way, you will need to avoid relative
 pathnames.  For example, the first two forms below are legal, but the third is
 not, assuming that @('my-acl2') is on your @('PATH') in a Unix-like
 environment (e.g., linux or MacOS) and that @('my-saved_acl2') is just a
 pathname relative to your ACL2 sources directory, which is not on your
 path.</p>

 @({
  make regression -j 8 ACL2=my-acl2
  make regression -j 8 ACL2=/u/smith/bin/acl2
  # The following only works if my-saved_acl2 is on your path (see above).
  make regression -j 8 ACL2=my-saved_acl2
 })

 <p><b>Cleaning</b></p>

 <p>You can delete files generated by book certification (including @('.cert')
 files, @('.out') files, compiled files, and more) by issuing the following
 command (again, in your ACL2 sources directory).</p>

 @({
  make clean-books
 })

 <p>Alternatively, if you want to cause such deletion and then do a regression,
 simply replace the `@('regression')' target by `@('regression-fresh').</p>

 <p>If however you only want to clean up generated files residing under a given
 directory (or its subdirectories, and recursively), you can issue the
 following command while standing in that directory, where @('DIR') is a
 pathname of your @('books') directory.</p>

 @({
  DIR/clean.pl
 })

 <p>For example, to clean up generated files under @('books/arithmetic'), you
 could do the following.</p>

 @({
  cd books/arithmetic
  ../clean.pl
  cd - # to return to the ACL2 sources directory, if you wish to do so
 })

 <p><b>Restricting to Specific Directories and Books</b></p>

 <p>You can specify which books you want certified by using any or all of the
 variables @('EXCLUDED_PREFIXES'), @('ACL2_BOOK_CERTS'), or
 @('ACL2_BOOK_DIRS').  First, the set of desired @('.cert') files is restricted
 to those that do not start with any string that is one of the words in the
 value of @('EXCLUDED_PREFIXES').  Then @('ACL2_BOOK_CERTS') and
 @('ACL2_BOOK_DIRS'), if supplied, specify which books should be certified, as
 illustrated by the following example.</p>

 @({
  make -j 8 regression-fresh \\
   ACL2_BOOK_DIRS=\"symbolic paco\" \\
   ACL2_BOOK_CERTS=\" \\
    workshops/2006/cowles-gamboa-euclid/Euclid/ed6a.cert \\
    workshops/2006/cowles-gamboa-euclid/Euclid/ed4bb.cert \\
    \"
 })

 <p>Then all book in directories @('symbolic') and @('paco') will be certified,
 as will the books @('workshops/2006/cowles-gamboa-euclid/Euclid/ed6a.lisp')
 and @('workshops/2006/cowles-gamboa-euclid/Euclid/ed4bb.lisp').  Note that all
 pathnames should be relative to your community books directory; in particular,
 they should not be absolute pathnames.  Also notice the @('.cert') extension
 used in files supplied for @('ACL2_BOOK_CERTS').</p>

 <p>Alternatively, you may wish to invoke @('books/cert.pl') while standing in
 a directory under which you want to certify books.  This will certify not only
 those books, but all supporting books &mdash; even those not under the current
 directory &mdash; that do not have up-to-date @('.cert') files.  The following
 is a simple command to invoke that will certify all books in the current
 directory, where if the @('books/') directory is not on your path, you will
 need to provide a suitable filename, e.g. @('../../cert.pl') or
 @('~/acl2/books/cert.pl').</p>

 @({
  cert.pl -j 4 *.lisp
 })

 <p>Here is a more complex command, which illustrates a way to certify books in
 subdirectories (as well as the current directory), the use of provisional
 certification (see @(see provisional-certification)), and `make'-level
 parallelism (in this case specifying four parallel processes).</p>

 @({
  cert.pl --pcert-all -j 4 `find . -name '*.lisp'`
 })

 <p>Note that with this approach, unlike classic ACL2 `make'-based
 certification (see @(see books-certification-classic), out-of-date @('.cert')
 files that are not under the current directory will also be built.  For
 documentation of @('cert.pl') invoke:</p>

 @({
  cert.pl -h
 })

 <p>See the top of @('cert.pl') for authorship and copyright information.</p>

 <p>Finally, we give a brief summary of how to use so-called ``classic ACL2
 `make'-based certification'' for community books; see @(see
 books-certification-classic) for details.  Note that support for this approach
 might be eliminated in a future ACL2 release.  We welcome comments from the
 ACL2 community about whether or not that would be a good thing to do.  See the
 discussion above about @('ACL2_BOOK_DIRS') for the ``modern'' way to
 accomplish the same thing.</p>

 <p>Many community book directories have a @('Makefile').  If you modify books
 only in such a directory, you can recertify by standing in that directory and
 issuing a `make' command.  This command can optionally specify an ACL2
 executable as well as parallelism, for example as follows, where the first
 line (@('make clean')) is optional.</p>

 @({
  make clean
  (time nice make -j 8 ACL2=my-acl2)
 })

 <p><b>ACL2 Customization Files</b></p>

 <p>By default, your acl2-customization file (see @(see acl2-customization)) is
 ignored by all flavors of ``@('make regression')''.  However, you can specify
 the use of an acl2-customization file by setting the value of environment
 variable @('ACL2_CUSTOMIZATION') to the empty string, indicating a default
 such file, or to the desired absolute pathname.  For example:</p>

 @({
  make regression ACL2_CUSTOMIZATION=''
  make regression ACL2_CUSTOMIZATION='~/acl2-customization.lisp'
 })

 <p><b>Regressions for Variants of ACL2</b></p>

 <p>The discussion above also pertains to using ACL2(p) (see @(see parallel))
 or ACL2(r) (see @(see real)), in which case the default is @('saved_acl2p') or
 @('saved_acl2r') respectively, rather than @('saved_acl2').  However, we
 recommend that you use ACL2, not ACL2(p), for your regression.  Then you can
 use ACL2(p) for your own proof developments.  However, if you want to use
 ACL2(p) or for your regression, see @(see
 waterfall-parallelism-for-book-certification).</p>

 <p><b>Provisional Certification</b></p>

 <p>To use provisional certification (see @(see provisional-certification)),
 supply @('ACL2_PCERT=t') with your `make' command.  Here is an example.</p>

 @({
  time nice make regression -j 4 ACL2_BOOK_DIRS=deduction ACL2_PCERT=t
 })

 <p><b>Miscellany</b></p>

 <p>Other control of the certification process may be found by perusing
 community books file @('books/make_cert').  In particular, the @('INHIBIT')
 variable may be set to a call of @(tsee set-inhibit-output-lst), for example
 as follows to obtain the output one would get by default in an (interactive)
 ACL2 session.</p>

 @({
  time nice make regression -j 4 ACL2_BOOK_DIRS=arithmetic \\
    INHIBIT='(set-inhibit-output-lst proof-tree)'
 })

 <p><b>Troubleshooting</b></p>

 <p>If you run into problems, you can get help by joining the @('acl2-help')
 email list (follow the link from the ACL2 home page) and sending a message to
 that list.  Also consider trying another version of GNU `make'; for example,
 we have found that versions 3.81 and 3.82 sometimes cause errors on Linux
 where version 3.80 does not.  Note however that Version 3.80 does not print
 certain informational messages that are printed by later versions.</p>")

(defxdoc books-certification-classic
  :parents (books-certification)
  :short "Classic ACL2 `make'-based certification of @(see books)"
  :long "<p>This @(see documentation) topic explains an approach to certifying
 directories of books, which we call ``classic ACL2 `make'-based
 certification''.</p>

 <p>Warning: The capability described in this section might be replaced at any
 time by a capability based on corresponding support for community books (see
 @(see books-certification)).  If you think that would be a hardship, please
 contact the ACL2 implementors.</p>

 <p>This topic discusses a way to certify a directory of books other than the
 ACL2 community books.  See @(see books-certification) for how to certify the
 set of ACL2 community @(see books).  There is also a section in that @(see
 documentation) topic, ``Restricting to Specific Directories and Books'', that
 provides an alternative to classic ACL2 `make'-based certification (as
 discussed in the present topic) for certifying specified sets of books.</p>

 <p>We assume here a familiarity with Unix/Linux `make'.  We also assume that
 you are using GNU `make' rather than some other flavor of `make'.  And
 finally, we assume, as is typically the case by following the standard
 installation instructions, that you install the ACL2 community books in the
 @('books/') subdirectory of your ACL2 distribution.  We will refer below to
 that directory as @('BOOKS').</p>

 <p>In summary: to use `make' to certify @(see books) under a given directory,
 you may create a simple Makefile in that directory (as explained below) so
 that when you stand in that directory, you can submit the command, `make', to
 certify those books.  If you have a multi-processor machine or the like, then
 you can use the `@('-j') flag `make'-level parallelism by specifying the
 number of concurrent processes.  For example:</p>

 @({
  make -j 4
 })

 <p>For each book @('foo.lisp'), a file @('foo.out') in the same directory as
 @('foo.lisp') will contain the output from the corresponding certification
 attempt.  If you have previously executed such a command, then you might first
 want to delete @(see certificate) files and other generated files by executing
 the following command.</p>

 @({
  make clean
 })

 <p>Note that when you run `make', then by default, the first error will cause
 the process to stop.  You can use @('make -i') to force `make' to ignore
 errors, thus continuing past them.  Or, use @('make -k') to keep going, but
 skipping certification for any book that includes another whose certification
 has failed.</p>

 <p>By default, your acl2-customization file (see @(see acl2-customization)) is
 ignored by such `make' commands.  However, you can specify the use of an
 acl2-customization file by setting the value of environment variable
 @('ACL2_CUSTOMIZATION') to the empty string, indicating a default such file,
 or to the desired absolute pathname.  For example:</p>

 @({
  make ACL2_CUSTOMIZATION=''
  make ACL2_CUSTOMIZATION='~/acl2-customization.lisp'
 })

 <p>We now discuss how to create makefiles to support `make' commands as
 discussed above.</p>

 <p>First we give five steps for creating a @('Makefile') to support
 certification of a directory of books, without subdirectories.  For examples
 of such Makefiles you can look in community book directories (which, however,
 might disappear in future versions of ACL2).</p>

 <blockquote><p>1. Include the file @('Makefile-generic') from the @('books/')
 subdirectory of your ACL2 sources directory, but first perhaps define the
 variable `@('ACL2')'.  Consider the following example.</p>

 @({
  ACL2 ?= /Users/john_doe/acl2/acl2-sources/saved_acl2
  include /Users/john_doe/acl2/acl2-sources/books/Makefile-generic
 })

 <p>In this example, you can omit the first line, because the default ACL2
 executable is file @('saved_acl2') in the directory immediately above the
 directory of the specified @('Makefile-generic') file.  Indeed, that is the
 common case.  Note the use of @('?=') instead of @('=') or @(':='), so that
 @('ACL2') can instead be defined by the environment or provided on the command
 line as part of the `make' command.</p>

 <p>2. (Optional; usually skipped.)  Set the @('INHIBIT') variable if you want
 to see more than the @(see summary) output.  For example, if you want to see
 the same output as you would normally see at the terminal, put this line in
 your Makefile after the `@('include')' lines.</p>

 @({
  INHIBIT = (assign inhibit-output-lst (list (quote proof-tree)))
 })

 <p>For other values to use for @('INHIBIT'), see @(see set-inhibit-output-lst)
 and see the original setting of @('INHIBIT') in
 @('books/Makefile-generic').</p>

 <p>3. Specify the books to be certified.  Normally, every file with extension
 @('.lisp') will be a book that you want to certify, in which case you can skip
 this step.  Otherwise, put a line in your @('Makefile') after the ones above
 that specifies the books to be certified.  The following example, from an old
 version of community books file @('books/finite-set-theory/osets/Makefile'),
 should make this clear.</p>

 @({
  BOOKS = computed-hints fast instance map membership outer primitives \\
          quantify set-order sets sort
 })

 <p>But better yet, use the extension @('.lsp') for any Lisp or ACL2 files that
 are not to be certified, so that the definition of @('BOOKS') can be
 omitted.</p>

 <p>4. Create @('.acl2') files for books that are to be certified in other than
 the initial ACL2 world (see @(see portcullis)).  For example, if you look in
 community books file @('books/arithmetic/equalities.acl2') you will see @(tsee
 defpkg) forms followed by a @(tsee certify-book) command, because it was
 determined that @(tsee defpkg) forms were necessary in the certification world
 in order to certify the @('equalities') book.  In general, for each
 @('<book-name>.lisp') whose certification requires a non-initial certification
 world, you will need a corresponding @('<book-name>.acl2') file that ends with
 the appropriate @(tsee certify-book) command.</p>

 <p>You also have the option of creating a file @('cert.acl2') that has a
 special role.  When file @('<book-name>.lisp') is certified, if there is no
 file @('<book-name>.acl2') but there is a file @('cert.acl2'), then
 @('cert.acl2') will be used as @('<book-name>.acl2') would have been used, as
 described in the preceding paragraph, except that the appropriate @(tsee
 certify-book) command will be generated automatically.  Thus, no
 @('certify-book') command should occur in @('cert.acl2').</p>

 <p>It is actually allowed to put raw lisp forms in a @('.acl2') file
 (presumably preceded by @(':q') or @('(value :q)') and followed by @('(lp)')).
 But this is not recommended; we make no guarantees about certification
 performed any time after raw Lisp has been entered in the ACL2 session.</p>

 <p>5. Generally, the next step is to include the following line after the
 `@('include')' of @('Makefile-generic') (see the first step above).</p>

 @({
  -include Makefile-deps
 })

 <p>This will cause `make' to create and then include a file @('Makefile-deps')
 that contains ``dependency'' lines needed by `make'.  If those dependencies
 are somehow flawed, it may be because you have @(tsee include-book) forms that
 are not truly including books, for example in multi-line comments
 (@('#|..|#')).  These will be ignored if preceded by a semicolon (@(';')), or
 if you add a line break after ``@('include-book').''  But instead of adding
 the `@('-include')' line above, you can create dependency lines yourself by
 running the command</p>

 @({
  make dependencies
 })

 <p>and pasting the result into the end of your @('Makefile'), and editing as
 you see fit.</p></blockquote>

 <p>This concludes the basic instructions for creating a @('Makefile') in a
 directory including books.  Here are some other capabilities offered by
 community books file @('books/Makefile-subdirs').  Not included below is a
 discussion of how to increase parallelism by avoiding the need to certify
 included books before certifying a given book; see @(see
 provisional-certification).</p>

 <p><b>Subdirectory Support</b></p>

 <p>There is support for using `make' to certify books in subdirectories.
 Consider the following example.</p>

 @({
  DIRS = pass1 bind-free floor-mod
  include ../Makefile-subdirs
 })

 <p>This indicates that we are to run `make' in subdirectories @('pass1/'),
 @('bind-free/'), and @('floor-mod/') of the current directory.</p>

 <p>You can combine this subdirectory support with the support already
 discussed for certifying books in the top-level directory.  Here is an
 example, which as of this writing is in community books file
 @('books/arithmetic-3/Makefile') contains the following lines.</p>

 @({
  arith-top: top all
  all: top

  DIRS = pass1 bind-free floor-mod
  include ../Makefile-subdirs
  include ../Makefile-generic

  -include Makefile-deps
 })

 <p>The `@('top')' target is defined in @('../Makefile-subdirs') to call `make'
 in each subdirectory specified in @('DIRS').  We have set the default target
 in the example above to a new name, @('arith-top'), that makes that @('top')
 target before making the `@('all')' target which, in turn, is the default
 target in any @('Makefile-generic'), and is responsible for certifying books
 in the current directory as discussed in the five steps displayed above.</p>

 <p>Use @('Makefile-psubdirs') instead of @('Makefile-subdirs') if
 certification of a book in a subdirectory never depends on certification of a
 book in a different subdirectory, because then the @('-j') option of `make'
 can allow subdirectories to be processed in parallel.</p>

 <p><b>Cleaning Up</b></p>

 <p>We note that there is a @('clean') target.  Thus,</p>

 @({
  make clean
 })

 <p>will remove generated files including @('.cert'), @('.out') files, and
 compiled files.</p>

 <p><b>System Books</b></p>

 <p>An environment variable @('ACL2_SYSTEM_BOOKS') is generally set
 automatically, so you can probably skip reading the following paragraph unless
 your attempt to certify books fails to locate those books properly.</p>

 <p>The environment variable @('ACL2_SYSTEM_BOOKS') can be set to the top-level
 directory of the ACL2 community books.  A Unix-style pathname, typically
 ending in @('books/') or @('books'), is permissible.  In most cases, your ACL2
 executable is a small script in which you can set this environment variable
 just above the line on which the actual ACL2 image is invoked, for
 example:</p>

 @({
  export ACL2_SYSTEM_BOOKS
  ACL2_SYSTEM_BOOKS=/home/acl2/v3-2/acl2-sources/books
 })

 <p>However, you can also set @('ACL2_SYSTEM_BOOKS') as a `make' variable, by
 setting it in your @('Makefile') before the first target definition, e.g.:</p>

 @({
  ACL2_SYSTEM_BOOKS ?= /home/acl2/v3-2/acl2-sources/books
 })

 <p><b>Compilation Support</b></p>

 <p>The file @('books/Makefile-generic') provides support for compiling books
 that are already certified (but see @(see compilation) for an exception).  For
 example, suppose that you have certified books using GCL as the host Lisp,
 resulting in compiled files with the @('.o') extension.  Now suppose you would
 like to compile the books for Allegro Common Lisp, whose compiled files have
 the @('.fasl') extension.  The following command will work if you have
 included @('books/Makefile-generic') in your @('Makefile').</p>

 @({
  make fasl
 })

 <p>In general, the compiled file extension for a Lisp supported by ACL2 will
 be a target name for building compiled files for all your books (after
 certifying the books, if not already up-to-date on certification).</p>

 <p>If you run into problems, you can get help by joining the @('acl2-help')
 email list (follow the link from the ACL2 home page) and sending a message to
 that list.  Also consider trying another version of GNU `make'; for example,
 we have found that versions 3.81 and 3.82 sometimes cause errors on Linux
 where version 3.80 does not.</p>")

(defxdoc books-reference
  :parents (books)
  :short "Reference guide for ACL2 functionality related to books, e.g.,
@(see include-book), @(see certify-book), @(see cbd), etc.")

(defxdoc books-tour
  :parents (books)
  :short "The <i>guided tour</i> of concepts related to ACL2 @(see books)."
  :long "<p>The tour begins with @(see book-example).</p>")

(defxdoc boole$
  :parents (numbers acl2-built-ins)
  :short "Perform a bit-wise logical operation on 2 two's complement integers"
  :long "<p>When integers @('x') and @('y') are viewed in their two's
 complement representation, @('(boole$ op x y)') returns the result of applying
 the bit-wise logical operation specified by @('op').  The following table is
 adapted from documentation for the analogous Common Lisp function
 @(`(:raw (clhs \"Body/f_boole.htm\" \"boole\"))`) in the @(`(:raw (clhs \"\"
 \"Common Lisp Hyperspec\"))`).  Note that the values of @('op') for
 @('boole$') are ACL2 constants, rather than corresponding values of @('op')
 for the Common Lisp function @('boole').</p>

 @({
  op               result
  -----------      ---------
  *boole-1*        x
  *boole-2*        y
  *boole-andc1*    and complement of x with y
  *boole-andc2*    and x with complement of y
  *boole-and*      and
  *boole-c1*       complement of x
  *boole-c2*       complement of y
  *boole-clr*      the constant 0 (all zero bits)
  *boole-eqv*      equivalence (exclusive nor)
  *boole-ior*      inclusive or
  *boole-nand*     not-and
  *boole-nor*      not-or
  *boole-orc1*     or complement of x with y
  *boole-orc2*     or x with complement of y
  *boole-set*      the constant -1 (all one bits)
  *boole-xor*      exclusive or
 })

 <p>The guard of @('boole$') specifies that @('op') is the value of one of the
 constants above and that @('x') and @('y') are integers.</p>

 <p>See any Common Lisp documentation for analogous information about Common
 Lisp function @('boole').</p>

 @(def boole$)")

(defxdoc boolean-listp
  :parents (booleanp lists acl2-built-ins)
  :short "Recognizer for a true list of booleans"
  :long "<p>The predicate @('boolean-listp') tests whether its argument is a
 @(tsee true-listp) of objects each or which satisfies @(tsee booleanp), i.e.,
 is @('t') or @('nil').</p>

 @(def boolean-listp)")

(defxdoc booleanp
  :parents (basics acl2-built-ins)
  :short "Recognizer for booleans"
  :long "<p>@('(Booleanp x)') is @('t') if @('x') is @('t') or @('nil'), and is
 @('nil') otherwise.</p>

 <p>See @(see generalized-booleans) for a discussion of a potential soundness
 problem for ACL2 related to the question: Which Common Lisp functions are
 known to return Boolean values?</p>

 @(def booleanp)")

(defxdoc bounders
  :parents (tau-system)
  :short "Intervals, bounder functions, and bounder correctness"
  :long "<code>
 <i>Bounder Forms 1 and 2</i>:
 (implies (and (tau-intervalp i1)
               ...
               (or (equal (tau-interval-dom i1) 'dom1-1)
                   ...)
               ...
               (in-tau-intervalp x1 i1)
               ...)
          (and (tau-intervalp (bounder-fn i1 ...))
               (in-tau-intervalp <i>target</i>
                                 (bounder-fn i1 ...))))
 </code>

 <p>where <i>target</i> is either @('(fn x1 ... y1 ...)') or @('(mv-nth 'n (fn
 x1 ... y1 ...))'), depending on whether we are in the <i>Form 1</i> or <i>Form
 2</i> case, respectively.  However, the shape above is meant just as a
 reminder.  Details are given below.</p>

 <p>This topic first explains the basic shape of <i>Bounder Form 1</i>.  Then
 it illustrates <i>Bounder Form 2</i>.  Finally, it deals briefly with proving
 bounder correctness theorems.  The community book
 @('tau-bounders/elementary-bounders') contains bounders for various elementary
 functions including @(tsee +), @(tsee *), @(tsee /), @(tsee FLOOR), @(tsee
 MOD), @(tsee LOGAND), @(tsee LOGNOT), @(tsee LOGIOR), @(tsee LOGORC1), @(tsee
 LOGEQV), @(tsee LOGXOR), and @(tsee ASH).  You might look at or include this
 book to see more example theorems, to see how proofs of such theorems are
 managed, and to experiment with their effects on proving theorems involving
 arithmetic over finite or half-finite intervals.</p>

 <p>A bounder correctness theorem establishes that @('bounder-fn') is a
 ``bounder'' for the function @('fn').  That means that when trying to compute
 a tau for a call of @('fn') (or, in the case of <i>Form 2</i>, for the
 @('n')th component of the multiple-value vector returned by a call of @('fn'))
 the tau system can call @('bounder-fn') on the intervals containing certain
 arguments of @('fn').</p>

 <p>Let us start with an example.  Let @('fn') be the addition function, @('+')
 (actually, @(tsee binary-+)).  Consider the target term @('(+ x y)') and
 contemplate the question: if you know intervals containing @('x') and @('y'),
 say @('intx') and @('inty') respectively, what is an interval containing their
 sum?  The answer is pretty easy to state in English: the domain of the answer
 interval is the less restrictive of the domains of @('intx') and @('inty').
 The lower bound of the answer interval is the sum of the lower bounds of
 @('intx') and @('inty'), and the lower relation is the stronger of the lower
 relations of @('intx') and @('inty').  Analogous comments define the upper
 bound and relation of the answer interval.  So for example, if @('x') is an
 @('INTEGERP') such that @('0 <= x <= 10') and @('y') is a @('RATIONALP') such
 that @('0 < y <= 20'), then @('(+ x y)') is a @('RATIONALP') such that @('0 <
 (+ x y) <= 30').</p>

 <p>Defining this precisely is more tedious than describing it in English
 because one must make precise the notions of ``less restrictive'' domains,
 ``weaker'' relations, and the possibility that either or both of the bounds
 could be ``infinite.''  But we can easily imagine defining the function
 @('bounder-for-+') that returns the answer interval described, given @('intx')
 and @('inty').</p>

 <p>Then the following <i>Bounder Form 1</i> formula establishes the
 correctness of @('bounder-for-+') and allows the tau system to use it to
 produce bounds in the tau computed for @('+')-expressions:</p>

 @({
  (implies (and (tau-intervalp intx)
                (tau-intervalp inty)
                (in-tau-intervalp x intx)
                (in-tau-intervalp y inty))
           (and (tau-intervalp (bounder-for-+ intx inty))
                (in-tau-intervalp (+ x y)
                                  (bounder-for-+ intx inty))))
 })

 <p>For example, suppose we have a formula with the following hypotheses</p>

 @({
  (and (integerp a)
       (<= 0 a)
       (<= a 10)
       (rationalp b)
       (< 0 b)
       (<= b 20))
 })

 <p>and suppose the tau system encounters the term @('(+ a b)').  When the term
 is encountered, the tau for @('a') would include an @('INTEGERP') interval such
 that @('0 <= a <= 10') and the tau for @('b') would include a @('RATIONALP')
 interval such that @('0 < b <= 20').  In its most primitive configuration, the
 tau system would only know that the tau for @('(+ a b)') includes the
 recognizer @('RATIONALP') (and all that it is known to imply).  But after the
 bounder theorem above is proved and available as a @(':tau-system') rule the
 tau system would infer that @('(+ a b)') was in the @('RATIONALP') interval
 such that @('0 < (+ a b) <= 30').</p>

 <p>Thus, by defining bounder functions and proving them correct the user can
 give the tau system the ability to compute the bounds on function calls as a
 function of the known bounds on their actuals.</p>

 <p>It is sometimes useful to restrict the domains of the intervals to be
 considered.  For example, in bounding @('*')-expressions it is simplifying to
 restrict one's attention to intervals over the integers or rationals (and thus
 exclude the complex rationals so one need not think about the getting negative
 bounds by multiplying two ``positive'' complex rationals or how to ``round
 up'' from complex bounds to the rationals required by our intervals).</p>

 <p>If we were to define @('bounder-for-*') so that it works correctly to bound
 @('*')-expressions, but only for integer or rational arguments, its
 correctness theorem would be:</p>

 @({
  (implies (and (tau-intervalp intx)                             ; (a)
                (tau-intervalp inty)
                (or (equal (tau-interval-dom intx) 'INTEGERP)    ; (b)
                    (equal (tau-interval-dom intx) 'RATIONALP))
                (or (equal (tau-interval-dom inty) 'INTEGERP)
                    (equal (tau-interval-dom inty) 'RATIONALP))
                (in-tau-intervalp x intx)                        ; (c)
                (in-tau-intervalp y inty))
           (and (tau-intervalp (bounder-for-* intx inty))       ; (d)
                (in-tau-intervalp (* x y)                        ; (e)
                                  (bounder-for-* intx inty))))
 })

 <p>In this case, @('bounder-for-*') would be applied to the intervals for
 @('x') and @('y') only if those intervals were over the integers or the
 rationals.</p>

 <p>The above theorem for @('bounder-for-*') begins to suggest the general form
 of a bounder theorem and we will use it to explain the general form.</p>

 <p>The hypotheses of a bounder theorem must be a conjunction and the conjuncts
 must be partitionable into three parts, (a), (b), and (c).  The conclusion,
 must be a conjunction, must contain at least two conjuncts, (d) and (e), and
 is allowed to contain others that are simply ignored for purposes of bounders.
 (See the note below about why we allow but ignore additional conjuncts in the
 conclusion.)</p>

 <p>Part (a) introduces some distinct ``interval variables,'' here called
 ``ivars,'' that are known to denote intervals; for the example above, the
 ivars are @('intx') and @('inty').  Each hypothesis in part (a) is of the form
 @('(TAU-INTERVALP ivar)').</p>

 <p>Part (b) allows us to restrict the domains of some of the intervals.  Each
 hypothesis in part (b) must be a disjunction and each of the disjuncts must be
 of the form @('(EQUAL (TAU-INTERVAL-DOM ivar) 'dom)'), where @('ivar') is one
 of the interval variables and @('dom') is one of @('INTEGERP'),
 @('RATIONALP'), @('ACL2-NUMBERP'), or @('NIL').  It is not necessary to
 restrict every interval variable.  Indeed, part (b) may be empty, as in the
 theorem for @('bounder-for-+') above.</p>

 <p>Part (c) consists of a set of @('(IN-TAU-INTERVALP avar ivar)') hypotheses
 where each @('avar') is a variable and no two hypotheses in part (c) use the
 same @('avar') or @('ivar').  We call the set of all such @('avar') the
 ``actual variables'' or ``avars.''  The avars and ivars must be distinct.
 Part (c) sets up a correspondence between the avars and the ivars, each avar
 is in an interval denoted by one ivar.</p>

 <p>Part (d) introduces the name of the bounder function, here
 @('bounder-for-*'), and the order of its ivar arguments.  We see that
 @('bounder-for-*') takes two arguments and they correspond, in order, to the
 intervals containing @('x') and @('y').  Part (d) also establishes that the
 bounder function always returns an interval under hypotheses (a), (b), and
 (c).  Note that it is sometimes useful to return the ``universal interval''
 (one that contains everything) if you don't want to compute a better interval
 for some case; see @(tsee tau-intervalp) or @(tsee in-tau-intervalp).</p>

 <p>Part (e) introduces the name of the function being bounded, here @('*'),
 and the order of its arguments.  It establishes that the function being
 bounded really is bounded by the interval computed by the bounder function.
 In general, the function being bounded may take additional arguments.  It is
 possible that the function being bounded takes some arguments that do not
 affect the bounds of its output.</p>

 <p>Thus, parts (c) and (e) together establish a mapping between the actuals of
 a call of the function being bounded and the intervals to be supplied to the
 bounder.</p>

 <p>The parts identified above may be presented in any order and the literals
 constituting those parts may be mingled.  Thus, for example, here is another
 version of the theorem above that generates the same bounding information for
 the tau system.  In this version, the hypotheses and conclusions are
 rearranged, @('bounder-for-*') takes its arguments in the opposite order, and
 the theorem includes an additional conclusion.</p>

 @({
  (implies (and (tau-intervalp intx)                             ; (a)
                (or (equal (tau-interval-dom intx) 'INTEGERP)    ; (b)
                    (equal (tau-interval-dom intx) 'RATIONALP))
                (in-tau-intervalp x intx)                        ; (c)

                (tau-intervalp inty)                             ; (a)
                (or (equal (tau-interval-dom inty) 'INTEGERP)    ; (b)
                    (equal (tau-interval-dom inty) 'RATIONALP))
                (in-tau-intervalp y inty))
           (and (in-tau-intervalp (* x y)                        ; (e)
                                  (bounder-for-* inty intx))
                (tau-intervalp (bounder-for-* inty intx))        ; (d)))

                (or (equal (tau-interval-dom (bounder-for-* inty intx))
                           'INTEGERP)
                    (equal (tau-interval-dom (bounder-for-* inty intx))
                           'RATIONALP))
 })

 <p><i>Note on why bounder forms allow additional conjuncts in the
 conclusion</i>: It is often the case that one creates bounders by composing
 other bounders.  To prove compositional bounds correct one must often prove
 more than the mere correctness of the components.  For example, one might need
 to prove that the domain of the new bounding interval is @('INTEGERP') or
 otherwise restricted.  We allow such ``unnecessary'' conclusions simply to
 save the user the burden of stating multiple theorems.</p>

 <p><i>An Illustration of Bounder Form 2</i>: Suppose @('(quad i)') is defined
 so that truncates the integer @('i') to the largest multiple of 4 weakly below
 @('i') and, additionally, returns the remainder.  For example, @('(quad 26)')
 returns @('(mv 24 2)').  Then here are bounders for each of its return
 values:</p>

 @({
  (defun quad-bounds-0 (i)
    (cond ((and (tau-interval-lo i)
                (<= 0 (tau-interval-lo i)))
           (make-tau-interval 'integerp nil 0 nil (tau-interval-hi i)))
          (t (make-tau-interval nil nil nil nil nil))))

  (defun quad-bounds-1 (i)
    (cond ((and (tau-interval-lo i)
                (<= 0 (tau-interval-lo i)))
           (make-tau-interval 'integerp nil 0 nil 3))
          (t (make-tau-interval nil nil nil nil nil))))
 })

 <p>Note that the bounders assume @('i') is an @('INTEGERP') and return the
 universal interval when @('i') is not a natural.</p>

 <p>As noted in the discussion below about how to prove bounder correctness
 theorems, proving these bounders correct will require an arithmetic book,
 e.g.,</p>

 @({
  (include-book \"arithmetic-5/top\" :dir :system)
 })

 <p>Here then are two bounder correctness theorems of <i>Form 2</i>:</p>

 @({
  (defthm quad-bounds-0-correct
    (implies (and (tau-intervalp i)
                  (equal (tau-interval-dom i) 'INTEGERP)
                  (in-tau-intervalp x i))
             (and (tau-intervalp (quad-bounds-0 i))
                  (in-tau-intervalp (mv-nth 0 (quad x))
                                    (quad-bounds-0 i))))
    :rule-classes :tau-system)

  (defthm quad-bounds-1-correct
    (implies (and (tau-intervalp i)
                  (equal (tau-interval-dom i) 'INTEGERP)
                  (in-tau-intervalp x i))
             (and (tau-intervalp (quad-bounds-1 i))
                  (in-tau-intervalp (mv-nth 1 (quad x)) (quad-bounds-1 i))))
    :rule-classes :tau-system)
 })

 <p>As noted above, if these bounders are to be used in constructing other
 bounders, we might include (in the first theorem) an additional concluding
 conjunct, such as</p>

 @({
  (equal (tau-interval-dom (quad-bounds-0 i)) 'INTEGERP)
 })

 <p>so that we can keep @('quad-bounds-0') disabled to allow us to use
 @('quad-bounds-0-correct') as a @(':rewrite') or other rule and still relieve
 hypotheses about the domain of the interval it produces.  These hypotheses
 would arise if some other verified bounder was called on the produced
 interval.  In addition, as noted below, we might replace the
 @(':rule-classes') above with</p>

 @({
  :rule-classes
   ((:rewrite)
    (:forward-chaining :trigger-terms ((quad-bounds-0 i))))
 })

 <p>Since the theorem is being stored as some kind of rule and since it
 satisfies the <i>Bounder Form 2</i> shape, it will additionally be stored as a
 @(':tau-system') rule.</p>

 <p><i>Note on proving bounder theorems</i>: Proving bounder theorems is just
 like proving any other arithmetic theorem and you will need whatever libraries
 are appropriate for the problem domain you are working in.  Do not expect the
 tau system to be of much use in proving bounder theorems.  A typical bounder
 theorem might require you to prove a subgoal like @('(< (fn x y) (g
 (tau-interval-hi int1) int2))').  But tau deals with inequalities relating
 terms to constants, e.g., @('(< ... 16)').  A bounder theorem is a sort of
 ``metatheorem'' about <i>how to construct</i> bounded intervals from other
 bounded intervals.  So when you undertake to define a bounder and prove it
 correct, go into the project with your eyes open!</p>

 <p>But bounder functions can be broadly divided into two classes, those
 defined in terms of arithmetic on the interval bounds and those defined in
 terms of other bounders.  For example, given that</p>

 @({
  (LOGXOR x y) = (LOGNOT (LOGEQV x y))
 })

 <p>an interval for bounding @('LOGXOR') can be constructed by composing the
 constructions of intervals for @('LOGEQV') and @('LOGNOT').  So some bounder
 correctness proofs will involve direct manipulation of arithmetic inequalities
 and others might involve appeal to the correctness of other bounders,
 depending on how the new bounder is defined.</p>

 <p>Regardless of which style of bounder we are dealing with, we have found it
 useful to prove the basic theorems relating the tau interval accessors to
 @(tsee MAKE-TAU-INTERVAL), e.g.,</p>

 @({
  (equal (tau-interval-dom (make-tau-interval dom lo-rel lo hi-rel hi)) dom)
 })

 <p>and then disable those functions to avoid seeing excessive @('car')s and
 @('cdr')s.</p>

 <p>When dealing with bounders defined in the direct, arithmetic style, we tend
 to keep @(tsee TAU-INTERVALP) and @(tsee IN-TAU-INTERVALP) enabled so they
 unfold and expose the algebra.</p>

 <p>When dealing with bounders defined compositionally in terms of other
 verified bounders, we tend to keep @(tsee TAU-INTERVALP) and @(tsee
 IN-TAU-INTERVALP) disabled so we can rely on the previously proved bounder
 theorems as rewrite and forward chaining rules.</p>

 <p>Note that this last remark means that when you prove bounder correctness
 theorems you should include corollaries that are useful @(':rewrite') and
 possibly @(':forward-chaining') rules if you anticipate using that bounder in
 more complex ones.  We tend to trigger the forward chaining with the bounder
 expression itself, rather than one of the hypotheses.  For example in the rule
 above for @('bounder-for-*') we would include @('(:forward-chaining
 :trigger-terms ((tau-bounder-expt2 int2)))') and let the @('in-tau-intervalp')
 hypotheses select the free variables @('x') and @('y').</p>")

(defxdoc break$
  :parents (errors acl2-built-ins)
  :short "Cause an immediate Lisp break"
  :long "<p>ACL2 users are generally advised to avoid breaking into raw Lisp.
 Advanced users may, on occasion, see the need to do so.  Evaluating
 @('(break$)') will have that effect.  (Exception: @('break$') is disabled
 after evaluation of @('(set-debugger-enable :never)'); see @(see
 set-debugger-enable).)  @('Break$') returns @('nil').</p>

 @(def break$)")

(defxdoc break-lemma
  :parents (break-rewrite)
  :short "A quick introduction to breaking rewrite rules in ACL2"
  :long "@({
  Example:
  :brr t                          ; if you haven't done that yet
  :monitor (:rewrite lemma12) t   ; to install a break point on the
                                  ;   rule named (:rewrite lemma12)
  :monitor! (:rewrite lemma12) t  ; quiet version of :monitor that
                                  ;   invokes :brr t
 })

 <p>ACL2 does not support Nqthm's @('break-lemma') but supports a very similar
 and more powerful break facility.  Suppose some proof is failing; apparently
 some particular rule is not being used and you wish to learn why.  Then you
 need the ACL2 @(see break-rewrite) facility.  See @(see break-rewrite) and all
 of its associated @(':')@(tsee doc) topics for details.  The following basic
 steps are required.</p>

 <p>(1) To enable the ``break rewrite'' feature, you must first execute</p>

 @({
  ACL2 !>:brr t
 })

 <p>at the top-level of ACL2.  Equivalently, evaluate @('(brr t)').  @(see
 Break-rewrite) stays enabled until you disable it with @('(brr nil)').  When
 @(see break-rewrite) is enabled the ACL2 rewriter will run slower than normal
 but you will be able to @(see monitor) the attempts to apply specified
 rules.</p>

 <p>(2) Decide what @(see rune)s (see @(see rune)) you wish to @(see monitor).
 For example, you might want to know why @('(:rewrite lemma12 . 2)') is not
 being used in the attempted proof.  That, by the way, is the name of the
 second rewrite rule generated from the event named @('lemma12').</p>

 <p>The command</p>

 @({
  ACL2 !>:monitor (:rewrite lemma12 . 2) t
 })

 <p>will install an ``unconditional'' break point on that rule.  The ``@('t')''
 at the end of the command means it is unconditional, i.e., a break will occur
 every time the rule is tried.  ACL2 supports conditional breaks also, in which
 case the @('t') is replaced by an expression that evaluates to non-@('nil')
 when you wish for a break to occur.  See @(see monitor).  The above keyword
 command is, of course, equivalent to</p>

 @({
  ACL2 !>(monitor '(:rewrite lemma12 . 2) t)
 })

 <p>which you may also type.  You may install breaks on as many rules as you
 wish.  You must use @(tsee monitor) on each rule.  You may also change the
 break condition on a rule with @(tsee monitor).  Use @(tsee unmonitor) (see
 @(see unmonitor)) to remove a rule from the list of @(see monitor)ed
 rules.</p>

 <p>(3) Then try the proof again.  When a @(see monitor)ed rule is tried by the
 rewriter you will enter an interactive break, called @(see break-rewrite).
 See @(see break-rewrite) for a detailed description.  Very simply, @(see
 break-rewrite) lets you inspect the context of the attempted application both
 before and after the attempt.  When @(see break-rewrite) is entered it will
 print out the ``target'' term being rewritten.  If you type @(':go') @(see
 break-rewrite) will try the rule and then exit, telling you (a) whether the
 rule was applied, (b) if so, how the target was rewritten, and (c) if not, why
 the rule failed.  There are many other commands.  See @(see brr-commands).</p>

 <p>(4) When you have finished using the @(see break-rewrite) feature you
 should disable it to speed up the rewriter.  You can disable it with</p>

 @({
  ACL2 !>:brr nil
 })

 <p>The list of @(see monitor)ed rules and their break conditions persists but
 is ignored.  If you enable @(see break-rewrite) later, the list of @(see
 monitor)ed rules will be displayed and will be used again by rewrite.</p>

 <p>You should disable the @(see break-rewrite) feature whenever you are not
 intending to use it, even if the list of @(see monitor)ed rules is empty,
 because the rewriter is slowed down as long as @(see break-rewrite) is
 enabled.</p>

 <p>If you get a stack overflow, see @(see cw-gstack).</p>")

(defxdoc break-on-error
  :parents (trace acl2-built-ins)
  :short "Break when encountering a hard or soft error caused by ACL2"
  :long "@({
  General forms:
  (break-on-error t)    ; installs a trace causing a continuable error (break)
                        ;   when an error is invoked by ACL2.
  (break-on-error)      ; same as above
  (break-on-error :all) ; same as above, but even when inside the prover
  (break-on-error nil)  ; uninstall any above trace
 })

 <p>@('(Break-on-error)') generates a suitable trace of error functions.
 Evaluate @('(trace$)') after @('(break-on-error)') if you want to see the
 specific trace forms (which you can modify and then submit directly to
 @('trace$'), if you wish).  This @(see trace) should cause entry to the Lisp
 debugger whenever ACL2 calls its error routines, except for certain errors
 when inside the theorem prover, and also at those times if option :all is
 supplied.</p>

 <p>NOTE: For technical reasons, you may see some error messages more than
 once.</p>

 <p>Finally, note that you are welcome to define your own version of
 @('break-on-error') by modifying a copy of the source definition (search for
 ``@('(defmacro break-on-error')'' in ACL2 source file other-events.lisp).
 Please feel free to send your version of @('break-on-error') to the ACL2
 implementors, for possible inclusion into ACL2.</p>

 <p>@('Break-on-error') is implemented using ACL2 @(tsee trace$).  See @(see
 trace!) if you want an explanation of the ``@('TTAG NOTE')'' that is
 printed.</p>

 <p>The argument, if supplied, is evaluated and must evaluate to @('t'),
 @('nil'), or @(':all').</p>

 <p>Also see @(see set-debugger-enable) for how to get raw-Lisp backtrace
 information when an error occurs as a result of @('break-on-error'), or even
 of a raw Lisp error, by calling @('set-debugger-enable') with argument
 @(':bt'), @(':bt-break'), or @(':break-bt').  Note that for ACL2 errors (as
 opposed to raw Lisp errors), i.e. errors affected by @('break-on-error'), all
 three of those keyword values are treated equivalently (and, all are ignored
 for non-ANSI GCL; see @(see set-debugger-enable)).</p>")

(defxdoc break-rewrite
  :parents (debugging)
  :short "The read-eval-print loop entered to @(see monitor) rules"
  :long "<p>ACL2 allows the user to @(see monitor) the application of @(see
 rewrite), @(see definition), and @(see linear) rules.  When @(see monitor)ed
 rules are about to be tried by the rewriter, an interactive break occurs and
 the user is allowed to watch and, in a limited sense, control the attempt to
 apply the rule.  This interactive loop, which is technically just a call of
 the standard top-level ACL2 read-eval-print loop, @(tsee ld), on a ``@(see
 wormhole) @(see state)'' (see @(see wormhole)), is called ``break-rewrite.''
 While in break-rewrite, certain keyword commands are available for accessing
 information about the context in which the lemma is being tried.  These
 keywords are called break-rewrite ``commands''; see @(see brr-commands).</p>

 <p>For a related utility, see @(see dmr) (Dynamically Monitor Rewrites), which
 allows you to watch progress of the rewriter in real time.</p>

 <p>To abort from inside break-rewrite at any time, execute @(':')@(tsee
 a!).</p>

 <p>Output from break-rewrite is abbreviated by default, but that can be
 changed.  See @(see set-brr-evisc-tuple).</p>

 <p>For further information, see the related @(':')@(tsee doc) topics listed
 below.</p>

 <p>It is possible to cause the ACL2 rewriter to @(see monitor) the attempted
 application of selected rules.  When such a rule is about to be tried, the
 rewriter evaluates its break condition and if the result is non-@('nil'),
 break-rewrite is entered.</p>

 <p>Break-rewrite permits the user to inspect the current @(see state) by
 evaluating break-rewrite commands.  Type @(':help') in break-rewrite to see
 what the break-rewrite commands are.  However, break-rewrite is actually just
 a call of the general ACL2 read-eval-print loop, @(tsee ld), on a certain
 @(see state) and the break-rewrite commands are simply aliases provided by
 @('ld-keyword-aliases') @(see table) (see @(see ld-keyword-aliases)).  See
 @(see ld) for details about this read-eval-print loop.  Thus, with a few
 exceptions, anything you can do at the ACL2 top-level can be done within
 break-rewrite.  For example, you can evaluate arbitrary expressions, use the
 keyword command hack, access @(see documentation), print @(see events), and
 even define functions and prove theorems.  However, the ``certain @(see
 state)'' upon which @(tsee ld) was called is a ``wormhole state'' (see @(see
 wormhole)) because break-rewrite is not allowed to have any effect upon the
 behavior of rewrite.  What this means, at a high level, is that break-rewrite
 operates on a copy of the @(see state) being used by rewrite and when
 break-rewrite exits the @(see wormhole) closes and the @(see state)
 ``produced'' by break-rewrite disappears.  For example, all invocations of
 @(tsee trace$) and @(tsee untrace$) that are made during a break at a @(see
 monitor)ed @(see rune) are undone when proceeding from that break.  Thus,
 break-rewrite lets you query the state of the rewriter and even do experiments
 involving proofs, etc., but these experiments have no effect on the ongoing
 proof attempt.</p>

 <p>There are however exceptions to this loss of state when exiting a break.
 One exception is that the effect of turning on iprinting in a break (see @(see
 set-iprint)) will persist even after exiting the break.  The other exceptions
 pertain to setting the @(tsee brr-evisc-tuple) or invoking @(tsee monitor) or
 @(tsee unmonitor): if these are done inside the break-rewrite loop at level 1
 of interaction (i.e., at the top level) then their effects will persist even
 after exiting the break.</p>

 <p>When you first enter break-rewrite a simple herald is printed such as:</p>

 @({
  (3 Breaking (:rewrite lemma12) on (delta a (+ 1 j)):
 })

 <p>The integer after the open parenthesis indicates the depth of nested
 break-rewrite calls.  In this discussion we use @('3') consistently for this
 integer.  Unless you abort or somehow enter unbalanced parentheses into the
 script, the entire session at a given depth will be enclosed in balanced
 parentheses, making it easy to skip over them in Emacs.</p>

 <p>You then will see the break-rewrite @(see prompt):</p>

 @({
  3 ACL2 !>
 })

 <p>The leading integer is, again, the depth.  Because breaks often occur
 recursively it is convenient always to know the level with which you are
 interacting.</p>

 <p>You may type arbitrary commands as in the top-level ACL2 loop.  For
 example, you might type:</p>

 @({
  3 ACL2 !>:help
 })

 <p>or</p>

 @({
  3 ACL2 !>:pe lemma12
 })

 <p>More likely than typing a history or @(tsee disabledp) command, upon
 entering break-rewrite you will determine the context of the attempted
 application.  Here are some useful commands:</p>

 @({
  3 ACL2 >:target           ; the term being rewritten
  3 ACL2 >:unify-subst      ; the unifying substitution
  3 ACL2 >:path             ; the stack of goals pursued by the rewriter
                            ; starting at the top-level clause being simplified
                            ; and ending with the current application
 })

 <p>At this point in the interaction the system has not yet tried to apply the
 @(see monitor)ed rule.  That is, it has not tried to establish the hypotheses,
 considered the heuristic cost of backchaining, rewritten the right-hand side
 of the conclusion, etc.  When you are ready for it to try the rule you can
 type one of several different ``proceed'' commands.  The basic proceed
 commands are @(':ok'), @(':go'), and @(':eval').</p>

 @({
  :ok
 })

 <p>exits break-rewrite without further interaction.  When break-rewrite exits
 it prints ``@('3)')'', closing the parenthesis that opened the level @('3')
 interaction.</p>

 @({
  :go
 })

 <p>exits break-rewrite without further interaction, but prints out the result
 of the application attempt, i.e., whether the application succeeded, if so,
 what the @(':target') term was rewritten to, and if not why the rule was not
 applicable.</p>

 @({
  :eval
 })

 <p>causes break-rewrite to attempt to apply the rule but interaction at this
 level of break-rewrite resumes when the attempt is complete.  When control
 returns to this level of break-rewrite a message indicating the result of the
 application attempt (just as in @(':go')) is printed, followed by the @(see
 prompt) for additional user input.</p>

 <p>Generally speaking, @(':ok') and @(':go') are used when the break in
 question is routine or uninteresting and @(':eval') is used when the break is
 one that the user anticipates is causing trouble.  For example, if you are
 trying to determine why a lemma isn't being applied to a given term and the
 @(':target') of the current break-rewrite is the term in question, you would
 usually @(':eval') the rule and if break-rewrite reports that the rule failed
 then you are in a position to determine why, for example by carefully
 inspecting the @(':')@(tsee type-alist) of governing assumptions or why some
 hypothesis of the rule could not be established.</p>

 <p>It is often the case that when you are in break-rewrite you wish to change
 the set of @(see monitor)ed @(see rune)s.  This can be done by using
 @(':')@(tsee monitor) and @(':')@(tsee unmonitor) as noted above.  For
 example, you might want to @(see monitor) a certain rule, say
 @('hyp-reliever'), just when it is being used while attempting to apply
 another rule, say @('main-lemma').  Typically then you would @(see monitor)
 @('main-lemma') at the ACL2 top-level, start the proof-attempt, and then in
 the break-rewrite in which @('main-lemma') is about to be tried, you would
 install a @(see monitor) on @('hyp-reliever').  If during the ensuing
 @(':eval') @('hyp-reliever') is broken you will know it is being used under
 the attempt to apply @('main-lemma').</p>

 <p>However, once @('hyp-reliever') is being @(see monitor)ed it will be @(see
 monitor)ed even after @('main-lemma') has been tried.  That is, if you let the
 proof attempt proceed then you may see many other breaks on @('hyp-reliever'),
 breaks that are not ``under'' the attempt to apply @('main-lemma').  One way
 to prevent this is to @(':eval') the application of @('main-lemma') and then
 @(':')@(tsee unmonitor) @('hyp-reliever') before exiting.  But this case
 arises so often that ACL2 supports several additional ``flavors'' of proceed
 commands.</p>

 <p>@(':Ok!'), @(':go!'), and @(':eval!') are just like their counterparts
 (@(':ok'), @(':go'), and @(':eval'), respectively), except that while
 processing the rule that is currently broken no @(see rune)s are @(see
 monitor)ed.  When consideration of the current rule is complete, the set of
 @(see monitor)ed @(see rune)s is restored to its original setting.</p>

 <p>@(':Ok$'), @(':go$'), and @(':eval$') are similar but take an additional
 argument which must be a list of @(see rune)s.  An example usage of
 @(':eval$') is</p>

 @({
  3 ACL2 !>:eval$ ((:rewrite hyp-reliever))
 })

 <p>These three commands temporarily install unconditional breaks on the @(see
 rune)s listed, proceed with the consideration of the currently broken rule,
 and then restore the set of @(see monitor)ed rules to its original
 setting.</p>

 <p>Thus, there are nine ways to proceed from the initial entry into
 break-rewrite although we often speak as though there are two, @(':ok') and
 @(':eval'), and leave the others implicit.  We group @(':go') with @(':ok')
 because in all their flavors they exit break-rewrite without further
 interaction (at the current level).  All the flavors of @(':eval') require
 further interaction after the rule has been tried.</p>

 <p>To abort a proof attempt and return to the top-level of ACL2 you may at any
 time type @('(a!)') followed by a carriage return.  If you are not in a raw
 Lisp break, you may type @(':a!') instead.  The utility @('p!') is completely
 analogous to @('a!') except that it pops up only one @(tsee ld) level.  If you
 have just entered the break-rewrite loop, this will pop you out of that loop,
 back to the proof.  See @(see a!) and see @(see p!).</p>

 <p>We now address ourselves to the post-@(':eval') interaction with
 break-rewrite.  As noted, that interaction begins with break-rewrite's report
 on the results of applying the rule: whether it worked and either what it
 produced or why it failed.  This information is also printed by certain
 keyword commands available after @(':eval'), namely @(':wonp'),
 @(':rewritten-rhs') or (for @(see linear) rules) @(':poly-list'), and
 @(':failure-reason').  In addition, by using @(tsee brr@) you can obtain this
 information in the form of ACL2 data objects.  This allows the development of
 more sophisticated ``break conditions''; see @(see monitor) for examples.  In
 this connection we point out the macro form @('(ok-if term)').  See @(see
 ok-if).  This command exits break-rewrite if @('term') evaluates to
 non-@('nil') and otherwise does not exit.  Thus it is possible to define
 macros that provide other kinds of exits from break-rewrite.  The only way to
 exit break-rewrite after @(':eval') is @(':ok') (or, equivalently, the use of
 @(tsee ok-if)).</p>

 <p>Note that when inside break-rewrite, all @(see history) commands, such as
 @(':')@(tsee pe), show the @(see enable)d status of rules with respect to the
 current point in the proof attempt.  For example, if you break while the
 prover is working on Subgoal 3, and the @(see hints) supplied for the proof
 specify @('(\"Subgoal 3\" :in-theory (disable foo))') for some rule @('foo'),
 then @(':')@(tsee pe) will indicate that @('foo') is @(see disable)d: even
 though @('foo') may be enabled globally, it is shown as disabled because it is
 disabled during Subgoal 3.  See subtopics of @(see history) for a list of all
 such history commands.  In addition to those commands, the function @(tsee
 disabledp) is also evaluated inside break-rewrite with respect to the current
 enabled state of the prover.</p>

 <p>ACL2 users who wish to know more about break-rewrite so that they can
 develop more convenient ways to @(see monitor) rules are encouraged to speak
 to J Moore.</p>

 <p>The rest of this @(see documentation) discusses a few implementation
 details of break-rewrite and may not be interesting to the typical user.</p>

 <p>There is no ACL2 function named break-rewrite.  It is an illusion created
 by appropriate calls to two functions named @('brkpt1') and @('brkpt2').  As
 previously noted, break-rewrite is @(tsee ld) operating on a @(see wormhole)
 @(see state).  One might therefore wonder how break-rewrite can apply a rule
 and then communicate the results back to the rewriter running in the external
 @(see state).  The answer is that it cannot.  Nothing can be communicated
 through a @(see wormhole).  In fact, @('brkpt1') and @('brkpt2') are each
 calls of @(tsee ld) running on @(see wormhole) @(see state)s.  @('Brkpt1')
 implements the pre-@(':eval') break-rewrite and @('brkpt2') implements the
 post-@(':eval') break-rewrite.  The rewriter actually calls @('brkpt1') before
 attempting to apply a rule and calls @('brkpt2') afterwards.  In both cases,
 the rewriter passes into the @(see wormhole) the relevant information about
 the current context.  Logically @('brkpt1') and @('brkpt2') are no-ops and
 @(tsee rewrite) ignores the @('nil') they return.  But while control is in
 them, the execution of @(tsee rewrite) is suspended and cannot proceed until
 the break-rewrite interactions complete.</p>

 <p>This design causes a certain anomaly that might be troubling.  Suppose that
 inside break-rewrite before @(':evaling') a rule (i.e., in the @('brkpt1')
 @(see wormhole) @(see state)) you define some function, @('foo').  Suppose
 then you @(':eval') the rule and eventually control returns to break-rewrite
 (i.e., to @('brkpt2') on a @(see wormhole) @(see state) with the results of
 the application in it).  You will discover that @('foo') is no longer defined!
 That is because the @(see wormhole) @(see state) created during your
 @('pre-:eval') interaction is lost when we exit the @(see wormhole) to resume
 the proof attempt.  The post-@(':eval') @(see wormhole) @(see state) is in
 fact identical to the initial pre-@(':eval') @(see state) (except for the
 results of the application) because @(tsee rewrite) did not change the
 external @(see state) and both @(see wormhole) @(see state)s are copies of it.
 A similar issue occurs with the use of @(see trace) utilities: all effects of
 calling @(tsee trace$) and @(tsee untrace$) are erased when you proceed from a
 break in the break-rewrite loop.</p>

 <p>There is a lot more to know about break-rewrite, most of which is fairly
 easy to learn from looking at the code, since it is all expressed in ACL2.
 Feel free to ask questions of J Moore.</p>")

(defxdoc breaks
  :parents (errors)
  :short "Common Lisp breaks"
  :long "@({
  Example:
  Broken at PROVE.  Type :H for Help.
  >>:Q

  ACL2 !>
 })

 <p>You may interrupt the system by typing various control character sequences.
 The precise sequences are determined by the host Lisp and operating system
 environment.  For example, in GCL and Allegro Common Lisp, a console interrupt
 is caused by typing ``@('ctrl-c')''.  If, however, the GCL or Allegro is
 running in an Emacs shell buffer, one must type ``ctrl-c ctrl-c''.</p>

 <p>If a break occurs, for example because of a bug in ACL2 or a user
 interrupt, the break will run a Common Lisp read-eval-print loop, not an ACL2
 read-eval-print loop.  This may not be obvious if the @(see prompt)s in the
 two loops are similar.  Because you are typing to a Common Lisp evaluator, you
 must be careful.  It is possible to damage your ACL2 state in irreparable ways
 by executing non-ACL2 Common Lisp.  It is even possible to disrupt and render
 inaccurate the interrupted evaluation of a simple ACL2 expression.</p>

 <p>For ACL2 built on most host Common Lisps, you will see the string @('[RAW
 LISP]') in the @(see prompt) at a break, to emphasize that one is inside a
 break and hence should quit from the break.  For some host Common Lisps, the
 top-level prompt also contains the string @('[RAW LISP]').  See @(see prompt)
 for how to control printing of that string.</p>

 <p>The most reliable way to return to the ACL2 top level is by executing the
 following command: @('(')@(tsee abort!)@(')').  Appropriate cleanup will then
 be done, which should leave you in an appropriate state.</p>

 <p>However, you may be able to quit from the break in the normal Lisp manner
 (as with @(':q') in GCL or CCL, @(':reset') in Allegro CL, and @('q') in CMU
 CL).  If this attempt to quit is successful, it will return you to the
 innermost ACL2 read-eval-print loop, with appropriate cleanup performed first.
 Note that if you are within a @(tsee brr) environment when the break occurs,
 quitting from the break will only return you to that environment, not to the
 top of ACL2's read-eval-print loop.</p>")

(defxdoc broken-link
  :parents (documentation)
  :short "Placeholder for link to documentation that resides in the community
 books"
  :long "<p>You may have attempted to access information about the ACL2 @(see
 community-books) while looking at the ACL2 User's Manual, which contains @(see
 documentation) only about the ACL2 <i>system</i>, and does not include
 documentation from the @(see community-books).  Please point your browser at
 the @(`(:raw (combined-manual-ref))`) (or if browsing in @(see ACL2-Doc),
 switch to that manual with meta-0 I) to access the desired topic.</p>

 <p>If you want information about the book where your missing topic is defined,
 see @(see broken-link-table).</p>")

(defxdoc broken-link-table
  :parents (broken-link)
  :short "Map @(see documentation) topics to the community books that define
 them"
  :long "<p>The table below maps topics to book locations that reside only in
 the ACL2+Books combined manual, not the ACL2 User's Manual.  For example, the
 entry</p>

 @({
 (b* \"[books]/std/util/bstar.lisp\")
 })

 <p>signifies that the topic @('B*') is documented in the community
 book @('std/util/bstar.lisp').</p>

 <p>@(`(:code *acl2-broken-links-alist*)`)</p>")

(defxdoc brr
  :parents (break-rewrite)
  :short "To enable or disable the breaking of rewrite rules"
  :long "@({
  Example:
  :brr t       ; enable
  (brr t)      ; enable (same as above)
  (brr t t)    ; enable with less output (rarely invoked interactively)
  :brr nil     ; disable

  General Form:
  (brr flg &optional quietp)
 })

 <p>where @('flg') and the optional @('quietp') argument evaluate to @('t') or
 @('nil').  This function modifies @(tsee state) so that the attempted
 application of certain rewrite rules are ``broken.'' ``@('Brr')'' stands for
 ``break-rewrite'' and can be thought of as a mode with two settings.  The
 normal mode is ``disabled.''</p>

 <p>For a more thorough introduction to the break rewrite system see @(see
 break-rewrite).</p>

 <p>When @('brr') mode is ``enabled'' the ACL2 rewriter monitors the attempts
 to apply certain rules and advises the user of those attempts by entering an
 interactive wormhole break.  From within this break the user can watch
 selected application attempts.  The user can also interact with the system
 during @('brr') breaks via @(tsee brr-commands).</p>

 <p>The rules monitored are selected by using the @(tsee monitor) and @(tsee
 unmonitor) commands.  It is possible to break a rune ``conditionally'' in the
 sense that an interactive break will occur only if a specified predicate is
 true of the environment at the time of the attempted application.  See @(see
 monitor) and see @(see unmonitor).</p>

 <p>Even if a non-empty set of rules has been selected, no breaks will occur
 unless @('brr') mode is enabled.  Thus, the first time in a session that you
 wish to monitor a rewrite rule, use @(':brr') @('t') to enable @('brr') mode.
 Thereafter you may select runes to be monitored with @(tsee monitor) and
 @(tsee unmonitor) with the effect that whenever monitored rules are tried (and
 their break conditions are met) an interactive break will occur.  Be advised
 that when @('brr') mode is enabled the rewriter is somewhat slower than
 normal.  Furthermore, that sluggishness persists even if no runes are
 monitored.  You may regain normal performance &mdash; regardless of what runes
 are monitored &mdash; by disabling @('brr') mode with @(':brr') @('nil').</p>

 <p>Why isn't @('brr') mode disabled automatically when no runes are monitored?
 More generally, why does ACL2 have @('brr') mode at all?  Why not just test
 whether there are monitored runes?  If you care about the answers, see @(see
 why-brr).</p>

 <p>BRR Mode, Console Interrupts, and Subsidiary Prover Calls: If the system is
 operating in @('brr') mode and you break into raw Lisp (as by causing a
 console interrupt or happening upon a signaled Lisp error; see @(see breaks)),
 you can return to the ACL2 top-level, outside any @('brr') environment, by
 executing @('(')@(tsee abort!)@(')').  Otherwise, the normal way to quit from
 such a break (for example @(':q') in GCL, @(':reset') in Allegro CL, and
 @('q') in CMU CL) will return to the innermost ACL2 read-eval-print loop,
 which may or may not be the top-level of your ACL2 session!  In particular, if
 the break happens to occur while ACL2 is within the @('brr') environment (in
 which it is preparing to read @(tsee brr-commands)), the abort will merely
 return to that @('brr') environment.  Upon exiting that environment, normal
 theorem proving is continued (and the @('brr') environment may be entered
 again in response to subsequent monitored rule applications).  Before
 returning to the @('brr') environment, ACL2 ``cleans up'' from the interrupted
 @('brr') processing.  However, it is not possible (given the current
 implementation) to clean up perfectly.  This may have two side-effects.
 First, the system may occasionally print the self-explanatory ``Cryptic BRR
 Message 1'' (or 2), informing you that the system has attempted to recover
 from an aborted @('brr') environment.  Second, it is possible that subsequent
 @('brr') behavior in that proof will be erroneous because the cleanup was done
 incorrectly.  The moral is that you should not trust what you learn from
 @('brr') if you have interrupted and aborted @('brr') processing during the
 proof.  Such ``clean up'' may also occur when you call the prover from within
 the @('brr') environment (for example using @(tsee defthm) or @(tsee thm), or
 even @(tsee defun) or any other @(see event) that invoke the prover), unless
 you invoke @(':brr nil') before making that call.  These issues do not affect
 the behavior or soundness of the theorem prover.</p>")

(defxdoc brr-commands
  :parents (break-rewrite)
  :short "@(see Break-Rewrite) Commands"
  :long "<p>Many commands display terms that are abbreviated (``eviscerated'')
  by default.  These have corresponding commands with a ``+'' suffix that avoid
  such abbreviation, as shown below; also see @(see brr-evisc-tuple).  For
  example, the notation ``@(':ancestors[+]')'' below indicates that the
  @(':ancestors') command may abbreviate terms but the @(':ancestors+') command
  does not.</p>

 @({
 :a!                abort to ACL2 top-level
 :ancestors[+]      negations of backchaining hypotheses being pursued
 :btm[+]            bottom-most frame in :path
 :eval              try rule and re-enter break afterwards
 :eval!             :eval but no recursive breaks
 :eval$ runes       :eval with runes monitored during recursion
 :failure-reason[+] reason rule failed (after :eval)
 :final-ttree[+]    ttree after :eval (see :DOC ttree)
 :frame[+] i        ith frame in :path
 :go                exit break, printing result
 :go!               :go but no recursive breaks
 :go$ runes         :go with runes monitored during recursion
 :help              this message
 :hyp i             ith hypothesis of the rule
 :hyps              hypotheses of the rule
 :initial-ttree[+]  ttree before :eval (see :DOC ttree)
 :lhs               left-hand side of rule's conclusion (or, in the case
                      of :rewrite-quoted-constant rules of form [2], the
                      right-hand side!)
 :ok                exit break
 :ok!               :ok but no recursive breaks
 :ok$ runes         :ok with runes monitored during recursion
 :p!                pop one level (exits a top-level break-rewrite loop)
 :path[+]           rewriter's path from top clause to :target
 :poly-list[+]      list of polynomials (after :eval) of a linear rule,
                      where the leading term of each is enclosed in an
                      extra set of parentheses
 :rewritten-rhs[+]  rewritten :rhs (after :eval) of a rewrite rule
 :rhs               right-hand side of rule's conclusion (or, in the case
                      of :rewrite-quoted-constant rules of form [2], the
                      left-hand side!)
 :standard-help     :help message from ACL2 top-level
 :target[+]         term being rewritten
 :top[+]            top-most frame in :path
 :type-alist[+]     type assumptions governing :target
 :unify-subst[+]    substitution making :lhs equal :target
 :wonp              indicates whether application succeeded (after :eval)
 })

 <p>See the discussion of form [2] @(':')@(tsee rewrite-quoted-constant) rules
 for an explanation of the swapped meanings of ``@(':lhs')'' and
 ``@(':rhs').''</p>

 <p>@(see Break-rewrite) is just a call of the standard ACL2 read-eval-print
 loop, @(tsee ld), on a ``@(see wormhole)'' @(see state).  Thus, you may
 execute most commands you might normally execute at the top-level of ACL2.
 However, all @(see state) changes you cause from within @(see break-rewrite)
 are lost when you exit or @(':eval') the rule.  You cannot modify @(see
 stobj)s from within the break.  See @(see break-rewrite) for more details and
 see @(see ld) for general information about the standard ACL2 read-eval-print
 loop.  Also see @(see brr@) for a utility that can return a value for many of
 the keywords above, instead of merely printing to the screen.</p>

 <p>Note that if you are breaking on a @(see monitor)ed @(see linear) rule,
 several of the commands listed above do not apply: @(':lhs'), @(':rhs'),
 @(':initial-ttree'), and @(':final-ttree').  Moreover, @(':rewritten-rhs')
 also does not apply, but instead, @(':poly-list') shows the result of applying
 the linear lemma as a list of polynomials, implicitly conjoined.  The leading
 term of each polynomial is enclosed in an extra set of parentheses.</p>")

(defxdoc brr-evisc-tuple
  :parents (brr evisc-tuple)
  :short "Determines partial suppression of output from @(see brr-commands)"
  :long "<p>See @(see evisc-tuple) for relevant background on ``evisceration'':
 eliding of subexpressions during printing.  Also see @(see break-rewrite) for
 background on the break-rewrite loop.</p>

 <p>One of the settable evisc-tuples (see @(see set-evisc-tuple)) can control
 output from @(see brr-commands): the @('brr-evisc-tuple').  Unlike most other
 evisc-tuples, if you set the @('brr-evisc-tuple') inside the break-rewrite
 loop at level 1 of interaction (i.e., at the top level; see @(see
 break-rewrite)), then its effect will persist even after you exit the
 break.</p>

 <p>A special value, @(':default'), is legal for this evisc-tuple, and is its
 initial value.  In that case the actual evisc-tuple used during output from
 @(see brr-commands) &mdash; which we call the <i>effective value</i> of the
 @('brr-evisc-tuple') &mdash; is the value of the evisc-tuple for terms.  See
 @(see set-evisc-tuple), in particular, the discussion of the @(':term') site
 for setting evisc-tuples.</p>

 <p>You can see the effective value of the @('brr-evisc-tuple') by evaluating
 the form, @('(show-brr-evisc-tuple)').  Note that this value is only printed
 by such evaluation as a side-effect, not returned.  (Technical note: This is
 because the @('brr-evisc-tuple') is maintained entirely within the
 break-rewrite @(see wormhole).  That implementation enables the persistence of
 this evisc-tuple within and without the break-rewrite loop.)</p>")

(defxdoc brr@
  :parents (break-rewrite)
  :short "To access context sensitive information within @(tsee break-rewrite)"
  :long "@({
  Example:
  (brr@ :target)      ; the term being rewritten
  (brr@ :unify-subst) ; the unifying substitution

  General Form:
  (brr@ :symbol)
 })

 <p>where @(':symbol') is one of the keywords displayed below.  This utility
 may be most useful for system hackers; see @(see brr-commands) for queries
 that are more at a user level.  In particular, keywords marked below with
 @('*') probably require an implementor's knowledge of the system to use
 effectively.  They are supported but not well documented.  More is said on
 this topic following the table.  For background on the notion of
 ``translated'' term, see @(see term).</p>

 @({
  :symbol             (brr@ :symbol)
  -------             ---------------------

  :target             the term to be rewritten.  This term is an
                      instantiation of the left-hand side of the
                      conclusion of the rewrite-rule being broken.
                      This term is in translated form!  Thus, if
                      you are expecting (equal x nil) -- and your
                      expectation is almost right -- you will see
                      (equal x 'nil); similarly, instead of (cadr a)
                      you will see (car (cdr a)).  In translated
                      forms, all constants are quoted (even nil, t,
                      strings and numbers) and all macros are
                      expanded.

  :unify-subst        the substitution that, when applied to :target,
                      produces the left-hand side of the rule being
                      broken.  This substitution is an alist pairing
                      variable symbols to translated (!) terms.

  :wonp               t or nil indicating whether the rune was
                      successfully applied.  (brr@ :wonp) returns
                      nil if evaluated before :EVALing the rule.

  :rewritten-rhs      the result of successfully applying the rewrite
                      rule or else nil if (brr@ :wonp) is nil.  The
                      result of successfully applying the rule is
                      always a translated (!) term and is never nil.

  :poly-list          the result of successfully applying the linear
                      rule or else nil if (brr@ :wonp) is nil.  This
                      result represents the list of polynomials
                      produced by the rule application.  The leading
                      term of each polynomial is enclosed in an extra
                      set of parentheses.

  :failure-reason     some non-nil lisp object indicating why the rule
                      was not applied or else nil.  Before the rule is
                      :EVALed, (brr@ :failure-reason) is nil.  After
                      :EVALing the rule, (brr@ :failure-reason) is nil
                      if (brr@ :wonp) is t.  Rather than document the
                      various non-nil objects returned as the failure
                      reason, we encourage you simply to evaluate
                      (brr@ :failure-reason) in the contexts of interest.
                      Alternatively, study the ACL2 function tilde-@-
                      failure-reason-phrase.

  :lemma           *  the rewrite rule being broken.  For example,
                      (access rewrite-rule (brr@ :lemma) :lhs) will
                      return the left-hand side of the conclusion
                      of the rule.

  :type-alist      *  a display of the type-alist governing :target.
                      Elements on the displayed list are of the form
                      (term type), where term is a term and type
                      describes information about term assumed to hold in
                      the current context.  (See also the documentation for
                      type-alist.)  The type-alist may be used to determine
                      the current assumptions, e.g., whether A is a CONSP.

  :ancestors       *  a stack of frames indicating the backchain history
                      of the current context.  The theorem prover is in
                      the process of trying to establish each hypothesis
                      in this stack.  Thus, the negation of each hypothesis
                      can be assumed false.  Each frame also records the
                      rules on behalf of which this backchaining is being
                      done and the weight (function symbol count) of the
                      hypothesis.  All three items are involved in the
                      heuristic for preventing infinite backchaining.
                      Exception:  Some frames are ``binding hypotheses''
                      (equal var term) or (equiv var (double-rewrite term))
                      that bind variable var to the result of rewriting
                      term.  The ACL2 source code has a definition
                      (defrec ancestor ...) that may provide some relevant
                      insight.

  :initial-ttree   *  the initial and (after :EVAL) final tag trees,
  :final-ttree        respectively.  (Tag trees are low-level data structures
                      that store lemmas used and other information, as
                      documented in topic TTREE.)

  :gstack          *  the current goal stack.  The gstack is maintained
                      by rewrite and is the data structure printed as the
                      current ``path.''  Thus, any information derivable
                      from the :path brr command is derivable from gstack.
                      For example, from gstack one might determine that
                      the current term is the second hypothesis of a
                      certain rewrite rule.
 })

 <p>In general @('brr@-expressions') are used in break conditions, the
 expressions that determine whether interactive breaks occur when @(see
 monitor)ed @(see rune)s are applied.  See @(see monitor).  For example, you
 might want to break only those attempts in which one particular term is being
 rewritten or only those attempts in which the binding for the variable @('a')
 is known to be a @(tsee consp).  Such conditions can be expressed using ACL2
 system functions and the information provided by @('brr@').  Unfortunately,
 digging some of this information out of the internal data structures may be
 awkward or may, at least, require intimate knowledge of the system functions.
 But since conditional expressions may employ arbitrary functions and macros,
 we anticipate that a set of convenient primitives will gradually evolve within
 the ACL2 community.  It is to encourage this evolution that @('brr@') provides
 access to the @('*')'d data.</p>")

(defxdoc building-acl2
  :parents (about-acl2)
  :short "How to build an ACL2 executable"
  :long "<p>To build an ACL2 executable, submit the following command while
 standing in the main ACL2 directory, where @('<my-lisp>') invokes your Lisp
 executable (default: @('ccl')).</p>

 @({
 make LISP=<my-lisp>
 })

 <p>You should find \"Initialization SUCCEEDED.\" near the end the of the log.
 Note: There may be ACL2 warnings, for example: \"ACL2 Warning [Skip-proofs]
 in....\".  These may be safely ignored.</p>

 <p>Note that you will want to certify @(see books) in order to take full
 advantage of ACL2.  See @(see books-certification).</p>")

(defxdoc built-in-clause
  :parents (rule-classes)
  :short "To build a clause into the simplifier"
  :long "<p>See @(see rule-classes) for a general discussion of rule classes,
 including how they are used to build rules from formulas and a discussion of
 the various keywords in a rule class description.</p>

 @({
  Example:
  (defthm acl2-count-abl
    (and (implies (and (true-listp x)
                       (not (equal x nil)))
                  (< (acl2-count (abl x))
                     (acl2-count x)))
         (implies (and (true-listp x)
                       (not (equal nil x)))
                  (< (acl2-count (abl x))
                     (acl2-count x))))
    :rule-classes :built-in-clause)
 })

 <p>A @(':built-in-clause') rule can be built from any formula other than
 propositional tautologies.  Roughly speaking, the system uses the list of
 built-in clauses as the first method of proof when attacking a new goal.  Any
 goal that is subsumed by a built in clause is proved ``silently.''</p>

 <p>ACL2 maintains a set of ``built-in'' clauses that are used to short-circuit
 certain theorem proving tasks.  We discuss this at length below.  When a
 theorem is given the rule class @(':built-in-clause') ACL2 flattens the @(tsee
 implies) and @(tsee and) structure of the @(':')@(tsee corollary) formula so
 as to obtain a set of formulas whose conjunction is equivalent to the given
 corollary.  It then converts each of these to clausal form and adds each
 clause to the set of built-in clauses.</p>

 <p>The example above (regardless of the definition of @('abl')) will build in
 two clauses,</p>

 @({
  {(not (true-listp x))
   (equal x nil)
   (< (acl2-count (abl x)) (acl2-count x))}
 })

 <p>and</p>

 @({
  {(not (true-listp x))
   (equal nil x)
   (< (acl2-count (abl x)) (acl2-count x))}.
 })

 <p>We now give more background.</p>

 <p>Recall that a clause is a set of terms, implicitly representing the
 disjunction of the terms.  Clause @('c1') is ``subsumed'' by clause @('c2') if
 some instance of @('c2') is a subset @('c1').</p>

 <p>For example, let @('c1') be</p>

 @({
  {(not (consp l))
   (equal a (car l))
   (< (acl2-count (cdr l)) (acl2-count l))}.
 })

 <p>Then @('c1') is subsumed by @('c2'), shown below,</p>

 @({
  {(not (consp x))
   ; second term omitted here
   (< (acl2-count (cdr x)) (acl2-count x))}
 })

 <p>because we can instantiate @('x') in @('c2') with @('l') to obtain a subset
 of @('c1').</p>

 <p>Observe that @('c1') is the clausal form of</p>

 @({
  (implies (and (consp l)
                (not (equal a (car l))))
           (< (acl2-count (cdr l)) (acl2-count l))),
 })

 <p>@('c2') is the clausal form of</p>

 @({
  (implies (consp l)
           (< (acl2-count (cdr l)) (acl2-count l)))
 })

 <p>and the subsumption property just means that @('c1') follows trivially from
 @('c2') by instantiation.</p>

 <p>The set of built-in clauses is just a set of known theorems in clausal
 form.  Any formula that is subsumed by a built-in clause is thus a theorem.
 If the set of built-in theorems is reasonably small, this little theorem
 prover is fast.  ACL2 uses the ``built-in clause check'' in four places: (1)
 at the top of the iteration in the prover &mdash; thus if a built-in clause is
 generated as a subgoal it will be recognized when that goal is considered, (2)
 within the simplifier so that no built-in clause is ever generated by
 simplification, (3) as a filter on the clauses generated to prove the
 termination of recursively @(tsee defun)'d functions and (4) as a filter on
 the clauses generated to verify the guards of a function.</p>

 <p>The latter two uses are the ones that most often motivate an extension to
 the set of built-in clauses.  Frequently a given formalization problem
 requires the definition of many functions which require virtually identical
 termination and/or guard proofs.  These proofs can be short-circuited by
 extending the set of built-in clauses to contain the most general forms of the
 clauses generated by the definitional schemes in use.</p>

 <p>The attentive user might have noticed that there are some recursive
 schemes, e.g., recursion by @(tsee cdr) after testing @(tsee consp), that ACL2
 just seems to ``know'' are ok, while for others it generates measure clauses
 to prove.  Actually, it always generates measure clauses but then filters out
 any that pass the built-in clause check.  When ACL2 is initialized, the clause
 justifying @(tsee cdr) recursion after a @(tsee consp) test is added to the
 set of built-in clauses.  (That clause is @('c2') above.)</p>

 <p>Note that only a subsumption check is made; no rewriting or simplification
 is done.  Thus, if we want the system to ``know'' that @(tsee cdr) recursion
 is ok after a negative @(tsee atom) test (which, by the definition of @(tsee
 atom), is the same as a @(tsee consp) test), we have to build in a second
 clause.  The subsumption algorithm does not ``know'' about commutative
 functions.  Thus, for predictability, we have built in commuted versions of
 each clause involving commutative functions.  For example, we build in
 both</p>

 @({
  {(not (integerp x))
   (< 0 x)
   (= x 0)
   (< (acl2-count (+ -1 x)) (acl2-count x))}
 })

 <p>and the commuted version</p>

 @({
  {(not (integerp x))
   (< 0 x)
   (= 0 x)
   (< (acl2-count (+ -1 x)) (acl2-count x))}
 })

 <p>so that the user need not worry whether to write @('(= x 0)') or @('(= 0
 x)') in definitions.</p>

 <p>@(':built-in-clause') rules added by the user can be enabled and
 disabled.</p>")

(defxdoc butlast
  :parents (lists acl2-built-ins)
  :short "All but a final segment of a list"
  :long "<p>@('(Butlast l n)') is the list obtained by removing the last @('n')
 elements from the true list @('l').  The following are theorems.</p>

 @({
  (implies (and (integerp n)
                (<= 0 n)
                (true-listp l))
           (equal (length (butlast l n))
                  (if (< n (length l))
                      (- (length l) n)
                    0)))

  (equal (len (butlast l n)) (nfix (- (len l) (nfix n)))))
 })

 <p>For related functions, see @(see take) and see @(see nthcdr).</p>

 <p>The @(see guard) for @('(butlast l n)') requires that @('n') is a
 nonnegative integer and @('lst') is a true list.</p>

 <p>@('Butlast') is a Common Lisp function.  See any Common Lisp documentation
 for more information.  Note: In Common Lisp the second argument of
 @('butlast') is optional, but in ACL2 it is required.</p>

 @(def butlast)")

(defxdoc caaaar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee caaar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caaadr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee caadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caaar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee caar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caadar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cadar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caaddr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee caddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caadr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee car)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cadaar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cdaar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cadadr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cdadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cadar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cdar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caddar
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cddar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cadddr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cdddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc caddr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cadr
  :parents (conses acl2-built-ins)
  :short "@(tsee car) of the @(tsee cdr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc calling-ld-in-bad-contexts
  :parents (ld)
  :short "Errors caused by calling @(tsee ld) in inappropriate contexts"
  :long "<p>The macro @(tsee ld) was designed to be called directly in the
 top-level ACL2 loop, although there may be a few occasions for calling it from
 functions.  ACL2 cannot cope with invocations of @(tsee ld) during the process
 of loading a compiled file for a book, so that is an error.</p>

 <p>Specifically: ACL2 will cause an error in the following two
 circumstances:</p>

 <ul>

 <li>when calling @(tsee ld) inside @(tsee progn!) unless state global
 @('ld-okp') is first set to @('t'), e.g., using @('(assign ld-okp t)');
 also,</li>

 <li>when calling @('ld') while inside raw Lisp, e.g., when loading a compiled
 file during an invocation of @(tsee include-book).</li>

 </ul>

 <p>Consider for example the following book, where file @('const.lsp') contains
 the single form @('(defconst *foo* '(a b))') after its initial @(tsee
 in-package) form.</p>

 @({
 (in-package \"ACL2\")
 (defttag t)
 (progn! (ld \"const.lsp\"))
 })

 <p>An attempt to certify this book as follows</p>

 @({
 (certify-book \"const-wrapper\" 0 t :ttags :all)
 })

 <p>will cause an error:</p>

 @({
 ACL2 Error in LD:  It is illegal to call LD in this context.  See :DOC
 calling-ld-in-bad-contexts.
 })

 <p>However, that error can be avoided by expanding the @(tsee progn!) call as
 follows.</p>

 @({
 (progn! (assign ld-okp t)
         (ld \"const.lsp\"))
 })

 <p>Now certification succeeds; however, any subsequent call of @(tsee
 include-book) will fail to load the compiled file for the book.  Again, that
 is necessary because of how ACL2 is designed; in this case, the @(tsee ld)
 call would interfere with tracking of constant definitions when loading the
 compiled file for the book.  To avoid warnings about loading compiled files,
 either certify the book without creating a compiled file or else include the
 book without loading the compiled file; see @(see certify-book) and @(see
 include-book).</p>

 <p>Note that it is legal to put a definition such as the following into a
 book, where @('ld') is called in the body of a function; the two conditions
 above do not prohibit this.</p>

 @({
 (defun foo (state)
   (declare (xargs :guard t :stobjs state :mode :program))
   (ld '((defun h (x) x)) :ld-user-stobjs-modified-warning t))
 })

 <p>One can then include the book, evaluate @('(foo state)'), and then evaluate
 calls of @('h').</p>")

(defxdoc canonical-pathname
  :parents (programming-with-state acl2-built-ins)
  :short "The true absolute filename, with soft links resolved"
  :long "<p>For the name @('fname') of a file, the form @('(Canonical-pathname
 fname nil state)') evaluates to a Unix-style absolute filename representing
 the same file as @('fname'), but generally without any use of soft links in
 the name.  (Below, we explain the qualifier ``generally''.)  If however the
 file indicated by @('fname') does not exist, @('(canonical-pathname fname nil
 state)') is @('nil').  Thus, @('canonical-pathname') can be used as one would
 use the raw Lisp function @('probe-file').</p>

 <p>The specification of @('(Canonical-pathname fname dir-p state)') when
 @('dir-p') is not @('nil') is similar, except that if the specified file
 exists but is not a directory, then the result is @('nil').</p>

 <p>The function @('canonical-pathname') has a guard of @('t'), though the
 second argument must be the ACL2 @(tsee state).  This function is introduced
 with the following properties.</p>

 @({
  (defthm canonical-pathname-is-idempotent
    (equal (canonical-pathname (canonical-pathname x dir-p state) dir-p state)
           (canonical-pathname x dir-p state)))
  (defthm canonical-pathname-type
    (or (equal (canonical-pathname x dir-p state) nil)
        (stringp (canonical-pathname x dir-p state)))
    :rule-classes :type-prescription)
 })

 <p>We use the qualifier ``generally'', above, because there is no guarantee
 that the filename will be canonical without soft links, though we expect this
 to be true in practice.  ACL2 attempts to compute the desired result and then
 checks that the input and result have the same Common Lisp ``@('truename')''.
 This check is expected to succeed, but if it fails then the input string is
 returned unchanged, and to be conservative, the value returned is @('nil') in
 this case if @('dir-p') is true.</p>")

(defxdoc car
  :parents (conses acl2-built-ins)
  :short "Returns the first element of a non-empty list, else @('nil')"
  :long "<p>Completion Axiom (@('completion-of-car')):</p>

 @({
  (equal (car x)
         (cond
          ((consp x)
           (car x))
          (t nil)))
 })

 <p>@(see Guard):</p>

 @({
  (or (consp x) (equal x nil))
 })

 <p>Notice that in the ACL2 logic, @('car') returns @('nil') for every @(see
 atom).</p>")

(defxdoc case
  :parents (basics acl2-built-ins)
  :short "Conditional based on if-then-else using @(tsee eql)"
  :long "@({
  Example Form:
  (case typ
    ((:character foo)
     (open file-name :direction :output))
    (bar (open-for-bar file-name))
    (otherwise
     (my-error \"Illegal.\")))
 })

 <p>is the same as</p>

 @({
  (cond ((member typ '(:character foo))
         (open file-name :direction :output))
        ((eql typ 'bar)
         (open-for-bar file-name))
        (t (my-error \"Illegal.\")))
 })

 <p>which in turn is the same as</p>

 @({
  (if (member typ '(:character foo))
      (open file-name :direction :output)
      (if (eql typ 'bar)
          (open-for-bar file-name)
          (my-error \"Illegal.\")))
 })

 <p>Notice the quotations that appear in the example above: @(''(:character
 foo)') and @(''bar').  Indeed, a @('case') expression expands to a @(tsee
 cond) expression in which each tested form is quoted, and @(tsee eql) is used
 as the test.</p>

 @({
  General Forms:
  (case expr
    (x1 val-1)
    ...
    (xk val-k)
    (otherwise val-k+1))

  (case expr
    (x1 val-1)
    ...
    (xk val-k)
    (t val-k+1))

  (case expr
    (x1 val-1)
    ...
    (xk val-k))
 })

 <p>where each @('xi') is either @(tsee eqlablep) or a true list of @(tsee
 eqlablep) objects.  The final @('otherwise') or @('t') case is optional.</p>

 <p>@('Case') is defined in Common Lisp.  See any Common Lisp documentation for
 more information.</p>")

(defxdoc case-match
  :parents (basics acl2-built-ins)
  :short "Pattern matching or destructuring"
  :long "@({
  General Form:
  (case-match x
    (pat1 dcl1 body1)
    ...
    (patk dclk bodyk))
 })

 <p>where @('x') is a variable symbol, the @('pati') are structural patterns as
 described below, the @('dcli') are optional @(tsee declare) forms and the
 @('bodyi') are terms.  Return the value(s) of the @('bodyi') corresponding to
 the first @('pati') matching @('x'), or @('nil') if none matches.</p>

 <p>Pattern Language:<br></br>

 With the few special exceptions described below, matching requires that the
 @(tsee cons) structure of @('x') be isomorphic to that of the pattern, down to
 the @(see atom)s in the pattern.  Non-symbol @(see atom)s in the pattern match
 only themselves.  Symbols in the pattern denote variables which match anything
 and which are bound by a successful match to the corresponding substructure of
 @('x').  Variables that occur more than once must match the same (@(tsee
 EQUAL)) structure in every occurrence.</p>

 @({
  Exceptions:
  &               Matches anything and is not bound.  Repeated
                    occurrences of & in a pattern may match different
                    structures.
  nil, t, *sym*, :sym
                  These symbols cannot be bound and match only their
                    global values.
  !sym            where sym is a symbol that is already bound in the
                    context of the case-match, matches only the
                    current binding of sym.
  'obj            Matches only itself.  This is the same as (QUOTE obj).
  (QUOTE~ sym)    where sym is a symbol, is like (QUOTE sym) except it
                    matches any symbol with the same symbol-name as sym.
                    Note that QUOTE~ is in the \"ACL2\" package.
 })

 <p>Some examples are shown below.</p>

 <p>Below we show some sample patterns and examples of things they match and do
 not match.</p>

 @({
  pattern       matches         non-matches
  (x y y)       (ABC 3 3)       (ABC 3 4)    ; 3 is not 4
  (fn x . rst)  (P (A I) B C)   (ABC)        ; NIL is not (x . rst)
                (J (A I))                    ; rst matches nil
  ('fn (g x) 3) (FN (H 4) 3)    (GN (G X) 3) ; 'fn matches only itself
  (& t & !x)    ((A) T (B) (C))              ; provided x is '(C)
 })

 <p>Consider the two binary trees that contain three leaves.  They might be
 described as @('(x . (y . z))') and @('((x . y) . z)'), where @('x'), @('y'),
 and @('z') are atomic.  Suppose we wished to recognize those trees.  The
 following @('case-match') would do:</p>

 @({
  (case-match tree
    ((x . (y . z))
     (and (atom x) (atom y) (atom z)))
    (((x . y) . z)
     (and (atom x) (atom y) (atom z))))
 })

 <p>Suppose we wished to recognize such trees where all three tips are
 identical.  Suppose further we wish to return the tip if the tree is one of
 those recognized ones and to return the number @('7') otherwise.</p>

 @({
  (case-match tree
    ((x . (x . x))
     (if (atom x) x 7))
    (((x . x) . x)
     (if (atom x) x 7))
    (& 7))
 })

 <p>Note that @('case-match') returns @('nil') if no @('pati') matches.  Thus
 if we must return @('7') in that case, we have to add as the final pattern the
 @('&'), which always matches anything.</p>

 <p>Technical point: The symbol @('sym') referenced by the symbol @('!sym') is
 in the same package as @('!sym') but with the leading exclamation point
 character, @('\\#!'), removed from the @(tsee symbol-name) of @('!sym').</p>")

(defxdoc case-split
  :parents (rewrite linear type-prescription definition meta forward-chaining)
  :short "Like force but immediately splits the top-level goal on the hypothesis"
  :long "<p>@('Case-split') is a variant of @(tsee force), which has similar
 special treatment in hypotheses of rules for the same @(see rule-classes) as
 for @('force') (see @(see force)).  This treatment takes place for rule
 classes @(':')@(tsee rewrite), @(':')@(tsee linear), @(':')@(tsee
 type-prescription), @(':')@(tsee definition), @(':')@(tsee meta) (actually in
 that case, the result of evaluating the hypothesis metafunction call), and
 @(':')@(tsee forward-chaining).</p>

 <p>When a hypothesis of a conditional rule (of one of the classes listed
 above) has the form @('(case-split hyp)') it is logically equivalent to
 @('hyp').  However it affects the application of the rule generated as
 follows: if ACL2 attempts to apply the rule but cannot establish that the
 required instance of @('hyp') holds in the current context, it considers the
 hypothesis true anyhow, but (assuming all hypotheses are seen to be true and
 the rule is applied) creates a subgoal in which that instance of @('hyp') is
 assumed false.  (There are exceptions, noted below.)</p>

 <p>For example, given the rule</p>

 @({
  (defthm p1->p2
    (implies (case-split (p1 x))
             (p2 x)))
 })

 <p>then an attempt to prove</p>

 @({
  (implies (p3 x) (p2 (car x)))
 })

 <p>can give rise to a single subgoal:</p>

 @({
  (IMPLIES (AND (NOT (P1 (CAR X))) (P3 X))
           (P2 (CAR X))).
 })

 <p>Unlike @(tsee force), @('case-split') does not delay the ``false case'' to
 a forcing round but tackles it more or less immediately.</p>

 <p>The special ``split'' treatment of @('case-split') can be disabled by
 disabling forcing: see @(see force) for a discussion of disabling forcing, and
 also see @(see disable-forcing).  Finally, we should mention that the rewriter
 is never willing to split when there is an @(tsee if) term present in the goal
 being simplified.  Since @(tsee and) terms and @(tsee or) terms are merely
 abbreviations for @(tsee if) terms, they also prevent splitting.  Note that
 @(tsee if) terms are ultimately eliminated using the ordinary flow of the
 proof (but see @(see set-case-split-limitations)), so @('case-split') will
 ultimately function as intended.</p>

 <p>When in the interactive @(see proof-builder), @('case-split') behaves like
 @('force').</p>

 @(def case-split)")

(defxdoc case-split-limitations
  :parents (miscellaneous)
  :short "Limiting the number of immediate subgoals"
  :long "@({
  Examples:
  ACL2 !>(case-split-limitations (w state))
  (500 100)
 })

 <p>With the setting above, which is the default, @('clausify') will not try
 subsumption/replacement if more than 500 clauses are involved.  Furthermore,
 the simplifier, as it sweeps over a clause, will inhibit further case splits
 when it has accumulated 100 subgoals.  To implement this inhibition, ACL2
 refuses to rewrite subsequent literals, although it continues to split on any
 @('IF') calls in those literals.</p>

 <p>The following example illustrates how the latter restriction &mdash;
 specifically, not rewriting subsequent literals to avoid further case splits
 &mdash; can work.  We define a (rather nonsensical) function whose body
 introduces several cases.</p>

 @({
 (defun f1 (a b c)
   (if a
       (if b (equal c 0) (equal c 1))
     (if b (equal c 2) (equal c 3))))

 (set-case-split-limitations '(500 10))

 (set-gag-mode nil)

 (thm (or (equal (f1 a b c) xxx)
          (equal (f1 d e f) yyy)
          (equal (f1 g h i) zzz))
      :hints ((\"Goal\" :in-theory (disable f1))
              (\"Goal'\" :in-theory (enable f1)))
      :otf-flg t)
 })

 <p>We show the output below.  The simplification of the original goal to Goal'
 replaces the original goal, which is the clause (i.e., disjunction) consisting
 of the single literal (i.e., term) shown above, to the clause consisting of
 three literals, namely, the list:</p>

 @({
 ; Goal', as a clause (disjunction of three literals)
 ((EQUAL (F1 A B C) XXX)
  (EQUAL (F1 D E F) YYY)
  (EQUAL (F1 G H I) ZZZ))
 })

 <p>That simplification is reflected in the value printed (as an implication)
 for Goal' in the output, below.  If we omit the call of @(tsee
 set-case-split-limitations) above, then we get 64 cases, from opening up the
 calls of @('f1') and splitting on the variables @('a'), @('b'), @('d'),
 @('e'), @('g'), and @('h').  But with the limit of 10 provided by
 set-case-split-limitations above, fewer cases are generated because rewriting
 of literals is inhibited, as explained below.  Here is the first part of the
 output for that limit of 10.</p>

 @({
 [Note:  A hint was supplied for our processing of the goal above.
 Thanks!]

 This simplifies, using trivial observations, to

 [Note:  A hint was supplied for our processing of the goal below.
 Thanks!]

 Goal'
 (IMPLIES (AND (NOT (EQUAL (F1 A B C) XXX))
               (NOT (EQUAL (F1 D E F) YYY)))
          (EQUAL (F1 G H I) ZZZ)).

 This simplifies, using the :definition F1 (if-intro), to the following
 thirteen conjectures.

 Subgoal 13
 (IMPLIES (AND A B (NOT (EQUAL (EQUAL C 0) XXX))
               (NOT (EQUAL (F1 D E F) YYY)))
          (EQUAL (F1 G H I) ZZZ)).
 })

 <p>Why, though, are there 13 cases, even though the limit was specified as 10?
 Initially, the first literal @('(EQUAL (F1 A B C) XXX)') was rewritten,
 splitting into four cases; and for each of those cases, the second literal was
 rewritten, splitting further; and so on.  However, the first time more than 10
 cases were generated was when there were 10 cases generated so far and a
 literal generated four cases &mdash; then since that one literal generated
 four cases, 4-1 = 3 cases were added to the 10 already generated.  From that
 point on, further rewriting of literals did not take place.</p>

 <p>In short: the first time a literal splits into enough cases so that the
 accumulated number of cases exceeds the limit, rewriting stops &mdash; but
 that last split can put us significantly past the limit specified.</p>

 <p>See @(see set-case-split-limitations) for a more general discussion.</p>")

(defxdoc cbd
  :parents (books-reference programming-with-state acl2-built-ins)
  :short "Connected book directory string"
  :long "@({
  Example:
  ACL2 !>:cbd
  \"/usr/home/smith/\"
 })

 <p>The connected book directory is a nonempty string that specifies a
 directory as an absolute pathname.  (See @(see pathname) for a discussion of
 file naming conventions.)  When @(tsee include-book) is given a relative book
 name it elaborates it into a full book name, essentially by appending the
 connected book directory string to the left and @('\".lisp\"') to the right.
 (For details, see @(see book-name) and also see @(see full-book-name).)
 Similarly, @(tsee ld) elaborates relative pathnames into full pathnames using
 the connected book directory string.  (The effect of the @('cbd') on @('ld')
 carries over to utilities that invoke @('ld') as well, notably, @(tsee
 rebuild).)  Furthermore, @(tsee include-book) and @(tsee ld) temporarily set
 the connected book directory to the directory string of the resulting full
 pathname so that references to files in the same directory may omit the
 directory.  See @(see set-cbd) for how to set the connected book directory
 string.</p>

 @({
  General Form:
  (cbd)
 })

 <p>This is a macro that expands into a term involving the single free variable
 @(tsee state).  It returns the connected book directory string.</p>

 <p>The connected book directory (henceforth called the ``@('cbd')'') is used
 by @(tsee include-book) to elaborate the supplied book name into a full book
 name (see @(see full-book-name)); similarly for @(tsee ld).  For example, if
 the @('cbd') is @('\"/usr/home/smith/\"') then the elaboration of the @(see
 book-name) @('\"project/task-1/arith\"') (to the @('\".lisp\"') extension) is
 @('\"/usr/home/smith/project/task-1/arith.lisp\"').  That @(see
 full-book-name) is what @(see include-book) opens to read the source text for
 the book.</p>

 <p>The @('cbd') may be changed using @(tsee set-cbd) (see @(see set-cbd)).
 Furthermore, during the processing of the @(see events) in a book, @(tsee
 include-book) sets the @('cbd') to be the directory string of the @(see
 full-book-name) of the book; similarly for @(tsee ld).  Thus, if the @('cbd')
 is @('\"/usr/home/smith/\"') then during the processing of @(see events)
 by</p>

 @({
  (include-book \"project/task-1/arith\")
 })

 <p>the @('cbd') will be set to @('\"/usr/home/smith/project/task-1/\"').  Note
 that if @('\"arith\"') recursively includes a sub-book, say @('\"naturals\"'),
 that resides on the same directory, the @(tsee include-book) event for it may
 omit the specification of that directory.  For example, @('\"arith\"') might
 contain the event</p>

 @({
    (include-book \"naturals\").
 })

 <p>In general, suppose we have a superior book and several inferior @(see
 books) which are included by @(see events) in the superior book.  Any inferior
 book residing on the same directory as the superior book may be referenced in
 the superior without specification of the directory.</p>

 <p>We call this a ``relative'' as opposed to ``absolute'' naming.  The use of
 relative naming is preferred because it permits @(see books) (and their
 accompanying inferiors) to be moved between directories while maintaining
 their @(see certificate)s and utility.  Certified @(see books) that reference
 inferiors by absolute file names are unusable (and rendered uncertified) if
 the inferiors are moved to new directories.</p>

 <p><i>Technical Note and a Challenge to Users:</i></p>

 <p>After elaborating the book name to a full book name, @(tsee include-book)
 opens a channel to the file to process the @(see events) in it.  In some host
 Common Lisps, the actual file opened depends upon a notion of ``connected
 directory'' similar to our connected book directory.  Our intention in always
 elaborating book names into absolute filename strings (see @(see pathname) for
 terminology) is to circumvent the sensitivity to the connected directory.  But
 we may have insufficient control over this since the ultimate file naming
 conventions are determined by the host operating system rather than Common
 Lisp (though, we do check that the operating system ``appears'' to be one that
 we ``know'' about).  Here is a question, which we'll pose assuming that we
 have an operating system that calls itself ``Unix.''  Suppose we have a file
 name, filename, that begins with a slash, e.g., @('\"/usr/home/smith/...\"').
 Consider two successive invocations of CLTL's</p>

 @({
  (open filename :direction :input)
 })

 <p>separated only by a change to the operating system's notion of connected
 directory.  Must these two invocations produce streams to the same file?  A
 candidate string might be something like
 @('\"/usr/home/smith/*/usr/local/src/foo.lisp\"') which includes some
 operating system-specific special character to mean ``here insert the
 connected directory'' or, more generally, ``here make the name dependent on
 some non-ACL2 aspect of the host's state.''  If such ``tricky'' name strings
 beginning with a slash exist, then we have failed to isolate ACL2 adequately
 from the operating system's file naming conventions.  Once upon a time, ACL2
 did not insist that the @('cbd') begin with a slash and that allowed the
 string @('\"foo.lisp\"') to be tricky because if one were connected to
 @('\"/usr/home/smith/\"') then with the empty @('cbd') @('\"foo.lisp\"') is a
 full book name that names the same file as @('\"/usr/home/smith/foo.lisp\"').
 If the actual file one reads is determined by the operating system's state
 then it is possible for ACL2 to have two distinct ``full book names'' for the
 same file, the ``real'' name and the ``tricky'' name.  This can cause ACL2 to
 include the same book twice, not recognizing the second one as
 redundant.</p>")

(defxdoc ccl-installation
  :parents (building-acl2)
  :short "Installing Clozure Common Lisp (CCL)"
  :long "<p>For those who use ACL2 built on CCL as the host Common Lisp
 implementation, it has been common practice to use the latest GitHub version
 of CCL.  We provide the following instructions for you to choose from.  The
 ``brief'' instructions for Linux or Mac (according to your operating system)
 might well suffice; the ``elaborate'' instructions have helped with version
 control.</p>

 <ul>

 <li>@(see ccl-installation-linux-brief)</li>

 <li>@(see ccl-installation-mac-brief)</li>

 <li>@(see ccl-installation-linux-elaborate)</li>

 <li>@(see ccl-installation-mac-elaborate)</li>

 </ul>

 <p>You may prefer instead to look at the <a
 href='https://github.com/Clozure/ccl/releases'>CCL Releases</a> page, using
 links above only as needed (e.g., for Linux-specific information or for
 discussion of @('CCL_DEFAULT_DIRECTORY')).</p>

 <p>One of the links listed above should generally suffice.  But if you
 would like additional information on CCL installation and implementation, see
 @(see ccl-installation-extra).</p>")

(defxdoc ccl-installation-extra
  :parents (ccl-installation)
  :short "Clozure Common Lisp (CCL) installation and implementation details"
  :long "<p>This topic, contributed by Warren A. Hunt, Jr., extends the basic
 information given in @(see ccl-installation).  It may be useful to some,
 especially those who use ACL2 in ways that particularly stress memory.
 Another resource may be found on <a
 href='https://github.com/Clozure/ccl/releases/'>this page</a>.</p>

 <p>Below we provide instructions to build CCL on FreeBSD and MacOS; building
 on Linux will be similar.  Before providing the build instructions, we mention
 a few facts about CCL's implementation.</p>

 <p>CCL's implementation can't expand stack space automatically.  The sizes of
 various stacks are set at thread creation time.  Most programs don't need very
 big stacks.  The CCL default value and temp stack sizes may be too small for
 compute-intensive applications.</p>

 <p>The various stack sizes are set when creating a thread with code in
 @('CCL::MAKE-PROCESS') and @('CCL::PROCESS-RUN-FUNCTION'), and they set the
 three stack sizes of the initial listener thread that is created when CCL
 starts.</p>

 <p>The value stack is used for data in deeply-nested lisp recursion.  ACL2 can
 benefit from an increase in the size of the value stack.</p>

 <p>The temp stack is used for dynamic-extent objects.  This might need need to
 be larger than the default.</p>

 <p>The control stack is used to run C and binary code; it would be surprising
 to need to increase its size.</p>

 <p>If only one or two threads are needed, giant (e.g., 4 GB) stacks can
 be OK, but with many threads large stacks use lots of memory.  Below
 are the stacks along with their current (April, 2020) default sizes.</p>

  @({
  ccl::*default-control-stack-size*                    ;; default 2^21
  ccl::*initial-listener-default-control-stack-size*   ;; default 2^21

  ccl::*default-value-stack-size*                      ;; default 2^21
  ccl::*initial-listener-default-value-stack-size*     ;; default 2^21

  ccl::*initial-listener-default-temp-stack-size*      ;; default 2^20
  ccl::*default-temp-stack-size*                       ;; default 2^20
  })

 <p>If we are running on a 32-bit platform, we set the stack sizes modestly.
 If we are on a 64-bit platform, we set the stack sizes to much larger
 values.  See the later discussion about ``configure-ccl.lisp'' below to
 see how to alter (increase) stack sizes.</p>

 <h3>MacOS Build Instructions:</h3>

 @({
 git clone https://github.com/Clozure/ccl.git ccl-dev
 curl -L -O https://github.com/Clozure/ccl/releases/download/v1.12-dev.5/darwinx86.tar.gz
 cd ccl-dev ; tar xf ../darwinx86.tar.gz
 })

 <p>Rebuild C-based, Lisp kernel</p>

 <p>To rebuild the Lisp-code part of the kernel, do...</p>

 @({
 cd lisp-kernel/darwinx8664 ; make ; cd ../..
 })

 <p>Unlike for FreeBSD and Linux, we exclude ``:full t'' from the
 ``rebuild-ccl'' command just below Matt Emerson (a CCL expert) writes:</p>

 <blockquote>

 <p>After looking at your log, I was able to duplicate the problem myself.
 For some reason I do not understand, it appears that on Catalina,
 removing the running lisp kernel binary causes run-program to break
 (trying to run external programs gets signal 9).  This surprises me
 very much.</p>

 <p>One of the effects of running (rebuild-ccl :full t), is that it first does
 a \"make clean\" in the lisp kernel directory, and then does a regular make.
 This has worked for years, and I do not know why it has stopped working.</p>

 </blockquote>

 <p>To avoid this, rebuild the lisp with (rebuild-ccl :clean t) instead.  Do
 not specify ``@(':full t')''.  Since you have already built the lisp kernel,
 you gain nothing from ``@(':full t')'' doing it again.  So, once the C-based,
 Lisp kernel is built, then do:</p>

 @({
 echo \"(in-package :ccl) (rebuild-ccl :verbose t :clean t)\" | \\
       ./dx86cl64 -n |& tee ./ccl-build-compile.log
 })

 <p>Matt Emerson suggests that one re-build again.  We asked Matt a
 long-simmering question: we have been told that it is a good idea to compile
 CCL twice.  Is that so that the CCL compiler produced by pass one on the
 target system is used to compile the CCL system that will be used (on the
 target system)?  Or, is compiling twice some silly myth?  Matt Emerson
 responded:</p>

 <blockquote>

 <p>It's not entirely mythic.  Some rare changes do need the lisp to be rebuilt
 twice for bootstrapping purposes, but usually it isn't required.</p>

 </blockquote>

 <p>Thus, we recommend you re-build (compile) the Lisp code again.</p>

 @({
 echo \"(in-package :ccl) (rebuild-ccl :verbose t :clean t)\" | \\
       ./dx86cl64 -n |& tee ./ccl-build-compile-2.log
 })

 <p>Finally, one may specialize the final image by:</p>

 @({
 cat configure-ccl.lisp | ./dx86cl64 -n |& tee ~/ccl-build-specialize.log
 })

 <p>where ``configure-ccl.lisp'' (not supplied) contains whatever CCL
 specialization commands you wish to have in the version of CCL you use for
 ACL2 or other work.  See the end of this note for an example of
 ``configure-ccl.lisp''.</p>

 <h3>FreeBSD Build Instructions:</h3>

 <p>The FreeBSD build instructions are similar to the MacOS build instruction,
 but the names are changed appropriately.</p>

 @({
 git clone https://github.com/Clozure/ccl.git ccl-dev
 curl -L -O https://github.com/Clozure/ccl/releases/download/v1.12-dev.5/freebsd12-x8664.tar.gz
 cd ccl-dev
 tar xf ../freebsd12-x8664.tar.gz
 })

 <p>Rebuild C-based Lisp kernel</p>

 @({
 cd lisp-kernel/freebsdx8664 ; make ; cd ../..
 })

 <p>To rebuild the Lisp-code part of the kernel, do...</p>

 @({
 echo \"(in-package :ccl) (rebuild-ccl :full t :verbose t :clean t)\" | \\
       ./fx86cl64 -n |& tee ./ccl-build-compile.log
 })

 <p>FreeBSD is OK with the ``:full t'' flag, which as of MacOS 10.15
 breaks (but used to work on earlier versions of MacOS).  So, this
 option persists on the FreeBSD build.</p>

 <p>Matt Emerson recommends building the Lisp code a second time; see
 the ``MacOS Build Instructions'' above for his rationale.</p>

 @({
 echo \"(in-package :ccl) (rebuild-ccl :full t :verbose t :clean t)\" | \\
       ./fx86cl64 -n |& tee ./ccl-build-compile-2.log
 })

 <p>Finally, one may specialize the final image by:</p>

 @({
 cat ~/a/scripts/configure-ccl.lisp | ./fx86cl64 -n |& tee ~/ccl-build-specialize.log
 })

 <p>where ``configure-ccl.lisp'' (not supplied) contains whatever CCL
 specialization commands you wish to have in the version of CCL you use for
 ACL2 or other work.</p>

 <code><h3>configure-ccl.lisp</h3></code>

 <p>Sample ``configure-ccl.lisp'' file for 64-bit implementation:</p>

 @({
 (progn
   ;; Parameters to configure CCL for use on FreeBSD and MacOS

   (in-package :ccl)

   ;; Enlarge stack sizes
   (setq *default-value-stack-size*                    (expt 2 28))
   (setq *initial-listener-value-stack-size*           (expt 2 28))

   (setq *default-temp-stack-size*                     (expt 2 24))
   (setq *initial-listener-temp-stack-size*            (expt 2 24))

   (setq *default-control-stack-size*                  (expt 2 24))
   (setq *initial-listener-default-control-stack-size* (expt 2 24))

   ;; For CCL double precision
   (setf *read-default-float-format* 'double-float)

   ;; Make DEFUN save the source code for later recovery via
   ;; FUNCTION-LAMBDA-EXPRESSION.
   (setq *save-definitions* t)
   (setq *fasl-save-definitions* t)

   ;; Make GC verbose; see ACL2 documentation topic GC-VERBOSE.
   (gc-verbose t t)

   ;; Dump executable heap image; see ACL2 documentation topic SAVE-EXEC.
   (save-exec *heap-image-name* \"Modification string to print at startup\")
   )
 })")

(defxdoc ccl-installation-linux-brief

; Warning: Keep this in sync with ccl-installation-mac-brief and, to a lesser
; extent, ccl-installation-linux-elaborate.

  :parents (ccl-installation)
  :short "Installing Clozure Common Lisp (CCL) on Linux (brief version)"
  :long "<p>See @(see ccl-installation) for introductory remarks.  The
 instructions below describe how to install CCL on Linux.  For more elaborate
 ``cookbook'' instructions see @(see ccl-installation-linux-elaborate).</p>

 <p><b>Note</b>: Linux users may need to install m4.</p>

 <p>Fetch CCL from GitHub into a fresh subdirectory, @('ccl/').</p>

 @({
 git clone https://github.com/Clozure/ccl
 })

 <p>Next fetch and extract a development snapshot in the new @('ccl')
 directory.  The version below is current as of this writing (April, 2021), but
 see <a
 href='https://github.com/Clozure/ccl/releases/'>https://github.com/Clozure/ccl/releases/</a>
 for the latest snapshots.</p>

 @({
 cd ccl
 wget https://github.com/Clozure/ccl/releases/download/v1.12/linuxx86.tar.gz
 tar xfz linuxx86.tar.gz
 })

 <p>Rebuild and quit, twice.</p>

 @({
 echo '(rebuild-ccl :full t)' | ./lx86cl64
 echo '(rebuild-ccl :full t)' | ./lx86cl64
 })

 <p>Create the following executable script, which you will probably want to
 name @('\"ccl\"'), where @('<DIR>') is the absolute pathname (without using
 ``@('~')'') of the directory in which you issued the ``@('git clone')''
 command.  Place this script wherever you wish, though putting it in a
 directory on your Unix @('PATH') might be helpful (since then you can invoke
 it as, say @('\"ccl\"')).</p>

 @({
 #!/bin/sh

 export CCL_DEFAULT_DIRECTORY=<DIR>/ccl
 ${CCL_DEFAULT_DIRECTORY}/scripts/ccl64 \"$@\"
 })

 <p>You're done!  (Note however that certification of @(see books) that use
 @(see Quicklisp) may require @('openssl') to be installed if it is not already
 on your system.)</p>")

(defxdoc ccl-installation-linux-elaborate

; Warning: Keep this in sync with ccl-installation-mac-elaborate and, to a
; lesser extent, ccl-installation-linux-brief.

  :parents (ccl-installation)
  :short "Installing Clozure Common Lisp (CCL) on Linux (elaborate version)"
  :long "<p>See @(see ccl-installation) for introductory remarks.  The
 ``cookbook'' instructions below give you one way to install CCL on Linux
 without any knowledge of git or CCL.  For more streamlined instructions see
 @(see ccl-installation-linux-brief).</p>

 <p><b>Note</b>: Linux users may need to install m4.</p>

 <p>First fetch CCL from GitHub as follows.  (You may prefer to use ``@('git
 pull')'' if you previously did this step.  In that case you probably won't
 want to do the optional renaming of the directory, mentioned below.)</p>

 @({
 # Obtain a ccl distribution in a fresh directory:
 mkdir temp
 cd temp
 git clone https://github.com/Clozure/ccl
 # Optionally rename that directory as suggested below, after
 # executing the following three commands.
 cd ccl
 git rev-parse HEAD
 cd ../../
 # Optionally change directory name, and then go back to ccl directory:
 # You'll want the last 10 hex digits to match those of the
 # output from the ``git rev-parse HEAD'' command above: do
 # that twice here and once further below.
 mv temp 2017-12-07-6be8298fe5
 cd 2017-12-07-6be8298fe5/ccl
 })

 <p>Next fetch and extract a development snapshot.  The version below is
 current as of this writing (April, 2021), but see <a
 href='https://github.com/Clozure/ccl/releases/'>https://github.com/Clozure/ccl/releases/</a>
 for the latest snapshots.</p>

 @({
 wget https://github.com/Clozure/ccl/releases/download/v1.12/linuxx86.tar.gz
 tar xfz linuxx86.tar.gz
 })

 <p>Rebuild and quit, twice.</p>

 @({
 echo '(rebuild-ccl :full t)' | ./lx86cl64
 echo '(rebuild-ccl :full t)' | ./lx86cl64
 })

 <p>Create an executable script like the following.  You might want to call it
 ``@('ccl')'' and put it into a directory on your path.  Be sure to change the
 name (shown as ``2017-12-07-6be8298fe5'' above) to match the name change
 already made above.</p>

 @({
 #!/bin/sh

 export CCL_DEFAULT_DIRECTORY=/projects/acl2/lisps/ccl/2017-12-07-6be8298fe5/ccl
 ${CCL_DEFAULT_DIRECTORY}/scripts/ccl64 \"$@\"
 })

 <p>Now ensure that your script is executable, e.g.:</p>

 @({
 chmod +x my-script
 })

 <p>You're done!  (Note however that certification of @(see books) that use
 @(see Quicklisp) may require @('openssl') to be installed if it is not already
 on your system.)</p>")

(defxdoc ccl-installation-mac-brief

; Warning: Keep this in sync with ccl-installation-linux-brief and, to a lesser
; extent, ccl-installation-macc-elaborate.

  :parents (ccl-installation)
  :short "Installing Clozure Common Lisp (CCL) on Mac (brief version)"
  :long "<p>See @(see ccl-installation) for introductory remarks.  The
 instructions below describe how to install CCL on a Mac (Darwin).  For more
 elaborate ``cookbook'' instructions see @(see
 ccl-installation-mac-elaborate).</p>

 <p>Fetch CCL from GitHub into a fresh subdirectory, @('ccl/').</p>

 @({
 git clone https://github.com/Clozure/ccl
 })

 <p>Next fetch and extract a development snapshot in the new @('ccl')
 directory.  The version below is current as of this writing (April, 2021), but
 see <a
 href='https://github.com/Clozure/ccl/releases/'>https://github.com/Clozure/ccl/releases/</a>
 for the latest snapshots.</p>

 @({
 cd ccl
 curl -O -L https://github.com/Clozure/ccl/releases/download/v1.12/darwinx86.tar.gz
 tar xfz darwinx86.tar.gz
 })

 <p>Rebuild the lisp kernel by hand before trying to rebuild the lisp.</p>

 @({
 cd lisp-kernel/darwinx8664; make clean; make
 cd -
 })

 <p>Rebuild and quit, twice.</p>

 @({
 echo '(rebuild-ccl :clean t)' | ./lx86cl64
 echo '(rebuild-ccl :clean t)' | ./lx86cl64
 })

 <p>Create the following executable script, which you will probably want to
 name @('\"ccl\"'), where @('<DIR>') is the absolute pathname (without using
 ``@('~')'') of the directory in which you issued the ``@('git clone')''
 command.  Place this script wherever you wish, though putting it in a
 directory on your Unix @('PATH') might be helpful (since then you can invoke
 it as, say @('\"ccl\"')).</p>

 @({
 #!/bin/sh

 export CCL_DEFAULT_DIRECTORY=<DIR>/ccl
 ${CCL_DEFAULT_DIRECTORY}/scripts/ccl64 \"$@\"
 })

 <p>You're done!  (Note however that certification of @(see books) that use
 @(see Quicklisp) may require @('openssl') to be installed if it is not already
 on your system.)</p>")

(defxdoc ccl-installation-mac-elaborate

; Warning: Keep this in sync with ccl-installation-mac-elaborate and, to a
; lesser extent, ccl-installation-mac-brief.

  :parents (ccl-installation)
  :short "Installing Clozure Common Lisp (CCL) on Mac (elaborate version)"
  :long "<p>See @(see ccl-installation) for introductory remarks.  The
 ``cookbook'' instructions below give you one way to install CCL on a Mac
 (Darwin) without any knowledge of git or CCL.  For more streamlined
 instructions see @(see ccl-installation-mac-brief).</p>

 <p>First fetch CCL from GitHub as follows.  (You may prefer to use ``@('git
 pull')'' if you previously did this step.  In that case you probably won't
 want to do the optional renaming of the directory, mentioned below.)</p>

 @({
 # Obtain a ccl distribution in a fresh directory:
 mkdir temp
 cd temp
 git clone https://github.com/Clozure/ccl
 # Optionally rename that directory as suggested below, after
 # executing the following three commands.
 cd ccl
 git rev-parse HEAD
 cd ../../
 # Optionally change directory name, and then go back to ccl directory:
 # You'll want the last 10 hex digits to match those of the
 # output from the ``git rev-parse HEAD'' command above: do
 # that twice here and once further below.
 mv temp 2017-12-07-6be8298fe5
 cd 2017-12-07-6be8298fe5/ccl
 })

 <p>Next fetch and extract a development snapshot.  The version below is
 current as of this writing (April, 2021), but see <a
 href='https://github.com/Clozure/ccl/releases/'>https://github.com/Clozure/ccl/releases/</a>
 for the latest snapshots.</p>

 @({
 curl -O -L https://github.com/Clozure/ccl/releases/download/v1.12/darwinx86.tar.gz
 })

 <p>Rebuild the lisp kernel by hand before trying to rebuild the lisp.
 (Note: This step was formerly unnecessary and might become unnecessary again,
 but it seems to have been necessary on MacOS Catalina (10.15).</p>

 @({
 cd lisp-kernel/darwinx8664; make clean; make
 cd -
 })

 <p>Rebuild and quit, twice.</p>

 @({
 echo '(rebuild-ccl :clean t)' | ./lx86cl64
 echo '(rebuild-ccl :clean t)' | ./lx86cl64
 })

 <p>Create an executable script like the following.  You might want to call it
 ``@('ccl')'' and put it into a directory on your path.  Be sure to change the
 name (shown as ``2017-12-07-6be8298fe5'' above) to match the name change
 already made above.</p>

 @({
 #!/bin/sh

 export CCL_DEFAULT_DIRECTORY=/projects/acl2/lisps/ccl/2017-12-07-6be8298fe5/ccl
 ${CCL_DEFAULT_DIRECTORY}/scripts/ccl64 \"$@\"
 })

 <p>Now ensure that your script is executable, e.g.:</p>

 @({
 chmod +x my-script
 })

 <p>You're done!  (Note however that certification of @(see books) that use
 @(see Quicklisp) may require @('openssl') to be installed if it is not already
 on your system.)</p>")

(defxdoc cdaaar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee caaar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdaadr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee caadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdaar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee caar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdadar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cadar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdaddr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee caddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdadr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee car)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cddaar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cdaar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cddadr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cdadr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cddar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cdar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdddar
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cddar)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cddddr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cdddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdddr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cddr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cddr
  :parents (conses acl2-built-ins)
  :short "@(tsee cdr) of the @(tsee cdr)"
  :long "<p>See any Common Lisp documentation for details.</p>")

(defxdoc cdr
  :parents (conses acl2-built-ins)
  :short "Returns the second element of a @(tsee cons) pair, else @('nil')"
  :long "<p>Completion Axiom (@('completion-of-cdr')):</p>

 @({
  (equal (cdr x)
         (cond
          ((consp x)
           (cdr x))
          (t nil)))
 })

 <p>@(see Guard):</p>

 @({
  (or (consp x) (equal x nil))
 })

 <p>Notice that in the ACL2 logic, @('cdr') returns @('nil') for every @(see
 atom).</p>")

(defxdoc ceiling
  :parents (numbers acl2-built-ins)
  :short "Division returning an integer by truncating toward positive infinity"
  :long "@({
  Example Forms:
  ACL2 !>(ceiling 14 3)
  5
  ACL2 !>(ceiling -14 3)
  -4
  ACL2 !>(ceiling 14 -3)
  -4
  ACL2 !>(ceiling -14 -3)
  5
  ACL2 !>(ceiling -15 -3)
  5
 })

 <p>@('(Ceiling i j)') is the result of taking the quotient of @('i') and
 @('j') and returning the smallest integer that is at least as great as that
 quotient.  For example, the quotient of @('-14') by @('3') is @('-4 2/3'), and
 the smallest integer at least that great is @('-4').</p>

 <p>The @(see guard) for @('(ceiling i j)') requires that @('i') and @('j') are
 rational (@(see real), in ACL2(r)) numbers and @('j') is non-zero.</p>

 <p>@('Ceiling') is a Common Lisp function.  See any Common Lisp documentation
 for more information.  However, note that unlike Common Lisp, the ACL2
 @('ceiling') function returns only a single value,</p>

 @(def ceiling)")

(defxdoc certificate
  :parents (books-tour)
  :short "A file specifying validity of a given book"
  :long "<p>A book, say @('\"arith\"'), is said to have a ``certificate'' if
 there is a file named @('\"arith.cert\"').  Certificates are created by the
 function @(tsee certify-book) and inspected by @(tsee include-book).  @(Csee
 book-hash) values are used to help ensure that certificates are legitimate and
 that the corresponding book has not been modified since certification.  But
 because the file system is insecure and book-hash values are not perfect, it
 is possible for the inclusion of a book to cause inconsistency even though the
 book carries an impeccable certificate.</p>

 <p>The certificate includes the version number of the certifying ACL2.  A book
 is considered uncertified if it is included in an ACL2 with a different @(see
 version) number.</p>

 <p>The presence of a ``valid'' certificate file for a book attests to two
 things: all of the @(see events) of the book are admissible in a certain
 extension of the initial ACL2 logic, and the non-@(tsee local) @(see events)
 of the book are independent of the @(tsee local) ones (see @(see
 local-incompatibility)).  In addition, the certificate contains the @(see
 command)s used to construct the @(see world) in which certification occurred.
 Among those @(see command)s, of course, are the @(tsee defpkg)s defining the
 packages used in the book.  When a book is included into a host @(see world),
 that @(see world) is first extended by the @(see command)s listed in the
 certificate for the book.  Unless that causes an error due to name conflicts,
 the extension ensures that all the packages used by the book are identically
 defined in the host @(see world).</p>

 <p><i>Security:</i></p>

 <p>Because the host file system is insecure, there is no way ACL2 can
 guarantee that the contents of a book remain the same as when its certificate
 was written.  That is, between the time a book is certified and the time it is
 used, it may be modified.  Furthermore, certificates can be counterfeited.
 @(csee Book-hash) values are used to help detect such problems, but provide
 imperfect security: two different files can have the same book-hash.</p>

 <p>Therefore, from the strictly logical point of view, one must consider even
 the inclusion of certified @(see books) as placing a burden on the user:</p>

 <blockquote>

 <p>The non-erroneous inclusion of a certified book is consistency preserving
 provided (a) the objects read by @(tsee include-book) from the certificate
 were the objects written there by a @(tsee certify-book) and (b) the forms
 read by @(tsee include-book) from the book itself are the forms read by the
 corresponding @(tsee certify-book).</p>

 </blockquote>

 <p>We say that a given execution of @(tsee include-book) is ``certified'' if a
 certificate file for the book is present and well-formed and the @(see
 book-hash) information contained within it supports the conclusion that the
 @(see events) read by the @(tsee include-book) are the ones checked by @(tsee
 certify-book).  When an uncertified @(tsee include-book) occurs, warnings are
 printed or errors are caused.  But even if no warning is printed, you must
 accept burdens (a) and (b) if you use @(see books).  These burdens are easier
 to live with if you protect your @(see books) so that other users cannot write
 to them, you abstain from running concurrent ACL2 jobs, and you abstain from
 counterfeiting certificates.  But even on a single user uniprocessor, you can
 shoot yourself in the foot by using the ACL2 @(see io) primitives to fabricate
 an inconsistent book and the corresponding certificate.</p>

 <p>Note that part (a) of the burden described above implies, in particular,
 that there are no guarantees when a certificate is copied.  When @(see books)
 are renamed (as by copying them), it is recommended that their certificates be
 removed and the @(see books) be recertified.  The expectation is that
 recertification will go through without a hitch if relative @(see pathname)s
 are used.  See @(see pathname), which is not on the guided tour.</p>

 <p>Certificates contain a @(see portcullis) and a @(see keep), each documented
 later in this guided tour through @(see books).  We mention here two other
 parts, which we do not document elsewhere because we view them as
 implementation details: an @('expansion-alist'), which records @(tsee
 make-event) expansions; and @('cert-data'), which contains logical information
 associated with the book.  (Those curious about @('cert-data') are invited to
 look at the ``Essay on Cert-data'' in the source code.)</p>

 <p>See @(see portcullis) to continue the guided tour through @(see
 books).</p>")

(defxdoc certify-book
  :parents (books-reference books-tour)
  :short "How to produce a @(see certificate) for a book"
  :long "<p>Also see @(see certificate) for information about the
 @('\".cert\"') file produced by @('certify-book'), in particular for
 information about the use of @(see book-hash) values to help ensure that the
 corresponding book has not been modified since certification.  See @(see
 certify-book-debug) for some potential remedies for failures of
 @('certify-book').</p>

 @({
  Examples:
  (certify-book \"my-arith\")          ; certify in a world with 0 commands
  (certify-book \"my-arith\" 3)        ; ... in a world with 3 commands
  (certify-book \"my-arith\" ?)        ; ... in a world without checking the
                                     ;     number of commands
  (certify-book \"my-arith\" 0 nil)    ; ... without compilation
  (certify-book \"my-arith\" 0 t)      ; ... with compilation (default)
  (certify-book \"my-arith\" 0 t :ttags (foo))
                                     ; ... allowing trust tag (ttag) foo
  (certify-book \"my-arith\" 0 t :ttags :all)
                                     ; ... allowing all trust tags (ttags)
  (certify-book \"my-arith\" t)        ; ... from world of old certificate
  (certify-book \"my-arith\" 0 nil :acl2x t)
                                     ; ... writing or reading a .acl2x file
  (certify-book \"my-arith\" 0 t :useless-runes :write)
                                     ; ... write file to speed up future proofs
  (certify-book \"my-arith\" 0 t :useless-runes :read)
                                     ; ... read file to speed up future proofs

  General Form:
  (certify-book book-name
                k                       ; [default 0]
                compile-flg             ; [default t]
                :defaxioms-okp t/nil    ; [default nil]
                :skip-proofs-okp t/nil  ; [default nil]
                :ttags ttags            ; [default nil]
                :acl2x t/nil            ; [default nil]
                :ttagsx ttags           ; [default nil]
                :pcert pcert            ; [default nil]
                :write-port t/nil       ; [default t unless pcert is non-nil]
                :useless-runes :write/:read/:read?/n/-n/nil
                                        ; (-100 < n < 0 or 0 < n <= 100)
                                        ; [default nil]
                )
 })

 <p>where @('book-name') is a book name (see @(see book-name)), @('k') is used
 to indicate your approval of the ``certification @(see world),'' and
 @('compile-flg') can control whether the book is to be compiled.  The defaults
 for @('compile-flg'), @('skip-proofs-okp'), @('acl2x'), @('write-port'),
 @('pcert'), and @(':useless-runes') can be affected by environment variables.
 All of these arguments are described in detail below, except for @(':pcert')
 and @(':useless-runes'): see @(see provisional-certification) and @(see
 useless-runes), respectively, for the effects of these two arguments and their
 corresponding environment variables, as we ignore those effects in the present
 topic.</p>

 <p>Certification occurs in some logical @(see world), called the
 ``certification @(see world).''  That @(see world) must contain the @(tsee
 defpkg)s needed to read and execute the forms in the book.  The @(see
 command)s necessary to recreate that @(see world) from the ACL2 initial @(see
 world) are called the ``@(see portcullis) commands,'' and will be copied into
 the @(see certificate) created for the book.  Those @(see command)s will be
 re-executed whenever the book is included, to ensure that the appropriate
 packages (and all other names used in the certification @(see world)) are
 correctly defined.  Note that Step 1 of @('certify-book') will fail if a
 package mentioned in the book is not defined before attempting the
 certification, i.e., defined by a portcullis command in the certification
 world.  For example, suppose that your book contains the symbol @('FOO::X'),
 but the package \"FOO\" is not currently defined.  Then an error message will
 likely complain either about a missing package @('\"FOO\"'), or about a symbol
 @('FOO::X') that is ``not in any of the packages known to ACL2.''  A solution
 is to define the package \"FOO\", perhaps directly using @(tsee defpkg) or by
 including a book that defines this package, before attempting the
 certification.</p>

 <p>The certified book will be more often usable if the certification @(see
 world) is kept to a minimal extension of the ACL2 initial @(see world) (for
 example, to prevent name clashes with functions defined in other books).
 Thus, before you call @('certify-book') for the first time on a book, you may
 wish to get into the initial ACL2 @(see world) (e.g., with @(':ubt 1') or just
 starting a new version of ACL2), @(tsee defpkg) the desired packages, and then
 invoke @('certify-book').</p>

 <p>The @('k') argument to @('certify-book') must be either a nonnegative
 integer or else one of the symbols @('t') or @('?') in any package.  If @('k')
 is an integer, then it must be the number of @(see command)s that have been
 executed after the initial ACL2 @(see world) to create the @(see world) in
 which @('certify-book') was called.  One way to obtain this number is by doing
 @(':pbt :start') to see all the @(see command)s back to the first one.</p>

 <p>If @('k') is @('t') (or any symbol whose @(tsee symbol-name) is
 @('\"T\"')), it means that @('certify-book') should use the same @(see world)
 used in the last certification of this book.  @('K') may have such a value
 only if you call @('certify-book') in the initial ACL2 @(see world) and there
 is a @(see certificate) on file for the book being certified.  (Of course, the
 @(see certificate) is probably invalid.)  In this case, @('certify-book')
 reads the old @(see certificate) to obtain the @(see portcullis) @(see
 command)s and executes them to recreate the certification @(see world).</p>

 <p>Finally, @('k') may be @('?') (or any symbol whose @(tsee symbol-name) is
 @('\"?\"')), in which case there is no check made on the certification world.
 That is, if @('k') is such a value then no action related to the preceding two
 paragraphs is performed, which can be a nice convenience but at the cost of
 eliminating a potentially valuable check that the certification @(see world)
 may be as expected.</p>

 <p>We next describe the meaning of @('compile-flg') and how it defaults.  If
 explicit compilation has been suppressed by @('(set-compiler-enabled nil
 state)'), then @('compile-flg') is coerced to @('nil'); see @(see
 compilation).  Otherwise @('compile-flg') may be given the value of @('t') (or
 @(':all'), which is equivalent to @('t') except during provisional
 certification; see @(see provisional-certification)), indicating that the book
 is to be compiled, or else @('nil').  (Note that compilation initially creates
 a compiled file with a temporary file name, and then moves that temporary file
 to the final compiled file name obtained by adding a suitable extension to the
 book name.  Thus, a compiled file will appear atomically in its intended
 location.)  Finally, suppose that @('compile-flg') is not supplied (or is
 @(':default')).  If environment variable @('ACL2_COMPILE_FLG') is defined and
 not the empty string, then its value should be @('T'), @('NIL'), or @('ALL')
 after converting to upper case, in which case @('compile-flg') is considered
 to have value @('t'), @('nil'), or @(':all') (respectively).  Otherwise
 @('compile-flg') defaults to @('t').  Note that the value @(':all') is
 equivalent to @('t') except for during the Convert procedure of provisional
 certification; see @(see provisional-certification).</p>

 <p>Two keyword arguments, @(':defaxioms-okp') and @(':skip-proofs-okp'),
 determine how the system handles the inclusion of @(tsee defaxiom) events and
 @(tsee skip-proofs) events, respectively, in the book.  The value @('t')
 allows such events, but prints a warning message.  The value @('nil') causes
 an error if such an event is found.  @('Nil') is the default unless keyword
 argument @(':acl2x t') is provided and state global @(''write-acl2x') is a
 cons (see @(see set-write-acl2x)), in which case the default is @('t').</p>

 <p>The keyword argument @(':ttags') may normally be omitted.  A few
 constructs, used for example if you are building your own system based on
 ACL2, may require it.  See @(see defttag) for an explanation of this
 argument.</p>

 <p>When book @('B') is certified with value @('t') (the default, unless the
 value used for @('pcert') is non-@('nil')) for keyword argument
 @(':write-port'), a file @('B.port') is written by certification process.
 This file contains all of the @(see portcullis) @(see command)s for @('B'),
 i.e., all user commands present in the ACL2 logical @(see world) at the time
 @('certify-book') is called.  If @('B.lisp') later becomes uncertified, say
 because @(see events) from that file or an included book have been edited,
 then @('(include-book \"B\")') will consult @('B.port') to evaluate forms in
 that file before evaluating the events in @('B.lisp').  On the other hand,
 @('B.port') is ignored when including @('B') if @('B') is certified.</p>

 <p>If you use @(see guard)s, please note @('certify-book') is executed as
 though @('(set-guard-checking t)') has been evaluated; see @(see
 set-guard-checking).  If you want to run with different guard-checking,
 consider using @('ld') instead, or in addition; see @(see ld).</p>

 <p>For a general discussion of books, see @(see books).  @('Certify-book') is
 akin to what we have historically called a ``proveall'': all the forms in the
 book are ``proved'' to guarantee their admissibility.  More precisely,
 @('certify-book') (1) reads the forms in the book, confirming that the
 appropriate packages are defined in the certification @(see world); (2) does
 the full admissibility checks on each form (proving termination of recursive
 functions, proving theorems, etc.), checking as it goes that each form is an
 embedded event form (see @(see embedded-event-form)); (3) may roll back the
 @(see world) (how far? &mdash; see below) and perform an @(tsee include-book)
 to check for @(tsee local) incompatibilities (see @(see
 local-incompatibility)); (4) writes a @(see certificate) recording not only
 that the book was certified but also recording the @(see command)s necessary
 to recreate the certification @(see world) (so the appropriate packages can be
 defined when the book is included in other @(see world)s) and a @(see
 book-hash) for each of the @(see books) involved (see @(see certificate)); and
 (5) compiles the book if so directed (and then loads the object file in that
 case).</p>

 <p>@('Certify-book') is a macro that returns an @(see error-triple), where
 success is indicated by an error component of @('nil') and has the effect of
 extending the @(see world) with a corresponding @(tsee include-book) event.
 If you don't want the included book's @(see events) in your present @(see
 world), simply execute @(':')@(tsee u).</p>

 <p>Technical Remark.  Step 3 above mentions rolling the logical @(see world)
 back to check for local incompatibilities.  For efficiency, this retraction to
 an initial segment of the the world is skipped if a local event is not
 encountered as described below; otherwise, the world is rolled back to the
 point just before the first such event was admitted.  For this purpose, a
 relevant local event is one at the top level of the book or under a sequences
 of @(tsee progn) calls in the book.  Here, we include forms generated by
 @(tsee make-event).  But this category does not include local @(see events)
 within an @(tsee encapsulate) event, since those are ignored in the final
 processing of the @('encapsulate') event.  We also do not consider local
 events within an @(tsee include-book) event.  End of Technical Remark.</p>

 <p>A utility is provided to assist in debugging failures of @('certify-book');
 see @(see redo-flat).)</p>

 <p>@('Certify-book') requires that the default @(see defun-mode) (see @(see
 default-defun-mode)) be @(':')@(tsee logic) when certification is attempted.
 If the mode is not @(':')@(tsee logic), an error is signaled.</p>

 <p>An error will occur if @('certify-book') has to deal with any uncertified
 book other than the one on which it was called.  For example, if the book
 being certified includes another book, that sub-book must already have been
 certified; that is, that sub-book must have a valid @(see certificate)
 file.</p>

 <p>If you have a certified book that has remained unchanged for some time you
 might well not remember the appropriate @(tsee defpkg)s for it, though they
 are stored in the @(see certificate) file and (by default) also in the
 @('.port') file.  If you begin to change the book, don't throw away its @(see
 certificate) file just because it has become invalid!  It is an important
 historical document until the book is re-certified.  More important, don't
 throw away the @('.port') file, as it will provide the @(see portcullis)
 commands when including the book as an uncertified book; see @(see
 include-book).</p>

 <p>When @('certify-book') is directed to produce a compiled file, it calls the
 Common Lisp function @('compile-file') on the original source file.  This
 creates a compiled file with an extension known to ACL2, e.g., if the book is
 named @('\"my-book\"') then the source file is @('\"my-book.lisp\"') and the
 compiled file under GCL will be @('\"my-book.o\"') while under SBCL it will be
 @('\"my-book.fasl\"').  The compiled file is then loaded.  When @(tsee
 include-book) is used later on @('\"my-book\"') it will automatically load the
 compiled file, provided the compiled file has a later write date than the
 source file.  The only effect of such @(see compilation) and loading is that
 the functions defined in the book execute faster.  See @(see guard) for a
 discussion of the issues, and if you want more details about @(see books) and
 compilation, see @(see book-compiled-file).</p>

 <p>When @('certify-book') is directed not to produce a compiled file, it will
 delete any existing compiled file for the book, so as not to mislead @(tsee
 include-book) into loading the now outdated compiled file.  Otherwise,
 @('certify-book') will create a temporary ``expansion file'' to compile,
 obtained by appending the string \"@@expansion.lsp\" to the end of the book
 name.  Remark: Users may ignore that file, which is automatically deleted
 unless @(see state) global variable @(''save-expansion-file') has been set,
 presumably by a system developer, to a non-@('nil') value; see @(see
 book-compiled-file) for more information about hit issue, including the role
 of environment variable @('ACL2_SAVE_EXPANSION').</p>

 <p>After execution of a @('certify-book') form, the value of @(tsee
 acl2-defaults-table) is restored to what it was immediately before that
 @('certify-book') form was executed.  See @(see acl2-defaults-table).</p>

 <p>Those who use the relatively advanced features of trust tags (see @(see
 defttag)) and @(tsee make-event) may wish to know how to create a @(see
 certificate) file that avoids dependence on trust tags that are used only
 during @(tsee make-event) expansion.  For this, including documentation of the
 @(':acl2x') and @(':ttagsx') keyword arguments for @('certify-book'), see
 @(see set-write-acl2x).</p>

 <p>This completes the tour through the @(see documentation) of @(see
 books).</p>")

(defxdoc certify-book-debug
  :parents (certify-book books-reference)
  :short "Some possible ways to work around @(tsee certify-book) failures"
  :long "<p>This topic provides some ideas for how to deal with @(tsee
 certify-book) failures.  Ideally, this topic will continue to grow over
 time.</p>

 <p>Stack overflows may show up as follows.</p>

 @({
 ***********************************************
 ************ ABORTING from raw Lisp ***********
 ********** (see :DOC raw-lisp-error) **********
 Error:  Stack overflow on value stack.
 ***********************************************
 })

 <p>When this occurs during book certification, it could be during an attempt
 to handle large objects, in particular by the @(see serialize) writer or by a
 @(see memoize)d version of the check for ``bad'' objects.  These two potential
 causes can be remedied by first evaluating the following forms,
 respectively.</p>

 @({
 (set-serialize-character-system nil state)
 (set-bad-lisp-consp-memoize nil)
 })

 <p>If the large object is in an event in the book under certification, then
 you may need to avoid printing it, as follows.</p>

 @({
 (set-inhibit-output-lst '(proof-tree event))
 })

 <p>Other failures of @('certify-book') may have error messages that point to
 the problem.  When that is not the case, it would be good to explain possible
 workarounds in this topic!</p>")

(defxdoc certify-book!
  :parents (certify-book)
  :short "A variant of @(tsee certify-book)"
  :long "@({
  Examples:
  (certify-book! \"my-arith\" 3)     ;Certify in a world with 3
                                     ; commands, starting in a world
                                     ; with at least 3 commands.
  (certify-book! \"my-arith\")       ;Certify in the initial world.

  General Form:
  (certify-book! book-name k compile-flg)
 })

 <p>where @('book-name') is a book name (see @(see book-name)), @('k') is a
 nonnegative integer used to indicate the ``certification @(see world),'' and
 @('compile-flg') indicates whether you wish to compile the (functions in the)
 book.</p>

 <p>This @(see command) is identical to @(tsee certify-book), except that the
 second argument @('k') may not be @('t') in @('certify-book!') and if @('k')
 exceeds the current @(see command) number, then an appropriate @(tsee ubt!)
 will be executed first.  See @(see certify-book) and see @(see ubt!).</p>")

(defxdoc change
  :parents (defrec acl2-built-ins)
  :short "Mutator macro for @(see defrec) structures."

  :long "<p>The @('change') macro is built into ACL2, and allows you to
 \"modify\" instances of structures that have been introduced with @(see
 defrec).  Of course, since ACL2 is applicative, the original structure is not
 actually changed&mdash;instead, a new structure is constructed, copying some
 fields from the original structure and installing new values for other
 fields.</p>

 <p>For instance, suppose we first use @(see make) to create an employee
 structure, e.g.,:</p>

 @({
    (defconst *jimmy* (make employee :name \"Jimmy\"
                                     :salary 0
                                     :position \"Unpaid Intern\")
 })

 <p>Then we can use @('change') to mutate this structure, e.g.,:</p>

 @({
     (change employee *jimmy* :salary 300000
                              :position \"Vice President\")
 })

 <p>Produces a new @('employee') structure where the @('name') is still
 \"Jimmy\", but where the @('salary') and @('position') have been updated to
 300000 and \"Vice President\", respectively.</p>

 <p>See @(see defrec) for more information.</p>")

(defxdoc char
  :parents (strings characters acl2-built-ins)
  :short "The @(see nth) element (zero-based) of a string"
  :long "<p>@('(Char s n)') is the @('n')th element of @('s'), zero-based.  If
 @('n') is greater than or equal to the length of @('s'), then @('char')
 returns @('nil').</p>

 <p>@('(Char s n)') has a @(see guard) that @('n') is a non-negative integer
 and @('s') is a @(tsee stringp).</p>

 <p>@('Char') is a Common Lisp function.  See any Common Lisp documentation for
 more information.</p>

 @(def char)")

(defxdoc char-code
  :parents (characters numbers acl2-built-ins)
  :short "The numeric code for a given character"
  :long "<p>This function maps a character to its code, for example:
  @('(char-code #\\A)') evaluates to @('65').  See @(tsee code-char) for a sort
  of inverse function, which maps a code to the corresponding character.</p>

 <p>Completion Axiom (@('completion-of-char-code')):</p>

 @({
  (equal (char-code x)
         (if (characterp x)
             (char-code x)
           0))
 })

 <p>@(see Guard) for @('(char-code x)'):</p>

 @({
  (characterp x)
 })

 <p>This function maps all non-characters to @('0').</p>")

(defxdoc char-downcase
  :parents (characters acl2-built-ins)
  :short "Turn upper-case @(see characters) into lower-case @(see characters)"
  :long "<p>@('(Char-downcase x)') is equal to @('#\\a') when @('x') is
 @('#\\A'), @('#\\b') when @('x') is @('#\\B'), ..., and @('#\\z') when @('x')
 is @('#\\Z'), and is @('x') for any other character.</p>

 <p>The @(see guard) for @('char-downcase') requires its argument to be a
 standard character (see @(see standard-char-p)).</p>

 <p>@('Char-downcase') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def char-downcase)")

(defxdoc char-equal
  :parents (characters acl2-built-ins)
  :short "Character equality without regard to case"
  :long "<p>For @(see characters) @('x') and @('y'), @('(char-equal x y)') is
 true if and only if @('x') and @('y') are the same except perhaps for their
 case.</p>

 <p>The @(see guard) on @('char-equal') requires that its arguments are both
 standard @(see characters) (see @(see standard-char-p)).</p>

 <p>@('Char-equal') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def char-equal)")

(defxdoc char-upcase
  :parents (characters acl2-built-ins)
  :short "Turn lower-case @(see characters) into upper-case @(see characters)"
  :long "<p>@('(Char-upcase x)') is equal to @('#\\A') when @('x') is
 @('#\\a'), @('#\\B') when @('x') is @('#\\b'), ..., and @('#\\Z') when @('x')
 is @('#\\z'), and is @('x') for any other character.</p>

 <p>The @(see guard) for @('char-upcase') requires its argument to be a
 standard character (see @(see standard-char-p)).</p>

 <p>@('Char-upcase') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def char-upcase)")

(defxdoc char<
  :parents (characters acl2-built-ins)
  :short "Less-than test for @(see characters)"
  :long "<p>@('(char< x y)') is true if and only if the character code of
 @('x') is less than that of @('y').  See @(see char-code).</p>

 <p>The @(see guard) for @('char<') specifies that its arguments are @(see
 characters).</p>

 <p>@('Char<') is a Common Lisp function.  See any Common Lisp documentation
 for more information.</p>

 @(def char<)")

(defxdoc char<=
  :parents (characters acl2-built-ins)
  :short "Less-than-or-equal test for @(see characters)"
  :long "<p>@('(char<= x y)') is true if and only if the character code of
 @('x') is less than or equal to that of @('y').  See @(see char-code).</p>

 <p>The @(see guard) for @('char<=') specifies that its arguments are @(see
 characters).</p>

 <p>@('Char<=') is a Common Lisp function.  See any Common Lisp documentation
 for more information.</p>

 @(def char<=)")

(defxdoc char>
  :parents (characters acl2-built-ins)
  :short "Greater-than test for @(see characters)"
  :long "<p>@('(char> x y)') is true if and only if the character code of
 @('x') is greater than that of @('y').  See @(see char-code).</p>

 <p>The @(see guard) for @('char>') specifies that its arguments are @(see
 characters).</p>

 <p>@('Char>') is a Common Lisp function.  See any Common Lisp documentation
 for more information.</p>

 @(def char>)")

(defxdoc char>=
  :parents (characters acl2-built-ins)
  :short "Greater-than-or-equal test for @(see characters)"
  :long "<p>@('(char>= x y)') is true if and only if the character code of
 @('x') is greater than or equal to that of @('y').  See @(see char-code).</p>

 <p>The @(see guard) for @('char>=') specifies that its arguments are @(see
 characters).</p>

 <p>@('Char>=') is a Common Lisp function.  See any Common Lisp documentation
 for more information.</p>

 @(def char>=)")

(defxdoc character-alistp
  :parents (characters alists acl2-built-ins)
  :short "Recognizer for association lists with characters as keys"
  :long "<p>@('(Character-alistp x)') is true if and only if @('x') is a list
 of pairs of the form @('(cons key val)') where @('key') is a @(tsee
 characterp).</p>

 @(def character-alistp)")

(defxdoc character-encoding

; The comment near the top of "PRINTING and READING" in ACL2 source file
; axioms.lisp pertains to Emacs variable save-buffer-coding-system, which is
; related to character encoding.

  :parents (io)
  :short "How bytes are parsed into characters"
  :long "<p>When the Common Lisp reader comes across bytes in a file or at the
 terminal, they are parsed into characters.  The simplest case is when each
 byte that is read is a standard character (see @(see standard-char-p)).  It is
 actually quite common that each byte that is read corresponds to a single
 character.  The parsing of bytes into characters is based on a <i>character
 encoding</i>, that is, a mapping that associates one or more bytes with each
 legal character.</p>

 <p>In order to help guarantee the portability of files (including @(see
 books)), ACL2 installs a common character encoding for reading files, often
 known as ISO-8859-1 or Latin-1.  For some host Lisps this character encoding
 is also used for reading from the terminal; but, sadly, this may not hold for
 all host Lisps, and may not even be possible for some of them.</p>

 <p>The use of the above encoding could in principle cause problems if one's
 editor produces files using an encoding other than ISO-8859-1, at least if one
 uses non-standard characters.  In particular, the default Emacs buffer
 encoding may be utf-8.  If your file has non-standard characters, then in
 Emacs you can evaluate the form</p>

 @({
  (setq save-buffer-coding-system 'iso-8859-1)
 })

 <p>before saving the buffer into a file.  This will happen automatically for
 users who load distributed file @('emacs/emacs-acl2.el') into their Emacs
 sessions.</p>

 <p>For an example of character encodings in action, see the community book
 @('books/misc/character-encoding-test.lisp').</p>")

(defxdoc character-listp
  :parents (characters lists acl2-built-ins)
  :short "Recognizer for a true list of characters"
  :long "<p>The predicate @('character-listp') tests whether its argument is a
 true list of @(see characters).</p>

 @(def character-listp)")

(defxdoc characterp
  :parents (characters acl2-built-ins)
  :short "Recognizer for @(see characters)"
  :long "<p>@('(characterp x)') is true if and only if @('x') is a
  character.</p>")

(defxdoc characters
  :parents (programming)
  :short "Characters in ACL2 and operations on them"
  :long "<p>ACL2 accepts 256 distinct characters, which are the characters
 obtained by applying the function @(tsee code-char) to each integer from
 @('0') to @('255').  Among these, Common Lisp designates certain ones as
 <i>standard characters</i>, namely those of the form @('(code-char n)') where
 @('n') is from @('33') to @('126'), together with @('#\\Newline') and
 @('#\\Space').  The actual standard characters may be viewed by evaluating the
 @(tsee defconst) @('*standard-chars*').</p>

 <p>To be more precise, Common Lisp does not specify the precise relationship
 between @(tsee code-char) and the standard characters.  However, we check that
 the underlying Common Lisp implementation uses a particular relationship that
 extends the usual ASCII coding of characters.  We also check that Space, Tab,
 Newline, Page, Rubout, and Return correspond to characters with respective
 @(tsee char-code)s @('32'), @('9'), @('10'), @('12'), @('127'), and
 @('13').</p>

 <p>@(tsee Code-char) has an inverse, @(tsee char-code).  Thus, when @(tsee
 char-code) is applied to an ACL2 character, @('c'), it returns a number @('n')
 between @('0') and @('255') inclusive such that @('(code-char n)') =
 @('c').</p>

 <p>The preceding paragraph implies that there is only one ACL2 character with
 a given character code.  CLTL allows for ``attributes'' for characters, which
 could allow distinct characters with the same code, but ACL2 does not allow
 this.</p>

 <p><i>The Character Reader</i></p>

 <p>ACL2 supports the `@('#\\')' notation for characters provided by Common
 Lisp, with some restrictions.  First of all, for every character @('c'), the
 notation</p>

 @({
  #\\c
 })

 <p>may be used to denote the character object @('c').  That is, the user may
 type in this notation and ACL2 will read it as denoting the character object
 @('c').  In this case, the character immediately following @('c') must be one
 of the following ``terminating characters'': a Tab, a Newline, a Page
 character, a space, or one of the characters:</p>

 @({
  \"  '  (  )  ;  `  ,
 })

 <p>Other than the notation above, ACL2 accepts alternate notation for five
 characters.</p>

 @({
  #\\Space
  #\\Tab
  #\\Newline
  #\\Page
  #\\Rubout
  #\\Return
 })

 <p>Again, in each of these cases the next character must be from among the set
 of ``terminating characters'' described in the single-character case.  Our
 implementation is consistent with ISO-8859-1, even though we don't provide
 @('#\\') syntax for entering characters other than that described above.</p>

 <p>Finally, we note that it is our intention that any object printed by ACL2's
 top-level-loop may be read back into ACL2.  Please notify the implementors if
 you find a counterexample to this claim.</p>")

(defxdoc check-vars-not-free
  :parents (macros)
  :short "Avoid variable capture in macro calls"
  :long "<p>@('Check-vars-not-free') is a macro that is useful for avoiding
 capture of variables in macro calls.  We explain using a running example,
 after which we give a precise explanation (at the end of this documentation
 topic).  These definitions are rather contrived but are designed to illustrate
 how @('check-vars-not-free') can be useful.</p>

 @({
 (defmacro has-length (x n)
   `(let ((k (len ,x)))
      (equal k ,n)))
 (defun f1 (u v)
   (and (natp v)
        (has-length u (* 2 v))))
 (defun f2 (lst k)
   (and (natp k)
        (has-length lst (* 2 k))))
 })

 <p>One might expect @('f1') and @('f2') to be the same function, since the
 only change seems to be from the choices of formal parameters.  However,
 consider these evaluations.</p>

 @({
 ACL2 !>(f1 '(a b c d e f) 3)
 T
 ACL2 !>(f2 '(a b c d e f) 3)
 NIL
 ACL2 !>
 })

 <p>The problem becomes clear if we expand the macro call in the body of
 @('f2').</p>

 @({
 ACL2 !>:trans1 (has-length lst (* 2 k))
  (LET ((K (LEN LST))) (EQUAL K (* 2 K)))
 ACL2 !>
 })

 <p>We see that when expanding the call of the macro @('has-length') in the
 body of @('f2'), the substitution of @('(* 2 k)') for @(',n') causes @('k') to
 be``captured'' by the binding of @('k') to @('(len lst)').</p>

 <p>The macro @('check-vars-not-free') is designed to enforce that no such
 capture takes place.  An improved definition of @('has-length') is as
 follows, since it insists that the variable @('k') must not appear in the term
 that replaces @(',n').</p>

 @({
 (defmacro has-length (x n)
   `(let ((k (len ,x)))
      (equal k (check-vars-not-free (k) ,n))))
 })

 <p>Now, the attempt to define @('f2') as above causes an error.</p>

 @({
 ACL2 !>(defun f2 (lst k)
    (and (natp k)
         (has-length lst (* 2 k))))


 ACL2 Error in ( DEFUN F2 ...):  CHECK-VARS-NOT-FREE failed:
 It is forbidden to use K in (BINARY-* '2 K).  Note:  this error occurred
 in the context (CHECK-VARS-NOT-FREE (K) (* 2 K)).


 Summary
 Form:  ( DEFUN F2 ...)
 Rules: NIL
 Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)

 ACL2 Error in ( DEFUN F2 ...):  See :DOC failure.

 ******** FAILED ********
 ACL2 !>
 })

 <p>The error message shows that the check performed by
 @('check-vars-not-free') has failed, because the variable @('k') occurs in the
 expression @('(* 2 k)').  More precisely, what is checked is that @('k') does
 not occur in the translation to a @(see term) of the expression @('(* 2 k)'),
 namely, @('(binary-* '2 k)').</p>

 <p>The general form of a call of @('check-vars-not-free') is</p>

 @({
 (check-vars-not-free (var_1 ... var_n) expr)
 })

 <p>where @('var_1'), ..., @('var_n') are variables and @('expr') is an
 expression.  This call is equivalent to @('expr') &mdash; indeed, there is no
 effect on the code generated when using this call in place of @('expr')
 &mdash; but ACL2 requires that the indicated variables do not occur free in
 the translation of @('expr') to a term.</p>")

(defxdoc checkpoint-summary-limit
  :parents (summary set-gag-mode)
  :short "Control printing of key checkpoints upon a proof's failure"
  :long "<p>See @(see set-checkpoint-summary-limit) for a discussion of the
 checkpoint-summary-limit.  Evaluation of the form
 @('(checkpoint-summary-limit)') produces the current checkpoint-summary-limit,
 and can be invoked with the keyword hack (see @(see keyword-commands)).  For
 example:</p>

 @({
 ACL2 !>:checkpoint-summary-limit
 (NIL . 3)
 ACL2 !>
 })")

(defxdoc checksum
  :parents (certificate)
  :short "Assigning ``often unique'' integers to files and objects"
  :long "<p>See @(see book-hash) for a discussion of how ACL2 can use checksums
 (though not by default) to increase security in the @(see books)
 mechanism.</p>

 <p>A <i>checksum</i> is an integer in some fixed range computed from the
 printed representation of an object, e.g., the sum, modulo @('2**32'), of the
 ascii codes of all the @(see characters) in the printed representation.</p>

 <p>Ideally, you would like the checksum of an object to be uniquely associated
 with that object, like a fingerprint.  It could then be used as a convenient
 way to recognize the object in the future: you could remember the
 checksum (which is relatively small) and when an object is presented to you
 and alleged to be the special one you could compute its checksum and see if
 indeed it was.  Alas, there are many more objects than checksums (after all,
 each checksum is an object, and then there's @('t')).  So you try to design a
 checksum algorithm that maps similar looking objects far apart, in the hopes
 that corruptions and counterfeits &mdash; which appear to be similar to the
 object &mdash; have different checksums.  Nevertheless, the best you can do is
 a many-to-one map.  If an object with a different checksum is presented, you
 can be positive it is not the special object.  But if an object with the same
 checksum is presented, you have no grounds for positive identification.</p>

 <p>The basic checksum algorithm in ACL2 is called @('check-sum-obj'), which
 computes the checksum of an ACL2 object.  Roughly speaking, we scan the print
 representation of the object and, for each character encountered, we multiply
 the ascii code of the character times its position in the stream (modulo a
 certain prime) and then add (modulo a certain prime) that into the running
 sum.  This is inaccurate in many senses (for example, we don't always use the
 ascii code and we see numbers as though they were printed in base 127) but
 indicates the basic idea.</p>")

(defxdoc checkpoint-forced-goals
  :parents (proof-tree)
  :short "Cause forcing goals to be checkpointed in proof trees"
  :long "@({
  Example forms:
  (checkpoint-forced-goals t)
  (checkpoint-forced-goals nil)
 })

 <p>Also see @(see proof-tree).</p>

 <p>By default, goals are not marked as checkpoints by a proof tree display (as
 described elsewhere; see @(see proof-tree)) merely because they @(see force)
 some hypotheses, thus possibly contributing to a forcing round.  However, some
 users may want such behavior, which will occur once the command
 @('(checkpoint-forced-goals') @('t')) has been executed.  To return to the
 default behavior, use the command @('(checkpoint-forced-goals nil)').</p>")

(defxdoc clause-identifier
  :parents (goal-spec)
  :short "The internal form of a @(see goal-spec)"
  :long "<p>To each goal-spec, @('str'), there corresponds a clause-identifier
 produced by @('(parse-clause-id str)').  For example,</p>

 @({
  (parse-clause-id \"[2]Subgoal *4.5.6/7.8.9'''\")
 })

 <p>returns @('((2 4 5 6) (7 8 9) . 3)').</p>

 <p>The function @('string-for-tilde-@-clause-id-phrase') inverts
 @('parse-clause-id') in the sense that given a clause identifier it returns
 the corresponding goal-spec.</p>

 <p>As noted in the documentation for @(see goal-spec), each clause printed in
 the theorem prover's proof attempt is identified by a name.  When these names
 are represented as strings they are called ``goal specs.''  Such strings are
 used to specify where in the proof attempt a given hint is to be applied.  The
 function @('parse-clause-id') converts goal-specs into clause identifiers,
 which are cons-trees containing natural numbers.</p>

 <p>Examples of goal-specs and their corresponding clause identifiers are shown
 below.</p>

 @({
               parse-clause-id
                     -->

  \"Goal\"                       ((0) NIL . 0)
  \"Subgoal 3.2.1'\"             ((0) (3 2 1) . 1)
  \"[2]Subgoal *4.5.6/7.8.9'''\" ((2 4 5 6) (7 8 9) . 3)

                     <--
        string-for-tilde-@-clause-id-phrase
 })

 <p>The caar of a clause id specifies the forcing round, the cdar specifies the
 goal being proved by induction, the cadr specifies the particular subgoal, and
 the cddr is the number of primes in that subgoal.</p>

 <p>Internally, the system maintains clause ids, not goal-specs.  The system
 prints clause ids in the form shown by goal-specs.  When a goal-spec is used
 in a hint, it is parsed (before the proof attempt begins) into a clause id.
 During the proof attempt, the system watches for the clause id and uses the
 corresponding hint when the id arises.  (Because of the expense of creating
 and garbage collecting a lot of strings, this design is more efficient than
 the alternative.)</p>")

(defxdoc clause-processor
  :parents (rule-classes)
  :short "Make or apply a @(':clause-processor') rule (goal-level simplifier)"
  :long "<p>See @(see rule-classes) for a general discussion of rule classes,
 including how they are used to build rules from formulas and a discussion of
 the various keywords in a rule class description.</p>

 <p>We will introduce clause-processor rules by way of the following example.
 But note that the clause-processor utility is more general than this example
 may suggest; for example, the second argument of @('evl0') in the hypothesis
 need not be the same as its second argument in the conclusion.</p>

 @({
  ; Example (which we'll return to, below):
  (defthm correctness-of-note-fact-clause-processor
    (implies (and (pseudo-term-listp cl)
                  (alistp a)
                  (evl0 (conjoin-clauses
                         (note-fact-clause-processor cl term))
                        a))
             (evl0 (disjoin cl) a))
    :rule-classes :clause-processor)
 })

 <p>We begin this documentation with an introduction, focusing on the example
 above, and then conclude with a detailed general discussion of
 clause-processor rules.  You might find it most useful simply to look at the
 examples in community books directory @('books/clause-processors/'); see file
 @('Readme.lsp') in that directory.</p>

 <p>Also see @(see define-trusted-clause-processor) for documentation of an
 analogous utility that does not require the clause-processor to be proved
 correct.  But please read the present documentation before reading about that
 utility.  Both utilities designate functions as ``clause-processors''.  Such
 functions must be executable &mdash; hence not constrained by virtue of being
 introduced in the @(see signature) of an @(tsee encapsulate) &mdash; and must
 respect @(see stobj) and output arity restrictions.  For example, something
 like @('(car (mv ...))') is illegal; also see @(see signature).</p>

 <h4>INTRODUCTION</h4>

 <p>A @(':clause-processor') rule installs a simplifier at the level of goals,
 where a goal is represented as a <i>clause</i>: a list of @(see term)s that is
 implicitly viewed as a disjunction (the application of @(tsee OR)).  For
 example, if ACL2 prints a goal in the form @('(implies (and p q) r)'), then
 the clause might be the one-element list containing the internal
 representation of this term &mdash; @('(implies (if p q 'nil) r)') &mdash; but
 more likely, the corresponding clause is @('((not p) (not q) r)').  Note that
 the members of a clause are <i>translated</i> terms; see @(see term) and
 @(tsee termp).  For example, they do not contain calls of the macro @('AND'),
 and constants are quoted.  The result of running a clause-processor must be a
 list of legal clauses; see @(see meta) for a discussion of translated terms,
 and for related discussion about ``forbidden'' function symbols, @(see
 set-skip-meta-termp-checks).</p>

 <p>The recognizer for a clause is the function @(tsee term-listp) and the
 recognizer for a list of clauses is @(tsee term-list-listp).</p>

 <p>Note that clause-processor simplifiers are similar to metafunctions, and
 similar efficiency considerations apply.  See @(see meta), in particular the
 discussion on how to ``make a metafunction maximally efficient.''</p>

 <p>Unlike rules of class @(':')@(tsee meta), rules of class
 @(':clause-processor') must be applied by explicit @(':clause-processor')
 @(see hints); they are not applied automatically (unless by way of computed
 hints; see @(see computed-hints)).  But @(':clause-processor') rules can be
 useful in situations for which it is more convenient to code a simplifier that
 manipulates the entire goal clause rather than individual subterms of terms in
 the clause.</p>

 <p>We begin with a simple illustrative example: a clause-processor that
 assumes an alleged fact (named @('term') in the example) and creates a
 separate goal to prove that fact.  We can extend the hypotheses of the current
 goal (named @('cl') in the example) with a term by adding the negation of that
 term to the clause (disjunctive) representation of that goal.  So the
 following returns a list of two clauses: the result of adding @('term') as a
 hypothesis to the input clause, as just described, and a second clause
 consisting only of that term.  This list of two clauses can be viewed as the
 conjunction of the first clause and the second clause (where again, each
 clause is viewed as a disjunction).</p>

 @({
  (defun note-fact-clause-processor (cl term)
    (declare (xargs :guard t)) ; optional, for better efficiency
    (list (cons (list 'not term)
                cl)
          (list term)))
 })

 <p>As with @(':')@(tsee meta) rules, we need to introduce a suitable
 evaluator; see @(see defevaluator) if you want details.  Since we expect to
 reason about the function @(tsee NOT), because of its role in
 @('note-fact-clause-processor') as defined above, we include @('NOT') in the
 set of functions known to this evaluator.  We also include @('IF'), as is
 often a good idea.</p>

 @({
  (defevaluator evl0 evl0-list
    ((not x) (if x y z)))
 })

 <p>ACL2 can now prove the following theorem automatically.  (This is the
 example displayed at the outset of this @(see documentation) topic.)  Of
 course, @(':clause-processor') rules about clause-processor functions less
 trivial than @('note-fact-clause-processor') may require lemmas to be proved
 first!  The function @('disjoin') takes a clause and returns its disjunction
 (the result of applying @(tsee OR) to its members), and @('conjoin-clauses')
 applies @('disjoin') to every element of a given list of clauses and then
 conjoins (applies @('AND')) to the corresponding list of resulting terms.</p>

 @({
  (defthm correctness-of-note-fact-clause-processor
    (implies (and (pseudo-term-listp cl)
                  (alistp a)
                  (evl0 (conjoin-clauses
                         (note-fact-clause-processor cl term))
                        a))
             (evl0 (disjoin cl) a))
    :rule-classes :clause-processor)
 })

 <p>Now let us submit a silly but illustrative example theorem to ACL2, to show
 how a corresponding @(':clause-processor') hint is applied.  The hint says to
 apply the clause-processor function, @('note-fact-clause-processor'), to the
 current goal clause and a ``user hint'' as the second argument of that
 function, in this case @('(equal a a)').  Thus, a specific variable,
 @('clause'), is always bound to the current goal clause for the evaluation of
 the @(':clause-processor') hint, to produce a list of clauses.  Since two
 subgoals are created below, we know that this list contained two clauses.
 Indeed, these are the clauses returned when @('note-fact-clause-processor') is
 applied to two arguments: the current clause, which is the one-element list
 @('((equal (car (cons x y)) x))'), and the user hint, @('(equal a a)').</p>

 @({
  ACL2 !>(thm (equal (car (cons x y))
                     x)
              :hints
              ((\"Goal\"
                :clause-processor
                (note-fact-clause-processor clause '(equal a a)))))

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  We now apply the verified :CLAUSE-PROCESSOR function NOTE-FACT-CLAUSE-
  PROCESSOR to produce two new subgoals.

  Subgoal 2
  (IMPLIES (EQUAL A A)
           (EQUAL (CAR (CONS X Y)) X)).

  But we reduce the conjecture to T, by the :executable-counterpart of
  IF and the simple :rewrite rule CAR-CONS.

  Subgoal 1
  (EQUAL A A).

  But we reduce the conjecture to T, by primitive type reasoning.

  Q.E.D.

  Summary
  Form:  ( THM ...)
  Rules: ((:EXECUTABLE-COUNTERPART IF)
          (:EXECUTABLE-COUNTERPART NOT)
          (:FAKE-RUNE-FOR-TYPE-SET NIL)
          (:REWRITE CAR-CONS))
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)

  Proof succeeded.
  ACL2 !>
 })

 <p>That concludes our introduction to clause-processor rules and hints.  We
 turn now to detailed documentation.</p>

 <h4>DETAILED DOCUMENTATION</h4>

 <p>The @(see signature) of a clause-processor function, @('CL-PROC'), must
 have one of the following forms.  Here, each @('st_i') is a @(see stobj)
 (possibly @(tsee state)) while the other parameters and results are not stobjs
 (see @(see stobj)).  Note that there need not be input stobjs in [3] &mdash;
 i.e., @('k') can be 0 &mdash; and even if there are, there need not be output
 stobjs.  In most ways [3] and [3+] are treated similarly; see @(see
 make-summary-data) for a discussion of the form of @('d') in [3+] and, more
 generally, for how [3+] differs from [3] by enhancing the @(see summary).</p>

 @({
  [1]  ((CL-PROC cl) => cl-list)

  [2]  ((CL-PROC cl hint) => cl-list)

  [3]  ((CL-PROC cl hint st_1 ... st_k) => (mv erp cl-list st_i1 ... st_in))

  [3+] ((CL-PROC cl hint st_1 ... st_k) => (mv erp cl-list st_i1 ... st_in d))
 })

 <p>We call @('cl-list') the <i>clauses-result</i>.  In [3] and [3+], we think
 of the first component of the result as an error flag.  Indeed, a proof will
 instantly abort if that error flag is not @('nil').</p>

 <p>We next discuss the legal forms of @(':clause-processor') rules, followed
 below by a discussion of @(':clause-processor') @(see hints).  In the
 discussion below, we use lower-case names to represent specific symbols, for
 example @('implies'), and also to represent @(see stobj) names; and we use
 upper-case names to represent arbitrary pieces of syntax (which we will
 describe), for example, @('CL').</p>

 <p>If a @(':')@(tsee rule-classes) specification includes
 @(':clause-processor'), then the corresponding term must have the following
 form.  (Additional ``meta-extract'' hypotheses, not shown or discussed below,
 may be included as desired in order to use facts from the logical @(tsee
 world) to help prove the rule; see @(see meta-extract) for explanation of this
 advanced feature.)</p>

 @({
  ; General Form (omitting possible meta-extract hypotheses)
  (implies (and (pseudo-term-listp CL)
                (alistp A)
                (EVL (conjoin-clauses <CL-LIST>)
                      B))
           (EVL (disjoin CL) A))
 })

 <p>Here @('EVL') is a known evaluator; @('CL') and @('A') are distinct
 non-stobj variables; and @('<CL-LIST>') is an expression representing the
 clauses returned by the clause-processor function @('CL-PROC'), whose form
 depends on the @(see signature) of that function, as follows.  Typically
 @('B') is @('A'), but it can be any term (useful when generalization is
 occurring; see the example ``Test generalizing alist'' in community book
 @('books/clause-processors/basic-examples.lisp')).  For cases [1] and [2]
 above, @('<CL-LIST>') is of the form @('(CL-PROC CL)') or @('(CL-PROC CL
 HINT)'), respectively, where in the latter case @('HINT') is a non-stobj
 variable distinct from the variables @('CL') and @('A').  For cases [3] and
 [3+], @('<CL-LIST>') is the result of wrapping the function
 @('clauses-result') around a call of @('CL-PROC'):</p>

 @({
  (clauses-result (CL-PROC CL HINT ...))
 })

 <p>Logically, @('clauses-result') returns the @(tsee cadr) if the @(tsee car)
 is @('NIL'), and otherwise (for the error case) returns a list containing the
 empty (false) clause.  So in the non-error case, @('clauses-result') picks out
 the second result, denoted @('cl-list') in [3] and [3+] above, and in the
 error case the implication above trivially holds.</p>

 <p>In the above theorem, we are asked to prove @('(EVL (disjoin CL) A)')
 assuming that the conjunction of all clauses produced by the clause processor
 evaluates to a non-@('nil') value under some alist @('B').  In fact, we can
 choose @('B') so as to allow us to assume evaluations of the generated clauses
 over many different alists.  This technique is discussed in the community book
 @('books/clause-processors/multi-env-trick.lisp'), which introduces some
 macros that may be helpful in accomplishing proofs of this type.</p>

 <p>The clause-processor function, @('CL'), must have a guard that ACL2 can
 trivially prove from the hypotheses that the first argument of @('CL') is
 known to be a @('pseudo-term-listp') and any @(see stobj) arguments are
 assumed to satisfy their stobj predicates.</p>

 <p>Next we specify the legal forms for @(':clause-processor') @(see hints).
 The basic form is @(':clause-processor TERM'), such that the translation of
 @('TERM') to a legal ACL2 term yields a call of a clause-processor function
 @('(CL-PROC clause)') or @('(CL-PROC clause HINT ...)'), where @('HINT') is a
 term whose only non-stobj free variable (if any) is the symbol, @('clause').
 Note that the first argument of the call is literally the variable,
 @('clause').  Recall that at the time the :clause-processor hint is applied to
 the clausal form @('CL') of the current subgoal, @('clause') is bound to
 @('CL'); moreover, any stobj names will be bound to the corresponding
 stobjs.</p>

 <p>But there are two additional general forms for @(':clause-processor')
 hints, which may viewed as abbreviations for the basic form discussed above.
 The first additional general form, which we call the ``@(':function') form'',
 includes the following two classes of expressions.</p>

 @({
 :clause-processor (:function F)
 :clause-processor (:function F :hint HINT)
 })

 <p>The first of these @(':function') forms abbreviates the following hint,
 where either @('F') is a macro-alias for the function symbol @('CL-PROC') (see
 @(see add-macro-alias)), and otherwise @('CL-PROC') is just the function
 symbol @('F').</p>

 @({
 :clause-processor (CL-PROC clause)
 })

 <p>Similarly the second of the @(':function') forms above abbreviates the
 following, where @('CL-PROC') is as above and the output signature of
 @('CL-PROC') is @('(nil nil st_1 ... st_k)').</p>

 @({
 :clause-processor (CL-PROC clause HINT st_1 ... st_k)
 })

 <p>Besides the ``@(':function') form'', there is one more additional general
 form for @(':clause-processor') hint: a symbol, as follows.</p>

 @({
 :clause-processor F
 })

 <p>This expands to a basic form as follows, depending on @('F').</p>

 <ul>

 <li>If @('F') is a macro with one required argument or a function symbol with
 one argument:<br/>
 @(':clause-processor (F clause)')</li>

 <li>If @('F') is a macro with two required arguments or a function symbol with
 two arguments:<br/>
 @(':clause-processor (F clause nil)')</li>

 <li>If @('F') is a function symbol with inputs of the form [3] or [3+] above,
 that is, with input signature @('(nil nil st_1 ... st_k)'):<br/>
 @(':clause-processor (F clause nil st_1 ... st_k)')</li>

 </ul>

 <p>For examples of these syntactic forms, see community book
 @('books/clause-processors/basic-examples.lisp')).  We turn now to discussion
 of when a @(':clause-processor') hint causes an abort.</p>

 <p>A @(':clause-processor') hint causes the proof to abort if the
 clauses-result is not a list of clauses, i.e., a list of (translated) @(see
 term) lists in the current logical @(see world).  This test is done explicitly
 every time a clause processor is run unless a @(':')@(tsee
 well-formedness-guarantee) has been provided with the @(':clause-processor')
 rule or @(tsee set-skip-meta-termp-checks) has been used with an active trust
 tag to skip the check at the user's risk.</p>

 <p>The proof also aborts when the clause-processor function returns at least
 two values and the first value returned &mdash; the ``@('erp')'' value from
 cases [3] and [3+] above &mdash; is not @('nil').  In that case, @('erp') is
 used for printing an error message as follows: if it is a string, then that
 string is printed; but if it is a non-empty true list whose first element is a
 string, then it is printed as though by @('(fmt ~@0 (list (cons #\\0 erp))
 ...)') (see @(see fmt)).  Otherwise, a non-@('nil') @('erp') value causes a
 generic error message to be printed.</p>

 <p>If there is no error as above, but the @('CL-PROC') call returns a clause
 list whose single element is equal to the input clause, then the hint is
 ignored since we are left with the goal with which we started.  In that case,
 the other prover processes are then applied as usual (see @(see
 hints-and-the-waterfall)).</p>

 <p>You can see all current @(':clause-processor') rules by issuing the
 following command: @('(print-clause-processor-rules)').</p>

 <p>The following paper discusses ACL2 clause-processors at a high level
 suitable for a non-ACL2 audience:</p>

 <blockquote>

 <p>M. Kaufmann, J S. Moore, S. Ray, and E. Reeber, ``Integrating External
 Deduction Tools with ACL2.''  <i>Journal of Applied Logic</i> (Special Issue:
 Empirically Successful Computerized Reasoning), Volume 7, Issue 1, March 2009,
 pp. 3&ndash;25.  Also published online (DOI @('10.1016/j.jal.2007.07.002')).
 Preliminary version in: Proceedings of the 6th International Workshop on the
 Implementation of Logics (IWIL 2006) (C. Benzmueller, B. Fischer, and
 G. Sutcliffe, editors), <a href='http://ceur-ws.org/Vol-212/'>CEUR Workshop
 Proceedings Vol. 212</a>, Phnom Penh, Cambodia, pp. 7&ndash;26, November
 2006.</p></blockquote>

 ")

(defxdoc clear-hash-tables
  :parents (memoize) ; skip programming as parent, since this is deprecated
  :short "Deprecated feature"
  :long "<p>Deprecated.  Calls @(tsee clear-memoize-tables) and then @(tsee
 hons-clear) or @(tsee hons-wash), whichever makes sense for the underlying
 Common Lisp.</p>")

(defxdoc clear-memoize-statistics
  :parents (memoize)
  :short "Clears all profiling info displayed by @('(')@(tsee
  memoize-summary)@(')')"
  :long "<p>Logically, this function just returns @('nil').  It clears all
 profiling info displayed by @('(')@(tsee memoize-summary)@(')')</p>

 @(def clear-memoize-statistics)")

(defxdoc clear-memoize-table
  :parents (memoize)
  :short "Forget values remembered for the given function"
  :long "<p>This function returns its argument, @('fn'), unchanged.  The values
 memoized for @('fn') are forgotten.</p>

 @(def clear-memoize-table)")

(defxdoc clear-memoize-tables
  :parents (memoize)
  :short "Forget values remembered for all the memoized functions"
  :long "<p>@('Clear-memoize-tables') is a logical no-op.  All memoized values
 are forgotten.  It returns @('nil'), invoking @(tsee clear-memoize-table) for
 each memoized function.</p>

 @(def clear-memoize-tables)")

(defxdoc close-trace-file
  :parents (trace)
  :short "Stop redirecting trace output to a file"
  :long "@({
  General Form:
  (close-trace-file) ; trace output is no longer redirected to a file
 })

 <p>Output from @(tsee trace$) normally goes to the screen, or more precisely,
 @(tsee standard-co).  It can be redirected to a file; see @(see
 open-trace-file).  Use @('close-trace-file') to redirect trace output to
 @(tsee standard-co).</p>")

(defxdoc code-char
  :parents (characters numbers acl2-built-ins)
  :short "The character corresponding to a given numeric code"
  :long "<p>This function maps a numeric code to the corresponding character,
  for example: @('(code-char 65)') evaluates to @('#\\A').  See @(tsee
  char-code) for a sort of inverse function, which maps a character to its
  code.</p>

 <p>Completion Axiom (@('completion-of-code-char')):</p>

 @({
  (equal (code-char x)
         (if (and (integerp x)
                  (>= x 0)
                  (< x 256))
             (code-char x)
           (code-char 0)))
 })

 <p>@(see Guard) for @('(code-char x)'):</p>

 @({
  (and (integerp x)
       (>= x 0)
       (< x 256))
 })

 <p>ACL2 supports 8-bit @(see characters).  Inputs not between @('0') and
 @('255') are treated as @('0').</p>")

(defxdoc coerce

; Jared Davis has written a faster version of coercing a character list to a
; string, which is displayed just below.  But we (the ACL2 authors) have
; decided not to try to meddle with the underlying Lisp implementation of
; coerce (though on 2/27/09 Bob Boyer temporarily added a patch from Gary Byers
; to hons-raw.lisp to speed this up for CCL).  Jared adds (6/30/09) that CCL
; now handles coerce efficiently, both to strings and to lists.

;  (defun my-coerce (chars)
;    (let* ((length (the integer (length (the list chars))))
;           (str    (the vector (make-string (the integer length))))
;           (i      (the integer 0)))
;      (loop for char in chars
;            do
;            (setf (aref (the vector str) (the integer i))
;                  (the character char))
;            (incf (the integer i)))
;      str))

  :parents (strings characters acl2-built-ins)
  :short "Coerce a character list to a string and a string to a list"
  :long "<p>Completion Axiom (@('completion-of-coerce')):</p>

 @({
  (equal (coerce x y)
         (cond
          ((equal y 'list)
           (if (stringp x)
               (coerce x 'list)
             nil))
          (t
           (coerce (make-character-list x) 'string))))
 })

 <p>@(see Guard) for @('(coerce x y)'):</p>

 @({
  (if (equal y 'list)
      (stringp x)
    (if (equal y 'string)
        (character-listp x)
      nil))
 })

 <p>Also see community book @('books/misc/fast-coerce.lisp'), contributed by
 Jared Davis, for a version of @('coerce') that may be faster for Common Lisp
 implementations other than CCL 1.3 or later, if the second argument is
 @(''list') (for coercing a string to a list).</p>

 <h4>Logical Note</h4>

 <p>The function @('coerce') can be viewed as the constructor for strings.  As
 discussed in Section 7 of \"<a
 href='http://www.cs.utexas.edu/users/moore/publications/km97a.pdf'>A Precise
 Description of the ACL2 Logic</a>\" (Matt Kaufmann and J Moore, April, 1998),
 a string may be built by coercing its list of characters: for example,
 @('\"abc\"') is @('(coerce '(#\\a #\\b #\\c) 'string)').  More precisely,
 @('\"abc\"') is an abbreviation for @('(coerce '(#\\a #\\b #\\c) 'string)'),
 where even more pedantically, @(''(#\\a #\\b #\\c)') is an abbreviation for
 @('(cons '#\\a (cons '#\\b (cons '#\\c 'nil)))').</p>")

(defxdoc command
  :parents (history)
  :short "Forms you type at the top-level, but..."
  :long "<p>...the word ``command'' usually refers to a top-level form whose
 evaluation produces a new logical @(see world).</p>

 @({
  Typical commands are:
  (defun foo (x) (cons x x))
  (defthm consp-foo (consp (foo x)))
  (defrec pair (hd . tl) nil)
 })

 <p>The first two forms are examples of commands that are in fact primitive
 @(see events).  See @(see events).  @('Defrec'), on the other hand, is a macro
 that expands into a @(tsee progn) of several primitive @(see events).  In
 general, a @(see world) extending command generates one or more @(see
 events).</p>

 <p>Both @(see events) and commands leave landmarks on the @(see world) that
 enable us to determine how the given @(see world) was created from the
 previous one.  Most of your interactions will occur at the command level,
 i.e., you type commands, you print previous commands, and you undo back
 through commands.  Commands are denoted by command descriptors.  See @(see
 command-descriptor).</p>")

(defxdoc command-descriptor
  :parents (history)
  :short "An object describing a particular @(see command) typed by the user"
  :long "@({
  Examples:

  :max      ; the command most recently typed by the user
  :x        ; synonymous with :max
  (:x -1)   ; the command before the most recent one
  (:x -2)   ; the command before that
  :x-2      ; synonymous with (:x -2)
  5         ; the fifth command typed by the user
  1         ; the first command typed by the user
  0         ; the last command of the system initialization
  -1        ; the next-to-last initialization command
  :min      ; the first command of the initialization
  :start    ; the last command of the initial ACL2 logical world
  fn        ; the command that introduced the logical name fn
  (:search (defmacro foo-bar))
            ; the first command encountered in a search from :max to
            ; 0 that either contains defmacro and foo-bar in the
            ; command form or contains defmacro and foo-bar in some
            ; event within its block.
 })

 <p>The recorded @(see history) of your interactions with the top-level ACL2
 @(see command) loop is marked by the @(see command)s you typed that changed
 the logical @(see world).  Each such @(see command) generated one or more
 @(see events), since the only way for you to change the logical @(see world)
 is to execute an event function.  See @(see command) and see @(see events).
 We divide @(see history) into ``@(see command) blocks,'' grouping together
 each @(see world) changing @(see command) and its @(see events).  A ``@(see
 command) descriptor'' is an object that can be used to describe a particular
 @(see command) in the @(see history) of the ongoing session.</p>

 <p>Each @(see command) is assigned a unique integer called its ``@(see
 command) number'' which indicates the @(see command)'s position in the
 chronological ordering of all of the @(see command)s ever executed in this
 session (including those executed to initialize the system).  We assign the
 number 1 to the first @(see command) you type to ACL2.  We assign 2 to the
 second and so on.  The non-positive integers are assigned to ``prehistoric''
 @(see command)s, i.e., the @(see command)s used to initialize the ACL2 system:
 0 is the last @(see command) of the initialization, -1 is the one before that,
 etc.</p>

 <p>The legal @(see command) descriptors are described below.  We use @('n') to
 denote any integer, @('sym') to denote any logical name (see @(see
 logical-name)), and @('cd') to denote, recursively, any @(see command)
 descriptor.</p>

 @({
   command                   command
  descriptor                described

  :max   -- the most recently executed command (i.e., the one with
            the largest command number)
  :x     -- synonymous with :max
  :x-k   -- synonymous with (:x -k), if k is an integer and k>0
  :min   -- the earliest command (i.e., the one with the smallest
            command number and hence the first command of the system
            initialization)
  :start -- the last command when ACL2 starts up
  n      -- command number n  (If n is not in the
            range :min<=n<=:max, n is replaced by the nearest of :min
            and :max.)
  sym    -- the command that introduced the logical name sym
  (cd n) -- the command whose number is n plus the command number of
            the command described by cd
  (:search pat cd1 cd2)
            In this command descriptor, pat must be either an atom or
            a true list of atoms and cd1 and cd2 must be command
            descriptors.  We search the interval from cd1 through cd2
            for the first command that matches pat.  Note that if cd1
            occurs chronologically after cd2, the search is
            ``backwards'' through history while if cd1 occurs
            chronologically before cd2, the search is ``forwards''.  A
            backwards search will find the most recent match; a
            forward search will find the chronologically earliest
            match.  A command matches pat if either the command form
            itself or one of the events in the block contains pat (or
            all of the atoms in pat if pat is a list).
  (:search pat)
            the command found by (:search pat :max 0), i.e., the most
            recent command matching pat that was part of the user's
            session, not part of the system initialization.
 })")

(defxdoc command-line
  :parents (interfacing-tools)
  :short "Handling of command-line arguments when ACL2 is invoked"
  :long "<p>You may provide command-line arguments when invoking ACL2, which
 are passed to the host Lisp.  For more information on this topic, along with a
 discussion of how to save an ACL2 executable that avoids passing command-line
 arguments to the host Lisp, see @(see save-exec).</p>")

(defxdoc comment
  :parents (hide acl2-built-ins)
  :short "Variant of @(tsee prog2$) to help debug evaluation failures during
 proofs"
  :long "<p>Semantically, @('(comment x y)') equals @('y'); the value of @('x')
 is ignored.  Thus @('comment') is much like @(tsee prog2$).  However, when you
 see a call of @('comment') in ACL2 proof output, it will likely be under a
 call of @(tsee hide), with information that may be helpful in understanding
 why the call of @('hide') was inserted.  Below we illustrate the various ways
 in which ACL2 may replace a term @('tm') by @('(hide (comment \"...\"
 tm))').  (On occasion you will simply see @('(hide tm)'); such cases are not
 discussed here.)</p>

 <p>Also see @(see hide) for further discussion of how to avoid such proof
 failures, and for how to keep the prover from inserting a @('comment') call
 under a call of @(tsee hide).</p>

 <h3>Evaluation during proofs</h3>

 <p>Forms:</p>

 @({
 (HIDE (COMMENT \"Failed attempt to call constrained function <fn>\" <term>))
 (HIDE (COMMENT \"Failed attempt to call non-executable function <fn>\" <term>))
 })

 <p>Consider the following example.</p>

 @({
 (defstub f (x) t)
 (defun g (x) (cons (f x) x))
 (defun h (x) (cons x (cdr (g x))))
 (thm (equal (h 3) '(3 . 3)))
 })

 <p>The proof attempt fails for the @(tsee thm) call, indicating the checkpoint
 shown below.</p>

 @({
 *** Key checkpoint at the top level: ***

 Goal'
 (EQUAL (HIDE (COMMENT \"Failed attempt to call constrained function F\"
                       (H 3)))
        '(3 . 3))
 })

 <p>The first argument of @('equal') is logically just @('(h 3)').  But the
 @('comment') and @('hide') wrappers are telling us that evaluation of @('(h
 3)') failed because it led to a call of the constrained function @('f').  It
 is easy to see why in this case, by looking at the definitions, where @('h')
 calls @('g'), which calls @('f').  But more complicated such failures may be
 difficult to understand without such information.  In very complicated cases,
 one might even want to use the Lisp debugger after designating a @(tsee
 break$) call using @(tsee trace$), like this (here, shown using host Lisp
 CCL).</p>

 @({
 ACL2 !>(trace$ (f :entry (break$)))
  ((F :ENTRY (BREAK$)))
 ACL2 !>(thm (equal (h 3) '(3 . 3)))

 > Break: Break
 > While executing: BREAK$, in process listener(1).
 > Type :GO to continue, :POP to abort, :R for a list of available restarts.
 > If continued: Return from BREAK.
 > Type :? for other options.
 1 > :b ; user input to get backtrace
  (262932A0) : 0 (BREAK$) 157
  (262932F0) : 1 (F 3) 141
  (26293338) : 2 (FUNCALL #'#<(:INTERNAL ACL2_*1*_ACL2::G ACL2_*1*_ACL2::G)> 3) 37
  (26293350) : 3 (FUNCALL #'#<(:INTERNAL ACL2_*1*_ACL2::H ACL2_*1*_ACL2::H)> 3) 37
  (26293368) : 4 (RAW-EV-FNCALL H (3) NIL NIL [[.. output elided ..]]
 })

 <p>This output from Lisp is quite low-level, but reading from the bottom up
 provides the following sequence of events.</p>

 <ul>

 <li>4. Call @('h') with argument list @('(3)').</li>

 <li>3. Call the @(see executable-counterpart) of @('h').</li>

 <li>2. Call the @(see executable-counterpart) of @('g').</li>

 <li>1. Attempt to call the constrained function, @('f').</li>

 </ul>

 <p>An easy way to avoid this proof failure is to avoid execution of calls of
 @('h') and @('g'), as follows.</p>

 @({
 (thm (equal (h 3) '(3 . 3))
      :hints ((\"Goal\" :in-theory (disable (:e g) (:e h)))))
 })

 <p>(It actually suffices to disable only @('(:e h)'), but the workings of the
 ACL2 rewriter are out of scope here.)</p>

 <p>Note that if the offending function is @(see non-executable) rather than
 constrained, in particular if that function is defined using @(tsee defun-nx),
 then in the first argument of comment you will see ``non-executable'' instead
 of ``constrained''.</p>

 <h3>Evaluation during building a term</h3>

 <p>Form:</p>

 @({
 (HIDE
  (COMMENT
   \"Failed attempt (when building a term) to call constrained function <fn>\"
   <term>))
 })

 <p>Consider how ACL2 approaches the proof of the non-theorem below.</p>

 @({
 (defstub foo (x) t)
 (defund bar (x) (foo x))
 (thm (implies (equal x 3) (equal (bar x) yyy)))
 })

 <p>The prover attacks the @(tsee thm) event by substituting the constant
 @(''3') for @('x').  But the prover attempts to evaluate @('(bar 3)') when
 doing that substitution, and the evaluation fails because @('bar') calls the
 undefined function @('foo').  The checkpoint is as follows.</p>

 @({
 (EQUAL
  (HIDE
   (COMMENT
      \"Failed attempt (when building a term) to call constrained function FOO\"
      (BAR 3)))
  YYY)
 })

 <h3>Failure to expand using a rule</h3>

 <p>Form:</p>

 @({
 (HIDE (COMMENT \"Unable to expand using the rule <name>\"
                <term>))
 })

 <p>Consider how ACL2 approaches the proof for the second event below.</p>

 @({
 (defthm nth-open (implies (and (consp x) (posp n))
                           (equal (nth n x) (nth (1- n) (cdr x))))
   :rule-classes ((:definition :controller-alist ((nth t t)) :install-body t)))
 (thm (equal (nth i y) zzz)
      :hints ((\"Goal\" :expand (nth i y) :do-not-induct t)))
 })

 <p>The checkpoint is as follows.  What happened is that the rule @('nth-open')
 had a hypothesis that was false when the rule was attempted for the term
 @('(nth i y)').</p>

 @({
 (IMPLIES (NOT (CONSP Y))
          (EQUAL (HIDE (COMMENT \"Unable to expand using the rule NTH-OPEN\"
                                (NTH I Y)))
                 ZZZ))
 })

 <h3>Failure due to disabled or missing warrants</h3>

 <p>Forms:</p>

 @({
 (HIDE (COMMENT \"Call failed because the rule apply$-<fn> is disabled\"
       <term>))
 (HIDE (COMMENT \"Call failed because the warrant for <fn> is not known to be true\"
       <term>))
 })

 <p>The first of these forms may appear when an attempt to evaluate a call of
 @(tsee apply$) fails because a necessary @(see warrant) is disable)d.  The
 second form may appear when the warrant is not known to be true in the present
 context, either because it is known to be false or because it cannot be
 assumed true because forcing is @(see disable)d.  In the following example,
 the attempt to simplify the call of @('apply$') in the theorem ultimately
 leads to an attempt to evaluate a call of @(tsee ev$), which ultimately fails
 because it leads to a call to evaluate @('(apply$ 'bar '(3))') @('bar').  That
 call causes an error because the warrant is unavailable, because the rule
 @('apply$-bar') is disabled, hence cannot rewrite a term @('(apply$ 'bar
 args)') to @('(bar (car args))').</p>

 @({
 (include-book \"projects/apply/top\" :dir :system)
 (defun$ bar (x) x)
 (thm (implies (warrant bar)
               (equal (apply$ '(lambda (y) (bar y)) '(3)) 3))
      :hints ((\"Goal\" :in-theory (disable apply$-bar ev$))))
 })

 <p>The checkpoint in the proof for the @('thm') just above is as follows.</p>

 @({
 (IMPLIES
   (APPLY$-WARRANT-BAR)
   (EQUAL (HIDE (COMMENT \"Call failed because the rule APPLY$-BAR is disabled\"
                         (EV$ '(BAR Y) '((Y . 3)))))
          3))
 })

 <p>Similarly, if we instead submit the following event, we see the other such
 message, in this case about a false warrant.</p>

 @({
 (thm (implies (not (warrant bar))
               (equal (apply$ '(lambda (y) (bar y)) '(3)) 3))
      :hints ((\"Goal\" :in-theory (disable ev$))))
 })

 <p>Here is the resulting checkpoint.</p>

 @({
 (IMPLIES
  (NOT (APPLY$-WARRANT-BAR))
  (EQUAL (HIDE (COMMENT \"Call failed because the warrant for BAR is not known to be true\"
                        (EV$ '(BAR Y) '((Y . 3)))))
         3))
 })

 <p>Our final example illustrates a failure due to forcing being disabled.  The
 use of @(tsee loop$) in the definition of @('bar') expands to create a call of
 @(tsee ev$), which cannot be simplified during the proof of the @(tsee thm)
 below because the necessary warrant hypothesis is missing and cannot be
 forced, since forcing is disabled (see also @(see disable-forcing)).</p>

 @({
 (defun$ hello (x)
    (declare (xargs :guard t))
    (list 'hi x))

 (defun bar (lst)
    (declare (xargs :guard (true-listp lst)))
    (loop$ for name in lst collect (hello name))))

 (thm (equal (bar '(john))
             '((hi john)))
      :hints ((\"Goal\" :in-theory (disable (:e force)))))
 })

 <p>Here is the resulting checkpoint.</p>

 @({
 (EQUAL
  (HIDE
      (COMMENT
           \"Call failed because the warrant for HELLO is not known to be true\"
           (EV$ '(HELLO LOOP$-IVAR)
                '((LOOP$-IVAR . JOHN)))))
  '(HI JOHN))
 })")

(defxdoc common-lisp
  :parents (about-acl2)
  :short "Relation to Common Lisp, including deviations from the spec"
  :long "<p>ACL2 is a logic, a theorem prover, and a programming language based
 on Common Lisp.  A connection with Common Lisp is established with guards (see
 @(see guard)).</p>

 <p>However, here we document potential deviations from Common Lisp semantics
 even in the presence of verified guards.  Our view is that these deviations
 are extremely unlikely to manifest; indeed, as of this writing we are unaware
 of any cases in which these issues arise in practice.  However, we feel
 obligated to acknowledge their possibility, which could result in surprises
 during evaluation or even proof.</p>

 <p>The Common Lisp spec allows certain predicates to return what it calls
 ``generalized Booleans,'' which are really arbitrary values that are to be
 viewed as either @('nil') or non-@('nil').  However, in ACL2 these functions
 are assumed to return @('nil') or @('t').  For details, see @(see
 generalized-booleans).</p>

 <p>The execution of forms with @(':')@(tsee program) mode functions can result
 in calls of functions on arguments that do not satisfy their @(see guard)s.
 In practice, this simply causes hard Lisp errors.  But in principle one could
 imagine a damaged Lisp image that operates incorrectly.  See @(see
 defun-mode-caveat).</p>

 <p>The Common Lisp spec, specifically
 Section @(`(:raw (clhs  \"Body/03_bbc.htm\" \"Section 3.2.2.3\"))`)
 of the @(`(:raw (clhs \"Front\" \"Common Lisp Hyperspec\"))`),
 allows for undefined results when a function is ``multiply defined'' in a
 compiled file.  ACL2 allows redundant @(tsee defun)s in a book, and in general
 @(see books) are compiled by @('certify-book') (but see @(see certify-book)
 and see @(see compilation) for how to control such compilation).  Moreover,
 ACL2 provides a redefinition capability (see @(see ld-redefinition-action) and
 see @(see redef)), and the above section also allows for undefined results
 when a function is defined in a compiled file and then redefined,
 presumably (for example) because of inlining.</p>")

(defxdoc community-books
  :parents (books)
  :short "Libraries of ACL2 @(see books) developed by the ACL2 community."

  :long "<p>ACL2 @(see books) are files of ACL2 @(see events) like definitions
 and theorems.</p>

 <p>The ACL2 <b>Community Books</b> are the canonical set of open-source books
 for ACL2, developed since the early 1990s by members of the ACL2 community.
 They include libraries for reasoning in many domains, macro libraries for more
 quickly writing and documenting code, interfacing tools for connecting ACL2 to
 other systems, productivity tools for better proof automation and debugging,
 and specialty libraries for areas like hardware verification.</p>

 <p>From the <a href='https://github.com/acl2/acl2/'>github ACL2 project</a> web
 site you can:</p>

 <ul>
 <li>Download the Community Books;</li>
 <li>Learn how to contribute books to the ACL2 community; and</li>
 <li>Obtain updates between ACL2 releases.</li>
 </ul>

 <p>See @(see git-quick-start) for information about how to download the
 ``bleeding edge'' ACL2 system and community books.</p>")

(defxdoc comp
  :parents (compilation events acl2-built-ins)
  :short "Compile some ACL2 functions"
  :long "<p>NOTE: @('Comp') is a no-op if explicit compilation is suppressed;
 see @(see compilation).  The documentation here assumes that this is not the
 case.  See @(see evaluation) for background on executable-counterpart and
 submitted functions.</p>

 @({
  Examples:
  :comp t          ; compile all uncompiled ACL2 functions
  (comp t)         ; same as above, but can be put into a book
  (comp :exec)     ; compile all uncompiled executable-counterpart definitions
  :comp foo        ; compile the defined function foo (both its submitted and
                   ; executable-counterpart functions)
  :comp (:raw foo) ; compile the submitted function for the defined function foo
                   ; but not the corresponding executable-counterpart
  :comp (foo bar)  ; compile the defined functions foo and bar
  :comp (foo (:raw bar))  ; compile the defined functions foo and bar, but for
                          ; bar do not compile the executable-counterpart

  General Form:
  :comp specifier
  where specifier is one of the following:

    t                     compile all user-defined ACL2 functions that are
                            currently uncompiled (redefined built-in functions
                            are not recompiled)
    :exec                 same as t, except that only executable-counterparts
                            are compiled (see below), not submitted definitions
    :raw                  same as t, except that only submitted definitions are
                            compiled, not executable-counterparts
    (name-1 ... name-k)   a non-empty list of names of functions defined by
                            DEFUN in ACL2, except that each name-i can be of
                            the form (:raw sym) or (:exec sym), where sym is
                          the name of such a function
    name                  same as (name)
 })

 <p>When you define a function in ACL2, you are really causing two definitions
 to be made ``under the hood'' in Common Lisp: the definition is submitted
 explicitly to raw Lisp, but so is a corresponding executable-counterpart; see
 @(see evaluation).  If guards have not been verified, then only the
 executable-counterpart will be evaluated; see @(see evaluation) and see @(see
 guards-and-evaluation), in particular the section titled ``Guards and
 evaluation V: efficiency issues''.</p>

 <p>Thus, if you are not verifying @(see guard)s and you want the benefit of
 Lisp compilation for speed and space efficiency, then you may want to place
 the form @('(comp :exec)') in your @(see books).</p>

 <p>Generally it is not necessary to place the form @('(comp t)'), or the form
 @('(comp :raw)'), in a book, because @(tsee certify-book) compiles the raw
 Lisp definitions anyhow, by default.  But you may wish to put @('(comp t)') or
 @('(comp fn1 fn2 ... fnk)') in a book when such a form precedes expensive
 calls of functions, for example for proofs involving calls of functions on
 large constants, or to support computationally expensive macroexpansion.</p>

 <p>As suggested by the examples above, if a function specifier is of the form
 @('(:raw fn)'), then @('fn') will be compiled in raw Common Lisp but its
 corresponding executable-counterpart definition will not be compiled; and for
 @('(:exec fn)'), it's the other way around.</p>

 <p>The use of @(':comp') may create various files whose names start with
 ``@('TMP*')'', but it then deletes them.  If you want to save these files,
 evaluate @('(assign keep-tmp-files t)').</p>

 <p>Also see @(see set-compile-fns) for a way to compile each function as it is
 defined.  But note that @('set-compile-fns') is ignored during @(tsee
 include-book).</p>

 <p>Note that if functions are traced (see @(see trace$)), then @('comp') will
 first untrace the functions that are to be compiled, then will do the
 compile(s), and finally will re-trace the functions that it untraced (using
 their original trace specs).  In particular, if you have traced a function and
 then you compile it using @(':comp'), the resulting traced function will be
 compiled as well unless you specified @(':compile nil') in your trace spec;
 and after you untrace the function it will definitely run compiled.</p>

 <p>We conclude with a technical remark only for those who use trust tags to
 write raw Lisp code.  @(':Comp') generally creates files to compile unless it
 is given a single function to compile.  Those files contain the ACL2
 definitions of all functions to compile, omitting those in the lists obtained
 by evaluating the forms @('(@ logic-fns-with-raw-code)') and @('(@
 program-fns-with-raw-code)').  @(':Comp') skips compilation for functions that
 are already compiled, as is typically the case when you redefine functions in
 raw Lisp using @(see include-raw).  But if you define interpreted (as opposed
 to compiled) functions with raw Lisp code, say by using trust tags (see @(see
 defttag)) and @(tsee progn!), then you are advised to add all such symbols to
 one of the lists stored in the two @(see state) globals above: to
 @('logic-fns-with-raw-code') if the function symbol is in @(':')@(tsee logic)
 mode, else to @('program-fns-with-raw-code').  Then, instead of the
 corresponding ACL2 definition (without raw Lisp code) being written to a file,
 the function symbol will be passed directly to the Lisp @('compile') function.
 Note that the above two state globals are both untouchable, so you may need to
 deal with that before modifying them, for example as follows (also see @(see
 remove-untouchable)).</p>

 @({
  (defttag t)
  (progn!
   :state-global-bindings
   ((acl2::temp-touchable-vars t acl2::set-temp-touchable-vars))
   (f-put-global 'acl2::logic-fns-with-raw-code
                 (cons 'my-fn (@ acl2::logic-fns-with-raw-code))
                 state))
 })")

(defxdoc comp-gcl
  :parents (compilation acl2-built-ins)
  :short "Compile some ACL2 functions leaving .c and .h files"
  :long "<p>@('Comp-gcl') is for use by experts who want to examine the results
 of GCL compilation, and it may only be used with ACL2 implementations built on
 top of GCL.  It takes exactly the same arguments as @(tsee comp), and has the
 same basic functionality (see @(see comp)), but has two additional effects.
 First, files @('\"TMP.lisp\"') and @('\"TMP1.lisp\"') are always created, even
 when a single function is specified.  Second, @('comp-gcl') always leaves
 files @('\"TMP.c\"'), @('\"TMP.h\"'), @('\"TMP1.c\"'), and @('\"TMP1.h\"')
 when compilation is complete.</p>")

(defxdoc compilation
  :parents (programming)
  :short "Compiling ACL2 functions"
  :long "<p>ACL2 has several mechanisms to speed up the evaluation of function
 calls by <i>compiling</i> functions: see @(see comp), see @(see
 set-compile-fns), and see @(see certify-book).  The intention is that
 compilation never changes the value returned by a function call, though it
 could cause the call to succeed rather than fail, for example by avoiding a
 stack overflow.</p>

 <p>The @(tsee state) global variable @(''compiler-enabled') is set
 automatically when the system is built, and may depend on the underlying Lisp
 implementation.  (In order to disable the compiler at build time, which will
 defeat the speed-up but usually be pretty harmless when the host Lisp is CCL
 or SBCL, see the discussion of @('ACL2_COMPILER_DISABLED') in distributed file
 @('GNUmakefile').)  The value of @(''compiler-enabled'), as returned by @('(@
 compiler-enabled)'), can be @('t'), @(':books'), or @('nil').  If the value is
 @('nil'), then @(tsee include-book) and @(tsee certify-book) coerce their
 arguments @(':load-compile-file') and @('compile-flg') arguments
 (respectively) to @('nil').  Otherwise, the value is @(':books') or @('t') and
 there is no such coercion; but if the value is not @('t'), then @(tsee comp)
 and @(tsee set-compile-fns) are no-ops, which is probably desirable for Lisps
 such as CCL and SBCL that compile on-the-fly even when the compiler is not
 explicitly invoked.</p>

 <p>However, you may have reason to want to change the above (default)
 behavior.  To enable compilation by default for @(tsee certify-book) and
 @(tsee include-book) but not for @(tsee comp) or @(tsee set-compile-fns):</p>

 @({
  (set-compiler-enabled :books state)
 })

 <p>To enable compilation not only as above but also for @(tsee comp) and
 @(tsee set-compile-fns):</p>

 @({
  (set-compiler-enabled t state)
 })

 <p>To suppress compilation and loading of compiled files by @(tsee
 include-book) (for example, if you get a raw Lisp error such as ``Wrong FASL
 version''):</p>

 @({
  (set-compiler-enabled nil state)
 })

 <p>Remark for users of @(tsee make-event).  If @('set-compiler-enabled') is
 invoked during @('make-event') expansion, its effect on @(tsee state) global
 variable @(''compiler-enabled') will persist after evaluation completes for
 that @('make-event') form.  So for example, one might use the following idiom
 in a book so that for all books included on behalf of a given @(tsee
 include-book) form, no compiled files are loaded, but (optionally) no such
 effect takes place for later @('include-book') forms in that book.</p>

 @({
  (make-event
   (pprogn (f-put-global 'saved-compiler-enabled (@ compiler-enabled) state)
           (set-compiler-enabled nil state)
           (value '(value-triple nil)))
   :check-expansion t)
  (include-book \"my-book\")
  ; optional
  (make-event
   (pprogn (set-compiler-enabled (@ saved-compiler-enabled) state)
           (value '(value-triple nil)))
   :check-expansion t)
 })

 <p>Upon completion of an invocation of @(tsee include-book) or @(tsee
 certify-book), the value of @(tsee state) global variable
 @(''compiler-enabled') is restored to the value it had immediately before that
 invocation.</p>

 <p>See @(see book-compiled-file) for more discussion about compilation and
 @(see books).</p>

 <p>We close with a discussion of a feature that allows control over the
 loading of @('.port') files in close analogy to how loading of compiled files
 is controlled by @('set-compiler-enabled'), as described above.
 (See @(see uncertified-books) for a discussion of @('.port') files.)  A @(see
 state) global variable, @(''port-file-enabled') exists for this purpose, and
 it is set as follows.</p>

 @({
  (set-port-file-enabled t state)   ; permit loading of .port files (default)
  (set-port-file-enabled nil state) ; skip loading of .port files
 })

 <p>Just as described above for state global @('compiler-enabled'), the value
 of @(''port-file-enabled') persists after @(tsee make-event) expansion and is
 restored after @(tsee certify-book) and @(tsee include-book).  The idiom
 displayed above, for avoiding loading of compiled files, can be modified or
 extended in the obvious way to avoid loading of @('.port') files.</p>")

(defxdoc compiling-acl2p
  :parents (parallelism)
  :short "Compiling ACL2(p)"
  :long "<p>This @(see documentation) topic relates to the experimental
 extension of ACL2 supporting parallel execution and proof; see @(see
 parallelism).  See @(see parallelism-tutorial) for an introduction to parallel
 programming in ACL2.</p>

 <p>You can build an experimental version of ACL2 that supports parallel
 execution in the following host Common Lisp implementations:</p>

 <blockquote><p>* CCL (OpenMCL)</p>

 <p>* Lispworks 6.0</p>

 <p>* SBCL with threads (feature @(':sb-thread'))</p></blockquote>

 <p>The command below will compile ACL2 to support parallel execution,
 including parallel execution during proofs.  Any non-empty string may be used
 in place of @('t'), and the value of @('LISP') (shown here as @('ccl')) is any
 Lisp executable on which one can build ACL2(p) (see @(see parallelism)).</p>

 @({
  make ACL2_PAR=t LISP=ccl
 })

 <p>So for example, to make an executable image and also documentation (which
 will appear in subdirectories @('doc/EMACS') and @('doc/HTML')), using the
 Lisp executable @('ccl'):</p>

 @({
  make large DOC ACL2_PAR=t LISP=ccl
 })")

(defxdoc complex
  :parents (numbers acl2-built-ins)
  :short "Create an ACL2 number"
  :long "@({
  Examples:
  (complex x 3) ; x + 3i, where i is the principal square root of -1
  (complex x y) ; x + yi
  (complex x 0) ; same as x, for rational numbers x
 })

 <p>The function @('complex') takes two rational number arguments and returns
 an ACL2 number.  This number will be of type @('(complex rational)') [as
 defined in the Common Lisp language], except that if the second argument is
 zero, then @('complex') returns its first argument.  The function @(tsee
 complex-rationalp) is a recognizer for complex rational numbers, i.e. for ACL2
 numbers that are not rational numbers.</p>

 <p>The reader macro @('#C') (which is the same as @('#c')) provides a
 convenient way for typing in complex numbers.  For explicit rational numbers
 @('x') and @('y'), @('#C(x y)') is read to the same value as @('(complex x
 y)').</p>

 <p>The functions @(tsee realpart) and @(tsee imagpart) return the real and
 imaginary parts (respectively) of a complex (possibly rational) number.  So
 for example, @('(realpart #C(3 4)) = 3'), @('(imagpart #C(3 4)) = 4'),
 @('(realpart 3/4) = 3/4'), and @('(imagpart 3/4) = 0').</p>

 <p>The following built-in axiom may be useful for reasoning about complex
 numbers.</p>

 @({
  (defaxiom complex-definition
    (implies (and (real/rationalp x)
                  (real/rationalp y))
             (equal (complex x y)
                    (+ x (* #c(0 1) y))))
    :rule-classes nil)
 })

 <p>A completion axiom that shows what @('complex') returns on arguments
 violating its @(see guard) (which says that both arguments are rational
 numbers) is the following, named @('completion-of-complex').</p>

 @({
  (equal (complex x y)
         (complex (if (rationalp x) x 0)
                  (if (rationalp y) y 0)))
 })")

(defxdoc complex-rationalp
  :parents (numbers acl2-built-ins)
  :short "Recognizes complex rational numbers"
  :long "@({
  Examples:
  (complex-rationalp 3)       ; nil, as 3 is rational, not complex rational
  (complex-rationalp #c(3 0)) ; nil, since #c(3 0) is the same as 3
  (complex-rationalp t)       ; nil
  (complex-rationalp #c(3 1)) ; t, as #c(3 1) is the complex number 3 + i
 })

 <p>See @(see complex) for more about complex rationals in ACL2.</p>")

(defxdoc complex/complex-rationalp
  :parents (numbers acl2-built-ins)
  :short "Recognizer for complex numbers"
  :long "<p>For most ACL2 users, this is a macro abbreviating
 @('complex-rationalp'); see @(see complex-rationalp).  In ACL2(r) (see @(see
 real)), a complex number @('x') may have irrational real and imaginary parts.
 This macro abbreviates the predicate @('complexp') in ACL2(r), which holds for
 such @('x').  Most ACL2 users can ignore this macro and use @(tsee
 complex-rationalp) instead.  Some community books use
 @('complex/complex-rationalp') so that they are suitable for ACL2(r) as
 well.</p>")

(defxdoc compound-recognizer
  :parents (rule-classes)
  :short "Make a rule used by the typing mechanism"
  :long "<p>See @(see rule-classes) for a general discussion of rule classes,
 including how they are used to build rules from formulas and a discussion of
 the various keywords in a rule class description.</p>

 @({
  Examples:
  (defthm alistp-implies-true-listp-compound-recognizer
    (implies (alistp x)                 ; When (alistp x) is assumed true, add
             (true-listp x))            ; the additional hypothesis that x is
    :rule-classes :compound-recognizer) ; of primitive type true-listp.

  (defthm natp-compound-recognizer      ; See discussion below.
    (equal (natp x)
           (and (integerp x)
                (<= 0 x)))
    :rule-classes :compound-recognizer)
 })

 <p>Before presenting the General Forms, we start with a motivating example:
 the second @(tsee defthm) form above, which provides a nice example of a
 @(':compound-recognizer') rule that is built into ACL2.  To see how this rule
 might be useful, consider the following (admittedly very simple) @(see
 events).</p>

 @({
  (defun triple (x)
    (* 3 x))

  (defthm triple-preserves-integerp
    (implies (integerp x)
             (integerp (triple x))))

  (in-theory (disable triple natp))
 })

 <p>If the above @(':compound-recognizer') rule is disabled, then the following
 trivial theorem fails as shown; we explain below.</p>

 @({
  (thm (implies (natp x)
                (integerp (triple x)))
    :hints ((\"Goal\" :in-theory (disable natp-compound-recognizer))))
 })

 <p>The problem is that when ACL2 tries to rewrite the term @('(integerp
 (triple x))') using the @(':')@(tsee rewrite) rule
 @('triple-preserves-integerp'), it needs to rewrite the hypothesis
 @('(integerp x)') to @('t'), but instead what is known is @('(natp x)').  If
 we remove the hint, then the proof succeeds because the above
 @(':compound-recognizer') rule tells ACL2 that when assuming @('(natp x)') to
 be true, it should actually assume both @('(integerp x)') and @('(<= 0 x)') to
 be true.</p>

 @({
  General Forms:
  (implies (fn x) concl)               ; (1)
  (implies (not (fn x)) concl)         ; (2)
  (and (implies (fn x) concl1)         ; (3)
       (implies (not (fn x)) concl2))
  (if (fn x) concl1 concl2)            ; (4)
  (iff (fn x) concl)                   ; (5)
  (equal (fn x) concl)                 ; (6)
 })

 <p>where @('fn') is a Boolean valued function of one argument, @('x') is a
 variable symbol, and the system can deduce some restriction on the primitive
 type of @('x') from the assumption that @('concl') holds.  The last
 restriction is vague but one way to understand it is to strengthen it a little
 to ``and @('concl') is a non-tautological disjunction of the primitive type
 recognizers listed below.''</p>

 <p>The primitive ACL2 types and a suitable primitive recognizing expression
 for each are listed below.</p>

 @({
    type                suitable primitive recognizer

    zero                    (equal x 0)
    one                     (equal x 1)
    negative integers       (and (integerp x) (< x 0))
    positive integers > 1   (and (integerp x) (> x 1))
    negative ratio          (and (rationalp x)
                                 (not (integerp x))
                                 (< x 0))
    positive ratio          (and (rationalp x)
                                 (not (integerp x))
                                 (> x 0))
    complex rational        (complex-rationalp x)
    nil                     (equal x nil)
    t                       (equal x t)
    other symbols           (and (symbolp x)
                                 (not (equal x nil))
                                 (not (equal x t)))
    proper conses           (and (consp x)
                                 (true-listp x))
    improper conses         (and (consp x)
                                 (not (true-listp x)))
    strings                 (stringp x)
    characters              (characterp x)
 })

 <p>Thus, since the naturals comprise the types @('zero'), @('one'), and
 @('positive integers > 1'), a suitable @('concl') to recognize the naturals
 would be</p>

 @({
    (or (equal x 0)
        (equal x 1)
        (and (integerp x) (> x 1)))
 })

 <p>However, it turns out that we also permit @('(and (integerp x) (>= x 0))'),
 i.e. @('concl') doesn't literally need to be formed as a direct disjunction of
 terms from the table above.</p>

 <p>Similarly, the true-lists could be specified by</p>

 @({
    (or (equal x nil)
        (and (consp x)
             (true-listp x)))
 })

 <p>but we in fact allow @('(true-listp x)') as well.  When time permits we
 may document more fully what is allowed or implement a macro that permits
 direct specification of the desired type in terms of the primitives.</p>

 <p>There are essentially four forms of @(':compound-recognizer') rules, as the
 forms labeled (3) and (4) above are equivalent, as are those labeled (5) and
 (6).  We explain how such rules are used by considering the individual
 forms.</p>

 <p>Consider form (1), @('(implies (fn x) concl)').  The effect of such a rule
 is that when the rewriter assumes @('(fn x)') true, as it would while diving
 through @('(if (fn x) xxx ...)') to rewrite @('xxx'), it restricts the type of
 @('x') as specified by @('concl').  For example, if @('concl') is the term
 @('(integerp x)'), then when rewriting @('xxx'), @('x') will be assumed to be
 an integer.  However, when assuming @('(fn x)') false, as necessary in @('(if
 (fn x) ... xxx)'), the rule permits no additional assumptions about the type
 of @('x').  For example, if @('fn') is @('primep'), i.e., the predicate that
 recognizes prime numbers, then @('(implies (primep x) (and (integerp x) (>= x
 0)))') is a compound recognizer rule of the first form.  When @('(primep x)')
 is assumed true, the rewriter gains the additional information that @('x') is
 a natural number.  When @('(primep x)') is assumed false, no additional
 information is gained &mdash; since @('x') may be a non-prime natural or may
 not even be a natural.</p>

 <p>Form (2) is the symmetric case, when assuming @('(fn x)') false permits
 type restrictions on @('x') but assuming @('(fn x)') true permits no such
 restrictions.  For example, if we defined @('exprp') to be the recognizer for
 well-formed expressions for some language in which all symbols, numbers,
 character objects and strings were well-formed &mdash; e.g., the
 well-formedness rules only put restrictions on expressions represented by
 @(tsee consp)s &mdash; then the theorem @('(implies (not (exprp x)) (consp
 x))') is a rule of the second form.  Assuming @('(exprp x)') true tells us
 nothing about the type of @('x'); assuming it false tells us @('x') is a
 @(tsee consp).</p>

 <p>Forms (3) and (4), which are really equivalent, address themselves to the
 case where one type may be deduced from @('(fn x)') and a generally unrelated
 type may be deduced from its negation.  If we modified the expression
 recognizer above so that character objects are illegal, then rules of the
 forms (3) and (4) are</p>

 @({
  (and (implies (exprp x) (not (characterp x)))
       (implies (not (exprp x)) (or (consp x) (characterp x)))).
  (if (exprp x)
      (not (characterp x))
    (or (consp x) (characterp x)))
 })

 <p>Finally, rules of forms (5) and (6) address the case where @('fn')
 recognizes all and only the objects whose type is described.  In these cases,
 @('fn') is really just a new name for some ``compound recognizers.''  The
 classic example is @('(booleanp x)'), which is just a handy combination of two
 primitive types:</p>

 @({
  (iff (booleanp x) (or (equal x t) (equal x nil))).
 })

 <p>Often it is best to disable @('fn') after proving that it is a compound
 recognizer, since otherwise the term @('(fn x)') will be expanded and thus
 disappear.</p>

 <p>Every time you prove a new compound recognizer rule about @('fn') it
 overrides all previously proved compound recognizer rules about @('fn').
 Thus, if you want to establish the type implied by @('(fn x)') and you want to
 establish the type implied by @('(not (fn x))'), you must prove a compound
 recognizer rule of the third, fourth, fifth, or sixth forms.  Proving a rule
 of the first form followed by one of the second only leaves the second fact in
 the database.</p>

 <p>Compound recognizer rules can be disabled with the effect that older rules
 about @('fn'), if any, are exposed.</p>

 <p>If you prove more than one compound recognizer rule for a function, you may
 see a <b>warning</b> message to the effect that the new rule is not as
 ``restrictive'' as the old.  That is, the new rules do not give the rewriter
 strictly more type information than it already had.  The new rule is stored
 anyway, overriding the old, if enabled.  You may be playing subtle games with
 enabling or rewriting.  But two other interpretations are more likely, we
 think.  One is that you have forgotten about an earlier rule and should merely
 print it out to make sure it says what you intend, and then discard your new
 rule.  The other is that you meant to give the system more information and the
 system has simply been unable to extract the intended type information from
 the term you placed in the conclusion of the new rule.  Given our lack of
 specificity in saying how type information is extracted from rules, you can
 hardly blame yourself for this problem.  Sorry.  If you suspect you've been
 burned this way, you should rephrase the new rule in terms of the primitive
 recognizing expressions above and see if the warning is still given.  It would
 also be helpful to let us see your example so we can consider it as we
 redesign this stuff.</p>

 <p>Compound recognizer rules are similar to @(':')@(tsee forward-chaining)
 rules in that the system deduces new information from the act of assuming
 something true or false.  If a compound recognizer rule were stored as a
 forward chaining rule it would have essentially the same effect as described,
 when it has any effect at all.  The important point is that @(':')@(tsee
 forward-chaining) rules, because of their more general and expensive form, are
 used ``at the top level'' of the simplification process: we forward chain from
 assumptions in the goal being proved.  But compound recognizer rules are built
 in at the bottom-most level of the simplifier, where type reasoning is
 done.</p>

 <p>All that said, compound recognizer rules are a rather fancy, specialized
 mechanism.  It may be more appropriate to create @(':')@(tsee
 forward-chaining) rules instead of @(':compound-recognizer') rules.</p>")

(defxdoc compress1
  :parents (arrays acl2-built-ins)
  :short "Remove irrelevant pairs from a 1-dimensional array"
  :long "@({
  Example Form:
  (compress1 'delta1 a)

  General Form:
  (compress1 name alist)
 })

 <p>where @('name') is a symbol and @('alist') is a 1-dimensional array,
 generally named @('name').  See @(see arrays) for details.  Logically
 speaking, this function can remove irrelevant pairs from @('alist'), possibly
 shortening it.  The function returns a new array, @('alist''), with the same
 @(tsee header) (including name and dimension) as @('alist'), that, under
 @(tsee aref1), is everywhere equal to @('alist').  That is, @('(aref1 name
 alist' i)') is @('(aref1 name alist i)'), for all legal indices @('i').
 @('Alist'') may be shorter than @('alist') and the non-irrelevant pairs may
 occur in a different order than in @('alist').</p>

 <p>Practically speaking, this function plays an important role in the
 efficient implementation of @(tsee aref1).  In addition to creating the new
 array, @('alist''), @('compress1') makes that array the ``semantic value'' of
 @('name') and allocates a raw lisp array to @('name').  For each legal index,
 @('i'), that raw lisp array contains @('(aref1 name alist' i)') in slot
 @('i').  Thus, subsequent @(tsee aref1) operations can be executed in
 virtually constant time provided they are given @('name') and the @('alist'')
 returned by the most recently executed @('compress1') or @(tsee aset1) on
 @('name').  See @(see arrays).</p>

 <p>In general, @('compress1') returns an alist whose @(tsee cdr) is an
 association list whose keys are nonnegative integers in ascending order.
 However, if the @(tsee header) specifies an @(':order') of @('>') then the
 keys will occur in descending order; and if the @(':order') is @(':none') or
 @('nil') then the keys will not be sorted and the header may appear anywhere
 (even more than once), i.e., @('compress1') is logically the identity
 function (though it still attaches an array under the hood).  Note however
 that a @(tsee compress1) call is replaced by a hard error if the header
 specifies an @(':order') of @(':none') or @('nil') and the array's length
 exceeds the @(tsee maximum-length) field of its @(tsee header).</p>

 <p>We close with a remark concerning efficiency in the case that the
 @(':ORDER') specified by the given @(see array)'s @(see header) is @('<') or
 @('>') and the alist is properly ordered: header occurring only first, then
 ascending (for @(':ORDER <')) or descending (for @(':ORDER >')) order of
 indices, with no value in the alist equal to the @(':DEFAULT') specified by
 the header.  In particular, this can cut the time to run @('compress1') on an
 alist containing only the header by more than half.</p>

 @(def compress1)")

(defxdoc compress2
  :parents (arrays acl2-built-ins)
  :short "Remove irrelevant pairs from a 2-dimensional array"
  :long "@({
  Example Form:
  (compress2 'delta1 a)

  General Form:
  (compress2 name alist)
 })

 <p>where @('name') is a symbol and @('alist') is a 2-dimensional array,
 generally named @('name').  See @(see arrays) for details.  Logically
 speaking, this function removes irrelevant pairs from @('alist'), possibly
 shortening it.  The function returns a new array, @('alist''), with the same
 @(tsee header) (including name and dimension) as @('alist'), that, under
 @(tsee aref2), is everywhere equal to @('alist').  That is, @('(aref2 name
 alist' i j)') is @('(aref2 name alist i j)'), for all legal indices @('i') and
 @('j').  @('Alist'') may be shorter than @('alist') and the non-irrelevant
 pairs may occur in a different order in @('alist'') than in @('alist').</p>

 <p>Practically speaking, this function plays an important role in the
 efficient implementation of @(tsee aref2).  In addition to creating the new
 array, @('alist''), @('compress2') makes that array the ``semantic value'' of
 @('name') and allocates a raw lisp array to @('name').  For all legal indices,
 @('i') and @('j'), that raw lisp array contains @('(aref2 name alist' i j)')
 in slot @('i'),@('j').  Thus, subsequent @(tsee aref2) operations can be
 executed in virtually constant time provided they are given @('name') and the
 @('alist'') returned by the most recently executed @('compress2') or @(tsee
 aset2) on @('name').  See @(see arrays).</p>

 @(def compress2)")

(defxdoc computed-hints
  :parents (hints)
  :short "Computing advice to the theorem proving process"
  :long "@({
  General Form of :hints:
  (hint1 hint2 ... hintk)
 })

 <p>Each element, hinti, must be either a common hint or a computed hint.</p>

 <p>A common hint is of the form</p>

 @({
  (goal-spec :key1 val1 ... :keyn valn)
 })

 <p>where @('goal-spec') is as specified in @(see goal-spec) and each
 @(':keyi') and @('vali') is as specified in @(see hints).  Among the ``common
 hints'' we include both the primitive hints and user-defined custom keyword
 hints (see @(see custom-keyword-hints)).</p>

 <p>A computed hint may be a function symbol, @('fn'), of three, four or seven
 arguments.  Otherwise, a computed hint is a term with the following
 properties:</p>

 <p>(a) the only free variables allowed in the term are @('ID'), @('CLAUSE'),
 @('WORLD'), @('STABLE-UNDER-SIMPLIFICATIONP'), @('HIST'), @('PSPV'), @('CTX'),
 and @(tsee STATE);</p>

 <p>(b) the output signature of the term is either @('(MV * * STATE)'), which
 is to be treated as an error triple (see below), or is @('*'), denoting a
 single non-@(see stobj) value; and</p>

 <p>(c) in the former case of (b) above, the term is single-threaded in @(tsee
 STATE).</p>

 <p>If a computed hint is a function symbol @('fn'), whose arity n is therefore
 three, four, or seven, then it is treated as the term resulting from applying
 that @('fn') to the first n variables shown in (a) above.  Notice that it must
 then return a single non-@(see stobj) value, not an error triple, since
 @('state') is not one of the first seven variables shown in (a).</p>

 <p>Note: Error triples are an ACL2 idiom for implementing ``errors''; see
 @(see error-triple).  If a computation returns @('(mv erp val state)') in a
 context in which ACL2 is respecting the error triple convention (see @(see
 ld-error-triples) and see @(see ld-error-action)), then an error is deemed to
 have occurred if @('erp') is non-@('nil').  The computation is expected to
 have printed an appropriate error message to @(tsee state) and further
 computation is aborted.  On the other hand, if a computation returns an error
 triple in which @('erp') is nil, then ``value'' of the computation is taken to
 be the second component, @('val'), of the triple (along with the possibly
 modified @(tsee state)), and computation continues.  For more information
 about programming with error triples, see @(see programming-with-state).</p>

 <p>The function symbol cases are treated as abbreviations of the term @('(fn
 ID CLAUSE WORLD)'), @('(fn ID CLAUSE WORLD STABLE-UNDER-SIMPLIFICATIONP)'), or
 @('(fn ID CLAUSE WORLD STABLE-UNDER-SIMPLIFICATIONP HIST PSPV CTX)') as
 appropriate for the arity of @('fn').  (Note that this tells you which
 argument of @('fn') is which.)  Moreover, in these cases the value returned
 must be a single ordinary (non-@(see stobj)) value, not an error triple.  In
 the discussion below we assume all computed hints are of the term form.
 Indeed, we almost assume all computed hints are of the 3 and 4 argument forms.
 We only comment briefly on the 7 argument form in @(see
 using-computed-hints-8).</p>

 <p>The semantics of a computed hint term is as follows.  On every subgoal, the
 term is evaluated in an environment in which the variables mentioned in (a)
 above are bound to context-sensitive values explained below.  Either the
 computed hint signals an error, in which the proof attempt aborts, or else it
 returns a value, @('val') and a new state, @(tsee state).  Any changes to
 those parts of @(tsee state) that affect logical soundness are undone; more
 specifically, the values of symbols (sometimes called ``state global
 variables'') in the list @('*protected-system-state-globals*') in the global
 table of the state (see @(see state)) are restored when changed during
 evaluation.  The value, @('val'), of a non-erroneous computed hint calculation
 is either @('nil'), which means the computed hint did not apply to the subgoal
 in question, or it is an alternating list of @(':keyi vali') pairs as
 specified in @(see hints).  With one exception, those new hints are applied to
 the given subgoal as though the user had typed them explicitly.</p>

 <p>The exception is that the first keyword in the returned @('val') is allowed
 to be @(':COMPUTED-HINT-REPLACEMENT').  Its value should be @('nil'), @('t'),
 or a list of terms.  If this keyword is not present, the default value of
 @('nil') is provided.  We explain @(':COMPUTED-HINT-REPLACEMENT') below.</p>

 <p>The evaluation of a hint term is done with guard checking turned off (see
 @(see set-guard-checking)); e.g., the form @('(car 23)') in a computed hint
 returns @('nil') as per the axioms.</p>

 <p>When a non-@('nil') value is returned, the keyword/value pairs (other than
 the optional @(':COMPUTED-HINT-REPLACEMENT')) are used as the hint for the
 subgoal in question.  Thus, your job as the programmer of computed hints is
 either to cause an error, typically by invoking @(tsee er), or to return a
 non-erroneous value triple whose value is the list of keys and values you
 would have typed had you supplied a common hint for the subgoal. (In
 particular, any theory expressions in it are evaluated with respect to the
 global current-theory, not whatever theory is active at the subgoal in
 question.)  If the generated list of keywords and values is illegal, an error
 will be signaled and the proof attempt will be aborted.</p>

 <p>The purpose of the @(':COMPUTED-HINT-REPLACEMENT') keyword and its value,
 @('chr'), is to change the list of hints.  If @('chr') is @('nil'), then the
 computed hint which was applied is removed from the list of hints that is
 passed down to the children of the subgoal in question.  This is the default.
 If @('chr') is @('t'), then the computed hint is left in the list of hints.
 This means that the same computed hint may act on the children of the subgoal.
 Otherwise, @('chr') must be a list of terms, each of which is treated as a
 computed hint.  The computed hint which was applied is deleted from the list
 of hints and the computed hints in @('chr') are added to the list of hints
 passed to the children of the subgoal.  The ability to generate new computed
 hints and pass them down allows strange and wonderful behavior.  Notice that
 certain hints, for example @(':in-theory') and @(':expand') hints, which
 appear in the computed hint will continue to take effect even when @('chr') is
 @('nil').  This might give one the false impression that a removed computed
 hint still hangs around.  See @(see hints-and-the-waterfall) for a more
 detailed discussion about how @(':in-theory') and other hints are handled in
 the waterfall.</p>

 <p>For these purposes, the goals produced by induction and the top-level goals
 of forcing rounds are not considered children; all original hints are
 available to them.</p>

 <p>Only the first hint applicable to a goal, as specified in the user-supplied
 list of @(':hints') followed by the @(tsee default-hints-table), will be
 applied to that goal.  (For an advanced exception, see @(see
 override-hints).)</p>

 <p>It remains only to describe the bindings of the free variables.</p>

 <p>Suppose the theorem prover is working on some clause, clause, named by some
 @(tsee goal-spec), e.g., \"Subgoal *1/2'''\" in some logical world, world.
 Corresponding to the printed @('goal-spec') is an internal data structure
 called a ``clause identifier'' id.  See @(see clause-identifier).</p>

 <p>In the case of a common hint, the hint applies if the goal-spec of the hint
 is the same as the goal-spec of the clause in question.</p>

 <p>In the case of a computed hint, the variable @('ID') is bound to the clause
 id, the variable @('CLAUSE') is bound to the (translated form of the) clause,
 and the variable @('WORLD') is bound to the current ACL2 world.  The variable
 @('STABLE-UNDER-SIMPLIFICATIONP') is bound to @('t') or @('nil').  It is bound
 to @('t') only if the clause is known to be stable under simplification.  That
 is, the simplifier has been applied to the clause and did not change it.  Such
 a clause is sometimes known as a ``simplification checkpoint.''  It is
 frequently useful to inject hints (e.g., to enable a rule or provide a
 @(':use') hint) only when the goal in question has stabilized.  If a hint is
 provided, the processing of the clause starts over with simplification.</p>

 <p>As for @('CTX') and @(tsee STATE), they are provided so that you can pass
 them to the @(tsee er) macro to print error messages.  We recommend not
 writing computed hints that otherwise change @(tsee STATE)!</p>

 <p>The remaining variables, @('HIST') and @('PSPV') are not documented yet.
 Only users familiar with the internals of ACL2 are likely to need them or
 understand their values.</p>

 <p>For some instruction about how to use computed hints, see @(see
 using-computed-hints).</p>")

(defxdoc concatenate
  :parents (lists strings acl2-built-ins)
  :short "Concatenate lists or strings together"
  :long "@({
  Examples:
  (concatenate 'string \"ab\" \"cd\" \"ef\")     ; equals \"abcdef\"
  (concatenate 'string \"ab\")               ; equals \"ab\"
  (concatenate 'list '(a b) '(c d) '(e f)) ; equals '(a b c d e f)
  (concatenate 'list)                      ; equals nil

  General Form:
  (concatenate result-type x1 x2 ... xn)
 })

 <p>where @('n >= 0') and either: @('result-type') is @(''')@(tsee string) and
 each @('xi') is a string; or @('result-type') is @(''')@(tsee list) and each
 @('xi') is a true list.  @('Concatenate') simply concatenates its arguments to
 form the result string or list.  Also see @(see append) and see @(see
 string-append).  (The latter immediately generates a call to @('concatenate')
 when applied to strings.)</p>

 <p>Note: We do *not* try to comply with the Lisp language's insistence that
 @('concatenate') copies its arguments.  Not only are we in an applicative
 setting, where this issue shouldn't matter for the logic, but also we do not
 actually modify the underlying lisp implementation of @('concatenate'); we
 merely provide a definition for it.</p>

 <p>@('Concatenate') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def concatenate)")

(defxdoc cond
  :parents (basics acl2-built-ins)
  :short "Conditional based on if-then-else"
  :long "<p>@('Cond') is the construct for IF, THEN, ELSE IF, ...  The test is
 against @('nil').  The argument list for @('cond') is a list of ``clauses'',
 each of which is a list.  In ACL2, clauses must have length 1 or 2.</p>

 @({
 ; Example 1.  The form
   (COND ((CONSP X) (FOO X Y))
         ((SYMBOLP X) (BAR X Y))
         (T (LIST X Y)))
 ; abbreviates the following.
   (IF (CONSP X)
       (FOO X Y)
       (IF (SYMBOLP X)
           (BAR X Y)
           (LIST X Y)))

 ; Example 2.  The form
   (COND ((CONSP X))
         ((SYMBOLP X) (BAR X Y)))
 ; abbreviates the following.
   (OR (CONSP X)
       (IF (SYMBOLP X) (BAR X Y) NIL))
 })

 <p>The results above were obtained by typing @(':trans1') followed by the form
 in the ACL2 loop, and then hitting @('<RETURN>').  See @(see trans1).  You can
 experiment in this way to see other such examples.</p>

 <p>@('Cond') is a Common Lisp macro.  See any Common Lisp documentation for
 more information.</p>

 @(def cond)
 @(def cond-macro)")

(defxdoc congruence
  :parents (rule-classes)
  :short "The relations to maintain while simplifying arguments"
  :long "<p>See @(see rule-classes) for a general discussion of rule classes
 and how they are used to build rules from formulas.  An example @(':')@(tsee
 corollary) formula from which a rule of class @(':congruence') might be built,
 assuming that @('set-equal') is a known @(see equivalence) relation, is:</p>

 @({
  Example:
  (defthm set-equal-implies-iff-memb-2
    (implies (set-equal x y)
             (iff (memb e x) (memb e y)))
    :rule-classes :congruence)
 })

 <p>Also see @(see defcong) and see @(see equivalence).</p>

 <p>NOTE: This topic discusses so-called ``classic'' congruence rules.  A more
 general class of rules, so-called ``patterned'' congruence rules, is
 supported.  We discuss only classic congruence rules below; for a discussion
 of patterned congruence rules, first read the present topic and then see @(see
 patterned-congruence).</p>

 @({
  General Form:
  (implies (equiv1 xk xk-equiv)
           (equiv2 (fn x1... xk       ...xn)
                   (fn x1... xk-equiv ...xn)))
 })

 <p>where @('equiv1') and @('equiv2') are known equivalence relations, @('fn')
 is an @('n-ary') function symbol other than @('if'), and the @('xi') and
 @('xk-equiv') are all distinct variables.  The effect of such a rule is to
 record that the @('equiv2')-equivalence of @('fn')-expressions can be
 maintained if, while rewriting the @('kth') argument position,
 @('equiv1')-equivalence is maintained.  See @(see equivalence) for a general
 discussion of the issues.  We say that @('equiv2'), above, is the ``outside
 equivalence'' in the rule and @('equiv1') is the ``inside equivalence for the
 @('k')th argument.''</p>

 <p>The macro form @('(defcong equiv1 equiv2 (fn x1 ... x1) k)') is an
 abbreviation for a @(tsee defthm) of rule-class @(':congruence') that attempts
 to establish that @('equiv2') is maintained by maintaining @('equiv1') in
 @('fn')'s @('k')th argument.  The @(tsee defcong) macro automatically
 generates the general formula shown above.  See @(see defcong).</p>

 <p>The @('memb') example above tells us that @('(memb e x)') is
 propositionally equivalent to @('(memb e y)'), provided @('x') and @('y') are
 @('set-equal').  The outside equivalence is @(tsee iff) and the inside
 equivalence for the second argument is @('set-equal').  If we see a @('memb')
 expression in a propositional context, e.g., as a literal of a clause or test
 of an @(tsee if) (but not, for example, as an argument to @(tsee cons)), we
 can rewrite its second argument maintaining @('set-equality').  For example, a
 rule stating the commutativity of @(tsee append) (modulo set-equality) could
 be applied in this context.  Since equality is a refinement of all equivalence
 relations, all equality rules are always available.  See @(see
 refinement).</p>

 <p>All known congruence rules about a given outside equivalence and @('fn')
 can be used independently.  That is, consider two congruence rules with the
 same outside equivalence, @('equiv'), and about the same function @('fn').
 Suppose one says that @('equiv1') is the inside equivalence for the first
 argument and the other says @('equiv2') is the inside equivalence for the
 second argument.  Then @('(fn a b)') is @('equiv') <tt>(fn a' b')</tt>
 provided @('a') is @('equiv1') to @('a'') and @('b') is @('equiv2') to
 @('b'').  This is an easy consequence of the transitivity of @('equiv').  It
 permits you to think independently about the inside equivalences.</p>

 <p>Furthermore, it is possible that more than one inside equivalence for a
 given argument slot will maintain a given outside equivalence.  For example,
 @('(length a)') is equal to <tt>(length a')</tt> if @('a') and @('a'') are
 related either by @('list-equal') or by @(tsee string-equal).  You may prove
 two (or more) congruence rules for the same slot of a function.  The result is
 that the system uses a new, ``generated'' equivalence relation for that slot
 with the result that rules of both (or all) kinds are available while
 rewriting.</p>

 <p>Congruence rules can be @(see disable)d.  For example, if you have two
 different inside equivalences for a given argument position and you find that
 the @(':')@(tsee rewrite) rules for one are unexpectedly preventing the
 application of the desired rule, you can disable the rule that introduced the
 unwanted inside equivalence.</p>

 <p><b>NOTE</b> however that unlike other rules, the tracking of congruence
 rules is incomplete.  Specifically: when congruence rules are used by the
 rewriter as it descends through terms, to maintain the generated equivalence
 relation used for rewriting, ACL2 does not track the congruence rules that are
 used, even though it is relevant that they are all @(see enable)d.  Congruence
 rules that are used only in this way will therefore not appear in the @(see
 summary).</p>

 <p><i>Remark on Replacing IFF by EQUAL.</i> You may encounter a warning
 suggesting that a congruence rule ``can be strengthened by replacing the
 second equivalence relation, IFF, by EQUAL.''  Suppose for example that this
 warning occurs when you submit the following rule:</p>

 @({
  (defcong equiv1 iff (fn x y) 2)
 })

 <p>which is shorthand for the following:</p>

 @({
  (defthm equiv1-implies-iff-fn-2
         (implies (equiv1 y y-equiv)
                  (iff (fn x y) (fn x y-equiv)))
         :rule-classes (:congruence))
 })

 <p>The warning is telling you that ACL2 was able to deduce that @('fn') always
 returns a Boolean, and hence a trivial but useful consequence is obtained by
 replacing @(tsee iff) by @(tsee equal) &mdash;</p>

 @({
  (defcong equiv1 equal (fn x y) 2)
 })

 <p>&mdash; which is shorthand for the following:</p>

 @({
  (defthm equiv1-implies-equal-fn-2
         (implies (equiv1 y y-equiv)
                  (equal (fn x y) (fn x y-equiv)))
         :rule-classes (:congruence))
 })

 <p>If you have difficulty proving the latter directly, you can derive it from
 the former by giving a suitable hint, minimally as follows.</p>

 @({
  (defcong equiv1 equal (fn x y) 2
    :hints ((\"Goal\"
             :use equiv1-implies-iff-fn-2
             :in-theory
             (union-theories '((:type-prescription fn))
                             (theory 'minimal-theory)))))
 })

 <p>By heeding this warning, you may avoid unnecessary @(tsee double-rewrite)
 warnings later.  We now explain why, but see @(see double-rewrite) for
 relevant background material.</p>

 <p>For example, suppose you have proved the ``@('iff')'' version of the
 congruence rule above, and later you submit the following rewrite rule.</p>

 @({
  (defthm equal-list-perm
    (implies (equiv1 x y)
             (fn x y)))
 })

 <p>Since @('fn') is known to return a Boolean, ACL2 performs an optimization
 that stores this rule as though it were the following.</p>

 @({
  (defthm equal-list-perm
    (implies (equiv1 x y)
             (equal (fn x y) t)))
 })

 <p>Thus, if ACL2's rewriter sees a term @('(fn a b)') in a context where the
 equivalence relation @(tsee iff) is not being maintained, then it cannot use
 rule @('equiv1-implies-iff-fn-2'), so it rewrites argument @('a') without the
 benefit of knowing that it suffices to maintain @('equiv1'); and then it
 caches the result.  When ACL2 subsequently attempts to relieve the hypothesis
 @('(equiv1 x y)'), it will rewrite @('x') simply by returning the rewritten
 value of @('a') from the result cache.  This is unfortunate if @('a') could
 have been rewritten more completely under maintenance of the equivalence
 relation @('equiv1') &mdash; which is legal in the hypothesis since @('a') is
 an argument of @('equiv1'), which is an @(see equivalence) relation.  The user
 who observes the warning from rule @('equiv1-implies-iff-fn-2'), and replaces
 it with @('equiv1-implies-equal-fn-2'), will avoid this unfortunate
 case.</p>")

(defxdoc conjugate
  :parents (numbers acl2-built-ins)
  :short "Complex number conjugate"
  :long "<p>@('Conjugate') takes an ACL2 number as an argument, and returns its
 complex conjugate (i.e., the result of negating its imaginary part.).</p>

 <p>@('Conjugate') is a Common Lisp function.  See any Common Lisp
 documentation for more information.</p>

 @(def conjugate)")

(defxdoc cons
  :parents (conses acl2-built-ins)
  :short "Pair and list constructor"
  :long "<p>@('(cons x y)') is a pair whose first component is @('x') and
 second component is @('y').  If @('y') is a list, then @('(cons x y)') is a
 list that has an additional element @('x') on the front.</p>")

(defxdoc cons-count-bounded
  :parents (conses acl2-built-ins)
  :short "Count the number of conses (up to a limit)"
  :long "<p>The call @('(cons-count-bounded x)') returns the number of cons
 nodes in @('x') (without accounting for sharing), but truncated above by the
 value of @('(fn-count-evg-max-val)'), which is 200,000 as of this
 writing.</p>")

(defxdoc cons-subtrees
  :parents (fast-alists acl2-built-ins)
  :short "Build a fast alist whose keys are the subtrees of X"
  :long "<p>@('(cons-subtrees x nil)') builds a fast alist that associates each
subtree of X with T, without duplication.</p>

 @(def cons-subtrees)")

(defxdoc cons-with-hint
  :parents (conses acl2-built-ins)
  :short "Alternative to @(tsee cons) that tries to avoid consing when a
 suitable @('cons') structure is provided as a hint."

  :long "<p>This is a special purpose function that is intended to help with
 reducing the memory usage of functions that modify existing cons tree
 structures.  Also see @(see hons) for a way to share @(see cons) structures;
 however, @('cons-with-hint') is likely much cheaper than @('hons') and hence
 can be useful for reducing consing without the overhead of @('hons').</p>

 <p>Logically @('(cons-with-hint x y hint)') is just @('(cons x y)'); @('hint')
 is completely irrelevant and ignored.  We generally expect that
 @('cons-with-hint') will just be left @(see enable)d, so you should never have
 to reason about it.</p>

 <p>But @('cons-with-hint') has a special raw Common Lisp definition that tries
 to avoid consing by using your @('hint').  Specifically: if @('hint') is the
 cons @('(x . y)'), then @('hint') is returned without creating a new cons.
 Equality checking against @('x') and @('y') is done using @(tsee eql), which
 makes it a fast but incomplete check for equality.</p>

 <p>What good is this?  A fairly common operation in ACL2 is to ``change'' an
 existing data structure by consing together a new structure that is similar to
 it, but perhaps with some subtrees replaced.  In many cases, some portion of
 the structure does not need to change.</p>

 <p>For instance, consider a function like @(tsee remove-equal), which updates
 a list by removing all copies of some element from it.  The definition of
 @('remove-equal') is as follows (in the logic; it has a slightly different
 definition in raw Lisp).</p>

 @(def remove-equal)

 <p>You can see that if @('l') doesn't have any copies of @('x'), this function
 will essentially make a fresh copy of the whole list @('x').  That could waste
 a lot of memory when @('x') is long.  The choice was made to define
 @('remove-equal') ``under the hood'' to call Common Lisp's function,
 @('remove'); but it is easy to write a new version of @('remove-equal') that
 uses @('cons-with-hint'):</p>

 @({
 (defun remove-equal-with-hint (x l)
   (declare (xargs :guard (true-listp l)))
   (mbe :logic (remove-equal x l)
        :exec (cond ((endp l) nil)
                    ((equal x (car l))
                     (remove-equal-with-hint x (cdr l)))
                    (t
                     (cons-with-hint (car l)
                                     (remove-equal-with-hint x (cdr l))
                                     l)))))
 })

 <p>This new version avoids consing in the case that we are not dropping an
 element.  For example, at the time of this writing, we found the following
 memory usages on our copy of ACL2 built on CCL:</p>

 @({
 :q

 ;; 16 MB of memory allocated
 (let ((list (make-list 1000 :initial-element 0)))
   (time (loop for i from 1 to 1000 do (remove-equal i list))))

 ;; 0 MB of memory allocated
 (let ((list (make-list 1000 :initial-element 0)))
   (time (loop for i from 1 to 1000 do (remove-equal-with-hint i list))))
 })

 <p>This memory usage is not very surprising when you consider that the list
 does not change when no removal takes place.  For example (still in raw
 Lisp):</p>

 @({
 ? (let ((x '(a b c d e))) (eq x (remove-equal-with-hint 3 x)))
 T
 ?
 })

 <p>Note that ACL2 asks Lisp to inline calls of @('cons-with-hint'), so there
 will likely be no function call overhead for using @('cons-with-hint').</p>")

(defxdoc conservativity-of-defchoose
  :parents (defchoose)
  :short "Proof of conservativity of @(tsee defchoose)"
  :long "<p>This documentation topic provides underlying theory.  It is of
 theoretical interest only; it has no relationship to the effective use of
 ACL2.</p>

 <p>The argument below for the conservativity of @(see defchoose) replaces the
 terse and somewhat misleading reference to a forcing argument in Appendix B of
 the paper by ACL2 authors Kaufmann and Moore, ``Structured Theory Development
 for a Mechanized Logic'' (Journal of Automated Reasoning 26, no. 2 (2001),
 pp. 161&ndash;203).</p>

 <p>Our basic idea is to take a (countable) first-order structure for ACL2,
 M, together with a function symbol, f, introduced by @(see defchoose), and
 find a way to expand M with an interpretation of f (without changing the
 universe of M) so that e0-induction continues to hold in the expansion.  A
 remark at the end of this documentation topic shows why care is necessary.  A
 concept called ``forcing'', originally introduced by Paul Cohen for set
 theory, has long since been adapted by logicians (in a simplified form) to
 model theory.  This simplified model-theoretic forcing provides the means for
 making our careful expansion.</p>

 <p>The forcing argument presented below is intended to be completely
 self-contained for those familiar with basic first-order logic and ACL2; in
 particular, see @(see defchoose).  No background in forcing (model-theoretic
 or otherwise) is expected, though we do expect a rudimentary background in
 first-order logic and familiarity with the following.</p>

 <p>Preliminaries.  We write s[p&lt;-p0] to denote the result of extending or
 modifying the assignment s by binding p to p0.  Now let A be a subset of the
 universe U of a first-order structure M.  A is said to be ``first-order
 definable with parameters'' in M if for some formula phi, variable x, and
 assignment s binding the free variables of phi except perhaps for x, A = {a
 \\in U: M |= phi[s[x&lt;-a]]}.  Note that we are writing ``\\in'' to denote
 set membership.  Finally, we indicate the end of a proof (or of a theorem
 statement, when the proof is omitted) with the symbol ``-|''.</p>

 <p>We gratefully acknowledge very helpful feedback from John Cowles, who found
 several errors in a draft of this note and suggested the exercises.  We also
 thank Ruben Gamboa for helpful feedback, and we thank Jim Schmerl for an
 observation that led us directly to this proof in the first place.</p>

 <p>We are given a consistent first-order theory T, extending the ACL2
 ground-zero theory, that satisfies the e0-induction scheme.  We wish to show
 that the extension of T by the following arbitrary defchoose event is
 conservative, where g is a new function symbol.</p>

 @({
       (defchoose g <bound-vars> <free-vars> <body>)
 })

 <p>Note that by ``the extension of T'' here we mean the extension of T by not
 only the new defchoose axiom displayed just below, but also the addition of
 e0-induction axioms for formulas in the expanded language obtained by
 including the new defchoose function symbol, g.</p>

 @({
       <body> -> (LET <bound-vars> = g(<free-vars>) in <body>)
 })

 <p>By definition of conservativity, since proofs are finite, it clearly
 suffices to consider an arbitrary finite subset of T.  Then by the
 completeness, soundness, and downward Lowenheim-Skolem theorems of first-order
 logic, it suffices to show that an arbitrary countable model of T can be
 expanded (i.e., by interpreting the new symbol g without changing the universe
 of the model) to a model of the corresponding defchoose axiom above, in which
 all e0-induction axioms hold in the language of that model (i.e., the expanded
 language).</p>

 <p>Below, we will carry out a so-called <i>forcing</i> construction, which
 allows us to expand any countable model M of T to a model M[G] for the
 expanded language that satisfies e0-induction (in the expanded language) and
 also satisfies the axiom displayed above, generated from the defchoose event.
 The ideas in this argument are standard in model theory; no novelty is claimed
 here.</p>

 <p>Fix a countable model M of a theory T that satisfies e0-induction for its
 countable language and extends the ACL2 ground-zero theory.  Also fix the
 above defchoose axiom, where g is not in the language of T.</p>

 <p>We start by defining a partial order P as follows.  Let Nb and Nf be the
 lengths of &lt;bound-vars&gt; and &lt;free-vars&gt;, respectively.  P consists
 of all fn in M such that the following formula is true in M.  Roughly
 speaking, it says that fn is a finite function that witnesses, on its domain,
 the requirement above for g.</p>

 @({
         alistp(fn) &
         no-duplicatesp-equal(strip-cars(fn)) &
         (forall <bound-vars>, <free-vars> .
            (member-equal(cons(<free-vars>,<bound-vars>), fn) ->
             (length(<bound-vars>) = Nb &
              length(<free-vars>)  = Nf &
              ((exists <bound-vars> . <body>) ->
               (LET <bound-vars> = g(<free-vars>) in <body>)))))
 })

 <p>P is ordered by subset, i.e., we say that p2 <i>extends</i> p1 if p1 is a
 subset (not necessarily proper) of p2 (more precisely, M |=
 subsetp-equal(p1,p2)).</p>

 <p>Remark.  The original argument in Appendix B of the aforementioned paper
 can essentially be salvaged, as we now show.  The key observation is that the
 particular choice of P is nearly irrelevant for the argument that follows
 below.  In particular, we can instead define P to consist of finite one-one
 functions with domain contained in the set of natural numbers.  More
 precisely, consider the following definitions.</p>

 @({
       (defun function-p (fn)
         (and (alistp fn)
              (no-duplicatesp-equal (strip-cars fn))))

       (defun nat-function-p (x)
         (and (function-p x)
              (nat-listp (strip-cars x))))
 })

 <p>and define an inverse function on alists as follows.</p>

 @({
       (defun inverse (fn)
         (if (endp fn)
             nil
           (cons (cons (cdar fn) (caar fn))
                 (inverse (cdr fn)))))
 })

 <p>Then P may instead be defined to consist of those fn for which
 nat-function-p(fn) &amp; function-p(inverse(fn)).  With this alternate
 definition of P, the argument below then goes through virtually unchanged, and
 we get an expansion M[G] of M in which there is a definable enumeration of the
 universe.  The conservativity of defchoose then follows easily because the
 function being introduced can be defined explicitly using that enumeration
 (namely, always pick the least witness in the sense of the enumeration).</p>

 <p>End of Remark.</p>

 <p>Next we present the relevant forcing concepts from model theory.</p>

 <p>A <i>dense</i> subset of P is a subset D of P such that for every p \\in P,
 there is d \\in D such that d extends p.  A subset G of P is <i>generic</i>
 with respect to a collection Ds of dense subsets of P, also written ``G is
 Ds-generic,'' if G is closed under subset (if p2 \\in G and p2 extends p1 then
 p1 \\in G), G is pairwise compatible (the union-equal of any two elements of G
 is in G), and every set in Ds has non-empty intersection with G.</p>

 <p>For p \\in P, we say that a subset D of P is <i>dense beyond</i> p if for
 all p1 extending p there exists p2 extending p1 such that p2 \\in D.  This
 notion makes sense even for D not a subset of P if we treat elements of D not
 in P as nil.</p>

 <p>Proposition 1.  For any partial order P and countable collection Ds of
 dense subsets of P, there is a Ds-generic subset of P.</p>

 <p>Proof.  Let Ds = {D0,D1,D2,...}.  Define a sequence &lt;p_0,p_1,...&gt;
 such that for all i, p_i \\in Di and p_(i+1) extends p_i.  Let G = {p \\in P:
 for some i, pi extends p}.  Then G is Ds-generic. -|</p>

 <p>Note that P is first-order definable (with parameters) in M.  Let Df be the
 set of dense subsets of P that are first-order definable (with parameters) in
 M.  A standard argument shows there are only countably many first-order
 definitions with parameters in a countable model M whose language is countable
 &mdash; for example, we can Goedel number all terms and then all formulas
 &mdash; hence, Df is countable.</p>

 <p>By Proposition 1, let G be Df-generic.  Notice that for any list x of
 length Nf in M, the set of elements f of P for which x is in the domain of f
 is dense and first-order definable.  We may thus define a function g0 as
 follows: g0(x_1,...,x_Nf) = y if there is some element of G containing the
 pair ((x_1 ... x_Nf) . y).  It is easy to see that g0 is a total function on
 M.  Let L be the language of T and let L[g] be the union of L with a set
 containing a single new function symbol, g.  Let M[G] be the expansion of M to
 L[g] obtained by interpreting g to be g0 (see also Proposition 5 below).</p>

 <p>So now we have fixed M, P, Df, G, and g0, where G is Df-generic.</p>

 <p>Proposition 2.  Let Df be the set of dense subsets of P that are
 first-order definable (with parameters) in M.  Suppose that p \\in G and D
 \\in Df.  Then for some q \\in G extending p, q \\in D.</p>

 <p>Proof.  Let D0 be the set of p' \\in D that either extend p or have no
 extension in D that extends p.  We leave it as a straightforward exercise to
 show that D0 is dense, and D0 is clearly first-order definable (with
 parameters) in M.  So by genericity of G, we may pick q \\in D0 such that q
 \\in G.  Thus q \\in D.  By definition of generic, some extension q1 of both p
 and q belongs to G.  Pick q2 \\in D extending q1; thus q has an extension in D
 that extends p (namely, q2), so by definition of D0, q extends p. -|</p>

 <p>Definition of forcing.  Let phi(x1,...,xk) be a first-order formula in L[g]
 and let p \\in P.  We define a formula of L, denoted ``p ||- phi'' (``p forces
 phi''), by recursion on phi (in the metatheory) as follows.  (Here, we view
 ``or'' and ``forall'' as abbreviations.)</p>

 <blockquote><p>If phi is atomic, then let phi'(A) be the result of replacing,
   inside-out, each subterm of the form g(x_1,...,x_Nb) with the term (cdr
   (assoc-equal (list x_1 ... x_Nb) A)), where A is neither p nor a variable
   occurring in phi.  Then p ||- phi is defined as follows: ``The set {A \\in
   P: A extends p and phi'(A)} is dense beyond p''.  That is, p ||- phi is the
   following formula:</p>

 @({
      (forall p1 \\in P extending p)
       (exists p2 \\in P extending p1) phi'(p2).
 })

 <p>p ||- ~phi is: (forall p' \\in P extending p) ~(p' ||- phi)</p>

 <p>p ||- phi_1 &amp; phi_2 is: (p ||- phi_1) &amp; (p ||- phi_2)</p>

 <p>p ||- (exists x) phi is: (exists x) (p ||- phi)</p></blockquote>

 <p>We will need the following definition later.</p>

 <p>Definition.  p ||-w phi (p <i>weakly forces</i> phi) is an abbreviation for
 p ||- ~~phi.</p>

 <p>The following exercises were suggested by John Cowles as a means for
 gaining familiarity with the definition of forcing.</p>

 <p>Exercise 1. Consider the formula (phi_1 OR phi_2) as an abbreviation for
 ~(~phi_1 &amp; ~phi_2), Show that p ||- (phi_1 OR phi_2) is equivalent to the
 following.</p>

 @({
       (forall p' \\in P extending p) (exists p'' \\in P extending p')
        ((p'' ||- phi_1) OR (p'' ||- phi_2))
 })

 <p>Exercise 2. Consider the formula (forall x)phi as an abbreviation for
 ~(exists x)~phi, Show that p ||- (forall x)phi is equivalent to the
 following.</p>

 @({
       (forall x)
        (forall p1 \\in P extending p)
         (exists p2 \\in P extending p1) (p2 ||- phi).
 })

 <p>Exercise 3. Prove that p ||-w phi is equivalent to the following.</p>

 @({
       (forall p' \\in P extending p)
        (exists p'' \\in P extending p') (p'' ||- phi).
 })

 <p>Exercise 4. Let phi be a formula of L[g].  Prove: M |= (p ||-
      phi)[s[p&lt;-p0]] implies M |= (p ||-w phi)[s[p&lt;-p0]].</p>

 <p>Exercise 5. Let phi be a formula of L[g].  Prove: M |= (p ||-
      ~phi)[s[p&lt;-p0]] iff M |= (p ||-w ~phi)[s[p&lt;-p0]].</p>

 <p>[End of exercises.]</p>

 <p>The definition of forcing stipulates how to view ``p ||- phi(x1,...,xk)''
 as a new formula theta(p,x1,...,xk).  That is, ``||-'' transforms formulas, so
 for any first-order formula phi, ``p ||- phi'' is just another first-order
 formula.  That observation shows that a formula such as ((p ||- phi) OR (p ||-
 ~phi)) is really just another first-order formula.  The following proposition
 thus follows easily.</p>

 <p>Proposition 3. For any formula phi of L[g], {p0: M |= ((p ||- phi) OR (p
 ||- ~phi))[s[p&lt;-p0]]]} is a dense subset of P, which (since it is
 first-order definable with parameters in M) intersects G. -|</p>

 <p>The following proposition is easily proved by a structural induction on
 phi, and is left to the reader.</p>

 <p>Proposition 4. Let phi be a formula of L[g].  Suppose p0 \\in P, p1
 \\in P,<br/>

 M |= (p ||- phi)[s[p&lt;-p0]],<br/>

 and p1 extends p0.  Then<br/>

 M |= (p ||- phi)[s[p&lt;-p1]]. -|</p>

 <p>We will also need the following.</p>

 <p>Proposition 5. The following is dense for any finite set S of Nf-tuples: {p
 \\in P: for some &lt;x_1 ... x_Nf&gt; \\in S, (list x_1 ... x_Nf) \\in
 strip-cars(p)}.  Thus, the function g0 is a total function. -|</p>

 <p>The next lemma tells us that the sentences true in M[G] are those that are
 forced by an element of G.</p>

 <p>Truth Lemma.  Let phi be a formula in L[g], let s be an assignment to the
 free variables of phi, and let p be a variable not in the domain of s.  Then
 M[G] |= phi[s] iff for some p0 \\in G, M |= (p ||- phi)[s[p&lt;-p0]].</p>

 <p>Proof.  The proof is by induction on the structure of phi.  First suppose
 phi is atomic.  Let D* be the set of elements p0 \\in P such that every
 assoc-equal evaluation from the definition of forcing phi returns a pair when
 A is bound to p0.  (Intuitively, this means that p0 is a sufficiently large
 approximation from any G containing p0 to make sense of phi in M[G].)  We make
 the following claim.</p>

 @({
  (*)   For all p0 \\in G such that p0 \\in D*,
        M[G] |= phi[s] iff M |= (p ||- phi)[s[p<-p0]].
 })

 <p>To prove the claim, fix p0 in both G and D*, and recall the function g0
 constructed from G in the definition of M[G].  Suppose that t_1, ..., t_Nf are
 terms and g(t_1, ..., t_Nf) is a subterm of phi.  Then s assigns a value in M
 to each of the t_i.  Let a_i be the value assigned by s to t_i.  Then g0(a_1,
 ..., a_Nf) = (cdr (assoc-equal (list a_1 ... a_Nf) p0)), as the assoc-equal is
 a pair (since p0 \\in D*) and has the indicated value (because p0 \\in G).  It
 follows by the definition of formula phi' in the definition of forcing:</p>

 @({
       M[G] |= phi[s]  iff  M |= phi'(p)[s[p<-p0]]
 })

 <p>Moreover, because p0 \\in D* it is clear that this holds if p0 is replaced
 by an arbitrary extension of p0.  Then (*) easily follows.</p>

 <p>By Proposition 5, D* is dense, so there is some p0 in the intersection of
 D* and G.  The forward direction of the conclusion then follows by (*).  The
 reverse direction is clear from (*) by application of Proposition 2 to D* and
 Proposition 4.</p>

 <p>Next, suppose M[G] |= ~phi[x].  Then it is not the case that M[G] |= phi,
 so by the inductive hypothesis, there is no p0 \\in G for which M |= (p ||-
 phi)[s[p&lt;-p0]].  By Proposition 3, there is p0 \\in G for which M |= (p ||-
 ~phi)[s[p&lt;-p0]].  For the other direction, suppose it is not the case that
 M[G] |= ~phi[s].  So M[G] |= phi[s], and by the inductive hypothesis, there is
 p0 \\in G for which M |= (p ||- phi)[s[p&lt;-p0]].  It follows that there is
 no p1 \\in G for which M |= (p ||- ~phi)[s[p&lt;-p1]], since from such p1 we
 can find a common extension p2 of p0 and p1 (since G is generic), and since p2
 extends p0 then by Proposition 4, M |= (p ||- phi)[s[p&lt;-p2]], contradicting
 (by definition of forcing) M |= (p ||- ~phi)[s[p&lt;-p1]] since p2 extends
 p1.</p>

 <p>The case (phi_1 &amp; phi_2) follows easily from the inductive hypothesis.
 For the forward direction, apply Proposition 4 and the observation that by
 genericity, if p0 \\in G and p1 \\in G then p0 and p1 they have a common
 extension in G.</p>

 <p>Finally, the case (exists x) phi follows trivially from the inductive
 hypothesis. -|</p>

 <p>Truth Lemma Corollary.  The Truth Lemma holds with ||-w replacing ||-.</p>

 <p>Proof.  This is clear by applying the Truth Lemma to ~~phi. -|</p>

 <p>Here is our main theorem.  Recall that all first-order theories in our ACL2
 context satisfy the e0-induction scheme.</p>

 <p>Theorem.  M[G] satisfies e0-induction.</p>

 <p>Proof.  We consider an arbitrary instance of e0-induction in L[g], stated
 using a strict well-founded relation &lt;| and a formula phi.  We write phi(y)
 to indicate that y may be among the free variables of phi, and phi(y&lt;-x) to
 denote the result of substituting x for y in phi.</p>

 @({
    theta:   (forall y) [((forall x <| y) phi(y<-x)) -> phi(y)]
          -> (forall y) phi(y)
 })

 <p>Our goal is to prove that theta holds in M[G].</p>

 <p>Below, we abuse notation by leaving assignments implicit and by writing ``p
 ||- phi(y0)'' to signify that the formula (p ||- phi(y)) is true in M under
 the extension of the explicit assignment that binds y to y0.  We believe that
 the intended meaning will be clear.</p>

 <p>Consider the following set D.</p>

 @({
    D = {p \\in P: either p ||-w phi(y0) for all y0,
                  or else
                  for some y0, p ||- ~phi(y0) and
                               for all y1 <| y0 p ||-w phi(y1)}.
 })

 <p>The set D is clearly first-order definable (with parameters) in M.  We
 claim that D is a dense subset of P.  For suppose p0 \\in P; we find p1 \\in D
 extending p0, as follows.  If p0 ||-w phi(y0) for all y0, then we may take p1
 to be p0.  Otherwise, by definition of ||-w and ||-, there is some y0 such
 that for some extension p0' of p0, p0' ||- ~phi(y0).  Pick a &lt;|-minimal
 such y0, and correspondingly pick p1 so that p1 extends p0 and p1 ||-
 ~phi(y0).  In order to show that p1 \\in D, it remains to show that for all y1
 &lt;| y0, p1 ||-w phi(y1), i.e., there is no q extending p1 such that q ||-
 ~phi(y1).  This is indeed the case since otherwise q and y1 would contradict
 the &lt;|-minimality of y0.</p>

 <p>Applying the genericity of G and just-proved density of D, pick p0 \\in G
 such that p0 \\in D.  If p0 ||-w phi(y0) for all y0, then by the Truth Lemma
 Corollary, M[G] |= phi(y0) for all y0, and thus M[G] |= theta.  Otherwise,
 since p0 \\in D we may choose y0 such that p0 ||- ~phi(y0) and for all y1
 &lt;| y0, p0 ||-w phi(y1).  By the Truth Lemma and its corollary, since p0
 \\in G we have:</p>

 @({
  (1)   M[G] |= ~phi(y0).
  (2)   For all y1 <| y0, M[G] |= phi(y1).
 })

 <p>It follows that the antecedent of theta is false in M[G], as witnessed by y
 = y0; thus M[G] |= theta. -|</p>

 <p>Remark.  We close by returning, as promised above, to the question of why
 so much care is necessary in constructing an expansion of M.  We assume
 familiarity here with the notion of a ``non-standard'' natural number of M,
 i.e., one that is greater than the interpretation of any term that has the
 form (+ 1 1 1 ... 1).  Here is a very simple example that illustrates the need
 for some care.  Consider the following event, which introduces a function foo
 with the following property: for all x, if natp(x) then natp(foo(x)).</p>

 @({
       (defchoose foo (y) (x)
         (implies (natp x) (natp y)))
 })

 <p>Certainly we can build a model of the above property from a model M of the
 ground-zero theory, by interpreting foo so that for all x for which M
 satisfies natp(x), foo(x) is also a natp in M.  But suppose we start with a
 non-standard model M of the ground-zero theory, and we happen to define foo(x)
 to be 1 for all non-standard natural numbers x and 0 for all other x.  The
 resulting expansion of M will not satisfy the e0-induction scheme or even the
 ordinary natural number induction scheme: foo(0)=0 holds in that expansion as
 does the implication foo(n)=0 =&gt; foo(n+1)=0 for every natural number n of
 M, standard or not; and yet foo(k)=0 fails for every non-standard natural
 number k of M.</p>")

(defxdoc conses
  :parents (programming)
  :short "A <b>cons</b> is an ordered pair.  In ACL2, data structures like
 @(see lists), @(see alists), etc., are made up of conses.")

(defxdoc consp
  :parents (conses acl2-built-ins)
  :short "Recognizer for @(see cons) pairs"
  :long "<p>@('(consp x)') is true if and only if @('x') is a @(see cons)
 pair.</p>")

(defxdoc constraint
  :parents (encapsulate)
  :short "Restrictions on certain functions introduced in @(tsee encapsulate) @(see events)"
  :long "<p>Suppose that a given theorem, @('thm'), is to be functionally
 instantiated using a given functional substitution, @('alist').  (See @(see
 lemma-instance), or for an example, see @(see
 functional-instantiation-example).)  What is the set of proof obligations
 generated?  It is the set obtained by applying @('alist') to all terms,
 @('tm'), such that (a) @('tm') mentions some function symbol in the domain of
 @('alist'), and (b) either (i) @('tm') is the ``constraint'' (see below) on a
 function symbol ancestral in @('thm') or in some @(tsee defaxiom) or (ii)
 @('tm') is the body of a @(tsee defaxiom).  Here, a function symbol is
 ``ancestral'' in @('thm') if either it occurs in @('thm'), or it occurs in the
 definition of some function symbol that occurs in @('thm'), and so on.</p>

 <p>The remainder of this note explains what we mean by ``constraint'' in the
 words above.  For a utility that obtains the constraint for a given function
 symbol, see @(see constraint-info).</p>

 <p>In a certain sense, function symbols are introduced in essentially two
 ways.  The most common way is to use @(tsee defun) (or when there is mutual
 recursion, @(tsee mutual-recursion) or @(tsee defuns)).  There is also a
 mechanism for introducing ``witness functions''; see @(see defchoose).  The
 documentation for these @(see events) describes the axioms they introduce,
 which we will call here their ``definitional axioms.''  These definitional
 axioms are generally the constraints on the function symbols that these axioms
 introduce.</p>

 <p>However, when a function symbol is introduced in the scope of an @(tsee
 encapsulate) event, its constraints may differ from the definitional axioms
 introduced for it.  For example, suppose that a function's definition is
 @(tsee local) to the @(tsee encapsulate); that is, suppose the function is
 introduced in the @(see signature) of the @(tsee encapsulate).  Then its
 constraints include, at the least, those non-@(tsee local) theorems and
 definitions in the @(tsee encapsulate) that mention the function symbol.</p>

 <p>Actually, it will follow from the discussion below that if the @(see
 signature) is empty for an @(tsee encapsulate), then the constraint on each of
 its new function symbols is exactly the definitional axiom introduced for it.
 Intuitively, we view such @('encapsulates') just as we view @(tsee
 include-book) @(see events).  But the general case, where the @(see signature)
 is not empty, is more complicated.</p>

 <p>In the discussion that follows we describe in detail exactly which
 constraints are associated with which function symbols that are introduced in
 the scope of an @(tsee encapsulate) event.  In order to simplify the
 exposition we make two cuts at it.  In the first cut we present an
 over-simplified explanation that nevertheless captures the main ideas.  In the
 second cut we complete our explanation by explaining how we view certain @(see
 events) as being ``lifted'' out of the @(tsee encapsulate), resulting in a
 possibly smaller @(tsee encapsulate), which becomes the target of the
 algorithm described in the first cut.</p>

 <p>At the end of this note we present an example showing why a more naive
 approach is unsound.</p>

 <p>Finally, before we start our ``first cut,'' we note that any information
 you want ``exported'' outside an @(tsee encapsulate) event must be there as an
 explicit definition or theorem.  For example, even if a function @('foo') has
 output type @('(mv t t)') in its @(see signature), the system will not know
 @('(true-listp (foo x))') merely on account of this information.  Thus, if you
 are using functions like @('foo') (constrained @(tsee mv) functions), then you
 may find it useful to prove (inside the @('encapsulate'), to be exported) a
 @(':')@(tsee type-prescription) rule for the constrained function, for
 example, the @(':')@(tsee type-prescription) rule @('(true-listp (foo
 x))').</p>

 <p><i>First cut at constraint-assigning algorithm.</i> Consider the set of
 formulas introduced by all @(see events) in the scope of an @(tsee
 encapsulate):</p>

 <ul>

 <li>for every @(tsee defthm) and @(tsee defaxiom) event, include its formula
 as well as the formulas of all its corollaries;</li>

 <li>for every @(tsee defun) event, include the formula that equates the body
 with the application of the function applied to the formals;</li>

 <li>for every @('defchoose') event, include the axiom that it adds (see @(see
 defchoose));</li>

 <li>and so on, including axioms added by @(tsee defpkg) and (recursively) any
 subsidiary @('encapsulate') events.</li>

 </ul>

 <p>Then quite simply, all such formulas are conjoined, and each function
 symbol introduced by the @(tsee encapsulate) is assigned that conjunction as
 its constraint.</p>

 <p>Clearly this is a rather severe algorithm.  Let us consider two possible
 optimizations in an informal manner before presenting our second cut.</p>

 <p>Consider the (rather artificial) event below.  The function @('before1')
 does not refer at all, even indirectly, to the locally-introduced function
 @('sig-fn'), so it is unfortunate to saddle it with constraints about
 @('sig-fn').</p>

 @({
  (encapsulate
   (((sig-fn *) => *))

   (defun before1 (x)
     (if (consp x)
         (before1 (cdr x))
       x))

   (local (defun sig-fn (x) (cons x x)))

   (defthm sig-fn-prop
     (consp (sig-fn x)))
   )
 })

 <p>We would like to imagine moving the definition of @('before1') to just in
 front of this @(tsee encapsulate), as follows.</p>

 @({
  (defun before1 (x)
    (if (consp x)
        (before1 (cdr x))
      x))

  (encapsulate
   (((sig-fn *) => *))

   (local (defun sig-fn (x) (cons x x)))

   (defthm sig-fn-prop
     (consp (sig-fn x)))
   )
 })

 <p>Thus, we will only assign the constraint @('(consp (sig-fn x))'), from the
 theorem @('sig-fn-prop'), to the function @('sig-fn'), not to the function
 @('before1').</p>

 <p>More generally, suppose an event in an @(tsee encapsulate) event does not
 mention any function symbol in the @(see signature) of the @(tsee
 encapsulate), nor any function symbol that mentions any such function symbol,
 and so on.  (We might say that no function symbol from the @(see signature) is
 an ``ancestor'' of any function symbol occurring in the event.)  Then we
 imagine moving the event, so that it appears in front of the @(tsee
 encapsulate).  We don't actually move it, but we pretend we do when it comes
 time to assign constraints.  Thus, such definitions only introduce
 definitional axioms as the constraints on the function symbols being defined.
 In the example above, the event @('sig-fn-prop') introduces no constraints on
 function @('before1').</p>

 <p>Once this first optimization is performed, we have in mind a set of
 ``constrained functions.''  These are the functions introduced in the @(tsee
 encapsulate) that would remain after moving some of them in front, as
 indicated above.  Consider the collection of all formulas introduced by the
 @(tsee encapsulate), except the definitional axioms, that mention these
 constrained functions.  So for example, in the event below, no such formula
 mentions the function symbol @('after1').</p>

 @({
  (encapsulate
   (((sig-fn *) => *))

   (local (defun sig-fn (x) (cons x x)))

   (defthm sig-fn-prop
     (consp (sig-fn x)))

   (defun after1 (x)
     (sig-fn x))
   )
 })

 <p>We can see that there is really no harm in imagining that we move the
 definition of @('after1') out of the @(tsee encapsulate), to just after the
 @(tsee encapsulate).</p>

 <p>Many subtle aspects of this rearrangement process have been omitted.  For
 example, suppose the function @('fn') uses @('sig-fn'), the latter being a
 function in the signature of the encapsulation.  Suppose a formula about
 @('fn') is proved in the encapsulation.  Then from the discussion above
 @('fn') is among the constrained functions of the encapsulate: it cannot be
 moved before the encapsulate and it cannot be moved after the encapsulation.
 But why is @('fn') constrained?  The reason is that the theorem proved about
 @('fn') may impose or express constraints on @('sig-fn').  That is, the
 theorem proved about @('fn') may depend upon properties of the witness used
 for @('sig-fn').  Here is a simple example:</p>

 @({
  (encapsulate
   (((sig-fn *) => *))

   (local (defun sig-fn (x) (declare (ignore x)) 0))

   (defun fn (lst)
     (if (endp lst)
         t
         (and (integerp (sig-fn (car lst)))
              (fn (cdr lst)))))

   (defthm fn-always-true
     (fn lst)))
 })

 <p>In this example, there are no @(tsee defthm) events that mention
 @('sig-fn') explicitly.  One might therefore conclude that it is completely
 unconstrained.  But the witness we chose for it always returns an integer.
 The function @('fn') uses @('sig-fn') and we prove that @('fn') always returns
 true.  Of course, the proof of this theorem depends upon the properties of the
 witness for @('sig-fn'), even though those properties were not explicitly
 ``called out'' in theorems proved about @('sig-fn').  It would be unsound to
 move @('fn-always-true') after the encapsulate.  It would also be unsound to
 constrain @('sig-fn') to satisfy just @('fn-always-true') without including in
 the constraint the relation between @('sig-fn') and @('fn').  Hence both
 @('sig-fn') and @('fn') are constrained by this encapsulation and the
 constraint imposed on each is the same and states the relation between the two
 as characterized by the equation defining @('fn') as well as the property that
 @('fn') always returns true.  Suppose, later, one proved a theorem about
 @('sig-fn') and wished to functionally instantiate it.  Then one must also
 functionally instantiate @('fn'), even if it is not involved in the theorem,
 because it is only through @('fn') that @('sig-fn') inherits its constrained
 properties.</p>

 <p>This is a pathological example that illustrate a trap into which one may
 easily fall: rather than identify the key properties of the constrained
 function the user has foreshadowed its intended application and constrained
 those notions.  Clearly, the user wishing to introduce the @('sig-fn') above
 would be well-advised to use the following instead:</p>

 @({
  (encapsulate
   (((sig-fn *) => *))
   (local (defun sig-fn (x) (declare (ignore x)) 0))
   (defthm integerp-sig-fn
     (integerp (sig-fn x))))

  (defun fn (lst)
    (if (endp lst)
        t
      (and (integerp (sig-fn (car lst)))
           (fn (cdr lst)))))

  (defthm fn-always-true
     (fn lst)))
 })

 <p>Note that @('sig-fn') is constrained merely to be an integer.  It is the
 only constrained function.  Now @('fn') is introduced after the encapsulation,
 as a simple function that uses @('sig-fn').  We prove that @('fn') always
 returns true, but this fact does not constrain @('sig-fn').  Future uses of
 @('sig-fn') do not have to consider @('fn') at all.</p>

 <p>Sometimes it is necessary to introduce a function such as @('fn') within
 the @('encapsulate') merely to state the key properties of the undefined
 function @('sig-fn').  But that is unusual and the user should understand that
 both functions are being constrained.</p>

 <p>Another subtle aspect of encapsulation that has been brushed over so far
 has to do with exactly how functions defined within the encapsulation use the
 signature functions.  For example, above we say ``Consider the collection of
 all formulas introduced by the encapsulate, <i>except the definitional
 axioms</i>, that mention these constrained functions.''  We seem to suggest
 that a definitional axiom which mentions a constrained function can be moved
 out of the encapsulation and considered part of the ``post-encapsulation''
 extension of the logical @(see world), if the defined function is not used in
 any non-definitional formula proved in the encapsulation.  For example, in the
 encapsulation above that constrained @('sig-fn') and introduced @('fn') within
 the encapsulation, @('fn') was constrained because we proved the formula
 @('fn-always-true') within the encapsulation.  Had we not proved
 @('fn-always-true') within the encapsulation, @('fn') could have been moved
 after the encapsulation.  But this suggests an unsound rule because whether
 such a function can be moved after the encapsulate depend on whether its
 <i>admission</i> used properties of the witnesses!  In particular, we say a
 function is ``subversive'' if any of its governing tests or the actuals in any
 recursive call involve a function in which the signature functions are
 ancestral.  See @(see infected-constraints) and see @(see
 subversive-recursions).</p>

 <p>(Aside: The definition of @('fn') in the first encapsulation above that
 defines @('fn'), i.e., the encapsulation with @('fn-always-true') inside, is
 subversive because the call of the macro @(tsee AND) expands to a call of
 @('IF') that governs a recursive call of @('fn'), in this case:</p>

 @({
  (defun fn (lst)
    (if (endp lst)
        t
        (if (integerp (sig-fn (car lst)))
            (fn (cdr lst))
          nil))).
 })

 <p>If we switch the order of conjuncts in @('fn'), then the definition of
 @('fn') is no longer subversive, but it still ``infects'' the constraint
 generated for the encapsulation, hence for @('sig-fn'), because
 @('fn-always-true') blocks the definition of @('fn') from being moved back (to
 after the encapsulation).  Also see @(see infected-constraints).  If we both
 switch the order of conjuncts and drop @('fn-always-true') from the
 encapsulation, then the definition of @('fn') is in essence moved back to
 after the encapsulation, and the constraint for @('sig-fn') no longer includes
 the definition of @('fn').  End of aside.)</p>

 <p>Another aspect we have not discussed is what happens to nested
 encapsulations when each introduces constrained functions.  We say an
 @('encapsulate') event is ``trivial'' if it introduces no constrained
 functions, i.e., if its signatures is @('nil').  Trivial encapsulations are
 just a way to wrap up a collection of events into a single event.</p>

 <p>From the foregoing discussion we see we are interested in exactly how we
 can ``rearrange'' the events in a non-trivial encapsulation &mdash; moving
 some ``before'' the encapsulation and others ``after'' the encapsulation.  We
 are also interested in which functions introduced by the encapsulation are
 ``constrained'' and what the ``constraints'' on each are.  We may summarize
 the observations above as follows, after which we conclude with a more
 elaborate example.</p>

 <p><i>Second cut at constraint-assigning algorithm.</i> First, we focus only
 on non-trivial encapsulations that neither contain nor are contained in
 non-trivial encapsulations.  (Nested non-trivial encapsulations are not
 rearranged at all: do not put anything in such a nest unless you mean for it
 to become part of the constraints generated.)  Second, in what follows we only
 consider the non-@('local') events of such an @('encapsulate'), assuming that
 they satisfy the restriction of using no locally defined function symbols
 other than the signature functions.  Given such an @('encapsulate') event,
 move, to just in front of it and in the same order, all definitions and
 theorems for which none of the signature functions is ancestral.  Now collect
 up all formulas (theorems) introduced in the @(tsee encapsulate) other than
 definitional axioms.  Add to this set any of those definitional equations that
 is either subversive or defines a function used in a formula in the set.  The
 conjunction of the resulting set of formulas is called the ``constraint'' and
 the set of all the signature functions of the @('encapsulate') together with
 all function symbols defined in the @('encapsulate') and mentioned in the
 constraint is called the ``constrained functions.''  Assign the constraint to
 each of the constrained functions.  Move, to just after the @('encapsulate'),
 the definitions of all function symbols defined in the @('encapsulate') that
 have been omitted from the constraint.</p>

 <p>Implementation note.  In the implementation we do not actually move @(see
 events), but we create constraints that pretend that we did.</p>

 <p>Here is an example illustrating our constraint-assigning algorithm.  It
 builds on the preceding examples.</p>

 @({
  (encapsulate
   (((sig-fn *) => *))

   (defun before1 (x)
     (if (consp x)
         (before1 (cdr x))
       x))

   (local (defun sig-fn (x) (cons x x)))

   (defthm sig-fn-prop
     (consp (sig-fn x)))

   (defun during (x)
     (if (consp x)
         x
       (cons (car (sig-fn x))
             17)))

   (defun before2 (x)
     (before1 x))

   (defthm before2-prop
     (atom (before2 x)))

   (defthm during-prop
     (implies (and (atom x)
                   (before2 x))
              (equal (car (during x))
                     (car (sig-fn x)))))

   (defun after1 (x)
     (sig-fn x))

   (defchoose after2 (x) (u)
     (and (< u x) (during x)))
   )
 })

 <p>Only the functions @('sig-fn') and @('during') receive extra constraints.
 The functions @('before1') and @('before2') are viewed as moving in front of
 the @(tsee encapsulate), as is the theorem @('before2-prop').  The functions
 @('after1') and @('after2') are viewed as being moved past the @(tsee
 encapsulate).  The implementation reports the following.</p>

 @({
  In addition to SIG-FN, we export AFTER2, AFTER1, BEFORE2, DURING and
  BEFORE1.

  The following constraint is associated with both of the functions DURING and
  SIG-FN:

  (AND (EQUAL (DURING X)
              (IF (CONSP X)
                  X (CONS (CAR (SIG-FN X)) 17)))
       (CONSP (SIG-FN X))
       (IMPLIES (AND (ATOM X) (BEFORE2 X))
                (EQUAL (CAR (DURING X))
                       (CAR (SIG-FN X)))))
 })

 <p>Notice that the formula @('(consp (during x))') is not a conjunct of the
 constraint.  During the first pass of the @('encapsulate'), this formula is
 stored as a @(':')@(tsee type-prescription) rule deduced during the definition
 of the function @('during').  However, the rule is not exported because of a
 rather subtle soundness issue.  (If you are interested in details, see the
 comments in source function @('putprop-type-prescription-lst').)</p>

 <p>We conclude by asking (and to a certain extent, answering) the following
 question: Isn't there an approach to assigning constraints that avoids
 over-constraining more simply than our ``second cut'' above?  Perhaps it seems
 that given an @(tsee encapsulate), we should simply assign to each locally
 defined function the theorems exported about that function.  If we adopted
 that simple approach the events below would be admissible.</p>

 @({
  (encapsulate
   (((foo *) => *))
   (local (defun foo (x) x))
   (defun bar (x)
     (foo x))
   (defthm bar-prop
     (equal (bar x) x)
     :rule-classes nil))

  (defthm foo-id
    (equal (foo x) x)
    :hints ((\"Goal\" :use bar-prop)))

  ; The following event is not admissible in ACL2.

  (defthm ouch!
    nil
    :rule-classes nil
    :hints
    ((\"Goal\" :use
      ((:functional-instance foo-id
                             (foo (lambda (x) (cons x x))))))))
 })

 <p>Under the simple approach we have in mind, @('bar') is constrained to
 satisfy both its definition and @('bar-prop') because @('bar') mentions a
 function declared in the signature list of the encapsulation.  In fact,
 @('bar') is so-constrained in the ACL2 semantics of encapsulation and the
 first two events above (the @('encapsulate') and the consequence that @('foo')
 must be the identity function) are actually admissible.  But under the simple
 approach to assigning constraints, @('foo') is unconstrained because no
 theorem about it is exported.  Under that approach, @('ouch!') is provable
 because @('foo') can be instantiated in @('foo-id') to a function other than
 the identity function.</p>

 <p>It's tempting to think we can fix this by including definitions, not just
 theorems, in constraints.  But consider the following slightly more elaborate
 example.  The problem is that we need to include as a constraint on @('foo')
 not only the definition of @('bar'), which mentions @('foo') explicitly, but
 also @('abc'), which has @('foo') as an ancestor.</p>

 @({
  (encapsulate
   (((foo *) => *))
   (local (defun foo (x) x))
   (local (defthm foo-prop
            (equal (foo x) x)))
   (defun bar (x)
     (foo x))
   (defun abc (x)
     (bar x))
   (defthm abc-prop
     (equal (abc x) x)
     :rule-classes nil))

  (defthm foo-id
    (equal (foo x) x)
    :hints ((\"Goal\" :use abc-prop)))

  ; The following event is not admissible in ACL2.

  (defthm ouch!
    nil
    :rule-classes nil
    :hints
    ((\"Goal\" :use
      ((:functional-instance foo-id
                             (foo (lambda (x) (cons x x)))
                             (bar (lambda (x) (cons x x))))))))
 })")

(defxdoc constraint-info
  :parents (system-utilities constraint)
  :short "Obtaining the @(see constraint) on a function symbol"
  :long "<p>For a function symbol, @('fn'), and a logical @(see world),
 @('wrld') &mdash; for example, the current world, @('(w state)') &mdash;
 evaluation of the form @('(constraint-info fn wrld)') returns @('(mv flg c)'),
 where @('c') is the list of @(see constraint)s on @('fn') (implicitly
 conjoined), and @('flg') is @('nil') if @('fn') is a defined function and
 otherwise is a function symbol with that same list of constraints (possibly
 @('fn') itself).  See @(see constraint) for relevant background.</p>

 <p>We illustrate with the following example.</p>

 @({
 (encapsulate
   (((f1 *) => *)
    ((f3 *) => *))
   (local (defun f1 (x) x))
   (defun f2 (x) (f1 x))
   (local (defun f3 (x) x))
   (defun f4 (x) (f3 x))
   (defthm f1-prop (equal (f1 x) (f4 x))))
 })

 <p>Then we can see the results of @('constraint-info') on each introduced
 function symbol, as follows.</p>

 @({
 ACL2 !>(let ((wrld (w state)))
           (list
            'result
            'f1 (mv-let (flg1 c1) (constraint-info 'f1 wrld) (list flg1 c1))
            'f2 (mv-let (flg2 c2) (constraint-info 'f2 wrld) (list flg2 c2))
            'f3 (mv-let (flg3 c3) (constraint-info 'f3 wrld) (list flg3 c3))
            'f4 (mv-let (flg4 c4) (constraint-info 'f4 wrld) (list flg4 c4))))
 (RESULT F1
         (F1 ((EQUAL (F4 X) (F3 X))
              (EQUAL (F1 X) (F4 X))))
         F2 (NIL (EQUAL (F2 X) (F1 X)))
         F3
         (F1 ((EQUAL (F4 X) (F3 X))
              (EQUAL (F1 X) (F4 X))))
         F4
         (F1 ((EQUAL (F4 X) (F3 X))
              (EQUAL (F1 X) (F4 X)))))
 ACL2 !>
 })

 <p>Notice that the flag (first result) for @('f2') is @('nil'), because even
 though the definition of @('f2') is lexically inside the @('encapsulate'), it
 doesn't affect the constraints because it can be safely moved to just after
 the @('encapsulate').  However, the definition of @('f4') does affect (or
 ``infect''; see @(see subversive-recursions)) the constraints: it can't be
 moved to after the @('encapsulate') because of the @('defthm') after it.</p>

 <p>Also see @(see constraint).  For more details, see comments in the
 definition of @('constraint-info') in the ACL2 source code.</p>")

(defxdoc copyright
  :parents (about-acl2)
  :short "ACL2 copyright, license, sponsorship"
  :long "<p>This topic provides information about copyright, license,
 authorship, and sponsorship of the ACL2 system.  For information about
 copyright and authorship of @(see documentation), see @(see
 documentation-copyright), which notes that there are many documentation
 authors.</p>

 <p>@(`(:raw (@ acl2-version))`) &mdash; A Computational Logic for Applicative
 Common Lisp</p>

 <p>Copyright (C) 2021, Regents of the University of Texas</p>

 <p>This version of ACL2 is a descendant of ACL2 Version 1.9, Copyright (C)
 1997 Computational Logic, Inc.  See the documentation topic NOTE-2-0.</p>

 <p>This program is free software; you can redistribute it and/or modify it
 under the terms of the LICENSE file distributed with ACL2.</p>

 <p>This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the LICENSE for more details.</p>

 <p>Written by: Matt Kaufmann and J Strother Moore<br/>
 email: Kaufmann@@cs.utexas.edu and Moore@@cs.utexas.edu<br/>
 Department of Computer Science<br/>
 University of Texas at Austin<br/>
 Austin, TX 78712 U.S.A.</p>

 <p>Please also see @(see acknowledgments).</p>")

(defxdoc corollary
  :parents (rule-classes)
  :short "The corollary formula of a @(see rune)"
  :long "<p>See @(see formula).  This is a low-level system function at the
 present time.  See @(see pr) and see @(see pr!) instead.  Also see @(see
 rule-classes) for the use of the symbol @(':corollary') in specifying a rule
 class.</p>")

(defxdoc count
  :parents (lists strings acl2-built-ins)
  :short "Count the number of occurrences of an item in a string or true-list"
  :long "@({
  Example Forms:
  (count #\\D \"DabcDefcDe\")                 ; = 3
  (count #\\D \"DabcDefcDe\" :start 1)        ; = 2
  (count #\\D \"DabcDefcDe\" :start 1 :end 5) ; = 1
  (count #\\D \"DabcDefcDe\" :start 1 :end 4) ; = 0
  (count #\\z \"DabcDefcDe\")                 ; = 0
  (count '(a b) '(17 (a b) 23 (a b) (c d)))   ; = 2

  General Form:
  (count item sequence &key start end)
 })

 <p>@('(Count item sequence)') returns the number of times @('item') occurs in
 @('sequence').  The @(see guard) for calls of @('count') (which is actually a
 macro in ACL2) specifies that @('sequence') is a string or a true-list, and
 that @('start'), which defaults to 0, and @('end'), which defaults to the
 length of @('sequence'), are valid indices into sequence.</p>

 <p>See any Common Lisp documentation for more information about @('count'),
 which is a Common Lisp utility.  At this time ACL2 does not support keyword
 arguments for @('count') other than @(':start') and @(':end'); we may add
 support for the @(':from-end') keyword upon request.</p>

 @(def count)")

(defxdoc count-keys
  :parents (stobj acl2-built-ins)
  :short "Count the number of keys in association list"
  :long "<p>@('(Count-keys al)') returns the number of distinct keys in an
 association list.</p>

 <p>@('Count-keys') has a guard of @('t').  This function is called in the body
 of function, @('<h>-count') where @('<h>') is a hash-table field of a @(see
 stobj).  See @(see defstobj).</p>

 @(def hons-remove-assoc)

 @(def count-keys)")

(defxdoc cpu-core-count
  :parents (parallelism acl2-built-ins)
  :short "The number of cpu cores"
  :long "<p>This @(see documentation) topic relates to the experimental
 extension of ACL2 supporting parallel execution and proof; see @(see
 parallelism).</p>

 <p>Unless the ACL2 executable supports parallel execution (see @(see
 parallelism)), this function returns @('(mv 1 state)').  Otherwise:</p>

 <p>@('(Cpu-core-count state)') returns @('(mv core-count state)'), where
 @('core-count') is determined as follows.  If environment variable
 @('ACL2_CORE_COUNT') has a non-empty value, then that value should represent a
 positive integer (else an error occurs), which is the returned
 @('core-count').  Otherwise, the returned @('core-count') may be obtained from
 the underlying Common Lisp implementation, else as a positive integer value
 from @(see state) global variable @(''cpu-core-count') (see @(see assign)).
 Otherwise an error occurs.</p>

 @({
  Example:
  (cpu-core-count state) ==> (mv 4 state)
 })

 <p>@('Cpu-core-count') has the following logical definition.</p>

 @(def cpu-core-count)")

(defxdoc ctx
  :parents (errors)
  :short "Context object for error messages"
  :long "<p>Calls of @(tsee er), such as @('(er soft ctx ...)'), take a context
 argument, typically called @('ctx'), for the initial part of the message:
 @('\"ACL2 Error in\"').  If @('ctx') is @('nil'), only @('\"ACL2 Error:\"') is
 printed for that initial part of the message.  Otherwise, the string printed
 after @('\"in\"') depends on the form of that context, as follows.  (See @(see
 ctxp) for the definition of a valid context.)</p>

 <ul>

 <li>If @('ctx') is a symbol, print it with @(tsee fmt) using @('\"~x\"').</li>

 <li>If @('ctx') is a pair whose @('car') is a symbol, use @(tsee fmt) to print
 @('\"(~x0 ~x1 ...)\"'), with @('#\\0') and @('#\\1') bound respectively to the
 @('car') and @('cdr') of @('ctx').  <b>Exception</b>: if the @('car') is a
 member of the value of constant @('*fmt-ctx-spacers*'), then a space is
 printed after the left parenthesis.  That explains why there is a space, for
 example, in @('\"( DEFUN\"') in error messages starting with:

 @({
 ACL2 Error in ( DEFUN FOO ...):
 })</li>

 <li>Otherwise, print @('ctx') with @('fmt') using @('\"~@\"').</li>

 </ul>")

(defxdoc ctxp
  :parents (errors)
  :short "Recognizer for context objects for error messages"
  :long "<p>See @(see ctx) for relevant background.  The function @('ctxp')
  returns @('t') when @('ctx') is a valid context according to the definition
  below (also see @(see msgp)), else @('nil').</p>

 @(def ctxp)")

(defxdoc current-package
  :parents (ld)
  :short "The package used for reading and printing"
  :long "<p>@('Current-package') is an @(tsee ld) special (see @(see ld)).  The
 accessor is @('(current-package state)') and the updater is
 @('(set-current-package val state)'), or more conventionally, @('(in-package
 val)').  The value of @('current-package') is actually the string that names
 the package.  (Common Lisp's ``package'' objects do not exist in ACL2.)  The
 current package must be known to ACL2, i.e., it must be one of the initial
 packages or a package defined with @(tsee defpkg) by the user.</p>

 <p>When printing symbols, the package prefix is displayed if it is not the
 @('current-package') and may be optionally displayed otherwise.  Thus, if
 @('current-package') is @('\"ACL2\"') then the symbol @(''ACL2::SYMB') may be
 printed as @('SYMB') or @('ACL2::SYMB'), while @(''MY-PKG::SYMB') must be
 printed as @('MY-PKG::SYMB').  But if @('current-package') is @('\"MY-PKG\"')
 then the former symbol must be printed as @('ACL2::SYMB') while the latter may
 be printed as @('SYMB').</p>

 <p>In Common Lisp, @('current-package') also affects how objects are read from
 character streams.  Roughly speaking, read and print are inverses if the
 @('current-package') is fixed, so reading from a stream produced by printing
 an object must produce an equal object.</p>

 <p>In ACL2, the situation is more complicated because we never read objects
 from character streams, we only read them from object ``streams'' (channels).
 Logically speaking, the objects in such a channel are fixed regardless of the
 setting of @('current-package').  However, our host file systems do not
 support the idea of Lisp object files and instead only support character
 files.  So when you open an object input channel to a given (character file)
 we must somehow convert it to a list of ACL2 objects.  This is done by a
 <i>deus ex machina</i> (``a person or thing that appears or is introduced
 suddenly and unexpectedly and provides a contrived solution to an apparently
 insoluble difficulty,'' Webster's Ninth New Collegiate Dictionary).  Roughly
 speaking, the <i>deus ex machina</i> determines what sequence of calls to
 @('read-object') will occur in the future and what the @('current-package')
 will be during each of those calls, and then produces a channel containing the
 sequence of objects produced by an analogous sequence of Common Lisp reads
 with @('*current-package*') bound appropriately for each.</p>

 <p>A simple rule suffices to make sane file @(see io) possible: before you
 read an object from an object channel to a file created by printing to a
 character channel, make sure the @('current-package') at read-time is the same
 as it was at print-time.</p>")

(defxdoc current-theory
  :parents (theories theory-functions)
  :short "Currently @(see enable)d rules as of logical name"
  :long "@({
  Examples:
  (current-theory :here)
  (current-theory 'lemma3)
 })

 <p>See @(see logical-name).</p>

 @({
  General Form:
  (current-theory logical-name)
 })

 <p>Returns the current theory as it existed immediately after the introduction
 of @(tsee logical-name) provided it is evaluated in an environment in which
 the variable symbol WORLD is bound to the current ACL2 logical world, @('(w
 state)').  Thus,</p>

 @({
  ACL2 !>(current-theory :here)
 })

 <p>will cause an (unbound variable) error while</p>

 @({
  ACL2 !>(let ((world (w state))) (current-theory :here))
 })

 <p>will return the current theory in world.</p>

 <p>See @(see theories) and see @(see logical-name) for a discussion of
 theories in general and why the commonly used ``theory functions'' such as
 @('current-theory') are really macros that expand into terms involving the
 variable @('world').</p>

 <p>The theory returned by @('current-theory') is in fact the theory selected
 by the @(tsee in-theory) event most recently preceding logical name, extended
 by the rules introduced up through @(tsee logical-name).</p>

 <p>You may experience a fencepost problem in deciding which logical name to
 use.  @(tsee Deflabel) can always be used to mark unambiguously for future
 reference a particular point in the development of your theory.  The order of
 @(see events) in the vicinity of an @(tsee encapsulate) is confusing.  See
 @(see encapsulate).</p>

 <p>This ``function'' is actually a macro that expands to a term mentioning the
 single free variable @(tsee world).  When theory expressions are evaluated by
 @(tsee in-theory) or the @(':')@(tsee in-theory) hint, @(tsee world) is bound
 to the current ACL2 @(see world).</p>")

(defxdoc custom-keyword-hints
  :parents (hints)
  :short "User-defined hints"
  :long "<p>See @(see add-custom-keyword-hint) for a discussion of how advanced
 users can define their own hint keywords.  For examples, see the community
 books directory @('books/hints/'), in particular @('basic-tests.lisp').</p>")

(defxdoc cw
  :parents (io acl2-built-ins)
  :short "Print to the comment window"
  :long "<p>@('Cw') is a macro that expands to a function whose guard is
 @('t').  For a guarded variant of @('cw'), see @(see fmx-cw).</p>

 <p>Example:</p>

 @({
  (cw \"The goal is ~p0 and the alist is ~x1.~%\"
      (untranslate term t nil)
      unify-subst)
 })

 <p>Logically, this expression is equivalent to @('nil').  However, it has the
 effect of first printing to the so-called ``comment window'' the @(tsee fmt)
 string as indicated.  Thus, @('cw') is like @('fmt') (see @(see fmt)) except
 in three important ways.  First, it is a macro whose calls expand to calls of
 a @(':')@(tsee logic) mode function.  Second, it neither takes nor returns the
 ACL2 @(tsee state); logically @('cw') simply returns @('nil'), although it
 prints to a <i>comment window</i> that just happens to share the terminal
 screen with the standard character output @(tsee *standard-co*).  Third, its
 @('fmt') args are positional references, so that for example</p>

 @({
  (cw \"Answers: ~p0 and ~p1\" ans1 ans2)
 })

 <p>prints in the same manner as:</p>

 @({
  (fmt \"Answers: ~p0 and ~p1\"
       (list (cons #\\0 ans1) (cons #\\1 ans2))
       *standard-co* state nil)
 })

 <p>Typically, calls of @('cw') are embedded in @(tsee prog2$) forms, e.g.,</p>

 @({
  (prog2$ (cw ...)
          (mv a b c))
 })

 <p>which has the side-effect of printing to the comment window and logically
 returning @('(mv a b c)').</p>

 @({
  General Form:
  (cw fmt-string arg1 arg2 ... argn)
 })

 <p>where n is between 0 and 9 (inclusive).  The macro uses @(tsee
 fmt-to-comment-window), passing it the column @('0') and @(see evisc-tuple)
 @('nil'), after assembling the appropriate alist binding the @(tsee fmt) vars
 #\\0 through #\\9; see @(see fmt).  If you want</p>

 @({
  (a) more than 10 vars,
  (b) vars other than the digit chars,
  (c) a different column, or
  (d) a different evisc-tuple,
 })

 <p>then call @(tsee fmt-to-comment-window) instead.</p>

 <p>Also see @(see cw!), which is useful if you want to be able to read the
 printed forms back in.</p>

 <p>Finally, we discuss another way to create formatted output that also avoids
 the need to pass in the ACL2 @(tsee state).  The idea is to use wormholes; see
 @(see wormhole).  Below is a function you can write, along with some calls,
 providing an illustration of this approach.</p>

 @({
  (defun my-fmt-to-comment-window (str alist)
    (wormhole 'my-fmt-to-comment-window
              '(lambda (whs) whs)
              (list str alist)
              '(pprogn
                (fms (car (@ wormhole-input))
                     (cadr (@ wormhole-input))
                     *standard-co*
                     state
                     nil)
                (value :q))
              :ld-verbose nil
              :ld-error-action :return ; harmless return on error
              :ld-prompt nil))

  ; A non-erroneous call:
  (my-fmt-to-comment-window \"Here is ~x0 for your inspection~%\"
                            (list (cons #\\0 'foo)))

  ; An error inside the fmt string (unbound fmt var); note that even
  ; with the error, the wormhole is exited.
  (my-fmt-to-comment-window \"Here is ~x1 for your inspection~%\"
                            (list (cons #\\0 'foo)))

  ; A guard violation in the binding; note that even with the error,
  ; the wormhole is exited.
  (my-fmt-to-comment-window \"Here is ~x0 for your inspection~%\"
                            (list (cons #\\0 (car 'foo))))
 })")

(defxdoc cw!
  :parents (io acl2-built-ins)
  :short "Print to the comment window"
  :long "<p>This is nearly the same as @(tsee cw), but @('cw!') avoids
 inserting backslash (\\) characters when forced to print past the right
 margin.  Use @('cw!') if you want to be able to read the forms back in.</p>")

(defxdoc cw-gstack
  :parents (break-rewrite debugging)
  :short "Debug a rewriting loop or stack overflow"
  :long "<p>After @(see break-rewrite) is enabled (with @(':')@(tsee brr)@('
 t')), @('cw-gstack') can be invoked to show a stack backtrace when a proof is
 interrupted (because of control-c or because of a loop).</p>

 @({
  Example Forms:
  (cw-gstack)
  (cw-gstack :frames 10)       ; show only the top 10 frames
  (cw-gstack :frames '(1 10))  ; same as above:  show only frames 1 through 10
  (cw-gstack :frames '(10 20)) ; show only frames 10 through 20
  (cw-gstack :evisc-tuple (evisc-tuple 3 4 nil nil))
                               ; print with print-level 3 and print-length 4
  (cw-gstack :evisc-tuple nil) ; print using default ``evisceration'',
                               ;   essentially the same as just above
  (cw-gstack :evisc-tuple '(nil 3 4 (hide)))
                               ; same as just above

  General Form:
  (cw-gstack :frames frames :evisc-tuple evisc-tuple)
 })

 <p>where @(':frames') and @(':evisc-tuple') are optional, but if they are
 supplied, their values are evaluated.  The value of @('frames') should be
 either a natural number or a list of two natural numbers, the first less than
 the second; and the value of @('evisc-tuple') should be an evisc-tuple (see
 @(see evisc-tuple)).  If @(':evisc-tuple') is omitted, then by default,
 substructures deeper than 3 are replaced by ``@('#')'' and those longer than 4
 are replaced by ``@('...')'', and terms of the form @('(hide ...)') are
 printed as @('<hidden>'); this behavior can be changed by setting the
 @(':TERM') @(see evisc-tuple) (see @(see set-evisc-tuple)).  Also see @(see
 set-iprint) for an alternative to printing ``@('#')'' and ``@('...')''.</p>

 <p>Stack overflows may occur, perhaps caused by looping rewrite rules.  In
 some Lisps, stack overflows may manifest themselves as segmentation faults,
 causing the entire ACL2 image to crash.  Finding looping rewrite rules can be
 tricky, especially if you are using books supplied by other people.  (However,
 see @(see set-rewrite-stack-limit) for a way to avoid stack overflows caused
 by rewriter loops.)</p>

 <p>Normally, a stack overflow will cause the printing of an error message that
 suggests how to proceed.  Just follow those instructions, and you will
 generally be able to see what is causing the loop.</p>

 <p>Suggestion: Once you have found the loop and fixed it, you should execute
 the ACL2 command @(':')@(tsee brr)@(' nil'), so that you don't slow down
 subsequent proof attempts.</p>")

(defxdoc cw-print-base-radix
  :parents (io acl2-built-ins)
  :short "Print to the comment window in a given print-base"
  :long "<p>See @(tsee cw) for relevant background.  This variant of @('cw')
 requires specification of a print-base and, optionally, a print-radix (see
 @(see set-print-base), @(see set-print-radix), and @(see
 set-print-base-radix)).</p>

 <p>The following examples show that @('cw-print-base-radix') is just like
 @(tsee cw), except that there is a new argument in the first position that
 specifies the print-base and can, for that print-base, override the default
 print-radix.</p>

 @({
 ACL2 !>(cw-print-base-radix 16 \"~x0~%\" '(3 12 16 17))
 (#x3 #xC #x10 #x11)
 NIL
 ACL2 !>(cw-print-base-radix '(16 . t) \"~x0~%\" '(3 12 16 17))
 (#x3 #xC #x10 #x11)
 NIL
 ACL2 !>(cw-print-base-radix '(16 . nil) \"~x0~%\" '(3 12 16 17))
 (3 C 10 11)
 NIL
 ACL2 !>(cw-print-base-radix 10 \"~x0~%\" '(3 12 16 17))
 (3 12 16 17)
 NIL
 ACL2 !>(cw-print-base-radix '(10 . t) \"~x0~%\" '(3 12 16 17))
 (3. 12. 16. 17.)
 NIL
 ACL2 !>(cw-print-base-radix '(10 . nil) \"~x0~%\" '(3 12 16 17))
 (3 12 16 17)
 NIL
 ACL2 !>

 @({
 General Forms:

 (cw-print-base-radix print-base fmt-string arg1 arg2 ... argn)
 (cw-print-base-radix print-base/print-radix fmt-string arg1 arg2 ... argn)
 })

 <p>where all arguments of this macro are evaluated; @('print-base') is a legal
 print-base as recognized by @(tsee print-base-p); @('print-base/print-radix')
 is a cons whose car is a legal print-base; @('fmt-string') is a string
 suitable for passing to @(tsee fmt); and @('arg1') through @('argn') (where n
 is at most 9) are corresponding arguments for @('fmt-string').  Printing is
 done according to the specified print-base, which is the first argument in the
 first general form and is the car of the first argument in the second general
 form.  The print-radix value that is used for printing in the first general
 form is the print-radix as specified for @(tsee set-print-base-radix), while
 in the second general form, it is the cdr.</p>")

(defxdoc cw-print-base-radix!
  :parents (io acl2-built-ins)
  :short "Print to the comment window in a given print-base"
  :long "<p>This is nearly the same as @(tsee cw-print-base-radix), but
  @('cw-print-base-radix!') avoids inserting backslash (\\) characters when
  forced to print past the right margin.  Use @('cw-print-base-radix!')  if you
  want to be able to read the forms back in.</p>")

(defxdoc |Common Lisp|
  :parents (|Pages Written Especially for the Tours|)
  :short "Common Lisp"
  :long "<p><see topic='@(url
 |An Example Common Lisp Function Definition|)'><img
 src='res/tours/walking.gif'></img></see></p>

 <p>The logic of ACL2 is based on Common Lisp.</p>

 <p>Common Lisp is the standard list processing programming language.  It is
 documented in: Guy L. Steele, <a
 href='https://www.cs.cmu.edu/Groups/AI/html/cltl/cltl2.html'><b>Common Lisp
 The Language</b></a>, Digital Press, 12 Crosby Drive, Bedford, MA 01730,
 1990.</p>

 <p>ACL2 formalizes only a subset of Common Lisp.  It includes such familiar
 Lisp functions as @('cons'), @('car') and @('cdr') for creating and
 manipulating list structures, various arithmetic primitives such as @('+'),
 @('*'), @('expt') and @('<='), and @('intern') and @('symbol-name') for
 creating and manipulating symbols.  Control primitives include @('cond'),
 @('case') and @('if'), as well as function call, including recursion.  New
 functions are defined with @('defun') and macros with @('defmacro').  See
 @(see programming) <see topic='ACL2____A_02Tiny_02Warning_02Sign'><icon
 src='res/tours/twarning.gif'/></see> for a list of the Common Lisp primitives
 supported by ACL2.</p>

 <p>ACL2 supports five of Common Lisp's datatypes:</p>

 <p>* the precisely represented, unbounded numbers (integers, rationals, and
 the complex numbers with rational components, called the ``complex rationals''
 here),</p>

 <p>* the characters with ASCII codes between 0 and 255</p>

 <p>* strings of such characters</p>

 <p>* symbols (including packages)</p>

 <p>* conses</p>

 <p>ACL2 is a very small subset of full Common Lisp.  ACL2 does not include the
 Common Lisp Object System (CLOS), higher order functions, circular structures,
 and other aspects of Common Lisp that are <b>non-applicative</b>.  Roughly
 speaking, a language is applicative if it follows the rules of function
 application.  For example, @('f(x)') must be equal to @('f(x)'), which means,
 among other things, that the value of @('f') must not be affected by ``global
 variables'' and the object @('x') must not change over time.</p>

 <p><see topic='@(url |An Example Common Lisp Function Definition|)'><img
 src='res/tours/walking.gif'></img></see></p>")

(defxdoc |Common Lisp as a Modeling Language|
  :parents (|Pages Written Especially for the Tours|)
  :short "Common Lisp as a Modeling Language"
  :long "<p>In ACL2 we have adopted Common Lisp as the basis of our modeling language.
 If you have already read our brief note on Common Lisp and recall the example
 of @('app'), please proceed.  Otherwise click <see topic='@(url
 |Common Lisp|)'>here</see> for an exceedingly brief introduction to Common
 Lisp and then come <b>back</b> here.</p>

 <p>In Common Lisp it is very easy to write systems of formulas that manipulate
 discrete, inductively constructed data objects.  In building a model you might
 need to formalize the notion of sequences and define such operations as
 concatenation, length, whether one is a permutation of the other, etc.  It is
 easy to do this in Common Lisp.  Furthermore, if you have a Common Lisp
 ``theory of sequences'' you can <b>run</b> the operations and relations you
 define.  That is, you can execute the functions on concrete data to see what
 results your formulas produce.</p>

 <p>If you define the function @('app') as shown above and then type</p>

 @({
  (app '(A B) '(C D E))
 })

 <p>in any Common Lisp, the answer will be computed and will be @('(A B C D
 E)').</p>

 <p>The <b>executable</b> nature of Common Lisp and thus of ACL2 is very handy
 when producing models.</p>

 <p>But executability is not enough for a modeling language because the purpose
 of models is to permit analysis.</p>

 <p>Click <see topic='@(url |Analyzing Common Lisp Models|)'>here</see> to
 continue.</p>")

(defxdoc |Conversion|
  :parents (|Pages Written Especially for the Tours|)
  :short "Conversion to Uppercase"
  :long "<p>When symbols are read by Common Lisp they are converted to upper
 case.  Note carefully that this remark applies to the characters in
 <i>symbols</i>.  The characters in strings are not converted upper case.</p>

 <p>To type a symbol containing lower case characters you can enclose the
 symbol in vertical bars, as in @('|AbC|') or you can put a ``backslash''
 before each lower case character you wish to preserve, as in @('A\\bC').
 @('|AbC|') and @('A\\bC') are two different ways of writing the same symbol
 (just like 2/4 and 1/2 are two different ways of writing the same rational and
 123 and 0123 are two different ways to write the same natural number).  The
 symbol has three characters in its name, the middle one of which is a lower
 case b.</p>")

(defxdoc |Corroborating Models|
  :parents (|Pages Written Especially for the Tours|)
  :short "Corroborating Models"
  :long "<p><see topic='@(url |Models of Computer Hardware and Software|)'><img
 src='res/tours/flying.gif'></img></see></p>

 <p>After producing a model, it must be <b>corroborated</b> against reality.
 The Falling Body Model has been corroborated by a vast number of experiments
 in which the time and distance were measured and compared according to the
 formula.  In general all models must be corroborated by experiment.</p>

 <p>The Falling Body Model can be derived from deeper models, namely Newton's
 laws of motion and the assertion that, over the limited distances concerned,
 gravitation exerts a constant acceleration on the object.  When the model in
 question can be derived from other models, it is the other models that are
 being corroborated by our experiments.</p>

 <p>Because nature is not formal, we cannot <b>prove</b> that our models of it
 are correct.  All we can do is test our models against nature's behavior.</p>

 <p>Such testing often exposes restrictions on the applicability of our models.
 For example, the Falling Body Model is inaccurate if air resistance is
 significant.  Thus, we learn not to use that model to predict how long it
 takes a feather to fall from a 200 foot tower in the earth's atmosphere.</p>

 <p>In addition, attempts at corroboration might reveal that the model is
 actually incorrect.  Careful measurements might expose the fact that the
 gravitational force increases as the body falls closer to earth.  Very careful
 measurements might reveal relativistic effects.  Technically, the familiar
 Falling Body Model is just wrong, even under excessive restrictions such as
 ``in a perfect vacuum'' and ``over small distances.''  But it is an incredibly
 useful model nonetheless.</p>

 <p>There are several morals here.</p>

 <p><b>Models need not be complete to be useful.</b></p>

 <p><b>Models need not be perfectly accurate to be useful.</b></p>

 <p><b>The user of a model must understand its limitations.</b></p>

 <p><see topic='@(url |Models of Computer Hardware and Software|)'><img
 src='res/tours/flying.gif'></img></see></p>")

(defxdoc dead-events
  :parents (debugging)
  :short "Using proof supporters to identify dead code and unused theorems"
  :long "<p>Below, when we talk about ``an event @('A')'', we mean an event
 whose name is @('A').</p>

 <p>When event @('A') is used in a proof performed to admit event @('B') that
 you submit to ACL2, we say that @('A') is a ``proof-supporter'' of @('B').
 ACL2 stores an association list such that for every event @('B') with at least
 one proof-supporter, @('B') is associated with a list of all of its
 proof-supporters, sorted by @(tsee symbol<).  The following form evaluates to
 that alist, which is called the ``proof-supporters-alist''.</p>

 @({
  (global-val 'proof-supporters-alist (w state))
 })

 <p>By ``used in a proof'' above, we mean: applied as a rule or supplied
 explicitly via @(see hints) of type @(':use'), @(':by'), or
 @(':clause-processor').  That is, the @(see events) ``used in a proof'' for
 admitting an event @('E') are those listed in the @(see summary) printed at
 the conclusion of admitting @('E').</p>

 <p>Note that if proofs are skipped when admitting event @('E'), say because
 the last admission of @('E') was done by @(tsee include-book) (or
 @('certify-book'), which ends with an @(tsee include-book)), then there will
 be no entry in that alist for @('E').  (An exception is made however for
 @(tsee encapsulate) @(see events), where proof-supporters are remembered from
 the first pass; see below.)  So if you want the proof-supporters-alist to
 include supporters for events in a book, use @(tsee ld) rather than @(tsee
 include-book) or @(tsee certify-book) to process the events in that book.  If
 however you are interested in the proof-supporters FROM a book that support a
 later event, then it is fine to include that book.</p>

 <p>The case for @(tsee encapsulate) is slightly tricky.  Consider an example
 of the following form.</p>

 @({
  A ; event preceding the encapsulate
  (encapsulate
   ()
   B
   (local C) ; uses A and B in a proof
   D ; uses C in a proof
   )
 })

 <p>At the conclusion of this @(tsee encapsulate) event, the
 proof-supporters-alist associates @('D') with @('A') and @('B'), but not
 @('C') (which has disappeared, since it is @(see local)).</p>

 <p>Note that this sort of ``transitive closure'' operation is only performed
 when necessary due to the disappearance of @(see local) @(see events).  For
 example, if we replace @('(local C)') above by just @('C'), then @('D') is
 associated in the proof-supporters-alist only with @('C'), not with @('A') or
 @('B').  If you want the transitive closure of the relation computed by the
 proof-supporters-alist, you have to compute it yourself. (This was a
 deliberate design decision, in order to avoid slowing down event processing.)
 However, there is help available on how to do such a computation:</p>

 <p>A community book, @('books/tools/dead-events.lisp'), does such a transitive
 closure, and moreover uses that information to find ``dead events'' relative
 to a list of ``desired'' events.  For example, suppose you use @(tsee LD) to
 process the events, with proofs, in a book intended to prove theorems
 @('MAIN-1') and @('MAIN-2').  (Remember, @(tsee certify-book) will not save
 such information.)  Suppose furthermore that the book begins with some @(tsee
 include-book) forms followed by @('(deflabel book-start)').  You could
 evaluate this form:</p>

 @({
  (dead-events '(main-1 main-2) :start 'book-start)
 })

 <p>The result is a list of events that you probably can delete from the book
 without causing any proofs to fail.  See the @('dead-events.lisp') book for
 further documentation.</p>

 <p>You might also find the code in the above book to be helpful for writing
 your own utilities based on the proof-supporters-alist.</p>")

(defxdoc dealing-with-key-combinations-of-function-symbols
  :parents (introduction-to-the-theorem-prover)
  :short "How to get rid of key combinations of function symbols"
  :long "<p>Suppose @('REV') reverses a list, @('MEMBER') checks that its first
 argument is an element of its second, and @('SQUARES-PLUS-3P') is some
 complicated predicate.  Suppose you're proving some Main Theorem that involves
 those concepts and the theorem prover presents you with the following hideous
 formula as a key checkpoint.  What action should you take?</p>

 <p>Hint: Don't read the formula ``for sense,'' i.e., don't try to understand
 what this formula is saying!  Just look at every subterm involving a nest of
 two function symbols and ask if you know something about those two symbols
 that allows you to <i>simplify</i> that one subterm.</p>

 @({
  (IMPLIES (AND (CONSP X)
                (MEMBER (+ 3 (* I I)) (REV X))
                (LIST-OF-INTEGERS X)
                (INTEGERP I)
                (<= 0 I)
                (INTEGERP K)
                (<= 0 K)
                (< I K)
                (SQUARES-PLUS-3P K X)
                (NOT (EQUAL (CAR X) (+ 3 (* I I))))
                (NOT (MEMBER (+ 3 (* I I)) X)))
           (SQUARES-PLUS-3P K (REV X)))?
 })

 <p>The experienced ACL2 user will stop reading at the second hypothesis!</p>

 @({
                (MEMBER (+ 3 (* I I)) (REV X))
 })

 <p>The combination of @('MEMBER') and @('REV') can be simplified.  The
 question ``is @('e') a member of @('(REV x)')'' can be answered by asking ``is
 @('e') a member of @('x')''.  The two questions are equivalent.  This insight
 comes from your intuition about the <i>semantics</i> of @('REV') &mdash; it
 just reorders the elements but doesn't add or delete any.  The second question
 is simpler since it doesn't mention @('REV'), so this is a good transformation
 to make.  And the theorem that they are equivalent is simpler than the key
 checkpoint above because it involves fewer functions and smaller
 expressions.</p>

 <p>You might formalize this insight as</p>

 @({
  (equal (member e (rev x))
         (member e x))
 })

 <p>But this conjecture is <i>not</i> a theorem, because @('(member e x)')
 returns the @('cdr') of @('x') that begins with @('e'), not just a Boolean
 (@('t') or @('nil')) indicating whether @('e') is an element of @('x').  The
 location of the first @('e') in @('(rev x)') is generally different than the
 location in @('x').  So when we say the two questions are ``equivalent'' we
 don't mean they are equal.  We mean that they're propositionally equivalent:
 both @('nil') or both non-@('nil').  This sense of equivalence is called ``if
 and only if'' and is checked by the function @('iff').</p>

 <p>So our intuitive insight can be phrased as this theorem:</p>

 @({
  (iff (member e (rev x))
       (member e x))
 })

 <p>Suggesting that this formulation of the insight is ``obvious'' begs many
 questions.  Mathematically, we could have avoided @('iff') and just written
 two implications:</p>

 @({
  (and (implies (member e x) (member e (rev x)))
       (implies (member e (rev x)) (member e x))).
 })

 <p>or</p>

 @({
  (and (implies (member e x) (member e (rev x)))
       (implies (not (member e x))  (not (member e (rev x))))).
 })

 <p>Or we could have used @('iff') but ``oriented'' it the other way:</p>

 @({
  (iff (member e x)
       (member e (rev x)))
 })

 <p>We choose to write</p>

 @({
  (iff (member e (rev x))
       (member e x))
 })

 <p>because of <i>our knowledge of how ACL2 turns formulas into rules!</i></p>

 <p>We deal with this at greater length later.  But just to drive the point
 home, if we issue the command:</p>

 @({
  (defthm member-rev
    (iff (member e (rev x))
         (member e x)))
 })

 <p>ACL2 will build in a rule that causes every propositional occurrence of
 @('(MEMBER e (REV x))') to be replaced by @('(MEMBER e x)').  (By
 ``propositional occurrence'' we mean an occurrence in which the value is
 tested, as by @('IF') or the propositional connectives.  Remember, one might
 use @('member') to determine the location of an element too.)</p>

 <p>Note carefully: <i>if you do not tell ACL2 how to make a rule</i> from a
 theorem, it makes a rewrite rule.  Rewrite rules always replace instances of
 the left-hand side by the corresponding instances of the right-hand side.
 That is, when interpreted as a rewrite rule, @('(iff ')<i>alpha</i>
 <i>beta</i>@(')') makes ACL2 replace <i>alpha</i> by <i>beta</i>.</p>

 <p>Probably the biggest mistake new users make is forgetting that every
 theorem they prove creates a very specific rule.  You must remember that you
 are <i>programming</i> ACL2 with these rules.  Being careless in your
 statement of theorems is tantamount to being careless in your programming.
 What you get is a mess.</p>

 <p>Had we proved the same equivalence, but with the @('iff') commuted, we
 would be giving ACL2 <i>bad advice</i>.  We would be telling it ``replace
 instances of @('(MEMBER e x)') by the corresponding instances of @('(MEMBER e
 (REV x))')''!  If ACL2 had that rule and ever tried to simplify any
 @('member') expression, e.g., @('(MEMBER A B)'), it would get into an infinite
 loop, e.g., producing the following sequence of transformations:</p>

 @({
  (MEMBER A B)
  (MEMBER A (REV B))
  (MEMBER A (REV (REV B)))
  ...
 })

 <p>until it eventually exhausted some resource.</p>

 <p>Recall that we entertained the idea of phrasing our insight about
 @('member') and @('rev') with implications rather than @('iff').  Generally
 speaking, implications produce weaker rules &mdash; rules that apply less
 often.  We discuss that later.</p>

 <p>Now suppose we've proved @('member-rev'), oriented so as to rewrite
 @('(member e (rev x))') to @('(member e x)'), and built it in as a rewrite
 rule.  Then suppose we repeated the attempt to prove our Main Theorem.  This
 time, when the prover is processing the hideous Key Checkpoint printed above,
 our new lemma, @('member-rev'), will hit it.  It will transform the formula
 to:</p>

 @({
  (IMPLIES (AND (CONSP X)
                (MEMBER (+ 3 (* I I)) X)   ; <-- the hyp has simplified
                (LIST-OF-INTEGERS X)
                (INTEGERP I)
                (<= 0 I)
                (INTEGERP K)
                (<= 0 K)
                (< I K)
                (SQUARES-PLUS-3P K X)
                (NOT (EQUAL (CAR X) (+ 3 (* I I))))
                (NOT (MEMBER (+ 3 (* I I)) X)))
           (SQUARES-PLUS-3P K (REV X)))?
 })

 <p>and then that will collapse to @('T'), since the @('IMPLIES') has
 contradictory hypotheses (note the last hypothesis above).</p>

 <p>By proving @('member-rev') we proved the hideous checkpoint.  We never had
 to look at the rest of the formula or think about why it is a theorem.
 Furthermore, attacking the main theorem again, from scratch, with
 @('member-rev') in the database, may eliminate other checkpoints that came up
 the last time we tried to prove our main goal.  So we recommend addressing one
 checkpoint at a time.</p>

 <p>This example illustrates that purely <i>local</i> thinking &mdash; looking
 for simplifiable combinations of function symbols &mdash; can sometimes lead
 to proofs and should always be your first reaction to a key checkpoint: what
 local fact do you know that would clean up the formula?  Don't think about
 deep questions like ``why is this true?'' until you can't see any way to make
 it simpler.</p>

 <p>It is important to train yourself to see combinations of function symbols
 and to create strong rules for eliminating them.  We will give you
 opportunities to practice this later in the tutorial.</p>

 <p>If you have been reading the tutorial introduction to the theorem prover,
 use your browser's <b>Back Button</b> now to return to @(see
 INTRODUCTION-TO-KEY-CHECKPOINTS).</p>")

(defxdoc dealing-with-tau-problems
  :parents (introduction-to-the-tau-system)
  :short "Some advice on dealing with problems caused by the tau system"
  :long "<p>For background on the tau system, see @(see
 introduction-to-the-tau-system).  The two most common problems caused by the
 tau system have to do with the system's interaction with ``legacy'' proof
 scripts.  Such scripts may suffer because they were not designed to exploit
 tau reasoning and which may configure the tau database in quite incomplete and
 arbitrary ways.  The two most common problems we have seen are (a) significant
 slow downs in a few proofs and (b) failed proof attempts due to hints being
 misapplied because the tau system caused subgoals to be renumbered.</p>

 <p>We discuss the rather limited means of dealing with these problems here.
 In @(see future-work-related-to-the-tau-system) we list some major
 inadequacies of the tau system.</p>

 <p>If the tau system contributes to a proof, the @(see rune) @('(:')@(tsee
 executable-counterpart)@(' tau-system)') will be listed among the Rules in the
 @(See Summary).  However, merely by being attempted the tau system can slow
 down proofs in which it makes no contribution.</p>

 <p>The most brutal and fool-proof way to isolate a proof from the tau system
 is to disable the entire system.  This can be done globally by</p>

 @({
  (in-theory (disable (tau-system)))  ; (:executable-counterpart tau-system)
 })

 <p>or locally with a subgoal specific hint:</p>

 @({
  ...
  :hints ((\"...subgoal id...\" :in-theory (disable (tau-system))))
 })

 <p>Conducting a proof with and without the participation of the tau system can
 help you determine whether tau reasoning is helping or hurting.</p>

 <p><i>Dealing with Slowdowns</i></p>

 <p>The @(tsee time-tracker) utility was added to allow users to investigate
 whether excessive amounts of time are being spent in a given function.  It was
 then used to annotate the code for the tau system as described in @(see
 time-tracker-tau).  The result is that if ``excessive'' time is spent in tau
 reasoning, messages to that effect will be printed to the proof log.  The
 question is: aside from disabling the tau system how can the proof be sped
 up?</p>

 <p>There are two common causes of slowdown in the tau system.  The first stems
 from the system's use of @(':')@(tsee executable-counterpart)s to determine
 whether a constant has a given tau.  Recall that a tau is a conjunction of
 monadic predicates.  To determine whether some constant satisfies the tau, the
 predicates are executed.  If you have a hard-to-compute predicate this can be
 very slow.  The most typical such predicates in ACL2 applications are those
 that check invariants, e.g., that recognize ``good states'' or ``well-formed
 data.''  These are often written inefficiently because they are intended only
 for used in theorems and, before the tau system was added, they may have never
 been applied to constants.  The most common constants tau predicates are
 applied to are @('0'), @('T'), and @('NIL'), although different models may
 stress other constants.  To understand why @('NIL') for example is frequently
 tested, if the test of an @('IF')-expression is computed to have tau <i>s</i>
 then the next question we ask is ``does @('nil') satisfy <i>s</i>?''</p>

 <p>You may determine whether the tau system is spending time executing tau
 predicates by observing the rewriter &mdash; see @(see dmr) &mdash; or by
 interrupting the system and getting a backtrace (see @(see
 set-debugger-enable)).</p>

 <p>If excessive time is being spent in a tau predicate, a draconian solution
 is to disable the @(':')@(tsee executable-counterpart) of that predicate, for
 example in either of these equivalent ways.  The tau system does not execute
 disabled @(':')@(tsee executable-counterpart)s.</p>

 @({
  (in-theory (disable (:executable-counterpart foo)))
  (in-theory (disable (foo)))
 })

 <p>In either case above, you may prefer to provide local @(':')@(tsee
 in-theory) @(':')@(tsee hints) rather than @(':in-theory') @(see events).</p>

 <p>Disabling the executable-counterpart of expensive tau predicates will
 weaken the tau system, probably only negligibly, because it can no longer run
 the predicates to determine whether they admits given constants.</p>

 <p>A more sophisticated solution is to make the tau system record values of
 the @(':')@(tsee logic)-mode function in question, so that the system will
 look up the necessary values rather than running the function every time the
 question arises.  It will look up recorded values whether the
 executable-counterpart of the tau predicate is enabled or disabled.  Here is
 an example of a lemma that can provide such a solution.  See the discussion of
 the <i>Eval</i> form of @(':')@(tsee tau-system) rules.</p>

 @({
  (defthm lemma
    (and (foo 0)
         (foo 17)
         (foo t)
         (not (foo '(a b c))))
    :rule-classes :tau-system)
 })

 <p>It might be difficult to determine <i>which</i> constants are being
 repeatedly tested, although tracing (@(tsee trace$)) suspected tau predicates
 will show what they are being called on.</p>

 <p>At the moment there are no better user-level tools to discover this.
 However, some users may wish to consider the following hack: In the ACL2
 source file @('tau.lisp'), immediately after the definition of the system
 function @('ev-fncall-w-tau-recog'), there is a comment which contains some
 raw Lisp code that can be used to investigate whether tau's use of evaluation
 on constants is causing a problem and to determine which constants are
 involved.</p>

 <p>The second main cause of slowdowns by the tau system is that the system
 contains ``too many'' conjunctive rules (see the <i>Conjunctive</i> form in
 @(tsee tau-system)).  Unfortunately, we have no tools for either identifying
 the problem or addressing it!  That said, let us tell you what we do know!</p>

 <p>Conjunctive rules are used to ``complete'' each tau as it is built.
 Referring to the @('weekdayp') example in @(tsee tau-system), if a tau is
 constructed that recognizes weekdays but not @('MON'), @('TUE'), @('THU'), or
 @('FRI'), it is completed by adding that the tau recognizes (only) @('WED').
 This means that when we construct a tau we scan all known conjunctive rules
 and see whether all but one of the literals of any conjunctive rule are
 present.  This can be expensive.  To mitigate this expense, the tau system
 caches the computation on a per proof basis (the cache is cleared after every
 proof).</p>

 <p>To learn what conjunctive rules there are in your system, evaluate</p>

 @({
  (assoc 'tau-conjunctive-rules (tau-database (w state)))
 })

 <p>Perhaps by sending the implementors that list, we can think of ways to
 index the conjunctive rules to save time.</p>

 <p><i>Dealing with Misapplied Hints</i></p>

 <p>The second common problem caused by the tau system in legacy proof scripts
 is that it can cause subgoals to be renumbered and thus cause hints to be
 missed.  The only ways to address this problem is either to disable the tau
 system (locally or globally by disabling @('(:executable-counterpart
 tau-system)')) or change the legacy hints to use the new subgoal names.</p>")

(defxdoc debugging
  :parents (top

; Including acl2 as a parent so that all ACL2 system topics can be found under
; the graph rooted at the acl2 node.

            acl2)
  :short "Tools for debugging failed or slow proofs, or misbehaving
 functions.")

(defxdoc declare

; Warning: Keep this in sync with ACL2 source function acceptable-dcls-alist.

  :parents (programming acl2-built-ins)
  :short "Extra declarations that can occur in function definitions, @(see let)
  bindings, and so forth."

  :long "<p>Common Lisp provides a declaration mechanism that allows the
 programmer to explain additional information to the compiler.  For
 instance:</p>

 <ul>

 <li>The programmer might declare that some variable always has some particular
 type.  The compiler might then, depending on its optimization/safety settings,
 either add run-time checks to ensure that this really is true, or optimize the
 compiled code by assuming the variable has the correct type.</li>

 <li>The programmer might declare that some variable is @('ignore')d.  The
 compiler might then, instead of warning the programmer that the variable is
 never used, explicitly check to make sure that it really is never used.</li>

 </ul>

 <p>ACL2 supports the above kinds of declarations, and also adds its own kinds
 of declarations for specifying things like the @(see guard)s and @(see
 measure)s of functions, as described in @(see xargs).</p>

 <p>There are also other kinds of Common Lisp declarations that ACL2 does not
 support, e.g., pertaining to inlining, safety settings, variable lifetime, and
 so forth.</p>

 <h3>Usage</h3>

 <p>Examples:</p>

 @({
 (declare (ignore x y z))
 (declare (ignorable x y z)
          (irrelevant w) ; for DEFUN only
          (type integer i j k)
          (type (satisfies integerp) m1 m2))
 (declare (xargs :guard (and (integerp i)
                             (<= 0 i))
                 :guard-hints ((\"Goal\" :use (:instance lemma3
                                               (x (+ i j)))))))
 })

 <p>General Form:</p>

 @({
     (declare d1 ... dn)
 })

 <p>where, in ACL2, each @('di') is of one of the following forms:</p>

 <dl>

 <dt>@('(ignore v1 ... vn)')</dt>

 <dd>where each @('vi') is a variable introduced in the immediately superior
 lexical environment.  These variables must not occur free in the scope of the
 declaration.  This declaration can be useful for inhibiting compiler warnings;
 see also @(see set-ignore-ok).</dd>

 <dt>@('(ignorable v1 ... vn)')</dt>

 <dd>where each @('vi') is a variable introduced in the immediately superior
 lexical environment.  These variables need not occur free in the scope of the
 declaration.  This declaration can be useful for inhibiting compiler warnings;
 see also @(see set-ignore-ok).</dd>

 <dt>@('(irrelevant v1 ... vn)')</dt>

 <dd>where each @('vi') is a formal parameter declared at the top level of a
 surrounding @(tsee defun) form, as shown below.  See @(see irrelevant-formals)
 for more information.</dd>

 <dt>@('(type type-spec v1 ... vn)')</dt>

 <dd>where each @('vi') is a variable introduced in the immediately superior
 lexical environment and @('type-spec') is a type specifier (as described in
 the documentation for @(see type-spec)).  This declaration can be useful for
 optimizing Common Lisp execution speed.  See also @(see the).</dd>

 <dt>@('(xargs :key1 val1 ... :keyn valn)')</dt>

 <dd>where the legal values of the keys and values are described in the
 documentation for @(see xargs).  These declarations are only allowed at the
 top level of definitions (@(tsee defun) and @(tsee defmacro), as shown below),
 and convey information such as the @(see guard) and @(see measure) for a
 function.</dd>

 <dt>@('(optimize ...)')</dt>

 <dd>for example, @('(optimize (safety 3))').  This is allowed only at the top
 level of @(tsee defun) forms and is probably only rarely of any interest.  See
 any Common Lisp documentation for more information.</dd>

 </dl>

 <p>Declarations in ACL2 may occur only where @('dcl') occurs in the following
 display (not including lambda objects, discussed later below):</p>

 <ul>
 <li>@('(DEFUN name args doc-string dcl ... dcl body)')</li>
 <li>@('(DEFMACRO name args doc-string dcl ... dcl body)')</li>
 <li>@('(LET ((v1 t1) ...) dcl ... dcl body)')</li>
 <li>@('(MV-LET (v1 ...) term dcl ... dcl body)')</li>
 <li>@('(FLET ((name args dcl ... dcl body) ...))')</li>
 </ul>

 <p>Of course, if a form macroexpands into one of these (e.g., as @(tsee let*)
 expands into nested @(tsee let)s and our @('er-let*') expands into nested
 @(tsee mv-let)s) then declarations are permitted as handled by the macros
 involved.</p>

 <p>Each of the cases above permits certain declarations, as follows.</p>

 <ul>

 <li>@('DEFUN'): @(`(cdr (assoc-eq 'defuns *acceptable-dcls-alist*))`)</li>
 <li>@('DEFMACRO'): @(`(cdr (assoc-eq 'defmacro *acceptable-dcls-alist*))`)</li>
 <li>@('LET'): @(`(cdr (assoc-eq 'let *acceptable-dcls-alist*))`)</li>
 <li>@('MV-LET'): @(`(cdr (assoc-eq 'mv-let *acceptable-dcls-alist*))`)</li>
 <li>@('FLET'): @(`(cdr (assoc-eq 'flet *acceptable-dcls-alist*))`)</li>
 </ul>

 <p>Also see @(see lambda) for discussion of lambda objects and their legal
 @('declare') forms.</p>

 <p>@('Declare') is defined in Common Lisp.  See any Common Lisp documentation
 for more information.</p>")

(defxdoc declare-stobjs
  :parents (stobj declare)
  :short "Declaring a formal parameter name to be a single-threaded object"
  :long "<p>When a @(tsee defun) uses one of its formals as a single-threaded
 object (@(see stobj)), the @('defun') <i>must</i> include a declaration that
 the formal is to be so used.  An exception is the formal ``@(tsee state),''
 which if not declared as explained below, may still be used provided an
 appropriate global ``declaration'' is issued: see @(see set-state-ok).</p>

 <p>If the formal in question is @('counters') then an appropriate declaration
 is</p>

 @({
  (declare (xargs :stobjs counters))
 })

 <p>or, more generally,</p>

 @({
  (declare (xargs :stobjs (... counters ...)))
 })

 <p>where all the single-threaded formals are listed.</p>

 <p>For such a declaration to be legal it must be the case that all the names
 have previously been defined as single-threaded objects with @(tsee
 defstobj).</p>

 <p>When an argument is declared to be single-threaded the guard of the
 function is augmented by conjoining to it the condition that the argument
 satisfy the recognizer for the single-threaded object.  Furthermore, the
 syntactic checks done to enforce the legal use of single-threaded objects are
 also sufficient to allow these guard conjuncts to be automatically proved.</p>

 <p>The obvious question arises: Why does ACL2 insist that you declare stobj
 names before using them in @('defun')s if you can only declare names that have
 already been defined with @('defstobj')?  What would go wrong if a formal were
 treated as a single-threaded object if and only if it had already been so
 defined?</p>

 <p>Suppose that one user, say Jones, creates a book in which @('counters') is
 defined as a single-threaded object.  Suppose another user, Smith, creates a
 book in which @('counters') is used as an ordinary formal parameter.  Finally,
 suppose a third user, Brown, wishes to use both books.  If Brown includes
 Jones' book first and then Smith's, then Smith's function treats @('counters')
 as single-threaded.  But if Brown includes Smith's book first, the argument is
 treated as ordinary.</p>

 <p>ACL2 insists on the declaration to ensure that the definition is processed
 the same way no matter what the context.</p>")

(defxdoc defabbrev
  :parents (macros events programming)
  :short "A convenient form of macro definition for simple expansions"
  :long "@({
  Examples:
  (defabbrev snoc (x y) (append y (list x)))
  (defabbrev sq (x) (declare (type (signed-byte 8) x)) (* x x))

  General Form:
  (defabbrev name (v1 ... vn) doc-string decl1 ... declk body)
 })

 <p>where @('name') is a new function symbol, the @('vi') are distinct variable
 symbols, and @('body') is a term.  The @('decli'), if supplied, should be
 legal @('declare') forms; see @(see declare).  @('Doc-string'), if
 non-@('nil'), is an optional string that can provide documentation but is
 essentially ignored by ACL2.</p>

 <p>Roughly speaking, the @('defabbrev') event is akin to defining @('f') so
 that @('(f v1 ... vn) = body').  But rather than do this by adding a new
 axiom, @('defabbrev') defines @('f') to be a macro so that @('(f a1 ... an)')
 expands to @('body'), with the ``formals,'' @('vi'), replaced by the
 ``actuals,'' @('ai').</p>

 <p>For example, if @('snoc') is defined as shown in the first example above,
 then @('(snoc (+ i j) temp)') is just an abbreviation for</p>

 @({
  (append temp (list (+ i j))).
 })

 <p>In order to generate efficiently executable Lisp code, the macro that
 @('defabbrev') introduces uses a @(tsee let) to bind the ``formals'' to the
 ``actuals.''  Consider the second example above.  Logically speaking, @('(sq
 (ack i j))') is an abbreviation for @('(* (ack i j) (ack i j))').  But in fact
 the macro for @('sq') introduced by @('defabbrev') actually arranges for
 @('(sq (ack i j))') to expand to:</p>

 @({
  (let ((x (ack i j)))
    (* x x))
 })

 <p>which executes more efficiently than @('(* (ack i j) (ack i j))').</p>

 <p>In the theorem prover, the @('let') above expands to</p>

 @({
  ((lambda (x) (* x x)) (ack i j))
 })

 <p>and thence to @('(* (ack i j) (ack i j))').</p>

 <p>It is important to note that the term in @('body') should not contain a
 call of @('name') &mdash; i.e., @('defabbrev') should not be used in place of
 @('defun') when the function is recursive.  ACL2 will not complain when the
 @('defabbrev') form is processed, but instead ACL2 will more than likely go
 into an infinite loop during macroexpansion of any form that has a call of
 @('name').</p>

 <p>It is also important to note that the parameters of any call of a macro
 defined by defabbrev will, as is the case for the parameters of a function
 call, be evaluated before the body is evaluated, since this is the evaluation
 order of @(tsee let).  This may lead to some errors or unexpected
 inefficiencies during evaluation if the body contains any conditionally
 evaluated forms like @('cond'), @('case'), or @('if').  Consider the following
 example.</p>

 @({
  (defabbrev foo (x y)
    (if (test x) (bar y) nil))
 })

 <p>Notice a typical one-step expansion of a call of @('foo') (see @(see
 trans1)):</p>

 @({
  ACL2 !>:trans1 (foo expr1 expr2)
   (LET ((X EXPR1) (Y EXPR2))
        (IF (TEST X) (BAR Y) NIL))
  ACL2 !>
 })

 <p>Now imagine that @('expr2') is a complicated expression whose evaluation is
 intended only when the predicate @('test') holds of @('expr1').  The expansion
 above suggests that @('expr2') will always be evaluated by the call @('(foo
 expr1 expr2)'), which may be inefficient (since perhaps we only need that
 value when @('test') is true of @('expr1')).  The evaluation of @('expr2') may
 even cause an error, for example in @(':')@(tsee program) mode if the
 expression @('expr2') has been constructed in a manner that could cause a
 guard violation unless @('test') holds of @('expr1').</p>")

(defxdoc defabsstobj

; Warning: If you change the discussion below of
; books/demos/defabsstobj-example-1.lisp, change that book accordingly.

; Warning: Do not replace the use of <code>...</code> with the use of the more
; common xdoc marking @({...}), because the presence of curly braces in the
; text will make the rendered text a mess.

  :parents (events stobj)
  :short "Define a new abstract single-threaded object"
  :long "<p>We assume familiarity with single-threaded objects; see @(see
 stobj) and see @(see defstobj).  The event @('defabsstobj') defines a
 so-called ``abstract stobj'', a notion we introduce briefly now and then
 explain in more depth below.</p>

 <p>Recall that a @(see defstobj) event produces logical definitions for
 several functions: a recognizer, which characterizes the @(see stobj) in terms
 of lists; a creator, which produces an initial suitable list structure; and
 field accessors and updaters, defined in terms of @(tsee nth) and @(tsee
 update-nth).  @('Defabsstobj') provides a way to define alternate definitions
 for ``stobj primitives'' for a corresponding single-threaded object.  These
 stobj primitives include a recognizer, a creator, and other ``exported''
 functions.  In essence, @('defabsstobj') establishes interface functions, or
 ``exports'', on a new stobj that is a copy of an existing stobj, its
 ``foundation'', which is either <i>concrete</i> (introduced by @(tsee
 defstobj)) or <i>abstract</i> (introduced by @('defabsstobj')).</p>

 <p>We begin below with an introduction to abstract @(see stobj)s.  We then
 explain the @(tsee defabsstobj) event by way of an example.  We conclude by
 giving summary documentation for the @('defabsstobj') event.</p>

 <p>For another introduction to abstract stobjs, see the paper ``Abstract
 Stobjs and Their Application to ISA Modeling'' by Shilpi Goel, Warren A. Hunt,
 Jr., and Matt Kaufmann, in the proceedings of <a
 href='http://www.cs.uwyo.edu/~ruben/acl2-13'>ACL2 Workshop 2013</a>.</p>

 <p><b>INTRODUCTION</b></p>

 <p>We start with a brief review of @(see stobj)s and some potential problems
 with them, followed by an introduction to abstract stobjs and how they can
 avoid these problems.  Prior experience with stobjs will probably help the
 reader to absorb the ideas below.</p>

 <p>Recall that single-threaded objects, or @(see stobj)s, provide a way for
 ACL2 users to stay within the ACL2 logic &mdash; where every data object is an
 atom or a @(tsee cons) of data objects &mdash; while obtaining the benefits of
 fast evaluation through destructive updates.  Consider for example this very
 simple event.</p>

 @({
  (defstobj st fld)
 })

 <p>This event introduces a recognizer, @('stp'), and a creator,
 @('create-st'), for a data structure consisting of a single field accessed and
 updated by functions @('fld') and @('update-fld'), respectively.  Each of
 these four primitive functions has both a logical definition, which is used
 when the prover reasons about the function, and an executable definition,
 which is used in raw Lisp.  In the logic, @('stp') recognizes objects that
 have the requisite fields.  In raw Lisp, there is a ``live stobj'', which is
 typically an array object whose fields correspond to those specified by the
 @(tsee defstobj) event.  (If there is a single stobj field that is an array or
 hash-table field, then that field is the entire stobj in raw Lisp; but we
 ignore that case below.)</p>

 <p>Here are the logical definition and the executable definition,
 respectively, that are introduced for the field accessor, @('fld'), introduced
 above.  Notice that since a stobj is represented in raw Lisp using an array,
 the raw Lisp accessor uses a raw Lisp array accessor, @('svref').  (You can
 see all the logical and executable definitions by evaluating the form
 @('(trace$ defstobj-axiomatic-defs defstobj-raw-defs)') before evaluating the
 @(tsee defstobj) form.)</p>

 @({
  ; logical definition
  (defun fld (st)
    (declare (xargs :guard (stp st)
                    :verify-guards t))
    (nth 0 st))

  ; executable (raw Lisp) definition
  (defun fld (st)
    (svref st 0))
 })

 <p>Sophisticated programming with stobjs can provide efficient implementations
 of algorithms, but may require the preservation of a complex invariant.  One
 can, of course, define a function to implement such an invariant after
 introducing the stobj, as follows.</p>

 @({
  ; Introduce a stobj.
  (defstobj st fld1 ... fldk)

  ; Define an invariant on that stobj.
  (defun good-stp (st)
    (declare (xargs :stobjs st))
    ...)

  ; Define some basic functions that update the stobj and preserve the
  ; invariant.
  (defun update-st (... st ...)
    (declare (xargs :stobjs st
                    :guard (and (good-stp st) ...)))
    ...)
  ...

  ; Prove that the invariant is indeed preserved by those basic functions.
  (defthm good-stp-update-st
    (implies (and (good-stp st)
                  ...)
             (good-stp (update-st ... st ...))))
  ...

  ; Implement algorithms built on the basic functions.
  (defun foo (... st ...)
    (declare (xargs :stobjs st
                    :guard (and (good-stp st) ...)))
    ... (update-st ... st ...) ...)

  ; Prove invariance theorems about these algorithms.
  (defthm good-stp-foo
    (implies (and (good-stp st)
                  ...)
             (good-stp (foo ... st ...))))
  ...

  ; Prove other properties of these algorithms.
  (defthm foo-is-correct
    (implies (and (good-stp st)
                  ...)
             (some-property (foo ... st ...))))
  ...
 })

 <p>But there are at least two potential difficulties in using stobjs as
 described above.</p>

 <ol>

 <li>When @('foo') is executed on concrete data in the ACL2 loop,
 the guard check may be expensive because @('(good-stp st)') is expensive.</li>

 <li>Reasoning about @('foo') (using rules like @('foo-is-correct') above)
 involves proving hypotheses of invariance theorems, which may be complicated
 for the user to manage or slow for the theorem prover.</li>

 </ol>

 <p>The @('defabsstobj') event offers an opportunity to address these issues.
 It introduces a new stobj, which we call an ``abstract stobj'', which is
 associated with a corresponding ``foundational stobj'' introduced by an
 earlier @(tsee defstobj) or @('defabsstobj') event.  The @('defabsstobj')
 event specifies a logical (@(':LOGIC')) and an executable (@(':EXEC'))
 definition for each primitive operation, or ``stobj primitive'', involving
 that stobj.  As is the case for @(tsee defstobj), the logical definition is
 what ACL2 reasons about, and is appropriate to apply to an ACL2 object
 satisfying the logical definition of the recognizer function for the stobj.
 The executable definition is applied in raw Lisp to a live stobj (as discussed
 above).</p>

 <p>Remark.  It is common to use ``a'' and ``c'' in a suffix to suggest
 ``abstract'' and ``concrete'', respectively.  The foundational stobj was, at
 one time, called the ``corresponding concrete stobj''.  That old terminology
 may still be appropriate in the common case that the foundational stobj is a
 concrete stobj (rather than another abstract stobj).  So below, a name like
 @('st$c0') suggests a foundational (``concrete'') stobj for an abstract stobj
 named @('st'), whose abstract stobj recognizer is @('st$ap'), and so on.  End
 of Remark.</p>

 <p>We can picture a sequence of updates to an abstract stobj and its
 foundational stobj.  Initially in this picture, @('st$a0') and @('st$c0') are
 an abstract stobj and its foundation (respectively).  Then an update, @('u1'),
 is applied with @(':LOGIC') and @(':EXEC') functions @('u$a1') and @('u$c1'),
 respectively.  The resulting abstract and foundational stobj, @('st$a1') and
 @('st$c1'), correspond as before.  Then a second update, @('u2'), is applied
 with @(':LOGIC') and @(':EXEC') functions @('u$a2') and @('u$c2'),
 respectively &mdash; again preserving the correspondence.  And so on.</p>

 @({
  Abstract               u$a1       u$a2       u$a3
  (:logic)         st$a0  --> st$a1  --> st$a2  -->   ...

                     ^          ^          ^               ^
  Correspondence     |          |          |          ...  |
                     v          v          v               v

                         u$c1       u$c2       u$c3
  Foundation       st$c0  --> st$c1  --> st$c2  -->   ...
  (:exec)
 })

 <p>We conclude this introduction with some remarks about implementation.
 Consider an abstract stobj @('st') with corresponding foundation @('st$c').
 The live stobjs for @('st') and @('st$c') have the same structure, but are
 distinct arrays.  Indeed, the raw Lisp creator function for @('st$c') is
 called to create a new initial live stobj for @('st').  As we will see below,
 reads and writes in raw Lisp to the live stobj for @('st') are ultimately
 performed using the primitive accessors and updaters defined for @('st$c').
 One might think of the live stobjs for @('st') and @('st$c') as being
 congruent stobjs (see @(see defstobj)), except that the stobjs themselves are
 not truly congruent: in particular, the stobj primitives introduced for
 @('st') may be applied to @('st'), but field updaters of @('st$c') may not.
 As one might expect, the @(':EXEC') function for an exported function is
 applied to the live stobj for @('st') in raw Lisp.</p>

 <p><b>EXAMPLE</b></p>

 <p>We present examples, with detailed comments intended to explain abstract
 stobjs, in two community books: @('books/demos/defabsstobj-example-1.lisp')
 and @('books/demos/defabsstobj-example-2.lisp').  In this section we outline
 the first of these.  We suggest that after you finish this @(see
 documentation) topic, you read through those two books.  There are other books
 @('books/dmeos/defabsstobj-example-*.lisp') that may be helpful to read; in
 particular, @('books/demos/defabsstobj-example-5.lisp') illustrates building
 an abstract stobj on top of another abstract stobj (as its so-called
 ``foundation'', as described below).</p>

 <p>Here is the first of two closely related @('defabsstobj') @(see events)
 from the book @('defabsstobj-example-1.lisp'), but in expanded form.  We will
 show the abbreviated form later, which omits most of the data in the form that
 is immediately below.  Thus most of the information shown here is default
 information.  We believe that the comments below explain most or all of what
 you need to know in order to start using @('defabsstobj'), and that you will
 learn the remainder when you see error messages.  For example, we do not say
 in the comments below that every @(':LOGIC') and @(':EXEC') function must be
 @(see guard)-verified, but that is indeed a requirement.</p>

 <code>
 (defabsstobj st ; The new abstract stobj is named st.

 ; The foundational stobj for st is st$c:

   :foundation st$c

 ; The recognizer for the new abstract stobj is stp, which is defined to be
 ; st$ap in the logic, and is executed on the live stobj in raw Lisp using
 ; st$cp.

   :recognizer (stp :logic st$ap :exec st$cp)

 ; The initial stobj is defined as create-st (a function of no arguments),
 ; which is defined logically as create-st$a, though create-st$c is invoked to
 ; create the initial live stobj for st.  The :correspondence and :preserved
 ; keywords refer to proof obligations, discussed below.

   :creator (create-st :logic create-st$a :exec create-st$c
                       :correspondence create-st{correspondence}
                       :preserved create-st{preserved})

 ; Proof obligations are generated that involve a correspondence between the
 ; new abstract stobj and corresponding foundational stobj.  The function
 ; st$corr, which need not be executable (see :DOC defun-nx), takes two
 ; arguments, a foundational stobj and an abstract stobj.  This function symbol
 ; is used in the statements of the proof obligations.

   :corr-fn st$corr

 ; In this example we have four exports.  In each case a new function is
 ; introduced that has the same signature as its :EXEC function, except that
 ; st$c is replaced by st.  The :LOGIC and :EXEC functions are as specified,
 ; and the other keywords refer to proof obligations that we discuss below.

   :exports ((lookup :logic lookup$a
                     :exec mem$ci
                     :correspondence lookup{correspondence}
                     :guard-thm lookup{guard-thm})
             (update :logic update$a
                     :exec update-mem$ci
                     :correspondence update{correspondence}
                     :preserved update{preserved}
                     :guard-thm update{guard-thm})
             (misc :logic misc$a
                   :exec misc$c
                   :correspondence misc{correspondence})
             (update-misc :logic update-misc$a
                          :exec update-misc$c
                          :correspondence update-misc{correspondence}
                          :preserved update-misc{preserved})))
 </code>

 <p>Note that all stobj primitives (recognizer, creator, and exported
 functions) are defined in the ACL2 loop in terms of their @(':LOGIC')
 functions and in raw Lisp in terms of their @(':EXEC') functions.  In the ACL2
 loop, a @(tsee defun) form defines a function, while in raw Lisp, a @(tsee
 defmacro) form defines a macro (for efficiency).  We first illustrate how that
 works for the recognizer.  (You can see all the logical and executable
 definitions by evaluating the form @('(trace$ defabsstobj-axiomatic-defs
 defabsstobj-raw-defs)') before evaluating the @(tsee defstobj) form.)</p>

 @({
  ; In the ACL2 loop:
  (defun stp (st)
    (declare (xargs :guard 't))
    (st$ap st))

  ; In raw Lisp:
  (defmacro stp (&rest args) (cons 'st$cp args))
 })

 <p>The definitions are made similarly for exported functions.  @(csee Guard)s
 are derived from their @(':LOGIC') functions as follows.  Consider the
 exported function @('update') in our example.  Its @(':LOGIC') function,
 @('update$a'), has formals @('(k val st$a)') and the following guard.</p>

 @({
  (and (and (integerp k) (<= 0 k) (<= k 49))
       (and (integerp val) (<= 0 val))
       (st$ap st$a)
       (mem$c-entryp val))
 })

 <p>The formals of @('update') are obtained by starting with the formals of its
 @(':EXEC') function, @('update-mem$ci') &mdash; which are @('(i v st$c)')
 &mdash; and replacing the foundational stobj name @('st$c') by the new stobj
 name @('st').  The formals of @('update') are thus @('(i v st)').  The guard
 for @('update') is obtained in two steps.  The first step is to substitute the
 formals of @('update') for the formals of @('update$a') in the guard for
 @('update$a'), to obtain the following.</p>

 @({
  (and (and (integerp i) (<= 0 i) (<= i 49))
       (and (integerp v) (<= 0 v))
       (st$ap st)
       (mem$c-entryp v))
 })

 <p>The second step is to replace, for each new stobj primitive @('p'), the
 @(':LOGIC') function for @('p') by @('p') itself.  The only @(':LOGIC')
 function occurring in the formula just above is @('st$ap'), which is the
 @(':LOGIC') function for @('stp').  The guard for @('update') is thus as
 follows.</p>

 @({
  (and (and (integerp i) (<= 0 i) (<= i 49))
       (and (integerp v) (<= 0 v))
       (stp st)
       (mem$c-entryp v))
 })

 <p>Note that the @(':EXEC') version of an abstract @(see stobj) export must
 not include the abstract stobj name among its formals.</p>

 <p>We turn now to the proof obligations, as promised above.  There are three
 types: @(':CORRESPONDENCE'), @(':PRESERVED'), and @(':GUARD-THM').  All
 required lemmas may be printed simply by defining the necessary @(':LOGIC')
 and @(':EXEC') functions and then submitting the @('defabsstobj') event.  (To
 advanced users: also see @(see defabsstobj-missing-events) for a utility that
 returns the required formulas in translated form.)  Although the
 @('defabsstobj') event will fail if the required lemmas have not been proved,
 first it will print the @(tsee defthm) forms that must be admitted in order to
 complete submission of the @('defabsstobj') event.  (Note that although those
 theorems are stated exactly in the form expected by the system, you are
 welcome to supply whatever @(':')@(tsee rule-classes) you prefer, even though
 the system creates @(':rule-classes nil') by default.)</p>

 <p>The detailed theory explaining the need for these lemmas may be found in
 ACL2 source file @('other-events.lisp'), in a comment entitled ``Essay on the
 Correctness of Abstract Stobjs''.  Here, we give an informal sense of the
 importance of these lemmas as we present examples of them.  Fundamental is the
 notion of evaluation in the logic versus evaluation using live stobjs, where
 one imagines tracking the current value of each abstract stobj during each of
 these two evaluations.</p>

 <p>We start with the @(':CORRESPONDENCE') lemmas.  These guarantee that
 evaluation in the logic agrees with evaluation using live stobjs, in the sense
 that the only difference is between a logical stobj and a live stobj, where
 the two correspond in the sense of the function specified by @(':CORR-FN').
 We start with the @(':CREATOR') function where the statement is quite simple,
 stating that the @(':CORR-FN') holds initially.</p>

 @({
  (defthm create-st{correspondence}
    (st$corr (create-st$c) (create-st$a)))
 })

 <p>For the exported functions, there are essentially two cases.  If an
 exported function returns other than the new abstract stobj, then the theorem
 asserts the equality of the results of applying the @(':LOGIC') and @(':EXEC')
 functions for the exported function.  Hypotheses include the @(':CORR-FN')
 correspondence followed by the @(see guard) for the @(':LOGIC') function,
 which is stated in terms of the formal parameters of the @(':EXEC') function
 except using the abstract stobj (here, @('st')) in place of the foundational
 stobj (here, @('st$c')).  The conclusion uses the @(':EXEC') formals, modified
 in the call of the @(':LOGIC') function (here, @('lookup$a')) to use the
 abstract stobj, as in the hypotheses.</p>

 @({
  (defthm lookup{correspondence}
    (implies (and (st$corr st$c st)
                  (integerp i) (<= 0 i) (<= i 49)
                  (st$ap st))
             (equal (mem$ci i st$c)
                    (lookup$a i st)))
    :rule-classes nil)
 })

 <p>By contrast, if the exported function returns the new abstract stobj, then
 the conclusion uses the correspondence function instead of @('EQUAL'), as in
 the following.</p>

 @({
  (defthm update{correspondence}
    (implies (and (st$corr st$c st)
                  (integerp i) (<= 0 i) (<= i 49)
                  (integerp v) (<= 0 v)
                  (st$ap st)
                  (mem$c-entryp v))
             (st$corr (update-mem$ci i v st$c)
                      (update$a i v st)))
    :rule-classes nil)
 })

 <p>For exported functions that return multiple values, such conclusions are
 conjoined together over the returned values.</p>

 <p>The @(':PRESERVED') lemmas guarantee that updates to the abstract stobj
 preserve its recognizer.  The fact that every exported function has this
 property provides justification for an optimization performed by ACL2 during
 generation of proof obligations for @(see guard) verification, by assuming
 that the recognizer always holds.  The @(':PRESERVED') lemma for the
 @(':CREATOR') shows that the recognizer holds initially.</p>

 @({
  (defthm create-st{preserved}
    (st$ap (create-st$a)))
 })

 <p>Here is a typical such lemma, for the exported function @('update').  Note
 that there is no such lemma for @('lookup'), since @('lookup') does not return
 @('st').</p>

 @({
  (defthm update{preserved}
    (implies (and (integerp i) (<= 0 i) (<= i 49)
                  (integerp v) (<= 0 v)
                  (st$ap st)
                  (mem$c-entryp v))
             (st$ap (update$a i v st))))
 })

 <p>Finally, we consider the @(':GUARD-THM') lemmas.  These serve to guarantee
 that the @(see guard) holds for each call of an @(':EXEC') function.  During
 guard verification, logical definitions are used; in particular, since each
 exported function is defined in the logic as the corresponding call of its
 @(':LOGIC') function, guard verification shows that each call of the
 @(':LOGIC') function for an exported function satisfies that function's guard.
 But why is this true for raw Lisp evaluation using live stobjs, where the
 @(':EXEC') function is called for an exported function?  The @(':GUARD-THM')
 lemmas provide the answer, as they state that if the @(':LOGIC') function's
 guard holds, then the @(':EXEC') function's guard holds.  Here is an example.
 Note that the hypotheses come from the correspondence of the foundational and
 abstract function as guaranteed by the @(':CORR') function, together with the
 guard of the @(':LOGIC') function; and the conclusion comes from the guard of
 the @(':EXEC') function.</p>

 @({
  (defthm lookup{guard-thm}
    (implies (and (st$corr st$c c)
                  (integerp i)
                  (<= 0 i)
                  (<= i 49)
                  (st$ap st))
             (and (integerp i)
                  (<= 0 i)
                  (< i (mem$c-length st$c))))
    :rule-classes nil)
 })

 <p>We conclude this EXAMPLE section by showing a short form for the
 @('defabsstobj') form displayed above.</p>

 @({
  (defabsstobj st
    :exports ((lookup :exec mem$ci)
              (update :exec update-mem$ci)
              misc update-misc))
 })

 <p><b>SUMMARY DOCUMENTATION</b></p>

 <p>The General Form is as shown below, where the order of keywords is
 unimportant.  Duplicate keywords are discouraged; while permitted, only the
 first (leftmost) occurrence of a given keyword is used.  Only the
 @(':exports') keyword is required.</p>

 @({
  (defabsstobj st
    :foundation foundation
    :recognizer recognizer
    :creator creator
    :corr-fn corr-fn
    :congruent-to congruent-to
    :protect-default protect-default
    :exports (e1 ... ek))
 })

 <p>The keyword argument @(':EXPORTS') must be supplied, and missing or
 @('nil') keyword arguments have defaults as indicated below.  All arguments
 must satisfy the conditions below.</p>

 <p>Before we describe the arguments, we define a notion of a ``function spec''
 and its ``completion''.  A function spec is either a symbol or else a list of
 the form</p>

 <code>
 (fn @(':kwd1') val1 ... @(':kwdn') valn), </code>

 <p>that is, a symbol followed by a @(tsee keyword-value-listp).  We view the
 case of a symbol, @('s'), as the function spec @('(s)'), with no keywords.
 There must be no duplicate keywords.  In each case that we expect a function
 spec, the context provides a set of valid keywords for that function spec; it
 is an error to provide any other keyword in the function spec.  Each function
 spec is interpreted as its ``completion'', obtained by extending the function
 spec with a default value for each valid keyword as indicated below.  With
 that interpretation, the ``exported function'' of a function spec is its
 @('car'), and that function symbol and each keyword value must be a
 guard-verified function symbol; and moreover, the @(':EXEC') function must not
 include the new abstract stobj name, @('st'), among its formals.</p>

 <p>We are ready to describe the arguments of @('defabsstobj').</p>

 <blockquote>

 <p>@('St') is a symbol, which names the new abstract stobj.</p>

 <p>@('Foundation') is the name of an existing stobj, which may have been
 introduced either with @(tsee defstobj) or with @('defabsstobj').</p>

 <p>@('Recognizer') is a function spec (for the recognizer function).  The
 valid keywords are @(':LOGIC') and @(':EXEC').  The default for
 @('recognizer') is obtained by adding the suffix @('\"P\"') to @('name').  The
 default value for @(':LOGIC') is formed by adding the suffix @('\"$AP\"') to
 @('recognizer'); for @(':EXEC'), by adding the suffix @('\"$CP\"').  The
 @(':EXEC') function must be the recognizer for the foundational stobj (which
 can be specified using the @(':FOUNDATION') keyword).</p>

 <p>@('Creator') is a function spec (for the creator function).  The valid
 keywords are @(':LOGIC') and @(':EXEC').  The default for @('creator') is
 obtained by adding the prefix @('\"CREATE-\"') to @('name').  The default
 value for @(':LOGIC') is formed by adding the suffix @('\"$A\"') to
 @('creator'); for @(':EXEC'), by adding the suffix @('\"$C\"').  The
 @(':EXEC') function must be the creator for the foundational stobj (which can
 be specified using the @(':FOUNDATION') keyword).</p>

 <p>@('Corr-fn') is a known function symbol that takes two arguments (for the
 correspondence theorems).  The default for @('corr-fn') is obtained by adding
 the suffix @('\"$CORR\"') to @('name').</p>

 <p>@('Congruent-to') should either be @('nil') (the default) or the name of an
 abstract stobj previously introduced (by @(tsee defabsstobj)).  In the latter
 case, the current and previous abstract stobj should have the same
 foundational stobj (not merely congruent foundational stobjs), and their
 @(':EXPORTS') fields should have the same length and also correspond, as
 follows: the ith export of each should have the same @(':LOGIC') and
 @(':EXEC') symbols.  See @(see defstobj) for more about congruent stobjs.
 Note that if two names are congruent, then they are either both ordinary
 stobjs or both abstract stobjs.</p>

 <p>@('Protect-default') should either be @('nil') (the default) or @('t').  It
 provides the value of keyword @(':PROTECT') for each member of @('exports')
 that does not explicitly specify @(':PROTECT').  See the discussion of
 @('exports') below.</p>

 <p>An important aspect of the @('congruent-to') parameter is that if it is not
 @('nil'), then the checks for lemmas &mdash; @('{CORRESPONDENCE}'),
 @('{GUARD-THM}'), and @('{PRESERVED}') &mdash; are omitted.  Thus, the values
 of keyword @(':CORR-FN'), and the values of keywords @(':CORRESPONDENCE'),
 @(':GUARD-THM'), and @(':PRESERVED') in each export (as we discuss next), are
 irrelevant; they are not inferred and they need not be supplied.</p>

 <p>The value of @(':EXPORTS') is a non-empty true list.  Each @('ei') is a
 function spec (for an exported function).  The valid keywords are @(':LOGIC'),
 @(':EXEC'), @(':CORRESPONDENCE'), and @(':GUARD-THM'), @(':PROTECT'),
 @(':UPDATER'), and also @(':PRESERVED') if and only if the specified
 @(':EXEC') function returns the foundational stobj.  The default values for
 all of these keywords except @(':UPDATER') and @(':PROTECT') are obtained by
 respectively adding the suffix @('\"$A\"') @('\"$C\"'),
 @('\"{CORRESPONDENCE}\"'), @('\"{GUARD-THM}\"'), or @('\"{PRESERVED}\"').  For
 @(':PROTECT'), the default is @('nil') unless the @('defabsstobj') event
 specifies @(':PROTECT-DEFAULT t').  If @(':UPDATER upd') is supplied and
 @('upd') is not @('nil'), then function exported by the function spec is a
 child stobj accessor whose corresponding updater is @('upd'); see the
 discussion of @(':UPDATER') in @(see nested-stobjs).</p>

 </blockquote>

 <p>Not shown is the keyword, @(':MISSING'); the effect of @(':missing t') is
 to turn the call of @('defabsstobj') into a corresponding call of @(tsee
 defabsstobj-missing-events).</p>

 <p>Note that a @('defabsstobj') event will fail if the required lemmas &mdash;
 that is, those for valid keywords @(':CORRESPONDENCE'), @(':GUARD-THM'), and
 @(':PRESERVED') &mdash; have not been proved, unless proofs are being skipped.
 The exemption when skipping proofs allows the supporting lemmas to be @(tsee
 local) to @(see books) and @(tsee encapsulate) @(see events).  If the @(tsee
 ld) special @(tsee ld-skip-proofsp) is @('t'), then the missing @(see events)
 are printed with a warning before the @('defabsstobj') event is admitted; but
 if @('ld-skip-proofsp') is the symbol @('INCLUDE-BOOK'), then that warning is
 omitted.  (Also see @(see skip-proofs) and see @(see ld-skip-proofsp).)  If
 however proofs are not being skipped, then the @('defabsstobj') event will
 fail after printing the missing events.  Advanced users may wish to see @(see
 defabsstobj-missing-events) for a utility that returns a data structure
 containing the missing lemmas.</p>

 <p>Let @('st') be an abstract stobj with corresponding foundational stobj
 @('st$c').  Let @('f') be an exported function for @('st') and let @('f$a')
 and @('f$c') be the corresponding @(':LOGIC') and @(':EXEC') functions,
 respectively.  The formals of @('f') are obtained by taking the formals of
 @('f$c') and replacing @('st$c') by @('st').  The @(see guard) for @('f') is
 derived as follows from the guard of @('f$a').  First, the formals of @('f$a')
 are replaced by the formals of @('f') in the guard of @('f$a'), to obtain a
 term we denote here as @('guard-pre').  Now for each exported function symbol
 @('g') of @('st') with corresponding @(':LOGIC') function @('g$a'), form a
 functional substitution by consing @('g$a') with @('g').  Finally, apply that
 functional substitution to @('guard-pre'); the result is the guard of @('f').
 That guard must satisfy the usual conditions of a guard: thus, it must return
 a single non-@(see stobj) value and satisfy normal syntactic restrictions,
 including single-threadedness in its handling of stobjs.</p>

 <p>@('Remark').  Because of how guards are created for exported functions, and
 in particular because @(':LOGIC') functions are replaced as discussed above, a
 good discipline is to define @(':LOGIC') functions that are not intended for
 general use, but are intended only for use as @(':LOGIC') functions of
 corresponding stobj primitives.  For example, suppose that you use @('length')
 as the @(':LOGIC') function for some stobj primitive, @('f') (as opposed to
 using your own function, say, @('foo-length') or @('foo$a')).  Then every call
 of @('length') will be replaced by @('f') when creating the guard of a stobj
 primitive from the guard of its @(':LOGIC') function.  This might not be what
 you intended if you were using @('length') in that guard simply to compute the
 length of an ordinary list.</p>

 <p>Additional restrictions include the following.</p>

 <ul>

 <li>All exported function names must be new (unless redefinition is on; see
 @(see ld-redefinition-action)), and there must be no duplicates among
 them.</li>

 <li>The foundational stobj name must be a formal parameter of the @(':EXEC')
 function of every function spec, except for the @(':CREATOR') function
 spec.</li>

 <li>The @(':LOGIC') and @(':EXEC') function for a function spec must agree on
 both the number of inputs and the number of outputs.</li>

 <li>The foundational stobj must not be a @(see declare)d stobj of the
 @(':LOGIC') function of any function spec.  (This restriction could perhaps be
 removed, but it is convenient for the implementation of the events generated
 by a call of @('defabsstobj').)</li>

 <li>The @(':PROTECT') keyword is something that you should ignore unless you
 get an error message about it, pertaining to modifying the foundational stobj
 non-atomically.  In that case, you can eliminate the error by providing
 @(':PROTECT t') in the function spec, or by providing @('defabsstobj') keyword
 argument @(':PROTECT-DEFAULT t') at the top level, in order to restore the
 required atomicity.  The above explanation is probably all you need to know
 about @(':PROTECT'), but just below is a more complete explanation for those
 who desire it.  Further information is also available if you need it; see
 @(see set-absstobj-debug), and see the example uses of these keywords in
 community book @('books/demos/defabsstobj-example-2.lisp').</li>

 </ul>

 <p>For those who are interested, here is a more detailed discussion of
 @(':PROTECT') and @(':PROTECT-DEFAULT'), as promised above.  It applies to any
 function spec for an export (hence not to the @(':CREATOR') function spec).
 If the @(':EXEC') function is a stobj primitive, then clearly the following
 property holds: any execution of a call of that function can only update the
 foundational stobj at most once &mdash; i.e., modification of the foundational
 stobj is atomic.  ACL2 can deduce this property not only for stobj primitives
 but for many other functions as well.  However, if ACL2 cannot deduce this
 property, then it will cause an error saying that the @(':EXEC') function
 ``appears capable of modifying the foundational stobj, @('<stobj_name>'),
 non-atomically.''  That message also explains how to eliminate this error:
 provide @(':PROTECT t') for the function spec.  Alternatively, all function
 specs without an explicit @(':PROTECT') keyword can be implicitly supplied
 @(':PROTECT t') by supplying the value @('t') for the @(':PROTECT-DEFAULT')
 keyword parameter of the @('defabsstobj') event.  However, beware that when
 @(':PROTECT') is @('t'), the generated raw Lisp code runs slightly less
 efficiently &mdash; though perhaps with negligible efficiency loss if the
 @(':EXEC') function is not trivial.  Community books
 @('books/demos/defabsstobj-example-3.lisp') and
 @('books/demos/defabsstobj-example-4.lisp') provide related information.  Also
 see @(see set-absstobj-debug) for a potentially dangerous way to eliminate
 that inefficiency using argument @(':ignore').</p>

 <p>We conclude with some remarks.</p>

 <p>Unlike @(tsee defstobj), there is no @(':renaming') argument.  Instead, the
 scheme described above provides a flexible way to assign names.  Also unlike
 @(tsee defstobj), there is no @(':inline') or @(':non-memoizable') argument;
 @(':inline') is essentially t, in the sense that stobj primitives are macros
 in raw Lisp; and the @(':non-memoizable') argument is derived implicitly, to
 agree with non-memoizability of the foundational stobj.</p>

 <p>Those who use @(see hons-enabled) features, including function
 memoization (see @(see memoize)), may be aware that the memo table for a
 function is flushed whenever it is the case that one of its stobj inputs is
 updated.  In fact, such flushing happens even when a stobj that is congruent
 to one of its stobj inputs is updated.  For that purpose, an abstract stobj is
 considered to be congruent to its foundational stobj.</p>")

(defxdoc defabsstobj-missing-events
  :parents (events)
  :short "Obtain the @(see events) needed to admit a @(tsee defabsstobj) event"
  :long "<p>We assume familiarity with @(tsee defabsstobj).
 @('Defabsstobj-missing-events') is a macro is for advanced users (who, for
 example, understand the role of the @('translate') and @('untranslate')
 functions), who want programmatic access to the @(tsee defthm) events required
 to admit a specific @('defabsstobj') event.</p>

 <p>This macro has the same syntax as @(tsee defabsstobj) &mdash; to use it,
 just replace a call of @(tsee defabsstobj) by a call of
 @('defabsstobj-missing-events') on the same arguments.  The result is an error
 triple @('(mv erp val state)').  If @('erp') is @('nil'), then @('val') is the
 list of all objects @('(name formula . old-formula)'), where a @(tsee defthm)
 event named @('name') remains to be admitted whose translated formula is
 @('formula'), and where @('old-formula') is @('nil') unless the indicated
 event already exists (hence with a different formula), in which case
 @('old-formula') is the existing translated formula.</p>

 <p>To build a @(tsee defthm) event from the above value, @('val'), we suggest
 evaluating a form like @('(untranslate formula t (w state))').</p>")

(defxdoc defattach
  :parents (events)
  :short "Execute constrained functions using corresponding attached functions"
  :long "
 @({
  General Forms:
  (defattach f g)   ; single attach or, if g is nil, unattach
  (defattach (f1 g1 :kwd val ...)
             ...
             (fk gk :kwd' val' ...)
             :kwd'' val'' ...)
 })

 <p>where each indicated keyword-value pair is optional and each keyword is in
 the list @(`*defattach-keys-extended*`).  More details are in the ``Syntax and
 Semantics'' section below.</p>

 <p>A related utility can cause a function call to be evaluated using an
 alternate, provably equal function.  See @(see memoize), option
 @(':INVOKE').</p>

 <p>This @(see documentation) topic is organized into the following
 sections:</p>

 <ul>

 <li>Introductory Example.</li>

 <li>Syntax and Semantics of Defattach.</li>

 <li>Three Primary Uses of Defattach.</li>

 <li>Miscellaneous Remarks, with discussion of possible user errors.</li>

 </ul>

 <p>Please see @(see encapsulate) if you intend to use @('defattach') but are
 not already familiar with the use of @('encapsulate') to introduce constrained
 functions.  It may also be helpful to see @(see evaluation).</p>

 <p>See community book @('books/misc/defattach-example.lisp') for a small
 example.  it illustrates how @('defattach') may be used to build something
 like ``higher-order'' programs, in which constrained functions may be refined
 to different executable functions.  More uses of @('defattach') may be found
 in the ACL2 source code, specifically, file @('boot-strap-pass-2-a.lisp').</p>

 <h3>Introductory Example.</h3>

 <p>We begin with a short log illustrating the use of @('defattach').  Notice
 that after evaluating the event @('(defattach f g)'), a call of the
 constrained function @('f') is evaluated by instead calling @('g') on the
 arguments.</p>

 @({
  ACL2 !>(encapsulate
          ((f (x) t :guard (true-listp x)))
          (local (defun f (x) x))
          (defthm f-property
            (implies (consp x) (consp (f x)))))
  [... output omitted ...]
   T
  ACL2 !>(defun g (x)
           (declare (xargs :guard (or (consp x) (null x))))
           (cons 17 (car x)))
  [... output omitted ...]
   G
  ACL2 !>(f '(3 4)) ; undefined function error

  ACL2 Error in TOP-LEVEL:  ACL2 cannot ev the call of undefined function
  F on argument list:

  ((3 4))

  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.

  ACL2 !>(defattach f g)
  [... output omitted ...]
   :ATTACHMENTS-RECORDED
  ACL2 !>(f '(3 4)) ; f is evaluated using g
  (17 . 3)
  ACL2 !>(trace$ f g)
   ((F) (G))
  ACL2 !>(f '(3 4)) ; f is evaluated using g
  1> (ACL2_*1*_ACL2::F (3 4))
    2> (ACL2_*1*_ACL2::G (3 4))
      3> (G (3 4))
      <3 (G (17 . 3))
    <2 (ACL2_*1*_ACL2::G (17 . 3))
  <1 (ACL2_*1*_ACL2::F (17 . 3))
  (17 . 3)
  ACL2 !>(defattach f nil) ; unattach f (remove its attachment)
  [... output omitted ...]
   :ATTACHMENTS-RECORDED
  ACL2 !>(f '(3 4)) ; undefined function error once again
  1> (ACL2_*1*_ACL2::F (3 4))

  ACL2 Error in TOP-LEVEL:  ACL2 cannot ev the call of undefined function
  F on argument list:

  ((3 4))

  To debug see :DOC print-gv, see :DOC trace, and see :DOC wet.

  ACL2 !>
 })

 <h3>Syntax and Semantics of Defattach.</h3>

 <p>The log above shows that the event @('(defattach f g)') allows @('g') to be
 used for evaluating calls of @('f').  From a logical perspective, the
 evaluation takes place in the addition to the current session of an
 ``attachment equation'' axiom (universally quantified over all @('x')) for
 each @('defattach') event:</p>

 @({
  (equal (f x) (g x)) ;;; attachment equation axiom for (defattach f g)
 })

 <p>Below we explain @('defattach') in some detail.  But it is important to
 keep in mind that evaluation with the attachment equations takes place in an
 extension of the logical theory of the session.  ACL2 guarantees that this
 so-called ``evaluation theory'' remains consistent, assuming the absence of
 @(tsee defaxiom) @(see events) from the user.  This guarantee is a consequence
 of a more general guarantee: an ACL2 logical @(see world) exists in which
 (loosely speaking) the attachment equation for @('(defattach f g)'), as
 @('(defun f (...) (g ...))'), takes the place of the original defining event
 for @('f'), for each @('defattach') event.  This more general guarantee holds
 even if there are @(tsee defaxiom) events, though as explained below, no
 function symbol that syntactically supports a @('defaxiom') formula is allowed
 to get an attachment.  A deeper discussion of the logical issues is available
 (but not intended to be read by most users) in a long comment in the ACL2
 source code labeled ``Essay on Defattach.''</p>

 @({
  Example Forms:
  (defattach f g)   ; call g in place of calling constrained function f
  (defattach (f g)) ; same as just above
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo)))))
                    ; equivalent to first form above, except with hints for the
                    ; proof that the guard of f implies the guard of g
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo)))
                  :otf-flg t))
                    ; as above, except with an :otf-flg of t for the proof that
                    ; the guard of f implies the guard of g
  (defattach (f g)
             :hints ((\"Goal\" :use my-thm)))
                    ; equivalent to first form above, except with hints for the
                    ; proof that the constraints on f hold for g
  (defattach (f g)
             :hints ((\"Goal\" :use my-thm))
             :otf-flg t)
                    ; as above, except with an :otf-flg of t for the proof that
                    ; the constraints on f hold for g
  (defattach (f g)
             (h j)) ; Attach g to f and attach j to h
  (defattach (f g :attach nil)
             (h j)) ; Same as just above, including the same proof obligations,
                    ; except for one difference: because of :attach nil, calls
                    ; of f will not be evaluated, i.e., there will be no
                    ; executable attachment of g to f
  (defattach (f nil)
             (h j)) ; Attach j to h and unattach f
  (defattach (f g :hints ((\"Goal\" :in-theory (enable foo))))
             (h j :hints ((\"Goal\" :in-theory (enable bar))))
             :hints ((\"Goal\" :use my-thm)))
                    ; Attach g to f and attach j to h, with hints:
                    ; - For proving that the guard of f implies the guard of g,
                    ;   enable foo;
                    ; - For proving that the guard of h implies the guard of j,
                    ;   enable bar; and
                    ; - For proving that the constraints on f and h hold for
                    ;   g and j (respectively), use theorem my-thm.

  (defattach f nil)   ; remove the attachment of f, if any (e.g., g above)
  (defattach (f nil)) ; same as just above

  General Forms:
  (defattach f g)   ; single attach or, if g is nil, unattach
  (defattach (f1 g1 :kwd11val11 ...)
             ...
             (fk gk :kwdk1 valk1 ...)
             :kwd1 val1 ...)
 })

 <p>where each indicated keyword-value pair is optional and each keyword is in
 the list @(`*defattach-keys-extended*`).  We distinguish between keywords
 within the @('(fi gi :kwdi1 vali1 ...)'), which we call <i>guard keywords</i>,
 and keywords at the top level, shown above as @(':kwd1 val1 ...'), which we
 call <i>top-level keywords</i>.</p>

 <ul>

 <li>The guard keywords are in the list @(`*defattach-keys*`).  The
 @(':')@(tsee hints), @(':')@(tsee instructions), and @(':')@(tsee otf-flg)
 keywords in @('(fi gi ...)') are used in the proofs of the guard proof
 obligation for the attachment of @('gi') to @('fi').  They have their usual
 values and meanings, as when used (for example) in @(tsee defthm) @(tsee
 events).  The value of each @(':attach') keyword is either @('t') or @('nil').
 We discuss the @(':attach') keyword later in this @(see documentation)
 topic.</li>

 <li>The top-level keywords @(':hints'), @(':instructions'), and @(':otf-flg')
 are used in the constraint proof obligations just as described above for the
 guard proof obligations.  When @(':attach') is used as a top-level keyword,
 its value serves as a default for entries @('(fi gi ...)') that do not specify
 @(':attach').  @(':Skip-checks') and @(':system-ok') are described below.</li>

 </ul>

 <p>No keyword may occur twice in the same context: that is, neither twice as a
 guard keyword in the same @('(fi gi ...)')  entry, nor twice as a top-level
 keyword.  Moreover, @(':instructions') may not occur in the same context with
 @(':hints') or @(':otf-flg').</p>

 <p>The argument @(':skip-checks t') enables easy experimentation with
 @('defattach'), by permitting use of @(':')@(tsee program) mode functions and
 the skipping of semantic checks.  Also permitted is @(':skip-checks nil') (the
 default) and @(':skip-checks :cycles'), which turns off only the update of the
 extended ancestor relation and hence the check for cycles in this relation;
 see below.  We do not make any logical claims when the value of
 @(':skip-checks') is non-@('nil'); indeed, a trust tag is then required (see
 @(see defttag)).  Note that the interaction of @(see memoization) and
 attachments is not tracked for attachments introduced with a non-@('nil')
 value of @(':skip-checks').  For more discussion of @(':skip-checks t'), see
 @(see defproxy); we do not discuss @(':skip-checks') further, here.</p>

 <p>The argument @(':system-ok t') allows attachment to system functions.
 Without this argument, the @('defattach') event will fail if any @('fi') is a
 built-in ACL2 function.  Rather than supplying this argument directly, it is
 recommended to use @(tsee defattach-system), which has the same syntax as
 @('defattach') with two exceptions: it adds @(':system-ok t') automatically,
 that is, @(':system-ok') is implicit; and it expands to a @(tsee local) call
 of @('defattach').  The latter is important so that the attachment does not
 affect system behavior outside a book containing the @('defattach') event.  Of
 course, if it is truly intended to affect such behavior, the argument
 @(':system-ok t') may be given directly to @('defattach'), without a
 surrounding use of @('local').</p>

 <p>The first General Form above is simply an abbreviation for the form
 @('(defattach (f g))'), which is an instance of the second General Form above.
 For the second General Form we say that @('gi') is ``attached to'' @('fi') (by
 the @('defattach') event) if @('gi') is not @('nil'), and otherwise we say
 that @('fi') is ``unattached'' (by the @('defattach') event).  It is also
 convenient to refer to @('<fi,gi>') as an ``attachment pair'' (of the event)
 if @('gi') is not @('nil').  We may refer to the set of @('fi') as the
 ``attachment nest'' of each @('fi').</p>

 <p>We start with a brief introduction to the first General Form in the case
 that @('g') is not @('nil').  This form arranges that during evaluation, with
 exceptions noted below, every call of the constrained function symbol @('f')
 will in essence be replaced by a call of the function symbol @('g') on the
 same arguments.  We may then refer to @('g') as the ``attachment of'' @('f'),
 or say that ``@('g') is attached to @('f').''  Notable exceptions, where we do
 not use attachments during evaluation, are for macroexpansion, evaluation of
 @(tsee defconst) and @(tsee defpkg) terms, evaluation during @(tsee table)
 events, some @(see stobj) operations including all <see topic='@(url
 STOBJ)'>updates</see>, and especially evaluation of ground terms (terms
 without free variables) during proofs.  However, even for these cases we allow
 the use of attachments in the first argument of @(tsee prog2$) and, more
 generally, the next-to-last (i.e., second) argument of @(tsee return-last)
 when its first argument is not of the form @(''m') for some macro, @('m').</p>

 <p>To see why attachments are disallowed during evaluation of ground terms
 during proofs (except for the @(tsee prog2$) and @(tsee return-last) cases
 mentioned above), consider the following example.</p>

 @({
  (defstub f (x) t)
  (defun g (x) (+ 3 x))
  (defattach f g)
 })

 <p>If the form @('(f 2)') is submitted at the ACL2 prompt, the result will be
 @('5') because the attachment @('g') of @('f') is called on the argument,
 @('2').  However, during a proof the term @('(f 2)') will not be simplified to
 @('5'), since that would be unsound, as there are no axioms about @('f') that
 would justify such a simplification.</p>

 <p>For the case that @('g') is @('nil') in the first General Form above, the
 result is the removal of the existing attachment to @('f'), if any.  After
 this removal, calls of @('f') will once again cause errors saying that ``ACL2
 cannot ev the call of undefined function @('f') ...''.  In this case not only
 is the previous attachment to @('f') removed; moreover, for every function
 symbol @('f'') in the attachment nest of @('f') in the @('defattach') event
 that introduced the existing attachment to @('f'), then @('f'') is unattached.
 (An example near the end of this @(see documentation) topic shows why this
 unattachment needs to be done.) Such removal takes place before the current
 @('defattach') is processed, but is restored if the new event fails to be
 admitted.</p>

 <p>We focus henceforth on the second General Form.  There must be at least one
 attachment, i.e., @('i') must be at least 1.  All keywords are optional; their
 role is described below.  The @('fi') must be distinct constrained function
 symbols, that is, function symbols all introduced in @(see signature)s of
 @(tsee encapsulate) @(see events) (or macros such as @(tsee defstub) that
 generate @(tsee encapsulate) events).  Each non-@('nil') @('gi') is a
 @(':')@(tsee logic)-mode function symbol that has had its guards verified,
 with the same @(see signature) as @('fi') (though formal parameters for
 @('fi') and @('gi') may have different names).  (Note: The macro
 @('defattach!'), defined in community book @('books/misc/defattach-bang'),
 avoids this restriction.)  This event generates proof obligations and an
 ordering check, both described below.  The effect of this event is first to
 remove any existing attachments for all the function symbols @('fi'), as
 described above for the first General Form, and then to attach each @('gi') to
 @('fi').</p>

 <p>Proof obligations must be checked before making attachments.  For this
 discussion we assume that each @('gi') is non-@('nil') (otherwise first remove
 all attachment pairs @('<fi,gi>') for which @('gi') is nil).  Let @('s') be
 the functional substitution mapping each @('fi') to @('gi').  For any term
 @('u'), we write @('u\\s') for the result of applying @('s') to @('u'); that
 is, @('u\\s') is the ``functional instance'' obtained by replacing each
 @('fi') by @('gi') in @('u').  Let @('G_fi') and @('G_gi') be the guards of
 @('fi') and @('gi'), respectively.  Let @('G_fi'') be the result of replacing
 each formal of @('fi') by the corresponding formal of @('gi') in @('G_fi').
 ACL2 first proves, for each @('i') (in order), the formula @('(implies G_fi'
 G_gi)\\s').  If this sequence of proofs succeeds, then the remaining formula
 to prove is the functional instance @('C\\s') of the conjunction @('C') of the
 constraints on the symbols @('fi'); see @(see constraint).  This last proof
 obligation is thus similar to the one generated by functional instantiation
 (see @(see constraint)).  As with functional instantiation, ACL2 stores the
 fact that such proofs have been done so that they are avoided in future events
 (see @(see lemma-instance)).  Thus, you will likely avoid some proofs with the
 sequence</p>

 @({
  (defattach f g)
  (defattach f nil)
  (defattach f g)
  (defattach f nil)
  ...
 })

 <p>rather than the sequence:</p>

 @({
  (defattach f g)
  :u
  (defattach f g)
  :u
  ...
 })

 <p>It remains to describe an ordering check.  We begin with the following
 motivating example.</p>

 @({
  (defstub f (x) t) ; constrained function with no constraints
  (defun g (x) (declare (xargs :guard t)) (not (f x)))
  (defattach f g) ; ILLEGAL!
 })

 <p>Were the above @('defattach') event to succeed, the evaluation theory
 (discussed above) would be inconsistent: @('(f x)') equals @('(g x)') by the
 new attachment equation, which in turn equals @('(not (f x))') by definition
 of @('g').  The evaluation would therefore be meaningless.  Also, from a
 practical perspective, there would be an infinite loop resulting from any call
 of @('f').</p>

 <p>We consider a function symbol @('g') to be an ``extended immediate ancestor
 of'' a function symbol @('f') if either of the following two criteria is met:
 (a) @('g') occurs in the formula that introduces @('f') (i.e., definition body
 or constraint) and @('g') is introduced by an event different from (earlier
 than) the event introducing @('f'); or (b) @('g') is attached to @('f').  For
 a proposed @('defattach') event, we check that this relation has no cycles,
 where for condition (b) we include all attachment pairs that would result,
 including those remaining from earlier @('defattach') events.</p>

 <p>Of course, a special case is that no function symbol may be attached to
 itself.  Similarly, no function symbol may be attached to any of its
 ``siblings'' &mdash; function symbols introduced by the same event &mdash; as
 siblings are considered equivalent for purposes of the acyclicity check.</p>

 <h3>Three Primary Uses of Defattach.</h3>

 <p>We anticipate three uses of @('defattach'):</p>

 <ol>

 <li>Constrained function execution</li>

 <li>Sound modification of the ACL2 system</li>

 <li>Program refinement</li>

 </ol>

 <p>We discuss these in turn.</p>

 <ol>

 <li>The example at the beginning of this @(see documentation) illustrates
 constrained function execution.</li>

 <li>ACL2 is written essentially in itself.  Thus, there is an opportunity
 to attaching to system functions.  For example, encapsulated function
 @('too-many-ifs-post-rewrite'), in the ACL2 source code, receives an
 attachment of @('too-many-ifs-post-rewrite-builtin'), which implements a
 heuristic used in the rewriter.  To find all such examples, search the source
 code for the string `-builtin'.<br/>

 Over time, we expect to continue replacing ACL2 source code in a similar
 manner.  We invite the ACL2 community to assist in this ``open architecture''
 enterprise; feel free to email the ACL2 implementors if you are interested in
 such activity.</li>

 <li>Recall that for an attachment pair @('<f,g>'), a proof obligation is
 (speaking informally) that @('g') satisfies the constraint on @('f').  Yet
 more informally speaking, @('g') is ``more defined'' than @('f'); we can think
 of @('g') as ``refining'' @('f').  With these informal notions as motivation,
 we can view defattach as providing refinement through the following formal
 observation: the evaluation theory extends the theory of the ACL2 session,
 specifically by the addition of all attachment equations.  For the
 logic-inclined, it may be useful to think model-theoretically: The class of
 models of the evaluation theory is non-empty but is a subset of the class of
 models of the current session theory.</li>

 </ol>

 <h3>Miscellaneous Remarks, with discussion of possible user errors.</h3>

 <p>We conclude with remarks on some details.</p>

 <p>A @('defattach') event is never redundant (see @(see redundant-events)); in
 that sense it is analogous to @(tsee in-theory).</p>

 <p>As mentioned above, the use of attachments is disabled for evaluation of
 ground terms during proofs.  However, attachments can be used on code during
 the proof process, essentially when the ``program refinement'' is on theorem
 prover code rather than on functions we are reasoning about.  The attachment
 to @('too-many-ifs-post-rewrite') described above provides one example of such
 attachments.  Meta functions and clause-processor functions can also have
 attachments, with the restriction that no common ancestor with the evaluator
 can have an attachment; see @(see evaluator-restrictions).</p>

 <p>For an attachment pair @('<f,g>'), evaluation of @('f') never consults the
 @(see guard) of @('f').  Rather, control passes to @('g'), whose guard is
 checked if necessary.  The proof obligation related to guards, as described
 above, guarantees that any legal call of @('f') is also a legal call of
 @('g').  Thus for guard-verified code that results in calls of @('f') in raw
 Lisp, it is sound to replace these calls with corresponding calls of
 @('g').</p>

 <p>@('Defattach') events are illegal inside any @(tsee encapsulate) event with
 a non-empty @(see signature) unless they are @(see local) to the @(tsee
 encapsulate).</p>

 <p>(Of interest only to users of @(tsee apply$).)  Special handling is applied
 when attempting to attach to a so-called <i>warrant</i>, which is produced by
 an appication of @(tsee defwarrant) (or @(tsee defun$)).  In that case it is
 legal to attach the function @('true-apply$-warrant') to the warrant, without
 any proof obligation.  This attachment is actually performed automatically by
 @('defwarrant'), so users (even users of @('apply$')) need not deal
 explicitly with such attachments.  However, these attachments make warrants
 executable in the loop; for example, after @('(defwarrant foo)'), @('(warrant
 foo)') will evaluate to @('t') in the loop.</p>

 <p>We next discuss a restriction based on a notion of a function symbol
 syntactically supporting an event.  Function symbol @('f') is <i>ancestral</i>
 in event @('E') if either @('f') occurs in @('E'), or (recursively) @('f')
 occurs in an event @('E'') that introduces some function symbol @('g') that is
 ancestral in @('E').  We require that no function symbol ancestral in the
 formula of a @(tsee defaxiom) event may have an attachment.  Theoretical
 reasons are discussed in comments in the ACL2 source code, but here we give a
 little example showing the need for some such restriction: without it, we show
 how to prove @('nil')!</p>

 @({
  (defn g1 () 1)
  (defn g2 () 2)
  (defstub f1 () t)
  (defstub f2 () t)
  (defund p (x)
    (declare (ignore x))
    t)
  (defevaluator evl evl-list
    ((p x)))
  (defaxiom f1-is-f2
    (equal (f1) (f2)))
  (defun meta-fn (x)
    (cond ((equal (f1) (f2))
           x)
          (t *nil*)))
  (defthm bad-meta-rule
    (equal (evl x a)
           (evl (meta-fn x) a))
    :rule-classes ((:meta :trigger-fns (p))))
  (defattach f1 g1)
  (defattach f2 g2)
  (defthm contradiction
    nil
    :hints ((\"Goal\" :use ((:instance (:theorem (not (p x)))
                                     (x t)))))
    :rule-classes nil)
 })

 <p>The form @('(all-attachments (w state))') evaluates to the list of all
 attachments except in two cases: @(see warrant)s, and attachments introduced
 with a non-@('nil') value of @(':skip-checks').  To obtain the attachment to a
 function symbol @('FN'), without the above restrictions and with value
 @('nil') if there is no attachment to @('FN'): @('(cdr (attachment-pair 'FN (w
 state)))').</p>

 <p>Next we discuss the @(':ATTACH') keyword.  There is rarely if ever a reason
 to specify @(':ATTACH T'), but the following (admittedly contrived) example
 shows why it may be necessary to specify @(':ATTACH NIL').  First we introduce
 three new function symbols.</p>

 @({
    (defstub f (x) t)

    (defun g (x)
      (f x))

    (encapsulate ((h (x) t))
      (local (defun h (x) (g x)))
      (defthm h-prop
        (equal (h x) (g x))))
 })

 <p>Now suppose we want to attach the function @(tsee acl2-numberp) to both
 @('f') and @('h').</p>

 @({
    (defattach (f acl2-numberp) (h acl2-numberp))
 })

 <p>Such an attempt fails, because the following constraint is generated but is
 not a theorem: @('(EQUAL (ACL2-NUMBERP X) (G X))').  Clearly we also need to
 attach to @('g') as well.</p>

 @({
    (defattach (f acl2-numberp) (h acl2-numberp) (g acl2-numberp))
 })

 <p>But this fails for a different reason, as explained by the error
 message:</p>

 @({
    ACL2 Error in ( DEFATTACH (F ACL2-NUMBERP) ...):  It is illegal to
    attach to function symbol G, because it was introduced with DEFUN.
    See :DOC defattach.
 })

 <p>That is: logically, we need to attach @('acl2-numberp') to @('g'), but we
 cannot actually attach to @('g') because it was not introduced with @(tsee
 encapsulate) (it was introduced with @(tsee defun)).  So we specify @(':ATTACH
 NIL') for the attachment to @('g'), saying that no actual attachment should be
 made to the code for @('g'), even though for logical purposes we should
 consider that @('g') has been given the indicated attachment.</p>

 @({
    (defattach (f acl2-numberp) (h acl2-numberp) (g acl2-numberp :attach nil))
 })

 <p>Finally, we can check that @('f'), @('g'), and @('h') execute as
 expected.</p>

 @({
      ACL2 !>(assert-event (and (f 3)
                         (not (f t))
                         (g 3)
                         (not (g t))
                         (h 3)
                         (not (h t))))
       :PASSED
      ACL2 !>
 })

 <p>We conclude with an example promised above, showing why it is necessary in
 general to unattach all function symbols in an existing attachment nest when
 unattaching any one of those function symbols.  Consider the following
 example.</p>

 @({
  (defstub f1 () t)
  (encapsulate ((f2 () t))
    (local (defun f2 () (f1)))
    (defthm f2=f1 (equal (f2) (f1))))
  (encapsulate ((f3 () t))
    (local (defun f3 () (f1)))
    (defthm f3=f1 (equal (f3) (f1))))
  (defun four () (declare (xargs :guard t)) 4)
  (defun five () (declare (xargs :guard t)) 5)
  (defattach (f1 four) (f2 four))
  (defattach (f1 five) (f3 five))
 })

 <p>The second @('defattach') erases the existing attachment pair
 @('<f1,four>') before installing the new attachment pairs @('<f1,five>') and
 @('<f3,five>').  After the second defattach, both @('(f1)') and @('(f3)')
 evaluate to 5.  Now suppose that the attachment pair @('<f2,four>') were not
 erased.  Then we would have @('(f1)') evaluating to 5 and @('(f2)') evaluating
 to 4, contradicting the constraint @('f2=f1').  The evaluation theory would
 thus be inconsistent, and at a more concrete level, the user might well be
 surprised by evaluation results if the code were written with the assumption
 specified in the constraint @('f2=f1').</p>")

(defxdoc defattach-system
  :parents (defattach)
  :short "Attach to built-in, system-level, constrained functions"
  :long "<p>For background on attachments, see @(see defattach).  The macro
 @('defattach-system') is a convenient way to attach to built-in functions.
 The event @('(defattach f g)') will fail if @('f') is built into ACL2.  This
 failure can be overcome by specifying top-level keyword argument @(':system-ok
 t'), for example: @('(defattach (f g) :system-ok t)').  However, rather than
 supplying this argument directly, it is recommended to use
 @('defattach-system'), which has the same syntax as @('defattach') with two
 exceptions: it adds @(':system-ok t') automatically, that is, @(':system-ok')
 is implicit; and it expands to a @(tsee local) call of @('defattach').  The
 latter is important so that the attachment does not affect system behavior
 outside a book containing the @('defattach') event.  Of course, if it is truly
 intended to affect such behavior, the argument @(':system-ok t') may be given
 directly to @('defattach'), without a surrounding use of @('local').</p>

 <p>See @(see system-attachments) for discussion of system attachments.  Also
 see @(see efficiency) for how to use attachments to modify the prover's
 behavior.</p>")

(defxdoc default
  :parents (arrays acl2-built-ins)
  :short "Return the @(':default') from the @(see header) of a 1- or
  2-dimensional array"
  :long "@({
  Example Form:
  (default 'delta1 a)

  General Form:
  (default name alist)
 })

 <p>where @('name') is an arbitrary object and @('alist') is a 1- or
 2-dimensional array.  This function returns the contents of the @(':default')
 field of the @(see header) of @('alist').  When @(tsee aref1) or @(tsee aref2)
 is used to obtain a value for an index (or index pair) not bound in
 @('alist'), the default value is returned instead.  Thus, the array @('alist')
 may be thought of as having been initialized with the default value.
 @('default') operates in virtually constant time if @('alist') is the semantic
 value of @('name').  See @(see arrays).</p>

 @(def default)")

(defxdoc default-backchain-limit
  :parents (rule-classes)
  :short "Specifying the backchain limit for a rule"
  :long "<p>See @(see backchain-limit).</p>

 <p>The initial value is @('(nil nil)').  To inspect the current value (as
 explained elsewhere; see @(see backchain-limit)):</p>

 @({
  (default-backchain-limit wrld :ts) ; for type-set reasoning
  (default-backchain-limit wrld :rewrite) ; for rewriting
 })")

(defxdoc default-defun-mode
  :parents (miscellaneous)
  :short "The default @(see defun-mode) of @(tsee defun)'d functions"
  :long "<p>When a @(tsee defun) is processed and no @(':mode') @('xarg') is
 supplied, the function @('default-defun-mode') is used.  To find the default
 @(see defun-mode) of the current ACL2 @(see world), type
 @('(default-defun-mode (w state))').  See @(see defun-mode) for a discussion
 of @(see defun-mode)s.  To change the default @(see defun-mode) of the ACL2
 @(see world), type one of the keywords @(':')@(tsee program) or @(':')@(tsee
 logic).</p>

 <p>The default ACL2 @(see prompt) displays the current default @(see
 defun-mode) by showing the character @('p') for @(':')@(tsee program) mode,
 and omitting it for @(':')@(tsee logic) mode; see @(see default-print-prompt).
 The default @(see defun-mode) may be changed using the keyword @(see command)s
 @(':')@(tsee program) and @(':')@(tsee logic), which are equivalent to the
 @(see command)s @('(program)') and @('(logic)').  Each of these names is
 documented separately: see @(see program) and see @(see logic).  The default
 @(see defun-mode) is stored in the @(see table) @(tsee acl2-defaults-table)
 and hence may also be changed by a @(tsee table) @(see command).  See @(see
 table) and also see @(see acl2-defaults-table).  Both mode-changing @(see
 command)s are @(see events).</p>

 <p>While @(see events) that change the default @(see defun-mode) are permitted
 within an @(tsee encapsulate) or the text of a book, their effects are @(tsee
 local) in scope to the duration of the encapsulation or inclusion.  For
 example, if the default @(see defun-mode) is @(':')@(tsee logic) and a book is
 included that contains the event @('(program)'), then subsequent @(see events)
 within the book are processed with the default @(see defun-mode) @(':')@(tsee
 program); but when the @(tsee include-book) event completes, the default @(see
 defun-mode) will still be @(':')@(tsee logic).  @(see Command)s that change
 the default @(see defun-mode) are not permitted inside @(tsee local)
 forms.</p>")

(defxdoc default-hints
  :parents (hints)
  :short "A list of hints added to every proof attempt"
  :long "@({
  Examples:
  ACL2 !>(default-hints (w state))
  ((computed-hint-1 clause)
   (computed-hint-2 clause stable-under-simplificationp))
 })

 <p>The value returned by this function is added to the right of the
 @(':')@(tsee hints) argument of every @(tsee defthm) and @(tsee thm) command,
 and to hints provided to @(tsee defun)s as well (@(':hints'),
 @(':guard-hints'), and (for ACL2(r)) @(':std-hints')).</p>

 <p>See @(see set-default-hints) for a more general discussion.  Advanced users
 only: see @(see override-hints) for an advanced variant of default hints that
 are not superseded by @(':')@(tsee hints) arguments.</p>")

(defxdoc default-hints-table
  :parents (default-hints)
  :short "A @(see table) used to provide @(see hints) for proofs"
  :long "<p>Please see @(see set-default-hints), see @(see add-default-hints),
 and see @(see remove-default-hints) for how to use this table.  For
 completeness, we mention here that under the hood, these events all update the
 @('default-hints-table') by updating its key, @('t'), for example as
 follows.</p>

 @({
  (table default-hints-table t
         '((computed-hint-1 clause)
           (computed-hint-2 clause
                            stable-under-simplificationp)))
 })

 <p>The use of default hints is explained elsewhere; see @(see
 set-default-hints).</p>

 <p>Advanced users only: see @(see override-hints) for an advanced variant of
 default hints.</p>")

(defxdoc default-print-prompt
  :parents (ld)
  :short "The default @(see prompt) printed by @(tsee ld)"
  :long "@({
  Example prompt:
  ACL2 p!s>
 })

 <p>The @(see prompt) printed by ACL2 displays the current package, followed by
 a space, followed by zero or more of the three @(see characters) as specified
 below, followed by the character @(tsee >) printed one or more times,
 reflecting the number of recursive calls of @(tsee ld).  The three @(see
 characters) in the middle are as follows:</p>

 @({
  p     ; when (default-defun-mode (w state)) is :program
  !     ; when guard checking is on
  s     ; when (ld-skip-proofsp state) is t
 })

 <p>See @(see default-defun-mode), see @(see set-guard-checking), and see @(see
 ld-skip-proofsp).</p>

 <p>Also see @(see ld-prompt) to see how to install your own @(see prompt).</p>

 <p>Here are some examples with @('ld-skip-proofsp nil').</p>

 @({
  ACL2 !>    ; logic mode with guard checking on
  ACL2 >     ; logic mode with guard checking off
  ACL2 p!>   ; program mode with guard checking on
  ACL2 p>    ; program mode with guard checking off
 })

 <p>Here are some examples with @(tsee default-defun-mode) of @(':')@(tsee
 logic).</p>

 @({
  ACL2 >     ; guard checking off, ld-skip-proofsp nil
  ACL2 s>    ; guard checking off, ld-skip-proofsp t
  ACL2 !>    ; guard checking on, ld-skip-proofsp nil
  ACL2 !s>   ; guard checking on, ld-skip-proofsp t
 })

 <p>Finally, here is the prompt in raw mode (see @(see set-raw-mode)),
 regardless of the settings above:</p>

 @({
  ACL2 P>
 })")

(defxdoc default-ruler-extenders
  :parents (rulers)
  :short "The default @(see ruler-extenders) for @(tsee defun)'d functions"
  :long "<p>When a @(tsee defun) is processed and no @(':ruler-extenders')
 @('xarg') is supplied, the function @('default-ruler-extenders') is used to
 obtain the current ruler-extenders; see @(see rulers).  To find the default
 ruler-extenders of the current ACL2 @(see world), type
 @('(default-ruler-extenders (w state))').</p>

 <p>While @(see events) that change the default ruler-extenders are permitted
 within an @(tsee encapsulate) or the text of a book, their effects are @(tsee
 local) in scope to the duration of the encapsulation or inclusion.  See @(see
 default-defun-mode) for an analogous discussion for defun-modes.</p>")

(defxdoc default-total-parallelism-work-limit
  :parents (parallel-proof)
  :short "For ACL2(p): returns the default value for global
  @('total-parallelism-work-limit')"
  :long "<p>See @(see set-total-parallelism-work-limit).</p>")

(defxdoc defaxiom
  :parents (events)
  :short "Add an axiom"
  :long "<p>WARNING: We strongly recommend that you not add axioms.  If at all
 possible you should use @(tsee defun) or @(tsee mutual-recursion) to define
 new concepts recursively or use @(tsee encapsulate) to constrain them
 constructively.  If your goal is to defer proving a formula that is, logically
 speaking, a theorem of the current theory, then consider using @(tsee
 skip-proofs) instead.  One such case is the use of a top-down style, as per
 the discussion on ``Top-Down Proof'' in Section B.1.2 of ``Computer-Aided
 Reasoning: An Approach.''  Note that adding new axioms may frequently render
 the logic inconsistent!</p>

 @({
  Example:
  (defaxiom sbar (equal t nil)
            :rule-classes nil)

  General Form:
  (defaxiom name term
           :rule-classes rule-classes)
 })

 <p>where @('name') is a new symbolic name (see @(see name)), @('term') is a
 term intended to be a new axiom, and @('rule-classes') is a legal value for
 @(':rule-classes') (see @(see rule-classes)).  The keyword argument is
 optional.  If @(':')@(tsee rule-classes) is not supplied, the list
 @('(:rewrite)') is used; if you wish the axiom to generate no rules, specify
 @(':')@(tsee rule-classes) @('nil').</p>")

(defxdoc defbadge
  :parents (apply$ acl2-built-ins)
  :short "Issue a badge for a function so @(tsee apply$) can evaluate with it"
  :long "<p>It is best to be somewhat familiar with the documentation of @(tsee
  apply$) before reading this topic.</p>

  <p>Before using @('defbadge') or a utility like @(tsee defun$) that relies on
  it:</p>

  @({
  (include-book \"projects/apply/top\" :dir :system)
  })

  <h3>Badges versus Warrants</h3>

  <p>It is easy to confuse badges, which are issued by @('defbadge'), with
  warrants, which are issued by @(tsee defwarrant).  The thing to keep in mind
  is that badges extend the syntax and evaluation capabilities of ACL2, while
  warrants extend the proof theory.  Some user-defined function symbols may be
  given badges even if they are in @(':')@(tsee program) mode (see @(see
  mixed-mode-functions)).  If @('fn') has a badge, you are allowed to write
  @('(apply$ 'fn (list a1 ... an))') and expect it evaluate ``correctly'' at
  the top level of the ACL2 read-eval-print loop, just as you expect @('(fn a1
  ... an)') to evaluate.  But you cannot <i>prove</i> anything interesting
  about @('(apply$ 'fn (list a1 ... an))') without the warrant for @('fn').  A
  badged but unwarranted function symbol might as well be undefined as far as
  the prover is concerned.  Warrants connect the quoted symbol to the axiomatic
  behavior of @('apply$'), in particular, they constrain @('(apply$ 'fn (list
  a1 ... an))') to be @('(fn a1 ... an)') under certain conditions.  Obviously,
  the very first requirement on @('fn') to have a warrant is that @('fn') must
  be in @(':')@(tsee logic) mode.  But there are other requirements because
  @('defwarrant') must make sure the extended proof theory is consistent.</p>

  <h3>Requirements of @('Defbadge')</h3>


  @({
  General Form:
  (defbadge fn)
  })

  <p>where @('fn') is a defined function name in either @(':')@(tsee program)
  or @(':')@(tsee logic) mode.  This command analyzes the body of @('fn') to
  determine whether it satisfies certain stringent syntactic conditions
  discussed below.  If the conditions are not met, @('defbadge') signals an
  error.  Otherwise, if records a @(see badge) for @('fn').  Badges record the
  input and output arities of @('fn') and specify which arguments are
  ``functions'' that may be applied with @('apply$'), which are ``expressions''
  that may be evaluated with @(tsee ev$), and which are neither.  The
  conditions checked are sufficient to allow @(tsee apply$) to run the function
  safely at the top level of the ACL2 read-eval-print loop.  However, in order
  to prove anything about the behavior of @('apply$') on @('fn') the function
  will need a @(see warrant) as issued by @(tsee defwarrant).  @('Defbadge')
  does not issue warrants, just badges.  @('Defwarrant') can issue both badges
  and warrants.</p>

  <p>The first condition on @('fn') is that it must be a defined function
  symbol that does not take or return @(tsee state) or any @(tsee stobj).
  Since @('fn') must be defined it may not be a constrained function such as
  one introduced by @(tsee defchoose) or @(tsee encapsulate).  In addition,
  @('fn') may not be one of a very few ``blacklisted'' symbols (see the value
  of @('*blacklisted-apply$-fns*')) like @(tsee sys-call) (which requires a
  trust tag) or an @(see untouchable).  (For technical reasons, untouchables
  are disallowed even if they are on @('temp-touchable-fns'); see @(tsee
  remove-untouchable).)</p>

  <p>The other conditions depend on whether @(tsee apply$) is reachable from
  @('fn').  That is, can a call of @('fn') lead to a call of @('apply$')?  If
  @('apply$') is not reachable from @('fn'), then there are no more conditions
  on @('fn').  A badge for fn is computed and stored.  We are more precise
  about ``reachability'' later.</p>

  <p>If @('apply$') is reachable from @('fn'), then there are additional
  conditions that must be checked.  First, @('fn') must not have been
  introduced with @(tsee mutual-recursion).  The current badging machinery is
  unable to enforce the syntactic restrictions for mutually-recursive cliques.
  Another restriction is that every function mentioned in the body of @('fn'),
  except @('fn') itself, must already have a badge.  Finally, @('fn') must
  respect certain conventions regarding its use of @(tsee apply$) and other
  @(see scion)s.  The basic idea of this last restriction is to make sure that
  @('apply$') is always called on a ``known'' function symbol or @(tsee lambda)
  object.  This restriction is enforced by checking the following
  conditions:</p>

  <p><i>(a)</i> It must be possible for each formal of @('fn') to be assigned
  one of three @(see ilk)s, @(':FN'), @(':EXPR'), or @('NIL'), as described
  below.  The basic idea is that a formal can be assigned ilk @(':FN') (or ilk
  @(':EXPR')) iff it is sometimes passed into a @(':FN') (or @(':EXPR')) slot
  in the body of @('fn') and is never passed into any other kind of slot.  A
  formal can be be assigned ilk @('NIL') iff it is never passed into a slot of
  ilk @(':FN') or @(':EXPR'), i.e., if it is used as an ``ordinary'' object.
  We are more precise below.</p>

  <p><i>(b)</i> Every @(':FN') and @(':EXPR') slot of every function called in
  the body of @('fn') is occupied either by a formal of @('fn') of the same ilk
  or, in the case of calls of functions other than @('fn'), a quoted @(see
  tame) function symbol or quoted tame (preferably well-formed) @('LAMBDA')
  object.</p>

  <p>This completes the list of restrictions imposed by @('defbadge').</p>

  <h3>Discussion and Examples</h3>

  <p>Note that if @('apply$') is not reachable from @('fn'), the restrictions
  imposed on @('fn') are comparatively generous.  Such a @('fn') could be
  badged despite being defined mutually recursively or in terms of unbadged or
  even unbadgeable functions.  (Functions with @(tsee stobj)s in their
  signatures are unbadgeable but could be used with @(tsee with-local-stobj) in
  @('fn') and @('fn') could still be badged.)</p>

  <p>After a successful @('defbadge') event for @('fn'), the function @(tsee
  badge) will return the computed badge and @(tsee apply$) will be able to
  accept the @('fn') as a functional argument.  Here is an annotated script.
  First, carry out these two events, defining @('foldr') as a @(':program')
  mode function.</p>

  @({
  (include-book \"projects/apply/top\" :dir :system)

  (defun foldr (lst fn init)
    (declare (xargs :mode :program))
    (if (endp lst)
        init
        (apply$ fn
                (list (car lst)
                      (foldr (cdr lst) fn init)))))
  })

  <p>Note the @('apply$') call in the definition.  We see that @('foldr')
  treats its middle argument, @('fn'), as a function of arity 2.  We can run
  @('foldr'), even without assigning a badge to @('foldr'), as long as we
  supply a badged function symbol of arity 2 as the middle argument.  Since
  the ACL2 primitive @('cons') has a badge and has arity 2, we can use it:</p>

  @({
  ACL2 !>(foldr '(a b c) 'cons '(d e f))
  (A B C D E F)
  })

  <p>Since @('foldr') has arity 3, we can try to apply it to a list of three
  things.</p>

  @({
  (apply$ 'foldr (list '(a b c) 'cons '(d e f)))

  ACL2 Error in TOP-LEVEL:  The value of APPLY$-USERFN is not specified
  on FOLDR because FOLDR has not been badged.
  })

  <p>However, we can use @('defbadge') to compute the badge for @('foldr').
  The badge says foldr has input arity 3, output arity 1, and treats its middle
  argument as a function.  We can recover the badge by calling the function
  @(tsee badge).  We can successfully apply @('foldr').  We can even use it in
  a @('lambda') expression that we pass as the middle argument to
  @('foldr').</p>

  @({
  ACL2 !>(defbadge foldr)

  FOLDR now has the badge (APPLY$-BADGE 3 1 NIL :FN NIL) but has no warrant.
  T

  ACL2 !>(badge 'foldr)
  (APPLY$-BADGE 3 1 NIL :FN NIL)

  ACL2 !>(apply$ 'foldr (list '(a b c) 'cons '(d e f)))
  (A B C D E F)

  ACL2 !>(foldr '((a b c) (d e) (f g h) (i j k))
                (lambda$ (x y)
                  (foldr x 'cons y))
                nil)
  (A B C D E F G H I J K)
  })

  <h3>The ``Reachability'' Test</h3>

  <p>We now clarify the test that we colloquially described above as whether
  @('apply$') is reachable from @('fn').  The actual test is whether
  @('apply$-userfn') is ancestral in @('fn').  That is, does @('fn') call
  @('apply$-userfn'), or a function that calls @('apply$-userfn'), or a
  function that calls a function that calls @('apply$-userfn'), etc.</p>

  <p>Since the only system functions that call @('apply$-userfn') are
  @('apply$'), @('ev$'), and @(see warrant)s, and since it is very unusual for
  a user-defined function to call directly @('apply$-userfn'), @('ev$'), or
  warrants, we think of this test colloquially as whether @('apply$') is
  ancestral in @('fn').</p>

  <p>The test and the onerous conditions imposed when the @('apply$') is
  reachable is crucial to the soundness of the ACL2 proof theory.  We discuss
  this further in the background material for @(tsee apply$), including <a
  href='http://www.cs.utexas.edu/users/kaufmann/papers/apply/index.html'>``Limited
  Second-Order Functionality in a First-Order Setting''</a> by Matt Kaufmann
  and J Strother Moore and offer a fully fleshed out metalevel proof that
  @('apply$') and all @(':logic')-mode scions can be modelled in the comment
  titled <tt>Essay on Admitting a Model for Apply$ and the Functions that Use
  It</tt> in the ACL2 source file @('apply-raw.lisp').</p>

  <p>But badges are more concerned with the evaluation theory than the proof
  theory.  Even if we convert @('foldr') to @(':logic') mode we cannot prove
  anything interesting about what happens when it is applied with
  @('apply$').</p>

  @({
  ACL2 !>(verify-termination foldr)
  [Successful.  Output deleted.]
   FOLDR

  ACL2 !>(thm
           (equal (apply$ 'foldr (list x 'cons z))
                  (append x z)))
  [Unsuccessful.  Output deleted.]
  ******** FAILED ********
  })

  <p>In order to prove anything about applying @('foldr') we need the @(tsee
  warrant) for @('foldr').  Warrants are issued by @(tsee defwarrant).</p>

  <h3>How Ilks Are Assigned</h3>

  <p>If a formal variable (or its slot among the actuals) has an ilk of
  @(':FN') then the variable is ``used as a function'' in the sense that it
  might eventually reach the first argument of a call of @('apply$') and is
  never passed into an ``ordinary'' slot like those for @('cons').  Similarly,
  an ilk of @(':EXPR') means the variable is ``used as an expression'' and may
  eventually reach the first argument of @('ev$').  An ilk of @('NIL') means
  the variable is never used as a function or an expression.  The correctness
  of this algorithm is crucial to the safe evaluation of @('apply$') on
  user-defined function symbols.  It also is crucial to the termination
  argument justifying the consistency of the proof theory created if and when
  @('fn') is warranted.</p>

  <p>The key to the inductive correctness of the algorithm implicity described
  below is the fact that initially the only function symbol with a slot of ilk
  @(':FN') is @('apply$') and the only function with a slot of ilk @(':EXPR')
  is @('ev$').  In both functions it is the first argument slot that is so
  distinguished.</p>

  <p>Let <i>v </i> be the <i>i </i>th formal parameter of a defined function
  <i>fn</i>.  Then the ilk of <i>v </i> is @(':FN') iff the value of <i>v </i>
  eventually makes its way into the first argument of @('apply$'), either in
  the definition of @('fn') or in some function ancestral to (i.e., eventually
  called by) @('fn').  Another way to say this is that there is an occurrence
  of <i>v </i> in a slot of ilk @(':FN').  Furthermore, <i>v </i> is never used
  any other way: every place <i>v </i> occurs in the body of <i>fn </i> is in a
  slot of ilk @(':FN').  And finally, in every recursive call of <i>fn </i>,
  <i>v </i> is passed identically in the <i>i </i>th argument position of the
  call.  We say such a <i>v </i> is ``used (exclusively) as a function.''</p>

  <p>The <i>i </i>th formal variable <i>v </i> has ilk @(':EXPR') under
  analogous conditions except that instead of eventually getting into the first
  argument of @('apply$') it eventually gets into the first argument of
  @('ev$').  We say such a <i>v </i> is ``used (exclusively) as an
  expression.''  Note: @(tsee ev$) is the natural notion of expression
  evaluation in this context: look up the values of variables in the alist
  argument to @('ev$'), return quoted constants, and otherwise @('apply$')
  function symbols and @('LAMBDA') objects to the recursively obtained list of
  values returned by evaluating the actuals.  However, @('ev$') first checks
  that the expression is @(tsee tamep).</p>

  <p>The <i>i </i>th formal variable <i>v </i> has ilk @('NIL') if it never
  occurs in a @(':FN') slot and never occurs in an @(':EXPR') slot.  We say
  such a <i>v </i> is ``used (exclusively) as an ordinary object.''</p>")

(defxdoc defchoose
  :parents (events)
  :short "Define a Skolem (witnessing) function"
  :long "@({
  Examples:
  (defchoose choose-x-for-p1-and-p2 (x) (y z)
    (and (p1 x y z)
         (p2 x y z)))

  ; Axiom added by the event above:
  (IMPLIES (AND (P1 X Y Z) (P2 X Y Z))
           (LET ((X (CHOOSE-X-FOR-P1-AND-P2 Y Z)))
                (AND (P1 X Y Z) (P2 X Y Z))))

  (defchoose choose-x-for-p1-and-p2 x (y z) ; equivalent to the above
    (and (p1 x y z)
         (p2 x y z)))

  ; The following is as above, but strengthens the axiom added to pick a sort
  ; of canonical witness, as described below.
  (defchoose choose-x-for-p1-and-p2 x (y z)
    (and (p1 x y z)
         (p2 x y z))
    :strengthen t)

  (defchoose choose-x-and-y-for-p1-and-p2 (x y) (z)
    (and (p1 x y z)
         (p2 x y z)))

  General Form:
  (defchoose fn
             (bound-var1 ... bound-varn)
             (free-var1 ... free-vark)
             body
             :strengthen b),
 })

 <p>where @('fn') is the symbol you wish to define and is a new symbolic name
 (see @(see name)), @('(bound-var1 ... bound-varn)') is a list of distinct
 `bound' variables (see below), @('(free-var1 ... free-vark)') is the list of
 formal parameters of @('fn') and is disjoint from the bound variables, and
 @('body') is a term.  The use of @('lambda-list') keywords (such as
 @('&optional')) is not allowed.  The @(':strengthen') keyword argument is
 optional; if supplied, it must be @('t') or @('nil').</p>

 <p>The system treats @('fn') very much as though it were declared in the @(see
 signature) of an @(tsee encapsulate) event, with a single axiom exported as
 described below.  If you supply a @(':use') hint (see @(see hints)), @(':use
 fn'), it will refer to that axiom.  No rule (of class @(':')@(tsee rewrite) or
 otherwise; see @(see rule-classes)) is created for @('fn').</p>

 <p>@('Defchoose') is only executed in @(see defun-mode) @(':')@(tsee logic);
 see @(see defun-mode).  Also see @(see defun-sk).</p>

 <p>In the most common case, where there is only one bound variable, it is
 permissible to omit the enclosing parentheses on that variable.  The effect is
 the same whether or not those parentheses are omitted.  We describe this case
 first, where there is only one bound variable, and then address the other
 case.  Both cases are discussed assuming @(':strengthen') is @('nil'), which
 is the default.  We deal with the case @(':strengthen t') at the end.</p>

 <p>The effect of the form</p>

 @({
  (defchoose fn bound-var (free-var1 ... free-vark)
    body)
 })

 <p>is to introduce a new function symbol, @('fn'), with formal parameters
 @('(free-var1 ... free-vark)').  Now consider the following axiom, which
 states that @('fn') picks a value of @('bound-var') so that the body will be
 true, if such a value exists:</p>

 @({
  (1)   (implies body
                 (let ((bound-var (fn free-var1 ... free-vark)))
                   body))
 })

 <p>This axiom is ``clearly conservative'' under the conditions expressed
 above: the function @('fn') simply picks out a ``witnessing'' value of
 @('bound-var') if there is one.  For a rigorous statement and proof of this
 conservativity claim, see @(see conservativity-of-defchoose).</p>

 <p>Next consider the case that there is more than one bound variable, i.e.,
 there is more than one bound-var in the following.</p>

 @({
  (defchoose fn
             (bound-var1 ... bound-varn)
             (free-var1 ... free-vark)
             body)
 })

 <p>Then @('fn') returns a multiple value with @('n') components, and formula
 (1) above is expressed using @(tsee mv-let) as follows:</p>

 @({
  (implies body
           (mv-let (bound-var1 ... bound-varn)
                   (fn free-var1 ... free-vark)
                   body))
 })

 <p>We now discuss the case that @(':strengthen t') is supplied.  For
 simplicity we return to our simplest case, with @('defchoose') applied to
 function @('fn'), a single free variable @('y'), and a single bound variable
 @('bound-var').  The idea is that if we pick the ``smallest'' witnessing
 @('bound-var') for two different free variables @('y') and @('y1'), then
 either those two witnesses are the same, or else one is less than the other,
 in which case the smaller one is a witness for its free variable but not for
 the other.  (See comments in source function @('defchoose-constraint-extra')
 for more details.)  Below, @('body1') is the result of replacing @('y') by
 @('y1') in @('body').</p>

 @({
  (2)   (or (equal (fn y) (fn y1))
            (let ((bound-var (fn y)))
              (and body
                   (not body1)))
            (let ((bound-var (fn y1)))
              (and body1
                   (not body))))
 })

 <p>An important application of this additional axiom is to be able to define a
 ``fixing'' function that picks a canonical representative of each equivalence
 class, for a given equivalence relation.  The following events illustrate this
 point.</p>

 @({
  (encapsulate
   ((equiv (x y) t))
   (local (defun equiv (x y) (equal x y)))
   (defequiv equiv))

  (defchoose efix (x) (y)
    (equiv x y)
    :strengthen t)

  (defthm equiv-implies-equal-efix-1
    (implies (equiv y y1)
             (equal (efix y) (efix y1)))
    :hints ((\"Goal\" :use efix))
    :rule-classes (:congruence))

  (defthm efix-fixes
    (equiv (efix x) x)
    :hints ((\"Goal\" :use ((:instance efix (y x))))))
 })

 <p>If there is more than one bound variable, then (2) is modified in complete
 analogy to (1) to use @(tsee mv-let) in place of @(tsee let).</p>

 <p>Comment for logicians: As we point out in the documentation for @(tsee
 defun-sk), @('defchoose') is ``appropriate,'' by which we mean that it is
 conservative, even in the presence of @('epsilon-0') induction.  For a proof,
 See @(see conservativity-of-defchoose).</p>")

(defxdoc defcong
  :parents (events)
  :short "Prove @(see congruence) rule"
  :long "<p>@('Defcong') is used to prove that one @(see equivalence) relation
 preserves another in a given argument position of a given function.</p>
 @({
 Example:
 (defcong set-equal iff (memb x y) 2)

 is an abbreviation for

 (defthm set-equal-implies-iff-memb-2
   (implies (set-equal y y-equiv)
            (iff (memb x y) (memb x y-equiv)))
   :rule-classes (:congruence))
 })

 <p>See @(see congruence) and also see @(see equivalence).</p>

 @({
  General Form:
  (defcong equiv1 equiv2 term k
    :package package
    :event-name event-name
    :rule-classes rule-classes
    :instructions instructions
    :hints hints
    :otf-flg otf-flg)
 })

 <p>where @('equiv1') and @('equiv2') are known @(see equivalence) relations;
 @('term') is a call of a function @('fn'), other than @('if'), on the correct
 number of distinct variable arguments, @('(fn x1 ... xn)'); @('k') is a
 positive integer less than or equal to the arity of @('fn'); @('package'), if
 supplied, is one of @(':current'), @(':equiv1'), @(':equiv2'), @(':function')
 or @(':legacy'); @('event-name'), if supplied, is a symbol; and other
 arguments are as specified in the documentation for @(tsee defthm).  The
 @('defcong') macro expands into a call of @(tsee defthm).  The name of the
 @(tsee defthm) event is @('equiv1-implies-equiv2-fn-k'), unless an
 @(':event-name') keyword argument is supplied, in which case @('event-name')
 is used.  The package of symbols generated, such as variables and @(tsee
 defthm) names, is determined by the @('package') argument: if it is not
 supplied or its value is @(':current'), then the @(tsee current-package) is
 used; if its value is @(':equiv1') or @(':legacy'), then the package of
 @(':equiv1') is used; if its value is @(':equiv2'), then the package of
 @(':equiv2') is used; and if its value is @(':function'), then the package of
 @('fn') is used.  The term of the theorem is</p>

 @({
  (implies (equiv1 xk yk)
           (equiv2 (fn x1... xk ...xn)
                   (fn x1... yk ...xn))).
 })

 <p>The rule-class @(':')@(tsee congruence) is added to the @(tsee
 rule-classes) specified, if it is not already there.  All other arguments to
 the generated @(tsee defthm) form are as specified by the keyword arguments
 above.</p>

 <p>NOTE: In normal usage, @('defcong') can only create classic @(see
 congruence) rules, not @(see patterned-congruence) rules.  However, if @('fn')
 is a macro, it's possible for a patterned congruence rule to be generated,
 depending on what @('term') expands to.</p>

 <p>For example, the following produces a shallow patterned congruence
 rule:</p>

 @({
 (defun f (x y) (and x y))
 (defmacro g (x) `(f ,x t))
 (defcong iff equal (g x) 1)
 })")

(defxdoc defconst
  :parents (events programming)
  :short "Define a constant"
  :long "@({
  Examples:
  (defconst *my-digits* '(0 1 2 3 4 5 6 7 8 9))
  (defconst *len-my-digits* (the unsigned-byte (length *my-digits*)))

  General Form:
  (defconst name term)
 })

 <p>where @('name') is a symbol beginning and ending with the character @('*')
 and @('term') is a variable-free term that is evaluated to determine the value
 of the constant.</p>

 <p>There are two restrictions on @('term') aside from it being variable-free.
 Both restrictions relate to ancestral uses of @(tsee apply$) in @('term'), i.e.,
 uses of @('apply$') by @('term') or any function that might be called during the
 evaluation of @('term').  First, only badged primitive functions may be
 applied.  See @(see badge) for a way to obtain the complete list of badged
 primitives.  Second, @('loop$') and @('lambda$') may not be used anywhere in
 the ancestry of term.  See @('ignored-attachment') and
 @('prohibition-of-loop$-and-lambda$') for more discussion.</p>

 <p>When a constant symbol is used as a @(see term), ACL2 replaces it by its
 value; see @(see term).</p>

 <p>Note that @('defconst') uses so-called @(see safe-mode) to evaluate its
 form, in order to avoids soundness issues but with an efficiency penalty
 (perhaps increasing the evaluation time by several hundred percent).  If
 efficiency is a concern, or if for some reason you need the form to be
 evaluated without safe mode (e.g., you are an advanced system hacker using
 trust tags to traffic in raw Lisp code), consider using @(see defconsts)
 instead.  Also see @(see using-tables-efficiently) for an analogous issue with
 @(tsee table) events.</p>

 <p>It may be of interest to note that @('defconst') is implemented at the lisp
 level using @('defparameter'), as opposed to @('defconstant').
 (Implementation note: this is important for proper support of undoing and
 redefinition.)</p>

 <p>We close with a technical remark, perhaps of interest only who make use of
 the @(see hons-enabled) features of ACL2.  For an event of the form
 @('(defconst *C* (quote OBJ))'), i.e., @('(defconst *C* 'OBJ)'), then the
 value associated with @('*C*') is @('OBJ'); that is, the value of @('*C*') is
 @(tsee eq) to the actual object @('OBJ') occurring in the @('defconst') form.
 So for example, if @(tsee make-event) is used to generate such a @('defconst')
 event, as it is in the two books mentioned above, and @('OBJ') is a fast
 alist (see @(see fast-alists)), then the value of @('*C*') is a fast alist.
 This guarantee disappears if the term in the @('defconst') form is not a
 quoted object, i.e., if it is not of the form @('(quote OBJ)').</p>")

(defxdoc defdoc
  :parents (events)
  :short "Deprecated event (formerly for adding documentation)"
  :long "<p>This event is deprecated; see @(see xdoc) for information about
 @(see documentation) in ACL2.  @('Defdoc') @(see events) are never considered
 redundant (see @(see redundant-events)).</p>")

(defxdoc defequiv
  :parents (events)
  :short "Prove that a function is an @(see equivalence) relation"
  :long "@({
  Example:
  (defequiv set-equal)

  is an abbreviation for
  (defthm set-equal-is-an-equivalence
    (and (booleanp (set-equal x y))
         (set-equal x x)
         (implies (set-equal x y) (set-equal y x))
         (implies (and (set-equal x y)
                       (set-equal y z))
                  (set-equal x z)))
    :rule-classes (:equivalence))
 })

 <p>See @(see equivalence).</p>

 @({
  General Form:
  (defequiv equiv
    :package package
    :event-name event-name
    :rule-classes rule-classes
    :instructions instructions
    :hints hints
    :otf-flg otf-flg)
 })

 <p>where @('equiv') is a function symbol of arity 2; @('package'), if
 supplied, is one of @(':current'), @(':equiv') or @(':legacy');
 @('event-name'), if supplied, is a symbol; and all other arguments are as
 specified in the documentation for @(tsee defthm).  The @('defequiv') macro
 expands into a call of @(tsee defthm).  The name of the @(tsee defthm) is
 @('equiv-is-an-equivalence'), unless an @(':event-name') keyword argument is
 supplied, in which case @('event-name') is used.  The package of symbols
 generated, such as variables and @(tsee defthm) names, is determined by the
 @('package') argument: if it is not supplied or its value is @(':current'),
 then the @(tsee current-package) is used; if its value is @(':equiv') or
 @(':legacy') (which are treated identically), then the package of @(':equiv')
 is used.  All other arguments to the generated @(tsee defthm) form are as
 specified by the other keyword arguments above.  The rule-class @(':')@(tsee
 equivalence) is added to the @(see rule-classes) specified, if it is not
 already there.  The term generated for the @(tsee defthm) event states that
 @('equiv') is Boolean, reflexive, symmetric, and transitive. </p>")

(defxdoc defevaluator
  :parents (events)
  :short "Introduce an evaluator function"
  :long "@({
  Example:
  (defevaluator evl evl-list
    ((length x) (member-equal x y)))
 })

 <p>See @(see meta).</p>

 @({
  General Form:
  (defevaluator ev ev-list
     ((g1 x1 ... xn_1)
      ...
      (gk x1 ... xn_k))
     :namedp flg)       ; [optional keyword argument]
 })

 <p>where @('ev') and @('ev-list') are new function symbols and @('g1'), ...,
 @('gk') are old function symbols with the indicated number of formals, i.e.,
 each @('gi') has @('n_i') formals.  If the @(':namedp') keyword argument
 is provided, its value should be Boolean.  If not provided, the default
 value for @('flg') is @('nil').</p>

 <p>This function provides a convenient way to constrain @('ev') and
 @('ev-list') to be mutually-recursive evaluator functions for the symbols
 @('g1'), ..., @('gk').  Roughly speaking, an evaluator function for a fixed,
 finite set of function symbols is a restriction of the universal evaluator to
 terms composed of variables, constants, lambda expressions, and applications
 of the given functions.  However, evaluator functions are constrained rather
 than defined, so that the proof that a given metafunction is correct vis-a-vis
 a particular evaluator function can be lifted (by functional instantiation) to
 a proof that it is correct for any larger evaluator function.  See @(see meta)
 for a discussion of metafunctions.</p>

 <p>If the @(':namedp') @('flg') is @('nil') (the default) constraints have
 names of the form <i>ev</i>@('-CONSTRAINT-')<i>i</i>, e.g.,
 @('EV-CONSTRAINT-0'), @('EV-CONSTRAINT-1'), etc.  If @('flg') is non-@('nil'),
 the constraints are named more mnemonically, e.g., @('EV-OF-VARIABLE'),
 @('EV-OF-REVAPPEND-CALL'), etc.  We illustrate the @(':namedp t') names
 below.</p>

 <p>@('Defevaluator') executes an @(tsee encapsulate) after generating the
 appropriate @(tsee defun) and @(tsee defthm) events.  Perhaps the easiest way
 to understand what @('defevaluator') does is to execute the keyword
 command</p>

 @({
  :trans1 (defevaluator evl evl-list ((length x) (member-equal x y)))
 })

 <p>and inspect the output.  This trick is also useful in the rare case that
 the event fails because a hint is needed.  In that case, the output of
 @(':')@(tsee trans1) can be edited by adding hints, then submitted
 directly.</p>

 <p>Formally, @('ev') is said to be an ``evaluator function for @('g1'), ...,
 @('gk'), with mutually-recursive counterpart @('ev-list')'' iff @('ev') and
 @('ev-list') are constrained functions satisfying just the @(see constraint)s
 discussed below.</p>

 <p>@('Ev') and @('ev-list') must satisfy @(see constraint)s (0)-(7) and
 (k) below.  When @(':namedp nil') is supplied, the <i>i</i> in the generated
 constraint names are the parenthesized numbers below.  When @(':namedp t')
 is supplied, the mnemonic names are those shown in brackets below.</p>

 @({
  (0) How to ev an arbitrary function application:
      [EV-OF-FNCALL-ARGS]
      (implies (and (consp x)
                    (syntaxp (not (equal a ''nil)))
                    (not (equal (car x) 'quote)))
               (equal (ev x a)
                      (ev (cons (car x)
                                (kwote-lst (ev-list (cdr x) a)))
                          nil)))

  (1) How to ev a variable symbol:
      [EV-OF-VARIABLE]
      (implies (symbolp x)
               (equal (ev x a) (and x (cdr (assoc-equal x a)))))

  (2) How to ev a constant:
      [EV-OF-QUOTE]
      (implies (and (consp x)
                    (equal (car x) 'quote))
               (equal (ev x a) (cadr x)))

  (3) How to ev a lambda application:
      [EV-OF-LAMBDA]
      (implies (and (consp x)
                    (consp (car x)))
               (equal (ev x a)
                      (ev (caddar x)
                          (pairlis$ (cadar x)
                                    (ev-list (cdr x) a)))))

  (4) How to ev an empty argument list:
      [EV-LIST-OF-ATOM]
      (implies (not (consp x-lst))
               (equal (ev-list x-lst a)
                      nil))

  (5) How to ev a non-empty argument list:
      [EV-LIST-OF-CONS]
      (implies (consp x-lst)
               (equal (ev-list x-lst a)
                      (cons (ev (car x-lst) a)
                            (ev-list (cdr x-lst) a))))

  (6) How to ev a non-symbol atom:
      [EV-OF-NONSYMBOL-ATOM]
      (implies (and (not (consp x))
                    (not (symbolp x)))
               (equal (ev x a)
                      nil))

  (7) How to ev a cons whose car is a non-symbol atom:
      [EV-OF-BAD-FNCALL]
      (implies (and (consp x)
                    (not (consp (car x)))
                    (not (symbolp (car x))))
               (equal (ev x a)
                      nil))

  (k) For each i from 1 to k, how to ev an application of gi,
      where gi is a function symbol of n arguments:
      [EV-OF-gi-CALL]
      (implies (and (consp x)
                    (equal (car x) 'gi))
               (equal (ev x a)
                      (gi (ev x1 a)
                          ...
                          (ev xn a)))),
      where xi is the (cad...dr x) expression equivalent to (nth i x).
 })

 <p>@('Defevaluator') defines suitable witnesses for @('ev') and @('ev-list'),
 proves the theorems about them, and constrains @('ev') and @('ev-list')
 appropriately.  We expect @('defevaluator') to work without assistance from
 you, though the proofs do take some time and generate a lot of output.  The
 proofs are done in the context of a fixed theory, namely the value of the
 constant @('*defevaluator-form-base-theory*').</p>

 <p>(Aside: (3) above may seem surprising, since the bindings of @('a') are not
 included in the environment that is used to evaluate the lambda body,
 @('(caddar x)').  However, ACL2 lambda expressions are all <i>closed</i>: in
 @('(lambda (v1 ... vn) body)'), the only free variables in @('body') are among
 the @('vi').  See @(see term).)</p>

 <p>Acknowledgment: We thank Sol Swords and Jared Davis for their community
 book @('tools/defevaluator-fast.lisp'), which provided the model on which the
 current @('defevaluator') is based.  The original @('defevaluator') was very
 inefficient, e.g., taking thousands of times longer than the current one on an
 evaluator interpreting 800 function symbols.  We refactored their
 @('defevaluator-fast') (with permission) and made it the new implementation as
 of ACL2 Version_7.2.  Both implementations produce the same constraints modulo
 the naming option provided by @(':namedp').</p>")

(defxdoc defexec
  :parents (events mbe)
  :short "Attach a terminating executable function to a definition"
  :long "<p>Suppose you define a function @('(fn x)') with a @(see guard) of
 @('(good-input-p x)'), and you know that when the guard holds, the measure
 decreases on each recursive call.  Unfortunately, the definitional principle
 (see @(see defun)) ignores the guard.  For example, if the definition has the
 form</p>

 @({
  (defun fn (x)
    (declare (xargs :guard (good-input-p x)))
    (if (not-done-yet x)
        (... (fn (destr x)) ...)
      ...))
 })

 <p>then in order to admit this definition, ACL2 must prove the appropriate
 formula asserting that @('(destr x)') is ``smaller than'' @('x') under the
 assumption @('(not-done-yet x)') but without the assumption @('(good-input-p
 x)'), even if @('(not-done-yet x)') is true.  In essence, it may be necessary
 to submit instead the following definition.</p>

 @({
  (defun fn (x)
    (declare (xargs :guard (good-input-p x)))
    (if (good-input-p x)
        (if (not-done-yet x)
            (... (fn (destr x)) ...)
          ...)
      nil)
 })

 <p>But it is unfortunate that when calls of @('fn') are evaluated, for example
 when @('fn') is applied to an explicit constant during a proof, then a call of
 @('good-input-p') must now be evaluated on each recursive call.</p>

 <p>Fortunately, @('defexec') provides a way to keep the execution efficient.
 For the example above we could use the following form.</p>

 @({
  (defexec fn (x)
    (declare (xargs :guard (good-input-p x)))
    (mbe :logic (if (good-input-p x)
                    (if (not-done-yet x)
                        (... (fn (destr x)) ...)
                      ...)
                  nil)
         :exec  (if (not-done-yet x)
                    (... (fn (destr x)) ...)
                  ...)))
 })

 <p>Here ``@(tsee mbe)'' stands for ``must be equal'' and, roughly speaking,
 its call above is logically equal to the @(':logic') form but is evaluated
 using the @(':exec') form when the guard holds.  See @(see mbe).  The effect
 is thus to define @('fn') as shown in the @(tsee defun) form above, but to
 cause execution of @('fn') using the @(':exec') body.  The use of @('defexec')
 instead of @(tsee defun) in the example above causes a termination proof to be
 performed, in order to guarantee that evaluation always theoretically
 terminates, even when using the @(':exec') form for evaluation.</p>

 @({
  Example:

  ; Some of the keyword arguments in the declarations below are irrelevant or
  ; unnecessary, but they serve to illustrate their use.

  (defexec f (x)
    (declare (xargs :measure (+ 15 (acl2-count x))
                    :ruler-extenders :basic
                    :hints ((\"Goal\" :in-theory (disable nth)))
                    :guard-hints ((\"Goal\" :in-theory (disable last)))
                    :guard (and (integerp x) (<= 0 x) (< x 25)))
             (exec-xargs
                    :test (and (integerp x) (<= 0 x))
                    :default-value 'undef ; defaults to nil
                    :measure (nfix x)
                    :ruler-extenders :basic
                    :well-founded-relation o<))
    (mbe :logic (if (zp x)
                    1
                  (* x (f (- x 1))))
         :exec  (if (= x 0)
                    1
                  (* x (f (- x 1))))))
 })

 <p>The above example macroexpands to the following.</p>

 @({
  (ENCAPSULATE ()
   (LOCAL
    (ENCAPSULATE ()
     (SET-IGNORE-OK T)
     (SET-IRRELEVANT-FORMALS-OK T)
     (LOCAL (DEFUN F (X)
              (DECLARE
               (XARGS :VERIFY-GUARDS NIL
                      :HINTS ((\"Goal\" :IN-THEORY (DISABLE NTH)))
                      :MEASURE (NFIX X)
                      :RULER-EXTENDERS :BASIC
                      :WELL-FOUNDED-RELATION O<))
              (IF (AND (INTEGERP X) (<= 0 X))
                  (IF (= X 0) 1 (* X (F (- X 1))))
                  'UNDEF)))
     (LOCAL (DEFTHM F-GUARD-IMPLIES-TEST
              (IMPLIES (AND (INTEGERP X) (<= 0 X) (< X 25))
                       (AND (INTEGERP X) (<= 0 X)))
              :RULE-CLASSES NIL))))
   (DEFUN F (X)
     (DECLARE (XARGS :MEASURE (+ 15 (ACL2-COUNT X))
                     :RULER-EXTENDERS :BASIC
                     :HINTS ((\"Goal\" :IN-THEORY (DISABLE NTH)))
                     :GUARD-HINTS ((\"Goal\" :IN-THEORY (DISABLE LAST)))
                     :GUARD (AND (INTEGERP X) (<= 0 X) (< X 25))))
     (MBE :LOGIC
          (IF (ZP X) 1 (* X (F (- X 1))))
          :EXEC
          (IF (= X 0) 1 (* X (F (- X 1)))))))
 })

 <p>Notice that in the example above, the @(':')@(tsee hints) in the @(tsee
 local) definition of @('F') are inherited from the @(':hints') in the @(tsee
 xargs) of the @('defexec') form.  We discuss such inheritance below.</p>

 <p>CAVEAT: Termination is not considered for calls of @(tsee mbe) under the
 top-level call.  Moreover, the @(':exec') part of an @(tsee mbe) call under
 the @(':logic') part of any superior @('mbe') call is completely ignored.</p>

 @({
  General Form:
  (defexec fn (var1 ... varn) doc-string dcl ... dcl
    (mbe :LOGIC logic-body
         :EXEC  exec-body))
 })

 <p>where the syntax is identical to the syntax of @(tsee defun) where the body
 is a call of @('mbe'), with the exceptions described below.  Thus, @('fn') is
 the symbol you wish to define and is a new symbolic name and @('(var1
 ... varn)') is its list of formal parameters (see @(see name)).  The first
 exception is that at least one @('dcl') (i.e., @(tsee declare) form) must
 specify a @(':guard'), @('guard').  The second exception is that one of the
 @('dcl')s is allowed to contain an element of the form @('(exec-xargs ...)').
 The @('exec-xargs') form, if present, must specify a non-empty @(tsee
 keyword-value-listp) each of whose keys is one of @(':test'),
 @(':default-value'), or one of the standard @(tsee xargs) keys of
 @(':measure'), @(':ruler-extenders'), @(':well-founded-relation'),
 @(':hints'), or @(':stobjs').  Any of these five standard @('xargs') keys that
 is present in an @('xargs') of some @('dcl') but is not specified in the
 (possibly nonexistent) @('exec-xargs') form is considered to be specified in
 the @('exec-xargs') form, as illustrated in the example above for @(':hints').
 (So for example, if you want @(':hints') in the final, non-local definition
 but not in the local definition, then specify the @(':hints') in the
 @('xargs') but specify @(':hints nil') in the @('exec-xargs').)  If @(':test')
 is specified and not @('nil'), let @('test') be its value; otherwise let
 @('test') default to @('guard').  If @(':default-value') is specified, let
 @('default-value') be its value; else @('default-value') is @('nil').
 @('Default-value') should have the same @(see signature) as @('exec-body');
 otherwise the @('defexec') form will fail to be admitted.</p>

 <p>The above General Form's macroexpansion is of the form @('(PROGN encap
 final-def)'), where @('encap') and @('final-def') are as follows.
 @('Final-def') is simply the result of removing the @('exec-xargs')
 declaration (if any) from its @(tsee declare) form, and is the result of
 evaluating the given @('defexec') form, since @('encap') is of the following
 form.</p>

 @({
  ; encap
  (ENCAPSULATE ()
    (set-ignore-ok t)             ; harmless for proving termination
    (set-irrelevant-formals-ok t) ; harmless for proving termination
    (local local-def)
    (local local-thm))
 })

 <p>The purpose of @('encap') is to ensure the the executable version of
 @('name') terminates on all arguments.  Thus, @('local-def') and
 @('local-thm') are as follows, where the @('xargs') of the @(tsee declare)
 form are the result of adding @(':VERIFY-GUARDS NIL') to the result of
 removing the @(':test') and (optional) @(':default-value') from the
 @('exec-xargs').</p>

 @({
  ; local-def
  (DEFUN fn formals
    (DECLARE (XARGS :VERIFY-GUARDS NIL ...))
    (IF test
        exec-body
      default-value))

  ; local-thm
  (DEFTHM fn-EXEC-GUARD-HOLDS
    (IMPLIES guard test)
    :RULE-CLASSES NIL)
 })

 <p>We claim that if the above @('local-def') and @('local-thm') are admitted,
 then all evaluations of calls of @('fn') terminate.  The concern is that the
 use of @(tsee mbe) in @('final-def') allows for the use of @('exec-body') for
 a call of @('fn'), as well as for subsequent recursive calls, when @('guard')
 holds and assuming that the guards have been verified for @('final-def').
 However, by @('local-thm') we can conclude in this case that @('test') holds,
 in which case the call of @('fn') may be viewed as a call of the version of
 @('fn') defined in @('local-def').  Moreover, since guards have been verified
 for @('final-def'), then guards hold for subsequent evaluation of
 @('exec-body'), and in particular for recursive calls of @('fn'), which can
 thus continue to be viewed as calls using @('local=def').</p>")

(defxdoc definductor
  :parents (loop$-recursion)
  :short "Create an induction scheme for a @('loop$')-recursive function"
  :long "<p>@('Definductor') is a utility provided as part of the community
  book @('projects/apply/top'), which should be included in any session dealing
  with @(tsee apply$), @(tsee loop$), or @(tsee loop$-recursion).
  @('(Definductor fn)') attempts to create an induction scheme appropriate for
  the previously defined @('loop$')-recursive function @('fn') and prove an
  @(':')@(tsee induction) rule so that certains calls of @('fn') suggest that
  induction.</p>

  <p><b>Warning:</b> @('Definductor') currently handles a very small class of
  @('loop$')-recursive functions and may produce unhelpful error messages when
  given a function name outside of that class!  We hope to improve it and this
  documentation as we all get more experience with @('loop$')s and
  @('loop$')-recursion.</p>

  @({
  Examples:
  (definductor copy-nat-tree)

  (definductor copy-nat-tree
               :measure (my-measure x)
               :hints ((\"Goal\" :use ...)))

  General Form:
  (definductor name &key measure well-founded-relation ruler-extenders hints)
  })

  <p>where @('name') is the name of a previously admitted @('loop$')-recursive
  function satisfying the restrictions listed below.  When successful it
  defines a function named @('name-INDUCTOR') that suggests an induction scheme
  that is supposedly appropriate for @('name'), admits it with a silent proof
  of its measure theorems, and then proves an @(':')@('induction') rule to
  associate that scheme with calls of @('name').  When omitted, the optional
  keyword arguments @('measure'), @('well-founded-relation'), and
  @('ruler-extenders') default to the measure, well-founded relation, and
  ruler-extender settings used in the admittance of @('name').  The keyword
  argument @('hints') defaults to @('nil').</p>

  <h3>Restrictions</h3>

  <p>The given function, @('name'), must satisfy the following restrictions.
  <b>Note:</b> Because we anticipate this utility being further developed in the
  near future this list may not correspond to the latest implementation!</p>

  <ul>

  <li>@('Name') must be a symbol naming a previously admitted @('loop$')-recursive
  function.</li>

  <li>Every recursive @('loop$') in the body of @('name') -- that is, every
  @('loop$') that calls @('name') recursively in the @('when'), @('until'), or
  @('body') clauses of the @('loop$') -- must have as its target(s) distinct
  measured variables or @('cdr')-nests around such variables.</li>

  <li>Every recursive @('loop$') must use @('IN')-iteration, not @('ON')- or
  @('FROM/TO/BY')-iteration.</li>

  </ul>

  <p>While @('definductor') can handle @('loop$') containing multiple @('AS')
  clauses (with targets as described above), it cannot handle @('loop$') such
  as</p>

  @({
  (loop$ for v in (target x) ...)
  (loop$ for v on x ...)
  (loop$ for i from 1 to max ...)
  })

  <p>To see the inductor function generated, type @(':pe name-INDUCTOR').  To
  see examples of the use of @('definductor') inspect the book
  @('projects/apply/definductor-tests.lisp').  To see the definition of
  @('definductor'), see @('projects/apply/definductor.lisp').</p>

  <p>Suggestions for improvements are welcome!  We know of many, including
  allowing the user to specify a different name for the inductor function,
  improving the error messages, printing out the generated @('defun') in the
  event of failure to admit it, and trying to expand the class of
  @('loop$')-recursive functions that can be successfully handled.  We have not
  yet even looked at inductions for @('ON') @('loop$')s and @('FROM/TO/BY')
  @('loop$')s, so that might be easy.  Induction for @('loop$')s over arbtrary
  target expressions may be infeasible!  We just need more examples of
  @('loop$')-recursive functions and successful (hand-written) induction hints
  for them.</p>")

(defxdoc define-pc-help
  :parents (proof-builder)
  :short "Define a macro command whose purpose is to print something"
  :long "@({
  Example:
  (define-pc-help pp ()
    (if (goals t)
        (io? proof-builder nil state
             (state-stack)
             (fms0 \"~|~y0~|\"
                   (list (cons #0
                               (fetch-term (conc t)
                                           (current-addr t))))))
      (print-all-goals-proved-message state)))

  General Form:
  (define-pc-help name args &rest body)
 })

 <p>This defines a macro command named @('name'), as explained further below.
 The @('body') should (after removing optional declarations) be a form that
 returns @('state') as its single value.  Typically, it will just print
 something.</p>

 <p>What @('(define-pc-help name args &rest body)') really does is to create a
 call of @('define-pc-macro') that defines @('name') to take arguments
 @('args'), to have the declarations indicated by all but the last form in
 @('body'), and to have a body that (via @('pprogn')) first executes the form
 in the last element of body and then returns a call to the command @('skip')
 (which will return @('(mv nil t state)')).</p>")

(defxdoc define-pc-macro
  :parents (proof-builder)
  :short "Define a proof-builder macro command"
  :long "<p>A call of @('define-pc-macro') defines a sort of macro, which is a
 tactic that generates @(see proof-builder) instructions.  This topic contains
 basic information about how to use this utility.  For somewhat sophisticated,
 but commented, examples, see the @(see community-book)
 @('books/kestrel/utilities/proof-builder-macros.lisp') and associated tests in
 the same directory, @('proof-builder-macros-tests.lisp').</p>

 <p>We begin with the following example.</p>

 @({
 (define-pc-macro ib (&optional term)
   (value
    (if term
        `(then (induct ,term) bash)
      `(then induct bash))))
 })

 <p>The example above captures a common paradigm: one attempts to prove the
 current goal by inducting and then simplifying the resulting goals.  (See
 @(see proof-builder-commands) for documentation of the command @('then'),
 which is itself a pc-macro command, and commands @('induct') and @('bash').)
 Rather than issuing @('(then induct bash)'), or worse yet issuing @('induct')
 and then issuing @('bash') for each resulting goal, the above definition of
 @('ib') would let you issue @('ib') and get the same effect.</p>

 @({
 General Form:
 (define-pc-macro cmd args doc-string dcl ... dcl body)
 })

 <p>where @('cmd') is the name of the pc-macro than you want to define,
 @('args') is its list of formal parameters.  @('Args') may include lambda-list
 keywords @('&optional') and @('&rest'); see @(see macro-args), but note that
 here, @('args') may not include @('&key') or @('&whole').</p>

 <p>The value of @('body') should be an @(see error-triple), of the form @('(mv
 erp xxx state)') for some @('erp') and @('xxx').  If @('erp') is @('nil'),
 then @('xxx') is handed off to the interactive proof-builder's instruction
 interpreter.  Otherwise, evaluation typically halts.</p>")

(defxdoc define-pc-meta
  :parents (proof-builder)
  :short "Define a proof-builder meta command"
  :long "<p>Built-in meta commands of the interactive @(see proof-builder)
 include @('undo') and @('restore'), and others (@('lisp'), @('exit'), and
 @('sequence')); see @(see proof-builder-commands).  The advanced proof-builder
 user can define these as well.  See ACL2 source file @('proof-builder-b.lisp')
 for examples, and contact the ACL2 implementors if those examples do not
 provide sufficient documentation.</p>")

(defxdoc define-trusted-clause-processor
  :parents (events)
  :short "Define a trusted (unverified) goal-level simplifier"
  :long "<p>This @(see documentation) assumes familiarity with
 @(':clause-processor') rules; see @(see clause-processor).  Briefly put, a
 <i>clause-processor</i> is a user-defined function that takes as input the
 ACL2 representation of a goal &mdash; a <i>clause</i> &mdash; and returns a
 list of goals (i.e., a list of clauses).  A @(':clause-processor') rule is a
 way to inform ACL2 that a clause-processor has been proved correct and now may
 be specified in @(':clause-processor') @(see hints).</p>

 <p>Here we describe a utility, @('define-trusted-clause-processor'), that
 provides another way to inform ACL2 that a function is to be considered a
 clause-processor that can be specified in a @(':clause-processor') hint.  You
 can find examples of correct and incorrect use of this utility in community
 book @('books/clause-processors/basic-examples').</p>

 <p>Consider the simple example already presented for @(':clause-processor')
 rules (again, see @(see clause-processor)), for a simple clause-processor
 named @('note-fact-clause-processor').  Instead of introducing an evaluator
 and proving a correctness theorem with @(':rule-classes :clause-processor'),
 we can simply inform ACL2 that we trust the function
 @('note-fact-clause-processor') to serve as a clause-processor.</p>

 @({
  (define-trusted-clause-processor
    note-fact-clause-processor
    nil
    :ttag my-ttag)
 })

 <p>A non-nil @(':ttag') argument generates a @(tsee defttag) event in order to
 acknowledge the dependence of the ACL2 session on the (unproved) correctness
 of this clause-processor.  That argument can be omitted if there is currently
 an active trust tag.  Note that the extra @(tsee defttag) event will be @(see
 local) to the @('define-trusted-clause-processor') event; that is, its effect
 will disappear after the @('define-trusted-clause-processor') event completes.
 This point becomes clear if one understands that a call of
 @('define-trusted-clause-processor') expands to a call of @(tsee encapsulate),
 and a @(tsee defttag) event is essentially @(see local) within any @(tsee
 encapsulate) event, as is any event that sets the @(tsee acl2-defaults-table).
 See @(see defttag).  Because we are trusting this clause-processor, rather
 than having proved it correct, we refer to it as a ``trusted''
 clause-processor to contrast with a proved-correct, or ``verified'',
 clause-processor.</p>

 <p>Now that the event displayed above has established
 @('note-fact-clause-processor') as a (trusted) clause-processor, we can use it
 in a @(':clause-processor') hint, for example as follows.  Notice that the
 output is identical to that for the corresponding example presented for the
 verified case (see @(see clause-processor)), except that the word ``verified''
 has been replaced by the word ``trusted''.</p>

 @({
  ACL2 !>(thm (equal (car (cons x y))
                     x)
              :hints
              ((\"Goal\"
                :clause-processor
                (note-fact-clause-processor clause '(equal a a)))))

  [Note:  A hint was supplied for our processing of the goal above.
  Thanks!]

  We now apply the trusted :CLAUSE-PROCESSOR function NOTE-FACT-CLAUSE-
  PROCESSOR to produce two new subgoals.

  Subgoal 2
  (IMPLIES (EQUAL A A)
           (EQUAL (CAR (CONS X Y)) X)).

  But we reduce the conjecture to T, by the :executable-counterpart of
  IF and the simple :rewrite rule CAR-CONS.

  Subgoal 1
  (EQUAL A A).

  But we reduce the conjecture to T, by primitive type reasoning.

  Q.E.D.

  Summary
  Form:  ( THM ...)
  Rules: ((:EXECUTABLE-COUNTERPART IF)
          (:EXECUTABLE-COUNTERPART NOT)
          (:FAKE-RUNE-FOR-TYPE-SET NIL)
          (:REWRITE CAR-CONS))
  Warnings:  None
  Time:  0.00 seconds (prove: 0.00, print: 0.00, other: 0.00)

  Proof succeeded.
  ACL2 !>
 })

 <p>Indeed, if one runs this example first and subsequently verifies the
 clause-processor, one will see the word ``trusted'' change to
 ``verified''.</p>

 <p>The general form is as follows.</p>

 @({
  (define-trusted-clause-processor
    cl-proc           ;;; clause-processor function
    supporters        ;;; see below
    &key
    label             ;;; optional; cl-proc$label by default
    ttag              ;;; discussed above
    partial-theory    ;;; optional encapsulate event
    )
 })

 <p>We discussed the @(':ttag') argument above, and we will discuss the
 @('supporters') and @(':partial-theory') arguments later below.  Let us turn
 our attention to the @('label') argument and its ramifications for undoing and
 redundancy.</p>

 <p>As mentioned above, a successful @('define-trusted-clause-processor') event
 results in an @(tsee encapsulate) event.  If the @(':label') argument is
 supplied with a non-@('nil') value @('L'), or if @(':label') is omitted and
 @('L') is the result of adding the suffix @('\"$LABEL\"') to @('cl-proc'),
 then the event @('(deflabel L)') will be included under the resulting
 @('encapsulate') form.  Thus, you will be able to undo this
 @('define-trusted-clause-processor') with @(':')@(tsee ubt)@(' L').  Also,
 because of the criteria for redundant encapsulate events (see @(see
 REDUNDANT-ENCAPSULATE)), the entire form is considered redundant (skipped) if
 it is identical to one already executed in the current ACL2 @(see world), with
 one exception: if @(':partial-theory') is @('nil') or omitted, and also
 @(':label nil') is supplied explicitly, then the event will not be redundant.
 If the event is not redundant, then @('cl-proc') must not already be
 designated as a trusted clause-processor.</p>

 <p>Note that @('cl-proc') may be defined either in @(':program')-mode or
 @(':logic')-mode.</p>

 <p>The @('supporters') argument should be a true list of function symbols in
 the current ACL2 world.  It is important that this list include user-defined
 functions whose definitions support the correctness of the clause-processor
 function.  Otherwise, @(tsee local) definitions of those missing supporters
 can render the use of this clause-processor unsound, as discussed in the paper
 referenced at the end of the @(see clause-processor) documentation topic.
 Below we discuss an additional reason that @('supporters') is critical for
 soundness, in the case of dependent clause-processors.</p>

 <p>(Remark.  There could have been two notions of supporters: one for
 functions whose definitions support the correctness of the clause-processor
 function, and, in the case of dependent clause-processors, one for supporters
 of the ``promised encapsulate'' discussed below.  But for simplicity, a single
 @('supporters') argument serves both purposes.)</p>

 <p><b>Dependent clause-processors and promised encapsulates</b>: The
 @(':partial-theory') argument</p>

 <p>Suppose you want to introduce a clause-processor to reason about a complex
 hardware simulator that is implemented outside ACL2.  Sawada and Reeber had
 just such a problem, as reported in their FMCAD 2006 paper.  Indeed, they used
 @(tsee sys-call) to implement a @(':')@(tsee program)-mode function in ACL2
 that can invoke that simulator.  (This has been changed to @(tsee sys-call*)
 since @('sys-call') cannot invoke the OS during proofs; see @(see sys-call).)
 In principle one could code the simulator directly in ACL2; but it would be a
 tremendous amount of work that has no practical purpose, given the interface
 to the external simulator.  So: In what sense can we have a clause-processor
 that proves properties about a simulator when that simulator is not fully
 axiomatized in ACL2?  Our answer, in a nutshell, is this: The above
 @(':partial-theory') argument provides a way to write merely some of the @(see
 constraint)s on the external tool (or even no constraints at all), with the
 understanding that such constraints are present implicitly in a stronger
 ``promised'' @('encapsulate'), for example by exporting the full
 definition.</p>

 <p>If a trusted clause-processor is introduced with a non-@('nil')
 @(':partial-theory') argument, we call it a ``dependent'' clause-processor,
 because its correctness is dependent on the constraints implicitly introduced
 by the @(':partial-theory') @('encapsulate') form.  The implicit constraints
 should logically imply the constraints actually introduced by the explicit
 @('encapsulate'), but they should also be sufficient to justify every possible
 invocation of the clause-processor in a @(':clause-processor') hint.  The user
 of a @('define-trusted-clause-processor') form is making a guarantee &mdash;
 or, is relying on a guarantee provided by the writer of that form &mdash; that
 in principle, there exists a so-called ``promised encapsulate'': an
 @('encapsulate') form with the same @(see signature) as the
 @(':partial-theory') @('encapsulate') form associated with the trusted
 clause-processor, but whose constraints introduced are the aforementioned
 implicit constraints.</p>

 <p>There are several additional requirements on a @(':partial-theory')
 argument.  First, it must be an @(tsee encapsulate) event with non-empty @(see
 signature).  Moreover, the functions introduced by that event must be exactly
 those specified in the signature, and no more.  And further still, the
 @('define-trusted-clause-processor') form cannot be executed inside any @(tsee
 encapsulate) form with non-empty @(see signature); we can think of this
 situation as attempting to associate more than one @('encapsulate') with the
 functions introduced in the inner @('encapsulate').</p>

 <p>Moreover, soundness depends on inclusion of enough function symbols in the
 @('supporters') argument, as follows.  Let @('S') be the set of specified
 @('supporters') augmented by the set of function symbols either introduced by,
 or in a property exported by, the @(':partial-theory') argument, which we call
 the ``promised encapsulate''.  Then every function symbol constrained by the
 promised encapsulate is in @('S').</p>

 <p>The @(':partial-theory') event will (in essence) be executed as part of the
 evaluation of the @('define-trusted-clause-processor') form.  Again, a
 critical obligation rests on the user who provides a @(':partial-theory'):
 there must exist (in principle at least) a corresponding promised encapsulate
 form with the same @(see signature) that could logically be admitted, whenever
 the above @('define-trusted-clause-processor') form is evaluated successfully,
 that justifies the designation of @('cl-proc') as a clause-processor.  See
 also the paper mentioned above for more about promised encapsulates.  A key
 consequence is that the @(see constraint)s are unknown for the functions
 introduced in (the signature of) a @(':partial-theory') @(tsee encapsulate)
 form.  Thus, functional instantiation (see @(see
 functional-instantiation-example)) is disabled for function in the signature
 of a @(':partial-theory') form.</p>

 <p><b>A remark on the underlying implementation</b></p>

 <p>You can see all of the current trusted clause-processors by issuing the
 command @('(table trusted-cl-proc-table)').  The resulting alist associates
 each trusted clause-processor with its supporters.</p>

 <p>Note that @('define-trusted-clause-processor') is actually a macro that
 generates (among other things) a @('table') event for extending
 @('trusted-cl-proc-table').  You are invited to use @(':')@(tsee trans1) to
 see expansions of calls of this macro.  In particular, you can see that the
 @(':partial-theory') argument results in an @('encapsulate') event that
 includes a call of the form @('(set-unknown-constraints-supporters f1
 ... fk)'), which in effect makes that call of @('encapsulate') into a call of
 @('partial-encapsulate') with supporters @('(f1 ... fk)').  See @(see
 partial-encapsulate).</p>

 <p><b>A technique for using raw Lisp to define a trusted
 clause-processor</b></p>

 <p>The following code is intended to give an idea for how one might define the
 ``guts'' of a trusted clause-processor in raw Lisp.  The idea is to stub out
 functions, such as @('acl2-my-prove below'), that you want to define in raw
 Lisp; and then, load a raw Lisp file to overwrite any such function with the
 real code.  But then we make any such overwritten function untouchable.  (This
 last step is important because otherwise, one can prove @('nil') using a
 @(':functional-instance') @(':use') hint, by exploiting the fact that this
 function has executable code for which there is no corresponding definitional
 axiom.)  Note: The point here is only to illustrate the use of raw Lisp, so we
 do not bother to define or explain functions @('hint-to-termlist') or
 @('disjoin-clause-segments-to-clause'), which this example assumes are defined
 elsewhere; their meanings are not important for this example.</p>

 @({
  (defstub acl2-my-prove (term hint) t)

  (program)

  (defttag :my-cl-proc)

  (progn

  ; We wrap everything here in a single progn, so that the entire form is
  ; atomic.  That's important because we want the use of push-untouchable to
  ; prevent anything besides my-clause-processor from calling acl2-my-prove.

    (progn!

     (set-raw-mode-on state)

     (load \"my-hint-raw.lsp\") ; defines my-prove in raw Lisp

     (defun acl2-my-prove (term hint)
       (my-prove term hint)))

    (defun my-clause-processor (cl hint)
      (declare (xargs :guard (pseudo-term-listp cl)
                      :mode :program))
      (if (acl2-my-prove (disjoin cl) hint)
          (disjoin-clause-segments-to-clause
           (pairlis$ (hint-to-termlist hint) nil)
           cl)
        (prog2$ (cw \"~|~%NOTE: Unable to prove goal with ~
                    my-clause-processor and indicated hint.~|\")
                (list cl))))

    (push-untouchable acl2-my-prove t)
    )
 })")

(defxdoc definition
  :parents (rule-classes)
  :short "Make a rule that acts like a function definition"
  :long "<p>See @(see rule-classes) for a general discussion of rule classes
 and how they are used to build rules from formulas.  An example @(':')@(tsee
 corollary) formula from which a @(':definition') rule might be built is:</p>

 @({
  Examples:
  (defthm open-len-twice
    (implies (true-listp x)
             (equal (len x)
                    (if (null x)
                        0
                      (if (null (cdr x))
                          1
                        (+ 2 (len (cddr x)))))))
    :rule-classes :definition)

  ; Same as above, with :controller-alist made explicit:
  (defthm open-len-twice
    (implies (true-listp x)
             (equal (len x)
                    (if (null x)
                        0
                      (if (null (cdr x))
                          1
                        (+ 2 (len (cddr x)))))))
    :rule-classes ((:definition :controller-alist ((len t)))))

  General Form:
  (implies hyp (equiv (fn a1 ... an) body))
 })

 <p>where @('equiv') is an equivalence relation and @('fn') is a function
 symbol other than @(tsee if), @(tsee hide), @(tsee force) or @(tsee
 case-split).  Such rules allow ``alternative'' definitions of @('fn') to be
 proved as theorems but used as definitions.  These rules are not true
 ``definitions'' in the sense that they (a) cannot introduce new function
 symbols and (b) do not have to be terminating recursion schemes.  They are
 just conditional rewrite rules that are controlled the same way we control
 recursive definitions.  We call these ``definition rules'' or ``generalized
 definitions''.</p>

 <p>Consider the general form above.  Generalized definitions are stored among
 the @(':')@(tsee rewrite) rules for the function ``defined,'' @('fn') above,
 but the procedure for applying them is a little different.  During rewriting,
 instances of @('(fn a1 ... an)') are replaced by corresponding instances of
 @('body') provided the @('hyp')s can be established as for a @(':')@(tsee
 rewrite) rule and the result of rewriting @('body') satisfies the criteria for
 function expansion.  There are two primary criteria, either of which permits
 expansion.  The first is that the ``recursive'' calls of @('fn') in the
 rewritten body have arguments that already occur in the goal conjecture.  The
 second is that the ``controlling'' arguments to @('fn') are simpler in the
 rewritten body.</p>

 <p>The notions of ``recursive call'' and ``controllers'' are complicated by
 the provisions for mutually recursive definitions.  Consider a ``clique'' of
 mutually recursive definitions.  Then a ``recursive call'' is a call to any
 function defined in the clique and an argument is a ``controller'' if it is
 involved in the measure that decreases in all recursive calls.  These notions
 are precisely defined by the definitional principle and do not necessarily
 make sense in the context of generalized definitional equations as implemented
 here.</p>

 <p>But because the heuristics governing the use of generalized definitions
 require these notions, it is generally up to the user to specify which calls
 in body are to be considered recursive and what the controlling arguments are.
 This information is specified in the @(':clique') and @(':controller-alist')
 fields of the @(':definition') rule class.</p>

 <p>The @(':clique') field is the list of function symbols to be considered
 recursive calls of @('fn').  In the case of a non-recursive definition, the
 @(':clique') field is empty; in a singly recursive definition, it should
 consist of the singleton list containing @('fn'); otherwise it should be a
 list of all of the functions in the mutually recursive clique with this
 definition of @('fn').</p>

 <p>If the @(':clique') field is not provided it defaults to @('nil') if
 @('fn') does not occur as a function symbol in @('body') and it defaults to
 the singleton list containing @('fn') otherwise.  Thus, @(':clique') must be
 supplied by the user only when the generalized definition rule is to be
 treated as one of several in a mutually recursive clique.</p>

 <p>The @(':controller-alist') is an alist that maps each function symbol in
 the @(':clique') to a mask specifying which arguments are considered
 controllers.  The mask for a given member of the clique, @('fn'), must be a
 list of @('t')'s and @('nil')'s of length equal to the arity of @('fn').  A
 @('t') should be in each argument position that is considered a ``controller''
 of the recursion.  For a function admitted under the principle of definition,
 an argument controls the recursion if it is one of the arguments measured in
 the termination argument for the function.  But in generalized definition
 rules, the user is free to designate any subset of the arguments as
 controllers.  Failure to choose wisely may result in the ``infinite
 expansion'' of definitional rules but cannot render ACL2 unsound since the
 rule being misused is a theorem.</p>

 <p>If the @(':controller-alist') is omitted it can sometimes be defaulted
 automatically by the system.  If the @(':clique') is @('nil'), the
 @(':controller-alist') defaults to @('nil').  If the @(':clique') is a
 singleton containing @('fn'), the @(':controller-alist') defaults to the
 controller alist computed by @('(defun fn args body)').  (The user can obtain
 some control over this analysis by setting the default ruler-extenders; see
 @(see rulers).)  If the @(':clique') contains more than one function, the user
 must supply the @(':controller-alist') specifying the controllers for each
 function in the clique.  This is necessary since the system cannot determine
 and thus cannot analyze the other definitional equations to be included in the
 clique.</p>

 <p>For example, suppose @('fn1') and @('fn2') have been defined one way and it
 is desired to make ``alternative'' mutually recursive definitions available to
 the rewriter.  Then one would prove two theorems and store each as a
 @(':definition') rule.  These two theorems would exhibit equations
 ``defining'' @('fn1') and @('fn2') in terms of each other.  No provision is
 here made for exhibiting these two equations as a system of equations.  One is
 proved and then the other.  It just so happens that the user intends them to
 be treated as mutually recursive definitions.  To achieve this end, both
 @(':definition') rules should specify the @(':clique') @('(fn1 fn2)') and
 should specify a suitable @(':controller-alist').  If, for example, the new
 definition of @('fn1') is controlled by its first argument and the new
 definition of @('fn2') is controlled by its second and third (and they each
 take three arguments) then a suitable @(':controller-alist') would be @('((fn1
 t nil nil) (fn2 nil t t))').  The order of the pairs in the alist is
 unimportant, but there must be a pair for each function in the clique.</p>

 <p>Inappropriate heuristic advice via @(':clique') and @(':controller-alist')
 can cause ``infinite expansion'' of generalized definitions, but cannot render
 ACL2 unsound.</p>

 <p>Note that the actual definition of @('fn1') has the runic name
 @('(:definition fn1)').  The runic name of the alternative definition is
 @('(:definition lemma)'), where @('lemma') is the name given to the event that
 created the generalized @(':definition') rule.  This allows theories to switch
 between various ``definitions'' of the functions.</p>

 <p>By default, a @(':definition') rule establishes the so-called ``body'' of a
 function.  The body is used by @(':expand') @(see hints), and it is also used
 heuristically by the theorem prover's preprocessing (the initial
 simplification using ``simple'' rules that is controlled by the
 @('preprocess') symbol in @(':do-not') @(see hints)), induction analysis, and
 the determination for when to warn about non-recursive functions in rules.
 The body is also used by some heuristics involving whether a function is
 recursively defined, and by the @('expand'), @('x'), and @('x-dumb') commands
 of the interactive @(see proof-builder).</p>

 <p>See @(see rule-classes) for a discussion of the optional field
 @(':install-body') of @(':definition') rules, which controls whether a
 @(':definition') rule is used as described in the paragraph above.  Note that
 even if @(':install-body nil') is supplied, the rewriter will still rewrite
 with the @(':definition') rule; in that case, ACL2 just won't install a new
 body for the top function symbol of the left-hand side of the rule, which for
 example affects the application of @(':expand') hints as described in the
 preceding paragraph.  Also see @(see set-body) and see @(see show-bodies) for
 how to change the body of a function symbol.</p>

 <p>Note only that if you prove a definition rule for function @('foo'), say,
 @('foo-new-def'), you will need to refer to that definition as
 @('foo-new-def') or as @('(:DEFINITION foo-new-def)').  That is because a
 @(':definition') rule does not change the meaning of the symbol @('foo') for
 @(':use') @(see hints), nor does it change the meaning of the symbol @('foo')
 in theory expressions; see @(see theories), in particular the discussion there
 of runic designators.  Similarly @(':')@(tsee pe) @('foo') and @(':')@(tsee
 pf) @('foo') will still show the original definition of @('foo').</p>

 <p>The definitional principle, @(tsee defun), actually adds @(':definition')
 rules.  Thus the handling of generalized definitions is exactly the same as
 for ``real'' definitions because no distinction is made in the implementation.
 Suppose @('(fn x y)') is @(tsee defun)'d to be @('body').  Note that @(tsee
 defun) (or @(tsee defuns) or @(tsee mutual-recursion)) can compute the clique
 for @('fn') from the syntactic presentation and it can compute the controllers
 from the termination analysis.  Provided the definition is admissible, @(tsee
 defun) adds the @(':definition') rule @('(equal (fn x y) body)').</p>")

(defxdoc deflabel
  :parents (events)
  :short "Build a landmark"
  :long "@({
 Examples:
 (deflabel interp-section)

 General Form:
 (deflabel name)
 })

 <p>where @('name') is a new symbolic name (see @(see name)).  By virtue of the
 fact that @('deflabel') is an event, it marks the current @(see history) with
 the @('name').  For example, you may wish to undo back through some label or
 compute a theory expression (see @(see theories)) in terms of some labels.
 @('Deflabel') @(see events) are never considered redundant.  See @(see
 redundant-events).</p>")

(defxdoc deflock
  :parents (parallel-programming)
  :short "Define a wrapper macro that provides mutual exclusion in ACL2(p)"
  :long "<p>This @(see documentation) topic relates to the experimental
 extension of ACL2 supporting parallel evaluation and proof; see @(see
 parallelism).</p>

 @({
  Example Form:
  (deflock *my-lock*)

  General Form:
  (deflock *symbol*)
 })

 <p>where @('*symbol*') is a symbol whose first and last characters are both
 the character @('#\\*').</p>

 <p>A call of this macro generates a definition of another macro, named
 @('with-<modified-lock-symbol>'), where @('<modified-lock-symbol>') is the
 given symbol with the leading and trailing @('*') characters removed.  This
 newly defined macro will guarantee mutually exclusive execution when called in
 the body of the raw Lisp definition of a function, as is typically the case
 for @(see guard)-verified functions, for @(':')@(tsee program) mode functions,
 and for calls of macro @(tsee top-level).  (See @(see guard-evaluation-table)
 for details of how raw Lisp code might not be invoked when guard-checking (see
 @(see set-guard-checking)) has value @(':none') or @(':all').)  Note that this
 macro is also simply the identity when invoked directly in the