use ui;
use layout;
use graphics;
use core:geometry;
use core:io;
use progvis:net;
use progvis:program;
use lang:bs:macro;

/**
 * Examples directory.
 */
Url? examplesDir() on Ui {
	if (x = named{progvis}.url()) {
		x = x / ".." / "progvis_demo";
		if (x.exists())
			return x;
	}

	null;
}

/**
 * Main window.
 */
class MainWin extends Frame {
	private MainPainter painter;

	// Currently open file(s).
	private Url[] currentFiles;

	// Current behavior.
	private Behavior behavior;

	// Host to connect to when going online.
	private Str onlineUrl;

	// Current client connection, if any.
	private Client? client;

	// The 'online' menu so that we can enable/disable items as we please.
	// The first one is the "sign in" item.
	private PopupMenu onlineMenu;

	// Menu element for selecting the memory model.
	private Menu:Check[] memOptions;

	// Menu element for selecting whether or not to report memory leaks.
	private Menu:Check reportLeaks;

	// Currently selected memory option.
	private Nat selectedMemOption;

	// Settings.
	private Settings settings;

	init() {
		// Set the icon globally.
		app.icon = progvis_icon;

		init("Progvis", Size(800, 800)) {
			settings = Settings:load();
			onlineUrl = "storm-lang.org";
			reportLeaks(mnemonic("Report _memory leaks"));
		}

		applySettings();

		PopupMenu fileMenu;
		fileMenu
			<< Menu:Text(mnemonic("_Open program..."), ctrlChord(Key:o), &this.onOpen())
			<< Menu:Text(mnemonic("_Reload program"), KeyChord(Key:r, Modifiers:ctrl + Modifiers:shift), &this.onReload())
			<< Menu:Text(mnemonic("Open in _editor"), ctrlChord(Key:e), &this.onOpenEditor());

		if (examplesDir())
			fileMenu
				<< Menu:Separator()
				<< Menu:Text("Open example...", &this.onOpenExample());

		fileMenu
			<< Menu:Separator()
			<< Menu:Text("Save screenshot...", ctrlChord(Key:s), &this.onSaveScreenshot)
			<< Menu:Separator()
			<< Menu:Text(mnemonic("_Settings..."), &this.onSettings());

		memOptions
			<< Menu:Check(mnemonic("_Disabled"))
			<< Menu:Check(mnemonic("Only _statements"))
			<< Menu:Check(mnemonic("_Full"));

		setMemoryCheck(2);

		PopupMenu memCheck;
		for (x in memOptions) {
			memCheck << x;
			var me = this;
			x.onClick = (Bool v) => me.onMemoryChange(x, v);
		}

		reportLeaks.onClick = &this.setReportLeaks;
		setReportLeaks(true);

		PopupMenu runMenu;
		runMenu
			<< Menu:Text(mnemonic("_Restart program"), ctrlChord(Key:r), &this.onRestart())
			<< Menu:Text(mnemonic("_Spawn new thread"), ctrlChord(Key:n), &this.onSpawnThread())
			<< Menu:Submenu(mnemonic("Report _data races"), memCheck)
			<< reportLeaks
			<< Menu:Separator()
			<< Menu:Text(mnemonic("_Look for errors"), ctrlChord(Key:l), &this.onFindErrors())
			<< Menu:Separator()
			<< Menu:Text(mnemonic("S_top all threads"), ctrlChord(Key:q), &this.onStopAll())
			<< Menu:Text(mnemonic("Start _all threads (slow)"), ctrlChord(Key:a), &this.onRunSlow())
			<< Menu:Text(mnemonic("Start all threads (_fast)"), ctrlChord(Key:f), &this.onRunFast());

		onlineMenu
			<< Menu:Text(mnemonic("_Connect"), &this.connect())
			<< Menu:Text(mnemonic("_Disconnect"), &this.disconnect())
			<< Menu:Text(mnemonic("_Highscores..."), &this.onOnlineStatus())
			<< Menu:Text(mnemonic("_Problems..."), ctrlChord(Key:p), &this.onOnlineProblems())
			<< Menu:Text("Submit problem...", &this.onOnlineSubmit())
			<< Menu:Text("Sign out", &this.logout());

		for (Nat i = 1; i < onlineMenu.count; i++)
			onlineMenu[i].enabled = false;

		PopupMenu helpMenu;
		helpMenu << Menu:Text(mnemonic("_About..."), &this.onAbout());

		MenuBar m;
		m
			<< Menu:Submenu(mnemonic("_File"), fileMenu)
			<< Menu:Submenu(mnemonic("_Run"), runMenu)
			<< Menu:Submenu(mnemonic("_Online"), onlineMenu)
			<< Menu:Submenu(mnemonic("_Help"), helpMenu);

		menu = m;

		painter(painter);
		create();
	}

	// Mostly for testing.
	void open(Url file) {
		open([file]);
	}

	// Called when the frame is closed.
	void close() : override {
		behavior.onDispose();
		painter.cleanup();
		super:close();
	}

	void applySettings() {
		painter.setZoom(settings.zoom);
		painter.repaint();
	}

	// For debug mode.
	assign onlineHost(Str to) {
		onlineUrl = to;
	}

	// Connect in the background.
	public void connect() {
		if (client.empty) {
			painter.showMessage("Connecting...");
			onlineMenu[0].enabled = false;
			(spawn bgConnect()).detach();
		}
	}

	// Function executed in the background to connect.
	private void bgConnect() {
		try {
			Client c = Client:connect(onlineUrl, settings.onlineId);
			this.client = c;

			for (Nat i = 1; i < onlineMenu.count; i++)
				onlineMenu[i].enabled = true;

			// This is only for testing.
			// if (onlineUrl == "localhost") {
			// 	painter.hideMessage();
			// 	onOnlineProblems();
			// 	return;
			// }

			painter.showMessage("Signed in as: ${c.username}");
			sleep(2 s);
		} catch (SignInRedirect redirect) {
			showMessage(this, "Sign in", "You need to sign in before using the online features. Click OK to continue.");
			try {
				core:sys:open(redirect.to);
			} catch (Exception e) {
				print("Trying to open: " + redirect.to.format);
				print(e.toS);
				showMessage(this, "Could not start web browser", "Failed to start a web browser to complete the sign in process. Make sure one is installed and try again.");
			}
			onlineMenu[0].enabled = true;
		} catch (ServerError error) {
			showMessage(this, "Error", "The server returned an error: ${error.message}");
			onlineMenu[0].enabled = true;
		} catch (Exception error) {
			print(error.toS);
			showMessage(this, "Error", "Error connecting. Check that your version of Progvis is up to date.");
			onlineMenu[0].enabled = true;
		}

		painter.hideMessage();
	}

	// For testing, to ensure that we can terminate the server cleanly.
	public void disconnect() {
		if (c = client) {
			c.close();
		}
		client = null;

		for (Nat i = 1; i < onlineMenu.count; i++)
			onlineMenu[i].enabled = false;
		onlineMenu[0].enabled = true;

		// Reset the behavior so that we don't try to use the client anymore.
		updateBehavior(DefaultBehavior(this));
	}

	private void logout() {
		if (c = client) {
			c.query(LogoutRequest());
			disconnect();
		}
	}

	Bool onMouseMove(Point pt) : override {
		painter.mouseMoved(pt);
		true;
	}

	void onMouseEnter() : override {}

	void onMouseLeave() : override {
		painter.mouseLeave();
	}

	Bool onClick(Bool down, Point pt, MouseButton button) {
		painter.mouseClicked(pt, down, button);
		true;
	}

	// Used to transition between behaviors. Typically set through "open", but some behaviors need
	// to switch during a simulation.
	public void updateBehavior(Behavior b) {
		behavior.onDispose();
		behavior = b;
		painter.updateBehavior(b);
		onNewBehavior();
		repaint();
	}

	// Set the default behavior again.
	public void resetBehavior() {
		updateBehavior(DefaultBehavior(this));
	}

	// Called when a new behavior has been set.
	private void onNewBehavior() {
		if (!behavior.allowTrackMemory()) {
			setMemoryCheck(2);
			for (x in memOptions)
				x.enabled = false;
		} else {
			for (i, x in memOptions) {
				x.enabled = true;
			}
		}

		if (!behavior.allowNoLeaks) {
			setReportLeaks(true);
			reportLeaks.enabled = false;
		} else {
			reportLeaks.enabled = true;
		}
	}

	private void onOpen() {
		FilePicker picker = createPicker();
		if (!picker.show(this))
			return;

		open(picker.results);
	}

	private void onOpenExample() {
		unless (folder = examplesDir())
			return;

		FilePicker picker = createPicker().defaultFolder(folder);
		if (!picker.show(this))
			return;

		open(picker.results);
	}

	// Create a file picker.
	private FilePicker createPicker() {
		FileTypes ft("Source code");
		for (k in Program:supportedFiles)
			ft.add("${k}-source", [k]);

		return FilePicker:open(ft).okLabel("Open").multiselect();
	}

	public void onReload() {
		if (currentFiles.empty) {
			showMessage(this, "Error", "You need to open a file before you can reload it.");
		} else if (msg = behavior.allowReload()) {
			showMessage(this, "Error", msg);
		} else {
			try {
				painter.open(currentFiles, behavior);
			} catch (core:lang:CodeError error) {
				CompileErrorDialog dlg(error.messageText, error.pos);
				dlg.show(this);
			} catch (Exception error) {
				// Print the stack trace in the terminal to make it easier to debug.
				print("Error:\n${error}");
				showMessage(this, "Error opening code", "Unable to open the selected files:\n${error.message}");
			}
		}
	}

	public void onOpenEditor() {
		if (currentFiles.empty()) {
			showMessage(this, "No files open", "You have no files open currently.");
		} else {
			openEditor(currentFiles[0]);
		}
	}

	// Open an editor for the specified file.
	public void openEditor(Url file) {
		try {
			settings.open(file);
		} catch (Exception e) {
			showMessage(this, "Failed to launch editor", "Failed to launch the editor: ${e}. Is your configuration correct?");
		}
	}

	private void onOnlineStatus() {
		if (c = client) {
			StatusDlg(c).show(this);
		}
	}

	private void onOnlineProblems() {
		if (c = client) {
			problems:ProblemsDlg dlg(c, settings);
			dlg.show(this);
			if (action = dlg.action) {
				open(action.files, "Solving: " + action.problem.title, action.behavior(this, c));
			}
		}
	}

	private void onOnlineSubmit() {
		unless (c = client)
			return;

		problems:UploadDlg(c).show(this);
	}

	// Get source for the current file.
	public Str currentSource() {
		if (currentFiles.count < 1 | !painter.hasProgram)
			throw InternalError("Failed to acquire source for the current file.");
		if (text = painter.sourceFor(currentFiles[0]))
			return text;
		throw InternalError("Failed to acquire source for the current file.");
	}

	// Get all source code, as files in a memory listing.
	public Url[] allCurrentSource() {
		MemoryProtocol memory;
		Url[] result;
		for (f in currentFiles) {
			if (text = painter.sourceFor(f)) {
				MemOStream out;
				Utf8Output tOut(out);
				tOut.write(text);
				tOut.flush();
				result << memory.put(f.name, out.buffer);
			}
		}
		result;
	}

	private void onSettings() {
		SettingsDlg(settings, &this.applySettings).show(this);
	}

	public void clearClient() {
		client = null;
	}

	public void onRestart() {
		painter.restart();
	}

	private void onSpawnThread() {
		if (error = behavior.allowSpawnThread) {
			showMessage(this, "Error", error);
			return;
		}
		painter.spawnThread();
	}

	private void onMemoryChange(Menu:Check check, Bool value) {
		if (!value) {
			Bool any = false;
			for (i, x in memOptions)
				any |= x.checked;

			if (!any)
				check.checked = true;
			return;
		}

		Nat selected = 0;
		for (i, x in memOptions)
			if (x is check)
				selected = i;

		setMemoryCheck(selected);
	}

	private void setMemoryCheck(Nat item) {
		selectedMemOption = item;

		for (i, x in memOptions)
			memOptions[i].checked = i == item;

		var v = if (item == 0) {
			data:MemoryChecks:sequential;
		} else if (item == 1) {
			data:MemoryChecks:statements;
		} else {
			data:MemoryChecks:barriers;
		};

		painter.checkMemory(v);
	}

	private void setReportLeaks(Bool enable) {
		if (reportLeaks.checked != enable)
			reportLeaks.checked = enable;

		painter.checkLeaks(enable);
	}

	private void onFindErrors() {
		if (msg = behavior.allowModelCheck) {
			showMessage(this, "Can not check for errors", msg);
			return;
		}

		// Note: 'allCurrentSource' picks the current version of the source, even if changed in the file system.
		Url[] source = allCurrentSource();
		if (source.empty) {
			showMessage(this, "No files open", "You have no files open currently.");
		} else if (result = check:findErrorsUi(this, source, settings.checks)) {
			onRestart();
			updateBehavior(ReplayBehavior(this, result.sequence));
		}
	}

	private void onStopAll() {
		painter.stopThreads();
	}

	private void onRunSlow() {
		painter.resumeThreads(1 s);
	}

	private void onRunFast() {
		painter.resumeThreads(500 ms);
	}

	private void onAbout() {
		var license = named{PROGVIS};
		var version = named{PROGVIS_VERSION};
		showLicenseDialog(this, ProgramInfo("Progvis", "Filip Strömbäck", version.version, license));
	}

	// Apply an action as if the user clicked something. The "step" received here is the same format
	// that the Behavior class receives in its "onUserAction".
	public Bool applyAction(Str step) {
		painter.applyAction(step);
	}

	// Open one or more files, shows nice messages on error.
	void open(Url[] files) {
		open(files, join(files, ", ", (x) => x.name).toS, DefaultBehavior(this));
	}

	private void open(Url[] files, Str title, Behavior behavior) {
		currentFiles = files;
		this.text = "Progvis - " + title;

		// Set it to default first, so that in case of failure we will behave as "normal".
		this.behavior.onDispose();
		this.behavior = DefaultBehavior(this);
		onNewBehavior();

		try {
			painter.cleanup();
			painter.open(files, behavior);
			this.behavior = behavior;
			onNewBehavior();
		} catch (core:lang:CodeError error) {
			CompileErrorDialog dlg(error.messageText, error.pos);
			dlg.show(this);
		} catch (Exception error) {
			// Print the stack trace in the terminal to make it easier to debug.
			print("Error:\n${error}");
			showMessage(this, "Error opening code", "Unable to open the selected files:\n${error.message}");
		}
	}

	// Called when an error is triggered from a program.
	private void onError(Nat threadId, Exception e) {
		showMessage(this, "Thread ${threadId} crashed", "Thread ${threadId} crashed with the following message:\n${e.message}");
	}

	// Called when a concurrency error is encountered.
	private void onConcurrencyError(Str[] messages) {
		StrBuf msg;
		msg << "Concurrency issues were encountered:\n";
		for (m in messages)
			msg << m << "\n";
		showMessage(this, "Concurrency issues", msg.toS);
	}

	// Save screenshot.
	private void onSaveScreenshot() {
		var sc = settings.screenshot;
		Url file = if (dir = sc.output) {
			Nat suffix = 0;
			Url file = dir;
			do {
				file = dir / "progvis_${suffix,f03}.png";
			} while (file.exists) {
				suffix++;
			}
			file;
		} else {
			FileTypes ft("Image files");
			for (fmt in supportedImageFormats)
				ft.add(fmt.name, fmt.extensions);
			FilePicker p = FilePicker:save(ft, null);
			if (!p.show(this))
				return;

			p.results[0];
		};

		try {
			if (sc.width == 0 | sc.height == 0) {
				painter.saveScreenshot(file, pos.size, sc.customScale);
			} else {
				painter.saveScreenshot(file, Size(sc.width.float, sc.height.float), sc.customScale);
			}
		} catch (Exception e) {
			showMessage(this, "Failed to save screenshot", e.message);
		}
	}
}


void main() on Compiler {
	named{progvis}.compile();

	MainWin win;

	{
		Url[] files;
		for (arg in core:argv) {
			Url file = parsePath(arg).makeAbsolute(cwdUrl);
			if (file.exists) {
				files << file;
			} else {
				print("WARNING: ${file} does not refer to a file.");
			}
		}

		if (files.any)
			win.open(files);
	}

	win.waitForClose();
}

void debug() on Compiler {
	// named{progvis}.compile();

	if (url = examplesDir()) {
		// var result = progvis:check:check([url / "test" / "busywait.c"]);
		// print(result.toS);
		// return;

		MainWin win;
		// win.open(url / "assert.c");
		// win.open(url / "atomics.c");
		// win.open(url / "test" / "atomic_stack.c");
		// win.open(url / "pointers" / "string.c");
		// win.open(url / "pointers" / "locals.c");
		// win.open(url / "pointers" / "ptr.c");
		// win.open(url / "concurrency" / "lock_sema.c");
		// win.open(url / "concurrency" / "bank" / "bank_good.c");
		// win.open(url / "test" / "error.c");
		// win.open(url / "test" / "multi_locks.c");
		// win.open(url / "test" / "step_error.c");
		// win.open(url / "test" / "livelock.c");
		// win.open(url / "test" / "alloc_in_nostep.c");
		// win.open(url / "test" / "globals.cpp");
		// win.open(url / "test" / "staticarrays.c");
		// win.open(url / "test" / "sizeof.c");
		// win.open(url / "test" / "complex-leaks.c");
		// win.open(url / "test" / "simple-leaks.c");
		// win.open(url / "test" / "thread_id.c");
		// win.open(url / "test" / "nested_atomics.c");
		// win.open(url / "test" / "restore_globals.c");
		// win.open(url / "test" / "busywait.c");
		// win.open(url / "test" / "atomic_busywait.c");
		win.open(url / "test" / "stack.c");
		win.waitForClose();
	}
}

// Debug the online capabilities with a local server.
void debugServer() {
	named{progvis}.compile();

	spawn simpleServer();

	MainWin win;
	win.onlineHost = "localhost";
	sleep(100 ms);
	win.connect();
	win.waitForClose();
	win.disconnect();

	// Wait for the other UThread to realize that we have disconnected.
	sleep(500 ms);
}
