﻿using NtApiDotNet;
using System;
using System.ComponentModel;
using System.Runtime.InteropServices;
using System.Text;

namespace PoC_NtlmAuthSession0_EoP
{
    enum SecBufferType
    {
        EMPTY = 0,
        DATA = 1,
        TOKEN = 2,
        PKG_PARAMS = 3,
        MISSING = 4,
        EXTRA = 5,
        STREAM_TRAILER = 6,
        STREAM_HEADER = 7,
        NEGOTIATION_INFO = 8,
        PADDING = 9,
        STREAM = 10,
        MECHLIST = 11,
        MECHLIST_SIGNATURE = 12,
        TARGET = 13,
        CHANNEL_BINDINGS = 14,
        CHANGE_PASS_RESPONSE = 15,
        TARGET_HOST = 16,
        ALERT = 17,
        APPLICATION_PROTOCOLS = 18,
        SRTP_PROTECTION_PROFILES = 19,
        SRTP_MASTER_KEY_IDENTIFIER = 20,
        TOKEN_BINDING = 21,
        PRESHARED_KEY = 22,
        PRESHARED_KEY_IDENTITY = 23,
        DTLS_MTU = 24,
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    sealed class SecBuffer : IDisposable
    {
        public int cbBuffer;
        public SecBufferType BufferType;
        public IntPtr pvBuffer;

        void IDisposable.Dispose()
        {
            if (pvBuffer != IntPtr.Zero)
            {
                Marshal.FreeHGlobal(pvBuffer);
            }
        }

        public SecBuffer()
        {
        }

        public SecBuffer(SecBufferType type, byte[] data) 
            : this(type, data.Length)
        {
            Marshal.Copy(data, 0, pvBuffer, data.Length);
        }

        public SecBuffer(SecBufferType type, int length)
        {
            cbBuffer = length;
            BufferType = type;
            pvBuffer = Marshal.AllocHGlobal(length);
        }

        public byte[] ToArray()
        {
            byte[] ret = new byte[cbBuffer];
            Marshal.Copy(pvBuffer, ret, 0, ret.Length);
            return ret;
        }
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    sealed class SecBufferDesc : IDisposable
    {
        const int SECBUFFER_VERSION = 0;

        public int ulVersion;
        public int cBuffers;
        public IntPtr pBuffers;

        void IDisposable.Dispose()
        {
            if (pBuffers != IntPtr.Zero)
            {
                Marshal.FreeHGlobal(pBuffers);
            }
        }

        public SecBufferDesc(SecBuffer buffer) : this(new SecBuffer[] { buffer })
        {
        }

        public SecBufferDesc(SecBuffer[] buffers)
        {
            int size = Marshal.SizeOf<SecBuffer>();
            ulVersion = SECBUFFER_VERSION;
            cBuffers = buffers.Length;
            pBuffers = Marshal.AllocHGlobal(buffers.Length * size);
            int offset = 0;
            foreach (var buffer in buffers)
            {
                Marshal.StructureToPtr(buffer, pBuffers + offset, false);
                offset += size;
            }
        }

        public SecBuffer[] ToArray()
        {
            SecBuffer[] buffers = new SecBuffer[cBuffers];
            int size = Marshal.SizeOf<SecBuffer>();
            for (int i = 0; i < cBuffers; ++i)
            {
                buffers[i] = Marshal.PtrToStructure<SecBuffer>(pBuffers + (i * size));
            }
            return buffers;
        }
    }

    enum SecWinNtAuthIdentityFlags
    {
        Ansi = 0x1,
        Unicode = 0x2,
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    class SEC_WINNT_AUTH_IDENTITY_EX
    {
        const int SEC_WINNT_AUTH_IDENTITY_VERSION = 0x200;

        public int Version;
        public int Length;
        [MarshalAs(UnmanagedType.LPWStr)]
        public string User;
        public int UserLength;
        [MarshalAs(UnmanagedType.LPWStr)]
        public string Domain;
        public int DomainLength;
        [MarshalAs(UnmanagedType.LPWStr)]
        public string Password;
        public int PasswordLength;
        public SecWinNtAuthIdentityFlags Flags;
        [MarshalAs(UnmanagedType.LPWStr)]
        public string PackageList;
        public int PackageListLength;

        public SEC_WINNT_AUTH_IDENTITY_EX(string user, string domain, string password)
        {
            Version = SEC_WINNT_AUTH_IDENTITY_VERSION;
            Length = Marshal.SizeOf(this);
            User = user;
            UserLength = user?.Length ?? 0;
            Domain = domain;
            DomainLength = domain?.Length ?? 0;
            Password = password;
            PasswordLength = password?.Length ?? 0;
            Flags = SecWinNtAuthIdentityFlags.Unicode;
        }
    }

    [StructLayout(LayoutKind.Sequential)]
    class SecHandle
    {
        public IntPtr dwLower;
        public IntPtr dwUpper;
    }

    enum SecDataRep
    {
        Native = 0x00000010,
        Network = 0x00000000
    }

    enum SecPkgCredFlag
    {
        INBOUND = 0x00000001,
        OUTBOUND = 0x00000002,
        BOTH = 0x00000003,
        DEFAULT = 0x00000004,
    }

    [Flags]
    enum SecContextReq
    {
        DELEGATE = 0x00000001,
        MUTUAL_AUTH = 0x00000002,
        REPLAY_DETECT = 0x00000004,
        SEQUENCE_DETECT = 0x00000008,
        CONFIDENTIALITY = 0x00000010,
        USE_SESSION_KEY = 0x00000020,
        PROMPT_FOR_CREDS = 0x00000040,
        USE_SUPPLIED_CREDS = 0x00000080,
        ALLOCATE_MEMORY = 0x00000100,
        USE_DCE_STYLE = 0x00000200,
        DATAGRAM = 0x00000400,
        CONNECTION = 0x00000800,
        CALL_LEVEL = 0x00001000,
        FRAGMENT_SUPPLIED = 0x00002000,
        EXTENDED_ERROR = 0x00004000,
        STREAM = 0x00008000,
        INTEGRITY = 0x00010000,
        IDENTIFY = 0x00020000,
        NULL_SESSION = 0x00040000,
        MANUAL_CRED_VALIDATION = 0x00080000,
        RESERVED1 = 0x00100000,
        FRAGMENT_TO_FIT = 0x00200000,
        FORWARD_CREDENTIALS = 0x00400000,
        NO_INTEGRITY = 0x00800000,
        USE_HTTP_STYLE = 0x01000000,
        UNVERIFIED_TARGET_NAME = 0x20000000,
        CONFIDENTIALITY_ONLY = 0x40000000,
    }

    enum SecStatusCode
    {
        Success = 0,
        ContinueNeeded =  0x00090312,
        CompleteNeeded = 0x00090313,
        CompleteAndContinue = 0x00090314,
    }

    [Flags]
    public enum CLSCTX : uint
    {
        INPROC_SERVER = 0x1,
        INPROC_HANDLER = 0x2,
        LOCAL_SERVER = 0x4,
        INPROC_SERVER16 = 0x8,
        REMOTE_SERVER = 0x10,
        INPROC_HANDLER16 = 0x20,
        RESERVED1 = 0x40,
        RESERVED2 = 0x80,
        RESERVED3 = 0x100,
        RESERVED4 = 0x200,
        NO_CODE_DOWNLOAD = 0x400,
        RESERVED5 = 0x800,
        NO_CUSTOM_MARSHAL = 0x1000,
        ENABLE_CODE_DOWNLOAD = 0x2000,
        NO_FAILURE_LOG = 0x4000,
        DISABLE_AAA = 0x8000,
        ENABLE_AAA = 0x10000,
        FROM_DEFAULT_CONTEXT = 0x20000,
        ACTIVATE_32_BIT_SERVER = 0x40000,
        ACTIVATE_64_BIT_SERVER = 0x80000,
        ENABLE_CLOAKING = 0x100000,
        APPCONTAINER = 0x400000,
        ACTIVATE_AAA_AS_IU = 0x800000,
        PS_DLL = 0x80000000,
        SERVER = INPROC_SERVER | LOCAL_SERVER | REMOTE_SERVER,
        ALL = INPROC_SERVER | INPROC_HANDLER | LOCAL_SERVER | REMOTE_SERVER
    }

    static class Win32NativeMethods
    {
        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode AcquireCredentialsHandle(
            string pszPrincipal,
            string pszPackage,
            SecPkgCredFlag fCredentialUse,
            IntPtr pvLogonId,
            SEC_WINNT_AUTH_IDENTITY_EX pAuthData,
            IntPtr pGetKeyFn,
            IntPtr pvGetKeyArgument,
            [Out] SecHandle phCredential,
            [Out] LargeInteger ptsExpiry
        );

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode FreeCredentialsHandle([In, Out] SecHandle phCredential);

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode InitializeSecurityContext(
            [In] CredentialsHandle phCredential,
            [In] SecHandle phContext,
            string pszTargetName,
            SecContextReq fContextReq,
            int Reserved1,
            SecDataRep TargetDataRep,
            SecBufferDesc pInput,
            int Reserved2,
            [Out] SecHandle phNewContext,
            [In, Out] SecBufferDesc pOutput,
            out int pfContextAttr,
            [Out] LargeInteger ptsExpiry
        );

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode CompleteAuthToken(SecHandle phContext,
            SecBufferDesc pToken
        );

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode DeleteSecurityContext(
            SecHandle phContext
        );

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode AcceptSecurityContext(
            [In] CredentialsHandle phCredential,
            [In] SecHandle phContext,
            SecBufferDesc pInput,
            int fContextReq,
            SecDataRep TargetDataRep,
            [In, Out] SecHandle phNewContext,
            [In, Out] SecBufferDesc pOutput,
            out int pfContextAttr,
            [Out] LargeInteger ptsExpiry
        );

        [DllImport("Secur32.dll", CharSet = CharSet.Unicode)]
        public static extern SecStatusCode QuerySecurityContextToken(SecHandle phContext, out SafeKernelObjectHandle Token);

        [DllImport("ole32.dll")]
        public static extern int CoCreateInstance(ref Guid rclsid, IntPtr pUnkOuter, CLSCTX dwClsContext, ref Guid riid, [MarshalAs(UnmanagedType.IUnknown)] out object ppv);

        public static SecStatusCode CheckResult(this SecStatusCode result)
        {
            if (result < 0)
            {
                throw new Win32Exception((int)result);
            }
            return result;
        }

        public static string BuildHexDump(byte[] ba)
        {
            StringBuilder builder = new StringBuilder();
            for (int i = 0; i < ba.Length; i += 16)
            {
                int line_length = Math.Min(16, ba.Length - i);
                int j = 0;
                for (; j < line_length; ++j)
                {
                    builder.AppendFormat("{0:X02} ", ba[i + j]);
                }
                for (; j < 16; ++j)
                {
                    builder.Append("   ");
                }
                builder.Append(" - ");
                for (j = 0; j < line_length; ++j)
                {
                    byte b = ba[i + j];
                    char c = b >= 32 && b < 127 ? (char)b : '.';
                    builder.Append(c);
                }
                builder.AppendLine();
            }
            return builder.ToString();
        }
    }
}
