/*
  decode_icq.c

  ICQ (note - ICQ2000 is actually AIM).
  
  Copyright (c) 2000 Dug Song <dugsong@monkey.org>
 
  $Id: decode_icq.c,v 1.6 2000/11/29 03:32:14 dugsong Exp $
*/

#include "config.h"

#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include "buf.h"
#include "decode.h"

#define ICQ2_UIN_OFFSET		6
#define ICQ2_CMD_OFFSET		2
#define ICQ2_PASS_OFFSET	16

#define ICQ5_UIN_OFFSET		6
#define ICQ5_CMD_OFFSET		14
#define ICQ5_CKSUM_OFFSET	20
#define ICQ5_PASS_OFFSET	34

const u_char icq5_table [] = {
	0x59, 0x60, 0x37, 0x6B, 0x65, 0x62, 0x46, 0x48, 0x53, 0x61, 0x4C,
	0x59, 0x60, 0x57, 0x5B, 0x3D, 0x5E, 0x34, 0x6D, 0x36, 0x50, 0x3F,
	0x6F, 0x67, 0x53, 0x61, 0x4C, 0x59, 0x40, 0x47, 0x63, 0x39, 0x50,
	0x5F, 0x5F, 0x3F, 0x6F, 0x47, 0x43, 0x69, 0x48, 0x33, 0x31, 0x64,
	0x35, 0x5A, 0x4A, 0x42, 0x56, 0x40, 0x67, 0x53, 0x41, 0x07, 0x6C,
	0x49, 0x58, 0x3B, 0x4D, 0x46, 0x68, 0x43, 0x69, 0x48, 0x33, 0x31,
	0x44, 0x65, 0x62, 0x46, 0x48, 0x53, 0x41, 0x07, 0x6C, 0x69, 0x48,
	0x33, 0x51, 0x54, 0x5D, 0x4E, 0x6C, 0x49, 0x38, 0x4B, 0x55, 0x4A,
	0x62, 0x46, 0x48, 0x33, 0x51, 0x34, 0x6D, 0x36, 0x50, 0x5F, 0x5F,
	0x5F, 0x3F, 0x6F, 0x47, 0x63, 0x59, 0x40, 0x67, 0x33, 0x31, 0x64,
	0x35, 0x5A, 0x6A, 0x52, 0x6E, 0x3C, 0x51, 0x34, 0x6D, 0x36, 0x50,
	0x5F, 0x5F, 0x3F, 0x4F, 0x37, 0x4B, 0x35, 0x5A, 0x4A, 0x62, 0x66,
	0x58, 0x3B, 0x4D, 0x66, 0x58, 0x5B, 0x5D, 0x4E, 0x6C, 0x49, 0x58,
	0x3B, 0x4D, 0x66, 0x58, 0x3B, 0x4D, 0x46, 0x48, 0x53, 0x61, 0x4C,
	0x59, 0x40, 0x67, 0x33, 0x31, 0x64, 0x55, 0x6A, 0x32, 0x3E, 0x44,
	0x45, 0x52, 0x6E, 0x3C, 0x31, 0x64, 0x55, 0x6A, 0x52, 0x4E, 0x6C,
	0x69, 0x48, 0x53, 0x61, 0x4C, 0x39, 0x30, 0x6F, 0x47, 0x63, 0x59,
	0x60, 0x57, 0x5B, 0x3D, 0x3E, 0x64, 0x35, 0x3A, 0x3A, 0x5A, 0x6A,
	0x52, 0x4E, 0x6C, 0x69, 0x48, 0x53, 0x61, 0x6C, 0x49, 0x58, 0x3B,
	0x4D, 0x46, 0x68, 0x63, 0x39, 0x50, 0x5F, 0x5F, 0x3F, 0x6F, 0x67,
	0x53, 0x41, 0x25, 0x41, 0x3C, 0x51, 0x54, 0x3D, 0x5E, 0x54, 0x5D,
	0x4E, 0x4C, 0x39, 0x50, 0x5F, 0x5F, 0x5F, 0x3F, 0x6F, 0x47, 0x43,
	0x69, 0x48, 0x33, 0x51, 0x54, 0x5D, 0x6E, 0x3C, 0x31, 0x64, 0x35,
	0x5A, 0x00, 0x00
};

int
decode_icq(u_char *buf, int len, u_char *obuf, int olen)
{
	struct buf inbuf, outbuf;
	u_short version, cmd;
	u_int32_t uin;
	u_char *p;

	buf_init(&inbuf, buf, len);
	buf_init(&outbuf, obuf, olen);

	if (buf_get(&inbuf, &version, sizeof(version)) != sizeof(version))
		return (0);

	version = pletohs(&version);
	
	switch (version) {
	case 2:
		if (buf_seek(&inbuf, ICQ2_CMD_OFFSET, SEEK_SET) < 0)
			return (0);

		if (buf_get(&inbuf, &cmd, sizeof(cmd)) != sizeof(cmd))
			return (0);

		if ((cmd = pletohs(&cmd)) != 1000)
			return (0);

		if (buf_seek(&inbuf, ICQ2_UIN_OFFSET, SEEK_SET) < 0)
			return (0);

		if (buf_get(&inbuf, &uin, sizeof(uin)) != sizeof(uin))
			return (0);
		
		uin = pletohl(&uin);

		if (buf_seek(&inbuf, ICQ2_PASS_OFFSET, SEEK_SET) < 0)
			return (0);
		break;
		
	case 5:
	{
		u_int32_t a1, a2, a3, a4, a5, c, key, i, k;
		
		if (buf_seek(&inbuf, ICQ5_CKSUM_OFFSET, SEEK_SET) < 0)
			return (0);
		
		if (buf_get(&inbuf, &c, sizeof(c)) != sizeof(c))
			return (0);
		
		c = pletohl(&c);
		
		a1 = c & 0x0001f000; a1 = a1 >> 0x0c;
		a2 = c & 0x07c007c0; a2 = a2 >> 0x01;
		a3 = c & 0x003e0001; a3 = a3 << 0x0a;
		a4 = c & 0xf8000000; a4 = a4 >> 0x10;
		a5 = c & 0x0000083e; a5 = a5 << 0x0f;
		
		key = len * 0x68656C6C;
		key += a1 + a2 + a3 + a4 + a5;

		p = inbuf.base;
		
		for (i = 0x0a; i < inbuf.end + 3; i += 4) {
			k = key + icq5_table[i & 0xff];
			if (i != 0x16) {
				p[i] ^= (u_char)(k & 0xff);
				p[i + 1] ^= (u_char)((k & 0xff00) >> 8);
			}
			if (i != 0x12) {
				p[i + 2] ^= (u_char)((k & 0xff0000) >> 16);
				p[i + 3] ^= (u_char)((k & 0xff000000) >> 24);
			}
		}
		if (buf_seek(&inbuf, ICQ5_CMD_OFFSET, SEEK_SET) < 0)
			return (0);

		if (buf_get(&inbuf, &cmd, sizeof(cmd)) != sizeof(cmd))
			return (0);

		if ((cmd = pletohs(&cmd)) != 1000)
			return (0);

		if (buf_seek(&inbuf, ICQ5_UIN_OFFSET, SEEK_SET) < 0)
			return (0);

		if (buf_get(&inbuf, &uin, sizeof(uin)) != sizeof(uin))
			return (0);
		
		uin = pletohl(&uin);

		if (buf_seek(&inbuf, ICQ5_PASS_OFFSET, SEEK_SET) < 0)
			return (0);
	}
	break;
	
	default:
		return (0);
	}
	buf_putf(&outbuf, "%d\n%.*s\n", uin, buf_len(&inbuf), buf_ptr(&inbuf));

	buf_end(&outbuf);
	
	return (buf_len(&outbuf));
}

