<?php
/****************************************************************************
 *  mmslib - a PHP library for encoding and decoding MMS
 *
 *  Copyright (C) 2003, 2004  Stefan Hellkvist
 *  additions Copyright (C) 2005, 2006  Collin Mulliner
 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *****************************************************************************/

/* If DEBUG is 1 the decoder will do printouts on stdout */
define( "DEBUG", 		  0 );

/* Field name constants */
define( "BCC", 			0x01 );
define( "CC", 			0x02 );
define( "CONTENT_LOCATION", 	0x03 );
define( "CONTENT_TYPE", 	0x04 );
define( "DATE", 		0x05 );
define( "DELIVERY_REPORT", 	0x06 );
define( "DELIVERY_TIME", 	0x07 );
define( "EXPIRY", 		0x08 );
define( "FROM", 		0x09 );
define( "MESSAGE_CLASS",	0x0A );
define( "MESSAGE_ID", 		0x0B );
define( "MESSAGE_TYPE", 	0x0C );
define( "MMS_VERSION", 		0x0D );
define( "MESSAGE_SIZE", 	0x0E );
define( "PRIORITY", 		0x0F );
define( "READ_REPLY", 		0x10 );
define( "REPORT_ALLOWED", 	0x11 );
define( "RESPONSE_STATUS", 	0x12 );
define( "RESPONSE_TEXT", 	0x13 );
define( "SENDER_VISIBILITY", 	0x14 );
define( "STATUS", 		0x15 );
define( "SUBJECT", 		0x16 );
define( "TO", 			0x17 );
define( "TRANSACTION_ID", 	0x18 );

/* Some constants for Content type (definitely not a complete list) */
define( "IMAGE_GIF", 		0x1D );
define( "IMAGE_JPEG", 		0x1E );
define( "IMAGE_PNG", 		0x20 );
define( "IMAGE_WBMP", 		0x21 );
define( "TEXT_PLAIN", 		0x03 );
define( "MULTIPART_MIXED", 	0x23 );
define( "MULTIPART_RELATED",  	0x33 );

/* Message types */
define("M_SEND_REQ", 0x80);
define("M_RECEIVE_CONF", 0x84);


/* Mapping from well-known content-type to string */
$content_types = array( "*/*", "text/*", "text/html", "text/plain",
	 	 	"text/x-hdml", "text/x-ttml", "text/x-vCalendar",
			"text/x-vCard", "text/vnd.wap.wml",
			"text/vnd.wap.wmlscript", "text/vnd.wap.wta-event",
			"multipart/*", "multipart/mixed",
			"multipart/form-data", "multipart/byterantes",
			"multipart/alternative", "application/*",
			"application/java-vm",
			"application/x-www-form-urlencoded",
			"application/x-hdmlc", "application/vnd.wap.wmlc",
			"application/vnd.wap.wmlscriptc",
			"application/vnd.wap.wta-eventc",
			"application/vnd.wap.uaprof",
			"application/vnd.wap.wtls-ca-certificate",
			"application/vnd.wap.wtls-user-certificate",
			"application/x-x509-ca-cert",
			"application/x-x509-user-cert",
			"image/*", "image/gif", "image/jpeg", "image/tiff",
			"image/png", "image/vnd.wap.wbmp", 
			"application/vnd.wap.multipart.*", 
			"application/vnd.wap.multipart.mixed", 
			"application/vnd.wap.multipart.form-data", 
			"application/vnd.wap.multipart.byteranges", 
			"application/vnd.wap.multipart.alternative", 
			"application/xml", "text/xml", 
			"application/vnd.wap.wbxml", 
			"application/x-x968-cross-cert", 
			"application/x-x968-ca-cert", 
			"application/x-x968-user-cert", 
			"text/vnd.wap.si", 
			"application/vnd.wap.sic", 
			"text/vnd.wap.sl", 
			"application/vnd.wap.slc", 
			"text/vnd.wap.co", 
			"application/vnd.wap.coc", 
			"application/vnd.wap.multipart.related",
			"application/vnd.wap.sia", 
			"text/vnd.wap.connectivity-xml", 
			"application/vnd.wap.connectivity-wbxml", 
			"application/pkcs7-mime", 
			"application/vnd.wap.hashed-certificate", 
			"application/vnd.wap.signed-certificate", 
			"application/vnd.wap.cert-response",
			"application/xhtml+xml",
			"application/wml+xml",
			"text/css", 
			"application/vnd.wap.mms-message",
			"application/vnd.wap.rollover-certificate", 
			"application/vnd.wap.locc+wbxml", 
			"application/vnd.wap.loc+xml", 
			"application/vnd.syncml.dm+wbxml", 
			"application/vnd.syncml.dm+xml", 
			"application/vnd.syncml.notification", 
			"application/vnd.wap.xhtml+xml", 
			"application/vnd.wv.csp.cir", 
			"application/vnd.oma.dd+xml", 
			"application/vnd.oma.drm.message", 
			"application/vnd.oma.drm.content", 
			"application/vnd.oma.drm.rights+xml",
			"application/vnd.oma.drm.rights+wbxml" );


/* Mapping from constant to string for field names */
$field_names = array( 	1 => "Bcc", "Cc", "Content-Location",
			"Content-Type", "Date", "Delivery-Report",
			"Delivery-Time", "Expiry", "From",
			"Message-Class", "Message-ID", "Message-Type",
			"MMS-Version", "Message-Size", "Priority",
			"Read-Reply", "Report-Allowed", "Response-Status",
			"Response-Text", "Sender-Visibility", "Status",
			"Subject", "To", "Transaction-ID" );



function fieldNameToString( $fieldName )
{
	global $field_names;
	return $field_names[$fieldName];
}

function messageTypeToString( $messageType )
{
	switch ( $messageType )
	{
		case 128: return "m-send-req"; 
		case 129: return "m-send-conf";
		case 130: return "m-notification-ind";
		case 131: return "m-notifyresp-ind";
		case 132: return "m-retrieve-conf";
		case 133: return "m-acknowledge-ind";
		case 134: return "m-delivery-ind";
		default: return "Unknown message type";
	}
}

function statusToString( $status )
{
	switch ( $status )
	{
		case 128: return "Expired";
		case 129: return "Retrieved";
		case 130: return "Rejected";
		case 131: return "Deferred";
		case 132: return "Unrecognized";
		default: return "Unknown status";
	}
}

function mmsVersionToString( $version )
{
	$major = ($version & 0x70) >> 4;
	$minor = ($version & 0x0f);
	return "" . $major . "." . $minor;
}

function dateToString( $date )
{
	return date( "Y-m-d H:i:s", $date ); 
}

function messageClassToString( $messageClass )
{
	switch ( $messageClass )
	{
		case 128: return "Personal";
		case 129: return "Advertisment";
		case 130: return "Informational";
		case 131: return "Auto";
		default: return "Unknown message class";
	}
}

function priorityToString( $priority )
{
	switch ( $priority )
	{
		case 128: return "Low";
		case 129: return "Normal";
		case 130: return "High";
		default: "Unknown priority";
	}
}		

function senderVisibilityToString( $vis )
{
	switch ( $vis )
	{
		case 128: return "Hide";
		case 129: return "Show";
		default: "Unknown sender visibility";
	}
}		

function deliveryReportToString( $dr )
{
	switch ( $dr )
	{
		case 128: return "Yes";
		case 129: return "No";
		default: "Unknown delivery report";
	}
}		


function readReplyToString( $readReply )
{
	return deliveryReportToString( $readReply );
}

function contentTypeToString( $contentType )
{
	global $content_types;
	if ( is_string( $contentType ) )
		return $contentType;
	return $content_types[$contentType];
}

/* encoding functions */
function encodeUintvar( $data, &$uintVar )
{
	$i = 0;
	$reversed[$i] = $data & 0x7f; // The lowest
	$data = $data >> 7;

	$i++;
	while ($data > 0){
		$reversed[$i] = 0x80 | ($data & 0x7f);
		$i++;
		$data = $data >> 7;
	}

	// Reverse it because it is in reverse order
	for ( $j = 0; $j < $i; $j++ ) {
		$uintVar[$j] = $reversed[$i - $j - 1];
	}
}

function encodeLongInteger( $data, &$longInt )
{
	$len = 0;

	while ($data > 0) {
		$tmp[$len] = $data & 0xff;
		$data = $data >> 8;
		$len++;
	}
	
	$longInt[0] = $len; /* Set short-length */

	/* tmp is the reverse of what we want so we reverse it */
	for ( $i = 0; $i < $len; $i++ ) {
		$longInt[$i+1] = $tmp[ $len - $i - 1 ];
	}
}

function hex_encode($text)    
{
	$encoded = strtoupper(bin2hex($text));
	return $encoded;
}

function hex2bin($hexdata)
{
	for ($i=0; $i<strlen($hexdata); $i+=2) {
		$bindata.=chr(hexdec(substr($hexdata,$i,2)));
	}					   
	return $bindata;
}

function encodeLengthval($len, &$ret)
{
	if ($len <= 30) {
		$ret[0] = $len;
		return sizeof($ret);
	}
	else {
		encodeUintvar($len, $out);
		$ret[0] = 31;
		for ($i = 0; $i < sizeof($out); $i++) {
			$ret[$i+1] = $out[$i];
		}
		return sizeof($ret);
	}
}

/*
 * Part class
 * 
 * The Part class is just a container for various attachments of different content types.
 * The data itself is not stored though, it merely contains a reference to the file      
 */

class Part
{
	var $contentType;
	var $dataLen;
	var $fname;
	var $data;
	var $fileID;
	var $sfname;

	function Part( $contentType = -1, $fileName = -1 , $shownfilename = -1, $fileID)
	{
		if ( $fileName != -1 && $contentType != -1 )
		{
			$this->contentType = $contentType;
			$this->dataLen = filesize( $fileName );
			$this->fname = $fileName;
			$this->fileID = $fileID;
			if ($shownfilename == -1) {
				$this->sfname = $fileName;
			}
			else {
				$this->sfname = $shownfilename;
			}
		}
	}

	function writeToFp( $fp )
	{
	  if ( isset( $this->fname ) && !isset( $this->data ) )
		{
			$cfp = fopen( $this->fname, "rb" );
			fwrite( $fp, fread( $cfp, $this->dataLen ), $this->dataLen );
			fclose( $cfp );
		}
		else
		if ( !isset( $this->fname ) && isset( $this->data ) )
		{
			for ( $i = 0; $i < $this->dataLen; $i++ )
				fwrite( $fp, chr( $this->data[$i] ), 1 );

		}
	}

	function getBytes()
	{
		$cfp = fopen( $this->fname, "rb" );
		$bytes = fread( $cfp, $this->dataLen );
		fclose( $cfp );
		return($bytes);
	}

	function writeToFile( $fileName )
	{
		$fp = fopen( $fileName, "wb" );
		for ( $i = 0; $i < $this->dataLen; $i++ )
			fwrite( $fp, chr( $this->data[$i] ), 1 );
		fclose( $fp );
	}

	function writeTopage()
	{

		for ( $i = 0; $i < $this->dataLen; $i++ )
		$vysl .= chr( $this->data[$i] );
		return $vysl;

	}
}

/*
 * MMSDecoder class
 * 
 * The MMSDecoder class decodes binary MMS chunks so that you can extract its parts
 * 
 * Limitations: The parsing of Content-type for parts is not complete so you won't get
 *              the filename for instance.  
 */

class MMSDecoder
{
	/* Parser variables */
	var $data;
	var $curp;

	/* Header variables as found when parsing */
	var $messageType;
	var $transactionId;
	var $mmsVersion;
	var $date;
	var $from;
	var $to;
	var $cc;
	var $bcc;
	var $subject;
	var $messageClass;
	var $priority;
	var $senderVisibility;
	var $deliveryReport;
	var $readReply;
	var $contentType;
	var $bodyStartsAt;
	var $expiryDate;
	var $expiryDeltaSeconds;
	var $status;

	/* Body variables as found when parsing */
	var $nparts;
	var $parts;
	var $druhy;
	var $jmena;

	function MMSDecoder( $data )
	{
		$datalen = strlen( $data );
		for ( $i = 0; $i < $datalen; $i++ )
			$this->data[$i] = ord( $data{$i} );
		$this->curp = 0;
	}


	function confirm($TRANSACTIONID,$cislo) 
	{
		$pos = 0;

		$confirm[$pos++] = 0x8C; // message-type
		$confirm[$pos++] = 129;  // m-send-conf
		$confirm[$pos++] = 0x98; // transaction-id

		for ($i = 0; $i < strlen($TRANSACTIONID); $i++)
			$confirm[$i+$pos] = ord(substr($TRANSACTIONID, $i, 1));

		$pos += $i;

		$confirm[$pos++] = 0x00; // end of string
		$confirm[$pos++] = 0x8D; // version
		$confirm[$pos++] = 0x90; // 1.0
		$confirm[$pos++] = 0x92; // response-status
		$confirm[$pos]   = $cislo;  // OK=128,
		//Ok = <Octet 128> Error-unspecified = <Octet 129> Error- service-denied = <Octet 130> Error-message-format-corrupt = <Octet 131> Error-sending-address-unresolved = <Octet 132> Error-message-not-found = <Octet 133> Error-network-problem = <Octet 134> Error- content-not-accepted = <Octet 135> Error-unsupported-message = <Octet 136>


		// respond with the m-send-conf
		foreach ($confirm as $byte)
			$sd .= chr($byte);
			//echo $byte;
			return $sd;
	}


	function parse()
	{
		while ( $this->parseHeader() );

		$this->bodyStartsAt = $this->curp;
	
		if ( $this->contentType == MULTIPART_MIXED ||
		     $this->contentType == MULTIPART_RELATED )
		{
			$this->parseBody();
		}
	}

	function parseHeader()
	{
		$res = $this->parseMMSHeader();
		if ( !$res )
			$res = $this->parseApplicationHeader();
		return $res;
	}

	function parseApplicationHeader()
	{
		$res = $this->parseToken( $token );
		if ( $res )
			$res = $this->parseTextString( $appspec );
		if ( DEBUG && $res )
			print( $token . ": " . $appspec );
		return $res;
	}

	function parseToken( &$token )
	{
		if ( $this->data[$this->curp] <= 31 ||
		     $this->isSeparator( $this->data[$this->curp] ) )
			return 0;
		while ( $this->data[$this->curp] > 31 && 
		        !$this->isSeparator( $this->data[$this->curp] ) )
		{
			$token .= chr( $this->data[$this->curp] );
			$this->curp++;
		}
		return 1;
	}

	function isSeparator( $ch )
	{
		return $ch == 40 || $ch == 41 || $ch == 60 || $ch == 62 || 
		       $ch == 64 || $ch == 44 || $ch == 58 || $ch == 59 || 
		       $ch == 92 || $ch == 47 || $ch == 123 || $ch == 125 || 
                       $ch == 91 || $ch == 93 || $ch == 63 || $ch == 61 || 
                       $ch == 32 || $ch == 11;
	}

	function parseMMSHeader()
	{
		if (!$this->parseShortInteger( $mmsFieldName ) )
		{
			return 0;
		}
		
		switch ( $mmsFieldName )
		{
			case BCC: 
				$this->parseBcc(); break;
			case CC:  
				$this->parseCc(); break;
			case CONTENT_LOCATION:
				$this->parseContentLocation(); break;
			case CONTENT_TYPE:
				$this->parseContentType( $this->contentType );
				break;
			case DATE: 
				$this->parseDate( $this->date ); break;
			case DELIVERY_REPORT: 
				$this->parseDeliveryReport(); break;
			case DELIVERY_TIME: 
				$this->parseDeliveryTime(); break;
			case EXPIRY: 
				$this->parseExpiry(); break;
			case FROM: 
				$this->parseFrom(); break;
			case MESSAGE_CLASS: 
				$this->parseMessageClass(); break;
			case MESSAGE_ID: 
				$this->parseMessageId(); break;
			case MESSAGE_TYPE: 
				$this->parseMessageType(); break;
			case MMS_VERSION: 
				$this->parseMmsVersion(); break;
			case MESSAGE_SIZE: 
				$this->parseMessageSize(); break;
			case PRIORITY: 
				$this->parsePriority(); break;
			case READ_REPLY: 
				$this->parseReadReply(); break;
			case REPORT_ALLOWED: 
				$this->parseReportAllowed(); break;
			case RESPONSE_STATUS:
				$this->parseResponseStatus(); break;
			case SENDER_VISIBILITY: 
				$this->parseSenderVisibility(); break;
			case STATUS: 
				$this->parseStatus(); break;
			case SUBJECT: 
				$this->parseSubject(); break;
			case TO: 
				$this->parseTo(); break;
			case TRANSACTION_ID: 
				$this->parseTransactionId(); break;
			default: 
				if ( DEBUG )
				{
					printf( "Unknown message field (" );
					printf( $mmsFieldName . ")\n");
				}
				break;
		}
		return 1;
	}

	function parseStatus()
	{
		$this->status = $this->data[$this->curp++];
		if ( DEBUG )
		{
			print( "X-Mms-Status: " . 
			  statusToString($this->status) . 
			  "\n" );		
		}
	}

	function parseShortInteger( &$shortInt )
	{
		if ( !($this->data[$this->curp] & 0x80) )
			return 0;
		$shortInt = $this->data[$this->curp] & 0x7f;
		$this->curp++;
		return 1;
	}

	function parseMessageType()
	{
		if ( !($this->data[$this->curp] & 0x80) )
			return 0;
		$this->messageType = $this->data[$this->curp];
		$this->curp++;

		if ( DEBUG )
		{
			print( "X-Mms-Message-Type: " . 
				messageTypeToString( $this->messageType ) .
				"\n" );
		}
		return 1;		
	}


	function parseTransactionId()
	{	
		$this->parseTextString( $this->transactionId );
		if ( DEBUG )
		{
			print( "X-Mms-Transaction-ID: " . 
				$this->transactionId . "\n" );
		}
	}

	function parseExpiry()
	{
		$this->parseValueLength($length);
		switch ( $this->data[$this->curp] )
		{
			case 128: 
				$this->curp++; 
				$this->parseDate( $this->expiryDate ); 
				break;
			case 129:
				$this->curp++; 
				$this->parseDeltaSeconds( $this->expiryDeltaSeconds );
				break;
			default:
		}
	}

	function parseDeltaSeconds( &$deltaSecs )
	{
		$this->parseDate( $deltaSecs );
	}

	function parseTextString( &$textString )
	{
		if ( $this->data[$this->curp] == 127 ) /* Remove quote */
			$this->curp++;
		while ( $this->data[$this->curp] )
		{
			$textString .= chr( $this->data[$this->curp] );
			$this->curp++;
		}
		$this->curp++;
		return 1;
	}

	function parseMmsVersion()
	{
		$this->parseShortInteger( $this->mmsVersion );
		if ( DEBUG )
		{
			print( "X-Mms-MMS-Version: " . 
				mmsVersionToString( $this->mmsVersion ) . 
				"\n" );
		}
	}


	function parseDate( &$date )
	{
		$this->parseLongInteger( $date );
		if ( DEBUG )
		{
			print( "Date: " . dateToString( $date ) . 
				"\n" );
		}
	}


	function parseLongInteger( &$longInt )
	{
		if ( !$this->parseShortLength( $length ) )
			return 0;
		return $this->parseMultiOctetInteger( $longInt, $length );
	}

	function parseShortLength( &$shortLength )
	{
		$shortLength = $this->data[$this->curp];
		if ( $shortLength > 30 )
			return 0;
 		$this->curp++;
		return 1;
	}

	function parseMultiOctetInteger( &$moint, $noctets )
	{
		$moint = 0;
		for ( $i = 0; $i < $noctets; $i++ )
		{
			$moint = $moint << 8;
			$moint |= $this->data[$this->curp];
			$this->curp++;
		}
		return 1;
	}

	function parseFrom()
	{
		$this->parseValueLength( $length );
		if ( $this->data[$this->curp] == 128 ) /* Address present? */
		{
			$this->curp++;
			$this->parseEncodedString( $this->from );
		}
		else 
		{
			$this->from = "Anonymous";
			$this->curp++;
		}
		
		if ( DEBUG )
			print( "From: " . $this->from . "\n" );
	}

	function parseEncodedString( &$encstring )
	{
		$isencoded = $this->parseValueLength( $length );
		if ( $isencoded )
		{
			$this->curp++;
			//die( "Encoded String not implemented fully" );
		}
		$this->parseTextString( $encstring );
	}

	function parseValueLength( &$length )
	{
		$lengthFound = $this->parseShortLength( $length );
		if ( !$lengthFound )
		{
			if ( $this->data[$this->curp] == 31 )
			{
				$this->curp++;
				$this->parseUintvar( $length );
				return 1;
			}	
		}
		return $lengthFound;
	}

	function parseUintvar( &$uintvar )
	{
		$uintvar = 0;
		while ( $this->data[$this->curp] & 0x80 )
		{
			$uintvar = $uintvar << 7;
			$uintvar |= $this->data[$this->curp] & 0x7f;
			$this->curp++;
		}
		$uintvar = $uintvar << 7;
		$uintvar |= $this->data[$this->curp] & 0x7f;
		$this->curp++;
	}

	function parseTo()
	{
		$this->parseEncodedString( $this->to );
		if ( DEBUG )
			print( "To: " . $this->to . "\n" );
	}

	function parseBcc()
	{
		$this->parseEncodedString( $this->bcc );
		if ( DEBUG )
			print( "Bcc: " . $this->bcc . "\n" );
	}

	function parseCc()
	{
		$this->parseEncodedString( $this->cc );
		if ( DEBUG )
			print( "Cc: " . $this->cc . "\n" );
	}
		

	function parseSubject()
	{
		$this->parseEncodedString( $this->subject );
		if ( DEBUG )
			print( "Subject: " . $this->subject . "\n" );
	}

	function parseMessageClass()
	{
		if ( $this->data[$this->curp] < 128 ||
			$this->data[$this->curp] > 131 )
		{
			die( "parseMessageClass not fully implemented" );
		}
		$this->messageClass = $this->data[$this->curp++];

		if ( DEBUG )
		{
			print( "X-Mms-Message-Class: " . 
				messageClassToString($this->messageClass) .
				"\n" );
		}
	}


	function parsePriority()
	{
		$this->priority = $this->data[$this->curp++];
		if ( DEBUG )
		{
			print( "X-Mms-Priority: " . 
				priorityToString( $this->priority ) . 
				"\n" );
		}
	}


	function parseSenderVisibility()
	{
		$this->senderVisibility = $this->data[$this->curp++];
		if ( DEBUG )
		{
			print( "X-Mms-Sender-Visibility: " . 
			  senderVisibilityToString($this->senderVisibility) . 
			  "\n" );		
		}
	}


	function parseDeliveryReport()
	{
		$this->deliveryReport = $this->data[$this->curp++];
		if ( DEBUG  )
		{
			print( "X-Mms-Delivery-Report: " . 
			   deliveryReportToString($this->deliveryReport) . 
			   "\n" );		
		}
	}


	function parseReadReply()
	{
		$this->readReply = $this->data[$this->curp++];
		if ( DEBUG )
		{
			print( "X-Mms-Read-Reply: " . 
				readReplyToString($this->readReply) . 
				"\n" );		
		}
	}	


	function parseContentType( &$contentType)
	{
		$typeFound = $this->parseConstrainedMedia( $contentType );
		if ( !$typeFound )
		{
			$this->parseContentGeneralForm($contentType);	
			$typeFound = 1;
		}
		
		if ( DEBUG )
		{
			printf( "Content-type: " .
				contentTypeToString( $contentType ) .
				"\n" );
		}
		return $typeFound;
	}


	function parseConstrainedMedia( &$contentType )
	{
		return $this->parseConstrainedEncoding( $contentType );
	}

	function parseConstrainedEncoding( &$encoding )
	{
		$res = $this->parseShortInteger( $encoding );
		if ( !$res )
			$res = $this->parseExtensionMedia( $encoding );
		return $res;
	}

	function parseExtensionMedia( &$encoding )
	{
		$ch = $this->data[$this->curp];
		if ( $ch < 32 || $ch == 127)
			return 0;
		$res = $this->parseTextString( $encoding );
		return $res;
	}

	function parseContentGeneralForm( &$encoding )
	{
		$res = $this->parseValueLength( $length );
		$tmp = $this->curp;
		if ( !$res )
			return 0;
		$res = $this->parseMediaType( $encoding );

		/* Jump over everything regardless of parameters */
		$this->curp = $tmp + $length;
		return $res;
	}

	function parseMediaType( &$encoding )
	{
		$res = $this->parseWellKnownMedia( $encoding );
		if ( !$res )
			$res = $this->parseExtensionMedia( $encoding );
		return $res;
	}

	function parseWellKnownMedia( &$encoding )
	{
		return $this->parseInteger( $encoding );
	}

	function parseInteger( &$integer )
	{
		$res = $this->parseShortInteger( $integer );
		if ( !$res )
			$res = $this->parseLongInteger( $integer );
		return $res;
	}

	function parseBody()
	{
		$this->parseUintvar( $this->nparts );
		if ( DEBUG )
		{
			printf( "Multipart with " . $this->nparts .
				" entries\n" );
		}
		for ( $i = 0; $i < $this->nparts; $i++ )
			$this->parsePart( $i );
	}

	function parsePart( $i )
	{
		$part = new Part;
		$this->parseUintvar( $headersLen );
		$this->parseUintvar( $dataLen );
		$part->dataLen = $dataLen;
		$tmp = $this->curp;
		$this->parseContentType( $part->contentType );

		$this->druhy[$i] =  contentTypeToString( $part->contentType ) ;
		$this->jmena[$i] =   $part->fname  ;

		/* Skip headers for now */
		$this->curp = $tmp + $headersLen;
		
		/* Store data */
		for ( $j = 0; $j < $dataLen; $j++)
		{
			$part->data[$j] = $this->data[$this->curp];
			$this->curp++;
		}
		$this->parts[$i] = $part;

	}
}

/*
 * MMSEncoder class
 * 
 * Full MMS enconder, all headers supported!
 */
class MMSEncoder
{
	/* Body variables */
	var $nparts;
	var $parts;

	/* message spec */
	var $subject;	
	var $subject_bin;	
  var $transaction_id;
	var $msg_type;
	var $from;
	var $from_bin;
	var $from_fillin;
	var $to;
	var $to_bin;
	var $cc;
	var $cc_bin;
	var $bcc;
	var $bcc_bin;
	var $date;
	var $start_content;
	var $start_info;
	var $start;
	var $msg_class;
	var $msg_class_string;
	var $msg_priority;
	var $expiry;
	var $expiry_type;
	var $expiry_bin;
	var $delivery_time;
	var $delivery_time_type;
	var $delivery_time_bin;
	var $sender_visibility;
	var $delivery_report;
	var $read_reply;
	var $header_data;
	var $body_data;
	var $url;
	var $url_bin;
	var $msg_size;
	var $msg_size_bin;
	var $message_id;
	var $message_id_bin;
	var $text_header;
	var $simple_enc;
	
	function MMSEncoder($simple = 0)
	{
		$this->nparts = 0;
		$this->simple_enc = $simple;
		
		// defaults
		$this->msg_type = "84";
		$this->start_content = "application/vnd.wap.multipart.related";
		$this->start_info = "<SMIL>";
		$this->start = "application/smil";
	}

	function writeToFile( $fileName, $hexorbin = 1 )
	{
		if ($hexorbin == 1) {
			$header = hex2bin($this->header_data);
			$body = hex2bin($this->body_data);
		}
		else {
			$header = $this->header_data;
			$body = $this->body_data;
		}
	
		$fp = fopen( $fileName, "wb" );
		fwrite($fp, $header );
		fwrite($fp, $body);
		fclose( $fp );
	}
	
	function headerWriteToFile($filename, $hexorbin = 1)
	{
		if ($hexorbin == 1) {
			$header = hex2bin($this->header_data);
		}
		else {
			$header = $this->header_data;
		}
	
		$fp = fopen( $fileName, "wb" );
		fwrite($fp, $header );
		fclose( $fp );
	}

	/* http://host.domain/filename (can have port ...domain:80..) */
	function setUrl($url, $bin = 0)
	{
		if ($bin == 1) {
			$this->url_bin = $url;
		}
		else {
			$this->url = $url;
		}
	}

	/* message size on server (ca.) */
	function setMsgSize($msgsize, $bin = 0)
	{
		if ($bin == 1) {
			$this->msg_size_bin = $msgsize;
		}
		else {
			$this->msg_size = $msgsize;
		}
	}

	function setSubject( $subj, $bin = 0 )
	{
		if ($bin == 1) {
			$this->subject_bin = $subj;
		}
		else {
			$this->subject = $subj;
		}
	}

	function setTransactionId( $transid, $bin = 0 )
	{
		if ($bin == 1) {
			$this->transaction_id_bin = $transid;
		}
		else {
			$this->transaction_id = $transid;
		}
	}

	/*
	 *  80 = m-send.req
	 *  84 = m-retrieve.conf
	 *  82 = notification.ind
	 */
	function setMsgType( $mtype )
	{
		$this->msg_type = $mtype;
	}

  function setTo( $to, $bin =0 )
	{
		if ($bin == 1) {
			$this->to_bin = $to;
		}
		else {
			$this->to = $to;
		}
	}

	function setCc( $cc, $bin = 0 )
	{
		if ($bin == 1) {
			$this->cc_bin = $cc;
		}
		else {
			$this->cc = $cc;
		}
	}
	
	function setBcc( $bcc, $bin = 0 )
	{
		if ($bin == 1) {
			$this->bcc_bin = $bcc;
		}
		else {
			$this->bcc = $bcc;
		}
	}

	/* fillin 80 = value present, 81 = please fillin */
	function setFrom( $from, $bin = 0, $fillin = "80" ) 
	{
		if ($bin == 1) {
			$this->from_bin = $from;
		}
		else {
			$this->from = $from;
			$this->from_fillin = $fillin;
		}
	}

  function setDate( $date, $bin = 0 ) 
	{
		if ($bin == 1) {
			$this->date_bin = $date;
		}
		else {
			$this->date = $date;
		}
	}
	
	// emailadress format, for normal using put "<" before and ">" after string
	function setMessageId($msgid, $bin = 0)
	{
		if ($bin == 1) {
			$this->message_id_bin = $msgid;
		}
		else {
			$this->message_id = $msgid;
		}
	}

	function setStart($content, $info, $start)
	{
		$this->start_content = $content;
		$this->start_info = $info;
		$this->start = $start;
	}

  /* message classes
	 * 80 = personal (default)
	 * 81 = ad
	 * 82 = info
	 * 83 = auto
	 *
	 */
	function setMsgClass($msgclass, $msgclass_type = 0)
	{
		if ($msgclass_type != 0) {
			$this->msg_class_string = $msgclass;
		}
		else {
			$this->msg_class = $msgclass;
		}
	}
	
	/* type 80 = absolute, 81 = relative */
	function setExpiry($expiry, $bin = 0, $expirytype = "80")
	{
		if ($bin == 1) { // fuzzing only
			$this->expiry_bin = $expiry;
		}
		else {
			$this->expiry = $expiry;
			$this->expiry_type = $expirytype;
		}
	}
	
	/* low=80,mid=81,high=82 */
	function setPriority($priority)
	{
		$this->msg_priority = $priority;
	}

	/* 80=yes, 81=no */
	function setSenderVisibility($sendervisibility)
	{
		$this->sender_visibility = $sendervisibility;
	}

	/* 80=yes, 81=no */
	function setDeliveryReport($report)
	{
		$this->delivery_report = $report;
	}
	
	/* type 80 = absolute, 81 = relative */
	function setDeliveryTime($deliverytime, $bin = 0, $deliverytype = "80")
	{
		if ($bin == 1) { // fuzzing only
			$this->delivery_time_bin = $deliverytime;
		}
		else {
			$this->delivery_time = $deliverytime;
			$this->delivery_time_type = $deliverytype;
		}
	}
	
	/* 80=yes, 81=no */
	function setReadReply($readreply)
	{
		$this->read_reply = $readreply;
	}

	/*
	 * used to set a text header
	 */
	function addTextHeader($headertext)
	{
		$this->text_header .= hex_encode($headertext);
		$this->text_header .= "00";
	}

	function buildHeader()
	{
		$header = "8C"; // X-Mms-Message-Type
		$header .= $this->msg_type; 
		
		if ($this->transaction_id_bin) {
			$header .= "98"; // transaction_id
  		$header .= $this->transaction_id_bin;
		}
		else {
    	if ($this->transaction_id) {
  			$header .= "98"; // transaction_id
				if ($this->transaction_id[0] > 127) $header .= "7F";;
  			$header .= hex_encode($this->transaction_id);
  			$header .= "00"; // terminator
			}
		}
		if ((!$this->transaction_id) && (!$this->transaction_id_bin))  {
			print "MMSEncoder: TansactionID-Header not set, this is a mandatory field\n";
		}

		$header .= "8D"; /* X-Mms-Version */
		$header .= "90"; /*    = 1.0      */

		// From
		if ($this->from_bin) {
			$header .= "89"; // from
			$header .= $this->from_bin;
		}
		else {
		  if ($this->from) {
				print "from " . $this->from . " " . $this->from_fillin . "\n";
				$header .= "89"; // from
				if ($this->from_fillin == "80") { // from field is filled
					$len = 1;
					$slen = strlen($this->from) + 2;
					if ($this->from[0] > 127) $slen++;
					encodeLengthval($slen, $slenval);
					if ($this->simple_enc == 0) {
						$len = $len + $slen + sizeof($slenval);
					}
					else if ($this->simple_enc == 1) {
						$len = $len + $slen - 1;
					}
					encodeLengthval($len, $lenval);
					// total length
					for ( $j = 0; $j < sizeof( $lenval ); $j++ ) {
						$header .=  hex_encode(chr($lenval[$j]));
					}
					$header .= $this->from_fillin; // value present/absent token
					if ($this->simple_enc == 0) {
						// length of string
						for ( $j = 0; $j < sizeof( $slenval ); $j++ ) {
							$header .=  hex_encode(chr($slenval[$j]));
						}
						$header .= "83"; // charset us-ascii (0x03 | 0x80)
					}
					if ($this->from[0] > 127) $header .= "7F";
					$header .= hex_encode($this->from);
					$header .= "00";    // terminator
				}
				else { // from field is not field (server please fill in)
					$header .= "01";
					$header .= $this->from_fillin; // value present/absent token
				}
			}
		}
		if ((!$this->from) && (!$this->from_bin)) {
			print "MMSEncoder: From-Header not set, this is a mandatory field\n";
		}

    // To (optional)
		if ($this->to_bin) {
			$header .= "97"; // to
			$header .= $this->to_bin;
		}
		else {
	  	if ($this->to) {
				$header .= "97"; // to
				if ($this->simple_enc == 0) {
					$len = strlen($this->to) + 2;
					encodeLengthval($len, $lenvar);
					for ($i = 0; $i < sizeof($lenvar); $i++) {
						$header .= hex_encode(chr($lenvar[$i]));
					}
					$header .= "83"; // us-ascii
				}
				$header .= hex_encode($this->to);
				$header .= "00";    // terminator
			}
		}
		// Cc (optional)
		if ($this->cc_bin) {
			$header .= "82"; // cc
			$header .= $this->cc_bin;
		}
		else {
	  	if ($this->cc) {
				$header .= "82"; // cc
				if ($this->simple_enc == 0) {
					$len = strlen($this->cc) + 2;
					encodeLengthval($len, $lenvar);
					for ($i = 0; $i < sizeof($lenvar); $i++) {
						$header .= hex_encode(chr($lenvar[$i]));
					}
					$header .= "83"; // us-ascii
				}
				$header .= hex_encode($this->cc);
				$header .= "00";    // terminator
			}
		}
		// Bcc (optional)
		if ($this->bcc_bin) {
			$header .= "81"; // bcc
	 	 	$header .= $this->bcc_bin;
		}
		else {
			if ($this->bcc) {
				$header .= "81"; // bcc
				if ($this->simple_enc == 0) {
					$len = strlen($this->bcc) + 2;
					encodeLengthval($len, $lenvar);
					for ($i = 0; $i < sizeof($lenvar); $i++) {
						$header .= hex_encode(chr($lenvar[$i]));
					}
					$header .= "83"; // us-ascii
				}
	 	 		$header .= hex_encode($this->bcc);
		 		$header .= "00";    // terminator
			}
		}
    // Date (optional)
		if ($this->date_bin) { // fuzzing only
			$header .= "85";  // date
			$header .= $this->date_bin;
		}
		else { // normal
			if ($this->date) {
				$header .= "85";  // date
				encodeLongInteger( $this->date, $longInt );
				$len = sizeof( $longInt );
				for ( $j = 0; $j < $len; $j++ ) {
					$header .= hex_encode($longInt[$j]);
				}
			}
		}
    // Subject (optional)
		if ($this->subject_bin) { // fuzzing only
			$header .= "96"; // subject
			$header .= $this->subject_bin;
		}
		else {
			if ($this->subject) { // normal
				$header .= "96"; // subject
				if ($this->simple_enc == 0) {
					$len = strlen($this->subject) + 2;
					encodeLengthval($len, $lenvar);
					for ($i = 0; $i < sizeof($lenvar); $i++) {
						$header .= hex_encode(chr($lenvar[$i]));
					}
					$header .= "83"; // us-ascii
				}
				$header .= hex_encode($this->subject);
				$header .= "00";    // terminator
			
			}
		}
		// MessageId (optional)
		if ($this->message_id_bin) { // fuzzing only
			$header .= "8B"; // message id
			$header .= $this->message_id_bin;
		}
		else {
			if ($this->message_id) { // normal
				$header .= "8B"; // message id
				$header .= hex_encode($this->message_id);
				$header .= "00";
			}
		}
		// msg class
		if ($this->msg_class_string) {
			$header .= "8A"; /* mes. class  */
			$header .= $this->msg_class_string;
			$header .= "00";
		}
		else {
			if ($this->msg_class) {
				$header .= "8A"; /* mes. class  */
				$header .= $this->msg_class;
			}
		}
		// msg priority
		if ($this->msg_priority) {
			$header .= "8F"; /* priority */
			print "Priority: " . $this->msg_priority . "\n";
			$header .= $this->msg_priority;
		}
		// sender visibility
		if ($this->sender_visibility) {
			$header .= "94"; /* sender visibility */
			$header .= $this->sender_visibility;
		}
		// delivery report
		if ($this->delivery_report) {
			$header .= "86"; /* delivery report  */
			$header .= $this->delivery_report;
		}
		// read reply
		if ($this->read_reply) {
			$header .= "90"; /* read reply  */
			$header .= $this->read_reply;
		}
		// delivery time
		if ($this->delivery_time_bin) { // for fuzzing only
			$header .= "87";
			$header .= $this->delivery_time_bin;
		}
		else { // normal
			if ($this->delivery_time) {
				$header .= "87";
				encodeLongInteger($this->delivery_time, $delivery);
				$len = sizeof($delivery) + 1;
				encodeLengthval($len, $lenvar2);
				for ($i = 0; $i < sizeof($lenvar2); $i++) {
					$header .= hex_encode(chr($lenvar2[$i]));
				}				
				$header .= $this->delivery_time_type; // 80=absolute 81=relative
				for ($i = 0; $i < sizeof($delivery); $i++) {
					$header .= bin2hex(chr($delivery[$i]));
				}
			}
		}
		// Expiry
		if ($this->expiry_bin) { // for fuzzing only
			$header .= "88";
			$header .= $this->expiry_bin;
		}
		else { // normal
			if ($this->expiry) {
				$header .= "88";
				encodeLongInteger($this->expiry, $expiry);
				$len = sizeof($expiry) + 1;
				encodeLengthval($len, $lenvar1);
				for ($i = 0; $i < sizeof($lenvar1); $i++) {
					$header .= hex_encode(chr($lenvar1[$i]));
				}
				$header .= $this->expiry_type; // 80=absolute 81=relative
				for ($i = 0; $i < sizeof($expiry); $i++) {
					$header .= bin2hex(chr($expiry[$i]));
				}
			}
		}
		// mms msg size (on server)
		if ($this->msg_size_bin) { // for fuzzing
			$header .= "8E"; // msg size
			$header .= $this->msg_size_bin;
		}
		else { // normal
			if ($this->msg_size) {
				$header .= "8E"; // msg size
				encodeLongInteger($this->msg_size, $msgsize);
				for ($i = 0; $i < sizeof($msgsize); $i++) {
					$header .= bin2hex(chr($msgsize[$i]));
				}
			}
		}
		
		// add text header, can really be anything
		if ($this->text_header)
		{
			$header .= $this->text_header;
		}
		
		// url / content location (only for nottification.ind)
		if ($this->url_bin) {
			$header .= "83"; // content location
			$header .= $this->url_bin;
		}
		else {
			if ($this->url) {
				$header .= "83"; // content location
				$header .= hex_encode($this->url);
				$header .= "00"; // terminator
			}
		}

		// ONLY if retrieve.conf or send.req
		if ($this->msg_type == "84" || $this->msg_type == "80") {	
    	//print "MSG type: " . $this->msg_type . "\n";
			// Content-type
			$header .= "84"; // content-type
			if ( $this->nparts == 0 )	{
				print( "MMSEncoder: No content added to message\n" );
			}
		
			$this->encodeContentType($this->start_content, $contentType );
		
			$len = strlen($contentType) + strlen($this->start_info) + strlen($this->start);
			//$len = 1;
			$header .= hex_encode(chr($len)); // length
			
			// content-type
			for ($i = 0; $i < sizeof($contentType); $i++) {
				$header .= hex_encode(chr($contentType[$i]));
			}
		
			$header .= "8A"; // Content-type-parameter: Type   
			$header .= hex_encode($this->start_info);
			$header .= "00"; // terminator
	
			$header .= "89"; // parametr Start
			$header .= hex_encode($this->start);
			$header .= "00"; // terminator
		}
		
		$this->header_data = $header;
	}

	function parseUintvar( $inp, &$uintvar )
	{
		$curp = 0;
		$uintvar = 0;
		while ( $inp[$curp] & 0x80 )
		{
			$uintvar = $uintvar << 7;
			$uintvar |= $inp[$curp] & 0x7f;
			$curp++;
		}
		$uintvar = $uintvar << 7;
		$uintvar |= $inp[$curp] & 0x7f;
	}

	function buildBody()
	{
		$body = "";
		// Write number of parts in the multipart message
		encodeUintvar( $this->nparts, $uintVar );
		for ( $j = 0; $j < sizeof( $uintVar ); $j++ ) {
			$body .= hex_encode(chr($uintVar[$j]));
		}

	  // write part-header and body
		for ( $i = 0; $i < $this->nparts; $i++ ) {
			$p = $this->parts[$i];
			$IDfile =  $p->fileID;
			$fname =  $p->sfname;

			$add_len = strlen( $fname ) + strlen( $IDfile ) +7;
			$this->encodeContentType( $p->contentType, $contentType );
			encodeUintvar( sizeof( $contentType )+$add_len, $headerLen );

			encodeUintvar( $p->dataLen, $dataLen );

			print "header len: " . (sizeof( $contentType )+$add_len) . "\n";
			// Write size of header
			for ( $j = 0; $j < sizeof( $headerLen ); $j++ ) {
				$body .= hex_encode(chr( $headerLen[$j]));
				print hex_encode(chr( $headerLen[$j]));
			}
			print "\n";

			print "data len: " .  $p->dataLen . "\n";
			// Write length of data
			for ( $j = 0; $j < sizeof( $dataLen ); $j++ ) {
				$body .= hex_encode(chr($dataLen[$j]));
				print hex_encode(chr($dataLen[$j]));
			}
			print "\n";
			$test = $dataLen;
			$this->parseUintvar($test, &$testo);
			print "data len test: " . $testo . "\n";

			// Write content type header (no other headers implemented right now)
			for ( $j = 0; $j < sizeof( $contentType ); $j++ ) {
				$body .= hex_encode(chr($contentType[$j]));
			}

      // content-id
			$body .= "C0"; // content-id .... 0x40 | 0x80 = 0xC0
			$body .= "22"; // " 
			$body .= "3C"; // <
			for ( $j = 0; $j < strlen( $IDfile ); $j++ ) {
				$body .= hex_encode(chr(ord($IDfile{$j})));
			}
			$body .= "3E"; // >
			$body .= "00"; // terminator
						
			// content name
      $body .= "8E"; // content name
			for ( $j = 0; $j < strlen(  $fname ); $j++ ) {
				$body .=  hex_encode(chr(ord($fname{$j})));
			}
			$body .= "00"; // terminator

			$body .= hex_encode($p->getBytes());

			$this->body_data = $body;
		}
	}

	function addPart( $contentType, $fileName, $shownfilename, $fileID )
	{
		if ( is_readable( $fileName ) )	{
			$p = new Part( $contentType, $fileName, $shownfilename, $fileID );
			$this->parts[ $this->nparts ] = $p;
			$this->nparts++;
		}
		else {
			print( "Could not find file: " . $fileName . "\n" );
		}
	}

	function encodeContentType( $textContentType, &$contentType )
	{
		$index = $this->findWellKnownContentType( $textContentType );
		if ( $index >= 0 ){
			$contentType[0] = 0x80 | $index;  /* Encode short-integer */
		}
		else {
			for ( $i = 0; $i < strlen( $textContentType ); $i++ ) {
				$contentType[$i] = ord( $textContentType[$i] );
			}
			/* Null terminated */
			$contentType[$i] = 0;
		}
	}

	function findWellKnownContentType( $contentType )
	{
		global $content_types;
		for ( $i = 0; $i < sizeof( $content_types ); $i++ ) {
			if ( !strcmp( $content_types[$i], $contentType ) ) {
				return $i;
			}
		}
		return -1;
	}
	
	// stuff is in HEX
	function getHeaderBytes()
	{
		return $this->header_data;
	}
	
	// stuff is in HEX
	function getBodyBytes()
	{
		return $this->body_data;
	}
}

class WAPMessage
{
	var $msg_type;
	var $header_length;
	var $transaction_id;
	var $headers;
	var $body;
	var $content_type;
	var $wap_application_id;

	function WAPMessage()
	{
		$this->msg_type = "06"; // PUSH
		$this->transaction_id = "00";
		$this->content_type = "application/vnd.wap.mms-message";
		$this->wap_application_id = "84"; // =MMS (0x04=MMS|0x80=well known field)
	}
	
	function setMsgType($msgtype)
	{
		$this->msg_type = $masgtype;
	}
	
	function setTransactionId($transid)
	{
		$this->transaction_id = $transid;
	}
	
	function setContentType($contenttype)
	{
		$this->content_type = $contenttype;
	}
	
	function setWapApplicationId($appid)
	{
		$this->wap_application_id = $appid;
	}
	
	/* only use for fuzzing!!! */
	function setHeaderLength($hlen)
	{
		$this->header_length = $hlen;
	}
	
	function setMiscHeaders($mischeaders)
	{
		$this->headers = $mischeaders;
	}
	
	function setBody($body)
	{
		$this->body = $body;
	}

	function build()
	{
		$this->message = $this->transaction_id;
		
		$this->message .= $this->msg_type;
		
		if ($this->header_length) {
			$this->message .= $this->header_length;
		}
		else {
			$len = strlen($this->content_type) + 3 + strlen($this->headers);
			encodeUintvar($len, $uintvlen);
			for ($i = 0; $i < sizeof($uintvlen); $i++) {
				$this->message .= bin2hex(chr($uintvlen[$i]));
			}
		}
		
		$this->message .= hex_encode($this->content_type);
		$this->message .= "00"; // terminator
		
		$this->message .= "AF"; // wap-application-id
		$this->message .= $this->wap_application_id;
		
		$this->message .= $this->body;
	}

	function writeToFile($filename, $hexorbin = 1)
	{
		if ($hexorbin) $data = hex2bin($this->message);
		else $data = $this->message;
		
		$fp = fopen($filename, "wb");
		fputs($fp, $data);
		fclose($fp);
	}
	
	/* returns stuff in HEX */
	function getBytes()
	{
		return $this->message;
	}
}

/*
 *  build just MMS m-notification.ind the WAPpush to transport it is build using
 *  WAPMessage
 */
class MMSNotify
{
	var $transaction_id;
	var $subject;
	var $from;
	var $from_fillin;
	var $url;
	var $url_bin;
	var $msg_class;
	var $msg_size;
	var $msg_size_bin;
	var $expiry;
	var $expiry_type;
	var $expiry_bin;
	var $message;
	
	function MMSNotify()
	{
		$this->msg_class = "80";
	}
	
	/* just some string */
	function setTransactionId($transid)
	{
		$this->transaction_id = $transid;
	}
		
	/* just some string */
	function setSubject($subject)
	{
		$this->subject = $subject;
	}
	
	function setFrom($from, $from_fillin = "80")
	{
		$this->from = $from;
		$this->from_fillin = $from_fillin;
	}
	
	/* http://host.domain/filename (can have port ...domain:80..) */
	function setUrl($url)
	{
		$this->url = $url;
	}
	
	/* message classes
	 * 80 = personal (default)
	 * 81 = ad
	 * 82 = info
	 * 83 = auto
	 */
	function setMsgClass($msgclass)
	{
		$this->msg_class = $msgclass;
	}
	
	/* message size on server (ca.) */
	function setMsgSize($msgsize, $bin = 0)
	{
		if ($bin == 1) {
			$this->msg_size_bin = $msgsize;
		}
		else {
			$this->msg_size = $msgsize;
		}
	}
		
	/* in seconds, type: 80 = absoute, 81 = relative */
	function setExpiry($expiry, $bin = 0, $expirytype = "81")
	{
		if ($bin == 1) { // fuzzing
			$this->expiry_bin = $expirybin;
		}
		else {
			$this->expiry = $expiry;
			$this->expiry_type = $expirytype;
		}
	}
	
	function build()
	{
		// this is a MMS notification.ind
		$message  = "8C"; // msg type
		$message .= "82"; // = notification.ind
		
		// transaction id
		$message .= "98";
		$message .= hex_encode($this->transaction_id);
		$message .= "00"; // terminator
		
		// mms version
		$message .= "8D"; // version
		$message .= "90"; // =1.0

		// From
	  if ($this->from) {
		 $message .= "89"; // from
		$len = strlen($this->from) + 2;
		 encodeUintvar( $len, $uintVar );
		 for ( $j = 0; $j < sizeof( $uintVar ); $j++ ) {
			$message .=  bin2hex(chr($uintVar[$j]));
		 }
		 $message .= $this->from_fillin; // value present/absent token
		 for ( $d = 0; $d < strlen( $this->from ); $d++ ) {
	 	 	$message .= bin2hex(chr(ord( $this->from{$d} )));
		 }
		 $message .= "00";    /* terminator */
		}

		// subject
		$message .= "96"; // subject
		$message .= hex_encode($this->subject);
		$message .= "00"; // terminator
	
		// mms msg class
		$message .= "8A"; // class
		$message .= $this->msg_class; // 80=personal, 81=ad, 82=info, 83=auto

		// expiry
		$message .= "88"; // expiry
		if ($this->expiry_bin) { // for fuzzing
			$message .= $this->expiry_bin;
		}
		else { // normal
			encodeLongInteger($this->expiry, $expiry);
			$message .= bin2hex(chr(sizeof($expiry) + 1));
			$message .= $this->expiry_type; // 80=absolute 81=relative
			for ($i = 0; $i < sizeof($expiry); $i++) {
				$message .= bin2hex(chr($expiry[$i]));
			}
		}

		// mms msg size (on server)
		$message .= "8E"; // msg size
		if ($this->msg_size_bin) { // for fuzzing
			$message .= $this->msg_size_bin;
		}
		else { // normal
			encodeLongInteger($this->msg_size, $msgsize);
			for ($i = 0; $i < sizeof($msgsize); $i++) {
				$message .= bin2hex(chr($msgsize[$i]));
			}
		}

		// url / content location
		$message .= "83"; // content location
		$message .= hex_encode($this->url);
		$message .= "00"; // terminator
		
		// copy to global
		$this->message = $message;
	}
	
	/* returns stuff in HEX */
	function getBytes()
	{
		return $this->message;
	}
}

/* OLD STUFF DOWN HERE */

/*
 * MMSNotifyer class
 * 
 * This class is used to send out the notification that will point the phone to the
 * MMS that is located at a certain URL. The SMS service is your simple web interface
 * using HTTP 1.1 GET. Change this to fit your system and your way of SMS sending
 */
class MMSNotifyer
{
	var $sms_host;
	var $sms_port;
	var $headers;
	var $data;
	var $length;
	var $contenttype;
	var $msgclass;
	
	
	function setContentType($type)
	{
		$this->contenttype = $type;
	}
	
	function setMsgClass($msgclass)
	{
		$this->msgclass = $msgclass;
	}
	
	function MMSNotifyer( $sms_host, $sms_port )
	{
		// MMS stuff
		$this->msgclass = "80";
			
		// SMS stuff (UDH user data header)
		$this->sms_host = $sms_host;
		$this->sms_port = $sms_port;
		
		// WSP stuff
		$this->headers = "E9";  // wsp transaction_id = something random
		$this->headers .= "06"; // type = push (0x06)
		$this->headers .= "22"; // length
		
		// move this
		$this->contenttype = hex_encode("application/vnd.wap.mms-message");
		$this->headers .= $this->contenttype;

		$this->headers .= "AF" . "84"; // wap-application-id: mms (0x04 | 0x80)
	}

	function sendNotify( $mms_url, $to, $subject, $size, $transid )
	{
		//---------- X-Mms-Message-Type ----------
		$message = "8C" . "82"; // m-notification-ind

		//---------- X-Mms-Transaction-Id ----------
		$message .= "98" . hex_encode($transid) . "00";

		//---------- X-Mms-Version ----------
		$message .= "8D" . "90"; // 1.0

		//---------- Subject ----------------
		$message .= "96";
		$message .= hex_encode( $subject ) . "00";

		//---------- X-Mms-Message-Class ----------
		$message .= "8A" . $this->msgclass; // 80 = personal, 81 = ad, 82=info, 83 = auto

		//---------- X-Mms-Message-Size ----------
		$message .= "8E" . "020329";
		//$message .= "8E" . "020B05";

		//---------- X-Mms-Expiry ----------
		$message .= "88" . "058103015180";

		//---------- X-MMS-Content-Location ----------
		$message .= "83" . hex_encode($mms_url) . "00";
		
		//print( "SMS size is: " . strlen( $this->headers . $message ) . "\n" );
		//print( "And the data part is: " . $this->headers . $message . "\n" );

		//$fp = fopen("test.bin", "wb");
		$this->$data = $this->hex2bin($this->headers . $message);
		//$data = $this->headers . $message;
		//fputs($fp, $data);
		//fclose($fp);

		//$this->httpSend( "/mmstry/sendmms1.html?phone=$to&UDH=0605040B8423F0&Data=" . 
		//		  $this->headers . $message );
	}

	function httpSend( $path )
	{
 		$connection = fsockopen( $this->sms_host, 
			$this->sms_port, 
			&$errno, 
			&$errdesc, 
			60 );

		fputs( $connection, "GET $path HTTP/1.1\r\nHost: " . $this->sms_host . 
			"\r\n\r\n");

		while (!feof($connection)) 
		{
			$myline = fgets($connection, 128);
			// possibly do something here to check reply
		}

		fclose ($connection);
	}

	function notify2file($filename)
	{
		$fp = fopen($filename, "wb");
		fputs($fp, $this->$data);
		fclose($fp);
	}
}

?>
